# @swagger-api/apidom-ns-json-schema-draft-4

`@swagger-api/apidom-ns-json-schema-draft-4` contains ApiDOM namespace specific to [JSON Schema Draft 4](https://tools.ietf.org/html/draft-wright-json-schema-00) specification.

> You might come across references to **Draft 5** a.k.a. **Wright Draft 00** ([core](https://tools.ietf.org/html/draft-wright-json-schema-00), [validation](https://datatracker.ietf.org/doc/html/draft-wright-json-schema-validation-00) and [hyper-schema](https://datatracker.ietf.org/doc/html/draft-wright-json-schema-hyperschema-00) vocabularies). There is no Draft 5 release of JSON Schema. Draft 5 refers to a no-change revision of the Draft 4 release. It does not add, remove, or change any functionality. It only updates references, makes clarifications, and fixes bugs. This package implements Draft 4 + no-change revision of Draft 5.

## Installation

You can install this package via [npm CLI](https://docs.npmjs.com/cli) by running the following command:

```sh
 $ npm install @swagger-api/apidom-ns-json-schema-draft-4
```

## JSON Schema Draft 4 namespace

JSON Schema Draft 4 namespace consists of [number of elements](https://github.com/swagger-api/apidom/tree/main/packages/apidom-ns-json-schema-draft-4/src/elements) implemented on top
of [primitive ones](https://github.com/refractproject/minim/tree/master/lib/primitives).

```js
import { createNamespace } from '@swagger-api/apidom-core';
import jsonShemaDraft4Namespace from '@swagger-api/apidom-ns-json-schema-draft-4';

const namespace = createNamespace(jsonShemaDraft4Namespace);

const objectElement = new namespace.elements.Object();
const jsonSchemaElement = new namespace.elements.JSONSchemaDraft4();
```

When namespace instance is created in this way, it will extend the base namespace
with the namespace provided as an argument.

Elements from the namespace can also be used directly by importing them.

```js
import { JSONSchemaElement, JSONReferenceElement, LinkDescriptionElement, MediaElement } from '@swagger-api/apidom-ns-json-schema-draft-4';

const jsonSchemaElement = new JSONSchemaElement();
const jsonReferenceElement = new JSONReferenceElement();
const linkDescriptionElement = new LinkDescriptionElement();
const mediaElement = new MediaElement();
```

## Predicates

This package exposes [predicates](https://github.com/swagger-api/apidom/blob/main/packages/apidom-ns-json-schema-draft-4/src/predicates.ts)
for all higher order elements that are part of this namespace.

```js
import { isJSONSchemaElement, JSONSchemaElement } from '@swagger-api/apidom-ns-json-schema-draft-4';

const jsonSchemaElement = new JSONSchemaElement();

isJSONSchemaElement(jsonSchemaElement); // => true
```

## Traversal

Traversing ApiDOM in this namespace is possible by using `visit` function from `apidom` package.
This package comes with its own [keyMap](https://github.com/swagger-api/apidom/blob/main/packages/apidom-ns-json-schema-draft-4/src/traversal/visitor.ts#L11) and [nodeTypeGetter](https://github.com/swagger-api/apidom/blob/main/packages/apidom-ns-json-schema-draft-4/src/traversal/visitor.ts#L4).
To learn more about these `visit` configuration options please refer to [@swagger-api/apidom-ast documentation](https://github.com/swagger-api/apidom/blob/main/packages/apidom-ast/README.md#visit).

```js
import { visit } from '@swagger-api/apidom-core';
import { JSONSchemaElement, keyMap, getNodeType } from '@swagger-api/apidom-ns-json-schema-draft-4';

const element = new JSONSchemaElement();

const visitor = {
  JSONSchemaDraft4Element(jsonSchemaElement) {
    console.dir(jsonSchemaElement);
  },
};

visit(element, visitor, { keyMap, nodeTypeGetter: getNodeType });
```

## Refractors

Refractor is a special layer inside the namespace that can transform either JavaScript structures
or generic ApiDOM structures into structures built from elements of this namespace.

**Refracting JavaScript structures**:

```js
import { MediaElement } from '@swagger-api/apidom-ns-json-schema-draft-4';

const object = {
  binaryEncoding: 'base64',
  type: 'image/png',
};

MediaElement.refract(object); // => MediaElement({ binaryEncoding, type })
```

**Refracting generic ApiDOM structures**:

```js
import { ObjectElement } from '@swagger-api/apidom-core';
import { MediaElement } from '@swagger-api/apidom-ns-json-schema-draft-4';

const objectElement = new ObjectElement({
  binaryEncoding: 'base64',
  type: 'image/png',
});

MediaElement.refract(objectElement); // => MediaElement({ binaryEncoding = 'base64', type = 'image/png' })
```

### Refractor plugins

Refractors can accept plugins as a second argument of refract static method.

```js
import { ObjectElement } from '@swagger-api/apidom-core';
import { MediaElement } from '@swagger-api/apidom-ns-json-schema-draft-4';

const objectElement = new ObjectElement({
  binaryEncoding: 'base64',
  type: 'image/png',
});

const plugin = ({ predicates, namespace }) => ({
  name: 'plugin',
  pre() {
      console.dir('runs before traversal');
  },
  visitor: {
    MediaElement(mediaElement) {
      mediaElement.type = 'image/gif';
    },
  },
  post() {
      console.dir('runs after traversal');
  },
});

MediaElement.refract(objectElement, { plugins: [plugin] }); // => MediaElement({ binaryEncoding = 'base64', type = 'image/gif' })
```

You can define as many plugins as needed to enhance the resulting namespaced ApiDOM structure.
If multiple plugins with the same visitor method are defined, they run in parallel (just like in Babel).

#### Replace Empty Element plugin

This plugin is specific to YAML 1.2 format, which allows defining key-value pairs with empty key,
empty value, or both. If the value is not provided in YAML format, this plugin compensates for
this missing value with the most appropriate semantic element type.

```js
import { parse } from '@swagger-api/apidom-parser-adapter-yaml-1-2';
import { refractorPluginReplaceEmptyElement, JSONSchemaElement } from '@swagger-api/apidom-ns-json-schema-draft-4';

const yamlDefinition = `
$schema: 'http://json-schema.org/draft-04/schema#'
additionalProperties:
`;
const apiDOM = await parse(yamlDefinition);
const jsonSchemaElement = JSONSchemaElement.refract(apiDOM.result, {
  plugins: [refractorPluginReplaceEmptyElement()],
});

// =>
// (JSONSchemaDraft4Element
//   (MemberElement
//     (StringElement)
//     (StringElement))
//   (MemberElement
//     (StringElement)
//     (JSONSchemaDraft4Element)))

// => without the plugin the result would be as follows:
// (JSONSchemaDraft4Element
//   (MemberElement
//     (StringElement)
//     (StringElement))
//   (MemberElement
//     (StringElement)
//     (StringElement)))
```

## Implementation progress

Only fully implemented specification objects should be checked here.

- [x] [JSON Schema Object](https://tools.ietf.org/html/draft-wright-json-schema-00)
- [x] [JSON Reference Object](https://datatracker.ietf.org/doc/html/draft-pbryan-zyp-json-ref-03)
- [x] [Link Description Object](https://datatracker.ietf.org/doc/html/draft-wright-json-schema-hyperschema-00#section-5)
- [x] [Media Object](https://datatracker.ietf.org/doc/html/draft-wright-json-schema-hyperschema-00#section-4.3)
