/*#############################################################################
#                                                                             #
# fireperf - A network benchmarking tool                                      #
# Copyright (C) 2021 IPFire Development Team                                  #
#                                                                             #
# This program is free software: you can redistribute it and/or modify        #
# it under the terms of the GNU General Public License as published by        #
# the Free Software Foundation, either version 3 of the License, or           #
# (at your option) any later version.                                         #
#                                                                             #
# This program is distributed in the hope that it will be useful,             #
# but WITHOUT ANY WARRANTY; without even the implied warranty of              #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               #
# GNU General Public License for more details.                                #
#                                                                             #
# You should have received a copy of the GNU General Public License           #
# along with this program.  If not, see <http://www.gnu.org/licenses/>.       #
#                                                                             #
#############################################################################*/

#include <stdlib.h>
#include <sys/random.h>

#include "logging.h"
#include "main.h"
#include "random.h"

struct fireperf_random_pool* fireperf_random_pool_create(struct fireperf_config* conf, size_t size) {
	struct fireperf_random_pool* pool = calloc(1, sizeof(*pool));
	if (!pool)
		return NULL;

	pool->size = size;

	// Allocate the data array
	pool->data = malloc(pool->size);
	if (!pool->data)
		goto ERROR;

	size_t offset = 0;
	while (offset < pool->size) {
		offset += getrandom(pool->data + offset, pool->size - offset, 0);
	}

	DEBUG(conf, "Allocated random pool of %zu bytes(s)\n", pool->size);

	return pool;

ERROR:
	fireperf_random_pool_free(pool);

	return NULL;
}

void fireperf_random_pool_free(struct fireperf_random_pool* pool) {
	if (pool->data)
		free(pool->data);

	free(pool);
}

const char* fireperf_random_pool_get_slice(struct fireperf_random_pool* pool, size_t size) {
	if (size > pool->size)
		return NULL;

	// Find a random value between the start and end of
	// the data region that is at least size bytes long.
	off_t offset = random() % (pool->size - size);

	return pool->data + offset;
}
