/*#############################################################################
#                                                                             #
# fireperf - A network benchmarking tool                                      #
# Copyright (C) 2021 IPFire Development Team                                  #
#                                                                             #
# This program is free software: you can redistribute it and/or modify        #
# it under the terms of the GNU General Public License as published by        #
# the Free Software Foundation, either version 3 of the License, or           #
# (at your option) any later version.                                         #
#                                                                             #
# This program is distributed in the hope that it will be useful,             #
# but WITHOUT ANY WARRANTY; without even the implied warranty of              #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               #
# GNU General Public License for more details.                                #
#                                                                             #
# You should have received a copy of the GNU General Public License           #
# along with this program.  If not, see <http://www.gnu.org/licenses/>.       #
#                                                                             #
#############################################################################*/

#include <errno.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <sys/epoll.h>
#include <unistd.h>

#include "client.h"
#include "logging.h"
#include "main.h"
#include "random.h"
#include "util.h"

// Set to one when the timeout has expired
static int timeout_expired = 0;

static void handle_SIGALRM(int signal) {
	switch (signal) {
		// Terminate after timeout has expired
		case SIGALRM:
			timeout_expired = 1;
			break;
	}
}

static int open_connection(struct fireperf_config* conf) {
	// Open a new socket
	int fd = socket(AF_INET6, SOCK_STREAM|SOCK_NONBLOCK|SOCK_CLOEXEC, 0);
	if (fd < 0) {
		ERROR(conf, "Could not open socket: %s\n", strerror(errno));
		goto ERROR;
	}

	// Chose a random port
	int port = conf->port + (random() % conf->listening_sockets);

	DEBUG(conf, "Opening socket %d (port %d)...\n", fd, port);

	// Define the peer
	struct sockaddr_in6 peer = {
		.sin6_family = AF_INET6,
		.sin6_addr = conf->address,
		.sin6_port = htons(port),
	};

	// Set socket buffer sizes
	int r = set_socket_buffer_sizes(conf, fd);
	if (r)
		goto ERROR;

	// Connect to the server
	r = connect(fd, &peer, sizeof(peer));
	if (r && (errno != EINPROGRESS)) {
		ERROR(conf, "Could not connect to server: %s\n", strerror(errno));
		goto ERROR;
	}

	return fd;

ERROR:
	if (fd > 0)
		close(fd);

	return -1;
}

int fireperf_client(struct fireperf_config* conf, struct fireperf_stats* stats,
		int epollfd, int timerfd) {
	DEBUG(conf, "Launching " PACKAGE_NAME " in client mode\n");

	int r = 1;

	struct epoll_event ev = {
		.events = EPOLLIN,
	};
	struct epoll_event events[EPOLL_MAX_EVENTS];

	// Let us know when the socket is ready for receiving data
	if (!conf->keepalive_only)
		ev.events |= EPOLLIN;

	// In duplex mode, we send data, too
	if (conf->duplex)
		ev.events |= EPOLLOUT;

	DEBUG(conf, "Opening %lu connections...\n", conf->parallel);

	// Configure timeout if set
	if (conf->timeout) {
		// Register signal handler
		signal(SIGALRM, handle_SIGALRM);

		alarm(conf->timeout);
	}

	DEBUG(conf, "Entering main loop...\n");

	while (!conf->terminated && !timeout_expired) {
		// Open connections
		while (stats->open_connections < conf->parallel) {
			int fd = open_connection(conf);
			if (fd < 0)
				continue;

			ev.data.fd = fd;

			if (epoll_ctl(epollfd, EPOLL_CTL_ADD, fd, &ev)) {
				ERROR(conf, "Could not add socket file descriptor to epoll(): %s\n",
					strerror(errno));
				goto ERROR;
			}

			stats->open_connections++;
			stats->connections++;
		}

		int fds = epoll_wait(epollfd, events, EPOLL_MAX_EVENTS, -1);
		if (fds < 1) {
			// We terminate gracefully when we receive a signal
			if (errno == EINTR)
				break;

			ERROR(conf, "epoll_wait() failed: %s\n", strerror(errno));
			goto ERROR;
		}

		for (int i = 0; i < fds; i++) {
			int fd = events[i].data.fd;

			// What type of event are we handling?

			// Handle timer events
			if (fd == timerfd) {
				uint64_t expirations;

				// Read from the timer to disarm it
				ssize_t bytes_read = read(timerfd, &expirations, sizeof(expirations));
				if (bytes_read <= 0) {
					ERROR(conf, "Could not read from timerfd: %s\n", strerror(errno));
					goto ERROR;
				}

				r = fireperf_dump_stats(conf, stats, FIREPERF_MODE_CLIENT);
				if (r)
					goto ERROR;

			// Handle connection sockets
			} else {
				// Has the socket been disconnected?
				if (events[i].events & EPOLLHUP) {
					if (epoll_ctl(epollfd, EPOLL_CTL_DEL, fd, NULL)) {
						ERROR(conf, "Could not remove socket file descriptor from epoll(): %s\n",
							strerror(errno));
						goto ERROR;
					}

					close(fd);

					stats->open_connections--;
				} else {
					// Close connections immediately when -x is set
					if (conf->close) {
						DEBUG(conf, "Closing connection %d\n", fd);
						close(fd);

						stats->open_connections--;
						continue;
					}

					// Handle incoming data
					if (events[i].events & EPOLLIN) {
						r = handle_connection_recv(conf, stats, fd);
						if (r < 0)
							goto ERROR;
					}

					// Handle outgoing data
					if (events[i].events & EPOLLOUT) {
						r = handle_connection_send(conf, stats, fd);
						if (r)
							goto ERROR;
					}
				}
			}
		}
	}

	// All okay
	r = 0;

ERROR:
	return r;
}
