/*#############################################################################
#                                                                             #
# fireperf - A network benchmarking tool                                      #
# Copyright (C) 2021 IPFire Development Team                                  #
#                                                                             #
# This program is free software: you can redistribute it and/or modify        #
# it under the terms of the GNU General Public License as published by        #
# the Free Software Foundation, either version 3 of the License, or           #
# (at your option) any later version.                                         #
#                                                                             #
# This program is distributed in the hope that it will be useful,             #
# but WITHOUT ANY WARRANTY; without even the implied warranty of              #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               #
# GNU General Public License for more details.                                #
#                                                                             #
# You should have received a copy of the GNU General Public License           #
# along with this program.  If not, see <http://www.gnu.org/licenses/>.       #
#                                                                             #
#############################################################################*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>

#include "util.h"

static const char* suffixes_bytes[] = { "B", "KiB", "MiB", "GiB", "TiB", NULL };
static const char* suffixes_bits[]  = { "b", "Kb", "Mb", "Gb", "Tb", NULL };

char* format_size(ssize_t size, int unit) {
	const char** suffixes;
	unsigned int divisor;

	switch (unit) {
		case FIREPERF_FORMAT_BYTES:
			suffixes = suffixes_bytes;
			divisor = 1024;
			break;

		case FIREPERF_FORMAT_BITS:
			suffixes = suffixes_bits;
			divisor = 1000;
			break;

		// Invalid input
		default:
			return NULL;
	}

	const char** suffix;
	char* retval = NULL;

	// Convert into double
	double s = size;

	for (suffix = suffixes; *suffix; suffix++) {
		if (abs(s) < divisor)
			break;

		s /= divisor;
	}

	if (!*suffix)
		return NULL;

	// Format the output string
	int r = asprintf(&retval, "%.02f %s", s, *suffix);
	if (r < 0)
		return NULL;

	return retval;
}

static char __timespec[20];

const char* format_timespec(const struct timespec* t) {
	// Convert to local time
	struct tm* tm = localtime(&t->tv_sec);

	size_t s = strftime(__timespec, sizeof(__timespec), "%F %T", tm);

	if (s)
		return __timespec;

	return NULL;
}

unsigned long timespec_delta(struct timespec* t1, struct timespec* t2) {
	// Compute delta in milliseconds
	return (
		((t1->tv_sec * 1000) + (t1->tv_nsec / 1000000))
		-
		((t2->tv_sec * 1000) + (t2->tv_nsec / 1000000))
	) / 1000.0;
}
