#!/usr/bin/python
###############################################################################
#                                                                             #
# Fireinfo                                                                    #
# Copyright (C) 2010, 2011 IPFire Team (www.ipfire.org)                       #
#                                                                             #
# This program is free software: you can redistribute it and/or modify        #
# it under the terms of the GNU General Public License as published by        #
# the Free Software Foundation, either version 3 of the License, or           #
# (at your option) any later version.                                         #
#                                                                             #
# This program is distributed in the hope that it will be useful,             #
# but WITHOUT ANY WARRANTY; without even the implied warranty of              #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               #
# GNU General Public License for more details.                                #
#                                                                             #
# You should have received a copy of the GNU General Public License           #
# along with this program.  If not, see <http://www.gnu.org/licenses/>.       #
#                                                                             #
###############################################################################

import _fireinfo
import system

class Hypervisor(object):
	def __init__(self):
		self.__hypervisor = _fireinfo.detect_hypervisor()

	@property
	def system(self):
		"""
			Return the current instance of the System class.

			We need to do that as a property because otherwise
			we get a recursion.
		"""
		return system.System()

	@property
	def vendor(self):
		"""
			Returns the name of the hypervisor vendor.
		"""
		# Citrix Xen says it is Microsoft Hv.
		if self.__hypervisor == "Microsoft" and self.system.bios_vendor == "Xen":
			return "Xen"

		# Some of the hypervisors can be detected in a right way.
		# We can return them at this place.
		if self.__hypervisor:
			return self.__hypervisor

		# Check DMI and BIOS information...
		if self.__bios_is_bochs():
			return "Bochs"
		elif self.__bios_is_microsoft():
			return "Microsoft"
		elif self.__bios_is_qemu():
			return "Qemu"
		elif self.__bios_is_virtualbox():
			return "VirtualBox"
		elif self.__bios_is_vmware():
			return "VMWare"
		elif self.__bios_is_xen():
			return "Xen"

	@property
	def virtual(self):
		"""
			Returns true if the host is running in a virtual environment.
			Otherwise: false.
		"""
		if self.vendor:
			return True

		return False

	def __bios_is_bochs(self):
		"""
			Check for Bochs emulator.
		"""
		return self.system.bios.check_vendor("Bochs")

	def __bios_is_microsoft(self):
		"""
			Check for Microsoft hypervisor.
		"""
		return self.system.bios.check_vendor("Microsoft Corporation")

	def __bios_is_qemu(self):
		"""
			Check for qemu emulator.
		"""
		return self.system.bios.check_vendor("QEMU")

	def __bios_is_virtualbox(self):
		"""
			Check for virtualbox hypervisor by comparing the bios vendor string
			to "innotek GmbH".
		"""
		return self.system.bios.check_vendor("innotek GmbH")

	def __bios_is_vmware(self):
		if self.system.bios.check_vendor("VMware-"):
			return True
		elif self.system.bios.check_vendor("VMW"):
			return True

		return False

	def __bios_is_xen(self):
		return self.system.bios.check_vendor("Xen")


if __name__ == "__main__":
	h = Hypervisor()

	print "Vendor:", h.vendor
	print "Virtual:", h.virtual
