#!/usr/bin/python
###############################################################################
#                                                                             #
# Fireinfo                                                                    #
# Copyright (C) 2010, 2011 IPFire Team (www.ipfire.org)                       #
#                                                                             #
# This program is free software: you can redistribute it and/or modify        #
# it under the terms of the GNU General Public License as published by        #
# the Free Software Foundation, either version 3 of the License, or           #
# (at your option) any later version.                                         #
#                                                                             #
# This program is distributed in the hope that it will be useful,             #
# but WITHOUT ANY WARRANTY; without even the implied warranty of              #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               #
# GNU General Public License for more details.                                #
#                                                                             #
# You should have received a copy of the GNU General Public License           #
# along with this program.  If not, see <http://www.gnu.org/licenses/>.       #
#                                                                             #
###############################################################################

import os.path

PROC_CPUINFO = "/proc/cpuinfo"
SYS_CLASS_CPUID = "/sys/class/cpuid/cpu%d"

class CPU(object):
	"""
		A class that represents the first CPU in a system.

		We get all information form the first CPU (or core) and assume that
		all other ones are equal.
	"""

	__cpuinfo = {}

	def __init__(self):
		"""
			Initialize this class by reading all data from /proc/cpuinfo.
		"""
		self.read_cpuinfo()

	def read_cpuinfo(self):
		"""
			Read information from PROC_CPUINFO and store
			it into a dictionary self.__cpuinfo.
		"""
		f = open(PROC_CPUINFO)
		while True:
			line = f.readline()

			if not line:
				break

			try:
				key, val = line.split(":", 1)
			except ValueError:
				# We got a line without key, pass that.
				pass

			key = key.strip().replace(" ", "_")
			val = val.strip()

			self.__cpuinfo[key] = val

		f.close()

	@property
	def bogomips(self):
		"""
			Return the bogomips of this CPU.
		"""
		return float(self.__cpuinfo["bogomips"])

	@property
	def model(self):
		"""
			Return the model id of this CPU.
		"""
		return int(self.__cpuinfo["model"])

	@property
	def model_string(self):
		"""
			Return the model string of this CPU.
		"""
		return self.__cpuinfo["model_name"]

	@property
	def vendor(self):
		"""
			Return the vendor string of this CPU.
		"""
		return self.__cpuinfo["vendor_id"]

	@property
	def stepping(self):
		"""
			Return the stepping id of this CPU.
		"""
		return int(self.__cpuinfo["stepping"])

	@property
	def flags(self):
		"""
			Return all flags of this CPU.
		"""
		return self.__cpuinfo["flags"].split()

	@property
	def speed(self):
		"""
			Return the speed (in MHz) of this CPU.
		"""
		return float(self.__cpuinfo["cpu_MHz"])

	@property
	def family(self):
		"""
			Return the family id of this CPU.
		"""
		return int(self.__cpuinfo["cpu_family"])
	
	@property
	def count(self):
		"""
			Count number of CPUs (cores).
		"""
		i = 0
		while (os.path.exists(SYS_CLASS_CPUID % i)):
			i += 1
		return i


if __name__ == "__main__":
	c = CPU()

	print "Vendor:", c.vendor
	print "Model:", c.model
	print "Stepping:", c.stepping
	print "Flags:", c.flags
	print "Bogomips:", c.bogomips
	print "Speed:", c.speed
	print "Hypervisor:", c.hypervisor
	print "Virtype:", c.virtype
	print "Family:", c.family
	print "Count:", c.count
	print "Model string:", c.model_string
