#!/usr/bin/python

import hashlib
import json
import os
import string

import cpu
import device
import hypervisor

class System(object):

	def __init__(self):
		# find all devices
		self.devices = []
		self.scan()
		self.cpu = cpu.CPU()
		self.hypervisor = hypervisor.Hypervisor()

	def profile(self):
		p = {
			"public_id" : self.public_id,
			"private_id" : self.private_id,
			
			"arch" : self.arch,
			"language" : self.language,
			"release" : self.release,
			"memory" : self.memory,
			"kernel" : self.kernel,
			"root_size" : self.root_size,
			"devices" : [],

			"virtual" : self.virtual,
			}
			
		for device in self.devices:
			p["devices"].append({
				"subsystem" : device.subsystem.lower(), 
				"vendor" : device.vendor.lower(), 
				"model" : device.model.lower(), 
				"deviceclass" : device.deviceclass
			})
		
		p["cpu"] = {
			"vendor" : self.cpu.vendor,
			"model" : self.cpu.model,
			"stepping" : self.cpu.stepping,
			"flags" : self.cpu.flags,
			"bogomips" : self.cpu.bogomips,
			"speed" : self.cpu.speed,
			"family" : self.cpu.family,
			"count" : self.cpu.count				
		}

		# Only append hypervisor information if we are virtualized.
		if self.virtual:
			p["hypervisor"] = {
				"type"   : self.hypervisor.type,
				"vendor" : self.hypervisor.vendor,
			}

		return p
				
		
	@property
	def arch(self):
		return os.uname()[4]

	@property
	def public_id(self):
		"""
			This returns a globally (hopefully) ID to identify the host
			later (by request) in the database.
		"""
		return hashlib.sha1(self._unique_id).hexdigest()
	
	@property
	def private_id(self):
		"""
			The private ID is built out of the _unique_id and used to
			permit a host to do changes on the database.

			No one could ever guess this without access to the host.
		"""
		return hashlib.sha1("%s" % reversed(self._unique_id)).hexdigest()

	@property
	def _unique_id(self):
		"""
			This is a helper ID which is generated out of some hardware information
			that is considered to be constant over a PC's lifetime.

			None of the data here is ever sent to the server.
		"""
		return "123456789" # XXX just a dummy

	@property
	def language(self):
		# Return "unknown" if settings file does not exist.
		filename = "/var/ipfire/main/settings"
		if not os.path.exists(filename):
			return "unknown"

		with open(filename, "r") as f:
			for line in f.readlines():
				key, val = line.split("=", 1)
				if key=="LANGUAGE":
					return val.strip()

	@property
	def release(self):
		with open("/etc/system-release", "r") as f:
			return f.read().strip()
		
	@property
	def memory(self):
		with open("/proc/meminfo", "r") as f:
			firstline = f.readline().strip()
			return firstline.split()[1]

	@property
	def kernel(self):
		return os.uname()[2]

	@property
	def root_disk(self):
		with open("/etc/mtab", "r") as f:
			dev, mountpoint, rest = f.readline().split(" ",2)
			if mountpoint == "/":
				dev = dev[5:]
				# cut off all digits at end of string
				while dev[-1] in string.digits:
					dev = dev[:-1]  
				return dev	

	@property
	def root_size(self):
		path="/sys/block/%s/size" %self.root_disk
		if not os.path.exists(path):
			return
		with open(path, "r") as f:
			return int(f.readline())*512/1024
					
	def scan(self):
		toscan = (("/sys/bus/pci/devices", device.PCIDevice),
		("/sys/bus/usb/devices", device.USBDevice))
		for path, cls in toscan:
			dirlist = os.listdir(path)
			for dir in dirlist:
				self.devices.append(cls(os.path.join(path, dir)))

	@property
	def virtual(self):
		"""
			Say if the host is running in a virtual environment.
		"""
		return self.hypervisor.virtual

		

if __name__ == "__main__":
	s=System()
	print s.arch
	print s.language
	print s.release
	print s.memory
	print s.kernel
	print s.root_disk
	print s.root_size
	print "------------\n", s.devices, "\n------------\n"
	print json.dumps(s.profile(), sort_keys=True, indent=4)
