/*
 * fireinfo.c
 *
 * Copyright (C) 2010 IPFire.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <Python.h>

#include <fcntl.h>
#include <linux/hdreg.h>
#include <stdbool.h>
#include <sys/ioctl.h>

/*
	Big parts of this were taken from
	http://git.kernel.org/?p=utils/util-linux-ng/util-linux-ng.git;a=blob;f=sys-utils/lscpu.c
*/

/* /sys paths */
#define _PATH_PROC_XEN		"/proc/xen"
#define _PATH_PROC_XENCAP	_PATH_PROC_XEN "/capabilities"
#define _PATH_PROC_PCIDEVS	"/proc/bus/pci/devices"

/* Used for the vmware hypervisor port detection */
#define VMWARE_HYPERVISOR_MAGIC 0x564D5868
#define VMWARE_HYPERVISOR_PORT  0x5658

#define VMWARE_PORT_CMD_GETVERSION      10

#define VMWARE_PORT(cmd, eax, ebx, ecx, edx)                            \
		__asm__("inl (%%dx)" :                                          \
						"=a"(eax), "=c"(ecx), "=d"(edx), "=b"(ebx) :    \
						"0"(VMWARE_HYPERVISOR_MAGIC),                   \
						"1"(VMWARE_PORT_CMD_##cmd),                     \
						"2"(VMWARE_HYPERVISOR_PORT), "3"(UINT_MAX) :    \
						"memory");

/* virtualization types */
enum {
	VIRT_NONE	= 0,
	VIRT_PARA,
	VIRT_FULL
};
const char *virt_types[] = {
	[VIRT_NONE]	= "none",
	[VIRT_PARA]	= "para",
	[VIRT_FULL]	= "full"
};

/* hypervisor vendors */
enum {
	HYPER_NONE	= 0,
	HYPER_XEN,
	HYPER_KVM,
	HYPER_MSHV,
	HYPER_VMWARE
};
const char *hv_vendors[] = {
	[HYPER_NONE]	= NULL,
	[HYPER_XEN]	= "Xen",
	[HYPER_KVM]	= "KVM",
	[HYPER_MSHV]	= "Microsoft",
	[HYPER_VMWARE]  = "VMWare"
};

struct hypervisor_desc {
	int hyper; 		/* hypervisor vendor ID */
	int virtype;	/* VIRT_PARA|FULL|NONE ? */
};

static size_t sysrootlen;
static char pathbuf[PATH_MAX];

static FILE *path_fopen(const char *mode, const char *path, ...)
		__attribute__ ((__format__ (__printf__, 2, 3)));
static int path_exist(const char *path, ...)
		__attribute__ ((__format__ (__printf__, 1, 2)));

static const char *
path_vcreate(const char *path, va_list ap)
{
	if (sysrootlen)
		vsnprintf(pathbuf + sysrootlen,
			  sizeof(pathbuf) - sysrootlen, path, ap);
	else
		vsnprintf(pathbuf, sizeof(pathbuf), path, ap);
	return pathbuf;
}

static FILE *
path_vfopen(const char *mode, const char *path, va_list ap)
{
	const char *p = path_vcreate(path, ap);

	return fopen(p, mode);
}

static FILE *
path_fopen(const char *mode, const char *path, ...)
{
	FILE *fd;
	va_list ap;

	va_start(ap, path);
	fd = path_vfopen(mode, path, ap);
	va_end(ap);

	return fd;
}

static int
path_exist(const char *path, ...)
{
	va_list ap;
	const char *p;

	va_start(ap, path);
	p = path_vcreate(path, ap);
	va_end(ap);

	return access(p, F_OK) == 0;
}

static int
has_pci_device(int vendor, int device)
{
	FILE *f;
	int num, fn, ven, dev;
	int res = 1;

	f = path_fopen("r", _PATH_PROC_PCIDEVS);
	if (!f)
		return 0;

	 /* for more details about bus/pci/devices format see
	  * drivers/pci/proc.c in linux kernel
	  */
	while(fscanf(f, "%02x%02x\t%04x%04x\t%*[^\n]",
			&num, &fn, &ven, &dev) == 4) {

		if (ven == vendor && dev == device)
			goto found;
	}

	res = 0;
found:
	fclose(f);
	return res;
}

#if defined(__x86_64__) || defined(__i386__)

/*
 * This CPUID leaf returns the information about the hypervisor.
 * EAX : maximum input value for CPUID supported by the hypervisor.
 * EBX, ECX, EDX : Hypervisor vendor ID signature. E.g. VMwareVMware.
 */
#define HYPERVISOR_INFO_LEAF   0x40000000

static inline void
cpuid(unsigned int op, unsigned int *eax, unsigned int *ebx,
			 unsigned int *ecx, unsigned int *edx)
{
	__asm__(
#if defined(__PIC__) && defined(__i386__)
		/* x86 PIC cannot clobber ebx -- gcc bitches */
		"pushl %%ebx;"
		"cpuid;"
		"movl %%ebx, %%esi;"
		"popl %%ebx;"
		: "=S" (*ebx),
#else
		"cpuid;"
		: "=b" (*ebx),
#endif
		  "=a" (*eax),
		  "=c" (*ecx),
		  "=d" (*edx)
		: "1" (op), "c"(0));
}

static void
read_hypervisor_cpuid(struct hypervisor_desc *desc)
{
	unsigned int eax = 0, ebx = 0, ecx = 0, edx = 0;
	char hyper_vendor_id[13];

	memset(hyper_vendor_id, 0, sizeof(hyper_vendor_id));

	cpuid(HYPERVISOR_INFO_LEAF, &eax, &ebx, &ecx, &edx);
	memcpy(hyper_vendor_id + 0, &ebx, 4);
	memcpy(hyper_vendor_id + 4, &ecx, 4);
	memcpy(hyper_vendor_id + 8, &edx, 4);
	hyper_vendor_id[12] = '\0';

	if (!hyper_vendor_id[0])
		return;

	if (!strncmp("XenVMMXenVMM", hyper_vendor_id, 12))
		desc->hyper = HYPER_XEN;
	else if (!strncmp("KVMKVMKVM", hyper_vendor_id, 9))
		desc->hyper = HYPER_KVM;
	else if (!strncmp("Microsoft Hv", hyper_vendor_id, 12))
		desc->hyper = HYPER_MSHV;
	else if (!strncmp("VMwareVMware", hyper_vendor_id, 12))
		desc->hyper = HYPER_VMWARE;
}

#else	/* ! __x86_64__ */
static void
read_hypervisor_cpuid(struct hypervisor_desc *desc)
{
}
#endif

static void
read_hypervisor(struct hypervisor_desc *desc)
{
	read_hypervisor_cpuid(desc);

	if (desc->hyper)
		/* hvm */
		desc->virtype = VIRT_FULL;

	else if (path_exist(_PATH_PROC_XEN)) {
		/* Xen para-virt or dom0 */
		FILE *fd = path_fopen("r", _PATH_PROC_XENCAP);
		int dom0 = 0;

		if (fd) {
			char buf[256];

			if (fscanf(fd, "%s", buf) == 1 &&
			    !strcmp(buf, "control_d"))
				dom0 = 1;
			fclose(fd);
		}
		desc->virtype = dom0 ? VIRT_NONE : VIRT_PARA;
		desc->hyper = HYPER_XEN;

	} else if (has_pci_device(0x5853, 0x0001)) {
		/* Xen full-virt on non-x86_64 */
		desc->hyper = HYPER_XEN;
		desc->virtype = VIRT_FULL;
	}
}

static void
read_harddisk_serial(char *device, char *serial) {
	static struct hd_driveid hd;
	int fd;

	if ((fd = open(device, O_RDONLY | O_NONBLOCK)) < 0) {
	    return;
	}

	if (!ioctl(fd, HDIO_GET_IDENTITY, &hd)) {
	    strncpy(serial, (const char *)hd.serial_no, 20);
	}
}

static bool
is_virtualized() {
	unsigned int eax, ebx, ecx, edx;

	cpuid(0x1, &eax, &ebx, &ecx, &edx);

	/*
		Bitwise detection of the 31st bit.
		This indicates if a host runs in a virtual environment.
	*/
	if (ecx & (1<<31))
		return true;

	return false;
}

int
hypervisor_port_check(void) {
        uint32_t eax, ebx, ecx, edx;
        VMWARE_PORT(GETVERSION, eax, ebx, ecx, edx);
        if (ebx == VMWARE_HYPERVISOR_MAGIC)
                return 1; // Success - running under VMware
        else
                return 0;
}

static PyObject *
do_get_hypervisor() {
	/*
		Get hypervisor from the cpuid command.
	*/
	struct hypervisor_desc _desc, *desc = &_desc;
	memset(desc, 0, sizeof(*desc));

	read_hypervisor(desc);

	PyObject *d = PyDict_New();
	PyObject *o;

	/* Hypervisor */
	if (desc->hyper == HYPER_NONE) {
		o = Py_None;
	} else {
		o = PyString_FromString((const char *)hv_vendors[desc->hyper]);
	}
	PyDict_SetItemString(d, "hypervisor", o);

	/* Virtualization type */
	if (desc->virtype == VIRT_NONE) {
		o = Py_None;
	} else {
		o = PyString_FromString((const char *)virt_types[desc->virtype]);
	}
	PyDict_SetItemString(d, "virtype", o);

	return d;
}

static PyObject *
do_is_virtualized() {
	/*
		Python wrapper around is_virtualized().
	*/

	if (is_virtualized())
		return Py_True;

	return Py_False;
}

static PyObject *
do_get_harddisk_serial(PyObject *o, PyObject *args) {
	/*
		Python wrapper around read_harddisk_serial.
	*/

	char serial[21];
	memset(serial, 0, sizeof(serial));

	char *device;
	if (!PyArg_ParseTuple(args, "s", &device))
		return NULL;

	read_harddisk_serial(device, serial);

	if (serial[0])
		return PyString_FromString(serial);

	return Py_None;
}

static PyObject *
do_hypervisor_port_check() {
	/*
		Python wrapper around hypervisor_port_check().
	*/

	if (hypervisor_port_check())
		return Py_True;

	return Py_False;
}

static PyMethodDef fireinfoModuleMethods[] = {
	{ "get_hypervisor", (PyCFunction) do_get_hypervisor, METH_NOARGS, NULL },
	{ "is_virtualized", (PyCFunction) do_is_virtualized, METH_NOARGS, NULL },
	{ "get_harddisk_serial", (PyCFunction) do_get_harddisk_serial, METH_VARARGS, NULL },
	{ "vmware_hypervisor_port_check", (PyCFunction) do_hypervisor_port_check, METH_NOARGS, NULL },
	{ NULL, NULL, 0, NULL }
};

void init_fireinfo(void) {
	PyObject *m;

	m = Py_InitModule("_fireinfo", fireinfoModuleMethods);
}
