#ifndef ALGO_COBALT___TRACEBACK__HPP
#define ALGO_COBALT___TRACEBACK__HPP

/* $Id: traceback.hpp 103491 2007-05-04 17:18:18Z kazimird $
* ===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's offical duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================*/

/*****************************************************************************

File name: traceback.hpp

Author: Jason Papadopoulos

Contents: Interface for CEditScript class

******************************************************************************/

/// @file traceback.hpp
/// Interface for CEditScript class

#include <objects/seqalign/Dense_seg.hpp>
#include <objects/seqalign/Dense_diag.hpp>
#include <algo/blast/core/gapinfo.h>
#include <algo/align/nw/nw_aligner.hpp>

#include <algo/cobalt/base.hpp>
#include <algo/cobalt/seq.hpp>

BEGIN_NCBI_SCOPE
BEGIN_SCOPE(cobalt)

/// Interface for the traceback from blast hits
class NCBI_COBALT_EXPORT CEditScript
{
public:
    /// Generate empty traceback
    ///
    CEditScript() {}

    /// Generate traceback from a blast alignment
    /// @param blast_tback Edit script for the alignment,
    ///                    computed by the blast engine [in]
    ///
    CEditScript(GapEditScript *blast_tback);

    /// Generate traceback from a Dense_seg
    /// @param denseg Dense_seg representing a single pairwise alignment [in]
    ///
    CEditScript(const objects::CDense_seg& denseg);

    /// Generate traceback from a Dense_diag
    /// @param dendiag Dense_diag representing a single 
    ///                ungapped pairwise alignment [in]
    ///
    CEditScript(const objects::CDense_diag& dendiag);

    /// Destructor
    ///
    ~CEditScript() {}

    /// Test whether edit script is empty
    /// @return true if there are no edit operations in the script
    ///
    bool Empty() { return m_Script.empty(); }

    /// Reverse an edit script; insertions become deletions
    /// and vice versa
    ///
    void ReverseEditScript()
    {
        for (size_t i = 0; i < m_Script.size(); i++) {
            if (m_Script[i].op_type == eGapAlignIns)
                m_Script[i].op_type = eGapAlignDel;
            else if (m_Script[i].op_type == eGapAlignDel)
                m_Script[i].op_type = eGapAlignIns;
        }
    }

    /// Return an edit script corresponding to a subset of 
    /// the complete traceback available
    /// @param tback_range The portion of the traceback desired.
    ///                    The starting and ending offsets in this
    ///                    range will also be included in the traceback
    ///                    structure returned [in]
    /// @return The subset of the traceback
    ///
    CEditScript MakeEditScript(TRange tback_range);

    /// Convert a CNWAligner edit script to a CEditScript
    /// @param tback The edit script generated by CNWAligner [in]
    /// @param tback_range The portion of the traceback desired.
    ///                    The starting and ending offsets in this
    ///                    range will also be included in the traceback
    ///                    structure returned [in]
    /// @return The subset of the traceback
    ///
    static CEditScript MakeEditScript(const CNWAligner::TTranscript& tback,
                                      TRange tback_range);

    /// Given a subject offset, find the corresponding query offset
    /// @param start_offsets The sequence offsets corresponding to the
    ///                      start of the edit script [in]
    /// @param new_offsets The offsets into the unaligned sequences
    ///                    where the specified subject offset occurs [out]
    /// @param seq2_target The subject offset to find [in]
    /// @param new_tback The offset of the traceback operation where
    ///                  where seq2_target was found [out]
    /// @param go_past_seq1_gap If seq2_target aligns with a gap in seq1,
    ///                         include the gap if true [in]
    ///
    void FindOffsetFromSeq2(TOffsetPair start_offsets,
                            TOffsetPair& new_offsets,
                            TOffset seq2_target, TOffset& new_tback,
                            bool go_past_seq1_gap);

    /// Given a query offset, find the corresponding subject offset
    /// @param start_offsets The sequence offsets corresponding to the
    ///                      start of the edit script [in]
    /// @param new_offsets The offsets into the unaligned sequences
    ///                    where the specified query offset occurs [out]
    /// @param seq1_target The query offset to find [in]
    /// @param new_tback The offset of the traceback operation where
    ///                  where seq1_target was found [out]
    /// @param go_past_seq2_gap If seq1_target aligns with a gap in seq2,
    ///                         include the gap if true [in]
    ///
    void FindOffsetFromSeq1(TOffsetPair start_offsets,
                            TOffsetPair& new_offsets,
                            TOffset seq1_target, TOffset& new_tback,
                            bool go_past_seq2_gap);

    /// Compute the score associated with (a portion of) an alignment
    /// Assumes that seq1 is a sequence and that seq2 is a PSSM
    /// @param tback_range The starting and ending traceback operation
    ///                    of the sub-alignment to score [in]
    /// @param start_offsets The sequence offsets of the beginning of the
    ///                      region described by the CEditScript [in]
    /// @param seq1 The complete first sequence [in]
    /// @param seq2_pssm PSSM representing the second sequence [in]
    /// @param gap_open Penalty for opening a gap [in]
    /// @param gap_extend Penalty for extending a gap [in]
    /// @return The score of the (sub-)alignment given by tback_range
    ///
    int GetScore(TRange tback_range, TOffsetPair start_offsets,
                 CSequence& seq1, int **seq2_pssm,
                 int gap_open, int gap_extend);

    /// Compile a list of regions in the current edit script that
    /// contain substitutions
    /// @param start_offsets The sequence offsets corresponding to the
    ///                   start of the range described by the CEditScript [in]
    /// @return List of regions within the CEditScript that contain 
    ///         substitutions
    ///
    vector<TOffsetPair> ListMatchRegions(TOffsetPair start_offsets);

    /// Validate that the alignment described by the CEditScript
    /// has the same size for each sequence as the input ranges
    /// @param seq1_range Start/stop offsets of the first sequence [in]
    /// @param seq2_range Start/stop offsets of the second sequence [in]
    ///
    void VerifyScript(TRange seq1_range, TRange seq2_range);

private:

    /// Runlength-encoded representation of a traceback
    /// operation. Note that we follow the blast convention,
    /// where a deletion is a gap in the first sequence and
    /// an insertion is a gap in the second sequence
    ///
    struct STracebackOp {
        EGapAlignOpType op_type;     ///< type of operation
        int num_ops;                 ///< number of such operations

        /// Create a new operation
        /// @param op Type of operation [in]
        /// @param num Number of operations [in]
        ///
        STracebackOp(EGapAlignOpType op, int num)
                        : op_type(op), num_ops(num) {}
    };

    /// Edit script type
    typedef vector<STracebackOp> TScriptOps;

    /// The list of edit operations in the current edit script
    TScriptOps m_Script;

    /// Add a new edit operation to the current list
    /// @param op_type Type of new operation [in]
    /// @param num_ops The number of such operations [in]
    /// 
    void AddOps(EGapAlignOpType op_type, int num_ops);
};


END_SCOPE(cobalt)
END_NCBI_SCOPE

#endif // ALGO_COBALT___TRACEBACK__HPP
