/*
  Copyright(C) 2002-2007 Pierre Mazire
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

/*
  scapi.c

  Simple Configuration API
*/

#include "common.h"
#include <stdio.h>
#include <scapi/scapi.h>

unsigned int Libscapi_WARNING_On=TRUE;
unsigned int Libscapi_ERROR_On=TRUE;
unsigned int Libscapi_OUTPUT_On=TRUE;
unsigned int Libscapi_HEADER_On=TRUE;
unsigned int Libscapi_HEADER_OUTPUT_On=TRUE;         // Print [OUTPUT] 
unsigned int Libscapi_HEADER_WARNING_On=TRUE;        // Print [WARNING]
unsigned int Libscapi_HEADER_ERROR_On=TRUE;          // Print [ERROR]
unsigned int Libscapi_HEADER_DEBUG_On=TRUE;
unsigned int Libscapi_HEADER_OUTPUT_LIBNAME_On=TRUE; // 
unsigned int Libscapi_HEADER_WARNING_LIBNAME_On=TRUE;// Print the lib name
unsigned int Libscapi_HEADER_ERROR_LIBNAME_On=TRUE;  //
unsigned int Libscapi_HEADER_DEBUG_LIBNAME_On=TRUE;

FILE *Libscapi_err=NULL;
FILE *Libscapi_out=NULL;


s_Options *InitOptions(unsigned char *Title,
		      unsigned char *Author,
		      unsigned char *Copyright,
		      unsigned int VerMaj,
		      unsigned int VerMin,
		      unsigned int VerMisc)
{
  s_Options *Options;

  Libscapi_err=stderr;
  Libscapi_out=stdout;

  Options=XMALLOC(s_Options,1);

  if(!Title)
    {
      LPRINTF(ERROR,"Title not defined");
      XFREE(Options);
      return NULL;
    }
  else
    Options->Title=XSTRDUP(Title);

  if(!Author)
    {
      LPRINTF(ERROR,"Author not defined");
      XFREE(Options);
      return NULL;
    }
  else
    Options->Author=XSTRDUP(Author);

  Options->Copyright=XSTRDUP(Copyright);

  Options->VerMaj=VerMaj;
  Options->VerMin=VerMin;
  Options->VerMisc=VerMisc;
  Options->NbrContexts=0;
  Options->Contexts=NULL;
  Options->NbrUnknownTokens=0;
  Options->UnknownTokens=NULL;

  return Options;
};

s_Options *ParseOptions(s_Options *Options,
			unsigned int argc,
			char **argv)
{
  unsigned int i,j,k;
  s_Context *Context;
  unsigned char *string=NULL;
  unsigned char *string2=NULL;
  unsigned char *value=NULL;

  for(i=0;i<Options->NbrContexts;i++)
    {
      ParseConfFile(Options,Options->Contexts[i],
		    Options->Contexts[i]->GlobalConfFile);
      ParseConfFile(Options,Options->Contexts[i],
		    Options->Contexts[i]->LocalConfFile);
    };

  for(i=1;i<argc;i++)
    {
      for(k=0;k<Options->NbrContexts;k++)
	{
	  for(j=0;j<Options->Contexts[k]->NbrOptions;j++)
	    if(strcmp(Options->Contexts[k]->Options[j]->Name,argv[i])==0 ||
	       (Options->Contexts[k]->Options[j]->ShortName!=NULL && 
		strcmp(Options->Contexts[k]->Options[j]->ShortName,argv[i])==0)
	       )
	      break;
	  if(j!=Options->Contexts[k]->NbrOptions)
	    break;
	};
      
      if(k==Options->NbrContexts)
	{
	  Options->UnknownTokens=XREALLOC(Options->UnknownTokens,
					  unsigned char*,
					  Options->NbrUnknownTokens+1);

	  Options->UnknownTokens[Options->NbrUnknownTokens]=XSTRDUP(argv[i]);
	  Options->NbrUnknownTokens++;
	}
      else
	{
	  Context=Options->Contexts[k];
	  switch(Context->Options[j]->Type)
	    {
	    case SET_ARG:
	    case FREE_ARG:
	      i++;
	      if(i>=argc-1)
		return NULL;
	      
	      for(k=0;k<Context->NbrOptions;k++)
		if(strcmp(Context->Options[k]->Name,argv[i])==0 ||
		   (Context->Options[k]->ShortName!=NULL && 
		    strcmp(Context->Options[k]->ShortName,argv[i])==0)
		   )
		  break;

	      if (k!=Context->NbrOptions)
		return NULL;
	      else
		{
		  if(Context->Options[j]->Type==SET_ARG)
		    {
		      for(k=0;k<Context->Options[j]->NbrSetArgs;k++)
			if(strcmp(Context->Options[j]->SetArgs[k],argv[i])==0)
			  break;
		      if(k==Context->Options[j]->NbrSetArgs)
			return NULL;
		    }
		  
		  Context->Options[j]->Flags|=SCAPI_OPTION_USED;
		  if((Context->Options[j]->Flags & SCAPI_MULTIPLE_ARGS)==0)
		    {
		      for(k=0;k<Context->Options[j]->NbrArgs;k++)
			XFREE(Context->Options[j]->Args[k]);
		      XFREE(Context->Options[j]->Args);
		      Context->Options[j]->NbrArgs=0;
		    }

		  if(Context->Options[j]->Separator!=NULL)
		    {
		      string=XSTRDUP(argv[i]);
		      string2=string;
		      value=strtok(string,Context->Options[j]->Separator);
		      while(value!=NULL)
			{
			  Context->Options[j]->Args=
			    XREALLOC(Context->Options[j]->Args,
				     unsigned char*,
				     Context->Options[j]->NbrArgs+1);
			  Context->Options[j]->Args[Context->Options[j]->NbrArgs]=XSTRDUP(value);
			  Context->Options[j]->NbrArgs++;
			  value=strtok(NULL,Context->Options[j]->Separator);
			}
		      string=NULL;
		      XFREE(string2);
		    }
		  else
		    {
		      Context->Options[j]->Args=
			XREALLOC(Context->Options[j]->Args,
				 unsigned char*,
				 Context->Options[j]->NbrArgs+1);
		      Context->Options[j]->Args[Context->Options[j]->NbrArgs]=XSTRDUP(argv[i]);
		      Context->Options[j]->NbrArgs++;
		    };
		};
	      break;

	    case NO_ARG:
	      Context->Options[j]->Flags|=SCAPI_OPTION_USED;
	      break;
	    };
	};
    };

  return Options;
};  

unsigned int GetOption(s_Options *Options,
		       unsigned char *Option,
		       unsigned char ***Value)
{
  unsigned int i,j;
  
  for(i=0;i<Options->NbrContexts;i++)
    {
      for(j=0;j<Options->Contexts[i]->NbrOptions;j++)
	if(strcmp(Options->Contexts[i]->Options[j]->Name,Option)==0 ||
	   (Options->Contexts[i]->Options[j]->ShortName!=NULL && 
	    strcmp(Options->Contexts[i]->Options[j]->ShortName,Option)==0)
	   )
	  break;
      if(j!=Options->Contexts[i]->NbrOptions)
	break;
    };

  if (i==Options->NbrContexts)
    return 0;

  *Value=Options->Contexts[i]->Options[j]->Args;
  switch(Options->Contexts[i]->Options[j]->Type)
    {
    case FREE_ARG:
    case SET_ARG:
      return(Options->Contexts[i]->Options[j]->NbrArgs);
    case NO_ARG:
      return (Options->Contexts[i]->Options[j]->Flags & SCAPI_OPTION_USED);
    };
  return 0;
};
	    
s_Options *SetOptionsUsage(s_Options *Options,unsigned char *Usage)
{
  if (!Options)
    {
      LPRINTF(ERROR,"There is no options defined");
      return NULL;
    }
  else
    Options->Usage=XSTRDUP(Usage);
  return Options;
};

s_Options *AddContext(s_Options *Options,
		      s_Context *Context)
{
  if (!Options)
    {
      LPRINTF(ERROR,"There is no options defined");
      return NULL;
    };

  if(!Context)
    {
      LPRINTF(WARNING,"This option context has not been defined");
      return Options;
    };


  Options->Contexts=XREALLOC(Options->Contexts,
			     s_Context*,
			     Options->NbrContexts+1);
  Options->Contexts[Options->NbrContexts]=Context;
  Options->NbrContexts++;

  return Options;
};

void DisplayOptionsHelp(s_Options *Options,
			unsigned int SizeColumnOptionNames,
			unsigned int SizeColumnOptionHelp)
{
  unsigned int i,j,k,l;
  unsigned char *Names=NULL;
  unsigned int NbrNameLines;
  unsigned char **NameLines=NULL;
  unsigned int NameLen;
  unsigned char *Help=NULL;
  unsigned int NbrHelpLines;
  unsigned char **HelpLines=NULL;
  unsigned char HelpLen;

  s_Context *Context=NULL;
;
  fprintf(stderr,"%s version %i.%i.%i ",
	  Options->Title,
	  Options->VerMaj,Options->VerMin,Options->VerMisc);
  if(Options->Copyright!=NULL)
    fprintf(stderr,"(c) %s ",Options->Copyright);
  fprintf(stderr,"by %s\n",Options->Author);
  fprintf(stderr,"\n");
  fprintf(stderr,"Usage: %s\n",Options->Usage);
  fprintf(stderr,"Options:\n");
  fprintf(stderr,"\n");

  for(k=0;k<Options->NbrContexts;k++)
    {
      Context=Options->Contexts[k];

      if(k!=0)
	fprintf(stderr,"[%s]\n",Context->Title);

      for(i=0;i<Context->NbrOptions;i++)
	{
	  NbrNameLines=0;
	  NbrHelpLines=0;
	  
	  NameLen=strlen(Context->Options[i]->Name);
	  if(Context->Options[i]->ShortName!=NULL)
	    NameLen+=strlen(" / ")+strlen(Context->Options[i]->ShortName);
	  switch(Context->Options[i]->Type)
	    {
	    case SET_ARG: 
	      NameLen+=strlen(" [");
	      for(j=0;j<Context->Options[i]->NbrSetArgs;j++)
		{
		  NameLen+=strlen(Context->Options[i]->SetArgs[j]);
		  if(j!=Context->Options[i]->NbrSetArgs-1)
		    NameLen+=strlen("|");
		}
	      NameLen+=strlen("]");
	      break;
	    case FREE_ARG:
	      NameLen+=1+strlen(Context->Options[i]->ArgDescription);
	      break;
	    };
	  Names=XCALLOC(unsigned char,NameLen+1);
	  strcat(Names,Context->Options[i]->Name);
	  if(Context->Options[i]->ShortName!=NULL)
	    {
	      strcat(Names," / ");
	      strcat(Names,Context->Options[i]->ShortName);
	    };
	  switch(Context->Options[i]->Type)
	    {
	    case SET_ARG: 
	      strcat(Names," [");
	      for(j=0;j<Context->Options[i]->NbrSetArgs;j++)
		{
		  strcat(Names,Context->Options[i]->SetArgs[j]);
		  if(j!=Context->Options[i]->NbrSetArgs-1)
		    strcat(Names,"|");
		}
	      strcat(Names,"]");
	      break;
	    case FREE_ARG:
	      strcat(Names," ");
	      strcat(Names,Context->Options[i]->ArgDescription);
	      break;
	    };      
	  
	  //      fprintf(stderr,"%s\n",Names);
	  if(NameLen>=SizeColumnOptionNames)
	    {
	      j=SizeColumnOptionNames-1;
	      while(Names[j]!='/')
		j--;
	      NameLines=XREALLOC(NameLines,unsigned char*,NbrNameLines+1);
	      NameLines[NbrNameLines]=XCALLOC(unsigned char,
					      SizeColumnOptionNames+1);
	      strncat(NameLines[NbrNameLines],Names,j+1);
	      memset(&NameLines[NbrNameLines][j+1],' ',
		     SizeColumnOptionNames-(j+1));
	      NbrNameLines++;
	      NameLines=XREALLOC(NameLines,unsigned char*,NbrNameLines+1);
	      NameLines[NbrNameLines]=XCALLOC(unsigned char,
					      SizeColumnOptionNames+1);
	      strncat(NameLines[NbrNameLines],&Names[j+1],strlen(&Names[j+1]));
	      memset(&NameLines[NbrNameLines][strlen(&Names[j+1])],' ',
		     SizeColumnOptionNames-strlen(&Names[j+1]));
	      NbrNameLines++;
	    }
	  else
	    {
	      NameLines=XREALLOC(NameLines,unsigned char*,NbrNameLines+1);
	      NameLines[NbrNameLines]=XCALLOC(unsigned char,
					      SizeColumnOptionNames+1);
	      strncat(NameLines[NbrNameLines],Names,strlen(Names));
	      memset(&NameLines[NbrNameLines][strlen(Names)],' ',
		     SizeColumnOptionNames-strlen(Names));
	      NbrNameLines++;
	    };
	  
	  XFREE(Names);
	  
	  Help=XSTRDUP(Context->Options[i]->Help);
	  //      fprintf(stderr,"%s\n",Help);
	  HelpLen=strlen(Context->Options[i]->Help);
	  k=0;
	  while(HelpLen>=SizeColumnOptionHelp)
	    {
	      j=SizeColumnOptionHelp-1;
	      while(Help[k+j]!=' ')
		j--;
	      HelpLines=XREALLOC(HelpLines,unsigned char*,NbrHelpLines+1);
	      HelpLines[NbrHelpLines]=XCALLOC(unsigned char,j+1);
	      strncpy(HelpLines[NbrHelpLines],&Help[k],j);
	      k+=j+1;
	      NbrHelpLines++;
	      HelpLen-=j;
	    };
	  
	  HelpLines=XREALLOC(HelpLines,unsigned char*,NbrHelpLines+1);
	  HelpLines[NbrHelpLines]=XCALLOC(unsigned char,HelpLen+1);
	  strncpy(HelpLines[NbrHelpLines],&Help[k],HelpLen);
	  NbrHelpLines++;
	  
	  XFREE(Help);
	  if(NbrNameLines<NbrHelpLines)
	    k=NbrHelpLines;
	  else
	    k=NbrNameLines;
	  
	  for(j=0;j<k;j++)
	    {
	      if(j<NbrNameLines)
		fprintf(stderr,"%s",NameLines[j]);
	      else
		for(l=0;l<SizeColumnOptionNames;l++)
		  fprintf(stderr," ");
	      
	      if(j<NbrHelpLines)
		fprintf(stderr,"%s",HelpLines[j]);
	      
	      fprintf(stderr,"\n");
	    };
	  
	  for(j=0;j<NbrNameLines;j++)
	    XFREE(NameLines[j]);
	  XFREE(NameLines);
	  
	  for(j=0;j<NbrHelpLines;j++)
	    XFREE(HelpLines[j]);
	  XFREE(HelpLines); 
	  fprintf(stderr,"\n");
	};
    };
};

void FreeOptions(s_Options *Options)
{
  unsigned int i;

  XFREE(Options->Title);
  XFREE(Options->Author);
  XFREE(Options->Copyright);
  XFREE(Options->Usage);
  Options->VerMaj=0;
  Options->VerMin=0;
  Options->VerMisc=0;

  for(i=0;i<Options->NbrContexts;i++)
    {
      FreeContext(Options->Contexts[i]);
      Options->Contexts[i]=NULL;
    };
  XFREE(Options->Contexts);

  for(i=0;i<Options->NbrUnknownTokens;i++)
    XFREE(Options->UnknownTokens[i]);
  XFREE(Options->UnknownTokens);  

  XFREE(Options);
};
