/*
  Copyright(C) 2002-2007 Pierre Mazire
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

/*
  readinfo.c

  Get games sources from different kind of sources
*/

#include "common.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>

#include <mamory/readinfo.h>
#include <sortbox/sortbox.h>
#include <mamory/zip.h>
#include <mamory/admameli.h>
#include <mamory/amlxml.h>
#include <mamory/dirzip.h>

#define ROMONLY 0
#define TOTALGAME 1
  
void FreeSharedRomInfo(s_SharedRomInfo *Shared)
{
  if(Shared!=NULL)
    {
      XFREE(Shared->CloneName);
      XFREE(Shared->RomName);
      if(Shared->Hardware!=NULL)
	{	
	  XFREE(Shared->Hardware->Region);
	  XFREE(Shared->Hardware->Flags);
	  Shared->Hardware->Offset=0;
	  XFREE(Shared->Hardware);
	};
      XFREE(Shared);
    };
};

void FreeRomInfo(s_RomInfo *Rom)
{
  unsigned int i;
  if (Rom!=NULL)
    {
      Rom->Source=NULL;
      XFREE(Rom->Name);
      XFREE(Rom->GameName);
      XFREE(Rom->Merge);
      if(Rom->Hardware!=NULL)
	{	
	  XFREE(Rom->Hardware->Region);
	  XFREE(Rom->Hardware->Flags);
	  Rom->Hardware->Offset=0;
	  XFREE(Rom->Hardware);
	};
      Rom->Content=NULL;
      for(i=0;i<Rom->NbrShared;i++)
	{
	  FreeSharedRomInfo(Rom->Shared[i]);
	  Rom->Shared[i]=NULL;
	};
      XFREE(Rom->Shared);
      Rom->NbrShared=0;
      XFREE(Rom);
    };
};

void FreeDiskInfo(s_DiskInfo *Disk)
{
  if (Disk==NULL)
    return;

  XFREE(Disk->Name);
  XFREE(Disk->md5);
  if (Disk->Hardware!=NULL)
    {
      XFREE(Disk->Hardware->Region);
      Disk->Hardware->Index=0;
      XFREE(Disk->Hardware);
    };
  XFREE(Disk);
};

void FreeChipInfo(s_ChipInfo *Chip)
{
  if(Chip!=NULL)
    {
      XFREE(Chip->Type);
      XFREE(Chip->Flags);
      XFREE(Chip->Name);
      Chip->Clock=0;
      XFREE(Chip);
    };
};

void FreeDisplayInfo(s_DisplayInfo *Display)
{
  if(Display!=NULL)
    {
      XFREE(Display->Type);
      Display->Rotate=0;
      Display->X=0;
      Display->Y=0;
      Display->AspectX=0;
      Display->AspectY=0;
      Display->Flipx=0;
      Display->Colors=0;
      Display->Freq=0;
      XFREE(Display);
    };
};

void FreeSoundInfo(s_SoundInfo *Sound)
{
  if(Sound!=NULL)
    {
      Sound->Channels=0;
      XFREE(Sound);
    };
};

void FreeControlInfo(s_ControlInfo *Control)
{
  if(Control!=NULL)
    XFREE(Control->Type);
}

void FreeInputInfo(s_InputInfo *Input)
{
  unsigned int i=0;

  if(Input!=NULL)
    {
      Input->Players=0;
      for(i=0;i<Input->NbrControls;i++)
	FreeControlInfo(Input->Controls[i]);
      XFREE(Input->Controls);
      Input->Buttons=0;
      Input->Coins=0;
      XFREE(Input->Tilt);
      XFREE(Input->Service);
      XFREE(Input);
    };
};

void FreeDipSwitchInfo(s_DipSwitchInfo *DS)
{
  int i;
  if(DS!=NULL)
    {
      XFREE(DS->Name);
      for(i=0;i<DS->NbrEntries;i++)
	XFREE(DS->Entries[i]);
      XFREE(DS->Entries);
      DS->NbrEntries=0;
      XFREE(DS->Default);
      XFREE(DS);
    };
};

void FreeDriverInfo(s_DriverInfo *Driver)
{
  if(Driver!=NULL)
    {
      XFREE(Driver->Status);
      XFREE(Driver->Color);
      XFREE(Driver->Sound);
      XFREE(Driver->Blit);
      Driver->PaletteSize=0;
      XFREE(Driver);
    };
};


void FreeMiscGameInfo(s_MiscGameInfo *Misc)
{
  unsigned int i;

  if(Misc!=NULL)
    {
      Misc->Year=0;
      XFREE(Misc->Manufacturer);
      for(i=0;i<Misc->NbrChips;i++)
	{
	  FreeChipInfo(Misc->Chips[i]);
	  Misc->Chips[i]=NULL;
	};
      XFREE(Misc->Chips);
      Misc->NbrChips=0;
      for(i=0;i<Misc->NbrDisplays;i++)
	{
	  FreeDisplayInfo(Misc->Displays[i]);
	  Misc->Displays[i]=NULL;
	};
      XFREE(Misc->Displays);
      Misc->NbrDisplays=0;
      FreeInputInfo(Misc->Input);
      Misc->Input=NULL;
      FreeSoundInfo(Misc->Sound);
      Misc->Sound=NULL;
      for(i=0;i<Misc->NbrDipSwitches;i++)
	{
	  FreeDipSwitchInfo(Misc->DipSwitches[i]);
	  Misc->DipSwitches[i]=NULL;
	};
      XFREE(Misc->DipSwitches);
      Misc->NbrDipSwitches=0;
      FreeDriverInfo(Misc->Driver);
      Misc->Driver=NULL;
      XFREE(Misc);
    };
};

void FreeGameInfo(s_GameInfo *GameInfo)
{
  int i;
  if (GameInfo!=NULL)
    {
      if(GameInfo->More!=NULL)
	{
	  FreeMiscGameInfo(GameInfo->More->Misc);
	  GameInfo->More->Misc=NULL;
	  XFREE(GameInfo->More->Description);
	  XFREE(GameInfo->More->CloneOf);
	  for(i=0;i<GameInfo->More->NbrMergedRoms;i++)
	    {	   
	      FreeRomInfo(GameInfo->More->MergedRoms[i]);
	      GameInfo->More->MergedRoms[i]=NULL;
	    };
	  XFREE(GameInfo->More->MergedRoms);
	  GameInfo->More->NbrMergedRoms=0;
	  XFREE(GameInfo->More->RomOf);
	  XFREE(GameInfo->More->Resource);
	  for(i=0;i<GameInfo->More->NbrClones;i++)
	    {
	      XFREE(GameInfo->More->Clones[i]->Name);
	      XFREE(GameInfo->More->Clones[i]->Description);
	      FreeMiscGameInfo(GameInfo->More->Clones[i]->Misc);
	      GameInfo->More->Clones[i]->Misc=NULL;
	      XFREE(GameInfo->More->Clones[i]);
	    };
	  XFREE(GameInfo->More->Clones);
	  GameInfo->More->NbrClones=0;	  
	  for(i=0;i<GameInfo->More->NbrSamples;i++)
	    GameInfo->More->Samples[i]=NULL;
	  XFREE(GameInfo->More->Samples);
	  GameInfo->More->NbrSamples=0;
	  XFREE(GameInfo->More->SampleOf);
	  
	  XFREE(GameInfo->More);
	};
      
      XFREE(GameInfo->Name);
      for(i=0;i<GameInfo->NbrDisks;i++)
	{
	  FreeDiskInfo(GameInfo->Disks[i]);
	  GameInfo->Disks[i]=NULL;
	};
      XFREE(GameInfo->Disks);

      for(i=0;i<GameInfo->NbrRoms;i++)
	{
	  FreeRomInfo(GameInfo->Roms[i]);
	  GameInfo->Roms[i]=NULL;
	};
      XFREE(GameInfo->Roms);
      GameInfo->NbrRoms=0;

      XFREE(GameInfo);
    };
};


void FreeGamesListSource(s_GamesListSource *Source)
{
  if(Source!=NULL)
    {
      Source->GamesList=NULL;
      XFREE(Source->Source);
      XFREE(Source->Target);
      XFREE(Source->TargetVersion);
      XFREE(Source->Author);
      XFREE(Source->Version);
      XFREE(Source->Comment);

      XFREE(Source);
    };
};


void FreeResourceInfo(s_ResourceInfo *Resource)
{
  if(Resource!=NULL)
    {
      XFREE(Resource->Name);
      XFREE(Resource->AssociatedGames);
      Resource->Game=NULL;
      XFREE(Resource);
    };
};

void FreeMergedGamesList(s_GamesList *GamesList)
{
  unsigned int i,j;

  if (GamesList!=NULL && GamesList->MergedGamesList!=FALSE)
    {    
      for(i=0;i<GamesList->NbrRomContents;i++)
	GamesList->RomContents[i]=NULL;
      
      XFREE(GamesList->RomContents);
      
      for(i=0;i<GamesList->NbrGames;i++)
	{
	  XFREE(GamesList->Games[i]->Name);
	  for(j=0;j<GamesList->Games[i]->NbrRoms;j++)
	    GamesList->Games[i]->Roms[j]=NULL;
	  XFREE(GamesList->Games[i]->Roms);
	  XFREE(GamesList->Games[i]);
	}
      
      XFREE(GamesList->Games);
      
      for(i=0;i<GamesList->NbrSources;i++)
	{
//	  FreeGamesListSource(GamesList->Sources[i]);
	  GamesList->Sources[i]=NULL;
	};
      XFREE(GamesList->Sources);
      
      for(i=0;i<GamesList->NbrResources;i++)
	{
	  FreeResourceInfo(GamesList->Resources[i]);
	  GamesList->Resources[i]=NULL;
	}
      XFREE(GamesList->Resources);
      XFREE(GamesList);
    };
};
 
void FreeGamesList(s_GamesList *GamesList)
{
  int i,j;

  if(GamesList!=NULL)
    {
      if(GamesList->MergedGamesList==TRUE)
	{
	  FreeMergedGamesList(GamesList);
	  GamesList=NULL;
	  return;
	};
      
      for(i=0;i<GamesList->NbrResources;i++)
	{
	  FreeResourceInfo(GamesList->Resources[i]);
	  GamesList->Resources[i]=NULL;
	}

      XFREE(GamesList->Resources);
      GamesList->NbrResources=0;
      
      for(i=0;i<GamesList->NbrRomContents;i++)
	{
	  for (j=0;j<GamesList->RomContents[i]->NbrAssociatedGames;j++)
	    {
	      GamesList->RomContents[i]->AssociatedGames[j]->Game=NULL;
	      XFREE(GamesList->RomContents[i]->AssociatedGames[j]);
	    };
	  XFREE(GamesList->RomContents[i]->AssociatedGames);
	  XFREE(GamesList->RomContents[i]);
	};
      XFREE(GamesList->RomContents);

      for(i=0;i<GamesList->NbrGames;i++)
	{
	  FreeGameInfo(GamesList->Games[i]);
	  GamesList->Games[i]=NULL;
	};
      XFREE(GamesList->Games);
      GamesList->NbrGames=0;
                  
      for(i=0;i<GamesList->NbrSoundSamples;i++)
	XFREE(GamesList->SoundSamples[i]);
      XFREE(GamesList->SoundSamples);

      for(i=0;i<GamesList->NbrSources;i++)
	{
	  FreeGamesListSource(GamesList->Sources[i]);
	  GamesList->Sources[i]=NULL;
	};
      XFREE(GamesList->Sources);

      XFREE(GamesList);
    };
};

s_GamesList *GetGamesList(unsigned char *MameListSrc,
			  unsigned char *TempDir)
{
  FILE *fp;

  s_GamesList *GamesList=NULL;
  unsigned char *TempFile="/mamory-GamesList.tmp";
  unsigned char *TempPath=NULL;
  unsigned char *commandline=NULL;
  unsigned char *opt_li="-listinfo";
  unsigned char *opt_lx="-listxml";
  unsigned char *opt_V="-V";
  unsigned char *opt_out="-out";
  unsigned char *opt_err="-err /dev/null";
  unsigned char *opt_no_load_config="-noloadconfig";
  unsigned char *redir="1 > /dev/null 2> /tmp/mame_stderr";
  unsigned char *redir1="1>";
  unsigned char *redir2= "2> /tmp/mame_stderr";
  unsigned char MameVersion[BUFSIZ]={0};
  unsigned int NumVersion=0;
  struct stat stat_buf;

  // Look at mamory.h for explanations about the 4 following lines
  if(Libmamory_out==NULL)
    Libmamory_out=stdout;
  if(Libmamory_err==NULL)
    Libmamory_err=stderr;

  Libsortbox_OUTPUT_On=FALSE;
  Libsortbox_WARNING_On=FALSE;
  Libsortbox_ERROR_On=FALSE;


  TempPath=XCALLOC(unsigned char,strlen(TempDir)+strlen(TempFile)+1);

  strcat(TempPath,TempDir);
  strcat(TempPath,TempFile);

//  getcwd(Buffer,BUFSIZ);
// TODO: Change relative pathway to absolute pathway

  if(!stat(MameListSrc,&stat_buf))
    {
      if(stat_buf.st_mode & S_IFDIR)
	{
	  LPRINTFW(OUTPUT,"Extract Games Infos from ZIP files in '%s' ...",
		  MameListSrc);
//	  GamesList=GetGamesListFromDir(MameListSrc);
	  GamesList=GetGamesListFromPath(MameListSrc,FALSE);
	  LPRINTF(OUTPUT," done");
	}
      else
	{
	  if(isAdMAMEliFile(MameListSrc))
	    {
	      LPRINTFW(OUTPUT,"Extract Games Infos from file '%s' ...",
		       MameListSrc);
	      GamesList=GetGamesListFromAdMAMEliFile(MameListSrc,
						     MameListSrc,
						     TYPE_FILE);
	      LPRINTF(OUTPUT," done");
	    }
	  else
	    if(isAMLXMLFile(MameListSrc))
	      {
		LPRINTFW(OUTPUT,"Extract Games Infos from file '%s' ...",
			 MameListSrc);
		GamesList=GetGamesListFromAMLXMLFile(MameListSrc,
						     MameListSrc,
						     TYPE_FILE);
		LPRINTF(OUTPUT," done");	      
	      }
	    else
	      if(stat_buf.st_mode & S_IXUSR ||
		 stat_buf.st_mode & S_IXGRP ||
		 stat_buf.st_mode & S_IXOTH)
		{
		  
		  commandline=XCALLOC(unsigned char,
				      strlen(MameListSrc)+1+
				      strlen(opt_no_load_config)+1+
				      strlen(opt_li)+1+
				      strlen(opt_out)+1+
				      strlen(TempPath)+1+
				      strlen(opt_err)+1+
				      strlen(redir)+1);

		  sprintf(commandline,"%s %s %s %s 1 > %s %s",
			  MameListSrc,opt_no_load_config,opt_V,opt_err,TempPath,redir2);

		  if(system(commandline)!=-1)
		    {
		      if(!(fp=fopen(TempPath,"r")))
			{
			  unlink(TempPath);
			  LPRINTF(WARNING,"'%s' error:",MameListSrc);
			  if((fp=fopen("/tmp/mame_stderr","r")))
			    {
			      while(!feof(fp))
				{
				  if(fgets(MameVersion,BUFSIZ,fp))
				    LPRINTF(WARNING,"%s",MameVersion);
				};
			      fclose(fp);
			      unlink("/tmp/mame_stderr");
			    };
			  
			  LPRINTF(ERROR,"'%s' is not a MAME executable",
				  MameListSrc);
			  LPRINTF(ERROR,"or the environment required by '%s' has not been found",
				  MameListSrc);
			  return NULL;
            };
		      
              while(fgets(MameVersion,BUFSIZ*sizeof(char),fp))
              {
                if(strncmp(MameVersion,"xmame",5)==0)
                  break;
              };
		      /*
		      fread(MameVersion,sizeof(char),BUFSIZ,fp);
		      */
		      fclose(fp);
		      
		      MameVersion[strlen(MameVersion)-1]=0;
		      if (strncmp(MameVersion,"xmame",5)==0)
			{
			  // Since xmame 0.84 the -listinfo option is deprecated
			  // in favor of the -listxml option
			  sscanf(MameVersion,"xmame %*s version %*u.%u.%*u %*s %*s %*s",&NumVersion);
			  if(NumVersion<84)
			    sprintf(commandline,"%s %s %s %s %s %s %s",
				    MameListSrc,
				    opt_no_load_config,
				    opt_li,
				    opt_out,TempPath,opt_err,redir);
			  else
			    sprintf(commandline,"%s %s %s %s %s %s %s",
				    MameListSrc,
				    opt_no_load_config,
				    opt_lx,
				    opt_out,TempPath,opt_err,redir);

			  if(system(commandline)==-1)
			    {
			      unlink(TempPath);
			      unlink("/tmp/mame_stderr");
			      LPRINTF(ERROR,"'%s' is not a MAME executable",
				      MameListSrc);
			      LPRINTF(ERROR,"or the environment required by '%s' is not found",
				      MameListSrc);		
			      return NULL;
			    };
			  
			  // xmame 0.68.1 specific:
			  // Because xmame 0.68.1 do not redirect correctly
			  // stdout to the file given with the -out option
			  if(stat(TempPath,&stat_buf))
			    {
			      unlink("/tmp/mame_stderr");
			      LPRINTF(ERROR,"Cannot read the temporary file from xmame output");
			      return NULL;
			    };
			  
			  if(stat_buf.st_size==0)
			    {
			      sprintf(commandline,"%s %s %s %s %s %s %s",
				      MameListSrc,
				      opt_no_load_config,
				      opt_li,
				      opt_err,redir1,TempPath,redir2);
			      if(system(commandline)==-1)
				{
				  unlink(TempPath);
				  unlink("/tmp/mame_stderr");
				  LPRINTF(ERROR,"'%s' is not a MAME executable",
					  MameListSrc);
				  LPRINTF(ERROR,"or the environment required by '%s' is not found",
					  MameListSrc);		
				  return NULL;
				};
			    };
			  // end of xmame 0.68.1 specific 
			}
		      else
			{
			  sprintf(commandline,"%s -help 1>%s %s",
				  MameListSrc,TempPath,redir2);

			  if(system(commandline)==-1)
			    {				 
			      LPRINTF(ERROR,strerror(errno));
			      return NULL;
			    }
			  else
			    {
			      if(!(fp=fopen(TempPath,"r")))
				{
				  unlink(TempPath);
				  LPRINTF(WARNING,"'%s' error:",MameListSrc);
				  if((fp=fopen("/tmp/mame_stderr","r")))
				    {
				      while(!feof(fp))
					{
					  if(fgets(MameVersion,BUFSIZ,fp))
					    LPRINTF(WARNING,"%s",MameVersion);
					};
				      fclose(fp);
				      unlink("/tmp/mame_stderr");
				    };
				  
				  LPRINTF(ERROR,"'%s' is not a MAME executable",
					  MameListSrc);
				  LPRINTF(ERROR,"or the environment required by '%s' has not been found",
					  MameListSrc);
				  return NULL;
				};
			      
			      while(fgets(MameVersion,BUFSIZ*sizeof(char),fp))
				{
				  if(strncmp(MameVersion,"M.A.M.E.",5)==0)
				    break;
				};
			      fclose(fp);
			      
			      MameVersion[strlen(MameVersion)-1]=0;
			      if (strncmp(MameVersion,"M.A.M.E.",8)==0)
				{
				  sscanf(MameVersion,"M.A.M.E. v %*u.%u",&NumVersion);
				  sprintf(commandline,"%s %s > %s",
				    MameListSrc,
				    opt_lx,
				    TempPath);

				  if(system(commandline)==-1)
				    {
				      unlink(TempPath);
				      unlink("/tmp/mame_stderr");
				      LPRINTF(ERROR,"'%s' is not a SDLMAME executable",
					      MameListSrc);
				      LPRINTF(ERROR,"or the environment required by '%s' is not found",
					      MameListSrc);		
				      return NULL;
				    };				  
				}
			      else
				{
				  unlink(TempPath);
				  unlink("/tmp/mame_stderr");
				  LPRINTF(ERROR,"'%s' is not a MAME executable",
					  MameListSrc);
				  LPRINTF(ERROR,"or the environment required by '%s' is not found",
					  MameListSrc);
				  return NULL;
				};
			    };
			};
		    }
		  else
		    {
		      LPRINTF(ERROR,strerror(errno));
		      return NULL;
		    };
		  LPRINTFW(OUTPUT,"Extract Games Infos from %s ...",
			   MameVersion);

		  // Since xmame 0.84 the -listinfo option is deprecated
		  // in favor of the -listxml option
		  if(NumVersion<84)
		    GamesList=GetGamesListFromAdMAMEliFile(TempPath,
							   MameVersion,
							   TYPE_EXEC);
		  else
		    GamesList=GetGamesListFromAMLXMLFile(TempPath,
							 MameVersion,
							 TYPE_EXEC);
		  unlink(TempPath);
		  unlink("/tmp/mame_stderr");
		  LPRINTF(OUTPUT," done");
		};
	  
	};
    }
  else
    LPRINTF(ERROR,strerror(errno));

  XFREE(commandline);
  XFREE(TempPath);
  return(GamesList);

};
     

s_GamesList *MergeGamesLists(s_GamesList *GamesList1, s_GamesList *GamesList2)
{
  unsigned int i,j,k,l,nbrdata=1;
  s_SortBox *GamesSortBox=NULL;
  s_SortBox *RomContentSortBox=NULL;
  s_GameInfo **Games=NULL;
  s_RomContent **Contents=NULL;
  if (GamesList1==NULL)
    {
      GamesList1=XCALLOC(s_GamesList,1);
      GamesList1->MergedGamesList=TRUE;
    };

  // TODO: Verify if source is not already in mergedgameslist
  GamesList1->Sources=
    XREALLOC(GamesList1->Sources,
	     s_GamesListSource*,
	     GamesList1->NbrSources+GamesList2->NbrSources);
  for(i=0;i<GamesList2->NbrSources;i++)
    {
      GamesList1->Sources[GamesList1->NbrSources]=GamesList2->Sources[i];
      GamesList1->NbrSources++;
    }
  
  RomContentSortBox=InitSortBox(SB_NUM,NULL);
  
  for(i=0;i<GamesList1->NbrRomContents;i++)
    if(GamesList1->RomContents[i]->CRC!=0)
      SortBox_AddOneUint(RomContentSortBox,
			 GamesList1->RomContents[i]->CRC,
			 GamesList1->RomContents[i]);

  for(i=0;i<GamesList2->NbrRomContents;i++)
    {
      nbrdata=0;
      if(GamesList2->RomContents[i]->CRC!=0)
	Contents=(s_RomContent**)SortBox_FindTheUint(RomContentSortBox,
						     GamesList2->RomContents[i]->CRC,
						     (int*)&nbrdata);
      
      if(Contents!=NULL)
	for(j=0;j<nbrdata;j++)
	  if (Contents[j]->Size==GamesList2->RomContents[i]->Size)
	    break;

      if(Contents==NULL || (Contents!=NULL && j==nbrdata))
	{
	  if(GamesList2->RomContents[i]->CRC!=0)
	    {
	      SortBox_AddOneUint(RomContentSortBox,
				 GamesList2->RomContents[i]->CRC,
				 GamesList2->RomContents[i]); 

	      GamesList1->RomContents=XREALLOC(GamesList1->RomContents,
					       s_RomContent*,
					       GamesList1->NbrRomContents+1);

	      GamesList1->RomContents[GamesList1->NbrRomContents]=
		GamesList2->RomContents[i];
	      GamesList1->NbrRomContents++;
	    };
	}; 
      for(j=0;j<nbrdata;j++)
	Contents[j]=NULL;
      XFREE(Contents);  
    };

  FreeSortBox(RomContentSortBox);
  RomContentSortBox=NULL;

  GamesSortBox=InitSortBox(SB_ALPHANUM,NULL);
  for(i=0;i<GamesList1->NbrGames;i++)
    SortBox_AddOneWord(GamesSortBox,GamesList1->Games[i]->Name,
		       GamesList1->Games[i]);
  
  for(i=0;i<GamesList2->NbrGames;i++)
    {
      Games=(s_GameInfo**)SortBox_FindTheWord(GamesSortBox,
					      GamesList2->Games[i]->Name,
					      (int*)&nbrdata);
      if(Games==NULL)
	{
	  GamesList1->Games=XREALLOC(GamesList1->Games,
				     s_GameInfo*,GamesList1->NbrGames+1);

	  GamesList1->Games[GamesList1->NbrGames]=XCALLOC(s_GameInfo,1);
	  GamesList1->Games[GamesList1->NbrGames]->Name=
	    XSTRDUP(GamesList2->Games[i]->Name);

	  if(GamesList2->Games[i]->NbrRoms!=0)
	    {
	      GamesList1->Games[GamesList1->NbrGames]->Roms=
		XCALLOC(s_RomInfo*,GamesList2->Games[i]->NbrRoms);
	      for(j=0;j<GamesList2->Games[i]->NbrRoms;j++)
		GamesList1->Games[GamesList1->NbrGames]->Roms[j]=
		  GamesList2->Games[i]->Roms[j];
	      GamesList1->Games[GamesList1->NbrGames]->NbrRoms=
		GamesList2->Games[i]->NbrRoms;
	    };

	  if(GamesList2->Games[i]->NbrDisks!=0)
	    {
	      GamesList1->Games[GamesList1->NbrGames]->Disks=
		XCALLOC(s_DiskInfo*,GamesList2->Games[i]->NbrDisks);
	      for(j=0;j<GamesList2->Games[i]->NbrDisks;j++)
		GamesList1->Games[GamesList1->NbrGames]->Disks[j]=
		  GamesList2->Games[i]->Disks[j];
	      GamesList1->Games[GamesList1->NbrGames]->NbrDisks=
		GamesList2->Games[i]->NbrDisks;
	    };
	  
	  GamesList1->Games[GamesList1->NbrGames]->More=
	    GamesList2->Games[i]->More;

	  SortBox_AddOneWord(GamesSortBox,
			     GamesList1->Games[GamesList1->NbrGames]->Name,
			     GamesList1->Games[GamesList1->NbrGames]);
	  
	  GamesList1->NbrGames++;
	}
      else
	{
	  for(j=0;j<GamesList2->Games[i]->NbrRoms;j++)
	    {
	      for(k=0;k<Games[0]->NbrRoms;k++)
		if(GamesList2->Games[i]->Roms[j]->Content->CRC==
		   Games[0]->Roms[k]->Content->CRC &&
		   GamesList2->Games[i]->Roms[j]->Content->Size==
		   Games[0]->Roms[k]->Content->Size &&
		   strcmp(GamesList2->Games[i]->Roms[j]->Name,
			  Games[0]->Roms[k]->Name)==0)
		  break;
	      if(k==Games[0]->NbrRoms)
		{
		  Games[0]->Roms=XREALLOC(Games[0]->Roms,
					  s_RomInfo*,
					  Games[0]->NbrRoms+1);
		  Games[0]->Roms[Games[0]->NbrRoms]=
		    GamesList2->Games[i]->Roms[j];
		  Games[0]->NbrRoms++;
		};
	    };
	  for(j=0;j<GamesList2->Games[i]->NbrDisks;j++)
	    {
	      for(k=0;k<Games[0]->NbrDisks;k++)
		if(strcmp(GamesList2->Games[i]->Disks[j]->Name,
			  Games[0]->Disks[k]->Name)==0)
		  break;
	      if(k==Games[0]->NbrDisks)
		{
		  Games[0]->Disks=XREALLOC(Games[0]->Disks,
					   s_DiskInfo*,
					   Games[0]->NbrDisks+1);
		  Games[0]->Disks[Games[0]->NbrDisks]=
		    GamesList2->Games[i]->Disks[j];
		  Games[0]->NbrDisks++;
		};
	    };
	};
      XFREE(Games);
    };

  for(i=0;i<GamesList2->NbrResources;i++)
    {
      for(j=0;j<GamesList1->NbrResources;j++)
	if(strcmp(GamesList2->Resources[i]->Name,
		  GamesList1->Resources[j]->Name)==0)
	  break;
      
      if(j==GamesList1->NbrResources)
	{
	  GamesList1->Resources=XREALLOC(GamesList1->Resources,
					 s_ResourceInfo*,
					 GamesList1->NbrResources+1);

	  GamesList1->Resources[GamesList1->NbrResources]=
	    XCALLOC(s_ResourceInfo,1);

	  GamesList1->Resources[GamesList1->NbrResources]->Name=
	    XSTRDUP(GamesList2->Resources[i]->Name);
	  GamesList1->Resources[GamesList1->NbrResources]->AssociatedGames=
	    XCALLOC(s_GameInfo*,GamesList2->Resources[i]->NbrAssociatedGames);
		   
	  for(k=0;k<GamesList2->Resources[i]->NbrAssociatedGames;k++)
	    {
	      Games=(s_GameInfo**)SortBox_FindTheWord(GamesSortBox,
						      GamesList2->Resources[i]->AssociatedGames[k]->Name,
						      (int*)&nbrdata);
	      if(Games==NULL)
		{
		  LPRINTF(ERROR,"The game '%s' associated to the resource '%s'"\
			  " has not been found in the merged gameslist",
			  GamesList2->Resources[i]->AssociatedGames[k]->Name,
			  GamesList2->Resources[i]->Name);
		  GamesList1->Resources[GamesList1->NbrResources]->AssociatedGames[k]=NULL;
		} 
	      else
		GamesList1->Resources[GamesList1->NbrResources]->AssociatedGames[k]=Games[0];
	      XFREE(Games);
	    }

	  GamesList1->Resources[GamesList1->NbrResources]->NbrAssociatedGames=GamesList2->Resources[i]->NbrAssociatedGames;
	  Games=
	    (s_GameInfo**)SortBox_FindTheWord(GamesSortBox,
					      GamesList2->Resources[i]->Game->Name,
					      (int*)&nbrdata);
	  if(Games==NULL)
	    {
	      LPRINTF(ERROR,"The game '%s' corresponding to the resource '%s'"\
		      " has not been found in the merged gameslist",
		      GamesList2->Resources[i]->Game->Name,
		      GamesList2->Resources[i]->Name);
	      GamesList1->Resources[GamesList1->NbrResources]->Game=NULL;
	    }
	  else
	    GamesList1->Resources[GamesList1->NbrResources]->Game=Games[0];
	  XFREE(Games);

	  GamesList1->NbrResources++;
	}
      else
	{
	  for(k=0;k<GamesList2->Resources[i]->NbrAssociatedGames;k++)
	    {
	      for(l=0;l<GamesList1->Resources[j]->NbrAssociatedGames;l++)
		if(strcmp(GamesList2->Resources[i]->AssociatedGames[k]->Name,
			  GamesList1->Resources[j]->AssociatedGames[l]->Name)==0)
		  break;
	      if(l==GamesList1->Resources[j]->NbrAssociatedGames)
		{
		  Games=(s_GameInfo**)SortBox_FindTheWord(GamesSortBox,
							  GamesList2->Resources[i]->AssociatedGames[k]->Name,
							  (int*)&nbrdata);
		  if(Games==NULL)
		    LPRINTF(ERROR,
			    "The game '%s' associated to the resource '%s'"\
			    " has not been found in the merged gameslist",
			    GamesList2->Resources[i]->AssociatedGames[k]->Name,
			    GamesList2->Resources[i]->Name);
		  else
		    {
		      GamesList1->Resources[j]->AssociatedGames=
			XREALLOC(GamesList1->Resources[j]->AssociatedGames,
				 s_GameInfo*,
				 GamesList1->Resources[j]->NbrAssociatedGames+1);
		      GamesList1->Resources[j]->AssociatedGames[GamesList1->Resources[j]->NbrAssociatedGames]=Games[0];
		      GamesList1->Resources[j]->NbrAssociatedGames++;
		    }
		  XFREE(Games);
		};
	    };
	};
    };

  FreeSortBox(GamesSortBox);
  return GamesList1;
};

