// Copyright 2020 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "generator/internal/connection_generator.h"
#include "generator/internal/codegen_utils.h"
#include "generator/internal/descriptor_utils.h"
#include "generator/internal/longrunning.h"
#include "generator/internal/pagination.h"
#include "generator/internal/predicate_utils.h"
#include "generator/internal/printer.h"
#include <google/protobuf/descriptor.h>

namespace google {
namespace cloud {
namespace generator_internal {

ConnectionGenerator::ConnectionGenerator(
    google::protobuf::ServiceDescriptor const* service_descriptor,
    VarsDictionary service_vars,
    std::map<std::string, VarsDictionary> service_method_vars,
    google::protobuf::compiler::GeneratorContext* context)
    : ServiceCodeGenerator("connection_header_path", "connection_cc_path",
                           service_descriptor, std::move(service_vars),
                           std::move(service_method_vars), context) {}

Status ConnectionGenerator::GenerateHeader() {
  HeaderPrint(CopyrightLicenseFileHeader());
  HeaderPrint(  // clang-format off
    "\n"
    "// Generated by the Codegen C++ plugin.\n"
    "// If you make any local changes, they will be lost.\n"
    "// source: $proto_file_name$\n"
    "\n"
    "#ifndef $header_include_guard$\n"
    "#define $header_include_guard$\n");
  // clang-format on

  auto endpoint_location_style = EndpointLocationStyle();

  // includes
  HeaderPrint("\n");
  HeaderLocalIncludes(
      {vars("idempotency_policy_header_path"), vars("retry_traits_header_path"),
       "google/cloud/backoff_policy.h",
       HasLongrunningMethod() || HasAsyncMethod() ? "google/cloud/future.h"
                                                  : "",
       "google/cloud/internal/retry_policy_impl.h", "google/cloud/options.h",
       HasLongrunningMethod() ? "google/cloud/polling_policy.h" : "",
       "google/cloud/status_or.h",
       HasStreamingReadMethod() || HasPaginatedMethod()
           ? "google/cloud/stream_range.h"
           : "",
       HasBidirStreamingMethod()
           ? "google/cloud/internal/async_read_write_stream_impl.h"
           : "",
       IsExperimental() ? "google/cloud/experimental_tag.h" : "",
       "google/cloud/version.h"});
  HeaderSystemIncludes(
      {vars("proto_header_path"), vars("additional_pb_header_paths"),
       HasGRPCLongrunningOperation() ? "google/longrunning/operations.grpc.pb.h"
                                     : "",
       "memory"});
  switch (endpoint_location_style) {
    case ServiceConfiguration::LOCATION_DEPENDENT:
    case ServiceConfiguration::LOCATION_DEPENDENT_COMPAT:
      HeaderSystemIncludes({"string"});
      break;
    default:
      break;
  }

  auto result = HeaderOpenNamespaces();
  if (!result.ok()) return result;

  HeaderPrint(R"""(
/// The retry policy for `$connection_class_name$`.
class $retry_policy_name$ : public ::google::cloud::RetryPolicy {
 public:
  /// Creates a new instance of the policy, reset to the initial state.
  virtual std::unique_ptr<$retry_policy_name$> clone() const = 0;
};

/**
 * A retry policy for `$connection_class_name$` based on counting errors.
 *
 * This policy stops retrying if:
 * - An RPC returns a non-transient error.
 * - More than a prescribed number of transient failures is detected.
 *$transient_errors_comment$
 */
class $limited_error_count_retry_policy_name$ : public $retry_policy_name$ {
 public:
  /**
   * Create an instance that tolerates up to @p maximum_failures transient
   * errors.
   *
   * @note Disable the retry loop by providing an instance of this policy with
   *     @p maximum_failures == 0.
   */
  explicit $limited_error_count_retry_policy_name$(int maximum_failures)
    : impl_(maximum_failures) {}

  $limited_error_count_retry_policy_name$(
      $limited_error_count_retry_policy_name$&& rhs) noexcept
    : $limited_error_count_retry_policy_name$(rhs.maximum_failures()) {}
  $limited_error_count_retry_policy_name$(
      $limited_error_count_retry_policy_name$ const& rhs) noexcept
    : $limited_error_count_retry_policy_name$(rhs.maximum_failures()) {}

  int maximum_failures() const { return impl_.maximum_failures(); }

  bool OnFailure(Status const& status) override {
    return impl_.OnFailure(status);
  }
  bool IsExhausted() const override { return impl_.IsExhausted(); }
  bool IsPermanentFailure(Status const& status) const override {
    return impl_.IsPermanentFailure(status);
  }
  std::unique_ptr<$retry_policy_name$> clone() const override {
    return std::make_unique<$limited_error_count_retry_policy_name$>(
        maximum_failures());
  }

  // This is provided only for backwards compatibility.
  using BaseType = $retry_policy_name$;

 private:
  google::cloud::internal::LimitedErrorCountRetryPolicy<$product_internal_namespace$::$retry_traits_name$> impl_;
};

/**
 * A retry policy for `$connection_class_name$` based on elapsed time.
 *
 * This policy stops retrying if:
 * - An RPC returns a non-transient error.
 * - The elapsed time in the retry loop exceeds a prescribed duration.
 *$transient_errors_comment$
 */
class $limited_time_retry_policy_name$ : public $retry_policy_name$ {
 public:
  /**
   * Constructor given a `std::chrono::duration<>` object.
   *
   * @tparam DurationRep a placeholder to match the `Rep` tparam for @p
   *     duration's type. The semantics of this template parameter are
   *     documented in `std::chrono::duration<>`. In brief, the underlying
   *     arithmetic type used to store the number of ticks. For our purposes it
   *     is simply a formal parameter.
   * @tparam DurationPeriod a placeholder to match the `Period` tparam for @p
   *     duration's type. The semantics of this template parameter are
   *     documented in `std::chrono::duration<>`. In brief, the length of the
   *     tick in seconds, expressed as a `std::ratio<>`. For our purposes it is
   *     simply a formal parameter.
   * @param maximum_duration the maximum time allowed before the policy expires.
   *     While the application can express this time in any units they desire,
   *     the class truncates to milliseconds.
   *
   * @see https://en.cppreference.com/w/cpp/chrono/duration for more information
   *     about `std::chrono::duration`.
   */
  template <typename DurationRep, typename DurationPeriod>
  explicit $limited_time_retry_policy_name$(
      std::chrono::duration<DurationRep, DurationPeriod> maximum_duration)
    : impl_(maximum_duration) {}

  $limited_time_retry_policy_name$($limited_time_retry_policy_name$&& rhs) noexcept
    : $limited_time_retry_policy_name$(rhs.maximum_duration()) {}
  $limited_time_retry_policy_name$($limited_time_retry_policy_name$ const& rhs) noexcept
    : $limited_time_retry_policy_name$(rhs.maximum_duration()) {}

  std::chrono::milliseconds maximum_duration() const {
    return impl_.maximum_duration();
  }

  bool OnFailure(Status const& status) override {
    return impl_.OnFailure(status);
  }
  bool IsExhausted() const override { return impl_.IsExhausted(); }
  bool IsPermanentFailure(Status const& status) const override {
    return impl_.IsPermanentFailure(status);
  }
  std::unique_ptr<$retry_policy_name$> clone() const override {
    return std::make_unique<$limited_time_retry_policy_name$>(
        maximum_duration());
  }

  // This is provided only for backwards compatibility.
  using BaseType = $retry_policy_name$;

 private:
  google::cloud::internal::LimitedTimeRetryPolicy<$product_internal_namespace$::$retry_traits_name$> impl_;
};

/**
 * The `$connection_class_name$` object for `$client_class_name$`.
 *
 * This interface defines virtual methods for each of the user-facing overload
 * sets in `$client_class_name$`. This allows users to inject custom behavior
 * (e.g., with a Google Mock object) when writing tests that use objects of type
 * `$client_class_name$`.
 *
 * To create a concrete instance, see `Make$connection_class_name$()`.
 *
 * For mocking, see `$product_namespace$_mocks::$mock_connection_class_name$`.
 */
class $connection_class_name$ {
 public:
  virtual ~$connection_class_name$() = 0;

  virtual Options options() { return Options{}; }
)""");

  for (auto const& method : methods()) {
    if (IsBidirStreaming(method)) {
      HeaderPrintMethod(method, __FILE__, __LINE__,
                        R"""(
  virtual std::unique_ptr<::google::cloud::AsyncStreamingReadWriteRpc<
      $request_type$,
      $response_type$>>
  Async$method_name$();
)""");
      continue;
    }
    HeaderPrintMethod(
        method,
        {MethodPattern(
             {
                 {IsResponseTypeEmpty,
                  // clang-format off
    "\n  virtual Status\n",
    "\n  virtual StatusOr<$response_type$>\n"},
   {"  $method_name$($request_type$ const& request);\n"}
                 // clang-format on
             },
             All(IsNonStreaming, Not(IsLongrunningOperation),
                 Not(IsPaginated))),
         MethodPattern(
             {
                 {IsResponseTypeEmpty,
                  // clang-format off
    "\n  virtual future<Status>\n",
    "\n  virtual future<StatusOr<$longrunning_deduced_response_type$>>\n"},
   {"  $method_name$($request_type$ const& request);\n"}
                 // clang-format on
             },
             All(IsNonStreaming, IsLongrunningOperation, Not(IsPaginated))),
         MethodPattern(
             {
                 // clang-format off
   {"\n  virtual StreamRange<$range_output_type$>\n"
    "  $method_name$($request_type$ request);\n"},
                 // clang-format on
             },
             All(IsNonStreaming, Not(IsLongrunningOperation), IsPaginated)),
         MethodPattern(
             {
                 // clang-format off
   {"\n  virtual StreamRange<$response_type$>\n"
    "  $method_name$($request_type$ const& request);\n"},
                 // clang-format on
             },
             IsStreamingRead)},
        __FILE__, __LINE__);
  }

  for (auto const& method : async_methods()) {
    if (IsStreamingRead(method)) continue;
    if (IsStreamingWrite(method)) continue;
    HeaderPrintMethod(
        method,
        {MethodPattern(
            {
                {IsResponseTypeEmpty,
                 // clang-format off
    "\n  virtual future<Status>\n",
    "\n  virtual future<StatusOr<$response_type$>>\n"},
   {"  Async$method_name$($request_type$ const& request);\n"}
                // clang-format on
            },
            All(IsNonStreaming, Not(IsLongrunningOperation),
                Not(IsPaginated)))},
        __FILE__, __LINE__);
  }

  // close abstract interface Connection base class
  HeaderPrint("};\n");

  if (HasGenerateGrpcTransport()) {
    EmitFactoryFunctionDeclaration(endpoint_location_style);
  }

  HeaderCloseNamespaces();

  // close header guard
  HeaderPrint("\n#endif  // $header_include_guard$\n");
  return {};
}

Status ConnectionGenerator::GenerateCc() {
  CcPrint(CopyrightLicenseFileHeader());
  CcPrint(  // clang-format off
    "\n"
    "// Generated by the Codegen C++ plugin.\n"
    "// If you make any local changes, they will be lost.\n"
    "// source: $proto_file_name$\n");
  // clang-format on

  // includes
  CcPrint("\n");
  CcLocalIncludes(
      {vars("connection_header_path"), vars("options_header_path"),
       HasGenerateGrpcTransport() ? vars("connection_impl_header_path") : "",
       vars("option_defaults_header_path"),
       vars("tracing_connection_header_path"),
       HasGenerateGrpcTransport() ? vars("stub_factory_header_path") : "",
       "google/cloud/background_threads.h", "google/cloud/common_options.h",
       "google/cloud/credentials.h", "google/cloud/grpc_options.h",
       HasPaginatedMethod() ? "google/cloud/internal/pagination_range.h" : ""});
  CcSystemIncludes({"memory"});

  auto result = CcOpenNamespaces();
  if (!result.ok()) return result;

  CcPrint(R"""(
$connection_class_name$::~$connection_class_name$() = default;
)""");

  for (auto const& method : methods()) {
    if (IsBidirStreaming(method)) {
      CcPrintMethod(method, __FILE__, __LINE__,
                    R"""(
std::unique_ptr<::google::cloud::AsyncStreamingReadWriteRpc<
    $request_type$,
    $response_type$>>
$connection_class_name$::Async$method_name$() {
  return std::make_unique<
      ::google::cloud::internal::AsyncStreamingReadWriteRpcError<
          $request_type$,
          $response_type$>>(
      Status(StatusCode::kUnimplemented, "not implemented"));
}
)""");
      continue;
    }
    CcPrintMethod(
        method,
        {MethodPattern(
             {
                 {IsResponseTypeEmpty,
                  // clang-format off
    "\nStatus\n",
    "\nStatusOr<$response_type$>\n"},
   {"$connection_class_name$::$method_name$(\n"
    "    $request_type$ const&) {\n"
    "  return Status(StatusCode::kUnimplemented, \"not implemented\");\n"
    "}\n"
    },
                 // clang-format on
             },
             All(IsNonStreaming, Not(IsLongrunningOperation),
                 Not(IsPaginated))),
         MethodPattern(
             {
                 {IsResponseTypeEmpty,
                  // clang-format off
    "\nfuture<Status>\n",
    "\nfuture<StatusOr<$longrunning_deduced_response_type$>>\n"},
   {"$connection_class_name$::$method_name$(\n"
    "    $request_type$ const&) {\n"
    "  return google::cloud::make_ready_future<\n"
    "    StatusOr<$longrunning_deduced_response_type$>>(\n"
    "    Status(StatusCode::kUnimplemented, \"not implemented\"));\n"
    "}\n"
    },
                 // clang-format on
             },
             All(IsNonStreaming, IsLongrunningOperation, Not(IsPaginated))),
         MethodPattern(
             {
                 // clang-format off
   {"\nStreamRange<$range_output_type$> $connection_class_name$::$method_name$(\n"
    "    $request_type$) {  // NOLINT(performance-unnecessary-value-param)\n"
    "  return google::cloud::internal::MakeUnimplementedPaginationRange<\n"
    "      StreamRange<$range_output_type$>>();\n"
    "}\n"
                     // clang-format on
                 },
             },
             All(IsNonStreaming, Not(IsLongrunningOperation), IsPaginated)),
         MethodPattern(
             {
                 // clang-format off
   {"\nStreamRange<$response_type$> $connection_class_name$::$method_name$(\n"
    "    $request_type$ const&) {\n"
    "  return google::cloud::internal::MakeStreamRange<\n"
    "      $response_type$>(\n"
    "      []() -> absl::variant<Status,\n"
    "      $response_type$>{\n"
    "        return Status(StatusCode::kUnimplemented, \"not implemented\");}\n"
    "      );\n"
    "}\n"
                     // clang-format on
                 },
             },
             IsStreamingRead)},
        __FILE__, __LINE__);
  }

  for (auto const& method : async_methods()) {
    if (IsStreamingRead(method)) continue;
    if (IsStreamingWrite(method)) continue;
    CcPrintMethod(
        method,
        {MethodPattern({{IsResponseTypeEmpty, R"""(
future<Status>
$connection_class_name$::Async$method_name$(
    $request_type$ const&) {
  return google::cloud::make_ready_future(
      Status(StatusCode::kUnimplemented, "not implemented"));
}
)""",
                         R"""(
future<StatusOr<$response_type$>>
$connection_class_name$::Async$method_name$(
    $request_type$ const&) {
  return google::cloud::make_ready_future<
    StatusOr<$response_type$>>(
    Status(StatusCode::kUnimplemented, "not implemented"));
}
)"""}},
                       All(IsNonStreaming, Not(IsLongrunningOperation),
                           Not(IsPaginated)))},
        __FILE__, __LINE__);
  }

  if (HasGenerateGrpcTransport()) {
    EmitFactoryFunctionDefinition(EndpointLocationStyle());
  }

  CcCloseNamespaces();

  // TODO(#8234): This is a special case for backwards compatibility of the
  //     streaming update function.
  if (vars().at("service_name") == "BigQueryRead") {
    CcOpenForwardingNamespaces();
    CcPrint(R"""(
void BigQueryReadReadRowsStreamingUpdater(
    google::cloud::bigquery::storage::v1::ReadRowsResponse const& response,
    google::cloud::bigquery::storage::v1::ReadRowsRequest& request) {
  return bigquery_storage_v1_internal::BigQueryReadReadRowsStreamingUpdater(response,
                                                                 request);
}
)""");
    CcCloseNamespaces();
  }

  return {};
}

std::string ConnectionGenerator::ConnectionFactoryFunctionArguments() const {
  std::string args;
  if (IsExperimental()) args += "ExperimentalTag, ";
  switch (EndpointLocationStyle()) {
    case ServiceConfiguration::LOCATION_DEPENDENT:
    case ServiceConfiguration::LOCATION_DEPENDENT_COMPAT:
      args += "std::string const& location, ";
      break;
    default:
      break;
  }
  args += "Options options";
  return args;
}

void ConnectionGenerator::EmitFactoryFunctionDeclaration(
    ServiceConfiguration::EndpointLocationStyle endpoint_location_style) {
  HeaderPrint(R"""(
/**
 * A factory function to construct an object of type `$connection_class_name$`.
 *
 * The returned connection object should not be used directly; instead it
 * should be passed as an argument to the constructor of $client_class_name$.
 *
 * The optional @p options argument may be used to configure aspects of the
 * returned `$connection_class_name$`. Expected options are any of the types in
 * the following option lists:
 *
 * - `google::cloud::CommonOptionList`
 * - `google::cloud::GrpcOptionList`
 * - `google::cloud::UnifiedCredentialsOptionList`
 * - `google::cloud::$product_namespace$::$service_name$PolicyOptionList`
 *
 * @note Unexpected options will be ignored. To log unexpected options instead,
 *     set `GOOGLE_CLOUD_CPP_ENABLE_CLOG=yes` in the environment.
 *)""");
  switch (endpoint_location_style) {
    case ServiceConfiguration::LOCATION_DEPENDENT:
    case ServiceConfiguration::LOCATION_DEPENDENT_COMPAT:
      HeaderPrint(R"""(
 * @param location Sets the prefix for the default `EndpointOption` value.)""");
      break;
    default:
      break;
  }
  HeaderPrint(R"""(
 * @param options (optional) Configure the `$connection_class_name$` created by
 * this function.
 */
std::shared_ptr<$connection_class_name$> Make$connection_class_name$(
)""");
  HeaderPrint("    " + ConnectionFactoryFunctionArguments() + " = {});\n");

  switch (endpoint_location_style) {
    case ServiceConfiguration::LOCATION_DEPENDENT_COMPAT:
      HeaderPrint(R"""(
/**
 * A backwards-compatible version of the previous factory function.  Unless
 * the service also offers a global endpoint, the default value of the
 * `EndpointOption` may be useless, in which case it must be overridden.
 *
 * @deprecated Please use the `location` overload instead.
 */
std::shared_ptr<$connection_class_name$> Make$connection_class_name$(
    Options options = {});
)""");
      break;
    default:
      break;
  }
}

void ConnectionGenerator::EmitFactoryFunctionDefinition(
    ServiceConfiguration::EndpointLocationStyle endpoint_location_style) {
  CcPrint(R"""(
std::shared_ptr<$connection_class_name$> Make$connection_class_name$(
)""");
  CcPrint("    " + ConnectionFactoryFunctionArguments() + ") {");
  CcPrint(R"""(
  internal::CheckExpectedOptions<CommonOptionList, GrpcOptionList,
      UnifiedCredentialsOptionList,
      $service_name$PolicyOptionList>(options, __func__);
  options = $product_internal_namespace$::$service_name$DefaultOptions(
)""");
  CcPrint("      ");
  switch (endpoint_location_style) {
    case ServiceConfiguration::LOCATION_DEPENDENT:
    case ServiceConfiguration::LOCATION_DEPENDENT_COMPAT:
      CcPrint("location, ");
      break;
    default:
      break;
  }
  CcPrint("std::move(options));");
  CcPrint(R"""(
  auto background = internal::MakeBackgroundThreadsFactory(options)();
  auto stub = $product_internal_namespace$::CreateDefault$stub_class_name$(
    background->cq(), options);
  return $product_internal_namespace$::Make$tracing_connection_class_name$(
      std::make_shared<$product_internal_namespace$::$connection_class_name$Impl>(
      std::move(background), std::move(stub), std::move(options)));
}
)""");

  switch (endpoint_location_style) {
    case ServiceConfiguration::LOCATION_DEPENDENT_COMPAT:
      CcPrint(R"""(
std::shared_ptr<$connection_class_name$> Make$connection_class_name$(
    Options options) {
  return Make$connection_class_name$(std::string{}, std::move(options));
}
)""");
      break;
    default:
      break;
  }
}

}  // namespace generator_internal
}  // namespace cloud
}  // namespace google
