/***************************************************************************
 *   Copyright (C) 2007 by Sergio Pistone                                  *
 *   sergio_pistone@yahoo.com.ar                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "input.h"

Input::Input( unsigned long timeLimitMS ):
	m_isOpened( false ),
	m_totalTimeMS( 0 ),
	m_currentTimeMS( 0 ),
	m_sampleRate( 0 ),
	m_channels( 0 ),
	m_bitsPerSample( 0 ),
	m_timeLimitMS( timeLimitMS ),
	m_bytesLimit( 0 ),
	m_bytesRead( 0 )
{
}

Input::~Input()
{
}

bool Input::open()
{
	if ( m_isOpened )
		return false;

	m_isOpened = _open();
	if ( m_isOpened )
	{
		m_bytesRead = 0;
		m_bytesLimit = getBlockAlign()*(unsigned long long)(getTimeLimit()*(getSampleRate()/1000.0));
	}

	return m_isOpened;
}

bool Input::close()
{
	if ( ! m_isOpened )
		return false;

	m_isOpened = ! _close();
	if ( ! m_isOpened )
	{
		m_bytesLimit = 0;
		m_bytesRead = 0;
	}

	return ! m_isOpened;
}

bool Input::isOpened()
{
	return m_isOpened;
}

bool Input::isClosed()
{
	return ! m_isOpened;
}

unsigned long Input::getTimeLimit()
{
	return m_timeLimitMS;
}

void Input::setTimeLimit( unsigned long timeLimitMS )
{
	m_timeLimitMS = timeLimitMS;
}

unsigned long Input::getTotalTime()
{
	unsigned long totalTimeMS = isOpened() ? m_totalTimeMS : 0;
	return m_timeLimitMS != NoLimit && totalTimeMS > m_timeLimitMS ? m_timeLimitMS : totalTimeMS;
}

unsigned long Input::getCurrentTime()
{
	return isOpened() ? m_currentTimeMS : 0;
}

unsigned long Input::getSampleRate()
{
	return isOpened() ? m_sampleRate : 0;
}

unsigned int Input::getChannels()
{
	return isOpened() ? m_channels : 0;
}

unsigned int Input::getBitsPerSample()
{
	return isOpened() ? m_bitsPerSample : 0;
}

unsigned int Input::getBlockAlign()
{
	return isOpened() ? Input::getBlockAlign( getChannels(), getBitsPerSample() ) : 0;
}

unsigned int Input::getBlockAlign( unsigned int channels, unsigned int bitsPerSample )
{
	return channels*(bitsPerSample/8);
}

bool Input::getPCMChunk( unsigned char* data, unsigned long limit, unsigned long &read )
{
	read = 0;

	if ( ! isOpened() )
		return false;

	if ( m_timeLimitMS != NoLimit && m_bytesRead + limit > m_bytesLimit )
		limit = m_bytesLimit - m_bytesRead;

	bool moreData = _getPCMChunk( data, limit, read );

	m_bytesRead += read;

	return m_timeLimitMS != NoLimit && m_bytesRead >= m_bytesLimit ? false : moreData;
}
