//go:build fips

package fips

import (
	"slices"

	"golang.org/x/crypto/ssh"
)

// fipsKexAlgos specifies FIPS approved key-exchange algorithms implemented
// by this package.
var fipsKexAlgos = []string{ssh.KeyExchangeECDHP256, ssh.KeyExchangeECDHP384}

// fipsCiphers specifies FIPS approved cipher algorithms implemented by this
// package.
var fipsCiphers = []string{
	ssh.CipherAES128GCM, ssh.CipherAES256GCM,
	ssh.CipherAES128CTR, ssh.CipherAES192CTR, ssh.CipherAES256CTR,
}

// fipsMACs specifies FIPS approved MAC algorithms implemented by this
// package.
var fipsMACs = []string{
	ssh.HMACSHA256ETM, ssh.HMACSHA512ETM,
	ssh.HMACSHA256, ssh.HMACSHA512,
}

// fipsHostKeyAlgos specifies FIPS approved host-key algorithms implemented
// by this package.
var fipsHostKeyAlgos = []string{
	ssh.CertAlgoRSASHA256v01, ssh.CertAlgoRSASHA512v01,
	ssh.CertAlgoECDSA256v01, ssh.CertAlgoECDSA384v01,
	ssh.KeyAlgoECDSA256, ssh.KeyAlgoECDSA384,
	ssh.KeyAlgoRSASHA256, ssh.KeyAlgoRSASHA512,
}

// fipsPubKeyAuthAlgos specifies FIPS approved public key authentication algorithms.
var fipsPubKeyAuthAlgos = []string{
	ssh.KeyAlgoRSASHA256, ssh.KeyAlgoRSASHA512,
	ssh.KeyAlgoECDSA256, ssh.KeyAlgoECDSA384,
}

type Algorithms struct {
	Ciphers        []string
	MACs           []string
	KeyExchanges   []string
	HostKeys       []string
	PublicKeyAuths []string
}

// SupportedAlgorithms returns algorithms currently implemented by this package,
// excluding those with security issues, which are returned by
// InsecureAlgorithms. The algorithms listed here are in preference order.
// FIPS 140-2 docs: https://csrc.nist.gov/pubs/fips/140-2/upd2/final
func SupportedAlgorithms() ssh.Algorithms {
	return ssh.Algorithms{
		Ciphers:        slices.Clone(fipsCiphers),
		MACs:           slices.Clone(fipsMACs),
		KeyExchanges:   slices.Clone(fipsKexAlgos),
		HostKeys:       slices.Clone(fipsHostKeyAlgos),
		PublicKeyAuths: slices.Clone(fipsPubKeyAuthAlgos),
	}
}
