#!/bin/sh

# Copyright 2012 The ChromiumOS Authors
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

# Generates a header file with a system call table made up of "name",
# syscall_nr entries by including the build target <asm/unistd.h> and
# emitting the list of defines.  Use of the compiler is needed to
# dereference the actual provider of syscall definitions.
#   E.g., asm/unistd_32.h or asm/unistd_64.h, etc.

set -e

if [ $# -ne 1 ] && [ $# -ne 2 ]; then
  echo "Usage: $(basename "$0") OUTFILE"
  echo "Usage: $(basename "$0") INFILE OUTFILE"
  exit 1
fi

build() {
  ${CC:-cc} -dD "${SRC:-.}"/gen_syscalls.c -E "$@"
}
GEN_DEPS=1

if [ $# -eq 2 ]; then
  build() {
    cat "${CAT_FILE}"
  }
  CAT_FILE="$1"
  GEN_DEPS=0
  shift
fi
OUTFILE="$1"

if [ "${GEN_DEPS}" -eq 1 ]; then
  # Generate a dependency file which helps the build tool to see when it
  # should regenerate ${OUTFILE}.
  build -M -MF "${OUTFILE}.d"
fi

# sed expression which extracts system calls that are
# defined via asm/unistd.h.  It converts them from:
#  #define __NR_read foo
# to:
# #ifdef __NR_read
#  { "read", __NR_read },
# #endif
SED_MULTILINE='s/#define __(ARM_)?(NR_)([[:lower:]0-9_]*) (.*)$/#ifdef __\1\2\3\
{ "\1\3", __\1\2\3 },\
#endif/g p;'

cat <<-EOF > "${OUTFILE}"
/* GENERATED BY MAKEFILE */
#include <stddef.h>
#include "gen_syscalls-inl.h"
#include "libsyscalls.h"
const struct syscall_entry syscall_table[] = {
$(build | sed -Ene "${SED_MULTILINE}")
  { NULL, -1 },
};

const size_t syscall_table_size =
    sizeof(syscall_table) / sizeof(syscall_table[0]);
EOF
