/*
 * config_window_handler.cpp
 *
 * Ralf Ulrich 17 Aug 2018 replaced snprintf with iostream: 
 *                         reduce warnings in gcc7, more save
 */
#include "config.h"

#ifdef STDC_HEADERS
# include <stdlib.h>
# include <stddef.h>
#else
# ifdef HAVE_STDLIB_H
#  include <stdlib.h>
# endif
#endif

#include <stdbool.h>
#include <math.h>

#ifdef HAVE_STRING_H
# if !defined STDC_HEADERS && defined HAVE_MEMORY_H
#  include <memory.h>
# endif
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif

#include <glib.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>

#include "interface_common.h"
#include "main_window_handler.h"
#include "select_frame_handler.h"
#include "dir_window_handler.h"
#include "misc_utils.h"
#include "config_rw.h"
#include "config_window_encoders_handler.h"
#include "config_window_handler.h"
#include "gtk_cpp_workaround.h"

#include <sstream> // for ostringstream
#include <iomanip> // for setw
#include <string>
#include <iostream>
#include <map>

extern ConfigRW config_data;

// Window config handlers
void cw_g_path_clicked(GtkWidget *widget, gpointer callback_data);
int cw_general_handler(int ops, _main_data *main_data, GtkWidget *notebook);
int cw_wav_handler(int ops, _main_data *main_data, GtkWidget *notebook);
int cw_encoders_handler(int ops, _main_data *main_data, GtkWidget *notebook);
int cw_players_handler(int ops, _main_data *main_data, GtkWidget *notebook);
int cw_cddb_handler(int ops, _main_data *main_data, GtkWidget *notebook);

void cw_ok_button_clicked(GtkWidget *widget, gpointer callback_data);
void cw_cancel_button_clicked(GtkWidget *widget, gpointer callback_data);

// Encoders page handlers
int cw_encoder_mp3_handler(int ops, _main_data *main_data, GtkWidget *notebook);
int cw_encoder_flac_handler(int ops, _main_data *main_data, GtkWidget *notebook);
int cw_encoder_ogg_handler(int ops, _main_data *main_data, GtkWidget *notebook);
int cw_encoder_musepack_handler(int ops, _main_data *main_data, GtkWidget *notebook);
int cw_encoder_opus_handler(int ops, _main_data *main_data, GtkWidget *notebook);
int cw_encoder_mp2_handler(int ops, _main_data *main_data, GtkWidget *notebook);

void cw_g_path_clicked(GtkWidget *widget, gpointer callback_data)
{
    GtkWidget *entry;
    char *temp;

    entry = (GtkWidget *) callback_data;
    gtk_widget_set_sensitive(widget, false);

    if ((temp = dir_window_handler(
        WIDGET_CREATE, gtk_entry_get_text(GTK_ENTRY(entry)))) != NULL)
    {
        gtk_entry_set_text(GTK_ENTRY(entry), temp);
    }

    gtk_widget_set_sensitive(widget, true);
}

int cw_general_handler(int ops, _main_data *main_data, GtkWidget *notebook)
{
    static GtkWidget *main_frame;
    static GtkWidget *wav_path_entry, *mp3_path_entry;
    static GtkWidget *wav_file_name_format_entry, *encoded_file_name_format_entry;
    static GtkWidget *prepend_char_entry;
    static GtkWidget *encode_from_existing_wav_check_button;
    static GtkWidget *ask_when_file_exists_check_button;
    static GtkWidget *keep_wav_check_button;
    static GtkWidget *eject_disk_check_button;
    char buf[2];

    switch (ops)
    {
        case WIDGET_CREATE :
        {
            GtkWidget *frame, *vbox, *table, *hbox, *label, *button;

            main_frame = gtk_frame_new(_("General Configuration"));
            gtk_container_set_border_width(GTK_CONTAINER(main_frame), MAIN_FRAME_BORDER_WIDTH);

            vbox = gtk_vbox_new(false, 0);
            gtk_container_add(GTK_CONTAINER(main_frame), vbox);

            // Frame for wav
            frame = gtk_frame_new(_("Wav file"));
            gtk_container_set_border_width(GTK_CONTAINER(frame), MAIN_FRAME_BORDER_WIDTH);
            gtk_box_pack_start(GTK_BOX(vbox), frame, false, false, 0);

            table = gtk_table_new(2, 2, false);
            gtk_container_set_border_width(GTK_CONTAINER(table), 3);
            gtk_container_add(GTK_CONTAINER(frame), table);

            label = gtk_label_new(_("File name format: "));
            gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,
                             0, GTK_EXPAND | GTK_FILL, 0, 0);

            wav_file_name_format_entry = gtk_entry_new_with_max_length(MAX_FILE_PATH_LENGTH - 3);
            gtk_entry_set_text(GTK_ENTRY(wav_file_name_format_entry),
                               config.wav_file_name_format.c_str());
            gtk_table_attach(GTK_TABLE(table),
                             wav_file_name_format_entry,
                             1, 2, 0, 1,
                             GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

            // Create entry first to give its address as callback data
            wav_path_entry = gtk_entry_new_with_max_length(MAX_FILE_PATH_LENGTH - 3);
            button = gtk_button_new_with_label(_("Target Directory: "));
            g_signal_connect(G_OBJECT(button), "clicked",
                             G_CALLBACK(cw_g_path_clicked),
                             wav_path_entry);
            gtk_table_attach(GTK_TABLE(table), button, 0, 1, 1, 2,
                             0, GTK_EXPAND | GTK_FILL, 2, 2);

            gtk_entry_set_text(GTK_ENTRY(wav_path_entry), config.wav_path.c_str());
            gtk_table_attach(GTK_TABLE(table),
                             wav_path_entry,
                             1, 2, 1, 2,
                             GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

            // Frame for encoded file
            frame = gtk_frame_new(_("Encoded file"));
            gtk_container_set_border_width(GTK_CONTAINER(frame), MAIN_FRAME_BORDER_WIDTH);
            gtk_box_pack_start(GTK_BOX(vbox), frame, false, false, 0);

            table = gtk_table_new(2, 2, false);
            gtk_container_set_border_width(GTK_CONTAINER(table), 3);
            gtk_container_add(GTK_CONTAINER(frame), table);

            label = gtk_label_new(_("File name format: "));
            gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,
                             0, GTK_EXPAND | GTK_FILL, 0, 0);

            encoded_file_name_format_entry = gtk_entry_new_with_max_length(MAX_FILE_PATH_LENGTH - 3);
            gtk_entry_set_text(GTK_ENTRY(encoded_file_name_format_entry),
                               config.encoded_file_name_format.c_str());
            gtk_table_attach(GTK_TABLE(table),
                             encoded_file_name_format_entry,
                             1, 2, 0, 1,
                             GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

            mp3_path_entry = gtk_entry_new_with_max_length(MAX_FILE_PATH_LENGTH - 3);

            button = gtk_button_new_with_label(_("Target Directory: "));
            g_signal_connect(G_OBJECT(button), "clicked",
                             G_CALLBACK(cw_g_path_clicked),
                             mp3_path_entry);
            gtk_table_attach(GTK_TABLE(table), button, 0, 1, 1, 2,
                             0, GTK_EXPAND | GTK_FILL, 2, 2);

            gtk_entry_set_text(GTK_ENTRY(mp3_path_entry),
                               config.mp3_path.c_str());
            gtk_table_attach(GTK_TABLE(table),
                             mp3_path_entry,
                             1, 2, 1, 2,
                             GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

            // Misc options
            hbox = gtk_hbox_new(false, 3);
            gtk_container_set_border_width(GTK_CONTAINER(hbox), VBOX_BORDER_WIDTH);
            gtk_box_pack_start(GTK_BOX(vbox), hbox, false, false, 0);

            label = gtk_label_new(_("Prepend character : "));
            gtk_box_pack_start(GTK_BOX(hbox), label, false, false, 0);

            buf[ 0 ] = config.prepend_char;
            buf[ 1 ] = '\0';
            prepend_char_entry = gtk_entry_new_with_max_length(1);
            gtk_widget_set_size_request(prepend_char_entry, 24, 24);
            gtk_entry_set_text(GTK_ENTRY(prepend_char_entry), buf);
            gtk_box_pack_start(GTK_BOX(hbox), prepend_char_entry, false, false, 0);

            encode_from_existing_wav_check_button = gtk_check_button_new_with_label(_("Encode from existing Wav file"));
            gtk_toggle_button_set_active(
                GTK_TOGGLE_BUTTON(encode_from_existing_wav_check_button),
                config.encode_from_existing_wav);
            gtk_box_pack_start(GTK_BOX(vbox),
                               encode_from_existing_wav_check_button,
                               false, false, 0);

            ask_when_file_exists_check_button = gtk_check_button_new_with_label(_("Ask user when specified file exists"));
            gtk_toggle_button_set_active(
                GTK_TOGGLE_BUTTON(ask_when_file_exists_check_button),
                config.ask_when_file_exists);
            gtk_box_pack_start(GTK_BOX(vbox),
                               ask_when_file_exists_check_button,
                               false, false, 0);

            keep_wav_check_button = gtk_check_button_new_with_label(_("Keep wav files"));
            gtk_toggle_button_set_active(
                GTK_TOGGLE_BUTTON(keep_wav_check_button),
                config.keep_wav);
            gtk_box_pack_start(GTK_BOX(vbox),
                               keep_wav_check_button,
                               false, false, 0);

            eject_disk_check_button = gtk_check_button_new_with_label(_("Eject disk when finished"));
            gtk_toggle_button_set_active(
                GTK_TOGGLE_BUTTON(eject_disk_check_button),
                config.eject_disk_when_finished);
            gtk_box_pack_start(GTK_BOX(vbox), eject_disk_check_button, false, false, 0);

            label = gtk_label_new(_("General"));
            gtk_notebook_append_page_menu(GTK_NOTEBOOK(notebook),
                                          main_frame, label, label);
            gtk_widget_show_all(main_frame);
            return 0;
        }

        case OP_OK :
        {
            // Wav
            if(strlen(gtk_entry_get_text(
                          GTK_ENTRY(wav_file_name_format_entry))) == 0
                    || strlen(gtk_entry_get_text(
                                  GTK_ENTRY(wav_path_entry))) == 0
                    || strlen(gtk_entry_get_text(
                                  GTK_ENTRY(encoded_file_name_format_entry))) == 0
                    || strlen(gtk_entry_get_text(
                                  GTK_ENTRY(mp3_path_entry))) == 0
                    || strlen(gtk_entry_get_text(
                                  GTK_ENTRY(prepend_char_entry))) == 0)
            {
                err_handler(EMPTY_ENTRY_ERR, _("You need to fill every entry in general page"));
                return -1;
            }

            config.wav_file_name_format = gtk_entry_get_text(GTK_ENTRY(wav_file_name_format_entry));
            config.wav_path = gtk_entry_get_text(GTK_ENTRY(wav_path_entry));
            config.encoded_file_name_format = gtk_entry_get_text(GTK_ENTRY(encoded_file_name_format_entry));
            config.mp3_path = gtk_entry_get_text(GTK_ENTRY(mp3_path_entry));
            config.prepend_char = gtk_entry_get_text(GTK_ENTRY(prepend_char_entry))[0];

            config.encode_from_existing_wav =
                GTK_TOGGLE_BUTTON(encode_from_existing_wav_check_button) ->active
                ? true : false;
            config.ask_when_file_exists =
                GTK_TOGGLE_BUTTON(ask_when_file_exists_check_button) ->active
                ? true : false;
            config.auto_append_extension = true;
            config.keep_wav = GTK_TOGGLE_BUTTON(keep_wav_check_button)->active ? true : false;
            config.eject_disk_when_finished = GTK_TOGGLE_BUTTON(eject_disk_check_button)->active
                ? true : false;

            return 0;
        }

        default: break;
    }

    // Just to avoid warning
    return -1;
}

int cw_wav_handler(int ops, _main_data *main_data, GtkWidget *notebook)
{
    static GtkWidget *main_frame, *ripper_menu, *menu;
    static GtkWidget *extra_options_entry;
    static int num_buttons, option_length;
    static int extra_options_offset, options_offset;
    static struct
    {
        GtkWidget *button;
        const char *arg;
        const char *text;
    } button[] =
    {
        { NULL, "-s ", N_("Force search for drive (ignore /dev/cdrom)") },
        { NULL, "-Z ", N_("Disable paranoia (will act like cdda)") },
        { NULL, "-Y ", N_("Disable extra paranoia") },
        { NULL, "-X ", N_("Disable scratch detection") },
        { NULL, "-W ", N_("Disable scratch repair") },
    };
    static struct
    {
        GtkWidget *menu_item;
        const char *ripper;
        const char *plugin;
        const char *description;
    } ripper_plugins[] =
    {
        { NULL, "cdparanoia", "ripperX_plugin-cdparanoia", "cdparanoia III" }
    };

    const int ripper_plugins_count = sizeof(ripper_plugins) / sizeof(ripper_plugins[0]);

    int i, temp;

    switch (ops)
    {
        case WIDGET_CREATE :
        {
            GtkWidget *vbox, *hbox, *label;

            main_frame = gtk_frame_new(_("Wav Configuration"));
            gtk_container_set_border_width(GTK_CONTAINER(main_frame), MAIN_FRAME_BORDER_WIDTH);

            vbox = gtk_vbox_new(false, 0);
            gtk_container_set_border_width(GTK_CONTAINER(vbox), VBOX_BORDER_WIDTH);
            gtk_container_add(GTK_CONTAINER(main_frame), vbox);

            num_buttons = std::size(button);           // C++ 17
            options_offset = strlen(ripper_plugins[0].ripper) + 1;     // 11
            option_length = strlen(button[0].arg);
            extra_options_offset = options_offset + option_length * num_buttons;

            /* Plugin selector menu */
            label = gtk_label_new(_("Ripper plugin"));
            gtk_box_pack_start(GTK_BOX(vbox), label, false, false, 0);
            menu = gtk_option_menu_new();
            ripper_menu = gtk_menu_new();

            for(i = 0; i < ripper_plugins_count; i++)
            {
                ripper_plugins[i].menu_item = gtk_menu_item_new_with_label(ripper_plugins[i].description);
                gtk_menu_append(GTK_MENU(ripper_menu), ripper_plugins[i].menu_item);
            }

            /* set menu to current plugin */
            for(i = 0; i < ripper_plugins_count; i++)
                if (config.ripper.plugin == ripper_plugins[i].plugin)
                    gtk_menu_set_active(GTK_MENU(ripper_menu), i);

            gtk_option_menu_set_menu(GTK_OPTION_MENU(menu), ripper_menu);
            gtk_box_pack_start(GTK_BOX(vbox), menu, false, false, 0);

            /* options */
            for(i = 0; i < num_buttons; i++)
            {
                temp = config.ripper.ripper[ options_offset + option_length * i + 1 ]
                       == button[ i ].arg[ 1 ] ? true : false;
                button[ i ].button = gtk_check_button_new_with_label(button[ i ].text);
                gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button[ i ].button),
                                             temp);
                gtk_box_pack_start(GTK_BOX(vbox), button[ i ].button, false, false, 0);
            }

            hbox = gtk_hbox_new(false, 0);
            gtk_box_pack_start(GTK_BOX(vbox), hbox, false, false, 0);

            label = gtk_label_new(_("Extra Options: "));
            gtk_box_pack_start(GTK_BOX(hbox), label, false, false, 0);

            extra_options_entry = gtk_entry_new_with_max_length(MAX_OPTIONS_LENGTH - 20);
            gtk_entry_set_text(GTK_ENTRY(extra_options_entry),
                    (config.ripper.ripper.substr(extra_options_offset)).c_str());
            gtk_box_pack_start(GTK_BOX(hbox), extra_options_entry, false, false, 0);

            label = gtk_label_new(" Wav ");
            gtk_notebook_append_page_menu(GTK_NOTEBOOK(notebook),
                                          main_frame, label, label);
            gtk_widget_show_all(main_frame);
            return 0;
        }

        case OP_OK :
        {
            GtkWidget *item;
            
            config.ripper.ripper = "";

            item = gtk_menu_get_active(GTK_MENU(ripper_menu));
            temp = 0;

            for(i = 0; i < ripper_plugins_count; i++)
            {
                if(item == ripper_plugins[i].menu_item)
                    temp = i;
            }

            std::ostringstream tmp_ripper;
            tmp_ripper << ripper_plugins[temp].ripper << ' ';
            
            config.ripper.plugin = ripper_plugins[temp].plugin;

            for(i = 0; i < num_buttons; i++)
            {
                if(GTK_TOGGLE_BUTTON(button[ i ].button) ->active)
                    tmp_ripper << button[i].arg;
                else
                    tmp_ripper << std::string(option_length, ' ');
            }

            tmp_ripper << gtk_entry_get_text(GTK_ENTRY(extra_options_entry));
            config.ripper.ripper = tmp_ripper.str();

            return 0;
        }

        default: break;
}

    // Just to avoid warning
    return -1;
}

int cw_encoders_handler(int ops, _main_data *main_data, GtkWidget *notebook)
{
    static GtkWidget *priority_scale;
    static GtkObject *priority_adj;

    switch (ops)
    {
        case WIDGET_CREATE :
        {
            GtkWidget *frame, *vbox, *vbox_pri, *label;

            // Main container
            //
            vbox = gtk_vbox_new(false, 3);
            gtk_container_set_border_width(GTK_CONTAINER(vbox), VBOX_BORDER_WIDTH);

            // Encoder priority
            //
            frame = gtk_frame_new(_("Encoder Priority"));
            gtk_box_pack_start(GTK_BOX(vbox), frame, false, false, 3);

            vbox_pri = gtk_vbox_new(false, 3);
            gtk_container_set_border_width(GTK_CONTAINER(vbox_pri), VBOX_BORDER_WIDTH);
            gtk_container_add(GTK_CONTAINER(frame), vbox_pri);

            priority_adj = gtk_adjustment_new(
                (gfloat) CLAMP(config.encoder.priority, MAX_NICE_LEVEL, MIN_NICE_LEVEL),
                (float) MAX_NICE_LEVEL, (float) MIN_NICE_LEVEL, 1.0, 2.0, 0.0);
            priority_scale = gtk_hscale_new(GTK_ADJUSTMENT(priority_adj));
            gtk_scale_set_digits(GTK_SCALE(priority_scale), 0);

            char str_q[20];
            for (int x = 0; x < 20; x++)
            {
                sprintf(str_q, "%d", x);
                gtk_scale_add_mark(GTK_SCALE(priority_scale), x, GTK_POS_BOTTOM, str_q);
            }
            gtk_box_pack_start(GTK_BOX(vbox_pri), priority_scale, false, false, 3);

            label = gtk_label_new(_("0=high priority, 19=system is responsive while encoding"));
            gtk_box_pack_start(GTK_BOX(vbox_pri), label, false, false, 3);

            // Tab - End of encoders
            //
            label = gtk_label_new(_("Encoder"));
            gtk_notebook_append_page_menu(GTK_NOTEBOOK(notebook), vbox, label, label);

            config_window_encoders_handler(WIDGET_CREATE, main_data, vbox);

            return 0;
        }

        case OP_OK :
        {
            config.encoder.priority = CLAMP((int) GTK_ADJUSTMENT(priority_adj)->value, 0, 19);
            config_window_encoders_handler(OP_OK, main_data, NULL);
            return 0;
        }

        default: break;
    }

    // Just to avoid warning
    return -1;
}

int cw_players_handler(int ops, _main_data *main_data, GtkWidget *notebook)
{
    static GtkWidget *main_frame;
    static GtkWidget *cd_play_entry, *cd_stop_entry, *wav_entry, *mp3_entry;

    switch (ops)
    {
        case WIDGET_CREATE :
        {
            GtkWidget *table, *label;

            main_frame = gtk_frame_new(_("Players"));
            gtk_container_set_border_width(GTK_CONTAINER(main_frame), MAIN_FRAME_BORDER_WIDTH);

            table = gtk_table_new(4, 2, false);
            gtk_table_set_row_spacings(GTK_TABLE(table), 5);
            gtk_container_set_border_width(GTK_CONTAINER(table), 3);
            gtk_container_add(GTK_CONTAINER(main_frame), table);

            label = gtk_label_new(_("CD play command: "));
            gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,
                             0, 0, 0, 0);
            cd_play_entry = gtk_entry_new_with_max_length(MAX_COMMAND_LENGTH - 2);
            gtk_entry_set_text(GTK_ENTRY(cd_play_entry),
                               config.cd_player.play_command.c_str());
            gtk_table_attach(GTK_TABLE(table), cd_play_entry, 1, 2, 0, 1,
                             GTK_EXPAND | GTK_FILL, 0, 0, 0);

            label = gtk_label_new(_("CD stop command: "));
            gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,
                             0, 0, 0, 0);
            cd_stop_entry = gtk_entry_new_with_max_length(MAX_COMMAND_LENGTH - 2);
            gtk_entry_set_text(GTK_ENTRY(cd_stop_entry),
                               config.cd_player.stop_command.c_str());
            gtk_table_attach(GTK_TABLE(table), cd_stop_entry, 1, 2, 1, 2,
                             GTK_EXPAND | GTK_FILL, 0, 0, 0);

            label = gtk_label_new(_("Wav play command: "));
            gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3,
                             0, 0, 0, 0);
            wav_entry = gtk_entry_new_with_max_length(MAX_COMMAND_LENGTH - 2);
            gtk_entry_set_text(GTK_ENTRY(wav_entry),
                               config.wav_player.command.c_str());
            gtk_table_attach(GTK_TABLE(table), wav_entry, 1, 2, 2, 3,
                             GTK_EXPAND | GTK_FILL, 0, 0, 0);

            label = gtk_label_new(_("Mp3 play command: "));
            gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4,
                             0, 0, 0, 0);
            mp3_entry = gtk_entry_new_with_max_length(MAX_COMMAND_LENGTH - 2);
            gtk_entry_set_text(GTK_ENTRY(mp3_entry),
                               config.mp3_player.command.c_str());
            gtk_table_attach(GTK_TABLE(table), mp3_entry, 1, 2, 3, 4,
                             GTK_EXPAND | GTK_FILL, 0, 0, 0);

            label = gtk_label_new(_("Players"));
            gtk_notebook_append_page_menu(GTK_NOTEBOOK(notebook),
                                          main_frame, label, label);
            gtk_widget_show_all(main_frame);
            return 0;
        }

        case OP_OK :
        {
            if(strlen(gtk_entry_get_text(
                          GTK_ENTRY(cd_play_entry))) == 0
                    || strlen(gtk_entry_get_text(
                                  GTK_ENTRY(cd_stop_entry))) == 0
                    || strlen(gtk_entry_get_text(
                                  GTK_ENTRY(wav_entry))) == 0
                    || strlen(gtk_entry_get_text(
                                  GTK_ENTRY(mp3_entry))) == 0)
            {
                err_handler(EMPTY_ENTRY_ERR, _("You need to fill every entry in players page"));
                return -1;
            }

            config.cd_player.play_command = gtk_entry_get_text(GTK_ENTRY(cd_play_entry));
            config.cd_player.stop_command = gtk_entry_get_text(GTK_ENTRY(cd_stop_entry));
            config.wav_player.command = gtk_entry_get_text(GTK_ENTRY(wav_entry));
            config.mp3_player.command = gtk_entry_get_text(GTK_ENTRY(mp3_entry));
            return 0;
        }

        default: break;
    }

    // Just to avoid warning
    return -1;
}

int cw_files_handler(int ops, _main_data *main_data, GtkWidget *notebook)
{
    static GtkWidget *main_frame, *table, *vbox, *hbox;
    static GtkWidget *format_string_entry, *dir_format_string_entry;
    static GtkWidget *convert_spaces_ckbx, *make_directories_ckbx, *create_id3_ckbx,
            *create_playlist_ckbx, *create_splitt_ckbx;

    switch (ops)
    {
        case WIDGET_CREATE :
        {
            GtkWidget *label;

            main_frame = gtk_frame_new(_("File Configuration"));
            gtk_container_set_border_width(GTK_CONTAINER(main_frame), MAIN_FRAME_BORDER_WIDTH);

            vbox = gtk_vbox_new(false, 0);
            gtk_container_set_border_width(GTK_CONTAINER(vbox), VBOX_BORDER_WIDTH);

            /* convert spaces */
            convert_spaces_ckbx = gtk_check_button_new_with_label(_("Convert spaces to underscores"));
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(convert_spaces_ckbx),
                                         config.cddb_config.convert_spaces);
            gtk_box_pack_start(GTK_BOX(vbox), convert_spaces_ckbx, false, false, 0);

            /* make directories */
            make_directories_ckbx = gtk_check_button_new_with_label(_("Create album subdirectory for each CD"));
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(make_directories_ckbx),
                                         config.cddb_config.make_directories);
            gtk_box_pack_start(GTK_BOX(vbox), make_directories_ckbx, false, false, 0);

            /* Create ID3 tag*/
            create_id3_ckbx = gtk_check_button_new_with_label(_("Create ID3 tag"));
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(create_id3_ckbx),
                                         config.cddb_config.create_id3);
            gtk_box_pack_start(GTK_BOX(vbox), create_id3_ckbx, false, false, 0);

            /* Split title on Slash tag*/
            create_splitt_ckbx = gtk_check_button_new_with_label(_("Split Title"));
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON( create_splitt_ckbx),
                                         config.cddb_config.split_title);
            gtk_box_pack_start(GTK_BOX(vbox), create_splitt_ckbx, false, false, 0);

            /* Create m3u playlist file */
            create_playlist_ckbx = gtk_check_button_new_with_label(_("Create m3u playlist"));
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(create_playlist_ckbx),
                                         config.cddb_config.create_playlist);
            gtk_box_pack_start(GTK_BOX(vbox), create_playlist_ckbx, false, false, 0);

            /* format string entry */
            hbox = gtk_hbox_new(false, 3);
            gtk_box_pack_start(GTK_BOX(vbox), hbox, false, false, 0);

            label = gtk_label_new(_("Filename format string: "));
            gtk_box_pack_start(GTK_BOX(hbox), label, false, false, 0);

            format_string_entry = gtk_entry_new_with_max_length(50);
            gtk_entry_set_text(GTK_ENTRY(format_string_entry),
                               config.cddb_config.format_string.c_str());
            gtk_box_pack_start(GTK_BOX(hbox), format_string_entry, false, false, 0);

            /* directory format string entry */
            hbox = gtk_hbox_new(false, 0);
            gtk_box_pack_start(GTK_BOX(vbox), hbox, false, false, 8);

            label = gtk_label_new(_("Directory format string: "));
            gtk_box_pack_start(GTK_BOX(hbox), label, false, false, 0);

            dir_format_string_entry = gtk_entry_new_with_max_length(50);
            gtk_entry_set_text(GTK_ENTRY(dir_format_string_entry),
                               config.cddb_config.dir_format_string.c_str());
            gtk_box_pack_start(GTK_BOX(hbox), dir_format_string_entry, false, false, 0);

            /* format string help */
            table = gtk_table_new(3, 2, false);
            gtk_table_set_row_spacings(GTK_TABLE(table), 5);
            gtk_container_set_border_width(GTK_CONTAINER(table), 3);
            gtk_box_pack_start(GTK_BOX(vbox), table, false, false, 0);

            label = gtk_label_new(_("%a = Artist"));
            gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,
                             GTK_EXPAND | GTK_FILL, 0, 0, 0);
            label = gtk_label_new(_("%v = Album"));
            gtk_table_attach(GTK_TABLE(table), label, 1, 2, 0, 1,
                             GTK_EXPAND | GTK_FILL, 0, 0, 0);
            label = gtk_label_new(_("%# = Track no."));
            gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,
                             GTK_EXPAND | GTK_FILL, 0, 0, 0);
            label = gtk_label_new(_("%s = Song title"));
            gtk_table_attach(GTK_TABLE(table), label, 1, 2, 1, 2,
                             GTK_EXPAND | GTK_FILL, 0, 0, 0);
            label = gtk_label_new(_("%y = Year"));
            gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3,
                             GTK_EXPAND | GTK_FILL, 0, 0, 0);

            gtk_container_add(GTK_CONTAINER(main_frame), vbox);

            label = gtk_label_new(_("Files"));
            gtk_notebook_append_page_menu(GTK_NOTEBOOK(notebook), main_frame, label, label);
            gtk_widget_show_all(main_frame);
            return 0;
        }

        case OP_OK :
        {
            config.cddb_config.convert_spaces = GTK_TOGGLE_BUTTON(convert_spaces_ckbx)->active ? 1 : 0;
            config.cddb_config.make_directories = GTK_TOGGLE_BUTTON(make_directories_ckbx)->active ? 1 : 0;
            config.cddb_config.create_id3 =  GTK_TOGGLE_BUTTON(create_id3_ckbx)->active ? 1 : 0;
            config.cddb_config.split_title =  GTK_TOGGLE_BUTTON(create_splitt_ckbx)->active ? 1 : 0;
            config.cddb_config.create_playlist = GTK_TOGGLE_BUTTON(create_playlist_ckbx)->active ? 1 : 0;
            config.cddb_config.format_string = gtk_entry_get_text(GTK_ENTRY(format_string_entry));
            config.cddb_config.dir_format_string = gtk_entry_get_text(GTK_ENTRY(dir_format_string_entry));

            return 0;
        }

        default: break;
    }

    // Just to avoid warning
    return -1;
}

int cw_cddb_handler(int ops, _main_data *main_data, GtkWidget *notebook)
{
    static GtkWidget *main_frame, *table, *vbox, *hbox;
    static GtkWidget *cddb_server_entry, *cddb_port_entry, *cddb_path_entry;
    static GtkWidget *cddb_proxy_server_entry, *cddb_proxy_port_entry;
    static GtkWidget *use_http_ckbx, *auto_lookup_ckbx;
    std::string cddb_port_num, cddb_proxy_port_num;

    switch (ops)
    {
        case WIDGET_CREATE :
        {
            GtkWidget *label;
	    
            main_frame = gtk_frame_new(_("CDDB Configuration"));
            gtk_container_set_border_width(GTK_CONTAINER(main_frame), MAIN_FRAME_BORDER_WIDTH);

            vbox = gtk_vbox_new(false, 0);
            gtk_container_set_border_width(GTK_CONTAINER(vbox), VBOX_BORDER_WIDTH);

            /* cddb server config table */
            table = gtk_table_new(6, 2, false);
            gtk_table_set_row_spacings(GTK_TABLE(table), 5);
            gtk_container_set_border_width(GTK_CONTAINER(table), 3);
            gtk_box_pack_start(GTK_BOX(vbox), table, false, false, 0);

            /* URL box */
            label = gtk_label_new(_("URL: "));
            cddb_server_entry = gtk_entry_new_with_max_length(MAX_COMMAND_LENGTH - 2);
            gtk_entry_set_text(GTK_ENTRY(cddb_server_entry),
                               config.cddb_config.server.c_str());
            gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,
                             0, 0, 0, 0);
            gtk_table_attach(GTK_TABLE(table), cddb_server_entry, 1, 2, 0, 1,
                             GTK_EXPAND | GTK_FILL, 0, 0, 0);

            /* port box */
            label = gtk_label_new(_("Port: "));
            cddb_port_entry = gtk_entry_new_with_max_length(5);
            cddb_port_num = int2str(config.cddb_config.port);
            gtk_entry_set_text(GTK_ENTRY(cddb_port_entry), cddb_port_num.c_str());
            gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,
                             0, 0, 0, 0);
            gtk_table_attach(GTK_TABLE(table), cddb_port_entry, 1, 2, 1, 2,
                             GTK_EXPAND | GTK_FILL, 0, 0, 0);

            /* use http */
            use_http_ckbx = gtk_check_button_new_with_label(_("Use HTTP"));
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(use_http_ckbx),
                                         config.cddb_config.use_http);
            gtk_table_attach(GTK_TABLE(table), use_http_ckbx, 0, 2, 2, 3,
                             0, 0, 0, 0);

            /* Proxy URL box */
            label = gtk_label_new(_("Proxy Server: "));
            cddb_proxy_server_entry = gtk_entry_new_with_max_length(MAX_COMMAND_LENGTH - 2);
            gtk_entry_set_text(GTK_ENTRY(cddb_proxy_server_entry),
                               config.cddb_config.proxy_server.c_str());
            gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4,
                             0, 0, 0, 0);
            gtk_table_attach(GTK_TABLE(table), cddb_proxy_server_entry, 1, 2, 3, 4,
                             GTK_EXPAND | GTK_FILL, 0, 0, 0);

            /* proxy port box */
            label = gtk_label_new(_("Proxy Port: "));
            cddb_proxy_port_entry = gtk_entry_new_with_max_length(5);
            cddb_proxy_port_num = int2str(config.cddb_config.proxy_port);
            gtk_entry_set_text(GTK_ENTRY(cddb_proxy_port_entry), cddb_proxy_port_num.c_str());

            gtk_table_attach(GTK_TABLE(table), label, 0, 1, 4, 5,
                             0, 0, 0, 0);
            gtk_table_attach(GTK_TABLE(table), cddb_proxy_port_entry, 1, 2, 4, 5,
                             GTK_EXPAND | GTK_FILL, 0, 0, 0);

            /* end cddb server config table */

            /* CDDB Path*/
            hbox = gtk_hbox_new(false, 3);
            label = gtk_label_new(_("CDDB Cache path: "));
            gtk_widget_show(label);
            gtk_box_pack_start(GTK_BOX(hbox), label, false, false, 0);

            cddb_path_entry = gtk_entry_new_with_max_length(MAX_FILE_NAME_LENGTH - 3);
            gtk_entry_set_text(GTK_ENTRY(cddb_path_entry),
                               config.cddb_path.c_str());
            gtk_widget_show(cddb_path_entry);

            gtk_box_pack_start(GTK_BOX(hbox), cddb_path_entry, false, false, 0);
            gtk_box_pack_start(GTK_BOX(vbox), hbox, false, false, 0);

            /* CDDB Autolookup */
            hbox = gtk_hbox_new(false, 3);
            auto_lookup_ckbx = gtk_check_button_new_with_label(_("Automatic lookup on startup"));
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(auto_lookup_ckbx),
                                         config.cddb_config.auto_lookup);
            gtk_widget_show(auto_lookup_ckbx);
            gtk_box_pack_start(GTK_BOX(hbox), auto_lookup_ckbx, false, false, 0);
            gtk_box_pack_start(GTK_BOX(vbox), hbox, false, false, 0);
            gtk_container_add(GTK_CONTAINER(main_frame), vbox);

            label = gtk_label_new("CDDB");
            gtk_notebook_append_page_menu(GTK_NOTEBOOK(notebook),
                                          main_frame, label, label);
            gtk_widget_show_all(main_frame);
            return 0;
        }

        case OP_OK :
        {
            if(strlen(gtk_entry_get_text(GTK_ENTRY(cddb_server_entry))) == 0
                    || strlen(gtk_entry_get_text(GTK_ENTRY(cddb_port_entry))) == 0)
            {
                err_handler(EMPTY_ENTRY_ERR, _("You need to specify a server and a port"));
                return -1;
            }

            config.cddb_config.server = gtk_entry_get_text(GTK_ENTRY(cddb_server_entry));
            config.cddb_config.port = std::stoi(gtk_entry_get_text(GTK_ENTRY(cddb_port_entry)));
            config.cddb_config.proxy_server = gtk_entry_get_text(GTK_ENTRY(cddb_proxy_server_entry));
            config.cddb_config.proxy_port = std::stoi(gtk_entry_get_text(GTK_ENTRY(cddb_proxy_port_entry)));
            config.cddb_path = gtk_entry_get_text(GTK_ENTRY(cddb_path_entry));

            config.cddb_config.use_http = GTK_TOGGLE_BUTTON(use_http_ckbx) ->active ? 1 : 0;
            config.cddb_config.auto_lookup = GTK_TOGGLE_BUTTON(auto_lookup_ckbx) ->active ? 1 : 0;

            return 0;
        }

        default: break;
    }

    /* Just to avoid warning */
    return -1;
}

void cw_ok_button_clicked(GtkWidget *widget, gpointer callback_data)
{
    config_window_handler(OP_OK, NULL);
}

void cw_cancel_button_clicked(GtkWidget *widget, gpointer callback_data)
{
    config_window_handler(OP_CANCEL, NULL);
}

void config_window_handler(enum InterfaceCommon ops, _main_data *main_data)
{
    static GtkWidget *window = nullptr;
    static GtkWidget *notebook = nullptr;
    static _main_data *saved_main_data;

    switch (ops)
    {
        case WIDGET_CREATE :
        {
            GtkWidget *vbox, *bbox, *separator, *button;

            saved_main_data = main_data;
            window = gtk_window_new(GTK_WINDOW_TOPLEVEL);

            g_signal_connect(G_OBJECT(window), "destroy",
                             G_CALLBACK(cw_cancel_button_clicked),
                             NULL);

            gtk_window_set_title(GTK_WINDOW(window), _("Configuration"));
            gtk_container_set_border_width(GTK_CONTAINER(window), 0);

            vbox = gtk_vbox_new(false, 0);
            gtk_container_add(GTK_CONTAINER(window), vbox);

            notebook = gtk_notebook_new();
            gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
            gtk_box_pack_start(GTK_BOX(vbox), notebook, true, true, 0);
            gtk_container_set_border_width(GTK_CONTAINER(notebook), 10);

            gtk_widget_realize(window);

            cw_general_handler(WIDGET_CREATE, main_data, notebook);
            cw_wav_handler(WIDGET_CREATE, main_data, notebook);
            cw_encoders_handler(WIDGET_CREATE, main_data, notebook);
            cw_players_handler(WIDGET_CREATE, main_data, notebook);
            cw_cddb_handler(WIDGET_CREATE, main_data, notebook);
            cw_files_handler(WIDGET_CREATE, main_data, notebook);

            separator = gtk_hseparator_new();
            gtk_box_pack_start(GTK_BOX(vbox), separator, false, true, 10);

            bbox = gtk_hbox_new(true, 5);
            gtk_container_set_border_width(GTK_CONTAINER(bbox), 10);
            gtk_box_pack_start(GTK_BOX(vbox), bbox, false, true, 0);

            button = gtk_button_new_from_stock(GTK_STOCK_OK);
            g_signal_connect(G_OBJECT(button), "clicked",
                             G_CALLBACK(cw_ok_button_clicked),
                             NULL);

            gtk_box_pack_end(GTK_BOX(bbox), button, true, true, 0);
            GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);

            button = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
            g_signal_connect(G_OBJECT(button), "clicked",
                             G_CALLBACK(cw_cancel_button_clicked),
                             NULL);

            gtk_box_pack_end(GTK_BOX(bbox), button, true, true, 0);
            GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
            gtk_widget_grab_default(button);

            gtk_widget_show_all(window);
            gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), config.last_config_page);
            break;
        }

        case OP_OK :
        {
            main_data = saved_main_data;

            if (cw_general_handler(OP_OK, main_data, NULL) < 0
                    || cw_wav_handler(OP_OK, main_data, NULL) < 0
                    || cw_encoders_handler(OP_OK, main_data, NULL) < 0
                    || cw_players_handler(OP_OK, main_data, NULL) < 0
                    || cw_cddb_handler(OP_OK, main_data, NULL) < 0
                    || cw_files_handler(OP_OK, main_data, NULL) < 0)
            {
                return;
            }

            config.last_config_page = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
            gtk_widget_destroy(window);
            main_window_handler(MW_MODE_SELECT, 0, main_data);
            config_data.write_config();
            break;
        }

        case OP_CANCEL :
        {
            gtk_widget_destroy(window);
            main_window_handler(MW_MODE_SELECT, 0, main_data);
            break;
        }

        default: break;
    }
}
