/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.searchablesnapshots.store.input;

import java.io.Closeable;
import java.io.EOFException;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.Objects;
import java.util.concurrent.atomic.LongAdder;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.elasticsearch.common.CheckedSupplier;
import org.elasticsearch.core.CheckedRunnable;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.internal.io.IOUtils;
import org.elasticsearch.index.snapshots.blobstore.BlobStoreIndexShardSnapshot;
import org.elasticsearch.xpack.searchablesnapshots.SearchableSnapshotsUtils;
import org.elasticsearch.xpack.searchablesnapshots.cache.common.ByteRange;
import org.elasticsearch.xpack.searchablesnapshots.store.IndexInputStats;
import org.elasticsearch.xpack.searchablesnapshots.store.SearchableSnapshotDirectory;
import org.elasticsearch.xpack.searchablesnapshots.store.input.BaseSearchableSnapshotIndexInput;

public class DirectBlobContainerIndexInput
extends BaseSearchableSnapshotIndexInput {
    private static final Logger logger = LogManager.getLogger(DirectBlobContainerIndexInput.class);
    private long position;
    @Nullable
    private StreamForSequentialReads streamForSequentialReads;
    private long sequentialReadSize;
    private static final long NO_SEQUENTIAL_READ_OPTIMIZATION = 0L;
    private static final int COPY_BUFFER_SIZE = 8192;

    public DirectBlobContainerIndexInput(String name, SearchableSnapshotDirectory directory, BlobStoreIndexShardSnapshot.FileInfo fileInfo, IOContext context, IndexInputStats stats, long sequentialReadSize, int bufferSize) {
        this(name, directory, fileInfo, context, stats, 0L, 0L, fileInfo.length(), sequentialReadSize, bufferSize);
        stats.incrementOpenCount();
    }

    private DirectBlobContainerIndexInput(String name, SearchableSnapshotDirectory directory, BlobStoreIndexShardSnapshot.FileInfo fileInfo, IOContext context, IndexInputStats stats, long position, long offset, long length, long sequentialReadSize, int bufferSize) {
        super(logger, name, directory, fileInfo, context, stats, offset, length, ByteRange.EMPTY, ByteRange.EMPTY);
        this.position = position;
        assert (sequentialReadSize >= 0L);
        this.sequentialReadSize = sequentialReadSize;
        this.setBufferSize(bufferSize);
    }

    @Override
    protected void doReadInternal(ByteBuffer b) throws IOException {
        this.ensureOpen();
        if (this.fileInfo.numberOfParts() == 1) {
            this.readInternalBytes(0, this.position, b, b.remaining());
        } else {
            while (b.hasRemaining()) {
                int currentPart = Math.toIntExact(this.position / this.fileInfo.partSize().getBytes());
                long remainingBytesInPart = currentPart < this.fileInfo.numberOfParts() - 1 ? (long)(currentPart + 1) * this.fileInfo.partSize().getBytes() - this.position : (long)SearchableSnapshotsUtils.toIntBytes(this.fileInfo.length() - this.position);
                int read = SearchableSnapshotsUtils.toIntBytes(Math.min((long)b.remaining(), remainingBytesInPart));
                this.readInternalBytes(currentPart, this.position % this.fileInfo.partSize().getBytes(), b, read);
            }
        }
    }

    private void readInternalBytes(int part, long pos, ByteBuffer b, int length) throws IOException {
        int optimizedReadSize = this.readOptimized(part, pos, b, length);
        assert (optimizedReadSize <= length);
        this.position += (long)optimizedReadSize;
        if (optimizedReadSize < length) {
            long startTimeNanos = this.stats.currentTimeNanos();
            try (InputStream inputStream = this.openBlobStream(part, pos + (long)optimizedReadSize, length - optimizedReadSize);){
                int directReadSize = DirectBlobContainerIndexInput.readFully(inputStream, b, length - optimizedReadSize, (CheckedRunnable<IOException>)((CheckedRunnable)() -> {
                    throw new EOFException("Read past EOF at [" + this.position + "] with length [" + this.fileInfo.partBytes(part) + "]");
                }));
                assert (optimizedReadSize + directReadSize == length) : optimizedReadSize + " and " + directReadSize + " vs " + length;
                this.position += (long)directReadSize;
                long endTimeNanos = this.stats.currentTimeNanos();
                this.stats.addDirectBytesRead(directReadSize, endTimeNanos - startTimeNanos);
            }
        }
    }

    private int readOptimized(int part, long pos, ByteBuffer b, int length) throws IOException {
        if (this.sequentialReadSize == 0L) {
            return 0;
        }
        int read = 0;
        if (this.streamForSequentialReads == null) {
            read = this.readFromNewSequentialStream(part, pos, b, length);
        } else if (this.streamForSequentialReads.canContinueSequentialRead(part, pos)) {
            read = this.streamForSequentialReads.read(b, length);
            if (this.streamForSequentialReads.isFullyRead()) {
                this.streamForSequentialReads.close();
                this.streamForSequentialReads = null;
            } else assert (read == length) : length + " remaining";
            if (read < length) {
                read += this.readFromNewSequentialStream(part, pos + (long)read, b, length - read);
            }
        } else {
            assert (!this.streamForSequentialReads.isFullyRead());
            this.sequentialReadSize = 0L;
            this.closeStreamForSequentialReads();
        }
        return read;
    }

    private void closeStreamForSequentialReads() throws IOException {
        try {
            IOUtils.close((Closeable)this.streamForSequentialReads);
        }
        finally {
            this.streamForSequentialReads = null;
        }
    }

    private int readFromNewSequentialStream(int part, long pos, ByteBuffer b, int length) throws IOException {
        assert (this.streamForSequentialReads == null) : "should only be called when a new stream is needed";
        assert (this.sequentialReadSize > 0L) : "should only be called if optimizing sequential reads";
        long streamLength = Math.min(this.sequentialReadSize, this.fileInfo.partBytes(part) - pos);
        if (streamLength <= (long)length) {
            return 0;
        }
        InputStream inputStream = this.openBlobStream(part, pos, streamLength);
        this.streamForSequentialReads = new StreamForSequentialReads(new FilterInputStream(inputStream){
            private final LongAdder bytesRead;
            private final LongAdder timeNanos;
            {
                this.bytesRead = new LongAdder();
                this.timeNanos = new LongAdder();
            }

            private int onOptimizedRead(CheckedSupplier<Integer, IOException> read) throws IOException {
                long startTimeNanos = DirectBlobContainerIndexInput.this.stats.currentTimeNanos();
                int result = (Integer)read.get();
                long endTimeNanos = DirectBlobContainerIndexInput.this.stats.currentTimeNanos();
                if (result != -1) {
                    this.bytesRead.add(result);
                    this.timeNanos.add(endTimeNanos - startTimeNanos);
                }
                return result;
            }

            @Override
            public int read() throws IOException {
                return this.onOptimizedRead((CheckedSupplier<Integer, IOException>)((CheckedSupplier)() -> super.read()));
            }

            @Override
            public int read(byte[] b, int off, int len) throws IOException {
                return this.onOptimizedRead((CheckedSupplier<Integer, IOException>)((CheckedSupplier)() -> super.read(b, off, len)));
            }

            @Override
            public void close() throws IOException {
                super.close();
                DirectBlobContainerIndexInput.this.stats.addOptimizedBytesRead(Math.toIntExact(this.bytesRead.sumThenReset()), this.timeNanos.sumThenReset());
            }
        }, part, pos, streamLength);
        int read = this.streamForSequentialReads.read(b, length);
        assert (read == length) : read + " vs " + length;
        assert (!this.streamForSequentialReads.isFullyRead());
        return read;
    }

    protected void seekInternal(long pos) throws IOException {
        if (pos > this.length()) {
            throw new EOFException("Reading past end of file [position=" + pos + ", length=" + this.length() + "] for " + this.toString());
        }
        if (pos < 0L) {
            throw new IOException("Seeking to negative position [" + pos + "] for " + this.toString());
        }
        if (this.position != this.offset + pos) {
            this.position = this.offset + pos;
            this.closeStreamForSequentialReads();
        }
    }

    @Override
    public DirectBlobContainerIndexInput clone() {
        DirectBlobContainerIndexInput clone = (DirectBlobContainerIndexInput)super.clone();
        clone.sequentialReadSize = 0L;
        clone.isClone = true;
        return clone;
    }

    public IndexInput slice(String sliceName, long offset, long length) throws IOException {
        if (offset >= 0L && length >= 0L && offset + length <= this.length()) {
            DirectBlobContainerIndexInput slice = new DirectBlobContainerIndexInput(sliceName, this.directory, this.fileInfo, this.context, this.stats, this.position, this.offset + offset, length, 0L, this.getBufferSize());
            slice.isClone = true;
            slice.seek(0L);
            return slice;
        }
        throw new IllegalArgumentException("slice() " + sliceName + " out of bounds: offset=" + offset + ",length=" + length + ",fileLength=" + this.length() + ": " + (Object)((Object)this));
    }

    @Override
    public void doClose() throws IOException {
        this.closeStreamForSequentialReads();
    }

    @Override
    public String toString() {
        return super.toString() + "[read seq=" + (this.streamForSequentialReads != null ? "yes" : "no") + ']';
    }

    private InputStream openBlobStream(int part, long pos, long length) throws IOException {
        assert (this.assertCurrentThreadMayAccessBlobStore());
        this.stats.addBlobStoreBytesRequested(length);
        return this.blobContainer.readBlob(this.fileInfo.partName(part), pos, length);
    }

    private static int readFully(InputStream inputStream, ByteBuffer b, int length, CheckedRunnable<IOException> onEOF) throws IOException {
        int totalRead;
        int read;
        byte[] buffer = new byte[Math.min(length, 8192)];
        for (totalRead = 0; totalRead < length; totalRead += read) {
            int len = Math.min(length - totalRead, 8192);
            read = inputStream.read(buffer, 0, len);
            if (read == -1) {
                onEOF.run();
                break;
            }
            b.put(buffer, 0, read);
        }
        return totalRead > 0 ? totalRead : -1;
    }

    private static class StreamForSequentialReads
    implements Closeable {
        private final InputStream inputStream;
        private final int part;
        private long pos;
        private final long maxPos;

        StreamForSequentialReads(InputStream inputStream, int part, long pos, long streamLength) {
            this.inputStream = Objects.requireNonNull(inputStream);
            this.part = part;
            this.pos = pos;
            this.maxPos = pos + streamLength;
        }

        boolean canContinueSequentialRead(int part, long pos) {
            return this.part == part && this.pos == pos;
        }

        int read(ByteBuffer b, int length) throws IOException {
            assert (this.pos < this.maxPos) : "should not try and read from a fully-read stream";
            int read = DirectBlobContainerIndexInput.readFully(this.inputStream, b, length, (CheckedRunnable<IOException>)() -> {});
            assert (read <= length) : read + " vs " + length;
            this.pos += (long)read;
            return read;
        }

        boolean isFullyRead() {
            assert (this.pos <= this.maxPos);
            return this.pos >= this.maxPos;
        }

        @Override
        public void close() throws IOException {
            this.inputStream.close();
        }
    }
}

