/* Fixed Huffman table as per RFC1951.
 *
 *       Lit Value    Bits        Codes
 *       ---------    ----        -----
 *         0 - 143     8          00110000 through  10111111
 *       144 - 255     9         110010000 through 111111111
 *       256 - 279     7           0000000 through   0010111
 *       280 - 287     8          11000000 through  11000111
 *
 * The codes are encoded in reverse, the high bit of the code appears encoded
 * as bit 0. The table is built by mkhuff.sh. The 16 bits are encoded this way :
 *  - bits 0..3  : bits
 *  - bits 4..12 : code
 */
static const uint16_t fixed_huff[288] = {
	0x00c8, 0x08c8, 0x04c8, 0x0cc8, 0x02c8, 0x0ac8, 0x06c8, 0x0ec8, //   0
	0x01c8, 0x09c8, 0x05c8, 0x0dc8, 0x03c8, 0x0bc8, 0x07c8, 0x0fc8, //   8
	0x0028, 0x0828, 0x0428, 0x0c28, 0x0228, 0x0a28, 0x0628, 0x0e28, //  16
	0x0128, 0x0928, 0x0528, 0x0d28, 0x0328, 0x0b28, 0x0728, 0x0f28, //  24
	0x00a8, 0x08a8, 0x04a8, 0x0ca8, 0x02a8, 0x0aa8, 0x06a8, 0x0ea8, //  32
	0x01a8, 0x09a8, 0x05a8, 0x0da8, 0x03a8, 0x0ba8, 0x07a8, 0x0fa8, //  40
	0x0068, 0x0868, 0x0468, 0x0c68, 0x0268, 0x0a68, 0x0668, 0x0e68, //  48
	0x0168, 0x0968, 0x0568, 0x0d68, 0x0368, 0x0b68, 0x0768, 0x0f68, //  56
	0x00e8, 0x08e8, 0x04e8, 0x0ce8, 0x02e8, 0x0ae8, 0x06e8, 0x0ee8, //  64
	0x01e8, 0x09e8, 0x05e8, 0x0de8, 0x03e8, 0x0be8, 0x07e8, 0x0fe8, //  72
	0x0018, 0x0818, 0x0418, 0x0c18, 0x0218, 0x0a18, 0x0618, 0x0e18, //  80
	0x0118, 0x0918, 0x0518, 0x0d18, 0x0318, 0x0b18, 0x0718, 0x0f18, //  88
	0x0098, 0x0898, 0x0498, 0x0c98, 0x0298, 0x0a98, 0x0698, 0x0e98, //  96
	0x0198, 0x0998, 0x0598, 0x0d98, 0x0398, 0x0b98, 0x0798, 0x0f98, // 104
	0x0058, 0x0858, 0x0458, 0x0c58, 0x0258, 0x0a58, 0x0658, 0x0e58, // 112
	0x0158, 0x0958, 0x0558, 0x0d58, 0x0358, 0x0b58, 0x0758, 0x0f58, // 120
	0x00d8, 0x08d8, 0x04d8, 0x0cd8, 0x02d8, 0x0ad8, 0x06d8, 0x0ed8, // 128
	0x01d8, 0x09d8, 0x05d8, 0x0dd8, 0x03d8, 0x0bd8, 0x07d8, 0x0fd8, // 136
	0x0139, 0x1139, 0x0939, 0x1939, 0x0539, 0x1539, 0x0d39, 0x1d39, // 144
	0x0339, 0x1339, 0x0b39, 0x1b39, 0x0739, 0x1739, 0x0f39, 0x1f39, // 152
	0x00b9, 0x10b9, 0x08b9, 0x18b9, 0x04b9, 0x14b9, 0x0cb9, 0x1cb9, // 160
	0x02b9, 0x12b9, 0x0ab9, 0x1ab9, 0x06b9, 0x16b9, 0x0eb9, 0x1eb9, // 168
	0x01b9, 0x11b9, 0x09b9, 0x19b9, 0x05b9, 0x15b9, 0x0db9, 0x1db9, // 176
	0x03b9, 0x13b9, 0x0bb9, 0x1bb9, 0x07b9, 0x17b9, 0x0fb9, 0x1fb9, // 184
	0x0079, 0x1079, 0x0879, 0x1879, 0x0479, 0x1479, 0x0c79, 0x1c79, // 192
	0x0279, 0x1279, 0x0a79, 0x1a79, 0x0679, 0x1679, 0x0e79, 0x1e79, // 200
	0x0179, 0x1179, 0x0979, 0x1979, 0x0579, 0x1579, 0x0d79, 0x1d79, // 208
	0x0379, 0x1379, 0x0b79, 0x1b79, 0x0779, 0x1779, 0x0f79, 0x1f79, // 216
	0x00f9, 0x10f9, 0x08f9, 0x18f9, 0x04f9, 0x14f9, 0x0cf9, 0x1cf9, // 224
	0x02f9, 0x12f9, 0x0af9, 0x1af9, 0x06f9, 0x16f9, 0x0ef9, 0x1ef9, // 232
	0x01f9, 0x11f9, 0x09f9, 0x19f9, 0x05f9, 0x15f9, 0x0df9, 0x1df9, // 240
	0x03f9, 0x13f9, 0x0bf9, 0x1bf9, 0x07f9, 0x17f9, 0x0ff9, 0x1ff9, // 248
	0x0007, 0x0407, 0x0207, 0x0607, 0x0107, 0x0507, 0x0307, 0x0707, // 256
	0x0087, 0x0487, 0x0287, 0x0687, 0x0187, 0x0587, 0x0387, 0x0787, // 264
	0x0047, 0x0447, 0x0247, 0x0647, 0x0147, 0x0547, 0x0347, 0x0747, // 272
	0x0038, 0x0838, 0x0438, 0x0c38, 0x0238, 0x0a38, 0x0638, 0x0e38  // 280
};

/* length from 3 to 258 converted to bit strings for use with fixed huffman
 * coding. It was built by tools/dump_len.c. The format is the following :
 *   - bits 0..15  = code
 *   - bits 16..19 = #bits
 */
static const uint32_t len_fh[259] = {
	0x000000,  0x000000,  0x000000,  0x070040,   /* 0-3 */
	0x070020,  0x070060,  0x070010,  0x070050,   /* 4-7 */
	0x070030,  0x070070,  0x070008,  0x080048,   /* 8-11 */
	0x0800c8,  0x080028,  0x0800a8,  0x080068,   /* 12-15 */
	0x0800e8,  0x080018,  0x080098,  0x090058,   /* 16-19 */
	0x0900d8,  0x090158,  0x0901d8,  0x090038,   /* 20-23 */
	0x0900b8,  0x090138,  0x0901b8,  0x090078,   /* 24-27 */
	0x0900f8,  0x090178,  0x0901f8,  0x090004,   /* 28-31 */
	0x090084,  0x090104,  0x090184,  0x0a0044,   /* 32-35 */
	0x0a00c4,  0x0a0144,  0x0a01c4,  0x0a0244,   /* 36-39 */
	0x0a02c4,  0x0a0344,  0x0a03c4,  0x0a0024,   /* 40-43 */
	0x0a00a4,  0x0a0124,  0x0a01a4,  0x0a0224,   /* 44-47 */
	0x0a02a4,  0x0a0324,  0x0a03a4,  0x0a0064,   /* 48-51 */
	0x0a00e4,  0x0a0164,  0x0a01e4,  0x0a0264,   /* 52-55 */
	0x0a02e4,  0x0a0364,  0x0a03e4,  0x0a0014,   /* 56-59 */
	0x0a0094,  0x0a0114,  0x0a0194,  0x0a0214,   /* 60-63 */
	0x0a0294,  0x0a0314,  0x0a0394,  0x0b0054,   /* 64-67 */
	0x0b00d4,  0x0b0154,  0x0b01d4,  0x0b0254,   /* 68-71 */
	0x0b02d4,  0x0b0354,  0x0b03d4,  0x0b0454,   /* 72-75 */
	0x0b04d4,  0x0b0554,  0x0b05d4,  0x0b0654,   /* 76-79 */
	0x0b06d4,  0x0b0754,  0x0b07d4,  0x0b0034,   /* 80-83 */
	0x0b00b4,  0x0b0134,  0x0b01b4,  0x0b0234,   /* 84-87 */
	0x0b02b4,  0x0b0334,  0x0b03b4,  0x0b0434,   /* 88-91 */
	0x0b04b4,  0x0b0534,  0x0b05b4,  0x0b0634,   /* 92-95 */
	0x0b06b4,  0x0b0734,  0x0b07b4,  0x0b0074,   /* 96-99 */
	0x0b00f4,  0x0b0174,  0x0b01f4,  0x0b0274,   /* 100-103 */
	0x0b02f4,  0x0b0374,  0x0b03f4,  0x0b0474,   /* 104-107 */
	0x0b04f4,  0x0b0574,  0x0b05f4,  0x0b0674,   /* 108-111 */
	0x0b06f4,  0x0b0774,  0x0b07f4,  0x0c0003,   /* 112-115 */
	0x0c0103,  0x0c0203,  0x0c0303,  0x0c0403,   /* 116-119 */
	0x0c0503,  0x0c0603,  0x0c0703,  0x0c0803,   /* 120-123 */
	0x0c0903,  0x0c0a03,  0x0c0b03,  0x0c0c03,   /* 124-127 */
	0x0c0d03,  0x0c0e03,  0x0c0f03,  0x0d0083,   /* 128-131 */
	0x0d0183,  0x0d0283,  0x0d0383,  0x0d0483,   /* 132-135 */
	0x0d0583,  0x0d0683,  0x0d0783,  0x0d0883,   /* 136-139 */
	0x0d0983,  0x0d0a83,  0x0d0b83,  0x0d0c83,   /* 140-143 */
	0x0d0d83,  0x0d0e83,  0x0d0f83,  0x0d1083,   /* 144-147 */
	0x0d1183,  0x0d1283,  0x0d1383,  0x0d1483,   /* 148-151 */
	0x0d1583,  0x0d1683,  0x0d1783,  0x0d1883,   /* 152-155 */
	0x0d1983,  0x0d1a83,  0x0d1b83,  0x0d1c83,   /* 156-159 */
	0x0d1d83,  0x0d1e83,  0x0d1f83,  0x0d0043,   /* 160-163 */
	0x0d0143,  0x0d0243,  0x0d0343,  0x0d0443,   /* 164-167 */
	0x0d0543,  0x0d0643,  0x0d0743,  0x0d0843,   /* 168-171 */
	0x0d0943,  0x0d0a43,  0x0d0b43,  0x0d0c43,   /* 172-175 */
	0x0d0d43,  0x0d0e43,  0x0d0f43,  0x0d1043,   /* 176-179 */
	0x0d1143,  0x0d1243,  0x0d1343,  0x0d1443,   /* 180-183 */
	0x0d1543,  0x0d1643,  0x0d1743,  0x0d1843,   /* 184-187 */
	0x0d1943,  0x0d1a43,  0x0d1b43,  0x0d1c43,   /* 188-191 */
	0x0d1d43,  0x0d1e43,  0x0d1f43,  0x0d00c3,   /* 192-195 */
	0x0d01c3,  0x0d02c3,  0x0d03c3,  0x0d04c3,   /* 196-199 */
	0x0d05c3,  0x0d06c3,  0x0d07c3,  0x0d08c3,   /* 200-203 */
	0x0d09c3,  0x0d0ac3,  0x0d0bc3,  0x0d0cc3,   /* 204-207 */
	0x0d0dc3,  0x0d0ec3,  0x0d0fc3,  0x0d10c3,   /* 208-211 */
	0x0d11c3,  0x0d12c3,  0x0d13c3,  0x0d14c3,   /* 212-215 */
	0x0d15c3,  0x0d16c3,  0x0d17c3,  0x0d18c3,   /* 216-219 */
	0x0d19c3,  0x0d1ac3,  0x0d1bc3,  0x0d1cc3,   /* 220-223 */
	0x0d1dc3,  0x0d1ec3,  0x0d1fc3,  0x0d0023,   /* 224-227 */
	0x0d0123,  0x0d0223,  0x0d0323,  0x0d0423,   /* 228-231 */
	0x0d0523,  0x0d0623,  0x0d0723,  0x0d0823,   /* 232-235 */
	0x0d0923,  0x0d0a23,  0x0d0b23,  0x0d0c23,   /* 236-239 */
	0x0d0d23,  0x0d0e23,  0x0d0f23,  0x0d1023,   /* 240-243 */
	0x0d1123,  0x0d1223,  0x0d1323,  0x0d1423,   /* 244-247 */
	0x0d1523,  0x0d1623,  0x0d1723,  0x0d1823,   /* 248-251 */
	0x0d1923,  0x0d1a23,  0x0d1b23,  0x0d1c23,   /* 252-255 */
	0x0d1d23,  0x0d1e23,  0x0800a3               /* 256-258 */
};

#ifndef PRECOMPUTE_TABLES

static uint32_t crc32_fast[4][256];
static uint32_t fh_dist_table[32768];

#else  /* ifndef PRECOMPUTE_TABLES */

/* Table of *inverted* CRC32 for each 8-bit quantity based on the position of
 * the byte being read relative to the last byte. Eg: [0] means we're on the
 * last byte, [1] on the previous one etc. These values have 8 inverted bits
 * at each position so that when processing 32-bit little endian quantities,
 * the CRC already appears inverted in each individual byte and doesn't need
 * to be inverted again in the loop. The precomputed version was generated by
 * mkcrc.sh.
 */
static const uint32_t crc32_fast[4][256] = {
	{ // [0]
		0xd202ef8d, 0xa505df1b, 0x3c0c8ea1, 0x4b0bbe37, // 0-3
		0xd56f2b94, 0xa2681b02, 0x3b614ab8, 0x4c667a2e, // 4-7
		0xdcd967bf, 0xabde5729, 0x32d70693, 0x45d03605, // 8-11
		0xdbb4a3a6, 0xacb39330, 0x35bac28a, 0x42bdf21c, // 12-15
		0xcfb5ffe9, 0xb8b2cf7f, 0x21bb9ec5, 0x56bcae53, // 16-19
		0xc8d83bf0, 0xbfdf0b66, 0x26d65adc, 0x51d16a4a, // 20-23
		0xc16e77db, 0xb669474d, 0x2f6016f7, 0x58672661, // 24-27
		0xc603b3c2, 0xb1048354, 0x280dd2ee, 0x5f0ae278, // 28-31
		0xe96ccf45, 0x9e6bffd3, 0x0762ae69, 0x70659eff, // 32-35
		0xee010b5c, 0x99063bca, 0x000f6a70, 0x77085ae6, // 36-39
		0xe7b74777, 0x90b077e1, 0x09b9265b, 0x7ebe16cd, // 40-43
		0xe0da836e, 0x97ddb3f8, 0x0ed4e242, 0x79d3d2d4, // 44-47
		0xf4dbdf21, 0x83dcefb7, 0x1ad5be0d, 0x6dd28e9b, // 48-51
		0xf3b61b38, 0x84b12bae, 0x1db87a14, 0x6abf4a82, // 52-55
		0xfa005713, 0x8d076785, 0x140e363f, 0x630906a9, // 56-59
		0xfd6d930a, 0x8a6aa39c, 0x1363f226, 0x6464c2b0, // 60-63
		0xa4deae1d, 0xd3d99e8b, 0x4ad0cf31, 0x3dd7ffa7, // 64-67
		0xa3b36a04, 0xd4b45a92, 0x4dbd0b28, 0x3aba3bbe, // 68-71
		0xaa05262f, 0xdd0216b9, 0x440b4703, 0x330c7795, // 72-75
		0xad68e236, 0xda6fd2a0, 0x4366831a, 0x3461b38c, // 76-79
		0xb969be79, 0xce6e8eef, 0x5767df55, 0x2060efc3, // 80-83
		0xbe047a60, 0xc9034af6, 0x500a1b4c, 0x270d2bda, // 84-87
		0xb7b2364b, 0xc0b506dd, 0x59bc5767, 0x2ebb67f1, // 88-91
		0xb0dff252, 0xc7d8c2c4, 0x5ed1937e, 0x29d6a3e8, // 92-95
		0x9fb08ed5, 0xe8b7be43, 0x71beeff9, 0x06b9df6f, // 96-99
		0x98dd4acc, 0xefda7a5a, 0x76d32be0, 0x01d41b76, // 100-103
		0x916b06e7, 0xe66c3671, 0x7f6567cb, 0x0862575d, // 104-107
		0x9606c2fe, 0xe101f268, 0x7808a3d2, 0x0f0f9344, // 108-111
		0x82079eb1, 0xf500ae27, 0x6c09ff9d, 0x1b0ecf0b, // 112-115
		0x856a5aa8, 0xf26d6a3e, 0x6b643b84, 0x1c630b12, // 116-119
		0x8cdc1683, 0xfbdb2615, 0x62d277af, 0x15d54739, // 120-123
		0x8bb1d29a, 0xfcb6e20c, 0x65bfb3b6, 0x12b88320, // 124-127
		0x3fba6cad, 0x48bd5c3b, 0xd1b40d81, 0xa6b33d17, // 128-131
		0x38d7a8b4, 0x4fd09822, 0xd6d9c998, 0xa1def90e, // 132-135
		0x3161e49f, 0x4666d409, 0xdf6f85b3, 0xa868b525, // 136-139
		0x360c2086, 0x410b1010, 0xd80241aa, 0xaf05713c, // 140-143
		0x220d7cc9, 0x550a4c5f, 0xcc031de5, 0xbb042d73, // 144-147
		0x2560b8d0, 0x52678846, 0xcb6ed9fc, 0xbc69e96a, // 148-151
		0x2cd6f4fb, 0x5bd1c46d, 0xc2d895d7, 0xb5dfa541, // 152-155
		0x2bbb30e2, 0x5cbc0074, 0xc5b551ce, 0xb2b26158, // 156-159
		0x04d44c65, 0x73d37cf3, 0xeada2d49, 0x9ddd1ddf, // 160-163
		0x03b9887c, 0x74beb8ea, 0xedb7e950, 0x9ab0d9c6, // 164-167
		0x0a0fc457, 0x7d08f4c1, 0xe401a57b, 0x930695ed, // 168-171
		0x0d62004e, 0x7a6530d8, 0xe36c6162, 0x946b51f4, // 172-175
		0x19635c01, 0x6e646c97, 0xf76d3d2d, 0x806a0dbb, // 176-179
		0x1e0e9818, 0x6909a88e, 0xf000f934, 0x8707c9a2, // 180-183
		0x17b8d433, 0x60bfe4a5, 0xf9b6b51f, 0x8eb18589, // 184-187
		0x10d5102a, 0x67d220bc, 0xfedb7106, 0x89dc4190, // 188-191
		0x49662d3d, 0x3e611dab, 0xa7684c11, 0xd06f7c87, // 192-195
		0x4e0be924, 0x390cd9b2, 0xa0058808, 0xd702b89e, // 196-199
		0x47bda50f, 0x30ba9599, 0xa9b3c423, 0xdeb4f4b5, // 200-203
		0x40d06116, 0x37d75180, 0xaede003a, 0xd9d930ac, // 204-207
		0x54d13d59, 0x23d60dcf, 0xbadf5c75, 0xcdd86ce3, // 208-211
		0x53bcf940, 0x24bbc9d6, 0xbdb2986c, 0xcab5a8fa, // 212-215
		0x5a0ab56b, 0x2d0d85fd, 0xb404d447, 0xc303e4d1, // 216-219
		0x5d677172, 0x2a6041e4, 0xb369105e, 0xc46e20c8, // 220-223
		0x72080df5, 0x050f3d63, 0x9c066cd9, 0xeb015c4f, // 224-227
		0x7565c9ec, 0x0262f97a, 0x9b6ba8c0, 0xec6c9856, // 228-231
		0x7cd385c7, 0x0bd4b551, 0x92dde4eb, 0xe5dad47d, // 232-235
		0x7bbe41de, 0x0cb97148, 0x95b020f2, 0xe2b71064, // 236-239
		0x6fbf1d91, 0x18b82d07, 0x81b17cbd, 0xf6b64c2b, // 240-243
		0x68d2d988, 0x1fd5e91e, 0x86dcb8a4, 0xf1db8832, // 244-247
		0x616495a3, 0x1663a535, 0x8f6af48f, 0xf86dc419, // 248-251
		0x660951ba, 0x110e612c, 0x88073096, 0xff000000, // 252-255
	},
	{ // [1]
		0x93dbfd72, 0x8ac0cc33, 0xa1ed9ff0, 0xb8f6aeb1, // 0-3
		0xf7b73876, 0xeeac0937, 0xc5815af4, 0xdc9a6bb5, // 4-7
		0x5b02777a, 0x4219463b, 0x693415f8, 0x702f24b9, // 8-11
		0x3f6eb27e, 0x2675833f, 0x0d58d0fc, 0x1443e1bd, // 12-15
		0xd919ef23, 0xc002de62, 0xeb2f8da1, 0xf234bce0, // 16-19
		0xbd752a27, 0xa46e1b66, 0x8f4348a5, 0x965879e4, // 20-23
		0x11c0652b, 0x08db546a, 0x23f607a9, 0x3aed36e8, // 24-27
		0x75aca02f, 0x6cb7916e, 0x479ac2ad, 0x5e81f3ec, // 28-31
		0x065fd9d0, 0x1f44e891, 0x3469bb52, 0x2d728a13, // 32-35
		0x62331cd4, 0x7b282d95, 0x50057e56, 0x491e4f17, // 36-39
		0xce8653d8, 0xd79d6299, 0xfcb0315a, 0xe5ab001b, // 40-43
		0xaaea96dc, 0xb3f1a79d, 0x98dcf45e, 0x81c7c51f, // 44-47
		0x4c9dcb81, 0x5586fac0, 0x7eaba903, 0x67b09842, // 48-51
		0x28f10e85, 0x31ea3fc4, 0x1ac76c07, 0x03dc5d46, // 52-55
		0x84444189, 0x9d5f70c8, 0xb672230b, 0xaf69124a, // 56-59
		0xe028848d, 0xf933b5cc, 0xd21ee60f, 0xcb05d74e, // 60-63
		0x63a2b277, 0x7ab98336, 0x5194d0f5, 0x488fe1b4, // 64-67
		0x07ce7773, 0x1ed54632, 0x35f815f1, 0x2ce324b0, // 68-71
		0xab7b387f, 0xb260093e, 0x994d5afd, 0x80566bbc, // 72-75
		0xcf17fd7b, 0xd60ccc3a, 0xfd219ff9, 0xe43aaeb8, // 76-79
		0x2960a026, 0x307b9167, 0x1b56c2a4, 0x024df3e5, // 80-83
		0x4d0c6522, 0x54175463, 0x7f3a07a0, 0x662136e1, // 84-87
		0xe1b92a2e, 0xf8a21b6f, 0xd38f48ac, 0xca9479ed, // 88-91
		0x85d5ef2a, 0x9ccede6b, 0xb7e38da8, 0xaef8bce9, // 92-95
		0xf62696d5, 0xef3da794, 0xc410f457, 0xdd0bc516, // 96-99
		0x924a53d1, 0x8b516290, 0xa07c3153, 0xb9670012, // 100-103
		0x3eff1cdd, 0x27e42d9c, 0x0cc97e5f, 0x15d24f1e, // 104-107
		0x5a93d9d9, 0x4388e898, 0x68a5bb5b, 0x71be8a1a, // 108-111
		0xbce48484, 0xa5ffb5c5, 0x8ed2e606, 0x97c9d747, // 112-115
		0xd8884180, 0xc19370c1, 0xeabe2302, 0xf3a51243, // 116-119
		0x743d0e8c, 0x6d263fcd, 0x460b6c0e, 0x5f105d4f, // 120-123
		0x1051cb88, 0x094afac9, 0x2267a90a, 0x3b7c984b, // 124-127
		0xa8586539, 0xb1435478, 0x9a6e07bb, 0x837536fa, // 128-131
		0xcc34a03d, 0xd52f917c, 0xfe02c2bf, 0xe719f3fe, // 132-135
		0x6081ef31, 0x799ade70, 0x52b78db3, 0x4bacbcf2, // 136-139
		0x04ed2a35, 0x1df61b74, 0x36db48b7, 0x2fc079f6, // 140-143
		0xe29a7768, 0xfb814629, 0xd0ac15ea, 0xc9b724ab, // 144-147
		0x86f6b26c, 0x9fed832d, 0xb4c0d0ee, 0xaddbe1af, // 148-151
		0x2a43fd60, 0x3358cc21, 0x18759fe2, 0x016eaea3, // 152-155
		0x4e2f3864, 0x57340925, 0x7c195ae6, 0x65026ba7, // 156-159
		0x3ddc419b, 0x24c770da, 0x0fea2319, 0x16f11258, // 160-163
		0x59b0849f, 0x40abb5de, 0x6b86e61d, 0x729dd75c, // 164-167
		0xf505cb93, 0xec1efad2, 0xc733a911, 0xde289850, // 168-171
		0x91690e97, 0x88723fd6, 0xa35f6c15, 0xba445d54, // 172-175
		0x771e53ca, 0x6e05628b, 0x45283148, 0x5c330009, // 176-179
		0x137296ce, 0x0a69a78f, 0x2144f44c, 0x385fc50d, // 180-183
		0xbfc7d9c2, 0xa6dce883, 0x8df1bb40, 0x94ea8a01, // 184-187
		0xdbab1cc6, 0xc2b02d87, 0xe99d7e44, 0xf0864f05, // 188-191
		0x58212a3c, 0x413a1b7d, 0x6a1748be, 0x730c79ff, // 192-195
		0x3c4def38, 0x2556de79, 0x0e7b8dba, 0x1760bcfb, // 196-199
		0x90f8a034, 0x89e39175, 0xa2cec2b6, 0xbbd5f3f7, // 200-203
		0xf4946530, 0xed8f5471, 0xc6a207b2, 0xdfb936f3, // 204-207
		0x12e3386d, 0x0bf8092c, 0x20d55aef, 0x39ce6bae, // 208-211
		0x768ffd69, 0x6f94cc28, 0x44b99feb, 0x5da2aeaa, // 212-215
		0xda3ab265, 0xc3218324, 0xe80cd0e7, 0xf117e1a6, // 216-219
		0xbe567761, 0xa74d4620, 0x8c6015e3, 0x957b24a2, // 220-223
		0xcda50e9e, 0xd4be3fdf, 0xff936c1c, 0xe6885d5d, // 224-227
		0xa9c9cb9a, 0xb0d2fadb, 0x9bffa918, 0x82e49859, // 228-231
		0x057c8496, 0x1c67b5d7, 0x374ae614, 0x2e51d755, // 232-235
		0x61104192, 0x780b70d3, 0x53262310, 0x4a3d1251, // 236-239
		0x87671ccf, 0x9e7c2d8e, 0xb5517e4d, 0xac4a4f0c, // 240-243
		0xe30bd9cb, 0xfa10e88a, 0xd13dbb49, 0xc8268a08, // 244-247
		0x4fbe96c7, 0x56a5a786, 0x7d88f445, 0x6493c504, // 248-251
		0x2bd253c3, 0x32c96282, 0x19e43141, 0x00ff0000, // 252-255
	},
	{ // [2]
		0xbe98cbed, 0xbf5aa1da, 0xbd1c1f83, 0xbcde75b4, // 0-3
		0xb9916331, 0xb8530906, 0xba15b75f, 0xbbd7dd68, // 4-7
		0xb08b9a55, 0xb149f062, 0xb30f4e3b, 0xb2cd240c, // 8-11
		0xb7823289, 0xb64058be, 0xb406e6e7, 0xb5c48cd0, // 12-15
		0xa2be689d, 0xa37c02aa, 0xa13abcf3, 0xa0f8d6c4, // 16-19
		0xa5b7c041, 0xa475aa76, 0xa633142f, 0xa7f17e18, // 20-23
		0xacad3925, 0xad6f5312, 0xaf29ed4b, 0xaeeb877c, // 24-27
		0xaba491f9, 0xaa66fbce, 0xa8204597, 0xa9e22fa0, // 28-31
		0x86d58d0d, 0x8717e73a, 0x85515963, 0x84933354, // 32-35
		0x81dc25d1, 0x801e4fe6, 0x8258f1bf, 0x839a9b88, // 36-39
		0x88c6dcb5, 0x8904b682, 0x8b4208db, 0x8a8062ec, // 40-43
		0x8fcf7469, 0x8e0d1e5e, 0x8c4ba007, 0x8d89ca30, // 44-47
		0x9af32e7d, 0x9b31444a, 0x9977fa13, 0x98b59024, // 48-51
		0x9dfa86a1, 0x9c38ec96, 0x9e7e52cf, 0x9fbc38f8, // 52-55
		0x94e07fc5, 0x952215f2, 0x9764abab, 0x96a6c19c, // 56-59
		0x93e9d719, 0x922bbd2e, 0x906d0377, 0x91af6940, // 60-63
		0xce02462d, 0xcfc02c1a, 0xcd869243, 0xcc44f874, // 64-67
		0xc90beef1, 0xc8c984c6, 0xca8f3a9f, 0xcb4d50a8, // 68-71
		0xc0111795, 0xc1d37da2, 0xc395c3fb, 0xc257a9cc, // 72-75
		0xc718bf49, 0xc6dad57e, 0xc49c6b27, 0xc55e0110, // 76-79
		0xd224e55d, 0xd3e68f6a, 0xd1a03133, 0xd0625b04, // 80-83
		0xd52d4d81, 0xd4ef27b6, 0xd6a999ef, 0xd76bf3d8, // 84-87
		0xdc37b4e5, 0xddf5ded2, 0xdfb3608b, 0xde710abc, // 88-91
		0xdb3e1c39, 0xdafc760e, 0xd8bac857, 0xd978a260, // 92-95
		0xf64f00cd, 0xf78d6afa, 0xf5cbd4a3, 0xf409be94, // 96-99
		0xf146a811, 0xf084c226, 0xf2c27c7f, 0xf3001648, // 100-103
		0xf85c5175, 0xf99e3b42, 0xfbd8851b, 0xfa1aef2c, // 104-107
		0xff55f9a9, 0xfe97939e, 0xfcd12dc7, 0xfd1347f0, // 108-111
		0xea69a3bd, 0xebabc98a, 0xe9ed77d3, 0xe82f1de4, // 112-115
		0xed600b61, 0xeca26156, 0xeee4df0f, 0xef26b538, // 116-119
		0xe47af205, 0xe5b89832, 0xe7fe266b, 0xe63c4c5c, // 120-123
		0xe3735ad9, 0xe2b130ee, 0xe0f78eb7, 0xe135e480, // 124-127
		0x5fadd06d, 0x5e6fba5a, 0x5c290403, 0x5deb6e34, // 128-131
		0x58a478b1, 0x59661286, 0x5b20acdf, 0x5ae2c6e8, // 132-135
		0x51be81d5, 0x507cebe2, 0x523a55bb, 0x53f83f8c, // 136-139
		0x56b72909, 0x5775433e, 0x5533fd67, 0x54f19750, // 140-143
		0x438b731d, 0x4249192a, 0x400fa773, 0x41cdcd44, // 144-147
		0x4482dbc1, 0x4540b1f6, 0x47060faf, 0x46c46598, // 148-151
		0x4d9822a5, 0x4c5a4892, 0x4e1cf6cb, 0x4fde9cfc, // 152-155
		0x4a918a79, 0x4b53e04e, 0x49155e17, 0x48d73420, // 156-159
		0x67e0968d, 0x6622fcba, 0x646442e3, 0x65a628d4, // 160-163
		0x60e93e51, 0x612b5466, 0x636dea3f, 0x62af8008, // 164-167
		0x69f3c735, 0x6831ad02, 0x6a77135b, 0x6bb5796c, // 168-171
		0x6efa6fe9, 0x6f3805de, 0x6d7ebb87, 0x6cbcd1b0, // 172-175
		0x7bc635fd, 0x7a045fca, 0x7842e193, 0x79808ba4, // 176-179
		0x7ccf9d21, 0x7d0df716, 0x7f4b494f, 0x7e892378, // 180-183
		0x75d56445, 0x74170e72, 0x7651b02b, 0x7793da1c, // 184-187
		0x72dccc99, 0x731ea6ae, 0x715818f7, 0x709a72c0, // 188-191
		0x2f375dad, 0x2ef5379a, 0x2cb389c3, 0x2d71e3f4, // 192-195
		0x283ef571, 0x29fc9f46, 0x2bba211f, 0x2a784b28, // 196-199
		0x21240c15, 0x20e66622, 0x22a0d87b, 0x2362b24c, // 200-203
		0x262da4c9, 0x27efcefe, 0x25a970a7, 0x246b1a90, // 204-207
		0x3311fedd, 0x32d394ea, 0x30952ab3, 0x31574084, // 208-211
		0x34185601, 0x35da3c36, 0x379c826f, 0x365ee858, // 212-215
		0x3d02af65, 0x3cc0c552, 0x3e867b0b, 0x3f44113c, // 216-219
		0x3a0b07b9, 0x3bc96d8e, 0x398fd3d7, 0x384db9e0, // 220-223
		0x177a1b4d, 0x16b8717a, 0x14fecf23, 0x153ca514, // 224-227
		0x1073b391, 0x11b1d9a6, 0x13f767ff, 0x12350dc8, // 228-231
		0x19694af5, 0x18ab20c2, 0x1aed9e9b, 0x1b2ff4ac, // 232-235
		0x1e60e229, 0x1fa2881e, 0x1de43647, 0x1c265c70, // 236-239
		0x0b5cb83d, 0x0a9ed20a, 0x08d86c53, 0x091a0664, // 240-243
		0x0c5510e1, 0x0d977ad6, 0x0fd1c48f, 0x0e13aeb8, // 244-247
		0x054fe985, 0x048d83b2, 0x06cb3deb, 0x070957dc, // 248-251
		0x02464159, 0x03842b6e, 0x01c29537, 0x0000ff00, // 252-255
	},
	{ // [3]
		0xde05060e, 0x66b9616b, 0x740cce85, 0xccb0a9e0, // 0-3
		0x51679159, 0xe9dbf63c, 0xfb6e59d2, 0x43d23eb7, // 4-7
		0x1bb12ee1, 0xa30d4984, 0xb1b8e66a, 0x0904810f, // 8-11
		0x94d3b9b6, 0x2c6fded3, 0x3eda713d, 0x86661658, // 12-15
		0x8e1c5191, 0x36a036f4, 0x2415991a, 0x9ca9fe7f, // 16-19
		0x017ec6c6, 0xb9c2a1a3, 0xab770e4d, 0x13cb6928, // 20-23
		0x4ba8797e, 0xf3141e1b, 0xe1a1b1f5, 0x591dd690, // 24-27
		0xc4caee29, 0x7c76894c, 0x6ec326a2, 0xd67f41c7, // 28-31
		0x7e37a930, 0xc68bce55, 0xd43e61bb, 0x6c8206de, // 32-35
		0xf1553e67, 0x49e95902, 0x5b5cf6ec, 0xe3e09189, // 36-39
		0xbb8381df, 0x033fe6ba, 0x118a4954, 0xa9362e31, // 40-43
		0x34e11688, 0x8c5d71ed, 0x9ee8de03, 0x2654b966, // 44-47
		0x2e2efeaf, 0x969299ca, 0x84273624, 0x3c9b5141, // 48-51
		0xa14c69f8, 0x19f00e9d, 0x0b45a173, 0xb3f9c616, // 52-55
		0xeb9ad640, 0x5326b125, 0x41931ecb, 0xf92f79ae, // 56-59
		0x64f84117, 0xdc442672, 0xcef1899c, 0x764deef9, // 60-63
		0x45115e33, 0xfdad3956, 0xef1896b8, 0x57a4f1dd, // 64-67
		0xca73c964, 0x72cfae01, 0x607a01ef, 0xd8c6668a, // 68-71
		0x80a576dc, 0x381911b9, 0x2aacbe57, 0x9210d932, // 72-75
		0x0fc7e18b, 0xb77b86ee, 0xa5ce2900, 0x1d724e65, // 76-79
		0x150809ac, 0xadb46ec9, 0xbf01c127, 0x07bda642, // 80-83
		0x9a6a9efb, 0x22d6f99e, 0x30635670, 0x88df3115, // 84-87
		0xd0bc2143, 0x68004626, 0x7ab5e9c8, 0xc2098ead, // 88-91
		0x5fdeb614, 0xe762d171, 0xf5d77e9f, 0x4d6b19fa, // 92-95
		0xe523f10d, 0x5d9f9668, 0x4f2a3986, 0xf7965ee3, // 96-99
		0x6a41665a, 0xd2fd013f, 0xc048aed1, 0x78f4c9b4, // 100-103
		0x2097d9e2, 0x982bbe87, 0x8a9e1169, 0x3222760c, // 104-107
		0xaff54eb5, 0x174929d0, 0x05fc863e, 0xbd40e15b, // 108-111
		0xb53aa692, 0x0d86c1f7, 0x1f336e19, 0xa78f097c, // 112-115
		0x3a5831c5, 0x82e456a0, 0x9051f94e, 0x28ed9e2b, // 116-119
		0x708e8e7d, 0xc832e918, 0xda8746f6, 0x623b2193, // 120-123
		0xffec192a, 0x47507e4f, 0x55e5d1a1, 0xed59b6c4, // 124-127
		0x335cb035, 0x8be0d750, 0x995578be, 0x21e91fdb, // 128-131
		0xbc3e2762, 0x04824007, 0x1637efe9, 0xae8b888c, // 132-135
		0xf6e898da, 0x4e54ffbf, 0x5ce15051, 0xe45d3734, // 136-139
		0x798a0f8d, 0xc13668e8, 0xd383c706, 0x6b3fa063, // 140-143
		0x6345e7aa, 0xdbf980cf, 0xc94c2f21, 0x71f04844, // 144-147
		0xec2770fd, 0x549b1798, 0x462eb876, 0xfe92df13, // 148-151
		0xa6f1cf45, 0x1e4da820, 0x0cf807ce, 0xb44460ab, // 152-155
		0x29935812, 0x912f3f77, 0x839a9099, 0x3b26f7fc, // 156-159
		0x936e1f0b, 0x2bd2786e, 0x3967d780, 0x81dbb0e5, // 160-163
		0x1c0c885c, 0xa4b0ef39, 0xb60540d7, 0x0eb927b2, // 164-167
		0x56da37e4, 0xee665081, 0xfcd3ff6f, 0x446f980a, // 168-171
		0xd9b8a0b3, 0x6104c7d6, 0x73b16838, 0xcb0d0f5d, // 172-175
		0xc3774894, 0x7bcb2ff1, 0x697e801f, 0xd1c2e77a, // 176-179
		0x4c15dfc3, 0xf4a9b8a6, 0xe61c1748, 0x5ea0702d, // 180-183
		0x06c3607b, 0xbe7f071e, 0xaccaa8f0, 0x1476cf95, // 184-187
		0x89a1f72c, 0x311d9049, 0x23a83fa7, 0x9b1458c2, // 188-191
		0xa848e808, 0x10f48f6d, 0x02412083, 0xbafd47e6, // 192-195
		0x272a7f5f, 0x9f96183a, 0x8d23b7d4, 0x359fd0b1, // 196-199
		0x6dfcc0e7, 0xd540a782, 0xc7f5086c, 0x7f496f09, // 200-203
		0xe29e57b0, 0x5a2230d5, 0x48979f3b, 0xf02bf85e, // 204-207
		0xf851bf97, 0x40edd8f2, 0x5258771c, 0xeae41079, // 208-211
		0x773328c0, 0xcf8f4fa5, 0xdd3ae04b, 0x6586872e, // 212-215
		0x3de59778, 0x8559f01d, 0x97ec5ff3, 0x2f503896, // 216-219
		0xb287002f, 0x0a3b674a, 0x188ec8a4, 0xa032afc1, // 220-223
		0x087a4736, 0xb0c62053, 0xa2738fbd, 0x1acfe8d8, // 224-227
		0x8718d061, 0x3fa4b704, 0x2d1118ea, 0x95ad7f8f, // 228-231
		0xcdce6fd9, 0x757208bc, 0x67c7a752, 0xdf7bc037, // 232-235
		0x42acf88e, 0xfa109feb, 0xe8a53005, 0x50195760, // 236-239
		0x586310a9, 0xe0df77cc, 0xf26ad822, 0x4ad6bf47, // 240-243
		0xd70187fe, 0x6fbde09b, 0x7d084f75, 0xc5b42810, // 244-247
		0x9dd73846, 0x256b5f23, 0x37def0cd, 0x8f6297a8, // 248-251
		0x12b5af11, 0xaa09c874, 0xb8bc679a, 0x000000ff, // 252-255
	}
};

/* This table contains the fixed huffman encoding of all individual match
 * distances in order not to have to compute them. It was generated by
 * mkfhdist.sh. Despite its apparent size, it's only 128kB and actually
 * improves performance compared to a static table computed at initialization
 * time. It also removes the need for any initialization phase at all in the
 * library.
 */
static const uint32_t fh_dist_table[32768] = {
	0x00000005, 0x00000205, 0x00000105, 0x00000305, 0x00000086, 0x00000486, 0x00000286, 0x00000686, // 0-7
	0x00000187, 0x00000587, 0x00000987, 0x00000d87, 0x00000387, 0x00000787, 0x00000b87, 0x00000f87, // 8-15
	0x00000048, 0x00000448, 0x00000848, 0x00000c48, 0x00001048, 0x00001448, 0x00001848, 0x00001c48, // 16-23
	0x00000248, 0x00000648, 0x00000a48, 0x00000e48, 0x00001248, 0x00001648, 0x00001a48, 0x00001e48, // 24-31
	0x00000149, 0x00000549, 0x00000949, 0x00000d49, 0x00001149, 0x00001549, 0x00001949, 0x00001d49, // 32-39
	0x00002149, 0x00002549, 0x00002949, 0x00002d49, 0x00003149, 0x00003549, 0x00003949, 0x00003d49, // 40-47
	0x00000349, 0x00000749, 0x00000b49, 0x00000f49, 0x00001349, 0x00001749, 0x00001b49, 0x00001f49, // 48-55
	0x00002349, 0x00002749, 0x00002b49, 0x00002f49, 0x00003349, 0x00003749, 0x00003b49, 0x00003f49, // 56-63
	0x000000ca, 0x000004ca, 0x000008ca, 0x00000cca, 0x000010ca, 0x000014ca, 0x000018ca, 0x00001cca, // 64-71
	0x000020ca, 0x000024ca, 0x000028ca, 0x00002cca, 0x000030ca, 0x000034ca, 0x000038ca, 0x00003cca, // 72-79
	0x000040ca, 0x000044ca, 0x000048ca, 0x00004cca, 0x000050ca, 0x000054ca, 0x000058ca, 0x00005cca, // 80-87
	0x000060ca, 0x000064ca, 0x000068ca, 0x00006cca, 0x000070ca, 0x000074ca, 0x000078ca, 0x00007cca, // 88-95
	0x000002ca, 0x000006ca, 0x00000aca, 0x00000eca, 0x000012ca, 0x000016ca, 0x00001aca, 0x00001eca, // 96-103
	0x000022ca, 0x000026ca, 0x00002aca, 0x00002eca, 0x000032ca, 0x000036ca, 0x00003aca, 0x00003eca, // 104-111
	0x000042ca, 0x000046ca, 0x00004aca, 0x00004eca, 0x000052ca, 0x000056ca, 0x00005aca, 0x00005eca, // 112-119
	0x000062ca, 0x000066ca, 0x00006aca, 0x00006eca, 0x000072ca, 0x000076ca, 0x00007aca, 0x00007eca, // 120-127
	0x000001cb, 0x000005cb, 0x000009cb, 0x00000dcb, 0x000011cb, 0x000015cb, 0x000019cb, 0x00001dcb, // 128-135
	0x000021cb, 0x000025cb, 0x000029cb, 0x00002dcb, 0x000031cb, 0x000035cb, 0x000039cb, 0x00003dcb, // 136-143
	0x000041cb, 0x000045cb, 0x000049cb, 0x00004dcb, 0x000051cb, 0x000055cb, 0x000059cb, 0x00005dcb, // 144-151
	0x000061cb, 0x000065cb, 0x000069cb, 0x00006dcb, 0x000071cb, 0x000075cb, 0x000079cb, 0x00007dcb, // 152-159
	0x000081cb, 0x000085cb, 0x000089cb, 0x00008dcb, 0x000091cb, 0x000095cb, 0x000099cb, 0x00009dcb, // 160-167
	0x0000a1cb, 0x0000a5cb, 0x0000a9cb, 0x0000adcb, 0x0000b1cb, 0x0000b5cb, 0x0000b9cb, 0x0000bdcb, // 168-175
	0x0000c1cb, 0x0000c5cb, 0x0000c9cb, 0x0000cdcb, 0x0000d1cb, 0x0000d5cb, 0x0000d9cb, 0x0000ddcb, // 176-183
	0x0000e1cb, 0x0000e5cb, 0x0000e9cb, 0x0000edcb, 0x0000f1cb, 0x0000f5cb, 0x0000f9cb, 0x0000fdcb, // 184-191
	0x000003cb, 0x000007cb, 0x00000bcb, 0x00000fcb, 0x000013cb, 0x000017cb, 0x00001bcb, 0x00001fcb, // 192-199
	0x000023cb, 0x000027cb, 0x00002bcb, 0x00002fcb, 0x000033cb, 0x000037cb, 0x00003bcb, 0x00003fcb, // 200-207
	0x000043cb, 0x000047cb, 0x00004bcb, 0x00004fcb, 0x000053cb, 0x000057cb, 0x00005bcb, 0x00005fcb, // 208-215
	0x000063cb, 0x000067cb, 0x00006bcb, 0x00006fcb, 0x000073cb, 0x000077cb, 0x00007bcb, 0x00007fcb, // 216-223
	0x000083cb, 0x000087cb, 0x00008bcb, 0x00008fcb, 0x000093cb, 0x000097cb, 0x00009bcb, 0x00009fcb, // 224-231
	0x0000a3cb, 0x0000a7cb, 0x0000abcb, 0x0000afcb, 0x0000b3cb, 0x0000b7cb, 0x0000bbcb, 0x0000bfcb, // 232-239
	0x0000c3cb, 0x0000c7cb, 0x0000cbcb, 0x0000cfcb, 0x0000d3cb, 0x0000d7cb, 0x0000dbcb, 0x0000dfcb, // 240-247
	0x0000e3cb, 0x0000e7cb, 0x0000ebcb, 0x0000efcb, 0x0000f3cb, 0x0000f7cb, 0x0000fbcb, 0x0000ffcb, // 248-255
	0x0000002c, 0x0000042c, 0x0000082c, 0x00000c2c, 0x0000102c, 0x0000142c, 0x0000182c, 0x00001c2c, // 256-263
	0x0000202c, 0x0000242c, 0x0000282c, 0x00002c2c, 0x0000302c, 0x0000342c, 0x0000382c, 0x00003c2c, // 264-271
	0x0000402c, 0x0000442c, 0x0000482c, 0x00004c2c, 0x0000502c, 0x0000542c, 0x0000582c, 0x00005c2c, // 272-279
	0x0000602c, 0x0000642c, 0x0000682c, 0x00006c2c, 0x0000702c, 0x0000742c, 0x0000782c, 0x00007c2c, // 280-287
	0x0000802c, 0x0000842c, 0x0000882c, 0x00008c2c, 0x0000902c, 0x0000942c, 0x0000982c, 0x00009c2c, // 288-295
	0x0000a02c, 0x0000a42c, 0x0000a82c, 0x0000ac2c, 0x0000b02c, 0x0000b42c, 0x0000b82c, 0x0000bc2c, // 296-303
	0x0000c02c, 0x0000c42c, 0x0000c82c, 0x0000cc2c, 0x0000d02c, 0x0000d42c, 0x0000d82c, 0x0000dc2c, // 304-311
	0x0000e02c, 0x0000e42c, 0x0000e82c, 0x0000ec2c, 0x0000f02c, 0x0000f42c, 0x0000f82c, 0x0000fc2c, // 312-319
	0x0001002c, 0x0001042c, 0x0001082c, 0x00010c2c, 0x0001102c, 0x0001142c, 0x0001182c, 0x00011c2c, // 320-327
	0x0001202c, 0x0001242c, 0x0001282c, 0x00012c2c, 0x0001302c, 0x0001342c, 0x0001382c, 0x00013c2c, // 328-335
	0x0001402c, 0x0001442c, 0x0001482c, 0x00014c2c, 0x0001502c, 0x0001542c, 0x0001582c, 0x00015c2c, // 336-343
	0x0001602c, 0x0001642c, 0x0001682c, 0x00016c2c, 0x0001702c, 0x0001742c, 0x0001782c, 0x00017c2c, // 344-351
	0x0001802c, 0x0001842c, 0x0001882c, 0x00018c2c, 0x0001902c, 0x0001942c, 0x0001982c, 0x00019c2c, // 352-359
	0x0001a02c, 0x0001a42c, 0x0001a82c, 0x0001ac2c, 0x0001b02c, 0x0001b42c, 0x0001b82c, 0x0001bc2c, // 360-367
	0x0001c02c, 0x0001c42c, 0x0001c82c, 0x0001cc2c, 0x0001d02c, 0x0001d42c, 0x0001d82c, 0x0001dc2c, // 368-375
	0x0001e02c, 0x0001e42c, 0x0001e82c, 0x0001ec2c, 0x0001f02c, 0x0001f42c, 0x0001f82c, 0x0001fc2c, // 376-383
	0x0000022c, 0x0000062c, 0x00000a2c, 0x00000e2c, 0x0000122c, 0x0000162c, 0x00001a2c, 0x00001e2c, // 384-391
	0x0000222c, 0x0000262c, 0x00002a2c, 0x00002e2c, 0x0000322c, 0x0000362c, 0x00003a2c, 0x00003e2c, // 392-399
	0x0000422c, 0x0000462c, 0x00004a2c, 0x00004e2c, 0x0000522c, 0x0000562c, 0x00005a2c, 0x00005e2c, // 400-407
	0x0000622c, 0x0000662c, 0x00006a2c, 0x00006e2c, 0x0000722c, 0x0000762c, 0x00007a2c, 0x00007e2c, // 408-415
	0x0000822c, 0x0000862c, 0x00008a2c, 0x00008e2c, 0x0000922c, 0x0000962c, 0x00009a2c, 0x00009e2c, // 416-423
	0x0000a22c, 0x0000a62c, 0x0000aa2c, 0x0000ae2c, 0x0000b22c, 0x0000b62c, 0x0000ba2c, 0x0000be2c, // 424-431
	0x0000c22c, 0x0000c62c, 0x0000ca2c, 0x0000ce2c, 0x0000d22c, 0x0000d62c, 0x0000da2c, 0x0000de2c, // 432-439
	0x0000e22c, 0x0000e62c, 0x0000ea2c, 0x0000ee2c, 0x0000f22c, 0x0000f62c, 0x0000fa2c, 0x0000fe2c, // 440-447
	0x0001022c, 0x0001062c, 0x00010a2c, 0x00010e2c, 0x0001122c, 0x0001162c, 0x00011a2c, 0x00011e2c, // 448-455
	0x0001222c, 0x0001262c, 0x00012a2c, 0x00012e2c, 0x0001322c, 0x0001362c, 0x00013a2c, 0x00013e2c, // 456-463
	0x0001422c, 0x0001462c, 0x00014a2c, 0x00014e2c, 0x0001522c, 0x0001562c, 0x00015a2c, 0x00015e2c, // 464-471
	0x0001622c, 0x0001662c, 0x00016a2c, 0x00016e2c, 0x0001722c, 0x0001762c, 0x00017a2c, 0x00017e2c, // 472-479
	0x0001822c, 0x0001862c, 0x00018a2c, 0x00018e2c, 0x0001922c, 0x0001962c, 0x00019a2c, 0x00019e2c, // 480-487
	0x0001a22c, 0x0001a62c, 0x0001aa2c, 0x0001ae2c, 0x0001b22c, 0x0001b62c, 0x0001ba2c, 0x0001be2c, // 488-495
	0x0001c22c, 0x0001c62c, 0x0001ca2c, 0x0001ce2c, 0x0001d22c, 0x0001d62c, 0x0001da2c, 0x0001de2c, // 496-503
	0x0001e22c, 0x0001e62c, 0x0001ea2c, 0x0001ee2c, 0x0001f22c, 0x0001f62c, 0x0001fa2c, 0x0001fe2c, // 504-511
	0x0000012d, 0x0000052d, 0x0000092d, 0x00000d2d, 0x0000112d, 0x0000152d, 0x0000192d, 0x00001d2d, // 512-519
	0x0000212d, 0x0000252d, 0x0000292d, 0x00002d2d, 0x0000312d, 0x0000352d, 0x0000392d, 0x00003d2d, // 520-527
	0x0000412d, 0x0000452d, 0x0000492d, 0x00004d2d, 0x0000512d, 0x0000552d, 0x0000592d, 0x00005d2d, // 528-535
	0x0000612d, 0x0000652d, 0x0000692d, 0x00006d2d, 0x0000712d, 0x0000752d, 0x0000792d, 0x00007d2d, // 536-543
	0x0000812d, 0x0000852d, 0x0000892d, 0x00008d2d, 0x0000912d, 0x0000952d, 0x0000992d, 0x00009d2d, // 544-551
	0x0000a12d, 0x0000a52d, 0x0000a92d, 0x0000ad2d, 0x0000b12d, 0x0000b52d, 0x0000b92d, 0x0000bd2d, // 552-559
	0x0000c12d, 0x0000c52d, 0x0000c92d, 0x0000cd2d, 0x0000d12d, 0x0000d52d, 0x0000d92d, 0x0000dd2d, // 560-567
	0x0000e12d, 0x0000e52d, 0x0000e92d, 0x0000ed2d, 0x0000f12d, 0x0000f52d, 0x0000f92d, 0x0000fd2d, // 568-575
	0x0001012d, 0x0001052d, 0x0001092d, 0x00010d2d, 0x0001112d, 0x0001152d, 0x0001192d, 0x00011d2d, // 576-583
	0x0001212d, 0x0001252d, 0x0001292d, 0x00012d2d, 0x0001312d, 0x0001352d, 0x0001392d, 0x00013d2d, // 584-591
	0x0001412d, 0x0001452d, 0x0001492d, 0x00014d2d, 0x0001512d, 0x0001552d, 0x0001592d, 0x00015d2d, // 592-599
	0x0001612d, 0x0001652d, 0x0001692d, 0x00016d2d, 0x0001712d, 0x0001752d, 0x0001792d, 0x00017d2d, // 600-607
	0x0001812d, 0x0001852d, 0x0001892d, 0x00018d2d, 0x0001912d, 0x0001952d, 0x0001992d, 0x00019d2d, // 608-615
	0x0001a12d, 0x0001a52d, 0x0001a92d, 0x0001ad2d, 0x0001b12d, 0x0001b52d, 0x0001b92d, 0x0001bd2d, // 616-623
	0x0001c12d, 0x0001c52d, 0x0001c92d, 0x0001cd2d, 0x0001d12d, 0x0001d52d, 0x0001d92d, 0x0001dd2d, // 624-631
	0x0001e12d, 0x0001e52d, 0x0001e92d, 0x0001ed2d, 0x0001f12d, 0x0001f52d, 0x0001f92d, 0x0001fd2d, // 632-639
	0x0002012d, 0x0002052d, 0x0002092d, 0x00020d2d, 0x0002112d, 0x0002152d, 0x0002192d, 0x00021d2d, // 640-647
	0x0002212d, 0x0002252d, 0x0002292d, 0x00022d2d, 0x0002312d, 0x0002352d, 0x0002392d, 0x00023d2d, // 648-655
	0x0002412d, 0x0002452d, 0x0002492d, 0x00024d2d, 0x0002512d, 0x0002552d, 0x0002592d, 0x00025d2d, // 656-663
	0x0002612d, 0x0002652d, 0x0002692d, 0x00026d2d, 0x0002712d, 0x0002752d, 0x0002792d, 0x00027d2d, // 664-671
	0x0002812d, 0x0002852d, 0x0002892d, 0x00028d2d, 0x0002912d, 0x0002952d, 0x0002992d, 0x00029d2d, // 672-679
	0x0002a12d, 0x0002a52d, 0x0002a92d, 0x0002ad2d, 0x0002b12d, 0x0002b52d, 0x0002b92d, 0x0002bd2d, // 680-687
	0x0002c12d, 0x0002c52d, 0x0002c92d, 0x0002cd2d, 0x0002d12d, 0x0002d52d, 0x0002d92d, 0x0002dd2d, // 688-695
	0x0002e12d, 0x0002e52d, 0x0002e92d, 0x0002ed2d, 0x0002f12d, 0x0002f52d, 0x0002f92d, 0x0002fd2d, // 696-703
	0x0003012d, 0x0003052d, 0x0003092d, 0x00030d2d, 0x0003112d, 0x0003152d, 0x0003192d, 0x00031d2d, // 704-711
	0x0003212d, 0x0003252d, 0x0003292d, 0x00032d2d, 0x0003312d, 0x0003352d, 0x0003392d, 0x00033d2d, // 712-719
	0x0003412d, 0x0003452d, 0x0003492d, 0x00034d2d, 0x0003512d, 0x0003552d, 0x0003592d, 0x00035d2d, // 720-727
	0x0003612d, 0x0003652d, 0x0003692d, 0x00036d2d, 0x0003712d, 0x0003752d, 0x0003792d, 0x00037d2d, // 728-735
	0x0003812d, 0x0003852d, 0x0003892d, 0x00038d2d, 0x0003912d, 0x0003952d, 0x0003992d, 0x00039d2d, // 736-743
	0x0003a12d, 0x0003a52d, 0x0003a92d, 0x0003ad2d, 0x0003b12d, 0x0003b52d, 0x0003b92d, 0x0003bd2d, // 744-751
	0x0003c12d, 0x0003c52d, 0x0003c92d, 0x0003cd2d, 0x0003d12d, 0x0003d52d, 0x0003d92d, 0x0003dd2d, // 752-759
	0x0003e12d, 0x0003e52d, 0x0003e92d, 0x0003ed2d, 0x0003f12d, 0x0003f52d, 0x0003f92d, 0x0003fd2d, // 760-767
	0x0000032d, 0x0000072d, 0x00000b2d, 0x00000f2d, 0x0000132d, 0x0000172d, 0x00001b2d, 0x00001f2d, // 768-775
	0x0000232d, 0x0000272d, 0x00002b2d, 0x00002f2d, 0x0000332d, 0x0000372d, 0x00003b2d, 0x00003f2d, // 776-783
	0x0000432d, 0x0000472d, 0x00004b2d, 0x00004f2d, 0x0000532d, 0x0000572d, 0x00005b2d, 0x00005f2d, // 784-791
	0x0000632d, 0x0000672d, 0x00006b2d, 0x00006f2d, 0x0000732d, 0x0000772d, 0x00007b2d, 0x00007f2d, // 792-799
	0x0000832d, 0x0000872d, 0x00008b2d, 0x00008f2d, 0x0000932d, 0x0000972d, 0x00009b2d, 0x00009f2d, // 800-807
	0x0000a32d, 0x0000a72d, 0x0000ab2d, 0x0000af2d, 0x0000b32d, 0x0000b72d, 0x0000bb2d, 0x0000bf2d, // 808-815
	0x0000c32d, 0x0000c72d, 0x0000cb2d, 0x0000cf2d, 0x0000d32d, 0x0000d72d, 0x0000db2d, 0x0000df2d, // 816-823
	0x0000e32d, 0x0000e72d, 0x0000eb2d, 0x0000ef2d, 0x0000f32d, 0x0000f72d, 0x0000fb2d, 0x0000ff2d, // 824-831
	0x0001032d, 0x0001072d, 0x00010b2d, 0x00010f2d, 0x0001132d, 0x0001172d, 0x00011b2d, 0x00011f2d, // 832-839
	0x0001232d, 0x0001272d, 0x00012b2d, 0x00012f2d, 0x0001332d, 0x0001372d, 0x00013b2d, 0x00013f2d, // 840-847
	0x0001432d, 0x0001472d, 0x00014b2d, 0x00014f2d, 0x0001532d, 0x0001572d, 0x00015b2d, 0x00015f2d, // 848-855
	0x0001632d, 0x0001672d, 0x00016b2d, 0x00016f2d, 0x0001732d, 0x0001772d, 0x00017b2d, 0x00017f2d, // 856-863
	0x0001832d, 0x0001872d, 0x00018b2d, 0x00018f2d, 0x0001932d, 0x0001972d, 0x00019b2d, 0x00019f2d, // 864-871
	0x0001a32d, 0x0001a72d, 0x0001ab2d, 0x0001af2d, 0x0001b32d, 0x0001b72d, 0x0001bb2d, 0x0001bf2d, // 872-879
	0x0001c32d, 0x0001c72d, 0x0001cb2d, 0x0001cf2d, 0x0001d32d, 0x0001d72d, 0x0001db2d, 0x0001df2d, // 880-887
	0x0001e32d, 0x0001e72d, 0x0001eb2d, 0x0001ef2d, 0x0001f32d, 0x0001f72d, 0x0001fb2d, 0x0001ff2d, // 888-895
	0x0002032d, 0x0002072d, 0x00020b2d, 0x00020f2d, 0x0002132d, 0x0002172d, 0x00021b2d, 0x00021f2d, // 896-903
	0x0002232d, 0x0002272d, 0x00022b2d, 0x00022f2d, 0x0002332d, 0x0002372d, 0x00023b2d, 0x00023f2d, // 904-911
	0x0002432d, 0x0002472d, 0x00024b2d, 0x00024f2d, 0x0002532d, 0x0002572d, 0x00025b2d, 0x00025f2d, // 912-919
	0x0002632d, 0x0002672d, 0x00026b2d, 0x00026f2d, 0x0002732d, 0x0002772d, 0x00027b2d, 0x00027f2d, // 920-927
	0x0002832d, 0x0002872d, 0x00028b2d, 0x00028f2d, 0x0002932d, 0x0002972d, 0x00029b2d, 0x00029f2d, // 928-935
	0x0002a32d, 0x0002a72d, 0x0002ab2d, 0x0002af2d, 0x0002b32d, 0x0002b72d, 0x0002bb2d, 0x0002bf2d, // 936-943
	0x0002c32d, 0x0002c72d, 0x0002cb2d, 0x0002cf2d, 0x0002d32d, 0x0002d72d, 0x0002db2d, 0x0002df2d, // 944-951
	0x0002e32d, 0x0002e72d, 0x0002eb2d, 0x0002ef2d, 0x0002f32d, 0x0002f72d, 0x0002fb2d, 0x0002ff2d, // 952-959
	0x0003032d, 0x0003072d, 0x00030b2d, 0x00030f2d, 0x0003132d, 0x0003172d, 0x00031b2d, 0x00031f2d, // 960-967
	0x0003232d, 0x0003272d, 0x00032b2d, 0x00032f2d, 0x0003332d, 0x0003372d, 0x00033b2d, 0x00033f2d, // 968-975
	0x0003432d, 0x0003472d, 0x00034b2d, 0x00034f2d, 0x0003532d, 0x0003572d, 0x00035b2d, 0x00035f2d, // 976-983
	0x0003632d, 0x0003672d, 0x00036b2d, 0x00036f2d, 0x0003732d, 0x0003772d, 0x00037b2d, 0x00037f2d, // 984-991
	0x0003832d, 0x0003872d, 0x00038b2d, 0x00038f2d, 0x0003932d, 0x0003972d, 0x00039b2d, 0x00039f2d, // 992-999
	0x0003a32d, 0x0003a72d, 0x0003ab2d, 0x0003af2d, 0x0003b32d, 0x0003b72d, 0x0003bb2d, 0x0003bf2d, // 1000-1007
	0x0003c32d, 0x0003c72d, 0x0003cb2d, 0x0003cf2d, 0x0003d32d, 0x0003d72d, 0x0003db2d, 0x0003df2d, // 1008-1015
	0x0003e32d, 0x0003e72d, 0x0003eb2d, 0x0003ef2d, 0x0003f32d, 0x0003f72d, 0x0003fb2d, 0x0003ff2d, // 1016-1023
	0x000000ae, 0x000004ae, 0x000008ae, 0x00000cae, 0x000010ae, 0x000014ae, 0x000018ae, 0x00001cae, // 1024-1031
	0x000020ae, 0x000024ae, 0x000028ae, 0x00002cae, 0x000030ae, 0x000034ae, 0x000038ae, 0x00003cae, // 1032-1039
	0x000040ae, 0x000044ae, 0x000048ae, 0x00004cae, 0x000050ae, 0x000054ae, 0x000058ae, 0x00005cae, // 1040-1047
	0x000060ae, 0x000064ae, 0x000068ae, 0x00006cae, 0x000070ae, 0x000074ae, 0x000078ae, 0x00007cae, // 1048-1055
	0x000080ae, 0x000084ae, 0x000088ae, 0x00008cae, 0x000090ae, 0x000094ae, 0x000098ae, 0x00009cae, // 1056-1063
	0x0000a0ae, 0x0000a4ae, 0x0000a8ae, 0x0000acae, 0x0000b0ae, 0x0000b4ae, 0x0000b8ae, 0x0000bcae, // 1064-1071
	0x0000c0ae, 0x0000c4ae, 0x0000c8ae, 0x0000ccae, 0x0000d0ae, 0x0000d4ae, 0x0000d8ae, 0x0000dcae, // 1072-1079
	0x0000e0ae, 0x0000e4ae, 0x0000e8ae, 0x0000ecae, 0x0000f0ae, 0x0000f4ae, 0x0000f8ae, 0x0000fcae, // 1080-1087
	0x000100ae, 0x000104ae, 0x000108ae, 0x00010cae, 0x000110ae, 0x000114ae, 0x000118ae, 0x00011cae, // 1088-1095
	0x000120ae, 0x000124ae, 0x000128ae, 0x00012cae, 0x000130ae, 0x000134ae, 0x000138ae, 0x00013cae, // 1096-1103
	0x000140ae, 0x000144ae, 0x000148ae, 0x00014cae, 0x000150ae, 0x000154ae, 0x000158ae, 0x00015cae, // 1104-1111
	0x000160ae, 0x000164ae, 0x000168ae, 0x00016cae, 0x000170ae, 0x000174ae, 0x000178ae, 0x00017cae, // 1112-1119
	0x000180ae, 0x000184ae, 0x000188ae, 0x00018cae, 0x000190ae, 0x000194ae, 0x000198ae, 0x00019cae, // 1120-1127
	0x0001a0ae, 0x0001a4ae, 0x0001a8ae, 0x0001acae, 0x0001b0ae, 0x0001b4ae, 0x0001b8ae, 0x0001bcae, // 1128-1135
	0x0001c0ae, 0x0001c4ae, 0x0001c8ae, 0x0001ccae, 0x0001d0ae, 0x0001d4ae, 0x0001d8ae, 0x0001dcae, // 1136-1143
	0x0001e0ae, 0x0001e4ae, 0x0001e8ae, 0x0001ecae, 0x0001f0ae, 0x0001f4ae, 0x0001f8ae, 0x0001fcae, // 1144-1151
	0x000200ae, 0x000204ae, 0x000208ae, 0x00020cae, 0x000210ae, 0x000214ae, 0x000218ae, 0x00021cae, // 1152-1159
	0x000220ae, 0x000224ae, 0x000228ae, 0x00022cae, 0x000230ae, 0x000234ae, 0x000238ae, 0x00023cae, // 1160-1167
	0x000240ae, 0x000244ae, 0x000248ae, 0x00024cae, 0x000250ae, 0x000254ae, 0x000258ae, 0x00025cae, // 1168-1175
	0x000260ae, 0x000264ae, 0x000268ae, 0x00026cae, 0x000270ae, 0x000274ae, 0x000278ae, 0x00027cae, // 1176-1183
	0x000280ae, 0x000284ae, 0x000288ae, 0x00028cae, 0x000290ae, 0x000294ae, 0x000298ae, 0x00029cae, // 1184-1191
	0x0002a0ae, 0x0002a4ae, 0x0002a8ae, 0x0002acae, 0x0002b0ae, 0x0002b4ae, 0x0002b8ae, 0x0002bcae, // 1192-1199
	0x0002c0ae, 0x0002c4ae, 0x0002c8ae, 0x0002ccae, 0x0002d0ae, 0x0002d4ae, 0x0002d8ae, 0x0002dcae, // 1200-1207
	0x0002e0ae, 0x0002e4ae, 0x0002e8ae, 0x0002ecae, 0x0002f0ae, 0x0002f4ae, 0x0002f8ae, 0x0002fcae, // 1208-1215
	0x000300ae, 0x000304ae, 0x000308ae, 0x00030cae, 0x000310ae, 0x000314ae, 0x000318ae, 0x00031cae, // 1216-1223
	0x000320ae, 0x000324ae, 0x000328ae, 0x00032cae, 0x000330ae, 0x000334ae, 0x000338ae, 0x00033cae, // 1224-1231
	0x000340ae, 0x000344ae, 0x000348ae, 0x00034cae, 0x000350ae, 0x000354ae, 0x000358ae, 0x00035cae, // 1232-1239
	0x000360ae, 0x000364ae, 0x000368ae, 0x00036cae, 0x000370ae, 0x000374ae, 0x000378ae, 0x00037cae, // 1240-1247
	0x000380ae, 0x000384ae, 0x000388ae, 0x00038cae, 0x000390ae, 0x000394ae, 0x000398ae, 0x00039cae, // 1248-1255
	0x0003a0ae, 0x0003a4ae, 0x0003a8ae, 0x0003acae, 0x0003b0ae, 0x0003b4ae, 0x0003b8ae, 0x0003bcae, // 1256-1263
	0x0003c0ae, 0x0003c4ae, 0x0003c8ae, 0x0003ccae, 0x0003d0ae, 0x0003d4ae, 0x0003d8ae, 0x0003dcae, // 1264-1271
	0x0003e0ae, 0x0003e4ae, 0x0003e8ae, 0x0003ecae, 0x0003f0ae, 0x0003f4ae, 0x0003f8ae, 0x0003fcae, // 1272-1279
	0x000400ae, 0x000404ae, 0x000408ae, 0x00040cae, 0x000410ae, 0x000414ae, 0x000418ae, 0x00041cae, // 1280-1287
	0x000420ae, 0x000424ae, 0x000428ae, 0x00042cae, 0x000430ae, 0x000434ae, 0x000438ae, 0x00043cae, // 1288-1295
	0x000440ae, 0x000444ae, 0x000448ae, 0x00044cae, 0x000450ae, 0x000454ae, 0x000458ae, 0x00045cae, // 1296-1303
	0x000460ae, 0x000464ae, 0x000468ae, 0x00046cae, 0x000470ae, 0x000474ae, 0x000478ae, 0x00047cae, // 1304-1311
	0x000480ae, 0x000484ae, 0x000488ae, 0x00048cae, 0x000490ae, 0x000494ae, 0x000498ae, 0x00049cae, // 1312-1319
	0x0004a0ae, 0x0004a4ae, 0x0004a8ae, 0x0004acae, 0x0004b0ae, 0x0004b4ae, 0x0004b8ae, 0x0004bcae, // 1320-1327
	0x0004c0ae, 0x0004c4ae, 0x0004c8ae, 0x0004ccae, 0x0004d0ae, 0x0004d4ae, 0x0004d8ae, 0x0004dcae, // 1328-1335
	0x0004e0ae, 0x0004e4ae, 0x0004e8ae, 0x0004ecae, 0x0004f0ae, 0x0004f4ae, 0x0004f8ae, 0x0004fcae, // 1336-1343
	0x000500ae, 0x000504ae, 0x000508ae, 0x00050cae, 0x000510ae, 0x000514ae, 0x000518ae, 0x00051cae, // 1344-1351
	0x000520ae, 0x000524ae, 0x000528ae, 0x00052cae, 0x000530ae, 0x000534ae, 0x000538ae, 0x00053cae, // 1352-1359
	0x000540ae, 0x000544ae, 0x000548ae, 0x00054cae, 0x000550ae, 0x000554ae, 0x000558ae, 0x00055cae, // 1360-1367
	0x000560ae, 0x000564ae, 0x000568ae, 0x00056cae, 0x000570ae, 0x000574ae, 0x000578ae, 0x00057cae, // 1368-1375
	0x000580ae, 0x000584ae, 0x000588ae, 0x00058cae, 0x000590ae, 0x000594ae, 0x000598ae, 0x00059cae, // 1376-1383
	0x0005a0ae, 0x0005a4ae, 0x0005a8ae, 0x0005acae, 0x0005b0ae, 0x0005b4ae, 0x0005b8ae, 0x0005bcae, // 1384-1391
	0x0005c0ae, 0x0005c4ae, 0x0005c8ae, 0x0005ccae, 0x0005d0ae, 0x0005d4ae, 0x0005d8ae, 0x0005dcae, // 1392-1399
	0x0005e0ae, 0x0005e4ae, 0x0005e8ae, 0x0005ecae, 0x0005f0ae, 0x0005f4ae, 0x0005f8ae, 0x0005fcae, // 1400-1407
	0x000600ae, 0x000604ae, 0x000608ae, 0x00060cae, 0x000610ae, 0x000614ae, 0x000618ae, 0x00061cae, // 1408-1415
	0x000620ae, 0x000624ae, 0x000628ae, 0x00062cae, 0x000630ae, 0x000634ae, 0x000638ae, 0x00063cae, // 1416-1423
	0x000640ae, 0x000644ae, 0x000648ae, 0x00064cae, 0x000650ae, 0x000654ae, 0x000658ae, 0x00065cae, // 1424-1431
	0x000660ae, 0x000664ae, 0x000668ae, 0x00066cae, 0x000670ae, 0x000674ae, 0x000678ae, 0x00067cae, // 1432-1439
	0x000680ae, 0x000684ae, 0x000688ae, 0x00068cae, 0x000690ae, 0x000694ae, 0x000698ae, 0x00069cae, // 1440-1447
	0x0006a0ae, 0x0006a4ae, 0x0006a8ae, 0x0006acae, 0x0006b0ae, 0x0006b4ae, 0x0006b8ae, 0x0006bcae, // 1448-1455
	0x0006c0ae, 0x0006c4ae, 0x0006c8ae, 0x0006ccae, 0x0006d0ae, 0x0006d4ae, 0x0006d8ae, 0x0006dcae, // 1456-1463
	0x0006e0ae, 0x0006e4ae, 0x0006e8ae, 0x0006ecae, 0x0006f0ae, 0x0006f4ae, 0x0006f8ae, 0x0006fcae, // 1464-1471
	0x000700ae, 0x000704ae, 0x000708ae, 0x00070cae, 0x000710ae, 0x000714ae, 0x000718ae, 0x00071cae, // 1472-1479
	0x000720ae, 0x000724ae, 0x000728ae, 0x00072cae, 0x000730ae, 0x000734ae, 0x000738ae, 0x00073cae, // 1480-1487
	0x000740ae, 0x000744ae, 0x000748ae, 0x00074cae, 0x000750ae, 0x000754ae, 0x000758ae, 0x00075cae, // 1488-1495
	0x000760ae, 0x000764ae, 0x000768ae, 0x00076cae, 0x000770ae, 0x000774ae, 0x000778ae, 0x00077cae, // 1496-1503
	0x000780ae, 0x000784ae, 0x000788ae, 0x00078cae, 0x000790ae, 0x000794ae, 0x000798ae, 0x00079cae, // 1504-1511
	0x0007a0ae, 0x0007a4ae, 0x0007a8ae, 0x0007acae, 0x0007b0ae, 0x0007b4ae, 0x0007b8ae, 0x0007bcae, // 1512-1519
	0x0007c0ae, 0x0007c4ae, 0x0007c8ae, 0x0007ccae, 0x0007d0ae, 0x0007d4ae, 0x0007d8ae, 0x0007dcae, // 1520-1527
	0x0007e0ae, 0x0007e4ae, 0x0007e8ae, 0x0007ecae, 0x0007f0ae, 0x0007f4ae, 0x0007f8ae, 0x0007fcae, // 1528-1535
	0x000002ae, 0x000006ae, 0x00000aae, 0x00000eae, 0x000012ae, 0x000016ae, 0x00001aae, 0x00001eae, // 1536-1543
	0x000022ae, 0x000026ae, 0x00002aae, 0x00002eae, 0x000032ae, 0x000036ae, 0x00003aae, 0x00003eae, // 1544-1551
	0x000042ae, 0x000046ae, 0x00004aae, 0x00004eae, 0x000052ae, 0x000056ae, 0x00005aae, 0x00005eae, // 1552-1559
	0x000062ae, 0x000066ae, 0x00006aae, 0x00006eae, 0x000072ae, 0x000076ae, 0x00007aae, 0x00007eae, // 1560-1567
	0x000082ae, 0x000086ae, 0x00008aae, 0x00008eae, 0x000092ae, 0x000096ae, 0x00009aae, 0x00009eae, // 1568-1575
	0x0000a2ae, 0x0000a6ae, 0x0000aaae, 0x0000aeae, 0x0000b2ae, 0x0000b6ae, 0x0000baae, 0x0000beae, // 1576-1583
	0x0000c2ae, 0x0000c6ae, 0x0000caae, 0x0000ceae, 0x0000d2ae, 0x0000d6ae, 0x0000daae, 0x0000deae, // 1584-1591
	0x0000e2ae, 0x0000e6ae, 0x0000eaae, 0x0000eeae, 0x0000f2ae, 0x0000f6ae, 0x0000faae, 0x0000feae, // 1592-1599
	0x000102ae, 0x000106ae, 0x00010aae, 0x00010eae, 0x000112ae, 0x000116ae, 0x00011aae, 0x00011eae, // 1600-1607
	0x000122ae, 0x000126ae, 0x00012aae, 0x00012eae, 0x000132ae, 0x000136ae, 0x00013aae, 0x00013eae, // 1608-1615
	0x000142ae, 0x000146ae, 0x00014aae, 0x00014eae, 0x000152ae, 0x000156ae, 0x00015aae, 0x00015eae, // 1616-1623
	0x000162ae, 0x000166ae, 0x00016aae, 0x00016eae, 0x000172ae, 0x000176ae, 0x00017aae, 0x00017eae, // 1624-1631
	0x000182ae, 0x000186ae, 0x00018aae, 0x00018eae, 0x000192ae, 0x000196ae, 0x00019aae, 0x00019eae, // 1632-1639
	0x0001a2ae, 0x0001a6ae, 0x0001aaae, 0x0001aeae, 0x0001b2ae, 0x0001b6ae, 0x0001baae, 0x0001beae, // 1640-1647
	0x0001c2ae, 0x0001c6ae, 0x0001caae, 0x0001ceae, 0x0001d2ae, 0x0001d6ae, 0x0001daae, 0x0001deae, // 1648-1655
	0x0001e2ae, 0x0001e6ae, 0x0001eaae, 0x0001eeae, 0x0001f2ae, 0x0001f6ae, 0x0001faae, 0x0001feae, // 1656-1663
	0x000202ae, 0x000206ae, 0x00020aae, 0x00020eae, 0x000212ae, 0x000216ae, 0x00021aae, 0x00021eae, // 1664-1671
	0x000222ae, 0x000226ae, 0x00022aae, 0x00022eae, 0x000232ae, 0x000236ae, 0x00023aae, 0x00023eae, // 1672-1679
	0x000242ae, 0x000246ae, 0x00024aae, 0x00024eae, 0x000252ae, 0x000256ae, 0x00025aae, 0x00025eae, // 1680-1687
	0x000262ae, 0x000266ae, 0x00026aae, 0x00026eae, 0x000272ae, 0x000276ae, 0x00027aae, 0x00027eae, // 1688-1695
	0x000282ae, 0x000286ae, 0x00028aae, 0x00028eae, 0x000292ae, 0x000296ae, 0x00029aae, 0x00029eae, // 1696-1703
	0x0002a2ae, 0x0002a6ae, 0x0002aaae, 0x0002aeae, 0x0002b2ae, 0x0002b6ae, 0x0002baae, 0x0002beae, // 1704-1711
	0x0002c2ae, 0x0002c6ae, 0x0002caae, 0x0002ceae, 0x0002d2ae, 0x0002d6ae, 0x0002daae, 0x0002deae, // 1712-1719
	0x0002e2ae, 0x0002e6ae, 0x0002eaae, 0x0002eeae, 0x0002f2ae, 0x0002f6ae, 0x0002faae, 0x0002feae, // 1720-1727
	0x000302ae, 0x000306ae, 0x00030aae, 0x00030eae, 0x000312ae, 0x000316ae, 0x00031aae, 0x00031eae, // 1728-1735
	0x000322ae, 0x000326ae, 0x00032aae, 0x00032eae, 0x000332ae, 0x000336ae, 0x00033aae, 0x00033eae, // 1736-1743
	0x000342ae, 0x000346ae, 0x00034aae, 0x00034eae, 0x000352ae, 0x000356ae, 0x00035aae, 0x00035eae, // 1744-1751
	0x000362ae, 0x000366ae, 0x00036aae, 0x00036eae, 0x000372ae, 0x000376ae, 0x00037aae, 0x00037eae, // 1752-1759
	0x000382ae, 0x000386ae, 0x00038aae, 0x00038eae, 0x000392ae, 0x000396ae, 0x00039aae, 0x00039eae, // 1760-1767
	0x0003a2ae, 0x0003a6ae, 0x0003aaae, 0x0003aeae, 0x0003b2ae, 0x0003b6ae, 0x0003baae, 0x0003beae, // 1768-1775
	0x0003c2ae, 0x0003c6ae, 0x0003caae, 0x0003ceae, 0x0003d2ae, 0x0003d6ae, 0x0003daae, 0x0003deae, // 1776-1783
	0x0003e2ae, 0x0003e6ae, 0x0003eaae, 0x0003eeae, 0x0003f2ae, 0x0003f6ae, 0x0003faae, 0x0003feae, // 1784-1791
	0x000402ae, 0x000406ae, 0x00040aae, 0x00040eae, 0x000412ae, 0x000416ae, 0x00041aae, 0x00041eae, // 1792-1799
	0x000422ae, 0x000426ae, 0x00042aae, 0x00042eae, 0x000432ae, 0x000436ae, 0x00043aae, 0x00043eae, // 1800-1807
	0x000442ae, 0x000446ae, 0x00044aae, 0x00044eae, 0x000452ae, 0x000456ae, 0x00045aae, 0x00045eae, // 1808-1815
	0x000462ae, 0x000466ae, 0x00046aae, 0x00046eae, 0x000472ae, 0x000476ae, 0x00047aae, 0x00047eae, // 1816-1823
	0x000482ae, 0x000486ae, 0x00048aae, 0x00048eae, 0x000492ae, 0x000496ae, 0x00049aae, 0x00049eae, // 1824-1831
	0x0004a2ae, 0x0004a6ae, 0x0004aaae, 0x0004aeae, 0x0004b2ae, 0x0004b6ae, 0x0004baae, 0x0004beae, // 1832-1839
	0x0004c2ae, 0x0004c6ae, 0x0004caae, 0x0004ceae, 0x0004d2ae, 0x0004d6ae, 0x0004daae, 0x0004deae, // 1840-1847
	0x0004e2ae, 0x0004e6ae, 0x0004eaae, 0x0004eeae, 0x0004f2ae, 0x0004f6ae, 0x0004faae, 0x0004feae, // 1848-1855
	0x000502ae, 0x000506ae, 0x00050aae, 0x00050eae, 0x000512ae, 0x000516ae, 0x00051aae, 0x00051eae, // 1856-1863
	0x000522ae, 0x000526ae, 0x00052aae, 0x00052eae, 0x000532ae, 0x000536ae, 0x00053aae, 0x00053eae, // 1864-1871
	0x000542ae, 0x000546ae, 0x00054aae, 0x00054eae, 0x000552ae, 0x000556ae, 0x00055aae, 0x00055eae, // 1872-1879
	0x000562ae, 0x000566ae, 0x00056aae, 0x00056eae, 0x000572ae, 0x000576ae, 0x00057aae, 0x00057eae, // 1880-1887
	0x000582ae, 0x000586ae, 0x00058aae, 0x00058eae, 0x000592ae, 0x000596ae, 0x00059aae, 0x00059eae, // 1888-1895
	0x0005a2ae, 0x0005a6ae, 0x0005aaae, 0x0005aeae, 0x0005b2ae, 0x0005b6ae, 0x0005baae, 0x0005beae, // 1896-1903
	0x0005c2ae, 0x0005c6ae, 0x0005caae, 0x0005ceae, 0x0005d2ae, 0x0005d6ae, 0x0005daae, 0x0005deae, // 1904-1911
	0x0005e2ae, 0x0005e6ae, 0x0005eaae, 0x0005eeae, 0x0005f2ae, 0x0005f6ae, 0x0005faae, 0x0005feae, // 1912-1919
	0x000602ae, 0x000606ae, 0x00060aae, 0x00060eae, 0x000612ae, 0x000616ae, 0x00061aae, 0x00061eae, // 1920-1927
	0x000622ae, 0x000626ae, 0x00062aae, 0x00062eae, 0x000632ae, 0x000636ae, 0x00063aae, 0x00063eae, // 1928-1935
	0x000642ae, 0x000646ae, 0x00064aae, 0x00064eae, 0x000652ae, 0x000656ae, 0x00065aae, 0x00065eae, // 1936-1943
	0x000662ae, 0x000666ae, 0x00066aae, 0x00066eae, 0x000672ae, 0x000676ae, 0x00067aae, 0x00067eae, // 1944-1951
	0x000682ae, 0x000686ae, 0x00068aae, 0x00068eae, 0x000692ae, 0x000696ae, 0x00069aae, 0x00069eae, // 1952-1959
	0x0006a2ae, 0x0006a6ae, 0x0006aaae, 0x0006aeae, 0x0006b2ae, 0x0006b6ae, 0x0006baae, 0x0006beae, // 1960-1967
	0x0006c2ae, 0x0006c6ae, 0x0006caae, 0x0006ceae, 0x0006d2ae, 0x0006d6ae, 0x0006daae, 0x0006deae, // 1968-1975
	0x0006e2ae, 0x0006e6ae, 0x0006eaae, 0x0006eeae, 0x0006f2ae, 0x0006f6ae, 0x0006faae, 0x0006feae, // 1976-1983
	0x000702ae, 0x000706ae, 0x00070aae, 0x00070eae, 0x000712ae, 0x000716ae, 0x00071aae, 0x00071eae, // 1984-1991
	0x000722ae, 0x000726ae, 0x00072aae, 0x00072eae, 0x000732ae, 0x000736ae, 0x00073aae, 0x00073eae, // 1992-1999
	0x000742ae, 0x000746ae, 0x00074aae, 0x00074eae, 0x000752ae, 0x000756ae, 0x00075aae, 0x00075eae, // 2000-2007
	0x000762ae, 0x000766ae, 0x00076aae, 0x00076eae, 0x000772ae, 0x000776ae, 0x00077aae, 0x00077eae, // 2008-2015
	0x000782ae, 0x000786ae, 0x00078aae, 0x00078eae, 0x000792ae, 0x000796ae, 0x00079aae, 0x00079eae, // 2016-2023
	0x0007a2ae, 0x0007a6ae, 0x0007aaae, 0x0007aeae, 0x0007b2ae, 0x0007b6ae, 0x0007baae, 0x0007beae, // 2024-2031
	0x0007c2ae, 0x0007c6ae, 0x0007caae, 0x0007ceae, 0x0007d2ae, 0x0007d6ae, 0x0007daae, 0x0007deae, // 2032-2039
	0x0007e2ae, 0x0007e6ae, 0x0007eaae, 0x0007eeae, 0x0007f2ae, 0x0007f6ae, 0x0007faae, 0x0007feae, // 2040-2047
	0x000001af, 0x000005af, 0x000009af, 0x00000daf, 0x000011af, 0x000015af, 0x000019af, 0x00001daf, // 2048-2055
	0x000021af, 0x000025af, 0x000029af, 0x00002daf, 0x000031af, 0x000035af, 0x000039af, 0x00003daf, // 2056-2063
	0x000041af, 0x000045af, 0x000049af, 0x00004daf, 0x000051af, 0x000055af, 0x000059af, 0x00005daf, // 2064-2071
	0x000061af, 0x000065af, 0x000069af, 0x00006daf, 0x000071af, 0x000075af, 0x000079af, 0x00007daf, // 2072-2079
	0x000081af, 0x000085af, 0x000089af, 0x00008daf, 0x000091af, 0x000095af, 0x000099af, 0x00009daf, // 2080-2087
	0x0000a1af, 0x0000a5af, 0x0000a9af, 0x0000adaf, 0x0000b1af, 0x0000b5af, 0x0000b9af, 0x0000bdaf, // 2088-2095
	0x0000c1af, 0x0000c5af, 0x0000c9af, 0x0000cdaf, 0x0000d1af, 0x0000d5af, 0x0000d9af, 0x0000ddaf, // 2096-2103
	0x0000e1af, 0x0000e5af, 0x0000e9af, 0x0000edaf, 0x0000f1af, 0x0000f5af, 0x0000f9af, 0x0000fdaf, // 2104-2111
	0x000101af, 0x000105af, 0x000109af, 0x00010daf, 0x000111af, 0x000115af, 0x000119af, 0x00011daf, // 2112-2119
	0x000121af, 0x000125af, 0x000129af, 0x00012daf, 0x000131af, 0x000135af, 0x000139af, 0x00013daf, // 2120-2127
	0x000141af, 0x000145af, 0x000149af, 0x00014daf, 0x000151af, 0x000155af, 0x000159af, 0x00015daf, // 2128-2135
	0x000161af, 0x000165af, 0x000169af, 0x00016daf, 0x000171af, 0x000175af, 0x000179af, 0x00017daf, // 2136-2143
	0x000181af, 0x000185af, 0x000189af, 0x00018daf, 0x000191af, 0x000195af, 0x000199af, 0x00019daf, // 2144-2151
	0x0001a1af, 0x0001a5af, 0x0001a9af, 0x0001adaf, 0x0001b1af, 0x0001b5af, 0x0001b9af, 0x0001bdaf, // 2152-2159
	0x0001c1af, 0x0001c5af, 0x0001c9af, 0x0001cdaf, 0x0001d1af, 0x0001d5af, 0x0001d9af, 0x0001ddaf, // 2160-2167
	0x0001e1af, 0x0001e5af, 0x0001e9af, 0x0001edaf, 0x0001f1af, 0x0001f5af, 0x0001f9af, 0x0001fdaf, // 2168-2175
	0x000201af, 0x000205af, 0x000209af, 0x00020daf, 0x000211af, 0x000215af, 0x000219af, 0x00021daf, // 2176-2183
	0x000221af, 0x000225af, 0x000229af, 0x00022daf, 0x000231af, 0x000235af, 0x000239af, 0x00023daf, // 2184-2191
	0x000241af, 0x000245af, 0x000249af, 0x00024daf, 0x000251af, 0x000255af, 0x000259af, 0x00025daf, // 2192-2199
	0x000261af, 0x000265af, 0x000269af, 0x00026daf, 0x000271af, 0x000275af, 0x000279af, 0x00027daf, // 2200-2207
	0x000281af, 0x000285af, 0x000289af, 0x00028daf, 0x000291af, 0x000295af, 0x000299af, 0x00029daf, // 2208-2215
	0x0002a1af, 0x0002a5af, 0x0002a9af, 0x0002adaf, 0x0002b1af, 0x0002b5af, 0x0002b9af, 0x0002bdaf, // 2216-2223
	0x0002c1af, 0x0002c5af, 0x0002c9af, 0x0002cdaf, 0x0002d1af, 0x0002d5af, 0x0002d9af, 0x0002ddaf, // 2224-2231
	0x0002e1af, 0x0002e5af, 0x0002e9af, 0x0002edaf, 0x0002f1af, 0x0002f5af, 0x0002f9af, 0x0002fdaf, // 2232-2239
	0x000301af, 0x000305af, 0x000309af, 0x00030daf, 0x000311af, 0x000315af, 0x000319af, 0x00031daf, // 2240-2247
	0x000321af, 0x000325af, 0x000329af, 0x00032daf, 0x000331af, 0x000335af, 0x000339af, 0x00033daf, // 2248-2255
	0x000341af, 0x000345af, 0x000349af, 0x00034daf, 0x000351af, 0x000355af, 0x000359af, 0x00035daf, // 2256-2263
	0x000361af, 0x000365af, 0x000369af, 0x00036daf, 0x000371af, 0x000375af, 0x000379af, 0x00037daf, // 2264-2271
	0x000381af, 0x000385af, 0x000389af, 0x00038daf, 0x000391af, 0x000395af, 0x000399af, 0x00039daf, // 2272-2279
	0x0003a1af, 0x0003a5af, 0x0003a9af, 0x0003adaf, 0x0003b1af, 0x0003b5af, 0x0003b9af, 0x0003bdaf, // 2280-2287
	0x0003c1af, 0x0003c5af, 0x0003c9af, 0x0003cdaf, 0x0003d1af, 0x0003d5af, 0x0003d9af, 0x0003ddaf, // 2288-2295
	0x0003e1af, 0x0003e5af, 0x0003e9af, 0x0003edaf, 0x0003f1af, 0x0003f5af, 0x0003f9af, 0x0003fdaf, // 2296-2303
	0x000401af, 0x000405af, 0x000409af, 0x00040daf, 0x000411af, 0x000415af, 0x000419af, 0x00041daf, // 2304-2311
	0x000421af, 0x000425af, 0x000429af, 0x00042daf, 0x000431af, 0x000435af, 0x000439af, 0x00043daf, // 2312-2319
	0x000441af, 0x000445af, 0x000449af, 0x00044daf, 0x000451af, 0x000455af, 0x000459af, 0x00045daf, // 2320-2327
	0x000461af, 0x000465af, 0x000469af, 0x00046daf, 0x000471af, 0x000475af, 0x000479af, 0x00047daf, // 2328-2335
	0x000481af, 0x000485af, 0x000489af, 0x00048daf, 0x000491af, 0x000495af, 0x000499af, 0x00049daf, // 2336-2343
	0x0004a1af, 0x0004a5af, 0x0004a9af, 0x0004adaf, 0x0004b1af, 0x0004b5af, 0x0004b9af, 0x0004bdaf, // 2344-2351
	0x0004c1af, 0x0004c5af, 0x0004c9af, 0x0004cdaf, 0x0004d1af, 0x0004d5af, 0x0004d9af, 0x0004ddaf, // 2352-2359
	0x0004e1af, 0x0004e5af, 0x0004e9af, 0x0004edaf, 0x0004f1af, 0x0004f5af, 0x0004f9af, 0x0004fdaf, // 2360-2367
	0x000501af, 0x000505af, 0x000509af, 0x00050daf, 0x000511af, 0x000515af, 0x000519af, 0x00051daf, // 2368-2375
	0x000521af, 0x000525af, 0x000529af, 0x00052daf, 0x000531af, 0x000535af, 0x000539af, 0x00053daf, // 2376-2383
	0x000541af, 0x000545af, 0x000549af, 0x00054daf, 0x000551af, 0x000555af, 0x000559af, 0x00055daf, // 2384-2391
	0x000561af, 0x000565af, 0x000569af, 0x00056daf, 0x000571af, 0x000575af, 0x000579af, 0x00057daf, // 2392-2399
	0x000581af, 0x000585af, 0x000589af, 0x00058daf, 0x000591af, 0x000595af, 0x000599af, 0x00059daf, // 2400-2407
	0x0005a1af, 0x0005a5af, 0x0005a9af, 0x0005adaf, 0x0005b1af, 0x0005b5af, 0x0005b9af, 0x0005bdaf, // 2408-2415
	0x0005c1af, 0x0005c5af, 0x0005c9af, 0x0005cdaf, 0x0005d1af, 0x0005d5af, 0x0005d9af, 0x0005ddaf, // 2416-2423
	0x0005e1af, 0x0005e5af, 0x0005e9af, 0x0005edaf, 0x0005f1af, 0x0005f5af, 0x0005f9af, 0x0005fdaf, // 2424-2431
	0x000601af, 0x000605af, 0x000609af, 0x00060daf, 0x000611af, 0x000615af, 0x000619af, 0x00061daf, // 2432-2439
	0x000621af, 0x000625af, 0x000629af, 0x00062daf, 0x000631af, 0x000635af, 0x000639af, 0x00063daf, // 2440-2447
	0x000641af, 0x000645af, 0x000649af, 0x00064daf, 0x000651af, 0x000655af, 0x000659af, 0x00065daf, // 2448-2455
	0x000661af, 0x000665af, 0x000669af, 0x00066daf, 0x000671af, 0x000675af, 0x000679af, 0x00067daf, // 2456-2463
	0x000681af, 0x000685af, 0x000689af, 0x00068daf, 0x000691af, 0x000695af, 0x000699af, 0x00069daf, // 2464-2471
	0x0006a1af, 0x0006a5af, 0x0006a9af, 0x0006adaf, 0x0006b1af, 0x0006b5af, 0x0006b9af, 0x0006bdaf, // 2472-2479
	0x0006c1af, 0x0006c5af, 0x0006c9af, 0x0006cdaf, 0x0006d1af, 0x0006d5af, 0x0006d9af, 0x0006ddaf, // 2480-2487
	0x0006e1af, 0x0006e5af, 0x0006e9af, 0x0006edaf, 0x0006f1af, 0x0006f5af, 0x0006f9af, 0x0006fdaf, // 2488-2495
	0x000701af, 0x000705af, 0x000709af, 0x00070daf, 0x000711af, 0x000715af, 0x000719af, 0x00071daf, // 2496-2503
	0x000721af, 0x000725af, 0x000729af, 0x00072daf, 0x000731af, 0x000735af, 0x000739af, 0x00073daf, // 2504-2511
	0x000741af, 0x000745af, 0x000749af, 0x00074daf, 0x000751af, 0x000755af, 0x000759af, 0x00075daf, // 2512-2519
	0x000761af, 0x000765af, 0x000769af, 0x00076daf, 0x000771af, 0x000775af, 0x000779af, 0x00077daf, // 2520-2527
	0x000781af, 0x000785af, 0x000789af, 0x00078daf, 0x000791af, 0x000795af, 0x000799af, 0x00079daf, // 2528-2535
	0x0007a1af, 0x0007a5af, 0x0007a9af, 0x0007adaf, 0x0007b1af, 0x0007b5af, 0x0007b9af, 0x0007bdaf, // 2536-2543
	0x0007c1af, 0x0007c5af, 0x0007c9af, 0x0007cdaf, 0x0007d1af, 0x0007d5af, 0x0007d9af, 0x0007ddaf, // 2544-2551
	0x0007e1af, 0x0007e5af, 0x0007e9af, 0x0007edaf, 0x0007f1af, 0x0007f5af, 0x0007f9af, 0x0007fdaf, // 2552-2559
	0x000801af, 0x000805af, 0x000809af, 0x00080daf, 0x000811af, 0x000815af, 0x000819af, 0x00081daf, // 2560-2567
	0x000821af, 0x000825af, 0x000829af, 0x00082daf, 0x000831af, 0x000835af, 0x000839af, 0x00083daf, // 2568-2575
	0x000841af, 0x000845af, 0x000849af, 0x00084daf, 0x000851af, 0x000855af, 0x000859af, 0x00085daf, // 2576-2583
	0x000861af, 0x000865af, 0x000869af, 0x00086daf, 0x000871af, 0x000875af, 0x000879af, 0x00087daf, // 2584-2591
	0x000881af, 0x000885af, 0x000889af, 0x00088daf, 0x000891af, 0x000895af, 0x000899af, 0x00089daf, // 2592-2599
	0x0008a1af, 0x0008a5af, 0x0008a9af, 0x0008adaf, 0x0008b1af, 0x0008b5af, 0x0008b9af, 0x0008bdaf, // 2600-2607
	0x0008c1af, 0x0008c5af, 0x0008c9af, 0x0008cdaf, 0x0008d1af, 0x0008d5af, 0x0008d9af, 0x0008ddaf, // 2608-2615
	0x0008e1af, 0x0008e5af, 0x0008e9af, 0x0008edaf, 0x0008f1af, 0x0008f5af, 0x0008f9af, 0x0008fdaf, // 2616-2623
	0x000901af, 0x000905af, 0x000909af, 0x00090daf, 0x000911af, 0x000915af, 0x000919af, 0x00091daf, // 2624-2631
	0x000921af, 0x000925af, 0x000929af, 0x00092daf, 0x000931af, 0x000935af, 0x000939af, 0x00093daf, // 2632-2639
	0x000941af, 0x000945af, 0x000949af, 0x00094daf, 0x000951af, 0x000955af, 0x000959af, 0x00095daf, // 2640-2647
	0x000961af, 0x000965af, 0x000969af, 0x00096daf, 0x000971af, 0x000975af, 0x000979af, 0x00097daf, // 2648-2655
	0x000981af, 0x000985af, 0x000989af, 0x00098daf, 0x000991af, 0x000995af, 0x000999af, 0x00099daf, // 2656-2663
	0x0009a1af, 0x0009a5af, 0x0009a9af, 0x0009adaf, 0x0009b1af, 0x0009b5af, 0x0009b9af, 0x0009bdaf, // 2664-2671
	0x0009c1af, 0x0009c5af, 0x0009c9af, 0x0009cdaf, 0x0009d1af, 0x0009d5af, 0x0009d9af, 0x0009ddaf, // 2672-2679
	0x0009e1af, 0x0009e5af, 0x0009e9af, 0x0009edaf, 0x0009f1af, 0x0009f5af, 0x0009f9af, 0x0009fdaf, // 2680-2687
	0x000a01af, 0x000a05af, 0x000a09af, 0x000a0daf, 0x000a11af, 0x000a15af, 0x000a19af, 0x000a1daf, // 2688-2695
	0x000a21af, 0x000a25af, 0x000a29af, 0x000a2daf, 0x000a31af, 0x000a35af, 0x000a39af, 0x000a3daf, // 2696-2703
	0x000a41af, 0x000a45af, 0x000a49af, 0x000a4daf, 0x000a51af, 0x000a55af, 0x000a59af, 0x000a5daf, // 2704-2711
	0x000a61af, 0x000a65af, 0x000a69af, 0x000a6daf, 0x000a71af, 0x000a75af, 0x000a79af, 0x000a7daf, // 2712-2719
	0x000a81af, 0x000a85af, 0x000a89af, 0x000a8daf, 0x000a91af, 0x000a95af, 0x000a99af, 0x000a9daf, // 2720-2727
	0x000aa1af, 0x000aa5af, 0x000aa9af, 0x000aadaf, 0x000ab1af, 0x000ab5af, 0x000ab9af, 0x000abdaf, // 2728-2735
	0x000ac1af, 0x000ac5af, 0x000ac9af, 0x000acdaf, 0x000ad1af, 0x000ad5af, 0x000ad9af, 0x000addaf, // 2736-2743
	0x000ae1af, 0x000ae5af, 0x000ae9af, 0x000aedaf, 0x000af1af, 0x000af5af, 0x000af9af, 0x000afdaf, // 2744-2751
	0x000b01af, 0x000b05af, 0x000b09af, 0x000b0daf, 0x000b11af, 0x000b15af, 0x000b19af, 0x000b1daf, // 2752-2759
	0x000b21af, 0x000b25af, 0x000b29af, 0x000b2daf, 0x000b31af, 0x000b35af, 0x000b39af, 0x000b3daf, // 2760-2767
	0x000b41af, 0x000b45af, 0x000b49af, 0x000b4daf, 0x000b51af, 0x000b55af, 0x000b59af, 0x000b5daf, // 2768-2775
	0x000b61af, 0x000b65af, 0x000b69af, 0x000b6daf, 0x000b71af, 0x000b75af, 0x000b79af, 0x000b7daf, // 2776-2783
	0x000b81af, 0x000b85af, 0x000b89af, 0x000b8daf, 0x000b91af, 0x000b95af, 0x000b99af, 0x000b9daf, // 2784-2791
	0x000ba1af, 0x000ba5af, 0x000ba9af, 0x000badaf, 0x000bb1af, 0x000bb5af, 0x000bb9af, 0x000bbdaf, // 2792-2799
	0x000bc1af, 0x000bc5af, 0x000bc9af, 0x000bcdaf, 0x000bd1af, 0x000bd5af, 0x000bd9af, 0x000bddaf, // 2800-2807
	0x000be1af, 0x000be5af, 0x000be9af, 0x000bedaf, 0x000bf1af, 0x000bf5af, 0x000bf9af, 0x000bfdaf, // 2808-2815
	0x000c01af, 0x000c05af, 0x000c09af, 0x000c0daf, 0x000c11af, 0x000c15af, 0x000c19af, 0x000c1daf, // 2816-2823
	0x000c21af, 0x000c25af, 0x000c29af, 0x000c2daf, 0x000c31af, 0x000c35af, 0x000c39af, 0x000c3daf, // 2824-2831
	0x000c41af, 0x000c45af, 0x000c49af, 0x000c4daf, 0x000c51af, 0x000c55af, 0x000c59af, 0x000c5daf, // 2832-2839
	0x000c61af, 0x000c65af, 0x000c69af, 0x000c6daf, 0x000c71af, 0x000c75af, 0x000c79af, 0x000c7daf, // 2840-2847
	0x000c81af, 0x000c85af, 0x000c89af, 0x000c8daf, 0x000c91af, 0x000c95af, 0x000c99af, 0x000c9daf, // 2848-2855
	0x000ca1af, 0x000ca5af, 0x000ca9af, 0x000cadaf, 0x000cb1af, 0x000cb5af, 0x000cb9af, 0x000cbdaf, // 2856-2863
	0x000cc1af, 0x000cc5af, 0x000cc9af, 0x000ccdaf, 0x000cd1af, 0x000cd5af, 0x000cd9af, 0x000cddaf, // 2864-2871
	0x000ce1af, 0x000ce5af, 0x000ce9af, 0x000cedaf, 0x000cf1af, 0x000cf5af, 0x000cf9af, 0x000cfdaf, // 2872-2879
	0x000d01af, 0x000d05af, 0x000d09af, 0x000d0daf, 0x000d11af, 0x000d15af, 0x000d19af, 0x000d1daf, // 2880-2887
	0x000d21af, 0x000d25af, 0x000d29af, 0x000d2daf, 0x000d31af, 0x000d35af, 0x000d39af, 0x000d3daf, // 2888-2895
	0x000d41af, 0x000d45af, 0x000d49af, 0x000d4daf, 0x000d51af, 0x000d55af, 0x000d59af, 0x000d5daf, // 2896-2903
	0x000d61af, 0x000d65af, 0x000d69af, 0x000d6daf, 0x000d71af, 0x000d75af, 0x000d79af, 0x000d7daf, // 2904-2911
	0x000d81af, 0x000d85af, 0x000d89af, 0x000d8daf, 0x000d91af, 0x000d95af, 0x000d99af, 0x000d9daf, // 2912-2919
	0x000da1af, 0x000da5af, 0x000da9af, 0x000dadaf, 0x000db1af, 0x000db5af, 0x000db9af, 0x000dbdaf, // 2920-2927
	0x000dc1af, 0x000dc5af, 0x000dc9af, 0x000dcdaf, 0x000dd1af, 0x000dd5af, 0x000dd9af, 0x000dddaf, // 2928-2935
	0x000de1af, 0x000de5af, 0x000de9af, 0x000dedaf, 0x000df1af, 0x000df5af, 0x000df9af, 0x000dfdaf, // 2936-2943
	0x000e01af, 0x000e05af, 0x000e09af, 0x000e0daf, 0x000e11af, 0x000e15af, 0x000e19af, 0x000e1daf, // 2944-2951
	0x000e21af, 0x000e25af, 0x000e29af, 0x000e2daf, 0x000e31af, 0x000e35af, 0x000e39af, 0x000e3daf, // 2952-2959
	0x000e41af, 0x000e45af, 0x000e49af, 0x000e4daf, 0x000e51af, 0x000e55af, 0x000e59af, 0x000e5daf, // 2960-2967
	0x000e61af, 0x000e65af, 0x000e69af, 0x000e6daf, 0x000e71af, 0x000e75af, 0x000e79af, 0x000e7daf, // 2968-2975
	0x000e81af, 0x000e85af, 0x000e89af, 0x000e8daf, 0x000e91af, 0x000e95af, 0x000e99af, 0x000e9daf, // 2976-2983
	0x000ea1af, 0x000ea5af, 0x000ea9af, 0x000eadaf, 0x000eb1af, 0x000eb5af, 0x000eb9af, 0x000ebdaf, // 2984-2991
	0x000ec1af, 0x000ec5af, 0x000ec9af, 0x000ecdaf, 0x000ed1af, 0x000ed5af, 0x000ed9af, 0x000eddaf, // 2992-2999
	0x000ee1af, 0x000ee5af, 0x000ee9af, 0x000eedaf, 0x000ef1af, 0x000ef5af, 0x000ef9af, 0x000efdaf, // 3000-3007
	0x000f01af, 0x000f05af, 0x000f09af, 0x000f0daf, 0x000f11af, 0x000f15af, 0x000f19af, 0x000f1daf, // 3008-3015
	0x000f21af, 0x000f25af, 0x000f29af, 0x000f2daf, 0x000f31af, 0x000f35af, 0x000f39af, 0x000f3daf, // 3016-3023
	0x000f41af, 0x000f45af, 0x000f49af, 0x000f4daf, 0x000f51af, 0x000f55af, 0x000f59af, 0x000f5daf, // 3024-3031
	0x000f61af, 0x000f65af, 0x000f69af, 0x000f6daf, 0x000f71af, 0x000f75af, 0x000f79af, 0x000f7daf, // 3032-3039
	0x000f81af, 0x000f85af, 0x000f89af, 0x000f8daf, 0x000f91af, 0x000f95af, 0x000f99af, 0x000f9daf, // 3040-3047
	0x000fa1af, 0x000fa5af, 0x000fa9af, 0x000fadaf, 0x000fb1af, 0x000fb5af, 0x000fb9af, 0x000fbdaf, // 3048-3055
	0x000fc1af, 0x000fc5af, 0x000fc9af, 0x000fcdaf, 0x000fd1af, 0x000fd5af, 0x000fd9af, 0x000fddaf, // 3056-3063
	0x000fe1af, 0x000fe5af, 0x000fe9af, 0x000fedaf, 0x000ff1af, 0x000ff5af, 0x000ff9af, 0x000ffdaf, // 3064-3071
	0x000003af, 0x000007af, 0x00000baf, 0x00000faf, 0x000013af, 0x000017af, 0x00001baf, 0x00001faf, // 3072-3079
	0x000023af, 0x000027af, 0x00002baf, 0x00002faf, 0x000033af, 0x000037af, 0x00003baf, 0x00003faf, // 3080-3087
	0x000043af, 0x000047af, 0x00004baf, 0x00004faf, 0x000053af, 0x000057af, 0x00005baf, 0x00005faf, // 3088-3095
	0x000063af, 0x000067af, 0x00006baf, 0x00006faf, 0x000073af, 0x000077af, 0x00007baf, 0x00007faf, // 3096-3103
	0x000083af, 0x000087af, 0x00008baf, 0x00008faf, 0x000093af, 0x000097af, 0x00009baf, 0x00009faf, // 3104-3111
	0x0000a3af, 0x0000a7af, 0x0000abaf, 0x0000afaf, 0x0000b3af, 0x0000b7af, 0x0000bbaf, 0x0000bfaf, // 3112-3119
	0x0000c3af, 0x0000c7af, 0x0000cbaf, 0x0000cfaf, 0x0000d3af, 0x0000d7af, 0x0000dbaf, 0x0000dfaf, // 3120-3127
	0x0000e3af, 0x0000e7af, 0x0000ebaf, 0x0000efaf, 0x0000f3af, 0x0000f7af, 0x0000fbaf, 0x0000ffaf, // 3128-3135
	0x000103af, 0x000107af, 0x00010baf, 0x00010faf, 0x000113af, 0x000117af, 0x00011baf, 0x00011faf, // 3136-3143
	0x000123af, 0x000127af, 0x00012baf, 0x00012faf, 0x000133af, 0x000137af, 0x00013baf, 0x00013faf, // 3144-3151
	0x000143af, 0x000147af, 0x00014baf, 0x00014faf, 0x000153af, 0x000157af, 0x00015baf, 0x00015faf, // 3152-3159
	0x000163af, 0x000167af, 0x00016baf, 0x00016faf, 0x000173af, 0x000177af, 0x00017baf, 0x00017faf, // 3160-3167
	0x000183af, 0x000187af, 0x00018baf, 0x00018faf, 0x000193af, 0x000197af, 0x00019baf, 0x00019faf, // 3168-3175
	0x0001a3af, 0x0001a7af, 0x0001abaf, 0x0001afaf, 0x0001b3af, 0x0001b7af, 0x0001bbaf, 0x0001bfaf, // 3176-3183
	0x0001c3af, 0x0001c7af, 0x0001cbaf, 0x0001cfaf, 0x0001d3af, 0x0001d7af, 0x0001dbaf, 0x0001dfaf, // 3184-3191
	0x0001e3af, 0x0001e7af, 0x0001ebaf, 0x0001efaf, 0x0001f3af, 0x0001f7af, 0x0001fbaf, 0x0001ffaf, // 3192-3199
	0x000203af, 0x000207af, 0x00020baf, 0x00020faf, 0x000213af, 0x000217af, 0x00021baf, 0x00021faf, // 3200-3207
	0x000223af, 0x000227af, 0x00022baf, 0x00022faf, 0x000233af, 0x000237af, 0x00023baf, 0x00023faf, // 3208-3215
	0x000243af, 0x000247af, 0x00024baf, 0x00024faf, 0x000253af, 0x000257af, 0x00025baf, 0x00025faf, // 3216-3223
	0x000263af, 0x000267af, 0x00026baf, 0x00026faf, 0x000273af, 0x000277af, 0x00027baf, 0x00027faf, // 3224-3231
	0x000283af, 0x000287af, 0x00028baf, 0x00028faf, 0x000293af, 0x000297af, 0x00029baf, 0x00029faf, // 3232-3239
	0x0002a3af, 0x0002a7af, 0x0002abaf, 0x0002afaf, 0x0002b3af, 0x0002b7af, 0x0002bbaf, 0x0002bfaf, // 3240-3247
	0x0002c3af, 0x0002c7af, 0x0002cbaf, 0x0002cfaf, 0x0002d3af, 0x0002d7af, 0x0002dbaf, 0x0002dfaf, // 3248-3255
	0x0002e3af, 0x0002e7af, 0x0002ebaf, 0x0002efaf, 0x0002f3af, 0x0002f7af, 0x0002fbaf, 0x0002ffaf, // 3256-3263
	0x000303af, 0x000307af, 0x00030baf, 0x00030faf, 0x000313af, 0x000317af, 0x00031baf, 0x00031faf, // 3264-3271
	0x000323af, 0x000327af, 0x00032baf, 0x00032faf, 0x000333af, 0x000337af, 0x00033baf, 0x00033faf, // 3272-3279
	0x000343af, 0x000347af, 0x00034baf, 0x00034faf, 0x000353af, 0x000357af, 0x00035baf, 0x00035faf, // 3280-3287
	0x000363af, 0x000367af, 0x00036baf, 0x00036faf, 0x000373af, 0x000377af, 0x00037baf, 0x00037faf, // 3288-3295
	0x000383af, 0x000387af, 0x00038baf, 0x00038faf, 0x000393af, 0x000397af, 0x00039baf, 0x00039faf, // 3296-3303
	0x0003a3af, 0x0003a7af, 0x0003abaf, 0x0003afaf, 0x0003b3af, 0x0003b7af, 0x0003bbaf, 0x0003bfaf, // 3304-3311
	0x0003c3af, 0x0003c7af, 0x0003cbaf, 0x0003cfaf, 0x0003d3af, 0x0003d7af, 0x0003dbaf, 0x0003dfaf, // 3312-3319
	0x0003e3af, 0x0003e7af, 0x0003ebaf, 0x0003efaf, 0x0003f3af, 0x0003f7af, 0x0003fbaf, 0x0003ffaf, // 3320-3327
	0x000403af, 0x000407af, 0x00040baf, 0x00040faf, 0x000413af, 0x000417af, 0x00041baf, 0x00041faf, // 3328-3335
	0x000423af, 0x000427af, 0x00042baf, 0x00042faf, 0x000433af, 0x000437af, 0x00043baf, 0x00043faf, // 3336-3343
	0x000443af, 0x000447af, 0x00044baf, 0x00044faf, 0x000453af, 0x000457af, 0x00045baf, 0x00045faf, // 3344-3351
	0x000463af, 0x000467af, 0x00046baf, 0x00046faf, 0x000473af, 0x000477af, 0x00047baf, 0x00047faf, // 3352-3359
	0x000483af, 0x000487af, 0x00048baf, 0x00048faf, 0x000493af, 0x000497af, 0x00049baf, 0x00049faf, // 3360-3367
	0x0004a3af, 0x0004a7af, 0x0004abaf, 0x0004afaf, 0x0004b3af, 0x0004b7af, 0x0004bbaf, 0x0004bfaf, // 3368-3375
	0x0004c3af, 0x0004c7af, 0x0004cbaf, 0x0004cfaf, 0x0004d3af, 0x0004d7af, 0x0004dbaf, 0x0004dfaf, // 3376-3383
	0x0004e3af, 0x0004e7af, 0x0004ebaf, 0x0004efaf, 0x0004f3af, 0x0004f7af, 0x0004fbaf, 0x0004ffaf, // 3384-3391
	0x000503af, 0x000507af, 0x00050baf, 0x00050faf, 0x000513af, 0x000517af, 0x00051baf, 0x00051faf, // 3392-3399
	0x000523af, 0x000527af, 0x00052baf, 0x00052faf, 0x000533af, 0x000537af, 0x00053baf, 0x00053faf, // 3400-3407
	0x000543af, 0x000547af, 0x00054baf, 0x00054faf, 0x000553af, 0x000557af, 0x00055baf, 0x00055faf, // 3408-3415
	0x000563af, 0x000567af, 0x00056baf, 0x00056faf, 0x000573af, 0x000577af, 0x00057baf, 0x00057faf, // 3416-3423
	0x000583af, 0x000587af, 0x00058baf, 0x00058faf, 0x000593af, 0x000597af, 0x00059baf, 0x00059faf, // 3424-3431
	0x0005a3af, 0x0005a7af, 0x0005abaf, 0x0005afaf, 0x0005b3af, 0x0005b7af, 0x0005bbaf, 0x0005bfaf, // 3432-3439
	0x0005c3af, 0x0005c7af, 0x0005cbaf, 0x0005cfaf, 0x0005d3af, 0x0005d7af, 0x0005dbaf, 0x0005dfaf, // 3440-3447
	0x0005e3af, 0x0005e7af, 0x0005ebaf, 0x0005efaf, 0x0005f3af, 0x0005f7af, 0x0005fbaf, 0x0005ffaf, // 3448-3455
	0x000603af, 0x000607af, 0x00060baf, 0x00060faf, 0x000613af, 0x000617af, 0x00061baf, 0x00061faf, // 3456-3463
	0x000623af, 0x000627af, 0x00062baf, 0x00062faf, 0x000633af, 0x000637af, 0x00063baf, 0x00063faf, // 3464-3471
	0x000643af, 0x000647af, 0x00064baf, 0x00064faf, 0x000653af, 0x000657af, 0x00065baf, 0x00065faf, // 3472-3479
	0x000663af, 0x000667af, 0x00066baf, 0x00066faf, 0x000673af, 0x000677af, 0x00067baf, 0x00067faf, // 3480-3487
	0x000683af, 0x000687af, 0x00068baf, 0x00068faf, 0x000693af, 0x000697af, 0x00069baf, 0x00069faf, // 3488-3495
	0x0006a3af, 0x0006a7af, 0x0006abaf, 0x0006afaf, 0x0006b3af, 0x0006b7af, 0x0006bbaf, 0x0006bfaf, // 3496-3503
	0x0006c3af, 0x0006c7af, 0x0006cbaf, 0x0006cfaf, 0x0006d3af, 0x0006d7af, 0x0006dbaf, 0x0006dfaf, // 3504-3511
	0x0006e3af, 0x0006e7af, 0x0006ebaf, 0x0006efaf, 0x0006f3af, 0x0006f7af, 0x0006fbaf, 0x0006ffaf, // 3512-3519
	0x000703af, 0x000707af, 0x00070baf, 0x00070faf, 0x000713af, 0x000717af, 0x00071baf, 0x00071faf, // 3520-3527
	0x000723af, 0x000727af, 0x00072baf, 0x00072faf, 0x000733af, 0x000737af, 0x00073baf, 0x00073faf, // 3528-3535
	0x000743af, 0x000747af, 0x00074baf, 0x00074faf, 0x000753af, 0x000757af, 0x00075baf, 0x00075faf, // 3536-3543
	0x000763af, 0x000767af, 0x00076baf, 0x00076faf, 0x000773af, 0x000777af, 0x00077baf, 0x00077faf, // 3544-3551
	0x000783af, 0x000787af, 0x00078baf, 0x00078faf, 0x000793af, 0x000797af, 0x00079baf, 0x00079faf, // 3552-3559
	0x0007a3af, 0x0007a7af, 0x0007abaf, 0x0007afaf, 0x0007b3af, 0x0007b7af, 0x0007bbaf, 0x0007bfaf, // 3560-3567
	0x0007c3af, 0x0007c7af, 0x0007cbaf, 0x0007cfaf, 0x0007d3af, 0x0007d7af, 0x0007dbaf, 0x0007dfaf, // 3568-3575
	0x0007e3af, 0x0007e7af, 0x0007ebaf, 0x0007efaf, 0x0007f3af, 0x0007f7af, 0x0007fbaf, 0x0007ffaf, // 3576-3583
	0x000803af, 0x000807af, 0x00080baf, 0x00080faf, 0x000813af, 0x000817af, 0x00081baf, 0x00081faf, // 3584-3591
	0x000823af, 0x000827af, 0x00082baf, 0x00082faf, 0x000833af, 0x000837af, 0x00083baf, 0x00083faf, // 3592-3599
	0x000843af, 0x000847af, 0x00084baf, 0x00084faf, 0x000853af, 0x000857af, 0x00085baf, 0x00085faf, // 3600-3607
	0x000863af, 0x000867af, 0x00086baf, 0x00086faf, 0x000873af, 0x000877af, 0x00087baf, 0x00087faf, // 3608-3615
	0x000883af, 0x000887af, 0x00088baf, 0x00088faf, 0x000893af, 0x000897af, 0x00089baf, 0x00089faf, // 3616-3623
	0x0008a3af, 0x0008a7af, 0x0008abaf, 0x0008afaf, 0x0008b3af, 0x0008b7af, 0x0008bbaf, 0x0008bfaf, // 3624-3631
	0x0008c3af, 0x0008c7af, 0x0008cbaf, 0x0008cfaf, 0x0008d3af, 0x0008d7af, 0x0008dbaf, 0x0008dfaf, // 3632-3639
	0x0008e3af, 0x0008e7af, 0x0008ebaf, 0x0008efaf, 0x0008f3af, 0x0008f7af, 0x0008fbaf, 0x0008ffaf, // 3640-3647
	0x000903af, 0x000907af, 0x00090baf, 0x00090faf, 0x000913af, 0x000917af, 0x00091baf, 0x00091faf, // 3648-3655
	0x000923af, 0x000927af, 0x00092baf, 0x00092faf, 0x000933af, 0x000937af, 0x00093baf, 0x00093faf, // 3656-3663
	0x000943af, 0x000947af, 0x00094baf, 0x00094faf, 0x000953af, 0x000957af, 0x00095baf, 0x00095faf, // 3664-3671
	0x000963af, 0x000967af, 0x00096baf, 0x00096faf, 0x000973af, 0x000977af, 0x00097baf, 0x00097faf, // 3672-3679
	0x000983af, 0x000987af, 0x00098baf, 0x00098faf, 0x000993af, 0x000997af, 0x00099baf, 0x00099faf, // 3680-3687
	0x0009a3af, 0x0009a7af, 0x0009abaf, 0x0009afaf, 0x0009b3af, 0x0009b7af, 0x0009bbaf, 0x0009bfaf, // 3688-3695
	0x0009c3af, 0x0009c7af, 0x0009cbaf, 0x0009cfaf, 0x0009d3af, 0x0009d7af, 0x0009dbaf, 0x0009dfaf, // 3696-3703
	0x0009e3af, 0x0009e7af, 0x0009ebaf, 0x0009efaf, 0x0009f3af, 0x0009f7af, 0x0009fbaf, 0x0009ffaf, // 3704-3711
	0x000a03af, 0x000a07af, 0x000a0baf, 0x000a0faf, 0x000a13af, 0x000a17af, 0x000a1baf, 0x000a1faf, // 3712-3719
	0x000a23af, 0x000a27af, 0x000a2baf, 0x000a2faf, 0x000a33af, 0x000a37af, 0x000a3baf, 0x000a3faf, // 3720-3727
	0x000a43af, 0x000a47af, 0x000a4baf, 0x000a4faf, 0x000a53af, 0x000a57af, 0x000a5baf, 0x000a5faf, // 3728-3735
	0x000a63af, 0x000a67af, 0x000a6baf, 0x000a6faf, 0x000a73af, 0x000a77af, 0x000a7baf, 0x000a7faf, // 3736-3743
	0x000a83af, 0x000a87af, 0x000a8baf, 0x000a8faf, 0x000a93af, 0x000a97af, 0x000a9baf, 0x000a9faf, // 3744-3751
	0x000aa3af, 0x000aa7af, 0x000aabaf, 0x000aafaf, 0x000ab3af, 0x000ab7af, 0x000abbaf, 0x000abfaf, // 3752-3759
	0x000ac3af, 0x000ac7af, 0x000acbaf, 0x000acfaf, 0x000ad3af, 0x000ad7af, 0x000adbaf, 0x000adfaf, // 3760-3767
	0x000ae3af, 0x000ae7af, 0x000aebaf, 0x000aefaf, 0x000af3af, 0x000af7af, 0x000afbaf, 0x000affaf, // 3768-3775
	0x000b03af, 0x000b07af, 0x000b0baf, 0x000b0faf, 0x000b13af, 0x000b17af, 0x000b1baf, 0x000b1faf, // 3776-3783
	0x000b23af, 0x000b27af, 0x000b2baf, 0x000b2faf, 0x000b33af, 0x000b37af, 0x000b3baf, 0x000b3faf, // 3784-3791
	0x000b43af, 0x000b47af, 0x000b4baf, 0x000b4faf, 0x000b53af, 0x000b57af, 0x000b5baf, 0x000b5faf, // 3792-3799
	0x000b63af, 0x000b67af, 0x000b6baf, 0x000b6faf, 0x000b73af, 0x000b77af, 0x000b7baf, 0x000b7faf, // 3800-3807
	0x000b83af, 0x000b87af, 0x000b8baf, 0x000b8faf, 0x000b93af, 0x000b97af, 0x000b9baf, 0x000b9faf, // 3808-3815
	0x000ba3af, 0x000ba7af, 0x000babaf, 0x000bafaf, 0x000bb3af, 0x000bb7af, 0x000bbbaf, 0x000bbfaf, // 3816-3823
	0x000bc3af, 0x000bc7af, 0x000bcbaf, 0x000bcfaf, 0x000bd3af, 0x000bd7af, 0x000bdbaf, 0x000bdfaf, // 3824-3831
	0x000be3af, 0x000be7af, 0x000bebaf, 0x000befaf, 0x000bf3af, 0x000bf7af, 0x000bfbaf, 0x000bffaf, // 3832-3839
	0x000c03af, 0x000c07af, 0x000c0baf, 0x000c0faf, 0x000c13af, 0x000c17af, 0x000c1baf, 0x000c1faf, // 3840-3847
	0x000c23af, 0x000c27af, 0x000c2baf, 0x000c2faf, 0x000c33af, 0x000c37af, 0x000c3baf, 0x000c3faf, // 3848-3855
	0x000c43af, 0x000c47af, 0x000c4baf, 0x000c4faf, 0x000c53af, 0x000c57af, 0x000c5baf, 0x000c5faf, // 3856-3863
	0x000c63af, 0x000c67af, 0x000c6baf, 0x000c6faf, 0x000c73af, 0x000c77af, 0x000c7baf, 0x000c7faf, // 3864-3871
	0x000c83af, 0x000c87af, 0x000c8baf, 0x000c8faf, 0x000c93af, 0x000c97af, 0x000c9baf, 0x000c9faf, // 3872-3879
	0x000ca3af, 0x000ca7af, 0x000cabaf, 0x000cafaf, 0x000cb3af, 0x000cb7af, 0x000cbbaf, 0x000cbfaf, // 3880-3887
	0x000cc3af, 0x000cc7af, 0x000ccbaf, 0x000ccfaf, 0x000cd3af, 0x000cd7af, 0x000cdbaf, 0x000cdfaf, // 3888-3895
	0x000ce3af, 0x000ce7af, 0x000cebaf, 0x000cefaf, 0x000cf3af, 0x000cf7af, 0x000cfbaf, 0x000cffaf, // 3896-3903
	0x000d03af, 0x000d07af, 0x000d0baf, 0x000d0faf, 0x000d13af, 0x000d17af, 0x000d1baf, 0x000d1faf, // 3904-3911
	0x000d23af, 0x000d27af, 0x000d2baf, 0x000d2faf, 0x000d33af, 0x000d37af, 0x000d3baf, 0x000d3faf, // 3912-3919
	0x000d43af, 0x000d47af, 0x000d4baf, 0x000d4faf, 0x000d53af, 0x000d57af, 0x000d5baf, 0x000d5faf, // 3920-3927
	0x000d63af, 0x000d67af, 0x000d6baf, 0x000d6faf, 0x000d73af, 0x000d77af, 0x000d7baf, 0x000d7faf, // 3928-3935
	0x000d83af, 0x000d87af, 0x000d8baf, 0x000d8faf, 0x000d93af, 0x000d97af, 0x000d9baf, 0x000d9faf, // 3936-3943
	0x000da3af, 0x000da7af, 0x000dabaf, 0x000dafaf, 0x000db3af, 0x000db7af, 0x000dbbaf, 0x000dbfaf, // 3944-3951
	0x000dc3af, 0x000dc7af, 0x000dcbaf, 0x000dcfaf, 0x000dd3af, 0x000dd7af, 0x000ddbaf, 0x000ddfaf, // 3952-3959
	0x000de3af, 0x000de7af, 0x000debaf, 0x000defaf, 0x000df3af, 0x000df7af, 0x000dfbaf, 0x000dffaf, // 3960-3967
	0x000e03af, 0x000e07af, 0x000e0baf, 0x000e0faf, 0x000e13af, 0x000e17af, 0x000e1baf, 0x000e1faf, // 3968-3975
	0x000e23af, 0x000e27af, 0x000e2baf, 0x000e2faf, 0x000e33af, 0x000e37af, 0x000e3baf, 0x000e3faf, // 3976-3983
	0x000e43af, 0x000e47af, 0x000e4baf, 0x000e4faf, 0x000e53af, 0x000e57af, 0x000e5baf, 0x000e5faf, // 3984-3991
	0x000e63af, 0x000e67af, 0x000e6baf, 0x000e6faf, 0x000e73af, 0x000e77af, 0x000e7baf, 0x000e7faf, // 3992-3999
	0x000e83af, 0x000e87af, 0x000e8baf, 0x000e8faf, 0x000e93af, 0x000e97af, 0x000e9baf, 0x000e9faf, // 4000-4007
	0x000ea3af, 0x000ea7af, 0x000eabaf, 0x000eafaf, 0x000eb3af, 0x000eb7af, 0x000ebbaf, 0x000ebfaf, // 4008-4015
	0x000ec3af, 0x000ec7af, 0x000ecbaf, 0x000ecfaf, 0x000ed3af, 0x000ed7af, 0x000edbaf, 0x000edfaf, // 4016-4023
	0x000ee3af, 0x000ee7af, 0x000eebaf, 0x000eefaf, 0x000ef3af, 0x000ef7af, 0x000efbaf, 0x000effaf, // 4024-4031
	0x000f03af, 0x000f07af, 0x000f0baf, 0x000f0faf, 0x000f13af, 0x000f17af, 0x000f1baf, 0x000f1faf, // 4032-4039
	0x000f23af, 0x000f27af, 0x000f2baf, 0x000f2faf, 0x000f33af, 0x000f37af, 0x000f3baf, 0x000f3faf, // 4040-4047
	0x000f43af, 0x000f47af, 0x000f4baf, 0x000f4faf, 0x000f53af, 0x000f57af, 0x000f5baf, 0x000f5faf, // 4048-4055
	0x000f63af, 0x000f67af, 0x000f6baf, 0x000f6faf, 0x000f73af, 0x000f77af, 0x000f7baf, 0x000f7faf, // 4056-4063
	0x000f83af, 0x000f87af, 0x000f8baf, 0x000f8faf, 0x000f93af, 0x000f97af, 0x000f9baf, 0x000f9faf, // 4064-4071
	0x000fa3af, 0x000fa7af, 0x000fabaf, 0x000fafaf, 0x000fb3af, 0x000fb7af, 0x000fbbaf, 0x000fbfaf, // 4072-4079
	0x000fc3af, 0x000fc7af, 0x000fcbaf, 0x000fcfaf, 0x000fd3af, 0x000fd7af, 0x000fdbaf, 0x000fdfaf, // 4080-4087
	0x000fe3af, 0x000fe7af, 0x000febaf, 0x000fefaf, 0x000ff3af, 0x000ff7af, 0x000ffbaf, 0x000fffaf, // 4088-4095
	0x00000070, 0x00000470, 0x00000870, 0x00000c70, 0x00001070, 0x00001470, 0x00001870, 0x00001c70, // 4096-4103
	0x00002070, 0x00002470, 0x00002870, 0x00002c70, 0x00003070, 0x00003470, 0x00003870, 0x00003c70, // 4104-4111
	0x00004070, 0x00004470, 0x00004870, 0x00004c70, 0x00005070, 0x00005470, 0x00005870, 0x00005c70, // 4112-4119
	0x00006070, 0x00006470, 0x00006870, 0x00006c70, 0x00007070, 0x00007470, 0x00007870, 0x00007c70, // 4120-4127
	0x00008070, 0x00008470, 0x00008870, 0x00008c70, 0x00009070, 0x00009470, 0x00009870, 0x00009c70, // 4128-4135
	0x0000a070, 0x0000a470, 0x0000a870, 0x0000ac70, 0x0000b070, 0x0000b470, 0x0000b870, 0x0000bc70, // 4136-4143
	0x0000c070, 0x0000c470, 0x0000c870, 0x0000cc70, 0x0000d070, 0x0000d470, 0x0000d870, 0x0000dc70, // 4144-4151
	0x0000e070, 0x0000e470, 0x0000e870, 0x0000ec70, 0x0000f070, 0x0000f470, 0x0000f870, 0x0000fc70, // 4152-4159
	0x00010070, 0x00010470, 0x00010870, 0x00010c70, 0x00011070, 0x00011470, 0x00011870, 0x00011c70, // 4160-4167
	0x00012070, 0x00012470, 0x00012870, 0x00012c70, 0x00013070, 0x00013470, 0x00013870, 0x00013c70, // 4168-4175
	0x00014070, 0x00014470, 0x00014870, 0x00014c70, 0x00015070, 0x00015470, 0x00015870, 0x00015c70, // 4176-4183
	0x00016070, 0x00016470, 0x00016870, 0x00016c70, 0x00017070, 0x00017470, 0x00017870, 0x00017c70, // 4184-4191
	0x00018070, 0x00018470, 0x00018870, 0x00018c70, 0x00019070, 0x00019470, 0x00019870, 0x00019c70, // 4192-4199
	0x0001a070, 0x0001a470, 0x0001a870, 0x0001ac70, 0x0001b070, 0x0001b470, 0x0001b870, 0x0001bc70, // 4200-4207
	0x0001c070, 0x0001c470, 0x0001c870, 0x0001cc70, 0x0001d070, 0x0001d470, 0x0001d870, 0x0001dc70, // 4208-4215
	0x0001e070, 0x0001e470, 0x0001e870, 0x0001ec70, 0x0001f070, 0x0001f470, 0x0001f870, 0x0001fc70, // 4216-4223
	0x00020070, 0x00020470, 0x00020870, 0x00020c70, 0x00021070, 0x00021470, 0x00021870, 0x00021c70, // 4224-4231
	0x00022070, 0x00022470, 0x00022870, 0x00022c70, 0x00023070, 0x00023470, 0x00023870, 0x00023c70, // 4232-4239
	0x00024070, 0x00024470, 0x00024870, 0x00024c70, 0x00025070, 0x00025470, 0x00025870, 0x00025c70, // 4240-4247
	0x00026070, 0x00026470, 0x00026870, 0x00026c70, 0x00027070, 0x00027470, 0x00027870, 0x00027c70, // 4248-4255
	0x00028070, 0x00028470, 0x00028870, 0x00028c70, 0x00029070, 0x00029470, 0x00029870, 0x00029c70, // 4256-4263
	0x0002a070, 0x0002a470, 0x0002a870, 0x0002ac70, 0x0002b070, 0x0002b470, 0x0002b870, 0x0002bc70, // 4264-4271
	0x0002c070, 0x0002c470, 0x0002c870, 0x0002cc70, 0x0002d070, 0x0002d470, 0x0002d870, 0x0002dc70, // 4272-4279
	0x0002e070, 0x0002e470, 0x0002e870, 0x0002ec70, 0x0002f070, 0x0002f470, 0x0002f870, 0x0002fc70, // 4280-4287
	0x00030070, 0x00030470, 0x00030870, 0x00030c70, 0x00031070, 0x00031470, 0x00031870, 0x00031c70, // 4288-4295
	0x00032070, 0x00032470, 0x00032870, 0x00032c70, 0x00033070, 0x00033470, 0x00033870, 0x00033c70, // 4296-4303
	0x00034070, 0x00034470, 0x00034870, 0x00034c70, 0x00035070, 0x00035470, 0x00035870, 0x00035c70, // 4304-4311
	0x00036070, 0x00036470, 0x00036870, 0x00036c70, 0x00037070, 0x00037470, 0x00037870, 0x00037c70, // 4312-4319
	0x00038070, 0x00038470, 0x00038870, 0x00038c70, 0x00039070, 0x00039470, 0x00039870, 0x00039c70, // 4320-4327
	0x0003a070, 0x0003a470, 0x0003a870, 0x0003ac70, 0x0003b070, 0x0003b470, 0x0003b870, 0x0003bc70, // 4328-4335
	0x0003c070, 0x0003c470, 0x0003c870, 0x0003cc70, 0x0003d070, 0x0003d470, 0x0003d870, 0x0003dc70, // 4336-4343
	0x0003e070, 0x0003e470, 0x0003e870, 0x0003ec70, 0x0003f070, 0x0003f470, 0x0003f870, 0x0003fc70, // 4344-4351
	0x00040070, 0x00040470, 0x00040870, 0x00040c70, 0x00041070, 0x00041470, 0x00041870, 0x00041c70, // 4352-4359
	0x00042070, 0x00042470, 0x00042870, 0x00042c70, 0x00043070, 0x00043470, 0x00043870, 0x00043c70, // 4360-4367
	0x00044070, 0x00044470, 0x00044870, 0x00044c70, 0x00045070, 0x00045470, 0x00045870, 0x00045c70, // 4368-4375
	0x00046070, 0x00046470, 0x00046870, 0x00046c70, 0x00047070, 0x00047470, 0x00047870, 0x00047c70, // 4376-4383
	0x00048070, 0x00048470, 0x00048870, 0x00048c70, 0x00049070, 0x00049470, 0x00049870, 0x00049c70, // 4384-4391
	0x0004a070, 0x0004a470, 0x0004a870, 0x0004ac70, 0x0004b070, 0x0004b470, 0x0004b870, 0x0004bc70, // 4392-4399
	0x0004c070, 0x0004c470, 0x0004c870, 0x0004cc70, 0x0004d070, 0x0004d470, 0x0004d870, 0x0004dc70, // 4400-4407
	0x0004e070, 0x0004e470, 0x0004e870, 0x0004ec70, 0x0004f070, 0x0004f470, 0x0004f870, 0x0004fc70, // 4408-4415
	0x00050070, 0x00050470, 0x00050870, 0x00050c70, 0x00051070, 0x00051470, 0x00051870, 0x00051c70, // 4416-4423
	0x00052070, 0x00052470, 0x00052870, 0x00052c70, 0x00053070, 0x00053470, 0x00053870, 0x00053c70, // 4424-4431
	0x00054070, 0x00054470, 0x00054870, 0x00054c70, 0x00055070, 0x00055470, 0x00055870, 0x00055c70, // 4432-4439
	0x00056070, 0x00056470, 0x00056870, 0x00056c70, 0x00057070, 0x00057470, 0x00057870, 0x00057c70, // 4440-4447
	0x00058070, 0x00058470, 0x00058870, 0x00058c70, 0x00059070, 0x00059470, 0x00059870, 0x00059c70, // 4448-4455
	0x0005a070, 0x0005a470, 0x0005a870, 0x0005ac70, 0x0005b070, 0x0005b470, 0x0005b870, 0x0005bc70, // 4456-4463
	0x0005c070, 0x0005c470, 0x0005c870, 0x0005cc70, 0x0005d070, 0x0005d470, 0x0005d870, 0x0005dc70, // 4464-4471
	0x0005e070, 0x0005e470, 0x0005e870, 0x0005ec70, 0x0005f070, 0x0005f470, 0x0005f870, 0x0005fc70, // 4472-4479
	0x00060070, 0x00060470, 0x00060870, 0x00060c70, 0x00061070, 0x00061470, 0x00061870, 0x00061c70, // 4480-4487
	0x00062070, 0x00062470, 0x00062870, 0x00062c70, 0x00063070, 0x00063470, 0x00063870, 0x00063c70, // 4488-4495
	0x00064070, 0x00064470, 0x00064870, 0x00064c70, 0x00065070, 0x00065470, 0x00065870, 0x00065c70, // 4496-4503
	0x00066070, 0x00066470, 0x00066870, 0x00066c70, 0x00067070, 0x00067470, 0x00067870, 0x00067c70, // 4504-4511
	0x00068070, 0x00068470, 0x00068870, 0x00068c70, 0x00069070, 0x00069470, 0x00069870, 0x00069c70, // 4512-4519
	0x0006a070, 0x0006a470, 0x0006a870, 0x0006ac70, 0x0006b070, 0x0006b470, 0x0006b870, 0x0006bc70, // 4520-4527
	0x0006c070, 0x0006c470, 0x0006c870, 0x0006cc70, 0x0006d070, 0x0006d470, 0x0006d870, 0x0006dc70, // 4528-4535
	0x0006e070, 0x0006e470, 0x0006e870, 0x0006ec70, 0x0006f070, 0x0006f470, 0x0006f870, 0x0006fc70, // 4536-4543
	0x00070070, 0x00070470, 0x00070870, 0x00070c70, 0x00071070, 0x00071470, 0x00071870, 0x00071c70, // 4544-4551
	0x00072070, 0x00072470, 0x00072870, 0x00072c70, 0x00073070, 0x00073470, 0x00073870, 0x00073c70, // 4552-4559
	0x00074070, 0x00074470, 0x00074870, 0x00074c70, 0x00075070, 0x00075470, 0x00075870, 0x00075c70, // 4560-4567
	0x00076070, 0x00076470, 0x00076870, 0x00076c70, 0x00077070, 0x00077470, 0x00077870, 0x00077c70, // 4568-4575
	0x00078070, 0x00078470, 0x00078870, 0x00078c70, 0x00079070, 0x00079470, 0x00079870, 0x00079c70, // 4576-4583
	0x0007a070, 0x0007a470, 0x0007a870, 0x0007ac70, 0x0007b070, 0x0007b470, 0x0007b870, 0x0007bc70, // 4584-4591
	0x0007c070, 0x0007c470, 0x0007c870, 0x0007cc70, 0x0007d070, 0x0007d470, 0x0007d870, 0x0007dc70, // 4592-4599
	0x0007e070, 0x0007e470, 0x0007e870, 0x0007ec70, 0x0007f070, 0x0007f470, 0x0007f870, 0x0007fc70, // 4600-4607
	0x00080070, 0x00080470, 0x00080870, 0x00080c70, 0x00081070, 0x00081470, 0x00081870, 0x00081c70, // 4608-4615
	0x00082070, 0x00082470, 0x00082870, 0x00082c70, 0x00083070, 0x00083470, 0x00083870, 0x00083c70, // 4616-4623
	0x00084070, 0x00084470, 0x00084870, 0x00084c70, 0x00085070, 0x00085470, 0x00085870, 0x00085c70, // 4624-4631
	0x00086070, 0x00086470, 0x00086870, 0x00086c70, 0x00087070, 0x00087470, 0x00087870, 0x00087c70, // 4632-4639
	0x00088070, 0x00088470, 0x00088870, 0x00088c70, 0x00089070, 0x00089470, 0x00089870, 0x00089c70, // 4640-4647
	0x0008a070, 0x0008a470, 0x0008a870, 0x0008ac70, 0x0008b070, 0x0008b470, 0x0008b870, 0x0008bc70, // 4648-4655
	0x0008c070, 0x0008c470, 0x0008c870, 0x0008cc70, 0x0008d070, 0x0008d470, 0x0008d870, 0x0008dc70, // 4656-4663
	0x0008e070, 0x0008e470, 0x0008e870, 0x0008ec70, 0x0008f070, 0x0008f470, 0x0008f870, 0x0008fc70, // 4664-4671
	0x00090070, 0x00090470, 0x00090870, 0x00090c70, 0x00091070, 0x00091470, 0x00091870, 0x00091c70, // 4672-4679
	0x00092070, 0x00092470, 0x00092870, 0x00092c70, 0x00093070, 0x00093470, 0x00093870, 0x00093c70, // 4680-4687
	0x00094070, 0x00094470, 0x00094870, 0x00094c70, 0x00095070, 0x00095470, 0x00095870, 0x00095c70, // 4688-4695
	0x00096070, 0x00096470, 0x00096870, 0x00096c70, 0x00097070, 0x00097470, 0x00097870, 0x00097c70, // 4696-4703
	0x00098070, 0x00098470, 0x00098870, 0x00098c70, 0x00099070, 0x00099470, 0x00099870, 0x00099c70, // 4704-4711
	0x0009a070, 0x0009a470, 0x0009a870, 0x0009ac70, 0x0009b070, 0x0009b470, 0x0009b870, 0x0009bc70, // 4712-4719
	0x0009c070, 0x0009c470, 0x0009c870, 0x0009cc70, 0x0009d070, 0x0009d470, 0x0009d870, 0x0009dc70, // 4720-4727
	0x0009e070, 0x0009e470, 0x0009e870, 0x0009ec70, 0x0009f070, 0x0009f470, 0x0009f870, 0x0009fc70, // 4728-4735
	0x000a0070, 0x000a0470, 0x000a0870, 0x000a0c70, 0x000a1070, 0x000a1470, 0x000a1870, 0x000a1c70, // 4736-4743
	0x000a2070, 0x000a2470, 0x000a2870, 0x000a2c70, 0x000a3070, 0x000a3470, 0x000a3870, 0x000a3c70, // 4744-4751
	0x000a4070, 0x000a4470, 0x000a4870, 0x000a4c70, 0x000a5070, 0x000a5470, 0x000a5870, 0x000a5c70, // 4752-4759
	0x000a6070, 0x000a6470, 0x000a6870, 0x000a6c70, 0x000a7070, 0x000a7470, 0x000a7870, 0x000a7c70, // 4760-4767
	0x000a8070, 0x000a8470, 0x000a8870, 0x000a8c70, 0x000a9070, 0x000a9470, 0x000a9870, 0x000a9c70, // 4768-4775
	0x000aa070, 0x000aa470, 0x000aa870, 0x000aac70, 0x000ab070, 0x000ab470, 0x000ab870, 0x000abc70, // 4776-4783
	0x000ac070, 0x000ac470, 0x000ac870, 0x000acc70, 0x000ad070, 0x000ad470, 0x000ad870, 0x000adc70, // 4784-4791
	0x000ae070, 0x000ae470, 0x000ae870, 0x000aec70, 0x000af070, 0x000af470, 0x000af870, 0x000afc70, // 4792-4799
	0x000b0070, 0x000b0470, 0x000b0870, 0x000b0c70, 0x000b1070, 0x000b1470, 0x000b1870, 0x000b1c70, // 4800-4807
	0x000b2070, 0x000b2470, 0x000b2870, 0x000b2c70, 0x000b3070, 0x000b3470, 0x000b3870, 0x000b3c70, // 4808-4815
	0x000b4070, 0x000b4470, 0x000b4870, 0x000b4c70, 0x000b5070, 0x000b5470, 0x000b5870, 0x000b5c70, // 4816-4823
	0x000b6070, 0x000b6470, 0x000b6870, 0x000b6c70, 0x000b7070, 0x000b7470, 0x000b7870, 0x000b7c70, // 4824-4831
	0x000b8070, 0x000b8470, 0x000b8870, 0x000b8c70, 0x000b9070, 0x000b9470, 0x000b9870, 0x000b9c70, // 4832-4839
	0x000ba070, 0x000ba470, 0x000ba870, 0x000bac70, 0x000bb070, 0x000bb470, 0x000bb870, 0x000bbc70, // 4840-4847
	0x000bc070, 0x000bc470, 0x000bc870, 0x000bcc70, 0x000bd070, 0x000bd470, 0x000bd870, 0x000bdc70, // 4848-4855
	0x000be070, 0x000be470, 0x000be870, 0x000bec70, 0x000bf070, 0x000bf470, 0x000bf870, 0x000bfc70, // 4856-4863
	0x000c0070, 0x000c0470, 0x000c0870, 0x000c0c70, 0x000c1070, 0x000c1470, 0x000c1870, 0x000c1c70, // 4864-4871
	0x000c2070, 0x000c2470, 0x000c2870, 0x000c2c70, 0x000c3070, 0x000c3470, 0x000c3870, 0x000c3c70, // 4872-4879
	0x000c4070, 0x000c4470, 0x000c4870, 0x000c4c70, 0x000c5070, 0x000c5470, 0x000c5870, 0x000c5c70, // 4880-4887
	0x000c6070, 0x000c6470, 0x000c6870, 0x000c6c70, 0x000c7070, 0x000c7470, 0x000c7870, 0x000c7c70, // 4888-4895
	0x000c8070, 0x000c8470, 0x000c8870, 0x000c8c70, 0x000c9070, 0x000c9470, 0x000c9870, 0x000c9c70, // 4896-4903
	0x000ca070, 0x000ca470, 0x000ca870, 0x000cac70, 0x000cb070, 0x000cb470, 0x000cb870, 0x000cbc70, // 4904-4911
	0x000cc070, 0x000cc470, 0x000cc870, 0x000ccc70, 0x000cd070, 0x000cd470, 0x000cd870, 0x000cdc70, // 4912-4919
	0x000ce070, 0x000ce470, 0x000ce870, 0x000cec70, 0x000cf070, 0x000cf470, 0x000cf870, 0x000cfc70, // 4920-4927
	0x000d0070, 0x000d0470, 0x000d0870, 0x000d0c70, 0x000d1070, 0x000d1470, 0x000d1870, 0x000d1c70, // 4928-4935
	0x000d2070, 0x000d2470, 0x000d2870, 0x000d2c70, 0x000d3070, 0x000d3470, 0x000d3870, 0x000d3c70, // 4936-4943
	0x000d4070, 0x000d4470, 0x000d4870, 0x000d4c70, 0x000d5070, 0x000d5470, 0x000d5870, 0x000d5c70, // 4944-4951
	0x000d6070, 0x000d6470, 0x000d6870, 0x000d6c70, 0x000d7070, 0x000d7470, 0x000d7870, 0x000d7c70, // 4952-4959
	0x000d8070, 0x000d8470, 0x000d8870, 0x000d8c70, 0x000d9070, 0x000d9470, 0x000d9870, 0x000d9c70, // 4960-4967
	0x000da070, 0x000da470, 0x000da870, 0x000dac70, 0x000db070, 0x000db470, 0x000db870, 0x000dbc70, // 4968-4975
	0x000dc070, 0x000dc470, 0x000dc870, 0x000dcc70, 0x000dd070, 0x000dd470, 0x000dd870, 0x000ddc70, // 4976-4983
	0x000de070, 0x000de470, 0x000de870, 0x000dec70, 0x000df070, 0x000df470, 0x000df870, 0x000dfc70, // 4984-4991
	0x000e0070, 0x000e0470, 0x000e0870, 0x000e0c70, 0x000e1070, 0x000e1470, 0x000e1870, 0x000e1c70, // 4992-4999
	0x000e2070, 0x000e2470, 0x000e2870, 0x000e2c70, 0x000e3070, 0x000e3470, 0x000e3870, 0x000e3c70, // 5000-5007
	0x000e4070, 0x000e4470, 0x000e4870, 0x000e4c70, 0x000e5070, 0x000e5470, 0x000e5870, 0x000e5c70, // 5008-5015
	0x000e6070, 0x000e6470, 0x000e6870, 0x000e6c70, 0x000e7070, 0x000e7470, 0x000e7870, 0x000e7c70, // 5016-5023
	0x000e8070, 0x000e8470, 0x000e8870, 0x000e8c70, 0x000e9070, 0x000e9470, 0x000e9870, 0x000e9c70, // 5024-5031
	0x000ea070, 0x000ea470, 0x000ea870, 0x000eac70, 0x000eb070, 0x000eb470, 0x000eb870, 0x000ebc70, // 5032-5039
	0x000ec070, 0x000ec470, 0x000ec870, 0x000ecc70, 0x000ed070, 0x000ed470, 0x000ed870, 0x000edc70, // 5040-5047
	0x000ee070, 0x000ee470, 0x000ee870, 0x000eec70, 0x000ef070, 0x000ef470, 0x000ef870, 0x000efc70, // 5048-5055
	0x000f0070, 0x000f0470, 0x000f0870, 0x000f0c70, 0x000f1070, 0x000f1470, 0x000f1870, 0x000f1c70, // 5056-5063
	0x000f2070, 0x000f2470, 0x000f2870, 0x000f2c70, 0x000f3070, 0x000f3470, 0x000f3870, 0x000f3c70, // 5064-5071
	0x000f4070, 0x000f4470, 0x000f4870, 0x000f4c70, 0x000f5070, 0x000f5470, 0x000f5870, 0x000f5c70, // 5072-5079
	0x000f6070, 0x000f6470, 0x000f6870, 0x000f6c70, 0x000f7070, 0x000f7470, 0x000f7870, 0x000f7c70, // 5080-5087
	0x000f8070, 0x000f8470, 0x000f8870, 0x000f8c70, 0x000f9070, 0x000f9470, 0x000f9870, 0x000f9c70, // 5088-5095
	0x000fa070, 0x000fa470, 0x000fa870, 0x000fac70, 0x000fb070, 0x000fb470, 0x000fb870, 0x000fbc70, // 5096-5103
	0x000fc070, 0x000fc470, 0x000fc870, 0x000fcc70, 0x000fd070, 0x000fd470, 0x000fd870, 0x000fdc70, // 5104-5111
	0x000fe070, 0x000fe470, 0x000fe870, 0x000fec70, 0x000ff070, 0x000ff470, 0x000ff870, 0x000ffc70, // 5112-5119
	0x00100070, 0x00100470, 0x00100870, 0x00100c70, 0x00101070, 0x00101470, 0x00101870, 0x00101c70, // 5120-5127
	0x00102070, 0x00102470, 0x00102870, 0x00102c70, 0x00103070, 0x00103470, 0x00103870, 0x00103c70, // 5128-5135
	0x00104070, 0x00104470, 0x00104870, 0x00104c70, 0x00105070, 0x00105470, 0x00105870, 0x00105c70, // 5136-5143
	0x00106070, 0x00106470, 0x00106870, 0x00106c70, 0x00107070, 0x00107470, 0x00107870, 0x00107c70, // 5144-5151
	0x00108070, 0x00108470, 0x00108870, 0x00108c70, 0x00109070, 0x00109470, 0x00109870, 0x00109c70, // 5152-5159
	0x0010a070, 0x0010a470, 0x0010a870, 0x0010ac70, 0x0010b070, 0x0010b470, 0x0010b870, 0x0010bc70, // 5160-5167
	0x0010c070, 0x0010c470, 0x0010c870, 0x0010cc70, 0x0010d070, 0x0010d470, 0x0010d870, 0x0010dc70, // 5168-5175
	0x0010e070, 0x0010e470, 0x0010e870, 0x0010ec70, 0x0010f070, 0x0010f470, 0x0010f870, 0x0010fc70, // 5176-5183
	0x00110070, 0x00110470, 0x00110870, 0x00110c70, 0x00111070, 0x00111470, 0x00111870, 0x00111c70, // 5184-5191
	0x00112070, 0x00112470, 0x00112870, 0x00112c70, 0x00113070, 0x00113470, 0x00113870, 0x00113c70, // 5192-5199
	0x00114070, 0x00114470, 0x00114870, 0x00114c70, 0x00115070, 0x00115470, 0x00115870, 0x00115c70, // 5200-5207
	0x00116070, 0x00116470, 0x00116870, 0x00116c70, 0x00117070, 0x00117470, 0x00117870, 0x00117c70, // 5208-5215
	0x00118070, 0x00118470, 0x00118870, 0x00118c70, 0x00119070, 0x00119470, 0x00119870, 0x00119c70, // 5216-5223
	0x0011a070, 0x0011a470, 0x0011a870, 0x0011ac70, 0x0011b070, 0x0011b470, 0x0011b870, 0x0011bc70, // 5224-5231
	0x0011c070, 0x0011c470, 0x0011c870, 0x0011cc70, 0x0011d070, 0x0011d470, 0x0011d870, 0x0011dc70, // 5232-5239
	0x0011e070, 0x0011e470, 0x0011e870, 0x0011ec70, 0x0011f070, 0x0011f470, 0x0011f870, 0x0011fc70, // 5240-5247
	0x00120070, 0x00120470, 0x00120870, 0x00120c70, 0x00121070, 0x00121470, 0x00121870, 0x00121c70, // 5248-5255
	0x00122070, 0x00122470, 0x00122870, 0x00122c70, 0x00123070, 0x00123470, 0x00123870, 0x00123c70, // 5256-5263
	0x00124070, 0x00124470, 0x00124870, 0x00124c70, 0x00125070, 0x00125470, 0x00125870, 0x00125c70, // 5264-5271
	0x00126070, 0x00126470, 0x00126870, 0x00126c70, 0x00127070, 0x00127470, 0x00127870, 0x00127c70, // 5272-5279
	0x00128070, 0x00128470, 0x00128870, 0x00128c70, 0x00129070, 0x00129470, 0x00129870, 0x00129c70, // 5280-5287
	0x0012a070, 0x0012a470, 0x0012a870, 0x0012ac70, 0x0012b070, 0x0012b470, 0x0012b870, 0x0012bc70, // 5288-5295
	0x0012c070, 0x0012c470, 0x0012c870, 0x0012cc70, 0x0012d070, 0x0012d470, 0x0012d870, 0x0012dc70, // 5296-5303
	0x0012e070, 0x0012e470, 0x0012e870, 0x0012ec70, 0x0012f070, 0x0012f470, 0x0012f870, 0x0012fc70, // 5304-5311
	0x00130070, 0x00130470, 0x00130870, 0x00130c70, 0x00131070, 0x00131470, 0x00131870, 0x00131c70, // 5312-5319
	0x00132070, 0x00132470, 0x00132870, 0x00132c70, 0x00133070, 0x00133470, 0x00133870, 0x00133c70, // 5320-5327
	0x00134070, 0x00134470, 0x00134870, 0x00134c70, 0x00135070, 0x00135470, 0x00135870, 0x00135c70, // 5328-5335
	0x00136070, 0x00136470, 0x00136870, 0x00136c70, 0x00137070, 0x00137470, 0x00137870, 0x00137c70, // 5336-5343
	0x00138070, 0x00138470, 0x00138870, 0x00138c70, 0x00139070, 0x00139470, 0x00139870, 0x00139c70, // 5344-5351
	0x0013a070, 0x0013a470, 0x0013a870, 0x0013ac70, 0x0013b070, 0x0013b470, 0x0013b870, 0x0013bc70, // 5352-5359
	0x0013c070, 0x0013c470, 0x0013c870, 0x0013cc70, 0x0013d070, 0x0013d470, 0x0013d870, 0x0013dc70, // 5360-5367
	0x0013e070, 0x0013e470, 0x0013e870, 0x0013ec70, 0x0013f070, 0x0013f470, 0x0013f870, 0x0013fc70, // 5368-5375
	0x00140070, 0x00140470, 0x00140870, 0x00140c70, 0x00141070, 0x00141470, 0x00141870, 0x00141c70, // 5376-5383
	0x00142070, 0x00142470, 0x00142870, 0x00142c70, 0x00143070, 0x00143470, 0x00143870, 0x00143c70, // 5384-5391
	0x00144070, 0x00144470, 0x00144870, 0x00144c70, 0x00145070, 0x00145470, 0x00145870, 0x00145c70, // 5392-5399
	0x00146070, 0x00146470, 0x00146870, 0x00146c70, 0x00147070, 0x00147470, 0x00147870, 0x00147c70, // 5400-5407
	0x00148070, 0x00148470, 0x00148870, 0x00148c70, 0x00149070, 0x00149470, 0x00149870, 0x00149c70, // 5408-5415
	0x0014a070, 0x0014a470, 0x0014a870, 0x0014ac70, 0x0014b070, 0x0014b470, 0x0014b870, 0x0014bc70, // 5416-5423
	0x0014c070, 0x0014c470, 0x0014c870, 0x0014cc70, 0x0014d070, 0x0014d470, 0x0014d870, 0x0014dc70, // 5424-5431
	0x0014e070, 0x0014e470, 0x0014e870, 0x0014ec70, 0x0014f070, 0x0014f470, 0x0014f870, 0x0014fc70, // 5432-5439
	0x00150070, 0x00150470, 0x00150870, 0x00150c70, 0x00151070, 0x00151470, 0x00151870, 0x00151c70, // 5440-5447
	0x00152070, 0x00152470, 0x00152870, 0x00152c70, 0x00153070, 0x00153470, 0x00153870, 0x00153c70, // 5448-5455
	0x00154070, 0x00154470, 0x00154870, 0x00154c70, 0x00155070, 0x00155470, 0x00155870, 0x00155c70, // 5456-5463
	0x00156070, 0x00156470, 0x00156870, 0x00156c70, 0x00157070, 0x00157470, 0x00157870, 0x00157c70, // 5464-5471
	0x00158070, 0x00158470, 0x00158870, 0x00158c70, 0x00159070, 0x00159470, 0x00159870, 0x00159c70, // 5472-5479
	0x0015a070, 0x0015a470, 0x0015a870, 0x0015ac70, 0x0015b070, 0x0015b470, 0x0015b870, 0x0015bc70, // 5480-5487
	0x0015c070, 0x0015c470, 0x0015c870, 0x0015cc70, 0x0015d070, 0x0015d470, 0x0015d870, 0x0015dc70, // 5488-5495
	0x0015e070, 0x0015e470, 0x0015e870, 0x0015ec70, 0x0015f070, 0x0015f470, 0x0015f870, 0x0015fc70, // 5496-5503
	0x00160070, 0x00160470, 0x00160870, 0x00160c70, 0x00161070, 0x00161470, 0x00161870, 0x00161c70, // 5504-5511
	0x00162070, 0x00162470, 0x00162870, 0x00162c70, 0x00163070, 0x00163470, 0x00163870, 0x00163c70, // 5512-5519
	0x00164070, 0x00164470, 0x00164870, 0x00164c70, 0x00165070, 0x00165470, 0x00165870, 0x00165c70, // 5520-5527
	0x00166070, 0x00166470, 0x00166870, 0x00166c70, 0x00167070, 0x00167470, 0x00167870, 0x00167c70, // 5528-5535
	0x00168070, 0x00168470, 0x00168870, 0x00168c70, 0x00169070, 0x00169470, 0x00169870, 0x00169c70, // 5536-5543
	0x0016a070, 0x0016a470, 0x0016a870, 0x0016ac70, 0x0016b070, 0x0016b470, 0x0016b870, 0x0016bc70, // 5544-5551
	0x0016c070, 0x0016c470, 0x0016c870, 0x0016cc70, 0x0016d070, 0x0016d470, 0x0016d870, 0x0016dc70, // 5552-5559
	0x0016e070, 0x0016e470, 0x0016e870, 0x0016ec70, 0x0016f070, 0x0016f470, 0x0016f870, 0x0016fc70, // 5560-5567
	0x00170070, 0x00170470, 0x00170870, 0x00170c70, 0x00171070, 0x00171470, 0x00171870, 0x00171c70, // 5568-5575
	0x00172070, 0x00172470, 0x00172870, 0x00172c70, 0x00173070, 0x00173470, 0x00173870, 0x00173c70, // 5576-5583
	0x00174070, 0x00174470, 0x00174870, 0x00174c70, 0x00175070, 0x00175470, 0x00175870, 0x00175c70, // 5584-5591
	0x00176070, 0x00176470, 0x00176870, 0x00176c70, 0x00177070, 0x00177470, 0x00177870, 0x00177c70, // 5592-5599
	0x00178070, 0x00178470, 0x00178870, 0x00178c70, 0x00179070, 0x00179470, 0x00179870, 0x00179c70, // 5600-5607
	0x0017a070, 0x0017a470, 0x0017a870, 0x0017ac70, 0x0017b070, 0x0017b470, 0x0017b870, 0x0017bc70, // 5608-5615
	0x0017c070, 0x0017c470, 0x0017c870, 0x0017cc70, 0x0017d070, 0x0017d470, 0x0017d870, 0x0017dc70, // 5616-5623
	0x0017e070, 0x0017e470, 0x0017e870, 0x0017ec70, 0x0017f070, 0x0017f470, 0x0017f870, 0x0017fc70, // 5624-5631
	0x00180070, 0x00180470, 0x00180870, 0x00180c70, 0x00181070, 0x00181470, 0x00181870, 0x00181c70, // 5632-5639
	0x00182070, 0x00182470, 0x00182870, 0x00182c70, 0x00183070, 0x00183470, 0x00183870, 0x00183c70, // 5640-5647
	0x00184070, 0x00184470, 0x00184870, 0x00184c70, 0x00185070, 0x00185470, 0x00185870, 0x00185c70, // 5648-5655
	0x00186070, 0x00186470, 0x00186870, 0x00186c70, 0x00187070, 0x00187470, 0x00187870, 0x00187c70, // 5656-5663
	0x00188070, 0x00188470, 0x00188870, 0x00188c70, 0x00189070, 0x00189470, 0x00189870, 0x00189c70, // 5664-5671
	0x0018a070, 0x0018a470, 0x0018a870, 0x0018ac70, 0x0018b070, 0x0018b470, 0x0018b870, 0x0018bc70, // 5672-5679
	0x0018c070, 0x0018c470, 0x0018c870, 0x0018cc70, 0x0018d070, 0x0018d470, 0x0018d870, 0x0018dc70, // 5680-5687
	0x0018e070, 0x0018e470, 0x0018e870, 0x0018ec70, 0x0018f070, 0x0018f470, 0x0018f870, 0x0018fc70, // 5688-5695
	0x00190070, 0x00190470, 0x00190870, 0x00190c70, 0x00191070, 0x00191470, 0x00191870, 0x00191c70, // 5696-5703
	0x00192070, 0x00192470, 0x00192870, 0x00192c70, 0x00193070, 0x00193470, 0x00193870, 0x00193c70, // 5704-5711
	0x00194070, 0x00194470, 0x00194870, 0x00194c70, 0x00195070, 0x00195470, 0x00195870, 0x00195c70, // 5712-5719
	0x00196070, 0x00196470, 0x00196870, 0x00196c70, 0x00197070, 0x00197470, 0x00197870, 0x00197c70, // 5720-5727
	0x00198070, 0x00198470, 0x00198870, 0x00198c70, 0x00199070, 0x00199470, 0x00199870, 0x00199c70, // 5728-5735
	0x0019a070, 0x0019a470, 0x0019a870, 0x0019ac70, 0x0019b070, 0x0019b470, 0x0019b870, 0x0019bc70, // 5736-5743
	0x0019c070, 0x0019c470, 0x0019c870, 0x0019cc70, 0x0019d070, 0x0019d470, 0x0019d870, 0x0019dc70, // 5744-5751
	0x0019e070, 0x0019e470, 0x0019e870, 0x0019ec70, 0x0019f070, 0x0019f470, 0x0019f870, 0x0019fc70, // 5752-5759
	0x001a0070, 0x001a0470, 0x001a0870, 0x001a0c70, 0x001a1070, 0x001a1470, 0x001a1870, 0x001a1c70, // 5760-5767
	0x001a2070, 0x001a2470, 0x001a2870, 0x001a2c70, 0x001a3070, 0x001a3470, 0x001a3870, 0x001a3c70, // 5768-5775
	0x001a4070, 0x001a4470, 0x001a4870, 0x001a4c70, 0x001a5070, 0x001a5470, 0x001a5870, 0x001a5c70, // 5776-5783
	0x001a6070, 0x001a6470, 0x001a6870, 0x001a6c70, 0x001a7070, 0x001a7470, 0x001a7870, 0x001a7c70, // 5784-5791
	0x001a8070, 0x001a8470, 0x001a8870, 0x001a8c70, 0x001a9070, 0x001a9470, 0x001a9870, 0x001a9c70, // 5792-5799
	0x001aa070, 0x001aa470, 0x001aa870, 0x001aac70, 0x001ab070, 0x001ab470, 0x001ab870, 0x001abc70, // 5800-5807
	0x001ac070, 0x001ac470, 0x001ac870, 0x001acc70, 0x001ad070, 0x001ad470, 0x001ad870, 0x001adc70, // 5808-5815
	0x001ae070, 0x001ae470, 0x001ae870, 0x001aec70, 0x001af070, 0x001af470, 0x001af870, 0x001afc70, // 5816-5823
	0x001b0070, 0x001b0470, 0x001b0870, 0x001b0c70, 0x001b1070, 0x001b1470, 0x001b1870, 0x001b1c70, // 5824-5831
	0x001b2070, 0x001b2470, 0x001b2870, 0x001b2c70, 0x001b3070, 0x001b3470, 0x001b3870, 0x001b3c70, // 5832-5839
	0x001b4070, 0x001b4470, 0x001b4870, 0x001b4c70, 0x001b5070, 0x001b5470, 0x001b5870, 0x001b5c70, // 5840-5847
	0x001b6070, 0x001b6470, 0x001b6870, 0x001b6c70, 0x001b7070, 0x001b7470, 0x001b7870, 0x001b7c70, // 5848-5855
	0x001b8070, 0x001b8470, 0x001b8870, 0x001b8c70, 0x001b9070, 0x001b9470, 0x001b9870, 0x001b9c70, // 5856-5863
	0x001ba070, 0x001ba470, 0x001ba870, 0x001bac70, 0x001bb070, 0x001bb470, 0x001bb870, 0x001bbc70, // 5864-5871
	0x001bc070, 0x001bc470, 0x001bc870, 0x001bcc70, 0x001bd070, 0x001bd470, 0x001bd870, 0x001bdc70, // 5872-5879
	0x001be070, 0x001be470, 0x001be870, 0x001bec70, 0x001bf070, 0x001bf470, 0x001bf870, 0x001bfc70, // 5880-5887
	0x001c0070, 0x001c0470, 0x001c0870, 0x001c0c70, 0x001c1070, 0x001c1470, 0x001c1870, 0x001c1c70, // 5888-5895
	0x001c2070, 0x001c2470, 0x001c2870, 0x001c2c70, 0x001c3070, 0x001c3470, 0x001c3870, 0x001c3c70, // 5896-5903
	0x001c4070, 0x001c4470, 0x001c4870, 0x001c4c70, 0x001c5070, 0x001c5470, 0x001c5870, 0x001c5c70, // 5904-5911
	0x001c6070, 0x001c6470, 0x001c6870, 0x001c6c70, 0x001c7070, 0x001c7470, 0x001c7870, 0x001c7c70, // 5912-5919
	0x001c8070, 0x001c8470, 0x001c8870, 0x001c8c70, 0x001c9070, 0x001c9470, 0x001c9870, 0x001c9c70, // 5920-5927
	0x001ca070, 0x001ca470, 0x001ca870, 0x001cac70, 0x001cb070, 0x001cb470, 0x001cb870, 0x001cbc70, // 5928-5935
	0x001cc070, 0x001cc470, 0x001cc870, 0x001ccc70, 0x001cd070, 0x001cd470, 0x001cd870, 0x001cdc70, // 5936-5943
	0x001ce070, 0x001ce470, 0x001ce870, 0x001cec70, 0x001cf070, 0x001cf470, 0x001cf870, 0x001cfc70, // 5944-5951
	0x001d0070, 0x001d0470, 0x001d0870, 0x001d0c70, 0x001d1070, 0x001d1470, 0x001d1870, 0x001d1c70, // 5952-5959
	0x001d2070, 0x001d2470, 0x001d2870, 0x001d2c70, 0x001d3070, 0x001d3470, 0x001d3870, 0x001d3c70, // 5960-5967
	0x001d4070, 0x001d4470, 0x001d4870, 0x001d4c70, 0x001d5070, 0x001d5470, 0x001d5870, 0x001d5c70, // 5968-5975
	0x001d6070, 0x001d6470, 0x001d6870, 0x001d6c70, 0x001d7070, 0x001d7470, 0x001d7870, 0x001d7c70, // 5976-5983
	0x001d8070, 0x001d8470, 0x001d8870, 0x001d8c70, 0x001d9070, 0x001d9470, 0x001d9870, 0x001d9c70, // 5984-5991
	0x001da070, 0x001da470, 0x001da870, 0x001dac70, 0x001db070, 0x001db470, 0x001db870, 0x001dbc70, // 5992-5999
	0x001dc070, 0x001dc470, 0x001dc870, 0x001dcc70, 0x001dd070, 0x001dd470, 0x001dd870, 0x001ddc70, // 6000-6007
	0x001de070, 0x001de470, 0x001de870, 0x001dec70, 0x001df070, 0x001df470, 0x001df870, 0x001dfc70, // 6008-6015
	0x001e0070, 0x001e0470, 0x001e0870, 0x001e0c70, 0x001e1070, 0x001e1470, 0x001e1870, 0x001e1c70, // 6016-6023
	0x001e2070, 0x001e2470, 0x001e2870, 0x001e2c70, 0x001e3070, 0x001e3470, 0x001e3870, 0x001e3c70, // 6024-6031
	0x001e4070, 0x001e4470, 0x001e4870, 0x001e4c70, 0x001e5070, 0x001e5470, 0x001e5870, 0x001e5c70, // 6032-6039
	0x001e6070, 0x001e6470, 0x001e6870, 0x001e6c70, 0x001e7070, 0x001e7470, 0x001e7870, 0x001e7c70, // 6040-6047
	0x001e8070, 0x001e8470, 0x001e8870, 0x001e8c70, 0x001e9070, 0x001e9470, 0x001e9870, 0x001e9c70, // 6048-6055
	0x001ea070, 0x001ea470, 0x001ea870, 0x001eac70, 0x001eb070, 0x001eb470, 0x001eb870, 0x001ebc70, // 6056-6063
	0x001ec070, 0x001ec470, 0x001ec870, 0x001ecc70, 0x001ed070, 0x001ed470, 0x001ed870, 0x001edc70, // 6064-6071
	0x001ee070, 0x001ee470, 0x001ee870, 0x001eec70, 0x001ef070, 0x001ef470, 0x001ef870, 0x001efc70, // 6072-6079
	0x001f0070, 0x001f0470, 0x001f0870, 0x001f0c70, 0x001f1070, 0x001f1470, 0x001f1870, 0x001f1c70, // 6080-6087
	0x001f2070, 0x001f2470, 0x001f2870, 0x001f2c70, 0x001f3070, 0x001f3470, 0x001f3870, 0x001f3c70, // 6088-6095
	0x001f4070, 0x001f4470, 0x001f4870, 0x001f4c70, 0x001f5070, 0x001f5470, 0x001f5870, 0x001f5c70, // 6096-6103
	0x001f6070, 0x001f6470, 0x001f6870, 0x001f6c70, 0x001f7070, 0x001f7470, 0x001f7870, 0x001f7c70, // 6104-6111
	0x001f8070, 0x001f8470, 0x001f8870, 0x001f8c70, 0x001f9070, 0x001f9470, 0x001f9870, 0x001f9c70, // 6112-6119
	0x001fa070, 0x001fa470, 0x001fa870, 0x001fac70, 0x001fb070, 0x001fb470, 0x001fb870, 0x001fbc70, // 6120-6127
	0x001fc070, 0x001fc470, 0x001fc870, 0x001fcc70, 0x001fd070, 0x001fd470, 0x001fd870, 0x001fdc70, // 6128-6135
	0x001fe070, 0x001fe470, 0x001fe870, 0x001fec70, 0x001ff070, 0x001ff470, 0x001ff870, 0x001ffc70, // 6136-6143
	0x00000270, 0x00000670, 0x00000a70, 0x00000e70, 0x00001270, 0x00001670, 0x00001a70, 0x00001e70, // 6144-6151
	0x00002270, 0x00002670, 0x00002a70, 0x00002e70, 0x00003270, 0x00003670, 0x00003a70, 0x00003e70, // 6152-6159
	0x00004270, 0x00004670, 0x00004a70, 0x00004e70, 0x00005270, 0x00005670, 0x00005a70, 0x00005e70, // 6160-6167
	0x00006270, 0x00006670, 0x00006a70, 0x00006e70, 0x00007270, 0x00007670, 0x00007a70, 0x00007e70, // 6168-6175
	0x00008270, 0x00008670, 0x00008a70, 0x00008e70, 0x00009270, 0x00009670, 0x00009a70, 0x00009e70, // 6176-6183
	0x0000a270, 0x0000a670, 0x0000aa70, 0x0000ae70, 0x0000b270, 0x0000b670, 0x0000ba70, 0x0000be70, // 6184-6191
	0x0000c270, 0x0000c670, 0x0000ca70, 0x0000ce70, 0x0000d270, 0x0000d670, 0x0000da70, 0x0000de70, // 6192-6199
	0x0000e270, 0x0000e670, 0x0000ea70, 0x0000ee70, 0x0000f270, 0x0000f670, 0x0000fa70, 0x0000fe70, // 6200-6207
	0x00010270, 0x00010670, 0x00010a70, 0x00010e70, 0x00011270, 0x00011670, 0x00011a70, 0x00011e70, // 6208-6215
	0x00012270, 0x00012670, 0x00012a70, 0x00012e70, 0x00013270, 0x00013670, 0x00013a70, 0x00013e70, // 6216-6223
	0x00014270, 0x00014670, 0x00014a70, 0x00014e70, 0x00015270, 0x00015670, 0x00015a70, 0x00015e70, // 6224-6231
	0x00016270, 0x00016670, 0x00016a70, 0x00016e70, 0x00017270, 0x00017670, 0x00017a70, 0x00017e70, // 6232-6239
	0x00018270, 0x00018670, 0x00018a70, 0x00018e70, 0x00019270, 0x00019670, 0x00019a70, 0x00019e70, // 6240-6247
	0x0001a270, 0x0001a670, 0x0001aa70, 0x0001ae70, 0x0001b270, 0x0001b670, 0x0001ba70, 0x0001be70, // 6248-6255
	0x0001c270, 0x0001c670, 0x0001ca70, 0x0001ce70, 0x0001d270, 0x0001d670, 0x0001da70, 0x0001de70, // 6256-6263
	0x0001e270, 0x0001e670, 0x0001ea70, 0x0001ee70, 0x0001f270, 0x0001f670, 0x0001fa70, 0x0001fe70, // 6264-6271
	0x00020270, 0x00020670, 0x00020a70, 0x00020e70, 0x00021270, 0x00021670, 0x00021a70, 0x00021e70, // 6272-6279
	0x00022270, 0x00022670, 0x00022a70, 0x00022e70, 0x00023270, 0x00023670, 0x00023a70, 0x00023e70, // 6280-6287
	0x00024270, 0x00024670, 0x00024a70, 0x00024e70, 0x00025270, 0x00025670, 0x00025a70, 0x00025e70, // 6288-6295
	0x00026270, 0x00026670, 0x00026a70, 0x00026e70, 0x00027270, 0x00027670, 0x00027a70, 0x00027e70, // 6296-6303
	0x00028270, 0x00028670, 0x00028a70, 0x00028e70, 0x00029270, 0x00029670, 0x00029a70, 0x00029e70, // 6304-6311
	0x0002a270, 0x0002a670, 0x0002aa70, 0x0002ae70, 0x0002b270, 0x0002b670, 0x0002ba70, 0x0002be70, // 6312-6319
	0x0002c270, 0x0002c670, 0x0002ca70, 0x0002ce70, 0x0002d270, 0x0002d670, 0x0002da70, 0x0002de70, // 6320-6327
	0x0002e270, 0x0002e670, 0x0002ea70, 0x0002ee70, 0x0002f270, 0x0002f670, 0x0002fa70, 0x0002fe70, // 6328-6335
	0x00030270, 0x00030670, 0x00030a70, 0x00030e70, 0x00031270, 0x00031670, 0x00031a70, 0x00031e70, // 6336-6343
	0x00032270, 0x00032670, 0x00032a70, 0x00032e70, 0x00033270, 0x00033670, 0x00033a70, 0x00033e70, // 6344-6351
	0x00034270, 0x00034670, 0x00034a70, 0x00034e70, 0x00035270, 0x00035670, 0x00035a70, 0x00035e70, // 6352-6359
	0x00036270, 0x00036670, 0x00036a70, 0x00036e70, 0x00037270, 0x00037670, 0x00037a70, 0x00037e70, // 6360-6367
	0x00038270, 0x00038670, 0x00038a70, 0x00038e70, 0x00039270, 0x00039670, 0x00039a70, 0x00039e70, // 6368-6375
	0x0003a270, 0x0003a670, 0x0003aa70, 0x0003ae70, 0x0003b270, 0x0003b670, 0x0003ba70, 0x0003be70, // 6376-6383
	0x0003c270, 0x0003c670, 0x0003ca70, 0x0003ce70, 0x0003d270, 0x0003d670, 0x0003da70, 0x0003de70, // 6384-6391
	0x0003e270, 0x0003e670, 0x0003ea70, 0x0003ee70, 0x0003f270, 0x0003f670, 0x0003fa70, 0x0003fe70, // 6392-6399
	0x00040270, 0x00040670, 0x00040a70, 0x00040e70, 0x00041270, 0x00041670, 0x00041a70, 0x00041e70, // 6400-6407
	0x00042270, 0x00042670, 0x00042a70, 0x00042e70, 0x00043270, 0x00043670, 0x00043a70, 0x00043e70, // 6408-6415
	0x00044270, 0x00044670, 0x00044a70, 0x00044e70, 0x00045270, 0x00045670, 0x00045a70, 0x00045e70, // 6416-6423
	0x00046270, 0x00046670, 0x00046a70, 0x00046e70, 0x00047270, 0x00047670, 0x00047a70, 0x00047e70, // 6424-6431
	0x00048270, 0x00048670, 0x00048a70, 0x00048e70, 0x00049270, 0x00049670, 0x00049a70, 0x00049e70, // 6432-6439
	0x0004a270, 0x0004a670, 0x0004aa70, 0x0004ae70, 0x0004b270, 0x0004b670, 0x0004ba70, 0x0004be70, // 6440-6447
	0x0004c270, 0x0004c670, 0x0004ca70, 0x0004ce70, 0x0004d270, 0x0004d670, 0x0004da70, 0x0004de70, // 6448-6455
	0x0004e270, 0x0004e670, 0x0004ea70, 0x0004ee70, 0x0004f270, 0x0004f670, 0x0004fa70, 0x0004fe70, // 6456-6463
	0x00050270, 0x00050670, 0x00050a70, 0x00050e70, 0x00051270, 0x00051670, 0x00051a70, 0x00051e70, // 6464-6471
	0x00052270, 0x00052670, 0x00052a70, 0x00052e70, 0x00053270, 0x00053670, 0x00053a70, 0x00053e70, // 6472-6479
	0x00054270, 0x00054670, 0x00054a70, 0x00054e70, 0x00055270, 0x00055670, 0x00055a70, 0x00055e70, // 6480-6487
	0x00056270, 0x00056670, 0x00056a70, 0x00056e70, 0x00057270, 0x00057670, 0x00057a70, 0x00057e70, // 6488-6495
	0x00058270, 0x00058670, 0x00058a70, 0x00058e70, 0x00059270, 0x00059670, 0x00059a70, 0x00059e70, // 6496-6503
	0x0005a270, 0x0005a670, 0x0005aa70, 0x0005ae70, 0x0005b270, 0x0005b670, 0x0005ba70, 0x0005be70, // 6504-6511
	0x0005c270, 0x0005c670, 0x0005ca70, 0x0005ce70, 0x0005d270, 0x0005d670, 0x0005da70, 0x0005de70, // 6512-6519
	0x0005e270, 0x0005e670, 0x0005ea70, 0x0005ee70, 0x0005f270, 0x0005f670, 0x0005fa70, 0x0005fe70, // 6520-6527
	0x00060270, 0x00060670, 0x00060a70, 0x00060e70, 0x00061270, 0x00061670, 0x00061a70, 0x00061e70, // 6528-6535
	0x00062270, 0x00062670, 0x00062a70, 0x00062e70, 0x00063270, 0x00063670, 0x00063a70, 0x00063e70, // 6536-6543
	0x00064270, 0x00064670, 0x00064a70, 0x00064e70, 0x00065270, 0x00065670, 0x00065a70, 0x00065e70, // 6544-6551
	0x00066270, 0x00066670, 0x00066a70, 0x00066e70, 0x00067270, 0x00067670, 0x00067a70, 0x00067e70, // 6552-6559
	0x00068270, 0x00068670, 0x00068a70, 0x00068e70, 0x00069270, 0x00069670, 0x00069a70, 0x00069e70, // 6560-6567
	0x0006a270, 0x0006a670, 0x0006aa70, 0x0006ae70, 0x0006b270, 0x0006b670, 0x0006ba70, 0x0006be70, // 6568-6575
	0x0006c270, 0x0006c670, 0x0006ca70, 0x0006ce70, 0x0006d270, 0x0006d670, 0x0006da70, 0x0006de70, // 6576-6583
	0x0006e270, 0x0006e670, 0x0006ea70, 0x0006ee70, 0x0006f270, 0x0006f670, 0x0006fa70, 0x0006fe70, // 6584-6591
	0x00070270, 0x00070670, 0x00070a70, 0x00070e70, 0x00071270, 0x00071670, 0x00071a70, 0x00071e70, // 6592-6599
	0x00072270, 0x00072670, 0x00072a70, 0x00072e70, 0x00073270, 0x00073670, 0x00073a70, 0x00073e70, // 6600-6607
	0x00074270, 0x00074670, 0x00074a70, 0x00074e70, 0x00075270, 0x00075670, 0x00075a70, 0x00075e70, // 6608-6615
	0x00076270, 0x00076670, 0x00076a70, 0x00076e70, 0x00077270, 0x00077670, 0x00077a70, 0x00077e70, // 6616-6623
	0x00078270, 0x00078670, 0x00078a70, 0x00078e70, 0x00079270, 0x00079670, 0x00079a70, 0x00079e70, // 6624-6631
	0x0007a270, 0x0007a670, 0x0007aa70, 0x0007ae70, 0x0007b270, 0x0007b670, 0x0007ba70, 0x0007be70, // 6632-6639
	0x0007c270, 0x0007c670, 0x0007ca70, 0x0007ce70, 0x0007d270, 0x0007d670, 0x0007da70, 0x0007de70, // 6640-6647
	0x0007e270, 0x0007e670, 0x0007ea70, 0x0007ee70, 0x0007f270, 0x0007f670, 0x0007fa70, 0x0007fe70, // 6648-6655
	0x00080270, 0x00080670, 0x00080a70, 0x00080e70, 0x00081270, 0x00081670, 0x00081a70, 0x00081e70, // 6656-6663
	0x00082270, 0x00082670, 0x00082a70, 0x00082e70, 0x00083270, 0x00083670, 0x00083a70, 0x00083e70, // 6664-6671
	0x00084270, 0x00084670, 0x00084a70, 0x00084e70, 0x00085270, 0x00085670, 0x00085a70, 0x00085e70, // 6672-6679
	0x00086270, 0x00086670, 0x00086a70, 0x00086e70, 0x00087270, 0x00087670, 0x00087a70, 0x00087e70, // 6680-6687
	0x00088270, 0x00088670, 0x00088a70, 0x00088e70, 0x00089270, 0x00089670, 0x00089a70, 0x00089e70, // 6688-6695
	0x0008a270, 0x0008a670, 0x0008aa70, 0x0008ae70, 0x0008b270, 0x0008b670, 0x0008ba70, 0x0008be70, // 6696-6703
	0x0008c270, 0x0008c670, 0x0008ca70, 0x0008ce70, 0x0008d270, 0x0008d670, 0x0008da70, 0x0008de70, // 6704-6711
	0x0008e270, 0x0008e670, 0x0008ea70, 0x0008ee70, 0x0008f270, 0x0008f670, 0x0008fa70, 0x0008fe70, // 6712-6719
	0x00090270, 0x00090670, 0x00090a70, 0x00090e70, 0x00091270, 0x00091670, 0x00091a70, 0x00091e70, // 6720-6727
	0x00092270, 0x00092670, 0x00092a70, 0x00092e70, 0x00093270, 0x00093670, 0x00093a70, 0x00093e70, // 6728-6735
	0x00094270, 0x00094670, 0x00094a70, 0x00094e70, 0x00095270, 0x00095670, 0x00095a70, 0x00095e70, // 6736-6743
	0x00096270, 0x00096670, 0x00096a70, 0x00096e70, 0x00097270, 0x00097670, 0x00097a70, 0x00097e70, // 6744-6751
	0x00098270, 0x00098670, 0x00098a70, 0x00098e70, 0x00099270, 0x00099670, 0x00099a70, 0x00099e70, // 6752-6759
	0x0009a270, 0x0009a670, 0x0009aa70, 0x0009ae70, 0x0009b270, 0x0009b670, 0x0009ba70, 0x0009be70, // 6760-6767
	0x0009c270, 0x0009c670, 0x0009ca70, 0x0009ce70, 0x0009d270, 0x0009d670, 0x0009da70, 0x0009de70, // 6768-6775
	0x0009e270, 0x0009e670, 0x0009ea70, 0x0009ee70, 0x0009f270, 0x0009f670, 0x0009fa70, 0x0009fe70, // 6776-6783
	0x000a0270, 0x000a0670, 0x000a0a70, 0x000a0e70, 0x000a1270, 0x000a1670, 0x000a1a70, 0x000a1e70, // 6784-6791
	0x000a2270, 0x000a2670, 0x000a2a70, 0x000a2e70, 0x000a3270, 0x000a3670, 0x000a3a70, 0x000a3e70, // 6792-6799
	0x000a4270, 0x000a4670, 0x000a4a70, 0x000a4e70, 0x000a5270, 0x000a5670, 0x000a5a70, 0x000a5e70, // 6800-6807
	0x000a6270, 0x000a6670, 0x000a6a70, 0x000a6e70, 0x000a7270, 0x000a7670, 0x000a7a70, 0x000a7e70, // 6808-6815
	0x000a8270, 0x000a8670, 0x000a8a70, 0x000a8e70, 0x000a9270, 0x000a9670, 0x000a9a70, 0x000a9e70, // 6816-6823
	0x000aa270, 0x000aa670, 0x000aaa70, 0x000aae70, 0x000ab270, 0x000ab670, 0x000aba70, 0x000abe70, // 6824-6831
	0x000ac270, 0x000ac670, 0x000aca70, 0x000ace70, 0x000ad270, 0x000ad670, 0x000ada70, 0x000ade70, // 6832-6839
	0x000ae270, 0x000ae670, 0x000aea70, 0x000aee70, 0x000af270, 0x000af670, 0x000afa70, 0x000afe70, // 6840-6847
	0x000b0270, 0x000b0670, 0x000b0a70, 0x000b0e70, 0x000b1270, 0x000b1670, 0x000b1a70, 0x000b1e70, // 6848-6855
	0x000b2270, 0x000b2670, 0x000b2a70, 0x000b2e70, 0x000b3270, 0x000b3670, 0x000b3a70, 0x000b3e70, // 6856-6863
	0x000b4270, 0x000b4670, 0x000b4a70, 0x000b4e70, 0x000b5270, 0x000b5670, 0x000b5a70, 0x000b5e70, // 6864-6871
	0x000b6270, 0x000b6670, 0x000b6a70, 0x000b6e70, 0x000b7270, 0x000b7670, 0x000b7a70, 0x000b7e70, // 6872-6879
	0x000b8270, 0x000b8670, 0x000b8a70, 0x000b8e70, 0x000b9270, 0x000b9670, 0x000b9a70, 0x000b9e70, // 6880-6887
	0x000ba270, 0x000ba670, 0x000baa70, 0x000bae70, 0x000bb270, 0x000bb670, 0x000bba70, 0x000bbe70, // 6888-6895
	0x000bc270, 0x000bc670, 0x000bca70, 0x000bce70, 0x000bd270, 0x000bd670, 0x000bda70, 0x000bde70, // 6896-6903
	0x000be270, 0x000be670, 0x000bea70, 0x000bee70, 0x000bf270, 0x000bf670, 0x000bfa70, 0x000bfe70, // 6904-6911
	0x000c0270, 0x000c0670, 0x000c0a70, 0x000c0e70, 0x000c1270, 0x000c1670, 0x000c1a70, 0x000c1e70, // 6912-6919
	0x000c2270, 0x000c2670, 0x000c2a70, 0x000c2e70, 0x000c3270, 0x000c3670, 0x000c3a70, 0x000c3e70, // 6920-6927
	0x000c4270, 0x000c4670, 0x000c4a70, 0x000c4e70, 0x000c5270, 0x000c5670, 0x000c5a70, 0x000c5e70, // 6928-6935
	0x000c6270, 0x000c6670, 0x000c6a70, 0x000c6e70, 0x000c7270, 0x000c7670, 0x000c7a70, 0x000c7e70, // 6936-6943
	0x000c8270, 0x000c8670, 0x000c8a70, 0x000c8e70, 0x000c9270, 0x000c9670, 0x000c9a70, 0x000c9e70, // 6944-6951
	0x000ca270, 0x000ca670, 0x000caa70, 0x000cae70, 0x000cb270, 0x000cb670, 0x000cba70, 0x000cbe70, // 6952-6959
	0x000cc270, 0x000cc670, 0x000cca70, 0x000cce70, 0x000cd270, 0x000cd670, 0x000cda70, 0x000cde70, // 6960-6967
	0x000ce270, 0x000ce670, 0x000cea70, 0x000cee70, 0x000cf270, 0x000cf670, 0x000cfa70, 0x000cfe70, // 6968-6975
	0x000d0270, 0x000d0670, 0x000d0a70, 0x000d0e70, 0x000d1270, 0x000d1670, 0x000d1a70, 0x000d1e70, // 6976-6983
	0x000d2270, 0x000d2670, 0x000d2a70, 0x000d2e70, 0x000d3270, 0x000d3670, 0x000d3a70, 0x000d3e70, // 6984-6991
	0x000d4270, 0x000d4670, 0x000d4a70, 0x000d4e70, 0x000d5270, 0x000d5670, 0x000d5a70, 0x000d5e70, // 6992-6999
	0x000d6270, 0x000d6670, 0x000d6a70, 0x000d6e70, 0x000d7270, 0x000d7670, 0x000d7a70, 0x000d7e70, // 7000-7007
	0x000d8270, 0x000d8670, 0x000d8a70, 0x000d8e70, 0x000d9270, 0x000d9670, 0x000d9a70, 0x000d9e70, // 7008-7015
	0x000da270, 0x000da670, 0x000daa70, 0x000dae70, 0x000db270, 0x000db670, 0x000dba70, 0x000dbe70, // 7016-7023
	0x000dc270, 0x000dc670, 0x000dca70, 0x000dce70, 0x000dd270, 0x000dd670, 0x000dda70, 0x000dde70, // 7024-7031
	0x000de270, 0x000de670, 0x000dea70, 0x000dee70, 0x000df270, 0x000df670, 0x000dfa70, 0x000dfe70, // 7032-7039
	0x000e0270, 0x000e0670, 0x000e0a70, 0x000e0e70, 0x000e1270, 0x000e1670, 0x000e1a70, 0x000e1e70, // 7040-7047
	0x000e2270, 0x000e2670, 0x000e2a70, 0x000e2e70, 0x000e3270, 0x000e3670, 0x000e3a70, 0x000e3e70, // 7048-7055
	0x000e4270, 0x000e4670, 0x000e4a70, 0x000e4e70, 0x000e5270, 0x000e5670, 0x000e5a70, 0x000e5e70, // 7056-7063
	0x000e6270, 0x000e6670, 0x000e6a70, 0x000e6e70, 0x000e7270, 0x000e7670, 0x000e7a70, 0x000e7e70, // 7064-7071
	0x000e8270, 0x000e8670, 0x000e8a70, 0x000e8e70, 0x000e9270, 0x000e9670, 0x000e9a70, 0x000e9e70, // 7072-7079
	0x000ea270, 0x000ea670, 0x000eaa70, 0x000eae70, 0x000eb270, 0x000eb670, 0x000eba70, 0x000ebe70, // 7080-7087
	0x000ec270, 0x000ec670, 0x000eca70, 0x000ece70, 0x000ed270, 0x000ed670, 0x000eda70, 0x000ede70, // 7088-7095
	0x000ee270, 0x000ee670, 0x000eea70, 0x000eee70, 0x000ef270, 0x000ef670, 0x000efa70, 0x000efe70, // 7096-7103
	0x000f0270, 0x000f0670, 0x000f0a70, 0x000f0e70, 0x000f1270, 0x000f1670, 0x000f1a70, 0x000f1e70, // 7104-7111
	0x000f2270, 0x000f2670, 0x000f2a70, 0x000f2e70, 0x000f3270, 0x000f3670, 0x000f3a70, 0x000f3e70, // 7112-7119
	0x000f4270, 0x000f4670, 0x000f4a70, 0x000f4e70, 0x000f5270, 0x000f5670, 0x000f5a70, 0x000f5e70, // 7120-7127
	0x000f6270, 0x000f6670, 0x000f6a70, 0x000f6e70, 0x000f7270, 0x000f7670, 0x000f7a70, 0x000f7e70, // 7128-7135
	0x000f8270, 0x000f8670, 0x000f8a70, 0x000f8e70, 0x000f9270, 0x000f9670, 0x000f9a70, 0x000f9e70, // 7136-7143
	0x000fa270, 0x000fa670, 0x000faa70, 0x000fae70, 0x000fb270, 0x000fb670, 0x000fba70, 0x000fbe70, // 7144-7151
	0x000fc270, 0x000fc670, 0x000fca70, 0x000fce70, 0x000fd270, 0x000fd670, 0x000fda70, 0x000fde70, // 7152-7159
	0x000fe270, 0x000fe670, 0x000fea70, 0x000fee70, 0x000ff270, 0x000ff670, 0x000ffa70, 0x000ffe70, // 7160-7167
	0x00100270, 0x00100670, 0x00100a70, 0x00100e70, 0x00101270, 0x00101670, 0x00101a70, 0x00101e70, // 7168-7175
	0x00102270, 0x00102670, 0x00102a70, 0x00102e70, 0x00103270, 0x00103670, 0x00103a70, 0x00103e70, // 7176-7183
	0x00104270, 0x00104670, 0x00104a70, 0x00104e70, 0x00105270, 0x00105670, 0x00105a70, 0x00105e70, // 7184-7191
	0x00106270, 0x00106670, 0x00106a70, 0x00106e70, 0x00107270, 0x00107670, 0x00107a70, 0x00107e70, // 7192-7199
	0x00108270, 0x00108670, 0x00108a70, 0x00108e70, 0x00109270, 0x00109670, 0x00109a70, 0x00109e70, // 7200-7207
	0x0010a270, 0x0010a670, 0x0010aa70, 0x0010ae70, 0x0010b270, 0x0010b670, 0x0010ba70, 0x0010be70, // 7208-7215
	0x0010c270, 0x0010c670, 0x0010ca70, 0x0010ce70, 0x0010d270, 0x0010d670, 0x0010da70, 0x0010de70, // 7216-7223
	0x0010e270, 0x0010e670, 0x0010ea70, 0x0010ee70, 0x0010f270, 0x0010f670, 0x0010fa70, 0x0010fe70, // 7224-7231
	0x00110270, 0x00110670, 0x00110a70, 0x00110e70, 0x00111270, 0x00111670, 0x00111a70, 0x00111e70, // 7232-7239
	0x00112270, 0x00112670, 0x00112a70, 0x00112e70, 0x00113270, 0x00113670, 0x00113a70, 0x00113e70, // 7240-7247
	0x00114270, 0x00114670, 0x00114a70, 0x00114e70, 0x00115270, 0x00115670, 0x00115a70, 0x00115e70, // 7248-7255
	0x00116270, 0x00116670, 0x00116a70, 0x00116e70, 0x00117270, 0x00117670, 0x00117a70, 0x00117e70, // 7256-7263
	0x00118270, 0x00118670, 0x00118a70, 0x00118e70, 0x00119270, 0x00119670, 0x00119a70, 0x00119e70, // 7264-7271
	0x0011a270, 0x0011a670, 0x0011aa70, 0x0011ae70, 0x0011b270, 0x0011b670, 0x0011ba70, 0x0011be70, // 7272-7279
	0x0011c270, 0x0011c670, 0x0011ca70, 0x0011ce70, 0x0011d270, 0x0011d670, 0x0011da70, 0x0011de70, // 7280-7287
	0x0011e270, 0x0011e670, 0x0011ea70, 0x0011ee70, 0x0011f270, 0x0011f670, 0x0011fa70, 0x0011fe70, // 7288-7295
	0x00120270, 0x00120670, 0x00120a70, 0x00120e70, 0x00121270, 0x00121670, 0x00121a70, 0x00121e70, // 7296-7303
	0x00122270, 0x00122670, 0x00122a70, 0x00122e70, 0x00123270, 0x00123670, 0x00123a70, 0x00123e70, // 7304-7311
	0x00124270, 0x00124670, 0x00124a70, 0x00124e70, 0x00125270, 0x00125670, 0x00125a70, 0x00125e70, // 7312-7319
	0x00126270, 0x00126670, 0x00126a70, 0x00126e70, 0x00127270, 0x00127670, 0x00127a70, 0x00127e70, // 7320-7327
	0x00128270, 0x00128670, 0x00128a70, 0x00128e70, 0x00129270, 0x00129670, 0x00129a70, 0x00129e70, // 7328-7335
	0x0012a270, 0x0012a670, 0x0012aa70, 0x0012ae70, 0x0012b270, 0x0012b670, 0x0012ba70, 0x0012be70, // 7336-7343
	0x0012c270, 0x0012c670, 0x0012ca70, 0x0012ce70, 0x0012d270, 0x0012d670, 0x0012da70, 0x0012de70, // 7344-7351
	0x0012e270, 0x0012e670, 0x0012ea70, 0x0012ee70, 0x0012f270, 0x0012f670, 0x0012fa70, 0x0012fe70, // 7352-7359
	0x00130270, 0x00130670, 0x00130a70, 0x00130e70, 0x00131270, 0x00131670, 0x00131a70, 0x00131e70, // 7360-7367
	0x00132270, 0x00132670, 0x00132a70, 0x00132e70, 0x00133270, 0x00133670, 0x00133a70, 0x00133e70, // 7368-7375
	0x00134270, 0x00134670, 0x00134a70, 0x00134e70, 0x00135270, 0x00135670, 0x00135a70, 0x00135e70, // 7376-7383
	0x00136270, 0x00136670, 0x00136a70, 0x00136e70, 0x00137270, 0x00137670, 0x00137a70, 0x00137e70, // 7384-7391
	0x00138270, 0x00138670, 0x00138a70, 0x00138e70, 0x00139270, 0x00139670, 0x00139a70, 0x00139e70, // 7392-7399
	0x0013a270, 0x0013a670, 0x0013aa70, 0x0013ae70, 0x0013b270, 0x0013b670, 0x0013ba70, 0x0013be70, // 7400-7407
	0x0013c270, 0x0013c670, 0x0013ca70, 0x0013ce70, 0x0013d270, 0x0013d670, 0x0013da70, 0x0013de70, // 7408-7415
	0x0013e270, 0x0013e670, 0x0013ea70, 0x0013ee70, 0x0013f270, 0x0013f670, 0x0013fa70, 0x0013fe70, // 7416-7423
	0x00140270, 0x00140670, 0x00140a70, 0x00140e70, 0x00141270, 0x00141670, 0x00141a70, 0x00141e70, // 7424-7431
	0x00142270, 0x00142670, 0x00142a70, 0x00142e70, 0x00143270, 0x00143670, 0x00143a70, 0x00143e70, // 7432-7439
	0x00144270, 0x00144670, 0x00144a70, 0x00144e70, 0x00145270, 0x00145670, 0x00145a70, 0x00145e70, // 7440-7447
	0x00146270, 0x00146670, 0x00146a70, 0x00146e70, 0x00147270, 0x00147670, 0x00147a70, 0x00147e70, // 7448-7455
	0x00148270, 0x00148670, 0x00148a70, 0x00148e70, 0x00149270, 0x00149670, 0x00149a70, 0x00149e70, // 7456-7463
	0x0014a270, 0x0014a670, 0x0014aa70, 0x0014ae70, 0x0014b270, 0x0014b670, 0x0014ba70, 0x0014be70, // 7464-7471
	0x0014c270, 0x0014c670, 0x0014ca70, 0x0014ce70, 0x0014d270, 0x0014d670, 0x0014da70, 0x0014de70, // 7472-7479
	0x0014e270, 0x0014e670, 0x0014ea70, 0x0014ee70, 0x0014f270, 0x0014f670, 0x0014fa70, 0x0014fe70, // 7480-7487
	0x00150270, 0x00150670, 0x00150a70, 0x00150e70, 0x00151270, 0x00151670, 0x00151a70, 0x00151e70, // 7488-7495
	0x00152270, 0x00152670, 0x00152a70, 0x00152e70, 0x00153270, 0x00153670, 0x00153a70, 0x00153e70, // 7496-7503
	0x00154270, 0x00154670, 0x00154a70, 0x00154e70, 0x00155270, 0x00155670, 0x00155a70, 0x00155e70, // 7504-7511
	0x00156270, 0x00156670, 0x00156a70, 0x00156e70, 0x00157270, 0x00157670, 0x00157a70, 0x00157e70, // 7512-7519
	0x00158270, 0x00158670, 0x00158a70, 0x00158e70, 0x00159270, 0x00159670, 0x00159a70, 0x00159e70, // 7520-7527
	0x0015a270, 0x0015a670, 0x0015aa70, 0x0015ae70, 0x0015b270, 0x0015b670, 0x0015ba70, 0x0015be70, // 7528-7535
	0x0015c270, 0x0015c670, 0x0015ca70, 0x0015ce70, 0x0015d270, 0x0015d670, 0x0015da70, 0x0015de70, // 7536-7543
	0x0015e270, 0x0015e670, 0x0015ea70, 0x0015ee70, 0x0015f270, 0x0015f670, 0x0015fa70, 0x0015fe70, // 7544-7551
	0x00160270, 0x00160670, 0x00160a70, 0x00160e70, 0x00161270, 0x00161670, 0x00161a70, 0x00161e70, // 7552-7559
	0x00162270, 0x00162670, 0x00162a70, 0x00162e70, 0x00163270, 0x00163670, 0x00163a70, 0x00163e70, // 7560-7567
	0x00164270, 0x00164670, 0x00164a70, 0x00164e70, 0x00165270, 0x00165670, 0x00165a70, 0x00165e70, // 7568-7575
	0x00166270, 0x00166670, 0x00166a70, 0x00166e70, 0x00167270, 0x00167670, 0x00167a70, 0x00167e70, // 7576-7583
	0x00168270, 0x00168670, 0x00168a70, 0x00168e70, 0x00169270, 0x00169670, 0x00169a70, 0x00169e70, // 7584-7591
	0x0016a270, 0x0016a670, 0x0016aa70, 0x0016ae70, 0x0016b270, 0x0016b670, 0x0016ba70, 0x0016be70, // 7592-7599
	0x0016c270, 0x0016c670, 0x0016ca70, 0x0016ce70, 0x0016d270, 0x0016d670, 0x0016da70, 0x0016de70, // 7600-7607
	0x0016e270, 0x0016e670, 0x0016ea70, 0x0016ee70, 0x0016f270, 0x0016f670, 0x0016fa70, 0x0016fe70, // 7608-7615
	0x00170270, 0x00170670, 0x00170a70, 0x00170e70, 0x00171270, 0x00171670, 0x00171a70, 0x00171e70, // 7616-7623
	0x00172270, 0x00172670, 0x00172a70, 0x00172e70, 0x00173270, 0x00173670, 0x00173a70, 0x00173e70, // 7624-7631
	0x00174270, 0x00174670, 0x00174a70, 0x00174e70, 0x00175270, 0x00175670, 0x00175a70, 0x00175e70, // 7632-7639
	0x00176270, 0x00176670, 0x00176a70, 0x00176e70, 0x00177270, 0x00177670, 0x00177a70, 0x00177e70, // 7640-7647
	0x00178270, 0x00178670, 0x00178a70, 0x00178e70, 0x00179270, 0x00179670, 0x00179a70, 0x00179e70, // 7648-7655
	0x0017a270, 0x0017a670, 0x0017aa70, 0x0017ae70, 0x0017b270, 0x0017b670, 0x0017ba70, 0x0017be70, // 7656-7663
	0x0017c270, 0x0017c670, 0x0017ca70, 0x0017ce70, 0x0017d270, 0x0017d670, 0x0017da70, 0x0017de70, // 7664-7671
	0x0017e270, 0x0017e670, 0x0017ea70, 0x0017ee70, 0x0017f270, 0x0017f670, 0x0017fa70, 0x0017fe70, // 7672-7679
	0x00180270, 0x00180670, 0x00180a70, 0x00180e70, 0x00181270, 0x00181670, 0x00181a70, 0x00181e70, // 7680-7687
	0x00182270, 0x00182670, 0x00182a70, 0x00182e70, 0x00183270, 0x00183670, 0x00183a70, 0x00183e70, // 7688-7695
	0x00184270, 0x00184670, 0x00184a70, 0x00184e70, 0x00185270, 0x00185670, 0x00185a70, 0x00185e70, // 7696-7703
	0x00186270, 0x00186670, 0x00186a70, 0x00186e70, 0x00187270, 0x00187670, 0x00187a70, 0x00187e70, // 7704-7711
	0x00188270, 0x00188670, 0x00188a70, 0x00188e70, 0x00189270, 0x00189670, 0x00189a70, 0x00189e70, // 7712-7719
	0x0018a270, 0x0018a670, 0x0018aa70, 0x0018ae70, 0x0018b270, 0x0018b670, 0x0018ba70, 0x0018be70, // 7720-7727
	0x0018c270, 0x0018c670, 0x0018ca70, 0x0018ce70, 0x0018d270, 0x0018d670, 0x0018da70, 0x0018de70, // 7728-7735
	0x0018e270, 0x0018e670, 0x0018ea70, 0x0018ee70, 0x0018f270, 0x0018f670, 0x0018fa70, 0x0018fe70, // 7736-7743
	0x00190270, 0x00190670, 0x00190a70, 0x00190e70, 0x00191270, 0x00191670, 0x00191a70, 0x00191e70, // 7744-7751
	0x00192270, 0x00192670, 0x00192a70, 0x00192e70, 0x00193270, 0x00193670, 0x00193a70, 0x00193e70, // 7752-7759
	0x00194270, 0x00194670, 0x00194a70, 0x00194e70, 0x00195270, 0x00195670, 0x00195a70, 0x00195e70, // 7760-7767
	0x00196270, 0x00196670, 0x00196a70, 0x00196e70, 0x00197270, 0x00197670, 0x00197a70, 0x00197e70, // 7768-7775
	0x00198270, 0x00198670, 0x00198a70, 0x00198e70, 0x00199270, 0x00199670, 0x00199a70, 0x00199e70, // 7776-7783
	0x0019a270, 0x0019a670, 0x0019aa70, 0x0019ae70, 0x0019b270, 0x0019b670, 0x0019ba70, 0x0019be70, // 7784-7791
	0x0019c270, 0x0019c670, 0x0019ca70, 0x0019ce70, 0x0019d270, 0x0019d670, 0x0019da70, 0x0019de70, // 7792-7799
	0x0019e270, 0x0019e670, 0x0019ea70, 0x0019ee70, 0x0019f270, 0x0019f670, 0x0019fa70, 0x0019fe70, // 7800-7807
	0x001a0270, 0x001a0670, 0x001a0a70, 0x001a0e70, 0x001a1270, 0x001a1670, 0x001a1a70, 0x001a1e70, // 7808-7815
	0x001a2270, 0x001a2670, 0x001a2a70, 0x001a2e70, 0x001a3270, 0x001a3670, 0x001a3a70, 0x001a3e70, // 7816-7823
	0x001a4270, 0x001a4670, 0x001a4a70, 0x001a4e70, 0x001a5270, 0x001a5670, 0x001a5a70, 0x001a5e70, // 7824-7831
	0x001a6270, 0x001a6670, 0x001a6a70, 0x001a6e70, 0x001a7270, 0x001a7670, 0x001a7a70, 0x001a7e70, // 7832-7839
	0x001a8270, 0x001a8670, 0x001a8a70, 0x001a8e70, 0x001a9270, 0x001a9670, 0x001a9a70, 0x001a9e70, // 7840-7847
	0x001aa270, 0x001aa670, 0x001aaa70, 0x001aae70, 0x001ab270, 0x001ab670, 0x001aba70, 0x001abe70, // 7848-7855
	0x001ac270, 0x001ac670, 0x001aca70, 0x001ace70, 0x001ad270, 0x001ad670, 0x001ada70, 0x001ade70, // 7856-7863
	0x001ae270, 0x001ae670, 0x001aea70, 0x001aee70, 0x001af270, 0x001af670, 0x001afa70, 0x001afe70, // 7864-7871
	0x001b0270, 0x001b0670, 0x001b0a70, 0x001b0e70, 0x001b1270, 0x001b1670, 0x001b1a70, 0x001b1e70, // 7872-7879
	0x001b2270, 0x001b2670, 0x001b2a70, 0x001b2e70, 0x001b3270, 0x001b3670, 0x001b3a70, 0x001b3e70, // 7880-7887
	0x001b4270, 0x001b4670, 0x001b4a70, 0x001b4e70, 0x001b5270, 0x001b5670, 0x001b5a70, 0x001b5e70, // 7888-7895
	0x001b6270, 0x001b6670, 0x001b6a70, 0x001b6e70, 0x001b7270, 0x001b7670, 0x001b7a70, 0x001b7e70, // 7896-7903
	0x001b8270, 0x001b8670, 0x001b8a70, 0x001b8e70, 0x001b9270, 0x001b9670, 0x001b9a70, 0x001b9e70, // 7904-7911
	0x001ba270, 0x001ba670, 0x001baa70, 0x001bae70, 0x001bb270, 0x001bb670, 0x001bba70, 0x001bbe70, // 7912-7919
	0x001bc270, 0x001bc670, 0x001bca70, 0x001bce70, 0x001bd270, 0x001bd670, 0x001bda70, 0x001bde70, // 7920-7927
	0x001be270, 0x001be670, 0x001bea70, 0x001bee70, 0x001bf270, 0x001bf670, 0x001bfa70, 0x001bfe70, // 7928-7935
	0x001c0270, 0x001c0670, 0x001c0a70, 0x001c0e70, 0x001c1270, 0x001c1670, 0x001c1a70, 0x001c1e70, // 7936-7943
	0x001c2270, 0x001c2670, 0x001c2a70, 0x001c2e70, 0x001c3270, 0x001c3670, 0x001c3a70, 0x001c3e70, // 7944-7951
	0x001c4270, 0x001c4670, 0x001c4a70, 0x001c4e70, 0x001c5270, 0x001c5670, 0x001c5a70, 0x001c5e70, // 7952-7959
	0x001c6270, 0x001c6670, 0x001c6a70, 0x001c6e70, 0x001c7270, 0x001c7670, 0x001c7a70, 0x001c7e70, // 7960-7967
	0x001c8270, 0x001c8670, 0x001c8a70, 0x001c8e70, 0x001c9270, 0x001c9670, 0x001c9a70, 0x001c9e70, // 7968-7975
	0x001ca270, 0x001ca670, 0x001caa70, 0x001cae70, 0x001cb270, 0x001cb670, 0x001cba70, 0x001cbe70, // 7976-7983
	0x001cc270, 0x001cc670, 0x001cca70, 0x001cce70, 0x001cd270, 0x001cd670, 0x001cda70, 0x001cde70, // 7984-7991
	0x001ce270, 0x001ce670, 0x001cea70, 0x001cee70, 0x001cf270, 0x001cf670, 0x001cfa70, 0x001cfe70, // 7992-7999
	0x001d0270, 0x001d0670, 0x001d0a70, 0x001d0e70, 0x001d1270, 0x001d1670, 0x001d1a70, 0x001d1e70, // 8000-8007
	0x001d2270, 0x001d2670, 0x001d2a70, 0x001d2e70, 0x001d3270, 0x001d3670, 0x001d3a70, 0x001d3e70, // 8008-8015
	0x001d4270, 0x001d4670, 0x001d4a70, 0x001d4e70, 0x001d5270, 0x001d5670, 0x001d5a70, 0x001d5e70, // 8016-8023
	0x001d6270, 0x001d6670, 0x001d6a70, 0x001d6e70, 0x001d7270, 0x001d7670, 0x001d7a70, 0x001d7e70, // 8024-8031
	0x001d8270, 0x001d8670, 0x001d8a70, 0x001d8e70, 0x001d9270, 0x001d9670, 0x001d9a70, 0x001d9e70, // 8032-8039
	0x001da270, 0x001da670, 0x001daa70, 0x001dae70, 0x001db270, 0x001db670, 0x001dba70, 0x001dbe70, // 8040-8047
	0x001dc270, 0x001dc670, 0x001dca70, 0x001dce70, 0x001dd270, 0x001dd670, 0x001dda70, 0x001dde70, // 8048-8055
	0x001de270, 0x001de670, 0x001dea70, 0x001dee70, 0x001df270, 0x001df670, 0x001dfa70, 0x001dfe70, // 8056-8063
	0x001e0270, 0x001e0670, 0x001e0a70, 0x001e0e70, 0x001e1270, 0x001e1670, 0x001e1a70, 0x001e1e70, // 8064-8071
	0x001e2270, 0x001e2670, 0x001e2a70, 0x001e2e70, 0x001e3270, 0x001e3670, 0x001e3a70, 0x001e3e70, // 8072-8079
	0x001e4270, 0x001e4670, 0x001e4a70, 0x001e4e70, 0x001e5270, 0x001e5670, 0x001e5a70, 0x001e5e70, // 8080-8087
	0x001e6270, 0x001e6670, 0x001e6a70, 0x001e6e70, 0x001e7270, 0x001e7670, 0x001e7a70, 0x001e7e70, // 8088-8095
	0x001e8270, 0x001e8670, 0x001e8a70, 0x001e8e70, 0x001e9270, 0x001e9670, 0x001e9a70, 0x001e9e70, // 8096-8103
	0x001ea270, 0x001ea670, 0x001eaa70, 0x001eae70, 0x001eb270, 0x001eb670, 0x001eba70, 0x001ebe70, // 8104-8111
	0x001ec270, 0x001ec670, 0x001eca70, 0x001ece70, 0x001ed270, 0x001ed670, 0x001eda70, 0x001ede70, // 8112-8119
	0x001ee270, 0x001ee670, 0x001eea70, 0x001eee70, 0x001ef270, 0x001ef670, 0x001efa70, 0x001efe70, // 8120-8127
	0x001f0270, 0x001f0670, 0x001f0a70, 0x001f0e70, 0x001f1270, 0x001f1670, 0x001f1a70, 0x001f1e70, // 8128-8135
	0x001f2270, 0x001f2670, 0x001f2a70, 0x001f2e70, 0x001f3270, 0x001f3670, 0x001f3a70, 0x001f3e70, // 8136-8143
	0x001f4270, 0x001f4670, 0x001f4a70, 0x001f4e70, 0x001f5270, 0x001f5670, 0x001f5a70, 0x001f5e70, // 8144-8151
	0x001f6270, 0x001f6670, 0x001f6a70, 0x001f6e70, 0x001f7270, 0x001f7670, 0x001f7a70, 0x001f7e70, // 8152-8159
	0x001f8270, 0x001f8670, 0x001f8a70, 0x001f8e70, 0x001f9270, 0x001f9670, 0x001f9a70, 0x001f9e70, // 8160-8167
	0x001fa270, 0x001fa670, 0x001faa70, 0x001fae70, 0x001fb270, 0x001fb670, 0x001fba70, 0x001fbe70, // 8168-8175
	0x001fc270, 0x001fc670, 0x001fca70, 0x001fce70, 0x001fd270, 0x001fd670, 0x001fda70, 0x001fde70, // 8176-8183
	0x001fe270, 0x001fe670, 0x001fea70, 0x001fee70, 0x001ff270, 0x001ff670, 0x001ffa70, 0x001ffe70, // 8184-8191
	0x00000171, 0x00000571, 0x00000971, 0x00000d71, 0x00001171, 0x00001571, 0x00001971, 0x00001d71, // 8192-8199
	0x00002171, 0x00002571, 0x00002971, 0x00002d71, 0x00003171, 0x00003571, 0x00003971, 0x00003d71, // 8200-8207
	0x00004171, 0x00004571, 0x00004971, 0x00004d71, 0x00005171, 0x00005571, 0x00005971, 0x00005d71, // 8208-8215
	0x00006171, 0x00006571, 0x00006971, 0x00006d71, 0x00007171, 0x00007571, 0x00007971, 0x00007d71, // 8216-8223
	0x00008171, 0x00008571, 0x00008971, 0x00008d71, 0x00009171, 0x00009571, 0x00009971, 0x00009d71, // 8224-8231
	0x0000a171, 0x0000a571, 0x0000a971, 0x0000ad71, 0x0000b171, 0x0000b571, 0x0000b971, 0x0000bd71, // 8232-8239
	0x0000c171, 0x0000c571, 0x0000c971, 0x0000cd71, 0x0000d171, 0x0000d571, 0x0000d971, 0x0000dd71, // 8240-8247
	0x0000e171, 0x0000e571, 0x0000e971, 0x0000ed71, 0x0000f171, 0x0000f571, 0x0000f971, 0x0000fd71, // 8248-8255
	0x00010171, 0x00010571, 0x00010971, 0x00010d71, 0x00011171, 0x00011571, 0x00011971, 0x00011d71, // 8256-8263
	0x00012171, 0x00012571, 0x00012971, 0x00012d71, 0x00013171, 0x00013571, 0x00013971, 0x00013d71, // 8264-8271
	0x00014171, 0x00014571, 0x00014971, 0x00014d71, 0x00015171, 0x00015571, 0x00015971, 0x00015d71, // 8272-8279
	0x00016171, 0x00016571, 0x00016971, 0x00016d71, 0x00017171, 0x00017571, 0x00017971, 0x00017d71, // 8280-8287
	0x00018171, 0x00018571, 0x00018971, 0x00018d71, 0x00019171, 0x00019571, 0x00019971, 0x00019d71, // 8288-8295
	0x0001a171, 0x0001a571, 0x0001a971, 0x0001ad71, 0x0001b171, 0x0001b571, 0x0001b971, 0x0001bd71, // 8296-8303
	0x0001c171, 0x0001c571, 0x0001c971, 0x0001cd71, 0x0001d171, 0x0001d571, 0x0001d971, 0x0001dd71, // 8304-8311
	0x0001e171, 0x0001e571, 0x0001e971, 0x0001ed71, 0x0001f171, 0x0001f571, 0x0001f971, 0x0001fd71, // 8312-8319
	0x00020171, 0x00020571, 0x00020971, 0x00020d71, 0x00021171, 0x00021571, 0x00021971, 0x00021d71, // 8320-8327
	0x00022171, 0x00022571, 0x00022971, 0x00022d71, 0x00023171, 0x00023571, 0x00023971, 0x00023d71, // 8328-8335
	0x00024171, 0x00024571, 0x00024971, 0x00024d71, 0x00025171, 0x00025571, 0x00025971, 0x00025d71, // 8336-8343
	0x00026171, 0x00026571, 0x00026971, 0x00026d71, 0x00027171, 0x00027571, 0x00027971, 0x00027d71, // 8344-8351
	0x00028171, 0x00028571, 0x00028971, 0x00028d71, 0x00029171, 0x00029571, 0x00029971, 0x00029d71, // 8352-8359
	0x0002a171, 0x0002a571, 0x0002a971, 0x0002ad71, 0x0002b171, 0x0002b571, 0x0002b971, 0x0002bd71, // 8360-8367
	0x0002c171, 0x0002c571, 0x0002c971, 0x0002cd71, 0x0002d171, 0x0002d571, 0x0002d971, 0x0002dd71, // 8368-8375
	0x0002e171, 0x0002e571, 0x0002e971, 0x0002ed71, 0x0002f171, 0x0002f571, 0x0002f971, 0x0002fd71, // 8376-8383
	0x00030171, 0x00030571, 0x00030971, 0x00030d71, 0x00031171, 0x00031571, 0x00031971, 0x00031d71, // 8384-8391
	0x00032171, 0x00032571, 0x00032971, 0x00032d71, 0x00033171, 0x00033571, 0x00033971, 0x00033d71, // 8392-8399
	0x00034171, 0x00034571, 0x00034971, 0x00034d71, 0x00035171, 0x00035571, 0x00035971, 0x00035d71, // 8400-8407
	0x00036171, 0x00036571, 0x00036971, 0x00036d71, 0x00037171, 0x00037571, 0x00037971, 0x00037d71, // 8408-8415
	0x00038171, 0x00038571, 0x00038971, 0x00038d71, 0x00039171, 0x00039571, 0x00039971, 0x00039d71, // 8416-8423
	0x0003a171, 0x0003a571, 0x0003a971, 0x0003ad71, 0x0003b171, 0x0003b571, 0x0003b971, 0x0003bd71, // 8424-8431
	0x0003c171, 0x0003c571, 0x0003c971, 0x0003cd71, 0x0003d171, 0x0003d571, 0x0003d971, 0x0003dd71, // 8432-8439
	0x0003e171, 0x0003e571, 0x0003e971, 0x0003ed71, 0x0003f171, 0x0003f571, 0x0003f971, 0x0003fd71, // 8440-8447
	0x00040171, 0x00040571, 0x00040971, 0x00040d71, 0x00041171, 0x00041571, 0x00041971, 0x00041d71, // 8448-8455
	0x00042171, 0x00042571, 0x00042971, 0x00042d71, 0x00043171, 0x00043571, 0x00043971, 0x00043d71, // 8456-8463
	0x00044171, 0x00044571, 0x00044971, 0x00044d71, 0x00045171, 0x00045571, 0x00045971, 0x00045d71, // 8464-8471
	0x00046171, 0x00046571, 0x00046971, 0x00046d71, 0x00047171, 0x00047571, 0x00047971, 0x00047d71, // 8472-8479
	0x00048171, 0x00048571, 0x00048971, 0x00048d71, 0x00049171, 0x00049571, 0x00049971, 0x00049d71, // 8480-8487
	0x0004a171, 0x0004a571, 0x0004a971, 0x0004ad71, 0x0004b171, 0x0004b571, 0x0004b971, 0x0004bd71, // 8488-8495
	0x0004c171, 0x0004c571, 0x0004c971, 0x0004cd71, 0x0004d171, 0x0004d571, 0x0004d971, 0x0004dd71, // 8496-8503
	0x0004e171, 0x0004e571, 0x0004e971, 0x0004ed71, 0x0004f171, 0x0004f571, 0x0004f971, 0x0004fd71, // 8504-8511
	0x00050171, 0x00050571, 0x00050971, 0x00050d71, 0x00051171, 0x00051571, 0x00051971, 0x00051d71, // 8512-8519
	0x00052171, 0x00052571, 0x00052971, 0x00052d71, 0x00053171, 0x00053571, 0x00053971, 0x00053d71, // 8520-8527
	0x00054171, 0x00054571, 0x00054971, 0x00054d71, 0x00055171, 0x00055571, 0x00055971, 0x00055d71, // 8528-8535
	0x00056171, 0x00056571, 0x00056971, 0x00056d71, 0x00057171, 0x00057571, 0x00057971, 0x00057d71, // 8536-8543
	0x00058171, 0x00058571, 0x00058971, 0x00058d71, 0x00059171, 0x00059571, 0x00059971, 0x00059d71, // 8544-8551
	0x0005a171, 0x0005a571, 0x0005a971, 0x0005ad71, 0x0005b171, 0x0005b571, 0x0005b971, 0x0005bd71, // 8552-8559
	0x0005c171, 0x0005c571, 0x0005c971, 0x0005cd71, 0x0005d171, 0x0005d571, 0x0005d971, 0x0005dd71, // 8560-8567
	0x0005e171, 0x0005e571, 0x0005e971, 0x0005ed71, 0x0005f171, 0x0005f571, 0x0005f971, 0x0005fd71, // 8568-8575
	0x00060171, 0x00060571, 0x00060971, 0x00060d71, 0x00061171, 0x00061571, 0x00061971, 0x00061d71, // 8576-8583
	0x00062171, 0x00062571, 0x00062971, 0x00062d71, 0x00063171, 0x00063571, 0x00063971, 0x00063d71, // 8584-8591
	0x00064171, 0x00064571, 0x00064971, 0x00064d71, 0x00065171, 0x00065571, 0x00065971, 0x00065d71, // 8592-8599
	0x00066171, 0x00066571, 0x00066971, 0x00066d71, 0x00067171, 0x00067571, 0x00067971, 0x00067d71, // 8600-8607
	0x00068171, 0x00068571, 0x00068971, 0x00068d71, 0x00069171, 0x00069571, 0x00069971, 0x00069d71, // 8608-8615
	0x0006a171, 0x0006a571, 0x0006a971, 0x0006ad71, 0x0006b171, 0x0006b571, 0x0006b971, 0x0006bd71, // 8616-8623
	0x0006c171, 0x0006c571, 0x0006c971, 0x0006cd71, 0x0006d171, 0x0006d571, 0x0006d971, 0x0006dd71, // 8624-8631
	0x0006e171, 0x0006e571, 0x0006e971, 0x0006ed71, 0x0006f171, 0x0006f571, 0x0006f971, 0x0006fd71, // 8632-8639
	0x00070171, 0x00070571, 0x00070971, 0x00070d71, 0x00071171, 0x00071571, 0x00071971, 0x00071d71, // 8640-8647
	0x00072171, 0x00072571, 0x00072971, 0x00072d71, 0x00073171, 0x00073571, 0x00073971, 0x00073d71, // 8648-8655
	0x00074171, 0x00074571, 0x00074971, 0x00074d71, 0x00075171, 0x00075571, 0x00075971, 0x00075d71, // 8656-8663
	0x00076171, 0x00076571, 0x00076971, 0x00076d71, 0x00077171, 0x00077571, 0x00077971, 0x00077d71, // 8664-8671
	0x00078171, 0x00078571, 0x00078971, 0x00078d71, 0x00079171, 0x00079571, 0x00079971, 0x00079d71, // 8672-8679
	0x0007a171, 0x0007a571, 0x0007a971, 0x0007ad71, 0x0007b171, 0x0007b571, 0x0007b971, 0x0007bd71, // 8680-8687
	0x0007c171, 0x0007c571, 0x0007c971, 0x0007cd71, 0x0007d171, 0x0007d571, 0x0007d971, 0x0007dd71, // 8688-8695
	0x0007e171, 0x0007e571, 0x0007e971, 0x0007ed71, 0x0007f171, 0x0007f571, 0x0007f971, 0x0007fd71, // 8696-8703
	0x00080171, 0x00080571, 0x00080971, 0x00080d71, 0x00081171, 0x00081571, 0x00081971, 0x00081d71, // 8704-8711
	0x00082171, 0x00082571, 0x00082971, 0x00082d71, 0x00083171, 0x00083571, 0x00083971, 0x00083d71, // 8712-8719
	0x00084171, 0x00084571, 0x00084971, 0x00084d71, 0x00085171, 0x00085571, 0x00085971, 0x00085d71, // 8720-8727
	0x00086171, 0x00086571, 0x00086971, 0x00086d71, 0x00087171, 0x00087571, 0x00087971, 0x00087d71, // 8728-8735
	0x00088171, 0x00088571, 0x00088971, 0x00088d71, 0x00089171, 0x00089571, 0x00089971, 0x00089d71, // 8736-8743
	0x0008a171, 0x0008a571, 0x0008a971, 0x0008ad71, 0x0008b171, 0x0008b571, 0x0008b971, 0x0008bd71, // 8744-8751
	0x0008c171, 0x0008c571, 0x0008c971, 0x0008cd71, 0x0008d171, 0x0008d571, 0x0008d971, 0x0008dd71, // 8752-8759
	0x0008e171, 0x0008e571, 0x0008e971, 0x0008ed71, 0x0008f171, 0x0008f571, 0x0008f971, 0x0008fd71, // 8760-8767
	0x00090171, 0x00090571, 0x00090971, 0x00090d71, 0x00091171, 0x00091571, 0x00091971, 0x00091d71, // 8768-8775
	0x00092171, 0x00092571, 0x00092971, 0x00092d71, 0x00093171, 0x00093571, 0x00093971, 0x00093d71, // 8776-8783
	0x00094171, 0x00094571, 0x00094971, 0x00094d71, 0x00095171, 0x00095571, 0x00095971, 0x00095d71, // 8784-8791
	0x00096171, 0x00096571, 0x00096971, 0x00096d71, 0x00097171, 0x00097571, 0x00097971, 0x00097d71, // 8792-8799
	0x00098171, 0x00098571, 0x00098971, 0x00098d71, 0x00099171, 0x00099571, 0x00099971, 0x00099d71, // 8800-8807
	0x0009a171, 0x0009a571, 0x0009a971, 0x0009ad71, 0x0009b171, 0x0009b571, 0x0009b971, 0x0009bd71, // 8808-8815
	0x0009c171, 0x0009c571, 0x0009c971, 0x0009cd71, 0x0009d171, 0x0009d571, 0x0009d971, 0x0009dd71, // 8816-8823
	0x0009e171, 0x0009e571, 0x0009e971, 0x0009ed71, 0x0009f171, 0x0009f571, 0x0009f971, 0x0009fd71, // 8824-8831
	0x000a0171, 0x000a0571, 0x000a0971, 0x000a0d71, 0x000a1171, 0x000a1571, 0x000a1971, 0x000a1d71, // 8832-8839
	0x000a2171, 0x000a2571, 0x000a2971, 0x000a2d71, 0x000a3171, 0x000a3571, 0x000a3971, 0x000a3d71, // 8840-8847
	0x000a4171, 0x000a4571, 0x000a4971, 0x000a4d71, 0x000a5171, 0x000a5571, 0x000a5971, 0x000a5d71, // 8848-8855
	0x000a6171, 0x000a6571, 0x000a6971, 0x000a6d71, 0x000a7171, 0x000a7571, 0x000a7971, 0x000a7d71, // 8856-8863
	0x000a8171, 0x000a8571, 0x000a8971, 0x000a8d71, 0x000a9171, 0x000a9571, 0x000a9971, 0x000a9d71, // 8864-8871
	0x000aa171, 0x000aa571, 0x000aa971, 0x000aad71, 0x000ab171, 0x000ab571, 0x000ab971, 0x000abd71, // 8872-8879
	0x000ac171, 0x000ac571, 0x000ac971, 0x000acd71, 0x000ad171, 0x000ad571, 0x000ad971, 0x000add71, // 8880-8887
	0x000ae171, 0x000ae571, 0x000ae971, 0x000aed71, 0x000af171, 0x000af571, 0x000af971, 0x000afd71, // 8888-8895
	0x000b0171, 0x000b0571, 0x000b0971, 0x000b0d71, 0x000b1171, 0x000b1571, 0x000b1971, 0x000b1d71, // 8896-8903
	0x000b2171, 0x000b2571, 0x000b2971, 0x000b2d71, 0x000b3171, 0x000b3571, 0x000b3971, 0x000b3d71, // 8904-8911
	0x000b4171, 0x000b4571, 0x000b4971, 0x000b4d71, 0x000b5171, 0x000b5571, 0x000b5971, 0x000b5d71, // 8912-8919
	0x000b6171, 0x000b6571, 0x000b6971, 0x000b6d71, 0x000b7171, 0x000b7571, 0x000b7971, 0x000b7d71, // 8920-8927
	0x000b8171, 0x000b8571, 0x000b8971, 0x000b8d71, 0x000b9171, 0x000b9571, 0x000b9971, 0x000b9d71, // 8928-8935
	0x000ba171, 0x000ba571, 0x000ba971, 0x000bad71, 0x000bb171, 0x000bb571, 0x000bb971, 0x000bbd71, // 8936-8943
	0x000bc171, 0x000bc571, 0x000bc971, 0x000bcd71, 0x000bd171, 0x000bd571, 0x000bd971, 0x000bdd71, // 8944-8951
	0x000be171, 0x000be571, 0x000be971, 0x000bed71, 0x000bf171, 0x000bf571, 0x000bf971, 0x000bfd71, // 8952-8959
	0x000c0171, 0x000c0571, 0x000c0971, 0x000c0d71, 0x000c1171, 0x000c1571, 0x000c1971, 0x000c1d71, // 8960-8967
	0x000c2171, 0x000c2571, 0x000c2971, 0x000c2d71, 0x000c3171, 0x000c3571, 0x000c3971, 0x000c3d71, // 8968-8975
	0x000c4171, 0x000c4571, 0x000c4971, 0x000c4d71, 0x000c5171, 0x000c5571, 0x000c5971, 0x000c5d71, // 8976-8983
	0x000c6171, 0x000c6571, 0x000c6971, 0x000c6d71, 0x000c7171, 0x000c7571, 0x000c7971, 0x000c7d71, // 8984-8991
	0x000c8171, 0x000c8571, 0x000c8971, 0x000c8d71, 0x000c9171, 0x000c9571, 0x000c9971, 0x000c9d71, // 8992-8999
	0x000ca171, 0x000ca571, 0x000ca971, 0x000cad71, 0x000cb171, 0x000cb571, 0x000cb971, 0x000cbd71, // 9000-9007
	0x000cc171, 0x000cc571, 0x000cc971, 0x000ccd71, 0x000cd171, 0x000cd571, 0x000cd971, 0x000cdd71, // 9008-9015
	0x000ce171, 0x000ce571, 0x000ce971, 0x000ced71, 0x000cf171, 0x000cf571, 0x000cf971, 0x000cfd71, // 9016-9023
	0x000d0171, 0x000d0571, 0x000d0971, 0x000d0d71, 0x000d1171, 0x000d1571, 0x000d1971, 0x000d1d71, // 9024-9031
	0x000d2171, 0x000d2571, 0x000d2971, 0x000d2d71, 0x000d3171, 0x000d3571, 0x000d3971, 0x000d3d71, // 9032-9039
	0x000d4171, 0x000d4571, 0x000d4971, 0x000d4d71, 0x000d5171, 0x000d5571, 0x000d5971, 0x000d5d71, // 9040-9047
	0x000d6171, 0x000d6571, 0x000d6971, 0x000d6d71, 0x000d7171, 0x000d7571, 0x000d7971, 0x000d7d71, // 9048-9055
	0x000d8171, 0x000d8571, 0x000d8971, 0x000d8d71, 0x000d9171, 0x000d9571, 0x000d9971, 0x000d9d71, // 9056-9063
	0x000da171, 0x000da571, 0x000da971, 0x000dad71, 0x000db171, 0x000db571, 0x000db971, 0x000dbd71, // 9064-9071
	0x000dc171, 0x000dc571, 0x000dc971, 0x000dcd71, 0x000dd171, 0x000dd571, 0x000dd971, 0x000ddd71, // 9072-9079
	0x000de171, 0x000de571, 0x000de971, 0x000ded71, 0x000df171, 0x000df571, 0x000df971, 0x000dfd71, // 9080-9087
	0x000e0171, 0x000e0571, 0x000e0971, 0x000e0d71, 0x000e1171, 0x000e1571, 0x000e1971, 0x000e1d71, // 9088-9095
	0x000e2171, 0x000e2571, 0x000e2971, 0x000e2d71, 0x000e3171, 0x000e3571, 0x000e3971, 0x000e3d71, // 9096-9103
	0x000e4171, 0x000e4571, 0x000e4971, 0x000e4d71, 0x000e5171, 0x000e5571, 0x000e5971, 0x000e5d71, // 9104-9111
	0x000e6171, 0x000e6571, 0x000e6971, 0x000e6d71, 0x000e7171, 0x000e7571, 0x000e7971, 0x000e7d71, // 9112-9119
	0x000e8171, 0x000e8571, 0x000e8971, 0x000e8d71, 0x000e9171, 0x000e9571, 0x000e9971, 0x000e9d71, // 9120-9127
	0x000ea171, 0x000ea571, 0x000ea971, 0x000ead71, 0x000eb171, 0x000eb571, 0x000eb971, 0x000ebd71, // 9128-9135
	0x000ec171, 0x000ec571, 0x000ec971, 0x000ecd71, 0x000ed171, 0x000ed571, 0x000ed971, 0x000edd71, // 9136-9143
	0x000ee171, 0x000ee571, 0x000ee971, 0x000eed71, 0x000ef171, 0x000ef571, 0x000ef971, 0x000efd71, // 9144-9151
	0x000f0171, 0x000f0571, 0x000f0971, 0x000f0d71, 0x000f1171, 0x000f1571, 0x000f1971, 0x000f1d71, // 9152-9159
	0x000f2171, 0x000f2571, 0x000f2971, 0x000f2d71, 0x000f3171, 0x000f3571, 0x000f3971, 0x000f3d71, // 9160-9167
	0x000f4171, 0x000f4571, 0x000f4971, 0x000f4d71, 0x000f5171, 0x000f5571, 0x000f5971, 0x000f5d71, // 9168-9175
	0x000f6171, 0x000f6571, 0x000f6971, 0x000f6d71, 0x000f7171, 0x000f7571, 0x000f7971, 0x000f7d71, // 9176-9183
	0x000f8171, 0x000f8571, 0x000f8971, 0x000f8d71, 0x000f9171, 0x000f9571, 0x000f9971, 0x000f9d71, // 9184-9191
	0x000fa171, 0x000fa571, 0x000fa971, 0x000fad71, 0x000fb171, 0x000fb571, 0x000fb971, 0x000fbd71, // 9192-9199
	0x000fc171, 0x000fc571, 0x000fc971, 0x000fcd71, 0x000fd171, 0x000fd571, 0x000fd971, 0x000fdd71, // 9200-9207
	0x000fe171, 0x000fe571, 0x000fe971, 0x000fed71, 0x000ff171, 0x000ff571, 0x000ff971, 0x000ffd71, // 9208-9215
	0x00100171, 0x00100571, 0x00100971, 0x00100d71, 0x00101171, 0x00101571, 0x00101971, 0x00101d71, // 9216-9223
	0x00102171, 0x00102571, 0x00102971, 0x00102d71, 0x00103171, 0x00103571, 0x00103971, 0x00103d71, // 9224-9231
	0x00104171, 0x00104571, 0x00104971, 0x00104d71, 0x00105171, 0x00105571, 0x00105971, 0x00105d71, // 9232-9239
	0x00106171, 0x00106571, 0x00106971, 0x00106d71, 0x00107171, 0x00107571, 0x00107971, 0x00107d71, // 9240-9247
	0x00108171, 0x00108571, 0x00108971, 0x00108d71, 0x00109171, 0x00109571, 0x00109971, 0x00109d71, // 9248-9255
	0x0010a171, 0x0010a571, 0x0010a971, 0x0010ad71, 0x0010b171, 0x0010b571, 0x0010b971, 0x0010bd71, // 9256-9263
	0x0010c171, 0x0010c571, 0x0010c971, 0x0010cd71, 0x0010d171, 0x0010d571, 0x0010d971, 0x0010dd71, // 9264-9271
	0x0010e171, 0x0010e571, 0x0010e971, 0x0010ed71, 0x0010f171, 0x0010f571, 0x0010f971, 0x0010fd71, // 9272-9279
	0x00110171, 0x00110571, 0x00110971, 0x00110d71, 0x00111171, 0x00111571, 0x00111971, 0x00111d71, // 9280-9287
	0x00112171, 0x00112571, 0x00112971, 0x00112d71, 0x00113171, 0x00113571, 0x00113971, 0x00113d71, // 9288-9295
	0x00114171, 0x00114571, 0x00114971, 0x00114d71, 0x00115171, 0x00115571, 0x00115971, 0x00115d71, // 9296-9303
	0x00116171, 0x00116571, 0x00116971, 0x00116d71, 0x00117171, 0x00117571, 0x00117971, 0x00117d71, // 9304-9311
	0x00118171, 0x00118571, 0x00118971, 0x00118d71, 0x00119171, 0x00119571, 0x00119971, 0x00119d71, // 9312-9319
	0x0011a171, 0x0011a571, 0x0011a971, 0x0011ad71, 0x0011b171, 0x0011b571, 0x0011b971, 0x0011bd71, // 9320-9327
	0x0011c171, 0x0011c571, 0x0011c971, 0x0011cd71, 0x0011d171, 0x0011d571, 0x0011d971, 0x0011dd71, // 9328-9335
	0x0011e171, 0x0011e571, 0x0011e971, 0x0011ed71, 0x0011f171, 0x0011f571, 0x0011f971, 0x0011fd71, // 9336-9343
	0x00120171, 0x00120571, 0x00120971, 0x00120d71, 0x00121171, 0x00121571, 0x00121971, 0x00121d71, // 9344-9351
	0x00122171, 0x00122571, 0x00122971, 0x00122d71, 0x00123171, 0x00123571, 0x00123971, 0x00123d71, // 9352-9359
	0x00124171, 0x00124571, 0x00124971, 0x00124d71, 0x00125171, 0x00125571, 0x00125971, 0x00125d71, // 9360-9367
	0x00126171, 0x00126571, 0x00126971, 0x00126d71, 0x00127171, 0x00127571, 0x00127971, 0x00127d71, // 9368-9375
	0x00128171, 0x00128571, 0x00128971, 0x00128d71, 0x00129171, 0x00129571, 0x00129971, 0x00129d71, // 9376-9383
	0x0012a171, 0x0012a571, 0x0012a971, 0x0012ad71, 0x0012b171, 0x0012b571, 0x0012b971, 0x0012bd71, // 9384-9391
	0x0012c171, 0x0012c571, 0x0012c971, 0x0012cd71, 0x0012d171, 0x0012d571, 0x0012d971, 0x0012dd71, // 9392-9399
	0x0012e171, 0x0012e571, 0x0012e971, 0x0012ed71, 0x0012f171, 0x0012f571, 0x0012f971, 0x0012fd71, // 9400-9407
	0x00130171, 0x00130571, 0x00130971, 0x00130d71, 0x00131171, 0x00131571, 0x00131971, 0x00131d71, // 9408-9415
	0x00132171, 0x00132571, 0x00132971, 0x00132d71, 0x00133171, 0x00133571, 0x00133971, 0x00133d71, // 9416-9423
	0x00134171, 0x00134571, 0x00134971, 0x00134d71, 0x00135171, 0x00135571, 0x00135971, 0x00135d71, // 9424-9431
	0x00136171, 0x00136571, 0x00136971, 0x00136d71, 0x00137171, 0x00137571, 0x00137971, 0x00137d71, // 9432-9439
	0x00138171, 0x00138571, 0x00138971, 0x00138d71, 0x00139171, 0x00139571, 0x00139971, 0x00139d71, // 9440-9447
	0x0013a171, 0x0013a571, 0x0013a971, 0x0013ad71, 0x0013b171, 0x0013b571, 0x0013b971, 0x0013bd71, // 9448-9455
	0x0013c171, 0x0013c571, 0x0013c971, 0x0013cd71, 0x0013d171, 0x0013d571, 0x0013d971, 0x0013dd71, // 9456-9463
	0x0013e171, 0x0013e571, 0x0013e971, 0x0013ed71, 0x0013f171, 0x0013f571, 0x0013f971, 0x0013fd71, // 9464-9471
	0x00140171, 0x00140571, 0x00140971, 0x00140d71, 0x00141171, 0x00141571, 0x00141971, 0x00141d71, // 9472-9479
	0x00142171, 0x00142571, 0x00142971, 0x00142d71, 0x00143171, 0x00143571, 0x00143971, 0x00143d71, // 9480-9487
	0x00144171, 0x00144571, 0x00144971, 0x00144d71, 0x00145171, 0x00145571, 0x00145971, 0x00145d71, // 9488-9495
	0x00146171, 0x00146571, 0x00146971, 0x00146d71, 0x00147171, 0x00147571, 0x00147971, 0x00147d71, // 9496-9503
	0x00148171, 0x00148571, 0x00148971, 0x00148d71, 0x00149171, 0x00149571, 0x00149971, 0x00149d71, // 9504-9511
	0x0014a171, 0x0014a571, 0x0014a971, 0x0014ad71, 0x0014b171, 0x0014b571, 0x0014b971, 0x0014bd71, // 9512-9519
	0x0014c171, 0x0014c571, 0x0014c971, 0x0014cd71, 0x0014d171, 0x0014d571, 0x0014d971, 0x0014dd71, // 9520-9527
	0x0014e171, 0x0014e571, 0x0014e971, 0x0014ed71, 0x0014f171, 0x0014f571, 0x0014f971, 0x0014fd71, // 9528-9535
	0x00150171, 0x00150571, 0x00150971, 0x00150d71, 0x00151171, 0x00151571, 0x00151971, 0x00151d71, // 9536-9543
	0x00152171, 0x00152571, 0x00152971, 0x00152d71, 0x00153171, 0x00153571, 0x00153971, 0x00153d71, // 9544-9551
	0x00154171, 0x00154571, 0x00154971, 0x00154d71, 0x00155171, 0x00155571, 0x00155971, 0x00155d71, // 9552-9559
	0x00156171, 0x00156571, 0x00156971, 0x00156d71, 0x00157171, 0x00157571, 0x00157971, 0x00157d71, // 9560-9567
	0x00158171, 0x00158571, 0x00158971, 0x00158d71, 0x00159171, 0x00159571, 0x00159971, 0x00159d71, // 9568-9575
	0x0015a171, 0x0015a571, 0x0015a971, 0x0015ad71, 0x0015b171, 0x0015b571, 0x0015b971, 0x0015bd71, // 9576-9583
	0x0015c171, 0x0015c571, 0x0015c971, 0x0015cd71, 0x0015d171, 0x0015d571, 0x0015d971, 0x0015dd71, // 9584-9591
	0x0015e171, 0x0015e571, 0x0015e971, 0x0015ed71, 0x0015f171, 0x0015f571, 0x0015f971, 0x0015fd71, // 9592-9599
	0x00160171, 0x00160571, 0x00160971, 0x00160d71, 0x00161171, 0x00161571, 0x00161971, 0x00161d71, // 9600-9607
	0x00162171, 0x00162571, 0x00162971, 0x00162d71, 0x00163171, 0x00163571, 0x00163971, 0x00163d71, // 9608-9615
	0x00164171, 0x00164571, 0x00164971, 0x00164d71, 0x00165171, 0x00165571, 0x00165971, 0x00165d71, // 9616-9623
	0x00166171, 0x00166571, 0x00166971, 0x00166d71, 0x00167171, 0x00167571, 0x00167971, 0x00167d71, // 9624-9631
	0x00168171, 0x00168571, 0x00168971, 0x00168d71, 0x00169171, 0x00169571, 0x00169971, 0x00169d71, // 9632-9639
	0x0016a171, 0x0016a571, 0x0016a971, 0x0016ad71, 0x0016b171, 0x0016b571, 0x0016b971, 0x0016bd71, // 9640-9647
	0x0016c171, 0x0016c571, 0x0016c971, 0x0016cd71, 0x0016d171, 0x0016d571, 0x0016d971, 0x0016dd71, // 9648-9655
	0x0016e171, 0x0016e571, 0x0016e971, 0x0016ed71, 0x0016f171, 0x0016f571, 0x0016f971, 0x0016fd71, // 9656-9663
	0x00170171, 0x00170571, 0x00170971, 0x00170d71, 0x00171171, 0x00171571, 0x00171971, 0x00171d71, // 9664-9671
	0x00172171, 0x00172571, 0x00172971, 0x00172d71, 0x00173171, 0x00173571, 0x00173971, 0x00173d71, // 9672-9679
	0x00174171, 0x00174571, 0x00174971, 0x00174d71, 0x00175171, 0x00175571, 0x00175971, 0x00175d71, // 9680-9687
	0x00176171, 0x00176571, 0x00176971, 0x00176d71, 0x00177171, 0x00177571, 0x00177971, 0x00177d71, // 9688-9695
	0x00178171, 0x00178571, 0x00178971, 0x00178d71, 0x00179171, 0x00179571, 0x00179971, 0x00179d71, // 9696-9703
	0x0017a171, 0x0017a571, 0x0017a971, 0x0017ad71, 0x0017b171, 0x0017b571, 0x0017b971, 0x0017bd71, // 9704-9711
	0x0017c171, 0x0017c571, 0x0017c971, 0x0017cd71, 0x0017d171, 0x0017d571, 0x0017d971, 0x0017dd71, // 9712-9719
	0x0017e171, 0x0017e571, 0x0017e971, 0x0017ed71, 0x0017f171, 0x0017f571, 0x0017f971, 0x0017fd71, // 9720-9727
	0x00180171, 0x00180571, 0x00180971, 0x00180d71, 0x00181171, 0x00181571, 0x00181971, 0x00181d71, // 9728-9735
	0x00182171, 0x00182571, 0x00182971, 0x00182d71, 0x00183171, 0x00183571, 0x00183971, 0x00183d71, // 9736-9743
	0x00184171, 0x00184571, 0x00184971, 0x00184d71, 0x00185171, 0x00185571, 0x00185971, 0x00185d71, // 9744-9751
	0x00186171, 0x00186571, 0x00186971, 0x00186d71, 0x00187171, 0x00187571, 0x00187971, 0x00187d71, // 9752-9759
	0x00188171, 0x00188571, 0x00188971, 0x00188d71, 0x00189171, 0x00189571, 0x00189971, 0x00189d71, // 9760-9767
	0x0018a171, 0x0018a571, 0x0018a971, 0x0018ad71, 0x0018b171, 0x0018b571, 0x0018b971, 0x0018bd71, // 9768-9775
	0x0018c171, 0x0018c571, 0x0018c971, 0x0018cd71, 0x0018d171, 0x0018d571, 0x0018d971, 0x0018dd71, // 9776-9783
	0x0018e171, 0x0018e571, 0x0018e971, 0x0018ed71, 0x0018f171, 0x0018f571, 0x0018f971, 0x0018fd71, // 9784-9791
	0x00190171, 0x00190571, 0x00190971, 0x00190d71, 0x00191171, 0x00191571, 0x00191971, 0x00191d71, // 9792-9799
	0x00192171, 0x00192571, 0x00192971, 0x00192d71, 0x00193171, 0x00193571, 0x00193971, 0x00193d71, // 9800-9807
	0x00194171, 0x00194571, 0x00194971, 0x00194d71, 0x00195171, 0x00195571, 0x00195971, 0x00195d71, // 9808-9815
	0x00196171, 0x00196571, 0x00196971, 0x00196d71, 0x00197171, 0x00197571, 0x00197971, 0x00197d71, // 9816-9823
	0x00198171, 0x00198571, 0x00198971, 0x00198d71, 0x00199171, 0x00199571, 0x00199971, 0x00199d71, // 9824-9831
	0x0019a171, 0x0019a571, 0x0019a971, 0x0019ad71, 0x0019b171, 0x0019b571, 0x0019b971, 0x0019bd71, // 9832-9839
	0x0019c171, 0x0019c571, 0x0019c971, 0x0019cd71, 0x0019d171, 0x0019d571, 0x0019d971, 0x0019dd71, // 9840-9847
	0x0019e171, 0x0019e571, 0x0019e971, 0x0019ed71, 0x0019f171, 0x0019f571, 0x0019f971, 0x0019fd71, // 9848-9855
	0x001a0171, 0x001a0571, 0x001a0971, 0x001a0d71, 0x001a1171, 0x001a1571, 0x001a1971, 0x001a1d71, // 9856-9863
	0x001a2171, 0x001a2571, 0x001a2971, 0x001a2d71, 0x001a3171, 0x001a3571, 0x001a3971, 0x001a3d71, // 9864-9871
	0x001a4171, 0x001a4571, 0x001a4971, 0x001a4d71, 0x001a5171, 0x001a5571, 0x001a5971, 0x001a5d71, // 9872-9879
	0x001a6171, 0x001a6571, 0x001a6971, 0x001a6d71, 0x001a7171, 0x001a7571, 0x001a7971, 0x001a7d71, // 9880-9887
	0x001a8171, 0x001a8571, 0x001a8971, 0x001a8d71, 0x001a9171, 0x001a9571, 0x001a9971, 0x001a9d71, // 9888-9895
	0x001aa171, 0x001aa571, 0x001aa971, 0x001aad71, 0x001ab171, 0x001ab571, 0x001ab971, 0x001abd71, // 9896-9903
	0x001ac171, 0x001ac571, 0x001ac971, 0x001acd71, 0x001ad171, 0x001ad571, 0x001ad971, 0x001add71, // 9904-9911
	0x001ae171, 0x001ae571, 0x001ae971, 0x001aed71, 0x001af171, 0x001af571, 0x001af971, 0x001afd71, // 9912-9919
	0x001b0171, 0x001b0571, 0x001b0971, 0x001b0d71, 0x001b1171, 0x001b1571, 0x001b1971, 0x001b1d71, // 9920-9927
	0x001b2171, 0x001b2571, 0x001b2971, 0x001b2d71, 0x001b3171, 0x001b3571, 0x001b3971, 0x001b3d71, // 9928-9935
	0x001b4171, 0x001b4571, 0x001b4971, 0x001b4d71, 0x001b5171, 0x001b5571, 0x001b5971, 0x001b5d71, // 9936-9943
	0x001b6171, 0x001b6571, 0x001b6971, 0x001b6d71, 0x001b7171, 0x001b7571, 0x001b7971, 0x001b7d71, // 9944-9951
	0x001b8171, 0x001b8571, 0x001b8971, 0x001b8d71, 0x001b9171, 0x001b9571, 0x001b9971, 0x001b9d71, // 9952-9959
	0x001ba171, 0x001ba571, 0x001ba971, 0x001bad71, 0x001bb171, 0x001bb571, 0x001bb971, 0x001bbd71, // 9960-9967
	0x001bc171, 0x001bc571, 0x001bc971, 0x001bcd71, 0x001bd171, 0x001bd571, 0x001bd971, 0x001bdd71, // 9968-9975
	0x001be171, 0x001be571, 0x001be971, 0x001bed71, 0x001bf171, 0x001bf571, 0x001bf971, 0x001bfd71, // 9976-9983
	0x001c0171, 0x001c0571, 0x001c0971, 0x001c0d71, 0x001c1171, 0x001c1571, 0x001c1971, 0x001c1d71, // 9984-9991
	0x001c2171, 0x001c2571, 0x001c2971, 0x001c2d71, 0x001c3171, 0x001c3571, 0x001c3971, 0x001c3d71, // 9992-9999
	0x001c4171, 0x001c4571, 0x001c4971, 0x001c4d71, 0x001c5171, 0x001c5571, 0x001c5971, 0x001c5d71, // 10000-10007
	0x001c6171, 0x001c6571, 0x001c6971, 0x001c6d71, 0x001c7171, 0x001c7571, 0x001c7971, 0x001c7d71, // 10008-10015
	0x001c8171, 0x001c8571, 0x001c8971, 0x001c8d71, 0x001c9171, 0x001c9571, 0x001c9971, 0x001c9d71, // 10016-10023
	0x001ca171, 0x001ca571, 0x001ca971, 0x001cad71, 0x001cb171, 0x001cb571, 0x001cb971, 0x001cbd71, // 10024-10031
	0x001cc171, 0x001cc571, 0x001cc971, 0x001ccd71, 0x001cd171, 0x001cd571, 0x001cd971, 0x001cdd71, // 10032-10039
	0x001ce171, 0x001ce571, 0x001ce971, 0x001ced71, 0x001cf171, 0x001cf571, 0x001cf971, 0x001cfd71, // 10040-10047
	0x001d0171, 0x001d0571, 0x001d0971, 0x001d0d71, 0x001d1171, 0x001d1571, 0x001d1971, 0x001d1d71, // 10048-10055
	0x001d2171, 0x001d2571, 0x001d2971, 0x001d2d71, 0x001d3171, 0x001d3571, 0x001d3971, 0x001d3d71, // 10056-10063
	0x001d4171, 0x001d4571, 0x001d4971, 0x001d4d71, 0x001d5171, 0x001d5571, 0x001d5971, 0x001d5d71, // 10064-10071
	0x001d6171, 0x001d6571, 0x001d6971, 0x001d6d71, 0x001d7171, 0x001d7571, 0x001d7971, 0x001d7d71, // 10072-10079
	0x001d8171, 0x001d8571, 0x001d8971, 0x001d8d71, 0x001d9171, 0x001d9571, 0x001d9971, 0x001d9d71, // 10080-10087
	0x001da171, 0x001da571, 0x001da971, 0x001dad71, 0x001db171, 0x001db571, 0x001db971, 0x001dbd71, // 10088-10095
	0x001dc171, 0x001dc571, 0x001dc971, 0x001dcd71, 0x001dd171, 0x001dd571, 0x001dd971, 0x001ddd71, // 10096-10103
	0x001de171, 0x001de571, 0x001de971, 0x001ded71, 0x001df171, 0x001df571, 0x001df971, 0x001dfd71, // 10104-10111
	0x001e0171, 0x001e0571, 0x001e0971, 0x001e0d71, 0x001e1171, 0x001e1571, 0x001e1971, 0x001e1d71, // 10112-10119
	0x001e2171, 0x001e2571, 0x001e2971, 0x001e2d71, 0x001e3171, 0x001e3571, 0x001e3971, 0x001e3d71, // 10120-10127
	0x001e4171, 0x001e4571, 0x001e4971, 0x001e4d71, 0x001e5171, 0x001e5571, 0x001e5971, 0x001e5d71, // 10128-10135
	0x001e6171, 0x001e6571, 0x001e6971, 0x001e6d71, 0x001e7171, 0x001e7571, 0x001e7971, 0x001e7d71, // 10136-10143
	0x001e8171, 0x001e8571, 0x001e8971, 0x001e8d71, 0x001e9171, 0x001e9571, 0x001e9971, 0x001e9d71, // 10144-10151
	0x001ea171, 0x001ea571, 0x001ea971, 0x001ead71, 0x001eb171, 0x001eb571, 0x001eb971, 0x001ebd71, // 10152-10159
	0x001ec171, 0x001ec571, 0x001ec971, 0x001ecd71, 0x001ed171, 0x001ed571, 0x001ed971, 0x001edd71, // 10160-10167
	0x001ee171, 0x001ee571, 0x001ee971, 0x001eed71, 0x001ef171, 0x001ef571, 0x001ef971, 0x001efd71, // 10168-10175
	0x001f0171, 0x001f0571, 0x001f0971, 0x001f0d71, 0x001f1171, 0x001f1571, 0x001f1971, 0x001f1d71, // 10176-10183
	0x001f2171, 0x001f2571, 0x001f2971, 0x001f2d71, 0x001f3171, 0x001f3571, 0x001f3971, 0x001f3d71, // 10184-10191
	0x001f4171, 0x001f4571, 0x001f4971, 0x001f4d71, 0x001f5171, 0x001f5571, 0x001f5971, 0x001f5d71, // 10192-10199
	0x001f6171, 0x001f6571, 0x001f6971, 0x001f6d71, 0x001f7171, 0x001f7571, 0x001f7971, 0x001f7d71, // 10200-10207
	0x001f8171, 0x001f8571, 0x001f8971, 0x001f8d71, 0x001f9171, 0x001f9571, 0x001f9971, 0x001f9d71, // 10208-10215
	0x001fa171, 0x001fa571, 0x001fa971, 0x001fad71, 0x001fb171, 0x001fb571, 0x001fb971, 0x001fbd71, // 10216-10223
	0x001fc171, 0x001fc571, 0x001fc971, 0x001fcd71, 0x001fd171, 0x001fd571, 0x001fd971, 0x001fdd71, // 10224-10231
	0x001fe171, 0x001fe571, 0x001fe971, 0x001fed71, 0x001ff171, 0x001ff571, 0x001ff971, 0x001ffd71, // 10232-10239
	0x00200171, 0x00200571, 0x00200971, 0x00200d71, 0x00201171, 0x00201571, 0x00201971, 0x00201d71, // 10240-10247
	0x00202171, 0x00202571, 0x00202971, 0x00202d71, 0x00203171, 0x00203571, 0x00203971, 0x00203d71, // 10248-10255
	0x00204171, 0x00204571, 0x00204971, 0x00204d71, 0x00205171, 0x00205571, 0x00205971, 0x00205d71, // 10256-10263
	0x00206171, 0x00206571, 0x00206971, 0x00206d71, 0x00207171, 0x00207571, 0x00207971, 0x00207d71, // 10264-10271
	0x00208171, 0x00208571, 0x00208971, 0x00208d71, 0x00209171, 0x00209571, 0x00209971, 0x00209d71, // 10272-10279
	0x0020a171, 0x0020a571, 0x0020a971, 0x0020ad71, 0x0020b171, 0x0020b571, 0x0020b971, 0x0020bd71, // 10280-10287
	0x0020c171, 0x0020c571, 0x0020c971, 0x0020cd71, 0x0020d171, 0x0020d571, 0x0020d971, 0x0020dd71, // 10288-10295
	0x0020e171, 0x0020e571, 0x0020e971, 0x0020ed71, 0x0020f171, 0x0020f571, 0x0020f971, 0x0020fd71, // 10296-10303
	0x00210171, 0x00210571, 0x00210971, 0x00210d71, 0x00211171, 0x00211571, 0x00211971, 0x00211d71, // 10304-10311
	0x00212171, 0x00212571, 0x00212971, 0x00212d71, 0x00213171, 0x00213571, 0x00213971, 0x00213d71, // 10312-10319
	0x00214171, 0x00214571, 0x00214971, 0x00214d71, 0x00215171, 0x00215571, 0x00215971, 0x00215d71, // 10320-10327
	0x00216171, 0x00216571, 0x00216971, 0x00216d71, 0x00217171, 0x00217571, 0x00217971, 0x00217d71, // 10328-10335
	0x00218171, 0x00218571, 0x00218971, 0x00218d71, 0x00219171, 0x00219571, 0x00219971, 0x00219d71, // 10336-10343
	0x0021a171, 0x0021a571, 0x0021a971, 0x0021ad71, 0x0021b171, 0x0021b571, 0x0021b971, 0x0021bd71, // 10344-10351
	0x0021c171, 0x0021c571, 0x0021c971, 0x0021cd71, 0x0021d171, 0x0021d571, 0x0021d971, 0x0021dd71, // 10352-10359
	0x0021e171, 0x0021e571, 0x0021e971, 0x0021ed71, 0x0021f171, 0x0021f571, 0x0021f971, 0x0021fd71, // 10360-10367
	0x00220171, 0x00220571, 0x00220971, 0x00220d71, 0x00221171, 0x00221571, 0x00221971, 0x00221d71, // 10368-10375
	0x00222171, 0x00222571, 0x00222971, 0x00222d71, 0x00223171, 0x00223571, 0x00223971, 0x00223d71, // 10376-10383
	0x00224171, 0x00224571, 0x00224971, 0x00224d71, 0x00225171, 0x00225571, 0x00225971, 0x00225d71, // 10384-10391
	0x00226171, 0x00226571, 0x00226971, 0x00226d71, 0x00227171, 0x00227571, 0x00227971, 0x00227d71, // 10392-10399
	0x00228171, 0x00228571, 0x00228971, 0x00228d71, 0x00229171, 0x00229571, 0x00229971, 0x00229d71, // 10400-10407
	0x0022a171, 0x0022a571, 0x0022a971, 0x0022ad71, 0x0022b171, 0x0022b571, 0x0022b971, 0x0022bd71, // 10408-10415
	0x0022c171, 0x0022c571, 0x0022c971, 0x0022cd71, 0x0022d171, 0x0022d571, 0x0022d971, 0x0022dd71, // 10416-10423
	0x0022e171, 0x0022e571, 0x0022e971, 0x0022ed71, 0x0022f171, 0x0022f571, 0x0022f971, 0x0022fd71, // 10424-10431
	0x00230171, 0x00230571, 0x00230971, 0x00230d71, 0x00231171, 0x00231571, 0x00231971, 0x00231d71, // 10432-10439
	0x00232171, 0x00232571, 0x00232971, 0x00232d71, 0x00233171, 0x00233571, 0x00233971, 0x00233d71, // 10440-10447
	0x00234171, 0x00234571, 0x00234971, 0x00234d71, 0x00235171, 0x00235571, 0x00235971, 0x00235d71, // 10448-10455
	0x00236171, 0x00236571, 0x00236971, 0x00236d71, 0x00237171, 0x00237571, 0x00237971, 0x00237d71, // 10456-10463
	0x00238171, 0x00238571, 0x00238971, 0x00238d71, 0x00239171, 0x00239571, 0x00239971, 0x00239d71, // 10464-10471
	0x0023a171, 0x0023a571, 0x0023a971, 0x0023ad71, 0x0023b171, 0x0023b571, 0x0023b971, 0x0023bd71, // 10472-10479
	0x0023c171, 0x0023c571, 0x0023c971, 0x0023cd71, 0x0023d171, 0x0023d571, 0x0023d971, 0x0023dd71, // 10480-10487
	0x0023e171, 0x0023e571, 0x0023e971, 0x0023ed71, 0x0023f171, 0x0023f571, 0x0023f971, 0x0023fd71, // 10488-10495
	0x00240171, 0x00240571, 0x00240971, 0x00240d71, 0x00241171, 0x00241571, 0x00241971, 0x00241d71, // 10496-10503
	0x00242171, 0x00242571, 0x00242971, 0x00242d71, 0x00243171, 0x00243571, 0x00243971, 0x00243d71, // 10504-10511
	0x00244171, 0x00244571, 0x00244971, 0x00244d71, 0x00245171, 0x00245571, 0x00245971, 0x00245d71, // 10512-10519
	0x00246171, 0x00246571, 0x00246971, 0x00246d71, 0x00247171, 0x00247571, 0x00247971, 0x00247d71, // 10520-10527
	0x00248171, 0x00248571, 0x00248971, 0x00248d71, 0x00249171, 0x00249571, 0x00249971, 0x00249d71, // 10528-10535
	0x0024a171, 0x0024a571, 0x0024a971, 0x0024ad71, 0x0024b171, 0x0024b571, 0x0024b971, 0x0024bd71, // 10536-10543
	0x0024c171, 0x0024c571, 0x0024c971, 0x0024cd71, 0x0024d171, 0x0024d571, 0x0024d971, 0x0024dd71, // 10544-10551
	0x0024e171, 0x0024e571, 0x0024e971, 0x0024ed71, 0x0024f171, 0x0024f571, 0x0024f971, 0x0024fd71, // 10552-10559
	0x00250171, 0x00250571, 0x00250971, 0x00250d71, 0x00251171, 0x00251571, 0x00251971, 0x00251d71, // 10560-10567
	0x00252171, 0x00252571, 0x00252971, 0x00252d71, 0x00253171, 0x00253571, 0x00253971, 0x00253d71, // 10568-10575
	0x00254171, 0x00254571, 0x00254971, 0x00254d71, 0x00255171, 0x00255571, 0x00255971, 0x00255d71, // 10576-10583
	0x00256171, 0x00256571, 0x00256971, 0x00256d71, 0x00257171, 0x00257571, 0x00257971, 0x00257d71, // 10584-10591
	0x00258171, 0x00258571, 0x00258971, 0x00258d71, 0x00259171, 0x00259571, 0x00259971, 0x00259d71, // 10592-10599
	0x0025a171, 0x0025a571, 0x0025a971, 0x0025ad71, 0x0025b171, 0x0025b571, 0x0025b971, 0x0025bd71, // 10600-10607
	0x0025c171, 0x0025c571, 0x0025c971, 0x0025cd71, 0x0025d171, 0x0025d571, 0x0025d971, 0x0025dd71, // 10608-10615
	0x0025e171, 0x0025e571, 0x0025e971, 0x0025ed71, 0x0025f171, 0x0025f571, 0x0025f971, 0x0025fd71, // 10616-10623
	0x00260171, 0x00260571, 0x00260971, 0x00260d71, 0x00261171, 0x00261571, 0x00261971, 0x00261d71, // 10624-10631
	0x00262171, 0x00262571, 0x00262971, 0x00262d71, 0x00263171, 0x00263571, 0x00263971, 0x00263d71, // 10632-10639
	0x00264171, 0x00264571, 0x00264971, 0x00264d71, 0x00265171, 0x00265571, 0x00265971, 0x00265d71, // 10640-10647
	0x00266171, 0x00266571, 0x00266971, 0x00266d71, 0x00267171, 0x00267571, 0x00267971, 0x00267d71, // 10648-10655
	0x00268171, 0x00268571, 0x00268971, 0x00268d71, 0x00269171, 0x00269571, 0x00269971, 0x00269d71, // 10656-10663
	0x0026a171, 0x0026a571, 0x0026a971, 0x0026ad71, 0x0026b171, 0x0026b571, 0x0026b971, 0x0026bd71, // 10664-10671
	0x0026c171, 0x0026c571, 0x0026c971, 0x0026cd71, 0x0026d171, 0x0026d571, 0x0026d971, 0x0026dd71, // 10672-10679
	0x0026e171, 0x0026e571, 0x0026e971, 0x0026ed71, 0x0026f171, 0x0026f571, 0x0026f971, 0x0026fd71, // 10680-10687
	0x00270171, 0x00270571, 0x00270971, 0x00270d71, 0x00271171, 0x00271571, 0x00271971, 0x00271d71, // 10688-10695
	0x00272171, 0x00272571, 0x00272971, 0x00272d71, 0x00273171, 0x00273571, 0x00273971, 0x00273d71, // 10696-10703
	0x00274171, 0x00274571, 0x00274971, 0x00274d71, 0x00275171, 0x00275571, 0x00275971, 0x00275d71, // 10704-10711
	0x00276171, 0x00276571, 0x00276971, 0x00276d71, 0x00277171, 0x00277571, 0x00277971, 0x00277d71, // 10712-10719
	0x00278171, 0x00278571, 0x00278971, 0x00278d71, 0x00279171, 0x00279571, 0x00279971, 0x00279d71, // 10720-10727
	0x0027a171, 0x0027a571, 0x0027a971, 0x0027ad71, 0x0027b171, 0x0027b571, 0x0027b971, 0x0027bd71, // 10728-10735
	0x0027c171, 0x0027c571, 0x0027c971, 0x0027cd71, 0x0027d171, 0x0027d571, 0x0027d971, 0x0027dd71, // 10736-10743
	0x0027e171, 0x0027e571, 0x0027e971, 0x0027ed71, 0x0027f171, 0x0027f571, 0x0027f971, 0x0027fd71, // 10744-10751
	0x00280171, 0x00280571, 0x00280971, 0x00280d71, 0x00281171, 0x00281571, 0x00281971, 0x00281d71, // 10752-10759
	0x00282171, 0x00282571, 0x00282971, 0x00282d71, 0x00283171, 0x00283571, 0x00283971, 0x00283d71, // 10760-10767
	0x00284171, 0x00284571, 0x00284971, 0x00284d71, 0x00285171, 0x00285571, 0x00285971, 0x00285d71, // 10768-10775
	0x00286171, 0x00286571, 0x00286971, 0x00286d71, 0x00287171, 0x00287571, 0x00287971, 0x00287d71, // 10776-10783
	0x00288171, 0x00288571, 0x00288971, 0x00288d71, 0x00289171, 0x00289571, 0x00289971, 0x00289d71, // 10784-10791
	0x0028a171, 0x0028a571, 0x0028a971, 0x0028ad71, 0x0028b171, 0x0028b571, 0x0028b971, 0x0028bd71, // 10792-10799
	0x0028c171, 0x0028c571, 0x0028c971, 0x0028cd71, 0x0028d171, 0x0028d571, 0x0028d971, 0x0028dd71, // 10800-10807
	0x0028e171, 0x0028e571, 0x0028e971, 0x0028ed71, 0x0028f171, 0x0028f571, 0x0028f971, 0x0028fd71, // 10808-10815
	0x00290171, 0x00290571, 0x00290971, 0x00290d71, 0x00291171, 0x00291571, 0x00291971, 0x00291d71, // 10816-10823
	0x00292171, 0x00292571, 0x00292971, 0x00292d71, 0x00293171, 0x00293571, 0x00293971, 0x00293d71, // 10824-10831
	0x00294171, 0x00294571, 0x00294971, 0x00294d71, 0x00295171, 0x00295571, 0x00295971, 0x00295d71, // 10832-10839
	0x00296171, 0x00296571, 0x00296971, 0x00296d71, 0x00297171, 0x00297571, 0x00297971, 0x00297d71, // 10840-10847
	0x00298171, 0x00298571, 0x00298971, 0x00298d71, 0x00299171, 0x00299571, 0x00299971, 0x00299d71, // 10848-10855
	0x0029a171, 0x0029a571, 0x0029a971, 0x0029ad71, 0x0029b171, 0x0029b571, 0x0029b971, 0x0029bd71, // 10856-10863
	0x0029c171, 0x0029c571, 0x0029c971, 0x0029cd71, 0x0029d171, 0x0029d571, 0x0029d971, 0x0029dd71, // 10864-10871
	0x0029e171, 0x0029e571, 0x0029e971, 0x0029ed71, 0x0029f171, 0x0029f571, 0x0029f971, 0x0029fd71, // 10872-10879
	0x002a0171, 0x002a0571, 0x002a0971, 0x002a0d71, 0x002a1171, 0x002a1571, 0x002a1971, 0x002a1d71, // 10880-10887
	0x002a2171, 0x002a2571, 0x002a2971, 0x002a2d71, 0x002a3171, 0x002a3571, 0x002a3971, 0x002a3d71, // 10888-10895
	0x002a4171, 0x002a4571, 0x002a4971, 0x002a4d71, 0x002a5171, 0x002a5571, 0x002a5971, 0x002a5d71, // 10896-10903
	0x002a6171, 0x002a6571, 0x002a6971, 0x002a6d71, 0x002a7171, 0x002a7571, 0x002a7971, 0x002a7d71, // 10904-10911
	0x002a8171, 0x002a8571, 0x002a8971, 0x002a8d71, 0x002a9171, 0x002a9571, 0x002a9971, 0x002a9d71, // 10912-10919
	0x002aa171, 0x002aa571, 0x002aa971, 0x002aad71, 0x002ab171, 0x002ab571, 0x002ab971, 0x002abd71, // 10920-10927
	0x002ac171, 0x002ac571, 0x002ac971, 0x002acd71, 0x002ad171, 0x002ad571, 0x002ad971, 0x002add71, // 10928-10935
	0x002ae171, 0x002ae571, 0x002ae971, 0x002aed71, 0x002af171, 0x002af571, 0x002af971, 0x002afd71, // 10936-10943
	0x002b0171, 0x002b0571, 0x002b0971, 0x002b0d71, 0x002b1171, 0x002b1571, 0x002b1971, 0x002b1d71, // 10944-10951
	0x002b2171, 0x002b2571, 0x002b2971, 0x002b2d71, 0x002b3171, 0x002b3571, 0x002b3971, 0x002b3d71, // 10952-10959
	0x002b4171, 0x002b4571, 0x002b4971, 0x002b4d71, 0x002b5171, 0x002b5571, 0x002b5971, 0x002b5d71, // 10960-10967
	0x002b6171, 0x002b6571, 0x002b6971, 0x002b6d71, 0x002b7171, 0x002b7571, 0x002b7971, 0x002b7d71, // 10968-10975
	0x002b8171, 0x002b8571, 0x002b8971, 0x002b8d71, 0x002b9171, 0x002b9571, 0x002b9971, 0x002b9d71, // 10976-10983
	0x002ba171, 0x002ba571, 0x002ba971, 0x002bad71, 0x002bb171, 0x002bb571, 0x002bb971, 0x002bbd71, // 10984-10991
	0x002bc171, 0x002bc571, 0x002bc971, 0x002bcd71, 0x002bd171, 0x002bd571, 0x002bd971, 0x002bdd71, // 10992-10999
	0x002be171, 0x002be571, 0x002be971, 0x002bed71, 0x002bf171, 0x002bf571, 0x002bf971, 0x002bfd71, // 11000-11007
	0x002c0171, 0x002c0571, 0x002c0971, 0x002c0d71, 0x002c1171, 0x002c1571, 0x002c1971, 0x002c1d71, // 11008-11015
	0x002c2171, 0x002c2571, 0x002c2971, 0x002c2d71, 0x002c3171, 0x002c3571, 0x002c3971, 0x002c3d71, // 11016-11023
	0x002c4171, 0x002c4571, 0x002c4971, 0x002c4d71, 0x002c5171, 0x002c5571, 0x002c5971, 0x002c5d71, // 11024-11031
	0x002c6171, 0x002c6571, 0x002c6971, 0x002c6d71, 0x002c7171, 0x002c7571, 0x002c7971, 0x002c7d71, // 11032-11039
	0x002c8171, 0x002c8571, 0x002c8971, 0x002c8d71, 0x002c9171, 0x002c9571, 0x002c9971, 0x002c9d71, // 11040-11047
	0x002ca171, 0x002ca571, 0x002ca971, 0x002cad71, 0x002cb171, 0x002cb571, 0x002cb971, 0x002cbd71, // 11048-11055
	0x002cc171, 0x002cc571, 0x002cc971, 0x002ccd71, 0x002cd171, 0x002cd571, 0x002cd971, 0x002cdd71, // 11056-11063
	0x002ce171, 0x002ce571, 0x002ce971, 0x002ced71, 0x002cf171, 0x002cf571, 0x002cf971, 0x002cfd71, // 11064-11071
	0x002d0171, 0x002d0571, 0x002d0971, 0x002d0d71, 0x002d1171, 0x002d1571, 0x002d1971, 0x002d1d71, // 11072-11079
	0x002d2171, 0x002d2571, 0x002d2971, 0x002d2d71, 0x002d3171, 0x002d3571, 0x002d3971, 0x002d3d71, // 11080-11087
	0x002d4171, 0x002d4571, 0x002d4971, 0x002d4d71, 0x002d5171, 0x002d5571, 0x002d5971, 0x002d5d71, // 11088-11095
	0x002d6171, 0x002d6571, 0x002d6971, 0x002d6d71, 0x002d7171, 0x002d7571, 0x002d7971, 0x002d7d71, // 11096-11103
	0x002d8171, 0x002d8571, 0x002d8971, 0x002d8d71, 0x002d9171, 0x002d9571, 0x002d9971, 0x002d9d71, // 11104-11111
	0x002da171, 0x002da571, 0x002da971, 0x002dad71, 0x002db171, 0x002db571, 0x002db971, 0x002dbd71, // 11112-11119
	0x002dc171, 0x002dc571, 0x002dc971, 0x002dcd71, 0x002dd171, 0x002dd571, 0x002dd971, 0x002ddd71, // 11120-11127
	0x002de171, 0x002de571, 0x002de971, 0x002ded71, 0x002df171, 0x002df571, 0x002df971, 0x002dfd71, // 11128-11135
	0x002e0171, 0x002e0571, 0x002e0971, 0x002e0d71, 0x002e1171, 0x002e1571, 0x002e1971, 0x002e1d71, // 11136-11143
	0x002e2171, 0x002e2571, 0x002e2971, 0x002e2d71, 0x002e3171, 0x002e3571, 0x002e3971, 0x002e3d71, // 11144-11151
	0x002e4171, 0x002e4571, 0x002e4971, 0x002e4d71, 0x002e5171, 0x002e5571, 0x002e5971, 0x002e5d71, // 11152-11159
	0x002e6171, 0x002e6571, 0x002e6971, 0x002e6d71, 0x002e7171, 0x002e7571, 0x002e7971, 0x002e7d71, // 11160-11167
	0x002e8171, 0x002e8571, 0x002e8971, 0x002e8d71, 0x002e9171, 0x002e9571, 0x002e9971, 0x002e9d71, // 11168-11175
	0x002ea171, 0x002ea571, 0x002ea971, 0x002ead71, 0x002eb171, 0x002eb571, 0x002eb971, 0x002ebd71, // 11176-11183
	0x002ec171, 0x002ec571, 0x002ec971, 0x002ecd71, 0x002ed171, 0x002ed571, 0x002ed971, 0x002edd71, // 11184-11191
	0x002ee171, 0x002ee571, 0x002ee971, 0x002eed71, 0x002ef171, 0x002ef571, 0x002ef971, 0x002efd71, // 11192-11199
	0x002f0171, 0x002f0571, 0x002f0971, 0x002f0d71, 0x002f1171, 0x002f1571, 0x002f1971, 0x002f1d71, // 11200-11207
	0x002f2171, 0x002f2571, 0x002f2971, 0x002f2d71, 0x002f3171, 0x002f3571, 0x002f3971, 0x002f3d71, // 11208-11215
	0x002f4171, 0x002f4571, 0x002f4971, 0x002f4d71, 0x002f5171, 0x002f5571, 0x002f5971, 0x002f5d71, // 11216-11223
	0x002f6171, 0x002f6571, 0x002f6971, 0x002f6d71, 0x002f7171, 0x002f7571, 0x002f7971, 0x002f7d71, // 11224-11231
	0x002f8171, 0x002f8571, 0x002f8971, 0x002f8d71, 0x002f9171, 0x002f9571, 0x002f9971, 0x002f9d71, // 11232-11239
	0x002fa171, 0x002fa571, 0x002fa971, 0x002fad71, 0x002fb171, 0x002fb571, 0x002fb971, 0x002fbd71, // 11240-11247
	0x002fc171, 0x002fc571, 0x002fc971, 0x002fcd71, 0x002fd171, 0x002fd571, 0x002fd971, 0x002fdd71, // 11248-11255
	0x002fe171, 0x002fe571, 0x002fe971, 0x002fed71, 0x002ff171, 0x002ff571, 0x002ff971, 0x002ffd71, // 11256-11263
	0x00300171, 0x00300571, 0x00300971, 0x00300d71, 0x00301171, 0x00301571, 0x00301971, 0x00301d71, // 11264-11271
	0x00302171, 0x00302571, 0x00302971, 0x00302d71, 0x00303171, 0x00303571, 0x00303971, 0x00303d71, // 11272-11279
	0x00304171, 0x00304571, 0x00304971, 0x00304d71, 0x00305171, 0x00305571, 0x00305971, 0x00305d71, // 11280-11287
	0x00306171, 0x00306571, 0x00306971, 0x00306d71, 0x00307171, 0x00307571, 0x00307971, 0x00307d71, // 11288-11295
	0x00308171, 0x00308571, 0x00308971, 0x00308d71, 0x00309171, 0x00309571, 0x00309971, 0x00309d71, // 11296-11303
	0x0030a171, 0x0030a571, 0x0030a971, 0x0030ad71, 0x0030b171, 0x0030b571, 0x0030b971, 0x0030bd71, // 11304-11311
	0x0030c171, 0x0030c571, 0x0030c971, 0x0030cd71, 0x0030d171, 0x0030d571, 0x0030d971, 0x0030dd71, // 11312-11319
	0x0030e171, 0x0030e571, 0x0030e971, 0x0030ed71, 0x0030f171, 0x0030f571, 0x0030f971, 0x0030fd71, // 11320-11327
	0x00310171, 0x00310571, 0x00310971, 0x00310d71, 0x00311171, 0x00311571, 0x00311971, 0x00311d71, // 11328-11335
	0x00312171, 0x00312571, 0x00312971, 0x00312d71, 0x00313171, 0x00313571, 0x00313971, 0x00313d71, // 11336-11343
	0x00314171, 0x00314571, 0x00314971, 0x00314d71, 0x00315171, 0x00315571, 0x00315971, 0x00315d71, // 11344-11351
	0x00316171, 0x00316571, 0x00316971, 0x00316d71, 0x00317171, 0x00317571, 0x00317971, 0x00317d71, // 11352-11359
	0x00318171, 0x00318571, 0x00318971, 0x00318d71, 0x00319171, 0x00319571, 0x00319971, 0x00319d71, // 11360-11367
	0x0031a171, 0x0031a571, 0x0031a971, 0x0031ad71, 0x0031b171, 0x0031b571, 0x0031b971, 0x0031bd71, // 11368-11375
	0x0031c171, 0x0031c571, 0x0031c971, 0x0031cd71, 0x0031d171, 0x0031d571, 0x0031d971, 0x0031dd71, // 11376-11383
	0x0031e171, 0x0031e571, 0x0031e971, 0x0031ed71, 0x0031f171, 0x0031f571, 0x0031f971, 0x0031fd71, // 11384-11391
	0x00320171, 0x00320571, 0x00320971, 0x00320d71, 0x00321171, 0x00321571, 0x00321971, 0x00321d71, // 11392-11399
	0x00322171, 0x00322571, 0x00322971, 0x00322d71, 0x00323171, 0x00323571, 0x00323971, 0x00323d71, // 11400-11407
	0x00324171, 0x00324571, 0x00324971, 0x00324d71, 0x00325171, 0x00325571, 0x00325971, 0x00325d71, // 11408-11415
	0x00326171, 0x00326571, 0x00326971, 0x00326d71, 0x00327171, 0x00327571, 0x00327971, 0x00327d71, // 11416-11423
	0x00328171, 0x00328571, 0x00328971, 0x00328d71, 0x00329171, 0x00329571, 0x00329971, 0x00329d71, // 11424-11431
	0x0032a171, 0x0032a571, 0x0032a971, 0x0032ad71, 0x0032b171, 0x0032b571, 0x0032b971, 0x0032bd71, // 11432-11439
	0x0032c171, 0x0032c571, 0x0032c971, 0x0032cd71, 0x0032d171, 0x0032d571, 0x0032d971, 0x0032dd71, // 11440-11447
	0x0032e171, 0x0032e571, 0x0032e971, 0x0032ed71, 0x0032f171, 0x0032f571, 0x0032f971, 0x0032fd71, // 11448-11455
	0x00330171, 0x00330571, 0x00330971, 0x00330d71, 0x00331171, 0x00331571, 0x00331971, 0x00331d71, // 11456-11463
	0x00332171, 0x00332571, 0x00332971, 0x00332d71, 0x00333171, 0x00333571, 0x00333971, 0x00333d71, // 11464-11471
	0x00334171, 0x00334571, 0x00334971, 0x00334d71, 0x00335171, 0x00335571, 0x00335971, 0x00335d71, // 11472-11479
	0x00336171, 0x00336571, 0x00336971, 0x00336d71, 0x00337171, 0x00337571, 0x00337971, 0x00337d71, // 11480-11487
	0x00338171, 0x00338571, 0x00338971, 0x00338d71, 0x00339171, 0x00339571, 0x00339971, 0x00339d71, // 11488-11495
	0x0033a171, 0x0033a571, 0x0033a971, 0x0033ad71, 0x0033b171, 0x0033b571, 0x0033b971, 0x0033bd71, // 11496-11503
	0x0033c171, 0x0033c571, 0x0033c971, 0x0033cd71, 0x0033d171, 0x0033d571, 0x0033d971, 0x0033dd71, // 11504-11511
	0x0033e171, 0x0033e571, 0x0033e971, 0x0033ed71, 0x0033f171, 0x0033f571, 0x0033f971, 0x0033fd71, // 11512-11519
	0x00340171, 0x00340571, 0x00340971, 0x00340d71, 0x00341171, 0x00341571, 0x00341971, 0x00341d71, // 11520-11527
	0x00342171, 0x00342571, 0x00342971, 0x00342d71, 0x00343171, 0x00343571, 0x00343971, 0x00343d71, // 11528-11535
	0x00344171, 0x00344571, 0x00344971, 0x00344d71, 0x00345171, 0x00345571, 0x00345971, 0x00345d71, // 11536-11543
	0x00346171, 0x00346571, 0x00346971, 0x00346d71, 0x00347171, 0x00347571, 0x00347971, 0x00347d71, // 11544-11551
	0x00348171, 0x00348571, 0x00348971, 0x00348d71, 0x00349171, 0x00349571, 0x00349971, 0x00349d71, // 11552-11559
	0x0034a171, 0x0034a571, 0x0034a971, 0x0034ad71, 0x0034b171, 0x0034b571, 0x0034b971, 0x0034bd71, // 11560-11567
	0x0034c171, 0x0034c571, 0x0034c971, 0x0034cd71, 0x0034d171, 0x0034d571, 0x0034d971, 0x0034dd71, // 11568-11575
	0x0034e171, 0x0034e571, 0x0034e971, 0x0034ed71, 0x0034f171, 0x0034f571, 0x0034f971, 0x0034fd71, // 11576-11583
	0x00350171, 0x00350571, 0x00350971, 0x00350d71, 0x00351171, 0x00351571, 0x00351971, 0x00351d71, // 11584-11591
	0x00352171, 0x00352571, 0x00352971, 0x00352d71, 0x00353171, 0x00353571, 0x00353971, 0x00353d71, // 11592-11599
	0x00354171, 0x00354571, 0x00354971, 0x00354d71, 0x00355171, 0x00355571, 0x00355971, 0x00355d71, // 11600-11607
	0x00356171, 0x00356571, 0x00356971, 0x00356d71, 0x00357171, 0x00357571, 0x00357971, 0x00357d71, // 11608-11615
	0x00358171, 0x00358571, 0x00358971, 0x00358d71, 0x00359171, 0x00359571, 0x00359971, 0x00359d71, // 11616-11623
	0x0035a171, 0x0035a571, 0x0035a971, 0x0035ad71, 0x0035b171, 0x0035b571, 0x0035b971, 0x0035bd71, // 11624-11631
	0x0035c171, 0x0035c571, 0x0035c971, 0x0035cd71, 0x0035d171, 0x0035d571, 0x0035d971, 0x0035dd71, // 11632-11639
	0x0035e171, 0x0035e571, 0x0035e971, 0x0035ed71, 0x0035f171, 0x0035f571, 0x0035f971, 0x0035fd71, // 11640-11647
	0x00360171, 0x00360571, 0x00360971, 0x00360d71, 0x00361171, 0x00361571, 0x00361971, 0x00361d71, // 11648-11655
	0x00362171, 0x00362571, 0x00362971, 0x00362d71, 0x00363171, 0x00363571, 0x00363971, 0x00363d71, // 11656-11663
	0x00364171, 0x00364571, 0x00364971, 0x00364d71, 0x00365171, 0x00365571, 0x00365971, 0x00365d71, // 11664-11671
	0x00366171, 0x00366571, 0x00366971, 0x00366d71, 0x00367171, 0x00367571, 0x00367971, 0x00367d71, // 11672-11679
	0x00368171, 0x00368571, 0x00368971, 0x00368d71, 0x00369171, 0x00369571, 0x00369971, 0x00369d71, // 11680-11687
	0x0036a171, 0x0036a571, 0x0036a971, 0x0036ad71, 0x0036b171, 0x0036b571, 0x0036b971, 0x0036bd71, // 11688-11695
	0x0036c171, 0x0036c571, 0x0036c971, 0x0036cd71, 0x0036d171, 0x0036d571, 0x0036d971, 0x0036dd71, // 11696-11703
	0x0036e171, 0x0036e571, 0x0036e971, 0x0036ed71, 0x0036f171, 0x0036f571, 0x0036f971, 0x0036fd71, // 11704-11711
	0x00370171, 0x00370571, 0x00370971, 0x00370d71, 0x00371171, 0x00371571, 0x00371971, 0x00371d71, // 11712-11719
	0x00372171, 0x00372571, 0x00372971, 0x00372d71, 0x00373171, 0x00373571, 0x00373971, 0x00373d71, // 11720-11727
	0x00374171, 0x00374571, 0x00374971, 0x00374d71, 0x00375171, 0x00375571, 0x00375971, 0x00375d71, // 11728-11735
	0x00376171, 0x00376571, 0x00376971, 0x00376d71, 0x00377171, 0x00377571, 0x00377971, 0x00377d71, // 11736-11743
	0x00378171, 0x00378571, 0x00378971, 0x00378d71, 0x00379171, 0x00379571, 0x00379971, 0x00379d71, // 11744-11751
	0x0037a171, 0x0037a571, 0x0037a971, 0x0037ad71, 0x0037b171, 0x0037b571, 0x0037b971, 0x0037bd71, // 11752-11759
	0x0037c171, 0x0037c571, 0x0037c971, 0x0037cd71, 0x0037d171, 0x0037d571, 0x0037d971, 0x0037dd71, // 11760-11767
	0x0037e171, 0x0037e571, 0x0037e971, 0x0037ed71, 0x0037f171, 0x0037f571, 0x0037f971, 0x0037fd71, // 11768-11775
	0x00380171, 0x00380571, 0x00380971, 0x00380d71, 0x00381171, 0x00381571, 0x00381971, 0x00381d71, // 11776-11783
	0x00382171, 0x00382571, 0x00382971, 0x00382d71, 0x00383171, 0x00383571, 0x00383971, 0x00383d71, // 11784-11791
	0x00384171, 0x00384571, 0x00384971, 0x00384d71, 0x00385171, 0x00385571, 0x00385971, 0x00385d71, // 11792-11799
	0x00386171, 0x00386571, 0x00386971, 0x00386d71, 0x00387171, 0x00387571, 0x00387971, 0x00387d71, // 11800-11807
	0x00388171, 0x00388571, 0x00388971, 0x00388d71, 0x00389171, 0x00389571, 0x00389971, 0x00389d71, // 11808-11815
	0x0038a171, 0x0038a571, 0x0038a971, 0x0038ad71, 0x0038b171, 0x0038b571, 0x0038b971, 0x0038bd71, // 11816-11823
	0x0038c171, 0x0038c571, 0x0038c971, 0x0038cd71, 0x0038d171, 0x0038d571, 0x0038d971, 0x0038dd71, // 11824-11831
	0x0038e171, 0x0038e571, 0x0038e971, 0x0038ed71, 0x0038f171, 0x0038f571, 0x0038f971, 0x0038fd71, // 11832-11839
	0x00390171, 0x00390571, 0x00390971, 0x00390d71, 0x00391171, 0x00391571, 0x00391971, 0x00391d71, // 11840-11847
	0x00392171, 0x00392571, 0x00392971, 0x00392d71, 0x00393171, 0x00393571, 0x00393971, 0x00393d71, // 11848-11855
	0x00394171, 0x00394571, 0x00394971, 0x00394d71, 0x00395171, 0x00395571, 0x00395971, 0x00395d71, // 11856-11863
	0x00396171, 0x00396571, 0x00396971, 0x00396d71, 0x00397171, 0x00397571, 0x00397971, 0x00397d71, // 11864-11871
	0x00398171, 0x00398571, 0x00398971, 0x00398d71, 0x00399171, 0x00399571, 0x00399971, 0x00399d71, // 11872-11879
	0x0039a171, 0x0039a571, 0x0039a971, 0x0039ad71, 0x0039b171, 0x0039b571, 0x0039b971, 0x0039bd71, // 11880-11887
	0x0039c171, 0x0039c571, 0x0039c971, 0x0039cd71, 0x0039d171, 0x0039d571, 0x0039d971, 0x0039dd71, // 11888-11895
	0x0039e171, 0x0039e571, 0x0039e971, 0x0039ed71, 0x0039f171, 0x0039f571, 0x0039f971, 0x0039fd71, // 11896-11903
	0x003a0171, 0x003a0571, 0x003a0971, 0x003a0d71, 0x003a1171, 0x003a1571, 0x003a1971, 0x003a1d71, // 11904-11911
	0x003a2171, 0x003a2571, 0x003a2971, 0x003a2d71, 0x003a3171, 0x003a3571, 0x003a3971, 0x003a3d71, // 11912-11919
	0x003a4171, 0x003a4571, 0x003a4971, 0x003a4d71, 0x003a5171, 0x003a5571, 0x003a5971, 0x003a5d71, // 11920-11927
	0x003a6171, 0x003a6571, 0x003a6971, 0x003a6d71, 0x003a7171, 0x003a7571, 0x003a7971, 0x003a7d71, // 11928-11935
	0x003a8171, 0x003a8571, 0x003a8971, 0x003a8d71, 0x003a9171, 0x003a9571, 0x003a9971, 0x003a9d71, // 11936-11943
	0x003aa171, 0x003aa571, 0x003aa971, 0x003aad71, 0x003ab171, 0x003ab571, 0x003ab971, 0x003abd71, // 11944-11951
	0x003ac171, 0x003ac571, 0x003ac971, 0x003acd71, 0x003ad171, 0x003ad571, 0x003ad971, 0x003add71, // 11952-11959
	0x003ae171, 0x003ae571, 0x003ae971, 0x003aed71, 0x003af171, 0x003af571, 0x003af971, 0x003afd71, // 11960-11967
	0x003b0171, 0x003b0571, 0x003b0971, 0x003b0d71, 0x003b1171, 0x003b1571, 0x003b1971, 0x003b1d71, // 11968-11975
	0x003b2171, 0x003b2571, 0x003b2971, 0x003b2d71, 0x003b3171, 0x003b3571, 0x003b3971, 0x003b3d71, // 11976-11983
	0x003b4171, 0x003b4571, 0x003b4971, 0x003b4d71, 0x003b5171, 0x003b5571, 0x003b5971, 0x003b5d71, // 11984-11991
	0x003b6171, 0x003b6571, 0x003b6971, 0x003b6d71, 0x003b7171, 0x003b7571, 0x003b7971, 0x003b7d71, // 11992-11999
	0x003b8171, 0x003b8571, 0x003b8971, 0x003b8d71, 0x003b9171, 0x003b9571, 0x003b9971, 0x003b9d71, // 12000-12007
	0x003ba171, 0x003ba571, 0x003ba971, 0x003bad71, 0x003bb171, 0x003bb571, 0x003bb971, 0x003bbd71, // 12008-12015
	0x003bc171, 0x003bc571, 0x003bc971, 0x003bcd71, 0x003bd171, 0x003bd571, 0x003bd971, 0x003bdd71, // 12016-12023
	0x003be171, 0x003be571, 0x003be971, 0x003bed71, 0x003bf171, 0x003bf571, 0x003bf971, 0x003bfd71, // 12024-12031
	0x003c0171, 0x003c0571, 0x003c0971, 0x003c0d71, 0x003c1171, 0x003c1571, 0x003c1971, 0x003c1d71, // 12032-12039
	0x003c2171, 0x003c2571, 0x003c2971, 0x003c2d71, 0x003c3171, 0x003c3571, 0x003c3971, 0x003c3d71, // 12040-12047
	0x003c4171, 0x003c4571, 0x003c4971, 0x003c4d71, 0x003c5171, 0x003c5571, 0x003c5971, 0x003c5d71, // 12048-12055
	0x003c6171, 0x003c6571, 0x003c6971, 0x003c6d71, 0x003c7171, 0x003c7571, 0x003c7971, 0x003c7d71, // 12056-12063
	0x003c8171, 0x003c8571, 0x003c8971, 0x003c8d71, 0x003c9171, 0x003c9571, 0x003c9971, 0x003c9d71, // 12064-12071
	0x003ca171, 0x003ca571, 0x003ca971, 0x003cad71, 0x003cb171, 0x003cb571, 0x003cb971, 0x003cbd71, // 12072-12079
	0x003cc171, 0x003cc571, 0x003cc971, 0x003ccd71, 0x003cd171, 0x003cd571, 0x003cd971, 0x003cdd71, // 12080-12087
	0x003ce171, 0x003ce571, 0x003ce971, 0x003ced71, 0x003cf171, 0x003cf571, 0x003cf971, 0x003cfd71, // 12088-12095
	0x003d0171, 0x003d0571, 0x003d0971, 0x003d0d71, 0x003d1171, 0x003d1571, 0x003d1971, 0x003d1d71, // 12096-12103
	0x003d2171, 0x003d2571, 0x003d2971, 0x003d2d71, 0x003d3171, 0x003d3571, 0x003d3971, 0x003d3d71, // 12104-12111
	0x003d4171, 0x003d4571, 0x003d4971, 0x003d4d71, 0x003d5171, 0x003d5571, 0x003d5971, 0x003d5d71, // 12112-12119
	0x003d6171, 0x003d6571, 0x003d6971, 0x003d6d71, 0x003d7171, 0x003d7571, 0x003d7971, 0x003d7d71, // 12120-12127
	0x003d8171, 0x003d8571, 0x003d8971, 0x003d8d71, 0x003d9171, 0x003d9571, 0x003d9971, 0x003d9d71, // 12128-12135
	0x003da171, 0x003da571, 0x003da971, 0x003dad71, 0x003db171, 0x003db571, 0x003db971, 0x003dbd71, // 12136-12143
	0x003dc171, 0x003dc571, 0x003dc971, 0x003dcd71, 0x003dd171, 0x003dd571, 0x003dd971, 0x003ddd71, // 12144-12151
	0x003de171, 0x003de571, 0x003de971, 0x003ded71, 0x003df171, 0x003df571, 0x003df971, 0x003dfd71, // 12152-12159
	0x003e0171, 0x003e0571, 0x003e0971, 0x003e0d71, 0x003e1171, 0x003e1571, 0x003e1971, 0x003e1d71, // 12160-12167
	0x003e2171, 0x003e2571, 0x003e2971, 0x003e2d71, 0x003e3171, 0x003e3571, 0x003e3971, 0x003e3d71, // 12168-12175
	0x003e4171, 0x003e4571, 0x003e4971, 0x003e4d71, 0x003e5171, 0x003e5571, 0x003e5971, 0x003e5d71, // 12176-12183
	0x003e6171, 0x003e6571, 0x003e6971, 0x003e6d71, 0x003e7171, 0x003e7571, 0x003e7971, 0x003e7d71, // 12184-12191
	0x003e8171, 0x003e8571, 0x003e8971, 0x003e8d71, 0x003e9171, 0x003e9571, 0x003e9971, 0x003e9d71, // 12192-12199
	0x003ea171, 0x003ea571, 0x003ea971, 0x003ead71, 0x003eb171, 0x003eb571, 0x003eb971, 0x003ebd71, // 12200-12207
	0x003ec171, 0x003ec571, 0x003ec971, 0x003ecd71, 0x003ed171, 0x003ed571, 0x003ed971, 0x003edd71, // 12208-12215
	0x003ee171, 0x003ee571, 0x003ee971, 0x003eed71, 0x003ef171, 0x003ef571, 0x003ef971, 0x003efd71, // 12216-12223
	0x003f0171, 0x003f0571, 0x003f0971, 0x003f0d71, 0x003f1171, 0x003f1571, 0x003f1971, 0x003f1d71, // 12224-12231
	0x003f2171, 0x003f2571, 0x003f2971, 0x003f2d71, 0x003f3171, 0x003f3571, 0x003f3971, 0x003f3d71, // 12232-12239
	0x003f4171, 0x003f4571, 0x003f4971, 0x003f4d71, 0x003f5171, 0x003f5571, 0x003f5971, 0x003f5d71, // 12240-12247
	0x003f6171, 0x003f6571, 0x003f6971, 0x003f6d71, 0x003f7171, 0x003f7571, 0x003f7971, 0x003f7d71, // 12248-12255
	0x003f8171, 0x003f8571, 0x003f8971, 0x003f8d71, 0x003f9171, 0x003f9571, 0x003f9971, 0x003f9d71, // 12256-12263
	0x003fa171, 0x003fa571, 0x003fa971, 0x003fad71, 0x003fb171, 0x003fb571, 0x003fb971, 0x003fbd71, // 12264-12271
	0x003fc171, 0x003fc571, 0x003fc971, 0x003fcd71, 0x003fd171, 0x003fd571, 0x003fd971, 0x003fdd71, // 12272-12279
	0x003fe171, 0x003fe571, 0x003fe971, 0x003fed71, 0x003ff171, 0x003ff571, 0x003ff971, 0x003ffd71, // 12280-12287
	0x00000371, 0x00000771, 0x00000b71, 0x00000f71, 0x00001371, 0x00001771, 0x00001b71, 0x00001f71, // 12288-12295
	0x00002371, 0x00002771, 0x00002b71, 0x00002f71, 0x00003371, 0x00003771, 0x00003b71, 0x00003f71, // 12296-12303
	0x00004371, 0x00004771, 0x00004b71, 0x00004f71, 0x00005371, 0x00005771, 0x00005b71, 0x00005f71, // 12304-12311
	0x00006371, 0x00006771, 0x00006b71, 0x00006f71, 0x00007371, 0x00007771, 0x00007b71, 0x00007f71, // 12312-12319
	0x00008371, 0x00008771, 0x00008b71, 0x00008f71, 0x00009371, 0x00009771, 0x00009b71, 0x00009f71, // 12320-12327
	0x0000a371, 0x0000a771, 0x0000ab71, 0x0000af71, 0x0000b371, 0x0000b771, 0x0000bb71, 0x0000bf71, // 12328-12335
	0x0000c371, 0x0000c771, 0x0000cb71, 0x0000cf71, 0x0000d371, 0x0000d771, 0x0000db71, 0x0000df71, // 12336-12343
	0x0000e371, 0x0000e771, 0x0000eb71, 0x0000ef71, 0x0000f371, 0x0000f771, 0x0000fb71, 0x0000ff71, // 12344-12351
	0x00010371, 0x00010771, 0x00010b71, 0x00010f71, 0x00011371, 0x00011771, 0x00011b71, 0x00011f71, // 12352-12359
	0x00012371, 0x00012771, 0x00012b71, 0x00012f71, 0x00013371, 0x00013771, 0x00013b71, 0x00013f71, // 12360-12367
	0x00014371, 0x00014771, 0x00014b71, 0x00014f71, 0x00015371, 0x00015771, 0x00015b71, 0x00015f71, // 12368-12375
	0x00016371, 0x00016771, 0x00016b71, 0x00016f71, 0x00017371, 0x00017771, 0x00017b71, 0x00017f71, // 12376-12383
	0x00018371, 0x00018771, 0x00018b71, 0x00018f71, 0x00019371, 0x00019771, 0x00019b71, 0x00019f71, // 12384-12391
	0x0001a371, 0x0001a771, 0x0001ab71, 0x0001af71, 0x0001b371, 0x0001b771, 0x0001bb71, 0x0001bf71, // 12392-12399
	0x0001c371, 0x0001c771, 0x0001cb71, 0x0001cf71, 0x0001d371, 0x0001d771, 0x0001db71, 0x0001df71, // 12400-12407
	0x0001e371, 0x0001e771, 0x0001eb71, 0x0001ef71, 0x0001f371, 0x0001f771, 0x0001fb71, 0x0001ff71, // 12408-12415
	0x00020371, 0x00020771, 0x00020b71, 0x00020f71, 0x00021371, 0x00021771, 0x00021b71, 0x00021f71, // 12416-12423
	0x00022371, 0x00022771, 0x00022b71, 0x00022f71, 0x00023371, 0x00023771, 0x00023b71, 0x00023f71, // 12424-12431
	0x00024371, 0x00024771, 0x00024b71, 0x00024f71, 0x00025371, 0x00025771, 0x00025b71, 0x00025f71, // 12432-12439
	0x00026371, 0x00026771, 0x00026b71, 0x00026f71, 0x00027371, 0x00027771, 0x00027b71, 0x00027f71, // 12440-12447
	0x00028371, 0x00028771, 0x00028b71, 0x00028f71, 0x00029371, 0x00029771, 0x00029b71, 0x00029f71, // 12448-12455
	0x0002a371, 0x0002a771, 0x0002ab71, 0x0002af71, 0x0002b371, 0x0002b771, 0x0002bb71, 0x0002bf71, // 12456-12463
	0x0002c371, 0x0002c771, 0x0002cb71, 0x0002cf71, 0x0002d371, 0x0002d771, 0x0002db71, 0x0002df71, // 12464-12471
	0x0002e371, 0x0002e771, 0x0002eb71, 0x0002ef71, 0x0002f371, 0x0002f771, 0x0002fb71, 0x0002ff71, // 12472-12479
	0x00030371, 0x00030771, 0x00030b71, 0x00030f71, 0x00031371, 0x00031771, 0x00031b71, 0x00031f71, // 12480-12487
	0x00032371, 0x00032771, 0x00032b71, 0x00032f71, 0x00033371, 0x00033771, 0x00033b71, 0x00033f71, // 12488-12495
	0x00034371, 0x00034771, 0x00034b71, 0x00034f71, 0x00035371, 0x00035771, 0x00035b71, 0x00035f71, // 12496-12503
	0x00036371, 0x00036771, 0x00036b71, 0x00036f71, 0x00037371, 0x00037771, 0x00037b71, 0x00037f71, // 12504-12511
	0x00038371, 0x00038771, 0x00038b71, 0x00038f71, 0x00039371, 0x00039771, 0x00039b71, 0x00039f71, // 12512-12519
	0x0003a371, 0x0003a771, 0x0003ab71, 0x0003af71, 0x0003b371, 0x0003b771, 0x0003bb71, 0x0003bf71, // 12520-12527
	0x0003c371, 0x0003c771, 0x0003cb71, 0x0003cf71, 0x0003d371, 0x0003d771, 0x0003db71, 0x0003df71, // 12528-12535
	0x0003e371, 0x0003e771, 0x0003eb71, 0x0003ef71, 0x0003f371, 0x0003f771, 0x0003fb71, 0x0003ff71, // 12536-12543
	0x00040371, 0x00040771, 0x00040b71, 0x00040f71, 0x00041371, 0x00041771, 0x00041b71, 0x00041f71, // 12544-12551
	0x00042371, 0x00042771, 0x00042b71, 0x00042f71, 0x00043371, 0x00043771, 0x00043b71, 0x00043f71, // 12552-12559
	0x00044371, 0x00044771, 0x00044b71, 0x00044f71, 0x00045371, 0x00045771, 0x00045b71, 0x00045f71, // 12560-12567
	0x00046371, 0x00046771, 0x00046b71, 0x00046f71, 0x00047371, 0x00047771, 0x00047b71, 0x00047f71, // 12568-12575
	0x00048371, 0x00048771, 0x00048b71, 0x00048f71, 0x00049371, 0x00049771, 0x00049b71, 0x00049f71, // 12576-12583
	0x0004a371, 0x0004a771, 0x0004ab71, 0x0004af71, 0x0004b371, 0x0004b771, 0x0004bb71, 0x0004bf71, // 12584-12591
	0x0004c371, 0x0004c771, 0x0004cb71, 0x0004cf71, 0x0004d371, 0x0004d771, 0x0004db71, 0x0004df71, // 12592-12599
	0x0004e371, 0x0004e771, 0x0004eb71, 0x0004ef71, 0x0004f371, 0x0004f771, 0x0004fb71, 0x0004ff71, // 12600-12607
	0x00050371, 0x00050771, 0x00050b71, 0x00050f71, 0x00051371, 0x00051771, 0x00051b71, 0x00051f71, // 12608-12615
	0x00052371, 0x00052771, 0x00052b71, 0x00052f71, 0x00053371, 0x00053771, 0x00053b71, 0x00053f71, // 12616-12623
	0x00054371, 0x00054771, 0x00054b71, 0x00054f71, 0x00055371, 0x00055771, 0x00055b71, 0x00055f71, // 12624-12631
	0x00056371, 0x00056771, 0x00056b71, 0x00056f71, 0x00057371, 0x00057771, 0x00057b71, 0x00057f71, // 12632-12639
	0x00058371, 0x00058771, 0x00058b71, 0x00058f71, 0x00059371, 0x00059771, 0x00059b71, 0x00059f71, // 12640-12647
	0x0005a371, 0x0005a771, 0x0005ab71, 0x0005af71, 0x0005b371, 0x0005b771, 0x0005bb71, 0x0005bf71, // 12648-12655
	0x0005c371, 0x0005c771, 0x0005cb71, 0x0005cf71, 0x0005d371, 0x0005d771, 0x0005db71, 0x0005df71, // 12656-12663
	0x0005e371, 0x0005e771, 0x0005eb71, 0x0005ef71, 0x0005f371, 0x0005f771, 0x0005fb71, 0x0005ff71, // 12664-12671
	0x00060371, 0x00060771, 0x00060b71, 0x00060f71, 0x00061371, 0x00061771, 0x00061b71, 0x00061f71, // 12672-12679
	0x00062371, 0x00062771, 0x00062b71, 0x00062f71, 0x00063371, 0x00063771, 0x00063b71, 0x00063f71, // 12680-12687
	0x00064371, 0x00064771, 0x00064b71, 0x00064f71, 0x00065371, 0x00065771, 0x00065b71, 0x00065f71, // 12688-12695
	0x00066371, 0x00066771, 0x00066b71, 0x00066f71, 0x00067371, 0x00067771, 0x00067b71, 0x00067f71, // 12696-12703
	0x00068371, 0x00068771, 0x00068b71, 0x00068f71, 0x00069371, 0x00069771, 0x00069b71, 0x00069f71, // 12704-12711
	0x0006a371, 0x0006a771, 0x0006ab71, 0x0006af71, 0x0006b371, 0x0006b771, 0x0006bb71, 0x0006bf71, // 12712-12719
	0x0006c371, 0x0006c771, 0x0006cb71, 0x0006cf71, 0x0006d371, 0x0006d771, 0x0006db71, 0x0006df71, // 12720-12727
	0x0006e371, 0x0006e771, 0x0006eb71, 0x0006ef71, 0x0006f371, 0x0006f771, 0x0006fb71, 0x0006ff71, // 12728-12735
	0x00070371, 0x00070771, 0x00070b71, 0x00070f71, 0x00071371, 0x00071771, 0x00071b71, 0x00071f71, // 12736-12743
	0x00072371, 0x00072771, 0x00072b71, 0x00072f71, 0x00073371, 0x00073771, 0x00073b71, 0x00073f71, // 12744-12751
	0x00074371, 0x00074771, 0x00074b71, 0x00074f71, 0x00075371, 0x00075771, 0x00075b71, 0x00075f71, // 12752-12759
	0x00076371, 0x00076771, 0x00076b71, 0x00076f71, 0x00077371, 0x00077771, 0x00077b71, 0x00077f71, // 12760-12767
	0x00078371, 0x00078771, 0x00078b71, 0x00078f71, 0x00079371, 0x00079771, 0x00079b71, 0x00079f71, // 12768-12775
	0x0007a371, 0x0007a771, 0x0007ab71, 0x0007af71, 0x0007b371, 0x0007b771, 0x0007bb71, 0x0007bf71, // 12776-12783
	0x0007c371, 0x0007c771, 0x0007cb71, 0x0007cf71, 0x0007d371, 0x0007d771, 0x0007db71, 0x0007df71, // 12784-12791
	0x0007e371, 0x0007e771, 0x0007eb71, 0x0007ef71, 0x0007f371, 0x0007f771, 0x0007fb71, 0x0007ff71, // 12792-12799
	0x00080371, 0x00080771, 0x00080b71, 0x00080f71, 0x00081371, 0x00081771, 0x00081b71, 0x00081f71, // 12800-12807
	0x00082371, 0x00082771, 0x00082b71, 0x00082f71, 0x00083371, 0x00083771, 0x00083b71, 0x00083f71, // 12808-12815
	0x00084371, 0x00084771, 0x00084b71, 0x00084f71, 0x00085371, 0x00085771, 0x00085b71, 0x00085f71, // 12816-12823
	0x00086371, 0x00086771, 0x00086b71, 0x00086f71, 0x00087371, 0x00087771, 0x00087b71, 0x00087f71, // 12824-12831
	0x00088371, 0x00088771, 0x00088b71, 0x00088f71, 0x00089371, 0x00089771, 0x00089b71, 0x00089f71, // 12832-12839
	0x0008a371, 0x0008a771, 0x0008ab71, 0x0008af71, 0x0008b371, 0x0008b771, 0x0008bb71, 0x0008bf71, // 12840-12847
	0x0008c371, 0x0008c771, 0x0008cb71, 0x0008cf71, 0x0008d371, 0x0008d771, 0x0008db71, 0x0008df71, // 12848-12855
	0x0008e371, 0x0008e771, 0x0008eb71, 0x0008ef71, 0x0008f371, 0x0008f771, 0x0008fb71, 0x0008ff71, // 12856-12863
	0x00090371, 0x00090771, 0x00090b71, 0x00090f71, 0x00091371, 0x00091771, 0x00091b71, 0x00091f71, // 12864-12871
	0x00092371, 0x00092771, 0x00092b71, 0x00092f71, 0x00093371, 0x00093771, 0x00093b71, 0x00093f71, // 12872-12879
	0x00094371, 0x00094771, 0x00094b71, 0x00094f71, 0x00095371, 0x00095771, 0x00095b71, 0x00095f71, // 12880-12887
	0x00096371, 0x00096771, 0x00096b71, 0x00096f71, 0x00097371, 0x00097771, 0x00097b71, 0x00097f71, // 12888-12895
	0x00098371, 0x00098771, 0x00098b71, 0x00098f71, 0x00099371, 0x00099771, 0x00099b71, 0x00099f71, // 12896-12903
	0x0009a371, 0x0009a771, 0x0009ab71, 0x0009af71, 0x0009b371, 0x0009b771, 0x0009bb71, 0x0009bf71, // 12904-12911
	0x0009c371, 0x0009c771, 0x0009cb71, 0x0009cf71, 0x0009d371, 0x0009d771, 0x0009db71, 0x0009df71, // 12912-12919
	0x0009e371, 0x0009e771, 0x0009eb71, 0x0009ef71, 0x0009f371, 0x0009f771, 0x0009fb71, 0x0009ff71, // 12920-12927
	0x000a0371, 0x000a0771, 0x000a0b71, 0x000a0f71, 0x000a1371, 0x000a1771, 0x000a1b71, 0x000a1f71, // 12928-12935
	0x000a2371, 0x000a2771, 0x000a2b71, 0x000a2f71, 0x000a3371, 0x000a3771, 0x000a3b71, 0x000a3f71, // 12936-12943
	0x000a4371, 0x000a4771, 0x000a4b71, 0x000a4f71, 0x000a5371, 0x000a5771, 0x000a5b71, 0x000a5f71, // 12944-12951
	0x000a6371, 0x000a6771, 0x000a6b71, 0x000a6f71, 0x000a7371, 0x000a7771, 0x000a7b71, 0x000a7f71, // 12952-12959
	0x000a8371, 0x000a8771, 0x000a8b71, 0x000a8f71, 0x000a9371, 0x000a9771, 0x000a9b71, 0x000a9f71, // 12960-12967
	0x000aa371, 0x000aa771, 0x000aab71, 0x000aaf71, 0x000ab371, 0x000ab771, 0x000abb71, 0x000abf71, // 12968-12975
	0x000ac371, 0x000ac771, 0x000acb71, 0x000acf71, 0x000ad371, 0x000ad771, 0x000adb71, 0x000adf71, // 12976-12983
	0x000ae371, 0x000ae771, 0x000aeb71, 0x000aef71, 0x000af371, 0x000af771, 0x000afb71, 0x000aff71, // 12984-12991
	0x000b0371, 0x000b0771, 0x000b0b71, 0x000b0f71, 0x000b1371, 0x000b1771, 0x000b1b71, 0x000b1f71, // 12992-12999
	0x000b2371, 0x000b2771, 0x000b2b71, 0x000b2f71, 0x000b3371, 0x000b3771, 0x000b3b71, 0x000b3f71, // 13000-13007
	0x000b4371, 0x000b4771, 0x000b4b71, 0x000b4f71, 0x000b5371, 0x000b5771, 0x000b5b71, 0x000b5f71, // 13008-13015
	0x000b6371, 0x000b6771, 0x000b6b71, 0x000b6f71, 0x000b7371, 0x000b7771, 0x000b7b71, 0x000b7f71, // 13016-13023
	0x000b8371, 0x000b8771, 0x000b8b71, 0x000b8f71, 0x000b9371, 0x000b9771, 0x000b9b71, 0x000b9f71, // 13024-13031
	0x000ba371, 0x000ba771, 0x000bab71, 0x000baf71, 0x000bb371, 0x000bb771, 0x000bbb71, 0x000bbf71, // 13032-13039
	0x000bc371, 0x000bc771, 0x000bcb71, 0x000bcf71, 0x000bd371, 0x000bd771, 0x000bdb71, 0x000bdf71, // 13040-13047
	0x000be371, 0x000be771, 0x000beb71, 0x000bef71, 0x000bf371, 0x000bf771, 0x000bfb71, 0x000bff71, // 13048-13055
	0x000c0371, 0x000c0771, 0x000c0b71, 0x000c0f71, 0x000c1371, 0x000c1771, 0x000c1b71, 0x000c1f71, // 13056-13063
	0x000c2371, 0x000c2771, 0x000c2b71, 0x000c2f71, 0x000c3371, 0x000c3771, 0x000c3b71, 0x000c3f71, // 13064-13071
	0x000c4371, 0x000c4771, 0x000c4b71, 0x000c4f71, 0x000c5371, 0x000c5771, 0x000c5b71, 0x000c5f71, // 13072-13079
	0x000c6371, 0x000c6771, 0x000c6b71, 0x000c6f71, 0x000c7371, 0x000c7771, 0x000c7b71, 0x000c7f71, // 13080-13087
	0x000c8371, 0x000c8771, 0x000c8b71, 0x000c8f71, 0x000c9371, 0x000c9771, 0x000c9b71, 0x000c9f71, // 13088-13095
	0x000ca371, 0x000ca771, 0x000cab71, 0x000caf71, 0x000cb371, 0x000cb771, 0x000cbb71, 0x000cbf71, // 13096-13103
	0x000cc371, 0x000cc771, 0x000ccb71, 0x000ccf71, 0x000cd371, 0x000cd771, 0x000cdb71, 0x000cdf71, // 13104-13111
	0x000ce371, 0x000ce771, 0x000ceb71, 0x000cef71, 0x000cf371, 0x000cf771, 0x000cfb71, 0x000cff71, // 13112-13119
	0x000d0371, 0x000d0771, 0x000d0b71, 0x000d0f71, 0x000d1371, 0x000d1771, 0x000d1b71, 0x000d1f71, // 13120-13127
	0x000d2371, 0x000d2771, 0x000d2b71, 0x000d2f71, 0x000d3371, 0x000d3771, 0x000d3b71, 0x000d3f71, // 13128-13135
	0x000d4371, 0x000d4771, 0x000d4b71, 0x000d4f71, 0x000d5371, 0x000d5771, 0x000d5b71, 0x000d5f71, // 13136-13143
	0x000d6371, 0x000d6771, 0x000d6b71, 0x000d6f71, 0x000d7371, 0x000d7771, 0x000d7b71, 0x000d7f71, // 13144-13151
	0x000d8371, 0x000d8771, 0x000d8b71, 0x000d8f71, 0x000d9371, 0x000d9771, 0x000d9b71, 0x000d9f71, // 13152-13159
	0x000da371, 0x000da771, 0x000dab71, 0x000daf71, 0x000db371, 0x000db771, 0x000dbb71, 0x000dbf71, // 13160-13167
	0x000dc371, 0x000dc771, 0x000dcb71, 0x000dcf71, 0x000dd371, 0x000dd771, 0x000ddb71, 0x000ddf71, // 13168-13175
	0x000de371, 0x000de771, 0x000deb71, 0x000def71, 0x000df371, 0x000df771, 0x000dfb71, 0x000dff71, // 13176-13183
	0x000e0371, 0x000e0771, 0x000e0b71, 0x000e0f71, 0x000e1371, 0x000e1771, 0x000e1b71, 0x000e1f71, // 13184-13191
	0x000e2371, 0x000e2771, 0x000e2b71, 0x000e2f71, 0x000e3371, 0x000e3771, 0x000e3b71, 0x000e3f71, // 13192-13199
	0x000e4371, 0x000e4771, 0x000e4b71, 0x000e4f71, 0x000e5371, 0x000e5771, 0x000e5b71, 0x000e5f71, // 13200-13207
	0x000e6371, 0x000e6771, 0x000e6b71, 0x000e6f71, 0x000e7371, 0x000e7771, 0x000e7b71, 0x000e7f71, // 13208-13215
	0x000e8371, 0x000e8771, 0x000e8b71, 0x000e8f71, 0x000e9371, 0x000e9771, 0x000e9b71, 0x000e9f71, // 13216-13223
	0x000ea371, 0x000ea771, 0x000eab71, 0x000eaf71, 0x000eb371, 0x000eb771, 0x000ebb71, 0x000ebf71, // 13224-13231
	0x000ec371, 0x000ec771, 0x000ecb71, 0x000ecf71, 0x000ed371, 0x000ed771, 0x000edb71, 0x000edf71, // 13232-13239
	0x000ee371, 0x000ee771, 0x000eeb71, 0x000eef71, 0x000ef371, 0x000ef771, 0x000efb71, 0x000eff71, // 13240-13247
	0x000f0371, 0x000f0771, 0x000f0b71, 0x000f0f71, 0x000f1371, 0x000f1771, 0x000f1b71, 0x000f1f71, // 13248-13255
	0x000f2371, 0x000f2771, 0x000f2b71, 0x000f2f71, 0x000f3371, 0x000f3771, 0x000f3b71, 0x000f3f71, // 13256-13263
	0x000f4371, 0x000f4771, 0x000f4b71, 0x000f4f71, 0x000f5371, 0x000f5771, 0x000f5b71, 0x000f5f71, // 13264-13271
	0x000f6371, 0x000f6771, 0x000f6b71, 0x000f6f71, 0x000f7371, 0x000f7771, 0x000f7b71, 0x000f7f71, // 13272-13279
	0x000f8371, 0x000f8771, 0x000f8b71, 0x000f8f71, 0x000f9371, 0x000f9771, 0x000f9b71, 0x000f9f71, // 13280-13287
	0x000fa371, 0x000fa771, 0x000fab71, 0x000faf71, 0x000fb371, 0x000fb771, 0x000fbb71, 0x000fbf71, // 13288-13295
	0x000fc371, 0x000fc771, 0x000fcb71, 0x000fcf71, 0x000fd371, 0x000fd771, 0x000fdb71, 0x000fdf71, // 13296-13303
	0x000fe371, 0x000fe771, 0x000feb71, 0x000fef71, 0x000ff371, 0x000ff771, 0x000ffb71, 0x000fff71, // 13304-13311
	0x00100371, 0x00100771, 0x00100b71, 0x00100f71, 0x00101371, 0x00101771, 0x00101b71, 0x00101f71, // 13312-13319
	0x00102371, 0x00102771, 0x00102b71, 0x00102f71, 0x00103371, 0x00103771, 0x00103b71, 0x00103f71, // 13320-13327
	0x00104371, 0x00104771, 0x00104b71, 0x00104f71, 0x00105371, 0x00105771, 0x00105b71, 0x00105f71, // 13328-13335
	0x00106371, 0x00106771, 0x00106b71, 0x00106f71, 0x00107371, 0x00107771, 0x00107b71, 0x00107f71, // 13336-13343
	0x00108371, 0x00108771, 0x00108b71, 0x00108f71, 0x00109371, 0x00109771, 0x00109b71, 0x00109f71, // 13344-13351
	0x0010a371, 0x0010a771, 0x0010ab71, 0x0010af71, 0x0010b371, 0x0010b771, 0x0010bb71, 0x0010bf71, // 13352-13359
	0x0010c371, 0x0010c771, 0x0010cb71, 0x0010cf71, 0x0010d371, 0x0010d771, 0x0010db71, 0x0010df71, // 13360-13367
	0x0010e371, 0x0010e771, 0x0010eb71, 0x0010ef71, 0x0010f371, 0x0010f771, 0x0010fb71, 0x0010ff71, // 13368-13375
	0x00110371, 0x00110771, 0x00110b71, 0x00110f71, 0x00111371, 0x00111771, 0x00111b71, 0x00111f71, // 13376-13383
	0x00112371, 0x00112771, 0x00112b71, 0x00112f71, 0x00113371, 0x00113771, 0x00113b71, 0x00113f71, // 13384-13391
	0x00114371, 0x00114771, 0x00114b71, 0x00114f71, 0x00115371, 0x00115771, 0x00115b71, 0x00115f71, // 13392-13399
	0x00116371, 0x00116771, 0x00116b71, 0x00116f71, 0x00117371, 0x00117771, 0x00117b71, 0x00117f71, // 13400-13407
	0x00118371, 0x00118771, 0x00118b71, 0x00118f71, 0x00119371, 0x00119771, 0x00119b71, 0x00119f71, // 13408-13415
	0x0011a371, 0x0011a771, 0x0011ab71, 0x0011af71, 0x0011b371, 0x0011b771, 0x0011bb71, 0x0011bf71, // 13416-13423
	0x0011c371, 0x0011c771, 0x0011cb71, 0x0011cf71, 0x0011d371, 0x0011d771, 0x0011db71, 0x0011df71, // 13424-13431
	0x0011e371, 0x0011e771, 0x0011eb71, 0x0011ef71, 0x0011f371, 0x0011f771, 0x0011fb71, 0x0011ff71, // 13432-13439
	0x00120371, 0x00120771, 0x00120b71, 0x00120f71, 0x00121371, 0x00121771, 0x00121b71, 0x00121f71, // 13440-13447
	0x00122371, 0x00122771, 0x00122b71, 0x00122f71, 0x00123371, 0x00123771, 0x00123b71, 0x00123f71, // 13448-13455
	0x00124371, 0x00124771, 0x00124b71, 0x00124f71, 0x00125371, 0x00125771, 0x00125b71, 0x00125f71, // 13456-13463
	0x00126371, 0x00126771, 0x00126b71, 0x00126f71, 0x00127371, 0x00127771, 0x00127b71, 0x00127f71, // 13464-13471
	0x00128371, 0x00128771, 0x00128b71, 0x00128f71, 0x00129371, 0x00129771, 0x00129b71, 0x00129f71, // 13472-13479
	0x0012a371, 0x0012a771, 0x0012ab71, 0x0012af71, 0x0012b371, 0x0012b771, 0x0012bb71, 0x0012bf71, // 13480-13487
	0x0012c371, 0x0012c771, 0x0012cb71, 0x0012cf71, 0x0012d371, 0x0012d771, 0x0012db71, 0x0012df71, // 13488-13495
	0x0012e371, 0x0012e771, 0x0012eb71, 0x0012ef71, 0x0012f371, 0x0012f771, 0x0012fb71, 0x0012ff71, // 13496-13503
	0x00130371, 0x00130771, 0x00130b71, 0x00130f71, 0x00131371, 0x00131771, 0x00131b71, 0x00131f71, // 13504-13511
	0x00132371, 0x00132771, 0x00132b71, 0x00132f71, 0x00133371, 0x00133771, 0x00133b71, 0x00133f71, // 13512-13519
	0x00134371, 0x00134771, 0x00134b71, 0x00134f71, 0x00135371, 0x00135771, 0x00135b71, 0x00135f71, // 13520-13527
	0x00136371, 0x00136771, 0x00136b71, 0x00136f71, 0x00137371, 0x00137771, 0x00137b71, 0x00137f71, // 13528-13535
	0x00138371, 0x00138771, 0x00138b71, 0x00138f71, 0x00139371, 0x00139771, 0x00139b71, 0x00139f71, // 13536-13543
	0x0013a371, 0x0013a771, 0x0013ab71, 0x0013af71, 0x0013b371, 0x0013b771, 0x0013bb71, 0x0013bf71, // 13544-13551
	0x0013c371, 0x0013c771, 0x0013cb71, 0x0013cf71, 0x0013d371, 0x0013d771, 0x0013db71, 0x0013df71, // 13552-13559
	0x0013e371, 0x0013e771, 0x0013eb71, 0x0013ef71, 0x0013f371, 0x0013f771, 0x0013fb71, 0x0013ff71, // 13560-13567
	0x00140371, 0x00140771, 0x00140b71, 0x00140f71, 0x00141371, 0x00141771, 0x00141b71, 0x00141f71, // 13568-13575
	0x00142371, 0x00142771, 0x00142b71, 0x00142f71, 0x00143371, 0x00143771, 0x00143b71, 0x00143f71, // 13576-13583
	0x00144371, 0x00144771, 0x00144b71, 0x00144f71, 0x00145371, 0x00145771, 0x00145b71, 0x00145f71, // 13584-13591
	0x00146371, 0x00146771, 0x00146b71, 0x00146f71, 0x00147371, 0x00147771, 0x00147b71, 0x00147f71, // 13592-13599
	0x00148371, 0x00148771, 0x00148b71, 0x00148f71, 0x00149371, 0x00149771, 0x00149b71, 0x00149f71, // 13600-13607
	0x0014a371, 0x0014a771, 0x0014ab71, 0x0014af71, 0x0014b371, 0x0014b771, 0x0014bb71, 0x0014bf71, // 13608-13615
	0x0014c371, 0x0014c771, 0x0014cb71, 0x0014cf71, 0x0014d371, 0x0014d771, 0x0014db71, 0x0014df71, // 13616-13623
	0x0014e371, 0x0014e771, 0x0014eb71, 0x0014ef71, 0x0014f371, 0x0014f771, 0x0014fb71, 0x0014ff71, // 13624-13631
	0x00150371, 0x00150771, 0x00150b71, 0x00150f71, 0x00151371, 0x00151771, 0x00151b71, 0x00151f71, // 13632-13639
	0x00152371, 0x00152771, 0x00152b71, 0x00152f71, 0x00153371, 0x00153771, 0x00153b71, 0x00153f71, // 13640-13647
	0x00154371, 0x00154771, 0x00154b71, 0x00154f71, 0x00155371, 0x00155771, 0x00155b71, 0x00155f71, // 13648-13655
	0x00156371, 0x00156771, 0x00156b71, 0x00156f71, 0x00157371, 0x00157771, 0x00157b71, 0x00157f71, // 13656-13663
	0x00158371, 0x00158771, 0x00158b71, 0x00158f71, 0x00159371, 0x00159771, 0x00159b71, 0x00159f71, // 13664-13671
	0x0015a371, 0x0015a771, 0x0015ab71, 0x0015af71, 0x0015b371, 0x0015b771, 0x0015bb71, 0x0015bf71, // 13672-13679
	0x0015c371, 0x0015c771, 0x0015cb71, 0x0015cf71, 0x0015d371, 0x0015d771, 0x0015db71, 0x0015df71, // 13680-13687
	0x0015e371, 0x0015e771, 0x0015eb71, 0x0015ef71, 0x0015f371, 0x0015f771, 0x0015fb71, 0x0015ff71, // 13688-13695
	0x00160371, 0x00160771, 0x00160b71, 0x00160f71, 0x00161371, 0x00161771, 0x00161b71, 0x00161f71, // 13696-13703
	0x00162371, 0x00162771, 0x00162b71, 0x00162f71, 0x00163371, 0x00163771, 0x00163b71, 0x00163f71, // 13704-13711
	0x00164371, 0x00164771, 0x00164b71, 0x00164f71, 0x00165371, 0x00165771, 0x00165b71, 0x00165f71, // 13712-13719
	0x00166371, 0x00166771, 0x00166b71, 0x00166f71, 0x00167371, 0x00167771, 0x00167b71, 0x00167f71, // 13720-13727
	0x00168371, 0x00168771, 0x00168b71, 0x00168f71, 0x00169371, 0x00169771, 0x00169b71, 0x00169f71, // 13728-13735
	0x0016a371, 0x0016a771, 0x0016ab71, 0x0016af71, 0x0016b371, 0x0016b771, 0x0016bb71, 0x0016bf71, // 13736-13743
	0x0016c371, 0x0016c771, 0x0016cb71, 0x0016cf71, 0x0016d371, 0x0016d771, 0x0016db71, 0x0016df71, // 13744-13751
	0x0016e371, 0x0016e771, 0x0016eb71, 0x0016ef71, 0x0016f371, 0x0016f771, 0x0016fb71, 0x0016ff71, // 13752-13759
	0x00170371, 0x00170771, 0x00170b71, 0x00170f71, 0x00171371, 0x00171771, 0x00171b71, 0x00171f71, // 13760-13767
	0x00172371, 0x00172771, 0x00172b71, 0x00172f71, 0x00173371, 0x00173771, 0x00173b71, 0x00173f71, // 13768-13775
	0x00174371, 0x00174771, 0x00174b71, 0x00174f71, 0x00175371, 0x00175771, 0x00175b71, 0x00175f71, // 13776-13783
	0x00176371, 0x00176771, 0x00176b71, 0x00176f71, 0x00177371, 0x00177771, 0x00177b71, 0x00177f71, // 13784-13791
	0x00178371, 0x00178771, 0x00178b71, 0x00178f71, 0x00179371, 0x00179771, 0x00179b71, 0x00179f71, // 13792-13799
	0x0017a371, 0x0017a771, 0x0017ab71, 0x0017af71, 0x0017b371, 0x0017b771, 0x0017bb71, 0x0017bf71, // 13800-13807
	0x0017c371, 0x0017c771, 0x0017cb71, 0x0017cf71, 0x0017d371, 0x0017d771, 0x0017db71, 0x0017df71, // 13808-13815
	0x0017e371, 0x0017e771, 0x0017eb71, 0x0017ef71, 0x0017f371, 0x0017f771, 0x0017fb71, 0x0017ff71, // 13816-13823
	0x00180371, 0x00180771, 0x00180b71, 0x00180f71, 0x00181371, 0x00181771, 0x00181b71, 0x00181f71, // 13824-13831
	0x00182371, 0x00182771, 0x00182b71, 0x00182f71, 0x00183371, 0x00183771, 0x00183b71, 0x00183f71, // 13832-13839
	0x00184371, 0x00184771, 0x00184b71, 0x00184f71, 0x00185371, 0x00185771, 0x00185b71, 0x00185f71, // 13840-13847
	0x00186371, 0x00186771, 0x00186b71, 0x00186f71, 0x00187371, 0x00187771, 0x00187b71, 0x00187f71, // 13848-13855
	0x00188371, 0x00188771, 0x00188b71, 0x00188f71, 0x00189371, 0x00189771, 0x00189b71, 0x00189f71, // 13856-13863
	0x0018a371, 0x0018a771, 0x0018ab71, 0x0018af71, 0x0018b371, 0x0018b771, 0x0018bb71, 0x0018bf71, // 13864-13871
	0x0018c371, 0x0018c771, 0x0018cb71, 0x0018cf71, 0x0018d371, 0x0018d771, 0x0018db71, 0x0018df71, // 13872-13879
	0x0018e371, 0x0018e771, 0x0018eb71, 0x0018ef71, 0x0018f371, 0x0018f771, 0x0018fb71, 0x0018ff71, // 13880-13887
	0x00190371, 0x00190771, 0x00190b71, 0x00190f71, 0x00191371, 0x00191771, 0x00191b71, 0x00191f71, // 13888-13895
	0x00192371, 0x00192771, 0x00192b71, 0x00192f71, 0x00193371, 0x00193771, 0x00193b71, 0x00193f71, // 13896-13903
	0x00194371, 0x00194771, 0x00194b71, 0x00194f71, 0x00195371, 0x00195771, 0x00195b71, 0x00195f71, // 13904-13911
	0x00196371, 0x00196771, 0x00196b71, 0x00196f71, 0x00197371, 0x00197771, 0x00197b71, 0x00197f71, // 13912-13919
	0x00198371, 0x00198771, 0x00198b71, 0x00198f71, 0x00199371, 0x00199771, 0x00199b71, 0x00199f71, // 13920-13927
	0x0019a371, 0x0019a771, 0x0019ab71, 0x0019af71, 0x0019b371, 0x0019b771, 0x0019bb71, 0x0019bf71, // 13928-13935
	0x0019c371, 0x0019c771, 0x0019cb71, 0x0019cf71, 0x0019d371, 0x0019d771, 0x0019db71, 0x0019df71, // 13936-13943
	0x0019e371, 0x0019e771, 0x0019eb71, 0x0019ef71, 0x0019f371, 0x0019f771, 0x0019fb71, 0x0019ff71, // 13944-13951
	0x001a0371, 0x001a0771, 0x001a0b71, 0x001a0f71, 0x001a1371, 0x001a1771, 0x001a1b71, 0x001a1f71, // 13952-13959
	0x001a2371, 0x001a2771, 0x001a2b71, 0x001a2f71, 0x001a3371, 0x001a3771, 0x001a3b71, 0x001a3f71, // 13960-13967
	0x001a4371, 0x001a4771, 0x001a4b71, 0x001a4f71, 0x001a5371, 0x001a5771, 0x001a5b71, 0x001a5f71, // 13968-13975
	0x001a6371, 0x001a6771, 0x001a6b71, 0x001a6f71, 0x001a7371, 0x001a7771, 0x001a7b71, 0x001a7f71, // 13976-13983
	0x001a8371, 0x001a8771, 0x001a8b71, 0x001a8f71, 0x001a9371, 0x001a9771, 0x001a9b71, 0x001a9f71, // 13984-13991
	0x001aa371, 0x001aa771, 0x001aab71, 0x001aaf71, 0x001ab371, 0x001ab771, 0x001abb71, 0x001abf71, // 13992-13999
	0x001ac371, 0x001ac771, 0x001acb71, 0x001acf71, 0x001ad371, 0x001ad771, 0x001adb71, 0x001adf71, // 14000-14007
	0x001ae371, 0x001ae771, 0x001aeb71, 0x001aef71, 0x001af371, 0x001af771, 0x001afb71, 0x001aff71, // 14008-14015
	0x001b0371, 0x001b0771, 0x001b0b71, 0x001b0f71, 0x001b1371, 0x001b1771, 0x001b1b71, 0x001b1f71, // 14016-14023
	0x001b2371, 0x001b2771, 0x001b2b71, 0x001b2f71, 0x001b3371, 0x001b3771, 0x001b3b71, 0x001b3f71, // 14024-14031
	0x001b4371, 0x001b4771, 0x001b4b71, 0x001b4f71, 0x001b5371, 0x001b5771, 0x001b5b71, 0x001b5f71, // 14032-14039
	0x001b6371, 0x001b6771, 0x001b6b71, 0x001b6f71, 0x001b7371, 0x001b7771, 0x001b7b71, 0x001b7f71, // 14040-14047
	0x001b8371, 0x001b8771, 0x001b8b71, 0x001b8f71, 0x001b9371, 0x001b9771, 0x001b9b71, 0x001b9f71, // 14048-14055
	0x001ba371, 0x001ba771, 0x001bab71, 0x001baf71, 0x001bb371, 0x001bb771, 0x001bbb71, 0x001bbf71, // 14056-14063
	0x001bc371, 0x001bc771, 0x001bcb71, 0x001bcf71, 0x001bd371, 0x001bd771, 0x001bdb71, 0x001bdf71, // 14064-14071
	0x001be371, 0x001be771, 0x001beb71, 0x001bef71, 0x001bf371, 0x001bf771, 0x001bfb71, 0x001bff71, // 14072-14079
	0x001c0371, 0x001c0771, 0x001c0b71, 0x001c0f71, 0x001c1371, 0x001c1771, 0x001c1b71, 0x001c1f71, // 14080-14087
	0x001c2371, 0x001c2771, 0x001c2b71, 0x001c2f71, 0x001c3371, 0x001c3771, 0x001c3b71, 0x001c3f71, // 14088-14095
	0x001c4371, 0x001c4771, 0x001c4b71, 0x001c4f71, 0x001c5371, 0x001c5771, 0x001c5b71, 0x001c5f71, // 14096-14103
	0x001c6371, 0x001c6771, 0x001c6b71, 0x001c6f71, 0x001c7371, 0x001c7771, 0x001c7b71, 0x001c7f71, // 14104-14111
	0x001c8371, 0x001c8771, 0x001c8b71, 0x001c8f71, 0x001c9371, 0x001c9771, 0x001c9b71, 0x001c9f71, // 14112-14119
	0x001ca371, 0x001ca771, 0x001cab71, 0x001caf71, 0x001cb371, 0x001cb771, 0x001cbb71, 0x001cbf71, // 14120-14127
	0x001cc371, 0x001cc771, 0x001ccb71, 0x001ccf71, 0x001cd371, 0x001cd771, 0x001cdb71, 0x001cdf71, // 14128-14135
	0x001ce371, 0x001ce771, 0x001ceb71, 0x001cef71, 0x001cf371, 0x001cf771, 0x001cfb71, 0x001cff71, // 14136-14143
	0x001d0371, 0x001d0771, 0x001d0b71, 0x001d0f71, 0x001d1371, 0x001d1771, 0x001d1b71, 0x001d1f71, // 14144-14151
	0x001d2371, 0x001d2771, 0x001d2b71, 0x001d2f71, 0x001d3371, 0x001d3771, 0x001d3b71, 0x001d3f71, // 14152-14159
	0x001d4371, 0x001d4771, 0x001d4b71, 0x001d4f71, 0x001d5371, 0x001d5771, 0x001d5b71, 0x001d5f71, // 14160-14167
	0x001d6371, 0x001d6771, 0x001d6b71, 0x001d6f71, 0x001d7371, 0x001d7771, 0x001d7b71, 0x001d7f71, // 14168-14175
	0x001d8371, 0x001d8771, 0x001d8b71, 0x001d8f71, 0x001d9371, 0x001d9771, 0x001d9b71, 0x001d9f71, // 14176-14183
	0x001da371, 0x001da771, 0x001dab71, 0x001daf71, 0x001db371, 0x001db771, 0x001dbb71, 0x001dbf71, // 14184-14191
	0x001dc371, 0x001dc771, 0x001dcb71, 0x001dcf71, 0x001dd371, 0x001dd771, 0x001ddb71, 0x001ddf71, // 14192-14199
	0x001de371, 0x001de771, 0x001deb71, 0x001def71, 0x001df371, 0x001df771, 0x001dfb71, 0x001dff71, // 14200-14207
	0x001e0371, 0x001e0771, 0x001e0b71, 0x001e0f71, 0x001e1371, 0x001e1771, 0x001e1b71, 0x001e1f71, // 14208-14215
	0x001e2371, 0x001e2771, 0x001e2b71, 0x001e2f71, 0x001e3371, 0x001e3771, 0x001e3b71, 0x001e3f71, // 14216-14223
	0x001e4371, 0x001e4771, 0x001e4b71, 0x001e4f71, 0x001e5371, 0x001e5771, 0x001e5b71, 0x001e5f71, // 14224-14231
	0x001e6371, 0x001e6771, 0x001e6b71, 0x001e6f71, 0x001e7371, 0x001e7771, 0x001e7b71, 0x001e7f71, // 14232-14239
	0x001e8371, 0x001e8771, 0x001e8b71, 0x001e8f71, 0x001e9371, 0x001e9771, 0x001e9b71, 0x001e9f71, // 14240-14247
	0x001ea371, 0x001ea771, 0x001eab71, 0x001eaf71, 0x001eb371, 0x001eb771, 0x001ebb71, 0x001ebf71, // 14248-14255
	0x001ec371, 0x001ec771, 0x001ecb71, 0x001ecf71, 0x001ed371, 0x001ed771, 0x001edb71, 0x001edf71, // 14256-14263
	0x001ee371, 0x001ee771, 0x001eeb71, 0x001eef71, 0x001ef371, 0x001ef771, 0x001efb71, 0x001eff71, // 14264-14271
	0x001f0371, 0x001f0771, 0x001f0b71, 0x001f0f71, 0x001f1371, 0x001f1771, 0x001f1b71, 0x001f1f71, // 14272-14279
	0x001f2371, 0x001f2771, 0x001f2b71, 0x001f2f71, 0x001f3371, 0x001f3771, 0x001f3b71, 0x001f3f71, // 14280-14287
	0x001f4371, 0x001f4771, 0x001f4b71, 0x001f4f71, 0x001f5371, 0x001f5771, 0x001f5b71, 0x001f5f71, // 14288-14295
	0x001f6371, 0x001f6771, 0x001f6b71, 0x001f6f71, 0x001f7371, 0x001f7771, 0x001f7b71, 0x001f7f71, // 14296-14303
	0x001f8371, 0x001f8771, 0x001f8b71, 0x001f8f71, 0x001f9371, 0x001f9771, 0x001f9b71, 0x001f9f71, // 14304-14311
	0x001fa371, 0x001fa771, 0x001fab71, 0x001faf71, 0x001fb371, 0x001fb771, 0x001fbb71, 0x001fbf71, // 14312-14319
	0x001fc371, 0x001fc771, 0x001fcb71, 0x001fcf71, 0x001fd371, 0x001fd771, 0x001fdb71, 0x001fdf71, // 14320-14327
	0x001fe371, 0x001fe771, 0x001feb71, 0x001fef71, 0x001ff371, 0x001ff771, 0x001ffb71, 0x001fff71, // 14328-14335
	0x00200371, 0x00200771, 0x00200b71, 0x00200f71, 0x00201371, 0x00201771, 0x00201b71, 0x00201f71, // 14336-14343
	0x00202371, 0x00202771, 0x00202b71, 0x00202f71, 0x00203371, 0x00203771, 0x00203b71, 0x00203f71, // 14344-14351
	0x00204371, 0x00204771, 0x00204b71, 0x00204f71, 0x00205371, 0x00205771, 0x00205b71, 0x00205f71, // 14352-14359
	0x00206371, 0x00206771, 0x00206b71, 0x00206f71, 0x00207371, 0x00207771, 0x00207b71, 0x00207f71, // 14360-14367
	0x00208371, 0x00208771, 0x00208b71, 0x00208f71, 0x00209371, 0x00209771, 0x00209b71, 0x00209f71, // 14368-14375
	0x0020a371, 0x0020a771, 0x0020ab71, 0x0020af71, 0x0020b371, 0x0020b771, 0x0020bb71, 0x0020bf71, // 14376-14383
	0x0020c371, 0x0020c771, 0x0020cb71, 0x0020cf71, 0x0020d371, 0x0020d771, 0x0020db71, 0x0020df71, // 14384-14391
	0x0020e371, 0x0020e771, 0x0020eb71, 0x0020ef71, 0x0020f371, 0x0020f771, 0x0020fb71, 0x0020ff71, // 14392-14399
	0x00210371, 0x00210771, 0x00210b71, 0x00210f71, 0x00211371, 0x00211771, 0x00211b71, 0x00211f71, // 14400-14407
	0x00212371, 0x00212771, 0x00212b71, 0x00212f71, 0x00213371, 0x00213771, 0x00213b71, 0x00213f71, // 14408-14415
	0x00214371, 0x00214771, 0x00214b71, 0x00214f71, 0x00215371, 0x00215771, 0x00215b71, 0x00215f71, // 14416-14423
	0x00216371, 0x00216771, 0x00216b71, 0x00216f71, 0x00217371, 0x00217771, 0x00217b71, 0x00217f71, // 14424-14431
	0x00218371, 0x00218771, 0x00218b71, 0x00218f71, 0x00219371, 0x00219771, 0x00219b71, 0x00219f71, // 14432-14439
	0x0021a371, 0x0021a771, 0x0021ab71, 0x0021af71, 0x0021b371, 0x0021b771, 0x0021bb71, 0x0021bf71, // 14440-14447
	0x0021c371, 0x0021c771, 0x0021cb71, 0x0021cf71, 0x0021d371, 0x0021d771, 0x0021db71, 0x0021df71, // 14448-14455
	0x0021e371, 0x0021e771, 0x0021eb71, 0x0021ef71, 0x0021f371, 0x0021f771, 0x0021fb71, 0x0021ff71, // 14456-14463
	0x00220371, 0x00220771, 0x00220b71, 0x00220f71, 0x00221371, 0x00221771, 0x00221b71, 0x00221f71, // 14464-14471
	0x00222371, 0x00222771, 0x00222b71, 0x00222f71, 0x00223371, 0x00223771, 0x00223b71, 0x00223f71, // 14472-14479
	0x00224371, 0x00224771, 0x00224b71, 0x00224f71, 0x00225371, 0x00225771, 0x00225b71, 0x00225f71, // 14480-14487
	0x00226371, 0x00226771, 0x00226b71, 0x00226f71, 0x00227371, 0x00227771, 0x00227b71, 0x00227f71, // 14488-14495
	0x00228371, 0x00228771, 0x00228b71, 0x00228f71, 0x00229371, 0x00229771, 0x00229b71, 0x00229f71, // 14496-14503
	0x0022a371, 0x0022a771, 0x0022ab71, 0x0022af71, 0x0022b371, 0x0022b771, 0x0022bb71, 0x0022bf71, // 14504-14511
	0x0022c371, 0x0022c771, 0x0022cb71, 0x0022cf71, 0x0022d371, 0x0022d771, 0x0022db71, 0x0022df71, // 14512-14519
	0x0022e371, 0x0022e771, 0x0022eb71, 0x0022ef71, 0x0022f371, 0x0022f771, 0x0022fb71, 0x0022ff71, // 14520-14527
	0x00230371, 0x00230771, 0x00230b71, 0x00230f71, 0x00231371, 0x00231771, 0x00231b71, 0x00231f71, // 14528-14535
	0x00232371, 0x00232771, 0x00232b71, 0x00232f71, 0x00233371, 0x00233771, 0x00233b71, 0x00233f71, // 14536-14543
	0x00234371, 0x00234771, 0x00234b71, 0x00234f71, 0x00235371, 0x00235771, 0x00235b71, 0x00235f71, // 14544-14551
	0x00236371, 0x00236771, 0x00236b71, 0x00236f71, 0x00237371, 0x00237771, 0x00237b71, 0x00237f71, // 14552-14559
	0x00238371, 0x00238771, 0x00238b71, 0x00238f71, 0x00239371, 0x00239771, 0x00239b71, 0x00239f71, // 14560-14567
	0x0023a371, 0x0023a771, 0x0023ab71, 0x0023af71, 0x0023b371, 0x0023b771, 0x0023bb71, 0x0023bf71, // 14568-14575
	0x0023c371, 0x0023c771, 0x0023cb71, 0x0023cf71, 0x0023d371, 0x0023d771, 0x0023db71, 0x0023df71, // 14576-14583
	0x0023e371, 0x0023e771, 0x0023eb71, 0x0023ef71, 0x0023f371, 0x0023f771, 0x0023fb71, 0x0023ff71, // 14584-14591
	0x00240371, 0x00240771, 0x00240b71, 0x00240f71, 0x00241371, 0x00241771, 0x00241b71, 0x00241f71, // 14592-14599
	0x00242371, 0x00242771, 0x00242b71, 0x00242f71, 0x00243371, 0x00243771, 0x00243b71, 0x00243f71, // 14600-14607
	0x00244371, 0x00244771, 0x00244b71, 0x00244f71, 0x00245371, 0x00245771, 0x00245b71, 0x00245f71, // 14608-14615
	0x00246371, 0x00246771, 0x00246b71, 0x00246f71, 0x00247371, 0x00247771, 0x00247b71, 0x00247f71, // 14616-14623
	0x00248371, 0x00248771, 0x00248b71, 0x00248f71, 0x00249371, 0x00249771, 0x00249b71, 0x00249f71, // 14624-14631
	0x0024a371, 0x0024a771, 0x0024ab71, 0x0024af71, 0x0024b371, 0x0024b771, 0x0024bb71, 0x0024bf71, // 14632-14639
	0x0024c371, 0x0024c771, 0x0024cb71, 0x0024cf71, 0x0024d371, 0x0024d771, 0x0024db71, 0x0024df71, // 14640-14647
	0x0024e371, 0x0024e771, 0x0024eb71, 0x0024ef71, 0x0024f371, 0x0024f771, 0x0024fb71, 0x0024ff71, // 14648-14655
	0x00250371, 0x00250771, 0x00250b71, 0x00250f71, 0x00251371, 0x00251771, 0x00251b71, 0x00251f71, // 14656-14663
	0x00252371, 0x00252771, 0x00252b71, 0x00252f71, 0x00253371, 0x00253771, 0x00253b71, 0x00253f71, // 14664-14671
	0x00254371, 0x00254771, 0x00254b71, 0x00254f71, 0x00255371, 0x00255771, 0x00255b71, 0x00255f71, // 14672-14679
	0x00256371, 0x00256771, 0x00256b71, 0x00256f71, 0x00257371, 0x00257771, 0x00257b71, 0x00257f71, // 14680-14687
	0x00258371, 0x00258771, 0x00258b71, 0x00258f71, 0x00259371, 0x00259771, 0x00259b71, 0x00259f71, // 14688-14695
	0x0025a371, 0x0025a771, 0x0025ab71, 0x0025af71, 0x0025b371, 0x0025b771, 0x0025bb71, 0x0025bf71, // 14696-14703
	0x0025c371, 0x0025c771, 0x0025cb71, 0x0025cf71, 0x0025d371, 0x0025d771, 0x0025db71, 0x0025df71, // 14704-14711
	0x0025e371, 0x0025e771, 0x0025eb71, 0x0025ef71, 0x0025f371, 0x0025f771, 0x0025fb71, 0x0025ff71, // 14712-14719
	0x00260371, 0x00260771, 0x00260b71, 0x00260f71, 0x00261371, 0x00261771, 0x00261b71, 0x00261f71, // 14720-14727
	0x00262371, 0x00262771, 0x00262b71, 0x00262f71, 0x00263371, 0x00263771, 0x00263b71, 0x00263f71, // 14728-14735
	0x00264371, 0x00264771, 0x00264b71, 0x00264f71, 0x00265371, 0x00265771, 0x00265b71, 0x00265f71, // 14736-14743
	0x00266371, 0x00266771, 0x00266b71, 0x00266f71, 0x00267371, 0x00267771, 0x00267b71, 0x00267f71, // 14744-14751
	0x00268371, 0x00268771, 0x00268b71, 0x00268f71, 0x00269371, 0x00269771, 0x00269b71, 0x00269f71, // 14752-14759
	0x0026a371, 0x0026a771, 0x0026ab71, 0x0026af71, 0x0026b371, 0x0026b771, 0x0026bb71, 0x0026bf71, // 14760-14767
	0x0026c371, 0x0026c771, 0x0026cb71, 0x0026cf71, 0x0026d371, 0x0026d771, 0x0026db71, 0x0026df71, // 14768-14775
	0x0026e371, 0x0026e771, 0x0026eb71, 0x0026ef71, 0x0026f371, 0x0026f771, 0x0026fb71, 0x0026ff71, // 14776-14783
	0x00270371, 0x00270771, 0x00270b71, 0x00270f71, 0x00271371, 0x00271771, 0x00271b71, 0x00271f71, // 14784-14791
	0x00272371, 0x00272771, 0x00272b71, 0x00272f71, 0x00273371, 0x00273771, 0x00273b71, 0x00273f71, // 14792-14799
	0x00274371, 0x00274771, 0x00274b71, 0x00274f71, 0x00275371, 0x00275771, 0x00275b71, 0x00275f71, // 14800-14807
	0x00276371, 0x00276771, 0x00276b71, 0x00276f71, 0x00277371, 0x00277771, 0x00277b71, 0x00277f71, // 14808-14815
	0x00278371, 0x00278771, 0x00278b71, 0x00278f71, 0x00279371, 0x00279771, 0x00279b71, 0x00279f71, // 14816-14823
	0x0027a371, 0x0027a771, 0x0027ab71, 0x0027af71, 0x0027b371, 0x0027b771, 0x0027bb71, 0x0027bf71, // 14824-14831
	0x0027c371, 0x0027c771, 0x0027cb71, 0x0027cf71, 0x0027d371, 0x0027d771, 0x0027db71, 0x0027df71, // 14832-14839
	0x0027e371, 0x0027e771, 0x0027eb71, 0x0027ef71, 0x0027f371, 0x0027f771, 0x0027fb71, 0x0027ff71, // 14840-14847
	0x00280371, 0x00280771, 0x00280b71, 0x00280f71, 0x00281371, 0x00281771, 0x00281b71, 0x00281f71, // 14848-14855
	0x00282371, 0x00282771, 0x00282b71, 0x00282f71, 0x00283371, 0x00283771, 0x00283b71, 0x00283f71, // 14856-14863
	0x00284371, 0x00284771, 0x00284b71, 0x00284f71, 0x00285371, 0x00285771, 0x00285b71, 0x00285f71, // 14864-14871
	0x00286371, 0x00286771, 0x00286b71, 0x00286f71, 0x00287371, 0x00287771, 0x00287b71, 0x00287f71, // 14872-14879
	0x00288371, 0x00288771, 0x00288b71, 0x00288f71, 0x00289371, 0x00289771, 0x00289b71, 0x00289f71, // 14880-14887
	0x0028a371, 0x0028a771, 0x0028ab71, 0x0028af71, 0x0028b371, 0x0028b771, 0x0028bb71, 0x0028bf71, // 14888-14895
	0x0028c371, 0x0028c771, 0x0028cb71, 0x0028cf71, 0x0028d371, 0x0028d771, 0x0028db71, 0x0028df71, // 14896-14903
	0x0028e371, 0x0028e771, 0x0028eb71, 0x0028ef71, 0x0028f371, 0x0028f771, 0x0028fb71, 0x0028ff71, // 14904-14911
	0x00290371, 0x00290771, 0x00290b71, 0x00290f71, 0x00291371, 0x00291771, 0x00291b71, 0x00291f71, // 14912-14919
	0x00292371, 0x00292771, 0x00292b71, 0x00292f71, 0x00293371, 0x00293771, 0x00293b71, 0x00293f71, // 14920-14927
	0x00294371, 0x00294771, 0x00294b71, 0x00294f71, 0x00295371, 0x00295771, 0x00295b71, 0x00295f71, // 14928-14935
	0x00296371, 0x00296771, 0x00296b71, 0x00296f71, 0x00297371, 0x00297771, 0x00297b71, 0x00297f71, // 14936-14943
	0x00298371, 0x00298771, 0x00298b71, 0x00298f71, 0x00299371, 0x00299771, 0x00299b71, 0x00299f71, // 14944-14951
	0x0029a371, 0x0029a771, 0x0029ab71, 0x0029af71, 0x0029b371, 0x0029b771, 0x0029bb71, 0x0029bf71, // 14952-14959
	0x0029c371, 0x0029c771, 0x0029cb71, 0x0029cf71, 0x0029d371, 0x0029d771, 0x0029db71, 0x0029df71, // 14960-14967
	0x0029e371, 0x0029e771, 0x0029eb71, 0x0029ef71, 0x0029f371, 0x0029f771, 0x0029fb71, 0x0029ff71, // 14968-14975
	0x002a0371, 0x002a0771, 0x002a0b71, 0x002a0f71, 0x002a1371, 0x002a1771, 0x002a1b71, 0x002a1f71, // 14976-14983
	0x002a2371, 0x002a2771, 0x002a2b71, 0x002a2f71, 0x002a3371, 0x002a3771, 0x002a3b71, 0x002a3f71, // 14984-14991
	0x002a4371, 0x002a4771, 0x002a4b71, 0x002a4f71, 0x002a5371, 0x002a5771, 0x002a5b71, 0x002a5f71, // 14992-14999
	0x002a6371, 0x002a6771, 0x002a6b71, 0x002a6f71, 0x002a7371, 0x002a7771, 0x002a7b71, 0x002a7f71, // 15000-15007
	0x002a8371, 0x002a8771, 0x002a8b71, 0x002a8f71, 0x002a9371, 0x002a9771, 0x002a9b71, 0x002a9f71, // 15008-15015
	0x002aa371, 0x002aa771, 0x002aab71, 0x002aaf71, 0x002ab371, 0x002ab771, 0x002abb71, 0x002abf71, // 15016-15023
	0x002ac371, 0x002ac771, 0x002acb71, 0x002acf71, 0x002ad371, 0x002ad771, 0x002adb71, 0x002adf71, // 15024-15031
	0x002ae371, 0x002ae771, 0x002aeb71, 0x002aef71, 0x002af371, 0x002af771, 0x002afb71, 0x002aff71, // 15032-15039
	0x002b0371, 0x002b0771, 0x002b0b71, 0x002b0f71, 0x002b1371, 0x002b1771, 0x002b1b71, 0x002b1f71, // 15040-15047
	0x002b2371, 0x002b2771, 0x002b2b71, 0x002b2f71, 0x002b3371, 0x002b3771, 0x002b3b71, 0x002b3f71, // 15048-15055
	0x002b4371, 0x002b4771, 0x002b4b71, 0x002b4f71, 0x002b5371, 0x002b5771, 0x002b5b71, 0x002b5f71, // 15056-15063
	0x002b6371, 0x002b6771, 0x002b6b71, 0x002b6f71, 0x002b7371, 0x002b7771, 0x002b7b71, 0x002b7f71, // 15064-15071
	0x002b8371, 0x002b8771, 0x002b8b71, 0x002b8f71, 0x002b9371, 0x002b9771, 0x002b9b71, 0x002b9f71, // 15072-15079
	0x002ba371, 0x002ba771, 0x002bab71, 0x002baf71, 0x002bb371, 0x002bb771, 0x002bbb71, 0x002bbf71, // 15080-15087
	0x002bc371, 0x002bc771, 0x002bcb71, 0x002bcf71, 0x002bd371, 0x002bd771, 0x002bdb71, 0x002bdf71, // 15088-15095
	0x002be371, 0x002be771, 0x002beb71, 0x002bef71, 0x002bf371, 0x002bf771, 0x002bfb71, 0x002bff71, // 15096-15103
	0x002c0371, 0x002c0771, 0x002c0b71, 0x002c0f71, 0x002c1371, 0x002c1771, 0x002c1b71, 0x002c1f71, // 15104-15111
	0x002c2371, 0x002c2771, 0x002c2b71, 0x002c2f71, 0x002c3371, 0x002c3771, 0x002c3b71, 0x002c3f71, // 15112-15119
	0x002c4371, 0x002c4771, 0x002c4b71, 0x002c4f71, 0x002c5371, 0x002c5771, 0x002c5b71, 0x002c5f71, // 15120-15127
	0x002c6371, 0x002c6771, 0x002c6b71, 0x002c6f71, 0x002c7371, 0x002c7771, 0x002c7b71, 0x002c7f71, // 15128-15135
	0x002c8371, 0x002c8771, 0x002c8b71, 0x002c8f71, 0x002c9371, 0x002c9771, 0x002c9b71, 0x002c9f71, // 15136-15143
	0x002ca371, 0x002ca771, 0x002cab71, 0x002caf71, 0x002cb371, 0x002cb771, 0x002cbb71, 0x002cbf71, // 15144-15151
	0x002cc371, 0x002cc771, 0x002ccb71, 0x002ccf71, 0x002cd371, 0x002cd771, 0x002cdb71, 0x002cdf71, // 15152-15159
	0x002ce371, 0x002ce771, 0x002ceb71, 0x002cef71, 0x002cf371, 0x002cf771, 0x002cfb71, 0x002cff71, // 15160-15167
	0x002d0371, 0x002d0771, 0x002d0b71, 0x002d0f71, 0x002d1371, 0x002d1771, 0x002d1b71, 0x002d1f71, // 15168-15175
	0x002d2371, 0x002d2771, 0x002d2b71, 0x002d2f71, 0x002d3371, 0x002d3771, 0x002d3b71, 0x002d3f71, // 15176-15183
	0x002d4371, 0x002d4771, 0x002d4b71, 0x002d4f71, 0x002d5371, 0x002d5771, 0x002d5b71, 0x002d5f71, // 15184-15191
	0x002d6371, 0x002d6771, 0x002d6b71, 0x002d6f71, 0x002d7371, 0x002d7771, 0x002d7b71, 0x002d7f71, // 15192-15199
	0x002d8371, 0x002d8771, 0x002d8b71, 0x002d8f71, 0x002d9371, 0x002d9771, 0x002d9b71, 0x002d9f71, // 15200-15207
	0x002da371, 0x002da771, 0x002dab71, 0x002daf71, 0x002db371, 0x002db771, 0x002dbb71, 0x002dbf71, // 15208-15215
	0x002dc371, 0x002dc771, 0x002dcb71, 0x002dcf71, 0x002dd371, 0x002dd771, 0x002ddb71, 0x002ddf71, // 15216-15223
	0x002de371, 0x002de771, 0x002deb71, 0x002def71, 0x002df371, 0x002df771, 0x002dfb71, 0x002dff71, // 15224-15231
	0x002e0371, 0x002e0771, 0x002e0b71, 0x002e0f71, 0x002e1371, 0x002e1771, 0x002e1b71, 0x002e1f71, // 15232-15239
	0x002e2371, 0x002e2771, 0x002e2b71, 0x002e2f71, 0x002e3371, 0x002e3771, 0x002e3b71, 0x002e3f71, // 15240-15247
	0x002e4371, 0x002e4771, 0x002e4b71, 0x002e4f71, 0x002e5371, 0x002e5771, 0x002e5b71, 0x002e5f71, // 15248-15255
	0x002e6371, 0x002e6771, 0x002e6b71, 0x002e6f71, 0x002e7371, 0x002e7771, 0x002e7b71, 0x002e7f71, // 15256-15263
	0x002e8371, 0x002e8771, 0x002e8b71, 0x002e8f71, 0x002e9371, 0x002e9771, 0x002e9b71, 0x002e9f71, // 15264-15271
	0x002ea371, 0x002ea771, 0x002eab71, 0x002eaf71, 0x002eb371, 0x002eb771, 0x002ebb71, 0x002ebf71, // 15272-15279
	0x002ec371, 0x002ec771, 0x002ecb71, 0x002ecf71, 0x002ed371, 0x002ed771, 0x002edb71, 0x002edf71, // 15280-15287
	0x002ee371, 0x002ee771, 0x002eeb71, 0x002eef71, 0x002ef371, 0x002ef771, 0x002efb71, 0x002eff71, // 15288-15295
	0x002f0371, 0x002f0771, 0x002f0b71, 0x002f0f71, 0x002f1371, 0x002f1771, 0x002f1b71, 0x002f1f71, // 15296-15303
	0x002f2371, 0x002f2771, 0x002f2b71, 0x002f2f71, 0x002f3371, 0x002f3771, 0x002f3b71, 0x002f3f71, // 15304-15311
	0x002f4371, 0x002f4771, 0x002f4b71, 0x002f4f71, 0x002f5371, 0x002f5771, 0x002f5b71, 0x002f5f71, // 15312-15319
	0x002f6371, 0x002f6771, 0x002f6b71, 0x002f6f71, 0x002f7371, 0x002f7771, 0x002f7b71, 0x002f7f71, // 15320-15327
	0x002f8371, 0x002f8771, 0x002f8b71, 0x002f8f71, 0x002f9371, 0x002f9771, 0x002f9b71, 0x002f9f71, // 15328-15335
	0x002fa371, 0x002fa771, 0x002fab71, 0x002faf71, 0x002fb371, 0x002fb771, 0x002fbb71, 0x002fbf71, // 15336-15343
	0x002fc371, 0x002fc771, 0x002fcb71, 0x002fcf71, 0x002fd371, 0x002fd771, 0x002fdb71, 0x002fdf71, // 15344-15351
	0x002fe371, 0x002fe771, 0x002feb71, 0x002fef71, 0x002ff371, 0x002ff771, 0x002ffb71, 0x002fff71, // 15352-15359
	0x00300371, 0x00300771, 0x00300b71, 0x00300f71, 0x00301371, 0x00301771, 0x00301b71, 0x00301f71, // 15360-15367
	0x00302371, 0x00302771, 0x00302b71, 0x00302f71, 0x00303371, 0x00303771, 0x00303b71, 0x00303f71, // 15368-15375
	0x00304371, 0x00304771, 0x00304b71, 0x00304f71, 0x00305371, 0x00305771, 0x00305b71, 0x00305f71, // 15376-15383
	0x00306371, 0x00306771, 0x00306b71, 0x00306f71, 0x00307371, 0x00307771, 0x00307b71, 0x00307f71, // 15384-15391
	0x00308371, 0x00308771, 0x00308b71, 0x00308f71, 0x00309371, 0x00309771, 0x00309b71, 0x00309f71, // 15392-15399
	0x0030a371, 0x0030a771, 0x0030ab71, 0x0030af71, 0x0030b371, 0x0030b771, 0x0030bb71, 0x0030bf71, // 15400-15407
	0x0030c371, 0x0030c771, 0x0030cb71, 0x0030cf71, 0x0030d371, 0x0030d771, 0x0030db71, 0x0030df71, // 15408-15415
	0x0030e371, 0x0030e771, 0x0030eb71, 0x0030ef71, 0x0030f371, 0x0030f771, 0x0030fb71, 0x0030ff71, // 15416-15423
	0x00310371, 0x00310771, 0x00310b71, 0x00310f71, 0x00311371, 0x00311771, 0x00311b71, 0x00311f71, // 15424-15431
	0x00312371, 0x00312771, 0x00312b71, 0x00312f71, 0x00313371, 0x00313771, 0x00313b71, 0x00313f71, // 15432-15439
	0x00314371, 0x00314771, 0x00314b71, 0x00314f71, 0x00315371, 0x00315771, 0x00315b71, 0x00315f71, // 15440-15447
	0x00316371, 0x00316771, 0x00316b71, 0x00316f71, 0x00317371, 0x00317771, 0x00317b71, 0x00317f71, // 15448-15455
	0x00318371, 0x00318771, 0x00318b71, 0x00318f71, 0x00319371, 0x00319771, 0x00319b71, 0x00319f71, // 15456-15463
	0x0031a371, 0x0031a771, 0x0031ab71, 0x0031af71, 0x0031b371, 0x0031b771, 0x0031bb71, 0x0031bf71, // 15464-15471
	0x0031c371, 0x0031c771, 0x0031cb71, 0x0031cf71, 0x0031d371, 0x0031d771, 0x0031db71, 0x0031df71, // 15472-15479
	0x0031e371, 0x0031e771, 0x0031eb71, 0x0031ef71, 0x0031f371, 0x0031f771, 0x0031fb71, 0x0031ff71, // 15480-15487
	0x00320371, 0x00320771, 0x00320b71, 0x00320f71, 0x00321371, 0x00321771, 0x00321b71, 0x00321f71, // 15488-15495
	0x00322371, 0x00322771, 0x00322b71, 0x00322f71, 0x00323371, 0x00323771, 0x00323b71, 0x00323f71, // 15496-15503
	0x00324371, 0x00324771, 0x00324b71, 0x00324f71, 0x00325371, 0x00325771, 0x00325b71, 0x00325f71, // 15504-15511
	0x00326371, 0x00326771, 0x00326b71, 0x00326f71, 0x00327371, 0x00327771, 0x00327b71, 0x00327f71, // 15512-15519
	0x00328371, 0x00328771, 0x00328b71, 0x00328f71, 0x00329371, 0x00329771, 0x00329b71, 0x00329f71, // 15520-15527
	0x0032a371, 0x0032a771, 0x0032ab71, 0x0032af71, 0x0032b371, 0x0032b771, 0x0032bb71, 0x0032bf71, // 15528-15535
	0x0032c371, 0x0032c771, 0x0032cb71, 0x0032cf71, 0x0032d371, 0x0032d771, 0x0032db71, 0x0032df71, // 15536-15543
	0x0032e371, 0x0032e771, 0x0032eb71, 0x0032ef71, 0x0032f371, 0x0032f771, 0x0032fb71, 0x0032ff71, // 15544-15551
	0x00330371, 0x00330771, 0x00330b71, 0x00330f71, 0x00331371, 0x00331771, 0x00331b71, 0x00331f71, // 15552-15559
	0x00332371, 0x00332771, 0x00332b71, 0x00332f71, 0x00333371, 0x00333771, 0x00333b71, 0x00333f71, // 15560-15567
	0x00334371, 0x00334771, 0x00334b71, 0x00334f71, 0x00335371, 0x00335771, 0x00335b71, 0x00335f71, // 15568-15575
	0x00336371, 0x00336771, 0x00336b71, 0x00336f71, 0x00337371, 0x00337771, 0x00337b71, 0x00337f71, // 15576-15583
	0x00338371, 0x00338771, 0x00338b71, 0x00338f71, 0x00339371, 0x00339771, 0x00339b71, 0x00339f71, // 15584-15591
	0x0033a371, 0x0033a771, 0x0033ab71, 0x0033af71, 0x0033b371, 0x0033b771, 0x0033bb71, 0x0033bf71, // 15592-15599
	0x0033c371, 0x0033c771, 0x0033cb71, 0x0033cf71, 0x0033d371, 0x0033d771, 0x0033db71, 0x0033df71, // 15600-15607
	0x0033e371, 0x0033e771, 0x0033eb71, 0x0033ef71, 0x0033f371, 0x0033f771, 0x0033fb71, 0x0033ff71, // 15608-15615
	0x00340371, 0x00340771, 0x00340b71, 0x00340f71, 0x00341371, 0x00341771, 0x00341b71, 0x00341f71, // 15616-15623
	0x00342371, 0x00342771, 0x00342b71, 0x00342f71, 0x00343371, 0x00343771, 0x00343b71, 0x00343f71, // 15624-15631
	0x00344371, 0x00344771, 0x00344b71, 0x00344f71, 0x00345371, 0x00345771, 0x00345b71, 0x00345f71, // 15632-15639
	0x00346371, 0x00346771, 0x00346b71, 0x00346f71, 0x00347371, 0x00347771, 0x00347b71, 0x00347f71, // 15640-15647
	0x00348371, 0x00348771, 0x00348b71, 0x00348f71, 0x00349371, 0x00349771, 0x00349b71, 0x00349f71, // 15648-15655
	0x0034a371, 0x0034a771, 0x0034ab71, 0x0034af71, 0x0034b371, 0x0034b771, 0x0034bb71, 0x0034bf71, // 15656-15663
	0x0034c371, 0x0034c771, 0x0034cb71, 0x0034cf71, 0x0034d371, 0x0034d771, 0x0034db71, 0x0034df71, // 15664-15671
	0x0034e371, 0x0034e771, 0x0034eb71, 0x0034ef71, 0x0034f371, 0x0034f771, 0x0034fb71, 0x0034ff71, // 15672-15679
	0x00350371, 0x00350771, 0x00350b71, 0x00350f71, 0x00351371, 0x00351771, 0x00351b71, 0x00351f71, // 15680-15687
	0x00352371, 0x00352771, 0x00352b71, 0x00352f71, 0x00353371, 0x00353771, 0x00353b71, 0x00353f71, // 15688-15695
	0x00354371, 0x00354771, 0x00354b71, 0x00354f71, 0x00355371, 0x00355771, 0x00355b71, 0x00355f71, // 15696-15703
	0x00356371, 0x00356771, 0x00356b71, 0x00356f71, 0x00357371, 0x00357771, 0x00357b71, 0x00357f71, // 15704-15711
	0x00358371, 0x00358771, 0x00358b71, 0x00358f71, 0x00359371, 0x00359771, 0x00359b71, 0x00359f71, // 15712-15719
	0x0035a371, 0x0035a771, 0x0035ab71, 0x0035af71, 0x0035b371, 0x0035b771, 0x0035bb71, 0x0035bf71, // 15720-15727
	0x0035c371, 0x0035c771, 0x0035cb71, 0x0035cf71, 0x0035d371, 0x0035d771, 0x0035db71, 0x0035df71, // 15728-15735
	0x0035e371, 0x0035e771, 0x0035eb71, 0x0035ef71, 0x0035f371, 0x0035f771, 0x0035fb71, 0x0035ff71, // 15736-15743
	0x00360371, 0x00360771, 0x00360b71, 0x00360f71, 0x00361371, 0x00361771, 0x00361b71, 0x00361f71, // 15744-15751
	0x00362371, 0x00362771, 0x00362b71, 0x00362f71, 0x00363371, 0x00363771, 0x00363b71, 0x00363f71, // 15752-15759
	0x00364371, 0x00364771, 0x00364b71, 0x00364f71, 0x00365371, 0x00365771, 0x00365b71, 0x00365f71, // 15760-15767
	0x00366371, 0x00366771, 0x00366b71, 0x00366f71, 0x00367371, 0x00367771, 0x00367b71, 0x00367f71, // 15768-15775
	0x00368371, 0x00368771, 0x00368b71, 0x00368f71, 0x00369371, 0x00369771, 0x00369b71, 0x00369f71, // 15776-15783
	0x0036a371, 0x0036a771, 0x0036ab71, 0x0036af71, 0x0036b371, 0x0036b771, 0x0036bb71, 0x0036bf71, // 15784-15791
	0x0036c371, 0x0036c771, 0x0036cb71, 0x0036cf71, 0x0036d371, 0x0036d771, 0x0036db71, 0x0036df71, // 15792-15799
	0x0036e371, 0x0036e771, 0x0036eb71, 0x0036ef71, 0x0036f371, 0x0036f771, 0x0036fb71, 0x0036ff71, // 15800-15807
	0x00370371, 0x00370771, 0x00370b71, 0x00370f71, 0x00371371, 0x00371771, 0x00371b71, 0x00371f71, // 15808-15815
	0x00372371, 0x00372771, 0x00372b71, 0x00372f71, 0x00373371, 0x00373771, 0x00373b71, 0x00373f71, // 15816-15823
	0x00374371, 0x00374771, 0x00374b71, 0x00374f71, 0x00375371, 0x00375771, 0x00375b71, 0x00375f71, // 15824-15831
	0x00376371, 0x00376771, 0x00376b71, 0x00376f71, 0x00377371, 0x00377771, 0x00377b71, 0x00377f71, // 15832-15839
	0x00378371, 0x00378771, 0x00378b71, 0x00378f71, 0x00379371, 0x00379771, 0x00379b71, 0x00379f71, // 15840-15847
	0x0037a371, 0x0037a771, 0x0037ab71, 0x0037af71, 0x0037b371, 0x0037b771, 0x0037bb71, 0x0037bf71, // 15848-15855
	0x0037c371, 0x0037c771, 0x0037cb71, 0x0037cf71, 0x0037d371, 0x0037d771, 0x0037db71, 0x0037df71, // 15856-15863
	0x0037e371, 0x0037e771, 0x0037eb71, 0x0037ef71, 0x0037f371, 0x0037f771, 0x0037fb71, 0x0037ff71, // 15864-15871
	0x00380371, 0x00380771, 0x00380b71, 0x00380f71, 0x00381371, 0x00381771, 0x00381b71, 0x00381f71, // 15872-15879
	0x00382371, 0x00382771, 0x00382b71, 0x00382f71, 0x00383371, 0x00383771, 0x00383b71, 0x00383f71, // 15880-15887
	0x00384371, 0x00384771, 0x00384b71, 0x00384f71, 0x00385371, 0x00385771, 0x00385b71, 0x00385f71, // 15888-15895
	0x00386371, 0x00386771, 0x00386b71, 0x00386f71, 0x00387371, 0x00387771, 0x00387b71, 0x00387f71, // 15896-15903
	0x00388371, 0x00388771, 0x00388b71, 0x00388f71, 0x00389371, 0x00389771, 0x00389b71, 0x00389f71, // 15904-15911
	0x0038a371, 0x0038a771, 0x0038ab71, 0x0038af71, 0x0038b371, 0x0038b771, 0x0038bb71, 0x0038bf71, // 15912-15919
	0x0038c371, 0x0038c771, 0x0038cb71, 0x0038cf71, 0x0038d371, 0x0038d771, 0x0038db71, 0x0038df71, // 15920-15927
	0x0038e371, 0x0038e771, 0x0038eb71, 0x0038ef71, 0x0038f371, 0x0038f771, 0x0038fb71, 0x0038ff71, // 15928-15935
	0x00390371, 0x00390771, 0x00390b71, 0x00390f71, 0x00391371, 0x00391771, 0x00391b71, 0x00391f71, // 15936-15943
	0x00392371, 0x00392771, 0x00392b71, 0x00392f71, 0x00393371, 0x00393771, 0x00393b71, 0x00393f71, // 15944-15951
	0x00394371, 0x00394771, 0x00394b71, 0x00394f71, 0x00395371, 0x00395771, 0x00395b71, 0x00395f71, // 15952-15959
	0x00396371, 0x00396771, 0x00396b71, 0x00396f71, 0x00397371, 0x00397771, 0x00397b71, 0x00397f71, // 15960-15967
	0x00398371, 0x00398771, 0x00398b71, 0x00398f71, 0x00399371, 0x00399771, 0x00399b71, 0x00399f71, // 15968-15975
	0x0039a371, 0x0039a771, 0x0039ab71, 0x0039af71, 0x0039b371, 0x0039b771, 0x0039bb71, 0x0039bf71, // 15976-15983
	0x0039c371, 0x0039c771, 0x0039cb71, 0x0039cf71, 0x0039d371, 0x0039d771, 0x0039db71, 0x0039df71, // 15984-15991
	0x0039e371, 0x0039e771, 0x0039eb71, 0x0039ef71, 0x0039f371, 0x0039f771, 0x0039fb71, 0x0039ff71, // 15992-15999
	0x003a0371, 0x003a0771, 0x003a0b71, 0x003a0f71, 0x003a1371, 0x003a1771, 0x003a1b71, 0x003a1f71, // 16000-16007
	0x003a2371, 0x003a2771, 0x003a2b71, 0x003a2f71, 0x003a3371, 0x003a3771, 0x003a3b71, 0x003a3f71, // 16008-16015
	0x003a4371, 0x003a4771, 0x003a4b71, 0x003a4f71, 0x003a5371, 0x003a5771, 0x003a5b71, 0x003a5f71, // 16016-16023
	0x003a6371, 0x003a6771, 0x003a6b71, 0x003a6f71, 0x003a7371, 0x003a7771, 0x003a7b71, 0x003a7f71, // 16024-16031
	0x003a8371, 0x003a8771, 0x003a8b71, 0x003a8f71, 0x003a9371, 0x003a9771, 0x003a9b71, 0x003a9f71, // 16032-16039
	0x003aa371, 0x003aa771, 0x003aab71, 0x003aaf71, 0x003ab371, 0x003ab771, 0x003abb71, 0x003abf71, // 16040-16047
	0x003ac371, 0x003ac771, 0x003acb71, 0x003acf71, 0x003ad371, 0x003ad771, 0x003adb71, 0x003adf71, // 16048-16055
	0x003ae371, 0x003ae771, 0x003aeb71, 0x003aef71, 0x003af371, 0x003af771, 0x003afb71, 0x003aff71, // 16056-16063
	0x003b0371, 0x003b0771, 0x003b0b71, 0x003b0f71, 0x003b1371, 0x003b1771, 0x003b1b71, 0x003b1f71, // 16064-16071
	0x003b2371, 0x003b2771, 0x003b2b71, 0x003b2f71, 0x003b3371, 0x003b3771, 0x003b3b71, 0x003b3f71, // 16072-16079
	0x003b4371, 0x003b4771, 0x003b4b71, 0x003b4f71, 0x003b5371, 0x003b5771, 0x003b5b71, 0x003b5f71, // 16080-16087
	0x003b6371, 0x003b6771, 0x003b6b71, 0x003b6f71, 0x003b7371, 0x003b7771, 0x003b7b71, 0x003b7f71, // 16088-16095
	0x003b8371, 0x003b8771, 0x003b8b71, 0x003b8f71, 0x003b9371, 0x003b9771, 0x003b9b71, 0x003b9f71, // 16096-16103
	0x003ba371, 0x003ba771, 0x003bab71, 0x003baf71, 0x003bb371, 0x003bb771, 0x003bbb71, 0x003bbf71, // 16104-16111
	0x003bc371, 0x003bc771, 0x003bcb71, 0x003bcf71, 0x003bd371, 0x003bd771, 0x003bdb71, 0x003bdf71, // 16112-16119
	0x003be371, 0x003be771, 0x003beb71, 0x003bef71, 0x003bf371, 0x003bf771, 0x003bfb71, 0x003bff71, // 16120-16127
	0x003c0371, 0x003c0771, 0x003c0b71, 0x003c0f71, 0x003c1371, 0x003c1771, 0x003c1b71, 0x003c1f71, // 16128-16135
	0x003c2371, 0x003c2771, 0x003c2b71, 0x003c2f71, 0x003c3371, 0x003c3771, 0x003c3b71, 0x003c3f71, // 16136-16143
	0x003c4371, 0x003c4771, 0x003c4b71, 0x003c4f71, 0x003c5371, 0x003c5771, 0x003c5b71, 0x003c5f71, // 16144-16151
	0x003c6371, 0x003c6771, 0x003c6b71, 0x003c6f71, 0x003c7371, 0x003c7771, 0x003c7b71, 0x003c7f71, // 16152-16159
	0x003c8371, 0x003c8771, 0x003c8b71, 0x003c8f71, 0x003c9371, 0x003c9771, 0x003c9b71, 0x003c9f71, // 16160-16167
	0x003ca371, 0x003ca771, 0x003cab71, 0x003caf71, 0x003cb371, 0x003cb771, 0x003cbb71, 0x003cbf71, // 16168-16175
	0x003cc371, 0x003cc771, 0x003ccb71, 0x003ccf71, 0x003cd371, 0x003cd771, 0x003cdb71, 0x003cdf71, // 16176-16183
	0x003ce371, 0x003ce771, 0x003ceb71, 0x003cef71, 0x003cf371, 0x003cf771, 0x003cfb71, 0x003cff71, // 16184-16191
	0x003d0371, 0x003d0771, 0x003d0b71, 0x003d0f71, 0x003d1371, 0x003d1771, 0x003d1b71, 0x003d1f71, // 16192-16199
	0x003d2371, 0x003d2771, 0x003d2b71, 0x003d2f71, 0x003d3371, 0x003d3771, 0x003d3b71, 0x003d3f71, // 16200-16207
	0x003d4371, 0x003d4771, 0x003d4b71, 0x003d4f71, 0x003d5371, 0x003d5771, 0x003d5b71, 0x003d5f71, // 16208-16215
	0x003d6371, 0x003d6771, 0x003d6b71, 0x003d6f71, 0x003d7371, 0x003d7771, 0x003d7b71, 0x003d7f71, // 16216-16223
	0x003d8371, 0x003d8771, 0x003d8b71, 0x003d8f71, 0x003d9371, 0x003d9771, 0x003d9b71, 0x003d9f71, // 16224-16231
	0x003da371, 0x003da771, 0x003dab71, 0x003daf71, 0x003db371, 0x003db771, 0x003dbb71, 0x003dbf71, // 16232-16239
	0x003dc371, 0x003dc771, 0x003dcb71, 0x003dcf71, 0x003dd371, 0x003dd771, 0x003ddb71, 0x003ddf71, // 16240-16247
	0x003de371, 0x003de771, 0x003deb71, 0x003def71, 0x003df371, 0x003df771, 0x003dfb71, 0x003dff71, // 16248-16255
	0x003e0371, 0x003e0771, 0x003e0b71, 0x003e0f71, 0x003e1371, 0x003e1771, 0x003e1b71, 0x003e1f71, // 16256-16263
	0x003e2371, 0x003e2771, 0x003e2b71, 0x003e2f71, 0x003e3371, 0x003e3771, 0x003e3b71, 0x003e3f71, // 16264-16271
	0x003e4371, 0x003e4771, 0x003e4b71, 0x003e4f71, 0x003e5371, 0x003e5771, 0x003e5b71, 0x003e5f71, // 16272-16279
	0x003e6371, 0x003e6771, 0x003e6b71, 0x003e6f71, 0x003e7371, 0x003e7771, 0x003e7b71, 0x003e7f71, // 16280-16287
	0x003e8371, 0x003e8771, 0x003e8b71, 0x003e8f71, 0x003e9371, 0x003e9771, 0x003e9b71, 0x003e9f71, // 16288-16295
	0x003ea371, 0x003ea771, 0x003eab71, 0x003eaf71, 0x003eb371, 0x003eb771, 0x003ebb71, 0x003ebf71, // 16296-16303
	0x003ec371, 0x003ec771, 0x003ecb71, 0x003ecf71, 0x003ed371, 0x003ed771, 0x003edb71, 0x003edf71, // 16304-16311
	0x003ee371, 0x003ee771, 0x003eeb71, 0x003eef71, 0x003ef371, 0x003ef771, 0x003efb71, 0x003eff71, // 16312-16319
	0x003f0371, 0x003f0771, 0x003f0b71, 0x003f0f71, 0x003f1371, 0x003f1771, 0x003f1b71, 0x003f1f71, // 16320-16327
	0x003f2371, 0x003f2771, 0x003f2b71, 0x003f2f71, 0x003f3371, 0x003f3771, 0x003f3b71, 0x003f3f71, // 16328-16335
	0x003f4371, 0x003f4771, 0x003f4b71, 0x003f4f71, 0x003f5371, 0x003f5771, 0x003f5b71, 0x003f5f71, // 16336-16343
	0x003f6371, 0x003f6771, 0x003f6b71, 0x003f6f71, 0x003f7371, 0x003f7771, 0x003f7b71, 0x003f7f71, // 16344-16351
	0x003f8371, 0x003f8771, 0x003f8b71, 0x003f8f71, 0x003f9371, 0x003f9771, 0x003f9b71, 0x003f9f71, // 16352-16359
	0x003fa371, 0x003fa771, 0x003fab71, 0x003faf71, 0x003fb371, 0x003fb771, 0x003fbb71, 0x003fbf71, // 16360-16367
	0x003fc371, 0x003fc771, 0x003fcb71, 0x003fcf71, 0x003fd371, 0x003fd771, 0x003fdb71, 0x003fdf71, // 16368-16375
	0x003fe371, 0x003fe771, 0x003feb71, 0x003fef71, 0x003ff371, 0x003ff771, 0x003ffb71, 0x003fff71, // 16376-16383
	0x000000f2, 0x000004f2, 0x000008f2, 0x00000cf2, 0x000010f2, 0x000014f2, 0x000018f2, 0x00001cf2, // 16384-16391
	0x000020f2, 0x000024f2, 0x000028f2, 0x00002cf2, 0x000030f2, 0x000034f2, 0x000038f2, 0x00003cf2, // 16392-16399
	0x000040f2, 0x000044f2, 0x000048f2, 0x00004cf2, 0x000050f2, 0x000054f2, 0x000058f2, 0x00005cf2, // 16400-16407
	0x000060f2, 0x000064f2, 0x000068f2, 0x00006cf2, 0x000070f2, 0x000074f2, 0x000078f2, 0x00007cf2, // 16408-16415
	0x000080f2, 0x000084f2, 0x000088f2, 0x00008cf2, 0x000090f2, 0x000094f2, 0x000098f2, 0x00009cf2, // 16416-16423
	0x0000a0f2, 0x0000a4f2, 0x0000a8f2, 0x0000acf2, 0x0000b0f2, 0x0000b4f2, 0x0000b8f2, 0x0000bcf2, // 16424-16431
	0x0000c0f2, 0x0000c4f2, 0x0000c8f2, 0x0000ccf2, 0x0000d0f2, 0x0000d4f2, 0x0000d8f2, 0x0000dcf2, // 16432-16439
	0x0000e0f2, 0x0000e4f2, 0x0000e8f2, 0x0000ecf2, 0x0000f0f2, 0x0000f4f2, 0x0000f8f2, 0x0000fcf2, // 16440-16447
	0x000100f2, 0x000104f2, 0x000108f2, 0x00010cf2, 0x000110f2, 0x000114f2, 0x000118f2, 0x00011cf2, // 16448-16455
	0x000120f2, 0x000124f2, 0x000128f2, 0x00012cf2, 0x000130f2, 0x000134f2, 0x000138f2, 0x00013cf2, // 16456-16463
	0x000140f2, 0x000144f2, 0x000148f2, 0x00014cf2, 0x000150f2, 0x000154f2, 0x000158f2, 0x00015cf2, // 16464-16471
	0x000160f2, 0x000164f2, 0x000168f2, 0x00016cf2, 0x000170f2, 0x000174f2, 0x000178f2, 0x00017cf2, // 16472-16479
	0x000180f2, 0x000184f2, 0x000188f2, 0x00018cf2, 0x000190f2, 0x000194f2, 0x000198f2, 0x00019cf2, // 16480-16487
	0x0001a0f2, 0x0001a4f2, 0x0001a8f2, 0x0001acf2, 0x0001b0f2, 0x0001b4f2, 0x0001b8f2, 0x0001bcf2, // 16488-16495
	0x0001c0f2, 0x0001c4f2, 0x0001c8f2, 0x0001ccf2, 0x0001d0f2, 0x0001d4f2, 0x0001d8f2, 0x0001dcf2, // 16496-16503
	0x0001e0f2, 0x0001e4f2, 0x0001e8f2, 0x0001ecf2, 0x0001f0f2, 0x0001f4f2, 0x0001f8f2, 0x0001fcf2, // 16504-16511
	0x000200f2, 0x000204f2, 0x000208f2, 0x00020cf2, 0x000210f2, 0x000214f2, 0x000218f2, 0x00021cf2, // 16512-16519
	0x000220f2, 0x000224f2, 0x000228f2, 0x00022cf2, 0x000230f2, 0x000234f2, 0x000238f2, 0x00023cf2, // 16520-16527
	0x000240f2, 0x000244f2, 0x000248f2, 0x00024cf2, 0x000250f2, 0x000254f2, 0x000258f2, 0x00025cf2, // 16528-16535
	0x000260f2, 0x000264f2, 0x000268f2, 0x00026cf2, 0x000270f2, 0x000274f2, 0x000278f2, 0x00027cf2, // 16536-16543
	0x000280f2, 0x000284f2, 0x000288f2, 0x00028cf2, 0x000290f2, 0x000294f2, 0x000298f2, 0x00029cf2, // 16544-16551
	0x0002a0f2, 0x0002a4f2, 0x0002a8f2, 0x0002acf2, 0x0002b0f2, 0x0002b4f2, 0x0002b8f2, 0x0002bcf2, // 16552-16559
	0x0002c0f2, 0x0002c4f2, 0x0002c8f2, 0x0002ccf2, 0x0002d0f2, 0x0002d4f2, 0x0002d8f2, 0x0002dcf2, // 16560-16567
	0x0002e0f2, 0x0002e4f2, 0x0002e8f2, 0x0002ecf2, 0x0002f0f2, 0x0002f4f2, 0x0002f8f2, 0x0002fcf2, // 16568-16575
	0x000300f2, 0x000304f2, 0x000308f2, 0x00030cf2, 0x000310f2, 0x000314f2, 0x000318f2, 0x00031cf2, // 16576-16583
	0x000320f2, 0x000324f2, 0x000328f2, 0x00032cf2, 0x000330f2, 0x000334f2, 0x000338f2, 0x00033cf2, // 16584-16591
	0x000340f2, 0x000344f2, 0x000348f2, 0x00034cf2, 0x000350f2, 0x000354f2, 0x000358f2, 0x00035cf2, // 16592-16599
	0x000360f2, 0x000364f2, 0x000368f2, 0x00036cf2, 0x000370f2, 0x000374f2, 0x000378f2, 0x00037cf2, // 16600-16607
	0x000380f2, 0x000384f2, 0x000388f2, 0x00038cf2, 0x000390f2, 0x000394f2, 0x000398f2, 0x00039cf2, // 16608-16615
	0x0003a0f2, 0x0003a4f2, 0x0003a8f2, 0x0003acf2, 0x0003b0f2, 0x0003b4f2, 0x0003b8f2, 0x0003bcf2, // 16616-16623
	0x0003c0f2, 0x0003c4f2, 0x0003c8f2, 0x0003ccf2, 0x0003d0f2, 0x0003d4f2, 0x0003d8f2, 0x0003dcf2, // 16624-16631
	0x0003e0f2, 0x0003e4f2, 0x0003e8f2, 0x0003ecf2, 0x0003f0f2, 0x0003f4f2, 0x0003f8f2, 0x0003fcf2, // 16632-16639
	0x000400f2, 0x000404f2, 0x000408f2, 0x00040cf2, 0x000410f2, 0x000414f2, 0x000418f2, 0x00041cf2, // 16640-16647
	0x000420f2, 0x000424f2, 0x000428f2, 0x00042cf2, 0x000430f2, 0x000434f2, 0x000438f2, 0x00043cf2, // 16648-16655
	0x000440f2, 0x000444f2, 0x000448f2, 0x00044cf2, 0x000450f2, 0x000454f2, 0x000458f2, 0x00045cf2, // 16656-16663
	0x000460f2, 0x000464f2, 0x000468f2, 0x00046cf2, 0x000470f2, 0x000474f2, 0x000478f2, 0x00047cf2, // 16664-16671
	0x000480f2, 0x000484f2, 0x000488f2, 0x00048cf2, 0x000490f2, 0x000494f2, 0x000498f2, 0x00049cf2, // 16672-16679
	0x0004a0f2, 0x0004a4f2, 0x0004a8f2, 0x0004acf2, 0x0004b0f2, 0x0004b4f2, 0x0004b8f2, 0x0004bcf2, // 16680-16687
	0x0004c0f2, 0x0004c4f2, 0x0004c8f2, 0x0004ccf2, 0x0004d0f2, 0x0004d4f2, 0x0004d8f2, 0x0004dcf2, // 16688-16695
	0x0004e0f2, 0x0004e4f2, 0x0004e8f2, 0x0004ecf2, 0x0004f0f2, 0x0004f4f2, 0x0004f8f2, 0x0004fcf2, // 16696-16703
	0x000500f2, 0x000504f2, 0x000508f2, 0x00050cf2, 0x000510f2, 0x000514f2, 0x000518f2, 0x00051cf2, // 16704-16711
	0x000520f2, 0x000524f2, 0x000528f2, 0x00052cf2, 0x000530f2, 0x000534f2, 0x000538f2, 0x00053cf2, // 16712-16719
	0x000540f2, 0x000544f2, 0x000548f2, 0x00054cf2, 0x000550f2, 0x000554f2, 0x000558f2, 0x00055cf2, // 16720-16727
	0x000560f2, 0x000564f2, 0x000568f2, 0x00056cf2, 0x000570f2, 0x000574f2, 0x000578f2, 0x00057cf2, // 16728-16735
	0x000580f2, 0x000584f2, 0x000588f2, 0x00058cf2, 0x000590f2, 0x000594f2, 0x000598f2, 0x00059cf2, // 16736-16743
	0x0005a0f2, 0x0005a4f2, 0x0005a8f2, 0x0005acf2, 0x0005b0f2, 0x0005b4f2, 0x0005b8f2, 0x0005bcf2, // 16744-16751
	0x0005c0f2, 0x0005c4f2, 0x0005c8f2, 0x0005ccf2, 0x0005d0f2, 0x0005d4f2, 0x0005d8f2, 0x0005dcf2, // 16752-16759
	0x0005e0f2, 0x0005e4f2, 0x0005e8f2, 0x0005ecf2, 0x0005f0f2, 0x0005f4f2, 0x0005f8f2, 0x0005fcf2, // 16760-16767
	0x000600f2, 0x000604f2, 0x000608f2, 0x00060cf2, 0x000610f2, 0x000614f2, 0x000618f2, 0x00061cf2, // 16768-16775
	0x000620f2, 0x000624f2, 0x000628f2, 0x00062cf2, 0x000630f2, 0x000634f2, 0x000638f2, 0x00063cf2, // 16776-16783
	0x000640f2, 0x000644f2, 0x000648f2, 0x00064cf2, 0x000650f2, 0x000654f2, 0x000658f2, 0x00065cf2, // 16784-16791
	0x000660f2, 0x000664f2, 0x000668f2, 0x00066cf2, 0x000670f2, 0x000674f2, 0x000678f2, 0x00067cf2, // 16792-16799
	0x000680f2, 0x000684f2, 0x000688f2, 0x00068cf2, 0x000690f2, 0x000694f2, 0x000698f2, 0x00069cf2, // 16800-16807
	0x0006a0f2, 0x0006a4f2, 0x0006a8f2, 0x0006acf2, 0x0006b0f2, 0x0006b4f2, 0x0006b8f2, 0x0006bcf2, // 16808-16815
	0x0006c0f2, 0x0006c4f2, 0x0006c8f2, 0x0006ccf2, 0x0006d0f2, 0x0006d4f2, 0x0006d8f2, 0x0006dcf2, // 16816-16823
	0x0006e0f2, 0x0006e4f2, 0x0006e8f2, 0x0006ecf2, 0x0006f0f2, 0x0006f4f2, 0x0006f8f2, 0x0006fcf2, // 16824-16831
	0x000700f2, 0x000704f2, 0x000708f2, 0x00070cf2, 0x000710f2, 0x000714f2, 0x000718f2, 0x00071cf2, // 16832-16839
	0x000720f2, 0x000724f2, 0x000728f2, 0x00072cf2, 0x000730f2, 0x000734f2, 0x000738f2, 0x00073cf2, // 16840-16847
	0x000740f2, 0x000744f2, 0x000748f2, 0x00074cf2, 0x000750f2, 0x000754f2, 0x000758f2, 0x00075cf2, // 16848-16855
	0x000760f2, 0x000764f2, 0x000768f2, 0x00076cf2, 0x000770f2, 0x000774f2, 0x000778f2, 0x00077cf2, // 16856-16863
	0x000780f2, 0x000784f2, 0x000788f2, 0x00078cf2, 0x000790f2, 0x000794f2, 0x000798f2, 0x00079cf2, // 16864-16871
	0x0007a0f2, 0x0007a4f2, 0x0007a8f2, 0x0007acf2, 0x0007b0f2, 0x0007b4f2, 0x0007b8f2, 0x0007bcf2, // 16872-16879
	0x0007c0f2, 0x0007c4f2, 0x0007c8f2, 0x0007ccf2, 0x0007d0f2, 0x0007d4f2, 0x0007d8f2, 0x0007dcf2, // 16880-16887
	0x0007e0f2, 0x0007e4f2, 0x0007e8f2, 0x0007ecf2, 0x0007f0f2, 0x0007f4f2, 0x0007f8f2, 0x0007fcf2, // 16888-16895
	0x000800f2, 0x000804f2, 0x000808f2, 0x00080cf2, 0x000810f2, 0x000814f2, 0x000818f2, 0x00081cf2, // 16896-16903
	0x000820f2, 0x000824f2, 0x000828f2, 0x00082cf2, 0x000830f2, 0x000834f2, 0x000838f2, 0x00083cf2, // 16904-16911
	0x000840f2, 0x000844f2, 0x000848f2, 0x00084cf2, 0x000850f2, 0x000854f2, 0x000858f2, 0x00085cf2, // 16912-16919
	0x000860f2, 0x000864f2, 0x000868f2, 0x00086cf2, 0x000870f2, 0x000874f2, 0x000878f2, 0x00087cf2, // 16920-16927
	0x000880f2, 0x000884f2, 0x000888f2, 0x00088cf2, 0x000890f2, 0x000894f2, 0x000898f2, 0x00089cf2, // 16928-16935
	0x0008a0f2, 0x0008a4f2, 0x0008a8f2, 0x0008acf2, 0x0008b0f2, 0x0008b4f2, 0x0008b8f2, 0x0008bcf2, // 16936-16943
	0x0008c0f2, 0x0008c4f2, 0x0008c8f2, 0x0008ccf2, 0x0008d0f2, 0x0008d4f2, 0x0008d8f2, 0x0008dcf2, // 16944-16951
	0x0008e0f2, 0x0008e4f2, 0x0008e8f2, 0x0008ecf2, 0x0008f0f2, 0x0008f4f2, 0x0008f8f2, 0x0008fcf2, // 16952-16959
	0x000900f2, 0x000904f2, 0x000908f2, 0x00090cf2, 0x000910f2, 0x000914f2, 0x000918f2, 0x00091cf2, // 16960-16967
	0x000920f2, 0x000924f2, 0x000928f2, 0x00092cf2, 0x000930f2, 0x000934f2, 0x000938f2, 0x00093cf2, // 16968-16975
	0x000940f2, 0x000944f2, 0x000948f2, 0x00094cf2, 0x000950f2, 0x000954f2, 0x000958f2, 0x00095cf2, // 16976-16983
	0x000960f2, 0x000964f2, 0x000968f2, 0x00096cf2, 0x000970f2, 0x000974f2, 0x000978f2, 0x00097cf2, // 16984-16991
	0x000980f2, 0x000984f2, 0x000988f2, 0x00098cf2, 0x000990f2, 0x000994f2, 0x000998f2, 0x00099cf2, // 16992-16999
	0x0009a0f2, 0x0009a4f2, 0x0009a8f2, 0x0009acf2, 0x0009b0f2, 0x0009b4f2, 0x0009b8f2, 0x0009bcf2, // 17000-17007
	0x0009c0f2, 0x0009c4f2, 0x0009c8f2, 0x0009ccf2, 0x0009d0f2, 0x0009d4f2, 0x0009d8f2, 0x0009dcf2, // 17008-17015
	0x0009e0f2, 0x0009e4f2, 0x0009e8f2, 0x0009ecf2, 0x0009f0f2, 0x0009f4f2, 0x0009f8f2, 0x0009fcf2, // 17016-17023
	0x000a00f2, 0x000a04f2, 0x000a08f2, 0x000a0cf2, 0x000a10f2, 0x000a14f2, 0x000a18f2, 0x000a1cf2, // 17024-17031
	0x000a20f2, 0x000a24f2, 0x000a28f2, 0x000a2cf2, 0x000a30f2, 0x000a34f2, 0x000a38f2, 0x000a3cf2, // 17032-17039
	0x000a40f2, 0x000a44f2, 0x000a48f2, 0x000a4cf2, 0x000a50f2, 0x000a54f2, 0x000a58f2, 0x000a5cf2, // 17040-17047
	0x000a60f2, 0x000a64f2, 0x000a68f2, 0x000a6cf2, 0x000a70f2, 0x000a74f2, 0x000a78f2, 0x000a7cf2, // 17048-17055
	0x000a80f2, 0x000a84f2, 0x000a88f2, 0x000a8cf2, 0x000a90f2, 0x000a94f2, 0x000a98f2, 0x000a9cf2, // 17056-17063
	0x000aa0f2, 0x000aa4f2, 0x000aa8f2, 0x000aacf2, 0x000ab0f2, 0x000ab4f2, 0x000ab8f2, 0x000abcf2, // 17064-17071
	0x000ac0f2, 0x000ac4f2, 0x000ac8f2, 0x000accf2, 0x000ad0f2, 0x000ad4f2, 0x000ad8f2, 0x000adcf2, // 17072-17079
	0x000ae0f2, 0x000ae4f2, 0x000ae8f2, 0x000aecf2, 0x000af0f2, 0x000af4f2, 0x000af8f2, 0x000afcf2, // 17080-17087
	0x000b00f2, 0x000b04f2, 0x000b08f2, 0x000b0cf2, 0x000b10f2, 0x000b14f2, 0x000b18f2, 0x000b1cf2, // 17088-17095
	0x000b20f2, 0x000b24f2, 0x000b28f2, 0x000b2cf2, 0x000b30f2, 0x000b34f2, 0x000b38f2, 0x000b3cf2, // 17096-17103
	0x000b40f2, 0x000b44f2, 0x000b48f2, 0x000b4cf2, 0x000b50f2, 0x000b54f2, 0x000b58f2, 0x000b5cf2, // 17104-17111
	0x000b60f2, 0x000b64f2, 0x000b68f2, 0x000b6cf2, 0x000b70f2, 0x000b74f2, 0x000b78f2, 0x000b7cf2, // 17112-17119
	0x000b80f2, 0x000b84f2, 0x000b88f2, 0x000b8cf2, 0x000b90f2, 0x000b94f2, 0x000b98f2, 0x000b9cf2, // 17120-17127
	0x000ba0f2, 0x000ba4f2, 0x000ba8f2, 0x000bacf2, 0x000bb0f2, 0x000bb4f2, 0x000bb8f2, 0x000bbcf2, // 17128-17135
	0x000bc0f2, 0x000bc4f2, 0x000bc8f2, 0x000bccf2, 0x000bd0f2, 0x000bd4f2, 0x000bd8f2, 0x000bdcf2, // 17136-17143
	0x000be0f2, 0x000be4f2, 0x000be8f2, 0x000becf2, 0x000bf0f2, 0x000bf4f2, 0x000bf8f2, 0x000bfcf2, // 17144-17151
	0x000c00f2, 0x000c04f2, 0x000c08f2, 0x000c0cf2, 0x000c10f2, 0x000c14f2, 0x000c18f2, 0x000c1cf2, // 17152-17159
	0x000c20f2, 0x000c24f2, 0x000c28f2, 0x000c2cf2, 0x000c30f2, 0x000c34f2, 0x000c38f2, 0x000c3cf2, // 17160-17167
	0x000c40f2, 0x000c44f2, 0x000c48f2, 0x000c4cf2, 0x000c50f2, 0x000c54f2, 0x000c58f2, 0x000c5cf2, // 17168-17175
	0x000c60f2, 0x000c64f2, 0x000c68f2, 0x000c6cf2, 0x000c70f2, 0x000c74f2, 0x000c78f2, 0x000c7cf2, // 17176-17183
	0x000c80f2, 0x000c84f2, 0x000c88f2, 0x000c8cf2, 0x000c90f2, 0x000c94f2, 0x000c98f2, 0x000c9cf2, // 17184-17191
	0x000ca0f2, 0x000ca4f2, 0x000ca8f2, 0x000cacf2, 0x000cb0f2, 0x000cb4f2, 0x000cb8f2, 0x000cbcf2, // 17192-17199
	0x000cc0f2, 0x000cc4f2, 0x000cc8f2, 0x000cccf2, 0x000cd0f2, 0x000cd4f2, 0x000cd8f2, 0x000cdcf2, // 17200-17207
	0x000ce0f2, 0x000ce4f2, 0x000ce8f2, 0x000cecf2, 0x000cf0f2, 0x000cf4f2, 0x000cf8f2, 0x000cfcf2, // 17208-17215
	0x000d00f2, 0x000d04f2, 0x000d08f2, 0x000d0cf2, 0x000d10f2, 0x000d14f2, 0x000d18f2, 0x000d1cf2, // 17216-17223
	0x000d20f2, 0x000d24f2, 0x000d28f2, 0x000d2cf2, 0x000d30f2, 0x000d34f2, 0x000d38f2, 0x000d3cf2, // 17224-17231
	0x000d40f2, 0x000d44f2, 0x000d48f2, 0x000d4cf2, 0x000d50f2, 0x000d54f2, 0x000d58f2, 0x000d5cf2, // 17232-17239
	0x000d60f2, 0x000d64f2, 0x000d68f2, 0x000d6cf2, 0x000d70f2, 0x000d74f2, 0x000d78f2, 0x000d7cf2, // 17240-17247
	0x000d80f2, 0x000d84f2, 0x000d88f2, 0x000d8cf2, 0x000d90f2, 0x000d94f2, 0x000d98f2, 0x000d9cf2, // 17248-17255
	0x000da0f2, 0x000da4f2, 0x000da8f2, 0x000dacf2, 0x000db0f2, 0x000db4f2, 0x000db8f2, 0x000dbcf2, // 17256-17263
	0x000dc0f2, 0x000dc4f2, 0x000dc8f2, 0x000dccf2, 0x000dd0f2, 0x000dd4f2, 0x000dd8f2, 0x000ddcf2, // 17264-17271
	0x000de0f2, 0x000de4f2, 0x000de8f2, 0x000decf2, 0x000df0f2, 0x000df4f2, 0x000df8f2, 0x000dfcf2, // 17272-17279
	0x000e00f2, 0x000e04f2, 0x000e08f2, 0x000e0cf2, 0x000e10f2, 0x000e14f2, 0x000e18f2, 0x000e1cf2, // 17280-17287
	0x000e20f2, 0x000e24f2, 0x000e28f2, 0x000e2cf2, 0x000e30f2, 0x000e34f2, 0x000e38f2, 0x000e3cf2, // 17288-17295
	0x000e40f2, 0x000e44f2, 0x000e48f2, 0x000e4cf2, 0x000e50f2, 0x000e54f2, 0x000e58f2, 0x000e5cf2, // 17296-17303
	0x000e60f2, 0x000e64f2, 0x000e68f2, 0x000e6cf2, 0x000e70f2, 0x000e74f2, 0x000e78f2, 0x000e7cf2, // 17304-17311
	0x000e80f2, 0x000e84f2, 0x000e88f2, 0x000e8cf2, 0x000e90f2, 0x000e94f2, 0x000e98f2, 0x000e9cf2, // 17312-17319
	0x000ea0f2, 0x000ea4f2, 0x000ea8f2, 0x000eacf2, 0x000eb0f2, 0x000eb4f2, 0x000eb8f2, 0x000ebcf2, // 17320-17327
	0x000ec0f2, 0x000ec4f2, 0x000ec8f2, 0x000eccf2, 0x000ed0f2, 0x000ed4f2, 0x000ed8f2, 0x000edcf2, // 17328-17335
	0x000ee0f2, 0x000ee4f2, 0x000ee8f2, 0x000eecf2, 0x000ef0f2, 0x000ef4f2, 0x000ef8f2, 0x000efcf2, // 17336-17343
	0x000f00f2, 0x000f04f2, 0x000f08f2, 0x000f0cf2, 0x000f10f2, 0x000f14f2, 0x000f18f2, 0x000f1cf2, // 17344-17351
	0x000f20f2, 0x000f24f2, 0x000f28f2, 0x000f2cf2, 0x000f30f2, 0x000f34f2, 0x000f38f2, 0x000f3cf2, // 17352-17359
	0x000f40f2, 0x000f44f2, 0x000f48f2, 0x000f4cf2, 0x000f50f2, 0x000f54f2, 0x000f58f2, 0x000f5cf2, // 17360-17367
	0x000f60f2, 0x000f64f2, 0x000f68f2, 0x000f6cf2, 0x000f70f2, 0x000f74f2, 0x000f78f2, 0x000f7cf2, // 17368-17375
	0x000f80f2, 0x000f84f2, 0x000f88f2, 0x000f8cf2, 0x000f90f2, 0x000f94f2, 0x000f98f2, 0x000f9cf2, // 17376-17383
	0x000fa0f2, 0x000fa4f2, 0x000fa8f2, 0x000facf2, 0x000fb0f2, 0x000fb4f2, 0x000fb8f2, 0x000fbcf2, // 17384-17391
	0x000fc0f2, 0x000fc4f2, 0x000fc8f2, 0x000fccf2, 0x000fd0f2, 0x000fd4f2, 0x000fd8f2, 0x000fdcf2, // 17392-17399
	0x000fe0f2, 0x000fe4f2, 0x000fe8f2, 0x000fecf2, 0x000ff0f2, 0x000ff4f2, 0x000ff8f2, 0x000ffcf2, // 17400-17407
	0x001000f2, 0x001004f2, 0x001008f2, 0x00100cf2, 0x001010f2, 0x001014f2, 0x001018f2, 0x00101cf2, // 17408-17415
	0x001020f2, 0x001024f2, 0x001028f2, 0x00102cf2, 0x001030f2, 0x001034f2, 0x001038f2, 0x00103cf2, // 17416-17423
	0x001040f2, 0x001044f2, 0x001048f2, 0x00104cf2, 0x001050f2, 0x001054f2, 0x001058f2, 0x00105cf2, // 17424-17431
	0x001060f2, 0x001064f2, 0x001068f2, 0x00106cf2, 0x001070f2, 0x001074f2, 0x001078f2, 0x00107cf2, // 17432-17439
	0x001080f2, 0x001084f2, 0x001088f2, 0x00108cf2, 0x001090f2, 0x001094f2, 0x001098f2, 0x00109cf2, // 17440-17447
	0x0010a0f2, 0x0010a4f2, 0x0010a8f2, 0x0010acf2, 0x0010b0f2, 0x0010b4f2, 0x0010b8f2, 0x0010bcf2, // 17448-17455
	0x0010c0f2, 0x0010c4f2, 0x0010c8f2, 0x0010ccf2, 0x0010d0f2, 0x0010d4f2, 0x0010d8f2, 0x0010dcf2, // 17456-17463
	0x0010e0f2, 0x0010e4f2, 0x0010e8f2, 0x0010ecf2, 0x0010f0f2, 0x0010f4f2, 0x0010f8f2, 0x0010fcf2, // 17464-17471
	0x001100f2, 0x001104f2, 0x001108f2, 0x00110cf2, 0x001110f2, 0x001114f2, 0x001118f2, 0x00111cf2, // 17472-17479
	0x001120f2, 0x001124f2, 0x001128f2, 0x00112cf2, 0x001130f2, 0x001134f2, 0x001138f2, 0x00113cf2, // 17480-17487
	0x001140f2, 0x001144f2, 0x001148f2, 0x00114cf2, 0x001150f2, 0x001154f2, 0x001158f2, 0x00115cf2, // 17488-17495
	0x001160f2, 0x001164f2, 0x001168f2, 0x00116cf2, 0x001170f2, 0x001174f2, 0x001178f2, 0x00117cf2, // 17496-17503
	0x001180f2, 0x001184f2, 0x001188f2, 0x00118cf2, 0x001190f2, 0x001194f2, 0x001198f2, 0x00119cf2, // 17504-17511
	0x0011a0f2, 0x0011a4f2, 0x0011a8f2, 0x0011acf2, 0x0011b0f2, 0x0011b4f2, 0x0011b8f2, 0x0011bcf2, // 17512-17519
	0x0011c0f2, 0x0011c4f2, 0x0011c8f2, 0x0011ccf2, 0x0011d0f2, 0x0011d4f2, 0x0011d8f2, 0x0011dcf2, // 17520-17527
	0x0011e0f2, 0x0011e4f2, 0x0011e8f2, 0x0011ecf2, 0x0011f0f2, 0x0011f4f2, 0x0011f8f2, 0x0011fcf2, // 17528-17535
	0x001200f2, 0x001204f2, 0x001208f2, 0x00120cf2, 0x001210f2, 0x001214f2, 0x001218f2, 0x00121cf2, // 17536-17543
	0x001220f2, 0x001224f2, 0x001228f2, 0x00122cf2, 0x001230f2, 0x001234f2, 0x001238f2, 0x00123cf2, // 17544-17551
	0x001240f2, 0x001244f2, 0x001248f2, 0x00124cf2, 0x001250f2, 0x001254f2, 0x001258f2, 0x00125cf2, // 17552-17559
	0x001260f2, 0x001264f2, 0x001268f2, 0x00126cf2, 0x001270f2, 0x001274f2, 0x001278f2, 0x00127cf2, // 17560-17567
	0x001280f2, 0x001284f2, 0x001288f2, 0x00128cf2, 0x001290f2, 0x001294f2, 0x001298f2, 0x00129cf2, // 17568-17575
	0x0012a0f2, 0x0012a4f2, 0x0012a8f2, 0x0012acf2, 0x0012b0f2, 0x0012b4f2, 0x0012b8f2, 0x0012bcf2, // 17576-17583
	0x0012c0f2, 0x0012c4f2, 0x0012c8f2, 0x0012ccf2, 0x0012d0f2, 0x0012d4f2, 0x0012d8f2, 0x0012dcf2, // 17584-17591
	0x0012e0f2, 0x0012e4f2, 0x0012e8f2, 0x0012ecf2, 0x0012f0f2, 0x0012f4f2, 0x0012f8f2, 0x0012fcf2, // 17592-17599
	0x001300f2, 0x001304f2, 0x001308f2, 0x00130cf2, 0x001310f2, 0x001314f2, 0x001318f2, 0x00131cf2, // 17600-17607
	0x001320f2, 0x001324f2, 0x001328f2, 0x00132cf2, 0x001330f2, 0x001334f2, 0x001338f2, 0x00133cf2, // 17608-17615
	0x001340f2, 0x001344f2, 0x001348f2, 0x00134cf2, 0x001350f2, 0x001354f2, 0x001358f2, 0x00135cf2, // 17616-17623
	0x001360f2, 0x001364f2, 0x001368f2, 0x00136cf2, 0x001370f2, 0x001374f2, 0x001378f2, 0x00137cf2, // 17624-17631
	0x001380f2, 0x001384f2, 0x001388f2, 0x00138cf2, 0x001390f2, 0x001394f2, 0x001398f2, 0x00139cf2, // 17632-17639
	0x0013a0f2, 0x0013a4f2, 0x0013a8f2, 0x0013acf2, 0x0013b0f2, 0x0013b4f2, 0x0013b8f2, 0x0013bcf2, // 17640-17647
	0x0013c0f2, 0x0013c4f2, 0x0013c8f2, 0x0013ccf2, 0x0013d0f2, 0x0013d4f2, 0x0013d8f2, 0x0013dcf2, // 17648-17655
	0x0013e0f2, 0x0013e4f2, 0x0013e8f2, 0x0013ecf2, 0x0013f0f2, 0x0013f4f2, 0x0013f8f2, 0x0013fcf2, // 17656-17663
	0x001400f2, 0x001404f2, 0x001408f2, 0x00140cf2, 0x001410f2, 0x001414f2, 0x001418f2, 0x00141cf2, // 17664-17671
	0x001420f2, 0x001424f2, 0x001428f2, 0x00142cf2, 0x001430f2, 0x001434f2, 0x001438f2, 0x00143cf2, // 17672-17679
	0x001440f2, 0x001444f2, 0x001448f2, 0x00144cf2, 0x001450f2, 0x001454f2, 0x001458f2, 0x00145cf2, // 17680-17687
	0x001460f2, 0x001464f2, 0x001468f2, 0x00146cf2, 0x001470f2, 0x001474f2, 0x001478f2, 0x00147cf2, // 17688-17695
	0x001480f2, 0x001484f2, 0x001488f2, 0x00148cf2, 0x001490f2, 0x001494f2, 0x001498f2, 0x00149cf2, // 17696-17703
	0x0014a0f2, 0x0014a4f2, 0x0014a8f2, 0x0014acf2, 0x0014b0f2, 0x0014b4f2, 0x0014b8f2, 0x0014bcf2, // 17704-17711
	0x0014c0f2, 0x0014c4f2, 0x0014c8f2, 0x0014ccf2, 0x0014d0f2, 0x0014d4f2, 0x0014d8f2, 0x0014dcf2, // 17712-17719
	0x0014e0f2, 0x0014e4f2, 0x0014e8f2, 0x0014ecf2, 0x0014f0f2, 0x0014f4f2, 0x0014f8f2, 0x0014fcf2, // 17720-17727
	0x001500f2, 0x001504f2, 0x001508f2, 0x00150cf2, 0x001510f2, 0x001514f2, 0x001518f2, 0x00151cf2, // 17728-17735
	0x001520f2, 0x001524f2, 0x001528f2, 0x00152cf2, 0x001530f2, 0x001534f2, 0x001538f2, 0x00153cf2, // 17736-17743
	0x001540f2, 0x001544f2, 0x001548f2, 0x00154cf2, 0x001550f2, 0x001554f2, 0x001558f2, 0x00155cf2, // 17744-17751
	0x001560f2, 0x001564f2, 0x001568f2, 0x00156cf2, 0x001570f2, 0x001574f2, 0x001578f2, 0x00157cf2, // 17752-17759
	0x001580f2, 0x001584f2, 0x001588f2, 0x00158cf2, 0x001590f2, 0x001594f2, 0x001598f2, 0x00159cf2, // 17760-17767
	0x0015a0f2, 0x0015a4f2, 0x0015a8f2, 0x0015acf2, 0x0015b0f2, 0x0015b4f2, 0x0015b8f2, 0x0015bcf2, // 17768-17775
	0x0015c0f2, 0x0015c4f2, 0x0015c8f2, 0x0015ccf2, 0x0015d0f2, 0x0015d4f2, 0x0015d8f2, 0x0015dcf2, // 17776-17783
	0x0015e0f2, 0x0015e4f2, 0x0015e8f2, 0x0015ecf2, 0x0015f0f2, 0x0015f4f2, 0x0015f8f2, 0x0015fcf2, // 17784-17791
	0x001600f2, 0x001604f2, 0x001608f2, 0x00160cf2, 0x001610f2, 0x001614f2, 0x001618f2, 0x00161cf2, // 17792-17799
	0x001620f2, 0x001624f2, 0x001628f2, 0x00162cf2, 0x001630f2, 0x001634f2, 0x001638f2, 0x00163cf2, // 17800-17807
	0x001640f2, 0x001644f2, 0x001648f2, 0x00164cf2, 0x001650f2, 0x001654f2, 0x001658f2, 0x00165cf2, // 17808-17815
	0x001660f2, 0x001664f2, 0x001668f2, 0x00166cf2, 0x001670f2, 0x001674f2, 0x001678f2, 0x00167cf2, // 17816-17823
	0x001680f2, 0x001684f2, 0x001688f2, 0x00168cf2, 0x001690f2, 0x001694f2, 0x001698f2, 0x00169cf2, // 17824-17831
	0x0016a0f2, 0x0016a4f2, 0x0016a8f2, 0x0016acf2, 0x0016b0f2, 0x0016b4f2, 0x0016b8f2, 0x0016bcf2, // 17832-17839
	0x0016c0f2, 0x0016c4f2, 0x0016c8f2, 0x0016ccf2, 0x0016d0f2, 0x0016d4f2, 0x0016d8f2, 0x0016dcf2, // 17840-17847
	0x0016e0f2, 0x0016e4f2, 0x0016e8f2, 0x0016ecf2, 0x0016f0f2, 0x0016f4f2, 0x0016f8f2, 0x0016fcf2, // 17848-17855
	0x001700f2, 0x001704f2, 0x001708f2, 0x00170cf2, 0x001710f2, 0x001714f2, 0x001718f2, 0x00171cf2, // 17856-17863
	0x001720f2, 0x001724f2, 0x001728f2, 0x00172cf2, 0x001730f2, 0x001734f2, 0x001738f2, 0x00173cf2, // 17864-17871
	0x001740f2, 0x001744f2, 0x001748f2, 0x00174cf2, 0x001750f2, 0x001754f2, 0x001758f2, 0x00175cf2, // 17872-17879
	0x001760f2, 0x001764f2, 0x001768f2, 0x00176cf2, 0x001770f2, 0x001774f2, 0x001778f2, 0x00177cf2, // 17880-17887
	0x001780f2, 0x001784f2, 0x001788f2, 0x00178cf2, 0x001790f2, 0x001794f2, 0x001798f2, 0x00179cf2, // 17888-17895
	0x0017a0f2, 0x0017a4f2, 0x0017a8f2, 0x0017acf2, 0x0017b0f2, 0x0017b4f2, 0x0017b8f2, 0x0017bcf2, // 17896-17903
	0x0017c0f2, 0x0017c4f2, 0x0017c8f2, 0x0017ccf2, 0x0017d0f2, 0x0017d4f2, 0x0017d8f2, 0x0017dcf2, // 17904-17911
	0x0017e0f2, 0x0017e4f2, 0x0017e8f2, 0x0017ecf2, 0x0017f0f2, 0x0017f4f2, 0x0017f8f2, 0x0017fcf2, // 17912-17919
	0x001800f2, 0x001804f2, 0x001808f2, 0x00180cf2, 0x001810f2, 0x001814f2, 0x001818f2, 0x00181cf2, // 17920-17927
	0x001820f2, 0x001824f2, 0x001828f2, 0x00182cf2, 0x001830f2, 0x001834f2, 0x001838f2, 0x00183cf2, // 17928-17935
	0x001840f2, 0x001844f2, 0x001848f2, 0x00184cf2, 0x001850f2, 0x001854f2, 0x001858f2, 0x00185cf2, // 17936-17943
	0x001860f2, 0x001864f2, 0x001868f2, 0x00186cf2, 0x001870f2, 0x001874f2, 0x001878f2, 0x00187cf2, // 17944-17951
	0x001880f2, 0x001884f2, 0x001888f2, 0x00188cf2, 0x001890f2, 0x001894f2, 0x001898f2, 0x00189cf2, // 17952-17959
	0x0018a0f2, 0x0018a4f2, 0x0018a8f2, 0x0018acf2, 0x0018b0f2, 0x0018b4f2, 0x0018b8f2, 0x0018bcf2, // 17960-17967
	0x0018c0f2, 0x0018c4f2, 0x0018c8f2, 0x0018ccf2, 0x0018d0f2, 0x0018d4f2, 0x0018d8f2, 0x0018dcf2, // 17968-17975
	0x0018e0f2, 0x0018e4f2, 0x0018e8f2, 0x0018ecf2, 0x0018f0f2, 0x0018f4f2, 0x0018f8f2, 0x0018fcf2, // 17976-17983
	0x001900f2, 0x001904f2, 0x001908f2, 0x00190cf2, 0x001910f2, 0x001914f2, 0x001918f2, 0x00191cf2, // 17984-17991
	0x001920f2, 0x001924f2, 0x001928f2, 0x00192cf2, 0x001930f2, 0x001934f2, 0x001938f2, 0x00193cf2, // 17992-17999
	0x001940f2, 0x001944f2, 0x001948f2, 0x00194cf2, 0x001950f2, 0x001954f2, 0x001958f2, 0x00195cf2, // 18000-18007
	0x001960f2, 0x001964f2, 0x001968f2, 0x00196cf2, 0x001970f2, 0x001974f2, 0x001978f2, 0x00197cf2, // 18008-18015
	0x001980f2, 0x001984f2, 0x001988f2, 0x00198cf2, 0x001990f2, 0x001994f2, 0x001998f2, 0x00199cf2, // 18016-18023
	0x0019a0f2, 0x0019a4f2, 0x0019a8f2, 0x0019acf2, 0x0019b0f2, 0x0019b4f2, 0x0019b8f2, 0x0019bcf2, // 18024-18031
	0x0019c0f2, 0x0019c4f2, 0x0019c8f2, 0x0019ccf2, 0x0019d0f2, 0x0019d4f2, 0x0019d8f2, 0x0019dcf2, // 18032-18039
	0x0019e0f2, 0x0019e4f2, 0x0019e8f2, 0x0019ecf2, 0x0019f0f2, 0x0019f4f2, 0x0019f8f2, 0x0019fcf2, // 18040-18047
	0x001a00f2, 0x001a04f2, 0x001a08f2, 0x001a0cf2, 0x001a10f2, 0x001a14f2, 0x001a18f2, 0x001a1cf2, // 18048-18055
	0x001a20f2, 0x001a24f2, 0x001a28f2, 0x001a2cf2, 0x001a30f2, 0x001a34f2, 0x001a38f2, 0x001a3cf2, // 18056-18063
	0x001a40f2, 0x001a44f2, 0x001a48f2, 0x001a4cf2, 0x001a50f2, 0x001a54f2, 0x001a58f2, 0x001a5cf2, // 18064-18071
	0x001a60f2, 0x001a64f2, 0x001a68f2, 0x001a6cf2, 0x001a70f2, 0x001a74f2, 0x001a78f2, 0x001a7cf2, // 18072-18079
	0x001a80f2, 0x001a84f2, 0x001a88f2, 0x001a8cf2, 0x001a90f2, 0x001a94f2, 0x001a98f2, 0x001a9cf2, // 18080-18087
	0x001aa0f2, 0x001aa4f2, 0x001aa8f2, 0x001aacf2, 0x001ab0f2, 0x001ab4f2, 0x001ab8f2, 0x001abcf2, // 18088-18095
	0x001ac0f2, 0x001ac4f2, 0x001ac8f2, 0x001accf2, 0x001ad0f2, 0x001ad4f2, 0x001ad8f2, 0x001adcf2, // 18096-18103
	0x001ae0f2, 0x001ae4f2, 0x001ae8f2, 0x001aecf2, 0x001af0f2, 0x001af4f2, 0x001af8f2, 0x001afcf2, // 18104-18111
	0x001b00f2, 0x001b04f2, 0x001b08f2, 0x001b0cf2, 0x001b10f2, 0x001b14f2, 0x001b18f2, 0x001b1cf2, // 18112-18119
	0x001b20f2, 0x001b24f2, 0x001b28f2, 0x001b2cf2, 0x001b30f2, 0x001b34f2, 0x001b38f2, 0x001b3cf2, // 18120-18127
	0x001b40f2, 0x001b44f2, 0x001b48f2, 0x001b4cf2, 0x001b50f2, 0x001b54f2, 0x001b58f2, 0x001b5cf2, // 18128-18135
	0x001b60f2, 0x001b64f2, 0x001b68f2, 0x001b6cf2, 0x001b70f2, 0x001b74f2, 0x001b78f2, 0x001b7cf2, // 18136-18143
	0x001b80f2, 0x001b84f2, 0x001b88f2, 0x001b8cf2, 0x001b90f2, 0x001b94f2, 0x001b98f2, 0x001b9cf2, // 18144-18151
	0x001ba0f2, 0x001ba4f2, 0x001ba8f2, 0x001bacf2, 0x001bb0f2, 0x001bb4f2, 0x001bb8f2, 0x001bbcf2, // 18152-18159
	0x001bc0f2, 0x001bc4f2, 0x001bc8f2, 0x001bccf2, 0x001bd0f2, 0x001bd4f2, 0x001bd8f2, 0x001bdcf2, // 18160-18167
	0x001be0f2, 0x001be4f2, 0x001be8f2, 0x001becf2, 0x001bf0f2, 0x001bf4f2, 0x001bf8f2, 0x001bfcf2, // 18168-18175
	0x001c00f2, 0x001c04f2, 0x001c08f2, 0x001c0cf2, 0x001c10f2, 0x001c14f2, 0x001c18f2, 0x001c1cf2, // 18176-18183
	0x001c20f2, 0x001c24f2, 0x001c28f2, 0x001c2cf2, 0x001c30f2, 0x001c34f2, 0x001c38f2, 0x001c3cf2, // 18184-18191
	0x001c40f2, 0x001c44f2, 0x001c48f2, 0x001c4cf2, 0x001c50f2, 0x001c54f2, 0x001c58f2, 0x001c5cf2, // 18192-18199
	0x001c60f2, 0x001c64f2, 0x001c68f2, 0x001c6cf2, 0x001c70f2, 0x001c74f2, 0x001c78f2, 0x001c7cf2, // 18200-18207
	0x001c80f2, 0x001c84f2, 0x001c88f2, 0x001c8cf2, 0x001c90f2, 0x001c94f2, 0x001c98f2, 0x001c9cf2, // 18208-18215
	0x001ca0f2, 0x001ca4f2, 0x001ca8f2, 0x001cacf2, 0x001cb0f2, 0x001cb4f2, 0x001cb8f2, 0x001cbcf2, // 18216-18223
	0x001cc0f2, 0x001cc4f2, 0x001cc8f2, 0x001cccf2, 0x001cd0f2, 0x001cd4f2, 0x001cd8f2, 0x001cdcf2, // 18224-18231
	0x001ce0f2, 0x001ce4f2, 0x001ce8f2, 0x001cecf2, 0x001cf0f2, 0x001cf4f2, 0x001cf8f2, 0x001cfcf2, // 18232-18239
	0x001d00f2, 0x001d04f2, 0x001d08f2, 0x001d0cf2, 0x001d10f2, 0x001d14f2, 0x001d18f2, 0x001d1cf2, // 18240-18247
	0x001d20f2, 0x001d24f2, 0x001d28f2, 0x001d2cf2, 0x001d30f2, 0x001d34f2, 0x001d38f2, 0x001d3cf2, // 18248-18255
	0x001d40f2, 0x001d44f2, 0x001d48f2, 0x001d4cf2, 0x001d50f2, 0x001d54f2, 0x001d58f2, 0x001d5cf2, // 18256-18263
	0x001d60f2, 0x001d64f2, 0x001d68f2, 0x001d6cf2, 0x001d70f2, 0x001d74f2, 0x001d78f2, 0x001d7cf2, // 18264-18271
	0x001d80f2, 0x001d84f2, 0x001d88f2, 0x001d8cf2, 0x001d90f2, 0x001d94f2, 0x001d98f2, 0x001d9cf2, // 18272-18279
	0x001da0f2, 0x001da4f2, 0x001da8f2, 0x001dacf2, 0x001db0f2, 0x001db4f2, 0x001db8f2, 0x001dbcf2, // 18280-18287
	0x001dc0f2, 0x001dc4f2, 0x001dc8f2, 0x001dccf2, 0x001dd0f2, 0x001dd4f2, 0x001dd8f2, 0x001ddcf2, // 18288-18295
	0x001de0f2, 0x001de4f2, 0x001de8f2, 0x001decf2, 0x001df0f2, 0x001df4f2, 0x001df8f2, 0x001dfcf2, // 18296-18303
	0x001e00f2, 0x001e04f2, 0x001e08f2, 0x001e0cf2, 0x001e10f2, 0x001e14f2, 0x001e18f2, 0x001e1cf2, // 18304-18311
	0x001e20f2, 0x001e24f2, 0x001e28f2, 0x001e2cf2, 0x001e30f2, 0x001e34f2, 0x001e38f2, 0x001e3cf2, // 18312-18319
	0x001e40f2, 0x001e44f2, 0x001e48f2, 0x001e4cf2, 0x001e50f2, 0x001e54f2, 0x001e58f2, 0x001e5cf2, // 18320-18327
	0x001e60f2, 0x001e64f2, 0x001e68f2, 0x001e6cf2, 0x001e70f2, 0x001e74f2, 0x001e78f2, 0x001e7cf2, // 18328-18335
	0x001e80f2, 0x001e84f2, 0x001e88f2, 0x001e8cf2, 0x001e90f2, 0x001e94f2, 0x001e98f2, 0x001e9cf2, // 18336-18343
	0x001ea0f2, 0x001ea4f2, 0x001ea8f2, 0x001eacf2, 0x001eb0f2, 0x001eb4f2, 0x001eb8f2, 0x001ebcf2, // 18344-18351
	0x001ec0f2, 0x001ec4f2, 0x001ec8f2, 0x001eccf2, 0x001ed0f2, 0x001ed4f2, 0x001ed8f2, 0x001edcf2, // 18352-18359
	0x001ee0f2, 0x001ee4f2, 0x001ee8f2, 0x001eecf2, 0x001ef0f2, 0x001ef4f2, 0x001ef8f2, 0x001efcf2, // 18360-18367
	0x001f00f2, 0x001f04f2, 0x001f08f2, 0x001f0cf2, 0x001f10f2, 0x001f14f2, 0x001f18f2, 0x001f1cf2, // 18368-18375
	0x001f20f2, 0x001f24f2, 0x001f28f2, 0x001f2cf2, 0x001f30f2, 0x001f34f2, 0x001f38f2, 0x001f3cf2, // 18376-18383
	0x001f40f2, 0x001f44f2, 0x001f48f2, 0x001f4cf2, 0x001f50f2, 0x001f54f2, 0x001f58f2, 0x001f5cf2, // 18384-18391
	0x001f60f2, 0x001f64f2, 0x001f68f2, 0x001f6cf2, 0x001f70f2, 0x001f74f2, 0x001f78f2, 0x001f7cf2, // 18392-18399
	0x001f80f2, 0x001f84f2, 0x001f88f2, 0x001f8cf2, 0x001f90f2, 0x001f94f2, 0x001f98f2, 0x001f9cf2, // 18400-18407
	0x001fa0f2, 0x001fa4f2, 0x001fa8f2, 0x001facf2, 0x001fb0f2, 0x001fb4f2, 0x001fb8f2, 0x001fbcf2, // 18408-18415
	0x001fc0f2, 0x001fc4f2, 0x001fc8f2, 0x001fccf2, 0x001fd0f2, 0x001fd4f2, 0x001fd8f2, 0x001fdcf2, // 18416-18423
	0x001fe0f2, 0x001fe4f2, 0x001fe8f2, 0x001fecf2, 0x001ff0f2, 0x001ff4f2, 0x001ff8f2, 0x001ffcf2, // 18424-18431
	0x002000f2, 0x002004f2, 0x002008f2, 0x00200cf2, 0x002010f2, 0x002014f2, 0x002018f2, 0x00201cf2, // 18432-18439
	0x002020f2, 0x002024f2, 0x002028f2, 0x00202cf2, 0x002030f2, 0x002034f2, 0x002038f2, 0x00203cf2, // 18440-18447
	0x002040f2, 0x002044f2, 0x002048f2, 0x00204cf2, 0x002050f2, 0x002054f2, 0x002058f2, 0x00205cf2, // 18448-18455
	0x002060f2, 0x002064f2, 0x002068f2, 0x00206cf2, 0x002070f2, 0x002074f2, 0x002078f2, 0x00207cf2, // 18456-18463
	0x002080f2, 0x002084f2, 0x002088f2, 0x00208cf2, 0x002090f2, 0x002094f2, 0x002098f2, 0x00209cf2, // 18464-18471
	0x0020a0f2, 0x0020a4f2, 0x0020a8f2, 0x0020acf2, 0x0020b0f2, 0x0020b4f2, 0x0020b8f2, 0x0020bcf2, // 18472-18479
	0x0020c0f2, 0x0020c4f2, 0x0020c8f2, 0x0020ccf2, 0x0020d0f2, 0x0020d4f2, 0x0020d8f2, 0x0020dcf2, // 18480-18487
	0x0020e0f2, 0x0020e4f2, 0x0020e8f2, 0x0020ecf2, 0x0020f0f2, 0x0020f4f2, 0x0020f8f2, 0x0020fcf2, // 18488-18495
	0x002100f2, 0x002104f2, 0x002108f2, 0x00210cf2, 0x002110f2, 0x002114f2, 0x002118f2, 0x00211cf2, // 18496-18503
	0x002120f2, 0x002124f2, 0x002128f2, 0x00212cf2, 0x002130f2, 0x002134f2, 0x002138f2, 0x00213cf2, // 18504-18511
	0x002140f2, 0x002144f2, 0x002148f2, 0x00214cf2, 0x002150f2, 0x002154f2, 0x002158f2, 0x00215cf2, // 18512-18519
	0x002160f2, 0x002164f2, 0x002168f2, 0x00216cf2, 0x002170f2, 0x002174f2, 0x002178f2, 0x00217cf2, // 18520-18527
	0x002180f2, 0x002184f2, 0x002188f2, 0x00218cf2, 0x002190f2, 0x002194f2, 0x002198f2, 0x00219cf2, // 18528-18535
	0x0021a0f2, 0x0021a4f2, 0x0021a8f2, 0x0021acf2, 0x0021b0f2, 0x0021b4f2, 0x0021b8f2, 0x0021bcf2, // 18536-18543
	0x0021c0f2, 0x0021c4f2, 0x0021c8f2, 0x0021ccf2, 0x0021d0f2, 0x0021d4f2, 0x0021d8f2, 0x0021dcf2, // 18544-18551
	0x0021e0f2, 0x0021e4f2, 0x0021e8f2, 0x0021ecf2, 0x0021f0f2, 0x0021f4f2, 0x0021f8f2, 0x0021fcf2, // 18552-18559
	0x002200f2, 0x002204f2, 0x002208f2, 0x00220cf2, 0x002210f2, 0x002214f2, 0x002218f2, 0x00221cf2, // 18560-18567
	0x002220f2, 0x002224f2, 0x002228f2, 0x00222cf2, 0x002230f2, 0x002234f2, 0x002238f2, 0x00223cf2, // 18568-18575
	0x002240f2, 0x002244f2, 0x002248f2, 0x00224cf2, 0x002250f2, 0x002254f2, 0x002258f2, 0x00225cf2, // 18576-18583
	0x002260f2, 0x002264f2, 0x002268f2, 0x00226cf2, 0x002270f2, 0x002274f2, 0x002278f2, 0x00227cf2, // 18584-18591
	0x002280f2, 0x002284f2, 0x002288f2, 0x00228cf2, 0x002290f2, 0x002294f2, 0x002298f2, 0x00229cf2, // 18592-18599
	0x0022a0f2, 0x0022a4f2, 0x0022a8f2, 0x0022acf2, 0x0022b0f2, 0x0022b4f2, 0x0022b8f2, 0x0022bcf2, // 18600-18607
	0x0022c0f2, 0x0022c4f2, 0x0022c8f2, 0x0022ccf2, 0x0022d0f2, 0x0022d4f2, 0x0022d8f2, 0x0022dcf2, // 18608-18615
	0x0022e0f2, 0x0022e4f2, 0x0022e8f2, 0x0022ecf2, 0x0022f0f2, 0x0022f4f2, 0x0022f8f2, 0x0022fcf2, // 18616-18623
	0x002300f2, 0x002304f2, 0x002308f2, 0x00230cf2, 0x002310f2, 0x002314f2, 0x002318f2, 0x00231cf2, // 18624-18631
	0x002320f2, 0x002324f2, 0x002328f2, 0x00232cf2, 0x002330f2, 0x002334f2, 0x002338f2, 0x00233cf2, // 18632-18639
	0x002340f2, 0x002344f2, 0x002348f2, 0x00234cf2, 0x002350f2, 0x002354f2, 0x002358f2, 0x00235cf2, // 18640-18647
	0x002360f2, 0x002364f2, 0x002368f2, 0x00236cf2, 0x002370f2, 0x002374f2, 0x002378f2, 0x00237cf2, // 18648-18655
	0x002380f2, 0x002384f2, 0x002388f2, 0x00238cf2, 0x002390f2, 0x002394f2, 0x002398f2, 0x00239cf2, // 18656-18663
	0x0023a0f2, 0x0023a4f2, 0x0023a8f2, 0x0023acf2, 0x0023b0f2, 0x0023b4f2, 0x0023b8f2, 0x0023bcf2, // 18664-18671
	0x0023c0f2, 0x0023c4f2, 0x0023c8f2, 0x0023ccf2, 0x0023d0f2, 0x0023d4f2, 0x0023d8f2, 0x0023dcf2, // 18672-18679
	0x0023e0f2, 0x0023e4f2, 0x0023e8f2, 0x0023ecf2, 0x0023f0f2, 0x0023f4f2, 0x0023f8f2, 0x0023fcf2, // 18680-18687
	0x002400f2, 0x002404f2, 0x002408f2, 0x00240cf2, 0x002410f2, 0x002414f2, 0x002418f2, 0x00241cf2, // 18688-18695
	0x002420f2, 0x002424f2, 0x002428f2, 0x00242cf2, 0x002430f2, 0x002434f2, 0x002438f2, 0x00243cf2, // 18696-18703
	0x002440f2, 0x002444f2, 0x002448f2, 0x00244cf2, 0x002450f2, 0x002454f2, 0x002458f2, 0x00245cf2, // 18704-18711
	0x002460f2, 0x002464f2, 0x002468f2, 0x00246cf2, 0x002470f2, 0x002474f2, 0x002478f2, 0x00247cf2, // 18712-18719
	0x002480f2, 0x002484f2, 0x002488f2, 0x00248cf2, 0x002490f2, 0x002494f2, 0x002498f2, 0x00249cf2, // 18720-18727
	0x0024a0f2, 0x0024a4f2, 0x0024a8f2, 0x0024acf2, 0x0024b0f2, 0x0024b4f2, 0x0024b8f2, 0x0024bcf2, // 18728-18735
	0x0024c0f2, 0x0024c4f2, 0x0024c8f2, 0x0024ccf2, 0x0024d0f2, 0x0024d4f2, 0x0024d8f2, 0x0024dcf2, // 18736-18743
	0x0024e0f2, 0x0024e4f2, 0x0024e8f2, 0x0024ecf2, 0x0024f0f2, 0x0024f4f2, 0x0024f8f2, 0x0024fcf2, // 18744-18751
	0x002500f2, 0x002504f2, 0x002508f2, 0x00250cf2, 0x002510f2, 0x002514f2, 0x002518f2, 0x00251cf2, // 18752-18759
	0x002520f2, 0x002524f2, 0x002528f2, 0x00252cf2, 0x002530f2, 0x002534f2, 0x002538f2, 0x00253cf2, // 18760-18767
	0x002540f2, 0x002544f2, 0x002548f2, 0x00254cf2, 0x002550f2, 0x002554f2, 0x002558f2, 0x00255cf2, // 18768-18775
	0x002560f2, 0x002564f2, 0x002568f2, 0x00256cf2, 0x002570f2, 0x002574f2, 0x002578f2, 0x00257cf2, // 18776-18783
	0x002580f2, 0x002584f2, 0x002588f2, 0x00258cf2, 0x002590f2, 0x002594f2, 0x002598f2, 0x00259cf2, // 18784-18791
	0x0025a0f2, 0x0025a4f2, 0x0025a8f2, 0x0025acf2, 0x0025b0f2, 0x0025b4f2, 0x0025b8f2, 0x0025bcf2, // 18792-18799
	0x0025c0f2, 0x0025c4f2, 0x0025c8f2, 0x0025ccf2, 0x0025d0f2, 0x0025d4f2, 0x0025d8f2, 0x0025dcf2, // 18800-18807
	0x0025e0f2, 0x0025e4f2, 0x0025e8f2, 0x0025ecf2, 0x0025f0f2, 0x0025f4f2, 0x0025f8f2, 0x0025fcf2, // 18808-18815
	0x002600f2, 0x002604f2, 0x002608f2, 0x00260cf2, 0x002610f2, 0x002614f2, 0x002618f2, 0x00261cf2, // 18816-18823
	0x002620f2, 0x002624f2, 0x002628f2, 0x00262cf2, 0x002630f2, 0x002634f2, 0x002638f2, 0x00263cf2, // 18824-18831
	0x002640f2, 0x002644f2, 0x002648f2, 0x00264cf2, 0x002650f2, 0x002654f2, 0x002658f2, 0x00265cf2, // 18832-18839
	0x002660f2, 0x002664f2, 0x002668f2, 0x00266cf2, 0x002670f2, 0x002674f2, 0x002678f2, 0x00267cf2, // 18840-18847
	0x002680f2, 0x002684f2, 0x002688f2, 0x00268cf2, 0x002690f2, 0x002694f2, 0x002698f2, 0x00269cf2, // 18848-18855
	0x0026a0f2, 0x0026a4f2, 0x0026a8f2, 0x0026acf2, 0x0026b0f2, 0x0026b4f2, 0x0026b8f2, 0x0026bcf2, // 18856-18863
	0x0026c0f2, 0x0026c4f2, 0x0026c8f2, 0x0026ccf2, 0x0026d0f2, 0x0026d4f2, 0x0026d8f2, 0x0026dcf2, // 18864-18871
	0x0026e0f2, 0x0026e4f2, 0x0026e8f2, 0x0026ecf2, 0x0026f0f2, 0x0026f4f2, 0x0026f8f2, 0x0026fcf2, // 18872-18879
	0x002700f2, 0x002704f2, 0x002708f2, 0x00270cf2, 0x002710f2, 0x002714f2, 0x002718f2, 0x00271cf2, // 18880-18887
	0x002720f2, 0x002724f2, 0x002728f2, 0x00272cf2, 0x002730f2, 0x002734f2, 0x002738f2, 0x00273cf2, // 18888-18895
	0x002740f2, 0x002744f2, 0x002748f2, 0x00274cf2, 0x002750f2, 0x002754f2, 0x002758f2, 0x00275cf2, // 18896-18903
	0x002760f2, 0x002764f2, 0x002768f2, 0x00276cf2, 0x002770f2, 0x002774f2, 0x002778f2, 0x00277cf2, // 18904-18911
	0x002780f2, 0x002784f2, 0x002788f2, 0x00278cf2, 0x002790f2, 0x002794f2, 0x002798f2, 0x00279cf2, // 18912-18919
	0x0027a0f2, 0x0027a4f2, 0x0027a8f2, 0x0027acf2, 0x0027b0f2, 0x0027b4f2, 0x0027b8f2, 0x0027bcf2, // 18920-18927
	0x0027c0f2, 0x0027c4f2, 0x0027c8f2, 0x0027ccf2, 0x0027d0f2, 0x0027d4f2, 0x0027d8f2, 0x0027dcf2, // 18928-18935
	0x0027e0f2, 0x0027e4f2, 0x0027e8f2, 0x0027ecf2, 0x0027f0f2, 0x0027f4f2, 0x0027f8f2, 0x0027fcf2, // 18936-18943
	0x002800f2, 0x002804f2, 0x002808f2, 0x00280cf2, 0x002810f2, 0x002814f2, 0x002818f2, 0x00281cf2, // 18944-18951
	0x002820f2, 0x002824f2, 0x002828f2, 0x00282cf2, 0x002830f2, 0x002834f2, 0x002838f2, 0x00283cf2, // 18952-18959
	0x002840f2, 0x002844f2, 0x002848f2, 0x00284cf2, 0x002850f2, 0x002854f2, 0x002858f2, 0x00285cf2, // 18960-18967
	0x002860f2, 0x002864f2, 0x002868f2, 0x00286cf2, 0x002870f2, 0x002874f2, 0x002878f2, 0x00287cf2, // 18968-18975
	0x002880f2, 0x002884f2, 0x002888f2, 0x00288cf2, 0x002890f2, 0x002894f2, 0x002898f2, 0x00289cf2, // 18976-18983
	0x0028a0f2, 0x0028a4f2, 0x0028a8f2, 0x0028acf2, 0x0028b0f2, 0x0028b4f2, 0x0028b8f2, 0x0028bcf2, // 18984-18991
	0x0028c0f2, 0x0028c4f2, 0x0028c8f2, 0x0028ccf2, 0x0028d0f2, 0x0028d4f2, 0x0028d8f2, 0x0028dcf2, // 18992-18999
	0x0028e0f2, 0x0028e4f2, 0x0028e8f2, 0x0028ecf2, 0x0028f0f2, 0x0028f4f2, 0x0028f8f2, 0x0028fcf2, // 19000-19007
	0x002900f2, 0x002904f2, 0x002908f2, 0x00290cf2, 0x002910f2, 0x002914f2, 0x002918f2, 0x00291cf2, // 19008-19015
	0x002920f2, 0x002924f2, 0x002928f2, 0x00292cf2, 0x002930f2, 0x002934f2, 0x002938f2, 0x00293cf2, // 19016-19023
	0x002940f2, 0x002944f2, 0x002948f2, 0x00294cf2, 0x002950f2, 0x002954f2, 0x002958f2, 0x00295cf2, // 19024-19031
	0x002960f2, 0x002964f2, 0x002968f2, 0x00296cf2, 0x002970f2, 0x002974f2, 0x002978f2, 0x00297cf2, // 19032-19039
	0x002980f2, 0x002984f2, 0x002988f2, 0x00298cf2, 0x002990f2, 0x002994f2, 0x002998f2, 0x00299cf2, // 19040-19047
	0x0029a0f2, 0x0029a4f2, 0x0029a8f2, 0x0029acf2, 0x0029b0f2, 0x0029b4f2, 0x0029b8f2, 0x0029bcf2, // 19048-19055
	0x0029c0f2, 0x0029c4f2, 0x0029c8f2, 0x0029ccf2, 0x0029d0f2, 0x0029d4f2, 0x0029d8f2, 0x0029dcf2, // 19056-19063
	0x0029e0f2, 0x0029e4f2, 0x0029e8f2, 0x0029ecf2, 0x0029f0f2, 0x0029f4f2, 0x0029f8f2, 0x0029fcf2, // 19064-19071
	0x002a00f2, 0x002a04f2, 0x002a08f2, 0x002a0cf2, 0x002a10f2, 0x002a14f2, 0x002a18f2, 0x002a1cf2, // 19072-19079
	0x002a20f2, 0x002a24f2, 0x002a28f2, 0x002a2cf2, 0x002a30f2, 0x002a34f2, 0x002a38f2, 0x002a3cf2, // 19080-19087
	0x002a40f2, 0x002a44f2, 0x002a48f2, 0x002a4cf2, 0x002a50f2, 0x002a54f2, 0x002a58f2, 0x002a5cf2, // 19088-19095
	0x002a60f2, 0x002a64f2, 0x002a68f2, 0x002a6cf2, 0x002a70f2, 0x002a74f2, 0x002a78f2, 0x002a7cf2, // 19096-19103
	0x002a80f2, 0x002a84f2, 0x002a88f2, 0x002a8cf2, 0x002a90f2, 0x002a94f2, 0x002a98f2, 0x002a9cf2, // 19104-19111
	0x002aa0f2, 0x002aa4f2, 0x002aa8f2, 0x002aacf2, 0x002ab0f2, 0x002ab4f2, 0x002ab8f2, 0x002abcf2, // 19112-19119
	0x002ac0f2, 0x002ac4f2, 0x002ac8f2, 0x002accf2, 0x002ad0f2, 0x002ad4f2, 0x002ad8f2, 0x002adcf2, // 19120-19127
	0x002ae0f2, 0x002ae4f2, 0x002ae8f2, 0x002aecf2, 0x002af0f2, 0x002af4f2, 0x002af8f2, 0x002afcf2, // 19128-19135
	0x002b00f2, 0x002b04f2, 0x002b08f2, 0x002b0cf2, 0x002b10f2, 0x002b14f2, 0x002b18f2, 0x002b1cf2, // 19136-19143
	0x002b20f2, 0x002b24f2, 0x002b28f2, 0x002b2cf2, 0x002b30f2, 0x002b34f2, 0x002b38f2, 0x002b3cf2, // 19144-19151
	0x002b40f2, 0x002b44f2, 0x002b48f2, 0x002b4cf2, 0x002b50f2, 0x002b54f2, 0x002b58f2, 0x002b5cf2, // 19152-19159
	0x002b60f2, 0x002b64f2, 0x002b68f2, 0x002b6cf2, 0x002b70f2, 0x002b74f2, 0x002b78f2, 0x002b7cf2, // 19160-19167
	0x002b80f2, 0x002b84f2, 0x002b88f2, 0x002b8cf2, 0x002b90f2, 0x002b94f2, 0x002b98f2, 0x002b9cf2, // 19168-19175
	0x002ba0f2, 0x002ba4f2, 0x002ba8f2, 0x002bacf2, 0x002bb0f2, 0x002bb4f2, 0x002bb8f2, 0x002bbcf2, // 19176-19183
	0x002bc0f2, 0x002bc4f2, 0x002bc8f2, 0x002bccf2, 0x002bd0f2, 0x002bd4f2, 0x002bd8f2, 0x002bdcf2, // 19184-19191
	0x002be0f2, 0x002be4f2, 0x002be8f2, 0x002becf2, 0x002bf0f2, 0x002bf4f2, 0x002bf8f2, 0x002bfcf2, // 19192-19199
	0x002c00f2, 0x002c04f2, 0x002c08f2, 0x002c0cf2, 0x002c10f2, 0x002c14f2, 0x002c18f2, 0x002c1cf2, // 19200-19207
	0x002c20f2, 0x002c24f2, 0x002c28f2, 0x002c2cf2, 0x002c30f2, 0x002c34f2, 0x002c38f2, 0x002c3cf2, // 19208-19215
	0x002c40f2, 0x002c44f2, 0x002c48f2, 0x002c4cf2, 0x002c50f2, 0x002c54f2, 0x002c58f2, 0x002c5cf2, // 19216-19223
	0x002c60f2, 0x002c64f2, 0x002c68f2, 0x002c6cf2, 0x002c70f2, 0x002c74f2, 0x002c78f2, 0x002c7cf2, // 19224-19231
	0x002c80f2, 0x002c84f2, 0x002c88f2, 0x002c8cf2, 0x002c90f2, 0x002c94f2, 0x002c98f2, 0x002c9cf2, // 19232-19239
	0x002ca0f2, 0x002ca4f2, 0x002ca8f2, 0x002cacf2, 0x002cb0f2, 0x002cb4f2, 0x002cb8f2, 0x002cbcf2, // 19240-19247
	0x002cc0f2, 0x002cc4f2, 0x002cc8f2, 0x002cccf2, 0x002cd0f2, 0x002cd4f2, 0x002cd8f2, 0x002cdcf2, // 19248-19255
	0x002ce0f2, 0x002ce4f2, 0x002ce8f2, 0x002cecf2, 0x002cf0f2, 0x002cf4f2, 0x002cf8f2, 0x002cfcf2, // 19256-19263
	0x002d00f2, 0x002d04f2, 0x002d08f2, 0x002d0cf2, 0x002d10f2, 0x002d14f2, 0x002d18f2, 0x002d1cf2, // 19264-19271
	0x002d20f2, 0x002d24f2, 0x002d28f2, 0x002d2cf2, 0x002d30f2, 0x002d34f2, 0x002d38f2, 0x002d3cf2, // 19272-19279
	0x002d40f2, 0x002d44f2, 0x002d48f2, 0x002d4cf2, 0x002d50f2, 0x002d54f2, 0x002d58f2, 0x002d5cf2, // 19280-19287
	0x002d60f2, 0x002d64f2, 0x002d68f2, 0x002d6cf2, 0x002d70f2, 0x002d74f2, 0x002d78f2, 0x002d7cf2, // 19288-19295
	0x002d80f2, 0x002d84f2, 0x002d88f2, 0x002d8cf2, 0x002d90f2, 0x002d94f2, 0x002d98f2, 0x002d9cf2, // 19296-19303
	0x002da0f2, 0x002da4f2, 0x002da8f2, 0x002dacf2, 0x002db0f2, 0x002db4f2, 0x002db8f2, 0x002dbcf2, // 19304-19311
	0x002dc0f2, 0x002dc4f2, 0x002dc8f2, 0x002dccf2, 0x002dd0f2, 0x002dd4f2, 0x002dd8f2, 0x002ddcf2, // 19312-19319
	0x002de0f2, 0x002de4f2, 0x002de8f2, 0x002decf2, 0x002df0f2, 0x002df4f2, 0x002df8f2, 0x002dfcf2, // 19320-19327
	0x002e00f2, 0x002e04f2, 0x002e08f2, 0x002e0cf2, 0x002e10f2, 0x002e14f2, 0x002e18f2, 0x002e1cf2, // 19328-19335
	0x002e20f2, 0x002e24f2, 0x002e28f2, 0x002e2cf2, 0x002e30f2, 0x002e34f2, 0x002e38f2, 0x002e3cf2, // 19336-19343
	0x002e40f2, 0x002e44f2, 0x002e48f2, 0x002e4cf2, 0x002e50f2, 0x002e54f2, 0x002e58f2, 0x002e5cf2, // 19344-19351
	0x002e60f2, 0x002e64f2, 0x002e68f2, 0x002e6cf2, 0x002e70f2, 0x002e74f2, 0x002e78f2, 0x002e7cf2, // 19352-19359
	0x002e80f2, 0x002e84f2, 0x002e88f2, 0x002e8cf2, 0x002e90f2, 0x002e94f2, 0x002e98f2, 0x002e9cf2, // 19360-19367
	0x002ea0f2, 0x002ea4f2, 0x002ea8f2, 0x002eacf2, 0x002eb0f2, 0x002eb4f2, 0x002eb8f2, 0x002ebcf2, // 19368-19375
	0x002ec0f2, 0x002ec4f2, 0x002ec8f2, 0x002eccf2, 0x002ed0f2, 0x002ed4f2, 0x002ed8f2, 0x002edcf2, // 19376-19383
	0x002ee0f2, 0x002ee4f2, 0x002ee8f2, 0x002eecf2, 0x002ef0f2, 0x002ef4f2, 0x002ef8f2, 0x002efcf2, // 19384-19391
	0x002f00f2, 0x002f04f2, 0x002f08f2, 0x002f0cf2, 0x002f10f2, 0x002f14f2, 0x002f18f2, 0x002f1cf2, // 19392-19399
	0x002f20f2, 0x002f24f2, 0x002f28f2, 0x002f2cf2, 0x002f30f2, 0x002f34f2, 0x002f38f2, 0x002f3cf2, // 19400-19407
	0x002f40f2, 0x002f44f2, 0x002f48f2, 0x002f4cf2, 0x002f50f2, 0x002f54f2, 0x002f58f2, 0x002f5cf2, // 19408-19415
	0x002f60f2, 0x002f64f2, 0x002f68f2, 0x002f6cf2, 0x002f70f2, 0x002f74f2, 0x002f78f2, 0x002f7cf2, // 19416-19423
	0x002f80f2, 0x002f84f2, 0x002f88f2, 0x002f8cf2, 0x002f90f2, 0x002f94f2, 0x002f98f2, 0x002f9cf2, // 19424-19431
	0x002fa0f2, 0x002fa4f2, 0x002fa8f2, 0x002facf2, 0x002fb0f2, 0x002fb4f2, 0x002fb8f2, 0x002fbcf2, // 19432-19439
	0x002fc0f2, 0x002fc4f2, 0x002fc8f2, 0x002fccf2, 0x002fd0f2, 0x002fd4f2, 0x002fd8f2, 0x002fdcf2, // 19440-19447
	0x002fe0f2, 0x002fe4f2, 0x002fe8f2, 0x002fecf2, 0x002ff0f2, 0x002ff4f2, 0x002ff8f2, 0x002ffcf2, // 19448-19455
	0x003000f2, 0x003004f2, 0x003008f2, 0x00300cf2, 0x003010f2, 0x003014f2, 0x003018f2, 0x00301cf2, // 19456-19463
	0x003020f2, 0x003024f2, 0x003028f2, 0x00302cf2, 0x003030f2, 0x003034f2, 0x003038f2, 0x00303cf2, // 19464-19471
	0x003040f2, 0x003044f2, 0x003048f2, 0x00304cf2, 0x003050f2, 0x003054f2, 0x003058f2, 0x00305cf2, // 19472-19479
	0x003060f2, 0x003064f2, 0x003068f2, 0x00306cf2, 0x003070f2, 0x003074f2, 0x003078f2, 0x00307cf2, // 19480-19487
	0x003080f2, 0x003084f2, 0x003088f2, 0x00308cf2, 0x003090f2, 0x003094f2, 0x003098f2, 0x00309cf2, // 19488-19495
	0x0030a0f2, 0x0030a4f2, 0x0030a8f2, 0x0030acf2, 0x0030b0f2, 0x0030b4f2, 0x0030b8f2, 0x0030bcf2, // 19496-19503
	0x0030c0f2, 0x0030c4f2, 0x0030c8f2, 0x0030ccf2, 0x0030d0f2, 0x0030d4f2, 0x0030d8f2, 0x0030dcf2, // 19504-19511
	0x0030e0f2, 0x0030e4f2, 0x0030e8f2, 0x0030ecf2, 0x0030f0f2, 0x0030f4f2, 0x0030f8f2, 0x0030fcf2, // 19512-19519
	0x003100f2, 0x003104f2, 0x003108f2, 0x00310cf2, 0x003110f2, 0x003114f2, 0x003118f2, 0x00311cf2, // 19520-19527
	0x003120f2, 0x003124f2, 0x003128f2, 0x00312cf2, 0x003130f2, 0x003134f2, 0x003138f2, 0x00313cf2, // 19528-19535
	0x003140f2, 0x003144f2, 0x003148f2, 0x00314cf2, 0x003150f2, 0x003154f2, 0x003158f2, 0x00315cf2, // 19536-19543
	0x003160f2, 0x003164f2, 0x003168f2, 0x00316cf2, 0x003170f2, 0x003174f2, 0x003178f2, 0x00317cf2, // 19544-19551
	0x003180f2, 0x003184f2, 0x003188f2, 0x00318cf2, 0x003190f2, 0x003194f2, 0x003198f2, 0x00319cf2, // 19552-19559
	0x0031a0f2, 0x0031a4f2, 0x0031a8f2, 0x0031acf2, 0x0031b0f2, 0x0031b4f2, 0x0031b8f2, 0x0031bcf2, // 19560-19567
	0x0031c0f2, 0x0031c4f2, 0x0031c8f2, 0x0031ccf2, 0x0031d0f2, 0x0031d4f2, 0x0031d8f2, 0x0031dcf2, // 19568-19575
	0x0031e0f2, 0x0031e4f2, 0x0031e8f2, 0x0031ecf2, 0x0031f0f2, 0x0031f4f2, 0x0031f8f2, 0x0031fcf2, // 19576-19583
	0x003200f2, 0x003204f2, 0x003208f2, 0x00320cf2, 0x003210f2, 0x003214f2, 0x003218f2, 0x00321cf2, // 19584-19591
	0x003220f2, 0x003224f2, 0x003228f2, 0x00322cf2, 0x003230f2, 0x003234f2, 0x003238f2, 0x00323cf2, // 19592-19599
	0x003240f2, 0x003244f2, 0x003248f2, 0x00324cf2, 0x003250f2, 0x003254f2, 0x003258f2, 0x00325cf2, // 19600-19607
	0x003260f2, 0x003264f2, 0x003268f2, 0x00326cf2, 0x003270f2, 0x003274f2, 0x003278f2, 0x00327cf2, // 19608-19615
	0x003280f2, 0x003284f2, 0x003288f2, 0x00328cf2, 0x003290f2, 0x003294f2, 0x003298f2, 0x00329cf2, // 19616-19623
	0x0032a0f2, 0x0032a4f2, 0x0032a8f2, 0x0032acf2, 0x0032b0f2, 0x0032b4f2, 0x0032b8f2, 0x0032bcf2, // 19624-19631
	0x0032c0f2, 0x0032c4f2, 0x0032c8f2, 0x0032ccf2, 0x0032d0f2, 0x0032d4f2, 0x0032d8f2, 0x0032dcf2, // 19632-19639
	0x0032e0f2, 0x0032e4f2, 0x0032e8f2, 0x0032ecf2, 0x0032f0f2, 0x0032f4f2, 0x0032f8f2, 0x0032fcf2, // 19640-19647
	0x003300f2, 0x003304f2, 0x003308f2, 0x00330cf2, 0x003310f2, 0x003314f2, 0x003318f2, 0x00331cf2, // 19648-19655
	0x003320f2, 0x003324f2, 0x003328f2, 0x00332cf2, 0x003330f2, 0x003334f2, 0x003338f2, 0x00333cf2, // 19656-19663
	0x003340f2, 0x003344f2, 0x003348f2, 0x00334cf2, 0x003350f2, 0x003354f2, 0x003358f2, 0x00335cf2, // 19664-19671
	0x003360f2, 0x003364f2, 0x003368f2, 0x00336cf2, 0x003370f2, 0x003374f2, 0x003378f2, 0x00337cf2, // 19672-19679
	0x003380f2, 0x003384f2, 0x003388f2, 0x00338cf2, 0x003390f2, 0x003394f2, 0x003398f2, 0x00339cf2, // 19680-19687
	0x0033a0f2, 0x0033a4f2, 0x0033a8f2, 0x0033acf2, 0x0033b0f2, 0x0033b4f2, 0x0033b8f2, 0x0033bcf2, // 19688-19695
	0x0033c0f2, 0x0033c4f2, 0x0033c8f2, 0x0033ccf2, 0x0033d0f2, 0x0033d4f2, 0x0033d8f2, 0x0033dcf2, // 19696-19703
	0x0033e0f2, 0x0033e4f2, 0x0033e8f2, 0x0033ecf2, 0x0033f0f2, 0x0033f4f2, 0x0033f8f2, 0x0033fcf2, // 19704-19711
	0x003400f2, 0x003404f2, 0x003408f2, 0x00340cf2, 0x003410f2, 0x003414f2, 0x003418f2, 0x00341cf2, // 19712-19719
	0x003420f2, 0x003424f2, 0x003428f2, 0x00342cf2, 0x003430f2, 0x003434f2, 0x003438f2, 0x00343cf2, // 19720-19727
	0x003440f2, 0x003444f2, 0x003448f2, 0x00344cf2, 0x003450f2, 0x003454f2, 0x003458f2, 0x00345cf2, // 19728-19735
	0x003460f2, 0x003464f2, 0x003468f2, 0x00346cf2, 0x003470f2, 0x003474f2, 0x003478f2, 0x00347cf2, // 19736-19743
	0x003480f2, 0x003484f2, 0x003488f2, 0x00348cf2, 0x003490f2, 0x003494f2, 0x003498f2, 0x00349cf2, // 19744-19751
	0x0034a0f2, 0x0034a4f2, 0x0034a8f2, 0x0034acf2, 0x0034b0f2, 0x0034b4f2, 0x0034b8f2, 0x0034bcf2, // 19752-19759
	0x0034c0f2, 0x0034c4f2, 0x0034c8f2, 0x0034ccf2, 0x0034d0f2, 0x0034d4f2, 0x0034d8f2, 0x0034dcf2, // 19760-19767
	0x0034e0f2, 0x0034e4f2, 0x0034e8f2, 0x0034ecf2, 0x0034f0f2, 0x0034f4f2, 0x0034f8f2, 0x0034fcf2, // 19768-19775
	0x003500f2, 0x003504f2, 0x003508f2, 0x00350cf2, 0x003510f2, 0x003514f2, 0x003518f2, 0x00351cf2, // 19776-19783
	0x003520f2, 0x003524f2, 0x003528f2, 0x00352cf2, 0x003530f2, 0x003534f2, 0x003538f2, 0x00353cf2, // 19784-19791
	0x003540f2, 0x003544f2, 0x003548f2, 0x00354cf2, 0x003550f2, 0x003554f2, 0x003558f2, 0x00355cf2, // 19792-19799
	0x003560f2, 0x003564f2, 0x003568f2, 0x00356cf2, 0x003570f2, 0x003574f2, 0x003578f2, 0x00357cf2, // 19800-19807
	0x003580f2, 0x003584f2, 0x003588f2, 0x00358cf2, 0x003590f2, 0x003594f2, 0x003598f2, 0x00359cf2, // 19808-19815
	0x0035a0f2, 0x0035a4f2, 0x0035a8f2, 0x0035acf2, 0x0035b0f2, 0x0035b4f2, 0x0035b8f2, 0x0035bcf2, // 19816-19823
	0x0035c0f2, 0x0035c4f2, 0x0035c8f2, 0x0035ccf2, 0x0035d0f2, 0x0035d4f2, 0x0035d8f2, 0x0035dcf2, // 19824-19831
	0x0035e0f2, 0x0035e4f2, 0x0035e8f2, 0x0035ecf2, 0x0035f0f2, 0x0035f4f2, 0x0035f8f2, 0x0035fcf2, // 19832-19839
	0x003600f2, 0x003604f2, 0x003608f2, 0x00360cf2, 0x003610f2, 0x003614f2, 0x003618f2, 0x00361cf2, // 19840-19847
	0x003620f2, 0x003624f2, 0x003628f2, 0x00362cf2, 0x003630f2, 0x003634f2, 0x003638f2, 0x00363cf2, // 19848-19855
	0x003640f2, 0x003644f2, 0x003648f2, 0x00364cf2, 0x003650f2, 0x003654f2, 0x003658f2, 0x00365cf2, // 19856-19863
	0x003660f2, 0x003664f2, 0x003668f2, 0x00366cf2, 0x003670f2, 0x003674f2, 0x003678f2, 0x00367cf2, // 19864-19871
	0x003680f2, 0x003684f2, 0x003688f2, 0x00368cf2, 0x003690f2, 0x003694f2, 0x003698f2, 0x00369cf2, // 19872-19879
	0x0036a0f2, 0x0036a4f2, 0x0036a8f2, 0x0036acf2, 0x0036b0f2, 0x0036b4f2, 0x0036b8f2, 0x0036bcf2, // 19880-19887
	0x0036c0f2, 0x0036c4f2, 0x0036c8f2, 0x0036ccf2, 0x0036d0f2, 0x0036d4f2, 0x0036d8f2, 0x0036dcf2, // 19888-19895
	0x0036e0f2, 0x0036e4f2, 0x0036e8f2, 0x0036ecf2, 0x0036f0f2, 0x0036f4f2, 0x0036f8f2, 0x0036fcf2, // 19896-19903
	0x003700f2, 0x003704f2, 0x003708f2, 0x00370cf2, 0x003710f2, 0x003714f2, 0x003718f2, 0x00371cf2, // 19904-19911
	0x003720f2, 0x003724f2, 0x003728f2, 0x00372cf2, 0x003730f2, 0x003734f2, 0x003738f2, 0x00373cf2, // 19912-19919
	0x003740f2, 0x003744f2, 0x003748f2, 0x00374cf2, 0x003750f2, 0x003754f2, 0x003758f2, 0x00375cf2, // 19920-19927
	0x003760f2, 0x003764f2, 0x003768f2, 0x00376cf2, 0x003770f2, 0x003774f2, 0x003778f2, 0x00377cf2, // 19928-19935
	0x003780f2, 0x003784f2, 0x003788f2, 0x00378cf2, 0x003790f2, 0x003794f2, 0x003798f2, 0x00379cf2, // 19936-19943
	0x0037a0f2, 0x0037a4f2, 0x0037a8f2, 0x0037acf2, 0x0037b0f2, 0x0037b4f2, 0x0037b8f2, 0x0037bcf2, // 19944-19951
	0x0037c0f2, 0x0037c4f2, 0x0037c8f2, 0x0037ccf2, 0x0037d0f2, 0x0037d4f2, 0x0037d8f2, 0x0037dcf2, // 19952-19959
	0x0037e0f2, 0x0037e4f2, 0x0037e8f2, 0x0037ecf2, 0x0037f0f2, 0x0037f4f2, 0x0037f8f2, 0x0037fcf2, // 19960-19967
	0x003800f2, 0x003804f2, 0x003808f2, 0x00380cf2, 0x003810f2, 0x003814f2, 0x003818f2, 0x00381cf2, // 19968-19975
	0x003820f2, 0x003824f2, 0x003828f2, 0x00382cf2, 0x003830f2, 0x003834f2, 0x003838f2, 0x00383cf2, // 19976-19983
	0x003840f2, 0x003844f2, 0x003848f2, 0x00384cf2, 0x003850f2, 0x003854f2, 0x003858f2, 0x00385cf2, // 19984-19991
	0x003860f2, 0x003864f2, 0x003868f2, 0x00386cf2, 0x003870f2, 0x003874f2, 0x003878f2, 0x00387cf2, // 19992-19999
	0x003880f2, 0x003884f2, 0x003888f2, 0x00388cf2, 0x003890f2, 0x003894f2, 0x003898f2, 0x00389cf2, // 20000-20007
	0x0038a0f2, 0x0038a4f2, 0x0038a8f2, 0x0038acf2, 0x0038b0f2, 0x0038b4f2, 0x0038b8f2, 0x0038bcf2, // 20008-20015
	0x0038c0f2, 0x0038c4f2, 0x0038c8f2, 0x0038ccf2, 0x0038d0f2, 0x0038d4f2, 0x0038d8f2, 0x0038dcf2, // 20016-20023
	0x0038e0f2, 0x0038e4f2, 0x0038e8f2, 0x0038ecf2, 0x0038f0f2, 0x0038f4f2, 0x0038f8f2, 0x0038fcf2, // 20024-20031
	0x003900f2, 0x003904f2, 0x003908f2, 0x00390cf2, 0x003910f2, 0x003914f2, 0x003918f2, 0x00391cf2, // 20032-20039
	0x003920f2, 0x003924f2, 0x003928f2, 0x00392cf2, 0x003930f2, 0x003934f2, 0x003938f2, 0x00393cf2, // 20040-20047
	0x003940f2, 0x003944f2, 0x003948f2, 0x00394cf2, 0x003950f2, 0x003954f2, 0x003958f2, 0x00395cf2, // 20048-20055
	0x003960f2, 0x003964f2, 0x003968f2, 0x00396cf2, 0x003970f2, 0x003974f2, 0x003978f2, 0x00397cf2, // 20056-20063
	0x003980f2, 0x003984f2, 0x003988f2, 0x00398cf2, 0x003990f2, 0x003994f2, 0x003998f2, 0x00399cf2, // 20064-20071
	0x0039a0f2, 0x0039a4f2, 0x0039a8f2, 0x0039acf2, 0x0039b0f2, 0x0039b4f2, 0x0039b8f2, 0x0039bcf2, // 20072-20079
	0x0039c0f2, 0x0039c4f2, 0x0039c8f2, 0x0039ccf2, 0x0039d0f2, 0x0039d4f2, 0x0039d8f2, 0x0039dcf2, // 20080-20087
	0x0039e0f2, 0x0039e4f2, 0x0039e8f2, 0x0039ecf2, 0x0039f0f2, 0x0039f4f2, 0x0039f8f2, 0x0039fcf2, // 20088-20095
	0x003a00f2, 0x003a04f2, 0x003a08f2, 0x003a0cf2, 0x003a10f2, 0x003a14f2, 0x003a18f2, 0x003a1cf2, // 20096-20103
	0x003a20f2, 0x003a24f2, 0x003a28f2, 0x003a2cf2, 0x003a30f2, 0x003a34f2, 0x003a38f2, 0x003a3cf2, // 20104-20111
	0x003a40f2, 0x003a44f2, 0x003a48f2, 0x003a4cf2, 0x003a50f2, 0x003a54f2, 0x003a58f2, 0x003a5cf2, // 20112-20119
	0x003a60f2, 0x003a64f2, 0x003a68f2, 0x003a6cf2, 0x003a70f2, 0x003a74f2, 0x003a78f2, 0x003a7cf2, // 20120-20127
	0x003a80f2, 0x003a84f2, 0x003a88f2, 0x003a8cf2, 0x003a90f2, 0x003a94f2, 0x003a98f2, 0x003a9cf2, // 20128-20135
	0x003aa0f2, 0x003aa4f2, 0x003aa8f2, 0x003aacf2, 0x003ab0f2, 0x003ab4f2, 0x003ab8f2, 0x003abcf2, // 20136-20143
	0x003ac0f2, 0x003ac4f2, 0x003ac8f2, 0x003accf2, 0x003ad0f2, 0x003ad4f2, 0x003ad8f2, 0x003adcf2, // 20144-20151
	0x003ae0f2, 0x003ae4f2, 0x003ae8f2, 0x003aecf2, 0x003af0f2, 0x003af4f2, 0x003af8f2, 0x003afcf2, // 20152-20159
	0x003b00f2, 0x003b04f2, 0x003b08f2, 0x003b0cf2, 0x003b10f2, 0x003b14f2, 0x003b18f2, 0x003b1cf2, // 20160-20167
	0x003b20f2, 0x003b24f2, 0x003b28f2, 0x003b2cf2, 0x003b30f2, 0x003b34f2, 0x003b38f2, 0x003b3cf2, // 20168-20175
	0x003b40f2, 0x003b44f2, 0x003b48f2, 0x003b4cf2, 0x003b50f2, 0x003b54f2, 0x003b58f2, 0x003b5cf2, // 20176-20183
	0x003b60f2, 0x003b64f2, 0x003b68f2, 0x003b6cf2, 0x003b70f2, 0x003b74f2, 0x003b78f2, 0x003b7cf2, // 20184-20191
	0x003b80f2, 0x003b84f2, 0x003b88f2, 0x003b8cf2, 0x003b90f2, 0x003b94f2, 0x003b98f2, 0x003b9cf2, // 20192-20199
	0x003ba0f2, 0x003ba4f2, 0x003ba8f2, 0x003bacf2, 0x003bb0f2, 0x003bb4f2, 0x003bb8f2, 0x003bbcf2, // 20200-20207
	0x003bc0f2, 0x003bc4f2, 0x003bc8f2, 0x003bccf2, 0x003bd0f2, 0x003bd4f2, 0x003bd8f2, 0x003bdcf2, // 20208-20215
	0x003be0f2, 0x003be4f2, 0x003be8f2, 0x003becf2, 0x003bf0f2, 0x003bf4f2, 0x003bf8f2, 0x003bfcf2, // 20216-20223
	0x003c00f2, 0x003c04f2, 0x003c08f2, 0x003c0cf2, 0x003c10f2, 0x003c14f2, 0x003c18f2, 0x003c1cf2, // 20224-20231
	0x003c20f2, 0x003c24f2, 0x003c28f2, 0x003c2cf2, 0x003c30f2, 0x003c34f2, 0x003c38f2, 0x003c3cf2, // 20232-20239
	0x003c40f2, 0x003c44f2, 0x003c48f2, 0x003c4cf2, 0x003c50f2, 0x003c54f2, 0x003c58f2, 0x003c5cf2, // 20240-20247
	0x003c60f2, 0x003c64f2, 0x003c68f2, 0x003c6cf2, 0x003c70f2, 0x003c74f2, 0x003c78f2, 0x003c7cf2, // 20248-20255
	0x003c80f2, 0x003c84f2, 0x003c88f2, 0x003c8cf2, 0x003c90f2, 0x003c94f2, 0x003c98f2, 0x003c9cf2, // 20256-20263
	0x003ca0f2, 0x003ca4f2, 0x003ca8f2, 0x003cacf2, 0x003cb0f2, 0x003cb4f2, 0x003cb8f2, 0x003cbcf2, // 20264-20271
	0x003cc0f2, 0x003cc4f2, 0x003cc8f2, 0x003cccf2, 0x003cd0f2, 0x003cd4f2, 0x003cd8f2, 0x003cdcf2, // 20272-20279
	0x003ce0f2, 0x003ce4f2, 0x003ce8f2, 0x003cecf2, 0x003cf0f2, 0x003cf4f2, 0x003cf8f2, 0x003cfcf2, // 20280-20287
	0x003d00f2, 0x003d04f2, 0x003d08f2, 0x003d0cf2, 0x003d10f2, 0x003d14f2, 0x003d18f2, 0x003d1cf2, // 20288-20295
	0x003d20f2, 0x003d24f2, 0x003d28f2, 0x003d2cf2, 0x003d30f2, 0x003d34f2, 0x003d38f2, 0x003d3cf2, // 20296-20303
	0x003d40f2, 0x003d44f2, 0x003d48f2, 0x003d4cf2, 0x003d50f2, 0x003d54f2, 0x003d58f2, 0x003d5cf2, // 20304-20311
	0x003d60f2, 0x003d64f2, 0x003d68f2, 0x003d6cf2, 0x003d70f2, 0x003d74f2, 0x003d78f2, 0x003d7cf2, // 20312-20319
	0x003d80f2, 0x003d84f2, 0x003d88f2, 0x003d8cf2, 0x003d90f2, 0x003d94f2, 0x003d98f2, 0x003d9cf2, // 20320-20327
	0x003da0f2, 0x003da4f2, 0x003da8f2, 0x003dacf2, 0x003db0f2, 0x003db4f2, 0x003db8f2, 0x003dbcf2, // 20328-20335
	0x003dc0f2, 0x003dc4f2, 0x003dc8f2, 0x003dccf2, 0x003dd0f2, 0x003dd4f2, 0x003dd8f2, 0x003ddcf2, // 20336-20343
	0x003de0f2, 0x003de4f2, 0x003de8f2, 0x003decf2, 0x003df0f2, 0x003df4f2, 0x003df8f2, 0x003dfcf2, // 20344-20351
	0x003e00f2, 0x003e04f2, 0x003e08f2, 0x003e0cf2, 0x003e10f2, 0x003e14f2, 0x003e18f2, 0x003e1cf2, // 20352-20359
	0x003e20f2, 0x003e24f2, 0x003e28f2, 0x003e2cf2, 0x003e30f2, 0x003e34f2, 0x003e38f2, 0x003e3cf2, // 20360-20367
	0x003e40f2, 0x003e44f2, 0x003e48f2, 0x003e4cf2, 0x003e50f2, 0x003e54f2, 0x003e58f2, 0x003e5cf2, // 20368-20375
	0x003e60f2, 0x003e64f2, 0x003e68f2, 0x003e6cf2, 0x003e70f2, 0x003e74f2, 0x003e78f2, 0x003e7cf2, // 20376-20383
	0x003e80f2, 0x003e84f2, 0x003e88f2, 0x003e8cf2, 0x003e90f2, 0x003e94f2, 0x003e98f2, 0x003e9cf2, // 20384-20391
	0x003ea0f2, 0x003ea4f2, 0x003ea8f2, 0x003eacf2, 0x003eb0f2, 0x003eb4f2, 0x003eb8f2, 0x003ebcf2, // 20392-20399
	0x003ec0f2, 0x003ec4f2, 0x003ec8f2, 0x003eccf2, 0x003ed0f2, 0x003ed4f2, 0x003ed8f2, 0x003edcf2, // 20400-20407
	0x003ee0f2, 0x003ee4f2, 0x003ee8f2, 0x003eecf2, 0x003ef0f2, 0x003ef4f2, 0x003ef8f2, 0x003efcf2, // 20408-20415
	0x003f00f2, 0x003f04f2, 0x003f08f2, 0x003f0cf2, 0x003f10f2, 0x003f14f2, 0x003f18f2, 0x003f1cf2, // 20416-20423
	0x003f20f2, 0x003f24f2, 0x003f28f2, 0x003f2cf2, 0x003f30f2, 0x003f34f2, 0x003f38f2, 0x003f3cf2, // 20424-20431
	0x003f40f2, 0x003f44f2, 0x003f48f2, 0x003f4cf2, 0x003f50f2, 0x003f54f2, 0x003f58f2, 0x003f5cf2, // 20432-20439
	0x003f60f2, 0x003f64f2, 0x003f68f2, 0x003f6cf2, 0x003f70f2, 0x003f74f2, 0x003f78f2, 0x003f7cf2, // 20440-20447
	0x003f80f2, 0x003f84f2, 0x003f88f2, 0x003f8cf2, 0x003f90f2, 0x003f94f2, 0x003f98f2, 0x003f9cf2, // 20448-20455
	0x003fa0f2, 0x003fa4f2, 0x003fa8f2, 0x003facf2, 0x003fb0f2, 0x003fb4f2, 0x003fb8f2, 0x003fbcf2, // 20456-20463
	0x003fc0f2, 0x003fc4f2, 0x003fc8f2, 0x003fccf2, 0x003fd0f2, 0x003fd4f2, 0x003fd8f2, 0x003fdcf2, // 20464-20471
	0x003fe0f2, 0x003fe4f2, 0x003fe8f2, 0x003fecf2, 0x003ff0f2, 0x003ff4f2, 0x003ff8f2, 0x003ffcf2, // 20472-20479
	0x004000f2, 0x004004f2, 0x004008f2, 0x00400cf2, 0x004010f2, 0x004014f2, 0x004018f2, 0x00401cf2, // 20480-20487
	0x004020f2, 0x004024f2, 0x004028f2, 0x00402cf2, 0x004030f2, 0x004034f2, 0x004038f2, 0x00403cf2, // 20488-20495
	0x004040f2, 0x004044f2, 0x004048f2, 0x00404cf2, 0x004050f2, 0x004054f2, 0x004058f2, 0x00405cf2, // 20496-20503
	0x004060f2, 0x004064f2, 0x004068f2, 0x00406cf2, 0x004070f2, 0x004074f2, 0x004078f2, 0x00407cf2, // 20504-20511
	0x004080f2, 0x004084f2, 0x004088f2, 0x00408cf2, 0x004090f2, 0x004094f2, 0x004098f2, 0x00409cf2, // 20512-20519
	0x0040a0f2, 0x0040a4f2, 0x0040a8f2, 0x0040acf2, 0x0040b0f2, 0x0040b4f2, 0x0040b8f2, 0x0040bcf2, // 20520-20527
	0x0040c0f2, 0x0040c4f2, 0x0040c8f2, 0x0040ccf2, 0x0040d0f2, 0x0040d4f2, 0x0040d8f2, 0x0040dcf2, // 20528-20535
	0x0040e0f2, 0x0040e4f2, 0x0040e8f2, 0x0040ecf2, 0x0040f0f2, 0x0040f4f2, 0x0040f8f2, 0x0040fcf2, // 20536-20543
	0x004100f2, 0x004104f2, 0x004108f2, 0x00410cf2, 0x004110f2, 0x004114f2, 0x004118f2, 0x00411cf2, // 20544-20551
	0x004120f2, 0x004124f2, 0x004128f2, 0x00412cf2, 0x004130f2, 0x004134f2, 0x004138f2, 0x00413cf2, // 20552-20559
	0x004140f2, 0x004144f2, 0x004148f2, 0x00414cf2, 0x004150f2, 0x004154f2, 0x004158f2, 0x00415cf2, // 20560-20567
	0x004160f2, 0x004164f2, 0x004168f2, 0x00416cf2, 0x004170f2, 0x004174f2, 0x004178f2, 0x00417cf2, // 20568-20575
	0x004180f2, 0x004184f2, 0x004188f2, 0x00418cf2, 0x004190f2, 0x004194f2, 0x004198f2, 0x00419cf2, // 20576-20583
	0x0041a0f2, 0x0041a4f2, 0x0041a8f2, 0x0041acf2, 0x0041b0f2, 0x0041b4f2, 0x0041b8f2, 0x0041bcf2, // 20584-20591
	0x0041c0f2, 0x0041c4f2, 0x0041c8f2, 0x0041ccf2, 0x0041d0f2, 0x0041d4f2, 0x0041d8f2, 0x0041dcf2, // 20592-20599
	0x0041e0f2, 0x0041e4f2, 0x0041e8f2, 0x0041ecf2, 0x0041f0f2, 0x0041f4f2, 0x0041f8f2, 0x0041fcf2, // 20600-20607
	0x004200f2, 0x004204f2, 0x004208f2, 0x00420cf2, 0x004210f2, 0x004214f2, 0x004218f2, 0x00421cf2, // 20608-20615
	0x004220f2, 0x004224f2, 0x004228f2, 0x00422cf2, 0x004230f2, 0x004234f2, 0x004238f2, 0x00423cf2, // 20616-20623
	0x004240f2, 0x004244f2, 0x004248f2, 0x00424cf2, 0x004250f2, 0x004254f2, 0x004258f2, 0x00425cf2, // 20624-20631
	0x004260f2, 0x004264f2, 0x004268f2, 0x00426cf2, 0x004270f2, 0x004274f2, 0x004278f2, 0x00427cf2, // 20632-20639
	0x004280f2, 0x004284f2, 0x004288f2, 0x00428cf2, 0x004290f2, 0x004294f2, 0x004298f2, 0x00429cf2, // 20640-20647
	0x0042a0f2, 0x0042a4f2, 0x0042a8f2, 0x0042acf2, 0x0042b0f2, 0x0042b4f2, 0x0042b8f2, 0x0042bcf2, // 20648-20655
	0x0042c0f2, 0x0042c4f2, 0x0042c8f2, 0x0042ccf2, 0x0042d0f2, 0x0042d4f2, 0x0042d8f2, 0x0042dcf2, // 20656-20663
	0x0042e0f2, 0x0042e4f2, 0x0042e8f2, 0x0042ecf2, 0x0042f0f2, 0x0042f4f2, 0x0042f8f2, 0x0042fcf2, // 20664-20671
	0x004300f2, 0x004304f2, 0x004308f2, 0x00430cf2, 0x004310f2, 0x004314f2, 0x004318f2, 0x00431cf2, // 20672-20679
	0x004320f2, 0x004324f2, 0x004328f2, 0x00432cf2, 0x004330f2, 0x004334f2, 0x004338f2, 0x00433cf2, // 20680-20687
	0x004340f2, 0x004344f2, 0x004348f2, 0x00434cf2, 0x004350f2, 0x004354f2, 0x004358f2, 0x00435cf2, // 20688-20695
	0x004360f2, 0x004364f2, 0x004368f2, 0x00436cf2, 0x004370f2, 0x004374f2, 0x004378f2, 0x00437cf2, // 20696-20703
	0x004380f2, 0x004384f2, 0x004388f2, 0x00438cf2, 0x004390f2, 0x004394f2, 0x004398f2, 0x00439cf2, // 20704-20711
	0x0043a0f2, 0x0043a4f2, 0x0043a8f2, 0x0043acf2, 0x0043b0f2, 0x0043b4f2, 0x0043b8f2, 0x0043bcf2, // 20712-20719
	0x0043c0f2, 0x0043c4f2, 0x0043c8f2, 0x0043ccf2, 0x0043d0f2, 0x0043d4f2, 0x0043d8f2, 0x0043dcf2, // 20720-20727
	0x0043e0f2, 0x0043e4f2, 0x0043e8f2, 0x0043ecf2, 0x0043f0f2, 0x0043f4f2, 0x0043f8f2, 0x0043fcf2, // 20728-20735
	0x004400f2, 0x004404f2, 0x004408f2, 0x00440cf2, 0x004410f2, 0x004414f2, 0x004418f2, 0x00441cf2, // 20736-20743
	0x004420f2, 0x004424f2, 0x004428f2, 0x00442cf2, 0x004430f2, 0x004434f2, 0x004438f2, 0x00443cf2, // 20744-20751
	0x004440f2, 0x004444f2, 0x004448f2, 0x00444cf2, 0x004450f2, 0x004454f2, 0x004458f2, 0x00445cf2, // 20752-20759
	0x004460f2, 0x004464f2, 0x004468f2, 0x00446cf2, 0x004470f2, 0x004474f2, 0x004478f2, 0x00447cf2, // 20760-20767
	0x004480f2, 0x004484f2, 0x004488f2, 0x00448cf2, 0x004490f2, 0x004494f2, 0x004498f2, 0x00449cf2, // 20768-20775
	0x0044a0f2, 0x0044a4f2, 0x0044a8f2, 0x0044acf2, 0x0044b0f2, 0x0044b4f2, 0x0044b8f2, 0x0044bcf2, // 20776-20783
	0x0044c0f2, 0x0044c4f2, 0x0044c8f2, 0x0044ccf2, 0x0044d0f2, 0x0044d4f2, 0x0044d8f2, 0x0044dcf2, // 20784-20791
	0x0044e0f2, 0x0044e4f2, 0x0044e8f2, 0x0044ecf2, 0x0044f0f2, 0x0044f4f2, 0x0044f8f2, 0x0044fcf2, // 20792-20799
	0x004500f2, 0x004504f2, 0x004508f2, 0x00450cf2, 0x004510f2, 0x004514f2, 0x004518f2, 0x00451cf2, // 20800-20807
	0x004520f2, 0x004524f2, 0x004528f2, 0x00452cf2, 0x004530f2, 0x004534f2, 0x004538f2, 0x00453cf2, // 20808-20815
	0x004540f2, 0x004544f2, 0x004548f2, 0x00454cf2, 0x004550f2, 0x004554f2, 0x004558f2, 0x00455cf2, // 20816-20823
	0x004560f2, 0x004564f2, 0x004568f2, 0x00456cf2, 0x004570f2, 0x004574f2, 0x004578f2, 0x00457cf2, // 20824-20831
	0x004580f2, 0x004584f2, 0x004588f2, 0x00458cf2, 0x004590f2, 0x004594f2, 0x004598f2, 0x00459cf2, // 20832-20839
	0x0045a0f2, 0x0045a4f2, 0x0045a8f2, 0x0045acf2, 0x0045b0f2, 0x0045b4f2, 0x0045b8f2, 0x0045bcf2, // 20840-20847
	0x0045c0f2, 0x0045c4f2, 0x0045c8f2, 0x0045ccf2, 0x0045d0f2, 0x0045d4f2, 0x0045d8f2, 0x0045dcf2, // 20848-20855
	0x0045e0f2, 0x0045e4f2, 0x0045e8f2, 0x0045ecf2, 0x0045f0f2, 0x0045f4f2, 0x0045f8f2, 0x0045fcf2, // 20856-20863
	0x004600f2, 0x004604f2, 0x004608f2, 0x00460cf2, 0x004610f2, 0x004614f2, 0x004618f2, 0x00461cf2, // 20864-20871
	0x004620f2, 0x004624f2, 0x004628f2, 0x00462cf2, 0x004630f2, 0x004634f2, 0x004638f2, 0x00463cf2, // 20872-20879
	0x004640f2, 0x004644f2, 0x004648f2, 0x00464cf2, 0x004650f2, 0x004654f2, 0x004658f2, 0x00465cf2, // 20880-20887
	0x004660f2, 0x004664f2, 0x004668f2, 0x00466cf2, 0x004670f2, 0x004674f2, 0x004678f2, 0x00467cf2, // 20888-20895
	0x004680f2, 0x004684f2, 0x004688f2, 0x00468cf2, 0x004690f2, 0x004694f2, 0x004698f2, 0x00469cf2, // 20896-20903
	0x0046a0f2, 0x0046a4f2, 0x0046a8f2, 0x0046acf2, 0x0046b0f2, 0x0046b4f2, 0x0046b8f2, 0x0046bcf2, // 20904-20911
	0x0046c0f2, 0x0046c4f2, 0x0046c8f2, 0x0046ccf2, 0x0046d0f2, 0x0046d4f2, 0x0046d8f2, 0x0046dcf2, // 20912-20919
	0x0046e0f2, 0x0046e4f2, 0x0046e8f2, 0x0046ecf2, 0x0046f0f2, 0x0046f4f2, 0x0046f8f2, 0x0046fcf2, // 20920-20927
	0x004700f2, 0x004704f2, 0x004708f2, 0x00470cf2, 0x004710f2, 0x004714f2, 0x004718f2, 0x00471cf2, // 20928-20935
	0x004720f2, 0x004724f2, 0x004728f2, 0x00472cf2, 0x004730f2, 0x004734f2, 0x004738f2, 0x00473cf2, // 20936-20943
	0x004740f2, 0x004744f2, 0x004748f2, 0x00474cf2, 0x004750f2, 0x004754f2, 0x004758f2, 0x00475cf2, // 20944-20951
	0x004760f2, 0x004764f2, 0x004768f2, 0x00476cf2, 0x004770f2, 0x004774f2, 0x004778f2, 0x00477cf2, // 20952-20959
	0x004780f2, 0x004784f2, 0x004788f2, 0x00478cf2, 0x004790f2, 0x004794f2, 0x004798f2, 0x00479cf2, // 20960-20967
	0x0047a0f2, 0x0047a4f2, 0x0047a8f2, 0x0047acf2, 0x0047b0f2, 0x0047b4f2, 0x0047b8f2, 0x0047bcf2, // 20968-20975
	0x0047c0f2, 0x0047c4f2, 0x0047c8f2, 0x0047ccf2, 0x0047d0f2, 0x0047d4f2, 0x0047d8f2, 0x0047dcf2, // 20976-20983
	0x0047e0f2, 0x0047e4f2, 0x0047e8f2, 0x0047ecf2, 0x0047f0f2, 0x0047f4f2, 0x0047f8f2, 0x0047fcf2, // 20984-20991
	0x004800f2, 0x004804f2, 0x004808f2, 0x00480cf2, 0x004810f2, 0x004814f2, 0x004818f2, 0x00481cf2, // 20992-20999
	0x004820f2, 0x004824f2, 0x004828f2, 0x00482cf2, 0x004830f2, 0x004834f2, 0x004838f2, 0x00483cf2, // 21000-21007
	0x004840f2, 0x004844f2, 0x004848f2, 0x00484cf2, 0x004850f2, 0x004854f2, 0x004858f2, 0x00485cf2, // 21008-21015
	0x004860f2, 0x004864f2, 0x004868f2, 0x00486cf2, 0x004870f2, 0x004874f2, 0x004878f2, 0x00487cf2, // 21016-21023
	0x004880f2, 0x004884f2, 0x004888f2, 0x00488cf2, 0x004890f2, 0x004894f2, 0x004898f2, 0x00489cf2, // 21024-21031
	0x0048a0f2, 0x0048a4f2, 0x0048a8f2, 0x0048acf2, 0x0048b0f2, 0x0048b4f2, 0x0048b8f2, 0x0048bcf2, // 21032-21039
	0x0048c0f2, 0x0048c4f2, 0x0048c8f2, 0x0048ccf2, 0x0048d0f2, 0x0048d4f2, 0x0048d8f2, 0x0048dcf2, // 21040-21047
	0x0048e0f2, 0x0048e4f2, 0x0048e8f2, 0x0048ecf2, 0x0048f0f2, 0x0048f4f2, 0x0048f8f2, 0x0048fcf2, // 21048-21055
	0x004900f2, 0x004904f2, 0x004908f2, 0x00490cf2, 0x004910f2, 0x004914f2, 0x004918f2, 0x00491cf2, // 21056-21063
	0x004920f2, 0x004924f2, 0x004928f2, 0x00492cf2, 0x004930f2, 0x004934f2, 0x004938f2, 0x00493cf2, // 21064-21071
	0x004940f2, 0x004944f2, 0x004948f2, 0x00494cf2, 0x004950f2, 0x004954f2, 0x004958f2, 0x00495cf2, // 21072-21079
	0x004960f2, 0x004964f2, 0x004968f2, 0x00496cf2, 0x004970f2, 0x004974f2, 0x004978f2, 0x00497cf2, // 21080-21087
	0x004980f2, 0x004984f2, 0x004988f2, 0x00498cf2, 0x004990f2, 0x004994f2, 0x004998f2, 0x00499cf2, // 21088-21095
	0x0049a0f2, 0x0049a4f2, 0x0049a8f2, 0x0049acf2, 0x0049b0f2, 0x0049b4f2, 0x0049b8f2, 0x0049bcf2, // 21096-21103
	0x0049c0f2, 0x0049c4f2, 0x0049c8f2, 0x0049ccf2, 0x0049d0f2, 0x0049d4f2, 0x0049d8f2, 0x0049dcf2, // 21104-21111
	0x0049e0f2, 0x0049e4f2, 0x0049e8f2, 0x0049ecf2, 0x0049f0f2, 0x0049f4f2, 0x0049f8f2, 0x0049fcf2, // 21112-21119
	0x004a00f2, 0x004a04f2, 0x004a08f2, 0x004a0cf2, 0x004a10f2, 0x004a14f2, 0x004a18f2, 0x004a1cf2, // 21120-21127
	0x004a20f2, 0x004a24f2, 0x004a28f2, 0x004a2cf2, 0x004a30f2, 0x004a34f2, 0x004a38f2, 0x004a3cf2, // 21128-21135
	0x004a40f2, 0x004a44f2, 0x004a48f2, 0x004a4cf2, 0x004a50f2, 0x004a54f2, 0x004a58f2, 0x004a5cf2, // 21136-21143
	0x004a60f2, 0x004a64f2, 0x004a68f2, 0x004a6cf2, 0x004a70f2, 0x004a74f2, 0x004a78f2, 0x004a7cf2, // 21144-21151
	0x004a80f2, 0x004a84f2, 0x004a88f2, 0x004a8cf2, 0x004a90f2, 0x004a94f2, 0x004a98f2, 0x004a9cf2, // 21152-21159
	0x004aa0f2, 0x004aa4f2, 0x004aa8f2, 0x004aacf2, 0x004ab0f2, 0x004ab4f2, 0x004ab8f2, 0x004abcf2, // 21160-21167
	0x004ac0f2, 0x004ac4f2, 0x004ac8f2, 0x004accf2, 0x004ad0f2, 0x004ad4f2, 0x004ad8f2, 0x004adcf2, // 21168-21175
	0x004ae0f2, 0x004ae4f2, 0x004ae8f2, 0x004aecf2, 0x004af0f2, 0x004af4f2, 0x004af8f2, 0x004afcf2, // 21176-21183
	0x004b00f2, 0x004b04f2, 0x004b08f2, 0x004b0cf2, 0x004b10f2, 0x004b14f2, 0x004b18f2, 0x004b1cf2, // 21184-21191
	0x004b20f2, 0x004b24f2, 0x004b28f2, 0x004b2cf2, 0x004b30f2, 0x004b34f2, 0x004b38f2, 0x004b3cf2, // 21192-21199
	0x004b40f2, 0x004b44f2, 0x004b48f2, 0x004b4cf2, 0x004b50f2, 0x004b54f2, 0x004b58f2, 0x004b5cf2, // 21200-21207
	0x004b60f2, 0x004b64f2, 0x004b68f2, 0x004b6cf2, 0x004b70f2, 0x004b74f2, 0x004b78f2, 0x004b7cf2, // 21208-21215
	0x004b80f2, 0x004b84f2, 0x004b88f2, 0x004b8cf2, 0x004b90f2, 0x004b94f2, 0x004b98f2, 0x004b9cf2, // 21216-21223
	0x004ba0f2, 0x004ba4f2, 0x004ba8f2, 0x004bacf2, 0x004bb0f2, 0x004bb4f2, 0x004bb8f2, 0x004bbcf2, // 21224-21231
	0x004bc0f2, 0x004bc4f2, 0x004bc8f2, 0x004bccf2, 0x004bd0f2, 0x004bd4f2, 0x004bd8f2, 0x004bdcf2, // 21232-21239
	0x004be0f2, 0x004be4f2, 0x004be8f2, 0x004becf2, 0x004bf0f2, 0x004bf4f2, 0x004bf8f2, 0x004bfcf2, // 21240-21247
	0x004c00f2, 0x004c04f2, 0x004c08f2, 0x004c0cf2, 0x004c10f2, 0x004c14f2, 0x004c18f2, 0x004c1cf2, // 21248-21255
	0x004c20f2, 0x004c24f2, 0x004c28f2, 0x004c2cf2, 0x004c30f2, 0x004c34f2, 0x004c38f2, 0x004c3cf2, // 21256-21263
	0x004c40f2, 0x004c44f2, 0x004c48f2, 0x004c4cf2, 0x004c50f2, 0x004c54f2, 0x004c58f2, 0x004c5cf2, // 21264-21271
	0x004c60f2, 0x004c64f2, 0x004c68f2, 0x004c6cf2, 0x004c70f2, 0x004c74f2, 0x004c78f2, 0x004c7cf2, // 21272-21279
	0x004c80f2, 0x004c84f2, 0x004c88f2, 0x004c8cf2, 0x004c90f2, 0x004c94f2, 0x004c98f2, 0x004c9cf2, // 21280-21287
	0x004ca0f2, 0x004ca4f2, 0x004ca8f2, 0x004cacf2, 0x004cb0f2, 0x004cb4f2, 0x004cb8f2, 0x004cbcf2, // 21288-21295
	0x004cc0f2, 0x004cc4f2, 0x004cc8f2, 0x004cccf2, 0x004cd0f2, 0x004cd4f2, 0x004cd8f2, 0x004cdcf2, // 21296-21303
	0x004ce0f2, 0x004ce4f2, 0x004ce8f2, 0x004cecf2, 0x004cf0f2, 0x004cf4f2, 0x004cf8f2, 0x004cfcf2, // 21304-21311
	0x004d00f2, 0x004d04f2, 0x004d08f2, 0x004d0cf2, 0x004d10f2, 0x004d14f2, 0x004d18f2, 0x004d1cf2, // 21312-21319
	0x004d20f2, 0x004d24f2, 0x004d28f2, 0x004d2cf2, 0x004d30f2, 0x004d34f2, 0x004d38f2, 0x004d3cf2, // 21320-21327
	0x004d40f2, 0x004d44f2, 0x004d48f2, 0x004d4cf2, 0x004d50f2, 0x004d54f2, 0x004d58f2, 0x004d5cf2, // 21328-21335
	0x004d60f2, 0x004d64f2, 0x004d68f2, 0x004d6cf2, 0x004d70f2, 0x004d74f2, 0x004d78f2, 0x004d7cf2, // 21336-21343
	0x004d80f2, 0x004d84f2, 0x004d88f2, 0x004d8cf2, 0x004d90f2, 0x004d94f2, 0x004d98f2, 0x004d9cf2, // 21344-21351
	0x004da0f2, 0x004da4f2, 0x004da8f2, 0x004dacf2, 0x004db0f2, 0x004db4f2, 0x004db8f2, 0x004dbcf2, // 21352-21359
	0x004dc0f2, 0x004dc4f2, 0x004dc8f2, 0x004dccf2, 0x004dd0f2, 0x004dd4f2, 0x004dd8f2, 0x004ddcf2, // 21360-21367
	0x004de0f2, 0x004de4f2, 0x004de8f2, 0x004decf2, 0x004df0f2, 0x004df4f2, 0x004df8f2, 0x004dfcf2, // 21368-21375
	0x004e00f2, 0x004e04f2, 0x004e08f2, 0x004e0cf2, 0x004e10f2, 0x004e14f2, 0x004e18f2, 0x004e1cf2, // 21376-21383
	0x004e20f2, 0x004e24f2, 0x004e28f2, 0x004e2cf2, 0x004e30f2, 0x004e34f2, 0x004e38f2, 0x004e3cf2, // 21384-21391
	0x004e40f2, 0x004e44f2, 0x004e48f2, 0x004e4cf2, 0x004e50f2, 0x004e54f2, 0x004e58f2, 0x004e5cf2, // 21392-21399
	0x004e60f2, 0x004e64f2, 0x004e68f2, 0x004e6cf2, 0x004e70f2, 0x004e74f2, 0x004e78f2, 0x004e7cf2, // 21400-21407
	0x004e80f2, 0x004e84f2, 0x004e88f2, 0x004e8cf2, 0x004e90f2, 0x004e94f2, 0x004e98f2, 0x004e9cf2, // 21408-21415
	0x004ea0f2, 0x004ea4f2, 0x004ea8f2, 0x004eacf2, 0x004eb0f2, 0x004eb4f2, 0x004eb8f2, 0x004ebcf2, // 21416-21423
	0x004ec0f2, 0x004ec4f2, 0x004ec8f2, 0x004eccf2, 0x004ed0f2, 0x004ed4f2, 0x004ed8f2, 0x004edcf2, // 21424-21431
	0x004ee0f2, 0x004ee4f2, 0x004ee8f2, 0x004eecf2, 0x004ef0f2, 0x004ef4f2, 0x004ef8f2, 0x004efcf2, // 21432-21439
	0x004f00f2, 0x004f04f2, 0x004f08f2, 0x004f0cf2, 0x004f10f2, 0x004f14f2, 0x004f18f2, 0x004f1cf2, // 21440-21447
	0x004f20f2, 0x004f24f2, 0x004f28f2, 0x004f2cf2, 0x004f30f2, 0x004f34f2, 0x004f38f2, 0x004f3cf2, // 21448-21455
	0x004f40f2, 0x004f44f2, 0x004f48f2, 0x004f4cf2, 0x004f50f2, 0x004f54f2, 0x004f58f2, 0x004f5cf2, // 21456-21463
	0x004f60f2, 0x004f64f2, 0x004f68f2, 0x004f6cf2, 0x004f70f2, 0x004f74f2, 0x004f78f2, 0x004f7cf2, // 21464-21471
	0x004f80f2, 0x004f84f2, 0x004f88f2, 0x004f8cf2, 0x004f90f2, 0x004f94f2, 0x004f98f2, 0x004f9cf2, // 21472-21479
	0x004fa0f2, 0x004fa4f2, 0x004fa8f2, 0x004facf2, 0x004fb0f2, 0x004fb4f2, 0x004fb8f2, 0x004fbcf2, // 21480-21487
	0x004fc0f2, 0x004fc4f2, 0x004fc8f2, 0x004fccf2, 0x004fd0f2, 0x004fd4f2, 0x004fd8f2, 0x004fdcf2, // 21488-21495
	0x004fe0f2, 0x004fe4f2, 0x004fe8f2, 0x004fecf2, 0x004ff0f2, 0x004ff4f2, 0x004ff8f2, 0x004ffcf2, // 21496-21503
	0x005000f2, 0x005004f2, 0x005008f2, 0x00500cf2, 0x005010f2, 0x005014f2, 0x005018f2, 0x00501cf2, // 21504-21511
	0x005020f2, 0x005024f2, 0x005028f2, 0x00502cf2, 0x005030f2, 0x005034f2, 0x005038f2, 0x00503cf2, // 21512-21519
	0x005040f2, 0x005044f2, 0x005048f2, 0x00504cf2, 0x005050f2, 0x005054f2, 0x005058f2, 0x00505cf2, // 21520-21527
	0x005060f2, 0x005064f2, 0x005068f2, 0x00506cf2, 0x005070f2, 0x005074f2, 0x005078f2, 0x00507cf2, // 21528-21535
	0x005080f2, 0x005084f2, 0x005088f2, 0x00508cf2, 0x005090f2, 0x005094f2, 0x005098f2, 0x00509cf2, // 21536-21543
	0x0050a0f2, 0x0050a4f2, 0x0050a8f2, 0x0050acf2, 0x0050b0f2, 0x0050b4f2, 0x0050b8f2, 0x0050bcf2, // 21544-21551
	0x0050c0f2, 0x0050c4f2, 0x0050c8f2, 0x0050ccf2, 0x0050d0f2, 0x0050d4f2, 0x0050d8f2, 0x0050dcf2, // 21552-21559
	0x0050e0f2, 0x0050e4f2, 0x0050e8f2, 0x0050ecf2, 0x0050f0f2, 0x0050f4f2, 0x0050f8f2, 0x0050fcf2, // 21560-21567
	0x005100f2, 0x005104f2, 0x005108f2, 0x00510cf2, 0x005110f2, 0x005114f2, 0x005118f2, 0x00511cf2, // 21568-21575
	0x005120f2, 0x005124f2, 0x005128f2, 0x00512cf2, 0x005130f2, 0x005134f2, 0x005138f2, 0x00513cf2, // 21576-21583
	0x005140f2, 0x005144f2, 0x005148f2, 0x00514cf2, 0x005150f2, 0x005154f2, 0x005158f2, 0x00515cf2, // 21584-21591
	0x005160f2, 0x005164f2, 0x005168f2, 0x00516cf2, 0x005170f2, 0x005174f2, 0x005178f2, 0x00517cf2, // 21592-21599
	0x005180f2, 0x005184f2, 0x005188f2, 0x00518cf2, 0x005190f2, 0x005194f2, 0x005198f2, 0x00519cf2, // 21600-21607
	0x0051a0f2, 0x0051a4f2, 0x0051a8f2, 0x0051acf2, 0x0051b0f2, 0x0051b4f2, 0x0051b8f2, 0x0051bcf2, // 21608-21615
	0x0051c0f2, 0x0051c4f2, 0x0051c8f2, 0x0051ccf2, 0x0051d0f2, 0x0051d4f2, 0x0051d8f2, 0x0051dcf2, // 21616-21623
	0x0051e0f2, 0x0051e4f2, 0x0051e8f2, 0x0051ecf2, 0x0051f0f2, 0x0051f4f2, 0x0051f8f2, 0x0051fcf2, // 21624-21631
	0x005200f2, 0x005204f2, 0x005208f2, 0x00520cf2, 0x005210f2, 0x005214f2, 0x005218f2, 0x00521cf2, // 21632-21639
	0x005220f2, 0x005224f2, 0x005228f2, 0x00522cf2, 0x005230f2, 0x005234f2, 0x005238f2, 0x00523cf2, // 21640-21647
	0x005240f2, 0x005244f2, 0x005248f2, 0x00524cf2, 0x005250f2, 0x005254f2, 0x005258f2, 0x00525cf2, // 21648-21655
	0x005260f2, 0x005264f2, 0x005268f2, 0x00526cf2, 0x005270f2, 0x005274f2, 0x005278f2, 0x00527cf2, // 21656-21663
	0x005280f2, 0x005284f2, 0x005288f2, 0x00528cf2, 0x005290f2, 0x005294f2, 0x005298f2, 0x00529cf2, // 21664-21671
	0x0052a0f2, 0x0052a4f2, 0x0052a8f2, 0x0052acf2, 0x0052b0f2, 0x0052b4f2, 0x0052b8f2, 0x0052bcf2, // 21672-21679
	0x0052c0f2, 0x0052c4f2, 0x0052c8f2, 0x0052ccf2, 0x0052d0f2, 0x0052d4f2, 0x0052d8f2, 0x0052dcf2, // 21680-21687
	0x0052e0f2, 0x0052e4f2, 0x0052e8f2, 0x0052ecf2, 0x0052f0f2, 0x0052f4f2, 0x0052f8f2, 0x0052fcf2, // 21688-21695
	0x005300f2, 0x005304f2, 0x005308f2, 0x00530cf2, 0x005310f2, 0x005314f2, 0x005318f2, 0x00531cf2, // 21696-21703
	0x005320f2, 0x005324f2, 0x005328f2, 0x00532cf2, 0x005330f2, 0x005334f2, 0x005338f2, 0x00533cf2, // 21704-21711
	0x005340f2, 0x005344f2, 0x005348f2, 0x00534cf2, 0x005350f2, 0x005354f2, 0x005358f2, 0x00535cf2, // 21712-21719
	0x005360f2, 0x005364f2, 0x005368f2, 0x00536cf2, 0x005370f2, 0x005374f2, 0x005378f2, 0x00537cf2, // 21720-21727
	0x005380f2, 0x005384f2, 0x005388f2, 0x00538cf2, 0x005390f2, 0x005394f2, 0x005398f2, 0x00539cf2, // 21728-21735
	0x0053a0f2, 0x0053a4f2, 0x0053a8f2, 0x0053acf2, 0x0053b0f2, 0x0053b4f2, 0x0053b8f2, 0x0053bcf2, // 21736-21743
	0x0053c0f2, 0x0053c4f2, 0x0053c8f2, 0x0053ccf2, 0x0053d0f2, 0x0053d4f2, 0x0053d8f2, 0x0053dcf2, // 21744-21751
	0x0053e0f2, 0x0053e4f2, 0x0053e8f2, 0x0053ecf2, 0x0053f0f2, 0x0053f4f2, 0x0053f8f2, 0x0053fcf2, // 21752-21759
	0x005400f2, 0x005404f2, 0x005408f2, 0x00540cf2, 0x005410f2, 0x005414f2, 0x005418f2, 0x00541cf2, // 21760-21767
	0x005420f2, 0x005424f2, 0x005428f2, 0x00542cf2, 0x005430f2, 0x005434f2, 0x005438f2, 0x00543cf2, // 21768-21775
	0x005440f2, 0x005444f2, 0x005448f2, 0x00544cf2, 0x005450f2, 0x005454f2, 0x005458f2, 0x00545cf2, // 21776-21783
	0x005460f2, 0x005464f2, 0x005468f2, 0x00546cf2, 0x005470f2, 0x005474f2, 0x005478f2, 0x00547cf2, // 21784-21791
	0x005480f2, 0x005484f2, 0x005488f2, 0x00548cf2, 0x005490f2, 0x005494f2, 0x005498f2, 0x00549cf2, // 21792-21799
	0x0054a0f2, 0x0054a4f2, 0x0054a8f2, 0x0054acf2, 0x0054b0f2, 0x0054b4f2, 0x0054b8f2, 0x0054bcf2, // 21800-21807
	0x0054c0f2, 0x0054c4f2, 0x0054c8f2, 0x0054ccf2, 0x0054d0f2, 0x0054d4f2, 0x0054d8f2, 0x0054dcf2, // 21808-21815
	0x0054e0f2, 0x0054e4f2, 0x0054e8f2, 0x0054ecf2, 0x0054f0f2, 0x0054f4f2, 0x0054f8f2, 0x0054fcf2, // 21816-21823
	0x005500f2, 0x005504f2, 0x005508f2, 0x00550cf2, 0x005510f2, 0x005514f2, 0x005518f2, 0x00551cf2, // 21824-21831
	0x005520f2, 0x005524f2, 0x005528f2, 0x00552cf2, 0x005530f2, 0x005534f2, 0x005538f2, 0x00553cf2, // 21832-21839
	0x005540f2, 0x005544f2, 0x005548f2, 0x00554cf2, 0x005550f2, 0x005554f2, 0x005558f2, 0x00555cf2, // 21840-21847
	0x005560f2, 0x005564f2, 0x005568f2, 0x00556cf2, 0x005570f2, 0x005574f2, 0x005578f2, 0x00557cf2, // 21848-21855
	0x005580f2, 0x005584f2, 0x005588f2, 0x00558cf2, 0x005590f2, 0x005594f2, 0x005598f2, 0x00559cf2, // 21856-21863
	0x0055a0f2, 0x0055a4f2, 0x0055a8f2, 0x0055acf2, 0x0055b0f2, 0x0055b4f2, 0x0055b8f2, 0x0055bcf2, // 21864-21871
	0x0055c0f2, 0x0055c4f2, 0x0055c8f2, 0x0055ccf2, 0x0055d0f2, 0x0055d4f2, 0x0055d8f2, 0x0055dcf2, // 21872-21879
	0x0055e0f2, 0x0055e4f2, 0x0055e8f2, 0x0055ecf2, 0x0055f0f2, 0x0055f4f2, 0x0055f8f2, 0x0055fcf2, // 21880-21887
	0x005600f2, 0x005604f2, 0x005608f2, 0x00560cf2, 0x005610f2, 0x005614f2, 0x005618f2, 0x00561cf2, // 21888-21895
	0x005620f2, 0x005624f2, 0x005628f2, 0x00562cf2, 0x005630f2, 0x005634f2, 0x005638f2, 0x00563cf2, // 21896-21903
	0x005640f2, 0x005644f2, 0x005648f2, 0x00564cf2, 0x005650f2, 0x005654f2, 0x005658f2, 0x00565cf2, // 21904-21911
	0x005660f2, 0x005664f2, 0x005668f2, 0x00566cf2, 0x005670f2, 0x005674f2, 0x005678f2, 0x00567cf2, // 21912-21919
	0x005680f2, 0x005684f2, 0x005688f2, 0x00568cf2, 0x005690f2, 0x005694f2, 0x005698f2, 0x00569cf2, // 21920-21927
	0x0056a0f2, 0x0056a4f2, 0x0056a8f2, 0x0056acf2, 0x0056b0f2, 0x0056b4f2, 0x0056b8f2, 0x0056bcf2, // 21928-21935
	0x0056c0f2, 0x0056c4f2, 0x0056c8f2, 0x0056ccf2, 0x0056d0f2, 0x0056d4f2, 0x0056d8f2, 0x0056dcf2, // 21936-21943
	0x0056e0f2, 0x0056e4f2, 0x0056e8f2, 0x0056ecf2, 0x0056f0f2, 0x0056f4f2, 0x0056f8f2, 0x0056fcf2, // 21944-21951
	0x005700f2, 0x005704f2, 0x005708f2, 0x00570cf2, 0x005710f2, 0x005714f2, 0x005718f2, 0x00571cf2, // 21952-21959
	0x005720f2, 0x005724f2, 0x005728f2, 0x00572cf2, 0x005730f2, 0x005734f2, 0x005738f2, 0x00573cf2, // 21960-21967
	0x005740f2, 0x005744f2, 0x005748f2, 0x00574cf2, 0x005750f2, 0x005754f2, 0x005758f2, 0x00575cf2, // 21968-21975
	0x005760f2, 0x005764f2, 0x005768f2, 0x00576cf2, 0x005770f2, 0x005774f2, 0x005778f2, 0x00577cf2, // 21976-21983
	0x005780f2, 0x005784f2, 0x005788f2, 0x00578cf2, 0x005790f2, 0x005794f2, 0x005798f2, 0x00579cf2, // 21984-21991
	0x0057a0f2, 0x0057a4f2, 0x0057a8f2, 0x0057acf2, 0x0057b0f2, 0x0057b4f2, 0x0057b8f2, 0x0057bcf2, // 21992-21999
	0x0057c0f2, 0x0057c4f2, 0x0057c8f2, 0x0057ccf2, 0x0057d0f2, 0x0057d4f2, 0x0057d8f2, 0x0057dcf2, // 22000-22007
	0x0057e0f2, 0x0057e4f2, 0x0057e8f2, 0x0057ecf2, 0x0057f0f2, 0x0057f4f2, 0x0057f8f2, 0x0057fcf2, // 22008-22015
	0x005800f2, 0x005804f2, 0x005808f2, 0x00580cf2, 0x005810f2, 0x005814f2, 0x005818f2, 0x00581cf2, // 22016-22023
	0x005820f2, 0x005824f2, 0x005828f2, 0x00582cf2, 0x005830f2, 0x005834f2, 0x005838f2, 0x00583cf2, // 22024-22031
	0x005840f2, 0x005844f2, 0x005848f2, 0x00584cf2, 0x005850f2, 0x005854f2, 0x005858f2, 0x00585cf2, // 22032-22039
	0x005860f2, 0x005864f2, 0x005868f2, 0x00586cf2, 0x005870f2, 0x005874f2, 0x005878f2, 0x00587cf2, // 22040-22047
	0x005880f2, 0x005884f2, 0x005888f2, 0x00588cf2, 0x005890f2, 0x005894f2, 0x005898f2, 0x00589cf2, // 22048-22055
	0x0058a0f2, 0x0058a4f2, 0x0058a8f2, 0x0058acf2, 0x0058b0f2, 0x0058b4f2, 0x0058b8f2, 0x0058bcf2, // 22056-22063
	0x0058c0f2, 0x0058c4f2, 0x0058c8f2, 0x0058ccf2, 0x0058d0f2, 0x0058d4f2, 0x0058d8f2, 0x0058dcf2, // 22064-22071
	0x0058e0f2, 0x0058e4f2, 0x0058e8f2, 0x0058ecf2, 0x0058f0f2, 0x0058f4f2, 0x0058f8f2, 0x0058fcf2, // 22072-22079
	0x005900f2, 0x005904f2, 0x005908f2, 0x00590cf2, 0x005910f2, 0x005914f2, 0x005918f2, 0x00591cf2, // 22080-22087
	0x005920f2, 0x005924f2, 0x005928f2, 0x00592cf2, 0x005930f2, 0x005934f2, 0x005938f2, 0x00593cf2, // 22088-22095
	0x005940f2, 0x005944f2, 0x005948f2, 0x00594cf2, 0x005950f2, 0x005954f2, 0x005958f2, 0x00595cf2, // 22096-22103
	0x005960f2, 0x005964f2, 0x005968f2, 0x00596cf2, 0x005970f2, 0x005974f2, 0x005978f2, 0x00597cf2, // 22104-22111
	0x005980f2, 0x005984f2, 0x005988f2, 0x00598cf2, 0x005990f2, 0x005994f2, 0x005998f2, 0x00599cf2, // 22112-22119
	0x0059a0f2, 0x0059a4f2, 0x0059a8f2, 0x0059acf2, 0x0059b0f2, 0x0059b4f2, 0x0059b8f2, 0x0059bcf2, // 22120-22127
	0x0059c0f2, 0x0059c4f2, 0x0059c8f2, 0x0059ccf2, 0x0059d0f2, 0x0059d4f2, 0x0059d8f2, 0x0059dcf2, // 22128-22135
	0x0059e0f2, 0x0059e4f2, 0x0059e8f2, 0x0059ecf2, 0x0059f0f2, 0x0059f4f2, 0x0059f8f2, 0x0059fcf2, // 22136-22143
	0x005a00f2, 0x005a04f2, 0x005a08f2, 0x005a0cf2, 0x005a10f2, 0x005a14f2, 0x005a18f2, 0x005a1cf2, // 22144-22151
	0x005a20f2, 0x005a24f2, 0x005a28f2, 0x005a2cf2, 0x005a30f2, 0x005a34f2, 0x005a38f2, 0x005a3cf2, // 22152-22159
	0x005a40f2, 0x005a44f2, 0x005a48f2, 0x005a4cf2, 0x005a50f2, 0x005a54f2, 0x005a58f2, 0x005a5cf2, // 22160-22167
	0x005a60f2, 0x005a64f2, 0x005a68f2, 0x005a6cf2, 0x005a70f2, 0x005a74f2, 0x005a78f2, 0x005a7cf2, // 22168-22175
	0x005a80f2, 0x005a84f2, 0x005a88f2, 0x005a8cf2, 0x005a90f2, 0x005a94f2, 0x005a98f2, 0x005a9cf2, // 22176-22183
	0x005aa0f2, 0x005aa4f2, 0x005aa8f2, 0x005aacf2, 0x005ab0f2, 0x005ab4f2, 0x005ab8f2, 0x005abcf2, // 22184-22191
	0x005ac0f2, 0x005ac4f2, 0x005ac8f2, 0x005accf2, 0x005ad0f2, 0x005ad4f2, 0x005ad8f2, 0x005adcf2, // 22192-22199
	0x005ae0f2, 0x005ae4f2, 0x005ae8f2, 0x005aecf2, 0x005af0f2, 0x005af4f2, 0x005af8f2, 0x005afcf2, // 22200-22207
	0x005b00f2, 0x005b04f2, 0x005b08f2, 0x005b0cf2, 0x005b10f2, 0x005b14f2, 0x005b18f2, 0x005b1cf2, // 22208-22215
	0x005b20f2, 0x005b24f2, 0x005b28f2, 0x005b2cf2, 0x005b30f2, 0x005b34f2, 0x005b38f2, 0x005b3cf2, // 22216-22223
	0x005b40f2, 0x005b44f2, 0x005b48f2, 0x005b4cf2, 0x005b50f2, 0x005b54f2, 0x005b58f2, 0x005b5cf2, // 22224-22231
	0x005b60f2, 0x005b64f2, 0x005b68f2, 0x005b6cf2, 0x005b70f2, 0x005b74f2, 0x005b78f2, 0x005b7cf2, // 22232-22239
	0x005b80f2, 0x005b84f2, 0x005b88f2, 0x005b8cf2, 0x005b90f2, 0x005b94f2, 0x005b98f2, 0x005b9cf2, // 22240-22247
	0x005ba0f2, 0x005ba4f2, 0x005ba8f2, 0x005bacf2, 0x005bb0f2, 0x005bb4f2, 0x005bb8f2, 0x005bbcf2, // 22248-22255
	0x005bc0f2, 0x005bc4f2, 0x005bc8f2, 0x005bccf2, 0x005bd0f2, 0x005bd4f2, 0x005bd8f2, 0x005bdcf2, // 22256-22263
	0x005be0f2, 0x005be4f2, 0x005be8f2, 0x005becf2, 0x005bf0f2, 0x005bf4f2, 0x005bf8f2, 0x005bfcf2, // 22264-22271
	0x005c00f2, 0x005c04f2, 0x005c08f2, 0x005c0cf2, 0x005c10f2, 0x005c14f2, 0x005c18f2, 0x005c1cf2, // 22272-22279
	0x005c20f2, 0x005c24f2, 0x005c28f2, 0x005c2cf2, 0x005c30f2, 0x005c34f2, 0x005c38f2, 0x005c3cf2, // 22280-22287
	0x005c40f2, 0x005c44f2, 0x005c48f2, 0x005c4cf2, 0x005c50f2, 0x005c54f2, 0x005c58f2, 0x005c5cf2, // 22288-22295
	0x005c60f2, 0x005c64f2, 0x005c68f2, 0x005c6cf2, 0x005c70f2, 0x005c74f2, 0x005c78f2, 0x005c7cf2, // 22296-22303
	0x005c80f2, 0x005c84f2, 0x005c88f2, 0x005c8cf2, 0x005c90f2, 0x005c94f2, 0x005c98f2, 0x005c9cf2, // 22304-22311
	0x005ca0f2, 0x005ca4f2, 0x005ca8f2, 0x005cacf2, 0x005cb0f2, 0x005cb4f2, 0x005cb8f2, 0x005cbcf2, // 22312-22319
	0x005cc0f2, 0x005cc4f2, 0x005cc8f2, 0x005cccf2, 0x005cd0f2, 0x005cd4f2, 0x005cd8f2, 0x005cdcf2, // 22320-22327
	0x005ce0f2, 0x005ce4f2, 0x005ce8f2, 0x005cecf2, 0x005cf0f2, 0x005cf4f2, 0x005cf8f2, 0x005cfcf2, // 22328-22335
	0x005d00f2, 0x005d04f2, 0x005d08f2, 0x005d0cf2, 0x005d10f2, 0x005d14f2, 0x005d18f2, 0x005d1cf2, // 22336-22343
	0x005d20f2, 0x005d24f2, 0x005d28f2, 0x005d2cf2, 0x005d30f2, 0x005d34f2, 0x005d38f2, 0x005d3cf2, // 22344-22351
	0x005d40f2, 0x005d44f2, 0x005d48f2, 0x005d4cf2, 0x005d50f2, 0x005d54f2, 0x005d58f2, 0x005d5cf2, // 22352-22359
	0x005d60f2, 0x005d64f2, 0x005d68f2, 0x005d6cf2, 0x005d70f2, 0x005d74f2, 0x005d78f2, 0x005d7cf2, // 22360-22367
	0x005d80f2, 0x005d84f2, 0x005d88f2, 0x005d8cf2, 0x005d90f2, 0x005d94f2, 0x005d98f2, 0x005d9cf2, // 22368-22375
	0x005da0f2, 0x005da4f2, 0x005da8f2, 0x005dacf2, 0x005db0f2, 0x005db4f2, 0x005db8f2, 0x005dbcf2, // 22376-22383
	0x005dc0f2, 0x005dc4f2, 0x005dc8f2, 0x005dccf2, 0x005dd0f2, 0x005dd4f2, 0x005dd8f2, 0x005ddcf2, // 22384-22391
	0x005de0f2, 0x005de4f2, 0x005de8f2, 0x005decf2, 0x005df0f2, 0x005df4f2, 0x005df8f2, 0x005dfcf2, // 22392-22399
	0x005e00f2, 0x005e04f2, 0x005e08f2, 0x005e0cf2, 0x005e10f2, 0x005e14f2, 0x005e18f2, 0x005e1cf2, // 22400-22407
	0x005e20f2, 0x005e24f2, 0x005e28f2, 0x005e2cf2, 0x005e30f2, 0x005e34f2, 0x005e38f2, 0x005e3cf2, // 22408-22415
	0x005e40f2, 0x005e44f2, 0x005e48f2, 0x005e4cf2, 0x005e50f2, 0x005e54f2, 0x005e58f2, 0x005e5cf2, // 22416-22423
	0x005e60f2, 0x005e64f2, 0x005e68f2, 0x005e6cf2, 0x005e70f2, 0x005e74f2, 0x005e78f2, 0x005e7cf2, // 22424-22431
	0x005e80f2, 0x005e84f2, 0x005e88f2, 0x005e8cf2, 0x005e90f2, 0x005e94f2, 0x005e98f2, 0x005e9cf2, // 22432-22439
	0x005ea0f2, 0x005ea4f2, 0x005ea8f2, 0x005eacf2, 0x005eb0f2, 0x005eb4f2, 0x005eb8f2, 0x005ebcf2, // 22440-22447
	0x005ec0f2, 0x005ec4f2, 0x005ec8f2, 0x005eccf2, 0x005ed0f2, 0x005ed4f2, 0x005ed8f2, 0x005edcf2, // 22448-22455
	0x005ee0f2, 0x005ee4f2, 0x005ee8f2, 0x005eecf2, 0x005ef0f2, 0x005ef4f2, 0x005ef8f2, 0x005efcf2, // 22456-22463
	0x005f00f2, 0x005f04f2, 0x005f08f2, 0x005f0cf2, 0x005f10f2, 0x005f14f2, 0x005f18f2, 0x005f1cf2, // 22464-22471
	0x005f20f2, 0x005f24f2, 0x005f28f2, 0x005f2cf2, 0x005f30f2, 0x005f34f2, 0x005f38f2, 0x005f3cf2, // 22472-22479
	0x005f40f2, 0x005f44f2, 0x005f48f2, 0x005f4cf2, 0x005f50f2, 0x005f54f2, 0x005f58f2, 0x005f5cf2, // 22480-22487
	0x005f60f2, 0x005f64f2, 0x005f68f2, 0x005f6cf2, 0x005f70f2, 0x005f74f2, 0x005f78f2, 0x005f7cf2, // 22488-22495
	0x005f80f2, 0x005f84f2, 0x005f88f2, 0x005f8cf2, 0x005f90f2, 0x005f94f2, 0x005f98f2, 0x005f9cf2, // 22496-22503
	0x005fa0f2, 0x005fa4f2, 0x005fa8f2, 0x005facf2, 0x005fb0f2, 0x005fb4f2, 0x005fb8f2, 0x005fbcf2, // 22504-22511
	0x005fc0f2, 0x005fc4f2, 0x005fc8f2, 0x005fccf2, 0x005fd0f2, 0x005fd4f2, 0x005fd8f2, 0x005fdcf2, // 22512-22519
	0x005fe0f2, 0x005fe4f2, 0x005fe8f2, 0x005fecf2, 0x005ff0f2, 0x005ff4f2, 0x005ff8f2, 0x005ffcf2, // 22520-22527
	0x006000f2, 0x006004f2, 0x006008f2, 0x00600cf2, 0x006010f2, 0x006014f2, 0x006018f2, 0x00601cf2, // 22528-22535
	0x006020f2, 0x006024f2, 0x006028f2, 0x00602cf2, 0x006030f2, 0x006034f2, 0x006038f2, 0x00603cf2, // 22536-22543
	0x006040f2, 0x006044f2, 0x006048f2, 0x00604cf2, 0x006050f2, 0x006054f2, 0x006058f2, 0x00605cf2, // 22544-22551
	0x006060f2, 0x006064f2, 0x006068f2, 0x00606cf2, 0x006070f2, 0x006074f2, 0x006078f2, 0x00607cf2, // 22552-22559
	0x006080f2, 0x006084f2, 0x006088f2, 0x00608cf2, 0x006090f2, 0x006094f2, 0x006098f2, 0x00609cf2, // 22560-22567
	0x0060a0f2, 0x0060a4f2, 0x0060a8f2, 0x0060acf2, 0x0060b0f2, 0x0060b4f2, 0x0060b8f2, 0x0060bcf2, // 22568-22575
	0x0060c0f2, 0x0060c4f2, 0x0060c8f2, 0x0060ccf2, 0x0060d0f2, 0x0060d4f2, 0x0060d8f2, 0x0060dcf2, // 22576-22583
	0x0060e0f2, 0x0060e4f2, 0x0060e8f2, 0x0060ecf2, 0x0060f0f2, 0x0060f4f2, 0x0060f8f2, 0x0060fcf2, // 22584-22591
	0x006100f2, 0x006104f2, 0x006108f2, 0x00610cf2, 0x006110f2, 0x006114f2, 0x006118f2, 0x00611cf2, // 22592-22599
	0x006120f2, 0x006124f2, 0x006128f2, 0x00612cf2, 0x006130f2, 0x006134f2, 0x006138f2, 0x00613cf2, // 22600-22607
	0x006140f2, 0x006144f2, 0x006148f2, 0x00614cf2, 0x006150f2, 0x006154f2, 0x006158f2, 0x00615cf2, // 22608-22615
	0x006160f2, 0x006164f2, 0x006168f2, 0x00616cf2, 0x006170f2, 0x006174f2, 0x006178f2, 0x00617cf2, // 22616-22623
	0x006180f2, 0x006184f2, 0x006188f2, 0x00618cf2, 0x006190f2, 0x006194f2, 0x006198f2, 0x00619cf2, // 22624-22631
	0x0061a0f2, 0x0061a4f2, 0x0061a8f2, 0x0061acf2, 0x0061b0f2, 0x0061b4f2, 0x0061b8f2, 0x0061bcf2, // 22632-22639
	0x0061c0f2, 0x0061c4f2, 0x0061c8f2, 0x0061ccf2, 0x0061d0f2, 0x0061d4f2, 0x0061d8f2, 0x0061dcf2, // 22640-22647
	0x0061e0f2, 0x0061e4f2, 0x0061e8f2, 0x0061ecf2, 0x0061f0f2, 0x0061f4f2, 0x0061f8f2, 0x0061fcf2, // 22648-22655
	0x006200f2, 0x006204f2, 0x006208f2, 0x00620cf2, 0x006210f2, 0x006214f2, 0x006218f2, 0x00621cf2, // 22656-22663
	0x006220f2, 0x006224f2, 0x006228f2, 0x00622cf2, 0x006230f2, 0x006234f2, 0x006238f2, 0x00623cf2, // 22664-22671
	0x006240f2, 0x006244f2, 0x006248f2, 0x00624cf2, 0x006250f2, 0x006254f2, 0x006258f2, 0x00625cf2, // 22672-22679
	0x006260f2, 0x006264f2, 0x006268f2, 0x00626cf2, 0x006270f2, 0x006274f2, 0x006278f2, 0x00627cf2, // 22680-22687
	0x006280f2, 0x006284f2, 0x006288f2, 0x00628cf2, 0x006290f2, 0x006294f2, 0x006298f2, 0x00629cf2, // 22688-22695
	0x0062a0f2, 0x0062a4f2, 0x0062a8f2, 0x0062acf2, 0x0062b0f2, 0x0062b4f2, 0x0062b8f2, 0x0062bcf2, // 22696-22703
	0x0062c0f2, 0x0062c4f2, 0x0062c8f2, 0x0062ccf2, 0x0062d0f2, 0x0062d4f2, 0x0062d8f2, 0x0062dcf2, // 22704-22711
	0x0062e0f2, 0x0062e4f2, 0x0062e8f2, 0x0062ecf2, 0x0062f0f2, 0x0062f4f2, 0x0062f8f2, 0x0062fcf2, // 22712-22719
	0x006300f2, 0x006304f2, 0x006308f2, 0x00630cf2, 0x006310f2, 0x006314f2, 0x006318f2, 0x00631cf2, // 22720-22727
	0x006320f2, 0x006324f2, 0x006328f2, 0x00632cf2, 0x006330f2, 0x006334f2, 0x006338f2, 0x00633cf2, // 22728-22735
	0x006340f2, 0x006344f2, 0x006348f2, 0x00634cf2, 0x006350f2, 0x006354f2, 0x006358f2, 0x00635cf2, // 22736-22743
	0x006360f2, 0x006364f2, 0x006368f2, 0x00636cf2, 0x006370f2, 0x006374f2, 0x006378f2, 0x00637cf2, // 22744-22751
	0x006380f2, 0x006384f2, 0x006388f2, 0x00638cf2, 0x006390f2, 0x006394f2, 0x006398f2, 0x00639cf2, // 22752-22759
	0x0063a0f2, 0x0063a4f2, 0x0063a8f2, 0x0063acf2, 0x0063b0f2, 0x0063b4f2, 0x0063b8f2, 0x0063bcf2, // 22760-22767
	0x0063c0f2, 0x0063c4f2, 0x0063c8f2, 0x0063ccf2, 0x0063d0f2, 0x0063d4f2, 0x0063d8f2, 0x0063dcf2, // 22768-22775
	0x0063e0f2, 0x0063e4f2, 0x0063e8f2, 0x0063ecf2, 0x0063f0f2, 0x0063f4f2, 0x0063f8f2, 0x0063fcf2, // 22776-22783
	0x006400f2, 0x006404f2, 0x006408f2, 0x00640cf2, 0x006410f2, 0x006414f2, 0x006418f2, 0x00641cf2, // 22784-22791
	0x006420f2, 0x006424f2, 0x006428f2, 0x00642cf2, 0x006430f2, 0x006434f2, 0x006438f2, 0x00643cf2, // 22792-22799
	0x006440f2, 0x006444f2, 0x006448f2, 0x00644cf2, 0x006450f2, 0x006454f2, 0x006458f2, 0x00645cf2, // 22800-22807
	0x006460f2, 0x006464f2, 0x006468f2, 0x00646cf2, 0x006470f2, 0x006474f2, 0x006478f2, 0x00647cf2, // 22808-22815
	0x006480f2, 0x006484f2, 0x006488f2, 0x00648cf2, 0x006490f2, 0x006494f2, 0x006498f2, 0x00649cf2, // 22816-22823
	0x0064a0f2, 0x0064a4f2, 0x0064a8f2, 0x0064acf2, 0x0064b0f2, 0x0064b4f2, 0x0064b8f2, 0x0064bcf2, // 22824-22831
	0x0064c0f2, 0x0064c4f2, 0x0064c8f2, 0x0064ccf2, 0x0064d0f2, 0x0064d4f2, 0x0064d8f2, 0x0064dcf2, // 22832-22839
	0x0064e0f2, 0x0064e4f2, 0x0064e8f2, 0x0064ecf2, 0x0064f0f2, 0x0064f4f2, 0x0064f8f2, 0x0064fcf2, // 22840-22847
	0x006500f2, 0x006504f2, 0x006508f2, 0x00650cf2, 0x006510f2, 0x006514f2, 0x006518f2, 0x00651cf2, // 22848-22855
	0x006520f2, 0x006524f2, 0x006528f2, 0x00652cf2, 0x006530f2, 0x006534f2, 0x006538f2, 0x00653cf2, // 22856-22863
	0x006540f2, 0x006544f2, 0x006548f2, 0x00654cf2, 0x006550f2, 0x006554f2, 0x006558f2, 0x00655cf2, // 22864-22871
	0x006560f2, 0x006564f2, 0x006568f2, 0x00656cf2, 0x006570f2, 0x006574f2, 0x006578f2, 0x00657cf2, // 22872-22879
	0x006580f2, 0x006584f2, 0x006588f2, 0x00658cf2, 0x006590f2, 0x006594f2, 0x006598f2, 0x00659cf2, // 22880-22887
	0x0065a0f2, 0x0065a4f2, 0x0065a8f2, 0x0065acf2, 0x0065b0f2, 0x0065b4f2, 0x0065b8f2, 0x0065bcf2, // 22888-22895
	0x0065c0f2, 0x0065c4f2, 0x0065c8f2, 0x0065ccf2, 0x0065d0f2, 0x0065d4f2, 0x0065d8f2, 0x0065dcf2, // 22896-22903
	0x0065e0f2, 0x0065e4f2, 0x0065e8f2, 0x0065ecf2, 0x0065f0f2, 0x0065f4f2, 0x0065f8f2, 0x0065fcf2, // 22904-22911
	0x006600f2, 0x006604f2, 0x006608f2, 0x00660cf2, 0x006610f2, 0x006614f2, 0x006618f2, 0x00661cf2, // 22912-22919
	0x006620f2, 0x006624f2, 0x006628f2, 0x00662cf2, 0x006630f2, 0x006634f2, 0x006638f2, 0x00663cf2, // 22920-22927
	0x006640f2, 0x006644f2, 0x006648f2, 0x00664cf2, 0x006650f2, 0x006654f2, 0x006658f2, 0x00665cf2, // 22928-22935
	0x006660f2, 0x006664f2, 0x006668f2, 0x00666cf2, 0x006670f2, 0x006674f2, 0x006678f2, 0x00667cf2, // 22936-22943
	0x006680f2, 0x006684f2, 0x006688f2, 0x00668cf2, 0x006690f2, 0x006694f2, 0x006698f2, 0x00669cf2, // 22944-22951
	0x0066a0f2, 0x0066a4f2, 0x0066a8f2, 0x0066acf2, 0x0066b0f2, 0x0066b4f2, 0x0066b8f2, 0x0066bcf2, // 22952-22959
	0x0066c0f2, 0x0066c4f2, 0x0066c8f2, 0x0066ccf2, 0x0066d0f2, 0x0066d4f2, 0x0066d8f2, 0x0066dcf2, // 22960-22967
	0x0066e0f2, 0x0066e4f2, 0x0066e8f2, 0x0066ecf2, 0x0066f0f2, 0x0066f4f2, 0x0066f8f2, 0x0066fcf2, // 22968-22975
	0x006700f2, 0x006704f2, 0x006708f2, 0x00670cf2, 0x006710f2, 0x006714f2, 0x006718f2, 0x00671cf2, // 22976-22983
	0x006720f2, 0x006724f2, 0x006728f2, 0x00672cf2, 0x006730f2, 0x006734f2, 0x006738f2, 0x00673cf2, // 22984-22991
	0x006740f2, 0x006744f2, 0x006748f2, 0x00674cf2, 0x006750f2, 0x006754f2, 0x006758f2, 0x00675cf2, // 22992-22999
	0x006760f2, 0x006764f2, 0x006768f2, 0x00676cf2, 0x006770f2, 0x006774f2, 0x006778f2, 0x00677cf2, // 23000-23007
	0x006780f2, 0x006784f2, 0x006788f2, 0x00678cf2, 0x006790f2, 0x006794f2, 0x006798f2, 0x00679cf2, // 23008-23015
	0x0067a0f2, 0x0067a4f2, 0x0067a8f2, 0x0067acf2, 0x0067b0f2, 0x0067b4f2, 0x0067b8f2, 0x0067bcf2, // 23016-23023
	0x0067c0f2, 0x0067c4f2, 0x0067c8f2, 0x0067ccf2, 0x0067d0f2, 0x0067d4f2, 0x0067d8f2, 0x0067dcf2, // 23024-23031
	0x0067e0f2, 0x0067e4f2, 0x0067e8f2, 0x0067ecf2, 0x0067f0f2, 0x0067f4f2, 0x0067f8f2, 0x0067fcf2, // 23032-23039
	0x006800f2, 0x006804f2, 0x006808f2, 0x00680cf2, 0x006810f2, 0x006814f2, 0x006818f2, 0x00681cf2, // 23040-23047
	0x006820f2, 0x006824f2, 0x006828f2, 0x00682cf2, 0x006830f2, 0x006834f2, 0x006838f2, 0x00683cf2, // 23048-23055
	0x006840f2, 0x006844f2, 0x006848f2, 0x00684cf2, 0x006850f2, 0x006854f2, 0x006858f2, 0x00685cf2, // 23056-23063
	0x006860f2, 0x006864f2, 0x006868f2, 0x00686cf2, 0x006870f2, 0x006874f2, 0x006878f2, 0x00687cf2, // 23064-23071
	0x006880f2, 0x006884f2, 0x006888f2, 0x00688cf2, 0x006890f2, 0x006894f2, 0x006898f2, 0x00689cf2, // 23072-23079
	0x0068a0f2, 0x0068a4f2, 0x0068a8f2, 0x0068acf2, 0x0068b0f2, 0x0068b4f2, 0x0068b8f2, 0x0068bcf2, // 23080-23087
	0x0068c0f2, 0x0068c4f2, 0x0068c8f2, 0x0068ccf2, 0x0068d0f2, 0x0068d4f2, 0x0068d8f2, 0x0068dcf2, // 23088-23095
	0x0068e0f2, 0x0068e4f2, 0x0068e8f2, 0x0068ecf2, 0x0068f0f2, 0x0068f4f2, 0x0068f8f2, 0x0068fcf2, // 23096-23103
	0x006900f2, 0x006904f2, 0x006908f2, 0x00690cf2, 0x006910f2, 0x006914f2, 0x006918f2, 0x00691cf2, // 23104-23111
	0x006920f2, 0x006924f2, 0x006928f2, 0x00692cf2, 0x006930f2, 0x006934f2, 0x006938f2, 0x00693cf2, // 23112-23119
	0x006940f2, 0x006944f2, 0x006948f2, 0x00694cf2, 0x006950f2, 0x006954f2, 0x006958f2, 0x00695cf2, // 23120-23127
	0x006960f2, 0x006964f2, 0x006968f2, 0x00696cf2, 0x006970f2, 0x006974f2, 0x006978f2, 0x00697cf2, // 23128-23135
	0x006980f2, 0x006984f2, 0x006988f2, 0x00698cf2, 0x006990f2, 0x006994f2, 0x006998f2, 0x00699cf2, // 23136-23143
	0x0069a0f2, 0x0069a4f2, 0x0069a8f2, 0x0069acf2, 0x0069b0f2, 0x0069b4f2, 0x0069b8f2, 0x0069bcf2, // 23144-23151
	0x0069c0f2, 0x0069c4f2, 0x0069c8f2, 0x0069ccf2, 0x0069d0f2, 0x0069d4f2, 0x0069d8f2, 0x0069dcf2, // 23152-23159
	0x0069e0f2, 0x0069e4f2, 0x0069e8f2, 0x0069ecf2, 0x0069f0f2, 0x0069f4f2, 0x0069f8f2, 0x0069fcf2, // 23160-23167
	0x006a00f2, 0x006a04f2, 0x006a08f2, 0x006a0cf2, 0x006a10f2, 0x006a14f2, 0x006a18f2, 0x006a1cf2, // 23168-23175
	0x006a20f2, 0x006a24f2, 0x006a28f2, 0x006a2cf2, 0x006a30f2, 0x006a34f2, 0x006a38f2, 0x006a3cf2, // 23176-23183
	0x006a40f2, 0x006a44f2, 0x006a48f2, 0x006a4cf2, 0x006a50f2, 0x006a54f2, 0x006a58f2, 0x006a5cf2, // 23184-23191
	0x006a60f2, 0x006a64f2, 0x006a68f2, 0x006a6cf2, 0x006a70f2, 0x006a74f2, 0x006a78f2, 0x006a7cf2, // 23192-23199
	0x006a80f2, 0x006a84f2, 0x006a88f2, 0x006a8cf2, 0x006a90f2, 0x006a94f2, 0x006a98f2, 0x006a9cf2, // 23200-23207
	0x006aa0f2, 0x006aa4f2, 0x006aa8f2, 0x006aacf2, 0x006ab0f2, 0x006ab4f2, 0x006ab8f2, 0x006abcf2, // 23208-23215
	0x006ac0f2, 0x006ac4f2, 0x006ac8f2, 0x006accf2, 0x006ad0f2, 0x006ad4f2, 0x006ad8f2, 0x006adcf2, // 23216-23223
	0x006ae0f2, 0x006ae4f2, 0x006ae8f2, 0x006aecf2, 0x006af0f2, 0x006af4f2, 0x006af8f2, 0x006afcf2, // 23224-23231
	0x006b00f2, 0x006b04f2, 0x006b08f2, 0x006b0cf2, 0x006b10f2, 0x006b14f2, 0x006b18f2, 0x006b1cf2, // 23232-23239
	0x006b20f2, 0x006b24f2, 0x006b28f2, 0x006b2cf2, 0x006b30f2, 0x006b34f2, 0x006b38f2, 0x006b3cf2, // 23240-23247
	0x006b40f2, 0x006b44f2, 0x006b48f2, 0x006b4cf2, 0x006b50f2, 0x006b54f2, 0x006b58f2, 0x006b5cf2, // 23248-23255
	0x006b60f2, 0x006b64f2, 0x006b68f2, 0x006b6cf2, 0x006b70f2, 0x006b74f2, 0x006b78f2, 0x006b7cf2, // 23256-23263
	0x006b80f2, 0x006b84f2, 0x006b88f2, 0x006b8cf2, 0x006b90f2, 0x006b94f2, 0x006b98f2, 0x006b9cf2, // 23264-23271
	0x006ba0f2, 0x006ba4f2, 0x006ba8f2, 0x006bacf2, 0x006bb0f2, 0x006bb4f2, 0x006bb8f2, 0x006bbcf2, // 23272-23279
	0x006bc0f2, 0x006bc4f2, 0x006bc8f2, 0x006bccf2, 0x006bd0f2, 0x006bd4f2, 0x006bd8f2, 0x006bdcf2, // 23280-23287
	0x006be0f2, 0x006be4f2, 0x006be8f2, 0x006becf2, 0x006bf0f2, 0x006bf4f2, 0x006bf8f2, 0x006bfcf2, // 23288-23295
	0x006c00f2, 0x006c04f2, 0x006c08f2, 0x006c0cf2, 0x006c10f2, 0x006c14f2, 0x006c18f2, 0x006c1cf2, // 23296-23303
	0x006c20f2, 0x006c24f2, 0x006c28f2, 0x006c2cf2, 0x006c30f2, 0x006c34f2, 0x006c38f2, 0x006c3cf2, // 23304-23311
	0x006c40f2, 0x006c44f2, 0x006c48f2, 0x006c4cf2, 0x006c50f2, 0x006c54f2, 0x006c58f2, 0x006c5cf2, // 23312-23319
	0x006c60f2, 0x006c64f2, 0x006c68f2, 0x006c6cf2, 0x006c70f2, 0x006c74f2, 0x006c78f2, 0x006c7cf2, // 23320-23327
	0x006c80f2, 0x006c84f2, 0x006c88f2, 0x006c8cf2, 0x006c90f2, 0x006c94f2, 0x006c98f2, 0x006c9cf2, // 23328-23335
	0x006ca0f2, 0x006ca4f2, 0x006ca8f2, 0x006cacf2, 0x006cb0f2, 0x006cb4f2, 0x006cb8f2, 0x006cbcf2, // 23336-23343
	0x006cc0f2, 0x006cc4f2, 0x006cc8f2, 0x006cccf2, 0x006cd0f2, 0x006cd4f2, 0x006cd8f2, 0x006cdcf2, // 23344-23351
	0x006ce0f2, 0x006ce4f2, 0x006ce8f2, 0x006cecf2, 0x006cf0f2, 0x006cf4f2, 0x006cf8f2, 0x006cfcf2, // 23352-23359
	0x006d00f2, 0x006d04f2, 0x006d08f2, 0x006d0cf2, 0x006d10f2, 0x006d14f2, 0x006d18f2, 0x006d1cf2, // 23360-23367
	0x006d20f2, 0x006d24f2, 0x006d28f2, 0x006d2cf2, 0x006d30f2, 0x006d34f2, 0x006d38f2, 0x006d3cf2, // 23368-23375
	0x006d40f2, 0x006d44f2, 0x006d48f2, 0x006d4cf2, 0x006d50f2, 0x006d54f2, 0x006d58f2, 0x006d5cf2, // 23376-23383
	0x006d60f2, 0x006d64f2, 0x006d68f2, 0x006d6cf2, 0x006d70f2, 0x006d74f2, 0x006d78f2, 0x006d7cf2, // 23384-23391
	0x006d80f2, 0x006d84f2, 0x006d88f2, 0x006d8cf2, 0x006d90f2, 0x006d94f2, 0x006d98f2, 0x006d9cf2, // 23392-23399
	0x006da0f2, 0x006da4f2, 0x006da8f2, 0x006dacf2, 0x006db0f2, 0x006db4f2, 0x006db8f2, 0x006dbcf2, // 23400-23407
	0x006dc0f2, 0x006dc4f2, 0x006dc8f2, 0x006dccf2, 0x006dd0f2, 0x006dd4f2, 0x006dd8f2, 0x006ddcf2, // 23408-23415
	0x006de0f2, 0x006de4f2, 0x006de8f2, 0x006decf2, 0x006df0f2, 0x006df4f2, 0x006df8f2, 0x006dfcf2, // 23416-23423
	0x006e00f2, 0x006e04f2, 0x006e08f2, 0x006e0cf2, 0x006e10f2, 0x006e14f2, 0x006e18f2, 0x006e1cf2, // 23424-23431
	0x006e20f2, 0x006e24f2, 0x006e28f2, 0x006e2cf2, 0x006e30f2, 0x006e34f2, 0x006e38f2, 0x006e3cf2, // 23432-23439
	0x006e40f2, 0x006e44f2, 0x006e48f2, 0x006e4cf2, 0x006e50f2, 0x006e54f2, 0x006e58f2, 0x006e5cf2, // 23440-23447
	0x006e60f2, 0x006e64f2, 0x006e68f2, 0x006e6cf2, 0x006e70f2, 0x006e74f2, 0x006e78f2, 0x006e7cf2, // 23448-23455
	0x006e80f2, 0x006e84f2, 0x006e88f2, 0x006e8cf2, 0x006e90f2, 0x006e94f2, 0x006e98f2, 0x006e9cf2, // 23456-23463
	0x006ea0f2, 0x006ea4f2, 0x006ea8f2, 0x006eacf2, 0x006eb0f2, 0x006eb4f2, 0x006eb8f2, 0x006ebcf2, // 23464-23471
	0x006ec0f2, 0x006ec4f2, 0x006ec8f2, 0x006eccf2, 0x006ed0f2, 0x006ed4f2, 0x006ed8f2, 0x006edcf2, // 23472-23479
	0x006ee0f2, 0x006ee4f2, 0x006ee8f2, 0x006eecf2, 0x006ef0f2, 0x006ef4f2, 0x006ef8f2, 0x006efcf2, // 23480-23487
	0x006f00f2, 0x006f04f2, 0x006f08f2, 0x006f0cf2, 0x006f10f2, 0x006f14f2, 0x006f18f2, 0x006f1cf2, // 23488-23495
	0x006f20f2, 0x006f24f2, 0x006f28f2, 0x006f2cf2, 0x006f30f2, 0x006f34f2, 0x006f38f2, 0x006f3cf2, // 23496-23503
	0x006f40f2, 0x006f44f2, 0x006f48f2, 0x006f4cf2, 0x006f50f2, 0x006f54f2, 0x006f58f2, 0x006f5cf2, // 23504-23511
	0x006f60f2, 0x006f64f2, 0x006f68f2, 0x006f6cf2, 0x006f70f2, 0x006f74f2, 0x006f78f2, 0x006f7cf2, // 23512-23519
	0x006f80f2, 0x006f84f2, 0x006f88f2, 0x006f8cf2, 0x006f90f2, 0x006f94f2, 0x006f98f2, 0x006f9cf2, // 23520-23527
	0x006fa0f2, 0x006fa4f2, 0x006fa8f2, 0x006facf2, 0x006fb0f2, 0x006fb4f2, 0x006fb8f2, 0x006fbcf2, // 23528-23535
	0x006fc0f2, 0x006fc4f2, 0x006fc8f2, 0x006fccf2, 0x006fd0f2, 0x006fd4f2, 0x006fd8f2, 0x006fdcf2, // 23536-23543
	0x006fe0f2, 0x006fe4f2, 0x006fe8f2, 0x006fecf2, 0x006ff0f2, 0x006ff4f2, 0x006ff8f2, 0x006ffcf2, // 23544-23551
	0x007000f2, 0x007004f2, 0x007008f2, 0x00700cf2, 0x007010f2, 0x007014f2, 0x007018f2, 0x00701cf2, // 23552-23559
	0x007020f2, 0x007024f2, 0x007028f2, 0x00702cf2, 0x007030f2, 0x007034f2, 0x007038f2, 0x00703cf2, // 23560-23567
	0x007040f2, 0x007044f2, 0x007048f2, 0x00704cf2, 0x007050f2, 0x007054f2, 0x007058f2, 0x00705cf2, // 23568-23575
	0x007060f2, 0x007064f2, 0x007068f2, 0x00706cf2, 0x007070f2, 0x007074f2, 0x007078f2, 0x00707cf2, // 23576-23583
	0x007080f2, 0x007084f2, 0x007088f2, 0x00708cf2, 0x007090f2, 0x007094f2, 0x007098f2, 0x00709cf2, // 23584-23591
	0x0070a0f2, 0x0070a4f2, 0x0070a8f2, 0x0070acf2, 0x0070b0f2, 0x0070b4f2, 0x0070b8f2, 0x0070bcf2, // 23592-23599
	0x0070c0f2, 0x0070c4f2, 0x0070c8f2, 0x0070ccf2, 0x0070d0f2, 0x0070d4f2, 0x0070d8f2, 0x0070dcf2, // 23600-23607
	0x0070e0f2, 0x0070e4f2, 0x0070e8f2, 0x0070ecf2, 0x0070f0f2, 0x0070f4f2, 0x0070f8f2, 0x0070fcf2, // 23608-23615
	0x007100f2, 0x007104f2, 0x007108f2, 0x00710cf2, 0x007110f2, 0x007114f2, 0x007118f2, 0x00711cf2, // 23616-23623
	0x007120f2, 0x007124f2, 0x007128f2, 0x00712cf2, 0x007130f2, 0x007134f2, 0x007138f2, 0x00713cf2, // 23624-23631
	0x007140f2, 0x007144f2, 0x007148f2, 0x00714cf2, 0x007150f2, 0x007154f2, 0x007158f2, 0x00715cf2, // 23632-23639
	0x007160f2, 0x007164f2, 0x007168f2, 0x00716cf2, 0x007170f2, 0x007174f2, 0x007178f2, 0x00717cf2, // 23640-23647
	0x007180f2, 0x007184f2, 0x007188f2, 0x00718cf2, 0x007190f2, 0x007194f2, 0x007198f2, 0x00719cf2, // 23648-23655
	0x0071a0f2, 0x0071a4f2, 0x0071a8f2, 0x0071acf2, 0x0071b0f2, 0x0071b4f2, 0x0071b8f2, 0x0071bcf2, // 23656-23663
	0x0071c0f2, 0x0071c4f2, 0x0071c8f2, 0x0071ccf2, 0x0071d0f2, 0x0071d4f2, 0x0071d8f2, 0x0071dcf2, // 23664-23671
	0x0071e0f2, 0x0071e4f2, 0x0071e8f2, 0x0071ecf2, 0x0071f0f2, 0x0071f4f2, 0x0071f8f2, 0x0071fcf2, // 23672-23679
	0x007200f2, 0x007204f2, 0x007208f2, 0x00720cf2, 0x007210f2, 0x007214f2, 0x007218f2, 0x00721cf2, // 23680-23687
	0x007220f2, 0x007224f2, 0x007228f2, 0x00722cf2, 0x007230f2, 0x007234f2, 0x007238f2, 0x00723cf2, // 23688-23695
	0x007240f2, 0x007244f2, 0x007248f2, 0x00724cf2, 0x007250f2, 0x007254f2, 0x007258f2, 0x00725cf2, // 23696-23703
	0x007260f2, 0x007264f2, 0x007268f2, 0x00726cf2, 0x007270f2, 0x007274f2, 0x007278f2, 0x00727cf2, // 23704-23711
	0x007280f2, 0x007284f2, 0x007288f2, 0x00728cf2, 0x007290f2, 0x007294f2, 0x007298f2, 0x00729cf2, // 23712-23719
	0x0072a0f2, 0x0072a4f2, 0x0072a8f2, 0x0072acf2, 0x0072b0f2, 0x0072b4f2, 0x0072b8f2, 0x0072bcf2, // 23720-23727
	0x0072c0f2, 0x0072c4f2, 0x0072c8f2, 0x0072ccf2, 0x0072d0f2, 0x0072d4f2, 0x0072d8f2, 0x0072dcf2, // 23728-23735
	0x0072e0f2, 0x0072e4f2, 0x0072e8f2, 0x0072ecf2, 0x0072f0f2, 0x0072f4f2, 0x0072f8f2, 0x0072fcf2, // 23736-23743
	0x007300f2, 0x007304f2, 0x007308f2, 0x00730cf2, 0x007310f2, 0x007314f2, 0x007318f2, 0x00731cf2, // 23744-23751
	0x007320f2, 0x007324f2, 0x007328f2, 0x00732cf2, 0x007330f2, 0x007334f2, 0x007338f2, 0x00733cf2, // 23752-23759
	0x007340f2, 0x007344f2, 0x007348f2, 0x00734cf2, 0x007350f2, 0x007354f2, 0x007358f2, 0x00735cf2, // 23760-23767
	0x007360f2, 0x007364f2, 0x007368f2, 0x00736cf2, 0x007370f2, 0x007374f2, 0x007378f2, 0x00737cf2, // 23768-23775
	0x007380f2, 0x007384f2, 0x007388f2, 0x00738cf2, 0x007390f2, 0x007394f2, 0x007398f2, 0x00739cf2, // 23776-23783
	0x0073a0f2, 0x0073a4f2, 0x0073a8f2, 0x0073acf2, 0x0073b0f2, 0x0073b4f2, 0x0073b8f2, 0x0073bcf2, // 23784-23791
	0x0073c0f2, 0x0073c4f2, 0x0073c8f2, 0x0073ccf2, 0x0073d0f2, 0x0073d4f2, 0x0073d8f2, 0x0073dcf2, // 23792-23799
	0x0073e0f2, 0x0073e4f2, 0x0073e8f2, 0x0073ecf2, 0x0073f0f2, 0x0073f4f2, 0x0073f8f2, 0x0073fcf2, // 23800-23807
	0x007400f2, 0x007404f2, 0x007408f2, 0x00740cf2, 0x007410f2, 0x007414f2, 0x007418f2, 0x00741cf2, // 23808-23815
	0x007420f2, 0x007424f2, 0x007428f2, 0x00742cf2, 0x007430f2, 0x007434f2, 0x007438f2, 0x00743cf2, // 23816-23823
	0x007440f2, 0x007444f2, 0x007448f2, 0x00744cf2, 0x007450f2, 0x007454f2, 0x007458f2, 0x00745cf2, // 23824-23831
	0x007460f2, 0x007464f2, 0x007468f2, 0x00746cf2, 0x007470f2, 0x007474f2, 0x007478f2, 0x00747cf2, // 23832-23839
	0x007480f2, 0x007484f2, 0x007488f2, 0x00748cf2, 0x007490f2, 0x007494f2, 0x007498f2, 0x00749cf2, // 23840-23847
	0x0074a0f2, 0x0074a4f2, 0x0074a8f2, 0x0074acf2, 0x0074b0f2, 0x0074b4f2, 0x0074b8f2, 0x0074bcf2, // 23848-23855
	0x0074c0f2, 0x0074c4f2, 0x0074c8f2, 0x0074ccf2, 0x0074d0f2, 0x0074d4f2, 0x0074d8f2, 0x0074dcf2, // 23856-23863
	0x0074e0f2, 0x0074e4f2, 0x0074e8f2, 0x0074ecf2, 0x0074f0f2, 0x0074f4f2, 0x0074f8f2, 0x0074fcf2, // 23864-23871
	0x007500f2, 0x007504f2, 0x007508f2, 0x00750cf2, 0x007510f2, 0x007514f2, 0x007518f2, 0x00751cf2, // 23872-23879
	0x007520f2, 0x007524f2, 0x007528f2, 0x00752cf2, 0x007530f2, 0x007534f2, 0x007538f2, 0x00753cf2, // 23880-23887
	0x007540f2, 0x007544f2, 0x007548f2, 0x00754cf2, 0x007550f2, 0x007554f2, 0x007558f2, 0x00755cf2, // 23888-23895
	0x007560f2, 0x007564f2, 0x007568f2, 0x00756cf2, 0x007570f2, 0x007574f2, 0x007578f2, 0x00757cf2, // 23896-23903
	0x007580f2, 0x007584f2, 0x007588f2, 0x00758cf2, 0x007590f2, 0x007594f2, 0x007598f2, 0x00759cf2, // 23904-23911
	0x0075a0f2, 0x0075a4f2, 0x0075a8f2, 0x0075acf2, 0x0075b0f2, 0x0075b4f2, 0x0075b8f2, 0x0075bcf2, // 23912-23919
	0x0075c0f2, 0x0075c4f2, 0x0075c8f2, 0x0075ccf2, 0x0075d0f2, 0x0075d4f2, 0x0075d8f2, 0x0075dcf2, // 23920-23927
	0x0075e0f2, 0x0075e4f2, 0x0075e8f2, 0x0075ecf2, 0x0075f0f2, 0x0075f4f2, 0x0075f8f2, 0x0075fcf2, // 23928-23935
	0x007600f2, 0x007604f2, 0x007608f2, 0x00760cf2, 0x007610f2, 0x007614f2, 0x007618f2, 0x00761cf2, // 23936-23943
	0x007620f2, 0x007624f2, 0x007628f2, 0x00762cf2, 0x007630f2, 0x007634f2, 0x007638f2, 0x00763cf2, // 23944-23951
	0x007640f2, 0x007644f2, 0x007648f2, 0x00764cf2, 0x007650f2, 0x007654f2, 0x007658f2, 0x00765cf2, // 23952-23959
	0x007660f2, 0x007664f2, 0x007668f2, 0x00766cf2, 0x007670f2, 0x007674f2, 0x007678f2, 0x00767cf2, // 23960-23967
	0x007680f2, 0x007684f2, 0x007688f2, 0x00768cf2, 0x007690f2, 0x007694f2, 0x007698f2, 0x00769cf2, // 23968-23975
	0x0076a0f2, 0x0076a4f2, 0x0076a8f2, 0x0076acf2, 0x0076b0f2, 0x0076b4f2, 0x0076b8f2, 0x0076bcf2, // 23976-23983
	0x0076c0f2, 0x0076c4f2, 0x0076c8f2, 0x0076ccf2, 0x0076d0f2, 0x0076d4f2, 0x0076d8f2, 0x0076dcf2, // 23984-23991
	0x0076e0f2, 0x0076e4f2, 0x0076e8f2, 0x0076ecf2, 0x0076f0f2, 0x0076f4f2, 0x0076f8f2, 0x0076fcf2, // 23992-23999
	0x007700f2, 0x007704f2, 0x007708f2, 0x00770cf2, 0x007710f2, 0x007714f2, 0x007718f2, 0x00771cf2, // 24000-24007
	0x007720f2, 0x007724f2, 0x007728f2, 0x00772cf2, 0x007730f2, 0x007734f2, 0x007738f2, 0x00773cf2, // 24008-24015
	0x007740f2, 0x007744f2, 0x007748f2, 0x00774cf2, 0x007750f2, 0x007754f2, 0x007758f2, 0x00775cf2, // 24016-24023
	0x007760f2, 0x007764f2, 0x007768f2, 0x00776cf2, 0x007770f2, 0x007774f2, 0x007778f2, 0x00777cf2, // 24024-24031
	0x007780f2, 0x007784f2, 0x007788f2, 0x00778cf2, 0x007790f2, 0x007794f2, 0x007798f2, 0x00779cf2, // 24032-24039
	0x0077a0f2, 0x0077a4f2, 0x0077a8f2, 0x0077acf2, 0x0077b0f2, 0x0077b4f2, 0x0077b8f2, 0x0077bcf2, // 24040-24047
	0x0077c0f2, 0x0077c4f2, 0x0077c8f2, 0x0077ccf2, 0x0077d0f2, 0x0077d4f2, 0x0077d8f2, 0x0077dcf2, // 24048-24055
	0x0077e0f2, 0x0077e4f2, 0x0077e8f2, 0x0077ecf2, 0x0077f0f2, 0x0077f4f2, 0x0077f8f2, 0x0077fcf2, // 24056-24063
	0x007800f2, 0x007804f2, 0x007808f2, 0x00780cf2, 0x007810f2, 0x007814f2, 0x007818f2, 0x00781cf2, // 24064-24071
	0x007820f2, 0x007824f2, 0x007828f2, 0x00782cf2, 0x007830f2, 0x007834f2, 0x007838f2, 0x00783cf2, // 24072-24079
	0x007840f2, 0x007844f2, 0x007848f2, 0x00784cf2, 0x007850f2, 0x007854f2, 0x007858f2, 0x00785cf2, // 24080-24087
	0x007860f2, 0x007864f2, 0x007868f2, 0x00786cf2, 0x007870f2, 0x007874f2, 0x007878f2, 0x00787cf2, // 24088-24095
	0x007880f2, 0x007884f2, 0x007888f2, 0x00788cf2, 0x007890f2, 0x007894f2, 0x007898f2, 0x00789cf2, // 24096-24103
	0x0078a0f2, 0x0078a4f2, 0x0078a8f2, 0x0078acf2, 0x0078b0f2, 0x0078b4f2, 0x0078b8f2, 0x0078bcf2, // 24104-24111
	0x0078c0f2, 0x0078c4f2, 0x0078c8f2, 0x0078ccf2, 0x0078d0f2, 0x0078d4f2, 0x0078d8f2, 0x0078dcf2, // 24112-24119
	0x0078e0f2, 0x0078e4f2, 0x0078e8f2, 0x0078ecf2, 0x0078f0f2, 0x0078f4f2, 0x0078f8f2, 0x0078fcf2, // 24120-24127
	0x007900f2, 0x007904f2, 0x007908f2, 0x00790cf2, 0x007910f2, 0x007914f2, 0x007918f2, 0x00791cf2, // 24128-24135
	0x007920f2, 0x007924f2, 0x007928f2, 0x00792cf2, 0x007930f2, 0x007934f2, 0x007938f2, 0x00793cf2, // 24136-24143
	0x007940f2, 0x007944f2, 0x007948f2, 0x00794cf2, 0x007950f2, 0x007954f2, 0x007958f2, 0x00795cf2, // 24144-24151
	0x007960f2, 0x007964f2, 0x007968f2, 0x00796cf2, 0x007970f2, 0x007974f2, 0x007978f2, 0x00797cf2, // 24152-24159
	0x007980f2, 0x007984f2, 0x007988f2, 0x00798cf2, 0x007990f2, 0x007994f2, 0x007998f2, 0x00799cf2, // 24160-24167
	0x0079a0f2, 0x0079a4f2, 0x0079a8f2, 0x0079acf2, 0x0079b0f2, 0x0079b4f2, 0x0079b8f2, 0x0079bcf2, // 24168-24175
	0x0079c0f2, 0x0079c4f2, 0x0079c8f2, 0x0079ccf2, 0x0079d0f2, 0x0079d4f2, 0x0079d8f2, 0x0079dcf2, // 24176-24183
	0x0079e0f2, 0x0079e4f2, 0x0079e8f2, 0x0079ecf2, 0x0079f0f2, 0x0079f4f2, 0x0079f8f2, 0x0079fcf2, // 24184-24191
	0x007a00f2, 0x007a04f2, 0x007a08f2, 0x007a0cf2, 0x007a10f2, 0x007a14f2, 0x007a18f2, 0x007a1cf2, // 24192-24199
	0x007a20f2, 0x007a24f2, 0x007a28f2, 0x007a2cf2, 0x007a30f2, 0x007a34f2, 0x007a38f2, 0x007a3cf2, // 24200-24207
	0x007a40f2, 0x007a44f2, 0x007a48f2, 0x007a4cf2, 0x007a50f2, 0x007a54f2, 0x007a58f2, 0x007a5cf2, // 24208-24215
	0x007a60f2, 0x007a64f2, 0x007a68f2, 0x007a6cf2, 0x007a70f2, 0x007a74f2, 0x007a78f2, 0x007a7cf2, // 24216-24223
	0x007a80f2, 0x007a84f2, 0x007a88f2, 0x007a8cf2, 0x007a90f2, 0x007a94f2, 0x007a98f2, 0x007a9cf2, // 24224-24231
	0x007aa0f2, 0x007aa4f2, 0x007aa8f2, 0x007aacf2, 0x007ab0f2, 0x007ab4f2, 0x007ab8f2, 0x007abcf2, // 24232-24239
	0x007ac0f2, 0x007ac4f2, 0x007ac8f2, 0x007accf2, 0x007ad0f2, 0x007ad4f2, 0x007ad8f2, 0x007adcf2, // 24240-24247
	0x007ae0f2, 0x007ae4f2, 0x007ae8f2, 0x007aecf2, 0x007af0f2, 0x007af4f2, 0x007af8f2, 0x007afcf2, // 24248-24255
	0x007b00f2, 0x007b04f2, 0x007b08f2, 0x007b0cf2, 0x007b10f2, 0x007b14f2, 0x007b18f2, 0x007b1cf2, // 24256-24263
	0x007b20f2, 0x007b24f2, 0x007b28f2, 0x007b2cf2, 0x007b30f2, 0x007b34f2, 0x007b38f2, 0x007b3cf2, // 24264-24271
	0x007b40f2, 0x007b44f2, 0x007b48f2, 0x007b4cf2, 0x007b50f2, 0x007b54f2, 0x007b58f2, 0x007b5cf2, // 24272-24279
	0x007b60f2, 0x007b64f2, 0x007b68f2, 0x007b6cf2, 0x007b70f2, 0x007b74f2, 0x007b78f2, 0x007b7cf2, // 24280-24287
	0x007b80f2, 0x007b84f2, 0x007b88f2, 0x007b8cf2, 0x007b90f2, 0x007b94f2, 0x007b98f2, 0x007b9cf2, // 24288-24295
	0x007ba0f2, 0x007ba4f2, 0x007ba8f2, 0x007bacf2, 0x007bb0f2, 0x007bb4f2, 0x007bb8f2, 0x007bbcf2, // 24296-24303
	0x007bc0f2, 0x007bc4f2, 0x007bc8f2, 0x007bccf2, 0x007bd0f2, 0x007bd4f2, 0x007bd8f2, 0x007bdcf2, // 24304-24311
	0x007be0f2, 0x007be4f2, 0x007be8f2, 0x007becf2, 0x007bf0f2, 0x007bf4f2, 0x007bf8f2, 0x007bfcf2, // 24312-24319
	0x007c00f2, 0x007c04f2, 0x007c08f2, 0x007c0cf2, 0x007c10f2, 0x007c14f2, 0x007c18f2, 0x007c1cf2, // 24320-24327
	0x007c20f2, 0x007c24f2, 0x007c28f2, 0x007c2cf2, 0x007c30f2, 0x007c34f2, 0x007c38f2, 0x007c3cf2, // 24328-24335
	0x007c40f2, 0x007c44f2, 0x007c48f2, 0x007c4cf2, 0x007c50f2, 0x007c54f2, 0x007c58f2, 0x007c5cf2, // 24336-24343
	0x007c60f2, 0x007c64f2, 0x007c68f2, 0x007c6cf2, 0x007c70f2, 0x007c74f2, 0x007c78f2, 0x007c7cf2, // 24344-24351
	0x007c80f2, 0x007c84f2, 0x007c88f2, 0x007c8cf2, 0x007c90f2, 0x007c94f2, 0x007c98f2, 0x007c9cf2, // 24352-24359
	0x007ca0f2, 0x007ca4f2, 0x007ca8f2, 0x007cacf2, 0x007cb0f2, 0x007cb4f2, 0x007cb8f2, 0x007cbcf2, // 24360-24367
	0x007cc0f2, 0x007cc4f2, 0x007cc8f2, 0x007cccf2, 0x007cd0f2, 0x007cd4f2, 0x007cd8f2, 0x007cdcf2, // 24368-24375
	0x007ce0f2, 0x007ce4f2, 0x007ce8f2, 0x007cecf2, 0x007cf0f2, 0x007cf4f2, 0x007cf8f2, 0x007cfcf2, // 24376-24383
	0x007d00f2, 0x007d04f2, 0x007d08f2, 0x007d0cf2, 0x007d10f2, 0x007d14f2, 0x007d18f2, 0x007d1cf2, // 24384-24391
	0x007d20f2, 0x007d24f2, 0x007d28f2, 0x007d2cf2, 0x007d30f2, 0x007d34f2, 0x007d38f2, 0x007d3cf2, // 24392-24399
	0x007d40f2, 0x007d44f2, 0x007d48f2, 0x007d4cf2, 0x007d50f2, 0x007d54f2, 0x007d58f2, 0x007d5cf2, // 24400-24407
	0x007d60f2, 0x007d64f2, 0x007d68f2, 0x007d6cf2, 0x007d70f2, 0x007d74f2, 0x007d78f2, 0x007d7cf2, // 24408-24415
	0x007d80f2, 0x007d84f2, 0x007d88f2, 0x007d8cf2, 0x007d90f2, 0x007d94f2, 0x007d98f2, 0x007d9cf2, // 24416-24423
	0x007da0f2, 0x007da4f2, 0x007da8f2, 0x007dacf2, 0x007db0f2, 0x007db4f2, 0x007db8f2, 0x007dbcf2, // 24424-24431
	0x007dc0f2, 0x007dc4f2, 0x007dc8f2, 0x007dccf2, 0x007dd0f2, 0x007dd4f2, 0x007dd8f2, 0x007ddcf2, // 24432-24439
	0x007de0f2, 0x007de4f2, 0x007de8f2, 0x007decf2, 0x007df0f2, 0x007df4f2, 0x007df8f2, 0x007dfcf2, // 24440-24447
	0x007e00f2, 0x007e04f2, 0x007e08f2, 0x007e0cf2, 0x007e10f2, 0x007e14f2, 0x007e18f2, 0x007e1cf2, // 24448-24455
	0x007e20f2, 0x007e24f2, 0x007e28f2, 0x007e2cf2, 0x007e30f2, 0x007e34f2, 0x007e38f2, 0x007e3cf2, // 24456-24463
	0x007e40f2, 0x007e44f2, 0x007e48f2, 0x007e4cf2, 0x007e50f2, 0x007e54f2, 0x007e58f2, 0x007e5cf2, // 24464-24471
	0x007e60f2, 0x007e64f2, 0x007e68f2, 0x007e6cf2, 0x007e70f2, 0x007e74f2, 0x007e78f2, 0x007e7cf2, // 24472-24479
	0x007e80f2, 0x007e84f2, 0x007e88f2, 0x007e8cf2, 0x007e90f2, 0x007e94f2, 0x007e98f2, 0x007e9cf2, // 24480-24487
	0x007ea0f2, 0x007ea4f2, 0x007ea8f2, 0x007eacf2, 0x007eb0f2, 0x007eb4f2, 0x007eb8f2, 0x007ebcf2, // 24488-24495
	0x007ec0f2, 0x007ec4f2, 0x007ec8f2, 0x007eccf2, 0x007ed0f2, 0x007ed4f2, 0x007ed8f2, 0x007edcf2, // 24496-24503
	0x007ee0f2, 0x007ee4f2, 0x007ee8f2, 0x007eecf2, 0x007ef0f2, 0x007ef4f2, 0x007ef8f2, 0x007efcf2, // 24504-24511
	0x007f00f2, 0x007f04f2, 0x007f08f2, 0x007f0cf2, 0x007f10f2, 0x007f14f2, 0x007f18f2, 0x007f1cf2, // 24512-24519
	0x007f20f2, 0x007f24f2, 0x007f28f2, 0x007f2cf2, 0x007f30f2, 0x007f34f2, 0x007f38f2, 0x007f3cf2, // 24520-24527
	0x007f40f2, 0x007f44f2, 0x007f48f2, 0x007f4cf2, 0x007f50f2, 0x007f54f2, 0x007f58f2, 0x007f5cf2, // 24528-24535
	0x007f60f2, 0x007f64f2, 0x007f68f2, 0x007f6cf2, 0x007f70f2, 0x007f74f2, 0x007f78f2, 0x007f7cf2, // 24536-24543
	0x007f80f2, 0x007f84f2, 0x007f88f2, 0x007f8cf2, 0x007f90f2, 0x007f94f2, 0x007f98f2, 0x007f9cf2, // 24544-24551
	0x007fa0f2, 0x007fa4f2, 0x007fa8f2, 0x007facf2, 0x007fb0f2, 0x007fb4f2, 0x007fb8f2, 0x007fbcf2, // 24552-24559
	0x007fc0f2, 0x007fc4f2, 0x007fc8f2, 0x007fccf2, 0x007fd0f2, 0x007fd4f2, 0x007fd8f2, 0x007fdcf2, // 24560-24567
	0x007fe0f2, 0x007fe4f2, 0x007fe8f2, 0x007fecf2, 0x007ff0f2, 0x007ff4f2, 0x007ff8f2, 0x007ffcf2, // 24568-24575
	0x000002f2, 0x000006f2, 0x00000af2, 0x00000ef2, 0x000012f2, 0x000016f2, 0x00001af2, 0x00001ef2, // 24576-24583
	0x000022f2, 0x000026f2, 0x00002af2, 0x00002ef2, 0x000032f2, 0x000036f2, 0x00003af2, 0x00003ef2, // 24584-24591
	0x000042f2, 0x000046f2, 0x00004af2, 0x00004ef2, 0x000052f2, 0x000056f2, 0x00005af2, 0x00005ef2, // 24592-24599
	0x000062f2, 0x000066f2, 0x00006af2, 0x00006ef2, 0x000072f2, 0x000076f2, 0x00007af2, 0x00007ef2, // 24600-24607
	0x000082f2, 0x000086f2, 0x00008af2, 0x00008ef2, 0x000092f2, 0x000096f2, 0x00009af2, 0x00009ef2, // 24608-24615
	0x0000a2f2, 0x0000a6f2, 0x0000aaf2, 0x0000aef2, 0x0000b2f2, 0x0000b6f2, 0x0000baf2, 0x0000bef2, // 24616-24623
	0x0000c2f2, 0x0000c6f2, 0x0000caf2, 0x0000cef2, 0x0000d2f2, 0x0000d6f2, 0x0000daf2, 0x0000def2, // 24624-24631
	0x0000e2f2, 0x0000e6f2, 0x0000eaf2, 0x0000eef2, 0x0000f2f2, 0x0000f6f2, 0x0000faf2, 0x0000fef2, // 24632-24639
	0x000102f2, 0x000106f2, 0x00010af2, 0x00010ef2, 0x000112f2, 0x000116f2, 0x00011af2, 0x00011ef2, // 24640-24647
	0x000122f2, 0x000126f2, 0x00012af2, 0x00012ef2, 0x000132f2, 0x000136f2, 0x00013af2, 0x00013ef2, // 24648-24655
	0x000142f2, 0x000146f2, 0x00014af2, 0x00014ef2, 0x000152f2, 0x000156f2, 0x00015af2, 0x00015ef2, // 24656-24663
	0x000162f2, 0x000166f2, 0x00016af2, 0x00016ef2, 0x000172f2, 0x000176f2, 0x00017af2, 0x00017ef2, // 24664-24671
	0x000182f2, 0x000186f2, 0x00018af2, 0x00018ef2, 0x000192f2, 0x000196f2, 0x00019af2, 0x00019ef2, // 24672-24679
	0x0001a2f2, 0x0001a6f2, 0x0001aaf2, 0x0001aef2, 0x0001b2f2, 0x0001b6f2, 0x0001baf2, 0x0001bef2, // 24680-24687
	0x0001c2f2, 0x0001c6f2, 0x0001caf2, 0x0001cef2, 0x0001d2f2, 0x0001d6f2, 0x0001daf2, 0x0001def2, // 24688-24695
	0x0001e2f2, 0x0001e6f2, 0x0001eaf2, 0x0001eef2, 0x0001f2f2, 0x0001f6f2, 0x0001faf2, 0x0001fef2, // 24696-24703
	0x000202f2, 0x000206f2, 0x00020af2, 0x00020ef2, 0x000212f2, 0x000216f2, 0x00021af2, 0x00021ef2, // 24704-24711
	0x000222f2, 0x000226f2, 0x00022af2, 0x00022ef2, 0x000232f2, 0x000236f2, 0x00023af2, 0x00023ef2, // 24712-24719
	0x000242f2, 0x000246f2, 0x00024af2, 0x00024ef2, 0x000252f2, 0x000256f2, 0x00025af2, 0x00025ef2, // 24720-24727
	0x000262f2, 0x000266f2, 0x00026af2, 0x00026ef2, 0x000272f2, 0x000276f2, 0x00027af2, 0x00027ef2, // 24728-24735
	0x000282f2, 0x000286f2, 0x00028af2, 0x00028ef2, 0x000292f2, 0x000296f2, 0x00029af2, 0x00029ef2, // 24736-24743
	0x0002a2f2, 0x0002a6f2, 0x0002aaf2, 0x0002aef2, 0x0002b2f2, 0x0002b6f2, 0x0002baf2, 0x0002bef2, // 24744-24751
	0x0002c2f2, 0x0002c6f2, 0x0002caf2, 0x0002cef2, 0x0002d2f2, 0x0002d6f2, 0x0002daf2, 0x0002def2, // 24752-24759
	0x0002e2f2, 0x0002e6f2, 0x0002eaf2, 0x0002eef2, 0x0002f2f2, 0x0002f6f2, 0x0002faf2, 0x0002fef2, // 24760-24767
	0x000302f2, 0x000306f2, 0x00030af2, 0x00030ef2, 0x000312f2, 0x000316f2, 0x00031af2, 0x00031ef2, // 24768-24775
	0x000322f2, 0x000326f2, 0x00032af2, 0x00032ef2, 0x000332f2, 0x000336f2, 0x00033af2, 0x00033ef2, // 24776-24783
	0x000342f2, 0x000346f2, 0x00034af2, 0x00034ef2, 0x000352f2, 0x000356f2, 0x00035af2, 0x00035ef2, // 24784-24791
	0x000362f2, 0x000366f2, 0x00036af2, 0x00036ef2, 0x000372f2, 0x000376f2, 0x00037af2, 0x00037ef2, // 24792-24799
	0x000382f2, 0x000386f2, 0x00038af2, 0x00038ef2, 0x000392f2, 0x000396f2, 0x00039af2, 0x00039ef2, // 24800-24807
	0x0003a2f2, 0x0003a6f2, 0x0003aaf2, 0x0003aef2, 0x0003b2f2, 0x0003b6f2, 0x0003baf2, 0x0003bef2, // 24808-24815
	0x0003c2f2, 0x0003c6f2, 0x0003caf2, 0x0003cef2, 0x0003d2f2, 0x0003d6f2, 0x0003daf2, 0x0003def2, // 24816-24823
	0x0003e2f2, 0x0003e6f2, 0x0003eaf2, 0x0003eef2, 0x0003f2f2, 0x0003f6f2, 0x0003faf2, 0x0003fef2, // 24824-24831
	0x000402f2, 0x000406f2, 0x00040af2, 0x00040ef2, 0x000412f2, 0x000416f2, 0x00041af2, 0x00041ef2, // 24832-24839
	0x000422f2, 0x000426f2, 0x00042af2, 0x00042ef2, 0x000432f2, 0x000436f2, 0x00043af2, 0x00043ef2, // 24840-24847
	0x000442f2, 0x000446f2, 0x00044af2, 0x00044ef2, 0x000452f2, 0x000456f2, 0x00045af2, 0x00045ef2, // 24848-24855
	0x000462f2, 0x000466f2, 0x00046af2, 0x00046ef2, 0x000472f2, 0x000476f2, 0x00047af2, 0x00047ef2, // 24856-24863
	0x000482f2, 0x000486f2, 0x00048af2, 0x00048ef2, 0x000492f2, 0x000496f2, 0x00049af2, 0x00049ef2, // 24864-24871
	0x0004a2f2, 0x0004a6f2, 0x0004aaf2, 0x0004aef2, 0x0004b2f2, 0x0004b6f2, 0x0004baf2, 0x0004bef2, // 24872-24879
	0x0004c2f2, 0x0004c6f2, 0x0004caf2, 0x0004cef2, 0x0004d2f2, 0x0004d6f2, 0x0004daf2, 0x0004def2, // 24880-24887
	0x0004e2f2, 0x0004e6f2, 0x0004eaf2, 0x0004eef2, 0x0004f2f2, 0x0004f6f2, 0x0004faf2, 0x0004fef2, // 24888-24895
	0x000502f2, 0x000506f2, 0x00050af2, 0x00050ef2, 0x000512f2, 0x000516f2, 0x00051af2, 0x00051ef2, // 24896-24903
	0x000522f2, 0x000526f2, 0x00052af2, 0x00052ef2, 0x000532f2, 0x000536f2, 0x00053af2, 0x00053ef2, // 24904-24911
	0x000542f2, 0x000546f2, 0x00054af2, 0x00054ef2, 0x000552f2, 0x000556f2, 0x00055af2, 0x00055ef2, // 24912-24919
	0x000562f2, 0x000566f2, 0x00056af2, 0x00056ef2, 0x000572f2, 0x000576f2, 0x00057af2, 0x00057ef2, // 24920-24927
	0x000582f2, 0x000586f2, 0x00058af2, 0x00058ef2, 0x000592f2, 0x000596f2, 0x00059af2, 0x00059ef2, // 24928-24935
	0x0005a2f2, 0x0005a6f2, 0x0005aaf2, 0x0005aef2, 0x0005b2f2, 0x0005b6f2, 0x0005baf2, 0x0005bef2, // 24936-24943
	0x0005c2f2, 0x0005c6f2, 0x0005caf2, 0x0005cef2, 0x0005d2f2, 0x0005d6f2, 0x0005daf2, 0x0005def2, // 24944-24951
	0x0005e2f2, 0x0005e6f2, 0x0005eaf2, 0x0005eef2, 0x0005f2f2, 0x0005f6f2, 0x0005faf2, 0x0005fef2, // 24952-24959
	0x000602f2, 0x000606f2, 0x00060af2, 0x00060ef2, 0x000612f2, 0x000616f2, 0x00061af2, 0x00061ef2, // 24960-24967
	0x000622f2, 0x000626f2, 0x00062af2, 0x00062ef2, 0x000632f2, 0x000636f2, 0x00063af2, 0x00063ef2, // 24968-24975
	0x000642f2, 0x000646f2, 0x00064af2, 0x00064ef2, 0x000652f2, 0x000656f2, 0x00065af2, 0x00065ef2, // 24976-24983
	0x000662f2, 0x000666f2, 0x00066af2, 0x00066ef2, 0x000672f2, 0x000676f2, 0x00067af2, 0x00067ef2, // 24984-24991
	0x000682f2, 0x000686f2, 0x00068af2, 0x00068ef2, 0x000692f2, 0x000696f2, 0x00069af2, 0x00069ef2, // 24992-24999
	0x0006a2f2, 0x0006a6f2, 0x0006aaf2, 0x0006aef2, 0x0006b2f2, 0x0006b6f2, 0x0006baf2, 0x0006bef2, // 25000-25007
	0x0006c2f2, 0x0006c6f2, 0x0006caf2, 0x0006cef2, 0x0006d2f2, 0x0006d6f2, 0x0006daf2, 0x0006def2, // 25008-25015
	0x0006e2f2, 0x0006e6f2, 0x0006eaf2, 0x0006eef2, 0x0006f2f2, 0x0006f6f2, 0x0006faf2, 0x0006fef2, // 25016-25023
	0x000702f2, 0x000706f2, 0x00070af2, 0x00070ef2, 0x000712f2, 0x000716f2, 0x00071af2, 0x00071ef2, // 25024-25031
	0x000722f2, 0x000726f2, 0x00072af2, 0x00072ef2, 0x000732f2, 0x000736f2, 0x00073af2, 0x00073ef2, // 25032-25039
	0x000742f2, 0x000746f2, 0x00074af2, 0x00074ef2, 0x000752f2, 0x000756f2, 0x00075af2, 0x00075ef2, // 25040-25047
	0x000762f2, 0x000766f2, 0x00076af2, 0x00076ef2, 0x000772f2, 0x000776f2, 0x00077af2, 0x00077ef2, // 25048-25055
	0x000782f2, 0x000786f2, 0x00078af2, 0x00078ef2, 0x000792f2, 0x000796f2, 0x00079af2, 0x00079ef2, // 25056-25063
	0x0007a2f2, 0x0007a6f2, 0x0007aaf2, 0x0007aef2, 0x0007b2f2, 0x0007b6f2, 0x0007baf2, 0x0007bef2, // 25064-25071
	0x0007c2f2, 0x0007c6f2, 0x0007caf2, 0x0007cef2, 0x0007d2f2, 0x0007d6f2, 0x0007daf2, 0x0007def2, // 25072-25079
	0x0007e2f2, 0x0007e6f2, 0x0007eaf2, 0x0007eef2, 0x0007f2f2, 0x0007f6f2, 0x0007faf2, 0x0007fef2, // 25080-25087
	0x000802f2, 0x000806f2, 0x00080af2, 0x00080ef2, 0x000812f2, 0x000816f2, 0x00081af2, 0x00081ef2, // 25088-25095
	0x000822f2, 0x000826f2, 0x00082af2, 0x00082ef2, 0x000832f2, 0x000836f2, 0x00083af2, 0x00083ef2, // 25096-25103
	0x000842f2, 0x000846f2, 0x00084af2, 0x00084ef2, 0x000852f2, 0x000856f2, 0x00085af2, 0x00085ef2, // 25104-25111
	0x000862f2, 0x000866f2, 0x00086af2, 0x00086ef2, 0x000872f2, 0x000876f2, 0x00087af2, 0x00087ef2, // 25112-25119
	0x000882f2, 0x000886f2, 0x00088af2, 0x00088ef2, 0x000892f2, 0x000896f2, 0x00089af2, 0x00089ef2, // 25120-25127
	0x0008a2f2, 0x0008a6f2, 0x0008aaf2, 0x0008aef2, 0x0008b2f2, 0x0008b6f2, 0x0008baf2, 0x0008bef2, // 25128-25135
	0x0008c2f2, 0x0008c6f2, 0x0008caf2, 0x0008cef2, 0x0008d2f2, 0x0008d6f2, 0x0008daf2, 0x0008def2, // 25136-25143
	0x0008e2f2, 0x0008e6f2, 0x0008eaf2, 0x0008eef2, 0x0008f2f2, 0x0008f6f2, 0x0008faf2, 0x0008fef2, // 25144-25151
	0x000902f2, 0x000906f2, 0x00090af2, 0x00090ef2, 0x000912f2, 0x000916f2, 0x00091af2, 0x00091ef2, // 25152-25159
	0x000922f2, 0x000926f2, 0x00092af2, 0x00092ef2, 0x000932f2, 0x000936f2, 0x00093af2, 0x00093ef2, // 25160-25167
	0x000942f2, 0x000946f2, 0x00094af2, 0x00094ef2, 0x000952f2, 0x000956f2, 0x00095af2, 0x00095ef2, // 25168-25175
	0x000962f2, 0x000966f2, 0x00096af2, 0x00096ef2, 0x000972f2, 0x000976f2, 0x00097af2, 0x00097ef2, // 25176-25183
	0x000982f2, 0x000986f2, 0x00098af2, 0x00098ef2, 0x000992f2, 0x000996f2, 0x00099af2, 0x00099ef2, // 25184-25191
	0x0009a2f2, 0x0009a6f2, 0x0009aaf2, 0x0009aef2, 0x0009b2f2, 0x0009b6f2, 0x0009baf2, 0x0009bef2, // 25192-25199
	0x0009c2f2, 0x0009c6f2, 0x0009caf2, 0x0009cef2, 0x0009d2f2, 0x0009d6f2, 0x0009daf2, 0x0009def2, // 25200-25207
	0x0009e2f2, 0x0009e6f2, 0x0009eaf2, 0x0009eef2, 0x0009f2f2, 0x0009f6f2, 0x0009faf2, 0x0009fef2, // 25208-25215
	0x000a02f2, 0x000a06f2, 0x000a0af2, 0x000a0ef2, 0x000a12f2, 0x000a16f2, 0x000a1af2, 0x000a1ef2, // 25216-25223
	0x000a22f2, 0x000a26f2, 0x000a2af2, 0x000a2ef2, 0x000a32f2, 0x000a36f2, 0x000a3af2, 0x000a3ef2, // 25224-25231
	0x000a42f2, 0x000a46f2, 0x000a4af2, 0x000a4ef2, 0x000a52f2, 0x000a56f2, 0x000a5af2, 0x000a5ef2, // 25232-25239
	0x000a62f2, 0x000a66f2, 0x000a6af2, 0x000a6ef2, 0x000a72f2, 0x000a76f2, 0x000a7af2, 0x000a7ef2, // 25240-25247
	0x000a82f2, 0x000a86f2, 0x000a8af2, 0x000a8ef2, 0x000a92f2, 0x000a96f2, 0x000a9af2, 0x000a9ef2, // 25248-25255
	0x000aa2f2, 0x000aa6f2, 0x000aaaf2, 0x000aaef2, 0x000ab2f2, 0x000ab6f2, 0x000abaf2, 0x000abef2, // 25256-25263
	0x000ac2f2, 0x000ac6f2, 0x000acaf2, 0x000acef2, 0x000ad2f2, 0x000ad6f2, 0x000adaf2, 0x000adef2, // 25264-25271
	0x000ae2f2, 0x000ae6f2, 0x000aeaf2, 0x000aeef2, 0x000af2f2, 0x000af6f2, 0x000afaf2, 0x000afef2, // 25272-25279
	0x000b02f2, 0x000b06f2, 0x000b0af2, 0x000b0ef2, 0x000b12f2, 0x000b16f2, 0x000b1af2, 0x000b1ef2, // 25280-25287
	0x000b22f2, 0x000b26f2, 0x000b2af2, 0x000b2ef2, 0x000b32f2, 0x000b36f2, 0x000b3af2, 0x000b3ef2, // 25288-25295
	0x000b42f2, 0x000b46f2, 0x000b4af2, 0x000b4ef2, 0x000b52f2, 0x000b56f2, 0x000b5af2, 0x000b5ef2, // 25296-25303
	0x000b62f2, 0x000b66f2, 0x000b6af2, 0x000b6ef2, 0x000b72f2, 0x000b76f2, 0x000b7af2, 0x000b7ef2, // 25304-25311
	0x000b82f2, 0x000b86f2, 0x000b8af2, 0x000b8ef2, 0x000b92f2, 0x000b96f2, 0x000b9af2, 0x000b9ef2, // 25312-25319
	0x000ba2f2, 0x000ba6f2, 0x000baaf2, 0x000baef2, 0x000bb2f2, 0x000bb6f2, 0x000bbaf2, 0x000bbef2, // 25320-25327
	0x000bc2f2, 0x000bc6f2, 0x000bcaf2, 0x000bcef2, 0x000bd2f2, 0x000bd6f2, 0x000bdaf2, 0x000bdef2, // 25328-25335
	0x000be2f2, 0x000be6f2, 0x000beaf2, 0x000beef2, 0x000bf2f2, 0x000bf6f2, 0x000bfaf2, 0x000bfef2, // 25336-25343
	0x000c02f2, 0x000c06f2, 0x000c0af2, 0x000c0ef2, 0x000c12f2, 0x000c16f2, 0x000c1af2, 0x000c1ef2, // 25344-25351
	0x000c22f2, 0x000c26f2, 0x000c2af2, 0x000c2ef2, 0x000c32f2, 0x000c36f2, 0x000c3af2, 0x000c3ef2, // 25352-25359
	0x000c42f2, 0x000c46f2, 0x000c4af2, 0x000c4ef2, 0x000c52f2, 0x000c56f2, 0x000c5af2, 0x000c5ef2, // 25360-25367
	0x000c62f2, 0x000c66f2, 0x000c6af2, 0x000c6ef2, 0x000c72f2, 0x000c76f2, 0x000c7af2, 0x000c7ef2, // 25368-25375
	0x000c82f2, 0x000c86f2, 0x000c8af2, 0x000c8ef2, 0x000c92f2, 0x000c96f2, 0x000c9af2, 0x000c9ef2, // 25376-25383
	0x000ca2f2, 0x000ca6f2, 0x000caaf2, 0x000caef2, 0x000cb2f2, 0x000cb6f2, 0x000cbaf2, 0x000cbef2, // 25384-25391
	0x000cc2f2, 0x000cc6f2, 0x000ccaf2, 0x000ccef2, 0x000cd2f2, 0x000cd6f2, 0x000cdaf2, 0x000cdef2, // 25392-25399
	0x000ce2f2, 0x000ce6f2, 0x000ceaf2, 0x000ceef2, 0x000cf2f2, 0x000cf6f2, 0x000cfaf2, 0x000cfef2, // 25400-25407
	0x000d02f2, 0x000d06f2, 0x000d0af2, 0x000d0ef2, 0x000d12f2, 0x000d16f2, 0x000d1af2, 0x000d1ef2, // 25408-25415
	0x000d22f2, 0x000d26f2, 0x000d2af2, 0x000d2ef2, 0x000d32f2, 0x000d36f2, 0x000d3af2, 0x000d3ef2, // 25416-25423
	0x000d42f2, 0x000d46f2, 0x000d4af2, 0x000d4ef2, 0x000d52f2, 0x000d56f2, 0x000d5af2, 0x000d5ef2, // 25424-25431
	0x000d62f2, 0x000d66f2, 0x000d6af2, 0x000d6ef2, 0x000d72f2, 0x000d76f2, 0x000d7af2, 0x000d7ef2, // 25432-25439
	0x000d82f2, 0x000d86f2, 0x000d8af2, 0x000d8ef2, 0x000d92f2, 0x000d96f2, 0x000d9af2, 0x000d9ef2, // 25440-25447
	0x000da2f2, 0x000da6f2, 0x000daaf2, 0x000daef2, 0x000db2f2, 0x000db6f2, 0x000dbaf2, 0x000dbef2, // 25448-25455
	0x000dc2f2, 0x000dc6f2, 0x000dcaf2, 0x000dcef2, 0x000dd2f2, 0x000dd6f2, 0x000ddaf2, 0x000ddef2, // 25456-25463
	0x000de2f2, 0x000de6f2, 0x000deaf2, 0x000deef2, 0x000df2f2, 0x000df6f2, 0x000dfaf2, 0x000dfef2, // 25464-25471
	0x000e02f2, 0x000e06f2, 0x000e0af2, 0x000e0ef2, 0x000e12f2, 0x000e16f2, 0x000e1af2, 0x000e1ef2, // 25472-25479
	0x000e22f2, 0x000e26f2, 0x000e2af2, 0x000e2ef2, 0x000e32f2, 0x000e36f2, 0x000e3af2, 0x000e3ef2, // 25480-25487
	0x000e42f2, 0x000e46f2, 0x000e4af2, 0x000e4ef2, 0x000e52f2, 0x000e56f2, 0x000e5af2, 0x000e5ef2, // 25488-25495
	0x000e62f2, 0x000e66f2, 0x000e6af2, 0x000e6ef2, 0x000e72f2, 0x000e76f2, 0x000e7af2, 0x000e7ef2, // 25496-25503
	0x000e82f2, 0x000e86f2, 0x000e8af2, 0x000e8ef2, 0x000e92f2, 0x000e96f2, 0x000e9af2, 0x000e9ef2, // 25504-25511
	0x000ea2f2, 0x000ea6f2, 0x000eaaf2, 0x000eaef2, 0x000eb2f2, 0x000eb6f2, 0x000ebaf2, 0x000ebef2, // 25512-25519
	0x000ec2f2, 0x000ec6f2, 0x000ecaf2, 0x000ecef2, 0x000ed2f2, 0x000ed6f2, 0x000edaf2, 0x000edef2, // 25520-25527
	0x000ee2f2, 0x000ee6f2, 0x000eeaf2, 0x000eeef2, 0x000ef2f2, 0x000ef6f2, 0x000efaf2, 0x000efef2, // 25528-25535
	0x000f02f2, 0x000f06f2, 0x000f0af2, 0x000f0ef2, 0x000f12f2, 0x000f16f2, 0x000f1af2, 0x000f1ef2, // 25536-25543
	0x000f22f2, 0x000f26f2, 0x000f2af2, 0x000f2ef2, 0x000f32f2, 0x000f36f2, 0x000f3af2, 0x000f3ef2, // 25544-25551
	0x000f42f2, 0x000f46f2, 0x000f4af2, 0x000f4ef2, 0x000f52f2, 0x000f56f2, 0x000f5af2, 0x000f5ef2, // 25552-25559
	0x000f62f2, 0x000f66f2, 0x000f6af2, 0x000f6ef2, 0x000f72f2, 0x000f76f2, 0x000f7af2, 0x000f7ef2, // 25560-25567
	0x000f82f2, 0x000f86f2, 0x000f8af2, 0x000f8ef2, 0x000f92f2, 0x000f96f2, 0x000f9af2, 0x000f9ef2, // 25568-25575
	0x000fa2f2, 0x000fa6f2, 0x000faaf2, 0x000faef2, 0x000fb2f2, 0x000fb6f2, 0x000fbaf2, 0x000fbef2, // 25576-25583
	0x000fc2f2, 0x000fc6f2, 0x000fcaf2, 0x000fcef2, 0x000fd2f2, 0x000fd6f2, 0x000fdaf2, 0x000fdef2, // 25584-25591
	0x000fe2f2, 0x000fe6f2, 0x000feaf2, 0x000feef2, 0x000ff2f2, 0x000ff6f2, 0x000ffaf2, 0x000ffef2, // 25592-25599
	0x001002f2, 0x001006f2, 0x00100af2, 0x00100ef2, 0x001012f2, 0x001016f2, 0x00101af2, 0x00101ef2, // 25600-25607
	0x001022f2, 0x001026f2, 0x00102af2, 0x00102ef2, 0x001032f2, 0x001036f2, 0x00103af2, 0x00103ef2, // 25608-25615
	0x001042f2, 0x001046f2, 0x00104af2, 0x00104ef2, 0x001052f2, 0x001056f2, 0x00105af2, 0x00105ef2, // 25616-25623
	0x001062f2, 0x001066f2, 0x00106af2, 0x00106ef2, 0x001072f2, 0x001076f2, 0x00107af2, 0x00107ef2, // 25624-25631
	0x001082f2, 0x001086f2, 0x00108af2, 0x00108ef2, 0x001092f2, 0x001096f2, 0x00109af2, 0x00109ef2, // 25632-25639
	0x0010a2f2, 0x0010a6f2, 0x0010aaf2, 0x0010aef2, 0x0010b2f2, 0x0010b6f2, 0x0010baf2, 0x0010bef2, // 25640-25647
	0x0010c2f2, 0x0010c6f2, 0x0010caf2, 0x0010cef2, 0x0010d2f2, 0x0010d6f2, 0x0010daf2, 0x0010def2, // 25648-25655
	0x0010e2f2, 0x0010e6f2, 0x0010eaf2, 0x0010eef2, 0x0010f2f2, 0x0010f6f2, 0x0010faf2, 0x0010fef2, // 25656-25663
	0x001102f2, 0x001106f2, 0x00110af2, 0x00110ef2, 0x001112f2, 0x001116f2, 0x00111af2, 0x00111ef2, // 25664-25671
	0x001122f2, 0x001126f2, 0x00112af2, 0x00112ef2, 0x001132f2, 0x001136f2, 0x00113af2, 0x00113ef2, // 25672-25679
	0x001142f2, 0x001146f2, 0x00114af2, 0x00114ef2, 0x001152f2, 0x001156f2, 0x00115af2, 0x00115ef2, // 25680-25687
	0x001162f2, 0x001166f2, 0x00116af2, 0x00116ef2, 0x001172f2, 0x001176f2, 0x00117af2, 0x00117ef2, // 25688-25695
	0x001182f2, 0x001186f2, 0x00118af2, 0x00118ef2, 0x001192f2, 0x001196f2, 0x00119af2, 0x00119ef2, // 25696-25703
	0x0011a2f2, 0x0011a6f2, 0x0011aaf2, 0x0011aef2, 0x0011b2f2, 0x0011b6f2, 0x0011baf2, 0x0011bef2, // 25704-25711
	0x0011c2f2, 0x0011c6f2, 0x0011caf2, 0x0011cef2, 0x0011d2f2, 0x0011d6f2, 0x0011daf2, 0x0011def2, // 25712-25719
	0x0011e2f2, 0x0011e6f2, 0x0011eaf2, 0x0011eef2, 0x0011f2f2, 0x0011f6f2, 0x0011faf2, 0x0011fef2, // 25720-25727
	0x001202f2, 0x001206f2, 0x00120af2, 0x00120ef2, 0x001212f2, 0x001216f2, 0x00121af2, 0x00121ef2, // 25728-25735
	0x001222f2, 0x001226f2, 0x00122af2, 0x00122ef2, 0x001232f2, 0x001236f2, 0x00123af2, 0x00123ef2, // 25736-25743
	0x001242f2, 0x001246f2, 0x00124af2, 0x00124ef2, 0x001252f2, 0x001256f2, 0x00125af2, 0x00125ef2, // 25744-25751
	0x001262f2, 0x001266f2, 0x00126af2, 0x00126ef2, 0x001272f2, 0x001276f2, 0x00127af2, 0x00127ef2, // 25752-25759
	0x001282f2, 0x001286f2, 0x00128af2, 0x00128ef2, 0x001292f2, 0x001296f2, 0x00129af2, 0x00129ef2, // 25760-25767
	0x0012a2f2, 0x0012a6f2, 0x0012aaf2, 0x0012aef2, 0x0012b2f2, 0x0012b6f2, 0x0012baf2, 0x0012bef2, // 25768-25775
	0x0012c2f2, 0x0012c6f2, 0x0012caf2, 0x0012cef2, 0x0012d2f2, 0x0012d6f2, 0x0012daf2, 0x0012def2, // 25776-25783
	0x0012e2f2, 0x0012e6f2, 0x0012eaf2, 0x0012eef2, 0x0012f2f2, 0x0012f6f2, 0x0012faf2, 0x0012fef2, // 25784-25791
	0x001302f2, 0x001306f2, 0x00130af2, 0x00130ef2, 0x001312f2, 0x001316f2, 0x00131af2, 0x00131ef2, // 25792-25799
	0x001322f2, 0x001326f2, 0x00132af2, 0x00132ef2, 0x001332f2, 0x001336f2, 0x00133af2, 0x00133ef2, // 25800-25807
	0x001342f2, 0x001346f2, 0x00134af2, 0x00134ef2, 0x001352f2, 0x001356f2, 0x00135af2, 0x00135ef2, // 25808-25815
	0x001362f2, 0x001366f2, 0x00136af2, 0x00136ef2, 0x001372f2, 0x001376f2, 0x00137af2, 0x00137ef2, // 25816-25823
	0x001382f2, 0x001386f2, 0x00138af2, 0x00138ef2, 0x001392f2, 0x001396f2, 0x00139af2, 0x00139ef2, // 25824-25831
	0x0013a2f2, 0x0013a6f2, 0x0013aaf2, 0x0013aef2, 0x0013b2f2, 0x0013b6f2, 0x0013baf2, 0x0013bef2, // 25832-25839
	0x0013c2f2, 0x0013c6f2, 0x0013caf2, 0x0013cef2, 0x0013d2f2, 0x0013d6f2, 0x0013daf2, 0x0013def2, // 25840-25847
	0x0013e2f2, 0x0013e6f2, 0x0013eaf2, 0x0013eef2, 0x0013f2f2, 0x0013f6f2, 0x0013faf2, 0x0013fef2, // 25848-25855
	0x001402f2, 0x001406f2, 0x00140af2, 0x00140ef2, 0x001412f2, 0x001416f2, 0x00141af2, 0x00141ef2, // 25856-25863
	0x001422f2, 0x001426f2, 0x00142af2, 0x00142ef2, 0x001432f2, 0x001436f2, 0x00143af2, 0x00143ef2, // 25864-25871
	0x001442f2, 0x001446f2, 0x00144af2, 0x00144ef2, 0x001452f2, 0x001456f2, 0x00145af2, 0x00145ef2, // 25872-25879
	0x001462f2, 0x001466f2, 0x00146af2, 0x00146ef2, 0x001472f2, 0x001476f2, 0x00147af2, 0x00147ef2, // 25880-25887
	0x001482f2, 0x001486f2, 0x00148af2, 0x00148ef2, 0x001492f2, 0x001496f2, 0x00149af2, 0x00149ef2, // 25888-25895
	0x0014a2f2, 0x0014a6f2, 0x0014aaf2, 0x0014aef2, 0x0014b2f2, 0x0014b6f2, 0x0014baf2, 0x0014bef2, // 25896-25903
	0x0014c2f2, 0x0014c6f2, 0x0014caf2, 0x0014cef2, 0x0014d2f2, 0x0014d6f2, 0x0014daf2, 0x0014def2, // 25904-25911
	0x0014e2f2, 0x0014e6f2, 0x0014eaf2, 0x0014eef2, 0x0014f2f2, 0x0014f6f2, 0x0014faf2, 0x0014fef2, // 25912-25919
	0x001502f2, 0x001506f2, 0x00150af2, 0x00150ef2, 0x001512f2, 0x001516f2, 0x00151af2, 0x00151ef2, // 25920-25927
	0x001522f2, 0x001526f2, 0x00152af2, 0x00152ef2, 0x001532f2, 0x001536f2, 0x00153af2, 0x00153ef2, // 25928-25935
	0x001542f2, 0x001546f2, 0x00154af2, 0x00154ef2, 0x001552f2, 0x001556f2, 0x00155af2, 0x00155ef2, // 25936-25943
	0x001562f2, 0x001566f2, 0x00156af2, 0x00156ef2, 0x001572f2, 0x001576f2, 0x00157af2, 0x00157ef2, // 25944-25951
	0x001582f2, 0x001586f2, 0x00158af2, 0x00158ef2, 0x001592f2, 0x001596f2, 0x00159af2, 0x00159ef2, // 25952-25959
	0x0015a2f2, 0x0015a6f2, 0x0015aaf2, 0x0015aef2, 0x0015b2f2, 0x0015b6f2, 0x0015baf2, 0x0015bef2, // 25960-25967
	0x0015c2f2, 0x0015c6f2, 0x0015caf2, 0x0015cef2, 0x0015d2f2, 0x0015d6f2, 0x0015daf2, 0x0015def2, // 25968-25975
	0x0015e2f2, 0x0015e6f2, 0x0015eaf2, 0x0015eef2, 0x0015f2f2, 0x0015f6f2, 0x0015faf2, 0x0015fef2, // 25976-25983
	0x001602f2, 0x001606f2, 0x00160af2, 0x00160ef2, 0x001612f2, 0x001616f2, 0x00161af2, 0x00161ef2, // 25984-25991
	0x001622f2, 0x001626f2, 0x00162af2, 0x00162ef2, 0x001632f2, 0x001636f2, 0x00163af2, 0x00163ef2, // 25992-25999
	0x001642f2, 0x001646f2, 0x00164af2, 0x00164ef2, 0x001652f2, 0x001656f2, 0x00165af2, 0x00165ef2, // 26000-26007
	0x001662f2, 0x001666f2, 0x00166af2, 0x00166ef2, 0x001672f2, 0x001676f2, 0x00167af2, 0x00167ef2, // 26008-26015
	0x001682f2, 0x001686f2, 0x00168af2, 0x00168ef2, 0x001692f2, 0x001696f2, 0x00169af2, 0x00169ef2, // 26016-26023
	0x0016a2f2, 0x0016a6f2, 0x0016aaf2, 0x0016aef2, 0x0016b2f2, 0x0016b6f2, 0x0016baf2, 0x0016bef2, // 26024-26031
	0x0016c2f2, 0x0016c6f2, 0x0016caf2, 0x0016cef2, 0x0016d2f2, 0x0016d6f2, 0x0016daf2, 0x0016def2, // 26032-26039
	0x0016e2f2, 0x0016e6f2, 0x0016eaf2, 0x0016eef2, 0x0016f2f2, 0x0016f6f2, 0x0016faf2, 0x0016fef2, // 26040-26047
	0x001702f2, 0x001706f2, 0x00170af2, 0x00170ef2, 0x001712f2, 0x001716f2, 0x00171af2, 0x00171ef2, // 26048-26055
	0x001722f2, 0x001726f2, 0x00172af2, 0x00172ef2, 0x001732f2, 0x001736f2, 0x00173af2, 0x00173ef2, // 26056-26063
	0x001742f2, 0x001746f2, 0x00174af2, 0x00174ef2, 0x001752f2, 0x001756f2, 0x00175af2, 0x00175ef2, // 26064-26071
	0x001762f2, 0x001766f2, 0x00176af2, 0x00176ef2, 0x001772f2, 0x001776f2, 0x00177af2, 0x00177ef2, // 26072-26079
	0x001782f2, 0x001786f2, 0x00178af2, 0x00178ef2, 0x001792f2, 0x001796f2, 0x00179af2, 0x00179ef2, // 26080-26087
	0x0017a2f2, 0x0017a6f2, 0x0017aaf2, 0x0017aef2, 0x0017b2f2, 0x0017b6f2, 0x0017baf2, 0x0017bef2, // 26088-26095
	0x0017c2f2, 0x0017c6f2, 0x0017caf2, 0x0017cef2, 0x0017d2f2, 0x0017d6f2, 0x0017daf2, 0x0017def2, // 26096-26103
	0x0017e2f2, 0x0017e6f2, 0x0017eaf2, 0x0017eef2, 0x0017f2f2, 0x0017f6f2, 0x0017faf2, 0x0017fef2, // 26104-26111
	0x001802f2, 0x001806f2, 0x00180af2, 0x00180ef2, 0x001812f2, 0x001816f2, 0x00181af2, 0x00181ef2, // 26112-26119
	0x001822f2, 0x001826f2, 0x00182af2, 0x00182ef2, 0x001832f2, 0x001836f2, 0x00183af2, 0x00183ef2, // 26120-26127
	0x001842f2, 0x001846f2, 0x00184af2, 0x00184ef2, 0x001852f2, 0x001856f2, 0x00185af2, 0x00185ef2, // 26128-26135
	0x001862f2, 0x001866f2, 0x00186af2, 0x00186ef2, 0x001872f2, 0x001876f2, 0x00187af2, 0x00187ef2, // 26136-26143
	0x001882f2, 0x001886f2, 0x00188af2, 0x00188ef2, 0x001892f2, 0x001896f2, 0x00189af2, 0x00189ef2, // 26144-26151
	0x0018a2f2, 0x0018a6f2, 0x0018aaf2, 0x0018aef2, 0x0018b2f2, 0x0018b6f2, 0x0018baf2, 0x0018bef2, // 26152-26159
	0x0018c2f2, 0x0018c6f2, 0x0018caf2, 0x0018cef2, 0x0018d2f2, 0x0018d6f2, 0x0018daf2, 0x0018def2, // 26160-26167
	0x0018e2f2, 0x0018e6f2, 0x0018eaf2, 0x0018eef2, 0x0018f2f2, 0x0018f6f2, 0x0018faf2, 0x0018fef2, // 26168-26175
	0x001902f2, 0x001906f2, 0x00190af2, 0x00190ef2, 0x001912f2, 0x001916f2, 0x00191af2, 0x00191ef2, // 26176-26183
	0x001922f2, 0x001926f2, 0x00192af2, 0x00192ef2, 0x001932f2, 0x001936f2, 0x00193af2, 0x00193ef2, // 26184-26191
	0x001942f2, 0x001946f2, 0x00194af2, 0x00194ef2, 0x001952f2, 0x001956f2, 0x00195af2, 0x00195ef2, // 26192-26199
	0x001962f2, 0x001966f2, 0x00196af2, 0x00196ef2, 0x001972f2, 0x001976f2, 0x00197af2, 0x00197ef2, // 26200-26207
	0x001982f2, 0x001986f2, 0x00198af2, 0x00198ef2, 0x001992f2, 0x001996f2, 0x00199af2, 0x00199ef2, // 26208-26215
	0x0019a2f2, 0x0019a6f2, 0x0019aaf2, 0x0019aef2, 0x0019b2f2, 0x0019b6f2, 0x0019baf2, 0x0019bef2, // 26216-26223
	0x0019c2f2, 0x0019c6f2, 0x0019caf2, 0x0019cef2, 0x0019d2f2, 0x0019d6f2, 0x0019daf2, 0x0019def2, // 26224-26231
	0x0019e2f2, 0x0019e6f2, 0x0019eaf2, 0x0019eef2, 0x0019f2f2, 0x0019f6f2, 0x0019faf2, 0x0019fef2, // 26232-26239
	0x001a02f2, 0x001a06f2, 0x001a0af2, 0x001a0ef2, 0x001a12f2, 0x001a16f2, 0x001a1af2, 0x001a1ef2, // 26240-26247
	0x001a22f2, 0x001a26f2, 0x001a2af2, 0x001a2ef2, 0x001a32f2, 0x001a36f2, 0x001a3af2, 0x001a3ef2, // 26248-26255
	0x001a42f2, 0x001a46f2, 0x001a4af2, 0x001a4ef2, 0x001a52f2, 0x001a56f2, 0x001a5af2, 0x001a5ef2, // 26256-26263
	0x001a62f2, 0x001a66f2, 0x001a6af2, 0x001a6ef2, 0x001a72f2, 0x001a76f2, 0x001a7af2, 0x001a7ef2, // 26264-26271
	0x001a82f2, 0x001a86f2, 0x001a8af2, 0x001a8ef2, 0x001a92f2, 0x001a96f2, 0x001a9af2, 0x001a9ef2, // 26272-26279
	0x001aa2f2, 0x001aa6f2, 0x001aaaf2, 0x001aaef2, 0x001ab2f2, 0x001ab6f2, 0x001abaf2, 0x001abef2, // 26280-26287
	0x001ac2f2, 0x001ac6f2, 0x001acaf2, 0x001acef2, 0x001ad2f2, 0x001ad6f2, 0x001adaf2, 0x001adef2, // 26288-26295
	0x001ae2f2, 0x001ae6f2, 0x001aeaf2, 0x001aeef2, 0x001af2f2, 0x001af6f2, 0x001afaf2, 0x001afef2, // 26296-26303
	0x001b02f2, 0x001b06f2, 0x001b0af2, 0x001b0ef2, 0x001b12f2, 0x001b16f2, 0x001b1af2, 0x001b1ef2, // 26304-26311
	0x001b22f2, 0x001b26f2, 0x001b2af2, 0x001b2ef2, 0x001b32f2, 0x001b36f2, 0x001b3af2, 0x001b3ef2, // 26312-26319
	0x001b42f2, 0x001b46f2, 0x001b4af2, 0x001b4ef2, 0x001b52f2, 0x001b56f2, 0x001b5af2, 0x001b5ef2, // 26320-26327
	0x001b62f2, 0x001b66f2, 0x001b6af2, 0x001b6ef2, 0x001b72f2, 0x001b76f2, 0x001b7af2, 0x001b7ef2, // 26328-26335
	0x001b82f2, 0x001b86f2, 0x001b8af2, 0x001b8ef2, 0x001b92f2, 0x001b96f2, 0x001b9af2, 0x001b9ef2, // 26336-26343
	0x001ba2f2, 0x001ba6f2, 0x001baaf2, 0x001baef2, 0x001bb2f2, 0x001bb6f2, 0x001bbaf2, 0x001bbef2, // 26344-26351
	0x001bc2f2, 0x001bc6f2, 0x001bcaf2, 0x001bcef2, 0x001bd2f2, 0x001bd6f2, 0x001bdaf2, 0x001bdef2, // 26352-26359
	0x001be2f2, 0x001be6f2, 0x001beaf2, 0x001beef2, 0x001bf2f2, 0x001bf6f2, 0x001bfaf2, 0x001bfef2, // 26360-26367
	0x001c02f2, 0x001c06f2, 0x001c0af2, 0x001c0ef2, 0x001c12f2, 0x001c16f2, 0x001c1af2, 0x001c1ef2, // 26368-26375
	0x001c22f2, 0x001c26f2, 0x001c2af2, 0x001c2ef2, 0x001c32f2, 0x001c36f2, 0x001c3af2, 0x001c3ef2, // 26376-26383
	0x001c42f2, 0x001c46f2, 0x001c4af2, 0x001c4ef2, 0x001c52f2, 0x001c56f2, 0x001c5af2, 0x001c5ef2, // 26384-26391
	0x001c62f2, 0x001c66f2, 0x001c6af2, 0x001c6ef2, 0x001c72f2, 0x001c76f2, 0x001c7af2, 0x001c7ef2, // 26392-26399
	0x001c82f2, 0x001c86f2, 0x001c8af2, 0x001c8ef2, 0x001c92f2, 0x001c96f2, 0x001c9af2, 0x001c9ef2, // 26400-26407
	0x001ca2f2, 0x001ca6f2, 0x001caaf2, 0x001caef2, 0x001cb2f2, 0x001cb6f2, 0x001cbaf2, 0x001cbef2, // 26408-26415
	0x001cc2f2, 0x001cc6f2, 0x001ccaf2, 0x001ccef2, 0x001cd2f2, 0x001cd6f2, 0x001cdaf2, 0x001cdef2, // 26416-26423
	0x001ce2f2, 0x001ce6f2, 0x001ceaf2, 0x001ceef2, 0x001cf2f2, 0x001cf6f2, 0x001cfaf2, 0x001cfef2, // 26424-26431
	0x001d02f2, 0x001d06f2, 0x001d0af2, 0x001d0ef2, 0x001d12f2, 0x001d16f2, 0x001d1af2, 0x001d1ef2, // 26432-26439
	0x001d22f2, 0x001d26f2, 0x001d2af2, 0x001d2ef2, 0x001d32f2, 0x001d36f2, 0x001d3af2, 0x001d3ef2, // 26440-26447
	0x001d42f2, 0x001d46f2, 0x001d4af2, 0x001d4ef2, 0x001d52f2, 0x001d56f2, 0x001d5af2, 0x001d5ef2, // 26448-26455
	0x001d62f2, 0x001d66f2, 0x001d6af2, 0x001d6ef2, 0x001d72f2, 0x001d76f2, 0x001d7af2, 0x001d7ef2, // 26456-26463
	0x001d82f2, 0x001d86f2, 0x001d8af2, 0x001d8ef2, 0x001d92f2, 0x001d96f2, 0x001d9af2, 0x001d9ef2, // 26464-26471
	0x001da2f2, 0x001da6f2, 0x001daaf2, 0x001daef2, 0x001db2f2, 0x001db6f2, 0x001dbaf2, 0x001dbef2, // 26472-26479
	0x001dc2f2, 0x001dc6f2, 0x001dcaf2, 0x001dcef2, 0x001dd2f2, 0x001dd6f2, 0x001ddaf2, 0x001ddef2, // 26480-26487
	0x001de2f2, 0x001de6f2, 0x001deaf2, 0x001deef2, 0x001df2f2, 0x001df6f2, 0x001dfaf2, 0x001dfef2, // 26488-26495
	0x001e02f2, 0x001e06f2, 0x001e0af2, 0x001e0ef2, 0x001e12f2, 0x001e16f2, 0x001e1af2, 0x001e1ef2, // 26496-26503
	0x001e22f2, 0x001e26f2, 0x001e2af2, 0x001e2ef2, 0x001e32f2, 0x001e36f2, 0x001e3af2, 0x001e3ef2, // 26504-26511
	0x001e42f2, 0x001e46f2, 0x001e4af2, 0x001e4ef2, 0x001e52f2, 0x001e56f2, 0x001e5af2, 0x001e5ef2, // 26512-26519
	0x001e62f2, 0x001e66f2, 0x001e6af2, 0x001e6ef2, 0x001e72f2, 0x001e76f2, 0x001e7af2, 0x001e7ef2, // 26520-26527
	0x001e82f2, 0x001e86f2, 0x001e8af2, 0x001e8ef2, 0x001e92f2, 0x001e96f2, 0x001e9af2, 0x001e9ef2, // 26528-26535
	0x001ea2f2, 0x001ea6f2, 0x001eaaf2, 0x001eaef2, 0x001eb2f2, 0x001eb6f2, 0x001ebaf2, 0x001ebef2, // 26536-26543
	0x001ec2f2, 0x001ec6f2, 0x001ecaf2, 0x001ecef2, 0x001ed2f2, 0x001ed6f2, 0x001edaf2, 0x001edef2, // 26544-26551
	0x001ee2f2, 0x001ee6f2, 0x001eeaf2, 0x001eeef2, 0x001ef2f2, 0x001ef6f2, 0x001efaf2, 0x001efef2, // 26552-26559
	0x001f02f2, 0x001f06f2, 0x001f0af2, 0x001f0ef2, 0x001f12f2, 0x001f16f2, 0x001f1af2, 0x001f1ef2, // 26560-26567
	0x001f22f2, 0x001f26f2, 0x001f2af2, 0x001f2ef2, 0x001f32f2, 0x001f36f2, 0x001f3af2, 0x001f3ef2, // 26568-26575
	0x001f42f2, 0x001f46f2, 0x001f4af2, 0x001f4ef2, 0x001f52f2, 0x001f56f2, 0x001f5af2, 0x001f5ef2, // 26576-26583
	0x001f62f2, 0x001f66f2, 0x001f6af2, 0x001f6ef2, 0x001f72f2, 0x001f76f2, 0x001f7af2, 0x001f7ef2, // 26584-26591
	0x001f82f2, 0x001f86f2, 0x001f8af2, 0x001f8ef2, 0x001f92f2, 0x001f96f2, 0x001f9af2, 0x001f9ef2, // 26592-26599
	0x001fa2f2, 0x001fa6f2, 0x001faaf2, 0x001faef2, 0x001fb2f2, 0x001fb6f2, 0x001fbaf2, 0x001fbef2, // 26600-26607
	0x001fc2f2, 0x001fc6f2, 0x001fcaf2, 0x001fcef2, 0x001fd2f2, 0x001fd6f2, 0x001fdaf2, 0x001fdef2, // 26608-26615
	0x001fe2f2, 0x001fe6f2, 0x001feaf2, 0x001feef2, 0x001ff2f2, 0x001ff6f2, 0x001ffaf2, 0x001ffef2, // 26616-26623
	0x002002f2, 0x002006f2, 0x00200af2, 0x00200ef2, 0x002012f2, 0x002016f2, 0x00201af2, 0x00201ef2, // 26624-26631
	0x002022f2, 0x002026f2, 0x00202af2, 0x00202ef2, 0x002032f2, 0x002036f2, 0x00203af2, 0x00203ef2, // 26632-26639
	0x002042f2, 0x002046f2, 0x00204af2, 0x00204ef2, 0x002052f2, 0x002056f2, 0x00205af2, 0x00205ef2, // 26640-26647
	0x002062f2, 0x002066f2, 0x00206af2, 0x00206ef2, 0x002072f2, 0x002076f2, 0x00207af2, 0x00207ef2, // 26648-26655
	0x002082f2, 0x002086f2, 0x00208af2, 0x00208ef2, 0x002092f2, 0x002096f2, 0x00209af2, 0x00209ef2, // 26656-26663
	0x0020a2f2, 0x0020a6f2, 0x0020aaf2, 0x0020aef2, 0x0020b2f2, 0x0020b6f2, 0x0020baf2, 0x0020bef2, // 26664-26671
	0x0020c2f2, 0x0020c6f2, 0x0020caf2, 0x0020cef2, 0x0020d2f2, 0x0020d6f2, 0x0020daf2, 0x0020def2, // 26672-26679
	0x0020e2f2, 0x0020e6f2, 0x0020eaf2, 0x0020eef2, 0x0020f2f2, 0x0020f6f2, 0x0020faf2, 0x0020fef2, // 26680-26687
	0x002102f2, 0x002106f2, 0x00210af2, 0x00210ef2, 0x002112f2, 0x002116f2, 0x00211af2, 0x00211ef2, // 26688-26695
	0x002122f2, 0x002126f2, 0x00212af2, 0x00212ef2, 0x002132f2, 0x002136f2, 0x00213af2, 0x00213ef2, // 26696-26703
	0x002142f2, 0x002146f2, 0x00214af2, 0x00214ef2, 0x002152f2, 0x002156f2, 0x00215af2, 0x00215ef2, // 26704-26711
	0x002162f2, 0x002166f2, 0x00216af2, 0x00216ef2, 0x002172f2, 0x002176f2, 0x00217af2, 0x00217ef2, // 26712-26719
	0x002182f2, 0x002186f2, 0x00218af2, 0x00218ef2, 0x002192f2, 0x002196f2, 0x00219af2, 0x00219ef2, // 26720-26727
	0x0021a2f2, 0x0021a6f2, 0x0021aaf2, 0x0021aef2, 0x0021b2f2, 0x0021b6f2, 0x0021baf2, 0x0021bef2, // 26728-26735
	0x0021c2f2, 0x0021c6f2, 0x0021caf2, 0x0021cef2, 0x0021d2f2, 0x0021d6f2, 0x0021daf2, 0x0021def2, // 26736-26743
	0x0021e2f2, 0x0021e6f2, 0x0021eaf2, 0x0021eef2, 0x0021f2f2, 0x0021f6f2, 0x0021faf2, 0x0021fef2, // 26744-26751
	0x002202f2, 0x002206f2, 0x00220af2, 0x00220ef2, 0x002212f2, 0x002216f2, 0x00221af2, 0x00221ef2, // 26752-26759
	0x002222f2, 0x002226f2, 0x00222af2, 0x00222ef2, 0x002232f2, 0x002236f2, 0x00223af2, 0x00223ef2, // 26760-26767
	0x002242f2, 0x002246f2, 0x00224af2, 0x00224ef2, 0x002252f2, 0x002256f2, 0x00225af2, 0x00225ef2, // 26768-26775
	0x002262f2, 0x002266f2, 0x00226af2, 0x00226ef2, 0x002272f2, 0x002276f2, 0x00227af2, 0x00227ef2, // 26776-26783
	0x002282f2, 0x002286f2, 0x00228af2, 0x00228ef2, 0x002292f2, 0x002296f2, 0x00229af2, 0x00229ef2, // 26784-26791
	0x0022a2f2, 0x0022a6f2, 0x0022aaf2, 0x0022aef2, 0x0022b2f2, 0x0022b6f2, 0x0022baf2, 0x0022bef2, // 26792-26799
	0x0022c2f2, 0x0022c6f2, 0x0022caf2, 0x0022cef2, 0x0022d2f2, 0x0022d6f2, 0x0022daf2, 0x0022def2, // 26800-26807
	0x0022e2f2, 0x0022e6f2, 0x0022eaf2, 0x0022eef2, 0x0022f2f2, 0x0022f6f2, 0x0022faf2, 0x0022fef2, // 26808-26815
	0x002302f2, 0x002306f2, 0x00230af2, 0x00230ef2, 0x002312f2, 0x002316f2, 0x00231af2, 0x00231ef2, // 26816-26823
	0x002322f2, 0x002326f2, 0x00232af2, 0x00232ef2, 0x002332f2, 0x002336f2, 0x00233af2, 0x00233ef2, // 26824-26831
	0x002342f2, 0x002346f2, 0x00234af2, 0x00234ef2, 0x002352f2, 0x002356f2, 0x00235af2, 0x00235ef2, // 26832-26839
	0x002362f2, 0x002366f2, 0x00236af2, 0x00236ef2, 0x002372f2, 0x002376f2, 0x00237af2, 0x00237ef2, // 26840-26847
	0x002382f2, 0x002386f2, 0x00238af2, 0x00238ef2, 0x002392f2, 0x002396f2, 0x00239af2, 0x00239ef2, // 26848-26855
	0x0023a2f2, 0x0023a6f2, 0x0023aaf2, 0x0023aef2, 0x0023b2f2, 0x0023b6f2, 0x0023baf2, 0x0023bef2, // 26856-26863
	0x0023c2f2, 0x0023c6f2, 0x0023caf2, 0x0023cef2, 0x0023d2f2, 0x0023d6f2, 0x0023daf2, 0x0023def2, // 26864-26871
	0x0023e2f2, 0x0023e6f2, 0x0023eaf2, 0x0023eef2, 0x0023f2f2, 0x0023f6f2, 0x0023faf2, 0x0023fef2, // 26872-26879
	0x002402f2, 0x002406f2, 0x00240af2, 0x00240ef2, 0x002412f2, 0x002416f2, 0x00241af2, 0x00241ef2, // 26880-26887
	0x002422f2, 0x002426f2, 0x00242af2, 0x00242ef2, 0x002432f2, 0x002436f2, 0x00243af2, 0x00243ef2, // 26888-26895
	0x002442f2, 0x002446f2, 0x00244af2, 0x00244ef2, 0x002452f2, 0x002456f2, 0x00245af2, 0x00245ef2, // 26896-26903
	0x002462f2, 0x002466f2, 0x00246af2, 0x00246ef2, 0x002472f2, 0x002476f2, 0x00247af2, 0x00247ef2, // 26904-26911
	0x002482f2, 0x002486f2, 0x00248af2, 0x00248ef2, 0x002492f2, 0x002496f2, 0x00249af2, 0x00249ef2, // 26912-26919
	0x0024a2f2, 0x0024a6f2, 0x0024aaf2, 0x0024aef2, 0x0024b2f2, 0x0024b6f2, 0x0024baf2, 0x0024bef2, // 26920-26927
	0x0024c2f2, 0x0024c6f2, 0x0024caf2, 0x0024cef2, 0x0024d2f2, 0x0024d6f2, 0x0024daf2, 0x0024def2, // 26928-26935
	0x0024e2f2, 0x0024e6f2, 0x0024eaf2, 0x0024eef2, 0x0024f2f2, 0x0024f6f2, 0x0024faf2, 0x0024fef2, // 26936-26943
	0x002502f2, 0x002506f2, 0x00250af2, 0x00250ef2, 0x002512f2, 0x002516f2, 0x00251af2, 0x00251ef2, // 26944-26951
	0x002522f2, 0x002526f2, 0x00252af2, 0x00252ef2, 0x002532f2, 0x002536f2, 0x00253af2, 0x00253ef2, // 26952-26959
	0x002542f2, 0x002546f2, 0x00254af2, 0x00254ef2, 0x002552f2, 0x002556f2, 0x00255af2, 0x00255ef2, // 26960-26967
	0x002562f2, 0x002566f2, 0x00256af2, 0x00256ef2, 0x002572f2, 0x002576f2, 0x00257af2, 0x00257ef2, // 26968-26975
	0x002582f2, 0x002586f2, 0x00258af2, 0x00258ef2, 0x002592f2, 0x002596f2, 0x00259af2, 0x00259ef2, // 26976-26983
	0x0025a2f2, 0x0025a6f2, 0x0025aaf2, 0x0025aef2, 0x0025b2f2, 0x0025b6f2, 0x0025baf2, 0x0025bef2, // 26984-26991
	0x0025c2f2, 0x0025c6f2, 0x0025caf2, 0x0025cef2, 0x0025d2f2, 0x0025d6f2, 0x0025daf2, 0x0025def2, // 26992-26999
	0x0025e2f2, 0x0025e6f2, 0x0025eaf2, 0x0025eef2, 0x0025f2f2, 0x0025f6f2, 0x0025faf2, 0x0025fef2, // 27000-27007
	0x002602f2, 0x002606f2, 0x00260af2, 0x00260ef2, 0x002612f2, 0x002616f2, 0x00261af2, 0x00261ef2, // 27008-27015
	0x002622f2, 0x002626f2, 0x00262af2, 0x00262ef2, 0x002632f2, 0x002636f2, 0x00263af2, 0x00263ef2, // 27016-27023
	0x002642f2, 0x002646f2, 0x00264af2, 0x00264ef2, 0x002652f2, 0x002656f2, 0x00265af2, 0x00265ef2, // 27024-27031
	0x002662f2, 0x002666f2, 0x00266af2, 0x00266ef2, 0x002672f2, 0x002676f2, 0x00267af2, 0x00267ef2, // 27032-27039
	0x002682f2, 0x002686f2, 0x00268af2, 0x00268ef2, 0x002692f2, 0x002696f2, 0x00269af2, 0x00269ef2, // 27040-27047
	0x0026a2f2, 0x0026a6f2, 0x0026aaf2, 0x0026aef2, 0x0026b2f2, 0x0026b6f2, 0x0026baf2, 0x0026bef2, // 27048-27055
	0x0026c2f2, 0x0026c6f2, 0x0026caf2, 0x0026cef2, 0x0026d2f2, 0x0026d6f2, 0x0026daf2, 0x0026def2, // 27056-27063
	0x0026e2f2, 0x0026e6f2, 0x0026eaf2, 0x0026eef2, 0x0026f2f2, 0x0026f6f2, 0x0026faf2, 0x0026fef2, // 27064-27071
	0x002702f2, 0x002706f2, 0x00270af2, 0x00270ef2, 0x002712f2, 0x002716f2, 0x00271af2, 0x00271ef2, // 27072-27079
	0x002722f2, 0x002726f2, 0x00272af2, 0x00272ef2, 0x002732f2, 0x002736f2, 0x00273af2, 0x00273ef2, // 27080-27087
	0x002742f2, 0x002746f2, 0x00274af2, 0x00274ef2, 0x002752f2, 0x002756f2, 0x00275af2, 0x00275ef2, // 27088-27095
	0x002762f2, 0x002766f2, 0x00276af2, 0x00276ef2, 0x002772f2, 0x002776f2, 0x00277af2, 0x00277ef2, // 27096-27103
	0x002782f2, 0x002786f2, 0x00278af2, 0x00278ef2, 0x002792f2, 0x002796f2, 0x00279af2, 0x00279ef2, // 27104-27111
	0x0027a2f2, 0x0027a6f2, 0x0027aaf2, 0x0027aef2, 0x0027b2f2, 0x0027b6f2, 0x0027baf2, 0x0027bef2, // 27112-27119
	0x0027c2f2, 0x0027c6f2, 0x0027caf2, 0x0027cef2, 0x0027d2f2, 0x0027d6f2, 0x0027daf2, 0x0027def2, // 27120-27127
	0x0027e2f2, 0x0027e6f2, 0x0027eaf2, 0x0027eef2, 0x0027f2f2, 0x0027f6f2, 0x0027faf2, 0x0027fef2, // 27128-27135
	0x002802f2, 0x002806f2, 0x00280af2, 0x00280ef2, 0x002812f2, 0x002816f2, 0x00281af2, 0x00281ef2, // 27136-27143
	0x002822f2, 0x002826f2, 0x00282af2, 0x00282ef2, 0x002832f2, 0x002836f2, 0x00283af2, 0x00283ef2, // 27144-27151
	0x002842f2, 0x002846f2, 0x00284af2, 0x00284ef2, 0x002852f2, 0x002856f2, 0x00285af2, 0x00285ef2, // 27152-27159
	0x002862f2, 0x002866f2, 0x00286af2, 0x00286ef2, 0x002872f2, 0x002876f2, 0x00287af2, 0x00287ef2, // 27160-27167
	0x002882f2, 0x002886f2, 0x00288af2, 0x00288ef2, 0x002892f2, 0x002896f2, 0x00289af2, 0x00289ef2, // 27168-27175
	0x0028a2f2, 0x0028a6f2, 0x0028aaf2, 0x0028aef2, 0x0028b2f2, 0x0028b6f2, 0x0028baf2, 0x0028bef2, // 27176-27183
	0x0028c2f2, 0x0028c6f2, 0x0028caf2, 0x0028cef2, 0x0028d2f2, 0x0028d6f2, 0x0028daf2, 0x0028def2, // 27184-27191
	0x0028e2f2, 0x0028e6f2, 0x0028eaf2, 0x0028eef2, 0x0028f2f2, 0x0028f6f2, 0x0028faf2, 0x0028fef2, // 27192-27199
	0x002902f2, 0x002906f2, 0x00290af2, 0x00290ef2, 0x002912f2, 0x002916f2, 0x00291af2, 0x00291ef2, // 27200-27207
	0x002922f2, 0x002926f2, 0x00292af2, 0x00292ef2, 0x002932f2, 0x002936f2, 0x00293af2, 0x00293ef2, // 27208-27215
	0x002942f2, 0x002946f2, 0x00294af2, 0x00294ef2, 0x002952f2, 0x002956f2, 0x00295af2, 0x00295ef2, // 27216-27223
	0x002962f2, 0x002966f2, 0x00296af2, 0x00296ef2, 0x002972f2, 0x002976f2, 0x00297af2, 0x00297ef2, // 27224-27231
	0x002982f2, 0x002986f2, 0x00298af2, 0x00298ef2, 0x002992f2, 0x002996f2, 0x00299af2, 0x00299ef2, // 27232-27239
	0x0029a2f2, 0x0029a6f2, 0x0029aaf2, 0x0029aef2, 0x0029b2f2, 0x0029b6f2, 0x0029baf2, 0x0029bef2, // 27240-27247
	0x0029c2f2, 0x0029c6f2, 0x0029caf2, 0x0029cef2, 0x0029d2f2, 0x0029d6f2, 0x0029daf2, 0x0029def2, // 27248-27255
	0x0029e2f2, 0x0029e6f2, 0x0029eaf2, 0x0029eef2, 0x0029f2f2, 0x0029f6f2, 0x0029faf2, 0x0029fef2, // 27256-27263
	0x002a02f2, 0x002a06f2, 0x002a0af2, 0x002a0ef2, 0x002a12f2, 0x002a16f2, 0x002a1af2, 0x002a1ef2, // 27264-27271
	0x002a22f2, 0x002a26f2, 0x002a2af2, 0x002a2ef2, 0x002a32f2, 0x002a36f2, 0x002a3af2, 0x002a3ef2, // 27272-27279
	0x002a42f2, 0x002a46f2, 0x002a4af2, 0x002a4ef2, 0x002a52f2, 0x002a56f2, 0x002a5af2, 0x002a5ef2, // 27280-27287
	0x002a62f2, 0x002a66f2, 0x002a6af2, 0x002a6ef2, 0x002a72f2, 0x002a76f2, 0x002a7af2, 0x002a7ef2, // 27288-27295
	0x002a82f2, 0x002a86f2, 0x002a8af2, 0x002a8ef2, 0x002a92f2, 0x002a96f2, 0x002a9af2, 0x002a9ef2, // 27296-27303
	0x002aa2f2, 0x002aa6f2, 0x002aaaf2, 0x002aaef2, 0x002ab2f2, 0x002ab6f2, 0x002abaf2, 0x002abef2, // 27304-27311
	0x002ac2f2, 0x002ac6f2, 0x002acaf2, 0x002acef2, 0x002ad2f2, 0x002ad6f2, 0x002adaf2, 0x002adef2, // 27312-27319
	0x002ae2f2, 0x002ae6f2, 0x002aeaf2, 0x002aeef2, 0x002af2f2, 0x002af6f2, 0x002afaf2, 0x002afef2, // 27320-27327
	0x002b02f2, 0x002b06f2, 0x002b0af2, 0x002b0ef2, 0x002b12f2, 0x002b16f2, 0x002b1af2, 0x002b1ef2, // 27328-27335
	0x002b22f2, 0x002b26f2, 0x002b2af2, 0x002b2ef2, 0x002b32f2, 0x002b36f2, 0x002b3af2, 0x002b3ef2, // 27336-27343
	0x002b42f2, 0x002b46f2, 0x002b4af2, 0x002b4ef2, 0x002b52f2, 0x002b56f2, 0x002b5af2, 0x002b5ef2, // 27344-27351
	0x002b62f2, 0x002b66f2, 0x002b6af2, 0x002b6ef2, 0x002b72f2, 0x002b76f2, 0x002b7af2, 0x002b7ef2, // 27352-27359
	0x002b82f2, 0x002b86f2, 0x002b8af2, 0x002b8ef2, 0x002b92f2, 0x002b96f2, 0x002b9af2, 0x002b9ef2, // 27360-27367
	0x002ba2f2, 0x002ba6f2, 0x002baaf2, 0x002baef2, 0x002bb2f2, 0x002bb6f2, 0x002bbaf2, 0x002bbef2, // 27368-27375
	0x002bc2f2, 0x002bc6f2, 0x002bcaf2, 0x002bcef2, 0x002bd2f2, 0x002bd6f2, 0x002bdaf2, 0x002bdef2, // 27376-27383
	0x002be2f2, 0x002be6f2, 0x002beaf2, 0x002beef2, 0x002bf2f2, 0x002bf6f2, 0x002bfaf2, 0x002bfef2, // 27384-27391
	0x002c02f2, 0x002c06f2, 0x002c0af2, 0x002c0ef2, 0x002c12f2, 0x002c16f2, 0x002c1af2, 0x002c1ef2, // 27392-27399
	0x002c22f2, 0x002c26f2, 0x002c2af2, 0x002c2ef2, 0x002c32f2, 0x002c36f2, 0x002c3af2, 0x002c3ef2, // 27400-27407
	0x002c42f2, 0x002c46f2, 0x002c4af2, 0x002c4ef2, 0x002c52f2, 0x002c56f2, 0x002c5af2, 0x002c5ef2, // 27408-27415
	0x002c62f2, 0x002c66f2, 0x002c6af2, 0x002c6ef2, 0x002c72f2, 0x002c76f2, 0x002c7af2, 0x002c7ef2, // 27416-27423
	0x002c82f2, 0x002c86f2, 0x002c8af2, 0x002c8ef2, 0x002c92f2, 0x002c96f2, 0x002c9af2, 0x002c9ef2, // 27424-27431
	0x002ca2f2, 0x002ca6f2, 0x002caaf2, 0x002caef2, 0x002cb2f2, 0x002cb6f2, 0x002cbaf2, 0x002cbef2, // 27432-27439
	0x002cc2f2, 0x002cc6f2, 0x002ccaf2, 0x002ccef2, 0x002cd2f2, 0x002cd6f2, 0x002cdaf2, 0x002cdef2, // 27440-27447
	0x002ce2f2, 0x002ce6f2, 0x002ceaf2, 0x002ceef2, 0x002cf2f2, 0x002cf6f2, 0x002cfaf2, 0x002cfef2, // 27448-27455
	0x002d02f2, 0x002d06f2, 0x002d0af2, 0x002d0ef2, 0x002d12f2, 0x002d16f2, 0x002d1af2, 0x002d1ef2, // 27456-27463
	0x002d22f2, 0x002d26f2, 0x002d2af2, 0x002d2ef2, 0x002d32f2, 0x002d36f2, 0x002d3af2, 0x002d3ef2, // 27464-27471
	0x002d42f2, 0x002d46f2, 0x002d4af2, 0x002d4ef2, 0x002d52f2, 0x002d56f2, 0x002d5af2, 0x002d5ef2, // 27472-27479
	0x002d62f2, 0x002d66f2, 0x002d6af2, 0x002d6ef2, 0x002d72f2, 0x002d76f2, 0x002d7af2, 0x002d7ef2, // 27480-27487
	0x002d82f2, 0x002d86f2, 0x002d8af2, 0x002d8ef2, 0x002d92f2, 0x002d96f2, 0x002d9af2, 0x002d9ef2, // 27488-27495
	0x002da2f2, 0x002da6f2, 0x002daaf2, 0x002daef2, 0x002db2f2, 0x002db6f2, 0x002dbaf2, 0x002dbef2, // 27496-27503
	0x002dc2f2, 0x002dc6f2, 0x002dcaf2, 0x002dcef2, 0x002dd2f2, 0x002dd6f2, 0x002ddaf2, 0x002ddef2, // 27504-27511
	0x002de2f2, 0x002de6f2, 0x002deaf2, 0x002deef2, 0x002df2f2, 0x002df6f2, 0x002dfaf2, 0x002dfef2, // 27512-27519
	0x002e02f2, 0x002e06f2, 0x002e0af2, 0x002e0ef2, 0x002e12f2, 0x002e16f2, 0x002e1af2, 0x002e1ef2, // 27520-27527
	0x002e22f2, 0x002e26f2, 0x002e2af2, 0x002e2ef2, 0x002e32f2, 0x002e36f2, 0x002e3af2, 0x002e3ef2, // 27528-27535
	0x002e42f2, 0x002e46f2, 0x002e4af2, 0x002e4ef2, 0x002e52f2, 0x002e56f2, 0x002e5af2, 0x002e5ef2, // 27536-27543
	0x002e62f2, 0x002e66f2, 0x002e6af2, 0x002e6ef2, 0x002e72f2, 0x002e76f2, 0x002e7af2, 0x002e7ef2, // 27544-27551
	0x002e82f2, 0x002e86f2, 0x002e8af2, 0x002e8ef2, 0x002e92f2, 0x002e96f2, 0x002e9af2, 0x002e9ef2, // 27552-27559
	0x002ea2f2, 0x002ea6f2, 0x002eaaf2, 0x002eaef2, 0x002eb2f2, 0x002eb6f2, 0x002ebaf2, 0x002ebef2, // 27560-27567
	0x002ec2f2, 0x002ec6f2, 0x002ecaf2, 0x002ecef2, 0x002ed2f2, 0x002ed6f2, 0x002edaf2, 0x002edef2, // 27568-27575
	0x002ee2f2, 0x002ee6f2, 0x002eeaf2, 0x002eeef2, 0x002ef2f2, 0x002ef6f2, 0x002efaf2, 0x002efef2, // 27576-27583
	0x002f02f2, 0x002f06f2, 0x002f0af2, 0x002f0ef2, 0x002f12f2, 0x002f16f2, 0x002f1af2, 0x002f1ef2, // 27584-27591
	0x002f22f2, 0x002f26f2, 0x002f2af2, 0x002f2ef2, 0x002f32f2, 0x002f36f2, 0x002f3af2, 0x002f3ef2, // 27592-27599
	0x002f42f2, 0x002f46f2, 0x002f4af2, 0x002f4ef2, 0x002f52f2, 0x002f56f2, 0x002f5af2, 0x002f5ef2, // 27600-27607
	0x002f62f2, 0x002f66f2, 0x002f6af2, 0x002f6ef2, 0x002f72f2, 0x002f76f2, 0x002f7af2, 0x002f7ef2, // 27608-27615
	0x002f82f2, 0x002f86f2, 0x002f8af2, 0x002f8ef2, 0x002f92f2, 0x002f96f2, 0x002f9af2, 0x002f9ef2, // 27616-27623
	0x002fa2f2, 0x002fa6f2, 0x002faaf2, 0x002faef2, 0x002fb2f2, 0x002fb6f2, 0x002fbaf2, 0x002fbef2, // 27624-27631
	0x002fc2f2, 0x002fc6f2, 0x002fcaf2, 0x002fcef2, 0x002fd2f2, 0x002fd6f2, 0x002fdaf2, 0x002fdef2, // 27632-27639
	0x002fe2f2, 0x002fe6f2, 0x002feaf2, 0x002feef2, 0x002ff2f2, 0x002ff6f2, 0x002ffaf2, 0x002ffef2, // 27640-27647
	0x003002f2, 0x003006f2, 0x00300af2, 0x00300ef2, 0x003012f2, 0x003016f2, 0x00301af2, 0x00301ef2, // 27648-27655
	0x003022f2, 0x003026f2, 0x00302af2, 0x00302ef2, 0x003032f2, 0x003036f2, 0x00303af2, 0x00303ef2, // 27656-27663
	0x003042f2, 0x003046f2, 0x00304af2, 0x00304ef2, 0x003052f2, 0x003056f2, 0x00305af2, 0x00305ef2, // 27664-27671
	0x003062f2, 0x003066f2, 0x00306af2, 0x00306ef2, 0x003072f2, 0x003076f2, 0x00307af2, 0x00307ef2, // 27672-27679
	0x003082f2, 0x003086f2, 0x00308af2, 0x00308ef2, 0x003092f2, 0x003096f2, 0x00309af2, 0x00309ef2, // 27680-27687
	0x0030a2f2, 0x0030a6f2, 0x0030aaf2, 0x0030aef2, 0x0030b2f2, 0x0030b6f2, 0x0030baf2, 0x0030bef2, // 27688-27695
	0x0030c2f2, 0x0030c6f2, 0x0030caf2, 0x0030cef2, 0x0030d2f2, 0x0030d6f2, 0x0030daf2, 0x0030def2, // 27696-27703
	0x0030e2f2, 0x0030e6f2, 0x0030eaf2, 0x0030eef2, 0x0030f2f2, 0x0030f6f2, 0x0030faf2, 0x0030fef2, // 27704-27711
	0x003102f2, 0x003106f2, 0x00310af2, 0x00310ef2, 0x003112f2, 0x003116f2, 0x00311af2, 0x00311ef2, // 27712-27719
	0x003122f2, 0x003126f2, 0x00312af2, 0x00312ef2, 0x003132f2, 0x003136f2, 0x00313af2, 0x00313ef2, // 27720-27727
	0x003142f2, 0x003146f2, 0x00314af2, 0x00314ef2, 0x003152f2, 0x003156f2, 0x00315af2, 0x00315ef2, // 27728-27735
	0x003162f2, 0x003166f2, 0x00316af2, 0x00316ef2, 0x003172f2, 0x003176f2, 0x00317af2, 0x00317ef2, // 27736-27743
	0x003182f2, 0x003186f2, 0x00318af2, 0x00318ef2, 0x003192f2, 0x003196f2, 0x00319af2, 0x00319ef2, // 27744-27751
	0x0031a2f2, 0x0031a6f2, 0x0031aaf2, 0x0031aef2, 0x0031b2f2, 0x0031b6f2, 0x0031baf2, 0x0031bef2, // 27752-27759
	0x0031c2f2, 0x0031c6f2, 0x0031caf2, 0x0031cef2, 0x0031d2f2, 0x0031d6f2, 0x0031daf2, 0x0031def2, // 27760-27767
	0x0031e2f2, 0x0031e6f2, 0x0031eaf2, 0x0031eef2, 0x0031f2f2, 0x0031f6f2, 0x0031faf2, 0x0031fef2, // 27768-27775
	0x003202f2, 0x003206f2, 0x00320af2, 0x00320ef2, 0x003212f2, 0x003216f2, 0x00321af2, 0x00321ef2, // 27776-27783
	0x003222f2, 0x003226f2, 0x00322af2, 0x00322ef2, 0x003232f2, 0x003236f2, 0x00323af2, 0x00323ef2, // 27784-27791
	0x003242f2, 0x003246f2, 0x00324af2, 0x00324ef2, 0x003252f2, 0x003256f2, 0x00325af2, 0x00325ef2, // 27792-27799
	0x003262f2, 0x003266f2, 0x00326af2, 0x00326ef2, 0x003272f2, 0x003276f2, 0x00327af2, 0x00327ef2, // 27800-27807
	0x003282f2, 0x003286f2, 0x00328af2, 0x00328ef2, 0x003292f2, 0x003296f2, 0x00329af2, 0x00329ef2, // 27808-27815
	0x0032a2f2, 0x0032a6f2, 0x0032aaf2, 0x0032aef2, 0x0032b2f2, 0x0032b6f2, 0x0032baf2, 0x0032bef2, // 27816-27823
	0x0032c2f2, 0x0032c6f2, 0x0032caf2, 0x0032cef2, 0x0032d2f2, 0x0032d6f2, 0x0032daf2, 0x0032def2, // 27824-27831
	0x0032e2f2, 0x0032e6f2, 0x0032eaf2, 0x0032eef2, 0x0032f2f2, 0x0032f6f2, 0x0032faf2, 0x0032fef2, // 27832-27839
	0x003302f2, 0x003306f2, 0x00330af2, 0x00330ef2, 0x003312f2, 0x003316f2, 0x00331af2, 0x00331ef2, // 27840-27847
	0x003322f2, 0x003326f2, 0x00332af2, 0x00332ef2, 0x003332f2, 0x003336f2, 0x00333af2, 0x00333ef2, // 27848-27855
	0x003342f2, 0x003346f2, 0x00334af2, 0x00334ef2, 0x003352f2, 0x003356f2, 0x00335af2, 0x00335ef2, // 27856-27863
	0x003362f2, 0x003366f2, 0x00336af2, 0x00336ef2, 0x003372f2, 0x003376f2, 0x00337af2, 0x00337ef2, // 27864-27871
	0x003382f2, 0x003386f2, 0x00338af2, 0x00338ef2, 0x003392f2, 0x003396f2, 0x00339af2, 0x00339ef2, // 27872-27879
	0x0033a2f2, 0x0033a6f2, 0x0033aaf2, 0x0033aef2, 0x0033b2f2, 0x0033b6f2, 0x0033baf2, 0x0033bef2, // 27880-27887
	0x0033c2f2, 0x0033c6f2, 0x0033caf2, 0x0033cef2, 0x0033d2f2, 0x0033d6f2, 0x0033daf2, 0x0033def2, // 27888-27895
	0x0033e2f2, 0x0033e6f2, 0x0033eaf2, 0x0033eef2, 0x0033f2f2, 0x0033f6f2, 0x0033faf2, 0x0033fef2, // 27896-27903
	0x003402f2, 0x003406f2, 0x00340af2, 0x00340ef2, 0x003412f2, 0x003416f2, 0x00341af2, 0x00341ef2, // 27904-27911
	0x003422f2, 0x003426f2, 0x00342af2, 0x00342ef2, 0x003432f2, 0x003436f2, 0x00343af2, 0x00343ef2, // 27912-27919
	0x003442f2, 0x003446f2, 0x00344af2, 0x00344ef2, 0x003452f2, 0x003456f2, 0x00345af2, 0x00345ef2, // 27920-27927
	0x003462f2, 0x003466f2, 0x00346af2, 0x00346ef2, 0x003472f2, 0x003476f2, 0x00347af2, 0x00347ef2, // 27928-27935
	0x003482f2, 0x003486f2, 0x00348af2, 0x00348ef2, 0x003492f2, 0x003496f2, 0x00349af2, 0x00349ef2, // 27936-27943
	0x0034a2f2, 0x0034a6f2, 0x0034aaf2, 0x0034aef2, 0x0034b2f2, 0x0034b6f2, 0x0034baf2, 0x0034bef2, // 27944-27951
	0x0034c2f2, 0x0034c6f2, 0x0034caf2, 0x0034cef2, 0x0034d2f2, 0x0034d6f2, 0x0034daf2, 0x0034def2, // 27952-27959
	0x0034e2f2, 0x0034e6f2, 0x0034eaf2, 0x0034eef2, 0x0034f2f2, 0x0034f6f2, 0x0034faf2, 0x0034fef2, // 27960-27967
	0x003502f2, 0x003506f2, 0x00350af2, 0x00350ef2, 0x003512f2, 0x003516f2, 0x00351af2, 0x00351ef2, // 27968-27975
	0x003522f2, 0x003526f2, 0x00352af2, 0x00352ef2, 0x003532f2, 0x003536f2, 0x00353af2, 0x00353ef2, // 27976-27983
	0x003542f2, 0x003546f2, 0x00354af2, 0x00354ef2, 0x003552f2, 0x003556f2, 0x00355af2, 0x00355ef2, // 27984-27991
	0x003562f2, 0x003566f2, 0x00356af2, 0x00356ef2, 0x003572f2, 0x003576f2, 0x00357af2, 0x00357ef2, // 27992-27999
	0x003582f2, 0x003586f2, 0x00358af2, 0x00358ef2, 0x003592f2, 0x003596f2, 0x00359af2, 0x00359ef2, // 28000-28007
	0x0035a2f2, 0x0035a6f2, 0x0035aaf2, 0x0035aef2, 0x0035b2f2, 0x0035b6f2, 0x0035baf2, 0x0035bef2, // 28008-28015
	0x0035c2f2, 0x0035c6f2, 0x0035caf2, 0x0035cef2, 0x0035d2f2, 0x0035d6f2, 0x0035daf2, 0x0035def2, // 28016-28023
	0x0035e2f2, 0x0035e6f2, 0x0035eaf2, 0x0035eef2, 0x0035f2f2, 0x0035f6f2, 0x0035faf2, 0x0035fef2, // 28024-28031
	0x003602f2, 0x003606f2, 0x00360af2, 0x00360ef2, 0x003612f2, 0x003616f2, 0x00361af2, 0x00361ef2, // 28032-28039
	0x003622f2, 0x003626f2, 0x00362af2, 0x00362ef2, 0x003632f2, 0x003636f2, 0x00363af2, 0x00363ef2, // 28040-28047
	0x003642f2, 0x003646f2, 0x00364af2, 0x00364ef2, 0x003652f2, 0x003656f2, 0x00365af2, 0x00365ef2, // 28048-28055
	0x003662f2, 0x003666f2, 0x00366af2, 0x00366ef2, 0x003672f2, 0x003676f2, 0x00367af2, 0x00367ef2, // 28056-28063
	0x003682f2, 0x003686f2, 0x00368af2, 0x00368ef2, 0x003692f2, 0x003696f2, 0x00369af2, 0x00369ef2, // 28064-28071
	0x0036a2f2, 0x0036a6f2, 0x0036aaf2, 0x0036aef2, 0x0036b2f2, 0x0036b6f2, 0x0036baf2, 0x0036bef2, // 28072-28079
	0x0036c2f2, 0x0036c6f2, 0x0036caf2, 0x0036cef2, 0x0036d2f2, 0x0036d6f2, 0x0036daf2, 0x0036def2, // 28080-28087
	0x0036e2f2, 0x0036e6f2, 0x0036eaf2, 0x0036eef2, 0x0036f2f2, 0x0036f6f2, 0x0036faf2, 0x0036fef2, // 28088-28095
	0x003702f2, 0x003706f2, 0x00370af2, 0x00370ef2, 0x003712f2, 0x003716f2, 0x00371af2, 0x00371ef2, // 28096-28103
	0x003722f2, 0x003726f2, 0x00372af2, 0x00372ef2, 0x003732f2, 0x003736f2, 0x00373af2, 0x00373ef2, // 28104-28111
	0x003742f2, 0x003746f2, 0x00374af2, 0x00374ef2, 0x003752f2, 0x003756f2, 0x00375af2, 0x00375ef2, // 28112-28119
	0x003762f2, 0x003766f2, 0x00376af2, 0x00376ef2, 0x003772f2, 0x003776f2, 0x00377af2, 0x00377ef2, // 28120-28127
	0x003782f2, 0x003786f2, 0x00378af2, 0x00378ef2, 0x003792f2, 0x003796f2, 0x00379af2, 0x00379ef2, // 28128-28135
	0x0037a2f2, 0x0037a6f2, 0x0037aaf2, 0x0037aef2, 0x0037b2f2, 0x0037b6f2, 0x0037baf2, 0x0037bef2, // 28136-28143
	0x0037c2f2, 0x0037c6f2, 0x0037caf2, 0x0037cef2, 0x0037d2f2, 0x0037d6f2, 0x0037daf2, 0x0037def2, // 28144-28151
	0x0037e2f2, 0x0037e6f2, 0x0037eaf2, 0x0037eef2, 0x0037f2f2, 0x0037f6f2, 0x0037faf2, 0x0037fef2, // 28152-28159
	0x003802f2, 0x003806f2, 0x00380af2, 0x00380ef2, 0x003812f2, 0x003816f2, 0x00381af2, 0x00381ef2, // 28160-28167
	0x003822f2, 0x003826f2, 0x00382af2, 0x00382ef2, 0x003832f2, 0x003836f2, 0x00383af2, 0x00383ef2, // 28168-28175
	0x003842f2, 0x003846f2, 0x00384af2, 0x00384ef2, 0x003852f2, 0x003856f2, 0x00385af2, 0x00385ef2, // 28176-28183
	0x003862f2, 0x003866f2, 0x00386af2, 0x00386ef2, 0x003872f2, 0x003876f2, 0x00387af2, 0x00387ef2, // 28184-28191
	0x003882f2, 0x003886f2, 0x00388af2, 0x00388ef2, 0x003892f2, 0x003896f2, 0x00389af2, 0x00389ef2, // 28192-28199
	0x0038a2f2, 0x0038a6f2, 0x0038aaf2, 0x0038aef2, 0x0038b2f2, 0x0038b6f2, 0x0038baf2, 0x0038bef2, // 28200-28207
	0x0038c2f2, 0x0038c6f2, 0x0038caf2, 0x0038cef2, 0x0038d2f2, 0x0038d6f2, 0x0038daf2, 0x0038def2, // 28208-28215
	0x0038e2f2, 0x0038e6f2, 0x0038eaf2, 0x0038eef2, 0x0038f2f2, 0x0038f6f2, 0x0038faf2, 0x0038fef2, // 28216-28223
	0x003902f2, 0x003906f2, 0x00390af2, 0x00390ef2, 0x003912f2, 0x003916f2, 0x00391af2, 0x00391ef2, // 28224-28231
	0x003922f2, 0x003926f2, 0x00392af2, 0x00392ef2, 0x003932f2, 0x003936f2, 0x00393af2, 0x00393ef2, // 28232-28239
	0x003942f2, 0x003946f2, 0x00394af2, 0x00394ef2, 0x003952f2, 0x003956f2, 0x00395af2, 0x00395ef2, // 28240-28247
	0x003962f2, 0x003966f2, 0x00396af2, 0x00396ef2, 0x003972f2, 0x003976f2, 0x00397af2, 0x00397ef2, // 28248-28255
	0x003982f2, 0x003986f2, 0x00398af2, 0x00398ef2, 0x003992f2, 0x003996f2, 0x00399af2, 0x00399ef2, // 28256-28263
	0x0039a2f2, 0x0039a6f2, 0x0039aaf2, 0x0039aef2, 0x0039b2f2, 0x0039b6f2, 0x0039baf2, 0x0039bef2, // 28264-28271
	0x0039c2f2, 0x0039c6f2, 0x0039caf2, 0x0039cef2, 0x0039d2f2, 0x0039d6f2, 0x0039daf2, 0x0039def2, // 28272-28279
	0x0039e2f2, 0x0039e6f2, 0x0039eaf2, 0x0039eef2, 0x0039f2f2, 0x0039f6f2, 0x0039faf2, 0x0039fef2, // 28280-28287
	0x003a02f2, 0x003a06f2, 0x003a0af2, 0x003a0ef2, 0x003a12f2, 0x003a16f2, 0x003a1af2, 0x003a1ef2, // 28288-28295
	0x003a22f2, 0x003a26f2, 0x003a2af2, 0x003a2ef2, 0x003a32f2, 0x003a36f2, 0x003a3af2, 0x003a3ef2, // 28296-28303
	0x003a42f2, 0x003a46f2, 0x003a4af2, 0x003a4ef2, 0x003a52f2, 0x003a56f2, 0x003a5af2, 0x003a5ef2, // 28304-28311
	0x003a62f2, 0x003a66f2, 0x003a6af2, 0x003a6ef2, 0x003a72f2, 0x003a76f2, 0x003a7af2, 0x003a7ef2, // 28312-28319
	0x003a82f2, 0x003a86f2, 0x003a8af2, 0x003a8ef2, 0x003a92f2, 0x003a96f2, 0x003a9af2, 0x003a9ef2, // 28320-28327
	0x003aa2f2, 0x003aa6f2, 0x003aaaf2, 0x003aaef2, 0x003ab2f2, 0x003ab6f2, 0x003abaf2, 0x003abef2, // 28328-28335
	0x003ac2f2, 0x003ac6f2, 0x003acaf2, 0x003acef2, 0x003ad2f2, 0x003ad6f2, 0x003adaf2, 0x003adef2, // 28336-28343
	0x003ae2f2, 0x003ae6f2, 0x003aeaf2, 0x003aeef2, 0x003af2f2, 0x003af6f2, 0x003afaf2, 0x003afef2, // 28344-28351
	0x003b02f2, 0x003b06f2, 0x003b0af2, 0x003b0ef2, 0x003b12f2, 0x003b16f2, 0x003b1af2, 0x003b1ef2, // 28352-28359
	0x003b22f2, 0x003b26f2, 0x003b2af2, 0x003b2ef2, 0x003b32f2, 0x003b36f2, 0x003b3af2, 0x003b3ef2, // 28360-28367
	0x003b42f2, 0x003b46f2, 0x003b4af2, 0x003b4ef2, 0x003b52f2, 0x003b56f2, 0x003b5af2, 0x003b5ef2, // 28368-28375
	0x003b62f2, 0x003b66f2, 0x003b6af2, 0x003b6ef2, 0x003b72f2, 0x003b76f2, 0x003b7af2, 0x003b7ef2, // 28376-28383
	0x003b82f2, 0x003b86f2, 0x003b8af2, 0x003b8ef2, 0x003b92f2, 0x003b96f2, 0x003b9af2, 0x003b9ef2, // 28384-28391
	0x003ba2f2, 0x003ba6f2, 0x003baaf2, 0x003baef2, 0x003bb2f2, 0x003bb6f2, 0x003bbaf2, 0x003bbef2, // 28392-28399
	0x003bc2f2, 0x003bc6f2, 0x003bcaf2, 0x003bcef2, 0x003bd2f2, 0x003bd6f2, 0x003bdaf2, 0x003bdef2, // 28400-28407
	0x003be2f2, 0x003be6f2, 0x003beaf2, 0x003beef2, 0x003bf2f2, 0x003bf6f2, 0x003bfaf2, 0x003bfef2, // 28408-28415
	0x003c02f2, 0x003c06f2, 0x003c0af2, 0x003c0ef2, 0x003c12f2, 0x003c16f2, 0x003c1af2, 0x003c1ef2, // 28416-28423
	0x003c22f2, 0x003c26f2, 0x003c2af2, 0x003c2ef2, 0x003c32f2, 0x003c36f2, 0x003c3af2, 0x003c3ef2, // 28424-28431
	0x003c42f2, 0x003c46f2, 0x003c4af2, 0x003c4ef2, 0x003c52f2, 0x003c56f2, 0x003c5af2, 0x003c5ef2, // 28432-28439
	0x003c62f2, 0x003c66f2, 0x003c6af2, 0x003c6ef2, 0x003c72f2, 0x003c76f2, 0x003c7af2, 0x003c7ef2, // 28440-28447
	0x003c82f2, 0x003c86f2, 0x003c8af2, 0x003c8ef2, 0x003c92f2, 0x003c96f2, 0x003c9af2, 0x003c9ef2, // 28448-28455
	0x003ca2f2, 0x003ca6f2, 0x003caaf2, 0x003caef2, 0x003cb2f2, 0x003cb6f2, 0x003cbaf2, 0x003cbef2, // 28456-28463
	0x003cc2f2, 0x003cc6f2, 0x003ccaf2, 0x003ccef2, 0x003cd2f2, 0x003cd6f2, 0x003cdaf2, 0x003cdef2, // 28464-28471
	0x003ce2f2, 0x003ce6f2, 0x003ceaf2, 0x003ceef2, 0x003cf2f2, 0x003cf6f2, 0x003cfaf2, 0x003cfef2, // 28472-28479
	0x003d02f2, 0x003d06f2, 0x003d0af2, 0x003d0ef2, 0x003d12f2, 0x003d16f2, 0x003d1af2, 0x003d1ef2, // 28480-28487
	0x003d22f2, 0x003d26f2, 0x003d2af2, 0x003d2ef2, 0x003d32f2, 0x003d36f2, 0x003d3af2, 0x003d3ef2, // 28488-28495
	0x003d42f2, 0x003d46f2, 0x003d4af2, 0x003d4ef2, 0x003d52f2, 0x003d56f2, 0x003d5af2, 0x003d5ef2, // 28496-28503
	0x003d62f2, 0x003d66f2, 0x003d6af2, 0x003d6ef2, 0x003d72f2, 0x003d76f2, 0x003d7af2, 0x003d7ef2, // 28504-28511
	0x003d82f2, 0x003d86f2, 0x003d8af2, 0x003d8ef2, 0x003d92f2, 0x003d96f2, 0x003d9af2, 0x003d9ef2, // 28512-28519
	0x003da2f2, 0x003da6f2, 0x003daaf2, 0x003daef2, 0x003db2f2, 0x003db6f2, 0x003dbaf2, 0x003dbef2, // 28520-28527
	0x003dc2f2, 0x003dc6f2, 0x003dcaf2, 0x003dcef2, 0x003dd2f2, 0x003dd6f2, 0x003ddaf2, 0x003ddef2, // 28528-28535
	0x003de2f2, 0x003de6f2, 0x003deaf2, 0x003deef2, 0x003df2f2, 0x003df6f2, 0x003dfaf2, 0x003dfef2, // 28536-28543
	0x003e02f2, 0x003e06f2, 0x003e0af2, 0x003e0ef2, 0x003e12f2, 0x003e16f2, 0x003e1af2, 0x003e1ef2, // 28544-28551
	0x003e22f2, 0x003e26f2, 0x003e2af2, 0x003e2ef2, 0x003e32f2, 0x003e36f2, 0x003e3af2, 0x003e3ef2, // 28552-28559
	0x003e42f2, 0x003e46f2, 0x003e4af2, 0x003e4ef2, 0x003e52f2, 0x003e56f2, 0x003e5af2, 0x003e5ef2, // 28560-28567
	0x003e62f2, 0x003e66f2, 0x003e6af2, 0x003e6ef2, 0x003e72f2, 0x003e76f2, 0x003e7af2, 0x003e7ef2, // 28568-28575
	0x003e82f2, 0x003e86f2, 0x003e8af2, 0x003e8ef2, 0x003e92f2, 0x003e96f2, 0x003e9af2, 0x003e9ef2, // 28576-28583
	0x003ea2f2, 0x003ea6f2, 0x003eaaf2, 0x003eaef2, 0x003eb2f2, 0x003eb6f2, 0x003ebaf2, 0x003ebef2, // 28584-28591
	0x003ec2f2, 0x003ec6f2, 0x003ecaf2, 0x003ecef2, 0x003ed2f2, 0x003ed6f2, 0x003edaf2, 0x003edef2, // 28592-28599
	0x003ee2f2, 0x003ee6f2, 0x003eeaf2, 0x003eeef2, 0x003ef2f2, 0x003ef6f2, 0x003efaf2, 0x003efef2, // 28600-28607
	0x003f02f2, 0x003f06f2, 0x003f0af2, 0x003f0ef2, 0x003f12f2, 0x003f16f2, 0x003f1af2, 0x003f1ef2, // 28608-28615
	0x003f22f2, 0x003f26f2, 0x003f2af2, 0x003f2ef2, 0x003f32f2, 0x003f36f2, 0x003f3af2, 0x003f3ef2, // 28616-28623
	0x003f42f2, 0x003f46f2, 0x003f4af2, 0x003f4ef2, 0x003f52f2, 0x003f56f2, 0x003f5af2, 0x003f5ef2, // 28624-28631
	0x003f62f2, 0x003f66f2, 0x003f6af2, 0x003f6ef2, 0x003f72f2, 0x003f76f2, 0x003f7af2, 0x003f7ef2, // 28632-28639
	0x003f82f2, 0x003f86f2, 0x003f8af2, 0x003f8ef2, 0x003f92f2, 0x003f96f2, 0x003f9af2, 0x003f9ef2, // 28640-28647
	0x003fa2f2, 0x003fa6f2, 0x003faaf2, 0x003faef2, 0x003fb2f2, 0x003fb6f2, 0x003fbaf2, 0x003fbef2, // 28648-28655
	0x003fc2f2, 0x003fc6f2, 0x003fcaf2, 0x003fcef2, 0x003fd2f2, 0x003fd6f2, 0x003fdaf2, 0x003fdef2, // 28656-28663
	0x003fe2f2, 0x003fe6f2, 0x003feaf2, 0x003feef2, 0x003ff2f2, 0x003ff6f2, 0x003ffaf2, 0x003ffef2, // 28664-28671
	0x004002f2, 0x004006f2, 0x00400af2, 0x00400ef2, 0x004012f2, 0x004016f2, 0x00401af2, 0x00401ef2, // 28672-28679
	0x004022f2, 0x004026f2, 0x00402af2, 0x00402ef2, 0x004032f2, 0x004036f2, 0x00403af2, 0x00403ef2, // 28680-28687
	0x004042f2, 0x004046f2, 0x00404af2, 0x00404ef2, 0x004052f2, 0x004056f2, 0x00405af2, 0x00405ef2, // 28688-28695
	0x004062f2, 0x004066f2, 0x00406af2, 0x00406ef2, 0x004072f2, 0x004076f2, 0x00407af2, 0x00407ef2, // 28696-28703
	0x004082f2, 0x004086f2, 0x00408af2, 0x00408ef2, 0x004092f2, 0x004096f2, 0x00409af2, 0x00409ef2, // 28704-28711
	0x0040a2f2, 0x0040a6f2, 0x0040aaf2, 0x0040aef2, 0x0040b2f2, 0x0040b6f2, 0x0040baf2, 0x0040bef2, // 28712-28719
	0x0040c2f2, 0x0040c6f2, 0x0040caf2, 0x0040cef2, 0x0040d2f2, 0x0040d6f2, 0x0040daf2, 0x0040def2, // 28720-28727
	0x0040e2f2, 0x0040e6f2, 0x0040eaf2, 0x0040eef2, 0x0040f2f2, 0x0040f6f2, 0x0040faf2, 0x0040fef2, // 28728-28735
	0x004102f2, 0x004106f2, 0x00410af2, 0x00410ef2, 0x004112f2, 0x004116f2, 0x00411af2, 0x00411ef2, // 28736-28743
	0x004122f2, 0x004126f2, 0x00412af2, 0x00412ef2, 0x004132f2, 0x004136f2, 0x00413af2, 0x00413ef2, // 28744-28751
	0x004142f2, 0x004146f2, 0x00414af2, 0x00414ef2, 0x004152f2, 0x004156f2, 0x00415af2, 0x00415ef2, // 28752-28759
	0x004162f2, 0x004166f2, 0x00416af2, 0x00416ef2, 0x004172f2, 0x004176f2, 0x00417af2, 0x00417ef2, // 28760-28767
	0x004182f2, 0x004186f2, 0x00418af2, 0x00418ef2, 0x004192f2, 0x004196f2, 0x00419af2, 0x00419ef2, // 28768-28775
	0x0041a2f2, 0x0041a6f2, 0x0041aaf2, 0x0041aef2, 0x0041b2f2, 0x0041b6f2, 0x0041baf2, 0x0041bef2, // 28776-28783
	0x0041c2f2, 0x0041c6f2, 0x0041caf2, 0x0041cef2, 0x0041d2f2, 0x0041d6f2, 0x0041daf2, 0x0041def2, // 28784-28791
	0x0041e2f2, 0x0041e6f2, 0x0041eaf2, 0x0041eef2, 0x0041f2f2, 0x0041f6f2, 0x0041faf2, 0x0041fef2, // 28792-28799
	0x004202f2, 0x004206f2, 0x00420af2, 0x00420ef2, 0x004212f2, 0x004216f2, 0x00421af2, 0x00421ef2, // 28800-28807
	0x004222f2, 0x004226f2, 0x00422af2, 0x00422ef2, 0x004232f2, 0x004236f2, 0x00423af2, 0x00423ef2, // 28808-28815
	0x004242f2, 0x004246f2, 0x00424af2, 0x00424ef2, 0x004252f2, 0x004256f2, 0x00425af2, 0x00425ef2, // 28816-28823
	0x004262f2, 0x004266f2, 0x00426af2, 0x00426ef2, 0x004272f2, 0x004276f2, 0x00427af2, 0x00427ef2, // 28824-28831
	0x004282f2, 0x004286f2, 0x00428af2, 0x00428ef2, 0x004292f2, 0x004296f2, 0x00429af2, 0x00429ef2, // 28832-28839
	0x0042a2f2, 0x0042a6f2, 0x0042aaf2, 0x0042aef2, 0x0042b2f2, 0x0042b6f2, 0x0042baf2, 0x0042bef2, // 28840-28847
	0x0042c2f2, 0x0042c6f2, 0x0042caf2, 0x0042cef2, 0x0042d2f2, 0x0042d6f2, 0x0042daf2, 0x0042def2, // 28848-28855
	0x0042e2f2, 0x0042e6f2, 0x0042eaf2, 0x0042eef2, 0x0042f2f2, 0x0042f6f2, 0x0042faf2, 0x0042fef2, // 28856-28863
	0x004302f2, 0x004306f2, 0x00430af2, 0x00430ef2, 0x004312f2, 0x004316f2, 0x00431af2, 0x00431ef2, // 28864-28871
	0x004322f2, 0x004326f2, 0x00432af2, 0x00432ef2, 0x004332f2, 0x004336f2, 0x00433af2, 0x00433ef2, // 28872-28879
	0x004342f2, 0x004346f2, 0x00434af2, 0x00434ef2, 0x004352f2, 0x004356f2, 0x00435af2, 0x00435ef2, // 28880-28887
	0x004362f2, 0x004366f2, 0x00436af2, 0x00436ef2, 0x004372f2, 0x004376f2, 0x00437af2, 0x00437ef2, // 28888-28895
	0x004382f2, 0x004386f2, 0x00438af2, 0x00438ef2, 0x004392f2, 0x004396f2, 0x00439af2, 0x00439ef2, // 28896-28903
	0x0043a2f2, 0x0043a6f2, 0x0043aaf2, 0x0043aef2, 0x0043b2f2, 0x0043b6f2, 0x0043baf2, 0x0043bef2, // 28904-28911
	0x0043c2f2, 0x0043c6f2, 0x0043caf2, 0x0043cef2, 0x0043d2f2, 0x0043d6f2, 0x0043daf2, 0x0043def2, // 28912-28919
	0x0043e2f2, 0x0043e6f2, 0x0043eaf2, 0x0043eef2, 0x0043f2f2, 0x0043f6f2, 0x0043faf2, 0x0043fef2, // 28920-28927
	0x004402f2, 0x004406f2, 0x00440af2, 0x00440ef2, 0x004412f2, 0x004416f2, 0x00441af2, 0x00441ef2, // 28928-28935
	0x004422f2, 0x004426f2, 0x00442af2, 0x00442ef2, 0x004432f2, 0x004436f2, 0x00443af2, 0x00443ef2, // 28936-28943
	0x004442f2, 0x004446f2, 0x00444af2, 0x00444ef2, 0x004452f2, 0x004456f2, 0x00445af2, 0x00445ef2, // 28944-28951
	0x004462f2, 0x004466f2, 0x00446af2, 0x00446ef2, 0x004472f2, 0x004476f2, 0x00447af2, 0x00447ef2, // 28952-28959
	0x004482f2, 0x004486f2, 0x00448af2, 0x00448ef2, 0x004492f2, 0x004496f2, 0x00449af2, 0x00449ef2, // 28960-28967
	0x0044a2f2, 0x0044a6f2, 0x0044aaf2, 0x0044aef2, 0x0044b2f2, 0x0044b6f2, 0x0044baf2, 0x0044bef2, // 28968-28975
	0x0044c2f2, 0x0044c6f2, 0x0044caf2, 0x0044cef2, 0x0044d2f2, 0x0044d6f2, 0x0044daf2, 0x0044def2, // 28976-28983
	0x0044e2f2, 0x0044e6f2, 0x0044eaf2, 0x0044eef2, 0x0044f2f2, 0x0044f6f2, 0x0044faf2, 0x0044fef2, // 28984-28991
	0x004502f2, 0x004506f2, 0x00450af2, 0x00450ef2, 0x004512f2, 0x004516f2, 0x00451af2, 0x00451ef2, // 28992-28999
	0x004522f2, 0x004526f2, 0x00452af2, 0x00452ef2, 0x004532f2, 0x004536f2, 0x00453af2, 0x00453ef2, // 29000-29007
	0x004542f2, 0x004546f2, 0x00454af2, 0x00454ef2, 0x004552f2, 0x004556f2, 0x00455af2, 0x00455ef2, // 29008-29015
	0x004562f2, 0x004566f2, 0x00456af2, 0x00456ef2, 0x004572f2, 0x004576f2, 0x00457af2, 0x00457ef2, // 29016-29023
	0x004582f2, 0x004586f2, 0x00458af2, 0x00458ef2, 0x004592f2, 0x004596f2, 0x00459af2, 0x00459ef2, // 29024-29031
	0x0045a2f2, 0x0045a6f2, 0x0045aaf2, 0x0045aef2, 0x0045b2f2, 0x0045b6f2, 0x0045baf2, 0x0045bef2, // 29032-29039
	0x0045c2f2, 0x0045c6f2, 0x0045caf2, 0x0045cef2, 0x0045d2f2, 0x0045d6f2, 0x0045daf2, 0x0045def2, // 29040-29047
	0x0045e2f2, 0x0045e6f2, 0x0045eaf2, 0x0045eef2, 0x0045f2f2, 0x0045f6f2, 0x0045faf2, 0x0045fef2, // 29048-29055
	0x004602f2, 0x004606f2, 0x00460af2, 0x00460ef2, 0x004612f2, 0x004616f2, 0x00461af2, 0x00461ef2, // 29056-29063
	0x004622f2, 0x004626f2, 0x00462af2, 0x00462ef2, 0x004632f2, 0x004636f2, 0x00463af2, 0x00463ef2, // 29064-29071
	0x004642f2, 0x004646f2, 0x00464af2, 0x00464ef2, 0x004652f2, 0x004656f2, 0x00465af2, 0x00465ef2, // 29072-29079
	0x004662f2, 0x004666f2, 0x00466af2, 0x00466ef2, 0x004672f2, 0x004676f2, 0x00467af2, 0x00467ef2, // 29080-29087
	0x004682f2, 0x004686f2, 0x00468af2, 0x00468ef2, 0x004692f2, 0x004696f2, 0x00469af2, 0x00469ef2, // 29088-29095
	0x0046a2f2, 0x0046a6f2, 0x0046aaf2, 0x0046aef2, 0x0046b2f2, 0x0046b6f2, 0x0046baf2, 0x0046bef2, // 29096-29103
	0x0046c2f2, 0x0046c6f2, 0x0046caf2, 0x0046cef2, 0x0046d2f2, 0x0046d6f2, 0x0046daf2, 0x0046def2, // 29104-29111
	0x0046e2f2, 0x0046e6f2, 0x0046eaf2, 0x0046eef2, 0x0046f2f2, 0x0046f6f2, 0x0046faf2, 0x0046fef2, // 29112-29119
	0x004702f2, 0x004706f2, 0x00470af2, 0x00470ef2, 0x004712f2, 0x004716f2, 0x00471af2, 0x00471ef2, // 29120-29127
	0x004722f2, 0x004726f2, 0x00472af2, 0x00472ef2, 0x004732f2, 0x004736f2, 0x00473af2, 0x00473ef2, // 29128-29135
	0x004742f2, 0x004746f2, 0x00474af2, 0x00474ef2, 0x004752f2, 0x004756f2, 0x00475af2, 0x00475ef2, // 29136-29143
	0x004762f2, 0x004766f2, 0x00476af2, 0x00476ef2, 0x004772f2, 0x004776f2, 0x00477af2, 0x00477ef2, // 29144-29151
	0x004782f2, 0x004786f2, 0x00478af2, 0x00478ef2, 0x004792f2, 0x004796f2, 0x00479af2, 0x00479ef2, // 29152-29159
	0x0047a2f2, 0x0047a6f2, 0x0047aaf2, 0x0047aef2, 0x0047b2f2, 0x0047b6f2, 0x0047baf2, 0x0047bef2, // 29160-29167
	0x0047c2f2, 0x0047c6f2, 0x0047caf2, 0x0047cef2, 0x0047d2f2, 0x0047d6f2, 0x0047daf2, 0x0047def2, // 29168-29175
	0x0047e2f2, 0x0047e6f2, 0x0047eaf2, 0x0047eef2, 0x0047f2f2, 0x0047f6f2, 0x0047faf2, 0x0047fef2, // 29176-29183
	0x004802f2, 0x004806f2, 0x00480af2, 0x00480ef2, 0x004812f2, 0x004816f2, 0x00481af2, 0x00481ef2, // 29184-29191
	0x004822f2, 0x004826f2, 0x00482af2, 0x00482ef2, 0x004832f2, 0x004836f2, 0x00483af2, 0x00483ef2, // 29192-29199
	0x004842f2, 0x004846f2, 0x00484af2, 0x00484ef2, 0x004852f2, 0x004856f2, 0x00485af2, 0x00485ef2, // 29200-29207
	0x004862f2, 0x004866f2, 0x00486af2, 0x00486ef2, 0x004872f2, 0x004876f2, 0x00487af2, 0x00487ef2, // 29208-29215
	0x004882f2, 0x004886f2, 0x00488af2, 0x00488ef2, 0x004892f2, 0x004896f2, 0x00489af2, 0x00489ef2, // 29216-29223
	0x0048a2f2, 0x0048a6f2, 0x0048aaf2, 0x0048aef2, 0x0048b2f2, 0x0048b6f2, 0x0048baf2, 0x0048bef2, // 29224-29231
	0x0048c2f2, 0x0048c6f2, 0x0048caf2, 0x0048cef2, 0x0048d2f2, 0x0048d6f2, 0x0048daf2, 0x0048def2, // 29232-29239
	0x0048e2f2, 0x0048e6f2, 0x0048eaf2, 0x0048eef2, 0x0048f2f2, 0x0048f6f2, 0x0048faf2, 0x0048fef2, // 29240-29247
	0x004902f2, 0x004906f2, 0x00490af2, 0x00490ef2, 0x004912f2, 0x004916f2, 0x00491af2, 0x00491ef2, // 29248-29255
	0x004922f2, 0x004926f2, 0x00492af2, 0x00492ef2, 0x004932f2, 0x004936f2, 0x00493af2, 0x00493ef2, // 29256-29263
	0x004942f2, 0x004946f2, 0x00494af2, 0x00494ef2, 0x004952f2, 0x004956f2, 0x00495af2, 0x00495ef2, // 29264-29271
	0x004962f2, 0x004966f2, 0x00496af2, 0x00496ef2, 0x004972f2, 0x004976f2, 0x00497af2, 0x00497ef2, // 29272-29279
	0x004982f2, 0x004986f2, 0x00498af2, 0x00498ef2, 0x004992f2, 0x004996f2, 0x00499af2, 0x00499ef2, // 29280-29287
	0x0049a2f2, 0x0049a6f2, 0x0049aaf2, 0x0049aef2, 0x0049b2f2, 0x0049b6f2, 0x0049baf2, 0x0049bef2, // 29288-29295
	0x0049c2f2, 0x0049c6f2, 0x0049caf2, 0x0049cef2, 0x0049d2f2, 0x0049d6f2, 0x0049daf2, 0x0049def2, // 29296-29303
	0x0049e2f2, 0x0049e6f2, 0x0049eaf2, 0x0049eef2, 0x0049f2f2, 0x0049f6f2, 0x0049faf2, 0x0049fef2, // 29304-29311
	0x004a02f2, 0x004a06f2, 0x004a0af2, 0x004a0ef2, 0x004a12f2, 0x004a16f2, 0x004a1af2, 0x004a1ef2, // 29312-29319
	0x004a22f2, 0x004a26f2, 0x004a2af2, 0x004a2ef2, 0x004a32f2, 0x004a36f2, 0x004a3af2, 0x004a3ef2, // 29320-29327
	0x004a42f2, 0x004a46f2, 0x004a4af2, 0x004a4ef2, 0x004a52f2, 0x004a56f2, 0x004a5af2, 0x004a5ef2, // 29328-29335
	0x004a62f2, 0x004a66f2, 0x004a6af2, 0x004a6ef2, 0x004a72f2, 0x004a76f2, 0x004a7af2, 0x004a7ef2, // 29336-29343
	0x004a82f2, 0x004a86f2, 0x004a8af2, 0x004a8ef2, 0x004a92f2, 0x004a96f2, 0x004a9af2, 0x004a9ef2, // 29344-29351
	0x004aa2f2, 0x004aa6f2, 0x004aaaf2, 0x004aaef2, 0x004ab2f2, 0x004ab6f2, 0x004abaf2, 0x004abef2, // 29352-29359
	0x004ac2f2, 0x004ac6f2, 0x004acaf2, 0x004acef2, 0x004ad2f2, 0x004ad6f2, 0x004adaf2, 0x004adef2, // 29360-29367
	0x004ae2f2, 0x004ae6f2, 0x004aeaf2, 0x004aeef2, 0x004af2f2, 0x004af6f2, 0x004afaf2, 0x004afef2, // 29368-29375
	0x004b02f2, 0x004b06f2, 0x004b0af2, 0x004b0ef2, 0x004b12f2, 0x004b16f2, 0x004b1af2, 0x004b1ef2, // 29376-29383
	0x004b22f2, 0x004b26f2, 0x004b2af2, 0x004b2ef2, 0x004b32f2, 0x004b36f2, 0x004b3af2, 0x004b3ef2, // 29384-29391
	0x004b42f2, 0x004b46f2, 0x004b4af2, 0x004b4ef2, 0x004b52f2, 0x004b56f2, 0x004b5af2, 0x004b5ef2, // 29392-29399
	0x004b62f2, 0x004b66f2, 0x004b6af2, 0x004b6ef2, 0x004b72f2, 0x004b76f2, 0x004b7af2, 0x004b7ef2, // 29400-29407
	0x004b82f2, 0x004b86f2, 0x004b8af2, 0x004b8ef2, 0x004b92f2, 0x004b96f2, 0x004b9af2, 0x004b9ef2, // 29408-29415
	0x004ba2f2, 0x004ba6f2, 0x004baaf2, 0x004baef2, 0x004bb2f2, 0x004bb6f2, 0x004bbaf2, 0x004bbef2, // 29416-29423
	0x004bc2f2, 0x004bc6f2, 0x004bcaf2, 0x004bcef2, 0x004bd2f2, 0x004bd6f2, 0x004bdaf2, 0x004bdef2, // 29424-29431
	0x004be2f2, 0x004be6f2, 0x004beaf2, 0x004beef2, 0x004bf2f2, 0x004bf6f2, 0x004bfaf2, 0x004bfef2, // 29432-29439
	0x004c02f2, 0x004c06f2, 0x004c0af2, 0x004c0ef2, 0x004c12f2, 0x004c16f2, 0x004c1af2, 0x004c1ef2, // 29440-29447
	0x004c22f2, 0x004c26f2, 0x004c2af2, 0x004c2ef2, 0x004c32f2, 0x004c36f2, 0x004c3af2, 0x004c3ef2, // 29448-29455
	0x004c42f2, 0x004c46f2, 0x004c4af2, 0x004c4ef2, 0x004c52f2, 0x004c56f2, 0x004c5af2, 0x004c5ef2, // 29456-29463
	0x004c62f2, 0x004c66f2, 0x004c6af2, 0x004c6ef2, 0x004c72f2, 0x004c76f2, 0x004c7af2, 0x004c7ef2, // 29464-29471
	0x004c82f2, 0x004c86f2, 0x004c8af2, 0x004c8ef2, 0x004c92f2, 0x004c96f2, 0x004c9af2, 0x004c9ef2, // 29472-29479
	0x004ca2f2, 0x004ca6f2, 0x004caaf2, 0x004caef2, 0x004cb2f2, 0x004cb6f2, 0x004cbaf2, 0x004cbef2, // 29480-29487
	0x004cc2f2, 0x004cc6f2, 0x004ccaf2, 0x004ccef2, 0x004cd2f2, 0x004cd6f2, 0x004cdaf2, 0x004cdef2, // 29488-29495
	0x004ce2f2, 0x004ce6f2, 0x004ceaf2, 0x004ceef2, 0x004cf2f2, 0x004cf6f2, 0x004cfaf2, 0x004cfef2, // 29496-29503
	0x004d02f2, 0x004d06f2, 0x004d0af2, 0x004d0ef2, 0x004d12f2, 0x004d16f2, 0x004d1af2, 0x004d1ef2, // 29504-29511
	0x004d22f2, 0x004d26f2, 0x004d2af2, 0x004d2ef2, 0x004d32f2, 0x004d36f2, 0x004d3af2, 0x004d3ef2, // 29512-29519
	0x004d42f2, 0x004d46f2, 0x004d4af2, 0x004d4ef2, 0x004d52f2, 0x004d56f2, 0x004d5af2, 0x004d5ef2, // 29520-29527
	0x004d62f2, 0x004d66f2, 0x004d6af2, 0x004d6ef2, 0x004d72f2, 0x004d76f2, 0x004d7af2, 0x004d7ef2, // 29528-29535
	0x004d82f2, 0x004d86f2, 0x004d8af2, 0x004d8ef2, 0x004d92f2, 0x004d96f2, 0x004d9af2, 0x004d9ef2, // 29536-29543
	0x004da2f2, 0x004da6f2, 0x004daaf2, 0x004daef2, 0x004db2f2, 0x004db6f2, 0x004dbaf2, 0x004dbef2, // 29544-29551
	0x004dc2f2, 0x004dc6f2, 0x004dcaf2, 0x004dcef2, 0x004dd2f2, 0x004dd6f2, 0x004ddaf2, 0x004ddef2, // 29552-29559
	0x004de2f2, 0x004de6f2, 0x004deaf2, 0x004deef2, 0x004df2f2, 0x004df6f2, 0x004dfaf2, 0x004dfef2, // 29560-29567
	0x004e02f2, 0x004e06f2, 0x004e0af2, 0x004e0ef2, 0x004e12f2, 0x004e16f2, 0x004e1af2, 0x004e1ef2, // 29568-29575
	0x004e22f2, 0x004e26f2, 0x004e2af2, 0x004e2ef2, 0x004e32f2, 0x004e36f2, 0x004e3af2, 0x004e3ef2, // 29576-29583
	0x004e42f2, 0x004e46f2, 0x004e4af2, 0x004e4ef2, 0x004e52f2, 0x004e56f2, 0x004e5af2, 0x004e5ef2, // 29584-29591
	0x004e62f2, 0x004e66f2, 0x004e6af2, 0x004e6ef2, 0x004e72f2, 0x004e76f2, 0x004e7af2, 0x004e7ef2, // 29592-29599
	0x004e82f2, 0x004e86f2, 0x004e8af2, 0x004e8ef2, 0x004e92f2, 0x004e96f2, 0x004e9af2, 0x004e9ef2, // 29600-29607
	0x004ea2f2, 0x004ea6f2, 0x004eaaf2, 0x004eaef2, 0x004eb2f2, 0x004eb6f2, 0x004ebaf2, 0x004ebef2, // 29608-29615
	0x004ec2f2, 0x004ec6f2, 0x004ecaf2, 0x004ecef2, 0x004ed2f2, 0x004ed6f2, 0x004edaf2, 0x004edef2, // 29616-29623
	0x004ee2f2, 0x004ee6f2, 0x004eeaf2, 0x004eeef2, 0x004ef2f2, 0x004ef6f2, 0x004efaf2, 0x004efef2, // 29624-29631
	0x004f02f2, 0x004f06f2, 0x004f0af2, 0x004f0ef2, 0x004f12f2, 0x004f16f2, 0x004f1af2, 0x004f1ef2, // 29632-29639
	0x004f22f2, 0x004f26f2, 0x004f2af2, 0x004f2ef2, 0x004f32f2, 0x004f36f2, 0x004f3af2, 0x004f3ef2, // 29640-29647
	0x004f42f2, 0x004f46f2, 0x004f4af2, 0x004f4ef2, 0x004f52f2, 0x004f56f2, 0x004f5af2, 0x004f5ef2, // 29648-29655
	0x004f62f2, 0x004f66f2, 0x004f6af2, 0x004f6ef2, 0x004f72f2, 0x004f76f2, 0x004f7af2, 0x004f7ef2, // 29656-29663
	0x004f82f2, 0x004f86f2, 0x004f8af2, 0x004f8ef2, 0x004f92f2, 0x004f96f2, 0x004f9af2, 0x004f9ef2, // 29664-29671
	0x004fa2f2, 0x004fa6f2, 0x004faaf2, 0x004faef2, 0x004fb2f2, 0x004fb6f2, 0x004fbaf2, 0x004fbef2, // 29672-29679
	0x004fc2f2, 0x004fc6f2, 0x004fcaf2, 0x004fcef2, 0x004fd2f2, 0x004fd6f2, 0x004fdaf2, 0x004fdef2, // 29680-29687
	0x004fe2f2, 0x004fe6f2, 0x004feaf2, 0x004feef2, 0x004ff2f2, 0x004ff6f2, 0x004ffaf2, 0x004ffef2, // 29688-29695
	0x005002f2, 0x005006f2, 0x00500af2, 0x00500ef2, 0x005012f2, 0x005016f2, 0x00501af2, 0x00501ef2, // 29696-29703
	0x005022f2, 0x005026f2, 0x00502af2, 0x00502ef2, 0x005032f2, 0x005036f2, 0x00503af2, 0x00503ef2, // 29704-29711
	0x005042f2, 0x005046f2, 0x00504af2, 0x00504ef2, 0x005052f2, 0x005056f2, 0x00505af2, 0x00505ef2, // 29712-29719
	0x005062f2, 0x005066f2, 0x00506af2, 0x00506ef2, 0x005072f2, 0x005076f2, 0x00507af2, 0x00507ef2, // 29720-29727
	0x005082f2, 0x005086f2, 0x00508af2, 0x00508ef2, 0x005092f2, 0x005096f2, 0x00509af2, 0x00509ef2, // 29728-29735
	0x0050a2f2, 0x0050a6f2, 0x0050aaf2, 0x0050aef2, 0x0050b2f2, 0x0050b6f2, 0x0050baf2, 0x0050bef2, // 29736-29743
	0x0050c2f2, 0x0050c6f2, 0x0050caf2, 0x0050cef2, 0x0050d2f2, 0x0050d6f2, 0x0050daf2, 0x0050def2, // 29744-29751
	0x0050e2f2, 0x0050e6f2, 0x0050eaf2, 0x0050eef2, 0x0050f2f2, 0x0050f6f2, 0x0050faf2, 0x0050fef2, // 29752-29759
	0x005102f2, 0x005106f2, 0x00510af2, 0x00510ef2, 0x005112f2, 0x005116f2, 0x00511af2, 0x00511ef2, // 29760-29767
	0x005122f2, 0x005126f2, 0x00512af2, 0x00512ef2, 0x005132f2, 0x005136f2, 0x00513af2, 0x00513ef2, // 29768-29775
	0x005142f2, 0x005146f2, 0x00514af2, 0x00514ef2, 0x005152f2, 0x005156f2, 0x00515af2, 0x00515ef2, // 29776-29783
	0x005162f2, 0x005166f2, 0x00516af2, 0x00516ef2, 0x005172f2, 0x005176f2, 0x00517af2, 0x00517ef2, // 29784-29791
	0x005182f2, 0x005186f2, 0x00518af2, 0x00518ef2, 0x005192f2, 0x005196f2, 0x00519af2, 0x00519ef2, // 29792-29799
	0x0051a2f2, 0x0051a6f2, 0x0051aaf2, 0x0051aef2, 0x0051b2f2, 0x0051b6f2, 0x0051baf2, 0x0051bef2, // 29800-29807
	0x0051c2f2, 0x0051c6f2, 0x0051caf2, 0x0051cef2, 0x0051d2f2, 0x0051d6f2, 0x0051daf2, 0x0051def2, // 29808-29815
	0x0051e2f2, 0x0051e6f2, 0x0051eaf2, 0x0051eef2, 0x0051f2f2, 0x0051f6f2, 0x0051faf2, 0x0051fef2, // 29816-29823
	0x005202f2, 0x005206f2, 0x00520af2, 0x00520ef2, 0x005212f2, 0x005216f2, 0x00521af2, 0x00521ef2, // 29824-29831
	0x005222f2, 0x005226f2, 0x00522af2, 0x00522ef2, 0x005232f2, 0x005236f2, 0x00523af2, 0x00523ef2, // 29832-29839
	0x005242f2, 0x005246f2, 0x00524af2, 0x00524ef2, 0x005252f2, 0x005256f2, 0x00525af2, 0x00525ef2, // 29840-29847
	0x005262f2, 0x005266f2, 0x00526af2, 0x00526ef2, 0x005272f2, 0x005276f2, 0x00527af2, 0x00527ef2, // 29848-29855
	0x005282f2, 0x005286f2, 0x00528af2, 0x00528ef2, 0x005292f2, 0x005296f2, 0x00529af2, 0x00529ef2, // 29856-29863
	0x0052a2f2, 0x0052a6f2, 0x0052aaf2, 0x0052aef2, 0x0052b2f2, 0x0052b6f2, 0x0052baf2, 0x0052bef2, // 29864-29871
	0x0052c2f2, 0x0052c6f2, 0x0052caf2, 0x0052cef2, 0x0052d2f2, 0x0052d6f2, 0x0052daf2, 0x0052def2, // 29872-29879
	0x0052e2f2, 0x0052e6f2, 0x0052eaf2, 0x0052eef2, 0x0052f2f2, 0x0052f6f2, 0x0052faf2, 0x0052fef2, // 29880-29887
	0x005302f2, 0x005306f2, 0x00530af2, 0x00530ef2, 0x005312f2, 0x005316f2, 0x00531af2, 0x00531ef2, // 29888-29895
	0x005322f2, 0x005326f2, 0x00532af2, 0x00532ef2, 0x005332f2, 0x005336f2, 0x00533af2, 0x00533ef2, // 29896-29903
	0x005342f2, 0x005346f2, 0x00534af2, 0x00534ef2, 0x005352f2, 0x005356f2, 0x00535af2, 0x00535ef2, // 29904-29911
	0x005362f2, 0x005366f2, 0x00536af2, 0x00536ef2, 0x005372f2, 0x005376f2, 0x00537af2, 0x00537ef2, // 29912-29919
	0x005382f2, 0x005386f2, 0x00538af2, 0x00538ef2, 0x005392f2, 0x005396f2, 0x00539af2, 0x00539ef2, // 29920-29927
	0x0053a2f2, 0x0053a6f2, 0x0053aaf2, 0x0053aef2, 0x0053b2f2, 0x0053b6f2, 0x0053baf2, 0x0053bef2, // 29928-29935
	0x0053c2f2, 0x0053c6f2, 0x0053caf2, 0x0053cef2, 0x0053d2f2, 0x0053d6f2, 0x0053daf2, 0x0053def2, // 29936-29943
	0x0053e2f2, 0x0053e6f2, 0x0053eaf2, 0x0053eef2, 0x0053f2f2, 0x0053f6f2, 0x0053faf2, 0x0053fef2, // 29944-29951
	0x005402f2, 0x005406f2, 0x00540af2, 0x00540ef2, 0x005412f2, 0x005416f2, 0x00541af2, 0x00541ef2, // 29952-29959
	0x005422f2, 0x005426f2, 0x00542af2, 0x00542ef2, 0x005432f2, 0x005436f2, 0x00543af2, 0x00543ef2, // 29960-29967
	0x005442f2, 0x005446f2, 0x00544af2, 0x00544ef2, 0x005452f2, 0x005456f2, 0x00545af2, 0x00545ef2, // 29968-29975
	0x005462f2, 0x005466f2, 0x00546af2, 0x00546ef2, 0x005472f2, 0x005476f2, 0x00547af2, 0x00547ef2, // 29976-29983
	0x005482f2, 0x005486f2, 0x00548af2, 0x00548ef2, 0x005492f2, 0x005496f2, 0x00549af2, 0x00549ef2, // 29984-29991
	0x0054a2f2, 0x0054a6f2, 0x0054aaf2, 0x0054aef2, 0x0054b2f2, 0x0054b6f2, 0x0054baf2, 0x0054bef2, // 29992-29999
	0x0054c2f2, 0x0054c6f2, 0x0054caf2, 0x0054cef2, 0x0054d2f2, 0x0054d6f2, 0x0054daf2, 0x0054def2, // 30000-30007
	0x0054e2f2, 0x0054e6f2, 0x0054eaf2, 0x0054eef2, 0x0054f2f2, 0x0054f6f2, 0x0054faf2, 0x0054fef2, // 30008-30015
	0x005502f2, 0x005506f2, 0x00550af2, 0x00550ef2, 0x005512f2, 0x005516f2, 0x00551af2, 0x00551ef2, // 30016-30023
	0x005522f2, 0x005526f2, 0x00552af2, 0x00552ef2, 0x005532f2, 0x005536f2, 0x00553af2, 0x00553ef2, // 30024-30031
	0x005542f2, 0x005546f2, 0x00554af2, 0x00554ef2, 0x005552f2, 0x005556f2, 0x00555af2, 0x00555ef2, // 30032-30039
	0x005562f2, 0x005566f2, 0x00556af2, 0x00556ef2, 0x005572f2, 0x005576f2, 0x00557af2, 0x00557ef2, // 30040-30047
	0x005582f2, 0x005586f2, 0x00558af2, 0x00558ef2, 0x005592f2, 0x005596f2, 0x00559af2, 0x00559ef2, // 30048-30055
	0x0055a2f2, 0x0055a6f2, 0x0055aaf2, 0x0055aef2, 0x0055b2f2, 0x0055b6f2, 0x0055baf2, 0x0055bef2, // 30056-30063
	0x0055c2f2, 0x0055c6f2, 0x0055caf2, 0x0055cef2, 0x0055d2f2, 0x0055d6f2, 0x0055daf2, 0x0055def2, // 30064-30071
	0x0055e2f2, 0x0055e6f2, 0x0055eaf2, 0x0055eef2, 0x0055f2f2, 0x0055f6f2, 0x0055faf2, 0x0055fef2, // 30072-30079
	0x005602f2, 0x005606f2, 0x00560af2, 0x00560ef2, 0x005612f2, 0x005616f2, 0x00561af2, 0x00561ef2, // 30080-30087
	0x005622f2, 0x005626f2, 0x00562af2, 0x00562ef2, 0x005632f2, 0x005636f2, 0x00563af2, 0x00563ef2, // 30088-30095
	0x005642f2, 0x005646f2, 0x00564af2, 0x00564ef2, 0x005652f2, 0x005656f2, 0x00565af2, 0x00565ef2, // 30096-30103
	0x005662f2, 0x005666f2, 0x00566af2, 0x00566ef2, 0x005672f2, 0x005676f2, 0x00567af2, 0x00567ef2, // 30104-30111
	0x005682f2, 0x005686f2, 0x00568af2, 0x00568ef2, 0x005692f2, 0x005696f2, 0x00569af2, 0x00569ef2, // 30112-30119
	0x0056a2f2, 0x0056a6f2, 0x0056aaf2, 0x0056aef2, 0x0056b2f2, 0x0056b6f2, 0x0056baf2, 0x0056bef2, // 30120-30127
	0x0056c2f2, 0x0056c6f2, 0x0056caf2, 0x0056cef2, 0x0056d2f2, 0x0056d6f2, 0x0056daf2, 0x0056def2, // 30128-30135
	0x0056e2f2, 0x0056e6f2, 0x0056eaf2, 0x0056eef2, 0x0056f2f2, 0x0056f6f2, 0x0056faf2, 0x0056fef2, // 30136-30143
	0x005702f2, 0x005706f2, 0x00570af2, 0x00570ef2, 0x005712f2, 0x005716f2, 0x00571af2, 0x00571ef2, // 30144-30151
	0x005722f2, 0x005726f2, 0x00572af2, 0x00572ef2, 0x005732f2, 0x005736f2, 0x00573af2, 0x00573ef2, // 30152-30159
	0x005742f2, 0x005746f2, 0x00574af2, 0x00574ef2, 0x005752f2, 0x005756f2, 0x00575af2, 0x00575ef2, // 30160-30167
	0x005762f2, 0x005766f2, 0x00576af2, 0x00576ef2, 0x005772f2, 0x005776f2, 0x00577af2, 0x00577ef2, // 30168-30175
	0x005782f2, 0x005786f2, 0x00578af2, 0x00578ef2, 0x005792f2, 0x005796f2, 0x00579af2, 0x00579ef2, // 30176-30183
	0x0057a2f2, 0x0057a6f2, 0x0057aaf2, 0x0057aef2, 0x0057b2f2, 0x0057b6f2, 0x0057baf2, 0x0057bef2, // 30184-30191
	0x0057c2f2, 0x0057c6f2, 0x0057caf2, 0x0057cef2, 0x0057d2f2, 0x0057d6f2, 0x0057daf2, 0x0057def2, // 30192-30199
	0x0057e2f2, 0x0057e6f2, 0x0057eaf2, 0x0057eef2, 0x0057f2f2, 0x0057f6f2, 0x0057faf2, 0x0057fef2, // 30200-30207
	0x005802f2, 0x005806f2, 0x00580af2, 0x00580ef2, 0x005812f2, 0x005816f2, 0x00581af2, 0x00581ef2, // 30208-30215
	0x005822f2, 0x005826f2, 0x00582af2, 0x00582ef2, 0x005832f2, 0x005836f2, 0x00583af2, 0x00583ef2, // 30216-30223
	0x005842f2, 0x005846f2, 0x00584af2, 0x00584ef2, 0x005852f2, 0x005856f2, 0x00585af2, 0x00585ef2, // 30224-30231
	0x005862f2, 0x005866f2, 0x00586af2, 0x00586ef2, 0x005872f2, 0x005876f2, 0x00587af2, 0x00587ef2, // 30232-30239
	0x005882f2, 0x005886f2, 0x00588af2, 0x00588ef2, 0x005892f2, 0x005896f2, 0x00589af2, 0x00589ef2, // 30240-30247
	0x0058a2f2, 0x0058a6f2, 0x0058aaf2, 0x0058aef2, 0x0058b2f2, 0x0058b6f2, 0x0058baf2, 0x0058bef2, // 30248-30255
	0x0058c2f2, 0x0058c6f2, 0x0058caf2, 0x0058cef2, 0x0058d2f2, 0x0058d6f2, 0x0058daf2, 0x0058def2, // 30256-30263
	0x0058e2f2, 0x0058e6f2, 0x0058eaf2, 0x0058eef2, 0x0058f2f2, 0x0058f6f2, 0x0058faf2, 0x0058fef2, // 30264-30271
	0x005902f2, 0x005906f2, 0x00590af2, 0x00590ef2, 0x005912f2, 0x005916f2, 0x00591af2, 0x00591ef2, // 30272-30279
	0x005922f2, 0x005926f2, 0x00592af2, 0x00592ef2, 0x005932f2, 0x005936f2, 0x00593af2, 0x00593ef2, // 30280-30287
	0x005942f2, 0x005946f2, 0x00594af2, 0x00594ef2, 0x005952f2, 0x005956f2, 0x00595af2, 0x00595ef2, // 30288-30295
	0x005962f2, 0x005966f2, 0x00596af2, 0x00596ef2, 0x005972f2, 0x005976f2, 0x00597af2, 0x00597ef2, // 30296-30303
	0x005982f2, 0x005986f2, 0x00598af2, 0x00598ef2, 0x005992f2, 0x005996f2, 0x00599af2, 0x00599ef2, // 30304-30311
	0x0059a2f2, 0x0059a6f2, 0x0059aaf2, 0x0059aef2, 0x0059b2f2, 0x0059b6f2, 0x0059baf2, 0x0059bef2, // 30312-30319
	0x0059c2f2, 0x0059c6f2, 0x0059caf2, 0x0059cef2, 0x0059d2f2, 0x0059d6f2, 0x0059daf2, 0x0059def2, // 30320-30327
	0x0059e2f2, 0x0059e6f2, 0x0059eaf2, 0x0059eef2, 0x0059f2f2, 0x0059f6f2, 0x0059faf2, 0x0059fef2, // 30328-30335
	0x005a02f2, 0x005a06f2, 0x005a0af2, 0x005a0ef2, 0x005a12f2, 0x005a16f2, 0x005a1af2, 0x005a1ef2, // 30336-30343
	0x005a22f2, 0x005a26f2, 0x005a2af2, 0x005a2ef2, 0x005a32f2, 0x005a36f2, 0x005a3af2, 0x005a3ef2, // 30344-30351
	0x005a42f2, 0x005a46f2, 0x005a4af2, 0x005a4ef2, 0x005a52f2, 0x005a56f2, 0x005a5af2, 0x005a5ef2, // 30352-30359
	0x005a62f2, 0x005a66f2, 0x005a6af2, 0x005a6ef2, 0x005a72f2, 0x005a76f2, 0x005a7af2, 0x005a7ef2, // 30360-30367
	0x005a82f2, 0x005a86f2, 0x005a8af2, 0x005a8ef2, 0x005a92f2, 0x005a96f2, 0x005a9af2, 0x005a9ef2, // 30368-30375
	0x005aa2f2, 0x005aa6f2, 0x005aaaf2, 0x005aaef2, 0x005ab2f2, 0x005ab6f2, 0x005abaf2, 0x005abef2, // 30376-30383
	0x005ac2f2, 0x005ac6f2, 0x005acaf2, 0x005acef2, 0x005ad2f2, 0x005ad6f2, 0x005adaf2, 0x005adef2, // 30384-30391
	0x005ae2f2, 0x005ae6f2, 0x005aeaf2, 0x005aeef2, 0x005af2f2, 0x005af6f2, 0x005afaf2, 0x005afef2, // 30392-30399
	0x005b02f2, 0x005b06f2, 0x005b0af2, 0x005b0ef2, 0x005b12f2, 0x005b16f2, 0x005b1af2, 0x005b1ef2, // 30400-30407
	0x005b22f2, 0x005b26f2, 0x005b2af2, 0x005b2ef2, 0x005b32f2, 0x005b36f2, 0x005b3af2, 0x005b3ef2, // 30408-30415
	0x005b42f2, 0x005b46f2, 0x005b4af2, 0x005b4ef2, 0x005b52f2, 0x005b56f2, 0x005b5af2, 0x005b5ef2, // 30416-30423
	0x005b62f2, 0x005b66f2, 0x005b6af2, 0x005b6ef2, 0x005b72f2, 0x005b76f2, 0x005b7af2, 0x005b7ef2, // 30424-30431
	0x005b82f2, 0x005b86f2, 0x005b8af2, 0x005b8ef2, 0x005b92f2, 0x005b96f2, 0x005b9af2, 0x005b9ef2, // 30432-30439
	0x005ba2f2, 0x005ba6f2, 0x005baaf2, 0x005baef2, 0x005bb2f2, 0x005bb6f2, 0x005bbaf2, 0x005bbef2, // 30440-30447
	0x005bc2f2, 0x005bc6f2, 0x005bcaf2, 0x005bcef2, 0x005bd2f2, 0x005bd6f2, 0x005bdaf2, 0x005bdef2, // 30448-30455
	0x005be2f2, 0x005be6f2, 0x005beaf2, 0x005beef2, 0x005bf2f2, 0x005bf6f2, 0x005bfaf2, 0x005bfef2, // 30456-30463
	0x005c02f2, 0x005c06f2, 0x005c0af2, 0x005c0ef2, 0x005c12f2, 0x005c16f2, 0x005c1af2, 0x005c1ef2, // 30464-30471
	0x005c22f2, 0x005c26f2, 0x005c2af2, 0x005c2ef2, 0x005c32f2, 0x005c36f2, 0x005c3af2, 0x005c3ef2, // 30472-30479
	0x005c42f2, 0x005c46f2, 0x005c4af2, 0x005c4ef2, 0x005c52f2, 0x005c56f2, 0x005c5af2, 0x005c5ef2, // 30480-30487
	0x005c62f2, 0x005c66f2, 0x005c6af2, 0x005c6ef2, 0x005c72f2, 0x005c76f2, 0x005c7af2, 0x005c7ef2, // 30488-30495
	0x005c82f2, 0x005c86f2, 0x005c8af2, 0x005c8ef2, 0x005c92f2, 0x005c96f2, 0x005c9af2, 0x005c9ef2, // 30496-30503
	0x005ca2f2, 0x005ca6f2, 0x005caaf2, 0x005caef2, 0x005cb2f2, 0x005cb6f2, 0x005cbaf2, 0x005cbef2, // 30504-30511
	0x005cc2f2, 0x005cc6f2, 0x005ccaf2, 0x005ccef2, 0x005cd2f2, 0x005cd6f2, 0x005cdaf2, 0x005cdef2, // 30512-30519
	0x005ce2f2, 0x005ce6f2, 0x005ceaf2, 0x005ceef2, 0x005cf2f2, 0x005cf6f2, 0x005cfaf2, 0x005cfef2, // 30520-30527
	0x005d02f2, 0x005d06f2, 0x005d0af2, 0x005d0ef2, 0x005d12f2, 0x005d16f2, 0x005d1af2, 0x005d1ef2, // 30528-30535
	0x005d22f2, 0x005d26f2, 0x005d2af2, 0x005d2ef2, 0x005d32f2, 0x005d36f2, 0x005d3af2, 0x005d3ef2, // 30536-30543
	0x005d42f2, 0x005d46f2, 0x005d4af2, 0x005d4ef2, 0x005d52f2, 0x005d56f2, 0x005d5af2, 0x005d5ef2, // 30544-30551
	0x005d62f2, 0x005d66f2, 0x005d6af2, 0x005d6ef2, 0x005d72f2, 0x005d76f2, 0x005d7af2, 0x005d7ef2, // 30552-30559
	0x005d82f2, 0x005d86f2, 0x005d8af2, 0x005d8ef2, 0x005d92f2, 0x005d96f2, 0x005d9af2, 0x005d9ef2, // 30560-30567
	0x005da2f2, 0x005da6f2, 0x005daaf2, 0x005daef2, 0x005db2f2, 0x005db6f2, 0x005dbaf2, 0x005dbef2, // 30568-30575
	0x005dc2f2, 0x005dc6f2, 0x005dcaf2, 0x005dcef2, 0x005dd2f2, 0x005dd6f2, 0x005ddaf2, 0x005ddef2, // 30576-30583
	0x005de2f2, 0x005de6f2, 0x005deaf2, 0x005deef2, 0x005df2f2, 0x005df6f2, 0x005dfaf2, 0x005dfef2, // 30584-30591
	0x005e02f2, 0x005e06f2, 0x005e0af2, 0x005e0ef2, 0x005e12f2, 0x005e16f2, 0x005e1af2, 0x005e1ef2, // 30592-30599
	0x005e22f2, 0x005e26f2, 0x005e2af2, 0x005e2ef2, 0x005e32f2, 0x005e36f2, 0x005e3af2, 0x005e3ef2, // 30600-30607
	0x005e42f2, 0x005e46f2, 0x005e4af2, 0x005e4ef2, 0x005e52f2, 0x005e56f2, 0x005e5af2, 0x005e5ef2, // 30608-30615
	0x005e62f2, 0x005e66f2, 0x005e6af2, 0x005e6ef2, 0x005e72f2, 0x005e76f2, 0x005e7af2, 0x005e7ef2, // 30616-30623
	0x005e82f2, 0x005e86f2, 0x005e8af2, 0x005e8ef2, 0x005e92f2, 0x005e96f2, 0x005e9af2, 0x005e9ef2, // 30624-30631
	0x005ea2f2, 0x005ea6f2, 0x005eaaf2, 0x005eaef2, 0x005eb2f2, 0x005eb6f2, 0x005ebaf2, 0x005ebef2, // 30632-30639
	0x005ec2f2, 0x005ec6f2, 0x005ecaf2, 0x005ecef2, 0x005ed2f2, 0x005ed6f2, 0x005edaf2, 0x005edef2, // 30640-30647
	0x005ee2f2, 0x005ee6f2, 0x005eeaf2, 0x005eeef2, 0x005ef2f2, 0x005ef6f2, 0x005efaf2, 0x005efef2, // 30648-30655
	0x005f02f2, 0x005f06f2, 0x005f0af2, 0x005f0ef2, 0x005f12f2, 0x005f16f2, 0x005f1af2, 0x005f1ef2, // 30656-30663
	0x005f22f2, 0x005f26f2, 0x005f2af2, 0x005f2ef2, 0x005f32f2, 0x005f36f2, 0x005f3af2, 0x005f3ef2, // 30664-30671
	0x005f42f2, 0x005f46f2, 0x005f4af2, 0x005f4ef2, 0x005f52f2, 0x005f56f2, 0x005f5af2, 0x005f5ef2, // 30672-30679
	0x005f62f2, 0x005f66f2, 0x005f6af2, 0x005f6ef2, 0x005f72f2, 0x005f76f2, 0x005f7af2, 0x005f7ef2, // 30680-30687
	0x005f82f2, 0x005f86f2, 0x005f8af2, 0x005f8ef2, 0x005f92f2, 0x005f96f2, 0x005f9af2, 0x005f9ef2, // 30688-30695
	0x005fa2f2, 0x005fa6f2, 0x005faaf2, 0x005faef2, 0x005fb2f2, 0x005fb6f2, 0x005fbaf2, 0x005fbef2, // 30696-30703
	0x005fc2f2, 0x005fc6f2, 0x005fcaf2, 0x005fcef2, 0x005fd2f2, 0x005fd6f2, 0x005fdaf2, 0x005fdef2, // 30704-30711
	0x005fe2f2, 0x005fe6f2, 0x005feaf2, 0x005feef2, 0x005ff2f2, 0x005ff6f2, 0x005ffaf2, 0x005ffef2, // 30712-30719
	0x006002f2, 0x006006f2, 0x00600af2, 0x00600ef2, 0x006012f2, 0x006016f2, 0x00601af2, 0x00601ef2, // 30720-30727
	0x006022f2, 0x006026f2, 0x00602af2, 0x00602ef2, 0x006032f2, 0x006036f2, 0x00603af2, 0x00603ef2, // 30728-30735
	0x006042f2, 0x006046f2, 0x00604af2, 0x00604ef2, 0x006052f2, 0x006056f2, 0x00605af2, 0x00605ef2, // 30736-30743
	0x006062f2, 0x006066f2, 0x00606af2, 0x00606ef2, 0x006072f2, 0x006076f2, 0x00607af2, 0x00607ef2, // 30744-30751
	0x006082f2, 0x006086f2, 0x00608af2, 0x00608ef2, 0x006092f2, 0x006096f2, 0x00609af2, 0x00609ef2, // 30752-30759
	0x0060a2f2, 0x0060a6f2, 0x0060aaf2, 0x0060aef2, 0x0060b2f2, 0x0060b6f2, 0x0060baf2, 0x0060bef2, // 30760-30767
	0x0060c2f2, 0x0060c6f2, 0x0060caf2, 0x0060cef2, 0x0060d2f2, 0x0060d6f2, 0x0060daf2, 0x0060def2, // 30768-30775
	0x0060e2f2, 0x0060e6f2, 0x0060eaf2, 0x0060eef2, 0x0060f2f2, 0x0060f6f2, 0x0060faf2, 0x0060fef2, // 30776-30783
	0x006102f2, 0x006106f2, 0x00610af2, 0x00610ef2, 0x006112f2, 0x006116f2, 0x00611af2, 0x00611ef2, // 30784-30791
	0x006122f2, 0x006126f2, 0x00612af2, 0x00612ef2, 0x006132f2, 0x006136f2, 0x00613af2, 0x00613ef2, // 30792-30799
	0x006142f2, 0x006146f2, 0x00614af2, 0x00614ef2, 0x006152f2, 0x006156f2, 0x00615af2, 0x00615ef2, // 30800-30807
	0x006162f2, 0x006166f2, 0x00616af2, 0x00616ef2, 0x006172f2, 0x006176f2, 0x00617af2, 0x00617ef2, // 30808-30815
	0x006182f2, 0x006186f2, 0x00618af2, 0x00618ef2, 0x006192f2, 0x006196f2, 0x00619af2, 0x00619ef2, // 30816-30823
	0x0061a2f2, 0x0061a6f2, 0x0061aaf2, 0x0061aef2, 0x0061b2f2, 0x0061b6f2, 0x0061baf2, 0x0061bef2, // 30824-30831
	0x0061c2f2, 0x0061c6f2, 0x0061caf2, 0x0061cef2, 0x0061d2f2, 0x0061d6f2, 0x0061daf2, 0x0061def2, // 30832-30839
	0x0061e2f2, 0x0061e6f2, 0x0061eaf2, 0x0061eef2, 0x0061f2f2, 0x0061f6f2, 0x0061faf2, 0x0061fef2, // 30840-30847
	0x006202f2, 0x006206f2, 0x00620af2, 0x00620ef2, 0x006212f2, 0x006216f2, 0x00621af2, 0x00621ef2, // 30848-30855
	0x006222f2, 0x006226f2, 0x00622af2, 0x00622ef2, 0x006232f2, 0x006236f2, 0x00623af2, 0x00623ef2, // 30856-30863
	0x006242f2, 0x006246f2, 0x00624af2, 0x00624ef2, 0x006252f2, 0x006256f2, 0x00625af2, 0x00625ef2, // 30864-30871
	0x006262f2, 0x006266f2, 0x00626af2, 0x00626ef2, 0x006272f2, 0x006276f2, 0x00627af2, 0x00627ef2, // 30872-30879
	0x006282f2, 0x006286f2, 0x00628af2, 0x00628ef2, 0x006292f2, 0x006296f2, 0x00629af2, 0x00629ef2, // 30880-30887
	0x0062a2f2, 0x0062a6f2, 0x0062aaf2, 0x0062aef2, 0x0062b2f2, 0x0062b6f2, 0x0062baf2, 0x0062bef2, // 30888-30895
	0x0062c2f2, 0x0062c6f2, 0x0062caf2, 0x0062cef2, 0x0062d2f2, 0x0062d6f2, 0x0062daf2, 0x0062def2, // 30896-30903
	0x0062e2f2, 0x0062e6f2, 0x0062eaf2, 0x0062eef2, 0x0062f2f2, 0x0062f6f2, 0x0062faf2, 0x0062fef2, // 30904-30911
	0x006302f2, 0x006306f2, 0x00630af2, 0x00630ef2, 0x006312f2, 0x006316f2, 0x00631af2, 0x00631ef2, // 30912-30919
	0x006322f2, 0x006326f2, 0x00632af2, 0x00632ef2, 0x006332f2, 0x006336f2, 0x00633af2, 0x00633ef2, // 30920-30927
	0x006342f2, 0x006346f2, 0x00634af2, 0x00634ef2, 0x006352f2, 0x006356f2, 0x00635af2, 0x00635ef2, // 30928-30935
	0x006362f2, 0x006366f2, 0x00636af2, 0x00636ef2, 0x006372f2, 0x006376f2, 0x00637af2, 0x00637ef2, // 30936-30943
	0x006382f2, 0x006386f2, 0x00638af2, 0x00638ef2, 0x006392f2, 0x006396f2, 0x00639af2, 0x00639ef2, // 30944-30951
	0x0063a2f2, 0x0063a6f2, 0x0063aaf2, 0x0063aef2, 0x0063b2f2, 0x0063b6f2, 0x0063baf2, 0x0063bef2, // 30952-30959
	0x0063c2f2, 0x0063c6f2, 0x0063caf2, 0x0063cef2, 0x0063d2f2, 0x0063d6f2, 0x0063daf2, 0x0063def2, // 30960-30967
	0x0063e2f2, 0x0063e6f2, 0x0063eaf2, 0x0063eef2, 0x0063f2f2, 0x0063f6f2, 0x0063faf2, 0x0063fef2, // 30968-30975
	0x006402f2, 0x006406f2, 0x00640af2, 0x00640ef2, 0x006412f2, 0x006416f2, 0x00641af2, 0x00641ef2, // 30976-30983
	0x006422f2, 0x006426f2, 0x00642af2, 0x00642ef2, 0x006432f2, 0x006436f2, 0x00643af2, 0x00643ef2, // 30984-30991
	0x006442f2, 0x006446f2, 0x00644af2, 0x00644ef2, 0x006452f2, 0x006456f2, 0x00645af2, 0x00645ef2, // 30992-30999
	0x006462f2, 0x006466f2, 0x00646af2, 0x00646ef2, 0x006472f2, 0x006476f2, 0x00647af2, 0x00647ef2, // 31000-31007
	0x006482f2, 0x006486f2, 0x00648af2, 0x00648ef2, 0x006492f2, 0x006496f2, 0x00649af2, 0x00649ef2, // 31008-31015
	0x0064a2f2, 0x0064a6f2, 0x0064aaf2, 0x0064aef2, 0x0064b2f2, 0x0064b6f2, 0x0064baf2, 0x0064bef2, // 31016-31023
	0x0064c2f2, 0x0064c6f2, 0x0064caf2, 0x0064cef2, 0x0064d2f2, 0x0064d6f2, 0x0064daf2, 0x0064def2, // 31024-31031
	0x0064e2f2, 0x0064e6f2, 0x0064eaf2, 0x0064eef2, 0x0064f2f2, 0x0064f6f2, 0x0064faf2, 0x0064fef2, // 31032-31039
	0x006502f2, 0x006506f2, 0x00650af2, 0x00650ef2, 0x006512f2, 0x006516f2, 0x00651af2, 0x00651ef2, // 31040-31047
	0x006522f2, 0x006526f2, 0x00652af2, 0x00652ef2, 0x006532f2, 0x006536f2, 0x00653af2, 0x00653ef2, // 31048-31055
	0x006542f2, 0x006546f2, 0x00654af2, 0x00654ef2, 0x006552f2, 0x006556f2, 0x00655af2, 0x00655ef2, // 31056-31063
	0x006562f2, 0x006566f2, 0x00656af2, 0x00656ef2, 0x006572f2, 0x006576f2, 0x00657af2, 0x00657ef2, // 31064-31071
	0x006582f2, 0x006586f2, 0x00658af2, 0x00658ef2, 0x006592f2, 0x006596f2, 0x00659af2, 0x00659ef2, // 31072-31079
	0x0065a2f2, 0x0065a6f2, 0x0065aaf2, 0x0065aef2, 0x0065b2f2, 0x0065b6f2, 0x0065baf2, 0x0065bef2, // 31080-31087
	0x0065c2f2, 0x0065c6f2, 0x0065caf2, 0x0065cef2, 0x0065d2f2, 0x0065d6f2, 0x0065daf2, 0x0065def2, // 31088-31095
	0x0065e2f2, 0x0065e6f2, 0x0065eaf2, 0x0065eef2, 0x0065f2f2, 0x0065f6f2, 0x0065faf2, 0x0065fef2, // 31096-31103
	0x006602f2, 0x006606f2, 0x00660af2, 0x00660ef2, 0x006612f2, 0x006616f2, 0x00661af2, 0x00661ef2, // 31104-31111
	0x006622f2, 0x006626f2, 0x00662af2, 0x00662ef2, 0x006632f2, 0x006636f2, 0x00663af2, 0x00663ef2, // 31112-31119
	0x006642f2, 0x006646f2, 0x00664af2, 0x00664ef2, 0x006652f2, 0x006656f2, 0x00665af2, 0x00665ef2, // 31120-31127
	0x006662f2, 0x006666f2, 0x00666af2, 0x00666ef2, 0x006672f2, 0x006676f2, 0x00667af2, 0x00667ef2, // 31128-31135
	0x006682f2, 0x006686f2, 0x00668af2, 0x00668ef2, 0x006692f2, 0x006696f2, 0x00669af2, 0x00669ef2, // 31136-31143
	0x0066a2f2, 0x0066a6f2, 0x0066aaf2, 0x0066aef2, 0x0066b2f2, 0x0066b6f2, 0x0066baf2, 0x0066bef2, // 31144-31151
	0x0066c2f2, 0x0066c6f2, 0x0066caf2, 0x0066cef2, 0x0066d2f2, 0x0066d6f2, 0x0066daf2, 0x0066def2, // 31152-31159
	0x0066e2f2, 0x0066e6f2, 0x0066eaf2, 0x0066eef2, 0x0066f2f2, 0x0066f6f2, 0x0066faf2, 0x0066fef2, // 31160-31167
	0x006702f2, 0x006706f2, 0x00670af2, 0x00670ef2, 0x006712f2, 0x006716f2, 0x00671af2, 0x00671ef2, // 31168-31175
	0x006722f2, 0x006726f2, 0x00672af2, 0x00672ef2, 0x006732f2, 0x006736f2, 0x00673af2, 0x00673ef2, // 31176-31183
	0x006742f2, 0x006746f2, 0x00674af2, 0x00674ef2, 0x006752f2, 0x006756f2, 0x00675af2, 0x00675ef2, // 31184-31191
	0x006762f2, 0x006766f2, 0x00676af2, 0x00676ef2, 0x006772f2, 0x006776f2, 0x00677af2, 0x00677ef2, // 31192-31199
	0x006782f2, 0x006786f2, 0x00678af2, 0x00678ef2, 0x006792f2, 0x006796f2, 0x00679af2, 0x00679ef2, // 31200-31207
	0x0067a2f2, 0x0067a6f2, 0x0067aaf2, 0x0067aef2, 0x0067b2f2, 0x0067b6f2, 0x0067baf2, 0x0067bef2, // 31208-31215
	0x0067c2f2, 0x0067c6f2, 0x0067caf2, 0x0067cef2, 0x0067d2f2, 0x0067d6f2, 0x0067daf2, 0x0067def2, // 31216-31223
	0x0067e2f2, 0x0067e6f2, 0x0067eaf2, 0x0067eef2, 0x0067f2f2, 0x0067f6f2, 0x0067faf2, 0x0067fef2, // 31224-31231
	0x006802f2, 0x006806f2, 0x00680af2, 0x00680ef2, 0x006812f2, 0x006816f2, 0x00681af2, 0x00681ef2, // 31232-31239
	0x006822f2, 0x006826f2, 0x00682af2, 0x00682ef2, 0x006832f2, 0x006836f2, 0x00683af2, 0x00683ef2, // 31240-31247
	0x006842f2, 0x006846f2, 0x00684af2, 0x00684ef2, 0x006852f2, 0x006856f2, 0x00685af2, 0x00685ef2, // 31248-31255
	0x006862f2, 0x006866f2, 0x00686af2, 0x00686ef2, 0x006872f2, 0x006876f2, 0x00687af2, 0x00687ef2, // 31256-31263
	0x006882f2, 0x006886f2, 0x00688af2, 0x00688ef2, 0x006892f2, 0x006896f2, 0x00689af2, 0x00689ef2, // 31264-31271
	0x0068a2f2, 0x0068a6f2, 0x0068aaf2, 0x0068aef2, 0x0068b2f2, 0x0068b6f2, 0x0068baf2, 0x0068bef2, // 31272-31279
	0x0068c2f2, 0x0068c6f2, 0x0068caf2, 0x0068cef2, 0x0068d2f2, 0x0068d6f2, 0x0068daf2, 0x0068def2, // 31280-31287
	0x0068e2f2, 0x0068e6f2, 0x0068eaf2, 0x0068eef2, 0x0068f2f2, 0x0068f6f2, 0x0068faf2, 0x0068fef2, // 31288-31295
	0x006902f2, 0x006906f2, 0x00690af2, 0x00690ef2, 0x006912f2, 0x006916f2, 0x00691af2, 0x00691ef2, // 31296-31303
	0x006922f2, 0x006926f2, 0x00692af2, 0x00692ef2, 0x006932f2, 0x006936f2, 0x00693af2, 0x00693ef2, // 31304-31311
	0x006942f2, 0x006946f2, 0x00694af2, 0x00694ef2, 0x006952f2, 0x006956f2, 0x00695af2, 0x00695ef2, // 31312-31319
	0x006962f2, 0x006966f2, 0x00696af2, 0x00696ef2, 0x006972f2, 0x006976f2, 0x00697af2, 0x00697ef2, // 31320-31327
	0x006982f2, 0x006986f2, 0x00698af2, 0x00698ef2, 0x006992f2, 0x006996f2, 0x00699af2, 0x00699ef2, // 31328-31335
	0x0069a2f2, 0x0069a6f2, 0x0069aaf2, 0x0069aef2, 0x0069b2f2, 0x0069b6f2, 0x0069baf2, 0x0069bef2, // 31336-31343
	0x0069c2f2, 0x0069c6f2, 0x0069caf2, 0x0069cef2, 0x0069d2f2, 0x0069d6f2, 0x0069daf2, 0x0069def2, // 31344-31351
	0x0069e2f2, 0x0069e6f2, 0x0069eaf2, 0x0069eef2, 0x0069f2f2, 0x0069f6f2, 0x0069faf2, 0x0069fef2, // 31352-31359
	0x006a02f2, 0x006a06f2, 0x006a0af2, 0x006a0ef2, 0x006a12f2, 0x006a16f2, 0x006a1af2, 0x006a1ef2, // 31360-31367
	0x006a22f2, 0x006a26f2, 0x006a2af2, 0x006a2ef2, 0x006a32f2, 0x006a36f2, 0x006a3af2, 0x006a3ef2, // 31368-31375
	0x006a42f2, 0x006a46f2, 0x006a4af2, 0x006a4ef2, 0x006a52f2, 0x006a56f2, 0x006a5af2, 0x006a5ef2, // 31376-31383
	0x006a62f2, 0x006a66f2, 0x006a6af2, 0x006a6ef2, 0x006a72f2, 0x006a76f2, 0x006a7af2, 0x006a7ef2, // 31384-31391
	0x006a82f2, 0x006a86f2, 0x006a8af2, 0x006a8ef2, 0x006a92f2, 0x006a96f2, 0x006a9af2, 0x006a9ef2, // 31392-31399
	0x006aa2f2, 0x006aa6f2, 0x006aaaf2, 0x006aaef2, 0x006ab2f2, 0x006ab6f2, 0x006abaf2, 0x006abef2, // 31400-31407
	0x006ac2f2, 0x006ac6f2, 0x006acaf2, 0x006acef2, 0x006ad2f2, 0x006ad6f2, 0x006adaf2, 0x006adef2, // 31408-31415
	0x006ae2f2, 0x006ae6f2, 0x006aeaf2, 0x006aeef2, 0x006af2f2, 0x006af6f2, 0x006afaf2, 0x006afef2, // 31416-31423
	0x006b02f2, 0x006b06f2, 0x006b0af2, 0x006b0ef2, 0x006b12f2, 0x006b16f2, 0x006b1af2, 0x006b1ef2, // 31424-31431
	0x006b22f2, 0x006b26f2, 0x006b2af2, 0x006b2ef2, 0x006b32f2, 0x006b36f2, 0x006b3af2, 0x006b3ef2, // 31432-31439
	0x006b42f2, 0x006b46f2, 0x006b4af2, 0x006b4ef2, 0x006b52f2, 0x006b56f2, 0x006b5af2, 0x006b5ef2, // 31440-31447
	0x006b62f2, 0x006b66f2, 0x006b6af2, 0x006b6ef2, 0x006b72f2, 0x006b76f2, 0x006b7af2, 0x006b7ef2, // 31448-31455
	0x006b82f2, 0x006b86f2, 0x006b8af2, 0x006b8ef2, 0x006b92f2, 0x006b96f2, 0x006b9af2, 0x006b9ef2, // 31456-31463
	0x006ba2f2, 0x006ba6f2, 0x006baaf2, 0x006baef2, 0x006bb2f2, 0x006bb6f2, 0x006bbaf2, 0x006bbef2, // 31464-31471
	0x006bc2f2, 0x006bc6f2, 0x006bcaf2, 0x006bcef2, 0x006bd2f2, 0x006bd6f2, 0x006bdaf2, 0x006bdef2, // 31472-31479
	0x006be2f2, 0x006be6f2, 0x006beaf2, 0x006beef2, 0x006bf2f2, 0x006bf6f2, 0x006bfaf2, 0x006bfef2, // 31480-31487
	0x006c02f2, 0x006c06f2, 0x006c0af2, 0x006c0ef2, 0x006c12f2, 0x006c16f2, 0x006c1af2, 0x006c1ef2, // 31488-31495
	0x006c22f2, 0x006c26f2, 0x006c2af2, 0x006c2ef2, 0x006c32f2, 0x006c36f2, 0x006c3af2, 0x006c3ef2, // 31496-31503
	0x006c42f2, 0x006c46f2, 0x006c4af2, 0x006c4ef2, 0x006c52f2, 0x006c56f2, 0x006c5af2, 0x006c5ef2, // 31504-31511
	0x006c62f2, 0x006c66f2, 0x006c6af2, 0x006c6ef2, 0x006c72f2, 0x006c76f2, 0x006c7af2, 0x006c7ef2, // 31512-31519
	0x006c82f2, 0x006c86f2, 0x006c8af2, 0x006c8ef2, 0x006c92f2, 0x006c96f2, 0x006c9af2, 0x006c9ef2, // 31520-31527
	0x006ca2f2, 0x006ca6f2, 0x006caaf2, 0x006caef2, 0x006cb2f2, 0x006cb6f2, 0x006cbaf2, 0x006cbef2, // 31528-31535
	0x006cc2f2, 0x006cc6f2, 0x006ccaf2, 0x006ccef2, 0x006cd2f2, 0x006cd6f2, 0x006cdaf2, 0x006cdef2, // 31536-31543
	0x006ce2f2, 0x006ce6f2, 0x006ceaf2, 0x006ceef2, 0x006cf2f2, 0x006cf6f2, 0x006cfaf2, 0x006cfef2, // 31544-31551
	0x006d02f2, 0x006d06f2, 0x006d0af2, 0x006d0ef2, 0x006d12f2, 0x006d16f2, 0x006d1af2, 0x006d1ef2, // 31552-31559
	0x006d22f2, 0x006d26f2, 0x006d2af2, 0x006d2ef2, 0x006d32f2, 0x006d36f2, 0x006d3af2, 0x006d3ef2, // 31560-31567
	0x006d42f2, 0x006d46f2, 0x006d4af2, 0x006d4ef2, 0x006d52f2, 0x006d56f2, 0x006d5af2, 0x006d5ef2, // 31568-31575
	0x006d62f2, 0x006d66f2, 0x006d6af2, 0x006d6ef2, 0x006d72f2, 0x006d76f2, 0x006d7af2, 0x006d7ef2, // 31576-31583
	0x006d82f2, 0x006d86f2, 0x006d8af2, 0x006d8ef2, 0x006d92f2, 0x006d96f2, 0x006d9af2, 0x006d9ef2, // 31584-31591
	0x006da2f2, 0x006da6f2, 0x006daaf2, 0x006daef2, 0x006db2f2, 0x006db6f2, 0x006dbaf2, 0x006dbef2, // 31592-31599
	0x006dc2f2, 0x006dc6f2, 0x006dcaf2, 0x006dcef2, 0x006dd2f2, 0x006dd6f2, 0x006ddaf2, 0x006ddef2, // 31600-31607
	0x006de2f2, 0x006de6f2, 0x006deaf2, 0x006deef2, 0x006df2f2, 0x006df6f2, 0x006dfaf2, 0x006dfef2, // 31608-31615
	0x006e02f2, 0x006e06f2, 0x006e0af2, 0x006e0ef2, 0x006e12f2, 0x006e16f2, 0x006e1af2, 0x006e1ef2, // 31616-31623
	0x006e22f2, 0x006e26f2, 0x006e2af2, 0x006e2ef2, 0x006e32f2, 0x006e36f2, 0x006e3af2, 0x006e3ef2, // 31624-31631
	0x006e42f2, 0x006e46f2, 0x006e4af2, 0x006e4ef2, 0x006e52f2, 0x006e56f2, 0x006e5af2, 0x006e5ef2, // 31632-31639
	0x006e62f2, 0x006e66f2, 0x006e6af2, 0x006e6ef2, 0x006e72f2, 0x006e76f2, 0x006e7af2, 0x006e7ef2, // 31640-31647
	0x006e82f2, 0x006e86f2, 0x006e8af2, 0x006e8ef2, 0x006e92f2, 0x006e96f2, 0x006e9af2, 0x006e9ef2, // 31648-31655
	0x006ea2f2, 0x006ea6f2, 0x006eaaf2, 0x006eaef2, 0x006eb2f2, 0x006eb6f2, 0x006ebaf2, 0x006ebef2, // 31656-31663
	0x006ec2f2, 0x006ec6f2, 0x006ecaf2, 0x006ecef2, 0x006ed2f2, 0x006ed6f2, 0x006edaf2, 0x006edef2, // 31664-31671
	0x006ee2f2, 0x006ee6f2, 0x006eeaf2, 0x006eeef2, 0x006ef2f2, 0x006ef6f2, 0x006efaf2, 0x006efef2, // 31672-31679
	0x006f02f2, 0x006f06f2, 0x006f0af2, 0x006f0ef2, 0x006f12f2, 0x006f16f2, 0x006f1af2, 0x006f1ef2, // 31680-31687
	0x006f22f2, 0x006f26f2, 0x006f2af2, 0x006f2ef2, 0x006f32f2, 0x006f36f2, 0x006f3af2, 0x006f3ef2, // 31688-31695
	0x006f42f2, 0x006f46f2, 0x006f4af2, 0x006f4ef2, 0x006f52f2, 0x006f56f2, 0x006f5af2, 0x006f5ef2, // 31696-31703
	0x006f62f2, 0x006f66f2, 0x006f6af2, 0x006f6ef2, 0x006f72f2, 0x006f76f2, 0x006f7af2, 0x006f7ef2, // 31704-31711
	0x006f82f2, 0x006f86f2, 0x006f8af2, 0x006f8ef2, 0x006f92f2, 0x006f96f2, 0x006f9af2, 0x006f9ef2, // 31712-31719
	0x006fa2f2, 0x006fa6f2, 0x006faaf2, 0x006faef2, 0x006fb2f2, 0x006fb6f2, 0x006fbaf2, 0x006fbef2, // 31720-31727
	0x006fc2f2, 0x006fc6f2, 0x006fcaf2, 0x006fcef2, 0x006fd2f2, 0x006fd6f2, 0x006fdaf2, 0x006fdef2, // 31728-31735
	0x006fe2f2, 0x006fe6f2, 0x006feaf2, 0x006feef2, 0x006ff2f2, 0x006ff6f2, 0x006ffaf2, 0x006ffef2, // 31736-31743
	0x007002f2, 0x007006f2, 0x00700af2, 0x00700ef2, 0x007012f2, 0x007016f2, 0x00701af2, 0x00701ef2, // 31744-31751
	0x007022f2, 0x007026f2, 0x00702af2, 0x00702ef2, 0x007032f2, 0x007036f2, 0x00703af2, 0x00703ef2, // 31752-31759
	0x007042f2, 0x007046f2, 0x00704af2, 0x00704ef2, 0x007052f2, 0x007056f2, 0x00705af2, 0x00705ef2, // 31760-31767
	0x007062f2, 0x007066f2, 0x00706af2, 0x00706ef2, 0x007072f2, 0x007076f2, 0x00707af2, 0x00707ef2, // 31768-31775
	0x007082f2, 0x007086f2, 0x00708af2, 0x00708ef2, 0x007092f2, 0x007096f2, 0x00709af2, 0x00709ef2, // 31776-31783
	0x0070a2f2, 0x0070a6f2, 0x0070aaf2, 0x0070aef2, 0x0070b2f2, 0x0070b6f2, 0x0070baf2, 0x0070bef2, // 31784-31791
	0x0070c2f2, 0x0070c6f2, 0x0070caf2, 0x0070cef2, 0x0070d2f2, 0x0070d6f2, 0x0070daf2, 0x0070def2, // 31792-31799
	0x0070e2f2, 0x0070e6f2, 0x0070eaf2, 0x0070eef2, 0x0070f2f2, 0x0070f6f2, 0x0070faf2, 0x0070fef2, // 31800-31807
	0x007102f2, 0x007106f2, 0x00710af2, 0x00710ef2, 0x007112f2, 0x007116f2, 0x00711af2, 0x00711ef2, // 31808-31815
	0x007122f2, 0x007126f2, 0x00712af2, 0x00712ef2, 0x007132f2, 0x007136f2, 0x00713af2, 0x00713ef2, // 31816-31823
	0x007142f2, 0x007146f2, 0x00714af2, 0x00714ef2, 0x007152f2, 0x007156f2, 0x00715af2, 0x00715ef2, // 31824-31831
	0x007162f2, 0x007166f2, 0x00716af2, 0x00716ef2, 0x007172f2, 0x007176f2, 0x00717af2, 0x00717ef2, // 31832-31839
	0x007182f2, 0x007186f2, 0x00718af2, 0x00718ef2, 0x007192f2, 0x007196f2, 0x00719af2, 0x00719ef2, // 31840-31847
	0x0071a2f2, 0x0071a6f2, 0x0071aaf2, 0x0071aef2, 0x0071b2f2, 0x0071b6f2, 0x0071baf2, 0x0071bef2, // 31848-31855
	0x0071c2f2, 0x0071c6f2, 0x0071caf2, 0x0071cef2, 0x0071d2f2, 0x0071d6f2, 0x0071daf2, 0x0071def2, // 31856-31863
	0x0071e2f2, 0x0071e6f2, 0x0071eaf2, 0x0071eef2, 0x0071f2f2, 0x0071f6f2, 0x0071faf2, 0x0071fef2, // 31864-31871
	0x007202f2, 0x007206f2, 0x00720af2, 0x00720ef2, 0x007212f2, 0x007216f2, 0x00721af2, 0x00721ef2, // 31872-31879
	0x007222f2, 0x007226f2, 0x00722af2, 0x00722ef2, 0x007232f2, 0x007236f2, 0x00723af2, 0x00723ef2, // 31880-31887
	0x007242f2, 0x007246f2, 0x00724af2, 0x00724ef2, 0x007252f2, 0x007256f2, 0x00725af2, 0x00725ef2, // 31888-31895
	0x007262f2, 0x007266f2, 0x00726af2, 0x00726ef2, 0x007272f2, 0x007276f2, 0x00727af2, 0x00727ef2, // 31896-31903
	0x007282f2, 0x007286f2, 0x00728af2, 0x00728ef2, 0x007292f2, 0x007296f2, 0x00729af2, 0x00729ef2, // 31904-31911
	0x0072a2f2, 0x0072a6f2, 0x0072aaf2, 0x0072aef2, 0x0072b2f2, 0x0072b6f2, 0x0072baf2, 0x0072bef2, // 31912-31919
	0x0072c2f2, 0x0072c6f2, 0x0072caf2, 0x0072cef2, 0x0072d2f2, 0x0072d6f2, 0x0072daf2, 0x0072def2, // 31920-31927
	0x0072e2f2, 0x0072e6f2, 0x0072eaf2, 0x0072eef2, 0x0072f2f2, 0x0072f6f2, 0x0072faf2, 0x0072fef2, // 31928-31935
	0x007302f2, 0x007306f2, 0x00730af2, 0x00730ef2, 0x007312f2, 0x007316f2, 0x00731af2, 0x00731ef2, // 31936-31943
	0x007322f2, 0x007326f2, 0x00732af2, 0x00732ef2, 0x007332f2, 0x007336f2, 0x00733af2, 0x00733ef2, // 31944-31951
	0x007342f2, 0x007346f2, 0x00734af2, 0x00734ef2, 0x007352f2, 0x007356f2, 0x00735af2, 0x00735ef2, // 31952-31959
	0x007362f2, 0x007366f2, 0x00736af2, 0x00736ef2, 0x007372f2, 0x007376f2, 0x00737af2, 0x00737ef2, // 31960-31967
	0x007382f2, 0x007386f2, 0x00738af2, 0x00738ef2, 0x007392f2, 0x007396f2, 0x00739af2, 0x00739ef2, // 31968-31975
	0x0073a2f2, 0x0073a6f2, 0x0073aaf2, 0x0073aef2, 0x0073b2f2, 0x0073b6f2, 0x0073baf2, 0x0073bef2, // 31976-31983
	0x0073c2f2, 0x0073c6f2, 0x0073caf2, 0x0073cef2, 0x0073d2f2, 0x0073d6f2, 0x0073daf2, 0x0073def2, // 31984-31991
	0x0073e2f2, 0x0073e6f2, 0x0073eaf2, 0x0073eef2, 0x0073f2f2, 0x0073f6f2, 0x0073faf2, 0x0073fef2, // 31992-31999
	0x007402f2, 0x007406f2, 0x00740af2, 0x00740ef2, 0x007412f2, 0x007416f2, 0x00741af2, 0x00741ef2, // 32000-32007
	0x007422f2, 0x007426f2, 0x00742af2, 0x00742ef2, 0x007432f2, 0x007436f2, 0x00743af2, 0x00743ef2, // 32008-32015
	0x007442f2, 0x007446f2, 0x00744af2, 0x00744ef2, 0x007452f2, 0x007456f2, 0x00745af2, 0x00745ef2, // 32016-32023
	0x007462f2, 0x007466f2, 0x00746af2, 0x00746ef2, 0x007472f2, 0x007476f2, 0x00747af2, 0x00747ef2, // 32024-32031
	0x007482f2, 0x007486f2, 0x00748af2, 0x00748ef2, 0x007492f2, 0x007496f2, 0x00749af2, 0x00749ef2, // 32032-32039
	0x0074a2f2, 0x0074a6f2, 0x0074aaf2, 0x0074aef2, 0x0074b2f2, 0x0074b6f2, 0x0074baf2, 0x0074bef2, // 32040-32047
	0x0074c2f2, 0x0074c6f2, 0x0074caf2, 0x0074cef2, 0x0074d2f2, 0x0074d6f2, 0x0074daf2, 0x0074def2, // 32048-32055
	0x0074e2f2, 0x0074e6f2, 0x0074eaf2, 0x0074eef2, 0x0074f2f2, 0x0074f6f2, 0x0074faf2, 0x0074fef2, // 32056-32063
	0x007502f2, 0x007506f2, 0x00750af2, 0x00750ef2, 0x007512f2, 0x007516f2, 0x00751af2, 0x00751ef2, // 32064-32071
	0x007522f2, 0x007526f2, 0x00752af2, 0x00752ef2, 0x007532f2, 0x007536f2, 0x00753af2, 0x00753ef2, // 32072-32079
	0x007542f2, 0x007546f2, 0x00754af2, 0x00754ef2, 0x007552f2, 0x007556f2, 0x00755af2, 0x00755ef2, // 32080-32087
	0x007562f2, 0x007566f2, 0x00756af2, 0x00756ef2, 0x007572f2, 0x007576f2, 0x00757af2, 0x00757ef2, // 32088-32095
	0x007582f2, 0x007586f2, 0x00758af2, 0x00758ef2, 0x007592f2, 0x007596f2, 0x00759af2, 0x00759ef2, // 32096-32103
	0x0075a2f2, 0x0075a6f2, 0x0075aaf2, 0x0075aef2, 0x0075b2f2, 0x0075b6f2, 0x0075baf2, 0x0075bef2, // 32104-32111
	0x0075c2f2, 0x0075c6f2, 0x0075caf2, 0x0075cef2, 0x0075d2f2, 0x0075d6f2, 0x0075daf2, 0x0075def2, // 32112-32119
	0x0075e2f2, 0x0075e6f2, 0x0075eaf2, 0x0075eef2, 0x0075f2f2, 0x0075f6f2, 0x0075faf2, 0x0075fef2, // 32120-32127
	0x007602f2, 0x007606f2, 0x00760af2, 0x00760ef2, 0x007612f2, 0x007616f2, 0x00761af2, 0x00761ef2, // 32128-32135
	0x007622f2, 0x007626f2, 0x00762af2, 0x00762ef2, 0x007632f2, 0x007636f2, 0x00763af2, 0x00763ef2, // 32136-32143
	0x007642f2, 0x007646f2, 0x00764af2, 0x00764ef2, 0x007652f2, 0x007656f2, 0x00765af2, 0x00765ef2, // 32144-32151
	0x007662f2, 0x007666f2, 0x00766af2, 0x00766ef2, 0x007672f2, 0x007676f2, 0x00767af2, 0x00767ef2, // 32152-32159
	0x007682f2, 0x007686f2, 0x00768af2, 0x00768ef2, 0x007692f2, 0x007696f2, 0x00769af2, 0x00769ef2, // 32160-32167
	0x0076a2f2, 0x0076a6f2, 0x0076aaf2, 0x0076aef2, 0x0076b2f2, 0x0076b6f2, 0x0076baf2, 0x0076bef2, // 32168-32175
	0x0076c2f2, 0x0076c6f2, 0x0076caf2, 0x0076cef2, 0x0076d2f2, 0x0076d6f2, 0x0076daf2, 0x0076def2, // 32176-32183
	0x0076e2f2, 0x0076e6f2, 0x0076eaf2, 0x0076eef2, 0x0076f2f2, 0x0076f6f2, 0x0076faf2, 0x0076fef2, // 32184-32191
	0x007702f2, 0x007706f2, 0x00770af2, 0x00770ef2, 0x007712f2, 0x007716f2, 0x00771af2, 0x00771ef2, // 32192-32199
	0x007722f2, 0x007726f2, 0x00772af2, 0x00772ef2, 0x007732f2, 0x007736f2, 0x00773af2, 0x00773ef2, // 32200-32207
	0x007742f2, 0x007746f2, 0x00774af2, 0x00774ef2, 0x007752f2, 0x007756f2, 0x00775af2, 0x00775ef2, // 32208-32215
	0x007762f2, 0x007766f2, 0x00776af2, 0x00776ef2, 0x007772f2, 0x007776f2, 0x00777af2, 0x00777ef2, // 32216-32223
	0x007782f2, 0x007786f2, 0x00778af2, 0x00778ef2, 0x007792f2, 0x007796f2, 0x00779af2, 0x00779ef2, // 32224-32231
	0x0077a2f2, 0x0077a6f2, 0x0077aaf2, 0x0077aef2, 0x0077b2f2, 0x0077b6f2, 0x0077baf2, 0x0077bef2, // 32232-32239
	0x0077c2f2, 0x0077c6f2, 0x0077caf2, 0x0077cef2, 0x0077d2f2, 0x0077d6f2, 0x0077daf2, 0x0077def2, // 32240-32247
	0x0077e2f2, 0x0077e6f2, 0x0077eaf2, 0x0077eef2, 0x0077f2f2, 0x0077f6f2, 0x0077faf2, 0x0077fef2, // 32248-32255
	0x007802f2, 0x007806f2, 0x00780af2, 0x00780ef2, 0x007812f2, 0x007816f2, 0x00781af2, 0x00781ef2, // 32256-32263
	0x007822f2, 0x007826f2, 0x00782af2, 0x00782ef2, 0x007832f2, 0x007836f2, 0x00783af2, 0x00783ef2, // 32264-32271
	0x007842f2, 0x007846f2, 0x00784af2, 0x00784ef2, 0x007852f2, 0x007856f2, 0x00785af2, 0x00785ef2, // 32272-32279
	0x007862f2, 0x007866f2, 0x00786af2, 0x00786ef2, 0x007872f2, 0x007876f2, 0x00787af2, 0x00787ef2, // 32280-32287
	0x007882f2, 0x007886f2, 0x00788af2, 0x00788ef2, 0x007892f2, 0x007896f2, 0x00789af2, 0x00789ef2, // 32288-32295
	0x0078a2f2, 0x0078a6f2, 0x0078aaf2, 0x0078aef2, 0x0078b2f2, 0x0078b6f2, 0x0078baf2, 0x0078bef2, // 32296-32303
	0x0078c2f2, 0x0078c6f2, 0x0078caf2, 0x0078cef2, 0x0078d2f2, 0x0078d6f2, 0x0078daf2, 0x0078def2, // 32304-32311
	0x0078e2f2, 0x0078e6f2, 0x0078eaf2, 0x0078eef2, 0x0078f2f2, 0x0078f6f2, 0x0078faf2, 0x0078fef2, // 32312-32319
	0x007902f2, 0x007906f2, 0x00790af2, 0x00790ef2, 0x007912f2, 0x007916f2, 0x00791af2, 0x00791ef2, // 32320-32327
	0x007922f2, 0x007926f2, 0x00792af2, 0x00792ef2, 0x007932f2, 0x007936f2, 0x00793af2, 0x00793ef2, // 32328-32335
	0x007942f2, 0x007946f2, 0x00794af2, 0x00794ef2, 0x007952f2, 0x007956f2, 0x00795af2, 0x00795ef2, // 32336-32343
	0x007962f2, 0x007966f2, 0x00796af2, 0x00796ef2, 0x007972f2, 0x007976f2, 0x00797af2, 0x00797ef2, // 32344-32351
	0x007982f2, 0x007986f2, 0x00798af2, 0x00798ef2, 0x007992f2, 0x007996f2, 0x00799af2, 0x00799ef2, // 32352-32359
	0x0079a2f2, 0x0079a6f2, 0x0079aaf2, 0x0079aef2, 0x0079b2f2, 0x0079b6f2, 0x0079baf2, 0x0079bef2, // 32360-32367
	0x0079c2f2, 0x0079c6f2, 0x0079caf2, 0x0079cef2, 0x0079d2f2, 0x0079d6f2, 0x0079daf2, 0x0079def2, // 32368-32375
	0x0079e2f2, 0x0079e6f2, 0x0079eaf2, 0x0079eef2, 0x0079f2f2, 0x0079f6f2, 0x0079faf2, 0x0079fef2, // 32376-32383
	0x007a02f2, 0x007a06f2, 0x007a0af2, 0x007a0ef2, 0x007a12f2, 0x007a16f2, 0x007a1af2, 0x007a1ef2, // 32384-32391
	0x007a22f2, 0x007a26f2, 0x007a2af2, 0x007a2ef2, 0x007a32f2, 0x007a36f2, 0x007a3af2, 0x007a3ef2, // 32392-32399
	0x007a42f2, 0x007a46f2, 0x007a4af2, 0x007a4ef2, 0x007a52f2, 0x007a56f2, 0x007a5af2, 0x007a5ef2, // 32400-32407
	0x007a62f2, 0x007a66f2, 0x007a6af2, 0x007a6ef2, 0x007a72f2, 0x007a76f2, 0x007a7af2, 0x007a7ef2, // 32408-32415
	0x007a82f2, 0x007a86f2, 0x007a8af2, 0x007a8ef2, 0x007a92f2, 0x007a96f2, 0x007a9af2, 0x007a9ef2, // 32416-32423
	0x007aa2f2, 0x007aa6f2, 0x007aaaf2, 0x007aaef2, 0x007ab2f2, 0x007ab6f2, 0x007abaf2, 0x007abef2, // 32424-32431
	0x007ac2f2, 0x007ac6f2, 0x007acaf2, 0x007acef2, 0x007ad2f2, 0x007ad6f2, 0x007adaf2, 0x007adef2, // 32432-32439
	0x007ae2f2, 0x007ae6f2, 0x007aeaf2, 0x007aeef2, 0x007af2f2, 0x007af6f2, 0x007afaf2, 0x007afef2, // 32440-32447
	0x007b02f2, 0x007b06f2, 0x007b0af2, 0x007b0ef2, 0x007b12f2, 0x007b16f2, 0x007b1af2, 0x007b1ef2, // 32448-32455
	0x007b22f2, 0x007b26f2, 0x007b2af2, 0x007b2ef2, 0x007b32f2, 0x007b36f2, 0x007b3af2, 0x007b3ef2, // 32456-32463
	0x007b42f2, 0x007b46f2, 0x007b4af2, 0x007b4ef2, 0x007b52f2, 0x007b56f2, 0x007b5af2, 0x007b5ef2, // 32464-32471
	0x007b62f2, 0x007b66f2, 0x007b6af2, 0x007b6ef2, 0x007b72f2, 0x007b76f2, 0x007b7af2, 0x007b7ef2, // 32472-32479
	0x007b82f2, 0x007b86f2, 0x007b8af2, 0x007b8ef2, 0x007b92f2, 0x007b96f2, 0x007b9af2, 0x007b9ef2, // 32480-32487
	0x007ba2f2, 0x007ba6f2, 0x007baaf2, 0x007baef2, 0x007bb2f2, 0x007bb6f2, 0x007bbaf2, 0x007bbef2, // 32488-32495
	0x007bc2f2, 0x007bc6f2, 0x007bcaf2, 0x007bcef2, 0x007bd2f2, 0x007bd6f2, 0x007bdaf2, 0x007bdef2, // 32496-32503
	0x007be2f2, 0x007be6f2, 0x007beaf2, 0x007beef2, 0x007bf2f2, 0x007bf6f2, 0x007bfaf2, 0x007bfef2, // 32504-32511
	0x007c02f2, 0x007c06f2, 0x007c0af2, 0x007c0ef2, 0x007c12f2, 0x007c16f2, 0x007c1af2, 0x007c1ef2, // 32512-32519
	0x007c22f2, 0x007c26f2, 0x007c2af2, 0x007c2ef2, 0x007c32f2, 0x007c36f2, 0x007c3af2, 0x007c3ef2, // 32520-32527
	0x007c42f2, 0x007c46f2, 0x007c4af2, 0x007c4ef2, 0x007c52f2, 0x007c56f2, 0x007c5af2, 0x007c5ef2, // 32528-32535
	0x007c62f2, 0x007c66f2, 0x007c6af2, 0x007c6ef2, 0x007c72f2, 0x007c76f2, 0x007c7af2, 0x007c7ef2, // 32536-32543
	0x007c82f2, 0x007c86f2, 0x007c8af2, 0x007c8ef2, 0x007c92f2, 0x007c96f2, 0x007c9af2, 0x007c9ef2, // 32544-32551
	0x007ca2f2, 0x007ca6f2, 0x007caaf2, 0x007caef2, 0x007cb2f2, 0x007cb6f2, 0x007cbaf2, 0x007cbef2, // 32552-32559
	0x007cc2f2, 0x007cc6f2, 0x007ccaf2, 0x007ccef2, 0x007cd2f2, 0x007cd6f2, 0x007cdaf2, 0x007cdef2, // 32560-32567
	0x007ce2f2, 0x007ce6f2, 0x007ceaf2, 0x007ceef2, 0x007cf2f2, 0x007cf6f2, 0x007cfaf2, 0x007cfef2, // 32568-32575
	0x007d02f2, 0x007d06f2, 0x007d0af2, 0x007d0ef2, 0x007d12f2, 0x007d16f2, 0x007d1af2, 0x007d1ef2, // 32576-32583
	0x007d22f2, 0x007d26f2, 0x007d2af2, 0x007d2ef2, 0x007d32f2, 0x007d36f2, 0x007d3af2, 0x007d3ef2, // 32584-32591
	0x007d42f2, 0x007d46f2, 0x007d4af2, 0x007d4ef2, 0x007d52f2, 0x007d56f2, 0x007d5af2, 0x007d5ef2, // 32592-32599
	0x007d62f2, 0x007d66f2, 0x007d6af2, 0x007d6ef2, 0x007d72f2, 0x007d76f2, 0x007d7af2, 0x007d7ef2, // 32600-32607
	0x007d82f2, 0x007d86f2, 0x007d8af2, 0x007d8ef2, 0x007d92f2, 0x007d96f2, 0x007d9af2, 0x007d9ef2, // 32608-32615
	0x007da2f2, 0x007da6f2, 0x007daaf2, 0x007daef2, 0x007db2f2, 0x007db6f2, 0x007dbaf2, 0x007dbef2, // 32616-32623
	0x007dc2f2, 0x007dc6f2, 0x007dcaf2, 0x007dcef2, 0x007dd2f2, 0x007dd6f2, 0x007ddaf2, 0x007ddef2, // 32624-32631
	0x007de2f2, 0x007de6f2, 0x007deaf2, 0x007deef2, 0x007df2f2, 0x007df6f2, 0x007dfaf2, 0x007dfef2, // 32632-32639
	0x007e02f2, 0x007e06f2, 0x007e0af2, 0x007e0ef2, 0x007e12f2, 0x007e16f2, 0x007e1af2, 0x007e1ef2, // 32640-32647
	0x007e22f2, 0x007e26f2, 0x007e2af2, 0x007e2ef2, 0x007e32f2, 0x007e36f2, 0x007e3af2, 0x007e3ef2, // 32648-32655
	0x007e42f2, 0x007e46f2, 0x007e4af2, 0x007e4ef2, 0x007e52f2, 0x007e56f2, 0x007e5af2, 0x007e5ef2, // 32656-32663
	0x007e62f2, 0x007e66f2, 0x007e6af2, 0x007e6ef2, 0x007e72f2, 0x007e76f2, 0x007e7af2, 0x007e7ef2, // 32664-32671
	0x007e82f2, 0x007e86f2, 0x007e8af2, 0x007e8ef2, 0x007e92f2, 0x007e96f2, 0x007e9af2, 0x007e9ef2, // 32672-32679
	0x007ea2f2, 0x007ea6f2, 0x007eaaf2, 0x007eaef2, 0x007eb2f2, 0x007eb6f2, 0x007ebaf2, 0x007ebef2, // 32680-32687
	0x007ec2f2, 0x007ec6f2, 0x007ecaf2, 0x007ecef2, 0x007ed2f2, 0x007ed6f2, 0x007edaf2, 0x007edef2, // 32688-32695
	0x007ee2f2, 0x007ee6f2, 0x007eeaf2, 0x007eeef2, 0x007ef2f2, 0x007ef6f2, 0x007efaf2, 0x007efef2, // 32696-32703
	0x007f02f2, 0x007f06f2, 0x007f0af2, 0x007f0ef2, 0x007f12f2, 0x007f16f2, 0x007f1af2, 0x007f1ef2, // 32704-32711
	0x007f22f2, 0x007f26f2, 0x007f2af2, 0x007f2ef2, 0x007f32f2, 0x007f36f2, 0x007f3af2, 0x007f3ef2, // 32712-32719
	0x007f42f2, 0x007f46f2, 0x007f4af2, 0x007f4ef2, 0x007f52f2, 0x007f56f2, 0x007f5af2, 0x007f5ef2, // 32720-32727
	0x007f62f2, 0x007f66f2, 0x007f6af2, 0x007f6ef2, 0x007f72f2, 0x007f76f2, 0x007f7af2, 0x007f7ef2, // 32728-32735
	0x007f82f2, 0x007f86f2, 0x007f8af2, 0x007f8ef2, 0x007f92f2, 0x007f96f2, 0x007f9af2, 0x007f9ef2, // 32736-32743
	0x007fa2f2, 0x007fa6f2, 0x007faaf2, 0x007faef2, 0x007fb2f2, 0x007fb6f2, 0x007fbaf2, 0x007fbef2, // 32744-32751
	0x007fc2f2, 0x007fc6f2, 0x007fcaf2, 0x007fcef2, 0x007fd2f2, 0x007fd6f2, 0x007fdaf2, 0x007fdef2, // 32752-32759
	0x007fe2f2, 0x007fe6f2, 0x007feaf2, 0x007feef2, 0x007ff2f2, 0x007ff6f2, 0x007ffaf2, 0x007ffef2, // 32760-32767
};
#endif /* ifndef PRECOMPUTE_TABLES */

/* Make the table for a fast CRC.
 * Not thread-safe, must be called exactly once.
 */
static inline void __slz_make_crc_table(void)
{
#ifndef PRECOMPUTE_TABLES
	uint32_t c;
	int n, k;

	for (n = 0; n < 256; n++) {
		c = (uint32_t) n ^ 255;
		for (k = 0; k < 8; k++) {
			if (c & 1) {
				c = 0xedb88320 ^ (c >> 1);
			} else {
				c = c >> 1;
			}
		}
		crc32_fast[0][n] = c ^ 0xff000000;
	}

	/* Note: here we *do not* have to invert the bits corresponding to the
	 * byte position, because [0] already has the 8 highest bits inverted,
	 * and these bits are shifted by 8 at the end of the operation, which
	 * results in having the next 8 bits shifted in turn. That's why we
	 * have the xor in the index used just after a computation.
	 */
	for (n = 0; n < 256; n++) {
		crc32_fast[1][n] = 0xff000000 ^ crc32_fast[0][(0xff000000 ^ crc32_fast[0][n] ^ 0xff) & 0xff] ^ (crc32_fast[0][n] >> 8);
		crc32_fast[2][n] = 0xff000000 ^ crc32_fast[0][(0x00ff0000 ^ crc32_fast[1][n] ^ 0xff) & 0xff] ^ (crc32_fast[1][n] >> 8);
		crc32_fast[3][n] = 0xff000000 ^ crc32_fast[0][(0x0000ff00 ^ crc32_fast[2][n] ^ 0xff) & 0xff] ^ (crc32_fast[2][n] >> 8);
	}
#endif
}

/* Returns code for lengths 1 to 32768. The bit size for the next value can be
 * found this way :
 *
 *	bits = code >> 1;
 *	if (bits)
 *		bits--;
 *
 */
static inline uint32_t dist_to_code(uint32_t l)
{
	uint32_t code;

	code = 0;
	switch (l) {
	case 24577 ... 32768: code++;
	case 16385 ... 24576: code++;
	case 12289 ... 16384: code++;
	case 8193 ... 12288: code++;
	case 6145 ... 8192: code++;
	case 4097 ... 6144: code++;
	case 3073 ... 4096: code++;
	case 2049 ... 3072: code++;
	case 1537 ... 2048: code++;
	case 1025 ... 1536: code++;
	case 769 ... 1024: code++;
	case 513 ... 768: code++;
	case 385 ... 512: code++;
	case 257 ... 384: code++;
	case 193 ... 256: code++;
	case 129 ... 192: code++;
	case 97 ... 128: code++;
	case 65 ... 96: code++;
	case 49 ... 64: code++;
	case 33 ... 48: code++;
	case 25 ... 32: code++;
	case 17 ... 24: code++;
	case 13 ... 16: code++;
	case 9 ... 12: code++;
	case 7 ... 8: code++;
	case 5 ... 6: code++;
	case 4: code++;
	case 3: code++;
	case 2: code++;
	}

	return code;
}

/* not thread-safe, must be called exactly once */
static inline void __slz_prepare_dist_table()
{
#ifndef PRECOMPUTE_TABLES
	uint32_t dist;
	uint32_t code;
	uint32_t bits;

	for (dist = 0; dist < sizeof(fh_dist_table) / sizeof(*fh_dist_table); dist++) {
		code = dist_to_code(dist + 1);
		bits = code >> 1;
		if (bits)
			bits--;

		/* Distance codes are stored on 5 bits reversed. The RFC
		 * doesn't state that they are reversed, but it's the only
		 * way it works.
		 */
		code = ((code & 0x01) << 4) | ((code & 0x02) << 2) |
		       (code & 0x04) |
		       ((code & 0x08) >> 2) | ((code & 0x10) >> 4);

		code += (dist & ((1 << bits) - 1)) << 5;
		fh_dist_table[dist] = (code << 5) + bits + 5;
	}
#endif
}
