// File and Version Information:
//      $Id: SprTrainedDecisionTree.hh,v 1.6 2008-01-03 20:51:58 narsky Exp $
//
// Description:
//      Class SprTrainedDecisionTree :
//         Implements a trained decision tree. The constructor input
//         is a vector of rectangular regions, where each region is in turn
//         represented by a vector of pairs, lower and upper bounds.
//         The tree can use as many rectangular regions as you'd like.
//         The length of each vector representing one rectangular region
//         must be equal to the dimensionality of the observable space.
//
// Environment:
//      Software developed for the BaBar Detector at the SLAC B-Factory.
//
// Author List:
//      Ilya Narsky                     Original author
//
// Copyright Information:
//      Copyright (C) 2005              California Institute of Technology
//
//------------------------------------------------------------------------
 
#ifndef _SprTrainedDecisionTree_HH
#define _SprTrainedDecisionTree_HH

#include "StatPatternRecognition/SprAbsTrainedClassifier.hh"
#include "StatPatternRecognition/SprUtils.hh"
#include "StatPatternRecognition/SprDefs.hh"

#include <iostream>
#include <string>
#include <vector>
#include <utility>


class SprTrainedDecisionTree : public SprAbsTrainedClassifier
{
public:
  virtual ~SprTrainedDecisionTree() {}

  /*
    The constructor takes a vector of terminal signal nodes. Each node
    is represented by a map, where the map key is the input dimension, 
    and the map element is a pair that gives the lower and upper
    bounds on the node in this dimension.
  */
  SprTrainedDecisionTree(const std::vector<SprBox>& nodes1)
    :
    SprAbsTrainedClassifier(),
    nodes1_(nodes1)
  {
    this->setCut(SprUtils::lowerBound(0.5));
  }

  SprTrainedDecisionTree(const SprTrainedDecisionTree& other)
    :
    SprAbsTrainedClassifier(other),
    nodes1_(other.nodes1_)
  {}

  /*
    Returns classifier name.
  */
  std::string name() const {
    return "DecisionTree";
  }

  /*
    Make a clone.
  */
  SprTrainedDecisionTree* clone() const {
    return new SprTrainedDecisionTree(*this);
  }

  /*
    Classifier response for a data point. 
  */
  double response(const std::vector<double>& v) const;

  /*
    Generate code.
  */
  bool generateCode(std::ostream& os) const {
    return false;
  }

  // Change normalization.
  void useStandard() {}
  void useNormalized() {}
  bool normalized() const { return true; }

  /*
    Print out.
  */
  void print(std::ostream& os) const;

  // Local methods.

  // return number of nodes
  unsigned nNodes() const { 
    return nodes1_.size(); 
  }

  // return box for the i-th node
  void box(int i, SprBox& limits) const {
    if( i<0 || i>=nodes1_.size() )
      limits.clear();
    else
      limits = nodes1_[i];
  }

  // return all nodes
  void nodes(std::vector<SprBox>& nodes) const {
    nodes = nodes1_;
  }

  // return the box number to which this vector belongs; -1 if none
  int nBox(const std::vector<double>& v) const;

protected:
  std::vector<SprBox> nodes1_;
};

#endif
