/*
 * Copyright 2018 Comcast Cable Communications Management, LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package strings

import (
	"strings"
	"testing"
)

func TestIndexOfString(t *testing.T) {

	arr := []string{"string0", "string1", "string2"}

	i := IndexOfString(arr, "string0")
	if i != 0 {
		t.Errorf(`expected 0. got %d`, i)
	}

	i = IndexOfString(arr, "string3")
	if i != -1 {
		t.Errorf(`expected -1. got %d`, i)
	}

}

func TestCloneMap(t *testing.T) {

	m1 := CloneMap(nil)
	if m1 != nil {
		t.Error("expected nil map")
	}

	const expected = "pass"

	m := map[string]string{"test": expected}
	m2 := CloneMap(m)

	v, ok := m2["test"]
	if !ok {
		t.Errorf("expected true got %t", ok)
	}

	if v != expected {
		t.Errorf("expected %s got %s", expected, v)
	}

}

func TestEqual(t *testing.T) {

	l1 := []string{"test1", "test2"}
	l2 := []string{"test1", "test2"}
	l3 := []string{"test3", "test4"}
	l4 := []string{}

	if !Equal(l1, l2) {
		t.Error("expected true got false")
	}

	if Equal(l1, l3) {
		t.Error("expected false got true")
	}

	if Equal(l1, l4) {
		t.Error("expected false got true")
	}
}

func TestStringMap(t *testing.T) {

	sm := StringMap(map[string]string{"test": "value"})
	s := sm.String()
	const expected = `{"test":"value"}`

	if s != expected {
		t.Errorf("expected %s got %s", expected, s)
	}

}

func TestCloneBoolMap(t *testing.T) {

	m1 := CloneBoolMap(nil)
	if m1 != nil {
		t.Error("expected nil map")
	}

	const expected = true

	m := map[string]bool{"test": expected}
	m2 := CloneBoolMap(m)

	v, ok := m2["test"]
	if !ok {
		t.Errorf("expected true got %t", ok)
	}

	if v != expected {
		t.Errorf("expected %t got %t", expected, v)
	}

}

func TestCloneList(t *testing.T) {

	m1 := CloneList(nil)
	if m1 != nil {
		t.Error("expected nil map")
	}

	m := []string{"test"}

	m2 := CloneList(m)
	if len(m2) != 1 {
		t.Errorf("expected %d got %d", 1, len(m2))
	}
	if m2[0] != "test" {
		t.Errorf("expected %s got %s", "test", m2[0])
	}

}

func TestUnique(t *testing.T) {
	initial := []string{"test", "test", "test1", "test2", "test2", "test", "test3"}
	expected := "test,test1,test2,test3"
	after := strings.Join(Unique(initial), ",")
	if expected != after {
		t.Errorf("expected %s got %s", expected, after)
	}

	empty := Unique(nil)
	if len(empty) != 0 {
		t.Error("expected empty list")
	}
}
