// MIT License
//
// Copyright (c) 2021 Augustin Husson
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
import { walk } from './walk';
import { match, render } from '@nexucis/fuzzy';
export function isQueryNode(q) {
    return q.right !== undefined;
}
export function isQuery(q) {
    return q.keyPath !== undefined;
}
// merge is called only when a.original and b.original are equal, and so the result must be merged
function merge(a, b) {
    const result = {
        original: a.original,
        rendered: a.rendered,
        score: a.score + b.score,
        index: a.index,
    };
    let matching;
    if (a.matched && b.matched) {
        matching = a.matched.concat(b.matched);
    }
    else if (a.matched) {
        matching = a.matched;
    }
    else if (b.matched) {
        matching = b.matched;
    }
    if (matching) {
        result.matched = matching;
    }
    return result;
}
// union will concat 'a' with 'b'.
// Note: this function is exported only for testing purpose.
export function union(a, b) {
    // here we don't copy a because it cost (a lot) to do it.
    const result = a;
    for (const [k, v] of Object.entries(b)) {
        const index = k;
        if (result[index] !== undefined) {
            result[index] = merge(result[index], v);
        }
        else {
            result[index] = v;
        }
    }
}
// intersect will keep only the results that are present in 'a' and 'b'
// Note: this function is exported only for testing purpose.
export function intersect(a, b) {
    const result = {};
    for (const [k, v] of Object.entries(b)) {
        const index = k;
        if (a[index] !== undefined) {
            result[index] = merge(a[index], v);
        }
    }
    return result;
}
function exactMatch(pattern, value, caseSensitive) {
    return genericMatch(pattern, value, (a, b) => a === b, caseSensitive);
}
function negativeMatch(pattern, value, caseSensitive) {
    return genericMatch(pattern, value, (a, b) => a !== b, caseSensitive);
}
function greaterMatch(pattern, value, caseSensitive) {
    return genericMatch(pattern, value, (a, b) => a > b, caseSensitive);
}
function greaterOrEqualMatch(pattern, value, caseSensitive) {
    return genericMatch(pattern, value, (a, b) => a >= b, caseSensitive);
}
function lessMatch(pattern, value, caseSensitive) {
    return genericMatch(pattern, value, (a, b) => a < b, caseSensitive);
}
function lessOrEqualMatch(pattern, value, caseSensitive) {
    return genericMatch(pattern, value, (a, b) => a <= b, caseSensitive);
}
function genericMatch(pattern, value, operator, caseSensitive) {
    let localPattern = pattern;
    let localValue = value;
    if (!caseSensitive && typeof localPattern === 'string' && typeof localValue == 'string') {
        localPattern = localPattern.toLocaleLowerCase();
        localValue = localValue.toLocaleLowerCase();
    }
    return operator(localValue, localPattern);
}
function buildQuery(pattern, key) {
    const query = { match: 'fuzzy', pattern: pattern, keyPath: [] };
    if (Array.isArray(key)) {
        query.keyPath = query.keyPath.concat(key);
    }
    else {
        query.keyPath = [key];
    }
    return query;
}
export class KVSearch {
    constructor(conf) {
        this.conf = {
            caseSensitive: (conf === null || conf === void 0 ? void 0 : conf.caseSensitive) === undefined ? false : conf.caseSensitive,
            includeMatches: (conf === null || conf === void 0 ? void 0 : conf.includeMatches) === undefined ? false : conf.includeMatches,
            shouldSort: (conf === null || conf === void 0 ? void 0 : conf.shouldSort) === undefined ? false : conf.shouldSort,
            escapeHTML: (conf === null || conf === void 0 ? void 0 : conf.escapeHTML) === undefined ? false : conf.escapeHTML,
            findAllMatches: (conf === null || conf === void 0 ? void 0 : conf.findAllMatches) == undefined ? false : conf.findAllMatches,
            pre: conf === null || conf === void 0 ? void 0 : conf.pre,
            post: conf === null || conf === void 0 ? void 0 : conf.post,
            indexedKeys: conf === null || conf === void 0 ? void 0 : conf.indexedKeys,
        };
    }
    filter(pattern, list, conf) {
        const indexedKeys = (conf === null || conf === void 0 ? void 0 : conf.indexedKeys) ? conf.indexedKeys : this.conf.indexedKeys;
        if (indexedKeys === undefined) {
            return [];
        }
        // build the query according to the pattern and the indexedKeys.
        const indexedKeysLength = indexedKeys.length;
        const firstQuery = buildQuery(pattern, indexedKeys[0]);
        if (indexedKeysLength === 1) {
            return this.filterWithQuery(firstQuery, list, conf);
        }
        const firstElement = {
            operator: 'or',
            left: firstQuery,
            right: firstQuery
        };
        let lastElement = firstElement;
        for (let i = 1; i < indexedKeysLength; i++) {
            const query = buildQuery(pattern, indexedKeys[i]);
            if (i + 1 < indexedKeysLength) {
                // here we have to create a new QueryNode
                const node = { operator: 'or', left: query, right: query };
                lastElement.right = node;
                lastElement = node;
            }
            else {
                lastElement.right = query;
            }
        }
        return this.filterWithQuery(firstElement, list, conf);
    }
    filterWithQuery(query, list, conf) {
        const shouldSort = (conf === null || conf === void 0 ? void 0 : conf.shouldSort) !== undefined ? conf.shouldSort : this.conf.shouldSort;
        const includeMatches = (conf === null || conf === void 0 ? void 0 : conf.includeMatches) !== undefined ? conf.includeMatches : this.conf.includeMatches;
        const findAllMatches = (conf === null || conf === void 0 ? void 0 : conf.findAllMatches) !== undefined ? conf.findAllMatches : this.conf.findAllMatches;
        const queryNodes = [{
                current: query,
                depth: 0
            }];
        // For example, you have the following tree :
        //            OR
        //           /  \
        //          /    \
        //   Query "left"  Query "right"
        //
        // So Each query, "Query "left"" and "Query "right"" are returning a map of KVSearchResult that are stored in results.
        // Once we arrived at the node "OR", we pop the result from both queries and we merged them.
        const results = [];
        while (queryNodes.length > 0) {
            const currentNode = queryNodes.shift();
            if (!currentNode) {
                break;
            }
            const currentQuery = currentNode.current;
            if (isQueryNode(currentQuery)) {
                // As we are doing a DFS (deep first searching, pre-order), we inject first the left node, then the right.
                // And as we need to merge the result coming from the left and right node according to the current node operator, we add as well the operator.
                queryNodes.unshift({
                    current: currentQuery.left,
                    parent: currentQuery.operator,
                    depth: currentNode.depth + 1
                }, {
                    current: currentQuery.right,
                    parent: currentQuery.operator,
                    depth: currentNode.depth + 1
                }, {
                    current: currentQuery.operator,
                    parent: currentNode.parent,
                    depth: currentNode.depth,
                });
                continue;
            }
            if (isQuery(currentQuery)) {
                // time to execute the actual query
                // In case we have a previous result at the same depth as the current query and we don't have to find all matches,
                // we should use the previous result and reduce the number of object to look at it. It will depend of the parent node is a "or" or a "and".
                // In case it's a "or", we are interesting to search/match every node not already present in the previous result.
                // In case it's a "and", we are interesting to search/match every node already present in the previous result.
                let previousResult = undefined;
                if (!findAllMatches && results.length > 1 && results[results.length - 1].depth === currentNode.depth && currentNode.parent !== undefined) {
                    previousResult = { result: results[results.length - 1].result, parent: currentNode.parent };
                }
                results.push({
                    result: this.executeQuery(currentQuery, list, previousResult, conf),
                    depth: currentNode.depth
                });
                continue;
            }
            // At this point the currentQuery is actually the operator coming from a previous QueryNode.
            // So we just have to merge the result from both branch.
            const a = results.pop();
            const b = results.pop();
            if (!a || !b) {
                // That would be weird since each branch of the queryNode should have provided a result
                break;
            }
            if (currentQuery === 'or') {
                union(a.result, b.result);
                results.push({ result: a.result, depth: currentNode.depth });
            }
            else {
                results.push({ result: intersect(a.result, b.result), depth: currentNode.depth });
            }
        }
        let finalResult = Object.values(results[0].result);
        for (let i = 0; i < finalResult.length; i++) {
            finalResult[i].rendered = this.render(finalResult[i].original, finalResult[i].matched, conf);
            if (!includeMatches) {
                finalResult[i].matched = undefined;
            }
        }
        if (shouldSort) {
            finalResult = finalResult.sort((a, b) => {
                return b.score - a.score;
            });
        }
        return finalResult ? finalResult : [];
    }
    match(query, obj, conf) {
        const includeMatches = (conf === null || conf === void 0 ? void 0 : conf.includeMatches) !== undefined ? conf.includeMatches : this.conf.includeMatches;
        const matched = this.internalMatch(query, obj, conf);
        if (matched !== null) {
            matched.original = obj;
            matched.rendered = this.render(obj, matched.matched, conf);
            if (!includeMatches) {
                matched.matched = undefined;
            }
        }
        return matched;
    }
    render(obj, matchingResults, conf) {
        const pre = (conf === null || conf === void 0 ? void 0 : conf.pre) ? conf.pre : this.conf.pre;
        const post = (conf === null || conf === void 0 ? void 0 : conf.post) ? conf.post : this.conf.post;
        if ((pre === undefined && post === undefined) || matchingResults === undefined) {
            return obj;
        }
        // copy the object to get a different reference
        // TODO find a more efficient way to deep copy an object
        const rendered = JSON.parse(JSON.stringify(obj));
        const associatePeer = {};
        for (const matchingResult of matchingResults) {
            this.renderingWalk(matchingResult, rendered, associatePeer, conf);
        }
        return rendered;
    }
    renderingWalk(matchingResult, obj, associatedPeer, conf) {
        // eslint-disable-next-line  @typescript-eslint/no-explicit-any
        let currentObj = obj;
        let i = 0;
        const path = matchingResult.path;
        while (i < path.length - 1) {
            //TODO manage array
            const tmp = currentObj[path[i]];
            if (tmp !== undefined) {
                currentObj = currentObj[path[i]];
            }
            else {
                currentObj = currentObj[associatedPeer[path.slice(0, i).toString()]];
            }
            i++;
        }
        const fuzzyConf = {
            pre: (conf === null || conf === void 0 ? void 0 : conf.pre) ? conf.pre : this.conf.pre,
            post: (conf === null || conf === void 0 ? void 0 : conf.post) ? conf.post : this.conf.post,
            escapeHTML: (conf === null || conf === void 0 ? void 0 : conf.escapeHTML) ? conf.escapeHTML : this.conf.escapeHTML
        };
        let lastKey = path[i];
        if (typeof currentObj[lastKey] === 'object' && typeof matchingResult.value === 'string') {
            currentObj = currentObj[lastKey];
            // in that case, the value is a key of object.
            // So we need to render this key, to add it to the object and then remove the previous one
            const renderedKey = render(matchingResult.value, matchingResult.intervals, fuzzyConf);
            currentObj[renderedKey] = currentObj[matchingResult.value];
            // Save the association between the old key and the new one.
            // It will be used in case we have other matching that are passing with this key.
            associatedPeer[path.concat(matchingResult.value).toString()] = renderedKey;
            delete (currentObj[matchingResult.value]);
        }
        else {
            if (currentObj[lastKey] === undefined) {
                // It means the key changed (aka rendered) in a previous iteration, so we need to get the new key
                lastKey = associatedPeer[path.toString()];
            }
            currentObj[lastKey] = render(currentObj[lastKey], matchingResult.intervals, fuzzyConf);
        }
    }
    executeQuery(query, list, previousResult, conf) {
        const result = {};
        for (let i = 0; i < list.length; i++) {
            const el = list[i];
            if (previousResult) {
                if (previousResult.parent === 'or') {
                    if (previousResult.result[i]) {
                        // in that case this object already matched and we don't need to check again
                        continue;
                    }
                }
                else {
                    if (!previousResult.result[i]) {
                        // In that case this object hasn't been matched by the previous query, so it's impossible with an "and" operator to have this object in the final result".
                        continue;
                    }
                }
            }
            const matched = this.internalMatch(query, el, conf);
            if (matched !== null) {
                matched.index = i;
                matched.original = el;
                result[i] = matched;
            }
        }
        return result;
    }
    internalMatch(query, obj, conf) {
        // walking through the object until finding the final key used to perform the query
        const endPath = walk(query.keyPath, obj);
        return this.processWalkingPath(endPath, query, conf);
    }
    processWalkingPath(walkingPath, query, conf) {
        let result = null;
        if (walkingPath === null) {
            return null;
        }
        if (Array.isArray(walkingPath)) {
            for (const existingPath of walkingPath) {
                const tmp = this.recursiveMatch(existingPath.value, existingPath.path, query, conf);
                if (tmp !== null) {
                    if (result != null) {
                        result = merge(tmp, result);
                    }
                    else {
                        result = tmp;
                    }
                }
            }
        }
        else {
            result = this.recursiveMatch(walkingPath.value, walkingPath.path, query, conf);
        }
        return result;
    }
    recursiveMatch(value, path, query, conf) {
        let result = null;
        if (typeof value === 'string' || typeof value === 'number') {
            result = this.matchSingleStringOrNumber(value, path, query, conf);
        }
        else if (Array.isArray(value)) {
            for (const r of value) {
                const tmp = this.recursiveMatch(r, path, query, conf);
                if (tmp !== null) {
                    if (result != null) {
                        result = merge(tmp, result);
                    }
                    else {
                        result = tmp;
                    }
                }
            }
        }
        else {
            for (const key of Object.keys(value)) {
                result = this.matchSingleStringOrNumber(key, path, query, conf);
                if (result !== null) {
                    // TODO by configuration decide if we should continue furthermore to grab all result.
                    break;
                }
            }
        }
        return result;
    }
    matchSingleStringOrNumber(value, path, query, conf) {
        const caseSensitive = (conf === null || conf === void 0 ? void 0 : conf.caseSensitive) !== undefined ? conf.caseSensitive : this.conf.caseSensitive;
        switch (query.match) {
            case 'exact': {
                if (!exactMatch(query.pattern, value, caseSensitive)) {
                    return null;
                }
                return {
                    // for scoring here, let's use the same value than the one used for the fuzzy search.
                    // It will be coherent when you are mixing query with fuzzy and exact match.
                    score: Infinity,
                    matched: [{
                            path: path,
                            value: value,
                            intervals: typeof value === 'string' ? [{ from: 0, to: value.length - 1 }] : []
                        }]
                };
            }
            case 'negative': {
                if (!negativeMatch(query.pattern, value, caseSensitive)) {
                    return null;
                }
                return {
                    score: 1,
                    matched: [{
                            path: path,
                            value: value,
                            intervals: []
                        }]
                };
            }
            case 'fuzzy': {
                if (typeof query.pattern !== 'string' || typeof value !== 'string') {
                    return null;
                }
                const fuzzyResult = match(query.pattern, value, {
                    includeMatches: true,
                    caseSensitive: caseSensitive
                });
                if (fuzzyResult === null) {
                    return null;
                }
                return {
                    score: fuzzyResult.score,
                    matched: [
                        {
                            path: path,
                            value: value,
                            intervals: fuzzyResult.intervals ? fuzzyResult.intervals : [],
                        }
                    ]
                };
            }
            case 'greater':
                if (!greaterMatch(query.pattern, value, caseSensitive)) {
                    return null;
                }
                return {
                    score: 1,
                    matched: [{
                            path: path,
                            value: value,
                            intervals: []
                        }]
                };
            case 'greaterEqual':
                if (!greaterOrEqualMatch(query.pattern, value, caseSensitive)) {
                    return null;
                }
                return {
                    score: 1,
                    matched: [{
                            path: path,
                            value: value,
                            intervals: []
                        }]
                };
            case 'less':
                if (!lessMatch(query.pattern, value, caseSensitive)) {
                    return null;
                }
                return {
                    score: 1,
                    matched: [{
                            path: path,
                            value: value,
                            intervals: []
                        }]
                };
            case 'lessEqual':
                if (!lessOrEqualMatch(query.pattern, value, caseSensitive)) {
                    return null;
                }
                return {
                    score: 1,
                    matched: [{
                            path: path,
                            value: value,
                            intervals: []
                        }]
                };
        }
    }
}
