;;; A small process-control library for Guile.
;;; Copyright (C) 1997, 2000, 2001 Free Software Foundation, Inc.
;;; version 1.3, 25 June 2001.

(define-module (goosh))

(export tail-call-program run run-concurrently run-with-pipe)
(export-syntax run-concurrently+ run+ tail-call-pipeline+
	       tail-call-pipeline)

;; these are exported because they appear in code generated by
;; macros.
(export pipe-make-redir-commands pipe-make-commands
	setup-redirected-port new-comm-pipes)
(export-syntax pipe-fork-child)

;; setup file descriptors 0, 1, 2 from the current Scheme ports, if
;; possible.  if some of these ports can not be used, open new
;; descriptors on /dev/null.

(define (stdports->stdio)

  ;; select the three file descriptors to be used as
  ;; standard descriptors 0, 1, 2 for the new process.

  (let* ((ensure-fdes (lambda (port mode)
			(or (false-if-exception (fileno port))
			    (open-fdes *null-device* mode))))

	 (input-fdes (ensure-fdes (current-input-port) O_RDONLY))
	 (output-fdes (ensure-fdes (current-output-port) O_WRONLY))
	 (error-fdes (ensure-fdes (current-error-port) O_WRONLY)))

    ;; copy the three selected descriptors to the standard
    ;; descriptors 0, 1, 2.  note that it's possible that
    ;; any of output-fdes, input-fdes and error-fdes are equal.

    (cond ((not (= input-fdes 0))
	   (if (= output-fdes 0)
	       (set! output-fdes (dup->fdes 0)))
	   (if (= error-fdes 0)
	       (set! error-fdes (dup->fdes 0)))
	   (dup2 input-fdes 0)))

    (cond ((not (= output-fdes 1))
	   (if (= error-fdes 1)
	       (set! error-fdes (dup->fdes 1)))
	   (dup2 output-fdes 1)))

    (dup2 error-fdes 2)))

(define (tail-call-program prog . args)
  (set-batch-mode?! #t)
  (stdports->stdio)
  (apply execlp (cons prog (cons prog args))))

;;; create a pipe with the writing end unbuffered.  the reading end doesn't
;;; matter, making it unbuffered would just slow things down.
(define (unbuffered-pipe)
  (let ((result (pipe)))
    (setvbuf (cdr result) _IONBF)
    result))

(defmacro-public run-concurrently+ (proc . connections)
  (let ((pid (gensym))
	(ports (gensym)))
    `(let ((,pid (primitive-fork))
	   (,ports (list)))
       (cond ((= ,pid 0)
	      ;; child
	      (set-batch-mode?! #t)
	      ,@(pipe-make-redir-commands connections ports)
	      ,proc
	      (primitive-exit 1))
	     (else
	      ,pid)))))

;;; generate the code needed to set up redirections for a child process.
(define (pipe-make-redir-commands connections portvar)
  (let next-conn ((conns connections)
		  (insert (list))         ;; result
		  (slave #f)
		  (no-auto-close #f))
    (cond ((null? conns)
	   (cond (slave
		  (next-conn conns
			     (append insert
				     (list
				      ;; make a new session, drop old ctty.
				      '(setsid)
				      ;; get a new ctty if possible.
				      '(cond ((isatty? (current-input-port))
					      ;; opening the tty should make
					      ;; it the ctty, now we are the
					      ;; session leader.
					      (let ((name
						     (ttyname 
						      (current-input-port)))
						    (mode 
							(port-mode
							 (current-input-port))))
						(close-port
						 (current-input-port))
						(set-current-input-port
						 (open-file name mode))))
					     ;; try this too -- required
					     ;; under BSD?.
					     ;(%set-ctty (current-input-port))
					     )))
			     #f
			     no-auto-close))
		 (no-auto-close
		  (append insert
			  (list
			   `(map (lambda (p)
				   (false-if-exception
				    (close-fdes (fileno p))))
				 ,portvar))))
		 (else
		  (append insert
			  (list
			   `(let loop ((pts (append
					     (list
					      (current-input-port)
					      (current-output-port)
					      (current-error-port))
					     ,portvar)) ; keep open.
				       (fds (list))) ; fdes keep open.
			      (if (null? pts)
				  (port-for-each (lambda (p)
						   (let ((f
							  (false-if-exception
							   (fileno p))))
						     (if (and f
							      (not
							       (memv f fds)))
							 (false-if-exception
							  (close-fdes f))))))
				  (loop (cdr pts)
					(let ((fd (false-if-exception 
						   (fileno (car pts)))))
					  (if fd
					      (cons fd fds)
					      fds))))))))))
	  (else
	   (let* ((c (car conns)))
	     (cond ((eq? c #:slave)
		    (next-conn (cdr conns)
			       insert
			       #t
			       no-auto-close))
		   ((eq? c #:no-auto-close)
		    (next-conn (cdr conns)
			       insert
			       slave
			       #t))
		   ((eq? c #:foreground) ; would be processed earlier.
		    (next-conn (cdr conns)
			       insert
			       slave
			       no-auto-close))
		   ((= (length c) 1)
		    (next-conn
		     (cdr conns)
		     (cons
		      `(set! ,portvar (cons ,(car c) ,portvar))
		      insert)
		     slave
		     no-auto-close))
		   (else
		    (let* ((reversed (number? (cadr c)))
			   (in (if reversed
				   (cadr c)
				   (car c)))
			   (out (if reversed
				    (car c)
				    (cadr c))))
		      (next-conn (cdr conns)
				 (append
				  (pipe-make-commands
				   in out portvar)
				  insert)
				 slave
				 no-auto-close)))))))))


;;; returns the commands for redirecting a single port in the child.
(define (pipe-make-commands fdes port portvar)
  (if (= fdes 0)
      `((let ((newport (setup-redirected-port ,port ,fdes)))
	  (set-current-input-port newport)))
      (if (= fdes 1)
	  `((let ((newport (setup-redirected-port ,port ,fdes)))
	      (set-current-output-port newport)))
	  (if (= fdes 2)
	      `((let ((newport (setup-redirected-port ,port ,fdes)))
		  (set-current-error-port newport)))
	      `((let ((newport (setup-redirected-port ,port ,fdes)))
		  (set! ,portvar (cons newport ,portvar))))))))

;;; safely redirect a port to a file descriptor.  it must usually be
;;; duplicated, in case it's redirected more than once.
(define (setup-redirected-port port fdes)
  (if (= (fileno port) fdes)
      port
      (let ((newport (duplicate-port port (port-mode port))))
	(primitive-move->fdes newport fdes)
	newport)))

(defmacro run+ (expr . connections)
  `(cdr (waitpid (run-concurrently+ ,expr #:foreground ,@connections))))

(define (run prog . args)
  (run+ (apply tail-call-program prog args)))

(define (run-concurrently . args)
  (run-concurrently+ (apply tail-call-program args)))

(define (run-with-pipe mode prog . args)
  (cond ((string=? mode OPEN_READ)
	 (let* ((upipe (unbuffered-pipe))
		(pid (run-concurrently+ (apply tail-call-program prog args)
					(1 (cdr upipe)))))
	   (close-port (cdr upipe))
	   (cons pid (car upipe))))
	((string=? mode OPEN_WRITE)
	 (let* ((upipe (unbuffered-pipe))
		(pid (run-concurrently+ (apply tail-call-program prog args)
					(0 (car upipe)))))
	   (close-port (car upipe))
	   (cons pid (cdr upipe))))
	((string=? mode OPEN_BOTH)
	 (let* ((upipe-r (unbuffered-pipe))
		(upipe-w (unbuffered-pipe))
		(pid (run-concurrently+ (apply tail-call-program prog args)
					(0 (car upipe-w))
					(1 (cdr upipe-r)))))
	   (close-port (car upipe-w))
	   (close-port (cdr upipe-r))
	   (cons pid (cons (car upipe-r) (cdr upipe-w)))))
	(else
	 (error "bad mode string: " mode))))
	
(defmacro tail-call-pipeline+ args
  (let* ((pipes (gensym))
	 (split-comps (pipe-split-components args))
	 (expressions (car split-comps))
	 (connections (cdr split-comps))
	 (pids (gensym)))
    `(let ((,pipes (cons (list) (list)))
	   (,pids (list)))
       ,@(let loop ((rem-exps expressions)
		    (rem-conns connections)
		    (insert (list)))
	   (cond ((null? rem-exps)
		  insert)
		 (else
		  (loop (cdr rem-exps)
			(cdr rem-conns)
			(append
			 insert
			 `(;; update the pipes used by this child.
			   (set! ,pipes (new-comm-pipes
					 ,pipes
					 ',(cadr rem-conns)))
			   ;; start one child process.
			   (set! ,pids (cons
					(pipe-fork-child ,(car rem-exps)
							 ,(car rem-conns)
							 ,(cadr rem-conns)
							 ,pipes)
					,pids))
			   ;; close used pipes in the parent.
			   (map (lambda (pipe-list)
				  (map close-port pipe-list))
				(car ,pipes))))))))
       ;; wait for all the processes to terminate and quit with the
       ;; exit status from the one at the tail of the pipe.
       ;; could save memory by exec'ing a tiny program to do the waiting.
       (set-batch-mode?! #t)
       (let next-pid ((waiting-for (length ,pids))
		      (result 0))
	 (cond ((> waiting-for 0)
		(let* ((report (waitpid WAIT_ANY))
		       (pid (car report))
		       ;; if normal termination return the exit status,
		       ;; otherwise 128 + the signal number.
		       (status (let ((exit-val (status:exit-val (cdr report)))
				     (term-sig (status:term-sig (cdr report))))
				 (or exit-val (+ term-sig 128)))))
		  (cond ((member pid ,pids)
			 ;; the pid list is reversed.
			 (if (= pid (car ,pids))
			     (next-pid (- waiting-for 1) status)
			     (next-pid (- waiting-for 1) result)))
			(else
			 (next-pid waiting-for result)))))
	       (else
		(primitive-exit result)))))))

;;; create pipes for communication: RHS connection list for a process.
;;; the previous set of pipes gets recycled to the LHS.
(define (new-comm-pipes old-pipes out-conns)
  (cons (cdr old-pipes)
	(map (lambda (conn)
	       (let ((rw-pair (unbuffered-pipe)))
		 (let next-dup ((new-pipes (list (cdr rw-pair) (car rw-pair)))
				(count (- (length conn) 2)))
		   (if (= count 0)
		       (reverse new-pipes)
		       (next-dup (cons (duplicate-port (car new-pipes) "w0")
				       new-pipes)
				 (- count 1))))))
	     out-conns)))

;;; fork a single child process, given redirections and pipes.
(defmacro pipe-fork-child (expr in-conns out-conns pipes)
  `(run-concurrently+
    ,expr #:no-auto-close
    ,@(append (let iloop ((count (- (length in-conns) 1))
			  (redirs (list)))
		(if (< count 0)
		    redirs
		    (iloop (- count 1)
			   (append
			    (let ((this-conn (list-ref in-conns count)))
			      ;; may be several ports to close (dups).
			      (let next-line ((dcount
					       (- (length this-conn) 2))
					      (lines (list)))
				(if (< dcount 0)
				    (append
				     lines
				     ;; redirect (port fdes).
				     `(((car (list-ref (car ,pipes) ,count))
					,(car (reverse this-conn))))
				     redirs)
				    (next-line
				     (- dcount 1)
				     (cons
				      ;; close the other pipe ends.
				      `((list-ref (list-ref (car ,pipes)
							    ,count)
						  ,(+ dcount 1)))
				      lines)))))))))
	      (let oloop ((count (- (length out-conns) 1))
			  (redirs (list)))
		(if (< count 0)
		    redirs
		    (oloop (- count 1)
			   ;; may need several redirections (dups).
			   (let ((this-conn (list-ref out-conns count)))
			     (let next-line ((dcount
					      (- (length this-conn) 2))
					     (lines (list)))
			       (if (< dcount 0)
				   (append lines
					   ;; close the other pipe ends.
					   `(((car
					       (list-ref (cdr ,pipes)
							 ,count))))
					   redirs)
				   (next-line
				    (- dcount 1)
				    (cons
				     ;; redirect (port fdes).
				     `((list-ref
					(list-ref (cdr ,pipes) ,count)
					,(+ dcount 1))
				       ,(list-ref this-conn dcount))
				     lines)))))))))))

;;; split a pipe into a process list and a connection list.
(define (pipe-split-components ppe)
  (let loop ((remaining ppe)
	     (do-expr? #t)  ; track alternating process / connection.
	     (exprs (list))
	     (connections (list)))
    (cond ((null? remaining)
	   (cons (reverse exprs)
		 ;; the null lists represent input and output from the pipe
		 ;; ends.
		 (cons (list) (reverse (cons (list) connections)))))
	  (do-expr? (loop (cdr remaining)
			  #f
			  (cons (car remaining) exprs)
			  connections))
	  (else (loop (cdr remaining)
		      #t
		      exprs
		      (cons (remove-dup-connections! (car remaining))
			    connections))))))

;;; convert connection spec like ((1 0)(2 0)) into ((1 2 0)).
;;; returns the mutated connection spec.
(define (remove-dup-connections! connections)
  (let ((r-connections (map reverse connections)))
    (let next-left ((left r-connections))
      (if (or (null? left) (null? (cdr left)))
	  (map reverse r-connections)
	  (let next-right ((right-1 left))
	    (let ((right (cdr right-1)))
	      (if (null? right)
		  (next-left (cdr left))
		  (cond ((= (caar left) (caar right))
			 (set-car! left (append (car left) (cdar right)))
			 (set-cdr! right-1 (cdr right))
			 (next-right right-1))
			(else
			 (next-right (cdr right-1)))))))))))

(defmacro tail-call-pipeline args
  `(tail-call-pipeline+
    ,@(let next-arg ((rem args)
		     (result (list)))
	(cond ((null? rem)
	       (reverse result))
	      (else
	       (next-arg (cdr rem)
			 (let ((temp (cons `(tail-call-program
					     ,@(car rem))
					   result)))
			   (if (null? (cdr rem))
			       temp
			       (cons '((1 0)) temp)))))))))

; try debugging a macro through a fork some day...
;(false-if-exception (delete-file "/tmp/goosh-debug"))
;(define-public (debug arg)
;  (let ((p (open-file "/tmp/goosh-debug" "a")))
;    (write arg p)
;    (newline p)
;    (close-port p)))
