/*
*  qm_trayicon.cpp
*  QUIMUP system-tray-icon class
*  © 2008-2024 Johan Spee
*  SPDX-License-Identifier: GPL-3.0-or-later
*/

#include "qm_trayicon.h"
#include "qm_widget_ids.h"


qm_trayIcon::qm_trayIcon(QMainWindow *parent, qm_Config *cfg) : QSystemTrayIcon(parent)
{
    if (objectName().isEmpty())
        setObjectName("quimup trayicon");

    set_themed_icons(cfg->set_dark_theme);

    mpd_cmd = nullptr;
    setIcon(pxb_noconn);
    b_mpd_connected = false;
    setToolTip("Idle");

    connect( this, SIGNAL(activated(QSystemTrayIcon::ActivationReason)), this, SLOT(on_activated(QSystemTrayIcon::ActivationReason)) );

    context_menu = new QMenu(parent);
    setup_widgets(); // context menu
    this->setContextMenu(context_menu);
}


void qm_trayIcon::setup_widgets()
{
    a_previous = new QAction("", this);
    a_previous->setIcon(pxb_prev);
    a_previous->setText(tr("Previous"));
    connect( a_previous, SIGNAL(triggered()), this, SLOT(on_prev_action()) );
    context_menu->addAction(a_previous);

    a_stop = new QAction("", this);
    a_stop->setIcon(pxb_stop);
    a_stop->setText(tr("Stop"));
    connect( a_stop, SIGNAL(triggered()), this, SLOT(on_stop_action()) );
    context_menu->addAction(a_stop);

    a_playpause = new QAction("", this);
    a_playpause->setIcon(pxb_play); // set in set_mode()
    a_playpause->setText(tr("Play"));
    connect( a_playpause, SIGNAL(triggered()), this, SLOT(on_play_action()) );
    context_menu->addAction(a_playpause);

    a_next = new QAction("", this);
    a_next->setIcon(pxb_next);
    a_next->setText(tr("Next"));
    connect( a_next, SIGNAL(triggered()), this, SLOT(on_next_action()) );
    context_menu->addAction(a_next);

    context_menu->addSeparator();

    a_quit = new QAction("", this);
    a_quit->setIcon(pxb_quit);
    a_quit->setText(tr("Quit"));
    connect( a_quit, SIGNAL(triggered()), this, SLOT(on_quit_action()) );
    context_menu->addAction(a_quit);
}


void qm_trayIcon::on_prev_action()
{
    if (b_mpd_connected)
        emit signal_tray_menu(ID_prev);
}


void qm_trayIcon::on_stop_action()
{
    if (b_mpd_connected)
        emit signal_tray_menu(ID_stop);
}


void qm_trayIcon::on_play_action()
{
    if (b_mpd_connected)
        emit signal_tray_menu(ID_play);
}


void qm_trayIcon::on_next_action()
{
    if (b_mpd_connected)
        emit signal_tray_menu(ID_next);
}


void qm_trayIcon::on_quit_action()
{
    emit signal_tray_menu(ID_exit);
}


void qm_trayIcon::on_activated(QSystemTrayIcon::ActivationReason reason)
{
    switch (reason)
    {
        case QSystemTrayIcon::Trigger:
        {
            emit clicked();
                    break;
        }
        case QSystemTrayIcon::MiddleClick:
        {
            on_play_action();
            break;
        }

        // Never happens. QSystemTrayIcon intercepts and shows context_menu.
        case QSystemTrayIcon::Context:
        {
            break;
        }
        default:
            break;
    }
}


void qm_trayIcon::set_tip(QString artist, QString title, bool b_message)
{
    if (b_message) // like 'Idle'
    {
        if (!artist.isNull() && !artist.isEmpty())
        {
           if (!title.isNull() && !title.isEmpty())
            setToolTip(artist + " - " + title);
           else
            setToolTip(artist);
        }
        else // no 'artist'
           if (!title.isNull() && !title.isEmpty())
                setToolTip(title);
           else // no artist, no title
                setToolTip("no info");
    }
    else
    {
        if (artist.isNull() || artist.isEmpty())
        {
           artist = tr("(no artist)");
        }

        if (title.isNull() || title.isEmpty())
        {
           title = tr("(no title)");

        }
        setToolTip(artist + " : " + title);
    }
}

// set a pointer to the parent's mpd_cmd
void qm_trayIcon::set_connected(qm_mpdCommand *control, bool isconnected)
{
    mpd_cmd = control;
    b_mpd_connected = isconnected;
    if (!b_mpd_connected)
        set_mode(ID_nocn);
    else
        set_mode(ID_idle);
}


void qm_trayIcon::set_mode(int mode)
{
    switch (mode)
    {
        case ID_play: // playing
        {
            a_playpause->setIcon(pxb_pause);
            a_playpause->setText(tr("Pause"));
            setIcon(pxb_playing);
            break;
        }

        case ID_paus: // paused
        {
            a_playpause->setIcon(pxb_play);
            a_playpause->setText(tr("Play"));
            setIcon(pxb_paused);
            break;
        }

        case ID_stop: // stopped
        {
            a_playpause->setIcon(pxb_play);
            a_playpause->setText(tr("Play"));
            setIcon(pxb_stopped);
            break;
        }

        case ID_nocn: // noconnection
        {
            a_playpause->setIcon(pxb_play);
            a_playpause->setText(tr("Play"));
            setIcon(pxb_noconn);
            set_tip(tr("Not Connected"),"", true);
            break;
        }

        default:
        {
            a_playpause->setIcon(pxb_play);
            a_playpause->setText(tr("Play"));
            setIcon(pxb_idle);
            set_tip(tr("Idle"), "", true);
            break;
        }
    }
}


void qm_trayIcon::set_themed_icons(bool dark_theme)
{
    if (!dark_theme)
    {
        pxb_prev = QPixmap(":/pl_sy_prev");
        pxb_stop = QPixmap(":/pl_sy_stop");
        pxb_play = QPixmap(":/pl_sy_play");
        pxb_pause = QPixmap(":/pl_sy_pause");
        pxb_next = QPixmap(":/pl_sy_next");
    }
    else
    {
        pxb_prev = QPixmap(":/pl_sy_prev.alt");
        pxb_stop = QPixmap(":/pl_sy_stop.alt");
        pxb_play = QPixmap(":/pl_sy_play.alt");
        pxb_pause = QPixmap(":/pl_sy_pause.alt");
        pxb_next = QPixmap(":/pl_sy_next.alt");
    }

    // not themed:
    pxb_quit =  QPixmap(":/sy_quit");
    pxb_playing = QPixmap(":/sy_playing");
    pxb_noconn = QPixmap(":/sy_noconn");
    pxb_paused = QPixmap(":/sy_paused");
    pxb_stopped = QPixmap(":/sy_stopped");
    pxb_idle = QPixmap(":/sy_idle");
}


qm_trayIcon::~qm_trayIcon()
{}
