/*
*  qm_core.cpp
*  QUIMUP core class
*  © 2008-2024 Johan Spee
*  SPDX-License-Identifier: GPL-3.0-or-later
*/


/*
    *************************************************************************
    The main purpose of the 'core' is to provide KDE with a program window
    (albeit invisible) that can be closed on shutdown (or logout).

    The player window ignores the close-event to allow close-to-tray behavior.

    But it will also ignore the close-event evoked by KDE on shutdown, which
    consequently aborts the logout procedure.

    This 'core' however accepts the close-event, and it's child, the player
    window, is closed with it.

    The player can close itself by calling qApp->quit().
    *************************************************************************
*/

#include "qm_core.h"


qm_Core::qm_Core (qm_Config *cnf)
{
    config = cnf;
    if (objectName().isEmpty())
        setObjectName("qm_Core");

    QGuiApplication::setDesktopFileName("Quimup");

    player = new qm_Player(cnf);
}


// called from main.cpp: 1st instance directly, 2nd via singleapplication
void qm_Core::on_message_from_main(QString msg_from_main)
{
    if (config->cout_extensive)
    {
        if (msg_from_main == "wakeupcall")
            printf ("Received wake up call from other instance\n");
        else
            printf ("Received arguments\n");
    }

    // will Wayland ever allow this?
    if (config->this_is_wayland)
    {
        // a hack
        player->hide();
        player->showNormal();
    }
    else
    {
        // Wayland does not restore from minimized
        player->showNormal();
        // Wayland does not put it in front
        player->activateWindow();
    }

    if (msg_from_main == "wakeupcall")
        return;

    int filecount = 0;

    // nb. splitter "*!#!*" is added in main and removed here
    QStringList split_args = QString(msg_from_main).split( "*!#!*", Qt::SkipEmptyParts);
    QStringList args_to_playlistview;

    QStringList::iterator iter;
    for (iter = split_args.begin(); iter != split_args.end(); ++iter)
    {
        // remove tailing spaces !
        QString arg = (*iter).trimmed(); // remove tailing spaces

        if( arg == "-play" )
        {
            args_to_playlistview.push_back(arg);
        }
        else
            if ( arg.startsWith("/")  && !arg.contains("quimup", Qt::CaseInsensitive))
            {

                arg.prepend("file://");
                filecount++;
                args_to_playlistview.push_back(arg);
            }
            else
                if ( arg.startsWith("file:///") )
                {
                    filecount++;
                    args_to_playlistview.push_back(arg);
                }
    }

    if (filecount > 0)
    {
        // send args to playlistview
        player->browser_window->playlist_view->on_open_with_request(args_to_playlistview);
    }
}

// called from main.cpp
void qm_Core::on_system_quit()
{
    player->on_shutdown();
    delete (player);
}


qm_Core::~qm_Core()
{
    player->on_shutdown();
    delete (player);
}
