/*
 * Routines for daemon, killproc, killall5, pidof, and runlevel.
 *
 * Version:	2.0 10-Nov-2000 Fink
 *
 * Copyright 1994-2000 Werner Fink, 1996-2000 SuSE GmbH Nuernberg, Germany.
 * Copyright 2005 Werner Fink, 2005 SUSE LINUX Products GmbH, Germany.
 *
 * Some parts of this software are copied out of killall5.c of the
 * sysvinit suite 2.57b, Copyright 1991-1995 Miquel van Smoorenburg.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Author:	Werner Fink <werner@suse.de>, 1994-2000
 *
 * 1998/09/29 Werner Fink: Add kernel thread handling.
 * 1999/02/24 Werner Fink: Advance script search
 * 1999/08/05 Werner Fink: Handle ignoring zombies
 * 2000/11/10 Werner Fink: LSB specs, logging
 */

#include "libinit.h"  /* Now get the inlined functions */
#ifndef  INITDIR
# define INITDIR	"/etc/init.d"
#endif

#ifndef __USE_BSD
extern void vsyslog (int, const char *, va_list);
#endif

char     * newenvp[MAXENV];
unsigned   newenvc = 0;
PROC     * remember = (PROC*)0;
PROC     * doignore = (PROC*)0;

char * we_are;
unsigned short stopped = 0;
pid_t p_pid, p_sid, p_ppid, p_pppid;

char *sys_signame [NSIG+1];
char *sys_sigalias[NSIG+1];

/* Buffers used herin */
static char tmp[CMDLLEN];
static char buf[CMDLLEN];
static char entry[MAXNAMLEN+1];
static char desti[MAXNAMLEN+1];
#define clean_buf(ent) memset(ent, (int)'\0', (size_t)(sizeof(char)*(MAXNAMLEN+1)))
static inline char * _handl_buf(char* buf)
{
    char * ptr = strchr(buf, (int)' ');
    if (ptr)
	*ptr = '\0';

    if ((ptr = strstr(buf, "-RPMDELETE")))
	*ptr = '\0';

    return buf;
}
#define handl_buf(ent) (_handl_buf(ent))

void init_signames(void)
{
    int n;

    for (n = 0; n < NSIG+1; n++) {
	sys_signame [n] = (char *)0;
	sys_sigalias[n] = (char *)0;
    }

    sys_signame [0] = "EXIT";
    sys_sigalias[0] = "EXIT";

/*     Signal        Value  Action     Comment */
#ifdef SIGHUP    /*    1       A       Hangup detected on controlling terminal or */
				    /* death of controlling process */
    if (!sys_signame[SIGHUP])
	sys_signame [SIGHUP]    = "HUP";
    else
	sys_sigalias[SIGHUP]    = "HUP";
#endif
#ifdef SIGINT    /*    2       A       Interrupt from keyboard */

    if (!sys_signame[SIGINT])
	sys_signame [SIGINT]    = "INT";
    else
	sys_sigalias[SIGINT]    = "INT";
#endif
#ifdef SIGQUIT   /*    3       A       Quit from keyboard */

    if (!sys_signame[SIGQUIT])
	sys_signame [SIGQUIT]   = "QUIT";
    else
	sys_sigalias[SIGQUIT]   = "QUIT";
#endif
#ifdef SIGILL    /*    4       A       Illegal Instruction */

    if (!sys_signame[SIGILL])
	sys_signame [SIGILL]    = "ILL";
    else
	sys_sigalias[SIGILL]    = "ILL";
#endif
#ifdef SIGABRT   /*    6       C       Abort signal from abort(3) */

    if (!sys_signame[SIGABRT])
	sys_signame [SIGABRT]   = "ABRT";
    else
	sys_sigalias[SIGABRT]   = "ABRT";
#endif
#ifdef SIGFPE    /*    8       C       Floating point exception */
    if (!sys_signame[SIGFPE])
	sys_signame [SIGFPE]    = "FPE";
    else
	sys_sigalias[SIGFPE]    = "FPE";
#endif
#ifdef SIGKILL   /*    9      AEF      Kill signal */

    if (!sys_signame[SIGKILL])
	sys_signame [SIGKILL]   = "KILL";
    else
	sys_sigalias[SIGKILL]   = "KILL";
#endif
#ifdef SIGSEGV   /*    11      C       Invalid memory reference */

    if (!sys_signame[SIGSEGV])
	sys_signame [SIGSEGV]   = "SEGV";
    else
	sys_sigalias[SIGSEGV]   = "SEGV";
#endif
#ifdef SIGPIPE   /*    13      A       Broken pipe: */

    if (!sys_signame[SIGPIPE])
	sys_signame [SIGPIPE]   = "PIPE";/* write to pipe with no readers */
    else
	sys_sigalias[SIGPIPE]   = "PIPE";/* write to pipe with no readers */
#endif
#ifdef SIGALRM   /*    14      A       Timer signal from alarm(1) */

    if (!sys_signame[SIGALRM])
	sys_signame [SIGALRM]   = "ALRM";
    else
	sys_sigalias[SIGALRM]   = "ALRM";
#endif
#ifdef SIGTERM   /*    15      A       Termination signal */

    if (!sys_signame[SIGTERM])
	sys_signame [SIGTERM]   = "TERM";
    else
	sys_sigalias[SIGTERM]   = "TERM";
#endif
#ifdef SIGUSR1   /* 30,10,16   A       User-defined signal 1 */

    if (!sys_signame[SIGUSR1])
	sys_signame [SIGUSR1]   = "USR1";
    else
	sys_sigalias[SIGUSR1]   = "USR1";
#endif
#ifdef SIGUSR2   /* 31,12,17   A       User-defined signal 2 */

    if (!sys_signame[SIGUSR2])
	sys_signame [SIGUSR2]   = "USR2";
    else
	sys_sigalias[SIGUSR2]   = "USR2";
#endif
#ifdef SIGCHLD   /* 20,17,18   B       Child stopped or terminated */

    if (!sys_signame[SIGCHLD])
	sys_signame [SIGCHLD]   = "CHLD";
    else
	sys_sigalias[SIGCHLD]   = "CHLD";
#endif
#ifdef SIGCONT   /* 19,18,25           Continue if stopped */

    if (!sys_signame[SIGCONT])
	sys_signame [SIGCONT]   = "CONT";
    else
	sys_sigalias[SIGCONT]   = "CONT";
#endif
#ifdef SIGSTOP   /* 17,19,23  DEF      Stop process */

    if (!sys_signame[SIGSTOP])
	sys_signame [SIGSTOP]   = "STOP";
    else
	sys_sigalias[SIGSTOP]   = "STOP";
#endif
#ifdef SIGTSTP   /* 18,20,24   D       Stop typed at tty */

    if (!sys_signame[SIGTSTP])
	sys_signame [SIGTSTP]   = "TSTP";
    else
	sys_sigalias[SIGTSTP]   = "TSTP";
#endif
#ifdef SIGTTIN   /* 21,21,26   D       tty input for background process */

    if (!sys_signame[SIGTTIN])
	sys_signame [SIGTTIN]   = "TTIN";
    else
	sys_sigalias[SIGTTIN]   = "TTIN";
#endif
#ifdef SIGTTOU   /* 22,22,27   D       tty output for background process */

    if (!sys_signame[SIGTTOU])
	sys_signame [SIGTTOU]   = "TTOU";
    else
	sys_sigalias[SIGTTOU]   = "TTOU";
#endif
#ifdef SIGTRAP   /*    5       CG      Trace/breakpoint trap */

    if (!sys_signame[SIGTRAP])
	sys_signame [SIGTRAP]   = "TRAP";
    else
	sys_sigalias[SIGTRAP]   = "TRAP";
#endif
#ifdef SIGIOT    /*    6       CG      IOT trap. A synonym for SIGABRT */

    if (!sys_signame[SIGIOT])
	sys_signame [SIGIOT]    = "IOT";
    else
	sys_sigalias[SIGIOT]    = "IOT";
#endif
#ifdef SIGEMT    /*  7,-,7     G */

    if (!sys_signame[SIGEMT])
	sys_signame [SIGEMT]    = "EMT";
    else
	sys_sigalias[SIGEMT]    = "EMT";
#endif
#ifdef SIGBUS    /* 10,7,10    AG      Bus error */

    if (!sys_signame[SIGBUS])
	sys_signame [SIGBUS]    = "BUS";
    else
	sys_sigalias[SIGBUS]    = "BUS";
#endif
#ifdef SIGSYS    /* 12,-,12    G       Bad argument to routine (SVID) */

    if (!sys_signame[SIGSYS])
	sys_signame [SIGSYS]    = "SYS";
    else
	sys_sigalias[SIGSYS]    = "SYS";
#endif
#ifdef SIGSTKFLT /*  -,16,-    AG      Stack fault on coprocessor */

    if (!sys_signame[SIGSTKFLT])
	sys_signame [SIGSTKFLT] = "STKFLT";
    else
	sys_sigalias[SIGSTKFLT] = "STKFLT";
#endif
#ifdef SIGURG    /* 16,23,21   BG      Urgent condition on socket (4.2 BSD) */

    if (!sys_signame[SIGURG])
	sys_signame [SIGURG]    = "URG";
    else
	sys_sigalias[SIGURG]    = "URG";
#endif
#ifdef SIGIO     /* 23,29,22   AG      I/O now possible (4.2 BSD) */

    if (!sys_signame[SIGIO])
	sys_signame [SIGIO]     = "IO";
    else
	sys_sigalias[SIGIO]     = "IO";
#endif
#ifdef SIGPOLL   /*            AG      A synonym for SIGIO (System V) */

    if (!sys_signame[SIGPOLL])
	sys_signame [SIGPOLL]   = "POLL";
    else
	sys_sigalias[SIGPOLL]   = "POLL";
#endif
#ifdef SIGCLD    /*  -,-,18    G       A synonym for SIGCHLD */

    if (!sys_signame[SIGCLD])
	sys_signame [SIGCLD]    = "CLD";
    else
	sys_sigalias[SIGCLD]    = "CLD";
#endif
#ifdef SIGXCPU   /* 24,24,30   AG      CPU time limit exceeded (4.2 BSD) */

    if (!sys_signame[SIGXCPU])
	sys_signame [SIGXCPU]   = "XCPU";
    else
	sys_sigalias[SIGXCPU]   = "XCPU";
#endif
#ifdef SIGXFSZ   /* 25,25,31   AG      File size limit exceeded (4.2 BSD) */

    if (!sys_signame[SIGXFSZ])
	sys_signame [SIGXFSZ]   = "XFSZ";
    else
	sys_sigalias[SIGXFSZ]   = "XFSZ";
#endif
#ifdef SIGVTALRM /* 26,26,28   AG      Virtual alarm clock (4.2 BSD) */

    if (!sys_signame[SIGVTALRM])
	sys_signame [SIGVTALRM] = "VTALRM";
    else
	sys_sigalias[SIGVTALRM] = "VTALRM";
#endif
#ifdef SIGPROF   /* 27,27,29   AG      Profile alarm clock */

    if (!sys_signame[SIGPROF])
	sys_signame [SIGPROF]   = "PROF";
    else
	sys_sigalias[SIGPROF]   = "PROF";
#endif
#ifdef SIGPWR    /* 29,30,19   AG      Power failure (System V) */

    if (!sys_signame[SIGPWR])
	sys_signame [SIGPWR]    = "PWR";
    else
	sys_sigalias[SIGPWR]    = "PWR";
#endif
#ifdef SIGINFO   /* 29,-,-     G       A synonym for SIGPWR */

    if (!sys_signame[SIGINFO])
	sys_signame [SIGINFO]   = "INFO";
    else
	sys_sigalias[SIGINFO]   = "INFO";
#endif
#ifdef SIGLOST   /*  -,-,-     AG      File lock lost */

    if (!sys_signame[SIGLOST])
	sys_signame [SIGLOST]   = "LOST";
    else
	sys_sigalias[SIGLOST]   = "LOST";
#endif
#ifdef SIGWINCH  /* 28,28,20   BG      Window resize signal (4.3 BSD, Sun) */

    if (!sys_signame[SIGWINCH])
	sys_signame [SIGWINCH]  = "WINCH";
    else
	sys_sigalias[SIGWINCH]  = "WINCH";
#endif
#ifdef SIGUNUSED /*  -,31,-    AG      Unused signal */

    if (!sys_signame[SIGUNUSED])
	sys_signame [SIGUNUSED] = "UNUSED";
    else
	sys_sigalias[SIGUNUSED] = "UNUSED";
#endif
}

/* write to syslog file if not open terminal */
static void nsyslog(int pri, const char *fmt, va_list args)
{
    char newfmt[strlen(we_are)+2+strlen(fmt)+1];

    strcat(strcat(strcpy(newfmt, we_are), ": "), fmt);

    /* file descriptor of stderr is 2 in most cases */
    if (ttyname(fileno(stderr)) == NULL)
	vsyslog(pri, newfmt, args);
    else
	vfprintf(stderr, newfmt, args);
}

void error(int stat, const char *fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    nsyslog(LOG_ERR, fmt, args);
    va_end(args);
    exit(stat);
}

void warn(const char *fmt, ...)
{
    int saveerr = errno;
    va_list args;
    va_start(args, fmt);
    nsyslog(LOG_WARNING, fmt, args);
    va_end(args);
    errno = saveerr;
}

static void dsyslog(int pri, const char *fmt, va_list args)
{
    char newfmt[strlen(we_are)+2+strlen(fmt)+1];

    strcat(strcat(strcpy(newfmt, we_are), ": "), fmt);
    vsyslog(pri, newfmt, args);
}

void logprogress(int prio, const char *fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    dsyslog(prio, fmt, args);
    va_end(args);
}

/* For mounting the /proc file system */
static void getproc()
{
    struct stat st;
    pid_t pid, wst;

    /* It's not there, so mount it. */
    errno = 0;
    if ((pid = fork()) < 0)
	error(100, "cannot fork: %s\n", strerror(errno));

    errno = 0;
    if (pid == 0) {
	/* Try a few mount binaries. */
	execl("/sbin/mount", "mount", "-n", "-t", "proc", "proc", "/proc", NULL);
	execl("/etc/mount",  "mount", "-n", "-t", "proc", "proc", "/proc", NULL);
	execl("/bin/mount",  "mount", "-n", "-t", "proc", "proc", "/proc", NULL);

	/* Okay, I give up. */
	error(100, "cannot execute mount: %s\n", strerror(errno));
    }
    /* Wait for child. */
    while(wait(&wst) != pid)
	__asm__ __volatile__("": : :"memory");
    if (WEXITSTATUS(wst) != 0)
	warn("mount returned not-zero exit status\n");

    /* See if mount succeeded. */
    errno = 0;
    if (stat("/proc/version", &st) < 0)
	error(100, "/proc not mounted, failed to mount: %s\n", strerror(errno));
}

/* Open the /proc directory, if necessary mounts it */
static DIR * openproc()
{
    struct stat st;
    DIR * dir = NULL;

    /* Stat /proc/version to see if /proc is mounted. */
    if (stat("/proc/version", &st) < 0)
	getproc();

    errno = 0;
    if ((dir = opendir("/proc")) == (DIR *)0)
	error(100, "cannot opendir(/proc): %s\n", strerror(errno));

    return dir;
}

/* secure read on EINTR */
static ssize_t xread(int fd, void *inbuf, size_t count)
{
    register ssize_t bytes;
    register int olderr = errno;

    bzero(inbuf, count);

    while (1) {
	errno = 0;
	bytes = read(fd, inbuf, count);
	if (bytes < 0 && (errno == EINTR || errno == EAGAIN))
	    continue;
	if (bytes < 0)
	    break;
	goto out;
    }

    warn("xread error: %s\n", strerror(errno));
out:
    errno = olderr;
    return bytes;
}

/* proof a given full path name is a script name */
static char * script_exe = NULL;
static boolean isscript(const char* fullname, const char *root)
{
    int fp = 0;
    boolean ret = true;
    char head[MAXNAMLEN];

    if (script_exe)	/* already done */
	goto out;

    ret = false;
    if ((fp = open(fullname, O_RDONLY, 0)) != -1 ) {
	if (xread(fp, head, sizeof(head)) > 0 && head[0] == '#' && head[1] == '!') {
	    if ((script_exe = strchr(head, '/'))) {
		char * ptr = strpbrk(script_exe, " \t\n");
		if (ptr && *ptr)
		    *ptr = '\0';
		if (root) {
		    char *ptr = (char *)xmalloc(strlen(root)+strlen(script_exe)+1);
		    ptr = strcat(strcpy(ptr,root),script_exe);
		    script_exe = ptr;
		} else {
		    script_exe = xstrdup(script_exe);
		}
	    }
	    ret = true;
	}
	close(fp);
    }
out:
    return ret;
}

/* check a given command line for a full path script name */
static const char * checkscripts(char* ent, const char* root, const size_t len, const char* pid)
{
    const char *scrpt = ent;
    size_t cnt = len;
    const char * ret = NULL;
    struct stat exe_st;
    struct stat scr_st;
    int search = 0;

    if (!len || !script_exe)
	/* do not check empty entries */
	goto out;

    if (readlink(exe_entry(tmp, pid),clean_buf(desti),MAXNAMLEN) < 0 ||
	stat(handl_buf(desti), &exe_st) < 0 ||
	stat(script_exe, &scr_st) < 0)
	goto out;

    if (exe_st.st_dev != scr_st.st_dev || exe_st.st_ino != scr_st.st_ino)
	goto out;

    /* The exe link is our interpreter */
    ret = scrpt;

    /* Some kernels skip the path of the interpreter */
    if (*scrpt != '/')
	search++;

    /* Some kernels use the interpreter as first argument */
    if (*scrpt  == '/' &&
	(strncmp(scrpt,handl_buf(desti),MAXNAMLEN) == 0 ||
	 strncmp(scrpt,script_exe,MAXNAMLEN) == 0))
	search++;

    if (!search)
	goto out;

    ret = NULL;
    do {
	/* After the first zero we have the first argument
	 * which may be the name of a so what ever script.
	 */
	scrpt = (char *)memchr(scrpt, 0, cnt);
	if (!scrpt || (cnt = len - (++scrpt - ent)) <= 0)
	    goto out;
	if (*scrpt == '/') {
	    ret = scrpt;
	    goto out;
	}
    } while (scrpt && cnt > 0);

out:
    if (scrpt && root) {
	char *ptr = strdupa(scrpt);
	if (!ptr)
	    error(100, "strdupa(): %s\n", strerror(errno));
	scrpt = strcat(strcpy(ent,root),ptr);
    }
    return ret;
}

/* Gets the parent's pid of the parent. */
static pid_t getpppid(const pid_t ppid)
{
    pid_t pppid = 1;
    int fp, c;
    char strpid[10];

    if ((c = snprintf(strpid, 9, "%d", ppid)) < 1 || c > 9)
	goto out;
    strpid[c] = '\0';

    if ((fp = open(stat_entry(tmp, strpid),O_RDONLY,0)) != -1) {
	xread(fp,entry,MAXNAMLEN);
	close(fp);
	if (sscanf(entry,"%*d %*s %*c %d %*d %*d", &pppid) != 1)
	    warn("can not read ppid for process %d!\n", ppid);
    }
out:
    return pppid;
}

/* Gets the parent's pid of the parent. */
static pid_t getsession(const pid_t pid)
{
    pid_t session = 1;
    int fp, c;
    char strpid[10];

    if ((c = snprintf(strpid, 9, "%d", pid)) < 1 || c > 9)
	goto out;
    strpid[c] = '\0';

    if ((fp = open(stat_entry(tmp, strpid),O_RDONLY,0)) != -1) {
	xread(fp,entry,MAXNAMLEN);
	close(fp);
	if (sscanf(entry,"%*d %*s %*c %*d %*d %d", &session) != 1)
	    warn("can not read session for process %d!\n", pid);
    }
out:
    return session;
}

#if 0
/* Remember all pids not being the caller and its parent */
int allpids (void)
{
    DIR *dir;
    struct dirent *d;
    int fp;
    unsigned num = 0;
    pid_t pid, sid;
    PROC *p, *n;

    dir = openproc();
    p_pppid = getpppid();

    n = remember;
    for (p = remember; n; p = n) {
	n = p->next;
	free(p);
    }
    remember = (PROC*)0;

    /* Real System5 killall (also known as killall5) need only this one,
     * this case is not used by killproc, daemon/startproc, pidof/pidofproc */
    while((d = readdir(dir)) != (struct dirent *)0) {
	if ((pid = (pid_t)atoi(d->d_name)) == 0)
	    continue;

	if ((fp = open(stat_entry(tmp, d->d_name),O_RDONLY,0)) != -1) {
	    xread(fp,entry,MAXNAMLEN);
	    close(fp);
	    /* see `man 5 proc' */
	    if (sscanf(entry,"%*d %*s %*c %*d %*d %d",&sid) != 1) {
		sid = 0;
		warn("can not read sid for process %d!\n", pid);
	    }
	    if (pid == p_pid) p_sid = sid;
	    do_list(pid,sid,false);
	    num++;
	}

    }
    closedir(dir);
    return num;
}
#endif

/* Search proc table with a gotten full path of a running programm
   and remember it */
int pidof (const char * inname, const char * root, unsigned short flags)
{
    DIR *dir;
    struct dirent *d;
    struct stat full_st, pid_st;
    int fp;
    boolean isscrpt = false;
    unsigned num = 0;
    pid_t pid;
    char *swapname = NULL;
    char *fullname = (char *)inname;
    PROC *p, *n;

    p_pid  = getpid();
    p_ppid = getppid();

    dir = openproc();		/* Open /proc and maybe do mount before */
    p_pppid = getpppid(p_ppid); /* Requires existence of /proc */

    if (!fullname) {
	warn("program or process name required\n");
	return -1;
    }

    n = remember;
    for (p = remember; n; p = n) {
	n = p->next;
	free(p);
    }
    remember = (PROC*)0;

    /* killproc, daemon/startproc, pidof/pidofproc: stat fullname if a
     * real program is handled, skip this if we handle a kernel thread */

    if (!(flags & (KTHREAD|KSHORT))) {
	errno = 0;
	if (rlstat(&fullname, &full_st, flags) < 0) {
	    /* stat() follows soft links -> security */
	    warn("cannot stat %s: %s\n", fullname, strerror(errno));
	    return -1;
	}
    }

    if (flags & (KTHREAD|KSHORT)) {
	if (flags & KBASE)
	    swapname = swap_name(base_name(fullname));
	else
	    swapname = swap_name(fullname);
    } else {
	isscrpt = isscript(fullname, root);
	swapname = swap_name(base_name(fullname));
    }

    /* killproc, daemon/startproc, pidof/pidofproc */
    while((d = readdir(dir)) != (struct dirent *)0) {

	/* Only directories with pid as names */
	if ((pid = (pid_t)atoi(d->d_name)) == 0)
	    continue;

	/* killproc and startproc should not touch calling process */
	if ((pid == p_pid || pid == p_ppid || pid == p_pppid || pid == 1) &&
	    (flags & (KILL|DAEMON)))
	    continue;

	/* Check for kernel threads, zombies or programs */
	if ((fp = open(statm_entry(tmp, d->d_name), O_RDONLY,0)) != -1) {
	    char ent[3];
	    boolean thread;
	    ssize_t len;
	    len = xread(fp,ent,3);
	    close(fp);

	    if (!len)
		continue;
    
	    thread = (strncmp(ent, "0 ", 2) == 0);

	    if ((flags & KTHREAD)  && !thread)
		continue; /* Threads do not have any memory size in user space */
	    
	    if (!(flags & KTHREAD) &&  thread)
		continue; /* Programs always show  _memory_ size in user space */
	}

	/*
	 * Kernels 2.1 and above do not lost this link even if the
	 * program is swapped out. But this link is lost if the file
	 * of the program is overwriten during the process. Kernels
	 * 2.2 and above do not lost the link name even if the original
	 * file is deleted. The link is marked as deleted.
	 */
	if (!(flags & (KTHREAD|KSHORT)) && !isscrpt &&
	     (stat(exe_entry(tmp, d->d_name), &pid_st) == 0)) {

	    const char *name;
	    char *realname;
	    boolean found;

	    if (pid_st.st_dev != full_st.st_dev)
		continue;		/* No processes below (kernel 2.2 and up) */

	    found = false;
	    switch (pid_st.st_nlink) {
		case 1:			/* One file on disk */

		    if (pid_st.st_ino == full_st.st_ino)
			found = true;
		    break;

		case 0:			/* file was deleted or */
		default:		/* has several hard links */

		    if (strlen(fullname) > MAXNAMLEN)
			continue;

		    if (readlink(exe_entry(tmp, d->d_name),clean_buf(entry),MAXNAMLEN) < 0)
			continue;
		    name = handl_buf(entry);

		    if (strncmp(fullname, name, MAXNAMLEN) == 0) {
			found = true;
			break;
		    }

		    if ((realname = realpath(fullname, NULL)) == (char*)0)
			continue;

		    if (strncmp(realname, name, MAXNAMLEN) == 0)
			found = true;

		    free(realname);

		    break;
	    }

	    if (found) {
		do_list(pid,getsession(pid),false);
		num++;			/* Found */
	    }

	    continue;			/* No processes below (kernel 2.2 and up) */
	}

	/*
	 * Here we check for scripts. Note that the command line gets lost if the
	 * corresponding process is swapped out.  Many script interpreters even
	 * do not hold a file descriptor opened on the script file.
	 */
	if (!(flags & (KTHREAD|KSHORT)) &&  isscrpt &&
	    (fp = open(cmd_entry(tmp, d->d_name),O_RDONLY,0)) != -1) {
	    const char *scrpt = NULL;
	    ssize_t len;

	    len = xread(fp,entry,MAXNAMLEN);
	    close(fp);

	    /* Seek for a script not for a binary */
	    if (!(scrpt = checkscripts(entry, root, len, d->d_name)))
		continue;

	    /* Don't blame our boot scripts having the same name */
	    if (   (flags & (KILL|DAEMON))
		&& (strncmp(scrpt, INITDIR, (sizeof(INITDIR) - 1)) == 0))
		continue;

	    if (scrpt && strcmp(scrpt,fullname) == 0) {
		do_list(pid,getsession(pid),false);
		num++;
		continue;
	    }
	}

	/*
	 * High risk ... the name in stat isn't exact enough to identify
	 * a swapped out script process, because only the name without
	 * its path is stated which may not be identical with the
	 * executable script its self.
	 */
	if ((fp = open(stat_entry(tmp, d->d_name),O_RDONLY,0)) != -1) {
	    char *comm, *state;
	    ssize_t len;

	    len = xread(fp,entry,MAXNAMLEN);
	    close(fp);

	    if (!len)
		continue;

	    comm  = index(entry,  ' ');
	    state = index(++comm, ' ');
	    *state++ = '\0';

	    if ( (flags & NZOMBIE) && state[0] == 'Z' )
		/* This is a zombie, ignore it */
		continue;

	    if ( strcmp(comm, swapname) == 0 ) {
		do_list(pid,getsession(pid),false);
		num++;
		continue;
	    }
	}
    }

    closedir(dir);
    free(swapname);
    return num;
}

/* Verify a given string of pids, and if found remember them */
int remember_pids(const char * pids, const char * inname,
		  const char * root, unsigned short flags)
{
    char *buf = strdupa(pids);
    char *bufp;

    if (!buf)
	error(100, "strdupa(): %s\n", strerror(errno));

    for (bufp = strsep(&buf, " "); bufp && *bufp; bufp = strsep(&buf, " ")) {
	const pid_t pid = (pid_t)atoi(bufp);
	if (!pid)
	    continue;
	if (pid == getpid())
	    continue;		/* Don't kill myself */
	do_list(pid,getsession(pid),false);
    }

    return check_pids (inname, root, flags);
}

/* Open, read, and verify pid file, if pid found remember it */
int verify_pidfile (const char * pid_file, const char * inname,
		    const char * root, unsigned short flags,
		    const boolean ignore)
{
    int fp, cnt;
    boolean isscrpt = false;
    pid_t pid;
    char *swapname = NULL, *bufp;
    char *fullname = (char *)inname;
    struct stat pid_st, full_st;
    
    if (!ignore) {
    	PROC *p, *n;
	n = remember;
	for (p = remember; n; p = n) {
	    n = p->next;
	    free(p);
	}
	remember = (PROC*)0;
    }

    errno = 0;
    if ((fp = open (pid_file, O_RDONLY,0)) < 0 ) {
	warn("Can not open pid file %s: %s\n", pid_file, strerror(errno));
	return -1;
    }

    errno = 0;
    if ((cnt = xread (fp, buf, CMDLLEN)) < 0) {
	warn("Can not read pid file %s: %s\n", pid_file, strerror(errno));
	return -1;
    }
    close(fp);
    buf[cnt] = '\0';

    bufp = buf;
    while (--cnt && isspace(*bufp)) bufp++;
    memmove(buf, bufp, sizeof(char)*(cnt+1));

    if ((bufp = strpbrk(buf, "\r\n\f\t\v \0")))
	*bufp = '\0';

    errno = 0;
    if ((pid = (pid_t)atoi(buf)) <= 0) {
	if (errno)
	    warn("Can not handle pid file %s with pid %s: %s\n", pid_file, buf, strerror(errno));
	if (!pid)
	    warn("Can not handle pid file %s with pid `%s\'\n", pid_file, buf);
	return -1;
    }

    if (!ignore && pid == getpid())
	return 0;		/* Don't kill myself */

    if (!fullname) {
	warn("program or process name required\n");
	return -1;
    }

    if (!(flags & (KTHREAD|KSHORT))) {
	errno = 0;
	if (rlstat(&fullname, &full_st, flags) < 0) {
	    /* stat() follows soft links -> security */
	    warn("cannot stat %s: %s\n", fullname, strerror(errno));
	    return -1;
	}
    }

    if (flags & (KTHREAD|KSHORT)) {
	if (flags & KBASE)
	    swapname = swap_name(base_name(fullname));
	else
	    swapname = swap_name(fullname);
    } else {
	isscrpt = isscript(fullname, root);
	swapname = swap_name(base_name(fullname));
    }

    /* Check for kernel threads, zombies or programs */
    if ((fp = open(statm_entry(tmp, buf), O_RDONLY,0)) != -1) {
	char ent[3];
	boolean thread;
	ssize_t len;
	len = xread(fp,ent,3);
	close(fp);

	if (!len)
	    goto out;
    
	thread = (strncmp(ent, "0 ", 2) == 0);

	if ((flags & KTHREAD)  && !thread)
	    goto out; /* Threads do not have any memory size in user space */

	if (!(flags & KTHREAD) &&  thread)
	    goto out; /* Programs always show  _memory_ size in user space */
    }

    errno = 0;
    if (!(flags & (KTHREAD|KSHORT)) && !isscrpt &&
	 (stat(exe_entry(tmp, buf), &pid_st) == 0)) {

	const char *name;
	char *realname;
	boolean found;

	if (pid_st.st_dev != full_st.st_dev)
	    goto out;

	found = false;
	switch (pid_st.st_nlink) {
	    case 1:			/* One file on disk */

		if (pid_st.st_ino == full_st.st_ino)
		    found = true;
		break;

	    case 0:			/* file was deleted or */
	    default:			/* has several hard links */

		if (strlen(fullname) > MAXNAMLEN)
		    goto out;

		if (readlink(exe_entry(tmp, buf),clean_buf(entry),MAXNAMLEN) < 0)
		    goto out;
		name = handl_buf(entry);

		if (strncmp(fullname, name, MAXNAMLEN) == 0) {
		    found = true;
		    break;
		}

		if ((realname = realpath(fullname, NULL)) == (char*)0)
		    goto out;

		if (strncmp(realname, name, MAXNAMLEN) == 0)
		    found = true;

		free(realname);

		break;
	}

	if (found)
	    do_list(pid,getsession(pid),ignore);

	goto out;
    }

    if (errno && errno != ENOENT) {
//	int saveerr = errno;
	warn("Can not read %s: %s\n", tmp, strerror(errno));
	free(swapname);
//	errno = saveerr;
	return -1;
    }

    if (!(flags & (KTHREAD|KSHORT))&&  isscrpt &&
	(fp = open(cmd_entry(tmp, buf),O_RDONLY,0)) != -1) {
	const char *scrpt = NULL;
	ssize_t len;

	len = xread(fp,entry,MAXNAMLEN);
	close(fp);

	/* Seek for a script not for a binary */
	if (!(scrpt = checkscripts(entry, root, len, buf)))
	    goto out;		/* Nothing found */

	if (scrpt && strcmp(scrpt,fullname) == 0) {
	    do_list(pid,getsession(pid),ignore);
	    goto out;		/* Done */
	}
    }

    if ((fp = open(stat_entry(tmp, buf),O_RDONLY,0)) != -1) {
	char *comm, *state;
	ssize_t len;
	len = xread(fp,entry,MAXNAMLEN);
	close(fp);

	if (!len)
	    goto out;

	comm  = index(entry,  ' ');
	state = index(++comm, ' ');
	*state++ = '\0';

	if ( (flags & NZOMBIE) && state[0] == 'Z' )
	    /* This is a zombie, ignore it */
	    goto out;		/* This is a zombie, ignore it */

	if (strcmp(comm, swapname) == 0) {
	    do_list(pid,getsession(pid),ignore);
	    goto out;		/* Done */
	}
    }
out:
    if (swapname)
	free(swapname);
    return 0;			/* Nothing found */
}

/*
 * Check remembered pids, every pid will be verified
 * We have to do the full stuff to avoid conflicts with
 * newer processes having similar pids.
 */
int check_pids (const char * inname, const char * root, unsigned short flags)
{
    int fp, c;
    boolean isscrpt = false;
    char *swapname = NULL, strpid[10];
    char *fullname = (char *)inname;
    struct stat pid_st, full_st;
    PROC *p, *n, *l;

    if (!fullname) {
	warn("program or process name required\n");
	return -1;
    }

    if (!(flags & (KTHREAD|KSHORT))) {
	errno = 0;
	if (rlstat(&fullname, &full_st, flags) < 0) {
	    /* stat() follows soft links -> security */
	    warn("cannot stat %s: %s\n", fullname, strerror(errno));
	    return -1;
	}
    }

    if (flags & (KTHREAD|KSHORT)) {
	if (flags & KBASE)
	    swapname = swap_name(base_name(fullname));
	else
	    swapname = swap_name(fullname);
    } else {
	isscrpt = isscript(fullname, root);
	swapname = swap_name(base_name(fullname));
    }

    n = remember;
    l = (PROC*)0;
    for(p = remember; n; p = n) {
	boolean skip = false;
	l = p->prev;
	n = p->next;

	errno = 0;
	if ((c = snprintf(strpid, 9, "%d", p->pid)) > 0)
	    strpid[c] = '\0';
	else {
	    int saveerr = errno;
	    warn("error in snprintf: %s\n", strerror(errno));
	    free(swapname);
	    errno = saveerr;
	    return -1;
	}

	/* Check for kernel threads, zombies or programs */
	if ((fp = open(statm_entry(tmp, strpid), O_RDONLY,0)) != -1) {
	    char ent[3];
	    int thread;
	    ssize_t len;

	    len = xread(fp,ent,3);
	    close(fp);

	    if (!len)
		goto ignore;		/* Bogus */

	    thread = (strncmp(ent, "0 ", 2) == 0);

	    if ((flags & KTHREAD)  && !thread)
		continue; /* Threads do not have any memory size in user space */

	    if (!(flags & KTHREAD) &&  thread)
		continue; /* Programs always show  _memory_ size in user space */
	}

	/* killproc and daemon/startproc should use the full path */
	errno = 0;
	if (!(flags & (KTHREAD|KSHORT)) && !isscrpt &&
	     (stat(exe_entry(tmp, strpid), &pid_st) == 0)) {

	    const char *name;
	    char *realname;

	    if (pid_st.st_dev != full_st.st_dev)
		goto ignore;		/* Does not belong to rembered list */

	    switch (pid_st.st_nlink) {
		case 1:			/* One file on disk */

		    if (pid_st.st_ino == full_st.st_ino)
			continue;	/* Found */
		    break;

		case 0:			/* file was deleted or */
		default:		/* has several hard links */

		    if (strlen(fullname) > MAXNAMLEN)
			goto ignore;	/* Bogus */

		    if (readlink(exe_entry(tmp, strpid),clean_buf(entry),MAXNAMLEN) < 0)
			goto ignore;	/* Bogus */
		    name = handl_buf(entry);

		    if (strncmp(fullname, name, MAXNAMLEN) == 0)
			continue;	/* Found */

		    if ((realname = realpath(fullname, NULL)) == (char*)0)
			goto ignore;	/* Bogus */

		    if (strncmp(realname, name, MAXNAMLEN) == 0) {
			free(realname);
			continue;	/* Found */
		    }
		    free(realname);

		    break;
	    }

	    skip = true;		/* No stat entry check needed */
	}

	if (!(flags & (KTHREAD|KSHORT)) &&  isscrpt &&
	    (fp = open(cmd_entry(tmp, strpid),O_RDONLY,0)) != -1) {
	    const char *scrpt;
	    ssize_t len;

	    len = xread(fp,entry,MAXNAMLEN);
	    close(fp);

	    if (!len)
		goto ignore;		/* Bogus */

	    /* Seek for a script not for a binary */
	    if ((scrpt = checkscripts(entry, root, len, strpid))) {
	        if (strcmp(scrpt,fullname) == 0)
		    continue;		/* Found */
		skip = true;		/* No stat entry check needed */
	    }
	}

	if (!skip && (fp = open(stat_entry(tmp, strpid),O_RDONLY,0)) != -1) {
	    char *comm, *state;
	    ssize_t len;

	    len = xread(fp,entry,MAXNAMLEN);
	    close(fp);

	    if (!len)
		goto ignore;		/* Bogus */

	    comm  = index(entry,  ' ');
	    state = index(++comm, ' ');
	    *state++ = '\0';

	    if ( ((flags & NZOMBIE) ? state[0] != 'Z' : 1) && strcmp(comm, swapname) == 0 ) {
		continue;		/* Found */
	    }
	}
    ignore:

	/* Remove this entry in remember */
	if (p == remember) {
	    if (n) n->prev = (PROC*)0;
	    remember = n;
	    free(p);
	} else if (l) {
	    if (n) n->prev = l;
	    l->next = n;
	    free(p);
	} else {
	    int saveerr = errno;
	    warn("error in linked list handling\n");
	    free(swapname);
	    errno = saveerr;
	    return -1;
	}

    }

    free(swapname);
    return 0;			/* Nothing found */
}

/*
 * Clear out the ignore proc list from the remember proc list.
 */
void clear_pids (void)
{
    PROC *p, *n, *l;

    n = remember;
    l = (PROC*)0;
    for(p = remember; n; p = n) {
	l = p->prev;
	n = p->next;

	if (!check_ignore(p->pid) && !check_ignore(p->sid))
	    continue;

	/* Remove this entry in remember */
	if (p == remember) {
	    if (n) n->prev = (PROC*)0;
	    remember = n;
	    free(p);
	} else if (l) {
	    if (n) n->prev = l;
	    l->next = n;
	    free(p);
	}
    }
}

void check_su()
{
#if DEBUG
    printf("Real user ID: %d\n",(int)getuid());
#else
    if ( 0 != (int)getuid() )
	error(1, "Only root sould run this program\n");
#endif
    return;
}

/* Used in libinit.h in the inlined function set_newenv */
void addnewenv ( const char * name, const char * entry )
{
    unsigned i, len = strlen(name);
    char *cp = (char*)xmalloc(len+2+strlen(entry));
    extern unsigned newenvc;

    (void)strcat(strcat(strcpy(cp,name),"="),entry);

    for (i=0; i < newenvc; i++)
	if (strncmp (cp, newenvp[i], len) == 0 &&
	    (newenvp[i][len] == '=' || newenvp[i][len] == '\0'))
		break;

    if (i == MAXENV) {
	puts ("Environment overflow");
	return;
    }

    if (i == newenvc) {
	newenvp[newenvc++] = cp;
	newenvp[newenvc] = (char*)0;
    } else {
	free (newenvp[i]);
	newenvp[i] = cp;
    }
}

/* Used in libinit.h in the inlined functions set_newenv set_environ */
char ** runlevel(const char *file)
{
    struct utmp *ut = (struct utmp*)0;
    char **ret = (char**)xmalloc(2 * sizeof(char*));

    if (file)
	utmpname(file);
    else
	utmpname(UTMP_FILE);

    setutent();
    while ( (ut = getutent()) != (struct utmp*)0 )
	if ( ut->ut_type == RUN_LVL )
	    break;
    endutent();

    ret[0] = (char*)xmalloc(sizeof(char[2]));
    ret[1] = (char*)xmalloc(sizeof(char[2]));

    ret[0][1] = ret[1][1] = '\0';
#if DEBUG
    ret[0][0] = '0';
    ret[1][0] = '3';
#else
    if (ut && ut->ut_pid) {
	ret[0][0] = ut->ut_pid / 256;
	ret[1][0] = ut->ut_pid % 256;
    } else {
	ret[0][0] = 'N';
	ret[1][0] = '?';
    }
#endif

    return ret;
}

/* Used in killproc.c to translate signal names into signal numbers */
int signame_to_signum (const char *sig)
{
    int n;

    init_signames();
    if (!strncasecmp (sig, "sig", 3))
	sig += 3;
    for (n = 1; n < NSIG+1; n++) {
	if ( sys_signame [n] && !strcasecmp (sys_signame [n], sig) )
	    return n;
	if ( sys_sigalias[n] && !strcasecmp (sys_sigalias[n], sig) )
	    return n;
    }

    return -1;
}

/*
 * Follow the link to its full deep, this because
 * to get the real file name stored in lnk[].
 */
static char lnk[PATH_MAX+1];
int rlstat(char ** file, struct stat *st, const unsigned short flags)
{
    int ret = -1;
    int deep = MAXSYMLINKS;
    char * cur_file = *file;

    if (lstat(cur_file, st) < 0)
	goto out;

    ret = 0;
    if (*file == (void *)&lnk[0])	/* already done that */
	goto out;

    do {
	const char *prev_file;
	int cnt;

	ret = 0;
	if (!S_ISLNK(st->st_mode))
	    goto out;
	ret = -1;

	if ((prev_file = strdupa(cur_file)) == NULL)
	    error(100, "strdupa(): %s\n", strerror(errno));

	if ((cnt = readlink(cur_file, lnk, PATH_MAX)) < 0)
	    goto out;
	lnk[cnt] = '\0';

	if (lnk[0] != '/') {		/* Construct a new valid file name */
	    const char *lastslash;

	    if ((lastslash = strrchr(prev_file, '/'))) {
		size_t dirname_len = lastslash - prev_file + 1;

		if (dirname_len + cnt > PATH_MAX)
		    cnt = PATH_MAX - dirname_len;

		memmove(&lnk[dirname_len], &lnk[0], cnt + 1);
		memcpy(&lnk[0], prev_file, dirname_len);
	    }
	}
	cur_file = &lnk[0];

	if (lstat(cur_file, st) < 0)
	    goto out;

	if (deep-- <= 0) {
	    errno = ELOOP;
	    goto out;
	}

    } while (S_ISLNK(st->st_mode));

    ret = 0;

    if (flags & FLWLINK)
	*file = cur_file;

out:
    return ret;
}

/*
 * Code below is for checking for binaries located on a NFS
 * based file system.  With this help we can switch off the
 * the various stat()'s checks become dead locked if the
 * corresponding NFS servers are not online.
 */

typedef struct _nfs_
{
    struct _nfs_ *next;		/* Pointer to next struct. */
    struct _nfs_ *prev;		/* Pointer to previous st. */
    char * name;
    size_t nlen;
} NFS;

NFS * nfs = (NFS*)0;

void init_nfs(void)
{
    struct stat st;
    struct mntent * ent;
    FILE * mnt;

    /* Stat /proc/version to see if /proc is mounted. */
    if (stat("/proc/version", &st) < 0)
	getproc();

    if ((mnt = setmntent("/proc/mounts", "r")) == (FILE*)0)
	return;

    while ((ent = getmntent(mnt))) {
	if (!strcasecmp(MNTTYPE_NFS, ent->mnt_type)) {
	    NFS * p = (NFS*)xmalloc(sizeof(NFS));
	    p->name = xstrdup(ent->mnt_dir);
	    p->nlen = strlen(p->name);
	    if (nfs)
		nfs->prev = p;
	    p->next = nfs;
	    p->prev = (NFS*)0;
	    nfs = p;
	}
    }

    endmntent(mnt);
}

boolean check4nfs(const char * path)
{
    char buf[PATH_MAX+1];
    const char *curr;
    int deep = MAXSYMLINKS;

    if (!nfs) goto out;

    curr = path;
    do {
	const char *prev;
	int len;

	if ((prev = strdupa(curr)) == NULL)
	    error(100, "strdupa(): %s\n", strerror(errno));

	errno = 0;
	if ((len = readlink(curr, buf, PATH_MAX)) < 0)
	    break;
	buf[len] = '\0';

	if (buf[0] != '/') {
	    const char *slash;

	    if ((slash = strrchr(prev, '/'))) {
		size_t off = slash - prev + 1;

		if (off + len > PATH_MAX)
		    len = PATH_MAX - off;

		memmove(&buf[off], &buf[0], len + 1);
		memcpy(&buf[0], prev, off);
	    }
	}
	curr = &buf[0];

	if (deep-- <= 0) {
	    errno = ELOOP;
	    return true;
	}

    } while (true);

    if (errno == EINVAL) {
	NFS *p, *n, *l;
	n = nfs;
	l = (NFS*)0;
	for (p = nfs; n; p = n) {
	    l = p->prev;
	    n = p->next;
	    if (!strncmp(curr, p->name, p->nlen))
		return true;
	}
    }
out:
    return false;
}
/* libinit.c ends here */
