#include "config.h"

#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <gtk/gtk.h>

#include "wmpasman.h"
#include "die.h"
#include "dock.h"
#include "popup_editpassword.h"
#include "secureentry.h"

typedef enum {
    GENERATE_INCLUDE_NEVER = 0,
    GENERATE_INCLUDE_OPTIONAL = 1,
    GENERATE_INCLUDE_REQUIRED = 2
} generate_include_t;


static void gen_responded(GObject *dialog, gint response_id, gpointer d G_GNUC_UNUSED) {
    if (response_id == GTK_RESPONSE_ACCEPT) {
        GObject *parent = G_OBJECT(gtk_window_get_transient_for(GTK_WINDOW(dialog)));
        GtkEntryBuffer *pw = GTK_ENTRY_BUFFER(g_object_get_data(parent, "pw"));

        gint len = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(g_object_get_data(dialog, "length")));
        generate_include_t inc_upper = GPOINTER_TO_INT(g_object_get_data(dialog, "inc-upper"));
        generate_include_t inc_lower = GPOINTER_TO_INT(g_object_get_data(dialog, "inc-lower"));
        generate_include_t inc_number = GPOINTER_TO_INT(g_object_get_data(dialog, "inc-number"));
        generate_include_t inc_other = GPOINTER_TO_INT(g_object_get_data(dialog, "inc-other"));
        const gchar *other_text = gtk_entry_get_text(GTK_ENTRY(g_object_get_data(dialog, "other")));
        gint other_len = gtk_entry_get_text_length(GTK_ENTRY(g_object_get_data(dialog, "other")));

        gint chars_len = (inc_upper ? 26 : 0) + (inc_lower ? 26 : 0) + (inc_number ? 26 : 0) + (inc_other ? g_utf8_strlen(other_text, other_len) : 0);
        struct sym {
            gunichar c;
            char type;
        } *chars = g_new(struct sym, chars_len);
        gint i = 0;

        if (inc_upper) {
            for (gunichar c = 'A'; c <= 'Z'; c++) {
                chars[i++] = (struct sym){ c, 0 };
            }
        }
        if (inc_lower) {
            for (gunichar c = 'a'; c <= 'z'; c++) {
                chars[i++] = (struct sym){ c, 1 };
            }
        }
        if (inc_number) {
            for (gunichar c = '0'; c <= '9'; c++) {
                chars[i++] = (struct sym){ c, 2 };
            }
        }
        if (inc_other) {
            gboolean any = FALSE;
            for (const gchar *p = other_text, *e = other_text + other_len; p < e; p = g_utf8_next_char(p)) {
                gunichar c = g_utf8_get_char(p);
                if (!(inc_upper && c >= 'A' && c <= 'Z') && !(inc_lower && c >= 'a' && c <= 'z') && !(inc_number && c >= '0' && c <= '1')) {
                    chars[i++] = (struct sym){c, 3};
                    any = TRUE;
                }
            }
            if (!any) {
                inc_other = GENERATE_INCLUDE_NEVER;
            }
        }
        chars_len = i;

        gint chars_bits = 0;
        for (gint b = 1; chars_len > b; b <<= 1, chars_bits++);
        gint chars_mask = (1 << chars_bits) - 1;

        gchar *dummy = g_malloc(len * 6);
        SecretValue *v = secret_value_new(dummy, len * 6, "text/plain; charset=utf-8");
        g_free(dummy);

        GError *err = NULL;
        errno = 0;
        int fd = open("/dev/urandom", O_RDONLY);
        if (fd < 0) {
            g_set_error(&err, APP_GENERIC_ERROR, 0, "Failed opening /dev/urandom: %s", strerror(errno));
            goto fail;
        }

        gchar *pass = (gchar *)secret_value_get(v, NULL);
        int tries = 0;
        while (1) {
            gint rand = 0;
            gint bits = 0;
            gchar *p = pass;
            gboolean used[4] = {FALSE, FALSE, FALSE, FALSE};
            for (gint i = 0; i < len; ) {
                while (bits < chars_bits) {
                    char r;
                    errno = 0;
                    if (read(fd, &r, 1) <= 0) {
                        g_set_error(&err, APP_GENERIC_ERROR, 0, "Failed reading from /dev/urandom: %s", strerror(errno));
                        goto fail;
                    }
                    rand = (rand << 8) | r;
                    bits += 8;
                }
                gint r = rand & chars_mask;
                rand >>= chars_bits;
                bits -= chars_bits;
                if (r >= chars_len) {
                    continue;
                }
                struct sym *s = &chars[r];
                p += g_unichar_to_utf8(s->c, p);
                used[(int)s->type] = TRUE;
                i++;
            }
            if ((inc_upper != GENERATE_INCLUDE_REQUIRED || used[0]) &&
                (inc_lower != GENERATE_INCLUDE_REQUIRED || used[1]) &&
                (inc_number != GENERATE_INCLUDE_REQUIRED || used[2]) &&
                (inc_other != GENERATE_INCLUDE_REQUIRED || used[3])
            ) {
                gtk_entry_buffer_set_text(pw, pass, p - pass);
                gtk_widget_destroy(GTK_WIDGET(dialog));
                break;
            }
            if (++tries >= 1000) {
                g_set_error(&err, APP_GENERIC_ERROR, 0, "Requirements not met after %d tries, giving up", tries);
                goto fail;
            }
        }
        
fail:
        if (err) {
            error_alert_with_parent(GTK_WINDOW(dialog), "Random password generation failed: %s", err);
            g_clear_error(&err);
        }
        if (fd > 0) {
            close(fd);
        }
        g_free(chars);
        secret_value_unref(v);
        return;
    }

    gtk_widget_destroy(GTK_WIDGET(dialog));
}

static void gen_check_valid(GObject *dialog, gpointer d G_GNUC_UNUSED) {
    gint len = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(g_object_get_data(dialog, "length")));
    generate_include_t inc_upper = GPOINTER_TO_INT(g_object_get_data(dialog, "inc-upper"));
    generate_include_t inc_lower = GPOINTER_TO_INT(g_object_get_data(dialog, "inc-lower"));
    generate_include_t inc_number = GPOINTER_TO_INT(g_object_get_data(dialog, "inc-number"));
    generate_include_t inc_other = GPOINTER_TO_INT(g_object_get_data(dialog, "inc-other"));
    gint other_len = gtk_entry_get_text_length(GTK_ENTRY(g_object_get_data(dialog, "other")));

    gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog), GTK_RESPONSE_ACCEPT,
        len > 3 && (inc_upper || inc_lower || inc_number || (inc_other && other_len > 0))
    );
}

static void include_button_toggled(GtkToggleButton *b, GObject *dialog) {
    if (gtk_toggle_button_get_active(b)) {
        const gchar *setting = g_object_get_data(G_OBJECT(b), "setting");
        gpointer value = g_object_get_data(G_OBJECT(b), "value");
        g_object_set_data(dialog, setting, value);
    }
    gen_check_valid(dialog, NULL);
}

static void add_include_buttons(GtkWidget *owner, GtkGrid *grid, gint row, gchar *setting, generate_include_t dflt) {
    GtkWidget *b1 = gtk_radio_button_new_with_label(NULL, "Never");
    g_object_set_data(G_OBJECT(b1), "setting", setting);
    g_object_set_data(G_OBJECT(b1), "value", GINT_TO_POINTER(GENERATE_INCLUDE_NEVER));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(b1), dflt == GENERATE_INCLUDE_NEVER);
    gtk_grid_attach(grid, b1, 1, row, 1, 1);

    GtkWidget *b2 = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(b1), "Optional");
    g_object_set_data(G_OBJECT(b2), "setting", setting);
    g_object_set_data(G_OBJECT(b2), "value", GINT_TO_POINTER(GENERATE_INCLUDE_OPTIONAL));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(b2), dflt == GENERATE_INCLUDE_OPTIONAL);
    gtk_grid_attach(grid, b2, 2, row, 1, 1);

    GtkWidget *b3 = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(b1), "Required");
    g_object_set_data(G_OBJECT(b3), "setting", setting);
    g_object_set_data(G_OBJECT(b3), "value", GINT_TO_POINTER(GENERATE_INCLUDE_REQUIRED));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(b3), dflt == GENERATE_INCLUDE_REQUIRED);
    gtk_grid_attach(grid, b3, 3, row, 1, 1);

    g_signal_connect(G_OBJECT(b1), "toggled", G_CALLBACK(include_button_toggled), owner);
    g_signal_connect(G_OBJECT(b2), "toggled", G_CALLBACK(include_button_toggled), owner);
    g_signal_connect(G_OBJECT(b3), "toggled", G_CALLBACK(include_button_toggled), owner);

    g_object_set_data(G_OBJECT(owner), setting, GINT_TO_POINTER(dflt));
}

static void gen_filter_input(GtkEditable *e, const gchar *text, gint length, gint *position, gpointer data) {
    const gchar *curtext = gtk_entry_get_text(GTK_ENTRY(e));
    gint curlen = gtk_entry_get_text_length(GTK_ENTRY(e));
    gchar *insert = g_new(gchar, length+7);
    gchar *inspos = insert;

    for (const gchar *p = text, *end = text + length; p < end; p = g_utf8_next_char(p)) {
        gunichar c = g_utf8_get_char(p);
        // Skip combining characters and soft-hyphen, too much trouble
        if (g_unichar_iszerowidth(c) || c == 0xad) {
            continue;
        }

        // Put the character at the end of the insert-buffer, and null-terminate
        gint l = g_unichar_to_utf8(c, inspos);
        inspos[l] = '\0';

        // If the inserted character already appears in the insert-buffer, skip
        // it.
        if (inspos != insert && g_strstr_len(insert, inspos-insert, inspos)) {
            continue;
        }

        // If the inserted character appears elsewhere in the entry, remove it.
        gchar *pp;
        while ((pp = g_strstr_len(curtext, curlen, inspos)) != NULL) {
            gtk_editable_delete_text(e, pp - curtext, pp - curtext + l);
            if (pp - curtext < *position) {
                (*position) -= l;
            }
            curtext = gtk_entry_get_text(GTK_ENTRY(e));
            curlen = gtk_entry_get_text_length(GTK_ENTRY(e));
        }

        // Now formally include the new character in the insert-buffer
        inspos += l;
    }
    if (inspos != insert) {
        g_signal_handlers_block_by_func(G_OBJECT(e), G_CALLBACK(gen_filter_input), data);
        gtk_editable_insert_text(e, insert, inspos - insert, position);
        g_signal_handlers_unblock_by_func(G_OBJECT(e), G_CALLBACK(gen_filter_input), data);
    }
    g_signal_stop_emission_by_name(G_OBJECT(e), "insert_text");
    g_free(insert);
}

static void do_generate_password(GtkWindow *parent, gpointer d G_GNUC_UNUSED) {
    GtkWidget *dialog = gtk_dialog_new_with_buttons(
        "Generate random password",
        parent,
        GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL,
        "_Cancel", GTK_RESPONSE_REJECT,
        "_Generate", GTK_RESPONSE_ACCEPT,
        NULL
    );
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_ACCEPT);
    GtkWidget *grid = gtk_grid_new();
    gtk_container_add(GTK_CONTAINER(gtk_dialog_get_content_area(GTK_DIALOG(dialog))), grid);

    gtk_grid_attach(GTK_GRID(grid), gtk_label_new("Length:"), 0, 0, 1, 1);

    GtkWidget *length_entry = gtk_spin_button_new_with_range(4, 100, 1);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(length_entry), 8);
    gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(length_entry), TRUE);
    gtk_spin_button_set_snap_to_ticks(GTK_SPIN_BUTTON(length_entry), TRUE);
    g_signal_connect_swapped(G_OBJECT(length_entry), "value-changed", G_CALLBACK(gen_check_valid), dialog);
    gtk_grid_attach(GTK_GRID(grid), length_entry, 1, 0, 3, 1);

    gtk_grid_attach(GTK_GRID(grid), gtk_label_new("Uppercase:"), 0, 1, 1, 1);
    add_include_buttons(dialog, GTK_GRID(grid), 1, "inc-upper", GENERATE_INCLUDE_OPTIONAL);

    gtk_grid_attach(GTK_GRID(grid), gtk_label_new("Lowercase:"), 0, 2, 1, 1);
    add_include_buttons(dialog, GTK_GRID(grid), 2, "inc-lower", GENERATE_INCLUDE_OPTIONAL);

    gtk_grid_attach(GTK_GRID(grid), gtk_label_new("Numbers:"), 0, 3, 1, 1);
    add_include_buttons(dialog, GTK_GRID(grid), 3, "inc-number", GENERATE_INCLUDE_OPTIONAL);

    gtk_grid_attach(GTK_GRID(grid), gtk_label_new("Other characters:"), 0, 4, 1, 1);
    add_include_buttons(dialog, GTK_GRID(grid), 4, "inc-other", GENERATE_INCLUDE_NEVER);

    GtkWidget *other_entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(other_entry), "!\"#$%&'()*+,-./:;<=>?@[\\]^_`{|}~");
    g_signal_connect_swapped(G_OBJECT(other_entry), "changed", G_CALLBACK(gen_check_valid), dialog);
    g_signal_connect(G_OBJECT(other_entry), "insert_text", G_CALLBACK(gen_filter_input), NULL);
    gtk_grid_attach(GTK_GRID(grid), other_entry, 1, 5, 3, 1);

    g_object_set_data(G_OBJECT(dialog), "length", length_entry);
    g_object_set_data(G_OBJECT(dialog), "other", other_entry);

    g_signal_connect(G_OBJECT(dialog), "response", G_CALLBACK(gen_responded), NULL);
    gtk_widget_show_all(dialog);
}



static void toggle_active(GtkDialog *dialog, gpointer d G_GNUC_UNUSED){
    GtkEntry *line1 = GTK_ENTRY(g_object_get_data(G_OBJECT(dialog), "line1"));
    GtkEntry *line2 = GTK_ENTRY(g_object_get_data(G_OBJECT(dialog), "line2"));
    GtkEntryBuffer *pw = GTK_ENTRY_BUFFER(g_object_get_data(G_OBJECT(dialog), "pw"));

    gtk_dialog_set_response_sensitive(dialog, GTK_RESPONSE_ACCEPT,
        (gtk_entry_get_text_length(line1) > 0 || gtk_entry_get_text_length(line2) > 0) && gtk_entry_buffer_get_length(pw) > 0
    );
}

static void toggle_visibility(GtkToggleButton *t, GtkEntry *b){
    gtk_entry_set_visibility(b, gtk_toggle_button_get_active(t));
}

static void save_cb(GError *err) {
    if (err) {
        error_alert("Save failed: %s", err);
    }
    app_finish_work();
}

static void responded(GObject *dialog, gint response_id, GObject *pw) {
    if (response_id == GTK_RESPONSE_ACCEPT) {
        const gchar *line1 = gtk_entry_get_text(GTK_ENTRY(g_object_get_data(dialog, "line1")));
        const gchar *line2 = gtk_entry_get_text(GTK_ENTRY(g_object_get_data(dialog, "line2")));
        SecretValue *v = secure_entry_buffer_get_secret_value(SECURE_ENTRY_BUFFER(g_object_get_data(dialog, "pw")));

        GCancellable *cancel = app_start_work();
        if (pw) {
            password_edit(pw, line1, line2, v, cancel, save_cb);
        } else {
            password_create(line1, line2, v, cancel, save_cb);
        }
    }

    gtk_widget_destroy(GTK_WIDGET(dialog));
    if (pw) {
        g_object_unref(pw);
    }
}

void do_edit_password(GtkWindow *parent, GObject *pw, const gchar *line1, const gchar *line2, SecretValue *v) {
    GtkWidget *dialog = gtk_dialog_new_with_buttons(
        pw ? "Edit password" : "Create password",
        parent,
        GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL,
        "_Cancel", GTK_RESPONSE_REJECT,
        "_Save", GTK_RESPONSE_ACCEPT,
        NULL
    );
    gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_ACCEPT);
    
    GtkWidget *grid = gtk_grid_new();
    gtk_container_add(GTK_CONTAINER(gtk_dialog_get_content_area(GTK_DIALOG(dialog))), grid);

    gtk_grid_attach(GTK_GRID(grid), gtk_label_new("Password Label:"), 0, 0, 2, 2);

    GtkWidget *entry_line1 = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(entry_line1), 40);
    if (line1) {
        gtk_entry_set_text(GTK_ENTRY(entry_line1), line1);
    }
    gtk_editable_set_position(GTK_EDITABLE(entry_line1), -1);
    gtk_widget_set_hexpand(entry_line1, TRUE);
    gtk_entry_set_activates_default(GTK_ENTRY(entry_line1), TRUE);
    gtk_grid_attach(GTK_GRID(grid), entry_line1, 2, 0, 2, 1);

    GtkWidget *entry_line2 = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(entry_line2), 40);
    if (line2) {
        gtk_entry_set_text(GTK_ENTRY(entry_line2), line2);
    }
    gtk_editable_set_position(GTK_EDITABLE(entry_line2), -1);
    gtk_widget_set_hexpand(entry_line2, TRUE);
    gtk_entry_set_activates_default(GTK_ENTRY(entry_line2), TRUE);
    gtk_grid_attach(GTK_GRID(grid), entry_line2, 2, 1, 2, 1);

    gtk_grid_attach(GTK_GRID(grid), gtk_label_new("Password:"), 0, 2, 1, 1);

    GtkWidget *check = gtk_check_button_new_with_label("Show");
    gtk_widget_set_halign(check, GTK_ALIGN_START);
    gtk_grid_attach(GTK_GRID(grid), check, 1, 2, 1, 1);

    GtkEntryBuffer *pw_buffer = secure_entry_buffer_new();
    if (v) {
        gsize len;
        const gchar *pw = secret_value_get(v, &len);
        gtk_entry_buffer_set_text(pw_buffer, pw, g_utf8_strlen(pw, len));
    }
    GtkWidget *entry_pw = gtk_entry_new_with_buffer(pw_buffer);
    gtk_entry_set_input_purpose(GTK_ENTRY(entry_pw), GTK_INPUT_PURPOSE_PASSWORD);
    gtk_entry_set_visibility(GTK_ENTRY(entry_pw), FALSE);
    gtk_editable_select_region(GTK_EDITABLE(entry_pw), 0, -1);
    gtk_widget_set_hexpand(entry_pw, TRUE);
    gtk_entry_set_activates_default(GTK_ENTRY(entry_pw), TRUE);
    gtk_grid_attach(GTK_GRID(grid), entry_pw, 0, 3, 3, 1);

    GtkWidget *random_button = gtk_button_new_with_mnemonic("_Random");
    g_signal_connect_swapped(G_OBJECT(random_button), "clicked", G_CALLBACK(do_generate_password), dialog);
    gtk_grid_attach(GTK_GRID(grid), random_button, 3, 3, 1, 1);

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), FALSE);
    g_signal_connect(G_OBJECT(check), "toggled", G_CALLBACK(toggle_visibility), entry_pw);

    g_object_set_data(G_OBJECT(dialog), "line1", (gpointer)entry_line1);
    g_object_set_data(G_OBJECT(dialog), "line2", (gpointer)entry_line2);
    g_object_set_data(G_OBJECT(dialog), "pw", (gpointer)pw_buffer);

    g_signal_connect_swapped(G_OBJECT(entry_line1), "changed", G_CALLBACK(toggle_active), dialog);
    g_signal_connect_swapped(G_OBJECT(entry_line2), "changed", G_CALLBACK(toggle_active), dialog);
    g_signal_connect_swapped(G_OBJECT(entry_pw), "changed", G_CALLBACK(toggle_active), dialog);
    toggle_active(GTK_DIALOG(dialog), NULL);

    if (pw) {
        g_object_ref(pw);
    }
    g_signal_connect(G_OBJECT(dialog), "response", G_CALLBACK(responded), pw);

    gtk_widget_show_all(dialog);
}
