// Copyright 2020-2025 Consensys Software Inc.
// Licensed under the Apache License, Version 2.0. See the LICENSE file for details.

// Code generated by consensys/gnark-crypto DO NOT EDIT

package fflonk

import (
	"crypto/sha256"

	"github.com/consensys/gnark-crypto/ecc/bn254/fr"
	"github.com/consensys/gnark-crypto/ecc/bn254/kzg"
)

// This example demonstrates how to open a list of polynomials on a list of points.
func Example_batchOpen() {

	// sample a list of polynomials, we have 5 packs of polynomials,
	// each pack will be opened on its own set of points.
	nbPacks := 5

	// The first set of polynomials contains 2 polynomials, the second 3, etc.
	// The i-th set of polynomials is opened on the i-th set of points. The first
	// set of point contains 4 points, the second 5, etc.
	nbPolynomialsPerPack := []int{2, 3, 4, 5, 6}
	nbPointsPerPack := []int{4, 5, 6, 7, 8}
	points := make([][]fr.Element, nbPacks)
	polynomials := make([][][]fr.Element, nbPacks)
	for i := 0; i < nbPacks; i++ {
		polynomials[i] = make([][]fr.Element, nbPolynomialsPerPack[i])
		for j := 0; j < nbPointsPerPack[i]; j++ {

			// random size for the polynomials
			polynomials[i][j] = make([]fr.Element, j+10)
		}

		// random number of points per pack
		points[i] = make([]fr.Element, i+5)
	}

	// commit to the folded Polynomials. In each pack, we fold the polynomials in a similar way
	// as in the FFT. If the given pack contains 3 polynomials P1,P2,P3, the folded polynomial
	// that we commit to is P1(X^t)+XP2(X^t)+X^2P3(X^t) where t is the smallest number dividing
	// r-1 bounding above the number of polynomials, which is 3 here.
	var err error
	digests := make([]kzg.Digest, nbPacks)
	for i := 0; i < nbPacks; i++ {
		digests[i], err = FoldAndCommit(polynomials[i], testSrs.Pk)
		if err != nil {
			panic(err)
		}
	}

	// compute the opening proof. We first pick a hash function that will be used for the FS challenge
	// derivation.
	hf := sha256.New()
	proof, err := BatchOpen(polynomials, digests, points, hf, testSrs.Pk)
	if err != nil {
		panic(err)
	}

	// Check the opening proof. The claimed values of the i-th pack of polynomials are the evaluation
	// of the i-th pack of polynomials, evaluated on the t-th powers of points[i], where t is the smallest
	// integer bounding above the number of polynomials in the pack that divides r-1, the field on which
	// the polynomials are defined.
	//
	// For instance, proof.ClaimedValues[i][j][k] contains the evaluation of the j-th polynomial of the i-th
	// pack, on points[i][k]^t, where t is defined as above.
	err = BatchVerify(proof, digests, points, hf, testSrs.Vk)
	if err != nil {
		panic(err)
	}
}
