/*******************************************************************************
*                                                                              *
* align_columns.c -- align text into columns                                   *
*                                                                              *
* Copyright (C) 1998 Steve Haehn                                               *
*                                                                              *
* This is free software; you can redistribute it and/or modify it under the    *
* terms of the GNU General Public License as published by the Free Software    *
* Foundation; either version 2 of the License, or (at your option) any later   *
* version.                                                                     *
*                                                                              *
* This software is distributed in the hope that it will be useful, but WITHOUT *
* ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or        *
* FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License *
* for more details.                                                            *
*                                                                              *
* You should have received a copy of the GNU General Public License along with *
* software; if not, write to the Free Software Foundation, Inc., 59 Temple     *
* Place, Suite 330, Boston, MA  02111-1307 USA                                 *
*                                                                              *
*******************************************************************************/

/*============================================================================*/
/*====================  OPERATING SYSTEM INCLUDE FILES  ======================*/
/*============================================================================*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

/*============================================================================*/
/*                              SYMBOL DEFINITIONS                            */
/*============================================================================*/

#define MAX_LINES 500
#define MAX_COLUMNS 50
#define MAX_LINE_SIZE 300
#define EOS '\0'

#define FPATH_SEPARATOR '/'
#define FALSE (bool_t)0
#define TRUE  (bool_t)!FALSE

/*============================================================================*/
/*                          VARIABLE TYPE DEFINITIONS                         */
/*============================================================================*/

typedef int bool_t;

/*============================================================================*/
/*                             PROGRAM PROTOTYPES                             */
/*============================================================================*/

#ifdef NEED_GETOPT_PROTO_TYPE
int   getopt( int, char **, char * );
#endif

/*============================================================================*/
/*================================= PROGRAMS =================================*/
/*============================================================================*/

#ifdef NEED_STRTOK_R

char * strtok_r
(
    char  *inoutp_src,       /* Source string for tokenizing, or NULL */
    char  *inp_separators,   /* List of token separators              */
    char **inoutpp_next_pos  /* Remainder of initial source string    */
)
{
    char           *crnt_token, *next_separator;
    int             sepLen;

    /*------------------------------------------
    * Use given source string, or when NULL use
    * remaining string from previous call.
    *------------------------------------------*/
    crnt_token = (inoutp_src) ? inoutp_src : *inoutpp_next_pos;
    
    /*---------------------------------------------------
    * When there is a possibility of a token present ...
    *---------------------------------------------------*/
    if (crnt_token != NULL)
    {
        /*-------------------------------------------
        * ... skip over all leading token separators 
        *     prior to determining position of token.
        *-------------------------------------------*/
        sepLen = strspn(crnt_token, inp_separators);
        
        if (crnt_token[sepLen] == EOS)
            crnt_token = *inoutpp_next_pos = NULL;
        else
            crnt_token = crnt_token + sepLen;

        /*---------------------------------------------------------
        * When there is a token present, locate the next separator
        * (if any) and prepare for retrieval of subsequent tokens.
        *--------------------------------------------------------*/
        if (crnt_token != NULL)
        {
            next_separator = strpbrk(crnt_token, inp_separators);
    
            if (next_separator == NULL)
                *inoutpp_next_pos = NULL;  /* no more tokens */
            else
            {
                *inoutpp_next_pos = next_separator + 1;
                *next_separator   = EOS;   /* terminate token string */
            }
        }
    }

    return ( crnt_token );
}

#endif /* NEED_STRTOK_R */

/*----------------------------------------------------------------------------*/

void pad_column(

    int in_max_column_size,
    int in_column_size
)
{
    int i;

    /*-----------------------------
    * Pad short fields with spaces.
    *-----------------------------*/
    for( i=in_max_column_size - in_column_size; i > 0; i-- )
        putchar( ' ' );
}

/*----------------------------------------------------------------------------*/

void align_columns( char * in_delimiters, bool_t in_right_align )
{
    char *newLines[MAX_LINES];
    int  columns_in_line[MAX_LINES];
    int  column_size[MAX_COLUMNS];
    char line[MAX_LINE_SIZE];
    char leader[MAX_LINE_SIZE];
    char *visible, *remainder;
    int invisibleLen;
    int shortestInvisible = MAX_LINE_SIZE + 1;
    int lineCnt = 0;
    int i, crntLine;
    int lineLen;
    int column_cnt = 0;
    
    /*------------------------------------------------
    * Do not know the sizes of any of the columns yet.
    *------------------------------------------------*/
    for( i=0; i< MAX_COLUMNS; i++ ) column_size[i] = 0;
    
    /*----------------------------------------------------------------
    * PASS 1: Gather text and statistics to be used in text alignment.
    *          Read all the lines from standard input. Determine the
    *          shortest leading white space to be used at the front 
    *          of each line. Select column components using delimiters
    *          and remove all intervening white space.
    *----------------------------------------------------------------*/
    while( fgets( line, MAX_LINE_SIZE, stdin ) && lineCnt < MAX_LINES )
    {
        char * delim_pos;
        
        /*--------------------------------------
        * Allocate space for a copy of the line.
        *--------------------------------------*/
        lineLen = strlen(line);
        newLines[lineCnt] = malloc( lineLen+1 );

        invisibleLen = strspn( line, " \t" );
        visible = line + invisibleLen;
	
        /*------------------------------------
        * Remember shortest leading whitespace.
        *-------------------------------------*/
        if( invisibleLen < shortestInvisible ) 
        {
            shortestInvisible = invisibleLen;
            memcpy( leader, line, shortestInvisible );
        }
        
        /*--------------------------------
        * Figure out maximum column sizes.
        *--------------------------------*/
        remainder = visible;
        column_cnt = 0;
        newLines[lineCnt][0] = EOS;
        
        while( remainder && *remainder != EOS && column_cnt < MAX_COLUMNS )
        {
            char  delimiter[2];
            char *column;
            int   col_size;
            
            /*--------------------------------------------------------
            * Get a copy of the data for the column and the delimiter.
            *--------------------------------------------------------*/
            if( (delim_pos = strpbrk( remainder, in_delimiters )) )
            {
                delimiter[0] = *delim_pos;
                delimiter[1] = EOS;
            } 

            column = strtok_r( NULL, in_delimiters, &remainder );
            
            /*---------------------------------------------
            * Protection against delimiters at end of line.
            *---------------------------------------------*/
            if( column == NULL )
            {
                /*---------------------------------------------------
                * When the string tokenization above eats the newline
                * it has to be put back into the structure.
                *---------------------------------------------------*/
                strcat( newLines[lineCnt], "\n" );
            }
            else
            {
                col_size = strlen( column );

                strcat( newLines[lineCnt], column ); /* add column to line */

                if( delim_pos != NULL )
                {
                    strcat( newLines[lineCnt], delimiter );
                    col_size++;  /* +1 includes delimiter */
                }

                /*------------------------------------------
                * If not working on the last column, 
                * ignore leading white space in next column.
                *------------------------------------------*/
                if( remainder != NULL && *remainder != EOS )
                {
                     remainder = remainder + strspn( remainder, " \t" );
                }

                /*--------------------------------------
                * Remember largest size of current field
                *--------------------------------------*/
                if( column_size[ column_cnt ] < col_size + 1 )
                {
                    if( delim_pos != NULL && !isspace( *delimiter )) 
                        column_size[ column_cnt ] = col_size + 1;
                    else
                         column_size[ column_cnt ] = col_size;
                }

                column_cnt++;
            }
        }
        
        columns_in_line[lineCnt] = column_cnt;
        lineCnt++;
    }

    /*--------------------------
    * PASS 2: Align column text.
    *--------------------------*/
    if( lineCnt > 0 )
    {
        for( crntLine = 0; crntLine < lineCnt; crntLine++ )
        {
            int current_column = 0;
            
            /*--------------------------
            * Output leading whitespace.
            *--------------------------*/
            for( i = 0; i < shortestInvisible; i++ )
                putchar( leader[i] );
            
            remainder = &newLines[crntLine][0];
            
            while( current_column < columns_in_line[ crntLine ] )
            {
                char * delim_pos = strpbrk( remainder, in_delimiters );
                int    col_size  = strcspn( remainder, in_delimiters );
		
               /*----------------------------------------------
                * The following will catch thoses ending columns
                * which do not have a delimiter character.
                *----------------------------------------------*/
                if( delim_pos != NULL || in_right_align )
                    col_size++;
		
                if( in_right_align )
                {
                    pad_column( column_size[current_column], col_size );
                }
                
                /*------------------------------------
                * Output data and delimiter in column.
                *------------------------------------*/
                for( i = 0; i < col_size; i++ )
                {
                    putchar( *(remainder++) );
                }
		
                if( ! in_right_align )
                {
                    /*------------------------------------------
                    * If this is not the last column, add space.
                    *------------------------------------------*/
                    if( current_column < columns_in_line[ crntLine ] - 1 )
                    {
                        pad_column( column_size[current_column], col_size );
                    }
                }
		
                current_column++;
            }
            
            /*-------------------------------------------
            * This forces out line when it has some other
            * delimiter preceding the end of the line.
            *-------------------------------------------*/
            if( remainder != NULL && *remainder == '\n' )
                putchar( '\n' );
        }
    }
}

/*----------------------------------------------------------------------------*/

char *
remove_file_path
(
    char * inp_file_path
)
{
    char * file_name = strrchr( inp_file_path, FPATH_SEPARATOR );
    
    return ( file_name ) ? file_name+1 : inp_file_path;
}

/*----------------------------------------------------------------------------*/

int main( int argc, char * argv[] )
{
    extern int   optind;
    
    char * pgm_name    = remove_file_path( argv[0] );
    char * delimiters  = ", "; /* default delimiter characters */
    bool_t right_align = FALSE;
    char * tmp;
    
    if( argc >= 2 )
    {
        int c;
        
        while( (c = getopt( argc, argv, "r" )) != -1 )
        {
            switch( c )
            {
              case 'r':
                right_align = TRUE;
                break;
            
              case '?':
                fprintf( stderr, "\n" );
                fprintf( stderr, "Usage: %s [-r] [column_delimiters]\n", pgm_name);
                fprintf( stderr, "\n" );
                fprintf( stderr, "  -r directs the program to right justify data in column\n" );
                fprintf( stderr, "     The default column delimiters are \"%s\".\n", delimiters );
                fprintf( stderr, "\n" );
                exit( 1 );
            }
        }
        
        if( optind < argc )
        {
            delimiters = argv[optind];
        }
    }
    /*---------------------------------------------
    * Append newline as line termination delimiter.
    *---------------------------------------------*/
    tmp = malloc( strlen( delimiters ) + 2 );
    strcpy( tmp, delimiters );
    strcat( tmp, "\n" );
    delimiters = tmp;
    
    align_columns( delimiters, right_align );
    return 0;
}
