/*
 *   dspquery.c -- query functions for the DSP Manager
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
#include <stdio.h>
#include <string.h>
#include "dspmgr.h"

#include "dspquery.h"
#include "dspgpc.h"
#include "dspmlvl.h"                   /* DEC                               */
#include "dspstcmp.h"                  /* CH01 Include function prototypes  */
#include "dspstruc.h"

ULONG ParseSegLabels(PRSEG,PVOID,PVOID,USHORT *,USHORT *,BOOL);

/**************************START OF SPECIFICATIONS **************************/
/* SUBROUTINE NAME: QueryDSPInfo                                            */
/*                                                                          */
/* FUNCTION: Returns DSP specific information.                              */
/*                                                                          */
/* INPUT: hDsp : Handle of DSP to be querried                               */
/*        pusSize : Size of pDspInfoBuf to hold the information             */
/*        pDspInfoBuf : pointer to RDSPINFO structure                       */
/*                                                                          */
/* OUTPUT: fill pDspInfoBuf with structure of RDSPINFO                      */
/*         If rc returns DSP_INS_BUFFER then pusSize returns the valid size */
/*                                                                          */
/*         return code: DSP_NOERROR                                         */
/*                      DSP_INS_BUFFER                                      */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/


RC FAR PASCAL QueryDSPInfo(HDSP hDsp,PUSHORT pusSize,PVOID pDspInfoBuf,PVOID
                            pRealBuffer)
{
   ULONG      ulRC;
   PRTSK      prtaskTemp;
   PRTSK      prtaskTempRoot;
   PRFM       prFMTemp;
   PRFM       prFMTempRoot;
   PRMOD      prmodTemp;
   PRMOD      prmodTempRoot;
   PBYTE      pBuf_Index = (BYTE *)pDspInfoBuf+sizeof(RDSPINFO);
   PBYTE      pBuf_End = (BYTE *)pDspInfoBuf+*pusSize;
   USHORT     usNumTasks = 0;
   USHORT     usNumFMs = 0;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspquery::QueryDSPInfo entry hDsp %x\n",(int)hDsp);

   ulRC = CheckDSPStatus((PRDSP) hDsp);       /* CH01 */
   if (ulRC != DSP_NOERROR)
      return (ulRC);

   if (pBuf_Index <= pBuf_End) {       /* only write if there is room in    */
                                       /* buffer                            */
      ((RDSPINFO *)pDspInfoBuf)->DSPINFO_ulIPSLeft = hDsp->DSP_ulCPSFree;
      ((RDSPINFO *)pDspInfoBuf)->DSPINFO_ulRemainingBusCap =
         pg->prmgrTail->MGR_ulRemBusCapacity;
      ((RDSPINFO *)pDspInfoBuf)->DSPINFO_hMod_MwaveOS = hDsp->DSP_prmodOS;
      ((RDSPINFO *)pDspInfoBuf)->DSPINFO_phTask = (PHTASK)((BYTE *)pRealBuffer
         +(pBuf_Index-(PBYTE)pDspInfoBuf));
   }

  /**************************************************************************/
  /* Find all tasks on this DSP: For each module on the Manager of this DSP */
  /* check every tasks of this module to see if parent DSP matchs this DSP, */
  /* if matchs and actual size of buffer is large enough then fill buffer.  */
  /**************************************************************************/

   if ((hDsp->DSP_prmgr->MGR_prmodTail) != NULL) {/* there are modules      */
      prmodTemp = hDsp->DSP_prmgr->MGR_prmodTail->MOD_prmodNext; // head of list

      prmodTempRoot = prmodTemp;

      do {                             /* Find the tasks (if any) on the    */
                                       /* modules in the module list        */
         if (prmodTemp->MOD_achValid != MODVALIDATE) // check mod validity
            return (DSP_INTERNAL_CORRUPT);
         if ((prmodTemp->MOD_prtskTail) != NULL) {/* there are Tasks on this*/
                                       /* module                            */
            prtaskTemp = prmodTemp->MOD_prtskTail->TSK_prtskNext; // head of list
            prtaskTempRoot = prtaskTemp; // root of task list
            do {                       /* copy the hTasks in this task list */
                                       /* into the buffer                   */
               if (prtaskTemp->TSK_achValid != TSKVALIDATE)
                  return (DSP_INTERNAL_CORRUPT);
               if (prtaskTemp->TSK_prdsp == hDsp) {/* task belongs to this  */
                                       /* DSP                               */
                  if ((pBuf_Index+sizeof(HTASK)) <= pBuf_End) // room in buffer?
                     *(PHTASK)pBuf_Index = prtaskTemp;
                  pBuf_Index += sizeof(HTASK);/* inc pointer as a PHTASK    */
                  usNumTasks++;
               }
               prtaskTemp = prtaskTemp->TSK_prtskNext; // next task on list
            }  while (prtaskTemp != prtaskTempRoot);/* not end of task list */
         }
         prmodTemp = prmodTemp->MOD_prmodNext;// next module on list
      }  while (prmodTemp != prmodTempRoot);  /* not end of module list     */
      if (pBuf_Index <= pBuf_End)             /* only write if there is room*/
                                              /* in buffer                  */
         {
         ((RDSPINFO *)pDspInfoBuf)->DSPINFO_usNumTasks = usNumTasks;
         if (usNumTasks == 0)                  /* if no tasks where found, */
                                               /* set phTask to NULL       */
            ((RDSPINFO *)pDspInfoBuf)->DSPINFO_phTask = NULL;
      }
   }
   else {                              /* there are no modules and therefore*/
                                       /* no tasks                          */
      if (pBuf_Index <= pBuf_End)      /* only write if there is room in    */
                                       /* buffer                            */
         {
         ((RDSPINFO *)pDspInfoBuf)->DSPINFO_phTask = NULL;
         ((RDSPINFO *)pDspInfoBuf)->DSPINFO_usNumTasks = 0;
      }
   }

  /**************************************************************************/
  /* Does any Frame Manager exist ?                                         */
  /* if so, fix up RDSPINFO_phFM and copy the hFMs into the buffer          */
  /* if not, assign 0 to RDSPINFO_usNumFM and NULL to RDSPINFO_phFM         */
  /**************************************************************************/

   if ((hDsp->DSP_prfmTail) != NULL) { /* there are FMs to copy into buffer */
      if (pBuf_Index <= pBuf_End)      /* only write if there is room in    */
                                       /* buffer                            */
         ((RDSPINFO *)pDspInfoBuf)->DSPINFO_phFM = (PHFM)((BYTE *)pRealBuffer+
            (pBuf_Index-(PBYTE)pDspInfoBuf));
      prFMTemp = hDsp->DSP_prfmTail->FM_prfmNext;/* Head of FM list         */
      prFMTempRoot = prFMTemp;         /* Root of FM linked list            */
      do {                             /* copy the FMs into the buffer      */
         if ((pBuf_Index+sizeof(HFM)) <= pBuf_End)/* still room in buffer?  */
            *(PHFM)pBuf_Index = prFMTemp;/* copy hFM to buffer              */
         pBuf_Index += sizeof(HFM);    /* inc pointer as a PHFM             */
         usNumFMs++;
      }  while ((prFMTemp = prFMTemp->FM_prfmNext) != prFMTempRoot);

      if (pBuf_Index <= pBuf_End)      /* only write if there is room in    */
                                       /* buffer                            */
         ((RDSPINFO *)pDspInfoBuf)->DSPINFO_usNumFM = usNumFMs;
   }
   else {                              /* there are no FMs                  */
      if (pBuf_Index <= pBuf_End)      /* only write if there is room in    */
                                       /* buffer                            */
         {
         ((RDSPINFO *)pDspInfoBuf)->DSPINFO_phFM = NULL;
         ((RDSPINFO *)pDspInfoBuf)->DSPINFO_usNumFM = 0;
      }
   }
   if (pBuf_Index > pBuf_End)          /* was the buffer to small?          */
      {                                /* the buffer was to small           */
      ulRC = DSP_INS_BUFFER;
      *pusSize = (USHORT)(pBuf_Index-(PBYTE)pDspInfoBuf);
   }
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspquery::QueryDSPInfo exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/* SUBROUTINE NAME: QueryManagerInfo                                        */
/*                                                                          */
/* FUNCTION: Returns the DSP Manager information.                           */
/*                                                                          */
/* INPUT: hMgr : Handle of Manager to be querried                           */
/*        pusSize : Size of pMgrInfoBuf to hold the information             */
/*        pMgrInfoBuf : pointer to RMGRINFO structure                       */
/*                                                                          */
/* OUTPUT: fill pMgrInfoBuf with structure of RMGRINFO                      */
/*         If rc returns DSP_INS_BUFFER then pusSize returns the valid size */
/*                                                                          */
/*         return code: DSP_NOERROR                                         */
/*                      DSP_INS_BUFFER                                      */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC FAR PASCAL QueryManagerInfo(PUSHORT pusSize,PVOID pMgrInfoBuf,PVOID
                                pRealBuffer)
{
   ULONG      ulRC = DSP_NOERROR;
   PRMOD      prmodTemp;
   PRMOD      prmodTempRoot;
   PRDSP      prdspTemp;
   PRDSP      prdspTempRoot;
   HMGR       hMgr = pg->prmgrTail;
   PBYTE      pBuf_Index = (BYTE *)pMgrInfoBuf+sizeof(RMGRINFO);
   PBYTE      pBuf_End = (BYTE *)pMgrInfoBuf+*pusSize;
   USHORT     usNumMods = 0;
   USHORT     usNumDSPs = 0;
                                       /* Note there's only ONE             */

   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspquery::QueryManagerInfo entry\n");

   if (pBuf_Index <= pBuf_End)         /* only write if there is room in    */
                                       /* buffer                            */
      {
      ((RMGRINFO *)pMgrInfoBuf)->MGRINFO_usMajorVersion = MGRMAJORVERSION;
      ((RMGRINFO *)pMgrInfoBuf)->MGRINFO_usMinorVersion = MGRMINORVERSION;
   }

   if (hMgr->MGR_prmodTail != NULL)
      {                                /* there are hmods to copy into      */
                                       /* buffer                            */
      prmodTemp = hMgr->MGR_prmodTail->MOD_prmodNext; // head of module list
      prmodTempRoot = prmodTemp;       /* Root of rmod linked list          */
      if (pBuf_Index <= pBuf_End)      /* only write if there is room in    */
                                       /* buffer                            */
         ((RMGRINFO *)pMgrInfoBuf)->MGRINFO_phMod = (PHMOD)((BYTE *)
            pRealBuffer+(pBuf_Index-(PBYTE)pMgrInfoBuf));
      do {                             /* copy hmods into buffer            */
         if (prmodTemp->MOD_achValid != MODVALIDATE) // check validity
            return (DSP_INTERNAL_CORRUPT);
         if ((pBuf_Index+sizeof(HMOD)) <= pBuf_End)/* still room in buffer? */
            *(PHMOD)pBuf_Index = prmodTemp;/* copy hmod to buffer           */
         pBuf_Index += sizeof(HMOD);   /* inc pointer as a PHMOD            */
         usNumMods++;                  /* increase count of module          */
      }  while ((prmodTemp = prmodTemp->MOD_prmodNext) != prmodTempRoot);

      if (pBuf_Index <= pBuf_End)      /* only write if there is room in    */
                                       /* buffer                            */
         ((RMGRINFO *)pMgrInfoBuf)->MGRINFO_usNumMod = usNumMods;
   }
        /*CH02*/                       /* there are no hmods                */
   else if (pBuf_Index <= pBuf_End) {  /* only write if there is room in    */
      ((RMGRINFO *)pMgrInfoBuf)->MGRINFO_phMod = NULL;
      ((RMGRINFO *)pMgrInfoBuf)->MGRINFO_usNumMod = 0;
   }

   if (hMgr->MGR_prdspTail != NULL)
      {                                /* there are hDSPs to copy into      */
                                       /* buffer                            */
      prdspTemp = hMgr->MGR_prdspTail;
      prdspTempRoot = prdspTemp;
      if (pBuf_Index <= pBuf_End)      /* only write if there is room in    */
                                       /* buffer                            */
         ((RMGRINFO *)pMgrInfoBuf)->MGRINFO_phDSP = (PHDSP)((BYTE *)
            pRealBuffer+(pBuf_Index-(PBYTE)pMgrInfoBuf));
      do {                             /* copy hmods into buffer            */
         if (prdspTemp->DSP_achValid != DSPVALIDATE) // check validity
            return (DSP_INTERNAL_CORRUPT);
         if ((pBuf_Index+sizeof(HDSP)) <= pBuf_End)/* still room in buffer? */
            *(PHDSP)pBuf_Index = prdspTemp;/* copy hmod to buffer           */
         pBuf_Index += sizeof(HDSP);   /* inc pointer as a PHDSP            */
         usNumDSPs++;                  /* increase count of module          */
      }  while ((prdspTemp = prdspTemp->DSP_prdspNext) != prdspTempRoot);

      if (pBuf_Index <= pBuf_End)      /* only write if there is room in    */
                                       /* buffer                            */
         ((RMGRINFO *)pMgrInfoBuf)->MGRINFO_usNumDSP = usNumDSPs;
   }
         /*CH02*/                      /* there are no hDSPs                */
   else if (pBuf_Index <= pBuf_End) {  /* only write if there is room in    */
      ((RMGRINFO *)pMgrInfoBuf)->MGRINFO_phDSP = NULL;
      ((RMGRINFO *)pMgrInfoBuf)->MGRINFO_usNumDSP = 0;
   }

  /**End copy hDSPs into buffer *********************************************/

   if (pBuf_Index > pBuf_End) {        /* was the buffer too small?         */
                                       /* the buffer was to small           */
      ulRC = DSP_INS_BUFFER;
      *pusSize = (USHORT)(pBuf_Index-(PBYTE)pMgrInfoBuf);
   }
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspquery::QueryManagerInfo exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/* SUBROUTINE NAME: QueryMiscInfo                                           */
/*                                                                          */
/* FUNCTION: Returns miscellaneous information.                             */
/*                                                                          */
/* INPUT: usType : Type of handle                                           */
/*        hMisc : Handle of Object to be querried                           */
/*        pusSize : Size of pMiscInfoBuf to hold the information            */
/*        pMiscInfoBuf : pointer to a structure that will contain           */
/*        information depend on usType                                      */
/*                                                                          */
/* OUTPUT: fill pMiscInfoBuf with structure that depends on usType          */
/*         If rc returns DSP_INS_BUFFER then pusSize returns the valid size */
/*                                                                          */
/*         return code: DSP_NOERROR                                         */
/*                      DSP_INSUFF_MEM  (Internal buffer size exceeded)     */
/*                      DSP_INS_BUFFER                                      */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC FAR PASCAL QueryMiscInfo(USHORT usType,PVOID hMisc,PUSHORT pusSize,PVOID
                             pMiscInfoBuf,PVOID pRealBuffer)
{
   ULONG      ulRC = DSP_NOERROR;
   PVOID      pBuffer;
   HSEG       hSeg;
   HTASK      hTask;
   USHORT     usbufsize;
   USHORT     usSizeTemp;
   USHORT     usinfosize;
   USHORT     uscnt;



   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspquery::QueryMiscInfo entry usType %x\n",usType);
   /*************************************************************************/
   /* Depend on usType, we return different informations in buffer          */
   /* that contains different struc.                                        */
   /*************************************************************************/

   switch (usType) {

      case  MISC_GPC :
         if (((PRGPC)hMisc)->GPC_achValid == GPCVALIDATE) {

            if (((PRGPC)hMisc)->GPC_pszName != NULL)
             usSizeTemp = (USHORT)(strlen(((PRGPC)hMisc)->GPC_pszName)+1);/**/

            else
               usSizeTemp = 1;
            usSizeTemp += sizeof(RGPCINFO);
            if (*pusSize >= usSizeTemp) {
               ((RGPCINFO *)pMiscInfoBuf)->GPCINFO_pszName = ((PRGPC)hMisc)->
                  GPC_pszName;
               ((RGPCINFO *)pMiscInfoBuf)->GPCINFO_usflgKind = ((PRGPC)hMisc)
                  ->GPC_usflgKind;
               ((RGPCINFO *)pMiscInfoBuf)->GPCINFO_lBufferSize = ((PRGPC)hMisc
                  )->GPC_lBufferSize;
               ((RGPCINFO *)pMiscInfoBuf)->GPCINFO_ulDSPAddr = ((PRGPC)hMisc)
                  ->GPC_prseg->SEG_prmem->MEM_ulSegAddr+((PRGPC)hMisc)->
                     GPC_ulOffset;
               ((RGPCINFO *)pMiscInfoBuf)->GPCINFO_usAddrMode = ((PRGPC)hMisc)
                  ->GPC_usAddrMode;
               ((RGPCINFO *)pMiscInfoBuf)->GPCINFO_usMode = ((PRGPC)hMisc)->
                  GPC_usMode;
               ((RGPCINFO *)pMiscInfoBuf)->GPCINFO_ulSamplerate = ((PRGPC)
                  hMisc)->GPC_ulSamplerate;
               ((RGPCINFO *)pMiscInfoBuf)->GPCINFO_usMinPtrSep = ((PRGPC)hMisc
                  )->GPC_usMinPtrSep;
               ((RGPCINFO *)pMiscInfoBuf)->GPCINFO_usMaxWPF = ((PRGPC)hMisc)->
                  GPC_usMaxWPF;
               ((RGPCINFO *)pMiscInfoBuf)->GPCINFO_usIntegerStride = ((PRGPC)
                  hMisc)->GPC_usIntegerStride;
               ((RGPCINFO *)pMiscInfoBuf)->GPCINFO_usFractStride = ((PRGPC)
                  hMisc)->GPC_usFractStride;
               ((RGPCINFO *)pMiscInfoBuf)->GPCINFO_lMaxdelay = ((PRGPC)hMisc)
                  ->GPC_lMaxdelay;
               ((RGPCINFO *)pMiscInfoBuf)->GPCINFO_usProtocol = ((PRGPC)hMisc)
                  ->GPC_usProtocol;
               ((RGPCINFO *)pMiscInfoBuf)->GPCINFO_hgpcOwner = ((PRGPC)hMisc)
                  ->GPC_prgpcOwner;
               ((RGPCINFO *)pMiscInfoBuf)->GPCINFO_hgpcUserListNext = ((PRGPC)
                  hMisc)->GPC_prgpcUserListNext;
               ((RGPCINFO *)pMiscInfoBuf)->GPCINFO_usUseCount = ((PRGPC)hMisc)
                  ->GPC_usUseCount;
               if (((PRGPC)hMisc)->GPC_prmemBuffer == NULL)/* user GPC does */
                  ((RGPCINFO *)pMiscInfoBuf)->GPCINFO_ulBufferAddr = 0;
               else
                  ((RGPCINFO *)pMiscInfoBuf)->GPCINFO_ulBufferAddr = ((PRGPC)
                     hMisc)->GPC_prmemBuffer->MEM_ulSegAddr;

               ((RGPCINFO *)pMiscInfoBuf)->GPCINFO_pszName = (PSZ)((PBYTE)
                  pRealBuffer+sizeof(RGPCINFO));

            /****************************************************************/
            /*     Add the Name        to the end of the buffer.            */
            /*     Increment to the end existing data and store the         */
            /*     string.  See buffer above                                */
            /****************************************************************/

               pBuffer = (BYTE *)pMiscInfoBuf+sizeof(RGPCINFO);

               if (((PRGPC)hMisc)->GPC_pszName != NULL)
                  strcpy(pBuffer, ((PRGPC)hMisc)->GPC_pszName);
               else
                  *(PSZ)pBuffer = 0;
            }
            else {
               *pusSize = usSizeTemp;
               ulRC = DSP_INS_BUFFER;
            }
         }
         else {
            ulRC = DSP_INV_HANDLE;
         }
         break;
      case  MISC_DMA :
         if (((PRLDMA)hMisc)->LDMA_achValid == LDMAVALIDATE) {

            if (((PRLDMA)hMisc)->LDMA_pszDMAName != NULL)
               usSizeTemp = (USHORT)(strlen(((PRLDMA)hMisc)->LDMA_pszDMAName)+
                  1);
            else
               usSizeTemp = 1;
            usSizeTemp += sizeof(RDMAINFO);
            if (*pusSize >= usSizeTemp) {

               pBuffer = (BYTE *)pMiscInfoBuf+sizeof(RDMAINFO);

               if (((PRLDMA)hMisc)->LDMA_pszDMAName != NULL)
                  strcpy(pBuffer, ((PRLDMA)hMisc)->LDMA_pszDMAName);
               else
                  *(PSZ)pBuffer = 0;
                                       /* buffer                            */

               ((RDMAINFO *)pMiscInfoBuf)->DMAINFO_pszDMAName = (PSZ)((BYTE *)
                  pRealBuffer+sizeof(RDMAINFO));
               ((RDMAINFO *)pMiscInfoBuf)->DMAINFO_ulPackets = ((PRLDMA)hMisc)
                  ->LDMA_lPackets;
               ((RDMAINFO *)pMiscInfoBuf)->DMAINFO_ulMaxPkSize = ((PRLDMA)
                  hMisc)->LDMA_ulMaxPkSize;
               ((RDMAINFO *)pMiscInfoBuf)->DMAINFO_ulMaxPerDMA = ((PRLDMA)
                  hMisc)->LDMA_lMaxPerDMA;
               ((RDMAINFO *)pMiscInfoBuf)->DMAINFO_ulStride = ((PRLDMA)hMisc)
                  ->LDMA_ulStride;
               ((RDMAINFO *)pMiscInfoBuf)->DMAINFO_ulHold = ((PRLDMA)hMisc)->
                  LDMA_ulHold;
               ((RDMAINFO *)pMiscInfoBuf)->DMAINFO_usFlag = ((PRLDMA)hMisc)->
                  LDMA_usflg;
            }
            else {
               *pusSize = usSizeTemp;
               ulRC = DSP_INS_BUFFER;
            }
         }
         else {
            ulRC = DSP_INV_HANDLE;
         }
         break;
      case  MISC_IPC :
         if (((HIPC)hMisc)->IPCTE_achValid == IPCTEVALIDATE) {
            if (*pusSize >= sizeof(RIPCINFO)) {

               ((RIPCINFO *)pMiscInfoBuf)->IPCINFO_ulCallBack = (ULONG)((HIPC)
                  hMisc)->IPCTE_pfnCallBack;
               ((RIPCINFO *)pMiscInfoBuf)->IPCINFO_usMask = (USHORT)((HIPC)
                  hMisc)->IPCTE_usMask;
            }
            else {
               *pusSize = sizeof(RIPCINFO);
               ulRC = DSP_INS_BUFFER;
            }
         }
         else {
            ulRC = DSP_INV_HANDLE;
         }
         break;
      case  MISC_ITCB :
         if (((PRITCB)hMisc)->ITCB_achValid == ITCBVALIDATE) {

            if (((PRITCB)hMisc)->ITCB_pszName != NULL)
               usSizeTemp = (USHORT)(strlen(((PRITCB)hMisc)->ITCB_pszName)+1);

            else
               usSizeTemp = 1;
            usSizeTemp += sizeof(RITCBINFO);
            if (*pusSize >= usSizeTemp) {

               ((RITCBINFO *)pMiscInfoBuf)->ITCBINFO_pszName = (PSZ)((BYTE *)
                  pRealBuffer+sizeof(RITCBINFO));
               ((RITCBINFO *)pMiscInfoBuf)->ITCBINFO_usflgKind = ((PRITCB)
                  hMisc)->ITCB_usflgKind;
               ((RITCBINFO *)pMiscInfoBuf)->ITCBINFO_ulSize = ((PRITCB)hMisc)
                  ->ITCB_ulSize/2;
               ((RITCBINFO *)pMiscInfoBuf)->ITCBINFO_ulDSPAddr = ((PRITCB)
                  hMisc)->ITCB_prseg->SEG_prmem->MEM_ulSegAddr+((PRITCB)hMisc)
                     ->ITCB_ulOffset;
               ((RITCBINFO *)pMiscInfoBuf)->ITCBINFO_hITCBPrimary = ((PRITCB)
                  hMisc)->ITCB_pritcbPrimary;
               ((RITCBINFO *)pMiscInfoBuf)->ITCBINFO_hITCBSecList = ((PRITCB)
                  hMisc)->ITCB_pritcbSecListNext;
               ((RITCBINFO *)pMiscInfoBuf)->ITCBINFO_usUseCount = ((PRITCB)
                  hMisc)->ITCB_usUseCount;

            /****************************************************************/
            /*     Add the Name        to the end of the buffer.            */
            /*     Increment to the end existing data and store the         */
            /*     string.  See buffer above                                */
            /****************************************************************/

               pBuffer = (BYTE *)pMiscInfoBuf+sizeof(RITCBINFO);

               if (((PRITCB)hMisc)->ITCB_pszName != NULL)
                  strcpy(pBuffer, ((PRITCB)hMisc)->ITCB_pszName);
               else
                  *(PSZ)pBuffer = 0;

            }
            else {
               *pusSize = usSizeTemp;
               ulRC = DSP_INS_BUFFER;
            }
         }
         else {
            ulRC = DSP_INV_HANDLE;
         }
         break;
      case  MISC_FM :
         if (((PRFM)hMisc)->FM_achValid == FMVALIDATE) {

      /**Start **************************************************************/
      /* typedef struct rfminfo {                                           */
      /*     PSZ    FMINFO_pszName;                                         */
      /*     ULONG  FMINFO_ulFMCB;                                          */
      /*     PSZ    FMINFO_pszHWintsource;                                  */
      /*     ULONG  FMINFO_ulSampleRate;                                    */
      /*     USHORT FMINFO_usSamplesPerFrame;                               */
      /*     USHORT FMINFO_usUseCount;                                      */
      /*     USFLG  FMINFO_usflgType;                                       */
      /*   } RFMINFO;                                                       */
      /**********************************************************************/

            usSizeTemp = sizeof(RFMINFO);

      /* add room for FM Name                                               */

            if (((PRFM)hMisc)->FM_pszName != NULL)
              usSizeTemp += (USHORT)(strlen(((PRFM)hMisc)->FM_pszName)+1);/**/

            else
               usSizeTemp += 1;

      /* set pointer to where the HWintsource will be put in MiscInfoBuf    */

            pBuffer = (BYTE *)pMiscInfoBuf+usSizeTemp;

      /* add room for HW interrupt source Name                              */

            if (((PRFM)hMisc)->FM_pszHWintsource != NULL)
               usSizeTemp += (USHORT)(strlen(((PRFM)hMisc)->FM_pszHWintsource)
                  +1);
            else
               usSizeTemp += 1;
            if (*pusSize >= usSizeTemp) {

               ((RFMINFO *)pMiscInfoBuf)->FMINFO_pszName = (PSZ)((BYTE *)
                  pRealBuffer+sizeof(RFMINFO));
               ((RFMINFO *)pMiscInfoBuf)->FMINFO_ulFMCB = ((PRFM)hMisc)->
                  FM_prmemFMCB->MEM_ulSegAddr;
               ((RFMINFO *)pMiscInfoBuf)->FMINFO_pszHWintsource = (PSZ)((BYTE
                  *)pRealBuffer+((PBYTE)pBuffer-(PBYTE)pMiscInfoBuf));
               ((RFMINFO *)pMiscInfoBuf)->FMINFO_ulSampleRate = ((PRFM)hMisc)
                  ->FM_ulSampleRate;
               ((RFMINFO *)pMiscInfoBuf)->FMINFO_usSamplesPerFrame = ((PRFM)
                  hMisc)->FM_usSamplesPerFrame;
               ((RFMINFO *)pMiscInfoBuf)->FMINFO_usUseCount = ((PRFM)hMisc)->
                  FM_usUseCount;
               ((RFMINFO *)pMiscInfoBuf)->FMINFO_usflgType = ((PRFM)hMisc)->
                  FM_usflgType;

         /* put HWintsource name into buffer                                */

               if (((PRFM)hMisc)->FM_pszHWintsource != NULL)
                  strcpy(pBuffer, ((PRFM)hMisc)->FM_pszHWintsource);
               else
                  *(PSZ)pBuffer = 0;


         /*******************************************************************/
         /*     Add the Name        to the end of the buffer.               */
         /*     Increment to the end existing data and store the            */
         /*     string.  See buffer above                                   */
         /*******************************************************************/

               pBuffer = (BYTE *)pMiscInfoBuf+sizeof(RFMINFO);
               if (((PRFM)hMisc)->FM_pszName != NULL)/* CH03                */
                  strcpy(pBuffer, ((PRFM)hMisc)->FM_pszName);
               else
                  *(PSZ)pBuffer = 0;

            }
            else {
               *pusSize = usSizeTemp;
               ulRC = DSP_INS_BUFFER;
            }
         }
         else {
            ulRC = DSP_INV_HANDLE;
         }
         break;
      case  MISC_SEG :

         if (((PRSEG)hMisc)->SEG_achValid == SEGVALIDATE) { // check validity
            hSeg = (PRSEG)hMisc;

         /**Start CH04**********************************                    */
         /*    typedef struct rseginfo {                                    */
         /*      PSZ    SEGINFO_pszInstanceName;                            */
         /*      PSZ    SEGINFO_pszRealName;                                */
         /*      RVER   SEGINFO_rver;                                       */
         /*      RVER   SEGINFO_rverDSP;                                    */
         /*      USFLG  SEGINFO_usflg;                                      */
         /*      ULONG  SEGINFO_ulDSPAddr;                                  */
         /*      ULONG  SEGINFO_ulSize;                                     */
         /*      } RSEGINFO;                                                */
         /*******************************************************************/

            usSizeTemp = sizeof(RSEGINFO);

         /* add room for Instance Name                                      */

            if (hSeg->SEG_pszVirtName != NULL)
               usSizeTemp += (USHORT)(strlen(hSeg->SEG_pszVirtName)+1);
            else
               usSizeTemp += 1;

         /* set pointer to where the Real Name will be put in MiscInfoBuf   */

            pBuffer = (BYTE *)pMiscInfoBuf+usSizeTemp;

         /* add room for Real Name                                          */

            if (hSeg->SEG_pszRealSegName != NULL)
               usSizeTemp += (USHORT)(strlen(hSeg->SEG_pszRealSegName)+1);/**/

            else
               usSizeTemp += 1;

         /*******************************************************************/
         /* If actual size of buffer is large enough then store information */
         /* in buffer and return.                                           */
         /* Return Segment address instead of DSP Address.                  */
         /*******************************************************************/

            if (*pusSize >= usSizeTemp) {
               ((RSEGINFO *)pMiscInfoBuf)->SEGINFO_pszInstanceName = (PSZ)
                  ((BYTE *)pRealBuffer+sizeof(RSEGINFO));
               ((RSEGINFO *)pMiscInfoBuf)->SEGINFO_pszRealName = (PSZ)((BYTE
                  *)pRealBuffer+((PBYTE)pBuffer-(PBYTE)pMiscInfoBuf));

               ((RSEGINFO *)pMiscInfoBuf)->SEGINFO_rver = hSeg->SEG_rver;
               ((RSEGINFO *)pMiscInfoBuf)->SEGINFO_rverDSP = hSeg->SEG_rverDSP
                  ;
               ((RSEGINFO *)pMiscInfoBuf)->SEGINFO_usflg = hSeg->SEG_usflg;

            /* Check For no Memory Allocated cases                          */


               if (hSeg->SEG_prmem != NULL) {
                  ((RSEGINFO *)pMiscInfoBuf)->SEGINFO_ulSize =
                     hSeg->SEG_prmem->MEM_ulSegSize;
                  ((RSEGINFO *)pMiscInfoBuf)->SEGINFO_ulDSPAddr =
                     hSeg->SEG_prmem->MEM_ulSegAddr;
               }
               else {
                  ((RSEGINFO *)pMiscInfoBuf)->SEGINFO_ulSize = 0;
                  ((RSEGINFO *)pMiscInfoBuf)->SEGINFO_ulDSPAddr = 0;
               }

            /* Copy the Real Name into the buffer                           */

               if (hSeg->SEG_pszRealSegName != NULL)
                  strcpy(pBuffer, hSeg->SEG_pszRealSegName);
               else
                  *(PSZ)pBuffer = 0;


          /* set pointer to where the Instance Name will be put in MiscInfoBuf*/

               pBuffer = (BYTE *)pMiscInfoBuf+sizeof(RSEGINFO);

            /* Copy the Instance Name into the buffer.                      */


               if (hSeg->SEG_pszVirtName != NULL)
                  strcpy(pBuffer, hSeg->SEG_pszVirtName);
               else
                  *(PSZ)pBuffer = 0;

            }
            else {

         /*******************************************************************/
         /* If actual size of buffer is not sufficien then returns the valid*/
         /* size of buffer.                                                 */
         /*******************************************************************/

               *pusSize = usSizeTemp;
               ulRC = DSP_INS_BUFFER;
            }
         }
         else {
            ulRC = DSP_INV_HANDLE;
         }
         break;
      case  MISC_LABELS :

         if (((PRTSK)hMisc)->TSK_achValid == TSKVALIDATE) {
            hTask = (PRTSK)hMisc;

         /*******************************************************************/
         /*    typedef struct rlabelinfo {                                  */
         /*       USHORT  LABELINFO_usNumDSLabels;                          */
         /*       PSZ    *LABELINFO_ppszDSLabels;                           */
         /*       USHORT  LABELINFO_usNumISLabels;                          */
         /*       PSZ    *LABELINFO_ppszISLabels;                           */
         /*       } RLABELINFO;                                             */
         /*******************************************************************/

            usinfosize = sizeof(RLABELINFO);
            pBuffer = (BYTE *)pMiscInfoBuf;
            if (usinfosize > *pusSize)
               usbufsize = 0;
            else {
               usbufsize = *pusSize-usinfosize;
               *((ULONG *)&pBuffer) += usinfosize;
               *((ULONG *)&pRealBuffer) += usinfosize;
            }

         /*                                                                 */
         /*  Scan Data Segments defined for public labels                   */
         /*                                                                 */

            ulRC = ParseSegLabels(hTask->TSK_prsegDTail, pBuffer, pRealBuffer,
               &uscnt, &usbufsize, TRUE); /* CH03 */
            usinfosize += usbufsize;   /* return size used                  */
            if (usinfosize > *pusSize)
               usbufsize = 0;
            else {
               ((RLABELINFO *)pMiscInfoBuf)->LABELINFO_usNumDSLabels = uscnt;
               ((RLABELINFO *)pMiscInfoBuf)->LABELINFO_ppszDSLabels =
                  pRealBuffer;

               *((ULONG *)&pBuffer) += usbufsize;/* return size used        */
               *((ULONG *)&pRealBuffer) += usbufsize;
               usbufsize = *pusSize-usinfosize;/* calc remaining size       */
            }

         /*                                                                 */
         /*  Scan Code Segments defined for public labels                   */
         /*                                                                 */

            ulRC = ParseSegLabels(hTask->TSK_prsegITail, pBuffer, pRealBuffer,
               &uscnt, &usbufsize, TRUE); /* CH03 */
            usinfosize += usbufsize;

            if (usinfosize > *pusSize)
               ulRC = DSP_INS_BUFFER;
            else {
               ulRC = DSP_NOERROR;
               ((RLABELINFO *)pMiscInfoBuf)->LABELINFO_usNumISLabels = uscnt;
               ((RLABELINFO *)pMiscInfoBuf)->LABELINFO_ppszISLabels =
                  pRealBuffer;
            }

            *pusSize = usinfosize;     /* return size of required           */
         }
         else {
            ulRC = DSP_INV_HANDLE;     /* invalid task handle for           */
                                       /* MISC_LABELS                       */
         }
         break;

      /**********************************************************************/
      /* CH03: MTS #2891 New function to query segment labels               */
      /**********************************************************************/
      case  MISC_SEGLABELS :

         if (((PRSEG)hMisc)->SEG_achValid == SEGVALIDATE) {
            hSeg = (PRSEG)hMisc;

         /*******************************************************************/
         /*    typedef struct rlabelinfo {                                  */
         /*       USHORT  LABELINFO_usNumDSLabels;  0    for IS segments    */
         /*       PSZ    *LABELINFO_ppszDSLabels;   NULL for IS segments    */
         /*       USHORT  LABELINFO_usNumISLabels;  0    for DS segments    */
         /*       PSZ    *LABELINFO_ppszISLabels;   NULL for DS segments    */
         /*       } RLABELINFO;                                             */
         /*******************************************************************/

            usinfosize = sizeof(RLABELINFO);
            pBuffer = (BYTE *)pMiscInfoBuf;
            if (usinfosize > *pusSize)
               usbufsize = 0;
            else {
               usbufsize = *pusSize-usinfosize;
               *((ULONG *)&pBuffer) += usinfosize;
               *((ULONG *)&pRealBuffer) += usinfosize;
            }

            ulRC = ParseSegLabels(hSeg, pBuffer, pRealBuffer,
               &uscnt, &usbufsize, FALSE);
            usinfosize += usbufsize;   /* return size used                  */
            if (usinfosize <= *pusSize) {

               /* Initialize counts to 0, and PPSZs to NULL */
               (void) memset( pMiscInfoBuf, 0, sizeof(RLABELINFO));

               /* Now set the appropriate count and PPSZ */
               if ((SEG_TYPEMASK & hSeg->SEG_usflg)==SEG_INST) {
                  ((RLABELINFO *)pMiscInfoBuf)->LABELINFO_usNumISLabels = uscnt;
                  ((RLABELINFO *)pMiscInfoBuf)->LABELINFO_ppszISLabels =
                     pRealBuffer;

               } else { /* Data Segment */
                  ((RLABELINFO *)pMiscInfoBuf)->LABELINFO_usNumDSLabels = uscnt;
                  ((RLABELINFO *)pMiscInfoBuf)->LABELINFO_ppszDSLabels =
                     pRealBuffer;
               } /* endif */
            }

            *pusSize = usinfosize;     /* return size of required           */
         }
         else {
            ulRC = DSP_INV_HANDLE;     /* invalid seg handle for function   */
         }
         break;

      default  :
         ulRC = DSP_INV_PARAMETER;
   }
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspquery::QueryMiscInfo exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/* SUBROUTINE NAME: ParseSegLabels                                          */
/*                                                                          */
/* FUNCTION: Parse segment labels for task and place in application buffer. */
/*                                                                          */
/* INPUT: prsegHead : head ptr to list of segments                          */
/*        pBuffer : ptr to buffer owned by daemon                           */
/*        pRealBuffer : ptr to application buffer (not accessable)          */
/*        usSize : size of application buffer remaining                     */
/*                                                                          */
/* OUTPUT: *pBuffer : filled with null terminated segment labels.           */
/*         usCnt : # of labels found                                        */
/*         usSize : buffer size required                                    */
/*                                                                          */
/*         Note: Only the size required returned if size available exceeded.*/
/*                                                                          */
/*         return code: DSP_NOERROR                                         */
/*                      DSP_INS_BUFFER                                      */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

ULONG ParseSegLabels(PRSEG prsegHead, PVOID pBuffer, PVOID pRealBuffer,
                     USHORT *usCnt, USHORT *usSize, BOOL fAllSegs)
{
   PARPUB     prpub;
   PARSTBL    prstring;
   PRSEG      prseg;
   ULONG      ulRC,pbuf1,pbuf2,realbuf;
   USHORT     usinfosize,uspubcnt,usstrlen;

   prseg = prsegHead;
   *usCnt = 0;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspquery::ParseSegLabels entry prsegHead %x\n",(int)prsegHead);
   /*
    *  Loop thru segement(s) to calculate total cnt & establish PSZ array
    *  size to skip over.
                                                                            */

   do {
      *usCnt += prseg->SEG_usPubsCount;
      prseg = prseg->SEG_prsegNext;

   }  while ((prseg != prsegHead) && fAllSegs); /* CH01 */
   usinfosize = *usCnt *sizeof(PSZ);

   pbuf1 = (ULONG)pBuffer;
   realbuf = (ULONG)pRealBuffer;

  /*                                                                        */
  /*  Setup offset ptrs to labels (daemon: pbuf2 and appl: realbuf)         */
  /*                                                                        */

   pbuf2 = *usCnt *sizeof(PSZ);        /* bump pbuf2 & realbuf pass array of*/
                                       /* PSZs                              */
   realbuf += pbuf2;
   pbuf2 += pbuf1;

   prseg = prsegHead;
   do                                  /* loop thru segment(s)              */
      {
      prpub = prseg->SEG_parpub;
      uspubcnt = prseg->SEG_usPubsCount;
      prstring = prseg->SEG_parstblStatic;

      for (; uspubcnt != 0; uspubcnt--, prpub++) {
         if ((prpub->PUB_usflg&PUB_PUBLICMASK) != PUB_PUBLIC)
            *usCnt--;                  /* not public.                       */
         else
            if (prpub->PUB_usLabel > prseg->SEG_ulStaticSize)

               *usCnt--;               /* offset beyond static str          */
            else {

           /*                                                               */
           /*  Label is valid.  Update buffer only if space is available.   */
           /*                                                               */
                                      /*  Add one to length for null char   */

               usstrlen = strlen((PSZ)((BYTE *)(prstring)+prpub->PUB_usLabel+1
                  ))+1;
               usinfosize += usstrlen;
               if (usinfosize <= *usSize) {
                  strcpy((PSZ)pbuf2, (PSZ)((BYTE *)(prstring)+
                     prpub->PUB_usLabel+1));
                  *((PSZ *)pbuf1) = (PSZ)realbuf;
                  pbuf1 += sizeof(PSZ);/* ptr to PSZ array                  */
                  pbuf2 += usstrlen;   /* ptr to LABELs in working buffer   */
                  realbuf += usstrlen; /* ptr to LABELs in user buffer      */
               }

            }
      }

      prseg = prseg->SEG_prsegNext;
   }  while ((prseg != prsegHead) && fAllSegs);  /* CH01 */

  /*                                                                        */
  /*  Check if space required exceeded available memory                     */
  /*                                                                        */

   if (usinfosize > *usSize)
      ulRC = DSP_INS_BUFFER;           /* Insufficient appl buffer          */
   else
      ulRC = DSP_NOERROR;

   *usSize = usinfosize;               /* return space required, available  */
                                       
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspquery::ParseSegLabels exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/* SUBROUTINE NAME: QueryModInfo                                            */
/*                                                                          */
/* FUNCTION: Returns module specific information.                           */
/*                                                                          */
/* INPUT: hMod : Handle of Module to be querried                            */
/*        pusSize : Size of pModInfoBuf to hold the information             */
/*        pModInfoBuf : pointer to RMODINFO structure                       */
/*                                                                          */
/* OUTPUT: fill pModInfoBuf with structure of RMODINFO                      */
/*         If rc returns DSP_INS_BUFFER then pusSize returns the valid size */
/*                                                                          */
/*         return code: DSP_NOERROR                                         */
/*                      DSP_INS_BUFFER                                      */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC FAR PASCAL QueryModuleInfo(HMOD hMod,PUSHORT pusSize,PVOID pModInfoBuf,
                               PVOID pRealBuffer)
{
   ULONG      ulRC = DSP_NOERROR;
   PSZ        pstring;
   PRTSK      prtaskTemp;
   PRTSK      prtaskTempRoot;
   PBYTE      pBuf_Index = (BYTE *)pModInfoBuf+sizeof(RMODINFO);
   PBYTE      pBuf_End = (BYTE *)pModInfoBuf+*pusSize;
   USHORT     usNumTasks = 0;

  /**************************************************************************/
  /*    typedef struct rmodinfo {                                           */
  /*      PSZ     MODINFO_pszModName;                                       */
  /*      USHORT  MODINFO_usNumTask;                                        */
  /*      HTASK   MODINFO_hTask[usNumTask];                                 */
  /*    } RMODINFO;                                                         */
  /* Find all tasks and store them in buffer if buffer is large enough.     */
  /* if not find the valid size of buffer to return later.                  */
  /**************************************************************************/

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspquery::QueryModuleInfo entry hMod %x\n",(int)hMod);

   if (hMod->MOD_achValid == MODVALIDATE)
      {                                /* the module is valid               */
      if ((hMod->MOD_prtskTail) != NULL) {/* there are Tasks on this module */
         prtaskTemp = hMod->MOD_prtskTail->TSK_prtskNext;
         prtaskTempRoot = prtaskTemp;
         do {                          /* copy hTasks into buffer           */
            if (prtaskTemp->TSK_achValid != TSKVALIDATE)
               return (DSP_INTERNAL_CORRUPT);
            if ((pBuf_Index+sizeof(HTASK)) <= pBuf_End) // room in buffer?
               *(PHTASK)pBuf_Index = prtaskTemp;
            pBuf_Index += sizeof(HTASK);/* inc pointer as a PHTASK          */
            usNumTasks++;
            prtaskTemp = prtaskTemp->TSK_prtskNext; // Root of linked list
         }  while (prtaskTemp != prtaskTempRoot); // is it end o the list ?
      }

      pstring = (PSZ)pBuf_Index;       /* set pointer to where string will  */
                                       /* be written                        */
      if (hMod->MOD_pszName != NULL)
         pBuf_Index += (strlen(hMod->MOD_pszName)+1);
      else
         pBuf_Index++;

      if (pBuf_Index <= pBuf_End)      // room in buffer?
         {                             /* there is room in the buffer       */
         ((RMODINFO *)pModInfoBuf)->MODINFO_pszModName = (BYTE *)pRealBuffer+
            (pstring-(PSZ)pModInfoBuf);
         ((RMODINFO *)pModInfoBuf)->MODINFO_usNumTask = usNumTasks;
         ((RMODINFO *)pModInfoBuf)->MODINFO_rver = hMod->MOD_rver;
                                       /* LBM                               */
         if (usNumTasks == 0)
            ((RMODINFO *)pModInfoBuf)->MODINFO_phTask = NULL;
         else
            ((RMODINFO *)pModInfoBuf)->MODINFO_phTask = (PHTASK)((BYTE *)
               pRealBuffer+sizeof(RMODINFO));

         if (hMod->MOD_pszName != NULL)
            strcpy(pstring, hMod->MOD_pszName);
         else
            *pstring = 0;
      }
      else {
         *pusSize = (USHORT)(pBuf_Index-(PBYTE)pModInfoBuf);
         ulRC = DSP_INS_BUFFER;
      }
   }
   else
      ulRC = DSP_INV_HANDLE;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspquery::QueryModuleInfo exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/* SUBROUTINE NAME: QueryTaskInfo                                           */
/*                                                                          */
/* FUNCTION: Returns task specific information.                             */
/*                                                                          */
/* INPUT: hTask : Handle of Task to be querried                             */
/*        pusSize : Size of pTaskInfoBuf to hold the information            */
/*        pTaskInfoBuf : pointer to RTASKINFO structure                     */
/*                                                                          */
/* OUTPUT: fill pTaskInfoBuf with structure of RTASKINFO                    */
/*         If rc returns DSP_INS_BUFFER then pusSize returns the valid size */
/*                                                                          */
/*         return code: DSP_NOERROR                                         */
/*                      DSP_INS_BUFFER                                      */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC FAR PASCAL QueryTaskInfo(HTASK hTask,PUSHORT pusSize,PVOID pTaskInfoBuf,
                             PVOID pRealBuffer)
{
   ULONG      ulRC = DSP_NOERROR;
   PRSEG      prsegTemp;
   PRSEG      prsegTempRoot;
   PRLDMA     prldmaTemp;
   PRLDMA     prldmaTempRoot;
   PRGPC      prgpcTemp;
   PRGPC      prgpcTempRoot;
   PRITCB     pritcbTemp;
   PRITCB     pritcbTempRoot;
   PSZ        pstring=0;
   PBYTE      pBuf_Index = (BYTE *)pTaskInfoBuf+sizeof(RTASKINFO);
   PBYTE      pBuf_End = (BYTE *)pTaskInfoBuf+*pusSize;
   USHORT     usNumSegs = 0;
   USHORT     usNumDMAs = 0;
   USHORT     usNumGPCs = 0;
   USHORT     usNumITCBs = 0;


   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspquery::QueryTaskInfo entry hTask %x\n",(int)hTask);

     /**************CH04*******************************************************/
     /*     typedef struct rtaskinfo {                                      */
     /*       PSZ     TSKINFO_pszInstanceName;                              */
     /*       HDSP    TSKINFO_hDSP;                                         */
     /*       RVER    TSKINFO_rver;                                         */
     /*       ULONG   TSKINFO_ulCPF;                                        */
     /*       USHORT  TSKINFO_usFlgType;                                    */
     /*       USHORT  TSKINFO_usFlgState;                                   */
     /*       HFM     TSKINFO_hFM;                                          */
     /*       HIPC    TSKINFO_hIPC;                                         */
     /*       USHORT  TSKINFO_usNumSeg;                                     */
     /*       HSEG    TSKINFO_hSeg[usNumSeg];                               */
     /*       USHORT  TSKINFO_usNumDMA;                                     */
     /*       HDMA    TSKINFO_hDMA[usNumDMA];                               */
     /*       USHORT  TSKINFO_usNumGPC;                                     */
     /*       HGPC    TSKINFO_hGPC[usNumGPC];                               */
     /*       USHORT  TSKINFO_usNumITCB;                                    */
     /*       HITCB   TSKINFO_hITCB[usNumITCB];                             */
     /*     } RTASKINFO;                                                    */
     /* Find all data and code segments and store them in buffer if the buffer*/
     /* large enough, if not find the valid size of buffer to return later  */
     /*      The first data segment on the list is the default data segment.*/
     /***********************************************************************/

   if (hTask->TSK_achValid == TSKVALIDATE)
      {
      if (pBuf_Index <= pBuf_End)      // room in buffer?
         ((RTASKINFO *)pTaskInfoBuf)->TSKINFO_phSeg = (PHSEG)((BYTE *)
            pRealBuffer+(pBuf_Index-(PBYTE)pTaskInfoBuf));
      prsegTemp = hTask->TSK_prsegDTail->SEG_prsegNext; // For data segments
      prsegTempRoot = prsegTemp;
      do {                             /* copy data hSegs into buffer       */
         if (prsegTemp->SEG_achValid != SEGVALIDATE)
            return (DSP_INTERNAL_CORRUPT);
         if ((pBuf_Index+sizeof(HSEG)) <= pBuf_End) // room in buffer?
            *(PHSEG)pBuf_Index = prsegTemp;
         pBuf_Index += sizeof(HSEG);   /* inc pointer as a PHSEG            */
         usNumSegs++;
         prsegTemp = prsegTemp->SEG_prsegNext; // Root of linked list
      }  while (prsegTemp != prsegTempRoot); // is it end o the list ?



      prsegTemp = hTask->TSK_prsegITail->SEG_prsegNext; // For code segments
      prsegTempRoot = prsegTemp;
      do {                             /* copy instruction hSegs into buffer*/
         if (prsegTemp->SEG_achValid != SEGVALIDATE)
            return (DSP_INTERNAL_CORRUPT);
         if ((pBuf_Index+sizeof(HSEG)) <= pBuf_End) // room in buffer?
            *(PHSEG)pBuf_Index = prsegTemp;
         pBuf_Index += sizeof(HSEG);   /* inc pointer as a PHSEG            */
         usNumSegs++;
         prsegTemp = prsegTemp->SEG_prsegNext;
      }  while (prsegTemp != prsegTempRoot);

      if (pBuf_Index <= pBuf_End)      // room in buffer?
         {                             /* copy NumSegs and phDMA into buffer*/
         ((RTASKINFO *)pTaskInfoBuf)->TSKINFO_usNumSeg = usNumSegs;
         ((RTASKINFO *)pTaskInfoBuf)->TSKINFO_phDMA = (PHDMA)((BYTE *)
            pRealBuffer+(pBuf_Index-(PBYTE)pTaskInfoBuf));
      }

      if (hTask->TSK_prldmaTail != NULL)
         {                             /* there are hDMAs                   */
         prldmaTemp = hTask->TSK_prldmaTail->LDMA_prldmaNext;
         prldmaTempRoot = prldmaTemp;
         do {                          /* copy the hDMAs into the buffer    */
            if (prldmaTemp->LDMA_achValid != LDMAVALIDATE)
               return (DSP_INTERNAL_CORRUPT);
            if ((pBuf_Index+sizeof(HDMA)) <= pBuf_End) // room in buffer?
               *(PHDMA)pBuf_Index = prldmaTemp;
            pBuf_Index += sizeof(HDMA);/* inc pointer as a PRLDMA           */
            usNumDMAs++;
            prldmaTemp = prldmaTemp->LDMA_prldmaNext;
         }  while (prldmaTemp != prldmaTempRoot);
      }
      else {
         if (pBuf_Index <= pBuf_End)   // room in buffer?
            ((RTASKINFO *)pTaskInfoBuf)->TSKINFO_phDMA = NULL;
      }
      if (pBuf_Index <= pBuf_End)      // room in buffer?
         {                             /* copy NumDMAs and phGPC into buffer*/
         ((RTASKINFO *)pTaskInfoBuf)->TSKINFO_usNumDMA = usNumDMAs;
         ((RTASKINFO *)pTaskInfoBuf)->TSKINFO_phGPC = (PHGPC)((BYTE *)
            pRealBuffer+(pBuf_Index-(PBYTE)pTaskInfoBuf));
      }

      /**********************************************************************/
      /*  Find hGPC's: they are on a task basis                             */
      /**********************************************************************/

      if (hTask->TSK_prgpcTail != NULL) {/* there are hGPCs                 */
         prgpcTemp = hTask->TSK_prgpcTail->GPC_prgpcNext;
         prgpcTempRoot = prgpcTemp;
         do {                          /* copy the hGPCs into the buffer    */
            if (prgpcTemp->GPC_achValid != GPCVALIDATE)
               return (DSP_INTERNAL_CORRUPT);
            if ((pBuf_Index+sizeof(HGPC)) <= pBuf_End) // room in buffer?
               *(PHGPC)pBuf_Index = prgpcTemp;
            pBuf_Index += sizeof(PRGPC);/* inc pointer as a PRGPC           */
            usNumGPCs++;
            prgpcTemp = prgpcTemp->GPC_prgpcNext;
         }  while (prgpcTemp != prgpcTempRoot);
      }                                /* end outer if                      */
      else {
         if (pBuf_Index <= pBuf_End)   // room in buffer?
            ((RTASKINFO *)pTaskInfoBuf)->TSKINFO_phGPC = NULL;
      }
      if (pBuf_Index <= pBuf_End)      // room in buffer?
         {                             /* copy NumGPCs and phITCB into      */
                                       /* buffer                            */
         ((RTASKINFO *)pTaskInfoBuf)->TSKINFO_usNumGPC = usNumGPCs;
         ((RTASKINFO *)pTaskInfoBuf)->TSKINFO_phITCB = (PHITCB)((BYTE *)
            pRealBuffer+(pBuf_Index-(PBYTE)pTaskInfoBuf));
      }

     /***********************************************************************/
      /*  Find hITCB's: CH03 they are on a Task Basis now.                  */
     /***********************************************************************/

      if (hTask->TSK_pritcbTail != NULL) {/* there are hITCBs               */
         pritcbTemp = hTask->TSK_pritcbTail->ITCB_pritcbNext;
         pritcbTempRoot = pritcbTemp;
         do {                          /* copy hITCBs into buffer           */
            if (pritcbTemp->ITCB_achValid != ITCBVALIDATE)
               return (DSP_INTERNAL_CORRUPT);
            if ((pBuf_Index+sizeof(HITCB)) <= pBuf_End) // room in buffer?
               *(PHITCB)pBuf_Index = pritcbTemp;
            pBuf_Index += sizeof(PRITCB);/* inc pointer as a PRITCB         */
            usNumITCBs++;
            pritcbTemp = pritcbTemp->ITCB_pritcbNext;
         }  while (pritcbTemp != pritcbTempRoot);
      }                                /* end outer if                      */

      else {
         if (pBuf_Index <= pBuf_End)   // room in buffer?
            ((RTASKINFO *)pTaskInfoBuf)->TSKINFO_phITCB = NULL;
      }
      if (pBuf_Index <= pBuf_End)      // room in buffer?
         {                             /* copy NumITCBs and pszInstanceName */
                                       /* into buffer                       */
         ((RTASKINFO *)pTaskInfoBuf)->TSKINFO_usNumITCB = usNumITCBs;
         ((RTASKINFO *)pTaskInfoBuf)->TSKINFO_pszInstanceName = (PSZ)((BYTE *)
            pRealBuffer+(pBuf_Index-(PBYTE)pTaskInfoBuf));
         pstring = (PSZ)pBuf_Index;    /* set pointer to string name        */
      }

      if (hTask->TSK_pszVirtName != NULL)
         pBuf_Index += (strlen(hTask->TSK_pszVirtName)+1);
      else
         pBuf_Index++;

      if (pBuf_Index <= pBuf_End)      // room in buffer?
         {                             /* there is room in the buffer       */
         if (hTask->TSK_pszVirtName != NULL)
            strcpy(pstring, hTask->TSK_pszVirtName);
         else
            *pstring = 0;

         ((RTASKINFO *)pTaskInfoBuf)->TSKINFO_hDSP =
            hTask->TSK_prdsp->DSP_hDSP;
         ((RTASKINFO *)pTaskInfoBuf)->TSKINFO_rver = hTask->TSK_rver;
         ((RTASKINFO *)pTaskInfoBuf)->TSKINFO_ulCPF = hTask->TSK_ulCPF;
         ((RTASKINFO *)pTaskInfoBuf)->TSKINFO_usFlgType = hTask->TSK_usflgType
            ;
         ((RTASKINFO *)pTaskInfoBuf)->TSKINFO_usFlgState =
            hTask->TSK_usflgState;
         ((RTASKINFO *)pTaskInfoBuf)->TSKINFO_hFM = hTask->TSK_prfm;
         ((RTASKINFO *)pTaskInfoBuf)->TSKINFO_hIPC = hTask->TSK_pripcte;
      }
      else {                           /* there is not enough room in the   */
                                       /* Buffer                            */
         *pusSize = (USHORT)(pBuf_Index-(PBYTE)pTaskInfoBuf);
         ulRC = DSP_INS_BUFFER;
      }
   }
   else
      ulRC = DSP_INV_HANDLE;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspquery::QueryTaskInfo exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/* SUBROUTINE NAME: GetDefaultDSPHandle                                     */
/*                                                                          */
/* FUNCTION:                                                                */
/*                                                                          */
/* INPUT:                                                                   */
/*                                                                          */
/* OUTPUT:                                                                  */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

HDSP FAR PASCAL GetDefaultDSPHandle(VOID)
{
   HMGR       hMgr = NULL;

   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspquery::GetDefaultDSPHandle entry\n");
   hMgr = pg->prmgrTail;
   return ((HDSP)hMgr->MGR_prdspTail);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: NameToTaskHandle                                        */
/*                                                                          */
/* FUNCTION: This routine will search the tasks in a module until a         */
/*           match is found for the Task Name.                              */
/*                                                                          */
/*                                                                          */
/* INPUT:                                                                   */
/*        HMOD  hMod        - Handle of Module to containing task           */
/*        PSZ   pszVirtName - Name of Task to search for                    */
/*        HTASK  hTask       - Handle of Task                               */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC NameToTaskHandle(HMOD hMod,PSZ pszVirtName,HTASK *phTask)

{
   RC         ulRC = DSP_NOERROR;      // Error return code
   PRTSK      prtskNode;               // general ptr to a Task Node.
   PRMOD      prmod;                   // ptr to the MOD
   BOOL       bFound;                  // If found, then TRUE
   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspquery::NameToTaskHandle entry hMod %x pszVirtName %s\n",(int)hMod,pszVirtName);

   /*************************************************************************/
   /* Verify that there is a Mod Handle.                                    */
   /* Verify that there are tasks to search.                                */
   /*************************************************************************/

   if (hMod == NULL)
      return (DSP_INV_HANDLE);
   else
      prmod = (PRMOD)hMod;

   if (prmod->MOD_achValid != MODVALIDATE)
      return (DSP_INV_HANDLE);
   if (prmod->MOD_prtskTail == NULL)
      return (DSP_NAME_NOT_FOUND);

   /*************************************************************************/
   /* Search for a Task that matches the Name input until                   */
   /* a match is found or the entire list has been checked.                 */
   /* Start at Tail->Next and go thru Tail.                                 */
   /* If a match is found, bFound will be set to TRUE and                   */
   /* the loop will end with prtskNode pointing to the                      */
   /* correct Task                                                          */
   /*************************************************************************/

   prtskNode = prmod->MOD_prtskTail;
   bFound = FALSE;

   do {
      prtskNode = prtskNode->TSK_prtskNext;

      if (prtskNode->TSK_pszVirtName != NULL) {
         if (mwstrcmp(pszVirtName, prtskNode->TSK_pszVirtName) == 0)
            bFound = TRUE;
      }
   }  while ((bFound == FALSE) && (prtskNode != prmod->MOD_prtskTail));/*   */
                                       /* enddo                             */
   if (bFound != TRUE) {
      return (DSP_NAME_NOT_FOUND);
   }

   *phTask = (HTASK)prtskNode;

   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspquery::NameToTaskHandle exit ulRC %lx hTask %x\n",ulRC,(int)*phTask);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: NameToSegmentHandle                                     */
/*                                                                          */
/* FUNCTION: This routine will search the tasks in a module until a         */
/*           match is found for the Segment Name.                           */
/*                                                                          */
/* INPUT:                                                                   */
/*        HTASK  hTask       - Handle of Task that contains segment         */
/*        PSZ   pszName     - Name of Segment to search for                 */
/*        HSEG  hSeg        - Handle of Segment                             */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC NameToSegmentHandle(HTASK hTask,PSZ pszName,HSEG *phSeg)

{
   RC         ulRC = DSP_NOERROR;      // Error return code
   PRSEG      prsegNode=0;             // general ptr to a Segment Node.
   PRTSK      prtsk;                   // ptr to the Task.
   BOOL       bFound;                  // If found, then TRUE

   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspquery::NameToSegmentHandle hTask %x pszName %s\n",(int)hTask,pszName);
   /*************************************************************************/
   /* Verify that there is a Tsk Handle.                                    */
   /* Verify that there are Segments to search.  Note that                  */
   /* a task can have code and data segments.                               */
   /*************************************************************************/

   if (hTask == NULL)
      return (DSP_INV_HANDLE);
   else
      prtsk = (PRTSK)hTask;

   if (prtsk->TSK_achValid != TSKVALIDATE)
      return (DSP_INV_HANDLE);

   /*************************************************************************/
   /* Search for a Seg  that matches the Name input until                   */
   /* a match is found or the entire list has been checked.                 */
   /* Start at Tail->Next and go thru Tail.                                 */
   /*                                                                       */
   /* If a match is found, bFound will be set to TRUE and                   */
   /* the loop will end with prsegNode pointing to the                      */
   /* correct Segment.                                                      */
   /*************************************************************************/
   /* Search data segments.                                                 */
   /*************************************************************************/

   bFound = FALSE;

   if (prtsk->TSK_prsegDTail != NULL) {

      prsegNode = prtsk->TSK_prsegDTail;
      do {
         prsegNode = prsegNode->SEG_prsegNext;

         if (prsegNode->SEG_pszVirtName != NULL) {
            if (mwstrcmp(pszName, prsegNode->SEG_pszVirtName) == 0)
               bFound = TRUE;
         }
     }  while ((bFound == FALSE) && (prsegNode != prtsk->TSK_prsegDTail));/**/
                                       /* enddo                             */
   }

   if ((bFound != TRUE) && (prtsk->TSK_prsegITail != NULL)) {

      /**********************************************************************/
      /* Search Instruction Segments.                                       */
      /**********************************************************************/

      prsegNode = prtsk->TSK_prsegITail;

      do {
         prsegNode = prsegNode->SEG_prsegNext;

         if (prsegNode->SEG_pszVirtName != NULL) {
            if (mwstrcmp(pszName, prsegNode->SEG_pszVirtName) == 0)
               bFound = TRUE;
         }
     }  while ((bFound == FALSE) && (prsegNode != prtsk->TSK_prsegITail));/**/
                                       /* enddo                             */
   }

   /*************************************************************************/
   /* If Match was found, return segment, otherwise error.                  */
   /*************************************************************************/

   if (bFound == TRUE)
      *phSeg = (HSEG)prsegNode;
   else {
      ulRC = DSP_NAME_NOT_FOUND;
      MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspquery::NameToSegmentHandle error ulRC %lx Segment: %s\n",ulRC, pszName);
   }

   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspquery::NameToSegmentHandle ulRC %lx hSeg %x\n",ulRC,(int)*phSeg);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: NameToModuleHandle                                      */
/*                                                                          */
/* FUNCTION: This routine will search the Modules in a Manager until a      */
/*           match is found for the Module Name.                            */
/*                                                                          */
/*                                                                          */
/* INPUT:                                                                   */
/*        HMGR  hMgr        - Handle of Manager to search                   */
/*        PSZ   pszVirtName - Name of Module to search f                    */
/*        HMOD  hMod        - Handle of Module                              */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC NameToModuleHandle(HMGR hMgr,PSZ pszVirtName,HMOD *phMod)

{
   RC         ulRC = DSP_NOERROR;      // Error return code
   PRMOD      prModNode;               // general ptr to a Module Node
   PRMGR      prmgr;                   // ptr to the Manager
   BOOL       bFound;                  // If found, then TRUE

   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspquery::NameToModuleHandle entry hMgr %x pszVirtName %s\n",(int)hMgr,pszVirtName);
   /*************************************************************************/
   /* Verify that there is a Mgr Handle.                                    */
   /* Verify that there are Modules to search.                              */
   /*************************************************************************/

   if (hMgr == NULL)
      return (DSP_INV_HANDLE);
   else
      prmgr = (PRMGR)hMgr;

   if (prmgr->MGR_achValid != MGRVALIDATE)
      return (DSP_INV_HANDLE);

   if (prmgr->MGR_prmodTail == NULL)
      return (DSP_NAME_NOT_FOUND);

   /*************************************************************************/
   /* Search for a Module that matches the Name input until                 */
   /* a match is found or the entire list has been checked.                 */
   /* Start at Tail->Next and go thru Tail.                                 */
   /* If a match is found, bFound will be set to TRUE and                   */
   /* the loop will end with prmodNode pointing to the                      */
   /* correct Task                                                          */
   /*************************************************************************/

   prModNode = prmgr->MGR_prmodTail;
   bFound = FALSE;

   do {
      prModNode = prModNode->MOD_prmodNext;
      if (prModNode->MOD_pszName != NULL) {
         if (mwstrcmp(pszVirtName, prModNode->MOD_pszName) == 0)
            bFound = TRUE;
      }
   }  while ((bFound == FALSE) && (prModNode != prmgr->MGR_prmodTail));/*   */
                                       /* enddo                             */
   if (bFound != TRUE) {
      ulRC = DSP_NAME_NOT_FOUND;
      MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspquery::NameToModuleHandle error ulRC %lx Module: %s",ulRC,pszVirtName);  
      return (ulRC);
   }

   *phMod = (HMOD)prModNode;
   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspquery::NameToModuleHandle exit ulRC %lx hMod %x\n",ulRC,(int)*phMod);
   return (ulRC);
}
