package merkletree2

import (
	"testing"

	"github.com/stretchr/testify/require"
)

func TestBlindedSHA512_256v1Encoder(t *testing.T) {

	tests := []struct {
		desc string
		enc  Encoder
	}{
		{"safe", NewBlindedSHA512_256v1Encoder()},
		{"non thread safe", NewUnsafeBlindedSHA512_256v1Encoder()},
	}

	for _, test := range tests {
		t.Run(test.desc, func(t *testing.T) {

			encoder := test.enc

			hashLength := 32

			val1 := "pizza"
			val2 := "pasta"

			enc0, err := encoder.Encode(val1)
			require.NoError(t, err)
			require.NotEmpty(t, enc0)

			dec := ""
			err = encoder.Decode(&dec, enc0)
			require.NoError(t, err)
			require.Equal(t, val1, dec)

			enc1, h, err := encoder.EncodeAndHashGeneric(val1)
			require.NoError(t, err)
			require.Len(t, h, hashLength)
			require.NotEmpty(t, enc1)
			require.Equal(t, enc0, enc1)

			var h2 Hash
			err = encoder.HashGeneric(val1, &h2)
			require.NoError(t, err)
			require.Equal(t, h, h2)

			enc2, h2, err := encoder.EncodeAndHashGeneric(val2)
			require.NoError(t, err)
			require.Len(t, h, hashLength)
			require.NotEqual(t, h, h2)
			require.NotEmpty(t, enc2)
			require.NotEqual(t, enc1, enc2)

			ms, err := encoder.GenerateMasterSecret(0)
			require.NoError(t, err)
			require.Len(t, ms, hashLength)

			ms2, err := encoder.GenerateMasterSecret(0)
			require.NoError(t, err)
			require.Len(t, ms2, hashLength)
			require.NotEqual(t, ms, ms2)

			ks := encoder.ComputeKeySpecificSecret(ms, Key([]byte{0x00, 0x01}))
			require.Len(t, ks, hashLength)
			ks2 := encoder.ComputeKeySpecificSecret(ms, Key([]byte{0x00, 0x01}))
			require.Len(t, ks, hashLength)
			require.Equal(t, ks, ks2)
			ks3 := encoder.ComputeKeySpecificSecret(ms, Key([]byte{0x00, 0x02}))
			require.Len(t, ks3, hashLength)
			require.NotEqual(t, ks, ks3)

			var ks4 KeySpecificSecret
			encoder.ComputeKeySpecificSecretTo(ms, Key([]byte{0x00, 0x01}), &ks4)
			require.Equal(t, ks, ks4)

			h0, err := encoder.HashKeyEncodedValuePairWithKeySpecificSecret(KeyEncodedValuePair{Key: Key([]byte{0x00, 0x01}), Value: enc0}, ks)
			require.NoError(t, err)
			require.Len(t, h, hashLength)

			err = encoder.HashKeyEncodedValuePairWithKeySpecificSecretTo(KeyEncodedValuePair{Key: Key([]byte{0x00, 0x01}), Value: enc0}, ks, &h)
			require.NoError(t, err)
			require.Equal(t, h0, h)

			h, err = encoder.HashKeyValuePairWithKeySpecificSecret(KeyValuePair{Key: Key([]byte{0x00, 0x01}), Value: val1}, ks)
			require.NoError(t, err)
			require.Len(t, h, hashLength)
			require.Equal(t, h0, h)

			h2, err = encoder.HashKeyValuePairWithKeySpecificSecret(KeyValuePair{Key: Key([]byte{0x00, 0x01}), Value: val1}, ks)
			require.NoError(t, err)
			require.Len(t, h2, hashLength)
			require.Equal(t, h, h2)

			h3, err := encoder.HashKeyValuePairWithKeySpecificSecret(KeyValuePair{Key: Key([]byte{0x00, 0x01}), Value: val1}, ks3)
			require.NoError(t, err)
			require.Len(t, h3, hashLength)
			require.NotEqual(t, h, h3)

			h4, err := encoder.HashKeyValuePairWithKeySpecificSecret(KeyValuePair{Key: Key([]byte{0x00, 0x02}), Value: val1}, ks)
			require.NoError(t, err)
			require.Len(t, h4, hashLength)
			require.NotEqual(t, h, h3)
		})
	}
}

func TestBlindedSHA512_256v1EncoderHardcodedValues(t *testing.T) {
	// To prevent changes to go-codec or similar from breaking existing merkle
	// proofs, this test hardcodes some example values for the encoder and
	// hasher.

	tests := []struct {
		desc string
		enc  Encoder
	}{
		{"safe", NewBlindedSHA512_256v1Encoder()},
		{"non thread safe", NewUnsafeBlindedSHA512_256v1Encoder()},
	}

	for _, test := range tests {
		t.Run(test.desc, func(t *testing.T) {

			encoder := test.enc

			hashLength := 32

			hashb1 := make([]byte, hashLength)
			hashb1[0] = 0x01
			hash1 := Hash(hashb1)

			hashb2 := make([]byte, hashLength)
			hashb2[0] = 0x02
			hash2 := Hash(hashb2)

			enc, err := encoder.Encode(Node{INodes: []Hash{hash1}})
			require.NoError(t, err)
			require.Equal(t, []byte{0x1, 0x91, 0xc4, 0x20, 0x1, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0}, enc)

			enc, err = encoder.Encode(Node{INodes: []Hash{hash1, hash2}})
			require.NoError(t, err)
			require.Equal(t, []byte{0x1, 0x92, 0xc4, 0x20, 0x1, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xc4, 0x20, 0x2, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0}, enc)

			enc, err = encoder.Encode(Node{LeafHashes: []KeyHashPair{{Key: []byte{0x01}, Hash: hash1}}})
			require.NoError(t, err)
			require.Equal(t, []byte{0x2, 0x91, 0x92, 0xc4, 0x1, 0x1, 0xc4, 0x20, 0x1, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0}, enc)

			enc, err = encoder.Encode(Node{LeafHashes: []KeyHashPair{{Key: []byte{0x01}, Hash: hash1}, {Key: []byte{0x03}, Hash: hash2}}})
			require.NoError(t, err)
			require.Equal(t, []byte{0x2, 0x92, 0x92, 0xc4, 0x1, 0x1, 0xc4, 0x20, 0x1, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x92, 0xc4, 0x1, 0x3, 0xc4, 0x20, 0x2, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0}, enc)

			_, hash, err := encoder.EncodeAndHashGeneric(Node{INodes: []Hash{hash1}})
			require.NoError(t, err)
			require.EqualValues(t, []byte{0x1d, 0x66, 0xcd, 0x98, 0xa1, 0x36, 0x3a, 0x96, 0x8c, 0xdf, 0x5f, 0xff, 0xb1, 0x8d, 0xff, 0xfa, 0xcf, 0xdb, 0xfe, 0x6, 0xc7, 0xbd, 0xe0, 0xe5, 0x4d, 0x58, 0x5d, 0x79, 0xd, 0x57, 0x50, 0x6d}, hash)

			_, hash, err = encoder.EncodeAndHashGeneric(Node{INodes: []Hash{hash1, hash2}})
			require.NoError(t, err)
			require.EqualValues(t, []byte{0x85, 0xd5, 0x4f, 0xf7, 0x4c, 0x11, 0x6f, 0x86, 0x46, 0x63, 0x25, 0xf9, 0xc8, 0x69, 0xfb, 0xb8, 0x26, 0x71, 0x50, 0x4f, 0x7d, 0xe9, 0x73, 0x9f, 0x2e, 0x66, 0x76, 0xa5, 0x1e, 0x1f, 0x16, 0x55}, hash)

			_, hash, err = encoder.EncodeAndHashGeneric(Node{LeafHashes: []KeyHashPair{{Key: []byte{0x01}, Hash: hash1}}})
			require.NoError(t, err)
			require.EqualValues(t, []byte{0x59, 0xfb, 0x86, 0x3b, 0x3a, 0x83, 0xdd, 0x92, 0x22, 0xa3, 0x85, 0x4a, 0x8a, 0x9d, 0x18, 0xb, 0x31, 0x4a, 0x9c, 0xd6, 0xb, 0x5e, 0x20, 0x79, 0xa9, 0xb8, 0xae, 0x41, 0xc9, 0xa1, 0xd8, 0x5d}, hash)

			_, hash, err = encoder.EncodeAndHashGeneric(Node{LeafHashes: []KeyHashPair{{Key: []byte{0x01}, Hash: hash1}, {Key: []byte{0x03}, Hash: hash2}}})
			require.NoError(t, err)
			require.EqualValues(t, []byte{0xf5, 0x19, 0xdb, 0x70, 0x5c, 0x4f, 0x84, 0x3f, 0xcc, 0x30, 0x62, 0x45, 0x7e, 0xfb, 0x91, 0x31, 0x59, 0x9, 0xca, 0x81, 0xc0, 0x78, 0x13, 0x82, 0xcf, 0x96, 0x37, 0xfd, 0xa7, 0xf3, 0x35, 0x81}, hash)

			ms := MasterSecret([]byte{0x00, 0x01})

			ks := encoder.ComputeKeySpecificSecret(ms, Key([]byte{0x00, 0x03}))
			require.Len(t, ks, hashLength)
			require.EqualValues(t, []byte{0xa1, 0x87, 0x3f, 0x2b, 0x2b, 0x1c, 0x76, 0xfe, 0x41, 0x64, 0x83, 0xef, 0x4d, 0xb5, 0x3f, 0x5d, 0xa2, 0x73, 0x84, 0x4d, 0x8d, 0x7e, 0x47, 0xc5, 0xd3, 0x53, 0x7e, 0xc4, 0x3, 0x2d, 0x56, 0xd5}, ks)

			value := "pasta"
			encValue, err := encoder.Encode(value)
			require.NoError(t, err)
			require.Equal(t, []byte{0xa5, 0x70, 0x61, 0x73, 0x74, 0x61}, encValue)

			h, err := encoder.HashKeyEncodedValuePairWithKeySpecificSecret(KeyEncodedValuePair{Key: Key([]byte{0x00, 0x01}), Value: encValue}, ks)
			require.NoError(t, err)
			require.EqualValues(t, []byte{0xc, 0x43, 0x55, 0x10, 0x67, 0xc3, 0x0, 0x13, 0x70, 0xfc, 0xab, 0x9e, 0xdd, 0x33, 0x70, 0x9a, 0x65, 0xb9, 0x8c, 0x34, 0xcc, 0xf0, 0x5, 0x23, 0x1d, 0x39, 0x2c, 0x0, 0x5d, 0xd4, 0xe9, 0x1d}, h)

			h, err = encoder.HashKeyValuePairWithKeySpecificSecret(KeyValuePair{Key: Key([]byte{0x00, 0x01}), Value: value}, ks)
			require.NoError(t, err)
			require.EqualValues(t, []byte{0xc, 0x43, 0x55, 0x10, 0x67, 0xc3, 0x0, 0x13, 0x70, 0xfc, 0xab, 0x9e, 0xdd, 0x33, 0x70, 0x9a, 0x65, 0xb9, 0x8c, 0x34, 0xcc, 0xf0, 0x5, 0x23, 0x1d, 0x39, 0x2c, 0x0, 0x5d, 0xd4, 0xe9, 0x1d}, h)
		})
	}
}
