/*
  File: main.c

  Copyright (C) 2007
  Marcin 'Morgoth' Kurek <morgoth6@box43.pl>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>

#include <sys/stat.h>

#include "bootcreator.h"

/*
 * Version
 */

#ifdef __MORPHOS__
static const char MVersion[] = "\0$VER: " VERSION_STRING;
#endif /* __MORPHOS__ */

/*
 * Private
 */

static void print_help(void);
static void print_version(void);
static void print_config(void);

static int parse_args(int argc, char *argv[]);

/*
 * Args
 */

struct Args args;

/*
 * Main
 */

int main(int argc, char *argv[])
{
	int res = RETURN_OK;
	int args_res = ARGS_MISSING;

	/* Parse Args */
	if(argc >= 2)
		args_res = parse_args(argc, argv);

	if(args_res == ARGS_OK)
	{
		/* Args OK */
		VERBOSE(printf("source file %s\ndestination file %s\n", args.source_file, args.destination_file));

		/* BootCreator */
		res = bootcreator();

		/* End */
		VERBOSE(printf("result %d\n", res));
	}
	else
	{
		switch (args_res)
		{
			case ARGS_VERSION:
			{
				/* Print program version */
				print_version();

				/* End */
				break;
			}
			case ARGS_HELP:
			{
				/* Print program help */
				print_help();

				/* End */
				break;
			}
			case ARGS_CONFIG:
			{
				/* Print some internals */
				print_config();

				/* End */
				break;
			}
			case ARGS_MISSING:
			{
				/* Required argument missing */
				printf("required argument missing\n");

				res = RETURN_FAILURE;

				/* End */
				break;
			}
			default:
			{
				/* Error parsing command line */
				printf("error parsing command line\n");

				res = RETURN_FAILURE;

				/* End */
				break;
			}
		}
	}

	/* End */
	return (res);
}

/*
 * Private
 */

static void print_help(void)
{
	printf("usage: " PROGRAM_NAME " [optopns] <source file> <destination file>\n" "-v, --verbose     be verbose.\n" "-V, --version     show version.\n" "-h, --help        show this usage message.\n" "-c, --config      show some internals.\n");

	/* End */
	return;
}

static void print_version(void)
{
	printf("%s\n", VERSION_STRING);

	/* End */
	return;
}

static void print_config(void)
{
	printf("program:\n" " args buffer size %d bytes\n" " line buffer size %d bytes\n" " default output file permisions 0%o\n" " parser comment character %c\n" "menu:\n" " default version %d\n" " default timeout %d\n" " default abort on key %s\n" " default entry %d\n" " default title %s\n", ARGS_BUF_SIZE, FILE_BUF_SIZE, DEF_PERMS, PARSE_COMMENT, DEF_VERSION, DEF_TIMEOUT, DEF_ABORTOK ? "true" : "false", DEF_DEFAULT, DEF_TITLE);

	/* End */
	return;
}

static int parse_args(int argc, char *argv[])
{
	int res = -1;
	unsigned int counter;

	for(counter = 1; counter < argc; ++counter)
	{
		/* Option */
		if(*argv[counter] == '-')
		{
			char *tmp = argv[counter] + 1;

			if(*tmp == '-')
			{
				/* Long format */
				if(strcmp(tmp, "-version") == 0)
				{
					/* Version */
					return (ARGS_VERSION);
				}

				if(strcmp(tmp, "-verbose") == 0)
				{
					/* Verbose */
					args.verbose = true;

					continue;
				}

				if(strcmp(tmp, "-help") == 0)
				{
					/* Help */
					return (ARGS_HELP);
				}

				if(strcmp(tmp, "-config") == 0)
				{
					/* Print some internals */
					return (ARGS_CONFIG);
				}
			}
			else
			{
				/* Short format */
				if(*tmp == 'V' && (*(tmp + 1) == '\0'))
				{
					/* Version */
					return (ARGS_VERSION);
				}

				if(*tmp == 'v' && (*(tmp + 1) == '\0'))
				{
					/* Verbose */
					args.verbose = true;

					continue;
				}

				if(*tmp == 'h' && (*(tmp + 1) == '\0'))
				{
					/* Help */
					return (ARGS_HELP);
				}

				if(*tmp == 'c' && (*(tmp + 1) == '\0'))
				{
					/* Config */
					return (ARGS_CONFIG);
				}
			}
		}

		if((counter >= (argc - 2)) && (*argv[counter] != '-'))	/* total arguments - required arguments [src, dst] */
		{
			/* Source - Destination */
			if(counter == (argc - 2))
				strcpy(args.source_file, argv[counter]);
			else
				strcpy(args.destination_file, argv[counter]);
		}
		else
		{
			/* Unknown argument */
			printf("%s: unknown argument\n", argv[counter]);

			res = 0;

			break;
		}
	}

	/* Sanity */
	if(res == -1)
	{
		if(strlen(args.source_file) && strlen(args.destination_file))
			res = ARGS_OK;
		else
			res = ARGS_MISSING;
	}

	/* End */
	return (res);
}
