// Copyright 2020-2025 Consensys Software Inc.
// Licensed under the Apache License, Version 2.0. See the LICENSE file for details.

// Code generated by consensys/gnark-crypto DO NOT EDIT

package fft

import (
	"math/big"
	"math/bits"

	"github.com/consensys/gnark-crypto/ecc"
	"github.com/consensys/gnark-crypto/internal/parallel"

	"github.com/consensys/gnark-crypto/field/babybear"
	fext "github.com/consensys/gnark-crypto/field/babybear/extensions"
)

// FFTExt computes the discrete Fourier transform of a slice of extension field elements.
// Coefficients and evaluations are extension field elements.
// The root of unity domain is the same as FFT.
func (domain *Domain) FFTExt(a []fext.E4, decimation Decimation, opts ...Option) {

	opt := fftOptions(opts)

	// find the stage where we should stop spawning go routines in our recursive calls
	// (ie when we have as many go routines running as we have available CPUs)
	maxSplits := bits.TrailingZeros64(ecc.NextPowerOfTwo(uint64(opt.nbTasks)))
	if opt.nbTasks == 1 {
		maxSplits = -1
	}

	// if coset != 0, scale by coset table
	if opt.coset {

		if decimation == DIT {

			// scale by coset table (in bit reversed order)
			cosetTable := domain.cosetTable
			if !domain.withPrecompute {
				// we need to build the full table or do a bit reverse dance.
				cosetTable = make([]babybear.Element, len(a))
				BuildExpTable(domain.FrMultiplicativeGen, cosetTable)
			}
			parallel.Execute(len(a), func(start, end int) {
				n := uint64(len(a))
				nn := uint64(64 - bits.TrailingZeros64(n))
				for i := start; i < end; i++ {
					irev := int(bits.Reverse64(uint64(i)) >> nn)
					a[i].MulByElement(&a[i], &cosetTable[irev])
				}
			}, opt.nbTasks)
		} else {

			if domain.withPrecompute {
				parallel.Execute(len(a), func(start, end int) {
					for i := start; i < end; i++ {
						a[i].MulByElement(&a[i], &domain.cosetTable[i])
					}
				}, opt.nbTasks)
			} else {
				c := domain.FrMultiplicativeGen
				parallel.Execute(len(a), func(start, end int) {
					var at babybear.Element
					at.Exp(c, big.NewInt(int64(start)))
					for i := start; i < end; i++ {
						a[i].MulByElement(&a[i], &at)
						at.Mul(&at, &c)
					}
				}, opt.nbTasks)
			}

		}
	}

	twiddles := domain.twiddles
	twiddlesStartStage := 0
	if !domain.withPrecompute {
		twiddlesStartStage = 3
		nbStages := int(bits.TrailingZeros64(domain.Cardinality))
		if nbStages-twiddlesStartStage > 0 {
			twiddles = make([][]babybear.Element, nbStages-twiddlesStartStage)
			w := domain.Generator
			w.Exp(w, big.NewInt(int64(1<<twiddlesStartStage)))
			buildTwiddles(twiddles, w, uint64(nbStages-twiddlesStartStage))
		} // else, we don't need twiddles
	}

	switch decimation {
	case DIF:
		difFFTExt(a, domain.Generator, twiddles, twiddlesStartStage, 0, maxSplits, nil, opt.nbTasks)
	case DIT:

		ditFFTExt(a, domain.Generator, twiddles, twiddlesStartStage, 0, maxSplits, nil, opt.nbTasks)
	default:
		panic("not implemented")
	}
}

// FFTInverseExt computes (recursively) the inverse discrete Fourier transform of a and stores the result in a
// if decimation == DIT (decimation in time), the input must be in bit-reversed order
// if decimation == DIF (decimation in frequency), the output will be in bit-reversed order
// coset sets the shift of the fft (0 = no shift, standard fft)
// len(a) must be a power of 2, and w must be a len(a)th root of unity in field F.
func (domain *Domain) FFTInverseExt(a []fext.E4, decimation Decimation, opts ...Option) {
	opt := fftOptions(opts)

	// find the stage where we should stop spawning go routines in our recursive calls
	// (ie when we have as many go routines running as we have available CPUs)
	maxSplits := bits.TrailingZeros64(ecc.NextPowerOfTwo(uint64(opt.nbTasks)))
	if opt.nbTasks == 1 {
		maxSplits = -1
	}

	twiddlesInv := domain.twiddlesInv
	twiddlesStartStage := 0
	if !domain.withPrecompute {
		twiddlesStartStage = 3
		nbStages := int(bits.TrailingZeros64(domain.Cardinality))
		if nbStages-twiddlesStartStage > 0 {
			twiddlesInv = make([][]babybear.Element, nbStages-twiddlesStartStage)
			w := domain.GeneratorInv
			w.Exp(w, big.NewInt(int64(1<<twiddlesStartStage)))
			buildTwiddles(twiddlesInv, w, uint64(nbStages-twiddlesStartStage))
		} // else, we don't need twiddles
	}

	switch decimation {
	case DIF:
		difFFTExt(a, domain.GeneratorInv, twiddlesInv, twiddlesStartStage, 0, maxSplits, nil, opt.nbTasks)
	case DIT:
		ditFFTExt(a, domain.GeneratorInv, twiddlesInv, twiddlesStartStage, 0, maxSplits, nil, opt.nbTasks)
	default:
		panic("not implemented")
	}

	// scale by CardinalityInv
	if !opt.coset {
		parallel.Execute(len(a), func(start, end int) {
			for i := start; i < end; i++ {
				a[i].MulByElement(&a[i], &domain.CardinalityInv)
			}
		}, opt.nbTasks)
		return
	}

	if decimation == DIT {
		if domain.withPrecompute {
			if opt.nbTasks == 1 {
				for i := 0; i < len(a); i++ {
					a[i].MulByElement(&a[i], &domain.cosetTableInv[i])
					a[i].MulByElement(&a[i], &domain.CardinalityInv)
				}
			} else {
				parallel.Execute(len(a), func(start, end int) {
					for i := start; i < end; i++ {
						a[i].MulByElement(&a[i], &domain.cosetTableInv[i]).
							MulByElement(&a[i], &domain.CardinalityInv)
					}
				}, opt.nbTasks)
			}
		} else {
			c := domain.FrMultiplicativeGenInv
			parallel.Execute(len(a), func(start, end int) {
				var at babybear.Element
				at.Exp(c, big.NewInt(int64(start)))
				at.Mul(&at, &domain.CardinalityInv)
				for i := start; i < end; i++ {
					a[i].MulByElement(&a[i], &at)
					at.Mul(&at, &c)
				}
			}, opt.nbTasks)
		}
		return
	}

	// decimation == DIF, need to access coset table in bit reversed order.
	cosetTableInv := domain.cosetTableInv
	if !domain.withPrecompute {
		// we need to build the full table or do a bit reverse dance.
		cosetTableInv = make([]babybear.Element, len(a))
		BuildExpTable(domain.FrMultiplicativeGenInv, cosetTableInv)
	}
	parallel.Execute(len(a), func(start, end int) {
		n := uint64(len(a))
		nn := uint64(64 - bits.TrailingZeros64(n))
		for i := start; i < end; i++ {
			irev := int(bits.Reverse64(uint64(i)) >> nn)
			a[i].MulByElement(&a[i], &cosetTableInv[irev]).
				MulByElement(&a[i], &domain.CardinalityInv)
		}
	}, opt.nbTasks)

}

func difFFTExt(a []fext.E4, w babybear.Element, twiddles [][]babybear.Element, twiddlesStartStage, stage, maxSplits int, chDone chan struct{}, nbTasks int) {
	if chDone != nil {
		defer close(chDone)
	}

	n := len(a)
	if n == 1 {
		return
	} else if stage >= twiddlesStartStage {
		if n == 1<<8 {
			kerDIFNP_256Ext(a, twiddles, stage-twiddlesStartStage)
			return
		}
	}
	m := n >> 1

	parallelButterfly := (m > butterflyThreshold) && (stage < maxSplits)

	if stage < twiddlesStartStage {
		if parallelButterfly {
			w := w
			parallel.Execute(m, func(start, end int) {
				if start == 0 {
					fext.Butterfly(&a[0], &a[m])
					start++
				}
				var at babybear.Element
				at.Exp(w, big.NewInt(int64(start)))
				innerDIFWithoutTwiddlesExt(a, at, w, start, end, m)
			}, nbTasks/(1<<(stage))) // 1 << stage == estimated used CPUs
		} else {
			innerDIFWithoutTwiddlesExt(a, w, w, 0, m, m)
		}
		// compute next twiddle
		w.Square(&w)
	} else {
		innerDIFWithTwiddlesExt(a, twiddles[stage-twiddlesStartStage], 0, m, m)
	}

	if m == 1 {
		return
	}

	nextStage := stage + 1
	if stage < maxSplits {
		chDone := make(chan struct{}, 1)
		go difFFTExt(a[m:n], w, twiddles, twiddlesStartStage, nextStage, maxSplits, chDone, nbTasks)
		difFFTExt(a[0:m], w, twiddles, twiddlesStartStage, nextStage, maxSplits, nil, nbTasks)
		<-chDone
	} else {
		difFFTExt(a[0:m], w, twiddles, twiddlesStartStage, nextStage, maxSplits, nil, nbTasks)
		difFFTExt(a[m:n], w, twiddles, twiddlesStartStage, nextStage, maxSplits, nil, nbTasks)
	}

}

func innerDIFWithTwiddlesGenericExt(a []fext.E4, twiddles []babybear.Element, start, end, m int) {
	if start == 0 {
		fext.Butterfly(&a[0], &a[m])
		start++
	}
	for i := start; i < end; i++ {
		fext.Butterfly(&a[i], &a[i+m])
		a[i+m].MulByElement(&a[i+m], &twiddles[i])
	}
}

func innerDIFWithoutTwiddlesExt(a []fext.E4, at, w babybear.Element, start, end, m int) {
	if start == 0 {
		fext.Butterfly(&a[0], &a[m])
		start++
	}
	for i := start; i < end; i++ {
		fext.Butterfly(&a[i], &a[i+m])
		a[i+m].MulByElement(&a[i+m], &at)
		at.Mul(&at, &w)
	}
}

func ditFFTExt(a []fext.E4, w babybear.Element, twiddles [][]babybear.Element, twiddlesStartStage, stage, maxSplits int, chDone chan struct{}, nbTasks int) {
	if chDone != nil {
		defer close(chDone)
	}
	n := len(a)
	if n == 1 {
		return
	} else if stage >= twiddlesStartStage {
		if n == 1<<8 {
			kerDITNP_256Ext(a, twiddles, stage-twiddlesStartStage)
			return
		}
	}

	m := n >> 1

	nextStage := stage + 1
	nextW := w
	nextW.Square(&nextW)

	if stage < maxSplits {
		// that's the only time we fire go routines
		chDone := make(chan struct{}, 1)

		go ditFFTExt(a[m:], nextW, twiddles, twiddlesStartStage, nextStage, maxSplits, chDone, nbTasks)
		ditFFTExt(a[0:m], nextW, twiddles, twiddlesStartStage, nextStage, maxSplits, nil, nbTasks)
		<-chDone
	} else {

		ditFFTExt(a[0:m], nextW, twiddles, twiddlesStartStage, nextStage, maxSplits, nil, nbTasks)
		ditFFTExt(a[m:n], nextW, twiddles, twiddlesStartStage, nextStage, maxSplits, nil, nbTasks)
	}

	parallelButterfly := (m > butterflyThreshold) && (stage < maxSplits)

	if stage < twiddlesStartStage {
		// we need to compute the twiddles for this stage on the fly.
		if parallelButterfly {
			w := w
			parallel.Execute(m, func(start, end int) {
				if start == 0 {
					fext.Butterfly(&a[0], &a[m])
					start++
				}
				var at babybear.Element
				at.Exp(w, big.NewInt(int64(start)))
				innerDITWithoutTwiddlesExt(a, at, w, start, end, m)
			}, nbTasks/(1<<(stage))) // 1 << stage == estimated used CPUs

		} else {
			innerDITWithoutTwiddlesExt(a, w, w, 0, m, m)
		}
		return
	}
	innerDITWithTwiddlesExt(a, twiddles[stage-twiddlesStartStage], 0, m, m)
}

func innerDITWithTwiddlesGenericExt(a []fext.E4, twiddles []babybear.Element, start, end, m int) {
	if start == 0 {
		fext.Butterfly(&a[0], &a[m])
		start++
	}
	for i := start; i < end; i++ {
		a[i+m].MulByElement(&a[i+m], &twiddles[i])
		fext.Butterfly(&a[i], &a[i+m])
	}
}

func innerDITWithoutTwiddlesExt(a []fext.E4, at, w babybear.Element, start, end, m int) {
	if start == 0 {
		fext.Butterfly(&a[0], &a[m])
		start++
	}
	for i := start; i < end; i++ {
		a[i+m].MulByElement(&a[i+m], &at)
		fext.Butterfly(&a[i], &a[i+m])
		at.Mul(&at, &w)
	}
}

func kerDIFNP_256genericExt(a []fext.E4, twiddles [][]babybear.Element, stage int) {
	// code unrolled & generated by internal/generator/fft/template/fftext.go.tmpl

	innerDIFWithTwiddlesGenericExt(a[:256], twiddles[stage+0], 0, 128, 128)
	for offset := 0; offset < 256; offset += 128 {
		innerDIFWithTwiddlesGenericExt(a[offset:offset+128], twiddles[stage+1], 0, 64, 64)
	}
	for offset := 0; offset < 256; offset += 64 {
		innerDIFWithTwiddlesGenericExt(a[offset:offset+64], twiddles[stage+2], 0, 32, 32)
	}
	for offset := 0; offset < 256; offset += 32 {
		innerDIFWithTwiddlesGenericExt(a[offset:offset+32], twiddles[stage+3], 0, 16, 16)
	}
	for offset := 0; offset < 256; offset += 16 {
		innerDIFWithTwiddlesGenericExt(a[offset:offset+16], twiddles[stage+4], 0, 8, 8)
	}
	for offset := 0; offset < 256; offset += 8 {
		innerDIFWithTwiddlesGenericExt(a[offset:offset+8], twiddles[stage+5], 0, 4, 4)
	}
	for offset := 0; offset < 256; offset += 4 {
		innerDIFWithTwiddlesGenericExt(a[offset:offset+4], twiddles[stage+6], 0, 2, 2)
	}
	for offset := 0; offset < 256; offset += 2 {
		fext.Butterfly(&a[offset], &a[offset+1])
	}
}

func kerDITNP_256genericExt(a []fext.E4, twiddles [][]babybear.Element, stage int) {
	// code unrolled & generated by internal/generator/fft/template/fftext.go.tmpl

	for offset := 0; offset < 256; offset += 2 {
		fext.Butterfly(&a[offset], &a[offset+1])
	}
	for offset := 0; offset < 256; offset += 4 {
		innerDITWithTwiddlesGenericExt(a[offset:offset+4], twiddles[stage+6], 0, 2, 2)
	}
	for offset := 0; offset < 256; offset += 8 {
		innerDITWithTwiddlesGenericExt(a[offset:offset+8], twiddles[stage+5], 0, 4, 4)
	}
	for offset := 0; offset < 256; offset += 16 {
		innerDITWithTwiddlesGenericExt(a[offset:offset+16], twiddles[stage+4], 0, 8, 8)
	}
	for offset := 0; offset < 256; offset += 32 {
		innerDITWithTwiddlesGenericExt(a[offset:offset+32], twiddles[stage+3], 0, 16, 16)
	}
	for offset := 0; offset < 256; offset += 64 {
		innerDITWithTwiddlesGenericExt(a[offset:offset+64], twiddles[stage+2], 0, 32, 32)
	}
	for offset := 0; offset < 256; offset += 128 {
		innerDITWithTwiddlesGenericExt(a[offset:offset+128], twiddles[stage+1], 0, 64, 64)
	}
	innerDITWithTwiddlesGenericExt(a[:256], twiddles[stage+0], 0, 128, 128)
}
