/*****************************************************************************
 * Project: RooFit                                                           *
 * Package: RooFitCore                                                       *
 * @(#)root/roofitcore:$Id$
 * Authors:                                                                  *
 *   WV, Wouter Verkerke, UC Santa Barbara, verkerke@slac.stanford.edu       *
 *   DK, David Kirkby,    UC Irvine,         dkirkby@uci.edu                 *
 *                                                                           *
 * Copyright (c) 2000-2005, Regents of the University of California          *
 *                          and Stanford University. All rights reserved.    *
 *                                                                           *
 * Redistribution and use in source and binary forms,                        *
 * with or without modification, are permitted according to the terms        *
 * listed in LICENSE (http://roofit.sourceforge.net/license.txt)             *
 *****************************************************************************/

//////////////////////////////////////////////////////////////////////////////
/// \class RooArgList
/// RooArgList is a container object that can hold multiple RooAbsArg objects.
/// The container has list semantics which means that:
///
///  - Contained objects are ordered, The iterator
///    follows the object insertion order.
///
///  - Objects can be retrieved by name and index
///
///  - Multiple objects with the same name are allowed
///
/// Ownership of contents.
///
/// Unowned objects are inserted with the add() method. Owned objects
/// are added with addOwned() or addClone(). A RooArgSet either owns all
/// of it contents, or none, which is determined by the first `<add>`
/// call. Once an ownership status is selected, inappropriate `<add>` calls
/// will return error status. Clearing the list via removeAll() resets the
/// ownership status. Arguments supplied in the constructor are always added
/// as unowned elements.
///
///

#include "RooArgList.h"

#include "RooStreamParser.h"
#include "RooAbsCategoryLValue.h"
#include "RooTrace.h"
#include "RooMsgService.h"
#include "RooConstVar.h"

#include <stdexcept>

using std::endl, std::istream, std::ostream;



////////////////////////////////////////////////////////////////////////////////
/// Default constructor

RooArgList::RooArgList()
{
  TRACE_CREATE;
}



////////////////////////////////////////////////////////////////////////////////
/// Constructor from another RooAbsCollection.

RooArgList::RooArgList(const RooAbsCollection& coll) :
  RooAbsCollection(coll.GetName())
{
  add(coll) ;
  TRACE_CREATE;
}



////////////////////////////////////////////////////////////////////////////////
/// Empty list constructor

RooArgList::RooArgList(const char *name) :
  RooAbsCollection(name)
{
  TRACE_CREATE;
}


////////////////////////////////////////////////////////////////////////////////
/// Constructor from a root TCollection. Elements in the collection that
/// do not inherit from RooAbsArg will be skipped. A warning message
/// will be printed for every skipped item.

RooArgList::RooArgList(const TCollection& tcoll, const char* name) :
  RooAbsCollection(name)
{
  for(TObject * obj : tcoll) {
    if (!dynamic_cast<RooAbsArg*>(obj)) {
      coutW(InputArguments) << "RooArgList::RooArgList(TCollection) element " << obj->GetName()
             << " is not a RooAbsArg, ignored" << std::endl ;
      continue ;
    }
    add(*static_cast<RooAbsArg*>(obj)) ;
  }
  TRACE_CREATE;
}



////////////////////////////////////////////////////////////////////////////////
/// Copy constructor. Note that a copy of a list is always non-owning,
/// even the source list is owning. To create an owning copy of
/// a list (owning or not), use the snapshot() method.

RooArgList::RooArgList(const RooArgList& other, const char *name)
  : RooAbsCollection(other,name)
{
  TRACE_CREATE;
}



////////////////////////////////////////////////////////////////////////////////
/// Destructor

RooArgList::~RooArgList()
{
  TRACE_DESTROY;
}


////////////////////////////////////////////////////////////////////////////////
/// Write the contents of the argset in ASCII form to given stream.
///
/// All elements will be printed on a single line separated by a single
/// white space. The contents of each element is written by the arguments'
/// writeToStream() function

void RooArgList::writeToStream(ostream& os, bool compact)
{
  if (!compact) {
    coutE(InputArguments) << "RooArgList::writeToStream(" << GetName() << ") non-compact mode not supported" << std::endl ;
    return ;
  }

  for (const auto obj : _list) {
    obj->writeToStream(os,true);
    os << " " ;
  }
  os << std::endl ;
}



////////////////////////////////////////////////////////////////////////////////
/// Read the contents of the argset in ASCII form from given stream.
///
/// A single line is read, and all elements are assumed to be separated
/// by white space. The value of each argument is read by the arguments
/// readFromStream function.

bool RooArgList::readFromStream(istream& is, bool compact, bool verbose)
{
  if (!compact) {
    coutE(InputArguments) << "RooArgList::readFromStream(" << GetName() << ") non-compact mode not supported" << std::endl ;
    return true ;
  }

  RooStreamParser parser(is) ;
  for (auto next : _list) {
    if (!next->getAttribute("Dynamic")) {
      if (next->readFromStream(is,true,verbose)) {
        parser.zapToEnd() ;

        return true ;
      }
    } else {
    }
  }

  if (!parser.atEOL()) {
    TString rest = parser.readLine() ;
    if (verbose) {
      coutW(InputArguments) << "RooArgSet::readFromStream(" << GetName()
             << "): ignoring extra characters at end of line: '" << rest << "'" << std::endl ;
    }
  }

  return false ;
}


void RooArgList::processArg(double value) { processArg(RooFit::RooConst(value)); }
