/*
 * xfce4-hdaps, an XFCE4 panel plugin for the HDAPS system.
 *
 * Copyright (C) 2019 Michael Orlitzky
 *
 *   http://michael.orlitzky.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details:
 *
 *   https://www.gnu.org/licenses/agpl-3.0.html
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <gtk/gtk.h>
#include <libxfce4ui/libxfce4ui.h>
#include <libxfce4panel/libxfce4panel.h>
#include "xfce4-hdaps-dialogs.h"
#include "hdaps.h"

/* Wild guess. Actually borrowed from some code
   I copy-and-pasted. Doesn't seem too ugly. */
#define DEFAULT_BORDER_WIDTH 8

/* If people can't read the README, well maybe
 *  they can do it /online/.
 */
#define PLUGIN_WEBSITE "http://michael.orlitzky.com/code/xfce4-hdaps.xhtml"

/* Launch a URI using either gtk_show_uri_on_window (gvfs), exo-open
 * or xdg-open. This is used both when you click the author's email
 * address, and when you visit the plugin's homepage.
 */
static gboolean launch_uri(GtkWindow *parent, const gchar *uri) {
  GError *error = NULL;
  gboolean result = gtk_show_uri_on_window(parent,
                                           uri,
                                           GDK_CURRENT_TIME,
                                           &error);
  /* First, let's try what the GTK docs say the default is. */
  if (result == TRUE) {
    return result;
  }
  else {
    g_warning(_("Error launching URI %s: %s"), uri, error->message);
    g_error_free(error);
  }

  /* Ok, GTK failed, on to the fallbacks. Let's try exo-open first,
     since it's part of XFCE. */
  gchar *cmd = g_strdup_printf("%s %s","exo-open", uri);
  result = g_spawn_command_line_async(cmd, NULL);

  if (G_UNLIKELY(result == FALSE)) {
    g_warning(_("Unable to open URI with exo-open, trying xdg-open: %s"), uri);
    g_free(cmd);
    cmd = g_strdup_printf("%s %s","xdg-open", uri);
    result = g_spawn_command_line_async(cmd, NULL);
    if (G_UNLIKELY(result == FALSE)) {
      g_warning(_("Unable to open URI with xdg-open, giving up: %s"), uri);
    }
  }

  g_free(cmd);

  return result;
}


static void hdaps_configure_response(GtkDialog   *dialog,
                                     gint         response,
                                     HdapsPlugin *hdaps) {
  if (response == GTK_RESPONSE_HELP) {
    launch_uri(GTK_WINDOW(dialog), PLUGIN_WEBSITE);
    return;
  }


  if (response == GTK_RESPONSE_OK) {
    /* This corresponds to the "Save" button, so we
       want to save any settings that may have changed. */
    g_object_set_data(G_OBJECT(hdaps->plugin), "dialog", NULL);
    hdaps->poll_frequency = gtk_spin_button_get_value_as_int((GtkSpinButton*)hdaps->sb_poll_frequency);
    hdaps->device_name = gtk_combo_box_text_get_active_text((GtkComboBoxText*)hdaps->cb_device_name);
    snprintf(hdaps->sysfs_file,
             FILENAME_MAX,
             UNLOAD_HEADS_FMT,
             hdaps->device_name);
    hdaps_save(hdaps->plugin, hdaps);
    hdaps_reset_timeout(hdaps);
  }


  /* Since there is already a "save" button, we should ignore
     any changes that were made if the user presses "cancel"
     instead. */
  g_object_set_data(G_OBJECT(hdaps->plugin), "dialog", NULL);
  xfce_panel_plugin_unblock_menu(hdaps->plugin);
  gtk_widget_destroy(GTK_WIDGET(dialog));

  return;
}



void hdaps_configure(XfcePanelPlugin *plugin,
                     HdapsPlugin     *hdaps) {

  /* Here comes a bunch of GTK garbage to create the
     settings dialog. */

  GtkWidget *dialog;
  GtkWidget *label;
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkSizeGroup *sg;

  /* Block the plugin menu while the configuration
     dialogue is open. Don't forget to unblock it before
     we close. */
  xfce_panel_plugin_block_menu(plugin);

  /* Create the dialog */
  dialog = xfce_titled_dialog_new_with_buttons(
             _("Hdaps Plugin"),
             GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(plugin))),
             GTK_DIALOG_DESTROY_WITH_PARENT,
             "gtk-help", GTK_RESPONSE_HELP,
             "gtk-save", GTK_RESPONSE_OK,
             "gtk-cancel", GTK_RESPONSE_CANCEL,
             NULL);

  /* Center the dialog on screen. */
  gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);

  /* Set dialog icon to a generic settings icon. */
  gtk_window_set_icon_name(GTK_WINDOW(dialog), "xfce4-settings");

  /* Link the dialog to the plugin, so we can destroy it in case
   * the plugin is closed while the dialog is still open. */
  g_object_set_data(G_OBJECT(plugin), "dialog", dialog);

  /* Connect the reponse signal to the dialog. */
  g_signal_connect(G_OBJECT (dialog), "response",
                   G_CALLBACK(hdaps_configure_response), hdaps);


  vbox = gtk_box_new(GTK_ORIENTATION_VERTICAL, DEFAULT_BORDER_WIDTH);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), DEFAULT_BORDER_WIDTH - 2);
  gtk_box_pack_start(GTK_BOX(gtk_dialog_get_content_area(GTK_DIALOG(dialog))),
                     vbox, TRUE, TRUE, 0);

  /* Create size group to keep widgets aligned */
  sg = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);

  hbox = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, DEFAULT_BORDER_WIDTH);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  label = gtk_label_new(_("Poll Frequency:"));
  gtk_size_group_add_widget(sg, label);
  gtk_widget_set_halign(label, 0.0);
  gtk_widget_set_valign(label, 0.5);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

  hdaps->sb_poll_frequency = gtk_spin_button_new_with_range(100, 5000, 100);
  gtk_spin_button_set_value((GtkSpinButton*)hdaps->sb_poll_frequency,
                            hdaps->poll_frequency);

  gtk_box_pack_start(GTK_BOX(hbox),
                     hdaps->sb_poll_frequency,
                     FALSE,
                     FALSE,
                     0);

  /* Create the device name dropdown. */
  hbox = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, DEFAULT_BORDER_WIDTH);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  label = gtk_label_new(_("Device Name:"));
  gtk_size_group_add_widget(sg, label);
  gtk_widget_set_halign(label, 0.0);
  gtk_widget_set_valign(label, 0.5);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

  hdaps->cb_device_name = gtk_combo_box_text_new();

  /* Add the current device name, it should always be available.
     Oh, and it should be the default, too. */
  gtk_combo_box_text_append_text((GtkComboBoxText*)hdaps->cb_device_name,
                                 hdaps->device_name);

  /* This function takes an index, and 0 should be the first
     (only) index at this point. */
  gtk_combo_box_set_active((GtkComboBox*)hdaps->cb_device_name, 0);

  /* Now loop through the list of available devices, adding each
     to the list as we go. */
  char hdaps_devices[MAX_HDAPS_DEVICES][FILENAME_MAX];
  int found_devices = get_hdaps_device_list(hdaps_devices);
  int list_idx = 0;

  for (list_idx = 0; list_idx < found_devices; list_idx++) {
    /* We don't want to add duplicate entries to the combo box.
       However, at this point, the current device name should be
       the only entry. Therefore, to avoid duplicates, we only
       have to avoid adding the current device name a second time. */
    if (strcmp(hdaps_devices[list_idx], hdaps->device_name) != 0) {
      gtk_combo_box_text_append_text((GtkComboBoxText*)hdaps->cb_device_name,
                                     hdaps_devices[list_idx]);
    }
  }

  gtk_box_pack_start(GTK_BOX(hbox), hdaps->cb_device_name, FALSE, FALSE, 0);

  /* Show the dialog and all of its widgets. */
  gtk_widget_show_all(dialog);
}




/* URI (http(s):// and mailto://) handler used on the About dialog. */
static void hdaps_uri_handler(GtkAboutDialog *about,
                              const gchar *uri,
                              gpointer data) {

  launch_uri(GTK_WINDOW(about), uri);
}

/* "Close" button handler for the About dialog. */
static void hdaps_close_handler (GtkAboutDialog *about,
                                 gint response_id,
                                 gpointer user_data) {
  /* This will cause the dialog to be destroyed */
  gtk_widget_destroy(GTK_WIDGET(about));
}

void hdaps_about(XfcePanelPlugin *plugin) {
  const gchar *authors[] = { "Michael Orlitzky <michael@orlitzky.com>", NULL };
  const gchar *copyright = "Copyright \302\251 2019 Michael Orlitzky";

  GtkAboutDialog *about = GTK_ABOUT_DIALOG(gtk_about_dialog_new());

  gtk_about_dialog_set_authors(about, authors);
  gtk_about_dialog_set_copyright(about, copyright);
  gtk_about_dialog_set_program_name(about, PACKAGE_NAME);
  gtk_about_dialog_set_logo_icon_name(about, PACKAGE_NAME);
  gtk_about_dialog_set_license(about, LICENSE_AGPL3);
  gtk_about_dialog_set_version(about, PACKAGE_VERSION);
  gtk_about_dialog_set_website(about, PLUGIN_WEBSITE);
  gtk_about_dialog_set_website_label(about, "xfce4-hdaps homepage");

  /* Use a custom URI handler for http(s):// and mailto:// URIs. The
     default behavior apparently needs gio/gvfs to work, so we want
     to add some fallbacks. */
  g_signal_connect(about,
                   "activate-link",
                   G_CALLBACK(hdaps_uri_handler),
                   NULL);

  /* To close the dialog when "close" is clicked, we have to connect
   * the response signal to a handler that actually does it.
   */
  g_signal_connect(about,
                   "response",
                   G_CALLBACK(hdaps_close_handler),
                   NULL);
  gtk_widget_show(GTK_WIDGET(about));
  return;
}
