/*                                                     -*- linux-c -*-
    Copyright (C) 2004 Tom Szilagyi

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: file_io.c,v 1.4 2004/06/12 13:57:24 tszilagyi Exp $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#include "common.h"
#include "file_io.h"


extern REVTYPE * root;


/* 0: between two REVTYPE records
 * 1: read REVTYPE keyword, name follows on next line
 * 2: read name, before COMBS keyword
 * 3: read COMBS keyword, lines for COMBS follow
 * 4: read ALLPS keyword, lines for ALLPS follow
 * 5: read BANDPS_LO keyword, next line contains BANDPS_LO data
 * 6: read BANDPS_HI keyword, next line contains BANDPS_HI data
 */
int parser_state = 0;


float convf(char * s) {

	float val, pow;
	int i, sign;

	for (i = 0; s[i] == ' ' || s[i] == '\n' || s[i] == '\t'; i++);
	sign = 1;
	if (s[i] == '+' || s[i] == '-')
		sign = (s[i++] == '+') ? 1 : -1;
	for (val = 0; s[i] >= '0' && s[i] <= '9'; i++)
		val = 10 * val + s[i] - '0';
	if ((s[i] == '.') || (s[i] == ','))
		i++;
	for (pow = 1; s[i] >= '0' && s[i] <= '9'; i++) {
		val = 10 * val + s[i] - '0';
		pow *= 10;
	}
	return(sign * val / pow);
}



REVTYPE *
parse_input_file(void) {

	char * home;
	char path[MAXLEN];
	FILE * infile;
	char line[MAXLEN];
	int c, i = 0;
	char str1[MAXLEN];
	char str2[MAXLEN];
	char str3[MAXLEN];
	int num_combs = 0;
	int num_allps = 0;
	REVTYPE * root = NULL;
	REVTYPE * item = NULL;
	REVTYPE * prev = NULL;

	
	if (!(home = getenv("HOME")))
		home = ".";
	sprintf(path, "%s/%s", home, NAME_REVERBED);
	if ((infile = fopen(path,"rt")) == NULL) {
		fprintf(stderr, "Unable to open %s\n", path);
		return(NULL);
	}

	if ((root = malloc(sizeof(REVTYPE))) == NULL) {
		fprintf(stderr, "file_io.c: couldn't alloc mem for root item\n");
		return(NULL);
	}
	root->next = NULL;
	root->name[0] = '\0';

	/* Here we read the whole file and fill up our list. */

	while ((c = fgetc(infile)) != EOF) {
		if (c != '\n')
			line[i++] = c;
		else {
			line[i] = '\0';
			i = 0;
			
			switch(parser_state) {

			case 0:
				if (strcmp(line, "REVTYPE") == 0) {
					if ((item = malloc(sizeof(REVTYPE))) == NULL) {
						fprintf(stderr, "file_io.c: malloc failed.\n");
						return(NULL);
					}
					item->next = NULL;

					if (root->next == NULL)
						root->next = item;
					else
						prev->next = item;
					prev = item;

					parser_state = 1;
					num_allps = 0;
				}
				break;

			case 1:
				strcpy(item->name, line);
				parser_state = 2;
				break;

			case 2:
				if (strcmp(line, "COMBS") == 0) {
					num_combs = 0;
					parser_state = 3;
				}
				break;
			case 3:
				if (strcmp(line, "ALLPS") == 0) {
					num_allps = 0;
					item->num_combs = num_combs;
					parser_state = 4;
				} else {
					sscanf(line, "%s %s %s", str1, str2, str3);
					item->combs_data[3 * num_combs] = 1000.0f * convf(str1);
					item->combs_data[3 * num_combs + 1] = convf(str2);
					item->combs_data[3 * num_combs + 2] = convf(str3);
					num_combs++;
				}
				break;
			case 4:
				if (strcmp(line, "BANDPS_LO") == 0) {
					item->num_allps = num_allps;
					parser_state = 5;
				} else {
					sscanf(line, "%s %s", str1, str2);
					item->allps_data[2 * num_allps] = 1000.0f * convf(str1);
					item->allps_data[2 * num_allps + 1] = convf(str2);
					num_allps++;
				}
				break;
			case 5: 
				if (strcmp(line, "BANDPS_HI") == 0) {
					parser_state = 6;
				} else {
					item->num_allps = num_allps;
					sscanf(line, "%s", str1);
					item->bandps_lo = convf(str1);
				}
				break;
			case 6:
				sscanf(line, "%s", str1);
				item->bandps_hi = convf(str1);
				parser_state = 0;
				break;
			}

		}
	}
	fclose(infile);
	return(root);
}



REVTYPE *
get_revtype_by_name(REVTYPE * root, const char * name) {

	REVTYPE * item = root;

	if (item->next != NULL)
		item = item->next;
	else
		return NULL;

	while (item != NULL) {
		if (strcmp(name, item->name) == 0)
			return item;
		item = item->next;
	}
	return NULL;
}



int
save_revtypes(const char * path) {

	FILE * outfile;
	char fullname[MAXLEN];
	char str[MAXLEN];
	REVTYPE * item = root;
	int i, j;

	sprintf(fullname, "%s", path);

	if ((outfile = fopen(fullname,"wt")) == NULL) {
		fprintf(stderr,"Error: unable to create file %s\n", fullname);
		return 1;
	}

	fprintf(outfile,"# DO NOT EDIT BY HAND!\n# This file is created and "
		"used by TAP Reverb Editor.\n\n");
	

	if (item->next != NULL)
		item = item->next;
	else
		return 1;

	while (item != NULL) {
		fprintf(outfile, "REVTYPE\n%s\n", item->name);
		fprintf(outfile, "COMBS\n");
		for (i = 0; i < item->num_combs; i++) {
			sprintf(str, "%.4ff %.2ff %.4ff",
				item->combs_data[3 * i] / 1000.0f,
				item->combs_data[3 * i + 1],
				item->combs_data[3 * i + 2]);
			for (j = 0; str[j] != '\0'; j++)
				if (str[j] == ',')
					str[j] = '.';
			fprintf(outfile, "%s\n", str);
		}
		fprintf(outfile, "ALLPS\n");
		for (i = 0; i < item->num_allps; i++) {
			sprintf(str, "%.4ff %.2ff",
				item->allps_data[2 * i] / 1000.0f,
				item->allps_data[2 * i + 1]);
			for (j = 0; str[j] != '\0'; j++)
				if (str[j] == ',')
					str[j] = '.';
			fprintf(outfile, "%s\n", str);
		}
		sprintf(str, "BANDPS_LO\n%.1ff", item->bandps_lo);
		for (j = 0; str[j] != '\0'; j++)
			if (str[j] == ',')
				str[j] = '.';
		fprintf(outfile, "%s\n", str);
		sprintf(str, "BANDPS_HI\n%.1ff\n", item->bandps_hi);
		for (j = 0; str[j] != '\0'; j++)
			if (str[j] == ',')
				str[j] = '.';
		fprintf(outfile, "%s\n", str);

		item = item->next;
	}

	fclose(outfile);
	return 0;
}


int
write_tap_reverb_presets_h(const char * path) {


	FILE * outfile;
	char fullname[MAXLEN];
	char n1[MAXLEN];
	char n2[MAXLEN];
	char n3[MAXLEN];
	char str[MAXLEN];
	REVTYPE * item = root;
	int num_types = 0;
	int i, j;

        if (item->next != NULL)
                item = item->next;
        else
                return 1;
	
        while (item != NULL) {
		num_types++;
		item = item->next;
	}

	item = root->next;


	sprintf(fullname, "%s", path);

	if ((outfile = fopen(fullname,"wt")) == NULL) {
		fprintf(stderr,"Error: unable to create file %s\n", fullname);
		return 1;
	}

	fprintf(outfile,
		"/*  \n"
		"    Copyright (C) 2004 Tom Szilagyi\n\n"

		"    This file is the output of TAP Reverb Editor, and is part of TAP Reverberator.\n"
		"    Please use TAP Reverb Editor to re-generate this file, rather than editing by hand.\n"
		"    Visit http://tap-plugins.sf.net for more info.\n\n\n"

		"    This program is free software; you can redistribute it and/or modify\n"
		"    it under the terms of the GNU General Public License as published by\n"
		"    the Free Software Foundation; either version 2 of the License, or\n"
		"    (at your option) any later version.\n\n"

		"    This program is distributed in the hope that it will be useful,\n"
		"    but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
		"    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
		"    GNU General Public License for more details.\n\n"

		"    You should have received a copy of the GNU General Public License\n"
		"    along with this program; if not, write to the Free Software\n"
		"    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n"
		"*/\n\n");
	
	fprintf(outfile,
		"#include \"tap_reverb.h\"\n\n\n"
		"/* Number of reverb presets */\n"
		"#define NUM_MODES %d\n\n\n", num_types);

	fprintf(outfile,
		"/* Reverb Type data */\n"
		"REVERB_DATA reverb_data[NUM_MODES] = {\n"
		);



	while (item != NULL) {
		fprintf(outfile, 
			"        {\n"
			"                %ld,\n"
			"                %ld,\n"
			"                {\n", item->num_combs, item->num_allps);

		for (i = 0; i < item->num_combs; i++) {
			sprintf(n1, "%.4ff", item->combs_data[3 * i] / 1000.0f);
			for (j = 0; n1[j] != '\0'; j++)
				if (n1[j] == ',')
					n1[j] = '.';
			sprintf(n2, "%.2ff", item->combs_data[3 * i + 1]);
			for (j = 0; n2[j] != '\0'; j++)
				if (n2[j] == ',')
					n2[j] = '.';
			sprintf(n3, "%.4ff", item->combs_data[3 * i + 2]);
			for (j = 0; n3[j] != '\0'; j++)
				if (n3[j] == ',')
					n3[j] = '.';
			sprintf(str, "                        {%s, %s, %s},", n1, n2, n3);
			fprintf(outfile, "%s\n", str);
		}
		fprintf(outfile,
			"                },\n"
			"                {\n");
		for (i = 0; i < item->num_allps; i++) {
			sprintf(n1, "%.4ff", item->allps_data[2 * i] / 1000.0f);
			for (j = 0; n1[j] != '\0'; j++)
				if (n1[j] == ',')
					n1[j] = '.';
			sprintf(n2, "%.2ff", item->allps_data[2 * i + 1]);
			for (j = 0; n2[j] != '\0'; j++)
				if (n2[j] == ',')
					n2[j] = '.';
			sprintf(str, "                        {%s, %s},", n1, n2);
			fprintf(outfile, "%s\n", str);
		}
		fprintf(outfile, "                },\n");

		sprintf(n1, "%.1ff", item->bandps_lo);
		for (j = 0; n1[j] != '\0'; j++)
			if (n1[j] == ',')
				n1[j] = '.';
		sprintf(str, "                %s,", n1);
		fprintf(outfile, "%s\n", str);

		sprintf(n1, "%.1ff", item->bandps_hi);
		for (j = 0; n1[j] != '\0'; j++)
			if (n1[j] == ',')
				n1[j] = '.';
		sprintf(str, "                %s,", n1);
		fprintf(outfile, "%s\n", str);

		fprintf(outfile, "        },\n");

		item = item->next;
	}

	fprintf(outfile, "};\n\n");

	fclose(outfile);
	return 0;
}


int
write_tap_reverb_rdf(const char * path) {

	FILE * outfile;
	char fullname[MAXLEN];
	REVTYPE * item = root;
	int i = 0;

	sprintf(fullname, "%s", path);

	if ((outfile = fopen(fullname,"wt")) == NULL) {
		fprintf(stderr,"Error: unable to create file %s\n", fullname);
		return 1;
	}

	fprintf(outfile,
		"<?xml version='1.0' encoding='ISO-8859-1'?>\n"
		"<!DOCTYPE rdf:RDF [\n"
		"        <!ENTITY rdf 'http://www.w3.org/1999/02/22-rdf-syntax-ns#'>\n"
		"        <!ENTITY rdfs 'http://www.w3.org/2000/01/rdf-schema#'>\n"
		"        <!ENTITY dc 'http://purl.org/dc/elements/1.1/'>\n"
		"        <!ENTITY ladspa 'http://ladspa.org/ontology#'>\n"
		"]>\n"
		"<rdf:RDF xmlns:rdf=\"&rdf;\"\n"
		"         xmlns:rdfs=\"&rdfs;\"\n"
		"         xmlns:dc=\"&dc;\"\n"
		"         xmlns:ladspa=\"&ladspa;\">\n\n");

	fprintf(outfile,
		"  <ladspa:ReverbPlugin rdf:about=\"&ladspa;2142\">\n"
		"    <dc:creator>Tom Szilagyi</dc:creator>\n"
		"    <dc:title>TAP Reverberator</dc:title>\n"
		"    <ladspa:hasPort>\n"
		"      <ladspa:InputControlPort rdf:about=\"&ladspa;2142.0\" ladspa:hasLabel=\"decay\" />\n"
		"    </ladspa:hasPort>\n"
		"    <ladspa:hasPort>\n"
		"      <ladspa:InputControlPort rdf:about=\"&ladspa;2142.1\" ladspa:hasLabel=\"drylevel\" />\n"
		"    </ladspa:hasPort>\n"
		"    <ladspa:hasPort>\n"
		"      <ladspa:InputControlPort rdf:about=\"&ladspa;2142.2\" ladspa:hasLabel=\"wetlevel\" />\n"
		"    </ladspa:hasPort>\n"
		"    <ladspa:hasPort>\n"
		"      <ladspa:InputControlPort rdf:about=\"&ladspa;2142.3\" ladspa:hasLabel=\"combs_en\" />\n"
		"    </ladspa:hasPort>\n"
		"    <ladspa:hasPort>\n"
		"      <ladspa:InputControlPort rdf:about=\"&ladspa;2142.4\" ladspa:hasLabel=\"allps_en\" />\n"
		"    </ladspa:hasPort>\n"
		"    <ladspa:hasPort>\n"
		"      <ladspa:InputControlPort rdf:about=\"&ladspa;2142.5\" ladspa:hasLabel=\"bandpass_en\" />\n"
		"    </ladspa:hasPort>\n"
		"    <ladspa:hasPort>\n"
		"      <ladspa:InputControlPort rdf:about=\"&ladspa;2142.6\" ladspa:hasLabel=\"stereo_enh\" />\n"
		"    </ladspa:hasPort>\n"
		"    <ladspa:hasPort>\n"
		"      <ladspa:InputControlPort rdf:about=\"&ladspa;2142.7\" ladspa:hasLabel=\"type\">\n"
		"        <ladspa:hasScale>\n"
		"          <ladspa:Scale>\n");
	
	if (item->next != NULL)
		item = item->next;
	else
		return 1;

	while (item != NULL) {
		fprintf(outfile, "            <ladspa:hasPoint>\n");
		fprintf(outfile, "              <ladspa:Point rdf:value=\"%d\" "
			"ladspa:hasLabel=\"%s\" />\n", i, item->name);
		fprintf(outfile, "            </ladspa:hasPoint>\n");
		i++;
		item = item->next;
	}

	fprintf(outfile,
		"          </ladspa:Scale>\n"
		"        </ladspa:hasScale>\n"
		"      </ladspa:InputControlPort>\n"
		"    </ladspa:hasPort>\n"
		"    <ladspa:hasSetting>\n"
		"      <ladspa:Default>\n"
		"        <ladspa:hasPortValue>\n"
		"          <ladspa:PortValue rdf:value=\"2500\">\n"
		"            <ladspa:forPort rdf:resource=\"&ladspa;2142.0\" />\n"
		"          </ladspa:PortValue>\n"
		"        </ladspa:hasPortValue>\n"
		"        <ladspa:hasPortValue>\n"
		"          <ladspa:PortValue rdf:value=\"0\">\n"
		"            <ladspa:forPort rdf:resource=\"&ladspa;2142.1\" />\n"
		"          </ladspa:PortValue>\n"
		"        </ladspa:hasPortValue>\n"
		"        <ladspa:hasPortValue>\n"
		"          <ladspa:PortValue rdf:value=\"0\">\n"
		"            <ladspa:forPort rdf:resource=\"&ladspa;2142.2\" />\n"
		"          </ladspa:PortValue>\n"
		"        </ladspa:hasPortValue>\n"
		"        <ladspa:hasPortValue>\n"
		"          <ladspa:PortValue rdf:value=\"1\">\n"
		"            <ladspa:forPort rdf:resource=\"&ladspa;2142.3\" />\n"
		"          </ladspa:PortValue>\n"
		"        </ladspa:hasPortValue>\n"
		"        <ladspa:hasPortValue>\n"
		"          <ladspa:PortValue rdf:value=\"1\">\n"
		"            <ladspa:forPort rdf:resource=\"&ladspa;2142.4\" />\n"
		"          </ladspa:PortValue>\n"
		"        </ladspa:hasPortValue>\n"
		"        <ladspa:hasPortValue>\n"
		"          <ladspa:PortValue rdf:value=\"1\">\n"
		"            <ladspa:forPort rdf:resource=\"&ladspa;2142.5\" />\n"
		"          </ladspa:PortValue>\n"
		"        </ladspa:hasPortValue>\n"
		"        <ladspa:hasPortValue>\n"
		"          <ladspa:PortValue rdf:value=\"1\">\n"
		"            <ladspa:forPort rdf:resource=\"&ladspa;2142.6\" />\n"
		"          </ladspa:PortValue>\n"
		"        </ladspa:hasPortValue>\n"
		"        <ladspa:hasPortValue>\n"
		"          <ladspa:PortValue rdf:value=\"0\">\n"
		"            <ladspa:forPort rdf:resource=\"&ladspa;2142.7\" />\n"
		"          </ladspa:PortValue>\n"
		"        </ladspa:hasPortValue>\n"
		"      </ladspa:Default>\n"
		"    </ladspa:hasSetting>\n"
		"  </ladspa:ReverbPlugin>\n\n"
		"</rdf:RDF>\n");

	fclose(outfile);

	return 0;
}

