/* ************************************************************************
 * Copyright (C) 2016-2024 Advanced Micro Devices, Inc. All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * ************************************************************************ */
#define ROCBLAS_NO_DEPRECATED_WARNINGS
#include "hipblas.h"
#include "exceptions.hpp"
#include "limits.h"
#include "rocblas/rocblas.h"
#ifdef __HIP_PLATFORM_SOLVER__
#include "rocsolver/rocsolver.h"
#endif
#include <algorithm>
#include <functional>
#include <hip/library_types.h>
#include <math.h>

extern "C" hipblasStatus_t rocBLASStatusToHIPStatus(rocblas_status_ error);

// Attempt a rocBLAS call; if it gets an allocation error, query the
// size needed and attempt to allocate it, retrying the operation
static hipblasStatus_t hipblasDemandAlloc(rocblas_handle                   handle,
                                          std::function<hipblasStatus_t()> func)
{
    hipblasStatus_t status = func();
    if(status == HIPBLAS_STATUS_ALLOC_FAILED)
    {
        rocblas_status blas_status = rocblas_start_device_memory_size_query(handle);
        if(blas_status != rocblas_status_success)
            status = rocBLASStatusToHIPStatus(blas_status);
        else
        {
            status = func();
            if(status == HIPBLAS_STATUS_SUCCESS)
            {
                size_t size;
                blas_status = rocblas_stop_device_memory_size_query(handle, &size);
                if(blas_status != rocblas_status_success)
                    status = rocBLASStatusToHIPStatus(blas_status);
                else
                {
                    blas_status = rocblas_set_device_memory_size(handle, size);
                    if(blas_status != rocblas_status_success)
                        status = rocBLASStatusToHIPStatus(blas_status);
                    else
                        status = func();
                }
            }
        }
    }
    return status;
}

#define HIPBLAS_DEMAND_ALLOC(status__) \
    hipblasDemandAlloc(rocblas_handle(handle), [&]() -> hipblasStatus_t { return status__; })

extern "C" {

rocblas_operation_ hipOperationToHCCOperation(hipblasOperation_t op)
{
    switch(op)
    {
    case HIPBLAS_OP_N:
        return rocblas_operation_none;
    case HIPBLAS_OP_T:
        return rocblas_operation_transpose;
    case HIPBLAS_OP_C:
        return rocblas_operation_conjugate_transpose;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

hipblasOperation_t HCCOperationToHIPOperation(rocblas_operation_ op)
{
    switch(op)
    {
    case rocblas_operation_none:
        return HIPBLAS_OP_N;
    case rocblas_operation_transpose:
        return HIPBLAS_OP_T;
    case rocblas_operation_conjugate_transpose:
        return HIPBLAS_OP_C;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

rocblas_fill_ hipFillToHCCFill(hipblasFillMode_t fill)
{
    switch(fill)
    {
    case HIPBLAS_FILL_MODE_UPPER:
        return rocblas_fill_upper;
    case HIPBLAS_FILL_MODE_LOWER:
        return rocblas_fill_lower;
    case HIPBLAS_FILL_MODE_FULL:
        return rocblas_fill_full;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

hipblasFillMode_t HCCFillToHIPFill(rocblas_fill_ fill)
{
    switch(fill)
    {
    case rocblas_fill_upper:
        return HIPBLAS_FILL_MODE_UPPER;
    case rocblas_fill_lower:
        return HIPBLAS_FILL_MODE_LOWER;
    case rocblas_fill_full:
        return HIPBLAS_FILL_MODE_FULL;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

rocblas_diagonal_ hipDiagonalToHCCDiagonal(hipblasDiagType_t diagonal)
{
    switch(diagonal)
    {
    case HIPBLAS_DIAG_NON_UNIT:
        return rocblas_diagonal_non_unit;
    case HIPBLAS_DIAG_UNIT:
        return rocblas_diagonal_unit;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

hipblasDiagType_t HCCDiagonalToHIPDiagonal(rocblas_diagonal_ diagonal)
{
    switch(diagonal)
    {
    case rocblas_diagonal_non_unit:
        return HIPBLAS_DIAG_NON_UNIT;
    case rocblas_diagonal_unit:
        return HIPBLAS_DIAG_UNIT;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

rocblas_side_ hipSideToHCCSide(hipblasSideMode_t side)
{
    switch(side)
    {
    case HIPBLAS_SIDE_LEFT:
        return rocblas_side_left;
    case HIPBLAS_SIDE_RIGHT:
        return rocblas_side_right;
    case HIPBLAS_SIDE_BOTH:
        return rocblas_side_both;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

hipblasSideMode_t HCCSideToHIPSide(rocblas_side_ side)
{
    switch(side)
    {
    case rocblas_side_left:
        return HIPBLAS_SIDE_LEFT;
    case rocblas_side_right:
        return HIPBLAS_SIDE_RIGHT;
    case rocblas_side_both:
        return HIPBLAS_SIDE_BOTH;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

rocblas_pointer_mode HIPPointerModeToRocblasPointerMode(hipblasPointerMode_t mode)
{
    switch(mode)
    {
    case HIPBLAS_POINTER_MODE_HOST:
        return rocblas_pointer_mode_host;

    case HIPBLAS_POINTER_MODE_DEVICE:
        return rocblas_pointer_mode_device;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

hipblasPointerMode_t RocblasPointerModeToHIPPointerMode(rocblas_pointer_mode mode)
{
    switch(mode)
    {
    case rocblas_pointer_mode_host:
        return HIPBLAS_POINTER_MODE_HOST;

    case rocblas_pointer_mode_device:
        return HIPBLAS_POINTER_MODE_DEVICE;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

rocblas_datatype HIPDatatypeToRocblasDatatype_v2(hipDataType type)
{
    switch(type)
    {
    case HIP_R_16F:
        return rocblas_datatype_f16_r;

    case HIP_R_32F:
        return rocblas_datatype_f32_r;

    case HIP_R_64F:
        return rocblas_datatype_f64_r;

    case HIP_C_16F:
        return rocblas_datatype_f16_c;

    case HIP_C_32F:
        return rocblas_datatype_f32_c;

    case HIP_C_64F:
        return rocblas_datatype_f64_c;

    case HIP_R_8I:
        return rocblas_datatype_i8_r;

    case HIP_R_8U:
        return rocblas_datatype_u8_r;

    case HIP_R_32I:
        return rocblas_datatype_i32_r;

    case HIP_R_32U:
        return rocblas_datatype_u32_r;

    case HIP_C_8I:
        return rocblas_datatype_i8_c;

    case HIP_C_8U:
        return rocblas_datatype_u8_c;

    case HIP_C_32I:
        return rocblas_datatype_i32_c;

    case HIP_C_32U:
        return rocblas_datatype_u32_c;

    case HIP_R_16BF:
        return rocblas_datatype_bf16_r;

    case HIP_C_16BF:
        return rocblas_datatype_bf16_c;

    default:
        throw HIPBLAS_STATUS_INVALID_ENUM;
    }
}

rocblas_datatype HIPDatatypeToRocblasDatatype(hipblasDatatype_t type)
{
    switch(type)
    {
    case HIPBLAS_R_16F:
        return rocblas_datatype_f16_r;

    case HIPBLAS_R_32F:
        return rocblas_datatype_f32_r;

    case HIPBLAS_R_64F:
        return rocblas_datatype_f64_r;

    case HIPBLAS_C_16F:
        return rocblas_datatype_f16_c;

    case HIPBLAS_C_32F:
        return rocblas_datatype_f32_c;

    case HIPBLAS_C_64F:
        return rocblas_datatype_f64_c;

    case HIPBLAS_R_8I:
        return rocblas_datatype_i8_r;

    case HIPBLAS_R_8U:
        return rocblas_datatype_u8_r;

    case HIPBLAS_R_32I:
        return rocblas_datatype_i32_r;

    case HIPBLAS_R_32U:
        return rocblas_datatype_u32_r;

    case HIPBLAS_C_8I:
        return rocblas_datatype_i8_c;

    case HIPBLAS_C_8U:
        return rocblas_datatype_u8_c;

    case HIPBLAS_C_32I:
        return rocblas_datatype_i32_c;

    case HIPBLAS_C_32U:
        return rocblas_datatype_u32_c;

    case HIPBLAS_R_16B:
        return rocblas_datatype_bf16_r;

    case HIPBLAS_C_16B:
        return rocblas_datatype_bf16_c;
    default:
        throw HIPBLAS_STATUS_INVALID_ENUM;
    }
}

hipblasDatatype_t RocblasDatatypeToHIPDatatype(rocblas_datatype type)
{
    switch(type)
    {
    case rocblas_datatype_f16_r:
        return HIPBLAS_R_16F;

    case rocblas_datatype_f32_r:
        return HIPBLAS_R_32F;

    case rocblas_datatype_f64_r:
        return HIPBLAS_R_64F;

    case rocblas_datatype_i8_r:
        return HIPBLAS_R_8I;

    case rocblas_datatype_i32_r:
        return HIPBLAS_R_32I;

    case rocblas_datatype_f16_c:
        return HIPBLAS_C_16F;

    case rocblas_datatype_f32_c:
        return HIPBLAS_C_32F;

    case rocblas_datatype_f64_c:
        return HIPBLAS_C_64F;
    default:
        throw HIPBLAS_STATUS_INVALID_ENUM;
    }
}

rocblas_gemm_algo HIPGemmAlgoToRocblasGemmAlgo(hipblasGemmAlgo_t algo)
{
    switch(algo)
    {
    case HIPBLAS_GEMM_DEFAULT:
        return rocblas_gemm_algo_standard;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

hipblasGemmAlgo_t RocblasGemmAlgoToHIPGemmAlgo(rocblas_gemm_algo algo)
{
    switch(algo)
    {
    case rocblas_gemm_algo_standard:
        return HIPBLAS_GEMM_DEFAULT;
    default:
        throw HIPBLAS_STATUS_INVALID_ENUM;
    }
}

rocblas_gemm_flags HIPGemmFlagsToRocblasGemmFlags(hipblasGemmFlags_t flags)
{
    switch(flags)
    {
    case HIPBLAS_GEMM_FLAGS_NONE:
        return rocblas_gemm_flags_none;
    case HIPBLAS_GEMM_FLAGS_USE_CU_EFFICIENCY:
        return rocblas_gemm_flags_use_cu_efficiency;
    case HIPBLAS_GEMM_FLAGS_FP16_ALT_IMPL:
        return rocblas_gemm_flags_fp16_alt_impl;
    case HIPBLAS_GEMM_FLAGS_CHECK_SOLUTION_INDEX:
        return rocblas_gemm_flags_check_solution_index;
    case HIPBLAS_GEMM_FLAGS_FP16_ALT_IMPL_RNZ:
        return rocblas_gemm_flags_fp16_alt_impl_rnz;
    default:
        throw HIPBLAS_STATUS_INVALID_ENUM;
    }
}

hipblasGemmFlags_t RocblasGemmFlagsToHIPGemmFlags(rocblas_gemm_flags flags)
{
    switch(flags)
    {
    case rocblas_gemm_flags_none:
        return HIPBLAS_GEMM_FLAGS_NONE;
    case rocblas_gemm_flags_use_cu_efficiency:
        return HIPBLAS_GEMM_FLAGS_USE_CU_EFFICIENCY;
    case rocblas_gemm_flags_fp16_alt_impl:
        return HIPBLAS_GEMM_FLAGS_FP16_ALT_IMPL;
    case rocblas_gemm_flags_check_solution_index:
        return HIPBLAS_GEMM_FLAGS_CHECK_SOLUTION_INDEX;
    case rocblas_gemm_flags_fp16_alt_impl_rnz:
        return HIPBLAS_GEMM_FLAGS_FP16_ALT_IMPL_RNZ;
    default:
        throw HIPBLAS_STATUS_INVALID_ENUM;
    }
}

rocblas_atomics_mode HIPAtomicsModeToRocblasAtomicsMode(hipblasAtomicsMode_t mode)
{
    switch(mode)
    {
    case HIPBLAS_ATOMICS_NOT_ALLOWED:
        return rocblas_atomics_not_allowed;
    case HIPBLAS_ATOMICS_ALLOWED:
        return rocblas_atomics_allowed;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

hipblasAtomicsMode_t RocblasAtomicsModeToHIPAtomicsMode(rocblas_atomics_mode mode)
{
    switch(mode)
    {
    case rocblas_atomics_not_allowed:
        return HIPBLAS_ATOMICS_NOT_ALLOWED;
    case rocblas_atomics_allowed:
        return HIPBLAS_ATOMICS_ALLOWED;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

rocblas_math_mode hipblasMathModeTorocblasMathMode(hipblasMath_t mode)
{
    switch(mode)
    {
    case HIPBLAS_DEFAULT_MATH:
        return rocblas_default_math;
    case HIPBLAS_XF32_XDL_MATH:
        return rocblas_xf32_xdl_math_op;
    }
    throw HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasMath_t rocblasMathModeTohipblasMathMode(rocblas_math_mode mode)
{
    switch(mode)
    {
    case rocblas_default_math:
        return HIPBLAS_DEFAULT_MATH;
    case rocblas_xf32_xdl_math_op:
        return HIPBLAS_XF32_XDL_MATH;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

hipblasStatus_t rocBLASStatusToHIPStatus(rocblas_status_ error)
{
    switch(error)
    {
    case rocblas_status_size_unchanged:
    case rocblas_status_size_increased:
    case rocblas_status_success:
        return HIPBLAS_STATUS_SUCCESS;
    case rocblas_status_invalid_handle:
        return HIPBLAS_STATUS_NOT_INITIALIZED;
    case rocblas_status_not_implemented:
        return HIPBLAS_STATUS_NOT_SUPPORTED;
    case rocblas_status_invalid_pointer:
    case rocblas_status_invalid_size:
    case rocblas_status_invalid_value:
        return HIPBLAS_STATUS_INVALID_VALUE;
    case rocblas_status_memory_error:
        return HIPBLAS_STATUS_ALLOC_FAILED;
    case rocblas_status_internal_error:
        return HIPBLAS_STATUS_INTERNAL_ERROR;
    default:
        return HIPBLAS_STATUS_UNKNOWN;
    }
}

hipblasStatus_t hipblasCreate(hipblasHandle_t* handle)
try
{
    if(!handle)
        return HIPBLAS_STATUS_HANDLE_IS_NULLPTR;

    // Create the rocBLAS handle
    return rocBLASStatusToHIPStatus(rocblas_create_handle((rocblas_handle*)handle));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDestroy(hipblasHandle_t handle)
try
{
    return rocBLASStatusToHIPStatus(rocblas_destroy_handle((rocblas_handle)handle));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasSetStream(hipblasHandle_t handle, hipStream_t streamId)
try
{
    if(handle == nullptr)
    {
        return HIPBLAS_STATUS_NOT_INITIALIZED;
    }
    return rocBLASStatusToHIPStatus(rocblas_set_stream((rocblas_handle)handle, streamId));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasGetStream(hipblasHandle_t handle, hipStream_t* streamId)
try
{
    if(handle == nullptr)
    {
        return HIPBLAS_STATUS_NOT_INITIALIZED;
    }
    return rocBLASStatusToHIPStatus(rocblas_get_stream((rocblas_handle)handle, streamId));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasSetPointerMode(hipblasHandle_t handle, hipblasPointerMode_t mode)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_set_pointer_mode((rocblas_handle)handle, HIPPointerModeToRocblasPointerMode(mode)));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasGetPointerMode(hipblasHandle_t handle, hipblasPointerMode_t* mode)
try
{
    rocblas_pointer_mode rocblas_mode;
    rocblas_status       status = rocblas_get_pointer_mode((rocblas_handle)handle, &rocblas_mode);
    *mode                       = RocblasPointerModeToHIPPointerMode(rocblas_mode);
    return rocBLASStatusToHIPStatus(status);
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasSetMathMode(hipblasHandle_t handle, hipblasMath_t mode)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_set_math_mode((rocblas_handle)handle, hipblasMathModeTorocblasMathMode(mode)));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasGetMathMode(hipblasHandle_t handle, hipblasMath_t* mode)
try
{
    rocblas_math_mode rocblas_mode;
    rocblas_status    status = rocblas_get_math_mode((rocblas_handle)handle, &rocblas_mode);
    *mode                    = rocblasMathModeTohipblasMathMode(rocblas_mode);
    return rocBLASStatusToHIPStatus(status);
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasSetVector(int n, int elemSize, const void* x, int incx, void* y, int incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_set_vector(n, elemSize, x, incx, y, incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasGetVector(int n, int elemSize, const void* x, int incx, void* y, int incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_get_vector(n, elemSize, x, incx, y, incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasSetMatrix(int rows, int cols, int elemSize, const void* A, int lda, void* B, int ldb)
try
{
    return rocBLASStatusToHIPStatus(rocblas_set_matrix(rows, cols, elemSize, A, lda, B, ldb));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasGetMatrix(int rows, int cols, int elemSize, const void* A, int lda, void* B, int ldb)
try
{
    return rocBLASStatusToHIPStatus(rocblas_get_matrix(rows, cols, elemSize, A, lda, B, ldb));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasSetVectorAsync(
    int n, int elemSize, const void* x, int incx, void* y, int incy, hipStream_t stream)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_set_vector_async(n, elemSize, x, incx, y, incy, stream));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasGetVectorAsync(
    int n, int elemSize, const void* x, int incx, void* y, int incy, hipStream_t stream)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_get_vector_async(n, elemSize, x, incx, y, incy, stream));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasSetMatrixAsync(
    int rows, int cols, int elemSize, const void* A, int lda, void* B, int ldb, hipStream_t stream)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_set_matrix_async(rows, cols, elemSize, A, lda, B, ldb, stream));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasGetMatrixAsync(
    int rows, int cols, int elemSize, const void* A, int lda, void* B, int ldb, hipStream_t stream)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_get_matrix_async(rows, cols, elemSize, A, lda, B, ldb, stream));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// atomics mode
hipblasStatus_t hipblasSetAtomicsMode(hipblasHandle_t handle, hipblasAtomicsMode_t atomics_mode)
try
{
    return rocBLASStatusToHIPStatus(rocblas_set_atomics_mode(
        (rocblas_handle)handle, HIPAtomicsModeToRocblasAtomicsMode(atomics_mode)));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasGetAtomicsMode(hipblasHandle_t handle, hipblasAtomicsMode_t* atomics_mode)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_get_atomics_mode((rocblas_handle)handle, (rocblas_atomics_mode*)atomics_mode));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// amax
hipblasStatus_t hipblasIsamax(hipblasHandle_t handle, int n, const float* x, int incx, int* result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_isamax((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIdamax(hipblasHandle_t handle, int n, const double* x, int incx, int* result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_idamax((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasIcamax(hipblasHandle_t handle, int n, const hipblasComplex* x, int incx, int* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_icamax((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzamax(
    hipblasHandle_t handle, int n, const hipblasDoubleComplex* x, int incx, int* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_izamax((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasIcamax_v2(hipblasHandle_t handle, int n, const hipComplex* x, int incx, int* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_icamax((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzamax_v2(
    hipblasHandle_t handle, int n, const hipDoubleComplex* x, int incx, int* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_izamax((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// amax_64
hipblasStatus_t hipblasIsamax_64(
    hipblasHandle_t handle, int64_t n, const float* x, int64_t incx, int64_t* result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_isamax_64((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIdamax_64(
    hipblasHandle_t handle, int64_t n, const double* x, int64_t incx, int64_t* result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_idamax_64((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIcamax_64(
    hipblasHandle_t handle, int64_t n, const hipblasComplex* x, int64_t incx, int64_t* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_icamax_64((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzamax_64(
    hipblasHandle_t handle, int64_t n, const hipblasDoubleComplex* x, int64_t incx, int64_t* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_izamax_64((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIcamax_v2_64(
    hipblasHandle_t handle, int64_t n, const hipComplex* x, int64_t incx, int64_t* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_icamax_64((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzamax_v2_64(
    hipblasHandle_t handle, int64_t n, const hipDoubleComplex* x, int64_t incx, int64_t* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_izamax_64((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// amax_batched
hipblasStatus_t hipblasIsamaxBatched(
    hipblasHandle_t handle, int n, const float* const x[], int incx, int batchCount, int* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_isamax_batched((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIdamaxBatched(
    hipblasHandle_t handle, int n, const double* const x[], int incx, int batchCount, int* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_idamax_batched((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIcamaxBatched(hipblasHandle_t             handle,
                                     int                         n,
                                     const hipblasComplex* const x[],
                                     int                         incx,
                                     int                         batchCount,
                                     int*                        result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_icamax_batched((rocblas_handle)handle,
                                                           n,
                                                           (const rocblas_float_complex* const*)x,
                                                           incx,
                                                           batchCount,
                                                           result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzamaxBatched(hipblasHandle_t                   handle,
                                     int                               n,
                                     const hipblasDoubleComplex* const x[],
                                     int                               incx,
                                     int                               batchCount,
                                     int*                              result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_izamax_batched((rocblas_handle)handle,
                                                           n,
                                                           (const rocblas_double_complex* const*)x,
                                                           incx,
                                                           batchCount,
                                                           result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIcamaxBatched_v2(hipblasHandle_t         handle,
                                        int                     n,
                                        const hipComplex* const x[],
                                        int                     incx,
                                        int                     batchCount,
                                        int*                    result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_icamax_batched((rocblas_handle)handle,
                                                           n,
                                                           (const rocblas_float_complex* const*)x,
                                                           incx,
                                                           batchCount,
                                                           result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzamaxBatched_v2(hipblasHandle_t               handle,
                                        int                           n,
                                        const hipDoubleComplex* const x[],
                                        int                           incx,
                                        int                           batchCount,
                                        int*                          result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_izamax_batched((rocblas_handle)handle,
                                                           n,
                                                           (const rocblas_double_complex* const*)x,
                                                           incx,
                                                           batchCount,
                                                           result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// amax_batched_64
hipblasStatus_t hipblasIsamaxBatched_64(hipblasHandle_t    handle,
                                        int64_t            n,
                                        const float* const x[],
                                        int64_t            incx,
                                        int64_t            batchCount,
                                        int64_t*           result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_isamax_batched_64((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIdamaxBatched_64(hipblasHandle_t     handle,
                                        int64_t             n,
                                        const double* const x[],
                                        int64_t             incx,
                                        int64_t             batchCount,
                                        int64_t*            result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_idamax_batched_64((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIcamaxBatched_64(hipblasHandle_t             handle,
                                        int64_t                     n,
                                        const hipblasComplex* const x[],
                                        int64_t                     incx,
                                        int64_t                     batchCount,
                                        int64_t*                    result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_icamax_batched_64((rocblas_handle)handle,
                                  n,
                                  (const rocblas_float_complex* const*)x,
                                  incx,
                                  batchCount,
                                  result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzamaxBatched_64(hipblasHandle_t                   handle,
                                        int64_t                           n,
                                        const hipblasDoubleComplex* const x[],
                                        int64_t                           incx,
                                        int64_t                           batchCount,
                                        int64_t*                          result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_izamax_batched_64((rocblas_handle)handle,
                                  n,
                                  (const rocblas_double_complex* const*)x,
                                  incx,
                                  batchCount,
                                  result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIcamaxBatched_v2_64(hipblasHandle_t         handle,
                                           int64_t                 n,
                                           const hipComplex* const x[],
                                           int64_t                 incx,
                                           int64_t                 batchCount,
                                           int64_t*                result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_icamax_batched_64((rocblas_handle)handle,
                                  n,
                                  (const rocblas_float_complex* const*)x,
                                  incx,
                                  batchCount,
                                  result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzamaxBatched_v2_64(hipblasHandle_t               handle,
                                           int64_t                       n,
                                           const hipDoubleComplex* const x[],
                                           int64_t                       incx,
                                           int64_t                       batchCount,
                                           int64_t*                      result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_izamax_batched_64((rocblas_handle)handle,
                                  n,
                                  (const rocblas_double_complex* const*)x,
                                  incx,
                                  batchCount,
                                  result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// amax_strided_batched
hipblasStatus_t hipblasIsamaxStridedBatched(hipblasHandle_t handle,
                                            int             n,
                                            const float*    x,
                                            int             incx,
                                            hipblasStride   stridex,
                                            int             batchCount,
                                            int*            result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_isamax_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIdamaxStridedBatched(hipblasHandle_t handle,
                                            int             n,
                                            const double*   x,
                                            int             incx,
                                            hipblasStride   stridex,
                                            int             batchCount,
                                            int*            result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_idamax_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIcamaxStridedBatched(hipblasHandle_t       handle,
                                            int                   n,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            hipblasStride         stridex,
                                            int                   batchCount,
                                            int*                  result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_icamax_strided_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzamaxStridedBatched(hipblasHandle_t             handle,
                                            int                         n,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            hipblasStride               stridex,
                                            int                         batchCount,
                                            int*                        result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_izamax_strided_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIcamaxStridedBatched_v2(hipblasHandle_t   handle,
                                               int               n,
                                               const hipComplex* x,
                                               int               incx,
                                               hipblasStride     stridex,
                                               int               batchCount,
                                               int*              result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_icamax_strided_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzamaxStridedBatched_v2(hipblasHandle_t         handle,
                                               int                     n,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               hipblasStride           stridex,
                                               int                     batchCount,
                                               int*                    result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_izamax_strided_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// amax_strided_batched_64
hipblasStatus_t hipblasIsamaxStridedBatched_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               const float*    x,
                                               int64_t         incx,
                                               hipblasStride   stridex,
                                               int64_t         batchCount,
                                               int64_t*        result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_isamax_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIdamaxStridedBatched_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               const double*   x,
                                               int64_t         incx,
                                               hipblasStride   stridex,
                                               int64_t         batchCount,
                                               int64_t*        result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_idamax_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIcamaxStridedBatched_64(hipblasHandle_t       handle,
                                               int64_t               n,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               hipblasStride         stridex,
                                               int64_t               batchCount,
                                               int64_t*              result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_icamax_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzamaxStridedBatched_64(hipblasHandle_t             handle,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               hipblasStride               stridex,
                                               int64_t                     batchCount,
                                               int64_t*                    result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_izamax_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIcamaxStridedBatched_v2_64(hipblasHandle_t   handle,
                                                  int64_t           n,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  hipblasStride     stridex,
                                                  int64_t           batchCount,
                                                  int64_t*          result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_icamax_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzamaxStridedBatched_v2_64(hipblasHandle_t         handle,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  hipblasStride           stridex,
                                                  int64_t                 batchCount,
                                                  int64_t*                result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_izamax_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// amin
hipblasStatus_t hipblasIsamin(hipblasHandle_t handle, int n, const float* x, int incx, int* result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_isamin((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIdamin(hipblasHandle_t handle, int n, const double* x, int incx, int* result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_idamin((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasIcamin(hipblasHandle_t handle, int n, const hipblasComplex* x, int incx, int* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_icamin((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzamin(
    hipblasHandle_t handle, int n, const hipblasDoubleComplex* x, int incx, int* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_izamin((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasIcamin_v2(hipblasHandle_t handle, int n, const hipComplex* x, int incx, int* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_icamin((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzamin_v2(
    hipblasHandle_t handle, int n, const hipDoubleComplex* x, int incx, int* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_izamin((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// amin_64
hipblasStatus_t hipblasIsamin_64(
    hipblasHandle_t handle, int64_t n, const float* x, int64_t incx, int64_t* result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_isamin_64((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIdamin_64(
    hipblasHandle_t handle, int64_t n, const double* x, int64_t incx, int64_t* result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_idamin_64((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIcamin_64(
    hipblasHandle_t handle, int64_t n, const hipblasComplex* x, int64_t incx, int64_t* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_icamin_64((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzamin_64(
    hipblasHandle_t handle, int64_t n, const hipblasDoubleComplex* x, int64_t incx, int64_t* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_izamin_64((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIcamin_v2_64(
    hipblasHandle_t handle, int64_t n, const hipComplex* x, int64_t incx, int64_t* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_icamin_64((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzamin_v2_64(
    hipblasHandle_t handle, int64_t n, const hipDoubleComplex* x, int64_t incx, int64_t* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_izamin_64((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// amin_batched
hipblasStatus_t hipblasIsaminBatched(
    hipblasHandle_t handle, int n, const float* const x[], int incx, int batchCount, int* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_isamin_batched((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIdaminBatched(
    hipblasHandle_t handle, int n, const double* const x[], int incx, int batchCount, int* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_idamin_batched((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIcaminBatched(hipblasHandle_t             handle,
                                     int                         n,
                                     const hipblasComplex* const x[],
                                     int                         incx,
                                     int                         batchCount,
                                     int*                        result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_icamin_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzaminBatched(hipblasHandle_t                   handle,
                                     int                               n,
                                     const hipblasDoubleComplex* const x[],
                                     int                               incx,
                                     int                               batchCount,
                                     int*                              result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_izamin_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIcaminBatched_v2(hipblasHandle_t         handle,
                                        int                     n,
                                        const hipComplex* const x[],
                                        int                     incx,
                                        int                     batchCount,
                                        int*                    result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_icamin_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzaminBatched_v2(hipblasHandle_t               handle,
                                        int                           n,
                                        const hipDoubleComplex* const x[],
                                        int                           incx,
                                        int                           batchCount,
                                        int*                          result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_izamin_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// amin_batched_64
hipblasStatus_t hipblasIsaminBatched_64(hipblasHandle_t    handle,
                                        int64_t            n,
                                        const float* const x[],
                                        int64_t            incx,
                                        int64_t            batchCount,
                                        int64_t*           result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_isamin_batched_64((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIdaminBatched_64(hipblasHandle_t     handle,
                                        int64_t             n,
                                        const double* const x[],
                                        int64_t             incx,
                                        int64_t             batchCount,
                                        int64_t*            result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_idamin_batched_64((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIcaminBatched_64(hipblasHandle_t             handle,
                                        int64_t                     n,
                                        const hipblasComplex* const x[],
                                        int64_t                     incx,
                                        int64_t                     batchCount,
                                        int64_t*                    result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_icamin_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzaminBatched_64(hipblasHandle_t                   handle,
                                        int64_t                           n,
                                        const hipblasDoubleComplex* const x[],
                                        int64_t                           incx,
                                        int64_t                           batchCount,
                                        int64_t*                          result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_izamin_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIcaminBatched_v2_64(hipblasHandle_t         handle,
                                           int64_t                 n,
                                           const hipComplex* const x[],
                                           int64_t                 incx,
                                           int64_t                 batchCount,
                                           int64_t*                result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_icamin_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzaminBatched_v2_64(hipblasHandle_t               handle,
                                           int64_t                       n,
                                           const hipDoubleComplex* const x[],
                                           int64_t                       incx,
                                           int64_t                       batchCount,
                                           int64_t*                      result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_izamin_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// amin_strided_batched
hipblasStatus_t hipblasIsaminStridedBatched(hipblasHandle_t handle,
                                            int             n,
                                            const float*    x,
                                            int             incx,
                                            hipblasStride   stridex,
                                            int             batchCount,
                                            int*            result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_isamin_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIdaminStridedBatched(hipblasHandle_t handle,
                                            int             n,
                                            const double*   x,
                                            int             incx,
                                            hipblasStride   stridex,
                                            int             batchCount,
                                            int*            result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_idamin_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIcaminStridedBatched(hipblasHandle_t       handle,
                                            int                   n,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            hipblasStride         stridex,
                                            int                   batchCount,
                                            int*                  result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_icamin_strided_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzaminStridedBatched(hipblasHandle_t             handle,
                                            int                         n,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            hipblasStride               stridex,
                                            int                         batchCount,
                                            int*                        result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_izamin_strided_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIcaminStridedBatched_v2(hipblasHandle_t   handle,
                                               int               n,
                                               const hipComplex* x,
                                               int               incx,
                                               hipblasStride     stridex,
                                               int               batchCount,
                                               int*              result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_icamin_strided_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzaminStridedBatched_v2(hipblasHandle_t         handle,
                                               int                     n,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               hipblasStride           stridex,
                                               int                     batchCount,
                                               int*                    result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_izamin_strided_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// amin_strided_batched_64
hipblasStatus_t hipblasIsaminStridedBatched_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               const float*    x,
                                               int64_t         incx,
                                               hipblasStride   stridex,
                                               int64_t         batchCount,
                                               int64_t*        result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_isamin_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIdaminStridedBatched_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               const double*   x,
                                               int64_t         incx,
                                               hipblasStride   stridex,
                                               int64_t         batchCount,
                                               int64_t*        result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_idamin_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIcaminStridedBatched_64(hipblasHandle_t       handle,
                                               int64_t               n,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               hipblasStride         stridex,
                                               int64_t               batchCount,
                                               int64_t*              result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_icamin_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzaminStridedBatched_64(hipblasHandle_t             handle,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               hipblasStride               stridex,
                                               int64_t                     batchCount,
                                               int64_t*                    result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_izamin_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIcaminStridedBatched_v2_64(hipblasHandle_t   handle,
                                                  int64_t           n,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  hipblasStride     stridex,
                                                  int64_t           batchCount,
                                                  int64_t*          result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_icamin_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasIzaminStridedBatched_v2_64(hipblasHandle_t         handle,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  hipblasStride           stridex,
                                                  int64_t                 batchCount,
                                                  int64_t*                result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_izamin_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// asum
hipblasStatus_t hipblasSasum(hipblasHandle_t handle, int n, const float* x, int incx, float* result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sasum((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasDasum(hipblasHandle_t handle, int n, const double* x, int incx, double* result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dasum((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasScasum(hipblasHandle_t handle, int n, const hipblasComplex* x, int incx, float* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_scasum((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDzasum(
    hipblasHandle_t handle, int n, const hipblasDoubleComplex* x, int incx, double* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dzasum((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasScasum_v2(hipblasHandle_t handle, int n, const hipComplex* x, int incx, float* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_scasum((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDzasum_v2(
    hipblasHandle_t handle, int n, const hipDoubleComplex* x, int incx, double* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dzasum((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// asum_64
hipblasStatus_t
    hipblasSasum_64(hipblasHandle_t handle, int64_t n, const float* x, int64_t incx, float* result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sasum_64((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDasum_64(
    hipblasHandle_t handle, int64_t n, const double* x, int64_t incx, double* result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dasum_64((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasScasum_64(
    hipblasHandle_t handle, int64_t n, const hipblasComplex* x, int64_t incx, float* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_scasum_64((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDzasum_64(
    hipblasHandle_t handle, int64_t n, const hipblasDoubleComplex* x, int64_t incx, double* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dzasum_64((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasScasum_v2_64(
    hipblasHandle_t handle, int64_t n, const hipComplex* x, int64_t incx, float* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_scasum_64((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDzasum_v2_64(
    hipblasHandle_t handle, int64_t n, const hipDoubleComplex* x, int64_t incx, double* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dzasum_64((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// asum_batched
hipblasStatus_t hipblasSasumBatched(
    hipblasHandle_t handle, int n, const float* const x[], int incx, int batchCount, float* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_sasum_batched((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDasumBatched(hipblasHandle_t     handle,
                                    int                 n,
                                    const double* const x[],
                                    int                 incx,
                                    int                 batchCount,
                                    double*             result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dasum_batched((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasScasumBatched(hipblasHandle_t             handle,
                                     int                         n,
                                     const hipblasComplex* const x[],
                                     int                         incx,
                                     int                         batchCount,
                                     float*                      result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_scasum_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDzasumBatched(hipblasHandle_t                   handle,
                                     int                               n,
                                     const hipblasDoubleComplex* const x[],
                                     int                               incx,
                                     int                               batchCount,
                                     double*                           result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dzasum_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasScasumBatched_v2(hipblasHandle_t         handle,
                                        int                     n,
                                        const hipComplex* const x[],
                                        int                     incx,
                                        int                     batchCount,
                                        float*                  result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_scasum_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDzasumBatched_v2(hipblasHandle_t               handle,
                                        int                           n,
                                        const hipDoubleComplex* const x[],
                                        int                           incx,
                                        int                           batchCount,
                                        double*                       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dzasum_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// asum_batched_64
hipblasStatus_t hipblasSasumBatched_64(hipblasHandle_t    handle,
                                       int64_t            n,
                                       const float* const x[],
                                       int64_t            incx,
                                       int64_t            batchCount,
                                       float*             result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_sasum_batched_64((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDasumBatched_64(hipblasHandle_t     handle,
                                       int64_t             n,
                                       const double* const x[],
                                       int64_t             incx,
                                       int64_t             batchCount,
                                       double*             result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dasum_batched_64((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasScasumBatched_64(hipblasHandle_t             handle,
                                        int64_t                     n,
                                        const hipblasComplex* const x[],
                                        int64_t                     incx,
                                        int64_t                     batchCount,
                                        float*                      result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_scasum_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDzasumBatched_64(hipblasHandle_t                   handle,
                                        int64_t                           n,
                                        const hipblasDoubleComplex* const x[],
                                        int64_t                           incx,
                                        int64_t                           batchCount,
                                        double*                           result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dzasum_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasScasumBatched_v2_64(hipblasHandle_t         handle,
                                           int64_t                 n,
                                           const hipComplex* const x[],
                                           int64_t                 incx,
                                           int64_t                 batchCount,
                                           float*                  result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_scasum_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDzasumBatched_v2_64(hipblasHandle_t               handle,
                                           int64_t                       n,
                                           const hipDoubleComplex* const x[],
                                           int64_t                       incx,
                                           int64_t                       batchCount,
                                           double*                       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dzasum_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// asum_strided_batched
hipblasStatus_t hipblasSasumStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const float*    x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           int             batchCount,
                                           float*          result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sasum_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDasumStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const double*   x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           int             batchCount,
                                           double*         result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dasum_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasScasumStridedBatched(hipblasHandle_t       handle,
                                            int                   n,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            hipblasStride         stridex,
                                            int                   batchCount,
                                            float*                result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_scasum_strided_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDzasumStridedBatched(hipblasHandle_t             handle,
                                            int                         n,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            hipblasStride               stridex,
                                            int                         batchCount,
                                            double*                     result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dzasum_strided_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasScasumStridedBatched_v2(hipblasHandle_t   handle,
                                               int               n,
                                               const hipComplex* x,
                                               int               incx,
                                               hipblasStride     stridex,
                                               int               batchCount,
                                               float*            result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_scasum_strided_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDzasumStridedBatched_v2(hipblasHandle_t         handle,
                                               int                     n,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               hipblasStride           stridex,
                                               int                     batchCount,
                                               double*                 result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dzasum_strided_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// asum_strided_batched_64
hipblasStatus_t hipblasSasumStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const float*    x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              int64_t         batchCount,
                                              float*          result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sasum_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDasumStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const double*   x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              int64_t         batchCount,
                                              double*         result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dasum_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasScasumStridedBatched_64(hipblasHandle_t       handle,
                                               int64_t               n,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               hipblasStride         stridex,
                                               int64_t               batchCount,
                                               float*                result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_scasum_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDzasumStridedBatched_64(hipblasHandle_t             handle,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               hipblasStride               stridex,
                                               int64_t                     batchCount,
                                               double*                     result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dzasum_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasScasumStridedBatched_v2_64(hipblasHandle_t   handle,
                                                  int64_t           n,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  hipblasStride     stridex,
                                                  int64_t           batchCount,
                                                  float*            result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_scasum_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDzasumStridedBatched_v2_64(hipblasHandle_t         handle,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  hipblasStride           stridex,
                                                  int64_t                 batchCount,
                                                  double*                 result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dzasum_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// axpy
hipblasStatus_t hipblasHaxpy(hipblasHandle_t    handle,
                             int                n,
                             const hipblasHalf* alpha,
                             const hipblasHalf* x,
                             int                incx,
                             hipblasHalf*       y,
                             int                incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_haxpy((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_half*)alpha,
                                                  (rocblas_half*)x,
                                                  incx,
                                                  (rocblas_half*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasSaxpy(
    hipblasHandle_t handle, int n, const float* alpha, const float* x, int incx, float* y, int incy)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_saxpy((rocblas_handle)handle, n, alpha, x, incx, y, incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDaxpy(hipblasHandle_t handle,
                             int             n,
                             const double*   alpha,
                             const double*   x,
                             int             incx,
                             double*         y,
                             int             incy)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_daxpy((rocblas_handle)handle, n, alpha, x, incx, y, incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCaxpy(hipblasHandle_t       handle,
                             int                   n,
                             const hipblasComplex* alpha,
                             const hipblasComplex* x,
                             int                   incx,
                             hipblasComplex*       y,
                             int                   incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_caxpy((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZaxpy(hipblasHandle_t             handle,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             hipblasDoubleComplex*       y,
                             int                         incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zaxpy((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCaxpy_v2(hipblasHandle_t   handle,
                                int               n,
                                const hipComplex* alpha,
                                const hipComplex* x,
                                int               incx,
                                hipComplex*       y,
                                int               incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_caxpy((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZaxpy_v2(hipblasHandle_t         handle,
                                int                     n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* x,
                                int                     incx,
                                hipDoubleComplex*       y,
                                int                     incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zaxpy((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// 64-bit interface
hipblasStatus_t hipblasHaxpy_64(hipblasHandle_t    handle,
                                int64_t            n,
                                const hipblasHalf* alpha,
                                const hipblasHalf* x,
                                int64_t            incx,
                                hipblasHalf*       y,
                                int64_t            incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_haxpy_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_half*)alpha,
                                                     (rocblas_half*)x,
                                                     incx,
                                                     (rocblas_half*)y,
                                                     incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasSaxpy_64(hipblasHandle_t handle,
                                int64_t         n,
                                const float*    alpha,
                                const float*    x,
                                int64_t         incx,
                                float*          y,
                                int64_t         incy)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_saxpy_64((rocblas_handle)handle, n, alpha, x, incx, y, incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDaxpy_64(hipblasHandle_t handle,
                                int64_t         n,
                                const double*   alpha,
                                const double*   x,
                                int64_t         incx,
                                double*         y,
                                int64_t         incy)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_daxpy_64((rocblas_handle)handle, n, alpha, x, incx, y, incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCaxpy_64(hipblasHandle_t       handle,
                                int64_t               n,
                                const hipblasComplex* alpha,
                                const hipblasComplex* x,
                                int64_t               incx,
                                hipblasComplex*       y,
                                int64_t               incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_caxpy_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_float_complex*)alpha,
                                                     (rocblas_float_complex*)x,
                                                     incx,
                                                     (rocblas_float_complex*)y,
                                                     incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZaxpy_64(hipblasHandle_t             handle,
                                int64_t                     n,
                                const hipblasDoubleComplex* alpha,
                                const hipblasDoubleComplex* x,
                                int64_t                     incx,
                                hipblasDoubleComplex*       y,
                                int64_t                     incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zaxpy_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_double_complex*)alpha,
                                                     (rocblas_double_complex*)x,
                                                     incx,
                                                     (rocblas_double_complex*)y,
                                                     incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCaxpy_v2_64(hipblasHandle_t   handle,
                                   int64_t           n,
                                   const hipComplex* alpha,
                                   const hipComplex* x,
                                   int64_t           incx,
                                   hipComplex*       y,
                                   int64_t           incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_caxpy_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_float_complex*)alpha,
                                                     (rocblas_float_complex*)x,
                                                     incx,
                                                     (rocblas_float_complex*)y,
                                                     incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZaxpy_v2_64(hipblasHandle_t         handle,
                                   int64_t                 n,
                                   const hipDoubleComplex* alpha,
                                   const hipDoubleComplex* x,
                                   int64_t                 incx,
                                   hipDoubleComplex*       y,
                                   int64_t                 incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zaxpy_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_double_complex*)alpha,
                                                     (rocblas_double_complex*)x,
                                                     incx,
                                                     (rocblas_double_complex*)y,
                                                     incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// axpy_batched
hipblasStatus_t hipblasHaxpyBatched(hipblasHandle_t          handle,
                                    int                      n,
                                    const hipblasHalf*       alpha,
                                    const hipblasHalf* const x[],
                                    int                      incx,
                                    hipblasHalf* const       y[],
                                    int                      incy,
                                    int                      batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_haxpy_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_half*)alpha,
                                                          (rocblas_half* const*)x,
                                                          incx,
                                                          (rocblas_half* const*)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasSaxpyBatched(hipblasHandle_t    handle,
                                    int                n,
                                    const float*       alpha,
                                    const float* const x[],
                                    int                incx,
                                    float* const       y[],
                                    int                incy,
                                    int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_saxpy_batched((rocblas_handle)handle, n, alpha, x, incx, y, incy, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDaxpyBatched(hipblasHandle_t     handle,
                                    int                 n,
                                    const double*       alpha,
                                    const double* const x[],
                                    int                 incx,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_daxpy_batched((rocblas_handle)handle, n, alpha, x, incx, y, incy, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCaxpyBatched(hipblasHandle_t             handle,
                                    int                         n,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    hipblasComplex* const       y[],
                                    int                         incy,
                                    int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_caxpy_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex* const*)x,
                                                          incx,
                                                          (rocblas_float_complex* const*)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZaxpyBatched(hipblasHandle_t                   handle,
                                    int                               n,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    hipblasDoubleComplex* const       y[],
                                    int                               incy,
                                    int                               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zaxpy_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex* const*)x,
                                                          incx,
                                                          (rocblas_double_complex* const*)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCaxpyBatched_v2(hipblasHandle_t         handle,
                                       int                     n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       hipComplex* const       y[],
                                       int                     incy,
                                       int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_caxpy_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex* const*)x,
                                                          incx,
                                                          (rocblas_float_complex* const*)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZaxpyBatched_v2(hipblasHandle_t               handle,
                                       int                           n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       hipDoubleComplex* const       y[],
                                       int                           incy,
                                       int                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zaxpy_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex* const*)x,
                                                          incx,
                                                          (rocblas_double_complex* const*)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// 64-bit interface
hipblasStatus_t hipblasHaxpyBatched_64(hipblasHandle_t          handle,
                                       int64_t                  n,
                                       const hipblasHalf*       alpha,
                                       const hipblasHalf* const x[],
                                       int64_t                  incx,
                                       hipblasHalf* const       y[],
                                       int64_t                  incy,
                                       int64_t                  batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_haxpy_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_half*)alpha,
                                                             (rocblas_half* const*)x,
                                                             incx,
                                                             (rocblas_half* const*)y,
                                                             incy,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasSaxpyBatched_64(hipblasHandle_t    handle,
                                       int64_t            n,
                                       const float*       alpha,
                                       const float* const x[],
                                       int64_t            incx,
                                       float* const       y[],
                                       int64_t            incy,
                                       int64_t            batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_saxpy_batched_64((rocblas_handle)handle, n, alpha, x, incx, y, incy, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDaxpyBatched_64(hipblasHandle_t     handle,
                                       int64_t             n,
                                       const double*       alpha,
                                       const double* const x[],
                                       int64_t             incx,
                                       double* const       y[],
                                       int64_t             incy,
                                       int64_t             batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_daxpy_batched_64((rocblas_handle)handle, n, alpha, x, incx, y, incy, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCaxpyBatched_64(hipblasHandle_t             handle,
                                       int64_t                     n,
                                       const hipblasComplex*       alpha,
                                       const hipblasComplex* const x[],
                                       int64_t                     incx,
                                       hipblasComplex* const       y[],
                                       int64_t                     incy,
                                       int64_t                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_caxpy_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_float_complex*)alpha,
                                                             (rocblas_float_complex* const*)x,
                                                             incx,
                                                             (rocblas_float_complex* const*)y,
                                                             incy,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZaxpyBatched_64(hipblasHandle_t                   handle,
                                       int64_t                           n,
                                       const hipblasDoubleComplex*       alpha,
                                       const hipblasDoubleComplex* const x[],
                                       int64_t                           incx,
                                       hipblasDoubleComplex* const       y[],
                                       int64_t                           incy,
                                       int64_t                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zaxpy_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_double_complex*)alpha,
                                                             (rocblas_double_complex* const*)x,
                                                             incx,
                                                             (rocblas_double_complex* const*)y,
                                                             incy,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCaxpyBatched_v2_64(hipblasHandle_t         handle,
                                          int64_t                 n,
                                          const hipComplex*       alpha,
                                          const hipComplex* const x[],
                                          int64_t                 incx,
                                          hipComplex* const       y[],
                                          int64_t                 incy,
                                          int64_t                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_caxpy_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_float_complex*)alpha,
                                                             (rocblas_float_complex* const*)x,
                                                             incx,
                                                             (rocblas_float_complex* const*)y,
                                                             incy,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZaxpyBatched_v2_64(hipblasHandle_t               handle,
                                          int64_t                       n,
                                          const hipDoubleComplex*       alpha,
                                          const hipDoubleComplex* const x[],
                                          int64_t                       incx,
                                          hipDoubleComplex* const       y[],
                                          int64_t                       incy,
                                          int64_t                       batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zaxpy_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_double_complex*)alpha,
                                                             (rocblas_double_complex* const*)x,
                                                             incx,
                                                             (rocblas_double_complex* const*)y,
                                                             incy,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// axpy_strided_batched
hipblasStatus_t hipblasHaxpyStridedBatched(hipblasHandle_t    handle,
                                           int                n,
                                           const hipblasHalf* alpha,
                                           const hipblasHalf* x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           hipblasHalf*       y,
                                           int                incy,
                                           hipblasStride      stridey,
                                           int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_haxpy_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_half*)alpha,
                                                                  (rocblas_half*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_half*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasSaxpyStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const float*    alpha,
                                           const float*    x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           float*          y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_saxpy_strided_batched(
        (rocblas_handle)handle, n, alpha, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDaxpyStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const double*   alpha,
                                           const double*   x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           double*         y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_daxpy_strided_batched(
        (rocblas_handle)handle, n, alpha, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCaxpyStridedBatched(hipblasHandle_t       handle,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           hipblasComplex*       y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_caxpy_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZaxpyStridedBatched(hipblasHandle_t             handle,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           hipblasDoubleComplex*       y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zaxpy_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCaxpyStridedBatched_v2(hipblasHandle_t   handle,
                                              int               n,
                                              const hipComplex* alpha,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              hipComplex*       y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_caxpy_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZaxpyStridedBatched_v2(hipblasHandle_t         handle,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              hipDoubleComplex*       y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zaxpy_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// 64-bit interface
hipblasStatus_t hipblasHaxpyStridedBatched_64(hipblasHandle_t    handle,
                                              int64_t            n,
                                              const hipblasHalf* alpha,
                                              const hipblasHalf* x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              hipblasHalf*       y,
                                              int64_t            incy,
                                              hipblasStride      stridey,
                                              int64_t            batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_haxpy_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_half*)alpha,
                                                                     (rocblas_half*)x,
                                                                     incx,
                                                                     stridex,
                                                                     (rocblas_half*)y,
                                                                     incy,
                                                                     stridey,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasSaxpyStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const float*    alpha,
                                              const float*    x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              float*          y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_saxpy_strided_batched_64(
        (rocblas_handle)handle, n, alpha, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDaxpyStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const double*   alpha,
                                              const double*   x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              double*         y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_daxpy_strided_batched_64(
        (rocblas_handle)handle, n, alpha, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCaxpyStridedBatched_64(hipblasHandle_t       handle,
                                              int64_t               n,
                                              const hipblasComplex* alpha,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              hipblasComplex*       y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              int64_t               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_caxpy_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_float_complex*)alpha,
                                                                     (rocblas_float_complex*)x,
                                                                     incx,
                                                                     stridex,
                                                                     (rocblas_float_complex*)y,
                                                                     incy,
                                                                     stridey,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZaxpyStridedBatched_64(hipblasHandle_t             handle,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* alpha,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              hipblasDoubleComplex*       y,
                                              int64_t                     incy,
                                              hipblasStride               stridey,
                                              int64_t                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zaxpy_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_double_complex*)alpha,
                                                                     (rocblas_double_complex*)x,
                                                                     incx,
                                                                     stridex,
                                                                     (rocblas_double_complex*)y,
                                                                     incy,
                                                                     stridey,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCaxpyStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 int64_t           n,
                                                 const hipComplex* alpha,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 hipComplex*       y,
                                                 int64_t           incy,
                                                 hipblasStride     stridey,
                                                 int64_t           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_caxpy_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_float_complex*)alpha,
                                                                     (rocblas_float_complex*)x,
                                                                     incx,
                                                                     stridex,
                                                                     (rocblas_float_complex*)y,
                                                                     incy,
                                                                     stridey,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZaxpyStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* alpha,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 hipDoubleComplex*       y,
                                                 int64_t                 incy,
                                                 hipblasStride           stridey,
                                                 int64_t                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zaxpy_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_double_complex*)alpha,
                                                                     (rocblas_double_complex*)x,
                                                                     incx,
                                                                     stridex,
                                                                     (rocblas_double_complex*)y,
                                                                     incy,
                                                                     stridey,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// copy
hipblasStatus_t
    hipblasScopy(hipblasHandle_t handle, int n, const float* x, int incx, float* y, int incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_scopy((rocblas_handle)handle, n, x, incx, y, incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasDcopy(hipblasHandle_t handle, int n, const double* x, int incx, double* y, int incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dcopy((rocblas_handle)handle, n, x, incx, y, incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCcopy(
    hipblasHandle_t handle, int n, const hipblasComplex* x, int incx, hipblasComplex* y, int incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ccopy((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZcopy(hipblasHandle_t             handle,
                             int                         n,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             hipblasDoubleComplex*       y,
                             int                         incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zcopy((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCcopy_v2(
    hipblasHandle_t handle, int n, const hipComplex* x, int incx, hipComplex* y, int incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ccopy((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZcopy_v2(hipblasHandle_t         handle,
                                int                     n,
                                const hipDoubleComplex* x,
                                int                     incx,
                                hipDoubleComplex*       y,
                                int                     incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zcopy((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// 64-bit interface
hipblasStatus_t hipblasScopy_64(
    hipblasHandle_t handle, int64_t n, const float* x, int64_t incx, float* y, int64_t incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_scopy_64((rocblas_handle)handle, n, x, incx, y, incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDcopy_64(
    hipblasHandle_t handle, int64_t n, const double* x, int64_t incx, double* y, int64_t incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dcopy_64((rocblas_handle)handle, n, x, incx, y, incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCcopy_64(hipblasHandle_t       handle,
                                int64_t               n,
                                const hipblasComplex* x,
                                int64_t               incx,
                                hipblasComplex*       y,
                                int64_t               incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ccopy_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_float_complex*)x,
                                                     incx,
                                                     (rocblas_float_complex*)y,
                                                     incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZcopy_64(hipblasHandle_t             handle,
                                int64_t                     n,
                                const hipblasDoubleComplex* x,
                                int64_t                     incx,
                                hipblasDoubleComplex*       y,
                                int64_t                     incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zcopy_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_double_complex*)x,
                                                     incx,
                                                     (rocblas_double_complex*)y,
                                                     incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCcopy_v2_64(hipblasHandle_t   handle,
                                   int64_t           n,
                                   const hipComplex* x,
                                   int64_t           incx,
                                   hipComplex*       y,
                                   int64_t           incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ccopy_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_float_complex*)x,
                                                     incx,
                                                     (rocblas_float_complex*)y,
                                                     incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZcopy_v2_64(hipblasHandle_t         handle,
                                   int64_t                 n,
                                   const hipDoubleComplex* x,
                                   int64_t                 incx,
                                   hipDoubleComplex*       y,
                                   int64_t                 incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zcopy_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_double_complex*)x,
                                                     incx,
                                                     (rocblas_double_complex*)y,
                                                     incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// copy_batched
hipblasStatus_t hipblasScopyBatched(hipblasHandle_t    handle,
                                    int                n,
                                    const float* const x[],
                                    int                incx,
                                    float* const       y[],
                                    int                incy,
                                    int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_scopy_batched((rocblas_handle)handle, n, x, incx, y, incy, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDcopyBatched(hipblasHandle_t     handle,
                                    int                 n,
                                    const double* const x[],
                                    int                 incx,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dcopy_batched((rocblas_handle)handle, n, x, incx, y, incy, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCcopyBatched(hipblasHandle_t             handle,
                                    int                         n,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    hipblasComplex* const       y[],
                                    int                         incy,
                                    int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ccopy_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZcopyBatched(hipblasHandle_t                   handle,
                                    int                               n,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    hipblasDoubleComplex* const       y[],
                                    int                               incy,
                                    int                               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zcopy_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCcopyBatched_v2(hipblasHandle_t         handle,
                                       int                     n,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       hipComplex* const       y[],
                                       int                     incy,
                                       int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ccopy_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZcopyBatched_v2(hipblasHandle_t               handle,
                                       int                           n,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       hipDoubleComplex* const       y[],
                                       int                           incy,
                                       int                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zcopy_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// 64-bit interface
hipblasStatus_t hipblasScopyBatched_64(hipblasHandle_t    handle,
                                       int64_t            n,
                                       const float* const x[],
                                       int64_t            incx,
                                       float* const       y[],
                                       int64_t            incy,
                                       int64_t            batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_scopy_batched_64((rocblas_handle)handle, n, x, incx, y, incy, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDcopyBatched_64(hipblasHandle_t     handle,
                                       int64_t             n,
                                       const double* const x[],
                                       int64_t             incx,
                                       double* const       y[],
                                       int64_t             incy,
                                       int64_t             batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dcopy_batched_64((rocblas_handle)handle, n, x, incx, y, incy, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCcopyBatched_64(hipblasHandle_t             handle,
                                       int64_t                     n,
                                       const hipblasComplex* const x[],
                                       int64_t                     incx,
                                       hipblasComplex* const       y[],
                                       int64_t                     incy,
                                       int64_t                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ccopy_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_float_complex**)x,
                                                             incx,
                                                             (rocblas_float_complex**)y,
                                                             incy,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZcopyBatched_64(hipblasHandle_t                   handle,
                                       int64_t                           n,
                                       const hipblasDoubleComplex* const x[],
                                       int64_t                           incx,
                                       hipblasDoubleComplex* const       y[],
                                       int64_t                           incy,
                                       int64_t                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zcopy_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_double_complex**)x,
                                                             incx,
                                                             (rocblas_double_complex**)y,
                                                             incy,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCcopyBatched_v2_64(hipblasHandle_t         handle,
                                          int64_t                 n,
                                          const hipComplex* const x[],
                                          int64_t                 incx,
                                          hipComplex* const       y[],
                                          int64_t                 incy,
                                          int64_t                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ccopy_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_float_complex**)x,
                                                             incx,
                                                             (rocblas_float_complex**)y,
                                                             incy,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZcopyBatched_v2_64(hipblasHandle_t               handle,
                                          int64_t                       n,
                                          const hipDoubleComplex* const x[],
                                          int64_t                       incx,
                                          hipDoubleComplex* const       y[],
                                          int64_t                       incy,
                                          int64_t                       batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zcopy_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_double_complex**)x,
                                                             incx,
                                                             (rocblas_double_complex**)y,
                                                             incy,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// copy_strided_batched
hipblasStatus_t hipblasScopyStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const float*    x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           float*          y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_scopy_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDcopyStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const double*   x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           double*         y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dcopy_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCcopyStridedBatched(hipblasHandle_t       handle,
                                           int                   n,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           hipblasComplex*       y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ccopy_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZcopyStridedBatched(hipblasHandle_t             handle,
                                           int                         n,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           hipblasDoubleComplex*       y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zcopy_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCcopyStridedBatched_v2(hipblasHandle_t   handle,
                                              int               n,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              hipComplex*       y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ccopy_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZcopyStridedBatched_v2(hipblasHandle_t         handle,
                                              int                     n,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              hipDoubleComplex*       y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zcopy_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// 64-bit interface
hipblasStatus_t hipblasScopyStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const float*    x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              float*          y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_scopy_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDcopyStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const double*   x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              double*         y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dcopy_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCcopyStridedBatched_64(hipblasHandle_t       handle,
                                              int64_t               n,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              hipblasComplex*       y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              int64_t               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ccopy_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_float_complex*)x,
                                                                     incx,
                                                                     stridex,
                                                                     (rocblas_float_complex*)y,
                                                                     incy,
                                                                     stridey,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZcopyStridedBatched_64(hipblasHandle_t             handle,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              hipblasDoubleComplex*       y,
                                              int64_t                     incy,
                                              hipblasStride               stridey,
                                              int64_t                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zcopy_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_double_complex*)x,
                                                                     incx,
                                                                     stridex,
                                                                     (rocblas_double_complex*)y,
                                                                     incy,
                                                                     stridey,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCcopyStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 int64_t           n,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 hipComplex*       y,
                                                 int64_t           incy,
                                                 hipblasStride     stridey,
                                                 int64_t           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ccopy_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_float_complex*)x,
                                                                     incx,
                                                                     stridex,
                                                                     (rocblas_float_complex*)y,
                                                                     incy,
                                                                     stridey,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZcopyStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 hipDoubleComplex*       y,
                                                 int64_t                 incy,
                                                 hipblasStride           stridey,
                                                 int64_t                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zcopy_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_double_complex*)x,
                                                                     incx,
                                                                     stridex,
                                                                     (rocblas_double_complex*)y,
                                                                     incy,
                                                                     stridey,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// dot
hipblasStatus_t hipblasHdot(hipblasHandle_t    handle,
                            int                n,
                            const hipblasHalf* x,
                            int                incx,
                            const hipblasHalf* y,
                            int                incy,
                            hipblasHalf*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_hdot((rocblas_handle)handle,
                                                 n,
                                                 (rocblas_half*)x,
                                                 incx,
                                                 (rocblas_half*)y,
                                                 incy,
                                                 (rocblas_half*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasBfdot(hipblasHandle_t        handle,
                             int                    n,
                             const hipblasBfloat16* x,
                             int                    incx,
                             const hipblasBfloat16* y,
                             int                    incy,
                             hipblasBfloat16*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_bfdot((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_bfloat16*)x,
                                                  incx,
                                                  (rocblas_bfloat16*)y,
                                                  incy,
                                                  (rocblas_bfloat16*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasSdot(hipblasHandle_t handle,
                            int             n,
                            const float*    x,
                            int             incx,
                            const float*    y,
                            int             incy,
                            float*          result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_sdot((rocblas_handle)handle, n, x, incx, y, incy, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDdot(hipblasHandle_t handle,
                            int             n,
                            const double*   x,
                            int             incx,
                            const double*   y,
                            int             incy,
                            double*         result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ddot((rocblas_handle)handle, n, x, incx, y, incy, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotc(hipblasHandle_t       handle,
                             int                   n,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* y,
                             int                   incy,
                             hipblasComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cdotc((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy,
                                                  (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotu(hipblasHandle_t       handle,
                             int                   n,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* y,
                             int                   incy,
                             hipblasComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cdotu((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy,
                                                  (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotc(hipblasHandle_t             handle,
                             int                         n,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* y,
                             int                         incy,
                             hipblasDoubleComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdotc((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy,
                                                  (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotu(hipblasHandle_t             handle,
                             int                         n,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* y,
                             int                         incy,
                             hipblasDoubleComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdotu((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy,
                                                  (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotc_v2(hipblasHandle_t   handle,
                                int               n,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* y,
                                int               incy,
                                hipComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cdotc((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy,
                                                  (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotu_v2(hipblasHandle_t   handle,
                                int               n,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* y,
                                int               incy,
                                hipComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cdotu((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy,
                                                  (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotc_v2(hipblasHandle_t         handle,
                                int                     n,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* y,
                                int                     incy,
                                hipDoubleComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdotc((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy,
                                                  (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotu_v2(hipblasHandle_t         handle,
                                int                     n,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* y,
                                int                     incy,
                                hipDoubleComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdotu((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy,
                                                  (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// dot_64
hipblasStatus_t hipblasHdot_64(hipblasHandle_t    handle,
                               int64_t            n,
                               const hipblasHalf* x,
                               int64_t            incx,
                               const hipblasHalf* y,
                               int64_t            incy,
                               hipblasHalf*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_hdot_64((rocblas_handle)handle,
                                                    n,
                                                    (rocblas_half*)x,
                                                    incx,
                                                    (rocblas_half*)y,
                                                    incy,
                                                    (rocblas_half*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasBfdot_64(hipblasHandle_t        handle,
                                int64_t                n,
                                const hipblasBfloat16* x,
                                int64_t                incx,
                                const hipblasBfloat16* y,
                                int64_t                incy,
                                hipblasBfloat16*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_bfdot_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_bfloat16*)x,
                                                     incx,
                                                     (rocblas_bfloat16*)y,
                                                     incy,
                                                     (rocblas_bfloat16*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasSdot_64(hipblasHandle_t handle,
                               int64_t         n,
                               const float*    x,
                               int64_t         incx,
                               const float*    y,
                               int64_t         incy,
                               float*          result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_sdot_64((rocblas_handle)handle, n, x, incx, y, incy, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDdot_64(hipblasHandle_t handle,
                               int64_t         n,
                               const double*   x,
                               int64_t         incx,
                               const double*   y,
                               int64_t         incy,
                               double*         result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ddot_64((rocblas_handle)handle, n, x, incx, y, incy, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotc_64(hipblasHandle_t       handle,
                                int64_t               n,
                                const hipblasComplex* x,
                                int64_t               incx,
                                const hipblasComplex* y,
                                int64_t               incy,
                                hipblasComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cdotc_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_float_complex*)x,
                                                     incx,
                                                     (rocblas_float_complex*)y,
                                                     incy,
                                                     (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotu_64(hipblasHandle_t       handle,
                                int64_t               n,
                                const hipblasComplex* x,
                                int64_t               incx,
                                const hipblasComplex* y,
                                int64_t               incy,
                                hipblasComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cdotu_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_float_complex*)x,
                                                     incx,
                                                     (rocblas_float_complex*)y,
                                                     incy,
                                                     (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotc_64(hipblasHandle_t             handle,
                                int64_t                     n,
                                const hipblasDoubleComplex* x,
                                int64_t                     incx,
                                const hipblasDoubleComplex* y,
                                int64_t                     incy,
                                hipblasDoubleComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdotc_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_double_complex*)x,
                                                     incx,
                                                     (rocblas_double_complex*)y,
                                                     incy,
                                                     (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotu_64(hipblasHandle_t             handle,
                                int64_t                     n,
                                const hipblasDoubleComplex* x,
                                int64_t                     incx,
                                const hipblasDoubleComplex* y,
                                int64_t                     incy,
                                hipblasDoubleComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdotu_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_double_complex*)x,
                                                     incx,
                                                     (rocblas_double_complex*)y,
                                                     incy,
                                                     (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotc_v2_64(hipblasHandle_t   handle,
                                   int64_t           n,
                                   const hipComplex* x,
                                   int64_t           incx,
                                   const hipComplex* y,
                                   int64_t           incy,
                                   hipComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cdotc_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_float_complex*)x,
                                                     incx,
                                                     (rocblas_float_complex*)y,
                                                     incy,
                                                     (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotu_v2_64(hipblasHandle_t   handle,
                                   int64_t           n,
                                   const hipComplex* x,
                                   int64_t           incx,
                                   const hipComplex* y,
                                   int64_t           incy,
                                   hipComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cdotu_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_float_complex*)x,
                                                     incx,
                                                     (rocblas_float_complex*)y,
                                                     incy,
                                                     (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotc_v2_64(hipblasHandle_t         handle,
                                   int64_t                 n,
                                   const hipDoubleComplex* x,
                                   int64_t                 incx,
                                   const hipDoubleComplex* y,
                                   int64_t                 incy,
                                   hipDoubleComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdotc_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_double_complex*)x,
                                                     incx,
                                                     (rocblas_double_complex*)y,
                                                     incy,
                                                     (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotu_v2_64(hipblasHandle_t         handle,
                                   int64_t                 n,
                                   const hipDoubleComplex* x,
                                   int64_t                 incx,
                                   const hipDoubleComplex* y,
                                   int64_t                 incy,
                                   hipDoubleComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdotu_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_double_complex*)x,
                                                     incx,
                                                     (rocblas_double_complex*)y,
                                                     incy,
                                                     (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// dot_batched
hipblasStatus_t hipblasHdotBatched(hipblasHandle_t          handle,
                                   int                      n,
                                   const hipblasHalf* const x[],
                                   int                      incx,
                                   const hipblasHalf* const y[],
                                   int                      incy,
                                   int                      batchCount,
                                   hipblasHalf*             result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_hdot_batched((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_half* const*)x,
                                                         incx,
                                                         (rocblas_half* const*)y,
                                                         incy,
                                                         batchCount,
                                                         (rocblas_half*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasBfdotBatched(hipblasHandle_t              handle,
                                    int                          n,
                                    const hipblasBfloat16* const x[],
                                    int                          incx,
                                    const hipblasBfloat16* const y[],
                                    int                          incy,
                                    int                          batchCount,
                                    hipblasBfloat16*             result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_bfdot_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_bfloat16* const*)x,
                                                          incx,
                                                          (rocblas_bfloat16* const*)y,
                                                          incy,
                                                          batchCount,
                                                          (rocblas_bfloat16*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasSdotBatched(hipblasHandle_t    handle,
                                   int                n,
                                   const float* const x[],
                                   int                incx,
                                   const float* const y[],
                                   int                incy,
                                   int                batchCount,
                                   float*             result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_sdot_batched((rocblas_handle)handle, n, x, incx, y, incy, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDdotBatched(hipblasHandle_t     handle,
                                   int                 n,
                                   const double* const x[],
                                   int                 incx,
                                   const double* const y[],
                                   int                 incy,
                                   int                 batchCount,
                                   double*             result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ddot_batched((rocblas_handle)handle, n, x, incx, y, incy, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotcBatched(hipblasHandle_t             handle,
                                    int                         n,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex* const y[],
                                    int                         incy,
                                    int                         batchCount,
                                    hipblasComplex*             result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cdotc_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          batchCount,
                                                          (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotuBatched(hipblasHandle_t             handle,
                                    int                         n,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex* const y[],
                                    int                         incy,
                                    int                         batchCount,
                                    hipblasComplex*             result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cdotu_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          batchCount,
                                                          (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotcBatched(hipblasHandle_t                   handle,
                                    int                               n,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex* const y[],
                                    int                               incy,
                                    int                               batchCount,
                                    hipblasDoubleComplex*             result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdotc_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          batchCount,
                                                          (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotuBatched(hipblasHandle_t                   handle,
                                    int                               n,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex* const y[],
                                    int                               incy,
                                    int                               batchCount,
                                    hipblasDoubleComplex*             result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdotu_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          batchCount,
                                                          (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotcBatched_v2(hipblasHandle_t         handle,
                                       int                     n,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex* const y[],
                                       int                     incy,
                                       int                     batchCount,
                                       hipComplex*             result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cdotc_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          batchCount,
                                                          (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotuBatched_v2(hipblasHandle_t         handle,
                                       int                     n,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex* const y[],
                                       int                     incy,
                                       int                     batchCount,
                                       hipComplex*             result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cdotu_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          batchCount,
                                                          (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotcBatched_v2(hipblasHandle_t               handle,
                                       int                           n,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex* const y[],
                                       int                           incy,
                                       int                           batchCount,
                                       hipDoubleComplex*             result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdotc_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          batchCount,
                                                          (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotuBatched_v2(hipblasHandle_t               handle,
                                       int                           n,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex* const y[],
                                       int                           incy,
                                       int                           batchCount,
                                       hipDoubleComplex*             result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdotu_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          batchCount,
                                                          (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// dot_batched_64
hipblasStatus_t hipblasHdotBatched_64(hipblasHandle_t          handle,
                                      int64_t                  n,
                                      const hipblasHalf* const x[],
                                      int64_t                  incx,
                                      const hipblasHalf* const y[],
                                      int64_t                  incy,
                                      int64_t                  batchCount,
                                      hipblasHalf*             result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_hdot_batched_64((rocblas_handle)handle,
                                                            n,
                                                            (rocblas_half* const*)x,
                                                            incx,
                                                            (rocblas_half* const*)y,
                                                            incy,
                                                            batchCount,
                                                            (rocblas_half*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasBfdotBatched_64(hipblasHandle_t              handle,
                                       int64_t                      n,
                                       const hipblasBfloat16* const x[],
                                       int64_t                      incx,
                                       const hipblasBfloat16* const y[],
                                       int64_t                      incy,
                                       int64_t                      batchCount,
                                       hipblasBfloat16*             result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_bfdot_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_bfloat16* const*)x,
                                                             incx,
                                                             (rocblas_bfloat16* const*)y,
                                                             incy,
                                                             batchCount,
                                                             (rocblas_bfloat16*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasSdotBatched_64(hipblasHandle_t    handle,
                                      int64_t            n,
                                      const float* const x[],
                                      int64_t            incx,
                                      const float* const y[],
                                      int64_t            incy,
                                      int64_t            batchCount,
                                      float*             result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_sdot_batched_64((rocblas_handle)handle, n, x, incx, y, incy, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDdotBatched_64(hipblasHandle_t     handle,
                                      int64_t             n,
                                      const double* const x[],
                                      int64_t             incx,
                                      const double* const y[],
                                      int64_t             incy,
                                      int64_t             batchCount,
                                      double*             result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ddot_batched_64((rocblas_handle)handle, n, x, incx, y, incy, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotcBatched_64(hipblasHandle_t             handle,
                                       int64_t                     n,
                                       const hipblasComplex* const x[],
                                       int64_t                     incx,
                                       const hipblasComplex* const y[],
                                       int64_t                     incy,
                                       int64_t                     batchCount,
                                       hipblasComplex*             result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cdotc_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_float_complex**)x,
                                                             incx,
                                                             (rocblas_float_complex**)y,
                                                             incy,
                                                             batchCount,
                                                             (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotuBatched_64(hipblasHandle_t             handle,
                                       int64_t                     n,
                                       const hipblasComplex* const x[],
                                       int64_t                     incx,
                                       const hipblasComplex* const y[],
                                       int64_t                     incy,
                                       int64_t                     batchCount,
                                       hipblasComplex*             result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cdotu_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_float_complex**)x,
                                                             incx,
                                                             (rocblas_float_complex**)y,
                                                             incy,
                                                             batchCount,
                                                             (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotcBatched_64(hipblasHandle_t                   handle,
                                       int64_t                           n,
                                       const hipblasDoubleComplex* const x[],
                                       int64_t                           incx,
                                       const hipblasDoubleComplex* const y[],
                                       int64_t                           incy,
                                       int64_t                           batchCount,
                                       hipblasDoubleComplex*             result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdotc_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_double_complex**)x,
                                                             incx,
                                                             (rocblas_double_complex**)y,
                                                             incy,
                                                             batchCount,
                                                             (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotuBatched_64(hipblasHandle_t                   handle,
                                       int64_t                           n,
                                       const hipblasDoubleComplex* const x[],
                                       int64_t                           incx,
                                       const hipblasDoubleComplex* const y[],
                                       int64_t                           incy,
                                       int64_t                           batchCount,
                                       hipblasDoubleComplex*             result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdotu_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_double_complex**)x,
                                                             incx,
                                                             (rocblas_double_complex**)y,
                                                             incy,
                                                             batchCount,
                                                             (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotcBatched_v2_64(hipblasHandle_t         handle,
                                          int64_t                 n,
                                          const hipComplex* const x[],
                                          int64_t                 incx,
                                          const hipComplex* const y[],
                                          int64_t                 incy,
                                          int64_t                 batchCount,
                                          hipComplex*             result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cdotc_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_float_complex**)x,
                                                             incx,
                                                             (rocblas_float_complex**)y,
                                                             incy,
                                                             batchCount,
                                                             (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotuBatched_v2_64(hipblasHandle_t         handle,
                                          int64_t                 n,
                                          const hipComplex* const x[],
                                          int64_t                 incx,
                                          const hipComplex* const y[],
                                          int64_t                 incy,
                                          int64_t                 batchCount,
                                          hipComplex*             result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cdotu_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_float_complex**)x,
                                                             incx,
                                                             (rocblas_float_complex**)y,
                                                             incy,
                                                             batchCount,
                                                             (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotcBatched_v2_64(hipblasHandle_t               handle,
                                          int64_t                       n,
                                          const hipDoubleComplex* const x[],
                                          int64_t                       incx,
                                          const hipDoubleComplex* const y[],
                                          int64_t                       incy,
                                          int64_t                       batchCount,
                                          hipDoubleComplex*             result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdotc_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_double_complex**)x,
                                                             incx,
                                                             (rocblas_double_complex**)y,
                                                             incy,
                                                             batchCount,
                                                             (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotuBatched_v2_64(hipblasHandle_t               handle,
                                          int64_t                       n,
                                          const hipDoubleComplex* const x[],
                                          int64_t                       incx,
                                          const hipDoubleComplex* const y[],
                                          int64_t                       incy,
                                          int64_t                       batchCount,
                                          hipDoubleComplex*             result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdotu_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_double_complex**)x,
                                                             incx,
                                                             (rocblas_double_complex**)y,
                                                             incy,
                                                             batchCount,
                                                             (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// dot_strided_batched
hipblasStatus_t hipblasHdotStridedBatched(hipblasHandle_t    handle,
                                          int                n,
                                          const hipblasHalf* x,
                                          int                incx,
                                          hipblasStride      stridex,
                                          const hipblasHalf* y,
                                          int                incy,
                                          hipblasStride      stridey,
                                          int                batchCount,
                                          hipblasHalf*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_hdot_strided_batched((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_half*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_half*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount,
                                                                 (rocblas_half*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasBfdotStridedBatched(hipblasHandle_t        handle,
                                           int                    n,
                                           const hipblasBfloat16* x,
                                           int                    incx,
                                           hipblasStride          stridex,
                                           const hipblasBfloat16* y,
                                           int                    incy,
                                           hipblasStride          stridey,
                                           int                    batchCount,
                                           hipblasBfloat16*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_bfdot_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_bfloat16*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_bfloat16*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount,
                                                                  (rocblas_bfloat16*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasSdotStridedBatched(hipblasHandle_t handle,
                                          int             n,
                                          const float*    x,
                                          int             incx,
                                          hipblasStride   stridex,
                                          const float*    y,
                                          int             incy,
                                          hipblasStride   stridey,
                                          int             batchCount,
                                          float*          result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sdot_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDdotStridedBatched(hipblasHandle_t handle,
                                          int             n,
                                          const double*   x,
                                          int             incx,
                                          hipblasStride   stridex,
                                          const double*   y,
                                          int             incy,
                                          hipblasStride   stridey,
                                          int             batchCount,
                                          double*         result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ddot_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotcStridedBatched(hipblasHandle_t       handle,
                                           int                   n,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           int                   batchCount,
                                           hipblasComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cdotc_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount,
                                                                  (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotuStridedBatched(hipblasHandle_t       handle,
                                           int                   n,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           int                   batchCount,
                                           hipblasComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cdotu_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount,
                                                                  (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotcStridedBatched(hipblasHandle_t             handle,
                                           int                         n,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           int                         batchCount,
                                           hipblasDoubleComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdotc_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount,
                                                                  (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotuStridedBatched(hipblasHandle_t             handle,
                                           int                         n,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           int                         batchCount,
                                           hipblasDoubleComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdotu_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount,
                                                                  (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotcStridedBatched_v2(hipblasHandle_t   handle,
                                              int               n,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              int               batchCount,
                                              hipComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cdotc_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount,
                                                                  (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotuStridedBatched_v2(hipblasHandle_t   handle,
                                              int               n,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              int               batchCount,
                                              hipComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cdotu_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount,
                                                                  (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotcStridedBatched_v2(hipblasHandle_t         handle,
                                              int                     n,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              int                     batchCount,
                                              hipDoubleComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdotc_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount,
                                                                  (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotuStridedBatched_v2(hipblasHandle_t         handle,
                                              int                     n,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              int                     batchCount,
                                              hipDoubleComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdotu_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount,
                                                                  (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// dot_strided_batched_64
hipblasStatus_t hipblasHdotStridedBatched_64(hipblasHandle_t    handle,
                                             int64_t            n,
                                             const hipblasHalf* x,
                                             int64_t            incx,
                                             hipblasStride      stridex,
                                             const hipblasHalf* y,
                                             int64_t            incy,
                                             hipblasStride      stridey,
                                             int64_t            batchCount,
                                             hipblasHalf*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_hdot_strided_batched_64((rocblas_handle)handle,
                                                                    n,
                                                                    (rocblas_half*)x,
                                                                    incx,
                                                                    stridex,
                                                                    (rocblas_half*)y,
                                                                    incy,
                                                                    stridey,
                                                                    batchCount,
                                                                    (rocblas_half*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasBfdotStridedBatched_64(hipblasHandle_t        handle,
                                              int64_t                n,
                                              const hipblasBfloat16* x,
                                              int64_t                incx,
                                              hipblasStride          stridex,
                                              const hipblasBfloat16* y,
                                              int64_t                incy,
                                              hipblasStride          stridey,
                                              int64_t                batchCount,
                                              hipblasBfloat16*       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_bfdot_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_bfloat16*)x,
                                                                     incx,
                                                                     stridex,
                                                                     (rocblas_bfloat16*)y,
                                                                     incy,
                                                                     stridey,
                                                                     batchCount,
                                                                     (rocblas_bfloat16*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasSdotStridedBatched_64(hipblasHandle_t handle,
                                             int64_t         n,
                                             const float*    x,
                                             int64_t         incx,
                                             hipblasStride   stridex,
                                             const float*    y,
                                             int64_t         incy,
                                             hipblasStride   stridey,
                                             int64_t         batchCount,
                                             float*          result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sdot_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDdotStridedBatched_64(hipblasHandle_t handle,
                                             int64_t         n,
                                             const double*   x,
                                             int64_t         incx,
                                             hipblasStride   stridex,
                                             const double*   y,
                                             int64_t         incy,
                                             hipblasStride   stridey,
                                             int64_t         batchCount,
                                             double*         result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ddot_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotcStridedBatched_64(hipblasHandle_t       handle,
                                              int64_t               n,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              const hipblasComplex* y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              int64_t               batchCount,
                                              hipblasComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_cdotc_strided_batched_64((rocblas_handle)handle,
                                         n,
                                         (rocblas_float_complex*)x,
                                         incx,
                                         stridex,
                                         (rocblas_float_complex*)y,
                                         incy,
                                         stridey,
                                         batchCount,
                                         (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotuStridedBatched_64(hipblasHandle_t       handle,
                                              int64_t               n,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              const hipblasComplex* y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              int64_t               batchCount,
                                              hipblasComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_cdotu_strided_batched_64((rocblas_handle)handle,
                                         n,
                                         (rocblas_float_complex*)x,
                                         incx,
                                         stridex,
                                         (rocblas_float_complex*)y,
                                         incy,
                                         stridey,
                                         batchCount,
                                         (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotcStridedBatched_64(hipblasHandle_t             handle,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              const hipblasDoubleComplex* y,
                                              int64_t                     incy,
                                              hipblasStride               stridey,
                                              int64_t                     batchCount,
                                              hipblasDoubleComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_zdotc_strided_batched_64((rocblas_handle)handle,
                                         n,
                                         (rocblas_double_complex*)x,
                                         incx,
                                         stridex,
                                         (rocblas_double_complex*)y,
                                         incy,
                                         stridey,
                                         batchCount,
                                         (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotuStridedBatched_64(hipblasHandle_t             handle,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              const hipblasDoubleComplex* y,
                                              int64_t                     incy,
                                              hipblasStride               stridey,
                                              int64_t                     batchCount,
                                              hipblasDoubleComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_zdotu_strided_batched_64((rocblas_handle)handle,
                                         n,
                                         (rocblas_double_complex*)x,
                                         incx,
                                         stridex,
                                         (rocblas_double_complex*)y,
                                         incy,
                                         stridey,
                                         batchCount,
                                         (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotcStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 int64_t           n,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 const hipComplex* y,
                                                 int64_t           incy,
                                                 hipblasStride     stridey,
                                                 int64_t           batchCount,
                                                 hipComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_cdotc_strided_batched_64((rocblas_handle)handle,
                                         n,
                                         (rocblas_float_complex*)x,
                                         incx,
                                         stridex,
                                         (rocblas_float_complex*)y,
                                         incy,
                                         stridey,
                                         batchCount,
                                         (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCdotuStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 int64_t           n,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 const hipComplex* y,
                                                 int64_t           incy,
                                                 hipblasStride     stridey,
                                                 int64_t           batchCount,
                                                 hipComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_cdotu_strided_batched_64((rocblas_handle)handle,
                                         n,
                                         (rocblas_float_complex*)x,
                                         incx,
                                         stridex,
                                         (rocblas_float_complex*)y,
                                         incy,
                                         stridey,
                                         batchCount,
                                         (rocblas_float_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotcStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 const hipDoubleComplex* y,
                                                 int64_t                 incy,
                                                 hipblasStride           stridey,
                                                 int64_t                 batchCount,
                                                 hipDoubleComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_zdotc_strided_batched_64((rocblas_handle)handle,
                                         n,
                                         (rocblas_double_complex*)x,
                                         incx,
                                         stridex,
                                         (rocblas_double_complex*)y,
                                         incy,
                                         stridey,
                                         batchCount,
                                         (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdotuStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 const hipDoubleComplex* y,
                                                 int64_t                 incy,
                                                 hipblasStride           stridey,
                                                 int64_t                 batchCount,
                                                 hipDoubleComplex*       result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_zdotu_strided_batched_64((rocblas_handle)handle,
                                         n,
                                         (rocblas_double_complex*)x,
                                         incx,
                                         stridex,
                                         (rocblas_double_complex*)y,
                                         incy,
                                         stridey,
                                         batchCount,
                                         (rocblas_double_complex*)result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// nrm2
hipblasStatus_t hipblasSnrm2(hipblasHandle_t handle, int n, const float* x, int incx, float* result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_snrm2((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasDnrm2(hipblasHandle_t handle, int n, const double* x, int incx, double* result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dnrm2((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasScnrm2(hipblasHandle_t handle, int n, const hipblasComplex* x, int incx, float* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_scnrm2((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDznrm2(
    hipblasHandle_t handle, int n, const hipblasDoubleComplex* x, int incx, double* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dznrm2((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasScnrm2_v2(hipblasHandle_t handle, int n, const hipComplex* x, int incx, float* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_scnrm2((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDznrm2_v2(
    hipblasHandle_t handle, int n, const hipDoubleComplex* x, int incx, double* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dznrm2((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// nrm2_64
hipblasStatus_t
    hipblasSnrm2_64(hipblasHandle_t handle, int64_t n, const float* x, int64_t incx, float* result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_snrm2_64((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDnrm2_64(
    hipblasHandle_t handle, int64_t n, const double* x, int64_t incx, double* result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dnrm2_64((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasScnrm2_64(
    hipblasHandle_t handle, int64_t n, const hipblasComplex* x, int64_t incx, float* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_scnrm2_64((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDznrm2_64(
    hipblasHandle_t handle, int64_t n, const hipblasDoubleComplex* x, int64_t incx, double* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dznrm2_64((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasScnrm2_v2_64(
    hipblasHandle_t handle, int64_t n, const hipComplex* x, int64_t incx, float* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_scnrm2_64((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDznrm2_v2_64(
    hipblasHandle_t handle, int64_t n, const hipDoubleComplex* x, int64_t incx, double* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dznrm2_64((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// nrm2_batched
hipblasStatus_t hipblasSnrm2Batched(
    hipblasHandle_t handle, int n, const float* const x[], int incx, int batchCount, float* result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_snrm2_batched((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDnrm2Batched(hipblasHandle_t     handle,
                                    int                 n,
                                    const double* const x[],
                                    int                 incx,
                                    int                 batchCount,
                                    double*             result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dnrm2_batched((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasScnrm2Batched(hipblasHandle_t             handle,
                                     int                         n,
                                     const hipblasComplex* const x[],
                                     int                         incx,
                                     int                         batchCount,
                                     float*                      result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_scnrm2_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDznrm2Batched(hipblasHandle_t                   handle,
                                     int                               n,
                                     const hipblasDoubleComplex* const x[],
                                     int                               incx,
                                     int                               batchCount,
                                     double*                           result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dznrm2_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasScnrm2Batched_v2(hipblasHandle_t         handle,
                                        int                     n,
                                        const hipComplex* const x[],
                                        int                     incx,
                                        int                     batchCount,
                                        float*                  result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_scnrm2_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDznrm2Batched_v2(hipblasHandle_t               handle,
                                        int                           n,
                                        const hipDoubleComplex* const x[],
                                        int                           incx,
                                        int                           batchCount,
                                        double*                       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dznrm2_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// nrm2_batched_64
hipblasStatus_t hipblasSnrm2Batched_64(hipblasHandle_t    handle,
                                       int64_t            n,
                                       const float* const x[],
                                       int64_t            incx,
                                       int64_t            batchCount,
                                       float*             result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_snrm2_batched_64((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDnrm2Batched_64(hipblasHandle_t     handle,
                                       int64_t             n,
                                       const double* const x[],
                                       int64_t             incx,
                                       int64_t             batchCount,
                                       double*             result)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dnrm2_batched_64((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasScnrm2Batched_64(hipblasHandle_t             handle,
                                        int64_t                     n,
                                        const hipblasComplex* const x[],
                                        int64_t                     incx,
                                        int64_t                     batchCount,
                                        float*                      result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_scnrm2_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDznrm2Batched_64(hipblasHandle_t                   handle,
                                        int64_t                           n,
                                        const hipblasDoubleComplex* const x[],
                                        int64_t                           incx,
                                        int64_t                           batchCount,
                                        double*                           result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dznrm2_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasScnrm2Batched_v2_64(hipblasHandle_t         handle,
                                           int64_t                 n,
                                           const hipComplex* const x[],
                                           int64_t                 incx,
                                           int64_t                 batchCount,
                                           float*                  result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_scnrm2_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDznrm2Batched_v2_64(hipblasHandle_t               handle,
                                           int64_t                       n,
                                           const hipDoubleComplex* const x[],
                                           int64_t                       incx,
                                           int64_t                       batchCount,
                                           double*                       result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dznrm2_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// nrm2_strided_batched
hipblasStatus_t hipblasSnrm2StridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const float*    x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           int             batchCount,
                                           float*          result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_snrm2_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDnrm2StridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const double*   x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           int             batchCount,
                                           double*         result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dnrm2_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasScnrm2StridedBatched(hipblasHandle_t       handle,
                                            int                   n,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            hipblasStride         stridex,
                                            int                   batchCount,
                                            float*                result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_scnrm2_strided_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDznrm2StridedBatched(hipblasHandle_t             handle,
                                            int                         n,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            hipblasStride               stridex,
                                            int                         batchCount,
                                            double*                     result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dznrm2_strided_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasScnrm2StridedBatched_v2(hipblasHandle_t   handle,
                                               int               n,
                                               const hipComplex* x,
                                               int               incx,
                                               hipblasStride     stridex,
                                               int               batchCount,
                                               float*            result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_scnrm2_strided_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDznrm2StridedBatched_v2(hipblasHandle_t         handle,
                                               int                     n,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               hipblasStride           stridex,
                                               int                     batchCount,
                                               double*                 result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dznrm2_strided_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// nrm2_strided_batched_64
hipblasStatus_t hipblasSnrm2StridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const float*    x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              int64_t         batchCount,
                                              float*          result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_snrm2_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDnrm2StridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const double*   x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              int64_t         batchCount,
                                              double*         result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dnrm2_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasScnrm2StridedBatched_64(hipblasHandle_t       handle,
                                               int64_t               n,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               hipblasStride         stridex,
                                               int64_t               batchCount,
                                               float*                result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_scnrm2_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDznrm2StridedBatched_64(hipblasHandle_t             handle,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               hipblasStride               stridex,
                                               int64_t                     batchCount,
                                               double*                     result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dznrm2_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasScnrm2StridedBatched_v2_64(hipblasHandle_t   handle,
                                                  int64_t           n,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  hipblasStride     stridex,
                                                  int64_t           batchCount,
                                                  float*            result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_scnrm2_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDznrm2StridedBatched_v2_64(hipblasHandle_t         handle,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  hipblasStride           stridex,
                                                  int64_t                 batchCount,
                                                  double*                 result)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dznrm2_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rot
hipblasStatus_t hipblasSrot(hipblasHandle_t handle,
                            int             n,
                            float*          x,
                            int             incx,
                            float*          y,
                            int             incy,
                            const float*    c,
                            const float*    s)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_srot((rocblas_handle)handle, n, x, incx, y, incy, c, s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrot(hipblasHandle_t handle,
                            int             n,
                            double*         x,
                            int             incx,
                            double*         y,
                            int             incy,
                            const double*   c,
                            const double*   s)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_drot((rocblas_handle)handle, n, x, incx, y, incy, c, s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrot(hipblasHandle_t       handle,
                            int                   n,
                            hipblasComplex*       x,
                            int                   incx,
                            hipblasComplex*       y,
                            int                   incy,
                            const float*          c,
                            const hipblasComplex* s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crot((rocblas_handle)handle,
                                                 n,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)y,
                                                 incy,
                                                 c,
                                                 (rocblas_float_complex*)s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsrot(hipblasHandle_t handle,
                             int             n,
                             hipblasComplex* x,
                             int             incx,
                             hipblasComplex* y,
                             int             incy,
                             const float*    c,
                             const float*    s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csrot((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy,
                                                  c,
                                                  s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrot(hipblasHandle_t             handle,
                            int                         n,
                            hipblasDoubleComplex*       x,
                            int                         incx,
                            hipblasDoubleComplex*       y,
                            int                         incy,
                            const double*               c,
                            const hipblasDoubleComplex* s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrot((rocblas_handle)handle,
                                                 n,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)y,
                                                 incy,
                                                 c,
                                                 (rocblas_double_complex*)s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdrot(hipblasHandle_t       handle,
                             int                   n,
                             hipblasDoubleComplex* x,
                             int                   incx,
                             hipblasDoubleComplex* y,
                             int                   incy,
                             const double*         c,
                             const double*         s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdrot((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy,
                                                  c,
                                                  s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrot_v2(hipblasHandle_t   handle,
                               int               n,
                               hipComplex*       x,
                               int               incx,
                               hipComplex*       y,
                               int               incy,
                               const float*      c,
                               const hipComplex* s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crot((rocblas_handle)handle,
                                                 n,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)y,
                                                 incy,
                                                 c,
                                                 (rocblas_float_complex*)s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsrot_v2(hipblasHandle_t handle,
                                int             n,
                                hipComplex*     x,
                                int             incx,
                                hipComplex*     y,
                                int             incy,
                                const float*    c,
                                const float*    s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csrot((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy,
                                                  c,
                                                  s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrot_v2(hipblasHandle_t         handle,
                               int                     n,
                               hipDoubleComplex*       x,
                               int                     incx,
                               hipDoubleComplex*       y,
                               int                     incy,
                               const double*           c,
                               const hipDoubleComplex* s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrot((rocblas_handle)handle,
                                                 n,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)y,
                                                 incy,
                                                 c,
                                                 (rocblas_double_complex*)s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdrot_v2(hipblasHandle_t   handle,
                                int               n,
                                hipDoubleComplex* x,
                                int               incx,
                                hipDoubleComplex* y,
                                int               incy,
                                const double*     c,
                                const double*     s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdrot((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy,
                                                  c,
                                                  s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rot_64
hipblasStatus_t hipblasSrot_64(hipblasHandle_t handle,
                               int64_t         n,
                               float*          x,
                               int64_t         incx,
                               float*          y,
                               int64_t         incy,
                               const float*    c,
                               const float*    s)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_srot_64((rocblas_handle)handle, n, x, incx, y, incy, c, s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrot_64(hipblasHandle_t handle,
                               int64_t         n,
                               double*         x,
                               int64_t         incx,
                               double*         y,
                               int64_t         incy,
                               const double*   c,
                               const double*   s)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_drot_64((rocblas_handle)handle, n, x, incx, y, incy, c, s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrot_64(hipblasHandle_t       handle,
                               int64_t               n,
                               hipblasComplex*       x,
                               int64_t               incx,
                               hipblasComplex*       y,
                               int64_t               incy,
                               const float*          c,
                               const hipblasComplex* s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crot_64((rocblas_handle)handle,
                                                    n,
                                                    (rocblas_float_complex*)x,
                                                    incx,
                                                    (rocblas_float_complex*)y,
                                                    incy,
                                                    c,
                                                    (rocblas_float_complex*)s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsrot_64(hipblasHandle_t handle,
                                int64_t         n,
                                hipblasComplex* x,
                                int64_t         incx,
                                hipblasComplex* y,
                                int64_t         incy,
                                const float*    c,
                                const float*    s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csrot_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_float_complex*)x,
                                                     incx,
                                                     (rocblas_float_complex*)y,
                                                     incy,
                                                     c,
                                                     s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrot_64(hipblasHandle_t             handle,
                               int64_t                     n,
                               hipblasDoubleComplex*       x,
                               int64_t                     incx,
                               hipblasDoubleComplex*       y,
                               int64_t                     incy,
                               const double*               c,
                               const hipblasDoubleComplex* s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrot_64((rocblas_handle)handle,
                                                    n,
                                                    (rocblas_double_complex*)x,
                                                    incx,
                                                    (rocblas_double_complex*)y,
                                                    incy,
                                                    c,
                                                    (rocblas_double_complex*)s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdrot_64(hipblasHandle_t       handle,
                                int64_t               n,
                                hipblasDoubleComplex* x,
                                int64_t               incx,
                                hipblasDoubleComplex* y,
                                int64_t               incy,
                                const double*         c,
                                const double*         s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdrot_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_double_complex*)x,
                                                     incx,
                                                     (rocblas_double_complex*)y,
                                                     incy,
                                                     c,
                                                     s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrot_v2_64(hipblasHandle_t   handle,
                                  int64_t           n,
                                  hipComplex*       x,
                                  int64_t           incx,
                                  hipComplex*       y,
                                  int64_t           incy,
                                  const float*      c,
                                  const hipComplex* s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crot_64((rocblas_handle)handle,
                                                    n,
                                                    (rocblas_float_complex*)x,
                                                    incx,
                                                    (rocblas_float_complex*)y,
                                                    incy,
                                                    c,
                                                    (rocblas_float_complex*)s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsrot_v2_64(hipblasHandle_t handle,
                                   int64_t         n,
                                   hipComplex*     x,
                                   int64_t         incx,
                                   hipComplex*     y,
                                   int64_t         incy,
                                   const float*    c,
                                   const float*    s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csrot_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_float_complex*)x,
                                                     incx,
                                                     (rocblas_float_complex*)y,
                                                     incy,
                                                     c,
                                                     s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrot_v2_64(hipblasHandle_t         handle,
                                  int64_t                 n,
                                  hipDoubleComplex*       x,
                                  int64_t                 incx,
                                  hipDoubleComplex*       y,
                                  int64_t                 incy,
                                  const double*           c,
                                  const hipDoubleComplex* s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrot_64((rocblas_handle)handle,
                                                    n,
                                                    (rocblas_double_complex*)x,
                                                    incx,
                                                    (rocblas_double_complex*)y,
                                                    incy,
                                                    c,
                                                    (rocblas_double_complex*)s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdrot_v2_64(hipblasHandle_t   handle,
                                   int64_t           n,
                                   hipDoubleComplex* x,
                                   int64_t           incx,
                                   hipDoubleComplex* y,
                                   int64_t           incy,
                                   const double*     c,
                                   const double*     s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdrot_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_double_complex*)x,
                                                     incx,
                                                     (rocblas_double_complex*)y,
                                                     incy,
                                                     c,
                                                     s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rot_batched
hipblasStatus_t hipblasSrotBatched(hipblasHandle_t handle,
                                   int             n,
                                   float* const    x[],
                                   int             incx,
                                   float* const    y[],
                                   int             incy,
                                   const float*    c,
                                   const float*    s,
                                   int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_srot_batched((rocblas_handle)handle, n, x, incx, y, incy, c, s, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotBatched(hipblasHandle_t handle,
                                   int             n,
                                   double* const   x[],
                                   int             incx,
                                   double* const   y[],
                                   int             incy,
                                   const double*   c,
                                   const double*   s,
                                   int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_drot_batched((rocblas_handle)handle, n, x, incx, y, incy, c, s, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrotBatched(hipblasHandle_t       handle,
                                   int                   n,
                                   hipblasComplex* const x[],
                                   int                   incx,
                                   hipblasComplex* const y[],
                                   int                   incy,
                                   const float*          c,
                                   const hipblasComplex* s,
                                   int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crot_batched((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex**)y,
                                                         incy,
                                                         c,
                                                         (rocblas_float_complex*)s,
                                                         batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsrotBatched(hipblasHandle_t       handle,
                                    int                   n,
                                    hipblasComplex* const x[],
                                    int                   incx,
                                    hipblasComplex* const y[],
                                    int                   incy,
                                    const float*          c,
                                    const float*          s,
                                    int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csrot_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          c,
                                                          s,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrotBatched(hipblasHandle_t             handle,
                                   int                         n,
                                   hipblasDoubleComplex* const x[],
                                   int                         incx,
                                   hipblasDoubleComplex* const y[],
                                   int                         incy,
                                   const double*               c,
                                   const hipblasDoubleComplex* s,
                                   int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrot_batched((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex**)y,
                                                         incy,
                                                         c,
                                                         (rocblas_double_complex*)s,
                                                         batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdrotBatched(hipblasHandle_t             handle,
                                    int                         n,
                                    hipblasDoubleComplex* const x[],
                                    int                         incx,
                                    hipblasDoubleComplex* const y[],
                                    int                         incy,
                                    const double*               c,
                                    const double*               s,
                                    int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdrot_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          c,
                                                          s,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrotBatched_v2(hipblasHandle_t   handle,
                                      int               n,
                                      hipComplex* const x[],
                                      int               incx,
                                      hipComplex* const y[],
                                      int               incy,
                                      const float*      c,
                                      const hipComplex* s,
                                      int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crot_batched((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex**)y,
                                                         incy,
                                                         c,
                                                         (rocblas_float_complex*)s,
                                                         batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsrotBatched_v2(hipblasHandle_t   handle,
                                       int               n,
                                       hipComplex* const x[],
                                       int               incx,
                                       hipComplex* const y[],
                                       int               incy,
                                       const float*      c,
                                       const float*      s,
                                       int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csrot_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          c,
                                                          s,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrotBatched_v2(hipblasHandle_t         handle,
                                      int                     n,
                                      hipDoubleComplex* const x[],
                                      int                     incx,
                                      hipDoubleComplex* const y[],
                                      int                     incy,
                                      const double*           c,
                                      const hipDoubleComplex* s,
                                      int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrot_batched((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex**)y,
                                                         incy,
                                                         c,
                                                         (rocblas_double_complex*)s,
                                                         batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdrotBatched_v2(hipblasHandle_t         handle,
                                       int                     n,
                                       hipDoubleComplex* const x[],
                                       int                     incx,
                                       hipDoubleComplex* const y[],
                                       int                     incy,
                                       const double*           c,
                                       const double*           s,
                                       int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdrot_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          c,
                                                          s,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rot_batched_64
hipblasStatus_t hipblasSrotBatched_64(hipblasHandle_t handle,
                                      int64_t         n,
                                      float* const    x[],
                                      int64_t         incx,
                                      float* const    y[],
                                      int64_t         incy,
                                      const float*    c,
                                      const float*    s,
                                      int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_srot_batched_64((rocblas_handle)handle, n, x, incx, y, incy, c, s, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotBatched_64(hipblasHandle_t handle,
                                      int64_t         n,
                                      double* const   x[],
                                      int64_t         incx,
                                      double* const   y[],
                                      int64_t         incy,
                                      const double*   c,
                                      const double*   s,
                                      int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_drot_batched_64((rocblas_handle)handle, n, x, incx, y, incy, c, s, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrotBatched_64(hipblasHandle_t       handle,
                                      int64_t               n,
                                      hipblasComplex* const x[],
                                      int64_t               incx,
                                      hipblasComplex* const y[],
                                      int64_t               incy,
                                      const float*          c,
                                      const hipblasComplex* s,
                                      int64_t               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crot_batched_64((rocblas_handle)handle,
                                                            n,
                                                            (rocblas_float_complex**)x,
                                                            incx,
                                                            (rocblas_float_complex**)y,
                                                            incy,
                                                            c,
                                                            (rocblas_float_complex*)s,
                                                            batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsrotBatched_64(hipblasHandle_t       handle,
                                       int64_t               n,
                                       hipblasComplex* const x[],
                                       int64_t               incx,
                                       hipblasComplex* const y[],
                                       int64_t               incy,
                                       const float*          c,
                                       const float*          s,
                                       int64_t               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csrot_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_float_complex**)x,
                                                             incx,
                                                             (rocblas_float_complex**)y,
                                                             incy,
                                                             c,
                                                             s,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrotBatched_64(hipblasHandle_t             handle,
                                      int64_t                     n,
                                      hipblasDoubleComplex* const x[],
                                      int64_t                     incx,
                                      hipblasDoubleComplex* const y[],
                                      int64_t                     incy,
                                      const double*               c,
                                      const hipblasDoubleComplex* s,
                                      int64_t                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrot_batched_64((rocblas_handle)handle,
                                                            n,
                                                            (rocblas_double_complex**)x,
                                                            incx,
                                                            (rocblas_double_complex**)y,
                                                            incy,
                                                            c,
                                                            (rocblas_double_complex*)s,
                                                            batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdrotBatched_64(hipblasHandle_t             handle,
                                       int64_t                     n,
                                       hipblasDoubleComplex* const x[],
                                       int64_t                     incx,
                                       hipblasDoubleComplex* const y[],
                                       int64_t                     incy,
                                       const double*               c,
                                       const double*               s,
                                       int64_t                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdrot_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_double_complex**)x,
                                                             incx,
                                                             (rocblas_double_complex**)y,
                                                             incy,
                                                             c,
                                                             s,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrotBatched_v2_64(hipblasHandle_t   handle,
                                         int64_t           n,
                                         hipComplex* const x[],
                                         int64_t           incx,
                                         hipComplex* const y[],
                                         int64_t           incy,
                                         const float*      c,
                                         const hipComplex* s,
                                         int64_t           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crot_batched_64((rocblas_handle)handle,
                                                            n,
                                                            (rocblas_float_complex**)x,
                                                            incx,
                                                            (rocblas_float_complex**)y,
                                                            incy,
                                                            c,
                                                            (rocblas_float_complex*)s,
                                                            batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsrotBatched_v2_64(hipblasHandle_t   handle,
                                          int64_t           n,
                                          hipComplex* const x[],
                                          int64_t           incx,
                                          hipComplex* const y[],
                                          int64_t           incy,
                                          const float*      c,
                                          const float*      s,
                                          int64_t           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csrot_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_float_complex**)x,
                                                             incx,
                                                             (rocblas_float_complex**)y,
                                                             incy,
                                                             c,
                                                             s,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrotBatched_v2_64(hipblasHandle_t         handle,
                                         int64_t                 n,
                                         hipDoubleComplex* const x[],
                                         int64_t                 incx,
                                         hipDoubleComplex* const y[],
                                         int64_t                 incy,
                                         const double*           c,
                                         const hipDoubleComplex* s,
                                         int64_t                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrot_batched_64((rocblas_handle)handle,
                                                            n,
                                                            (rocblas_double_complex**)x,
                                                            incx,
                                                            (rocblas_double_complex**)y,
                                                            incy,
                                                            c,
                                                            (rocblas_double_complex*)s,
                                                            batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdrotBatched_v2_64(hipblasHandle_t         handle,
                                          int64_t                 n,
                                          hipDoubleComplex* const x[],
                                          int64_t                 incx,
                                          hipDoubleComplex* const y[],
                                          int64_t                 incy,
                                          const double*           c,
                                          const double*           s,
                                          int64_t                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdrot_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_double_complex**)x,
                                                             incx,
                                                             (rocblas_double_complex**)y,
                                                             incy,
                                                             c,
                                                             s,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rot_strided_batched
hipblasStatus_t hipblasSrotStridedBatched(hipblasHandle_t handle,
                                          int             n,
                                          float*          x,
                                          int             incx,
                                          hipblasStride   stridex,
                                          float*          y,
                                          int             incy,
                                          hipblasStride   stridey,
                                          const float*    c,
                                          const float*    s,
                                          int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_srot_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, c, s, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotStridedBatched(hipblasHandle_t handle,
                                          int             n,
                                          double*         x,
                                          int             incx,
                                          hipblasStride   stridex,
                                          double*         y,
                                          int             incy,
                                          hipblasStride   stridey,
                                          const double*   c,
                                          const double*   s,
                                          int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_drot_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, c, s, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrotStridedBatched(hipblasHandle_t       handle,
                                          int                   n,
                                          hipblasComplex*       x,
                                          int                   incx,
                                          hipblasStride         stridex,
                                          hipblasComplex*       y,
                                          int                   incy,
                                          hipblasStride         stridey,
                                          const float*          c,
                                          const hipblasComplex* s,
                                          int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crot_strided_batched((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 c,
                                                                 (rocblas_float_complex*)s,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsrotStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           hipblasComplex* x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           hipblasComplex* y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           const float*    c,
                                           const float*    s,
                                           int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csrot_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  c,
                                                                  s,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrotStridedBatched(hipblasHandle_t             handle,
                                          int                         n,
                                          hipblasDoubleComplex*       x,
                                          int                         incx,
                                          hipblasStride               stridex,
                                          hipblasDoubleComplex*       y,
                                          int                         incy,
                                          hipblasStride               stridey,
                                          const double*               c,
                                          const hipblasDoubleComplex* s,
                                          int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrot_strided_batched((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 c,
                                                                 (rocblas_double_complex*)s,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdrotStridedBatched(hipblasHandle_t       handle,
                                           int                   n,
                                           hipblasDoubleComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           hipblasDoubleComplex* y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           const double*         c,
                                           const double*         s,
                                           int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdrot_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  c,
                                                                  s,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrotStridedBatched_v2(hipblasHandle_t   handle,
                                             int               n,
                                             hipComplex*       x,
                                             int               incx,
                                             hipblasStride     stridex,
                                             hipComplex*       y,
                                             int               incy,
                                             hipblasStride     stridey,
                                             const float*      c,
                                             const hipComplex* s,
                                             int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crot_strided_batched((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 c,
                                                                 (rocblas_float_complex*)s,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsrotStridedBatched_v2(hipblasHandle_t handle,
                                              int             n,
                                              hipComplex*     x,
                                              int             incx,
                                              hipblasStride   stridex,
                                              hipComplex*     y,
                                              int             incy,
                                              hipblasStride   stridey,
                                              const float*    c,
                                              const float*    s,
                                              int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csrot_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  c,
                                                                  s,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrotStridedBatched_v2(hipblasHandle_t         handle,
                                             int                     n,
                                             hipDoubleComplex*       x,
                                             int                     incx,
                                             hipblasStride           stridex,
                                             hipDoubleComplex*       y,
                                             int                     incy,
                                             hipblasStride           stridey,
                                             const double*           c,
                                             const hipDoubleComplex* s,
                                             int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrot_strided_batched((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 c,
                                                                 (rocblas_double_complex*)s,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdrotStridedBatched_v2(hipblasHandle_t   handle,
                                              int               n,
                                              hipDoubleComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              hipDoubleComplex* y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              const double*     c,
                                              const double*     s,
                                              int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdrot_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  c,
                                                                  s,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rot_strided_batched_64
hipblasStatus_t hipblasSrotStridedBatched_64(hipblasHandle_t handle,
                                             int64_t         n,
                                             float*          x,
                                             int64_t         incx,
                                             hipblasStride   stridex,
                                             float*          y,
                                             int64_t         incy,
                                             hipblasStride   stridey,
                                             const float*    c,
                                             const float*    s,
                                             int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_srot_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, c, s, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotStridedBatched_64(hipblasHandle_t handle,
                                             int64_t         n,
                                             double*         x,
                                             int64_t         incx,
                                             hipblasStride   stridex,
                                             double*         y,
                                             int64_t         incy,
                                             hipblasStride   stridey,
                                             const double*   c,
                                             const double*   s,
                                             int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_drot_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, c, s, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrotStridedBatched_64(hipblasHandle_t       handle,
                                             int64_t               n,
                                             hipblasComplex*       x,
                                             int64_t               incx,
                                             hipblasStride         stridex,
                                             hipblasComplex*       y,
                                             int64_t               incy,
                                             hipblasStride         stridey,
                                             const float*          c,
                                             const hipblasComplex* s,
                                             int64_t               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crot_strided_batched_64((rocblas_handle)handle,
                                                                    n,
                                                                    (rocblas_float_complex*)x,
                                                                    incx,
                                                                    stridex,
                                                                    (rocblas_float_complex*)y,
                                                                    incy,
                                                                    stridey,
                                                                    c,
                                                                    (rocblas_float_complex*)s,
                                                                    batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsrotStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              hipblasComplex* x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              hipblasComplex* y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              const float*    c,
                                              const float*    s,
                                              int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csrot_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_float_complex*)x,
                                                                     incx,
                                                                     stridex,
                                                                     (rocblas_float_complex*)y,
                                                                     incy,
                                                                     stridey,
                                                                     c,
                                                                     s,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrotStridedBatched_64(hipblasHandle_t             handle,
                                             int64_t                     n,
                                             hipblasDoubleComplex*       x,
                                             int64_t                     incx,
                                             hipblasStride               stridex,
                                             hipblasDoubleComplex*       y,
                                             int64_t                     incy,
                                             hipblasStride               stridey,
                                             const double*               c,
                                             const hipblasDoubleComplex* s,
                                             int64_t                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrot_strided_batched_64((rocblas_handle)handle,
                                                                    n,
                                                                    (rocblas_double_complex*)x,
                                                                    incx,
                                                                    stridex,
                                                                    (rocblas_double_complex*)y,
                                                                    incy,
                                                                    stridey,
                                                                    c,
                                                                    (rocblas_double_complex*)s,
                                                                    batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdrotStridedBatched_64(hipblasHandle_t       handle,
                                              int64_t               n,
                                              hipblasDoubleComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              hipblasDoubleComplex* y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              const double*         c,
                                              const double*         s,
                                              int64_t               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdrot_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_double_complex*)x,
                                                                     incx,
                                                                     stridex,
                                                                     (rocblas_double_complex*)y,
                                                                     incy,
                                                                     stridey,
                                                                     c,
                                                                     s,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrotStridedBatched_v2_64(hipblasHandle_t   handle,
                                                int64_t           n,
                                                hipComplex*       x,
                                                int64_t           incx,
                                                hipblasStride     stridex,
                                                hipComplex*       y,
                                                int64_t           incy,
                                                hipblasStride     stridey,
                                                const float*      c,
                                                const hipComplex* s,
                                                int64_t           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crot_strided_batched_64((rocblas_handle)handle,
                                                                    n,
                                                                    (rocblas_float_complex*)x,
                                                                    incx,
                                                                    stridex,
                                                                    (rocblas_float_complex*)y,
                                                                    incy,
                                                                    stridey,
                                                                    c,
                                                                    (rocblas_float_complex*)s,
                                                                    batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsrotStridedBatched_v2_64(hipblasHandle_t handle,
                                                 int64_t         n,
                                                 hipComplex*     x,
                                                 int64_t         incx,
                                                 hipblasStride   stridex,
                                                 hipComplex*     y,
                                                 int64_t         incy,
                                                 hipblasStride   stridey,
                                                 const float*    c,
                                                 const float*    s,
                                                 int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csrot_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_float_complex*)x,
                                                                     incx,
                                                                     stridex,
                                                                     (rocblas_float_complex*)y,
                                                                     incy,
                                                                     stridey,
                                                                     c,
                                                                     s,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrotStridedBatched_v2_64(hipblasHandle_t         handle,
                                                int64_t                 n,
                                                hipDoubleComplex*       x,
                                                int64_t                 incx,
                                                hipblasStride           stridex,
                                                hipDoubleComplex*       y,
                                                int64_t                 incy,
                                                hipblasStride           stridey,
                                                const double*           c,
                                                const hipDoubleComplex* s,
                                                int64_t                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrot_strided_batched_64((rocblas_handle)handle,
                                                                    n,
                                                                    (rocblas_double_complex*)x,
                                                                    incx,
                                                                    stridex,
                                                                    (rocblas_double_complex*)y,
                                                                    incy,
                                                                    stridey,
                                                                    c,
                                                                    (rocblas_double_complex*)s,
                                                                    batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdrotStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 int64_t           n,
                                                 hipDoubleComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 hipDoubleComplex* y,
                                                 int64_t           incy,
                                                 hipblasStride     stridey,
                                                 const double*     c,
                                                 const double*     s,
                                                 int64_t           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdrot_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_double_complex*)x,
                                                                     incx,
                                                                     stridex,
                                                                     (rocblas_double_complex*)y,
                                                                     incy,
                                                                     stridey,
                                                                     c,
                                                                     s,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rotg
hipblasStatus_t hipblasSrotg(hipblasHandle_t handle, float* a, float* b, float* c, float* s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_srotg((rocblas_handle)handle, a, b, c, s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotg(hipblasHandle_t handle, double* a, double* b, double* c, double* s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_drotg((rocblas_handle)handle, a, b, c, s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrotg(
    hipblasHandle_t handle, hipblasComplex* a, hipblasComplex* b, float* c, hipblasComplex* s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crotg((rocblas_handle)handle,
                                                  (rocblas_float_complex*)a,
                                                  (rocblas_float_complex*)b,
                                                  c,
                                                  (rocblas_float_complex*)s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrotg(hipblasHandle_t       handle,
                             hipblasDoubleComplex* a,
                             hipblasDoubleComplex* b,
                             double*               c,
                             hipblasDoubleComplex* s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrotg((rocblas_handle)handle,
                                                  (rocblas_double_complex*)a,
                                                  (rocblas_double_complex*)b,
                                                  c,
                                                  (rocblas_double_complex*)s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasCrotg_v2(hipblasHandle_t handle, hipComplex* a, hipComplex* b, float* c, hipComplex* s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crotg((rocblas_handle)handle,
                                                  (rocblas_float_complex*)a,
                                                  (rocblas_float_complex*)b,
                                                  c,
                                                  (rocblas_float_complex*)s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrotg_v2(hipblasHandle_t   handle,
                                hipDoubleComplex* a,
                                hipDoubleComplex* b,
                                double*           c,
                                hipDoubleComplex* s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrotg((rocblas_handle)handle,
                                                  (rocblas_double_complex*)a,
                                                  (rocblas_double_complex*)b,
                                                  c,
                                                  (rocblas_double_complex*)s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rotg_64
hipblasStatus_t hipblasSrotg_64(hipblasHandle_t handle, float* a, float* b, float* c, float* s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_srotg_64((rocblas_handle)handle, a, b, c, s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotg_64(hipblasHandle_t handle, double* a, double* b, double* c, double* s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_drotg_64((rocblas_handle)handle, a, b, c, s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrotg_64(
    hipblasHandle_t handle, hipblasComplex* a, hipblasComplex* b, float* c, hipblasComplex* s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crotg_64((rocblas_handle)handle,
                                                     (rocblas_float_complex*)a,
                                                     (rocblas_float_complex*)b,
                                                     c,
                                                     (rocblas_float_complex*)s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrotg_64(hipblasHandle_t       handle,
                                hipblasDoubleComplex* a,
                                hipblasDoubleComplex* b,
                                double*               c,
                                hipblasDoubleComplex* s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrotg_64((rocblas_handle)handle,
                                                     (rocblas_double_complex*)a,
                                                     (rocblas_double_complex*)b,
                                                     c,
                                                     (rocblas_double_complex*)s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrotg_v2_64(
    hipblasHandle_t handle, hipComplex* a, hipComplex* b, float* c, hipComplex* s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crotg_64((rocblas_handle)handle,
                                                     (rocblas_float_complex*)a,
                                                     (rocblas_float_complex*)b,
                                                     c,
                                                     (rocblas_float_complex*)s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrotg_v2_64(hipblasHandle_t   handle,
                                   hipDoubleComplex* a,
                                   hipDoubleComplex* b,
                                   double*           c,
                                   hipDoubleComplex* s)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrotg_64((rocblas_handle)handle,
                                                     (rocblas_double_complex*)a,
                                                     (rocblas_double_complex*)b,
                                                     c,
                                                     (rocblas_double_complex*)s));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rotg_batched
hipblasStatus_t hipblasSrotgBatched(hipblasHandle_t handle,
                                    float* const    a[],
                                    float* const    b[],
                                    float* const    c[],
                                    float* const    s[],
                                    int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_srotg_batched((rocblas_handle)handle, a, b, c, s, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotgBatched(hipblasHandle_t handle,
                                    double* const   a[],
                                    double* const   b[],
                                    double* const   c[],
                                    double* const   s[],
                                    int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_drotg_batched((rocblas_handle)handle, a, b, c, s, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrotgBatched(hipblasHandle_t       handle,
                                    hipblasComplex* const a[],
                                    hipblasComplex* const b[],
                                    float* const          c[],
                                    hipblasComplex* const s[],
                                    int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crotg_batched((rocblas_handle)handle,
                                                          (rocblas_float_complex**)a,
                                                          (rocblas_float_complex**)b,
                                                          c,
                                                          (rocblas_float_complex**)s,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrotgBatched(hipblasHandle_t             handle,
                                    hipblasDoubleComplex* const a[],
                                    hipblasDoubleComplex* const b[],
                                    double* const               c[],
                                    hipblasDoubleComplex* const s[],
                                    int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrotg_batched((rocblas_handle)handle,
                                                          (rocblas_double_complex**)a,
                                                          (rocblas_double_complex**)b,
                                                          c,
                                                          (rocblas_double_complex**)s,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrotgBatched_v2(hipblasHandle_t   handle,
                                       hipComplex* const a[],
                                       hipComplex* const b[],
                                       float* const      c[],
                                       hipComplex* const s[],
                                       int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crotg_batched((rocblas_handle)handle,
                                                          (rocblas_float_complex**)a,
                                                          (rocblas_float_complex**)b,
                                                          c,
                                                          (rocblas_float_complex**)s,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrotgBatched_v2(hipblasHandle_t         handle,
                                       hipDoubleComplex* const a[],
                                       hipDoubleComplex* const b[],
                                       double* const           c[],
                                       hipDoubleComplex* const s[],
                                       int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrotg_batched((rocblas_handle)handle,
                                                          (rocblas_double_complex**)a,
                                                          (rocblas_double_complex**)b,
                                                          c,
                                                          (rocblas_double_complex**)s,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rotg_batched_64
hipblasStatus_t hipblasSrotgBatched_64(hipblasHandle_t handle,
                                       float* const    a[],
                                       float* const    b[],
                                       float* const    c[],
                                       float* const    s[],
                                       int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_srotg_batched_64((rocblas_handle)handle, a, b, c, s, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotgBatched_64(hipblasHandle_t handle,
                                       double* const   a[],
                                       double* const   b[],
                                       double* const   c[],
                                       double* const   s[],
                                       int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_drotg_batched_64((rocblas_handle)handle, a, b, c, s, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrotgBatched_64(hipblasHandle_t       handle,
                                       hipblasComplex* const a[],
                                       hipblasComplex* const b[],
                                       float* const          c[],
                                       hipblasComplex* const s[],
                                       int64_t               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crotg_batched_64((rocblas_handle)handle,
                                                             (rocblas_float_complex**)a,
                                                             (rocblas_float_complex**)b,
                                                             c,
                                                             (rocblas_float_complex**)s,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrotgBatched_64(hipblasHandle_t             handle,
                                       hipblasDoubleComplex* const a[],
                                       hipblasDoubleComplex* const b[],
                                       double* const               c[],
                                       hipblasDoubleComplex* const s[],
                                       int64_t                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrotg_batched_64((rocblas_handle)handle,
                                                             (rocblas_double_complex**)a,
                                                             (rocblas_double_complex**)b,
                                                             c,
                                                             (rocblas_double_complex**)s,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrotgBatched_v2_64(hipblasHandle_t   handle,
                                          hipComplex* const a[],
                                          hipComplex* const b[],
                                          float* const      c[],
                                          hipComplex* const s[],
                                          int64_t           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crotg_batched_64((rocblas_handle)handle,
                                                             (rocblas_float_complex**)a,
                                                             (rocblas_float_complex**)b,
                                                             c,
                                                             (rocblas_float_complex**)s,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrotgBatched_v2_64(hipblasHandle_t         handle,
                                          hipDoubleComplex* const a[],
                                          hipDoubleComplex* const b[],
                                          double* const           c[],
                                          hipDoubleComplex* const s[],
                                          int64_t                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrotg_batched_64((rocblas_handle)handle,
                                                             (rocblas_double_complex**)a,
                                                             (rocblas_double_complex**)b,
                                                             c,
                                                             (rocblas_double_complex**)s,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rotg_strided_batched
hipblasStatus_t hipblasSrotgStridedBatched(hipblasHandle_t handle,
                                           float*          a,
                                           hipblasStride   stride_a,
                                           float*          b,
                                           hipblasStride   stride_b,
                                           float*          c,
                                           hipblasStride   stride_c,
                                           float*          s,
                                           hipblasStride   stride_s,
                                           int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_srotg_strided_batched(
        (rocblas_handle)handle, a, stride_a, b, stride_b, c, stride_c, s, stride_s, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotgStridedBatched(hipblasHandle_t handle,
                                           double*         a,
                                           hipblasStride   stride_a,
                                           double*         b,
                                           hipblasStride   stride_b,
                                           double*         c,
                                           hipblasStride   stride_c,
                                           double*         s,
                                           hipblasStride   stride_s,
                                           int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_drotg_strided_batched(
        (rocblas_handle)handle, a, stride_a, b, stride_b, c, stride_c, s, stride_s, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrotgStridedBatched(hipblasHandle_t handle,
                                           hipblasComplex* a,
                                           hipblasStride   stride_a,
                                           hipblasComplex* b,
                                           hipblasStride   stride_b,
                                           float*          c,
                                           hipblasStride   stride_c,
                                           hipblasComplex* s,
                                           hipblasStride   stride_s,
                                           int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crotg_strided_batched((rocblas_handle)handle,
                                                                  (rocblas_float_complex*)a,
                                                                  stride_a,
                                                                  (rocblas_float_complex*)b,
                                                                  stride_b,
                                                                  c,
                                                                  stride_c,
                                                                  (rocblas_float_complex*)s,
                                                                  stride_s,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrotgStridedBatched(hipblasHandle_t       handle,
                                           hipblasDoubleComplex* a,
                                           hipblasStride         stride_a,
                                           hipblasDoubleComplex* b,
                                           hipblasStride         stride_b,
                                           double*               c,
                                           hipblasStride         stride_c,
                                           hipblasDoubleComplex* s,
                                           hipblasStride         stride_s,
                                           int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrotg_strided_batched((rocblas_handle)handle,
                                                                  (rocblas_double_complex*)a,
                                                                  stride_a,
                                                                  (rocblas_double_complex*)b,
                                                                  stride_b,
                                                                  c,
                                                                  stride_c,
                                                                  (rocblas_double_complex*)s,
                                                                  stride_s,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrotgStridedBatched_v2(hipblasHandle_t handle,
                                              hipComplex*     a,
                                              hipblasStride   stride_a,
                                              hipComplex*     b,
                                              hipblasStride   stride_b,
                                              float*          c,
                                              hipblasStride   stride_c,
                                              hipComplex*     s,
                                              hipblasStride   stride_s,
                                              int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crotg_strided_batched((rocblas_handle)handle,
                                                                  (rocblas_float_complex*)a,
                                                                  stride_a,
                                                                  (rocblas_float_complex*)b,
                                                                  stride_b,
                                                                  c,
                                                                  stride_c,
                                                                  (rocblas_float_complex*)s,
                                                                  stride_s,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrotgStridedBatched_v2(hipblasHandle_t   handle,
                                              hipDoubleComplex* a,
                                              hipblasStride     stride_a,
                                              hipDoubleComplex* b,
                                              hipblasStride     stride_b,
                                              double*           c,
                                              hipblasStride     stride_c,
                                              hipDoubleComplex* s,
                                              hipblasStride     stride_s,
                                              int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrotg_strided_batched((rocblas_handle)handle,
                                                                  (rocblas_double_complex*)a,
                                                                  stride_a,
                                                                  (rocblas_double_complex*)b,
                                                                  stride_b,
                                                                  c,
                                                                  stride_c,
                                                                  (rocblas_double_complex*)s,
                                                                  stride_s,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rotg_strided_batched_64
hipblasStatus_t hipblasSrotgStridedBatched_64(hipblasHandle_t handle,
                                              float*          a,
                                              hipblasStride   stride_a,
                                              float*          b,
                                              hipblasStride   stride_b,
                                              float*          c,
                                              hipblasStride   stride_c,
                                              float*          s,
                                              hipblasStride   stride_s,
                                              int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_srotg_strided_batched_64(
        (rocblas_handle)handle, a, stride_a, b, stride_b, c, stride_c, s, stride_s, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotgStridedBatched_64(hipblasHandle_t handle,
                                              double*         a,
                                              hipblasStride   stride_a,
                                              double*         b,
                                              hipblasStride   stride_b,
                                              double*         c,
                                              hipblasStride   stride_c,
                                              double*         s,
                                              hipblasStride   stride_s,
                                              int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_drotg_strided_batched_64(
        (rocblas_handle)handle, a, stride_a, b, stride_b, c, stride_c, s, stride_s, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrotgStridedBatched_64(hipblasHandle_t handle,
                                              hipblasComplex* a,
                                              hipblasStride   stride_a,
                                              hipblasComplex* b,
                                              hipblasStride   stride_b,
                                              float*          c,
                                              hipblasStride   stride_c,
                                              hipblasComplex* s,
                                              hipblasStride   stride_s,
                                              int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crotg_strided_batched_64((rocblas_handle)handle,
                                                                     (rocblas_float_complex*)a,
                                                                     stride_a,
                                                                     (rocblas_float_complex*)b,
                                                                     stride_b,
                                                                     c,
                                                                     stride_c,
                                                                     (rocblas_float_complex*)s,
                                                                     stride_s,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrotgStridedBatched_64(hipblasHandle_t       handle,
                                              hipblasDoubleComplex* a,
                                              hipblasStride         stride_a,
                                              hipblasDoubleComplex* b,
                                              hipblasStride         stride_b,
                                              double*               c,
                                              hipblasStride         stride_c,
                                              hipblasDoubleComplex* s,
                                              hipblasStride         stride_s,
                                              int64_t               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrotg_strided_batched_64((rocblas_handle)handle,
                                                                     (rocblas_double_complex*)a,
                                                                     stride_a,
                                                                     (rocblas_double_complex*)b,
                                                                     stride_b,
                                                                     c,
                                                                     stride_c,
                                                                     (rocblas_double_complex*)s,
                                                                     stride_s,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCrotgStridedBatched_v2_64(hipblasHandle_t handle,
                                                 hipComplex*     a,
                                                 hipblasStride   stride_a,
                                                 hipComplex*     b,
                                                 hipblasStride   stride_b,
                                                 float*          c,
                                                 hipblasStride   stride_c,
                                                 hipComplex*     s,
                                                 hipblasStride   stride_s,
                                                 int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_crotg_strided_batched_64((rocblas_handle)handle,
                                                                     (rocblas_float_complex*)a,
                                                                     stride_a,
                                                                     (rocblas_float_complex*)b,
                                                                     stride_b,
                                                                     c,
                                                                     stride_c,
                                                                     (rocblas_float_complex*)s,
                                                                     stride_s,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZrotgStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 hipDoubleComplex* a,
                                                 hipblasStride     stride_a,
                                                 hipDoubleComplex* b,
                                                 hipblasStride     stride_b,
                                                 double*           c,
                                                 hipblasStride     stride_c,
                                                 hipDoubleComplex* s,
                                                 hipblasStride     stride_s,
                                                 int64_t           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zrotg_strided_batched_64((rocblas_handle)handle,
                                                                     (rocblas_double_complex*)a,
                                                                     stride_a,
                                                                     (rocblas_double_complex*)b,
                                                                     stride_b,
                                                                     c,
                                                                     stride_c,
                                                                     (rocblas_double_complex*)s,
                                                                     stride_s,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rotm
hipblasStatus_t hipblasSrotm(
    hipblasHandle_t handle, int n, float* x, int incx, float* y, int incy, const float* param)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_srotm((rocblas_handle)handle, n, x, incx, y, incy, param));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotm(
    hipblasHandle_t handle, int n, double* x, int incx, double* y, int incy, const double* param)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_drotm((rocblas_handle)handle, n, x, incx, y, incy, param));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rotm_64
hipblasStatus_t hipblasSrotm_64(hipblasHandle_t handle,
                                int64_t         n,
                                float*          x,
                                int64_t         incx,
                                float*          y,
                                int64_t         incy,
                                const float*    param)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_srotm_64((rocblas_handle)handle, n, x, incx, y, incy, param));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotm_64(hipblasHandle_t handle,
                                int64_t         n,
                                double*         x,
                                int64_t         incx,
                                double*         y,
                                int64_t         incy,
                                const double*   param)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_drotm_64((rocblas_handle)handle, n, x, incx, y, incy, param));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rotm_batched
hipblasStatus_t hipblasSrotmBatched(hipblasHandle_t    handle,
                                    int                n,
                                    float* const       x[],
                                    int                incx,
                                    float* const       y[],
                                    int                incy,
                                    const float* const param[],
                                    int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_srotm_batched((rocblas_handle)handle, n, x, incx, y, incy, param, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotmBatched(hipblasHandle_t     handle,
                                    int                 n,
                                    double* const       x[],
                                    int                 incx,
                                    double* const       y[],
                                    int                 incy,
                                    const double* const param[],
                                    int                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_drotm_batched((rocblas_handle)handle, n, x, incx, y, incy, param, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rotm_batched_64
hipblasStatus_t hipblasSrotmBatched_64(hipblasHandle_t    handle,
                                       int64_t            n,
                                       float* const       x[],
                                       int64_t            incx,
                                       float* const       y[],
                                       int64_t            incy,
                                       const float* const param[],
                                       int64_t            batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_srotm_batched_64((rocblas_handle)handle, n, x, incx, y, incy, param, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotmBatched_64(hipblasHandle_t     handle,
                                       int64_t             n,
                                       double* const       x[],
                                       int64_t             incx,
                                       double* const       y[],
                                       int64_t             incy,
                                       const double* const param[],
                                       int64_t             batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_drotm_batched_64((rocblas_handle)handle, n, x, incx, y, incy, param, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rotm_strided_batched
hipblasStatus_t hipblasSrotmStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           float*          x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           float*          y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           const float*    param,
                                           hipblasStride   strideParam,
                                           int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_srotm_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  x,
                                                                  incx,
                                                                  stridex,
                                                                  y,
                                                                  incy,
                                                                  stridey,
                                                                  param,
                                                                  strideParam,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotmStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           double*         x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           double*         y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           const double*   param,
                                           hipblasStride   strideParam,
                                           int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_drotm_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  x,
                                                                  incx,
                                                                  stridex,
                                                                  y,
                                                                  incy,
                                                                  stridey,
                                                                  param,
                                                                  strideParam,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rotm_strided_batched_64
hipblasStatus_t hipblasSrotmStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              float*          x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              float*          y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              const float*    param,
                                              hipblasStride   strideParam,
                                              int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_srotm_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     x,
                                                                     incx,
                                                                     stridex,
                                                                     y,
                                                                     incy,
                                                                     stridey,
                                                                     param,
                                                                     strideParam,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotmStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              double*         x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              double*         y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              const double*   param,
                                              hipblasStride   strideParam,
                                              int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_drotm_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     x,
                                                                     incx,
                                                                     stridex,
                                                                     y,
                                                                     incy,
                                                                     stridey,
                                                                     param,
                                                                     strideParam,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rotmg
hipblasStatus_t hipblasSrotmg(
    hipblasHandle_t handle, float* d1, float* d2, float* x1, const float* y1, float* param)
try
{
    return rocBLASStatusToHIPStatus(rocblas_srotmg((rocblas_handle)handle, d1, d2, x1, y1, param));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotmg(
    hipblasHandle_t handle, double* d1, double* d2, double* x1, const double* y1, double* param)
try
{
    return rocBLASStatusToHIPStatus(rocblas_drotmg((rocblas_handle)handle, d1, d2, x1, y1, param));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rotmg_64
hipblasStatus_t hipblasSrotmg_64(
    hipblasHandle_t handle, float* d1, float* d2, float* x1, const float* y1, float* param)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_srotmg_64((rocblas_handle)handle, d1, d2, x1, y1, param));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotmg_64(
    hipblasHandle_t handle, double* d1, double* d2, double* x1, const double* y1, double* param)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_drotmg_64((rocblas_handle)handle, d1, d2, x1, y1, param));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rotmg_batched
hipblasStatus_t hipblasSrotmgBatched(hipblasHandle_t    handle,
                                     float* const       d1[],
                                     float* const       d2[],
                                     float* const       x1[],
                                     const float* const y1[],
                                     float* const       param[],
                                     int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_srotmg_batched((rocblas_handle)handle, d1, d2, x1, y1, param, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotmgBatched(hipblasHandle_t     handle,
                                     double* const       d1[],
                                     double* const       d2[],
                                     double* const       x1[],
                                     const double* const y1[],
                                     double* const       param[],
                                     int                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_drotmg_batched((rocblas_handle)handle, d1, d2, x1, y1, param, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rotmg_batched_64
hipblasStatus_t hipblasSrotmgBatched_64(hipblasHandle_t    handle,
                                        float* const       d1[],
                                        float* const       d2[],
                                        float* const       x1[],
                                        const float* const y1[],
                                        float* const       param[],
                                        int64_t            batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_srotmg_batched_64((rocblas_handle)handle, d1, d2, x1, y1, param, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotmgBatched_64(hipblasHandle_t     handle,
                                        double* const       d1[],
                                        double* const       d2[],
                                        double* const       x1[],
                                        const double* const y1[],
                                        double* const       param[],
                                        int64_t             batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_drotmg_batched_64((rocblas_handle)handle, d1, d2, x1, y1, param, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rotmg_strided_batched
hipblasStatus_t hipblasSrotmgStridedBatched(hipblasHandle_t handle,
                                            float*          d1,
                                            hipblasStride   stride_d1,
                                            float*          d2,
                                            hipblasStride   stride_d2,
                                            float*          x1,
                                            hipblasStride   stride_x1,
                                            const float*    y1,
                                            hipblasStride   stride_y1,
                                            float*          param,
                                            hipblasStride   strideParam,
                                            int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_srotmg_strided_batched((rocblas_handle)handle,
                                                                   d1,
                                                                   stride_d1,
                                                                   d2,
                                                                   stride_d2,
                                                                   x1,
                                                                   stride_x1,
                                                                   y1,
                                                                   stride_y1,
                                                                   param,
                                                                   strideParam,
                                                                   batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotmgStridedBatched(hipblasHandle_t handle,
                                            double*         d1,
                                            hipblasStride   stride_d1,
                                            double*         d2,
                                            hipblasStride   stride_d2,
                                            double*         x1,
                                            hipblasStride   stride_x1,
                                            const double*   y1,
                                            hipblasStride   stride_y1,
                                            double*         param,
                                            hipblasStride   strideParam,
                                            int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_drotmg_strided_batched((rocblas_handle)handle,
                                                                   d1,
                                                                   stride_d1,
                                                                   d2,
                                                                   stride_d2,
                                                                   x1,
                                                                   stride_x1,
                                                                   y1,
                                                                   stride_y1,
                                                                   param,
                                                                   strideParam,
                                                                   batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// rotmg_strided_batched_64
hipblasStatus_t hipblasSrotmgStridedBatched_64(hipblasHandle_t handle,
                                               float*          d1,
                                               hipblasStride   stride_d1,
                                               float*          d2,
                                               hipblasStride   stride_d2,
                                               float*          x1,
                                               hipblasStride   stride_x1,
                                               const float*    y1,
                                               hipblasStride   stride_y1,
                                               float*          param,
                                               hipblasStride   strideParam,
                                               int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_srotmg_strided_batched_64((rocblas_handle)handle,
                                                                      d1,
                                                                      stride_d1,
                                                                      d2,
                                                                      stride_d2,
                                                                      x1,
                                                                      stride_x1,
                                                                      y1,
                                                                      stride_y1,
                                                                      param,
                                                                      strideParam,
                                                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDrotmgStridedBatched_64(hipblasHandle_t handle,
                                               double*         d1,
                                               hipblasStride   stride_d1,
                                               double*         d2,
                                               hipblasStride   stride_d2,
                                               double*         x1,
                                               hipblasStride   stride_x1,
                                               const double*   y1,
                                               hipblasStride   stride_y1,
                                               double*         param,
                                               hipblasStride   strideParam,
                                               int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_drotmg_strided_batched_64((rocblas_handle)handle,
                                                                      d1,
                                                                      stride_d1,
                                                                      d2,
                                                                      stride_d2,
                                                                      x1,
                                                                      stride_x1,
                                                                      y1,
                                                                      stride_y1,
                                                                      param,
                                                                      strideParam,
                                                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// scal
hipblasStatus_t hipblasSscal(hipblasHandle_t handle, int n, const float* alpha, float* x, int incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sscal((rocblas_handle)handle, n, alpha, x, incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasDscal(hipblasHandle_t handle, int n, const double* alpha, double* x, int incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dscal((rocblas_handle)handle, n, alpha, x, incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCscal(
    hipblasHandle_t handle, int n, const hipblasComplex* alpha, hipblasComplex* x, int incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cscal(
        (rocblas_handle)handle, n, (rocblas_float_complex*)alpha, (rocblas_float_complex*)x, incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasCsscal(hipblasHandle_t handle, int n, const float* alpha, hipblasComplex* x, int incx)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_csscal((rocblas_handle)handle, n, alpha, (rocblas_float_complex*)x, incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZscal(hipblasHandle_t             handle,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             hipblasDoubleComplex*       x,
                             int                         incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zscal((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdscal(
    hipblasHandle_t handle, int n, const double* alpha, hipblasDoubleComplex* x, int incx)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_zdscal((rocblas_handle)handle, n, alpha, (rocblas_double_complex*)x, incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasCscal_v2(hipblasHandle_t handle, int n, const hipComplex* alpha, hipComplex* x, int incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cscal(
        (rocblas_handle)handle, n, (rocblas_float_complex*)alpha, (rocblas_float_complex*)x, incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasCsscal_v2(hipblasHandle_t handle, int n, const float* alpha, hipComplex* x, int incx)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_csscal((rocblas_handle)handle, n, alpha, (rocblas_float_complex*)x, incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZscal_v2(
    hipblasHandle_t handle, int n, const hipDoubleComplex* alpha, hipDoubleComplex* x, int incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zscal((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdscal_v2(
    hipblasHandle_t handle, int n, const double* alpha, hipDoubleComplex* x, int incx)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_zdscal((rocblas_handle)handle, n, alpha, (rocblas_double_complex*)x, incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// scal_64
hipblasStatus_t
    hipblasSscal_64(hipblasHandle_t handle, int64_t n, const float* alpha, float* x, int64_t incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sscal_64((rocblas_handle)handle, n, alpha, x, incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasDscal_64(hipblasHandle_t handle, int64_t n, const double* alpha, double* x, int64_t incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dscal_64((rocblas_handle)handle, n, alpha, x, incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCscal_64(
    hipblasHandle_t handle, int64_t n, const hipblasComplex* alpha, hipblasComplex* x, int64_t incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cscal_64(
        (rocblas_handle)handle, n, (rocblas_float_complex*)alpha, (rocblas_float_complex*)x, incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsscal_64(
    hipblasHandle_t handle, int64_t n, const float* alpha, hipblasComplex* x, int64_t incx)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_csscal_64((rocblas_handle)handle, n, alpha, (rocblas_float_complex*)x, incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZscal_64(hipblasHandle_t             handle,
                                int64_t                     n,
                                const hipblasDoubleComplex* alpha,
                                hipblasDoubleComplex*       x,
                                int64_t                     incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zscal_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_double_complex*)alpha,
                                                     (rocblas_double_complex*)x,
                                                     incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdscal_64(
    hipblasHandle_t handle, int64_t n, const double* alpha, hipblasDoubleComplex* x, int64_t incx)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_zdscal_64((rocblas_handle)handle, n, alpha, (rocblas_double_complex*)x, incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCscal_v2_64(
    hipblasHandle_t handle, int64_t n, const hipComplex* alpha, hipComplex* x, int64_t incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cscal_64(
        (rocblas_handle)handle, n, (rocblas_float_complex*)alpha, (rocblas_float_complex*)x, incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsscal_v2_64(
    hipblasHandle_t handle, int64_t n, const float* alpha, hipComplex* x, int64_t incx)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_csscal_64((rocblas_handle)handle, n, alpha, (rocblas_float_complex*)x, incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZscal_v2_64(hipblasHandle_t         handle,
                                   int64_t                 n,
                                   const hipDoubleComplex* alpha,
                                   hipDoubleComplex*       x,
                                   int64_t                 incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zscal_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_double_complex*)alpha,
                                                     (rocblas_double_complex*)x,
                                                     incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdscal_v2_64(
    hipblasHandle_t handle, int64_t n, const double* alpha, hipDoubleComplex* x, int64_t incx)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_zdscal_64((rocblas_handle)handle, n, alpha, (rocblas_double_complex*)x, incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// scal_batched
hipblasStatus_t hipblasSscalBatched(
    hipblasHandle_t handle, int n, const float* alpha, float* const x[], int incx, int batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_sscal_batched((rocblas_handle)handle, n, alpha, x, incx, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDscalBatched(
    hipblasHandle_t handle, int n, const double* alpha, double* const x[], int incx, int batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dscal_batched((rocblas_handle)handle, n, alpha, x, incx, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCscalBatched(hipblasHandle_t       handle,
                                    int                   n,
                                    const hipblasComplex* alpha,
                                    hipblasComplex* const x[],
                                    int                   incx,
                                    int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cscal_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex* const*)x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZscalBatched(hipblasHandle_t             handle,
                                    int                         n,
                                    const hipblasDoubleComplex* alpha,
                                    hipblasDoubleComplex* const x[],
                                    int                         incx,
                                    int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zscal_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex* const*)x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsscalBatched(hipblasHandle_t       handle,
                                     int                   n,
                                     const float*          alpha,
                                     hipblasComplex* const x[],
                                     int                   incx,
                                     int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csscal_batched(
        (rocblas_handle)handle, n, alpha, (rocblas_float_complex* const*)x, incx, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdscalBatched(hipblasHandle_t             handle,
                                     int                         n,
                                     const double*               alpha,
                                     hipblasDoubleComplex* const x[],
                                     int                         incx,
                                     int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdscal_batched(
        (rocblas_handle)handle, n, alpha, (rocblas_double_complex* const*)x, incx, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCscalBatched_v2(hipblasHandle_t   handle,
                                       int               n,
                                       const hipComplex* alpha,
                                       hipComplex* const x[],
                                       int               incx,
                                       int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cscal_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex* const*)x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZscalBatched_v2(hipblasHandle_t         handle,
                                       int                     n,
                                       const hipDoubleComplex* alpha,
                                       hipDoubleComplex* const x[],
                                       int                     incx,
                                       int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zscal_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex* const*)x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsscalBatched_v2(hipblasHandle_t   handle,
                                        int               n,
                                        const float*      alpha,
                                        hipComplex* const x[],
                                        int               incx,
                                        int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csscal_batched(
        (rocblas_handle)handle, n, alpha, (rocblas_float_complex* const*)x, incx, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdscalBatched_v2(hipblasHandle_t         handle,
                                        int                     n,
                                        const double*           alpha,
                                        hipDoubleComplex* const x[],
                                        int                     incx,
                                        int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdscal_batched(
        (rocblas_handle)handle, n, alpha, (rocblas_double_complex* const*)x, incx, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// scal_batched_64
hipblasStatus_t hipblasSscalBatched_64(hipblasHandle_t handle,
                                       int64_t         n,
                                       const float*    alpha,
                                       float* const    x[],
                                       int64_t         incx,
                                       int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_sscal_batched_64((rocblas_handle)handle, n, alpha, x, incx, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDscalBatched_64(hipblasHandle_t handle,
                                       int64_t         n,
                                       const double*   alpha,
                                       double* const   x[],
                                       int64_t         incx,
                                       int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dscal_batched_64((rocblas_handle)handle, n, alpha, x, incx, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCscalBatched_64(hipblasHandle_t       handle,
                                       int64_t               n,
                                       const hipblasComplex* alpha,
                                       hipblasComplex* const x[],
                                       int64_t               incx,
                                       int64_t               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cscal_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_float_complex*)alpha,
                                                             (rocblas_float_complex* const*)x,
                                                             incx,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZscalBatched_64(hipblasHandle_t             handle,
                                       int64_t                     n,
                                       const hipblasDoubleComplex* alpha,
                                       hipblasDoubleComplex* const x[],
                                       int64_t                     incx,
                                       int64_t                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zscal_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_double_complex*)alpha,
                                                             (rocblas_double_complex* const*)x,
                                                             incx,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsscalBatched_64(hipblasHandle_t       handle,
                                        int64_t               n,
                                        const float*          alpha,
                                        hipblasComplex* const x[],
                                        int64_t               incx,
                                        int64_t               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csscal_batched_64(
        (rocblas_handle)handle, n, alpha, (rocblas_float_complex* const*)x, incx, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdscalBatched_64(hipblasHandle_t             handle,
                                        int64_t                     n,
                                        const double*               alpha,
                                        hipblasDoubleComplex* const x[],
                                        int64_t                     incx,
                                        int64_t                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdscal_batched_64(
        (rocblas_handle)handle, n, alpha, (rocblas_double_complex* const*)x, incx, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCscalBatched_v2_64(hipblasHandle_t   handle,
                                          int64_t           n,
                                          const hipComplex* alpha,
                                          hipComplex* const x[],
                                          int64_t           incx,
                                          int64_t           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cscal_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_float_complex*)alpha,
                                                             (rocblas_float_complex* const*)x,
                                                             incx,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZscalBatched_v2_64(hipblasHandle_t         handle,
                                          int64_t                 n,
                                          const hipDoubleComplex* alpha,
                                          hipDoubleComplex* const x[],
                                          int64_t                 incx,
                                          int64_t                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zscal_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_double_complex*)alpha,
                                                             (rocblas_double_complex* const*)x,
                                                             incx,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsscalBatched_v2_64(hipblasHandle_t   handle,
                                           int64_t           n,
                                           const float*      alpha,
                                           hipComplex* const x[],
                                           int64_t           incx,
                                           int64_t           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csscal_batched_64(
        (rocblas_handle)handle, n, alpha, (rocblas_float_complex* const*)x, incx, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdscalBatched_v2_64(hipblasHandle_t         handle,
                                           int64_t                 n,
                                           const double*           alpha,
                                           hipDoubleComplex* const x[],
                                           int64_t                 incx,
                                           int64_t                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdscal_batched_64(
        (rocblas_handle)handle, n, alpha, (rocblas_double_complex* const*)x, incx, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// scal_strided_batched
hipblasStatus_t hipblasSscalStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const float*    alpha,
                                           float*          x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sscal_strided_batched(
        (rocblas_handle)handle, n, alpha, x, incx, stridex, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDscalStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const double*   alpha,
                                           double*         x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dscal_strided_batched(
        (rocblas_handle)handle, n, alpha, x, incx, stridex, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCscalStridedBatched(hipblasHandle_t       handle,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           hipblasComplex*       x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cscal_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZscalStridedBatched(hipblasHandle_t             handle,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           hipblasDoubleComplex*       x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zscal_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsscalStridedBatched(hipblasHandle_t handle,
                                            int             n,
                                            const float*    alpha,
                                            hipblasComplex* x,
                                            int             incx,
                                            hipblasStride   stridex,
                                            int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csscal_strided_batched(
        (rocblas_handle)handle, n, alpha, (rocblas_float_complex*)x, incx, stridex, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdscalStridedBatched(hipblasHandle_t       handle,
                                            int                   n,
                                            const double*         alpha,
                                            hipblasDoubleComplex* x,
                                            int                   incx,
                                            hipblasStride         stridex,
                                            int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdscal_strided_batched(
        (rocblas_handle)handle, n, alpha, (rocblas_double_complex*)x, incx, stridex, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCscalStridedBatched_v2(hipblasHandle_t   handle,
                                              int               n,
                                              const hipComplex* alpha,
                                              hipComplex*       x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cscal_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZscalStridedBatched_v2(hipblasHandle_t         handle,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              hipDoubleComplex*       x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zscal_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsscalStridedBatched_v2(hipblasHandle_t handle,
                                               int             n,
                                               const float*    alpha,
                                               hipComplex*     x,
                                               int             incx,
                                               hipblasStride   stridex,
                                               int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csscal_strided_batched(
        (rocblas_handle)handle, n, alpha, (rocblas_float_complex*)x, incx, stridex, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdscalStridedBatched_v2(hipblasHandle_t   handle,
                                               int               n,
                                               const double*     alpha,
                                               hipDoubleComplex* x,
                                               int               incx,
                                               hipblasStride     stridex,
                                               int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdscal_strided_batched(
        (rocblas_handle)handle, n, alpha, (rocblas_double_complex*)x, incx, stridex, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// scal_strided_batched_64
hipblasStatus_t hipblasSscalStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const float*    alpha,
                                              float*          x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sscal_strided_batched_64(
        (rocblas_handle)handle, n, alpha, x, incx, stridex, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDscalStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const double*   alpha,
                                              double*         x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dscal_strided_batched_64(
        (rocblas_handle)handle, n, alpha, x, incx, stridex, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCscalStridedBatched_64(hipblasHandle_t       handle,
                                              int64_t               n,
                                              const hipblasComplex* alpha,
                                              hipblasComplex*       x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              int64_t               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cscal_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_float_complex*)alpha,
                                                                     (rocblas_float_complex*)x,
                                                                     incx,
                                                                     stridex,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZscalStridedBatched_64(hipblasHandle_t             handle,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* alpha,
                                              hipblasDoubleComplex*       x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              int64_t                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zscal_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_double_complex*)alpha,
                                                                     (rocblas_double_complex*)x,
                                                                     incx,
                                                                     stridex,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsscalStridedBatched_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               const float*    alpha,
                                               hipblasComplex* x,
                                               int64_t         incx,
                                               hipblasStride   stridex,
                                               int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csscal_strided_batched_64(
        (rocblas_handle)handle, n, alpha, (rocblas_float_complex*)x, incx, stridex, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdscalStridedBatched_64(hipblasHandle_t       handle,
                                               int64_t               n,
                                               const double*         alpha,
                                               hipblasDoubleComplex* x,
                                               int64_t               incx,
                                               hipblasStride         stridex,
                                               int64_t               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdscal_strided_batched_64(
        (rocblas_handle)handle, n, alpha, (rocblas_double_complex*)x, incx, stridex, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCscalStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 int64_t           n,
                                                 const hipComplex* alpha,
                                                 hipComplex*       x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 int64_t           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cscal_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_float_complex*)alpha,
                                                                     (rocblas_float_complex*)x,
                                                                     incx,
                                                                     stridex,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZscalStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* alpha,
                                                 hipDoubleComplex*       x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 int64_t                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zscal_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_double_complex*)alpha,
                                                                     (rocblas_double_complex*)x,
                                                                     incx,
                                                                     stridex,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsscalStridedBatched_v2_64(hipblasHandle_t handle,
                                                  int64_t         n,
                                                  const float*    alpha,
                                                  hipComplex*     x,
                                                  int64_t         incx,
                                                  hipblasStride   stridex,
                                                  int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csscal_strided_batched_64(
        (rocblas_handle)handle, n, alpha, (rocblas_float_complex*)x, incx, stridex, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZdscalStridedBatched_v2_64(hipblasHandle_t   handle,
                                                  int64_t           n,
                                                  const double*     alpha,
                                                  hipDoubleComplex* x,
                                                  int64_t           incx,
                                                  hipblasStride     stridex,
                                                  int64_t           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zdscal_strided_batched_64(
        (rocblas_handle)handle, n, alpha, (rocblas_double_complex*)x, incx, stridex, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// swap
hipblasStatus_t hipblasSswap(hipblasHandle_t handle, int n, float* x, int incx, float* y, int incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sswap((rocblas_handle)handle, n, x, incx, y, incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasDswap(hipblasHandle_t handle, int n, double* x, int incx, double* y, int incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dswap((rocblas_handle)handle, n, x, incx, y, incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCswap(
    hipblasHandle_t handle, int n, hipblasComplex* x, int incx, hipblasComplex* y, int incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cswap((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZswap(hipblasHandle_t       handle,
                             int                   n,
                             hipblasDoubleComplex* x,
                             int                   incx,
                             hipblasDoubleComplex* y,
                             int                   incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zswap((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t
    hipblasCswap_v2(hipblasHandle_t handle, int n, hipComplex* x, int incx, hipComplex* y, int incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cswap((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZswap_v2(
    hipblasHandle_t handle, int n, hipDoubleComplex* x, int incx, hipDoubleComplex* y, int incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zswap((rocblas_handle)handle,
                                                  n,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// swap_64
hipblasStatus_t hipblasSswap_64(
    hipblasHandle_t handle, int64_t n, float* x, int64_t incx, float* y, int64_t incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sswap_64((rocblas_handle)handle, n, x, incx, y, incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDswap_64(
    hipblasHandle_t handle, int64_t n, double* x, int64_t incx, double* y, int64_t incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dswap_64((rocblas_handle)handle, n, x, incx, y, incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCswap_64(hipblasHandle_t handle,
                                int64_t         n,
                                hipblasComplex* x,
                                int64_t         incx,
                                hipblasComplex* y,
                                int64_t         incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cswap_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_float_complex*)x,
                                                     incx,
                                                     (rocblas_float_complex*)y,
                                                     incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZswap_64(hipblasHandle_t       handle,
                                int64_t               n,
                                hipblasDoubleComplex* x,
                                int64_t               incx,
                                hipblasDoubleComplex* y,
                                int64_t               incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zswap_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_double_complex*)x,
                                                     incx,
                                                     (rocblas_double_complex*)y,
                                                     incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCswap_v2_64(
    hipblasHandle_t handle, int64_t n, hipComplex* x, int64_t incx, hipComplex* y, int64_t incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cswap_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_float_complex*)x,
                                                     incx,
                                                     (rocblas_float_complex*)y,
                                                     incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZswap_v2_64(hipblasHandle_t   handle,
                                   int64_t           n,
                                   hipDoubleComplex* x,
                                   int64_t           incx,
                                   hipDoubleComplex* y,
                                   int64_t           incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zswap_64((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_double_complex*)x,
                                                     incx,
                                                     (rocblas_double_complex*)y,
                                                     incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// swap_batched
hipblasStatus_t hipblasSswapBatched(hipblasHandle_t handle,
                                    int             n,
                                    float* const    x[],
                                    int             incx,
                                    float* const    y[],
                                    int             incy,
                                    int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_sswap_batched((rocblas_handle)handle, n, x, incx, y, incy, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDswapBatched(hipblasHandle_t handle,
                                    int             n,
                                    double* const   x[],
                                    int             incx,
                                    double* const   y[],
                                    int             incy,
                                    int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dswap_batched((rocblas_handle)handle, n, x, incx, y, incy, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCswapBatched(hipblasHandle_t       handle,
                                    int                   n,
                                    hipblasComplex* const x[],
                                    int                   incx,
                                    hipblasComplex* const y[],
                                    int                   incy,
                                    int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cswap_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZswapBatched(hipblasHandle_t             handle,
                                    int                         n,
                                    hipblasDoubleComplex* const x[],
                                    int                         incx,
                                    hipblasDoubleComplex* const y[],
                                    int                         incy,
                                    int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zswap_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCswapBatched_v2(hipblasHandle_t   handle,
                                       int               n,
                                       hipComplex* const x[],
                                       int               incx,
                                       hipComplex* const y[],
                                       int               incy,
                                       int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cswap_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZswapBatched_v2(hipblasHandle_t         handle,
                                       int                     n,
                                       hipDoubleComplex* const x[],
                                       int                     incx,
                                       hipDoubleComplex* const y[],
                                       int                     incy,
                                       int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zswap_batched((rocblas_handle)handle,
                                                          n,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// swap_batched_64
hipblasStatus_t hipblasSswapBatched_64(hipblasHandle_t handle,
                                       int64_t         n,
                                       float* const    x[],
                                       int64_t         incx,
                                       float* const    y[],
                                       int64_t         incy,
                                       int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_sswap_batched_64((rocblas_handle)handle, n, x, incx, y, incy, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDswapBatched_64(hipblasHandle_t handle,
                                       int64_t         n,
                                       double* const   x[],
                                       int64_t         incx,
                                       double* const   y[],
                                       int64_t         incy,
                                       int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dswap_batched_64((rocblas_handle)handle, n, x, incx, y, incy, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCswapBatched_64(hipblasHandle_t       handle,
                                       int64_t               n,
                                       hipblasComplex* const x[],
                                       int64_t               incx,
                                       hipblasComplex* const y[],
                                       int64_t               incy,
                                       int64_t               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cswap_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_float_complex**)x,
                                                             incx,
                                                             (rocblas_float_complex**)y,
                                                             incy,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZswapBatched_64(hipblasHandle_t             handle,
                                       int64_t                     n,
                                       hipblasDoubleComplex* const x[],
                                       int64_t                     incx,
                                       hipblasDoubleComplex* const y[],
                                       int64_t                     incy,
                                       int64_t                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zswap_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_double_complex**)x,
                                                             incx,
                                                             (rocblas_double_complex**)y,
                                                             incy,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCswapBatched_v2_64(hipblasHandle_t   handle,
                                          int64_t           n,
                                          hipComplex* const x[],
                                          int64_t           incx,
                                          hipComplex* const y[],
                                          int64_t           incy,
                                          int64_t           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cswap_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_float_complex**)x,
                                                             incx,
                                                             (rocblas_float_complex**)y,
                                                             incy,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZswapBatched_v2_64(hipblasHandle_t         handle,
                                          int64_t                 n,
                                          hipDoubleComplex* const x[],
                                          int64_t                 incx,
                                          hipDoubleComplex* const y[],
                                          int64_t                 incy,
                                          int64_t                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zswap_batched_64((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_double_complex**)x,
                                                             incx,
                                                             (rocblas_double_complex**)y,
                                                             incy,
                                                             batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// swap_strided_batched
hipblasStatus_t hipblasSswapStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           float*          x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           float*          y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sswap_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDswapStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           double*         x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           double*         y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dswap_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCswapStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           hipblasComplex* x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           hipblasComplex* y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cswap_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZswapStridedBatched(hipblasHandle_t       handle,
                                           int                   n,
                                           hipblasDoubleComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           hipblasDoubleComplex* y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zswap_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCswapStridedBatched_v2(hipblasHandle_t handle,
                                              int             n,
                                              hipComplex*     x,
                                              int             incx,
                                              hipblasStride   stridex,
                                              hipComplex*     y,
                                              int             incy,
                                              hipblasStride   stridey,
                                              int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cswap_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZswapStridedBatched_v2(hipblasHandle_t   handle,
                                              int               n,
                                              hipDoubleComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              hipDoubleComplex* y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zswap_strided_batched((rocblas_handle)handle,
                                                                  n,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// swap_strided_batched_64
hipblasStatus_t hipblasSswapStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              float*          x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              float*          y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sswap_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDswapStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              double*         x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              double*         y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dswap_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCswapStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              hipblasComplex* x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              hipblasComplex* y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cswap_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_float_complex*)x,
                                                                     incx,
                                                                     stridex,
                                                                     (rocblas_float_complex*)y,
                                                                     incy,
                                                                     stridey,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZswapStridedBatched_64(hipblasHandle_t       handle,
                                              int64_t               n,
                                              hipblasDoubleComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              hipblasDoubleComplex* y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              int64_t               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zswap_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_double_complex*)x,
                                                                     incx,
                                                                     stridex,
                                                                     (rocblas_double_complex*)y,
                                                                     incy,
                                                                     stridey,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCswapStridedBatched_v2_64(hipblasHandle_t handle,
                                                 int64_t         n,
                                                 hipComplex*     x,
                                                 int64_t         incx,
                                                 hipblasStride   stridex,
                                                 hipComplex*     y,
                                                 int64_t         incy,
                                                 hipblasStride   stridey,
                                                 int64_t         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cswap_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_float_complex*)x,
                                                                     incx,
                                                                     stridex,
                                                                     (rocblas_float_complex*)y,
                                                                     incy,
                                                                     stridey,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZswapStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 int64_t           n,
                                                 hipDoubleComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 hipDoubleComplex* y,
                                                 int64_t           incy,
                                                 hipblasStride     stridey,
                                                 int64_t           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zswap_strided_batched_64((rocblas_handle)handle,
                                                                     n,
                                                                     (rocblas_double_complex*)x,
                                                                     incx,
                                                                     stridex,
                                                                     (rocblas_double_complex*)y,
                                                                     incy,
                                                                     stridey,
                                                                     batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// gbmv
hipblasStatus_t hipblasSgbmv(hipblasHandle_t    handle,
                             hipblasOperation_t trans,
                             int                m,
                             int                n,
                             int                kl,
                             int                ku,
                             const float*       alpha,
                             const float*       A,
                             int                lda,
                             const float*       x,
                             int                incx,
                             const float*       beta,
                             float*             y,
                             int                incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sgbmv((rocblas_handle)handle,
                                                  hipOperationToHCCOperation(trans),
                                                  m,
                                                  n,
                                                  kl,
                                                  ku,
                                                  alpha,
                                                  A,
                                                  lda,
                                                  x,
                                                  incx,
                                                  beta,
                                                  y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDgbmv(hipblasHandle_t    handle,
                             hipblasOperation_t trans,
                             int                m,
                             int                n,
                             int                kl,
                             int                ku,
                             const double*      alpha,
                             const double*      A,
                             int                lda,
                             const double*      x,
                             int                incx,
                             const double*      beta,
                             double*            y,
                             int                incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dgbmv((rocblas_handle)handle,
                                                  hipOperationToHCCOperation(trans),
                                                  m,
                                                  n,
                                                  kl,
                                                  ku,
                                                  alpha,
                                                  A,
                                                  lda,
                                                  x,
                                                  incx,
                                                  beta,
                                                  y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgbmv(hipblasHandle_t       handle,
                             hipblasOperation_t    trans,
                             int                   m,
                             int                   n,
                             int                   kl,
                             int                   ku,
                             const hipblasComplex* alpha,
                             const hipblasComplex* A,
                             int                   lda,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* beta,
                             hipblasComplex*       y,
                             int                   incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgbmv((rocblas_handle)handle,
                                                  hipOperationToHCCOperation(trans),
                                                  m,
                                                  n,
                                                  kl,
                                                  ku,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)A,
                                                  lda,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)beta,
                                                  (rocblas_float_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgbmv(hipblasHandle_t             handle,
                             hipblasOperation_t          trans,
                             int                         m,
                             int                         n,
                             int                         kl,
                             int                         ku,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* A,
                             int                         lda,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* beta,
                             hipblasDoubleComplex*       y,
                             int                         incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgbmv((rocblas_handle)handle,
                                                  hipOperationToHCCOperation(trans),
                                                  m,
                                                  n,
                                                  kl,
                                                  ku,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)A,
                                                  lda,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)beta,
                                                  (rocblas_double_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgbmv_v2(hipblasHandle_t    handle,
                                hipblasOperation_t trans,
                                int                m,
                                int                n,
                                int                kl,
                                int                ku,
                                const hipComplex*  alpha,
                                const hipComplex*  A,
                                int                lda,
                                const hipComplex*  x,
                                int                incx,
                                const hipComplex*  beta,
                                hipComplex*        y,
                                int                incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgbmv((rocblas_handle)handle,
                                                  hipOperationToHCCOperation(trans),
                                                  m,
                                                  n,
                                                  kl,
                                                  ku,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)A,
                                                  lda,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)beta,
                                                  (rocblas_float_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgbmv_v2(hipblasHandle_t         handle,
                                hipblasOperation_t      trans,
                                int                     m,
                                int                     n,
                                int                     kl,
                                int                     ku,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* A,
                                int                     lda,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* beta,
                                hipDoubleComplex*       y,
                                int                     incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgbmv((rocblas_handle)handle,
                                                  hipOperationToHCCOperation(trans),
                                                  m,
                                                  n,
                                                  kl,
                                                  ku,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)A,
                                                  lda,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)beta,
                                                  (rocblas_double_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// gbmv_batched
hipblasStatus_t hipblasSgbmvBatched(hipblasHandle_t    handle,
                                    hipblasOperation_t trans,
                                    int                m,
                                    int                n,
                                    int                kl,
                                    int                ku,
                                    const float*       alpha,
                                    const float* const A[],
                                    int                lda,
                                    const float* const x[],
                                    int                incx,
                                    const float*       beta,
                                    float* const       y[],
                                    int                incy,
                                    int                batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sgbmv_batched((rocblas_handle)handle,
                                                          hipOperationToHCCOperation(trans),
                                                          m,
                                                          n,
                                                          kl,
                                                          ku,
                                                          alpha,
                                                          A,
                                                          lda,
                                                          x,
                                                          incx,
                                                          beta,
                                                          y,
                                                          incy,
                                                          batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDgbmvBatched(hipblasHandle_t     handle,
                                    hipblasOperation_t  trans,
                                    int                 m,
                                    int                 n,
                                    int                 kl,
                                    int                 ku,
                                    const double*       alpha,
                                    const double* const A[],
                                    int                 lda,
                                    const double* const x[],
                                    int                 incx,
                                    const double*       beta,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dgbmv_batched((rocblas_handle)handle,
                                                          hipOperationToHCCOperation(trans),
                                                          m,
                                                          n,
                                                          kl,
                                                          ku,
                                                          alpha,
                                                          A,
                                                          lda,
                                                          x,
                                                          incx,
                                                          beta,
                                                          y,
                                                          incy,
                                                          batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgbmvBatched(hipblasHandle_t             handle,
                                    hipblasOperation_t          trans,
                                    int                         m,
                                    int                         n,
                                    int                         kl,
                                    int                         ku,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const A[],
                                    int                         lda,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex*       beta,
                                    hipblasComplex* const       y[],
                                    int                         incy,
                                    int                         batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgbmv_batched((rocblas_handle)handle,
                                                          hipOperationToHCCOperation(trans),
                                                          m,
                                                          n,
                                                          kl,
                                                          ku,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex*)beta,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgbmvBatched(hipblasHandle_t                   handle,
                                    hipblasOperation_t                trans,
                                    int                               m,
                                    int                               n,
                                    int                               kl,
                                    int                               ku,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const A[],
                                    int                               lda,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex*       beta,
                                    hipblasDoubleComplex* const       y[],
                                    int                               incy,
                                    int                               batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgbmv_batched((rocblas_handle)handle,
                                                          hipOperationToHCCOperation(trans),
                                                          m,
                                                          n,
                                                          kl,
                                                          ku,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex*)beta,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgbmvBatched_v2(hipblasHandle_t         handle,
                                       hipblasOperation_t      trans,
                                       int                     m,
                                       int                     n,
                                       int                     kl,
                                       int                     ku,
                                       const hipComplex*       alpha,
                                       const hipComplex* const A[],
                                       int                     lda,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex*       beta,
                                       hipComplex* const       y[],
                                       int                     incy,
                                       int                     batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgbmv_batched((rocblas_handle)handle,
                                                          hipOperationToHCCOperation(trans),
                                                          m,
                                                          n,
                                                          kl,
                                                          ku,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex*)beta,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgbmvBatched_v2(hipblasHandle_t               handle,
                                       hipblasOperation_t            trans,
                                       int                           m,
                                       int                           n,
                                       int                           kl,
                                       int                           ku,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const A[],
                                       int                           lda,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex*       beta,
                                       hipDoubleComplex* const       y[],
                                       int                           incy,
                                       int                           batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgbmv_batched((rocblas_handle)handle,
                                                          hipOperationToHCCOperation(trans),
                                                          m,
                                                          n,
                                                          kl,
                                                          ku,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex*)beta,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// gbmv_strided_batched
hipblasStatus_t hipblasSgbmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasOperation_t trans,
                                           int                m,
                                           int                n,
                                           int                kl,
                                           int                ku,
                                           const float*       alpha,
                                           const float*       A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           const float*       x,
                                           int                incx,
                                           hipblasStride      stride_x,
                                           const float*       beta,
                                           float*             y,
                                           int                incy,
                                           hipblasStride      stride_y,
                                           int                batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sgbmv_strided_batched((rocblas_handle)handle,
                                                                  hipOperationToHCCOperation(trans),
                                                                  m,
                                                                  n,
                                                                  kl,
                                                                  ku,
                                                                  alpha,
                                                                  A,
                                                                  lda,
                                                                  stride_a,
                                                                  x,
                                                                  incx,
                                                                  stride_x,
                                                                  beta,
                                                                  y,
                                                                  incy,
                                                                  stride_y,
                                                                  batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDgbmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasOperation_t trans,
                                           int                m,
                                           int                n,
                                           int                kl,
                                           int                ku,
                                           const double*      alpha,
                                           const double*      A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           const double*      x,
                                           int                incx,
                                           hipblasStride      stride_x,
                                           const double*      beta,
                                           double*            y,
                                           int                incy,
                                           hipblasStride      stride_y,
                                           int                batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dgbmv_strided_batched((rocblas_handle)handle,
                                                                  hipOperationToHCCOperation(trans),
                                                                  m,
                                                                  n,
                                                                  kl,
                                                                  ku,
                                                                  alpha,
                                                                  A,
                                                                  lda,
                                                                  stride_a,
                                                                  x,
                                                                  incx,
                                                                  stride_x,
                                                                  beta,
                                                                  y,
                                                                  incy,
                                                                  stride_y,
                                                                  batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgbmvStridedBatched(hipblasHandle_t       handle,
                                           hipblasOperation_t    trans,
                                           int                   m,
                                           int                   n,
                                           int                   kl,
                                           int                   ku,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* A,
                                           int                   lda,
                                           hipblasStride         stride_a,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stride_x,
                                           const hipblasComplex* beta,
                                           hipblasComplex*       y,
                                           int                   incy,
                                           hipblasStride         stride_y,
                                           int                   batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgbmv_strided_batched((rocblas_handle)handle,
                                                                  hipOperationToHCCOperation(trans),
                                                                  m,
                                                                  n,
                                                                  kl,
                                                                  ku,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)A,
                                                                  lda,
                                                                  stride_a,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stride_x,
                                                                  (rocblas_float_complex*)beta,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stride_y,
                                                                  batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgbmvStridedBatched(hipblasHandle_t             handle,
                                           hipblasOperation_t          trans,
                                           int                         m,
                                           int                         n,
                                           int                         kl,
                                           int                         ku,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* A,
                                           int                         lda,
                                           hipblasStride               stride_a,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stride_x,
                                           const hipblasDoubleComplex* beta,
                                           hipblasDoubleComplex*       y,
                                           int                         incy,
                                           hipblasStride               stride_y,
                                           int                         batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgbmv_strided_batched((rocblas_handle)handle,
                                                                  hipOperationToHCCOperation(trans),
                                                                  m,
                                                                  n,
                                                                  kl,
                                                                  ku,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)A,
                                                                  lda,
                                                                  stride_a,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stride_x,
                                                                  (rocblas_double_complex*)beta,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stride_y,
                                                                  batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgbmvStridedBatched_v2(hipblasHandle_t    handle,
                                              hipblasOperation_t trans,
                                              int                m,
                                              int                n,
                                              int                kl,
                                              int                ku,
                                              const hipComplex*  alpha,
                                              const hipComplex*  A,
                                              int                lda,
                                              hipblasStride      stride_a,
                                              const hipComplex*  x,
                                              int                incx,
                                              hipblasStride      stride_x,
                                              const hipComplex*  beta,
                                              hipComplex*        y,
                                              int                incy,
                                              hipblasStride      stride_y,
                                              int                batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgbmv_strided_batched((rocblas_handle)handle,
                                                                  hipOperationToHCCOperation(trans),
                                                                  m,
                                                                  n,
                                                                  kl,
                                                                  ku,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)A,
                                                                  lda,
                                                                  stride_a,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stride_x,
                                                                  (rocblas_float_complex*)beta,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stride_y,
                                                                  batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgbmvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasOperation_t      trans,
                                              int                     m,
                                              int                     n,
                                              int                     kl,
                                              int                     ku,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* A,
                                              int                     lda,
                                              hipblasStride           stride_a,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stride_x,
                                              const hipDoubleComplex* beta,
                                              hipDoubleComplex*       y,
                                              int                     incy,
                                              hipblasStride           stride_y,
                                              int                     batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgbmv_strided_batched((rocblas_handle)handle,
                                                                  hipOperationToHCCOperation(trans),
                                                                  m,
                                                                  n,
                                                                  kl,
                                                                  ku,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)A,
                                                                  lda,
                                                                  stride_a,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stride_x,
                                                                  (rocblas_double_complex*)beta,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stride_y,
                                                                  batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// gemv
hipblasStatus_t hipblasSgemv(hipblasHandle_t    handle,
                             hipblasOperation_t trans,
                             int                m,
                             int                n,
                             const float*       alpha,
                             const float*       A,
                             int                lda,
                             const float*       x,
                             int                incx,
                             const float*       beta,
                             float*             y,
                             int                incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sgemv((rocblas_handle)handle,
                                                  hipOperationToHCCOperation(trans),
                                                  m,
                                                  n,
                                                  alpha,
                                                  A,
                                                  lda,
                                                  x,
                                                  incx,
                                                  beta,
                                                  y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDgemv(hipblasHandle_t    handle,
                             hipblasOperation_t trans,
                             int                m,
                             int                n,
                             const double*      alpha,
                             const double*      A,
                             int                lda,
                             const double*      x,
                             int                incx,
                             const double*      beta,
                             double*            y,
                             int                incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dgemv((rocblas_handle)handle,
                                                  hipOperationToHCCOperation(trans),
                                                  m,
                                                  n,
                                                  alpha,
                                                  A,
                                                  lda,
                                                  x,
                                                  incx,
                                                  beta,
                                                  y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgemv(hipblasHandle_t       handle,
                             hipblasOperation_t    trans,
                             int                   m,
                             int                   n,
                             const hipblasComplex* alpha,
                             const hipblasComplex* A,
                             int                   lda,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* beta,
                             hipblasComplex*       y,
                             int                   incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgemv((rocblas_handle)handle,
                                                  hipOperationToHCCOperation(trans),
                                                  m,
                                                  n,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)A,
                                                  lda,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)beta,
                                                  (rocblas_float_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgemv(hipblasHandle_t             handle,
                             hipblasOperation_t          trans,
                             int                         m,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* A,
                             int                         lda,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* beta,
                             hipblasDoubleComplex*       y,
                             int                         incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgemv((rocblas_handle)handle,
                                                  hipOperationToHCCOperation(trans),
                                                  m,
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)A,
                                                  lda,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)beta,
                                                  (rocblas_double_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgemv_v2(hipblasHandle_t    handle,
                                hipblasOperation_t trans,
                                int                m,
                                int                n,
                                const hipComplex*  alpha,
                                const hipComplex*  A,
                                int                lda,
                                const hipComplex*  x,
                                int                incx,
                                const hipComplex*  beta,
                                hipComplex*        y,
                                int                incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgemv((rocblas_handle)handle,
                                                  hipOperationToHCCOperation(trans),
                                                  m,
                                                  n,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)A,
                                                  lda,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)beta,
                                                  (rocblas_float_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgemv_v2(hipblasHandle_t         handle,
                                hipblasOperation_t      trans,
                                int                     m,
                                int                     n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* A,
                                int                     lda,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* beta,
                                hipDoubleComplex*       y,
                                int                     incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgemv((rocblas_handle)handle,
                                                  hipOperationToHCCOperation(trans),
                                                  m,
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)A,
                                                  lda,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)beta,
                                                  (rocblas_double_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// gemv_batched
hipblasStatus_t hipblasSgemvBatched(hipblasHandle_t    handle,
                                    hipblasOperation_t trans,
                                    int                m,
                                    int                n,
                                    const float*       alpha,
                                    const float* const A[],
                                    int                lda,
                                    const float* const x[],
                                    int                incx,
                                    const float*       beta,
                                    float* const       y[],
                                    int                incy,
                                    int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sgemv_batched((rocblas_handle)handle,
                                                          hipOperationToHCCOperation(trans),
                                                          m,
                                                          n,
                                                          alpha,
                                                          A,
                                                          lda,
                                                          x,
                                                          incx,
                                                          beta,
                                                          y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDgemvBatched(hipblasHandle_t     handle,
                                    hipblasOperation_t  trans,
                                    int                 m,
                                    int                 n,
                                    const double*       alpha,
                                    const double* const A[],
                                    int                 lda,
                                    const double* const x[],
                                    int                 incx,
                                    const double*       beta,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dgemv_batched((rocblas_handle)handle,
                                                          hipOperationToHCCOperation(trans),
                                                          m,
                                                          n,
                                                          alpha,
                                                          A,
                                                          lda,
                                                          x,
                                                          incx,
                                                          beta,
                                                          y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgemvBatched(hipblasHandle_t             handle,
                                    hipblasOperation_t          trans,
                                    int                         m,
                                    int                         n,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const A[],
                                    int                         lda,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex*       beta,
                                    hipblasComplex* const       y[],
                                    int                         incy,
                                    int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgemv_batched((rocblas_handle)handle,
                                                          hipOperationToHCCOperation(trans),
                                                          m,
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex*)beta,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgemvBatched(hipblasHandle_t                   handle,
                                    hipblasOperation_t                trans,
                                    int                               m,
                                    int                               n,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const A[],
                                    int                               lda,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex*       beta,
                                    hipblasDoubleComplex* const       y[],
                                    int                               incy,
                                    int                               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgemv_batched((rocblas_handle)handle,
                                                          hipOperationToHCCOperation(trans),
                                                          m,
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex*)beta,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgemvBatched_v2(hipblasHandle_t         handle,
                                       hipblasOperation_t      trans,
                                       int                     m,
                                       int                     n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const A[],
                                       int                     lda,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex*       beta,
                                       hipComplex* const       y[],
                                       int                     incy,
                                       int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgemv_batched((rocblas_handle)handle,
                                                          hipOperationToHCCOperation(trans),
                                                          m,
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex*)beta,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgemvBatched_v2(hipblasHandle_t               handle,
                                       hipblasOperation_t            trans,
                                       int                           m,
                                       int                           n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const A[],
                                       int                           lda,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex*       beta,
                                       hipDoubleComplex* const       y[],
                                       int                           incy,
                                       int                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgemv_batched((rocblas_handle)handle,
                                                          hipOperationToHCCOperation(trans),
                                                          m,
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex*)beta,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// gemv_strided_batched
hipblasStatus_t hipblasSgemvStridedBatched(hipblasHandle_t    handle,
                                           hipblasOperation_t trans,
                                           int                m,
                                           int                n,
                                           const float*       alpha,
                                           const float*       A,
                                           int                lda,
                                           hipblasStride      strideA,
                                           const float*       x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           const float*       beta,
                                           float*             y,
                                           int                incy,
                                           hipblasStride      stridey,
                                           int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sgemv_strided_batched((rocblas_handle)handle,
                                                                  hipOperationToHCCOperation(trans),
                                                                  m,
                                                                  n,
                                                                  alpha,
                                                                  A,
                                                                  lda,
                                                                  strideA,
                                                                  x,
                                                                  incx,
                                                                  stridex,
                                                                  beta,
                                                                  y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDgemvStridedBatched(hipblasHandle_t    handle,
                                           hipblasOperation_t trans,
                                           int                m,
                                           int                n,
                                           const double*      alpha,
                                           const double*      A,
                                           int                lda,
                                           hipblasStride      strideA,
                                           const double*      x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           const double*      beta,
                                           double*            y,
                                           int                incy,
                                           hipblasStride      stridey,
                                           int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dgemv_strided_batched((rocblas_handle)handle,
                                                                  hipOperationToHCCOperation(trans),
                                                                  m,
                                                                  n,
                                                                  alpha,
                                                                  A,
                                                                  lda,
                                                                  strideA,
                                                                  x,
                                                                  incx,
                                                                  stridex,
                                                                  beta,
                                                                  y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgemvStridedBatched(hipblasHandle_t       handle,
                                           hipblasOperation_t    trans,
                                           int                   m,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* A,
                                           int                   lda,
                                           hipblasStride         strideA,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* beta,
                                           hipblasComplex*       y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgemv_strided_batched((rocblas_handle)handle,
                                                                  hipOperationToHCCOperation(trans),
                                                                  m,
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)beta,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgemvStridedBatched(hipblasHandle_t             handle,
                                           hipblasOperation_t          trans,
                                           int                         m,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* A,
                                           int                         lda,
                                           hipblasStride               strideA,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* beta,
                                           hipblasDoubleComplex*       y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgemv_strided_batched((rocblas_handle)handle,
                                                                  hipOperationToHCCOperation(trans),
                                                                  m,
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)beta,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgemvStridedBatched_v2(hipblasHandle_t    handle,
                                              hipblasOperation_t trans,
                                              int                m,
                                              int                n,
                                              const hipComplex*  alpha,
                                              const hipComplex*  A,
                                              int                lda,
                                              hipblasStride      strideA,
                                              const hipComplex*  x,
                                              int                incx,
                                              hipblasStride      stridex,
                                              const hipComplex*  beta,
                                              hipComplex*        y,
                                              int                incy,
                                              hipblasStride      stridey,
                                              int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgemv_strided_batched((rocblas_handle)handle,
                                                                  hipOperationToHCCOperation(trans),
                                                                  m,
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)beta,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgemvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasOperation_t      trans,
                                              int                     m,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* A,
                                              int                     lda,
                                              hipblasStride           strideA,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* beta,
                                              hipDoubleComplex*       y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgemv_strided_batched((rocblas_handle)handle,
                                                                  hipOperationToHCCOperation(trans),
                                                                  m,
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)beta,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// ger
hipblasStatus_t hipblasSger(hipblasHandle_t handle,
                            int             m,
                            int             n,
                            const float*    alpha,
                            const float*    x,
                            int             incx,
                            const float*    y,
                            int             incy,
                            float*          A,
                            int             lda)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_sger((rocblas_handle)handle, m, n, alpha, x, incx, y, incy, A, lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDger(hipblasHandle_t handle,
                            int             m,
                            int             n,
                            const double*   alpha,
                            const double*   x,
                            int             incx,
                            const double*   y,
                            int             incy,
                            double*         A,
                            int             lda)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dger((rocblas_handle)handle, m, n, alpha, x, incx, y, incy, A, lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgeru(hipblasHandle_t       handle,
                             int                   m,
                             int                   n,
                             const hipblasComplex* alpha,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* y,
                             int                   incy,
                             hipblasComplex*       A,
                             int                   lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgeru((rocblas_handle)handle,
                                                  m,
                                                  n,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy,
                                                  (rocblas_float_complex*)A,
                                                  lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgerc(hipblasHandle_t       handle,
                             int                   m,
                             int                   n,
                             const hipblasComplex* alpha,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* y,
                             int                   incy,
                             hipblasComplex*       A,
                             int                   lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgerc((rocblas_handle)handle,
                                                  m,
                                                  n,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy,
                                                  (rocblas_float_complex*)A,
                                                  lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgeru(hipblasHandle_t             handle,
                             int                         m,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* y,
                             int                         incy,
                             hipblasDoubleComplex*       A,
                             int                         lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgeru((rocblas_handle)handle,
                                                  m,
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy,
                                                  (rocblas_double_complex*)A,
                                                  lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgerc(hipblasHandle_t             handle,
                             int                         m,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* y,
                             int                         incy,
                             hipblasDoubleComplex*       A,
                             int                         lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgerc((rocblas_handle)handle,
                                                  m,
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy,
                                                  (rocblas_double_complex*)A,
                                                  lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgeru_v2(hipblasHandle_t   handle,
                                int               m,
                                int               n,
                                const hipComplex* alpha,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* y,
                                int               incy,
                                hipComplex*       A,
                                int               lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgeru((rocblas_handle)handle,
                                                  m,
                                                  n,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy,
                                                  (rocblas_float_complex*)A,
                                                  lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgerc_v2(hipblasHandle_t   handle,
                                int               m,
                                int               n,
                                const hipComplex* alpha,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* y,
                                int               incy,
                                hipComplex*       A,
                                int               lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgerc((rocblas_handle)handle,
                                                  m,
                                                  n,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy,
                                                  (rocblas_float_complex*)A,
                                                  lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgeru_v2(hipblasHandle_t         handle,
                                int                     m,
                                int                     n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* y,
                                int                     incy,
                                hipDoubleComplex*       A,
                                int                     lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgeru((rocblas_handle)handle,
                                                  m,
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy,
                                                  (rocblas_double_complex*)A,
                                                  lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgerc_v2(hipblasHandle_t         handle,
                                int                     m,
                                int                     n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* y,
                                int                     incy,
                                hipDoubleComplex*       A,
                                int                     lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgerc((rocblas_handle)handle,
                                                  m,
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy,
                                                  (rocblas_double_complex*)A,
                                                  lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// ger_batched
hipblasStatus_t hipblasSgerBatched(hipblasHandle_t    handle,
                                   int                m,
                                   int                n,
                                   const float*       alpha,
                                   const float* const x[],
                                   int                incx,
                                   const float* const y[],
                                   int                incy,
                                   float* const       A[],
                                   int                lda,
                                   int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sger_batched(
        (rocblas_handle)handle, m, n, alpha, x, incx, y, incy, A, lda, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDgerBatched(hipblasHandle_t     handle,
                                   int                 m,
                                   int                 n,
                                   const double*       alpha,
                                   const double* const x[],
                                   int                 incx,
                                   const double* const y[],
                                   int                 incy,
                                   double* const       A[],
                                   int                 lda,
                                   int                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dger_batched(
        (rocblas_handle)handle, m, n, alpha, x, incx, y, incy, A, lda, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgeruBatched(hipblasHandle_t             handle,
                                    int                         m,
                                    int                         n,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex* const y[],
                                    int                         incy,
                                    hipblasComplex* const       A[],
                                    int                         lda,
                                    int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgeru_batched((rocblas_handle)handle,
                                                          m,
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgercBatched(hipblasHandle_t             handle,
                                    int                         m,
                                    int                         n,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex* const y[],
                                    int                         incy,
                                    hipblasComplex* const       A[],
                                    int                         lda,
                                    int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgerc_batched((rocblas_handle)handle,
                                                          m,
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgeruBatched(hipblasHandle_t                   handle,
                                    int                               m,
                                    int                               n,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex* const y[],
                                    int                               incy,
                                    hipblasDoubleComplex* const       A[],
                                    int                               lda,
                                    int                               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgeru_batched((rocblas_handle)handle,
                                                          m,
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgercBatched(hipblasHandle_t                   handle,
                                    int                               m,
                                    int                               n,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex* const y[],
                                    int                               incy,
                                    hipblasDoubleComplex* const       A[],
                                    int                               lda,
                                    int                               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgerc_batched((rocblas_handle)handle,
                                                          m,
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgeruBatched_v2(hipblasHandle_t         handle,
                                       int                     m,
                                       int                     n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex* const y[],
                                       int                     incy,
                                       hipComplex* const       A[],
                                       int                     lda,
                                       int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgeru_batched((rocblas_handle)handle,
                                                          m,
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgercBatched_v2(hipblasHandle_t         handle,
                                       int                     m,
                                       int                     n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex* const y[],
                                       int                     incy,
                                       hipComplex* const       A[],
                                       int                     lda,
                                       int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgerc_batched((rocblas_handle)handle,
                                                          m,
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgeruBatched_v2(hipblasHandle_t               handle,
                                       int                           m,
                                       int                           n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex* const y[],
                                       int                           incy,
                                       hipDoubleComplex* const       A[],
                                       int                           lda,
                                       int                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgeru_batched((rocblas_handle)handle,
                                                          m,
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgercBatched_v2(hipblasHandle_t               handle,
                                       int                           m,
                                       int                           n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex* const y[],
                                       int                           incy,
                                       hipDoubleComplex* const       A[],
                                       int                           lda,
                                       int                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgerc_batched((rocblas_handle)handle,
                                                          m,
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// ger_strided_batched
hipblasStatus_t hipblasSgerStridedBatched(hipblasHandle_t handle,
                                          int             m,
                                          int             n,
                                          const float*    alpha,
                                          const float*    x,
                                          int             incx,
                                          hipblasStride   stridex,
                                          const float*    y,
                                          int             incy,
                                          hipblasStride   stridey,
                                          float*          A,
                                          int             lda,
                                          hipblasStride   strideA,
                                          int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sger_strided_batched((rocblas_handle)handle,
                                                                 m,
                                                                 n,
                                                                 alpha,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 y,
                                                                 incy,
                                                                 stridey,
                                                                 A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDgerStridedBatched(hipblasHandle_t handle,
                                          int             m,
                                          int             n,
                                          const double*   alpha,
                                          const double*   x,
                                          int             incx,
                                          hipblasStride   stridex,
                                          const double*   y,
                                          int             incy,
                                          hipblasStride   stridey,
                                          double*         A,
                                          int             lda,
                                          hipblasStride   strideA,
                                          int             batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dger_strided_batched((rocblas_handle)handle,
                                                                 m,
                                                                 n,
                                                                 alpha,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 y,
                                                                 incy,
                                                                 stridey,
                                                                 A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgeruStridedBatched(hipblasHandle_t       handle,
                                           int                   m,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           hipblasComplex*       A,
                                           int                   lda,
                                           hipblasStride         strideA,
                                           int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgeru_strided_batched((rocblas_handle)handle,
                                                                  m,
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  (rocblas_float_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgercStridedBatched(hipblasHandle_t       handle,
                                           int                   m,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           hipblasComplex*       A,
                                           int                   lda,
                                           hipblasStride         strideA,
                                           int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgerc_strided_batched((rocblas_handle)handle,
                                                                  m,
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  (rocblas_float_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgeruStridedBatched(hipblasHandle_t             handle,
                                           int                         m,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           hipblasDoubleComplex*       A,
                                           int                         lda,
                                           hipblasStride               strideA,
                                           int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgeru_strided_batched((rocblas_handle)handle,
                                                                  m,
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  (rocblas_double_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgercStridedBatched(hipblasHandle_t             handle,
                                           int                         m,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           hipblasDoubleComplex*       A,
                                           int                         lda,
                                           hipblasStride               strideA,
                                           int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgerc_strided_batched((rocblas_handle)handle,
                                                                  m,
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  (rocblas_double_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgeruStridedBatched_v2(hipblasHandle_t   handle,
                                              int               m,
                                              int               n,
                                              const hipComplex* alpha,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              hipComplex*       A,
                                              int               lda,
                                              hipblasStride     strideA,
                                              int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgeru_strided_batched((rocblas_handle)handle,
                                                                  m,
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  (rocblas_float_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCgercStridedBatched_v2(hipblasHandle_t   handle,
                                              int               m,
                                              int               n,
                                              const hipComplex* alpha,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              hipComplex*       A,
                                              int               lda,
                                              hipblasStride     strideA,
                                              int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cgerc_strided_batched((rocblas_handle)handle,
                                                                  m,
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  (rocblas_float_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgeruStridedBatched_v2(hipblasHandle_t         handle,
                                              int                     m,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              hipDoubleComplex*       A,
                                              int                     lda,
                                              hipblasStride           strideA,
                                              int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgeru_strided_batched((rocblas_handle)handle,
                                                                  m,
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  (rocblas_double_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZgercStridedBatched_v2(hipblasHandle_t         handle,
                                              int                     m,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              hipDoubleComplex*       A,
                                              int                     lda,
                                              hipblasStride           strideA,
                                              int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zgerc_strided_batched((rocblas_handle)handle,
                                                                  m,
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  (rocblas_double_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// hbmv
hipblasStatus_t hipblasChbmv(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             int                   n,
                             int                   k,
                             const hipblasComplex* alpha,
                             const hipblasComplex* A,
                             int                   lda,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* beta,
                             hipblasComplex*       y,
                             int                   incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chbmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  k,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)A,
                                                  lda,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)beta,
                                                  (rocblas_float_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhbmv(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             int                         n,
                             int                         k,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* A,
                             int                         lda,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* beta,
                             hipblasDoubleComplex*       y,
                             int                         incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhbmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  k,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)A,
                                                  lda,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)beta,
                                                  (rocblas_double_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasChbmv_v2(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int               n,
                                int               k,
                                const hipComplex* alpha,
                                const hipComplex* A,
                                int               lda,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* beta,
                                hipComplex*       y,
                                int               incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chbmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  k,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)A,
                                                  lda,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)beta,
                                                  (rocblas_float_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhbmv_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int                     n,
                                int                     k,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* A,
                                int                     lda,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* beta,
                                hipDoubleComplex*       y,
                                int                     incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhbmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  k,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)A,
                                                  lda,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)beta,
                                                  (rocblas_double_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// hbmv_batched
hipblasStatus_t hipblasChbmvBatched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    int                         n,
                                    int                         k,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const A[],
                                    int                         lda,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex*       beta,
                                    hipblasComplex* const       y[],
                                    int                         incy,
                                    int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chbmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          k,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex*)beta,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhbmvBatched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    int                               n,
                                    int                               k,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const A[],
                                    int                               lda,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex*       beta,
                                    hipblasDoubleComplex* const       y[],
                                    int                               incy,
                                    int                               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhbmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          k,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex*)beta,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasChbmvBatched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int                     n,
                                       int                     k,
                                       const hipComplex*       alpha,
                                       const hipComplex* const A[],
                                       int                     lda,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex*       beta,
                                       hipComplex* const       y[],
                                       int                     incy,
                                       int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chbmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          k,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex*)beta,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhbmvBatched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int                           n,
                                       int                           k,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const A[],
                                       int                           lda,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex*       beta,
                                       hipDoubleComplex* const       y[],
                                       int                           incy,
                                       int                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhbmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          k,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex*)beta,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// hbmv_strided_batched
hipblasStatus_t hipblasChbmvStridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           int                   n,
                                           int                   k,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* A,
                                           int                   lda,
                                           hipblasStride         strideA,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* beta,
                                           hipblasComplex*       y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chbmv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  k,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)beta,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhbmvStridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           int                         n,
                                           int                         k,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* A,
                                           int                         lda,
                                           hipblasStride               strideA,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* beta,
                                           hipblasDoubleComplex*       y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhbmv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  k,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)beta,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasChbmvStridedBatched_v2(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int               n,
                                              int               k,
                                              const hipComplex* alpha,
                                              const hipComplex* A,
                                              int               lda,
                                              hipblasStride     strideA,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              const hipComplex* beta,
                                              hipComplex*       y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chbmv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  k,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)beta,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhbmvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int                     n,
                                              int                     k,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* A,
                                              int                     lda,
                                              hipblasStride           strideA,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* beta,
                                              hipDoubleComplex*       y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhbmv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  k,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)beta,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// hemv
hipblasStatus_t hipblasChemv(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             int                   n,
                             const hipblasComplex* alpha,
                             const hipblasComplex* A,
                             int                   lda,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* beta,
                             hipblasComplex*       y,
                             int                   incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chemv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)A,
                                                  lda,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)beta,
                                                  (rocblas_float_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhemv(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* A,
                             int                         lda,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* beta,
                             hipblasDoubleComplex*       y,
                             int                         incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhemv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)A,
                                                  lda,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)beta,
                                                  (rocblas_double_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasChemv_v2(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int               n,
                                const hipComplex* alpha,
                                const hipComplex* A,
                                int               lda,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* beta,
                                hipComplex*       y,
                                int               incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chemv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)A,
                                                  lda,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)beta,
                                                  (rocblas_float_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhemv_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int                     n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* A,
                                int                     lda,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* beta,
                                hipDoubleComplex*       y,
                                int                     incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhemv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)A,
                                                  lda,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)beta,
                                                  (rocblas_double_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// hemv_batched
hipblasStatus_t hipblasChemvBatched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    int                         n,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const A[],
                                    int                         lda,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex*       beta,
                                    hipblasComplex* const       y[],
                                    int                         incy,
                                    int                         batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chemv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex*)beta,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhemvBatched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    int                               n,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const A[],
                                    int                               lda,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex*       beta,
                                    hipblasDoubleComplex* const       y[],
                                    int                               incy,
                                    int                               batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhemv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex*)beta,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasChemvBatched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int                     n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const A[],
                                       int                     lda,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex*       beta,
                                       hipComplex* const       y[],
                                       int                     incy,
                                       int                     batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chemv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex*)beta,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhemvBatched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int                           n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const A[],
                                       int                           lda,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex*       beta,
                                       hipDoubleComplex* const       y[],
                                       int                           incy,
                                       int                           batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhemv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex*)beta,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// hemv_strided_batched
hipblasStatus_t hipblasChemvStridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* A,
                                           int                   lda,
                                           hipblasStride         stride_a,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stride_x,
                                           const hipblasComplex* beta,
                                           hipblasComplex*       y,
                                           int                   incy,
                                           hipblasStride         stride_y,
                                           int                   batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chemv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)A,
                                                                  lda,
                                                                  stride_a,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stride_x,
                                                                  (rocblas_float_complex*)beta,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stride_y,
                                                                  batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhemvStridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* A,
                                           int                         lda,
                                           hipblasStride               stride_a,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stride_x,
                                           const hipblasDoubleComplex* beta,
                                           hipblasDoubleComplex*       y,
                                           int                         incy,
                                           hipblasStride               stride_y,
                                           int                         batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhemv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)A,
                                                                  lda,
                                                                  stride_a,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stride_x,
                                                                  (rocblas_double_complex*)beta,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stride_y,
                                                                  batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasChemvStridedBatched_v2(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int               n,
                                              const hipComplex* alpha,
                                              const hipComplex* A,
                                              int               lda,
                                              hipblasStride     stride_a,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stride_x,
                                              const hipComplex* beta,
                                              hipComplex*       y,
                                              int               incy,
                                              hipblasStride     stride_y,
                                              int               batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chemv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)A,
                                                                  lda,
                                                                  stride_a,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stride_x,
                                                                  (rocblas_float_complex*)beta,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stride_y,
                                                                  batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhemvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* A,
                                              int                     lda,
                                              hipblasStride           stride_a,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stride_x,
                                              const hipDoubleComplex* beta,
                                              hipDoubleComplex*       y,
                                              int                     incy,
                                              hipblasStride           stride_y,
                                              int                     batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhemv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)A,
                                                                  lda,
                                                                  stride_a,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stride_x,
                                                                  (rocblas_double_complex*)beta,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stride_y,
                                                                  batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// her
hipblasStatus_t hipblasCher(hipblasHandle_t       handle,
                            hipblasFillMode_t     uplo,
                            int                   n,
                            const float*          alpha,
                            const hipblasComplex* x,
                            int                   incx,
                            hipblasComplex*       A,
                            int                   lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cher((rocblas_handle)handle,
                                                 hipFillToHCCFill(uplo),
                                                 n,
                                                 alpha,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)A,
                                                 lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZher(hipblasHandle_t             handle,
                            hipblasFillMode_t           uplo,
                            int                         n,
                            const double*               alpha,
                            const hipblasDoubleComplex* x,
                            int                         incx,
                            hipblasDoubleComplex*       A,
                            int                         lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zher((rocblas_handle)handle,
                                                 hipFillToHCCFill(uplo),
                                                 n,
                                                 alpha,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)A,
                                                 lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCher_v2(hipblasHandle_t   handle,
                               hipblasFillMode_t uplo,
                               int               n,
                               const float*      alpha,
                               const hipComplex* x,
                               int               incx,
                               hipComplex*       A,
                               int               lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cher((rocblas_handle)handle,
                                                 hipFillToHCCFill(uplo),
                                                 n,
                                                 alpha,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)A,
                                                 lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZher_v2(hipblasHandle_t         handle,
                               hipblasFillMode_t       uplo,
                               int                     n,
                               const double*           alpha,
                               const hipDoubleComplex* x,
                               int                     incx,
                               hipDoubleComplex*       A,
                               int                     lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zher((rocblas_handle)handle,
                                                 hipFillToHCCFill(uplo),
                                                 n,
                                                 alpha,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)A,
                                                 lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// her_batched
hipblasStatus_t hipblasCherBatched(hipblasHandle_t             handle,
                                   hipblasFillMode_t           uplo,
                                   int                         n,
                                   const float*                alpha,
                                   const hipblasComplex* const x[],
                                   int                         incx,
                                   hipblasComplex* const       A[],
                                   int                         lda,
                                   int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cher_batched((rocblas_handle)handle,
                                                         hipFillToHCCFill(uplo),
                                                         n,
                                                         alpha,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex**)A,
                                                         lda,
                                                         batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZherBatched(hipblasHandle_t                   handle,
                                   hipblasFillMode_t                 uplo,
                                   int                               n,
                                   const double*                     alpha,
                                   const hipblasDoubleComplex* const x[],
                                   int                               incx,
                                   hipblasDoubleComplex* const       A[],
                                   int                               lda,
                                   int                               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zher_batched((rocblas_handle)handle,
                                                         hipFillToHCCFill(uplo),
                                                         n,
                                                         alpha,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex**)A,
                                                         lda,
                                                         batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCherBatched_v2(hipblasHandle_t         handle,
                                      hipblasFillMode_t       uplo,
                                      int                     n,
                                      const float*            alpha,
                                      const hipComplex* const x[],
                                      int                     incx,
                                      hipComplex* const       A[],
                                      int                     lda,
                                      int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cher_batched((rocblas_handle)handle,
                                                         hipFillToHCCFill(uplo),
                                                         n,
                                                         alpha,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex**)A,
                                                         lda,
                                                         batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZherBatched_v2(hipblasHandle_t               handle,
                                      hipblasFillMode_t             uplo,
                                      int                           n,
                                      const double*                 alpha,
                                      const hipDoubleComplex* const x[],
                                      int                           incx,
                                      hipDoubleComplex* const       A[],
                                      int                           lda,
                                      int                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zher_batched((rocblas_handle)handle,
                                                         hipFillToHCCFill(uplo),
                                                         n,
                                                         alpha,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex**)A,
                                                         lda,
                                                         batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// her_strided_batched
hipblasStatus_t hipblasCherStridedBatched(hipblasHandle_t       handle,
                                          hipblasFillMode_t     uplo,
                                          int                   n,
                                          const float*          alpha,
                                          const hipblasComplex* x,
                                          int                   incx,
                                          hipblasStride         stridex,
                                          hipblasComplex*       A,
                                          int                   lda,
                                          hipblasStride         strideA,
                                          int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cher_strided_batched((rocblas_handle)handle,
                                                                 hipFillToHCCFill(uplo),
                                                                 n,
                                                                 alpha,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZherStridedBatched(hipblasHandle_t             handle,
                                          hipblasFillMode_t           uplo,
                                          int                         n,
                                          const double*               alpha,
                                          const hipblasDoubleComplex* x,
                                          int                         incx,
                                          hipblasStride               stridex,
                                          hipblasDoubleComplex*       A,
                                          int                         lda,
                                          hipblasStride               strideA,
                                          int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zher_strided_batched((rocblas_handle)handle,
                                                                 hipFillToHCCFill(uplo),
                                                                 n,
                                                                 alpha,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCherStridedBatched_v2(hipblasHandle_t   handle,
                                             hipblasFillMode_t uplo,
                                             int               n,
                                             const float*      alpha,
                                             const hipComplex* x,
                                             int               incx,
                                             hipblasStride     stridex,
                                             hipComplex*       A,
                                             int               lda,
                                             hipblasStride     strideA,
                                             int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cher_strided_batched((rocblas_handle)handle,
                                                                 hipFillToHCCFill(uplo),
                                                                 n,
                                                                 alpha,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZherStridedBatched_v2(hipblasHandle_t         handle,
                                             hipblasFillMode_t       uplo,
                                             int                     n,
                                             const double*           alpha,
                                             const hipDoubleComplex* x,
                                             int                     incx,
                                             hipblasStride           stridex,
                                             hipDoubleComplex*       A,
                                             int                     lda,
                                             hipblasStride           strideA,
                                             int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zher_strided_batched((rocblas_handle)handle,
                                                                 hipFillToHCCFill(uplo),
                                                                 n,
                                                                 alpha,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// her2
hipblasStatus_t hipblasCher2(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             int                   n,
                             const hipblasComplex* alpha,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* y,
                             int                   incy,
                             hipblasComplex*       A,
                             int                   lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cher2((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy,
                                                  (rocblas_float_complex*)A,
                                                  lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZher2(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* y,
                             int                         incy,
                             hipblasDoubleComplex*       A,
                             int                         lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zher2((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy,
                                                  (rocblas_double_complex*)A,
                                                  lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCher2_v2(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int               n,
                                const hipComplex* alpha,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* y,
                                int               incy,
                                hipComplex*       A,
                                int               lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cher2((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy,
                                                  (rocblas_float_complex*)A,
                                                  lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZher2_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int                     n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* y,
                                int                     incy,
                                hipDoubleComplex*       A,
                                int                     lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zher2((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy,
                                                  (rocblas_double_complex*)A,
                                                  lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// her2_batched
hipblasStatus_t hipblasCher2Batched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    int                         n,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex* const y[],
                                    int                         incy,
                                    hipblasComplex* const       A[],
                                    int                         lda,
                                    int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cher2_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZher2Batched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    int                               n,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex* const y[],
                                    int                               incy,
                                    hipblasDoubleComplex* const       A[],
                                    int                               lda,
                                    int                               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zher2_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCher2Batched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int                     n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex* const y[],
                                       int                     incy,
                                       hipComplex* const       A[],
                                       int                     lda,
                                       int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cher2_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZher2Batched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int                           n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex* const y[],
                                       int                           incy,
                                       hipDoubleComplex* const       A[],
                                       int                           lda,
                                       int                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zher2_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// her2_strided_batched
hipblasStatus_t hipblasCher2StridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           hipblasComplex*       A,
                                           int                   lda,
                                           hipblasStride         strideA,
                                           int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cher2_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  (rocblas_float_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZher2StridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           hipblasDoubleComplex*       A,
                                           int                         lda,
                                           hipblasStride               strideA,
                                           int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zher2_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  (rocblas_double_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCher2StridedBatched_v2(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int               n,
                                              const hipComplex* alpha,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              hipComplex*       A,
                                              int               lda,
                                              hipblasStride     strideA,
                                              int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cher2_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  (rocblas_float_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZher2StridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              hipDoubleComplex*       A,
                                              int                     lda,
                                              hipblasStride           strideA,
                                              int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zher2_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  (rocblas_double_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// hpmv
hipblasStatus_t hipblasChpmv(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             int                   n,
                             const hipblasComplex* alpha,
                             const hipblasComplex* AP,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* beta,
                             hipblasComplex*       y,
                             int                   incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chpmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)AP,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)beta,
                                                  (rocblas_float_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhpmv(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* AP,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* beta,
                             hipblasDoubleComplex*       y,
                             int                         incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhpmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)AP,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)beta,
                                                  (rocblas_double_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasChpmv_v2(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int               n,
                                const hipComplex* alpha,
                                const hipComplex* AP,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* beta,
                                hipComplex*       y,
                                int               incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chpmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)AP,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)beta,
                                                  (rocblas_float_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhpmv_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int                     n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* AP,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* beta,
                                hipDoubleComplex*       y,
                                int                     incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhpmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)AP,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)beta,
                                                  (rocblas_double_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// hpmv_batched
hipblasStatus_t hipblasChpmvBatched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    int                         n,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const AP[],
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex*       beta,
                                    hipblasComplex* const       y[],
                                    int                         incy,
                                    int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chpmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)AP,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex*)beta,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhpmvBatched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    int                               n,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const AP[],
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex*       beta,
                                    hipblasDoubleComplex* const       y[],
                                    int                               incy,
                                    int                               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhpmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)AP,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex*)beta,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasChpmvBatched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int                     n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const AP[],
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex*       beta,
                                       hipComplex* const       y[],
                                       int                     incy,
                                       int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chpmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)AP,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex*)beta,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhpmvBatched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int                           n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const AP[],
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex*       beta,
                                       hipDoubleComplex* const       y[],
                                       int                           incy,
                                       int                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhpmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)AP,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex*)beta,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// hpmv_strided_batched
hipblasStatus_t hipblasChpmvStridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* AP,
                                           hipblasStride         strideAP,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* beta,
                                           hipblasComplex*       y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chpmv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)AP,
                                                                  strideAP,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)beta,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhpmvStridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* AP,
                                           hipblasStride               strideAP,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* beta,
                                           hipblasDoubleComplex*       y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhpmv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)AP,
                                                                  strideAP,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)beta,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasChpmvStridedBatched_v2(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int               n,
                                              const hipComplex* alpha,
                                              const hipComplex* AP,
                                              hipblasStride     strideAP,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              const hipComplex* beta,
                                              hipComplex*       y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chpmv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)AP,
                                                                  strideAP,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)beta,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhpmvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* AP,
                                              hipblasStride           strideAP,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* beta,
                                              hipDoubleComplex*       y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhpmv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)AP,
                                                                  strideAP,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)beta,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// hpr
hipblasStatus_t hipblasChpr(hipblasHandle_t       handle,
                            hipblasFillMode_t     uplo,
                            int                   n,
                            const float*          alpha,
                            const hipblasComplex* x,
                            int                   incx,
                            hipblasComplex*       AP)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chpr((rocblas_handle)handle,
                                                 hipFillToHCCFill(uplo),
                                                 n,
                                                 alpha,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)AP));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhpr(hipblasHandle_t             handle,
                            hipblasFillMode_t           uplo,
                            int                         n,
                            const double*               alpha,
                            const hipblasDoubleComplex* x,
                            int                         incx,
                            hipblasDoubleComplex*       AP)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhpr((rocblas_handle)handle,
                                                 hipFillToHCCFill(uplo),
                                                 n,
                                                 alpha,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)AP));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasChpr_v2(hipblasHandle_t   handle,
                               hipblasFillMode_t uplo,
                               int               n,
                               const float*      alpha,
                               const hipComplex* x,
                               int               incx,
                               hipComplex*       AP)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chpr((rocblas_handle)handle,
                                                 hipFillToHCCFill(uplo),
                                                 n,
                                                 alpha,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)AP));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhpr_v2(hipblasHandle_t         handle,
                               hipblasFillMode_t       uplo,
                               int                     n,
                               const double*           alpha,
                               const hipDoubleComplex* x,
                               int                     incx,
                               hipDoubleComplex*       AP)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhpr((rocblas_handle)handle,
                                                 hipFillToHCCFill(uplo),
                                                 n,
                                                 alpha,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)AP));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// hpr_batched
hipblasStatus_t hipblasChprBatched(hipblasHandle_t             handle,
                                   hipblasFillMode_t           uplo,
                                   int                         n,
                                   const float*                alpha,
                                   const hipblasComplex* const x[],
                                   int                         incx,
                                   hipblasComplex* const       AP[],
                                   int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chpr_batched((rocblas_handle)handle,
                                                         hipFillToHCCFill(uplo),
                                                         n,
                                                         alpha,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex**)AP,
                                                         batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhprBatched(hipblasHandle_t                   handle,
                                   hipblasFillMode_t                 uplo,
                                   int                               n,
                                   const double*                     alpha,
                                   const hipblasDoubleComplex* const x[],
                                   int                               incx,
                                   hipblasDoubleComplex* const       AP[],
                                   int                               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhpr_batched((rocblas_handle)handle,
                                                         hipFillToHCCFill(uplo),
                                                         n,
                                                         alpha,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex**)AP,
                                                         batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasChprBatched_v2(hipblasHandle_t         handle,
                                      hipblasFillMode_t       uplo,
                                      int                     n,
                                      const float*            alpha,
                                      const hipComplex* const x[],
                                      int                     incx,
                                      hipComplex* const       AP[],
                                      int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chpr_batched((rocblas_handle)handle,
                                                         hipFillToHCCFill(uplo),
                                                         n,
                                                         alpha,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex**)AP,
                                                         batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhprBatched_v2(hipblasHandle_t               handle,
                                      hipblasFillMode_t             uplo,
                                      int                           n,
                                      const double*                 alpha,
                                      const hipDoubleComplex* const x[],
                                      int                           incx,
                                      hipDoubleComplex* const       AP[],
                                      int                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhpr_batched((rocblas_handle)handle,
                                                         hipFillToHCCFill(uplo),
                                                         n,
                                                         alpha,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex**)AP,
                                                         batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// hpr_strided_batched
hipblasStatus_t hipblasChprStridedBatched(hipblasHandle_t       handle,
                                          hipblasFillMode_t     uplo,
                                          int                   n,
                                          const float*          alpha,
                                          const hipblasComplex* x,
                                          int                   incx,
                                          hipblasStride         stridex,
                                          hipblasComplex*       AP,
                                          hipblasStride         strideAP,
                                          int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chpr_strided_batched((rocblas_handle)handle,
                                                                 hipFillToHCCFill(uplo),
                                                                 n,
                                                                 alpha,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)AP,
                                                                 strideAP,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhprStridedBatched(hipblasHandle_t             handle,
                                          hipblasFillMode_t           uplo,
                                          int                         n,
                                          const double*               alpha,
                                          const hipblasDoubleComplex* x,
                                          int                         incx,
                                          hipblasStride               stridex,
                                          hipblasDoubleComplex*       AP,
                                          hipblasStride               strideAP,
                                          int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhpr_strided_batched((rocblas_handle)handle,
                                                                 hipFillToHCCFill(uplo),
                                                                 n,
                                                                 alpha,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)AP,
                                                                 strideAP,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasChprStridedBatched_v2(hipblasHandle_t   handle,
                                             hipblasFillMode_t uplo,
                                             int               n,
                                             const float*      alpha,
                                             const hipComplex* x,
                                             int               incx,
                                             hipblasStride     stridex,
                                             hipComplex*       AP,
                                             hipblasStride     strideAP,
                                             int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chpr_strided_batched((rocblas_handle)handle,
                                                                 hipFillToHCCFill(uplo),
                                                                 n,
                                                                 alpha,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)AP,
                                                                 strideAP,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhprStridedBatched_v2(hipblasHandle_t         handle,
                                             hipblasFillMode_t       uplo,
                                             int                     n,
                                             const double*           alpha,
                                             const hipDoubleComplex* x,
                                             int                     incx,
                                             hipblasStride           stridex,
                                             hipDoubleComplex*       AP,
                                             hipblasStride           strideAP,
                                             int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhpr_strided_batched((rocblas_handle)handle,
                                                                 hipFillToHCCFill(uplo),
                                                                 n,
                                                                 alpha,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)AP,
                                                                 strideAP,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// hpr2
hipblasStatus_t hipblasChpr2(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             int                   n,
                             const hipblasComplex* alpha,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* y,
                             int                   incy,
                             hipblasComplex*       AP)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chpr2((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy,
                                                  (rocblas_float_complex*)AP));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhpr2(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* y,
                             int                         incy,
                             hipblasDoubleComplex*       AP)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhpr2((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy,
                                                  (rocblas_double_complex*)AP));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasChpr2_v2(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int               n,
                                const hipComplex* alpha,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* y,
                                int               incy,
                                hipComplex*       AP)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chpr2((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy,
                                                  (rocblas_float_complex*)AP));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhpr2_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int                     n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* y,
                                int                     incy,
                                hipDoubleComplex*       AP)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhpr2((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy,
                                                  (rocblas_double_complex*)AP));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// hpr2_batched
hipblasStatus_t hipblasChpr2Batched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    int                         n,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex* const y[],
                                    int                         incy,
                                    hipblasComplex* const       AP[],
                                    int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chpr2_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          (rocblas_float_complex**)AP,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhpr2Batched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    int                               n,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex* const y[],
                                    int                               incy,
                                    hipblasDoubleComplex* const       AP[],
                                    int                               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhpr2_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          (rocblas_double_complex**)AP,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasChpr2Batched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int                     n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex* const y[],
                                       int                     incy,
                                       hipComplex* const       AP[],
                                       int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chpr2_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          (rocblas_float_complex**)AP,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhpr2Batched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int                           n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex* const y[],
                                       int                           incy,
                                       hipDoubleComplex* const       AP[],
                                       int                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhpr2_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          (rocblas_double_complex**)AP,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// hpr2_strided_batched
hipblasStatus_t hipblasChpr2StridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           hipblasComplex*       AP,
                                           hipblasStride         strideAP,
                                           int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chpr2_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  (rocblas_float_complex*)AP,
                                                                  strideAP,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhpr2StridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           hipblasDoubleComplex*       AP,
                                           hipblasStride               strideAP,
                                           int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhpr2_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  (rocblas_double_complex*)AP,
                                                                  strideAP,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasChpr2StridedBatched_v2(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int               n,
                                              const hipComplex* alpha,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              hipComplex*       AP,
                                              hipblasStride     strideAP,
                                              int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_chpr2_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  (rocblas_float_complex*)AP,
                                                                  strideAP,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZhpr2StridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              hipDoubleComplex*       AP,
                                              hipblasStride           strideAP,
                                              int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zhpr2_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  (rocblas_double_complex*)AP,
                                                                  strideAP,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// sbmv
hipblasStatus_t hipblasSsbmv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             int               k,
                             const float*      alpha,
                             const float*      A,
                             int               lda,
                             const float*      x,
                             int               incx,
                             const float*      beta,
                             float*            y,
                             int               incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ssbmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  k,
                                                  alpha,
                                                  A,
                                                  lda,
                                                  x,
                                                  incx,
                                                  beta,
                                                  y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDsbmv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             int               k,
                             const double*     alpha,
                             const double*     A,
                             int               lda,
                             const double*     x,
                             int               incx,
                             const double*     beta,
                             double*           y,
                             int               incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dsbmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  k,
                                                  alpha,
                                                  A,
                                                  lda,
                                                  x,
                                                  incx,
                                                  beta,
                                                  y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// sbmv_batched
hipblasStatus_t hipblasSsbmvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    int                n,
                                    int                k,
                                    const float*       alpha,
                                    const float* const A[],
                                    int                lda,
                                    const float* const x[],
                                    int                incx,
                                    const float*       beta,
                                    float* const       y[],
                                    int                incy,
                                    int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ssbmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          k,
                                                          alpha,
                                                          A,
                                                          lda,
                                                          x,
                                                          incx,
                                                          beta,
                                                          y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDsbmvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    int                 n,
                                    int                 k,
                                    const double*       alpha,
                                    const double* const A[],
                                    int                 lda,
                                    const double* const x[],
                                    int                 incx,
                                    const double*       beta,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dsbmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          k,
                                                          alpha,
                                                          A,
                                                          lda,
                                                          x,
                                                          incx,
                                                          beta,
                                                          y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// sbmv_strided_batched
hipblasStatus_t hipblasSsbmvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           int               k,
                                           const float*      alpha,
                                           const float*      A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           const float*      x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const float*      beta,
                                           float*            y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ssbmv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  k,
                                                                  alpha,
                                                                  A,
                                                                  lda,
                                                                  strideA,
                                                                  x,
                                                                  incx,
                                                                  stridex,
                                                                  beta,
                                                                  y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDsbmvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           int               k,
                                           const double*     alpha,
                                           const double*     A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           const double*     x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const double*     beta,
                                           double*           y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dsbmv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  k,
                                                                  alpha,
                                                                  A,
                                                                  lda,
                                                                  strideA,
                                                                  x,
                                                                  incx,
                                                                  stridex,
                                                                  beta,
                                                                  y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// spmv
hipblasStatus_t hipblasSspmv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const float*      alpha,
                             const float*      AP,
                             const float*      x,
                             int               incx,
                             const float*      beta,
                             float*            y,
                             int               incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sspmv(
        (rocblas_handle)handle, hipFillToHCCFill(uplo), n, alpha, AP, x, incx, beta, y, incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDspmv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const double*     alpha,
                             const double*     AP,
                             const double*     x,
                             int               incx,
                             const double*     beta,
                             double*           y,
                             int               incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dspmv(
        (rocblas_handle)handle, hipFillToHCCFill(uplo), n, alpha, AP, x, incx, beta, y, incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// spmv_batched
hipblasStatus_t hipblasSspmvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    int                n,
                                    const float*       alpha,
                                    const float* const AP[],
                                    const float* const x[],
                                    int                incx,
                                    const float*       beta,
                                    float* const       y[],
                                    int                incy,
                                    int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sspmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          alpha,
                                                          AP,
                                                          x,
                                                          incx,
                                                          beta,
                                                          y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDspmvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    int                 n,
                                    const double*       alpha,
                                    const double* const AP[],
                                    const double* const x[],
                                    int                 incx,
                                    const double*       beta,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dspmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          alpha,
                                                          AP,
                                                          x,
                                                          incx,
                                                          beta,
                                                          y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// spmv_strided_batched
hipblasStatus_t hipblasSspmvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const float*      alpha,
                                           const float*      AP,
                                           hipblasStride     strideAP,
                                           const float*      x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const float*      beta,
                                           float*            y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sspmv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  alpha,
                                                                  AP,
                                                                  strideAP,
                                                                  x,
                                                                  incx,
                                                                  stridex,
                                                                  beta,
                                                                  y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDspmvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const double*     alpha,
                                           const double*     AP,
                                           hipblasStride     strideAP,
                                           const double*     x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const double*     beta,
                                           double*           y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dspmv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  alpha,
                                                                  AP,
                                                                  strideAP,
                                                                  x,
                                                                  incx,
                                                                  stridex,
                                                                  beta,
                                                                  y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// spr
hipblasStatus_t hipblasSspr(hipblasHandle_t   handle,
                            hipblasFillMode_t uplo,
                            int               n,
                            const float*      alpha,
                            const float*      x,
                            int               incx,
                            float*            AP)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_sspr((rocblas_handle)handle, hipFillToHCCFill(uplo), n, alpha, x, incx, AP));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDspr(hipblasHandle_t   handle,
                            hipblasFillMode_t uplo,
                            int               n,
                            const double*     alpha,
                            const double*     x,
                            int               incx,
                            double*           AP)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dspr((rocblas_handle)handle, hipFillToHCCFill(uplo), n, alpha, x, incx, AP));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCspr(hipblasHandle_t       handle,
                            hipblasFillMode_t     uplo,
                            int                   n,
                            const hipblasComplex* alpha,
                            const hipblasComplex* x,
                            int                   incx,
                            hipblasComplex*       AP)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cspr((rocblas_handle)handle,
                                                 hipFillToHCCFill(uplo),
                                                 n,
                                                 (rocblas_float_complex*)alpha,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)AP));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZspr(hipblasHandle_t             handle,
                            hipblasFillMode_t           uplo,
                            int                         n,
                            const hipblasDoubleComplex* alpha,
                            const hipblasDoubleComplex* x,
                            int                         incx,
                            hipblasDoubleComplex*       AP)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zspr((rocblas_handle)handle,
                                                 hipFillToHCCFill(uplo),
                                                 n,
                                                 (rocblas_double_complex*)alpha,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)AP));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCspr_v2(hipblasHandle_t   handle,
                               hipblasFillMode_t uplo,
                               int               n,
                               const hipComplex* alpha,
                               const hipComplex* x,
                               int               incx,
                               hipComplex*       AP)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cspr((rocblas_handle)handle,
                                                 hipFillToHCCFill(uplo),
                                                 n,
                                                 (rocblas_float_complex*)alpha,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)AP));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZspr_v2(hipblasHandle_t         handle,
                               hipblasFillMode_t       uplo,
                               int                     n,
                               const hipDoubleComplex* alpha,
                               const hipDoubleComplex* x,
                               int                     incx,
                               hipDoubleComplex*       AP)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zspr((rocblas_handle)handle,
                                                 hipFillToHCCFill(uplo),
                                                 n,
                                                 (rocblas_double_complex*)alpha,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)AP));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// spr_batched
hipblasStatus_t hipblasSsprBatched(hipblasHandle_t    handle,
                                   hipblasFillMode_t  uplo,
                                   int                n,
                                   const float*       alpha,
                                   const float* const x[],
                                   int                incx,
                                   float* const       AP[],
                                   int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sspr_batched(
        (rocblas_handle)handle, hipFillToHCCFill(uplo), n, alpha, x, incx, AP, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDsprBatched(hipblasHandle_t     handle,
                                   hipblasFillMode_t   uplo,
                                   int                 n,
                                   const double*       alpha,
                                   const double* const x[],
                                   int                 incx,
                                   double* const       AP[],
                                   int                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dspr_batched(
        (rocblas_handle)handle, hipFillToHCCFill(uplo), n, alpha, x, incx, AP, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsprBatched(hipblasHandle_t             handle,
                                   hipblasFillMode_t           uplo,
                                   int                         n,
                                   const hipblasComplex*       alpha,
                                   const hipblasComplex* const x[],
                                   int                         incx,
                                   hipblasComplex* const       AP[],
                                   int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cspr_batched((rocblas_handle)handle,
                                                         hipFillToHCCFill(uplo),
                                                         n,
                                                         (rocblas_float_complex*)alpha,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex**)AP,
                                                         batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsprBatched(hipblasHandle_t                   handle,
                                   hipblasFillMode_t                 uplo,
                                   int                               n,
                                   const hipblasDoubleComplex*       alpha,
                                   const hipblasDoubleComplex* const x[],
                                   int                               incx,
                                   hipblasDoubleComplex* const       AP[],
                                   int                               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zspr_batched((rocblas_handle)handle,
                                                         hipFillToHCCFill(uplo),
                                                         n,
                                                         (rocblas_double_complex*)alpha,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex**)AP,
                                                         batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsprBatched_v2(hipblasHandle_t         handle,
                                      hipblasFillMode_t       uplo,
                                      int                     n,
                                      const hipComplex*       alpha,
                                      const hipComplex* const x[],
                                      int                     incx,
                                      hipComplex* const       AP[],
                                      int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cspr_batched((rocblas_handle)handle,
                                                         hipFillToHCCFill(uplo),
                                                         n,
                                                         (rocblas_float_complex*)alpha,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex**)AP,
                                                         batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsprBatched_v2(hipblasHandle_t               handle,
                                      hipblasFillMode_t             uplo,
                                      int                           n,
                                      const hipDoubleComplex*       alpha,
                                      const hipDoubleComplex* const x[],
                                      int                           incx,
                                      hipDoubleComplex* const       AP[],
                                      int                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zspr_batched((rocblas_handle)handle,
                                                         hipFillToHCCFill(uplo),
                                                         n,
                                                         (rocblas_double_complex*)alpha,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex**)AP,
                                                         batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// spr_strided_batched
hipblasStatus_t hipblasSsprStridedBatched(hipblasHandle_t   handle,
                                          hipblasFillMode_t uplo,
                                          int               n,
                                          const float*      alpha,
                                          const float*      x,
                                          int               incx,
                                          hipblasStride     stridex,
                                          float*            AP,
                                          hipblasStride     strideAP,
                                          int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sspr_strided_batched((rocblas_handle)handle,
                                                                 hipFillToHCCFill(uplo),
                                                                 n,
                                                                 alpha,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 AP,
                                                                 strideAP,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDsprStridedBatched(hipblasHandle_t   handle,
                                          hipblasFillMode_t uplo,
                                          int               n,
                                          const double*     alpha,
                                          const double*     x,
                                          int               incx,
                                          hipblasStride     stridex,
                                          double*           AP,
                                          hipblasStride     strideAP,
                                          int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dspr_strided_batched((rocblas_handle)handle,
                                                                 hipFillToHCCFill(uplo),
                                                                 n,
                                                                 alpha,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 AP,
                                                                 strideAP,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsprStridedBatched(hipblasHandle_t       handle,
                                          hipblasFillMode_t     uplo,
                                          int                   n,
                                          const hipblasComplex* alpha,
                                          const hipblasComplex* x,
                                          int                   incx,
                                          hipblasStride         stridex,
                                          hipblasComplex*       AP,
                                          hipblasStride         strideAP,
                                          int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cspr_strided_batched((rocblas_handle)handle,
                                                                 hipFillToHCCFill(uplo),
                                                                 n,
                                                                 (rocblas_float_complex*)alpha,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)AP,
                                                                 strideAP,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsprStridedBatched(hipblasHandle_t             handle,
                                          hipblasFillMode_t           uplo,
                                          int                         n,
                                          const hipblasDoubleComplex* alpha,
                                          const hipblasDoubleComplex* x,
                                          int                         incx,
                                          hipblasStride               stridex,
                                          hipblasDoubleComplex*       AP,
                                          hipblasStride               strideAP,
                                          int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zspr_strided_batched((rocblas_handle)handle,
                                                                 hipFillToHCCFill(uplo),
                                                                 n,
                                                                 (rocblas_double_complex*)alpha,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)AP,
                                                                 strideAP,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsprStridedBatched_v2(hipblasHandle_t   handle,
                                             hipblasFillMode_t uplo,
                                             int               n,
                                             const hipComplex* alpha,
                                             const hipComplex* x,
                                             int               incx,
                                             hipblasStride     stridex,
                                             hipComplex*       AP,
                                             hipblasStride     strideAP,
                                             int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_cspr_strided_batched((rocblas_handle)handle,
                                                                 hipFillToHCCFill(uplo),
                                                                 n,
                                                                 (rocblas_float_complex*)alpha,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)AP,
                                                                 strideAP,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsprStridedBatched_v2(hipblasHandle_t         handle,
                                             hipblasFillMode_t       uplo,
                                             int                     n,
                                             const hipDoubleComplex* alpha,
                                             const hipDoubleComplex* x,
                                             int                     incx,
                                             hipblasStride           stridex,
                                             hipDoubleComplex*       AP,
                                             hipblasStride           strideAP,
                                             int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zspr_strided_batched((rocblas_handle)handle,
                                                                 hipFillToHCCFill(uplo),
                                                                 n,
                                                                 (rocblas_double_complex*)alpha,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)AP,
                                                                 strideAP,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// spr2
hipblasStatus_t hipblasSspr2(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const float*      alpha,
                             const float*      x,
                             int               incx,
                             const float*      y,
                             int               incy,
                             float*            AP)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sspr2(
        (rocblas_handle)handle, hipFillToHCCFill(uplo), n, alpha, x, incx, y, incy, AP));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDspr2(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const double*     alpha,
                             const double*     x,
                             int               incx,
                             const double*     y,
                             int               incy,
                             double*           AP)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dspr2(
        (rocblas_handle)handle, hipFillToHCCFill(uplo), n, alpha, x, incx, y, incy, AP));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// spr2_batched
hipblasStatus_t hipblasSspr2Batched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    int                n,
                                    const float*       alpha,
                                    const float* const x[],
                                    int                incx,
                                    const float* const y[],
                                    int                incy,
                                    float* const       AP[],
                                    int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sspr2_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          alpha,
                                                          x,
                                                          incx,
                                                          y,
                                                          incy,
                                                          AP,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDspr2Batched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    int                 n,
                                    const double*       alpha,
                                    const double* const x[],
                                    int                 incx,
                                    const double* const y[],
                                    int                 incy,
                                    double* const       AP[],
                                    int                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dspr2_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          alpha,
                                                          x,
                                                          incx,
                                                          y,
                                                          incy,
                                                          AP,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// spr2_strided_batched
hipblasStatus_t hipblasSspr2StridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const float*      alpha,
                                           const float*      x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const float*      y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           float*            AP,
                                           hipblasStride     strideAP,
                                           int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_sspr2_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  alpha,
                                                                  x,
                                                                  incx,
                                                                  stridex,
                                                                  y,
                                                                  incy,
                                                                  stridey,
                                                                  AP,
                                                                  strideAP,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDspr2StridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const double*     alpha,
                                           const double*     x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const double*     y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           double*           AP,
                                           hipblasStride     strideAP,
                                           int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dspr2_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  alpha,
                                                                  x,
                                                                  incx,
                                                                  stridex,
                                                                  y,
                                                                  incy,
                                                                  stridey,
                                                                  AP,
                                                                  strideAP,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// symv
hipblasStatus_t hipblasSsymv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const float*      alpha,
                             const float*      A,
                             int               lda,
                             const float*      x,
                             int               incx,
                             const float*      beta,
                             float*            y,
                             int               incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ssymv(
        (rocblas_handle)handle, hipFillToHCCFill(uplo), n, alpha, A, lda, x, incx, beta, y, incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDsymv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const double*     alpha,
                             const double*     A,
                             int               lda,
                             const double*     x,
                             int               incx,
                             const double*     beta,
                             double*           y,
                             int               incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dsymv(
        (rocblas_handle)handle, hipFillToHCCFill(uplo), n, alpha, A, lda, x, incx, beta, y, incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsymv(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             int                   n,
                             const hipblasComplex* alpha,
                             const hipblasComplex* A,
                             int                   lda,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* beta,
                             hipblasComplex*       y,
                             int                   incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csymv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)A,
                                                  lda,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)beta,
                                                  (rocblas_float_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsymv(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* A,
                             int                         lda,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* beta,
                             hipblasDoubleComplex*       y,
                             int                         incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zsymv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)A,
                                                  lda,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)beta,
                                                  (rocblas_double_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsymv_v2(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int               n,
                                const hipComplex* alpha,
                                const hipComplex* A,
                                int               lda,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* beta,
                                hipComplex*       y,
                                int               incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csymv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)A,
                                                  lda,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)beta,
                                                  (rocblas_float_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsymv_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int                     n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* A,
                                int                     lda,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* beta,
                                hipDoubleComplex*       y,
                                int                     incy)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zsymv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)A,
                                                  lda,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)beta,
                                                  (rocblas_double_complex*)y,
                                                  incy));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// symv_batched
hipblasStatus_t hipblasSsymvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    int                n,
                                    const float*       alpha,
                                    const float* const A[],
                                    int                lda,
                                    const float* const x[],
                                    int                incx,
                                    const float*       beta,
                                    float* const       y[],
                                    int                incy,
                                    int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ssymv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          alpha,
                                                          A,
                                                          lda,
                                                          x,
                                                          incx,
                                                          beta,
                                                          y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDsymvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    int                 n,
                                    const double*       alpha,
                                    const double* const A[],
                                    int                 lda,
                                    const double* const x[],
                                    int                 incx,
                                    const double*       beta,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dsymv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          alpha,
                                                          A,
                                                          lda,
                                                          x,
                                                          incx,
                                                          beta,
                                                          y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsymvBatched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    int                         n,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const A[],
                                    int                         lda,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex*       beta,
                                    hipblasComplex* const       y[],
                                    int                         incy,
                                    int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csymv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex*)beta,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsymvBatched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    int                               n,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const A[],
                                    int                               lda,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex*       beta,
                                    hipblasDoubleComplex* const       y[],
                                    int                               incy,
                                    int                               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zsymv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex*)beta,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsymvBatched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int                     n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const A[],
                                       int                     lda,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex*       beta,
                                       hipComplex* const       y[],
                                       int                     incy,
                                       int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csymv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex*)beta,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsymvBatched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int                           n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const A[],
                                       int                           lda,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex*       beta,
                                       hipDoubleComplex* const       y[],
                                       int                           incy,
                                       int                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zsymv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex*)beta,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// symv_strided_batched
hipblasStatus_t hipblasSsymvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const float*      alpha,
                                           const float*      A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           const float*      x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const float*      beta,
                                           float*            y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ssymv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  alpha,
                                                                  A,
                                                                  lda,
                                                                  strideA,
                                                                  x,
                                                                  incx,
                                                                  stridex,
                                                                  beta,
                                                                  y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDsymvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const double*     alpha,
                                           const double*     A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           const double*     x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const double*     beta,
                                           double*           y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dsymv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  alpha,
                                                                  A,
                                                                  lda,
                                                                  strideA,
                                                                  x,
                                                                  incx,
                                                                  stridex,
                                                                  beta,
                                                                  y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsymvStridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* A,
                                           int                   lda,
                                           hipblasStride         strideA,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* beta,
                                           hipblasComplex*       y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csymv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)beta,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsymvStridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* A,
                                           int                         lda,
                                           hipblasStride               strideA,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* beta,
                                           hipblasDoubleComplex*       y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zsymv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)beta,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsymvStridedBatched_v2(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int               n,
                                              const hipComplex* alpha,
                                              const hipComplex* A,
                                              int               lda,
                                              hipblasStride     strideA,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              const hipComplex* beta,
                                              hipComplex*       y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csymv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)beta,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsymvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* A,
                                              int                     lda,
                                              hipblasStride           strideA,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* beta,
                                              hipDoubleComplex*       y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zsymv_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)beta,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// syr
hipblasStatus_t hipblasSsyr(hipblasHandle_t   handle,
                            hipblasFillMode_t uplo,
                            int               n,
                            const float*      alpha,
                            const float*      x,
                            int               incx,
                            float*            A,
                            int               lda)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ssyr((rocblas_handle)handle, hipFillToHCCFill(uplo), n, alpha, x, incx, A, lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDsyr(hipblasHandle_t   handle,
                            hipblasFillMode_t uplo,
                            int               n,
                            const double*     alpha,
                            const double*     x,
                            int               incx,
                            double*           A,
                            int               lda)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dsyr((rocblas_handle)handle, hipFillToHCCFill(uplo), n, alpha, x, incx, A, lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsyr(hipblasHandle_t       handle,
                            hipblasFillMode_t     uplo,
                            int                   n,
                            const hipblasComplex* alpha,
                            const hipblasComplex* x,
                            int                   incx,
                            hipblasComplex*       A,
                            int                   lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csyr((rocblas_handle)handle,
                                                 hipFillToHCCFill(uplo),
                                                 n,
                                                 (rocblas_float_complex*)alpha,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)A,
                                                 lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsyr(hipblasHandle_t             handle,
                            hipblasFillMode_t           uplo,
                            int                         n,
                            const hipblasDoubleComplex* alpha,
                            const hipblasDoubleComplex* x,
                            int                         incx,
                            hipblasDoubleComplex*       A,
                            int                         lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zsyr((rocblas_handle)handle,
                                                 hipFillToHCCFill(uplo),
                                                 n,
                                                 (rocblas_double_complex*)alpha,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)A,
                                                 lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsyr_v2(hipblasHandle_t   handle,
                               hipblasFillMode_t uplo,
                               int               n,
                               const hipComplex* alpha,
                               const hipComplex* x,
                               int               incx,
                               hipComplex*       A,
                               int               lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csyr((rocblas_handle)handle,
                                                 hipFillToHCCFill(uplo),
                                                 n,
                                                 (rocblas_float_complex*)alpha,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)A,
                                                 lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsyr_v2(hipblasHandle_t         handle,
                               hipblasFillMode_t       uplo,
                               int                     n,
                               const hipDoubleComplex* alpha,
                               const hipDoubleComplex* x,
                               int                     incx,
                               hipDoubleComplex*       A,
                               int                     lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zsyr((rocblas_handle)handle,
                                                 hipFillToHCCFill(uplo),
                                                 n,
                                                 (rocblas_double_complex*)alpha,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)A,
                                                 lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// syr_batched
hipblasStatus_t hipblasSsyrBatched(hipblasHandle_t    handle,
                                   hipblasFillMode_t  uplo,
                                   int                n,
                                   const float*       alpha,
                                   const float* const x[],
                                   int                incx,
                                   float* const       A[],
                                   int                lda,
                                   int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ssyr_batched(
        (rocblas_handle)handle, hipFillToHCCFill(uplo), n, alpha, x, incx, A, lda, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDsyrBatched(hipblasHandle_t     handle,
                                   hipblasFillMode_t   uplo,
                                   int                 n,
                                   const double*       alpha,
                                   const double* const x[],
                                   int                 incx,
                                   double* const       A[],
                                   int                 lda,
                                   int                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dsyr_batched(
        (rocblas_handle)handle, hipFillToHCCFill(uplo), n, alpha, x, incx, A, lda, batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsyrBatched(hipblasHandle_t             handle,
                                   hipblasFillMode_t           uplo,
                                   int                         n,
                                   const hipblasComplex*       alpha,
                                   const hipblasComplex* const x[],
                                   int                         incx,
                                   hipblasComplex* const       A[],
                                   int                         lda,
                                   int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csyr_batched((rocblas_handle)handle,
                                                         hipFillToHCCFill(uplo),
                                                         n,
                                                         (rocblas_float_complex*)alpha,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex**)A,
                                                         lda,
                                                         batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsyrBatched(hipblasHandle_t                   handle,
                                   hipblasFillMode_t                 uplo,
                                   int                               n,
                                   const hipblasDoubleComplex*       alpha,
                                   const hipblasDoubleComplex* const x[],
                                   int                               incx,
                                   hipblasDoubleComplex* const       A[],
                                   int                               lda,
                                   int                               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zsyr_batched((rocblas_handle)handle,
                                                         hipFillToHCCFill(uplo),
                                                         n,
                                                         (rocblas_double_complex*)alpha,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex**)A,
                                                         lda,
                                                         batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsyrBatched_v2(hipblasHandle_t         handle,
                                      hipblasFillMode_t       uplo,
                                      int                     n,
                                      const hipComplex*       alpha,
                                      const hipComplex* const x[],
                                      int                     incx,
                                      hipComplex* const       A[],
                                      int                     lda,
                                      int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csyr_batched((rocblas_handle)handle,
                                                         hipFillToHCCFill(uplo),
                                                         n,
                                                         (rocblas_float_complex*)alpha,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex**)A,
                                                         lda,
                                                         batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsyrBatched_v2(hipblasHandle_t               handle,
                                      hipblasFillMode_t             uplo,
                                      int                           n,
                                      const hipDoubleComplex*       alpha,
                                      const hipDoubleComplex* const x[],
                                      int                           incx,
                                      hipDoubleComplex* const       A[],
                                      int                           lda,
                                      int                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zsyr_batched((rocblas_handle)handle,
                                                         hipFillToHCCFill(uplo),
                                                         n,
                                                         (rocblas_double_complex*)alpha,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex**)A,
                                                         lda,
                                                         batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// syr_strided_batched
hipblasStatus_t hipblasSsyrStridedBatched(hipblasHandle_t   handle,
                                          hipblasFillMode_t uplo,
                                          int               n,
                                          const float*      alpha,
                                          const float*      x,
                                          int               incx,
                                          hipblasStride     stridex,
                                          float*            A,
                                          int               lda,
                                          hipblasStride     strideA,
                                          int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ssyr_strided_batched((rocblas_handle)handle,
                                                                 hipFillToHCCFill(uplo),
                                                                 n,
                                                                 alpha,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDsyrStridedBatched(hipblasHandle_t   handle,
                                          hipblasFillMode_t uplo,
                                          int               n,
                                          const double*     alpha,
                                          const double*     x,
                                          int               incx,
                                          hipblasStride     stridex,
                                          double*           A,
                                          int               lda,
                                          hipblasStride     strideA,
                                          int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dsyr_strided_batched((rocblas_handle)handle,
                                                                 hipFillToHCCFill(uplo),
                                                                 n,
                                                                 alpha,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsyrStridedBatched(hipblasHandle_t       handle,
                                          hipblasFillMode_t     uplo,
                                          int                   n,
                                          const hipblasComplex* alpha,
                                          const hipblasComplex* x,
                                          int                   incx,
                                          hipblasStride         stridex,
                                          hipblasComplex*       A,
                                          int                   lda,
                                          hipblasStride         strideA,
                                          int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csyr_strided_batched((rocblas_handle)handle,
                                                                 hipFillToHCCFill(uplo),
                                                                 n,
                                                                 (rocblas_float_complex*)alpha,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsyrStridedBatched(hipblasHandle_t             handle,
                                          hipblasFillMode_t           uplo,
                                          int                         n,
                                          const hipblasDoubleComplex* alpha,
                                          const hipblasDoubleComplex* x,
                                          int                         incx,
                                          hipblasStride               stridex,
                                          hipblasDoubleComplex*       A,
                                          int                         lda,
                                          hipblasStride               strideA,
                                          int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zsyr_strided_batched((rocblas_handle)handle,
                                                                 hipFillToHCCFill(uplo),
                                                                 n,
                                                                 (rocblas_double_complex*)alpha,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsyrStridedBatched_v2(hipblasHandle_t   handle,
                                             hipblasFillMode_t uplo,
                                             int               n,
                                             const hipComplex* alpha,
                                             const hipComplex* x,
                                             int               incx,
                                             hipblasStride     stridex,
                                             hipComplex*       A,
                                             int               lda,
                                             hipblasStride     strideA,
                                             int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csyr_strided_batched((rocblas_handle)handle,
                                                                 hipFillToHCCFill(uplo),
                                                                 n,
                                                                 (rocblas_float_complex*)alpha,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsyrStridedBatched_v2(hipblasHandle_t         handle,
                                             hipblasFillMode_t       uplo,
                                             int                     n,
                                             const hipDoubleComplex* alpha,
                                             const hipDoubleComplex* x,
                                             int                     incx,
                                             hipblasStride           stridex,
                                             hipDoubleComplex*       A,
                                             int                     lda,
                                             hipblasStride           strideA,
                                             int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zsyr_strided_batched((rocblas_handle)handle,
                                                                 hipFillToHCCFill(uplo),
                                                                 n,
                                                                 (rocblas_double_complex*)alpha,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// syr2
hipblasStatus_t hipblasSsyr2(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const float*      alpha,
                             const float*      x,
                             int               incx,
                             const float*      y,
                             int               incy,
                             float*            A,
                             int               lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ssyr2(
        (rocblas_handle)handle, hipFillToHCCFill(uplo), n, alpha, x, incx, y, incy, A, lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDsyr2(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const double*     alpha,
                             const double*     x,
                             int               incx,
                             const double*     y,
                             int               incy,
                             double*           A,
                             int               lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dsyr2(
        (rocblas_handle)handle, hipFillToHCCFill(uplo), n, alpha, x, incx, y, incy, A, lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsyr2(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             int                   n,
                             const hipblasComplex* alpha,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* y,
                             int                   incy,
                             hipblasComplex*       A,
                             int                   lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csyr2((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy,
                                                  (rocblas_float_complex*)A,
                                                  lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsyr2(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* y,
                             int                         incy,
                             hipblasDoubleComplex*       A,
                             int                         lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zsyr2((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy,
                                                  (rocblas_double_complex*)A,
                                                  lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsyr2_v2(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int               n,
                                const hipComplex* alpha,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* y,
                                int               incy,
                                hipComplex*       A,
                                int               lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csyr2((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_float_complex*)alpha,
                                                  (rocblas_float_complex*)x,
                                                  incx,
                                                  (rocblas_float_complex*)y,
                                                  incy,
                                                  (rocblas_float_complex*)A,
                                                  lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsyr2_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int                     n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* y,
                                int                     incy,
                                hipDoubleComplex*       A,
                                int                     lda)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zsyr2((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  n,
                                                  (rocblas_double_complex*)alpha,
                                                  (rocblas_double_complex*)x,
                                                  incx,
                                                  (rocblas_double_complex*)y,
                                                  incy,
                                                  (rocblas_double_complex*)A,
                                                  lda));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// syr2_batched
hipblasStatus_t hipblasSsyr2Batched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    int                n,
                                    const float*       alpha,
                                    const float* const x[],
                                    int                incx,
                                    const float* const y[],
                                    int                incy,
                                    float* const       A[],
                                    int                lda,
                                    int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ssyr2_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          alpha,
                                                          x,
                                                          incx,
                                                          y,
                                                          incy,
                                                          A,
                                                          lda,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDsyr2Batched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    int                 n,
                                    const double*       alpha,
                                    const double* const x[],
                                    int                 incx,
                                    const double* const y[],
                                    int                 incy,
                                    double* const       A[],
                                    int                 lda,
                                    int                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dsyr2_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          alpha,
                                                          x,
                                                          incx,
                                                          y,
                                                          incy,
                                                          A,
                                                          lda,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsyr2Batched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    int                         n,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex* const y[],
                                    int                         incy,
                                    hipblasComplex* const       A[],
                                    int                         lda,
                                    int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csyr2_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsyr2Batched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    int                               n,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex* const y[],
                                    int                               incy,
                                    hipblasDoubleComplex* const       A[],
                                    int                               lda,
                                    int                               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zsyr2_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsyr2Batched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int                     n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex* const y[],
                                       int                     incy,
                                       hipComplex* const       A[],
                                       int                     lda,
                                       int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csyr2_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_float_complex*)alpha,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          (rocblas_float_complex**)y,
                                                          incy,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsyr2Batched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int                           n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex* const y[],
                                       int                           incy,
                                       hipDoubleComplex* const       A[],
                                       int                           lda,
                                       int                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zsyr2_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          n,
                                                          (rocblas_double_complex*)alpha,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          (rocblas_double_complex**)y,
                                                          incy,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// syr2_strided_batched
hipblasStatus_t hipblasSsyr2StridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const float*      alpha,
                                           const float*      x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const float*      y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           float*            A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ssyr2_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  alpha,
                                                                  x,
                                                                  incx,
                                                                  stridex,
                                                                  y,
                                                                  incy,
                                                                  stridey,
                                                                  A,
                                                                  lda,
                                                                  strideA,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDsyr2StridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const double*     alpha,
                                           const double*     x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const double*     y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           double*           A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dsyr2_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  alpha,
                                                                  x,
                                                                  incx,
                                                                  stridex,
                                                                  y,
                                                                  incy,
                                                                  stridey,
                                                                  A,
                                                                  lda,
                                                                  strideA,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsyr2StridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           hipblasComplex*       A,
                                           int                   lda,
                                           hipblasStride         strideA,
                                           int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csyr2_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  (rocblas_float_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsyr2StridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           hipblasDoubleComplex*       A,
                                           int                         lda,
                                           hipblasStride               strideA,
                                           int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zsyr2_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  (rocblas_double_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCsyr2StridedBatched_v2(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int               n,
                                              const hipComplex* alpha,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              hipComplex*       A,
                                              int               lda,
                                              hipblasStride     strideA,
                                              int               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_csyr2_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_float_complex*)alpha,
                                                                  (rocblas_float_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_float_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  (rocblas_float_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZsyr2StridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              hipDoubleComplex*       A,
                                              int                     lda,
                                              hipblasStride           strideA,
                                              int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_zsyr2_strided_batched((rocblas_handle)handle,
                                                                  hipFillToHCCFill(uplo),
                                                                  n,
                                                                  (rocblas_double_complex*)alpha,
                                                                  (rocblas_double_complex*)x,
                                                                  incx,
                                                                  stridex,
                                                                  (rocblas_double_complex*)y,
                                                                  incy,
                                                                  stridey,
                                                                  (rocblas_double_complex*)A,
                                                                  lda,
                                                                  strideA,
                                                                  batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// tbmv
hipblasStatus_t hipblasStbmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             int                k,
                             const float*       A,
                             int                lda,
                             float*             x,
                             int                incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_stbmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  k,
                                                  A,
                                                  lda,
                                                  x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDtbmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             int                k,
                             const double*      A,
                             int                lda,
                             double*            x,
                             int                incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dtbmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  k,
                                                  A,
                                                  lda,
                                                  x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtbmv(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             hipblasOperation_t    transA,
                             hipblasDiagType_t     diag,
                             int                   n,
                             int                   k,
                             const hipblasComplex* A,
                             int                   lda,
                             hipblasComplex*       x,
                             int                   incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ctbmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  k,
                                                  (rocblas_float_complex*)A,
                                                  lda,
                                                  (rocblas_float_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtbmv(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             hipblasOperation_t          transA,
                             hipblasDiagType_t           diag,
                             int                         n,
                             int                         k,
                             const hipblasDoubleComplex* A,
                             int                         lda,
                             hipblasDoubleComplex*       x,
                             int                         incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ztbmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  k,
                                                  (rocblas_double_complex*)A,
                                                  lda,
                                                  (rocblas_double_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtbmv_v2(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int                n,
                                int                k,
                                const hipComplex*  A,
                                int                lda,
                                hipComplex*        x,
                                int                incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ctbmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  k,
                                                  (rocblas_float_complex*)A,
                                                  lda,
                                                  (rocblas_float_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtbmv_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                hipblasOperation_t      transA,
                                hipblasDiagType_t       diag,
                                int                     n,
                                int                     k,
                                const hipDoubleComplex* A,
                                int                     lda,
                                hipDoubleComplex*       x,
                                int                     incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ztbmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  k,
                                                  (rocblas_double_complex*)A,
                                                  lda,
                                                  (rocblas_double_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// tbmv_batched
hipblasStatus_t hipblasStbmvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    hipblasOperation_t transA,
                                    hipblasDiagType_t  diag,
                                    int                n,
                                    int                k,
                                    const float* const A[],
                                    int                lda,
                                    float* const       x[],
                                    int                incx,
                                    int                batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_stbmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          k,
                                                          A,
                                                          lda,
                                                          x,
                                                          incx,
                                                          batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDtbmvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    hipblasOperation_t  transA,
                                    hipblasDiagType_t   diag,
                                    int                 n,
                                    int                 k,
                                    const double* const A[],
                                    int                 lda,
                                    double* const       x[],
                                    int                 incx,
                                    int                 batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dtbmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          k,
                                                          A,
                                                          lda,
                                                          x,
                                                          incx,
                                                          batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtbmvBatched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    hipblasOperation_t          transA,
                                    hipblasDiagType_t           diag,
                                    int                         n,
                                    int                         k,
                                    const hipblasComplex* const A[],
                                    int                         lda,
                                    hipblasComplex* const       x[],
                                    int                         incx,
                                    int                         batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ctbmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          k,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtbmvBatched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    hipblasOperation_t                transA,
                                    hipblasDiagType_t                 diag,
                                    int                               n,
                                    int                               k,
                                    const hipblasDoubleComplex* const A[],
                                    int                               lda,
                                    hipblasDoubleComplex* const       x[],
                                    int                               incx,
                                    int                               batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ztbmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          k,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtbmvBatched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       hipblasOperation_t      transA,
                                       hipblasDiagType_t       diag,
                                       int                     n,
                                       int                     k,
                                       const hipComplex* const A[],
                                       int                     lda,
                                       hipComplex* const       x[],
                                       int                     incx,
                                       int                     batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ctbmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          k,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtbmvBatched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       hipblasOperation_t            transA,
                                       hipblasDiagType_t             diag,
                                       int                           n,
                                       int                           k,
                                       const hipDoubleComplex* const A[],
                                       int                           lda,
                                       hipDoubleComplex* const       x[],
                                       int                           incx,
                                       int                           batch_count)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ztbmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          k,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// tbmv_strided_batched
hipblasStatus_t hipblasStbmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           int                k,
                                           const float*       A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           float*             x,
                                           int                incx,
                                           hipblasStride      stride_x,
                                           int                batch_count)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_stbmv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      k,
                                      A,
                                      lda,
                                      stride_a,
                                      x,
                                      incx,
                                      stride_x,
                                      batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDtbmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           int                k,
                                           const double*      A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           double*            x,
                                           int                incx,
                                           hipblasStride      stride_x,
                                           int                batch_count)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dtbmv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      k,
                                      A,
                                      lda,
                                      stride_a,
                                      x,
                                      incx,
                                      stride_x,
                                      batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtbmvStridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           hipblasOperation_t    transA,
                                           hipblasDiagType_t     diag,
                                           int                   n,
                                           int                   k,
                                           const hipblasComplex* A,
                                           int                   lda,
                                           hipblasStride         stride_a,
                                           hipblasComplex*       x,
                                           int                   incx,
                                           hipblasStride         stride_x,
                                           int                   batch_count)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ctbmv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      k,
                                      (rocblas_float_complex*)A,
                                      lda,
                                      stride_a,
                                      (rocblas_float_complex*)x,
                                      incx,
                                      stride_x,
                                      batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtbmvStridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           hipblasOperation_t          transA,
                                           hipblasDiagType_t           diag,
                                           int                         n,
                                           int                         k,
                                           const hipblasDoubleComplex* A,
                                           int                         lda,
                                           hipblasStride               stride_a,
                                           hipblasDoubleComplex*       x,
                                           int                         incx,
                                           hipblasStride               stride_x,
                                           int                         batch_count)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ztbmv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      k,
                                      (rocblas_double_complex*)A,
                                      lda,
                                      stride_a,
                                      (rocblas_double_complex*)x,
                                      incx,
                                      stride_x,
                                      batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtbmvStridedBatched_v2(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int                n,
                                              int                k,
                                              const hipComplex*  A,
                                              int                lda,
                                              hipblasStride      stride_a,
                                              hipComplex*        x,
                                              int                incx,
                                              hipblasStride      stride_x,
                                              int                batch_count)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ctbmv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      k,
                                      (rocblas_float_complex*)A,
                                      lda,
                                      stride_a,
                                      (rocblas_float_complex*)x,
                                      incx,
                                      stride_x,
                                      batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtbmvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              hipblasOperation_t      transA,
                                              hipblasDiagType_t       diag,
                                              int                     n,
                                              int                     k,
                                              const hipDoubleComplex* A,
                                              int                     lda,
                                              hipblasStride           stride_a,
                                              hipDoubleComplex*       x,
                                              int                     incx,
                                              hipblasStride           stride_x,
                                              int                     batch_count)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ztbmv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      k,
                                      (rocblas_double_complex*)A,
                                      lda,
                                      stride_a,
                                      (rocblas_double_complex*)x,
                                      incx,
                                      stride_x,
                                      batch_count));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// tbsv
hipblasStatus_t hipblasStbsv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             int                k,
                             const float*       A,
                             int                lda,
                             float*             x,
                             int                incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_stbsv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  k,
                                                  A,
                                                  lda,
                                                  x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDtbsv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             int                k,
                             const double*      A,
                             int                lda,
                             double*            x,
                             int                incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dtbsv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  k,
                                                  A,
                                                  lda,
                                                  x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtbsv(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             hipblasOperation_t    transA,
                             hipblasDiagType_t     diag,
                             int                   n,
                             int                   k,
                             const hipblasComplex* A,
                             int                   lda,
                             hipblasComplex*       x,
                             int                   incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ctbsv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  k,
                                                  (rocblas_float_complex*)A,
                                                  lda,
                                                  (rocblas_float_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtbsv(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             hipblasOperation_t          transA,
                             hipblasDiagType_t           diag,
                             int                         n,
                             int                         k,
                             const hipblasDoubleComplex* A,
                             int                         lda,
                             hipblasDoubleComplex*       x,
                             int                         incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ztbsv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  k,
                                                  (rocblas_double_complex*)A,
                                                  lda,
                                                  (rocblas_double_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtbsv_v2(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int                n,
                                int                k,
                                const hipComplex*  A,
                                int                lda,
                                hipComplex*        x,
                                int                incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ctbsv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  k,
                                                  (rocblas_float_complex*)A,
                                                  lda,
                                                  (rocblas_float_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtbsv_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                hipblasOperation_t      transA,
                                hipblasDiagType_t       diag,
                                int                     n,
                                int                     k,
                                const hipDoubleComplex* A,
                                int                     lda,
                                hipDoubleComplex*       x,
                                int                     incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ztbsv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  k,
                                                  (rocblas_double_complex*)A,
                                                  lda,
                                                  (rocblas_double_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// tbsv_batched
hipblasStatus_t hipblasStbsvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    hipblasOperation_t transA,
                                    hipblasDiagType_t  diag,
                                    int                n,
                                    int                k,
                                    const float* const A[],
                                    int                lda,
                                    float* const       x[],
                                    int                incx,
                                    int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_stbsv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          k,
                                                          A,
                                                          lda,
                                                          x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDtbsvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    hipblasOperation_t  transA,
                                    hipblasDiagType_t   diag,
                                    int                 n,
                                    int                 k,
                                    const double* const A[],
                                    int                 lda,
                                    double* const       x[],
                                    int                 incx,
                                    int                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dtbsv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          k,
                                                          A,
                                                          lda,
                                                          x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtbsvBatched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    hipblasOperation_t          transA,
                                    hipblasDiagType_t           diag,
                                    int                         n,
                                    int                         k,
                                    const hipblasComplex* const A[],
                                    int                         lda,
                                    hipblasComplex* const       x[],
                                    int                         incx,
                                    int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ctbsv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          k,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtbsvBatched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    hipblasOperation_t                transA,
                                    hipblasDiagType_t                 diag,
                                    int                               n,
                                    int                               k,
                                    const hipblasDoubleComplex* const A[],
                                    int                               lda,
                                    hipblasDoubleComplex* const       x[],
                                    int                               incx,
                                    int                               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ztbsv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          k,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtbsvBatched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       hipblasOperation_t      transA,
                                       hipblasDiagType_t       diag,
                                       int                     n,
                                       int                     k,
                                       const hipComplex* const A[],
                                       int                     lda,
                                       hipComplex* const       x[],
                                       int                     incx,
                                       int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ctbsv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          k,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtbsvBatched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       hipblasOperation_t            transA,
                                       hipblasDiagType_t             diag,
                                       int                           n,
                                       int                           k,
                                       const hipDoubleComplex* const A[],
                                       int                           lda,
                                       hipDoubleComplex* const       x[],
                                       int                           incx,
                                       int                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ztbsv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          k,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// tbsv_strided_batched
hipblasStatus_t hipblasStbsvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           int                k,
                                           const float*       A,
                                           int                lda,
                                           hipblasStride      strideA,
                                           float*             x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_stbsv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      k,
                                      A,
                                      lda,
                                      strideA,
                                      x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDtbsvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           int                k,
                                           const double*      A,
                                           int                lda,
                                           hipblasStride      strideA,
                                           double*            x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dtbsv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      k,
                                      A,
                                      lda,
                                      strideA,
                                      x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtbsvStridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           hipblasOperation_t    transA,
                                           hipblasDiagType_t     diag,
                                           int                   n,
                                           int                   k,
                                           const hipblasComplex* A,
                                           int                   lda,
                                           hipblasStride         strideA,
                                           hipblasComplex*       x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ctbsv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      k,
                                      (rocblas_float_complex*)A,
                                      lda,
                                      strideA,
                                      (rocblas_float_complex*)x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtbsvStridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           hipblasOperation_t          transA,
                                           hipblasDiagType_t           diag,
                                           int                         n,
                                           int                         k,
                                           const hipblasDoubleComplex* A,
                                           int                         lda,
                                           hipblasStride               strideA,
                                           hipblasDoubleComplex*       x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ztbsv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      k,
                                      (rocblas_double_complex*)A,
                                      lda,
                                      strideA,
                                      (rocblas_double_complex*)x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtbsvStridedBatched_v2(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int                n,
                                              int                k,
                                              const hipComplex*  A,
                                              int                lda,
                                              hipblasStride      strideA,
                                              hipComplex*        x,
                                              int                incx,
                                              hipblasStride      stridex,
                                              int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ctbsv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      k,
                                      (rocblas_float_complex*)A,
                                      lda,
                                      strideA,
                                      (rocblas_float_complex*)x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtbsvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              hipblasOperation_t      transA,
                                              hipblasDiagType_t       diag,
                                              int                     n,
                                              int                     k,
                                              const hipDoubleComplex* A,
                                              int                     lda,
                                              hipblasStride           strideA,
                                              hipDoubleComplex*       x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ztbsv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      k,
                                      (rocblas_double_complex*)A,
                                      lda,
                                      strideA,
                                      (rocblas_double_complex*)x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// tpmv
hipblasStatus_t hipblasStpmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const float*       AP,
                             float*             x,
                             int                incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_stpmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  AP,
                                                  x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDtpmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const double*      AP,
                             double*            x,
                             int                incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dtpmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  AP,
                                                  x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtpmv(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             hipblasOperation_t    transA,
                             hipblasDiagType_t     diag,
                             int                   n,
                             const hipblasComplex* AP,
                             hipblasComplex*       x,
                             int                   incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ctpmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  (rocblas_float_complex*)AP,
                                                  (rocblas_float_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtpmv(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             hipblasOperation_t          transA,
                             hipblasDiagType_t           diag,
                             int                         n,
                             const hipblasDoubleComplex* AP,
                             hipblasDoubleComplex*       x,
                             int                         incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ztpmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  (rocblas_double_complex*)AP,
                                                  (rocblas_double_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtpmv_v2(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int                n,
                                const hipComplex*  AP,
                                hipComplex*        x,
                                int                incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ctpmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  (rocblas_float_complex*)AP,
                                                  (rocblas_float_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtpmv_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                hipblasOperation_t      transA,
                                hipblasDiagType_t       diag,
                                int                     n,
                                const hipDoubleComplex* AP,
                                hipDoubleComplex*       x,
                                int                     incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ztpmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  (rocblas_double_complex*)AP,
                                                  (rocblas_double_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// tpmv_batched
hipblasStatus_t hipblasStpmvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    hipblasOperation_t transA,
                                    hipblasDiagType_t  diag,
                                    int                n,
                                    const float* const AP[],
                                    float* const       x[],
                                    int                incx,
                                    int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_stpmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          AP,
                                                          x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDtpmvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    hipblasOperation_t  transA,
                                    hipblasDiagType_t   diag,
                                    int                 n,
                                    const double* const AP[],
                                    double* const       x[],
                                    int                 incx,
                                    int                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dtpmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          AP,
                                                          x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtpmvBatched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    hipblasOperation_t          transA,
                                    hipblasDiagType_t           diag,
                                    int                         n,
                                    const hipblasComplex* const AP[],
                                    hipblasComplex* const       x[],
                                    int                         incx,
                                    int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ctpmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          (rocblas_float_complex**)AP,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtpmvBatched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    hipblasOperation_t                transA,
                                    hipblasDiagType_t                 diag,
                                    int                               n,
                                    const hipblasDoubleComplex* const AP[],
                                    hipblasDoubleComplex* const       x[],
                                    int                               incx,
                                    int                               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ztpmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          (rocblas_double_complex**)AP,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtpmvBatched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       hipblasOperation_t      transA,
                                       hipblasDiagType_t       diag,
                                       int                     n,
                                       const hipComplex* const AP[],
                                       hipComplex* const       x[],
                                       int                     incx,
                                       int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ctpmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          (rocblas_float_complex**)AP,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtpmvBatched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       hipblasOperation_t            transA,
                                       hipblasDiagType_t             diag,
                                       int                           n,
                                       const hipDoubleComplex* const AP[],
                                       hipDoubleComplex* const       x[],
                                       int                           incx,
                                       int                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ztpmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          (rocblas_double_complex**)AP,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// tpmv_strided_batched
hipblasStatus_t hipblasStpmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const float*       AP,
                                           hipblasStride      strideAP,
                                           float*             x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_stpmv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      AP,
                                      strideAP,
                                      x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDtpmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const double*      AP,
                                           hipblasStride      strideAP,
                                           double*            x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dtpmv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      AP,
                                      strideAP,
                                      x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtpmvStridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           hipblasOperation_t    transA,
                                           hipblasDiagType_t     diag,
                                           int                   n,
                                           const hipblasComplex* AP,
                                           hipblasStride         strideAP,
                                           hipblasComplex*       x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ctpmv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      (rocblas_float_complex*)AP,
                                      strideAP,
                                      (rocblas_float_complex*)x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtpmvStridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           hipblasOperation_t          transA,
                                           hipblasDiagType_t           diag,
                                           int                         n,
                                           const hipblasDoubleComplex* AP,
                                           hipblasStride               strideAP,
                                           hipblasDoubleComplex*       x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ztpmv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      (rocblas_double_complex*)AP,
                                      strideAP,
                                      (rocblas_double_complex*)x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtpmvStridedBatched_v2(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int                n,
                                              const hipComplex*  AP,
                                              hipblasStride      strideAP,
                                              hipComplex*        x,
                                              int                incx,
                                              hipblasStride      stridex,
                                              int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ctpmv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      (rocblas_float_complex*)AP,
                                      strideAP,
                                      (rocblas_float_complex*)x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtpmvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              hipblasOperation_t      transA,
                                              hipblasDiagType_t       diag,
                                              int                     n,
                                              const hipDoubleComplex* AP,
                                              hipblasStride           strideAP,
                                              hipDoubleComplex*       x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ztpmv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      (rocblas_double_complex*)AP,
                                      strideAP,
                                      (rocblas_double_complex*)x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// tpsv
hipblasStatus_t hipblasStpsv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const float*       AP,
                             float*             x,
                             int                incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_stpsv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  AP,
                                                  x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDtpsv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const double*      AP,
                             double*            x,
                             int                incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dtpsv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  AP,
                                                  x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtpsv(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             hipblasOperation_t    transA,
                             hipblasDiagType_t     diag,
                             int                   n,
                             const hipblasComplex* AP,
                             hipblasComplex*       x,
                             int                   incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ctpsv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  (rocblas_float_complex*)AP,
                                                  (rocblas_float_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtpsv(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             hipblasOperation_t          transA,
                             hipblasDiagType_t           diag,
                             int                         n,
                             const hipblasDoubleComplex* AP,
                             hipblasDoubleComplex*       x,
                             int                         incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ztpsv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  (rocblas_double_complex*)AP,
                                                  (rocblas_double_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtpsv_v2(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int                n,
                                const hipComplex*  AP,
                                hipComplex*        x,
                                int                incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ctpsv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  (rocblas_float_complex*)AP,
                                                  (rocblas_float_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtpsv_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                hipblasOperation_t      transA,
                                hipblasDiagType_t       diag,
                                int                     n,
                                const hipDoubleComplex* AP,
                                hipDoubleComplex*       x,
                                int                     incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ztpsv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  (rocblas_double_complex*)AP,
                                                  (rocblas_double_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// tpsv_batched
hipblasStatus_t hipblasStpsvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    hipblasOperation_t transA,
                                    hipblasDiagType_t  diag,
                                    int                n,
                                    const float* const AP[],
                                    float* const       x[],
                                    int                incx,
                                    int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_stpsv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          AP,
                                                          x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDtpsvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    hipblasOperation_t  transA,
                                    hipblasDiagType_t   diag,
                                    int                 n,
                                    const double* const AP[],
                                    double* const       x[],
                                    int                 incx,
                                    int                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dtpsv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          AP,
                                                          x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtpsvBatched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    hipblasOperation_t          transA,
                                    hipblasDiagType_t           diag,
                                    int                         n,
                                    const hipblasComplex* const AP[],
                                    hipblasComplex* const       x[],
                                    int                         incx,
                                    int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ctpsv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          (rocblas_float_complex**)AP,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtpsvBatched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    hipblasOperation_t                transA,
                                    hipblasDiagType_t                 diag,
                                    int                               n,
                                    const hipblasDoubleComplex* const AP[],
                                    hipblasDoubleComplex* const       x[],
                                    int                               incx,
                                    int                               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ztpsv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          (rocblas_double_complex**)AP,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtpsvBatched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       hipblasOperation_t      transA,
                                       hipblasDiagType_t       diag,
                                       int                     n,
                                       const hipComplex* const AP[],
                                       hipComplex* const       x[],
                                       int                     incx,
                                       int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ctpsv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          (rocblas_float_complex**)AP,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtpsvBatched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       hipblasOperation_t            transA,
                                       hipblasDiagType_t             diag,
                                       int                           n,
                                       const hipDoubleComplex* const AP[],
                                       hipDoubleComplex* const       x[],
                                       int                           incx,
                                       int                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ztpsv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          (rocblas_double_complex**)AP,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// tpsv_strided_batched
hipblasStatus_t hipblasStpsvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const float*       AP,
                                           hipblasStride      strideAP,
                                           float*             x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_stpsv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      AP,
                                      strideAP,
                                      x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDtpsvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const double*      AP,
                                           hipblasStride      strideAP,
                                           double*            x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dtpsv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      AP,
                                      strideAP,
                                      x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtpsvStridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           hipblasOperation_t    transA,
                                           hipblasDiagType_t     diag,
                                           int                   n,
                                           const hipblasComplex* AP,
                                           hipblasStride         strideAP,
                                           hipblasComplex*       x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ctpsv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      (rocblas_float_complex*)AP,
                                      strideAP,
                                      (rocblas_float_complex*)x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtpsvStridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           hipblasOperation_t          transA,
                                           hipblasDiagType_t           diag,
                                           int                         n,
                                           const hipblasDoubleComplex* AP,
                                           hipblasStride               strideAP,
                                           hipblasDoubleComplex*       x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ztpsv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      (rocblas_double_complex*)AP,
                                      strideAP,
                                      (rocblas_double_complex*)x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtpsvStridedBatched_v2(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int                n,
                                              const hipComplex*  AP,
                                              hipblasStride      strideAP,
                                              hipComplex*        x,
                                              int                incx,
                                              hipblasStride      stridex,
                                              int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ctpsv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      (rocblas_float_complex*)AP,
                                      strideAP,
                                      (rocblas_float_complex*)x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtpsvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              hipblasOperation_t      transA,
                                              hipblasDiagType_t       diag,
                                              int                     n,
                                              const hipDoubleComplex* AP,
                                              hipblasStride           strideAP,
                                              hipDoubleComplex*       x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ztpsv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      (rocblas_double_complex*)AP,
                                      strideAP,
                                      (rocblas_double_complex*)x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// trmv
hipblasStatus_t hipblasStrmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const float*       A,
                             int                lda,
                             float*             x,
                             int                incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_strmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  A,
                                                  lda,
                                                  x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDtrmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const double*      A,
                             int                lda,
                             double*            x,
                             int                incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dtrmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  A,
                                                  lda,
                                                  x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtrmv(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             hipblasOperation_t    transA,
                             hipblasDiagType_t     diag,
                             int                   n,
                             const hipblasComplex* A,
                             int                   lda,
                             hipblasComplex*       x,
                             int                   incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ctrmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  (rocblas_float_complex*)A,
                                                  lda,
                                                  (rocblas_float_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtrmv(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             hipblasOperation_t          transA,
                             hipblasDiagType_t           diag,
                             int                         n,
                             const hipblasDoubleComplex* A,
                             int                         lda,
                             hipblasDoubleComplex*       x,
                             int                         incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ztrmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  (rocblas_double_complex*)A,
                                                  lda,
                                                  (rocblas_double_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtrmv_v2(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int                n,
                                const hipComplex*  A,
                                int                lda,
                                hipComplex*        x,
                                int                incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ctrmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  (rocblas_float_complex*)A,
                                                  lda,
                                                  (rocblas_float_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtrmv_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                hipblasOperation_t      transA,
                                hipblasDiagType_t       diag,
                                int                     n,
                                const hipDoubleComplex* A,
                                int                     lda,
                                hipDoubleComplex*       x,
                                int                     incx)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ztrmv((rocblas_handle)handle,
                                                  hipFillToHCCFill(uplo),
                                                  hipOperationToHCCOperation(transA),
                                                  hipDiagonalToHCCDiagonal(diag),
                                                  n,
                                                  (rocblas_double_complex*)A,
                                                  lda,
                                                  (rocblas_double_complex*)x,
                                                  incx));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// trmv_batched
hipblasStatus_t hipblasStrmvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    hipblasOperation_t transA,
                                    hipblasDiagType_t  diag,
                                    int                n,
                                    const float* const A[],
                                    int                lda,
                                    float* const       x[],
                                    int                incx,
                                    int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_strmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          A,
                                                          lda,
                                                          x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDtrmvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    hipblasOperation_t  transA,
                                    hipblasDiagType_t   diag,
                                    int                 n,
                                    const double* const A[],
                                    int                 lda,
                                    double* const       x[],
                                    int                 incx,
                                    int                 batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_dtrmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          A,
                                                          lda,
                                                          x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtrmvBatched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    hipblasOperation_t          transA,
                                    hipblasDiagType_t           diag,
                                    int                         n,
                                    const hipblasComplex* const A[],
                                    int                         lda,
                                    hipblasComplex* const       x[],
                                    int                         incx,
                                    int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ctrmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtrmvBatched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    hipblasOperation_t                transA,
                                    hipblasDiagType_t                 diag,
                                    int                               n,
                                    const hipblasDoubleComplex* const A[],
                                    int                               lda,
                                    hipblasDoubleComplex* const       x[],
                                    int                               incx,
                                    int                               batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ztrmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtrmvBatched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       hipblasOperation_t      transA,
                                       hipblasDiagType_t       diag,
                                       int                     n,
                                       const hipComplex* const A[],
                                       int                     lda,
                                       hipComplex* const       x[],
                                       int                     incx,
                                       int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ctrmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          (rocblas_float_complex**)A,
                                                          lda,
                                                          (rocblas_float_complex**)x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtrmvBatched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       hipblasOperation_t            transA,
                                       hipblasDiagType_t             diag,
                                       int                           n,
                                       const hipDoubleComplex* const A[],
                                       int                           lda,
                                       hipDoubleComplex* const       x[],
                                       int                           incx,
                                       int                           batchCount)
try
{
    return rocBLASStatusToHIPStatus(rocblas_ztrmv_batched((rocblas_handle)handle,
                                                          hipFillToHCCFill(uplo),
                                                          hipOperationToHCCOperation(transA),
                                                          hipDiagonalToHCCDiagonal(diag),
                                                          n,
                                                          (rocblas_double_complex**)A,
                                                          lda,
                                                          (rocblas_double_complex**)x,
                                                          incx,
                                                          batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

// trmv_strided_batched
hipblasStatus_t hipblasStrmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const float*       A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           float*             x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_strmv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      A,
                                      lda,
                                      stride_a,
                                      x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasDtrmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const double*      A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           double*            x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_dtrmv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      A,
                                      lda,
                                      stride_a,
                                      x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtrmvStridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           hipblasOperation_t    transA,
                                           hipblasDiagType_t     diag,
                                           int                   n,
                                           const hipblasComplex* A,
                                           int                   lda,
                                           hipblasStride         stride_a,
                                           hipblasComplex*       x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           int                   batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ctrmv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      (rocblas_float_complex*)A,
                                      lda,
                                      stride_a,
                                      (rocblas_float_complex*)x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtrmvStridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           hipblasOperation_t          transA,
                                           hipblasDiagType_t           diag,
                                           int                         n,
                                           const hipblasDoubleComplex* A,
                                           int                         lda,
                                           hipblasStride               stride_a,
                                           hipblasDoubleComplex*       x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           int                         batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ztrmv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      (rocblas_double_complex*)A,
                                      lda,
                                      stride_a,
                                      (rocblas_double_complex*)x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasCtrmvStridedBatched_v2(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int                n,
                                              const hipComplex*  A,
                                              int                lda,
                                              hipblasStride      stride_a,
                                              hipComplex*        x,
                                              int                incx,
                                              hipblasStride      stridex,
                                              int                batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ctrmv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperation(transA),
                                      hipDiagonalToHCCDiagonal(diag),
                                      n,
                                      (rocblas_float_complex*)A,
                                      lda,
                                      stride_a,
                                      (rocblas_float_complex*)x,
                                      incx,
                                      stridex,
                                      batchCount));
}
catch(...)
{
    return exception_to_hipblas_status();
}

hipblasStatus_t hipblasZtrmvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              hipblasOperation_t      transA,
                                              hipblasDiagType_t       diag,
                                              int                     n,
                                              const hipDoubleComplex* A,
                                              int                     lda,
                                              hipblasStride           stride_a,
                                              hipDoubleComplex*       x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              int                     batchCount)
try
{
    return rocBLASStatusToHIPStatus(
        rocblas_ztrmv_strided_batched((rocblas_handle)handle,
                                      hipFillToHCCFill(uplo),
                                      hipOperationToHCCOperati