/*
 *  CUPS add-on module for Canon LIPSLX/UFR2/NCAP printer.
 *  Copyright CANON INC. 2018
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <cups/cups.h>
#include <cups/ppd.h>
#include <cups/raster.h>
#include <fcntl.h>
#include <string.h>
#include <stdbool.h>
#include <sys/wait.h>
#include <signal.h>
#include "cnrasterproc.h"
#include <stdarg.h>

static char* makePageInfoStr( cups_page_header2_t *header, unsigned int dstWidth, unsigned int dstHeight, unsigned int margin_x, unsigned int margin_y, int nowBand, int totalBandNum, int bandHeight);
static int processPoster(cups_raster_t *ras, CnRasterPipeFds *pFds,  int poster, bool *pIsStartJob);

bool g_sigterm_received = false;

#define DPI_72 72.0
#define RESOLUTION_600 600
#define PIXEL_600DPI_1MM	24
#define POSTER_2X2	2
#define POSTER_3X3 	3
#define POSTER_4X4 	4

#ifdef _SFP_

#define	LINE_BUF_SIZE	1024
#define IS_BLANK(c)		(c == ' '  || c == '\t')
#define IS_RETURN(c)	(c == '\r' || c == '\n')
static
int GetKey(char *ptr, char *key, int size)
{
	int	key_len = 0;
	char	*p_key = NULL;

	p_key = key;
	while(1){
		if(*ptr == '*'){
			ptr++;
		}
		if(*ptr == '\n' || *ptr == '\r' || *ptr == '\0'){
			break;
		}
		if(*ptr == ' ' || *ptr == '\t'){
			break;
		}
		if(*ptr == ':'){
			break;
		}
		if(p_key - key >= size - 1){
			break;
		}
		*p_key = *ptr;
		ptr++;
		p_key++;
		key_len++;
	}
	*p_key = '\0';

	return key_len;
}

static
int GetValue(char *p_line_buf, char *p_value_buf, int value_len)
{
	int pos = 0;
	int val_pos = 0;

	while( IS_BLANK(p_line_buf[pos]) )
		pos++;
	while( p_line_buf[pos] == ':' )
		pos++;
	while( IS_BLANK(p_line_buf[pos]) )
		pos++;
	while( p_line_buf[pos] == '"')
		pos++;

	while( !IS_RETURN(p_line_buf[pos])
		  && p_line_buf[pos] != '"'
		  && p_line_buf[pos] != '\n'
		  && val_pos < value_len - 1 )
		p_value_buf[val_pos++] = p_line_buf[pos++];
	p_value_buf[val_pos++] = 0;

	return val_pos;
}

static
void zFreeCUPSOptions(int num_options, cups_option_t **options)
{
	if ( *options != NULL )
	{
		cupsFreeOptions(num_options, *options);
		*options = NULL;
	}
}

static
char *strdup2(char *str1, char *str2)
{
	int len;
	char *p_str;

	if( str1 == NULL || str2 == NULL )
		return NULL;

	len = strlen(str1) + strlen(str2) + 1;
	if( (p_str = malloc(len)) != NULL )
	{
		strcpy(p_str, str1);
		strcat(p_str, str2);
	}
	return p_str;
}

static
char* get_driver_rootpath( const char* const ppd_file )
{
	FILE* fp = NULL;
	char* pDriverPath = NULL;

	if( ppd_file == NULL )
	{
		return NULL;
	}
	fp = fopen(ppd_file, "r");
	if( fp != NULL )
	{
		char line_buf[LINE_BUF_SIZE];
		char cKey[BUFSIZE];
		char cValue[BUFSIZE];
		char* const ppdKeyName = "*CNDriverRootPath";

		while( fgets(line_buf, sizeof(line_buf), fp) != NULL )
		{
			if( line_buf[0] == '*' )
			{
				int key_len = 0;
				int value_len = 0;
				size_t ppdKeyNameLen = 0;

				memset(cKey, 0, sizeof(cKey));
				memset(cValue, 0, sizeof(cValue));

				ppdKeyNameLen = strlen(ppdKeyName);
				if( strncmp(line_buf, ppdKeyName, ppdKeyNameLen ) == 0 )
				{
					key_len = GetKey(line_buf + 1, cKey, sizeof(cKey));
					if(key_len > 0)
					{
						(void)GetValue(line_buf + 1 + key_len, cValue, sizeof(cValue));
					}
				}

				value_len = (int)strlen(cValue);

				if (value_len > 0)
				{
					pDriverPath = strdup(cValue);
					break;
				}
			}
		}
		fclose(fp);
	}
	return pDriverPath;
}

static
int exec_commandfilefilter( char *argv[] )
{
	int child_pid = -1;
	char *pDriverPath = NULL;
	char* const p_ppd_name = getenv("PPD");

	child_pid = fork();

	if( child_pid == 0 )
	{
		pDriverPath = get_driver_rootpath(p_ppd_name);
		if( pDriverPath != NULL )
		{
			char* pExePath = NULL;

			pExePath = strdup2(pDriverPath, "/Bins/commandfilefilterr");
			if( pExePath != NULL )
			{
				execv(pExePath, argv);
				free(pExePath);
			}
			free(pDriverPath);
		}
	}

	return child_pid;
}
#endif

static void sigterm_handler(int sigcode)
{
	g_sigterm_received = true;
	debuglog(INDENT_NONE, ">> Catching [Sigterm] \n");
}

#ifdef DEBUGLOG
void debuglog(int indent, const char *message, ...)
{
	static int debug_indent;
	va_list	ap;
	FILE		*fp = NULL;
	int		i;

	fp = fopen("/tmp/rasterfilter.log", "a+");
	if ( fp )
	{
		if (indent == INDENT_LEFT)
		{
			debug_indent--;
			if (debug_indent < 0){
				debug_indent = 0;
			}
		}

		for (i = 0; i < debug_indent; i++)
		{
			fprintf(fp, "\t");
		}

		va_start(ap, message);
		vfprintf(fp, message, ap);

		fclose(fp);

		if (indent == INDENT_RIGHT)
		{
			debug_indent++;
		}
	}
}
#endif

static int processPoster(cups_raster_t *ras, CnRasterPipeFds *pFds, int poster, bool *pIsSendJobStart)
{
	int err = DEF_NO_ERR;
	char *pPageInfoStr = NULL;
	cups_page_header2_t header, pageHeader;
	unsigned char *pImgBuf = NULL, *pTemp = NULL, *pPageBuf = NULL;
	unsigned char **pImgPtrArray = NULL;
	unsigned char **pPagePtrArray = NULL;
	unsigned int tblsize = 0;
	unsigned int imageableWidth = 0, imageableHeight = 0;
	unsigned int updateWidth = 0, updateHeight = 0;
	unsigned int posterWidth = 0, posterHeight = 0;
	unsigned int margin_x = 0, margin_y = 0;
	unsigned short shift_x = 0, shift_y = 0;

	int index = 0, row = 0, col = 0;
	char rawPath[255];
	int page = 0;

	*pIsSendJobStart = false;

	memset(&header, 0, sizeof( cups_page_header2_t ));

	if (ras == NULL || pFds == NULL) {
		return err;
	}

	while ( cupsRasterReadHeader2(ras, &header) && (err == DEF_NO_ERR) && (g_sigterm_received == false))
	{
		imageableWidth =  (unsigned int)(((header.PageSize[0] - 2 * header.Margins[0]) * header.HWResolution[0] / DPI_72) + 0.5);
		imageableHeight =  (unsigned int)(((header.PageSize[1] - 2 * header.Margins[1]) * header.HWResolution[1] / DPI_72) + 0.5);

		margin_x = (unsigned int)((header.Margins[0] * header.HWResolution[0] / DPI_72) + 0.5);
		margin_y = (unsigned int)((header.Margins[1] * header.HWResolution[1] / DPI_72) + 0.5);

		if ((imageableWidth > header.cupsWidth) && (imageableWidth - header.cupsWidth >= PIXEL_600DPI_1MM)) {
			updateWidth = imageableWidth;
		}else {
			updateWidth = header.cupsWidth;
		}

		if ((imageableHeight > header.cupsHeight) && (imageableHeight - header.cupsHeight >= PIXEL_600DPI_1MM)) {
			updateHeight = imageableHeight;
		}else {
			updateHeight = header.cupsHeight;
		}

		if (updateWidth % poster != 0) {
			updateWidth += (poster - (updateWidth % poster));
		}

		if (updateHeight % poster != 0) {
			updateHeight += (poster - (updateHeight % poster));
		}

		shift_x = (unsigned short)(updateWidth - header.cupsWidth) / 2;
		shift_y = (unsigned short)(updateHeight - header.cupsHeight) / 2;

		posterWidth = updateWidth / poster;
		posterHeight = updateHeight / poster;

		debuglog(INDENT_NONE, "Raster Header's cupsWidth = %d\n", header.cupsWidth);
		debuglog(INDENT_NONE, "Raster Header's cupsHeight = %d\n", header.cupsHeight);
		debuglog(INDENT_NONE, "Raster Header's Imageable Area Width = %d\n", imageableWidth);
		debuglog(INDENT_NONE, "Raster Header's Imageable Area Height = %d\n", imageableHeight);
		debuglog(INDENT_NONE, "Raster Header's margin_x = %d\n", margin_x);
		debuglog(INDENT_NONE, "Raster Header's margin_y = %d\n", margin_y);
		debuglog(INDENT_NONE, "UpdateWidth for poster = %d\n", updateWidth);
		debuglog(INDENT_NONE, "updateHeight for poster = %d\n", updateHeight);
		debuglog(INDENT_NONE, "posterRowPixel = %d\n", posterWidth);
		debuglog(INDENT_NONE, "posterColPixel = %d\n", posterHeight);
		debuglog(INDENT_NONE, "shift_x = %d\n", shift_x);
		debuglog(INDENT_NONE, "shift_y = %d\n", shift_y);

		tblsize = updateWidth * updateHeight * header.cupsNumColors;
		pImgBuf = ( unsigned char *)malloc(tblsize * sizeof(unsigned char));
		pImgPtrArray = ( unsigned char ** )malloc( updateHeight * sizeof(unsigned char*));

		pPageBuf = (unsigned char *)malloc(posterWidth * posterHeight * header.cupsNumColors * sizeof(unsigned char));
		pPagePtrArray = ( unsigned char ** )malloc( posterHeight * sizeof(unsigned char*));

		if (pImgPtrArray == NULL || pImgBuf == NULL || pPageBuf == NULL || pPagePtrArray == NULL)
		{
			err = DEF_ERR;
		}
		else
		{
			memset(pImgBuf, 0xff, tblsize * sizeof(unsigned char));
			memset(pImgPtrArray, 0, updateHeight * sizeof(unsigned char*) );
			pTemp = pImgBuf;
			for (index = 0; index < updateHeight; index++)
			{
				pImgPtrArray[index] = pTemp;
				pTemp += (updateWidth * header.cupsNumColors);
			}

			for (index = 0; index < header.cupsHeight; index++)
			{
				pTemp = pImgPtrArray[shift_y + index] + shift_x * header.cupsNumColors;
				cupsRasterReadPixels(ras, pTemp, header.cupsBytesPerLine);
			}

#ifdef DEBUGLOG
			{
				memset(rawPath, 0, 255);
				page++;
				sprintf(rawPath, "/tmp/rasterdata_%d.data", page);

				int fd_raw = open(rawPath, O_RDWR | O_CREAT | O_TRUNC, S_IRUSR|S_IWUSR|S_IROTH|S_IWOTH);
				if (fd_raw != -1)
				{
					for (index = 0; index < updateHeight; index++)
					{
						write(fd_raw, pImgPtrArray[index],  updateWidth * header.cupsNumColors);
					}

					close(fd_raw);
				}
			}
#endif
			memset(pPagePtrArray, 0, posterHeight * sizeof(unsigned char*) );
			pTemp = pPageBuf;
			for (index = 0; index < posterHeight; index++)
			{
				pPagePtrArray[index] = pTemp;
				pTemp += (posterWidth * header.cupsNumColors);
			}

			for (row = 0; row < poster; row++)
			{
				for ( col = 0; col < poster; col++)
				{
					int startRow = row * posterHeight;
					int startCol = col * posterWidth;

					int nowBand = 0,  bandHeight = BAND_HEIGHT;
					int totalBandNum = posterHeight / bandHeight;
					int lastBandHeight = posterHeight % bandHeight;

					if (lastBandHeight != 0) {
						totalBandNum++;
					}else {
						lastBandHeight = bandHeight;
					}

					debuglog(INDENT_NONE, " # Process Poster Page row %d, col %d, total Band is %d # \n", row, col, totalBandNum);

					memset(pPageBuf, 0xff, posterWidth * posterHeight * header.cupsNumColors * sizeof(unsigned char));

					for (index = 0; index < posterHeight; index++)
					{
						memcpy(pPagePtrArray[index], pImgPtrArray[startRow] + startCol * header.cupsNumColors, posterWidth * header.cupsNumColors);
						startRow++;
					}

#ifdef DEBUGLOG
					{
						memset(rawPath, 0, 255);
						sprintf(rawPath, "/tmp/rasterdata_%d_%d%d.data", page, row+1, col+1);

						int fd_raw = open(rawPath, O_RDWR | O_CREAT | O_TRUNC, S_IRUSR|S_IWUSR|S_IROTH|S_IWOTH);
						if (fd_raw != -1)
						{
							for (index = 0; index < posterHeight; index++)
							{
								write(fd_raw, pPagePtrArray[index], posterWidth * header.cupsNumColors);
							}

							close(fd_raw);
						}
					}
#endif

					if ((*pIsSendJobStart == false) && (g_sigterm_received == false))
					{
						char pNumCopies[BUFSIZE] = {0};
						snprintf(pNumCopies, sizeof(pNumCopies), "%u", header.NumCopies);
						err = cnprocWriteCommand(*pFds, CNRS_ID_START_JOB, pNumCopies, (int)(strlen(pNumCopies)+1));
						debuglog(INDENT_NONE, "# Write Command [CNRS_ID_START_JOB]\n");
						*pIsSendJobStart = true;
					}

					if (err == DEF_NO_ERR)
					{
						for ( nowBand = 0; nowBand < totalBandNum; nowBand++)
						{
							if (err != DEF_NO_ERR) {
								break;
							}

							if( (g_sigterm_received == true) && (*pIsSendJobStart == true) )
							{
								cnprocWriteCommand(*pFds, CNRS_ID_JOB_CANCEL, NULL, 0);
								debuglog(INDENT_NONE, "# Write Command [CNRS_ID_JOB_CANCEL]\n");
								break;
							}

							if (nowBand == totalBandNum - 1) {
								bandHeight = lastBandHeight;
							}

							memset(&pageHeader, 0, sizeof(cups_page_header2_t));
							pageHeader.cupsWidth =  posterWidth;
							pageHeader.cupsHeight = posterHeight;
							pageHeader.cupsBytesPerLine = posterWidth * header.cupsNumColors;
							pageHeader.HWResolution[0] = header.HWResolution[0];
							pageHeader.HWResolution[1] = header.HWResolution[1];

							pPageInfoStr = makePageInfoStr(&pageHeader, updateWidth, updateHeight, margin_x, margin_y, nowBand, totalBandNum, bandHeight);
							if (pPageInfoStr == NULL)
							{
								err = DEF_ERR;
							}
							else
							{
								err = cnprocWriteCommand(*pFds, CNRS_ID_SEND_DATA, pPageInfoStr, (int)(strlen(pPageInfoStr)+1));
								if(err == DEF_NO_ERR)
								{
									err = cnprocWriteData( *pFds, (char*)pPagePtrArray[nowBand * BAND_HEIGHT],  pageHeader.cupsBytesPerLine * bandHeight);
								}
								free(pPageInfoStr);
								pPageInfoStr = NULL;
							}

							debuglog(INDENT_NONE, "# Write Command [POSTER CNRS_ID_SEND_DATA]: now poster row is %d, col is %d, now Band is %d\n\n", row, col, nowBand);
						}
					}
				}
			}
		}

		if (pImgBuf != NULL)
		{
			free(pImgBuf);
			pImgBuf = NULL;
		}

		if (pImgPtrArray != NULL)
		{
			free(pImgPtrArray);
			pImgPtrArray = NULL;
		}

		if (pPageBuf != NULL)
		{
			free(pPageBuf);
			pPageBuf = NULL;
		}

		if (pPagePtrArray != NULL)
		{
			free(pPagePtrArray);
			pPagePtrArray = NULL;
		}
	}

	return err;
}

int getBandData(cups_raster_t *ras, unsigned char **pBandBuf, cups_page_header2_t header, cups_page_header2_t* updatedHeader, int nowBand, int bandHeight)
{
	int ret = DEF_NO_ERR;

	if((ras == NULL) || (pBandBuf == NULL) || (*pBandBuf == NULL) ||(updatedHeader == NULL))
	{
		return DEF_ERR;
	}
	unsigned  imageHeight_pixels 	 = header.cupsHeight;
	unsigned  imageWidth_pixels 	 = header.cupsWidth;
	unsigned  imageWidthBytesPerLine = header.cupsBytesPerLine;
	unsigned  pageWidth_pixels 		 = (unsigned int)((header.cupsPageSize[0] * header.HWResolution[0] / DPI_72) + 0.5);
	unsigned  pageheight_pixels		 = (unsigned int)((header.cupsPageSize[1] * header.HWResolution[1] / DPI_72) + 0.5);
	unsigned  pageWidthBytesPerLine  = pageWidth_pixels * header.cupsNumColors;

	if( pageheight_pixels < imageHeight_pixels )
	{
		pageheight_pixels = imageHeight_pixels;
	}
	if( pageWidth_pixels <  imageWidth_pixels )
	{
		pageWidth_pixels = imageWidth_pixels;
		pageWidthBytesPerLine = imageWidthBytesPerLine;
	}
	debuglog(INDENT_NONE, "# header.cupsWidth = %u\n", header.cupsWidth);
	debuglog(INDENT_NONE, "# header.cupsHeight = %u\n", header.cupsHeight);
	debuglog(INDENT_NONE, "# pageWidth_pixels = %u\n", pageWidth_pixels);
	debuglog(INDENT_NONE, "# pageheight_pixels = %u\n", pageheight_pixels);

	unsigned int now_y_pos = nowBand * BAND_HEIGHT;

	unsigned int x_offset 	   = ((pageWidth_pixels - imageWidth_pixels)  / 2) * header.cupsNumColors;
	unsigned int y_offsetLines = (pageheight_pixels - imageHeight_pixels) / 2;
	debuglog(INDENT_NONE, "# x_offset = %d  \n", x_offset );
	debuglog(INDENT_NONE, "# y_offsetLines = %d\n", y_offsetLines);
	debuglog(INDENT_NONE, "# now_y_pos = %d\n", now_y_pos);

	if(pageheight_pixels > imageHeight_pixels)
	{
		if ( now_y_pos < y_offsetLines )
		{
			if( ((y_offsetLines - now_y_pos) / bandHeight) == 0 )
			{
				int lastFillWhiteLines = y_offsetLines - now_y_pos;
				memset( *pBandBuf, 0xff, imageWidthBytesPerLine * lastFillWhiteLines );
				cupsRasterReadPixels(ras, *pBandBuf+(imageWidthBytesPerLine * lastFillWhiteLines), imageWidthBytesPerLine * (bandHeight - lastFillWhiteLines) );
			}
			else
			{
				memset( *pBandBuf, 0xff, imageWidthBytesPerLine * bandHeight);
			}
		}
		else if ( now_y_pos > y_offsetLines + imageHeight_pixels)
		{
			memset( *pBandBuf, 0xff, imageWidthBytesPerLine * bandHeight );
		}
		else
		{
			unsigned int remainingImageLines = (y_offsetLines + imageHeight_pixels) - now_y_pos;
			if ((remainingImageLines / bandHeight) == 0)
			{
				cupsRasterReadPixels(ras, *pBandBuf, imageWidthBytesPerLine * remainingImageLines);
				memset( *pBandBuf+(imageWidthBytesPerLine * remainingImageLines), 0xff, imageWidthBytesPerLine * (bandHeight - remainingImageLines));
			}
			else
			{
				cupsRasterReadPixels(ras, *pBandBuf, imageWidthBytesPerLine * bandHeight);
			}
		}

		updatedHeader->cupsHeight = pageheight_pixels;
	}
	else
	{
		cupsRasterReadPixels(ras, *pBandBuf, imageWidthBytesPerLine * bandHeight);
	}

	if( pageWidth_pixels > imageWidth_pixels )
	{
		unsigned char *resizeImage = (unsigned char *)malloc(pageWidthBytesPerLine * bandHeight);
		if(resizeImage != NULL)
		{
			memset(resizeImage, 0xff, pageWidthBytesPerLine * bandHeight );
			unsigned int x_pos_src = 0;
			int line = 0;
			for(line =0 ; line < bandHeight; line++)
			{
				memcpy(&resizeImage[x_offset], *pBandBuf+x_pos_src, imageWidthBytesPerLine);
				x_offset += pageWidthBytesPerLine;
				x_pos_src += imageWidthBytesPerLine;
			}
			free(*pBandBuf);
			*pBandBuf = resizeImage;

			updatedHeader->cupsWidth        = pageWidth_pixels;
			updatedHeader->cupsBytesPerLine = pageWidthBytesPerLine;
		}
		else
		{
			ret = DEF_ERR;
		}
	}

	return ret;
}


int main(int argc, char * argv[])
{
	int err = DEF_NO_ERR;
	int ifd = 0;
	int poster = 1;
	bool isSendJobStart = false;
	struct sigaction sigact;
	cups_raster_t *ras = NULL;
	cups_page_header2_t header;

	cups_option_t *p_cups_opt = NULL;
	ppd_file_t *ppd = NULL;
	ppd_choice_t *p_Choice = NULL;
	int num_opt = 0;

	memset(&header, 0, sizeof( cups_page_header2_t ));

	debuglog(INDENT_NONE, "rasterfilter Start\n\n");

	if( (argc < 6) || (argc > 7) )
	{
		return DEF_ERR;
	}

#ifdef DEBUGLOG
		if (argv[5] != NULL)
			debuglog(INDENT_NONE, "argv[5] = %s\n", argv[5]);
#endif

	ppd = ppdOpenFile(getenv("PPD"));
	num_opt = cupsParseOptions(argv[5], 0, &p_cups_opt);
	if (ppd != NULL)
	{
		ppdMarkDefaults(ppd);
		cupsMarkOptions(ppd, num_opt, p_cups_opt);

		p_Choice = ppdFindMarkedChoice(ppd, "CNPoster");
		if (p_Choice != NULL && p_Choice->choice != NULL)
		{
			debuglog(INDENT_NONE, "CNPoster <p_Choice->choice> = %s\n", p_Choice->choice);
			poster = atoi(p_Choice->choice);
		}

		ppdClose(ppd);
	}
	else
	{
		if (p_cups_opt != NULL)
		{
			const char *strPoster = cupsGetOption("CNPoster", num_opt, p_cups_opt);
			if (strPoster != NULL)
			{
				debuglog(INDENT_NONE, "CNPoster = %s\n", strPoster);
				poster = atoi(strPoster);
			}
		}
	}

	if (p_cups_opt != NULL) {
		cupsFreeOptions(num_opt, p_cups_opt);
	}

	if (poster != POSTER_2X2 && poster != POSTER_3X3 && poster != POSTER_4X4)  {
		poster = 1;
	}

#ifdef _SFP_
	if( argv[5] != NULL )
	{
		int index = 0;
		bool bCall = false;
		int nChild_pid = 0;
		num_opt = cupsParseOptions(argv[5], 0, &p_cups_opt);
		if( num_opt < 0 ){
			zFreeCUPSOptions(num_opt, &p_cups_opt);
			return DEF_ERR;
		}
		if( p_cups_opt != NULL ){
			for (index = 0; index < num_opt; index++){
				if( strcmp( p_cups_opt[index].name, "CNCallCommandFileFilter") == 0 ){
					if(argc == 7){
						argv[6] = p_cups_opt[index].value;
					}
					bCall = true;
					break;
				}
			}
		}
		if( bCall != false ){
			nChild_pid = exec_commandfilefilter(argv);
			if( nChild_pid != -1 ){
				waitpid(nChild_pid, NULL, 0);
			}
			zFreeCUPSOptions(num_opt, &p_cups_opt);
			return DEF_NO_ERR;
		}
	}
#endif

	if(argc == 6)
	{
		ifd = 0;
	}
	else
	{
		if((ifd = open(argv[6], O_RDONLY)) == DEF_ERR)
		{
			return DEF_ERR;
		}
	}

	memset(&sigact, 0, sizeof(sigact));
	sigact.sa_handler = sigterm_handler;
	sigaction(SIGTERM, &sigact, NULL);

	CnRasterPipeFds *pFds = (CnRasterPipeFds *)calloc( sizeof(CnRasterPipeFds), 1 );
	if(pFds == NULL)
	{
		err = DEF_ERR;
	}
	else
	{
		int pid = 0;
		err = cnprocCreateProcess(&pid, pFds, argv);
		if(err == DEF_NO_ERR)
		{
			ras = cupsRasterOpen(ifd, CUPS_RASTER_READ);
			if( ras != NULL )
			{
				if (poster != 1) {
					err = processPoster(ras, pFds, poster, &isSendJobStart);
				}
				else
				{
					while ( cupsRasterReadHeader2(ras, &header) && (err == DEF_NO_ERR) && (g_sigterm_received == false))
					{
						unsigned char *pBandBuf = NULL;
						int nowBand = 0;
						char *pPageInfoStr = NULL;

						int bandHeight = BAND_HEIGHT;
						unsigned  pageheight_pixels		 = (unsigned int)((header.cupsPageSize[1] * header.HWResolution[1] / DPI_72) + 0.5);
						if( header.cupsHeight > pageheight_pixels )
						{
							pageheight_pixels = header.cupsHeight;
						}
						int totalBandNum   = pageheight_pixels / bandHeight;
						int lastBandHeight = pageheight_pixels % bandHeight;
						if(lastBandHeight != 0) {
							totalBandNum++;
						}else {
							lastBandHeight = bandHeight;
						}
						debuglog(INDENT_NONE, " # Total Band Num = %d # \n", totalBandNum);

						pBandBuf = (unsigned char*)malloc( header.cupsBytesPerLine * BAND_HEIGHT) ;
						if(pBandBuf == NULL)
						{
							err = DEF_ERR;
						}
						else
						{
							if((isSendJobStart == false) && (g_sigterm_received == false))
							{
								char pNumCopies[BUFSIZE]  = {0};
								snprintf(pNumCopies, sizeof(pNumCopies), "%u", header.NumCopies);
								err = cnprocWriteCommand(*pFds, CNRS_ID_START_JOB, pNumCopies, (int)(strlen(pNumCopies)+1));
								debuglog(INDENT_NONE, "# Write Command [CNRS_ID_START_JOB]\n");
								isSendJobStart = true;
							}

							if(err == DEF_NO_ERR)
							{
								for( nowBand = 0; nowBand < totalBandNum; nowBand++)
								{
									if(err != DEF_NO_ERR)
									{
										break;
									}
									if( (g_sigterm_received == true) && (isSendJobStart == true) )
									{
										cnprocWriteCommand(*pFds, CNRS_ID_JOB_CANCEL, NULL, 0);
										debuglog(INDENT_NONE, "# Write Command [CNRS_ID_JOB_CANCEL]\n");
										break;
									}

									if(nowBand == totalBandNum -1 )
									{
										bandHeight = lastBandHeight;
									}
									cups_page_header2_t updatedHeader = header;
									err = getBandData(ras, &pBandBuf, header, &updatedHeader, nowBand, bandHeight);
									if(err == DEF_NO_ERR)
									{
										pPageInfoStr = makePageInfoStr( &updatedHeader, updatedHeader.cupsWidth, updatedHeader.cupsHeight, 0, 0, nowBand, totalBandNum, bandHeight);
										if(pPageInfoStr == NULL)
										{
											err = DEF_ERR;
										}
										else
										{
											err = cnprocWriteCommand(*pFds, CNRS_ID_SEND_DATA, pPageInfoStr, (int)(strlen(pPageInfoStr)+1));
											if(err == DEF_NO_ERR)
											{
												err = cnprocWriteData( *pFds, (char*)pBandBuf,  updatedHeader.cupsBytesPerLine * bandHeight);
											}
											free(pPageInfoStr);
											pPageInfoStr = NULL;
										}
										debuglog(INDENT_NONE, "# Write Command [CNRS_ID_SEND_DATA]: now Band is %d\n\n", nowBand);
									}
								}
							}
							free(pBandBuf);
							pBandBuf = NULL;
						}
					}
				}
				cupsRasterClose(ras);
				ras = NULL;
			}

			if( (g_sigterm_received == true) && (isSendJobStart == true) )
			{
				cnprocWriteCommand(*pFds, CNRS_ID_JOB_CANCEL, NULL, 0);
				debuglog(INDENT_NONE, "# Write Command [CNRS_ID_JOB_CANCEL]\n");
			}
			if(isSendJobStart == true)
			{
				cnprocWriteCommand(*pFds, CNRS_ID_END_JOB, NULL, 0);
				debuglog(INDENT_NONE, "# Write Command [CNRS_ID_END_JOB]\n");
			}
		}

		if( pid > 0 )
		{
			waitpid(pid, NULL, 0);
		}

		free(pFds);
		pFds = NULL;
	}
	if(ifd)
	{
		close(ifd);
	}

	debuglog(INDENT_NONE, "rasterfilter End\n");
	return err;
}

static char* makePageInfoStr( cups_page_header2_t *header,  unsigned int dstWidth, unsigned int dstHeight, unsigned int marginX, unsigned int marginY, int nowBand, int totalBandNum, int bandHeight)
{
	char *pPageInfoStr = NULL;
	char pTmpBuf[BUFSIZE]  = {0};
	char pTmpStr[OPTION_STRING_MAXLENGTH] = {0};

	debuglog(INDENT_RIGHT, ">> makePageInfoStr\n");

	if(header == NULL)
	{
		return NULL;
	}

	strncpy(pTmpStr, "cupsWidth=", OPTION_STRING_MAXLENGTH-1 );
	snprintf(pTmpBuf, sizeof(pTmpBuf), "%u", header->cupsWidth);
	strncat(pTmpStr, pTmpBuf, (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));
	strncat(pTmpStr, ";", (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));

	strncat(pTmpStr, "cupsHeight=", (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));
	snprintf(pTmpBuf, sizeof(pTmpBuf), "%u", header->cupsHeight);
	strncat(pTmpStr, pTmpBuf, (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));
	strncat(pTmpStr, ";", (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));

	strncat(pTmpStr, "cupsBytesPerLine=", (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));
	snprintf(pTmpBuf, sizeof(pTmpBuf), "%u", header->cupsBytesPerLine);
	strncat(pTmpStr, pTmpBuf, (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));
	strncat(pTmpStr, ";", (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));

	strncat(pTmpStr, "resolution=", (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));
	snprintf(pTmpBuf, sizeof(pTmpBuf), "%u", header->HWResolution[0]);
	strncat(pTmpStr, pTmpBuf, (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));
	strncat(pTmpStr, ";", (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));

	strncat(pTmpStr, "dstWidth=", (OPTION_STRING_MAXLENGTH-1)- strlen(pTmpStr) );
	snprintf(pTmpBuf, sizeof(pTmpBuf), "%u", dstWidth);
	strncat(pTmpStr, pTmpBuf, (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));
	strncat(pTmpStr, ";", (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));

	strncat(pTmpStr, "dstHeight=", (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));
	snprintf(pTmpBuf, sizeof(pTmpBuf), "%u", dstHeight);
	strncat(pTmpStr, pTmpBuf, (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));
	strncat(pTmpStr, ";", (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));

	strncat(pTmpStr, "marginX=", (OPTION_STRING_MAXLENGTH-1)- strlen(pTmpStr) );
	snprintf(pTmpBuf, sizeof(pTmpBuf), "%u", marginX);
	strncat(pTmpStr, pTmpBuf, (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));
	strncat(pTmpStr, ";", (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));

	strncat(pTmpStr, "marginY=", (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));
	snprintf(pTmpBuf, sizeof(pTmpBuf), "%u", marginY);
	strncat(pTmpStr, pTmpBuf, (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));
	strncat(pTmpStr, ";", (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));

	strncat(pTmpStr, "totalBandNum=", (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));
	snprintf(pTmpBuf, sizeof(pTmpBuf), "%d", totalBandNum);
	strncat(pTmpStr, pTmpBuf, (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));
	strncat(pTmpStr, ";", (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));

	strncat(pTmpStr, "nowBandPos=", (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));
	snprintf(pTmpBuf, sizeof(pTmpBuf), "%d", nowBand);
	strncat(pTmpStr, pTmpBuf, (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));
	strncat(pTmpStr, ";", (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));

	strncat(pTmpStr, "bandheight=", (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));
	snprintf(pTmpBuf, sizeof(pTmpBuf), "%d", bandHeight);
	strncat(pTmpStr, pTmpBuf, (OPTION_STRING_MAXLENGTH - 1) - strlen(pTmpStr));

	pPageInfoStr = strdup(pTmpStr);

	debuglog(INDENT_NONE, "PageInfoStr = %s\n", pPageInfoStr);
	debuglog(INDENT_LEFT, "<< makePageInfoStr\n");

	return pPageInfoStr;
}

