// @HEADER
//
// ***********************************************************************
//
//        MueLu: A package for multigrid based preconditioning
//                  Copyright 2012 Sandia Corporation
//
// Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
// the U.S. Government retains certain rights in this software.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact
//                    Jonathan Hu       (jhu@sandia.gov)
//                    Andrey Prokopenko (aprokop@sandia.gov)
//                    Ray Tuminaro      (rstumin@sandia.gov)
//
// ***********************************************************************
//
// @HEADER

#ifndef MUELU_MAPTRANSFERFACTORY_DECL_HPP_
#define MUELU_MAPTRANSFERFACTORY_DECL_HPP_

#include "MueLu_ConfigDefs.hpp"
#include "MueLu_TwoLevelFactoryBase.hpp"

namespace MueLu {

/*!
  @class MapTransferFactory class.
  @brief Factory to transfer a map from a fine to a coarse level

  Factory that transfers a map (given by a variable name and a generating factory) for building
  a coarse version of the map. The coarse map is stored on the coarse level using the same variable name
  and generating factory than the original fine level map.

  The transfer is based on the prolongator maps. The prolongator/nullspace can also contain rotational modes,
  that might not be of interest for the map of interest. Use the option "nullspace vectors: limit to" to exclude such modes.

  ## Input/output ##

  ### User parameters ###
  Parameter | type | default | master.xml | validated | requested | description
  ----------|------|---------|:----------:|:---------:|:---------:|------------
  map: name                   | string  | ""     |   | * | * | Name of the map
  map: factory                | string  | "null" |   | * | * | Name of the generating factory
  P                           | Factory | null   |   | * | * | Generating factory of prolongator
  nullspace vectors: limit to | string  | "all"  |   | * | * | Use only these nullspace vectors/columns of P to transfer the map (e.g. to drop rotations)

  The * in the @c master.xml column denotes that the parameter is defined in the @c master.xml file.<br>
  The * in the @c validated column means that the parameter is declared in the list of valid input parameters (see @c GetValidParameters() ).<br>
  The * in the @c requested column states that the data is requested as input with all dependencies (see @c DeclareInput() ).

  ### Variables provided by this factory ###

  After \c Build() , the following data is available (if requested):

  Parameter | generated by | description
  ----------|--------------|------------
  | map: name | MapTransferFactory | Coarse version of the input map

*/

template <class Scalar        = DefaultScalar,
          class LocalOrdinal  = DefaultLocalOrdinal,
          class GlobalOrdinal = DefaultGlobalOrdinal,
          class Node          = DefaultNode>
class MapTransferFactory : public TwoLevelFactoryBase {
#undef MUELU_MAPTRANSFERFACTORY_SHORT
#include "MueLu_UseShortNames.hpp"

 public:
  //! Input
  //@{

  RCP<const ParameterList> GetValidParameterList() const override;

  void DeclareInput(Level& fineLevel, Level& coarseLevel) const override;

  //@}

  //@{
  //! @name Build methods.

  //! Build an object with this factory.
  void Build(Level& fineLevel, Level& coarseLevel) const override;

  //@}

 private:
  /*!
    @brief Get the max number of entries per row of P to be considered for map transfer

    To exclude some nullspace vectors (e.g. rotations in 2D or 3D elasticity), when doing the map transfer,
    this routine translates the user wish to a maximal number of entries per row of P to be considered during the map transfer.

    \warning Rows of P are looped from left to right, so we rely on the usual ordering of the nullspace vectors (translations in x/y/z, then rotations around x,y,z).

    @param[in] pL Parameter list with user-given configuration
    @return Number of entries per row of the prolongator to be used for the map transfer
  */
  int GetLimitOfProlongatorColumns(const ParameterList& pL) const;

  //! Generating factory of input variable
  mutable RCP<const FactoryBase> mapFact_;

};  // class MapTransferFactory

}  // namespace MueLu

#define MUELU_MAPTRANSFERFACTORY_SHORT
#endif /* MUELU_MAPTRANSFERFACTORY_DECL_HPP_ */
