//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// GEANT4 Class header file
//
//
// File name:     G4AtimaEnergyLossModel
//
// Author:        Jose Luis Rodriguez Sanchez on base of ATIMA code
//
// Creation date: 16.01.2018
//
// Modifications:
// 09/10/2018: Solved bug in the determination of the material atomic number
//
// Class Description:
//
// This model calculates the stopping power of ions according 
// to the ATIMA code developed at GSI, Darmstadt, Germany.
// For details: http://web-docs.gsi.de/~weick/atima/
//
// Helmut Weick, GSI (responsible for fortran version)
// Andrej Prochazka, GSI (responsible for C version)
// Christoph Scheidenberger, GSI (project coordination)
//
// -------------------------------------------------------------------
//
//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

#include "G4AtimaEnergyLossModel.hh"
#include "Randomize.hh"
#include "G4PhysicalConstants.hh"
#include "G4SystemOfUnits.hh"
#include "G4Electron.hh"
#include "G4GenericIon.hh"
#include "G4LossTableManager.hh"
#include "G4EmCorrections.hh"
#include "G4ParticleChangeForLoss.hh"
#include "G4DeltaAngle.hh"
#include "G4Log.hh"
#include "G4Exp.hh"
#include "G4Pow.hh"

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

using namespace std;

G4double G4AtimaEnergyLossModel::stepE = 0.0;
G4double G4AtimaEnergyLossModel::tableE[] = {0.0};

G4AtimaEnergyLossModel::G4AtimaEnergyLossModel(const G4ParticleDefinition*, 
                                     const G4String& nam)
  : G4VEmModel(nam),
    particle(nullptr)
{
  g4calc = G4Pow::GetInstance();
  fParticleChange = nullptr;
  theElectron = G4Electron::Electron();
  corr = G4LossTableManager::Instance()->EmCorrections();  
  nist = G4NistManager::Instance();
  SetLowEnergyLimit(2.0*MeV);
  MLN10 = 2.30258509299;
  atomic_mass_unit = 931.4940954; // MeV/c^2
  dedx_constant = 0.3070749187;  //4*pi*Na*me*c^2*r_e^2  //MeV cm^2
  electron_mass = 0.510998928;   // MeV/c^2
  fine_structure = 1.0/137.035999139;
  domega2dx_constant = dedx_constant*electron_mass;  //4*pi*Na*me*c^2*r_e^2  //MeV^2 cm^2
  if(tableE[0] == 0.0) {
    G4double logmin = 0.;
    G4double logmax = 5.;
    stepE = (logmax-logmin)/199.;
    for(G4int i=0; i<200; ++i){
      tableE[i] = G4Exp(MLN10*(logmin + ((G4double)i)*stepE));
    }
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4AtimaEnergyLossModel::~G4AtimaEnergyLossModel() = default;

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void G4AtimaEnergyLossModel::Initialise(const G4ParticleDefinition* p,
                                   const G4DataVector&)
{
  SetGenericIon(p);
  SetParticle(p);

  //G4cout << "G4AtimaEnergyLossModel::Initialise for " << p->GetParticleName()
  //         << "  isIon= " << isIon
  //         << G4endl;

  // always false before the run
  SetDeexcitationFlag(false);

  if(nullptr == fParticleChange) {
    fParticleChange = GetParticleChangeForLoss();
    if(UseAngularGeneratorFlag() && !GetAngularDistribution()) {
      SetAngularDistribution(new G4DeltaAngle());
    }
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4double G4AtimaEnergyLossModel::GetChargeSquareRatio(const G4ParticleDefinition* p,
                                                 const G4Material* mat,
                                                 G4double kineticEnergy)
{
  // this method is called only for ions
  G4double q2 = corr->EffectiveChargeSquareRatio(p,mat,kineticEnergy);
  corrFactor = q2*corr->EffectiveChargeCorrection(p,mat,kineticEnergy);
  return corrFactor;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4double G4AtimaEnergyLossModel::GetParticleCharge(const G4ParticleDefinition* p,
                                              const G4Material* mat,
                                              G4double kineticEnergy)
{
  //G4cout<<"G4AtimaEnergyLossModel::GetParticleCharge e= "<<kineticEnergy <<
  //  " q= " <<  corr->GetParticleCharge(p,mat,kineticEnergy) <<G4endl;
  // this method is called only for ions, so no check if it is an ion
  return corr->GetParticleCharge(p,mat,kineticEnergy);
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo....

void G4AtimaEnergyLossModel::SetupParameters()
{
  mass = particle->GetPDGMass();
  spin = particle->GetPDGSpin();
  G4double q = particle->GetPDGCharge()/eplus;
  chargeSquare = q*q;
  corrFactor = chargeSquare;
  ratio = electron_mass_c2/mass;
  static const G4double aMag = 1./(0.5*eplus*hbar_Planck*c_squared);
  G4double magmom = particle->GetPDGMagneticMoment()*mass*aMag;
  magMoment2 = magmom*magmom - 1.0;
  formfact = 0.0;
  tlimit = DBL_MAX;
  if(particle->GetLeptonNumber() == 0) {
    G4int iz = G4lrint(q);
    if(iz <= 1) {
      formfact = (spin == 0.0 && mass < GeV) ? 1.181e-6 : 1.548e-6;	
    } else {
      G4double x = nist->GetA27(iz);
      formfact = 3.969e-6*x*x;
    }
    tlimit = std::sqrt(0.414/formfact +
                       electron_mass_c2*electron_mass_c2) - electron_mass_c2;
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4double G4AtimaEnergyLossModel::MinEnergyCut(const G4ParticleDefinition*,
                                         const G4MaterialCutsCouple* couple)
{
  return couple->GetMaterial()->GetIonisation()->GetMeanExcitationEnergy();
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4double 
G4AtimaEnergyLossModel::ComputeCrossSectionPerElectron(const G4ParticleDefinition* p,
                                                  G4double kineticEnergy,
                                                  G4double cutEnergy,
                                                  G4double maxKinEnergy)        
{
  G4double cross = 0.0;
  G4double tmax = MaxSecondaryEnergy(p, kineticEnergy);
  G4double maxEnergy = std::min(tmax,maxKinEnergy);
  if(cutEnergy < maxEnergy) {

    G4double totEnergy = kineticEnergy + mass;
    G4double energy2   = totEnergy*totEnergy;
    G4double beta2     = kineticEnergy*(kineticEnergy + 2.0*mass)/energy2;

    cross = (maxEnergy - cutEnergy)/(cutEnergy*maxEnergy) 
      - beta2*G4Log(maxEnergy/cutEnergy)/tmax;

    // +term for spin=1/2 particle
    if( 0.0 < spin ) { cross += 0.5*(maxEnergy - cutEnergy)/energy2; }

    cross *= twopi_mc2_rcl2*chargeSquare/beta2;
  }
  
   // G4cout << "BB: e= " << kineticEnergy << " tmin= " << cutEnergy 
   //        << " tmax= " << tmax << " cross= " << cross << G4endl;
  
  return cross;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4double G4AtimaEnergyLossModel::ComputeCrossSectionPerAtom(
                                           const G4ParticleDefinition* p,
                                                 G4double kineticEnergy,
                                                 G4double Z, G4double,
                                                 G4double cutEnergy,
                                                 G4double maxEnergy)
{
  G4double cross = Z*ComputeCrossSectionPerElectron
                                         (p,kineticEnergy,cutEnergy,maxEnergy);
  return cross;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4double G4AtimaEnergyLossModel::CrossSectionPerVolume(
                                           const G4Material* material,
                                           const G4ParticleDefinition* p,
                                                 G4double kineticEnergy,
                                                 G4double cutEnergy,
                                                 G4double maxEnergy)
{
  G4double eDensity = material->GetElectronDensity();
  G4double cross = eDensity*ComputeCrossSectionPerElectron
                                         (p,kineticEnergy,cutEnergy,maxEnergy);
  return cross;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4double G4AtimaEnergyLossModel::ComputeDEDXPerVolume(const G4Material* material,
                                                 const G4ParticleDefinition* p,
                                                 G4double kineticEnergy,
                                                 G4double)
{
  //Call to ATIMA Stopping Power function
  G4double zt = material->GetIonisation()->GetZeffective();
  zt = std::min(zt,93.); 
  G4double at = nist->GetAtomicMassAmu(G4lrint(zt));
  G4double dedx = StoppingPower(p->GetPDGMass(), p->GetPDGCharge(), 
    kineticEnergy/(MeV), at, zt) *material->GetDensity()/(g/cm3);
  dedx = std::max(dedx, 0.0);

  //  G4cout << "E(MeV)= " << kineticEnergy/MeV << " dedx= " << dedx 
  //          << "  " << material->GetName() << G4endl;
  return dedx;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void G4AtimaEnergyLossModel::CorrectionsAlongStep(const G4MaterialCutsCouple* couple,
                                                  const G4DynamicParticle* dp,
                                                  const G4double& length,
                                                  G4double& eloss)
{
  if(isIon) {
    const G4ParticleDefinition* p = dp->GetDefinition();
    const G4Material* mat = couple->GetMaterial();
    G4double cutEnergy = DBL_MAX;
    G4double kineticEnergy = dp->GetKineticEnergy();
    eloss = ComputeDEDXPerVolume(mat, p, kineticEnergy, cutEnergy)*length/(cm);
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void G4AtimaEnergyLossModel::SampleSecondaries(vector<G4DynamicParticle*>* vdp,
                                          const G4MaterialCutsCouple* couple,
                                          const G4DynamicParticle* dp,
                                          G4double minKinEnergy,
                                          G4double maxEnergy)
{
  G4double kineticEnergy = dp->GetKineticEnergy();
  G4double tmax = MaxSecondaryEnergy(dp->GetDefinition(),kineticEnergy);

  G4double maxKinEnergy = std::min(maxEnergy,tmax);
  if(minKinEnergy >= maxKinEnergy) { return; }

  //G4cout << "G4AtimaEnergyLossModel::SampleSecondaries Emin= " << minKinEnergy
  //         << " Emax= " << maxKinEnergy << G4endl;

  G4double totEnergy     = kineticEnergy + mass;
  G4double etot2         = totEnergy*totEnergy;
  G4double beta2         = kineticEnergy*(kineticEnergy + 2.0*mass)/etot2;

  G4double deltaKinEnergy, f; 
  G4double f1 = 0.0;
  G4double fmax = 1.0;
  if( 0.0 < spin ) { fmax += 0.5*maxKinEnergy*maxKinEnergy/etot2; }

  CLHEP::HepRandomEngine* rndmEngineMod = G4Random::getTheEngine();
  G4double rndm[2];

  // sampling without nuclear size effect
  do {
    rndmEngineMod->flatArray(2, rndm);
    deltaKinEnergy = minKinEnergy*maxKinEnergy
                    /(minKinEnergy*(1.0 - rndm[0]) + maxKinEnergy*rndm[0]);

    f = 1.0 - beta2*deltaKinEnergy/tmax;
    if( 0.0 < spin ) {
      f1 = 0.5*deltaKinEnergy*deltaKinEnergy/etot2;
      f += f1;
    }

    // Loop checking, 03-Aug-2015, Vladimir Ivanchenko
  } while( fmax*rndm[1] > f);

  // projectile formfactor - suppresion of high energy
  // delta-electron production at high energy
  
  G4double x = formfact*deltaKinEnergy*(deltaKinEnergy + 2*electron_mass_c2);
  if(x > 1.e-6) {

    G4double x10 = 1.0 + x;
    G4double grej  = 1.0/(x10*x10);
    if( 0.0 < spin ) {
      G4double x2 = 0.5*electron_mass_c2*deltaKinEnergy/(mass*mass);
      grej *= (1.0 + magMoment2*(x2 - f1/f)/(1.0 + x2));
    }
    if(grej > 1.1) {
      G4cout << "### G4AtimaEnergyLossModel WARNING: grej= " << grej
             << "  " << dp->GetDefinition()->GetParticleName()
             << " Ekin(MeV)= " <<  kineticEnergy
             << " delEkin(MeV)= " << deltaKinEnergy
             << G4endl;
    }
    if(rndmEngineMod->flat() > grej) { return; }
  }

  G4ThreeVector deltaDirection;

  if(UseAngularGeneratorFlag()) {

    const G4Material* mat =  couple->GetMaterial();
    G4int Z = SelectRandomAtomNumber(mat);

    deltaDirection = 
      GetAngularDistribution()->SampleDirection(dp, deltaKinEnergy, Z, mat);

  } else {
 
    G4double deltaMomentum =
      std::sqrt(deltaKinEnergy * (deltaKinEnergy + 2.0*electron_mass_c2));
    G4double cost = deltaKinEnergy * (totEnergy + electron_mass_c2) /
      (deltaMomentum * dp->GetTotalMomentum());
    if(cost > 1.0) { cost = 1.0; }
    G4double sint = std::sqrt((1.0 - cost)*(1.0 + cost));

    G4double phi = twopi*rndmEngineMod->flat();

    deltaDirection.set(sint*cos(phi),sint*sin(phi), cost) ;
    deltaDirection.rotateUz(dp->GetMomentumDirection());
  }  
  /*
    G4cout << "### G4AtimaEnergyLossModel " 
           << dp->GetDefinition()->GetParticleName()
           << " Ekin(MeV)= " <<  kineticEnergy
           << " delEkin(MeV)= " << deltaKinEnergy
           << " tmin(MeV)= " << minKinEnergy
           << " tmax(MeV)= " << maxKinEnergy
           << " dir= " << dp->GetMomentumDirection()
           << " dirDelta= " << deltaDirection
           << G4endl;
  */
  // create G4DynamicParticle object for delta ray
  auto delta = new G4DynamicParticle(theElectron,deltaDirection,deltaKinEnergy);

  vdp->push_back(delta);

  // Change kinematics of primary particle
  kineticEnergy -= deltaKinEnergy;
  G4ThreeVector finalP = dp->GetMomentum() - delta->GetMomentum();
  finalP               = finalP.unit();
  
  fParticleChange->SetProposedKineticEnergy(kineticEnergy);
  fParticleChange->SetProposedMomentumDirection(finalP);
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo....

G4double G4AtimaEnergyLossModel::MaxSecondaryEnergy(const G4ParticleDefinition* pd,
                                               G4double kinEnergy) 
{
  // here particle type is checked for any method
  SetParticle(pd);
  G4double tau  = kinEnergy/mass;
  G4double tmax = 2.0*electron_mass_c2*tau*(tau + 2.) /
                  (1. + 2.0*(tau + 1.)*ratio + ratio*ratio);
  return std::min(tmax,tlimit);
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4double G4AtimaEnergyLossModel::StoppingPower(G4double ap, G4double zp, 
         G4double ep, G4double at, G4double zt){
  //
  if(ep==0)return 0.0;
  ap=ap/atomic_mass_unit;
  ep=ep/ap;
  G4double se = 0.0;
  // ep in MeV
  if(ep<=10.){
        se = sezi_dedx_e(zp, ep, at, zt);
  }
  else if(ep>10. && ep<30.){
    G4double factor = 0.05 * ( ep - 10.0 );
    se = (1.0-factor)*sezi_dedx_e(zp, ep, at, zt) + factor*Bethek_dedx_e(ap, zp, ep, at, zt);
  }
  else {
    se = Bethek_dedx_e(ap, zp, ep, at, zt);
  }
  return se + dedx_n(ap, zp, ep, at, zt);
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4double G4AtimaEnergyLossModel::dedx_n(const G4double ap, const G4double zp, 
         const G4double ep, const G4double at, const G4double zt){
    
  G4double zpowers = g4calc->powA(zp,0.23)+g4calc->powA(zt,0.23);
  G4double asum = ap + at;
  G4double epsilon = 32.53*at*1000.*ep*ap/(zp*zt*asum*zpowers); //projectile energy is converted from MeV/u to keV
  G4double sn=0.;
    
  if(epsilon<=30.0){
    sn = G4Log(1.+(1.1383*epsilon))/ (2.*(epsilon + 0.01321*g4calc->powA(epsilon,0.21226) + 0.19593*sqrt(epsilon)));
  }
  else{
    sn = G4Log(epsilon)/(2.*epsilon);
  }
  sn = 100.*8.4621*zp*zt*ap*sn*Avogadro/1.e+23/(asum*zpowers*at);
  return sn;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4double G4AtimaEnergyLossModel::sezi_dedx_e(const G4double zp, const G4double ep, const G4double at, const G4double zt){
  G4double e = ep*1000.; // e in keV/u
  G4double se = 0.;

  // heavy ion Z > 2
  G4double h1,h4;
  G4double a,q,b;
  G4double l1,l0,l;
  G4double YRmin = 0.130; //  YRmin = VR / ZP**0.67 <= 0.13 OR VR <= 1.0
  G4double VRmin = 1.0;
  G4double vfermi = atima_vfermi[(G4int)zt-1];
  G4double yr=0;
  G4double zeta = 0;
        
  G4double v = std::sqrt(e/25.0)/vfermi;
  G4double v2= v*v;
        
  G4double vr = (v >= 1)? v*vfermi*(1.+ 1./(5.*v2)) : 3.0*vfermi/4.0*(1.0+v2*(2.0/3.0-v2/15.0));
        
  h1= 1./g4calc->powA(zp,0.6667);
  yr = std::max(YRmin,vr*h1);
  yr = std::max(yr, VRmin*h1);
        
  //--  CALCULATE ZEFF
  a = -0.803*g4calc->powA(yr,0.3) + 1.3167*g4calc->powA(yr,0.6) + 0.38157*yr + 0.008983*yr*yr;
  q = std::min(1.0, std::max(0.0 , (1.0 - G4Exp(-std::min(a, 50.0))))); //-- Q = IONIZATION LEVEL OF THE ION AT RELATIVE VELOCITY YR
        
  //-- IONIZATION LEVEL TO EFFECTIVE CHARGE
  h1 = 1./g4calc->powA(zp,0.3333);
        
  b = (std::min(0.43, std::max(0.32,0.12 + 0.025*zp)))*h1;
  l0 = (0.8 - q * std::min(1.2,0.6 +zp/30.0))*h1;
        if(q < 0.2){
            l1 = 0;
            }
        else{
            if (q < std::max(0.0,0.9-0.025*zp)){
                l1 = b*(q-0.2)/std::abs(std::max(0.0,0.9-0.025*zp)-0.2000001);
                }
	     	 else{
             	if(q < std::max(0.0,1.0 - 0.025*std::min(16.,zp))) l1 = b;
                else l1 = b*(1.0 - q)/(0.025*std::min(16.,zp));
                }
        }
        // calculate screening
        l = std::max(l1,l0*atima_lambda_screening[(G4int)zp-1]);
        h1 =4.0*l*vfermi/1.919;
        zeta = q + (1./(2.*(vfermi*vfermi)))*(1. - q)* G4Log(1. + h1*h1);
         
         // ZP**3 EFFECT AS IN REF. 779?
        a = 7.6 - std::max(0.0, G4Log(e));
        zeta = zeta*(1. + (1./(zp*zp))*(0.18 + .0015*zt)*G4Exp(-a*a));
        
        h1= 1./g4calc->powA(zp,0.6667);
        if (yr <= ( std::max(YRmin, VRmin*h1))){
            VRmin=std::max(VRmin, YRmin/h1);
            //--C        ..CALCULATE VELOCITY STOPPING FOR YR < YRmin
            G4double vmin =.5*(VRmin + std::sqrt(std::max(0.0,VRmin*VRmin - 0.8*vfermi*vfermi)));
            G4double eee = 25.0*vmin*vmin;
            G4double eval = 1.0;
            if((zt == 6) || (((zt == 14) || (zt == 32)) &&  (zp <= 19))) eval = 0.35;
            else eval = 0.5;
            
            h1 = zeta *zp;
            h4 = g4calc->powA(e / eee,eval);
            se = sezi_p_se(eee*0.001,at,zt) * h1*h1*h4;
            return se;
        }
        else {
            se = sezi_p_se(ep,at,zt)*g4calc->powA(zeta*zp,2.0);
            return se;
        }

   return se;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4double G4AtimaEnergyLossModel::sezi_p_se(const G4double energy, const G4double at, const G4double zt){
  G4double sp = 0.;
  G4double e = 1000*energy; //e in keV/u
  G4int i = zt - 1;
    
  if(e<=25)e=25;
   G4double sl = (proton_stopping_coef[i][0]*g4calc->powA(e,proton_stopping_coef[i][1])) + (proton_stopping_coef[i][2]*g4calc->powA(e,proton_stopping_coef[i][3]));
   G4double sh = proton_stopping_coef[i][4]/g4calc->powA(e,proton_stopping_coef[i][5]) * G4Log( (proton_stopping_coef[i][6]/e) + (proton_stopping_coef[i][7]*e));
   sp = sl*sh/(sl+sh);
   e=1000*energy;
   if(e<=25){
       sp *=(zt>6)?g4calc->powA(e/25.,0.45):g4calc->powA(e/25.,0.25);
   }
  return 100.*sp*Avogadro/1.e+23/at;
}


//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4double G4AtimaEnergyLossModel::Bethek_dedx_e(G4double ap, G4double zp, G4double ep, G4double at, G4double zt){
//
// According to C. Scheidenberger et al., Phys. Rev. Lett. 73, 50 (1994).
//
    G4double gamma=1.0 + ep/atomic_mass_unit;
    G4double beta2=1.0-1.0/(gamma*gamma);
    //
    G4double beta = std::sqrt(beta2);
    G4double zp_eff = zp*(1.0-G4Exp(-0.95/fine_structure*beta/g4calc->A23(zp)));
    G4int zi = std::min(std::max((G4int)zt, 1), 120);
    G4double Ipot = ionisation_potentials_z[zi];
    G4double f1 = dedx_constant*g4calc->powA(zp_eff,2.)*zt/(beta2*at);

    //
    G4double f2 = G4Log(2.0*electron_mass*1000000*beta2/Ipot);
    G4double eta = beta*gamma;

    if(!(eta>=0.13)){ //shell corrections
        G4double cor = (+0.422377*g4calc->powA(eta,-2.)
                    +0.0304043*g4calc->powA(eta,-4.)
                    -0.00038106*g4calc->powA(eta,-6.))*1e-6*g4calc->powA(Ipot,2.) 
                  +(+3.858019*g4calc->powA(eta,-2.) 
                    -0.1667989*(g4calc->powA(eta,-4.))
                    +0.00157955*(g4calc->powA(eta,-6.)))*1.0e-9*g4calc->powA(Ipot,3.); 
        f2 = f2 -cor/zt;
    }

    f2+=2*G4Log(gamma) -beta2;

    //Barkas correction
    G4double barkas=1.0;
    G4double V2FVA[4]={0.33,0.30,0.26,0.23};
    G4double VA[4]={1.,2.,3.,4.};
    G4double v1 = eta/(fine_structure*std::sqrt(zt));
    G4double v2fv;
    if(v1 >= 4.){
        v2fv = 0.45 / std::sqrt(v1);
        }
    else if(v1 > 1. && v1 < 4.){//VALUES FROM THE JACKSON MC CARTHY FUNCTION //PHYS. REV. B 6 4131
        G4int i=1;
        for(; i<4; i++){
	  if(VA[i] >= v1) break;
	}
        i = std::min(i, 3);
	v2fv = V2FVA[i-1]+(v1-VA[i-1])*(V2FVA[i]-V2FVA[i-1])/(VA[i]-VA[i-1]);
    }
    else{
        v2fv=0;
    }
    barkas= 1.0+2.0 * zp_eff * v2fv /(v1*v1*std::sqrt(zt));

    //Fermi-density effect for relativistic velocities
    gamma = 1./std::sqrt(1-(beta*beta));
    G4double x = G4Log(beta * gamma) / 2.302585;
    G4int i = std::min(std::max(zi-1,0), 91);
    G4double del = 0.;
    
    if (x < x0[i] ){
        if(del_0[i] > 0.)del = del_0[i] * g4calc->powA(10.0,(2.*(x-x0[i])));
        }
    else {
        del = 4.6052 * x - c[i];
        if ( x0[i]<= x &&  x <= x1[i] ) del +=  afermi[i] * g4calc->powA((x1[i] - x),m0[i]);
        }

    //Precalculated lindhard correction
    G4double LS = 0.;
    G4int z = G4lrint(zp);
    if(z>109)z=109;
    if(ep<tableE[0])ep=tableE[0];

    G4double da = (ap - element_atomic_weights[z])/element_atomic_weights[z];
    z = z-1;

    G4double v3 = EnergyTable_interpolate(ep,ls_coefficients_a[z]);
    G4double v4 = EnergyTable_interpolate(ep,ls_coefficients_ahi[z]);
    G4double dif = v4 - v3;
    LS = v3+(dif*da/0.05);

    //Final stopping power
    G4double result  = (f2)*barkas + LS - del/2.;
    result *=f1;
    
    return result;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4double G4AtimaEnergyLossModel::EnergyTable_interpolate(G4double xval,const G4double* y){
     G4double r;
     G4int num=200;
     G4double lxval = G4Log(xval)/MLN10;
     if(xval<tableE[0] || xval>tableE[num-1])return 0.0;
     if(xval==tableE[num-1])return y[num-1];
     G4int i = (G4int)(lxval/stepE);
     i = std::min(std::max(i, 0), num-2);
     G4double linstep = tableE[i+1] - tableE[i];
     G4double x = 1.0 - ((xval - tableE[i])/linstep);
     r = (x*y[i]) + ((1-x)*y[i+1]);
     return r;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

const G4double G4AtimaEnergyLossModel::proton_stopping_coef[92][8] = { // proton in material stopping coefficient
{  .0091827, .0053496, .69741, .48493, 316.07,1.0143, 9329.3, .0539890}, //H
{  .11393, .0051984, 1.0822, .39252,  1081.0, 1.0645, 4068.5, .0176990}, //He
{  .85837, .0050147, 1.6044, .38844,  1337.3, 1.0470, 2659.2, .01898},
{  .8781,  .0051049, 5.4232, .2032,   1200.6, 1.0211, 1401.8, .0385290},
{ 1.4608,  .0048836, 2.338,  .44249, 1801.3, 1.0352, 1784.1, .02024},
{ 3.2579,  .0049148, 2.7156, .36473, 2092.2, 1.0291, 2643.6, .0182370}, //C
{  .59674, .0050837, 4.2073, .30612,  2394.2, 1.0255, 4892.1, .0160060},
{  .75253, .0050314, 4.0824, .30067, 2455.8, 1.0181, 5069.7, .0174260}, //O
{ 1.226,   .0051385, 3.2246, .32703, 2525.0, 1.0142, 7563.6, .0194690},
{ 1.0332,  .0051645, 3.004,  .33889, 2338.6, .99997, 6991.2, .0217990}, //Ne
{ 6.0972,  .0044292, 3.1929, .45763, 1363.3, .95182, 2380.6, .0818350},
{14.013,   .0043646, 2.2641, .36326, 2187.4, .99098, 6264.8, .0462},
{ .039001, .0045415, 5.5463, .39562, 1589.2, .95316, 816.16, .0474840},
{ 2.072,   .0044516, 3.5585, .53933, 1515.2, .93161, 1790.3, .0351980},
{17.575,   .0038346, .078694, 1.2388,2806.0, .97284, 1037.6, .0128790},
{16.126,   .0038315, .054164, 1.3104,2813.3, .96587, 1251.4, .0118470},
{ 3.217,   .0044579, 3.6696, .5091,  2734.6, .96253, 2187.5, .0169070},
{ 2.0379,  .0044775, 3.0743, .54773, 3505.0, .97575, 1714.00, .0117010},
{ .74171, .0043051, 1.1515, .95083,   917.21, .8782, 389.93, .18926},
{ 9.1316,  .0043809, 5.4611, .31327, 3891.8, .97933, 6267.9, .0151960}, //Ca
{ 7.2247,  .0043718, 6.1017, .37511, 2829.2, .95218, 6376.1, .0203980},
{  .147,   .0048456, 6.3485, .41057, 2164.1, .94028, 5292.6, .0502630},
{ 5.0611,  .0039867, 2.6174, .57957, 2218.9, .92361, 6323.00, .0256690},
{  .53267, .0042968, .39005, 1.2725, 1872.7, .90776, 64.166, .0301070},
{  .47697, .0043038, .31452, 1.3289, 1920.5, .90649, 45.576, .0274690},
{  .027426, .0035443, .031563, 2.1755,  1919.5, .90099, 23.902, .0253630},
{  .16383, .0043042, .073454, 1.8592, 1918.4, .89678, 27.61, .0231840},
{ 4.2562,  .0043737, 1.5606, .72067, 1546.8, .87958, 302.02, .0409440}, //Ni
{ 2.3508,  .0043237, 2.882,  .50113, 1837.7, .89992, 2377.00, .04965},
{ 3.1095,  .0038455, .11477, 1.5037, 2184.7, .89309, 67.306, .0165880},
{15.322,   .0040306, .65391, .67668, 3001.7, .92484, 3344.2, .0163660},
{ 3.6932,  .0044813, 8.608,  .27638,  2982.7, .9276, 3166.6, .0308740},
{ 7.1373,  .0043134, 9.4247, .27937,  2725.8, .91597, 3166.1, .0250080},
{ 4.8979,  .0042937, 3.7793, .50004,  2824.5, .91028, 1282.4, .0170610},
{ 1.3683,  .0043024, 2.5679, .60822,  6907.8, .9817, 628.01, .0068055},
{ 1.8301,  .0042983, 2.9057, .6038,   4744.6, .94722, 936.64, .0092242},
{  .42056, .0041169, .01695, 2.3616, 2252.7, .89192, 39.752, .0277570},
{30.78,    .0037736, .55813, .76816, 7113.2, .97697, 1604.4, .0065268},
{11.576,   .0042119, 7.0244, .37764, 4713.5, .94264, 2493.2, .01127},
{ 6.2406,  .0041916, 5.2701, .49453, 4234.6, .93232, 2063.9, .0118440},
{  .33073, .0041243, 1.7246, 1.1062, 1930.2, .86907, 27.416, .0382080},
{  .017747, .0041715, .14586, 1.7305,1803.6, .86315, 29.669, .0321230},
{ 3.7229,  .0041768, 4.6286, .56769, 1678.0, .86202, 3094.00, .06244},
{  .13998, .0041329, .25573, 1.4241, 1919.3, .86326, 72.797, .0322350},
{  .2859,  .0041386, .31301, 1.3424, 1954.8, .86175, 115.18, .0293420},
{ .76,    .0042179, 3.386,  .76285,  1867.4, .85805, 69.994, .0364480},
{ 6.3957,  .0041935, 5.4689, .41378, 1712.6, .85397,18493.00, .0564710},
{ 3.4717,  .0041344, 3.2337, .63788, 1116.4, .81959, 4766.0, .1179},
{ 2.5265,  .0042282, 4.532,  .53562, 1030.8, .81652,16252.0, .19722},
{ 7.3683,  .0041007, 4.6791, .51428, 1160.0, .82454,17965.0, .13316},
{ 7.7197,  .004388, 3.242,   .68434, 1428.1, .83398, 1786.7, .0665120},
{16.78,    .0041918, 9.3198, .29568, 3370.9, .90289, 7431.7, .02616},
{ 4.2132,  .0042098, 4.6753, .57945, 3503.9, .89261, 1468.9, .0143590},
{ 4.0818,  .004214, 4.4425,  .58393, 3945.3, .90281, 1340.5, .0134140}, //Xe
{.18517, .0036215, .00058788,3.5315, 2931.3, .88936, 26.18, .0263930},
{ 4.8248,  .0041458, 6.0934, .57026, 2300.1, .86359, 2980.7, .0386790},
{  .49857, .0041054, 1.9775, .95877, 786.55, .78509, 806.6, .40882},
{ 3.2754,  .0042177, 5.768,  .54054, 6631.3, .94282, 744.07, .0083026},
{ 2.9978,  .0040901, 4.5299, .62025, 2161.2, .85669, 1268.6, .0430310},
{ 2.8701,  .004096, 4.2568,  .6138,  2130.4, .85235, 1704.1, .0393850},
{10.853,   .0041149, 5.8907, .46834, 2857.2, .8755, 3654.2, .0299550},
{ 3.6407,  .0041782, 4.8742, .57861, 1267.7, .82211, 3508.2, .24174},
{17.645,   .0040992, 6.5855, .32734, 3931.3, .90754, 5156.7, .0362780},
{ 7.5309,  .0040814, 4.9389, .50679, 2519.7, .85819, 3314.6, .0305140},
{ 5.4742,  .0040829, 4.897,  .51113, 2340.1, .85296, 2342.7, .0356620},
{ 4.2661,  .0040667, 4.5032, .55257, 2076.4, .84151, 1666.6, .0408010},
{ 6.8313,  .0040486, 4.3987, .51675, 2003.0, .83437, 1410.4, .03478},
{ 1.2707,  .0040553, 4.6295, .57428, 1626.3, .81858, 995.68, .0553190},
{ 5.7561,  .0040491, 4.357,  .52496, 2207.3, .83796, 1579.5, .0271650},
{14.127,   .0040596, 5.8304, .37755, 3645.9, .87823, 3411.8, .0163920},
{ 6.6948,  .0040603, 4.9361, .47961, 2719.0, .85249, 1885.8, .0197130},
{ 3.0619,  .0040511, 3.5803, .59082, 2346.1, .83713, 1222.0, .0200720},
{10.811,   .0033008, 1.3767, .76512, 2003.7, .82269, 1110.6, .0249580},
{ 2.7101,  .0040961, 1.2289, .98598, 1232.4, .79066, 155.42, .0472940},
{  .52345, .0040244, 1.4038, .8551,  1461.4, .79677, 503.34, .0367890},
{  .4616,  .0040203, 1.3014, .87043, 1473.5, .79687, 443.09, .0363010},
{  .97814, .0040374, 2.0127, .7225,  1890.8, .81747, 930.7, .02769},
{ 3.2086,  .004051, 3.6658,  .53618, 3091.2, .85602, 1508.1, .0154010},
{ 2.0035,  .0040431, 7.4882, .3561,  4464.3, .88836, 3966.5, .0128390},
{15.43,    .0039432, 1.1237, .70703, 4595.7, .88437, 1576.5, .0088534},
{ 3.1512,  .0040524, 4.0996, .5425,  3246.3, .85772, 1691.8, .0150580},
{ 7.1896,  .0040588, 8.6927, .35842, 4760.6, .88833, 2888.3, .0110290}, //Pb
{ 9.3209,  .004054, 11.543,  .32027, 4866.2, .89124, 3213.4, .0119350},
{29.242,   .0036195, .16864, 1.1226, 5688.0, .89812, 1033.3, .0071303},
{ 1.8522,  .0039973, 3.1556, .65096, 3755.0, .86383, 1602.0, .0120420},
{ 3.222,   .0040041, 5.9024, .52678, 4040.2, .86804, 1658.4, .0117470},
{ 9.3412,  .0039661, 7.921,  .42977, 5180.9, .88773, 2173.2, .0092007},
{36.183,   .0036003, .58341, .86747, 6990.2, .91082, 1417.1, .0062187},
{ 5.9284,  .0039695, 6.4082, .52122, 4619.5, .88083, 2323.5, .0116270},
{ 5.2454,  .0039744, 6.7969, .48542, 4586.3, .87794, 2481.5, .0112820},
{33.702,   .0036901, .47257, .89235, 5295.7, .8893, 2053.3, .0091908},
{2.7589,  .0039806, 3.2092, .66122,  2505.4, .82863, 2065.1, .0228160} //U
};

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

const G4double G4AtimaEnergyLossModel::atima_vfermi[92] = {
1.0309, 
0.15976,
0.59782,
1.0781, 
1.0486, 
1.00,   
1.058,  
0.93942,
0.74562,
0.3424, 
0.45259,
0.71074,
0.90519,
0.97411,
0.97184,
0.89852,
0.70827,
0.39816,
0.36552,
0.62712,
0.81707,
0.9943, 
1.1423, 
1.2381, 
1.1222, 
0.92705,
1.0047, 
1.2,    
1.0661, 
0.97411,
0.84912,
0.95,   
1.0903, 
1.0429, 
0.49715,
0.37755,
0.35211,
0.57801,
0.77773,
1.0207, 
1.029,  
1.2542, 
1.122,  
1.1241, 
1.0882, 
1.2709, 
1.2542, 
0.90094,
0.74093,
0.86054,
0.93155,
1.0047, 
0.55379,
0.43289,
0.32636,
0.5131, 
0.6950, 
0.72591,
0.71202,
0.67413,
0.71418,
0.71453,
0.5911, 
0.70263,
0.68049,
0.68203,
0.68121,
0.68532,
0.68715,
0.61884,
0.71801,
0.83048,
1.1222, 
1.2381, 
1.045,  
1.0733, 
1.0953, 
1.2381, 
1.2879, 
0.78654,
0.66401,
0.84912,
0.88433,
0.80746,
0.43357,
0.41923,
0.43638,
0.51464,
0.73087,
0.81065,
1.9578, 
1.0257
};

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

const G4double G4AtimaEnergyLossModel::atima_lambda_screening[92]= {
 1.00,
 1.00,
 1.10,
 1.06,
 1.01,
 1.03,
 1.04,
 0.99,
 0.95,
 0.90,
 0.82,
 0.81,
 0.83,
 0.88,
 1.00,
 0.95,
 0.97,
 0.99,
 0.98,
 0.97,
 0.98,
 0.97,
 0.96,
 0.93,
 0.91,
 0.9, 
 0.88,
 0.9, 
 0.9, 
 0.9, 
 0.9, 
 0.85,
 0.9, 
 0.9, 
 0.91,
 0.92,
 0.9, 
 0.9, 
 0.9, 
 0.9, 
 0.9, 
 0.88,
 0.9, 
 0.88,
 0.88,
 0.9, 
 0.9, 
 0.88,
 0.9, 
 0.9, 
 0.9, 
 0.9, 
 0.96,
 1.2, 
 0.9, 
 0.88,
 0.88,
 0.85,
 0.90,
 0.90,
 0.92,
 0.95,
 0.99,
 1.03,
 1.05,
 1.07,
 1.08,
 1.10,
 1.08,
 1.08,
 1.08,
 1.08,
 1.09,
 1.09,
 1.10,
 1.11,
 1.12,
 1.13,
 1.14,
 1.15,
 1.17,
 1.20,
 1.18,
 1.17,
 1.17,
 1.16,
 1.16,
 1.16,
 1.16,
 1.16,
 1.16,
 1.16
};

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

const G4double G4AtimaEnergyLossModel::element_atomic_weights[110]={
 0.0,
 1.00794, //H
 4.0026, //He
 6.941, //Li
 9.01218, //Be
 10.811, //B
 12.0107, //C
 14.0067, //N
 15.9994, //O
 18.9984, //F
 20.1797, //Ne
 22.9898, //Na
 24.305, //Mg
 26.9815, //Al
 28.0855, //Si
 30.9738, //P
 32.065, //S
 35.453, //Cl
 39.948, //Ar
 39.0983, //K
 40.078, //Ca
 44.9559, //Sc
 47.867, //Ti
 50.9415, //V
 51.9961, //Cr
 54.938, //Mn
 55.845, //Fe
 58.9332, //Co
 58.6934, //Ni
 63.546, //Cu
 65.409, //Zn
 69.723, //Ga
 72.64, //Ge
 74.9216, //As
 78.96, //Se
 79.904, //Br
 83.798, //Kr
 85.4678, //Rb
 87.62, //Sr
 88.9059, //Y
 91.224, //Zr
 92.9064, //Nb
 95.94, //Mo
 97.9072, //Tc
 101.07, //Ru
 102.906, //Rh
 106.42, //Pd
 107.868, //Ag
 112.411, //Cd
 114.818, //In
 118.71, //Sn
 121.76, //Sb
 127.6, //Te
 126.904, //I
 131.293, //Xe
 132.905, //Cs
 137.327, //Ba
 138.905, //La
 140.116, //Ce
 140.908, //Pr
 144.24, //Nd
 144.913, //Pm
 150.36, //Sm
 151.964, //Eu
 157.25, //Gd
 158.925, //Tb
 162.5, //Dy
 164.93, //Ho
 167.259, //Er
 168.934, //Tm
 173.04, //Yb
 174.967, //Lu
 178.49, //Hf
 180.948, //Ta
 183.84, //W
 186.207, //Re
 190.23, //Os
 192.217, //Ir
 195.078, //Pt
 196.967, //Au
 200.59, //Hg
 204.383, //Tl
 207.2, //Pb
 208.98, //Bi
 208.982, //Po
 209.987, //At
 222.018, //Rn
 223.02, //Fr
 226.025, //Ra
 227.028, //Ac
 232.038, //Th
 231.036, //Pa
 238.029, //U
 237.048, //Np
 244.064, //Pu
 243.061, //Am
 247.07, //Cm
 247.07, //Bk
 251.08, //Cf
 252.083, //Es
 257.095, //Fm
 258.098, //Md
 259.101, //No
 262.11, //Lr
 261.109, //Rf
 262.114, //Db
 266.122, //Sg
 264.125, //Bh
 269.134, //Hs
 268.139 //Mt
};

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

//LS coefficient for A=atomic weight
const G4double G4AtimaEnergyLossModel::ls_coefficients_a[110][200]=
{	{-0.0286005,-0.0269762,-0.0254384,-0.0239826,-0.0226044,-0.0212998,-0.0200648,-0.0188958,-0.0177891,-0.0167414,-0.0157495,-0.0148105,-0.0139213,-0.0130793,-0.0122818,-0.0115265,-0.0108109,-0.0101329,-0.00949027,-0.00888115,-0.0083036,-0.00775583,-0.00723615,-0.00674295,-0.0062747,-0.00582996,-0.00540738,-0.00500564,-0.00462353,-0.00425989,-0.00391361,-0.00358365,-0.00326903,-0.0029688,-0.00268208,-0.00240803,-0.00214586,-0.00189481,-0.00165416,-0.00142324,-0.0012014,-0.000988038,-0.000782569,-0.000584449,-0.000393155,-0.000208196,-2.91012e-05,0.000144572,0.000313247,0.000477324,0.000637183,0.000793185,0.000945673,0.00109497,0.0012414,0.00138523,0.00152676,0.00166625,0.00180395,0.00194009,0.0020749,0.0022086,0.00234139,0.00247346,0.00260498,0.00273612,0.00286706,0.00299792,0.00312886,0.00325999,0.00339145,0.00352333,0.00365574,0.00378876,0.00392249,0.00405698,0.00419229,0.00432849,0.0044656,0.00460367,0.0047427,0.00488271,0.00502369,0.00516564,0.00530853,0.00545232,0.00559697,0.00574243,0.00588862,0.00603547,0.00618289,0.00633076,0.00647899,0.00662745,0.006776,0.00692451,0.00707282,0.00722077,0.00736819,0.00751491,0.00766074,0.0078055,0.007949,0.00809104,0.00823143,0.00836998,0.00850649,0.00864077,0.00877264,0.0089019,0.0090284,0.00915196,0.00927243,0.00938966,0.00950352,0.00961389,0.00972066,0.00982374,0.00992306,0.0100185,0.0101102,0.0101979,0.0102817,0.0103616,0.0104376,0.0105097,0.010578,0.0106425,0.0107033,0.0107605,0.0108141,0.0108643,0.0109111,0.0109547,0.0109951,0.0110324,0.0110669,0.0110984,0.0111273,0.0111535,0.0111772,0.0111984,0.0112173,0.0112339,0.0112483,0.0112605,0.0112705,0.0112785,0.0112844,0.0112882,0.0112899,0.0112895,0.0112869,0.0112821,0.0112749,0.0112652,0.011253,0.011238,0.01122,0.0111988,0.011174,0.0111455,0.0111127,0.0110753,0.0110328,0.0109847,0.0109303,0.0108689,0.0107998,0.010722,0.0106346,0.0105364,0.0104261,0.0103024,0.0101636,0.0100079,0.00983338,0.00963767,0.00941827,0.0091723,0.0088966,0.00858758,0.00824121,0.00785299,0.00741787,0.0069302,0.00638364,0.00577111,0.00508464,0.00431533,0.00345395,0.00248834,0.00140654,0.000194686,-0.00116268,-0.00268283,-0.00438502,-0.00629005,-0.00842259,-0.0108087},
	{-0.108683,-0.102922,-0.097437,-0.0922162,-0.0872489,-0.0825243,-0.0780321,-0.073762,-0.0697043,-0.0658493,-0.0621879,-0.0587109,-0.0554098,-0.0522762,-0.049302,-0.0464796,-0.0438014,-0.0412604,-0.0388496,-0.0365626,-0.0343929,-0.0323346,-0.0303819,-0.0285294,-0.0267717,-0.0251038,-0.023521,-0.0220186,-0.0205924,-0.0192382,-0.017952,-0.0167301,-0.015569,-0.0144651,-0.0134154,-0.0124167,-0.011466,-0.0105607,-0.00969814,-0.00887572,-0.00809112,-0.00734207,-0.00662644,-0.00594219,-0.00528738,-0.00466019,-0.00405889,-0.00348181,-0.00292739,-0.00239414,-0.00188067,-0.00138562,-0.000907732,-0.000445801,1.31405e-06,0.000434696,0.000855371,0.00126431,0.00166244,0.00205062,0.0024297,0.00280044,0.00316359,0.00351985,0.00386987,0.00421428,0.00455367,0.00488858,0.00521953,0.00554701,0.00587147,0.00619333,0.00651298,0.00683078,0.00714706,0.00746212,0.00777624,0.00808966,0.00840259,0.00871523,0.00902774,0.00934025,0.00965286,0.00996565,0.0102787,0.010592,0.0109055,0.0112192,0.0115331,0.0118471,0.012161,0.0124748,0.0127882,0.013101,0.0134132,0.0137243,0.0140342,0.0143426,0.0146492,0.0149537,0.0152557,0.015555,0.0158511,0.0161438,0.0164326,0.0167172,0.0169973,0.0172725,0.0175424,0.0178067,0.0180651,0.0183173,0.0185629,0.0188017,0.0190335,0.019258,0.019475,0.0196843,0.0198859,0.0200796,0.0202654,0.0204431,0.0206127,0.0207744,0.020928,0.0210737,0.0212116,0.0213417,0.0214643,0.0215794,0.0216873,0.021788,0.0218819,0.0219691,0.0220498,0.0221242,0.0221925,0.022255,0.0223118,0.0223631,0.0224091,0.0224499,0.0224857,0.0225167,0.0225428,0.0225643,0.0225811,0.0225932,0.0226007,0.0226036,0.0226016,0.0225948,0.0225828,0.0225656,0.0225427,0.022514,0.0224789,0.022437,0.0223877,0.0223304,0.0222644,0.0221887,0.0221025,0.0220046,0.0218937,0.0217684,0.0216273,0.0214683,0.0212895,0.0210887,0.0208632,0.0206101,0.0203263,0.020008,0.0196512,0.0192513,0.0188031,0.018301,0.0177384,0.0171081,0.0164021,0.0156115,0.0147258,0.0137341,0.0126235,0.0113801,0.00998803,0.00842986,0.00668603,0.00473508,0.00255242,0.000111379,-0.00261781,-0.0056681,-0.00907593,-0.0128834,-0.0171323,-0.0218726,-0.0271577,-0.0330461},
	{-0.223884,-0.213009,-0.202568,-0.192553,-0.182953,-0.173755,-0.16495,-0.156525,-0.148469,-0.140769,-0.133415,-0.126394,-0.119694,-0.113303,-0.10721,-0.101404,-0.0958714,-0.0906027,-0.0855866,-0.0808123,-0.0762693,-0.0719475,-0.0678369,-0.063928,-0.0602115,-0.0566782,-0.0533196,-0.0501272,-0.0470929,-0.0442089,-0.0414678,-0.0388623,-0.0363856,-0.034031,-0.0317921,-0.029663,-0.0276377,-0.0257108,-0.0238769,-0.0221309,-0.0204681,-0.0188837,-0.0173734,-0.0159331,-0.0145586,-0.0132462,-0.0119923,-0.0107934,-0.00964627,-0.00854773,-0.00749484,-0.00648478,-0.00551487,-0.00458256,-0.00368542,-0.00282117,-0.00198762,-0.0011827,-0.000404448,0.000348997,0.0010794,0.00178843,0.00247768,0.00314864,0.00380272,0.00444127,0.00506554,0.00567674,0.00627597,0.00686428,0.00744268,0.00801208,0.00857334,0.00912727,0.00967462,0.0102161,0.0107522,0.0112837,0.011811,0.0123347,0.012855,0.0133725,0.0138873,0.0143999,0.0149103,0.0154188,0.0159254,0.0164303,0.0169335,0.017435,0.0179346,0.0184323,0.018928,0.0194213,0.0199121,0.0204002,0.0208851,0.0213666,0.0218443,0.0223178,0.0227866,0.0232502,0.0237084,0.0241604,0.024606,0.0250444,0.0254754,0.0258983,0.0263127,0.0267181,0.027114,0.0275001,0.0278758,0.0282409,0.028595,0.0289377,0.0292689,0.0295882,0.0298954,0.0301905,0.0304734,0.0307439,0.0310021,0.031248,0.0314816,0.0317031,0.0319127,0.0321105,0.0322967,0.0324716,0.0326355,0.0327885,0.0329311,0.0330635,0.0331861,0.0332991,0.033403,0.0334979,0.0335842,0.0336623,0.0337323,0.0337945,0.0338491,0.0338964,0.0339365,0.0339694,0.0339954,0.0340144,0.0340265,0.0340315,0.0340294,0.0340199,0.0340028,0.0339778,0.0339445,0.0339023,0.0338507,0.033789,0.0337163,0.0336318,0.0335342,0.0334224,0.0332949,0.03315,0.032986,0.0328007,0.0325918,0.0323567,0.0320922,0.0317951,0.0314616,0.0310874,0.0306677,0.0301973,0.02967,0.0290791,0.0284172,0.0276758,0.0268457,0.025916,0.0248751,0.0237098,0.0224056,0.0209459,0.0193125,0.0174852,0.0154416,0.0131562,0.0106012,0.00774588,0.00455597,0.000993742,-0.00298393,-0.00742085,-0.0123683,-0.0178814,-0.0240203,-0.030853,-0.0384474,-0.0468817},
	{-0.356328,-0.340686,-0.325539,-0.310886,-0.296723,-0.283047,-0.269853,-0.257134,-0.244884,-0.233095,-0.221759,-0.210866,-0.200406,-0.19037,-0.180747,-0.171526,-0.162694,-0.154242,-0.146157,-0.138426,-0.131039,-0.123984,-0.117247,-0.110818,-0.104685,-0.0988357,-0.0932592,-0.0879442,-0.0828796,-0.0780547,-0.073459,-0.0690823,-0.0649147,-0.0609465,-0.0571684,-0.0535713,-0.0501465,-0.0468855,-0.0437803,-0.040823,-0.0380061,-0.0353223,-0.0327648,-0.0303268,-0.0280019,-0.0257842,-0.0236676,-0.0216467,-0.0197161,-0.0178706,-0.0161054,-0.0144159,-0.0127976,-0.0112463,-0.00975795,-0.00832875,-0.00695503,-0.00563334,-0.00436036,-0.00313296,-0.00194816,-0.000803139,0.000304789,0.00137817,0.00241941,0.00343079,0.00441446,0.00537247,0.00630674,0.00721908,0.0081112,0.00898469,0.00984106,0.0106817,0.011508,0.012321,0.013122,0.013912,0.0146919,0.0154626,0.0162249,0.0169795,0.017727,0.018468,0.019203,0.0199323,0.0206563,0.0213752,0.0220892,0.0227985,0.023503,0.0242028,0.0248977,0.0255876,0.0262723,0.0269516,0.027625,0.0282923,0.028953,0.0296067,0.0302529,0.030891,0.0315205,0.0321408,0.0327514,0.0333515,0.0339407,0.0345182,0.0350836,0.0356361,0.0361754,0.0367007,0.0372117,0.0377078,0.0381886,0.0386538,0.039103,0.0395359,0.0399523,0.0403521,0.0407351,0.0411013,0.0414507,0.0417834,0.0420995,0.0423992,0.0426827,0.0429502,0.0432021,0.0434387,0.0436603,0.0438675,0.0440605,0.0442399,0.0444061,0.0445595,0.0447005,0.0448297,0.0449474,0.045054,0.0451499,0.0452355,0.0453111,0.045377,0.0454334,0.0454806,0.0455186,0.0455475,0.0455675,0.0455784,0.0455801,0.0455724,0.0455552,0.0455279,0.0454901,0.0454413,0.0453806,0.0453074,0.0452206,0.0451191,0.0450015,0.0448663,0.0447118,0.0445361,0.0443368,0.0441115,0.0438573,0.0435709,0.0432487,0.0428867,0.0424802,0.042024,0.0415125,0.040939,0.0402963,0.0395763,0.0387698,0.0378668,0.0368557,0.0357238,0.034457,0.0330394,0.0314533,0.0296791,0.027695,0.0254765,0.0229968,0.0202258,0.0171305,0.0136742,0.00981578,0.00551186,0.000713059,-0.00463419,-0.0105884,-0.0172145,-0.0245801,-0.0327606,-0.0418365,-0.0518915},
	{-0.492346,-0.472912,-0.453947,-0.435458,-0.417451,-0.399931,-0.382902,-0.366367,-0.350327,-0.334782,-0.319731,-0.305173,-0.291103,-0.277517,-0.264411,-0.251777,-0.239609,-0.227898,-0.216636,-0.205814,-0.195421,-0.185447,-0.175882,-0.166714,-0.157932,-0.149525,-0.141479,-0.133784,-0.126427,-0.119396,-0.11268,-0.106266,-0.100142,-0.0942985,-0.0887221,-0.0834023,-0.078328,-0.0734885,-0.0688733,-0.0644723,-0.0602755,-0.0562732,-0.0524563,-0.0488157,-0.0453427,-0.0420288,-0.0388661,-0.0358467,-0.0329632,-0.0302082,-0.027575,-0.025057,-0.0226476,-0.020341,-0.0181312,-0.0160128,-0.0139804,-0.0120289,-0.0101535,-0.00834964,-0.00661289,-0.00493909,-0.00332425,-0.00176459,-0.00025654,0.00120332,0.00261821,0.00399118,0.00532514,0.00662282,0.00788678,0.00911946,0.0103232,0.0115,0.012652,0.013781,0.0148888,0.015977,0.0170471,0.0181005,0.0191384,0.0201619,0.0211722,0.0221702,0.0231566,0.0241321,0.0250975,0.0260531,0.0269994,0.0279367,0.0288652,0.029785,0.0306962,0.0315988,0.0324925,0.0333773,0.0342527,0.0351185,0.0359742,0.0368194,0.0376535,0.038476,0.0392863,0.0400837,0.0408675,0.0416372,0.0423919,0.043131,0.0438538,0.0445596,0.0452478,0.0459178,0.046569,0.0472008,0.0478128,0.0484045,0.0489756,0.0495258,0.0500547,0.0505623,0.0510485,0.0515131,0.0519564,0.0523783,0.0527791,0.053159,0.0535183,0.0538574,0.0541767,0.0544766,0.0547576,0.0550203,0.0552651,0.0554928,0.0557037,0.0558986,0.0560779,0.0562423,0.0563924,0.0565286,0.0566514,0.0567613,0.0568588,0.0569443,0.057018,0.0570802,0.0571313,0.0571712,0.0572001,0.0572181,0.0572249,0.0572204,0.0572043,0.0571763,0.0571358,0.0570822,0.0570146,0.0569321,0.0568337,0.0567179,0.0565834,0.0564283,0.0562508,0.0560484,0.0558186,0.0555586,0.055265,0.0549341,0.0545617,0.0541431,0.053673,0.0531454,0.0525537,0.0518904,0.0511472,0.0503146,0.0493824,0.0483386,0.0471701,0.0458626,0.0443996,0.042763,0.0409327,0.0388863,0.0365988,0.0340425,0.0311868,0.0279978,0.0244381,0.0204657,0.0160361,0.011099,0.00559988,-0.000520862,-0.00732909,-0.0148937,-0.023291,-0.0326025,-0.0429131,-0.0543173},
	{-0.623826,-0.601642,-0.579854,-0.558476,-0.537521,-0.517,-0.496926,-0.477307,-0.458152,-0.439471,-0.421268,-0.403551,-0.386322,-0.369586,-0.353343,-0.337595,-0.32234,-0.307577,-0.293302,-0.279513,-0.266202,-0.253366,-0.240995,-0.229084,-0.217623,-0.206603,-0.196014,-0.185847,-0.176089,-0.166731,-0.157761,-0.149166,-0.140936,-0.133058,-0.125521,-0.118311,-0.111418,-0.104829,-0.0985323,-0.0925163,-0.0867694,-0.0812803,-0.0760379,-0.0710314,-0.06625,-0.0616836,-0.0573221,-0.0531557,-0.0491749,-0.0453706,-0.0417341,-0.0382566,-0.0349301,-0.0317466,-0.0286985,-0.0257784,-0.0229794,-0.0202946,-0.0177178,-0.0152425,-0.0128631,-0.0105738,-0.00836925,-0.0062443,-0.00419408,-0.00221393,-0.000299416,0.00155365,0.00334927,0.00509121,0.00678305,0.00842817,0.0100297,0.0115908,0.0131141,0.0146023,0.016058,0.0174833,0.0188804,0.0202514,0.021598,0.0229219,0.0242247,0.0255076,0.026772,0.028019,0.0292495,0.0304643,0.0316641,0.0328496,0.0340211,0.035179,0.0363234,0.0374546,0.0385725,0.039677,0.0407678,0.0418448,0.0429075,0.0439555,0.0449882,0.0460051,0.0470056,0.047989,0.0489545,0.0499015,0.0508292,0.0517368,0.0526236,0.0534888,0.0543318,0.0551518,0.0559483,0.0567206,0.0574682,0.0581907,0.0588876,0.0595587,0.0602036,0.0608223,0.0614146,0.0619806,0.0625203,0.063034,0.0635218,0.0639842,0.0644214,0.064834,0.0652225,0.0655874,0.0659295,0.0662492,0.0665474,0.0668247,0.0670818,0.0673194,0.0675384,0.0677393,0.067923,0.06809,0.068241,0.0683765,0.0684972,0.0686036,0.068696,0.0687749,0.0688405,0.0688931,0.0689328,0.0689597,0.0689738,0.0689748,0.0689625,0.0689365,0.0688963,0.0688412,0.0687703,0.0686828,0.0685773,0.0684525,0.0683067,0.068138,0.0679443,0.0677232,0.0674717,0.0671868,0.0668648,0.0665016,0.0660926,0.0656327,0.0651161,0.0645363,0.063886,0.0631569,0.06234,0.0614251,0.0604005,0.0592536,0.05797,0.0565339,0.0549276,0.0531311,0.0511227,0.048878,0.0463697,0.043568,0.0404396,0.0369479,0.0330526,0.0287081,0.0238669,0.0184751,0.0124742,0.0058007,-0.00161559,-0.00984726,-0.018975,-0.0290842,-0.040263,-0.0526102},
	{-0.747011,-0.722922,-0.699148,-0.675705,-0.652609,-0.629874,-0.607515,-0.585548,-0.563985,-0.542839,-0.522123,-0.501848,-0.482024,-0.462662,-0.443768,-0.425351,-0.407415,-0.389966,-0.373007,-0.35654,-0.340566,-0.325085,-0.310095,-0.295594,-0.281577,-0.26804,-0.254978,-0.242383,-0.230248,-0.218565,-0.207324,-0.196516,-0.186132,-0.176159,-0.166588,-0.157406,-0.148603,-0.140166,-0.132084,-0.124343,-0.116933,-0.109841,-0.103055,-0.0965628,-0.0903531,-0.0844141,-0.0787344,-0.0733026,-0.0681078,-0.0631393,-0.0583866,-0.0538395,-0.0494881,-0.0453228,-0.0413344,-0.0375139,-0.0338526,-0.0303421,-0.0269745,-0.0237419,-0.020637,-0.0176526,-0.0147818,-0.0120183,-0.00935561,-0.00678788,-0.00430938,-0.0019147,0.000401332,0.00264362,0.00481683,0.00692537,0.0089734,0.0109649,0.0129035,0.0147928,0.0166361,0.0184363,0.0201965,0.0219191,0.0236068,0.0252618,0.0268861,0.0284817,0.0300503,0.0315934,0.0331125,0.0346087,0.0360832,0.0375367,0.03897,0.0403837,0.0417783,0.043154,0.044511,0.0458494,0.0471691,0.0484699,0.0497515,0.0510134,0.0522554,0.0534767,0.0546768,0.055855,0.0570105,0.0581426,0.0592506,0.0603336,0.0613909,0.0624216,0.0634251,0.0644005,0.0653473,0.0662648,0.0671524,0.0680097,0.0688362,0.0696317,0.0703959,0.0711287,0.07183,0.0724999,0.0731385,0.0737461,0.074323,0.0748697,0.0753865,0.0758742,0.0763332,0.0767645,0.0771685,0.0775463,0.0778985,0.0782261,0.0785299,0.0788108,0.0790696,0.0793072,0.0795244,0.079722,0.0799009,0.0800616,0.0802049,0.0803313,0.0804414,0.0805358,0.0806146,0.0806782,0.0807269,0.0807607,0.0807795,0.0807832,0.0807715,0.0807441,0.0807003,0.0806393,0.0805602,0.080462,0.0803432,0.0802023,0.0800375,0.0798465,0.079627,0.0793762,0.0790909,0.0787674,0.0784018,0.0779893,0.0775248,0.0770025,0.0764158,0.0757574,0.0750189,0.0741912,0.0732641,0.0722257,0.0710633,0.0697624,0.0683071,0.0666793,0.0648593,0.0628248,0.0605513,0.0580115,0.0551751,0.0520088,0.0484756,0.044535,0.0401414,0.0352468,0.0297969,0.0237333,0.0169922,0.00950342,0.00119386,-0.00801709,-0.0182149,-0.0294879,-0.0419344,-0.0556523},
	{-0.860779,-0.835389,-0.810244,-0.785358,-0.760745,-0.736422,-0.712404,-0.688706,-0.665343,-0.642332,-0.619687,-0.597422,-0.575552,-0.554091,-0.53305,-0.512443,-0.492279,-0.472569,-0.453323,-0.434546,-0.416247,-0.39843,-0.381101,-0.36426,-0.347911,-0.332054,-0.316687,-0.301809,-0.287417,-0.273507,-0.260073,-0.247109,-0.234608,-0.222562,-0.210963,-0.199802,-0.189067,-0.17875,-0.168839,-0.159323,-0.150191,-0.14143,-0.133028,-0.124975,-0.117257,-0.109862,-0.102778,-0.0959937,-0.0894965,-0.0832748,-0.0773168,-0.0716113,-0.066147,-0.060913,-0.0558986,-0.0510935,-0.0464875,-0.0420708,-0.0378339,-0.0337677,-0.0298632,-0.026112,-0.0225058,-0.0190367,-0.015697,-0.0124795,-0.00937729,-0.00638357,-0.00349201,-0.000696547,0.00200862,0.00462898,0.00716976,0.00963591,0.0120321,0.0143627,0.016632,0.0188438,0.0210017,0.0231092,0.0251695,0.0271854,0.0291598,0.0310951,0.0329936,0.0348573,0.0366882,0.0384877,0.0402575,0.0419988,0.0437126,0.0453998,0.0470612,0.0486972,0.0503084,0.0518949,0.0534567,0.054994,0.0565064,0.0579936,0.0594554,0.0608911,0.0623003,0.0636821,0.0650361,0.0663613,0.067657,0.0689224,0.0701567,0.0713591,0.0725288,0.0736651,0.0747673,0.0758347,0.0768669,0.0778632,0.0788233,0.0797469,0.0806338,0.0814839,0.0822972,0.0830738,0.0838139,0.0845179,0.0851861,0.0858191,0.0864175,0.086982,0.0875134,0.0880124,0.0884799,0.088917,0.0893246,0.0897036,0.0900551,0.0903801,0.0906796,0.0909546,0.0912061,0.0914351,0.0916423,0.0918287,0.0919951,0.0921421,0.0922703,0.0923804,0.0924727,0.0925477,0.0926055,0.0926463,0.0926701,0.0926766,0.0926657,0.0926368,0.0925893,0.0925225,0.0924352,0.0923262,0.092194,0.0920369,0.0918528,0.0916394,0.0913938,0.0911131,0.0907936,0.0904313,0.0900216,0.0895595,0.0890391,0.0884539,0.0877967,0.0870591,0.086232,0.0853052,0.0842671,0.0831048,0.0818041,0.0803489,0.0787214,0.0769018,0.074868,0.0725956,0.0700575,0.0672234,0.0640603,0.0605314,0.0565962,0.0522097,0.047324,0.0418852,0.0358351,0.0291104,0.0216416,0.0133558,0.00417311,-0.00599137,-0.0172259,-0.0296272,-0.0432931,-0.0583229},
	{-0.965337,-0.939056,-0.912961,-0.887067,-0.861386,-0.835932,-0.81072,-0.785764,-0.761079,-0.736681,-0.712586,-0.688808,-0.665364,-0.642268,-0.619536,-0.597182,-0.57522,-0.553665,-0.532528,-0.511822,-0.491557,-0.471744,-0.452392,-0.433509,-0.415101,-0.397174,-0.379731,-0.362777,-0.346312,-0.330337,-0.314851,-0.299853,-0.28534,-0.271306,-0.257748,-0.244658,-0.232031,-0.219857,-0.208129,-0.196837,-0.185971,-0.175521,-0.165476,-0.155825,-0.146556,-0.137657,-0.129117,-0.120923,-0.113063,-0.105524,-0.098296,-0.0913652,-0.08472,-0.0783485,-0.0722392,-0.0663806,-0.0607614,-0.0553706,-0.0501976,-0.0452318,-0.0404633,-0.0358821,-0.0314787,-0.027244,-0.023169,-0.0192453,-0.0154645,-0.0118188,-0.00830071,-0.00490286,-0.00161839,0.00155931,0.00463654,0.00761927,0.0105132,0.0133236,0.0160556,0.0187141,0.0213034,0.0238278,0.0262912,0.0286973,0.0310494,0.0333508,0.0356043,0.0378124,0.0399777,0.0421022,0.0441878,0.0462362,0.0482489,0.0502271,0.0521718,0.0540839,0.055964,0.0578126,0.0596299,0.0614161,0.0631712,0.0648949,0.0665871,0.0682472,0.0698748,0.0714693,0.07303,0.0745561,0.076047,0.0775018,0.0789196,0.0802999,0.0816416,0.0829441,0.0842068,0.0854289,0.0866099,0.0877494,0.0888469,0.0899023,0.0909152,0.0918857,0.0928139,0.0936998,0.0945439,0.0953464,0.096108,0.0968293,0.0975109,0.0981538,0.0987587,0.0993267,0.0998588,0.100356,0.10082,0.101251,0.10165,0.10202,0.10236,0.102673,0.102958,0.103218,0.103453,0.103665,0.103853,0.10402,0.104165,0.104289,0.104393,0.104478,0.104542,0.104588,0.104613,0.104619,0.104605,0.104571,0.104515,0.104437,0.104335,0.104209,0.104056,0.103874,0.103662,0.103415,0.103132,0.102808,0.10244,0.102023,0.101551,0.101019,0.10042,0.0997472,0.0989913,0.0981434,0.097193,0.0961282,0.094936,0.0936018,0.0921093,0.0904402,0.0885743,0.0864892,0.0841601,0.0815591,0.0786557,0.0754161,0.0718029,0.0677751,0.063287,0.0582897,0.052729,0.0465457,0.0396757,0.032049,0.0235919,0.0142237,0.00385915,-0.00759068,-0.0202227,-0.0341349,-0.0494269,-0.066204},
	{-1.06145,-1.03454,-1.00778,-0.981172,-0.95473,-0.928466,-0.902393,-0.876522,-0.850868,-0.825445,-0.800267,-0.775349,-0.750707,-0.726356,-0.702311,-0.678587,-0.655201,-0.632168,-0.609501,-0.587217,-0.565328,-0.543848,-0.52279,-0.502165,-0.481984,-0.462256,-0.442991,-0.424195,-0.405874,-0.388035,-0.37068,-0.353811,-0.337431,-0.32154,-0.306135,-0.291215,-0.276777,-0.262815,-0.249324,-0.236297,-0.223728,-0.211608,-0.199929,-0.188679,-0.177851,-0.167432,-0.157412,-0.147779,-0.138523,-0.12963,-0.121088,-0.112887,-0.105012,-0.0974527,-0.0901962,-0.0832306,-0.0765437,-0.070124,-0.0639596,-0.0580392,-0.0523517,-0.0468862,-0.0416321,-0.0365791,-0.0317171,-0.0270364,-0.0225278,-0.0181821,-0.0139907,-0.00994514,-0.00603749,-0.00226005,0.00139454,0.00493328,0.00836284,0.0116895,0.0149194,0.0180581,0.021111,0.0240831,0.0269792,0.0298036,0.0325605,0.0352536,0.0378866,0.0404626,0.0429845,0.0454551,0.0478766,0.0502514,0.0525812,0.0548676,0.0571121,0.0593158,0.0614796,0.0636043,0.0656904,0.0677382,0.0697479,0.0717194,0.0736526,0.0755472,0.0774028,0.0792189,0.0809948,0.08273,0.0844236,0.0860749,0.0876831,0.0892475,0.0907673,0.0922418,0.0936702,0.0950521,0.0963868,0.0976739,0.0989131,0.100104,0.101247,0.102341,0.103388,0.104386,0.105337,0.106241,0.107099,0.107911,0.108678,0.109402,0.110083,0.110722,0.111321,0.111881,0.112402,0.112887,0.113337,0.113753,0.114137,0.114489,0.114811,0.115104,0.11537,0.115609,0.115823,0.116012,0.116177,0.116319,0.116439,0.116536,0.116613,0.116667,0.116701,0.116712,0.116702,0.116669,0.116614,0.116533,0.116428,0.116296,0.116134,0.115942,0.115717,0.115455,0.115154,0.114809,0.114417,0.113972,0.113469,0.112901,0.112262,0.111544,0.110737,0.109832,0.108817,0.107681,0.106409,0.104985,0.103393,0.101613,0.0996236,0.0974013,0.0949196,0.0921493,0.0890581,0.0856104,0.0817668,0.0774843,0.0727148,0.0674074,0.061505,0.0549462,0.0476642,0.0395863,0.0306361,0.0207306,0.00978198,-0.00230042,-0.0156157,-0.0302629,-0.0463418,-0.0639582},
	{-1.15003,-1.12268,-1.09543,-1.06831,-1.04132,-1.01446,-0.987758,-0.961215,-0.934845,-0.908659,-0.882671,-0.856894,-0.831343,-0.80603,-0.780972,-0.756183,-0.731678,-0.707474,-0.683585,-0.660027,-0.636815,-0.613965,-0.59149,-0.569405,-0.547724,-0.52646,-0.505623,-0.485225,-0.465276,-0.445786,-0.42676,-0.408208,-0.390132,-0.372539,-0.35543,-0.338807,-0.322671,-0.30702,-0.291853,-0.277167,-0.262957,-0.249217,-0.235942,-0.223125,-0.210757,-0.19883,-0.187334,-0.17626,-0.165596,-0.155333,-0.145458,-0.135959,-0.126826,-0.118046,-0.109606,-0.101495,-0.0937,-0.086209,-0.0790099,-0.0720906,-0.0654393,-0.0590443,-0.0528941,-0.0469775,-0.0412836,-0.0358017,-0.0305215,-0.0254328,-0.020526,-0.0157917,-0.0112208,-0.00680457,-0.00253474,0.00159671,0.00559744,0.00947472,0.0132355,0.0168863,0.0204334,0.0238827,0.0272397,0.0305095,0.033697,0.0368068,0.039843,0.0428094,0.0457097,0.048547,0.0513242,0.054044,0.0567087,0.0593204,0.0618807,0.0643913,0.0668534,0.069268,0.0716359,0.0739577,0.0762337,0.0784641,0.0806488,0.0827879,0.0848809,0.0869274,0.088927,0.0908789,0.0927827,0.0946375,0.0964427,0.0981974,0.0999011,0.101553,0.103152,0.104699,0.106191,0.10763,0.109015,0.110345,0.111621,0.112843,0.11401,0.115124,0.116184,0.117192,0.118147,0.119052,0.119907,0.120712,0.12147,0.122182,0.122848,0.123471,0.124051,0.124591,0.125091,0.125554,0.12598,0.126371,0.126729,0.127055,0.127349,0.127615,0.127852,0.128061,0.128244,0.128402,0.128534,0.128642,0.128726,0.128786,0.128822,0.128834,0.128821,0.128783,0.12872,0.128629,0.12851,0.128361,0.12818,0.127964,0.127711,0.127417,0.127079,0.126692,0.126253,0.125754,0.12519,0.124555,0.123839,0.123035,0.122133,0.12112,0.119986,0.118716,0.117295,0.115705,0.113928,0.111941,0.109723,0.107245,0.10448,0.101396,0.0979556,0.0941214,0.0898501,0.0850942,0.0798029,0.0739196,0.0673835,0.0601281,0.0520815,0.0431677,0.0333043,0.0224044,0.0103774,-0.00287441,-0.0174497,-0.0334482,-0.0509744,-0.0701288},
	{-1.23199,-1.2043,-1.1767,-1.14919,-1.12179,-1.0945,-1.06732,-1.04028,-1.01337,-0.986609,-0.960009,-0.93358,-0.907333,-0.881282,-0.855441,-0.829823,-0.804442,-0.779313,-0.754451,-0.729872,-0.705592,-0.681624,-0.657986,-0.634692,-0.611758,-0.589197,-0.567025,-0.545253,-0.523896,-0.502966,-0.482472,-0.462425,-0.442835,-0.423708,-0.405051,-0.38687,-0.369169,-0.35195,-0.335216,-0.318966,-0.3032,-0.287915,-0.273109,-0.258777,-0.244914,-0.231514,-0.218569,-0.206071,-0.194012,-0.182382,-0.171171,-0.160369,-0.149965,-0.139947,-0.130303,-0.121023,-0.112092,-0.103501,-0.0952348,-0.087283,-0.0796327,-0.072272,-0.0651887,-0.058371,-0.0518072,-0.045486,-0.0393962,-0.0335268,-0.0278672,-0.0224072,-0.0171368,-0.0120463,-0.00712654,-0.00236848,0.00223639,0.00669628,0.011019,0.015212,0.0192824,0.0232369,0.0270818,0.030823,0.0344662,0.0380166,0.041479,0.044858,0.0481577,0.0513818,0.054534,0.0576172,0.0606344,0.063588,0.0664802,0.069313,0.0720878,0.074806,0.0774688,0.0800769,0.082631,0.0851314,0.0875783,0.0899718,0.0923117,0.0945977,0.0968294,0.0990063,0.101128,0.103193,0.105202,0.107154,0.109047,0.110882,0.112658,0.114374,0.116029,0.117625,0.119159,0.120633,0.122045,0.123397,0.124689,0.125921,0.127094,0.128208,0.129265,0.130264,0.131209,0.132099,0.132936,0.133722,0.134458,0.135146,0.135787,0.136383,0.136935,0.137446,0.137917,0.138349,0.138745,0.139105,0.139432,0.139726,0.139988,0.140221,0.140425,0.1406,0.140748,0.14087,0.140965,0.141034,0.141077,0.141094,0.141084,0.141048,0.140983,0.140889,0.140765,0.140608,0.140417,0.140188,0.13992,0.139608,0.139249,0.138838,0.13837,0.13784,0.13724,0.136563,0.135802,0.134946,0.133985,0.132908,0.131701,0.130349,0.128837,0.127146,0.125257,0.123145,0.120788,0.118156,0.11522,0.111946,0.108296,0.10423,0.0997023,0.094664,0.0890617,0.0828366,0.0759251,0.0682584,0.0597628,0.0503591,0.039963,0.0284861,0.0158337,0.00190907,-0.0133882,-0.0301575,-0.0485032,-0.0685241},
	{-1.30813,-1.28018,-1.25231,-1.22451,-1.1968,-1.16917,-1.14164,-1.11421,-1.08689,-1.05969,-1.03262,-1.00569,-0.978905,-0.952283,-0.925832,-0.899565,-0.873494,-0.847635,-0.822,-0.796603,-0.771461,-0.746587,-0.721998,-0.697708,-0.673733,-0.650089,-0.626791,-0.603854,-0.581292,-0.559119,-0.537348,-0.515993,-0.495064,-0.474573,-0.454529,-0.434941,-0.415817,-0.397162,-0.378981,-0.361279,-0.344059,-0.32732,-0.311064,-0.295289,-0.279993,-0.265172,-0.250823,-0.236938,-0.223512,-0.210538,-0.198007,-0.18591,-0.174238,-0.16298,-0.152126,-0.141664,-0.131584,-0.121873,-0.11252,-0.103512,-0.0948375,-0.0864836,-0.0784384,-0.0706895,-0.0632249,-0.0560327,-0.0491012,-0.0424186,-0.0359738,-0.0297558,-0.0237536,-0.017957,-0.0123558,-0.00694027,-0.00170096,0.00337116,0.00828473,0.0130481,0.017669,0.0221552,0.0265136,0.0307509,0.0348736,0.0388876,0.0427984,0.0466111,0.0503306,0.0539612,0.057507,0.0609716,0.0643584,0.0676703,0.07091,0.0740797,0.0771814,0.0802168,0.0831872,0.0860939,0.0889376,0.0917191,0.0944386,0.0970964,0.0996925,0.102227,0.104699,0.107109,0.109455,0.111739,0.113958,0.116112,0.118202,0.120225,0.122182,0.124073,0.125896,0.127651,0.12934,0.13096,0.132513,0.133999,0.135418,0.136771,0.138059,0.139282,0.140441,0.141538,0.142574,0.14355,0.144468,0.145329,0.146136,0.146889,0.147592,0.148244,0.148849,0.149409,0.149924,0.150398,0.15083,0.151225,0.151582,0.151903,0.15219,0.152444,0.152667,0.152858,0.15302,0.153152,0.153255,0.15333,0.153376,0.153393,0.153381,0.15334,0.153268,0.153163,0.153025,0.152852,0.15264,0.152388,0.152092,0.151748,0.151352,0.150899,0.150384,0.1498,0.149139,0.148394,0.147556,0.146615,0.145558,0.144374,0.143047,0.141563,0.139902,0.138046,0.135973,0.133657,0.131072,0.128189,0.124974,0.12139,0.117398,0.112953,0.108008,0.102509,0.0963999,0.0896176,0.0820949,0.0737586,0.0645318,0.0543312,0.043069,0.0306541,0.0169886,0.00197411,-0.014489,-0.0325038,-0.0521696,-0.0735837},
	{-1.37915,-1.351,-1.32292,-1.29489,-1.26693,-1.23904,-1.21123,-1.18351,-1.15587,-1.12833,-1.10089,-1.07357,-1.04636,-1.01929,-0.99236,-0.965581,-0.938965,-0.912523,-0.88627,-0.860217,-0.834379,-0.808769,-0.783403,-0.758295,-0.73346,-0.708914,-0.684673,-0.660751,-0.637165,-0.613928,-0.591057,-0.568565,-0.546466,-0.524773,-0.503498,-0.482653,-0.462247,-0.44229,-0.422791,-0.403755,-0.38519,-0.367099,-0.349485,-0.332351,-0.315697,-0.299523,-0.283827,-0.268607,-0.253858,-0.239575,-0.225752,-0.212383,-0.199459,-0.186973,-0.174914,-0.163273,-0.15204,-0.141203,-0.130752,-0.120674,-0.110958,-0.101592,-0.0925639,-0.0838611,-0.0754715,-0.0673829,-0.0595832,-0.0520604,-0.0448027,-0.0377986,-0.0310366,-0.0245056,-0.0181949,-0.0120939,-0.00619263,-0.000481135,0.00504996,0.0104097,0.0156068,0.0206494,0.0255456,0.0303026,0.0349276,0.0394271,0.0438075,0.0480745,0.0522335,0.0562896,0.0602473,0.0641108,0.067884,0.0715703,0.0751728,0.0786941,0.0821368,0.0855027,0.0887936,0.092011,0.095156,0.0982295,0.101232,0.104164,0.107026,0.109817,0.112539,0.115189,0.117769,0.120277,0.122713,0.125078,0.127369,0.129586,0.13173,0.1338,0.135796,0.137717,0.139563,0.141335,0.143032,0.144655,0.146205,0.147682,0.149088,0.150422,0.151687,0.152884,0.154014,0.155078,0.156079,0.157019,0.157898,0.158719,0.159485,0.160196,0.160856,0.161466,0.162028,0.162544,0.163016,0.163447,0.163836,0.164188,0.164502,0.16478,0.165024,0.165234,0.165413,0.165559,0.165674,0.165759,0.165813,0.165836,0.165827,0.165787,0.165714,0.165607,0.165464,0.165284,0.165063,0.164798,0.164487,0.164126,0.163709,0.163233,0.16269,0.162074,0.161378,0.160593,0.15971,0.158718,0.157604,0.156356,0.154959,0.153395,0.151647,0.149693,0.14751,0.145074,0.142355,0.139323,0.135943,0.132178,0.127986,0.12332,0.118131,0.112365,0.105962,0.0988585,0.0909843,0.0822645,0.0726204,0.0619669,0.0502145,0.0372712,0.0230378,0.00741538,-0.0096953,-0.0283969,-0.0487867,-0.07096},
	{-1.44565,-1.41735,-1.38909,-1.36088,-1.33273,-1.30464,-1.2766,-1.24864,-1.22075,-1.19293,-1.1652,-1.13757,-1.11003,-1.0826,-1.05529,-1.0281,-1.00105,-0.974139,-0.947387,-0.920804,-0.894401,-0.868192,-0.84219,-0.816408,-0.790862,-0.765566,-0.740535,-0.715785,-0.691331,-0.667189,-0.643373,-0.6199,-0.596784,-0.57404,-0.55168,-0.52972,-0.508171,-0.487045,-0.466353,-0.446104,-0.426306,-0.406969,-0.388096,-0.369694,-0.351767,-0.334316,-0.317343,-0.300848,-0.284829,-0.269285,-0.254212,-0.239604,-0.225457,-0.211764,-0.198517,-0.185708,-0.173329,-0.161369,-0.149818,-0.138665,-0.127901,-0.117512,-0.107487,-0.0978144,-0.0884821,-0.0794777,-0.0707892,-0.0624042,-0.0543108,-0.046497,-0.038951,-0.0316613,-0.0246164,-0.0178053,-0.0112172,-0.00484177,0.00133118,0.00731137,0.0131082,0.0187305,0.024187,0.0294857,0.0346345,0.0396406,0.0445108,0.0492517,0.0538692,0.058369,0.0627562,0.0670356,0.0712114,0.0752877,0.079268,0.0831553,0.0869526,0.0906621,0.0942861,0.0978261,0.101284,0.10466,0.107956,0.111172,0.114309,0.117366,0.120345,0.123244,0.126064,0.128804,0.131465,0.134044,0.136543,0.138961,0.141297,0.143551,0.145724,0.147814,0.149822,0.151749,0.153594,0.155358,0.157041,0.158646,0.160171,0.16162,0.162993,0.164291,0.165516,0.16667,0.167756,0.168774,0.169727,0.170616,0.171446,0.172216,0.172931,0.173591,0.174199,0.174758,0.175269,0.175734,0.176155,0.176534,0.176874,0.177174,0.177437,0.177663,0.177855,0.178012,0.178135,0.178224,0.17828,0.178303,0.178291,0.178245,0.178163,0.178043,0.177885,0.177685,0.177441,0.177149,0.176807,0.17641,0.175952,0.175428,0.174833,0.174157,0.173394,0.172533,0.171566,0.170479,0.16926,0.167894,0.166365,0.164655,0.162743,0.160608,0.158224,0.155565,0.152598,0.149292,0.145609,0.141508,0.136945,0.13187,0.126231,0.11997,0.113024,0.105324,0.0967979,0.0873676,0.0769498,0.0654566,0.0527968,0.0388731,0.0235875,0.00683949,-0.0114689,-0.0314394,-0.0531654,-0.0767371},
	{-1.50815,-1.47972,-1.45132,-1.42297,-1.39466,-1.36639,-1.33818,-1.31002,-1.28192,-1.25389,-1.22593,-1.19804,-1.17023,-1.14251,-1.11489,-1.08737,-1.05996,-1.03268,-1.00552,-0.978506,-0.951642,-0.924941,-0.898415,-0.872077,-0.845941,-0.82002,-0.794329,-0.768882,-0.743694,-0.718781,-0.694158,-0.66984,-0.645844,-0.622184,-0.598875,-0.575932,-0.553369,-0.531199,-0.509436,-0.48809,-0.467173,-0.446695,-0.426665,-0.40709,-0.387977,-0.369331,-0.351156,-0.333455,-0.316229,-0.299479,-0.283204,-0.267402,-0.252068,-0.2372,-0.222791,-0.208835,-0.195325,-0.182253,-0.16961,-0.157386,-0.145572,-0.134156,-0.123128,-0.112477,-0.10219,-0.0922568,-0.0826641,-0.0734002,-0.0644529,-0.0558103,-0.0474602,-0.0393908,-0.0315903,-0.0240474,-0.0167507,-0.00968922,-0.00285237,0.0037702,0.0101885,0.016412,0.02245,0.0283112,0.0340041,0.0395365,0.0449159,0.0501495,0.0552438,0.0602049,0.0650387,0.0697504,0.0743448,0.0788264,0.0831992,0.0874667,0.0916321,0.0956983,0.0996676,0.103542,0.107324,0.111014,0.114613,0.118123,0.121544,0.124877,0.128121,0.131277,0.134345,0.137325,0.140216,0.143018,0.145731,0.148354,0.150888,0.153332,0.155686,0.15795,0.160125,0.162211,0.164207,0.166116,0.167937,0.169671,0.171321,0.172886,0.174369,0.175772,0.177095,0.178342,0.179514,0.180613,0.181642,0.182603,0.183498,0.18433,0.185101,0.185814,0.18647,0.187073,0.187625,0.188128,0.188583,0.188993,0.18936,0.189685,0.18997,0.190217,0.190425,0.190596,0.190732,0.190831,0.190895,0.190923,0.190914,0.190869,0.190786,0.190663,0.190498,0.19029,0.190035,0.189731,0.189372,0.188955,0.188475,0.187925,0.1873,0.18659,0.185788,0.184885,0.183868,0.182727,0.181446,0.180012,0.178407,0.176612,0.174607,0.172367,0.169867,0.167079,0.16397,0.160507,0.15665,0.152357,0.147583,0.142277,0.136383,0.129843,0.122591,0.114558,0.105669,0.0958441,0.0849987,0.0730435,0.0598863,0.0454288,0.0295728,0.0122193,-0.00673262,-0.0273793,-0.0498142,-0.0741205},
	{-1.56708,-1.53854,-1.51003,-1.48156,-1.45312,-1.42471,-1.39635,-1.36803,-1.33977,-1.31155,-1.28339,-1.2553,-1.22727,-1.19931,-1.17144,-1.14365,-1.11595,-1.08835,-1.06086,-1.03349,-1.00625,-0.979139,-0.952179,-0.92538,-0.898752,-0.872308,-0.846062,-0.820028,-0.794218,-0.76865,-0.743336,-0.718294,-0.693537,-0.669082,-0.644944,-0.621138,-0.597679,-0.574583,-0.551862,-0.529531,-0.507602,-0.486088,-0.464999,-0.444346,-0.424137,-0.40438,-0.385082,-0.366248,-0.347883,-0.329988,-0.312567,-0.295619,-0.279144,-0.263139,-0.247602,-0.232527,-0.21791,-0.203745,-0.190024,-0.176739,-0.163882,-0.151443,-0.139412,-0.127779,-0.116533,-0.105662,-0.0951547,-0.0849998,-0.0751852,-0.0656986,-0.0565283,-0.0476621,-0.0390882,-0.0307948,-0.0227703,-0.0150034,-0.00748304,-0.000198342,0.00686113,0.0137055,0.0203445,0.0267875,0.0330434,0.0391208,0.0450277,0.0507718,0.0563601,0.0617996,0.0670963,0.0722561,0.0772844,0.0821861,0.0869656,0.0916269,0.0961737,0.100609,0.104936,0.109157,0.113273,0.117287,0.121201,0.125014,0.128729,0.132345,0.135863,0.139284,0.142608,0.145833,0.148962,0.151992,0.154925,0.15776,0.160496,0.163135,0.165675,0.168118,0.170463,0.172711,0.174863,0.176919,0.17888,0.180748,0.182523,0.184208,0.185804,0.187312,0.188735,0.190076,0.191335,0.192517,0.193622,0.194654,0.195615,0.196508,0.197336,0.198101,0.198805,0.199452,0.200043,0.200582,0.201069,0.201508,0.2019,0.202247,0.202551,0.202813,0.203034,0.203215,0.203356,0.203459,0.203524,0.203549,0.203535,0.203481,0.203385,0.203246,0.203062,0.20283,0.202547,0.20221,0.201814,0.201354,0.200824,0.200219,0.19953,0.19875,0.197869,0.196877,0.195761,0.194508,0.193104,0.191532,0.189773,0.187808,0.185612,0.183162,0.180429,0.177382,0.173986,0.170206,0.165999,0.16132,0.15612,0.150345,0.143937,0.136832,0.128963,0.120254,0.110629,0.100004,0.0882906,0.0753983,0.0612302,0.0456888,0.0286744,0.0100895,-0.0101654,-0.0321827,-0.0560503,-0.0818434},
	{-1.62282,-1.59419,-1.56558,-1.537,-1.50845,-1.47993,-1.45145,-1.423,-1.39459,-1.36622,-1.3379,-1.30964,-1.28142,-1.25327,-1.22518,-1.19717,-1.16923,-1.14137,-1.1136,-1.08594,-1.05837,-1.03093,-1.00361,-0.97642,-0.94938,-0.922497,-0.895783,-0.869251,-0.842915,-0.816787,-0.790883,-0.765217,-0.739804,-0.714659,-0.689798,-0.665236,-0.640988,-0.617071,-0.593499,-0.570286,-0.547447,-0.524996,-0.502944,-0.481304,-0.460088,-0.439304,-0.418963,-0.399071,-0.379636,-0.360663,-0.342156,-0.324117,-0.306549,-0.289452,-0.272826,-0.256667,-0.240973,-0.225739,-0.210961,-0.196632,-0.182745,-0.169292,-0.156263,-0.143651,-0.131445,-0.119634,-0.108207,-0.0971539,-0.0864625,-0.0761212,-0.0661182,-0.0564416,-0.0470795,-0.0380201,-0.0292516,-0.0207624,-0.0125411,-0.00457666,0.00314185,0.010625,0.0178829,0.0249255,0.0317622,0.038402,0.0448535,0.051125,0.0572239,0.0631578,0.0689331,0.0745564,0.0800335,0.0853696,0.0905697,0.0956382,0.100579,0.105396,0.110093,0.114671,0.119133,0.123482,0.127719,0.131846,0.135863,0.139772,0.143573,0.147266,0.150852,0.154331,0.157703,0.160969,0.164127,0.167179,0.170124,0.172962,0.175694,0.178319,0.180839,0.183254,0.185564,0.187771,0.189876,0.191879,0.193783,0.19559,0.1973,0.198917,0.200442,0.201877,0.203226,0.204491,0.205674,0.206778,0.207806,0.208761,0.209646,0.210463,0.211215,0.211906,0.212536,0.21311,0.213629,0.214096,0.214512,0.21488,0.215201,0.215476,0.215708,0.215896,0.216041,0.216145,0.216206,0.216225,0.216201,0.216133,0.216019,0.215858,0.215647,0.215384,0.215064,0.214684,0.21424,0.213725,0.213133,0.212457,0.211689,0.21082,0.209839,0.208735,0.207495,0.206104,0.204545,0.202801,0.200851,0.198672,0.196241,0.193529,0.190505,0.187137,0.183387,0.179214,0.174573,0.169417,0.163692,0.15734,0.150298,0.1425,0.133872,0.124336,0.113812,0.102211,0.0894433,0.0754143,0.0600252,0.043178,0.024775,0.0047172,-0.017088,-0.040729,-0.0662824,-0.0938151},
	{-1.67568,-1.64697,-1.61829,-1.58962,-1.56098,-1.53236,-1.50377,-1.4752,-1.44668,-1.41818,-1.38973,-1.36131,-1.33294,-1.30462,-1.27636,-1.24815,-1.22001,-1.19193,-1.16393,-1.13602,-1.10819,-1.08046,-1.05283,-1.02532,-0.99793,-0.970675,-0.943564,-0.916609,-0.889823,-0.863217,-0.836806,-0.810603,-0.784623,-0.758879,-0.733388,-0.708164,-0.683223,-0.65858,-0.634251,-0.610252,-0.586597,-0.5633,-0.540376,-0.517838,-0.4957,-0.473972,-0.452666,-0.431791,-0.411357,-0.391371,-0.37184,-0.352769,-0.334162,-0.316021,-0.298349,-0.281145,-0.264408,-0.248137,-0.232328,-0.216977,-0.202077,-0.187624,-0.173609,-0.160025,-0.146863,-0.134114,-0.121767,-0.109812,-0.0982391,-0.0870359,-0.0761915,-0.0656942,-0.0555323,-0.0456941,-0.0361679,-0.026942,-0.0180048,-0.00934502,-0.000951511,0.0071866,0.0150799,0.0227385,0.0301724,0.0373911,0.0444035,0.0512183,0.0578437,0.0642874,0.0705567,0.0766582,0.0825984,0.088383,0.0940174,0.0995064,0.104854,0.110066,0.115143,0.12009,0.12491,0.129604,0.134176,0.138625,0.142954,0.147164,0.151256,0.15523,0.159087,0.162827,0.16645,0.169957,0.173348,0.176623,0.179782,0.182826,0.185754,0.188567,0.191267,0.193853,0.196326,0.198689,0.200941,0.203085,0.205122,0.207053,0.208883,0.210611,0.212241,0.213776,0.215218,0.21657,0.217834,0.219014,0.220114,0.221135,0.222081,0.222955,0.22376,0.224499,0.225174,0.225789,0.226346,0.226848,0.227296,0.227692,0.22804,0.228339,0.228591,0.228799,0.228961,0.22908,0.229154,0.229185,0.22917,0.22911,0.229003,0.228847,0.22864,0.228378,0.228059,0.227679,0.227232,0.226713,0.226116,0.225434,0.224658,0.223779,0.222787,0.22167,0.220414,0.219005,0.217427,0.215661,0.213687,0.211482,0.209021,0.206276,0.203217,0.199809,0.196016,0.191796,0.187105,0.181894,0.176108,0.169692,0.162581,0.154708,0.146,0.13638,0.125766,0.11407,0.101201,0.0870664,0.0715673,0.0546057,0.0360843,0.0159051,-0.0060242,-0.0297911,-0.0554715,-0.0831313},
	{-1.72594,-1.69717,-1.66841,-1.63967,-1.61094,-1.58224,-1.55356,-1.5249,-1.49627,-1.46767,-1.4391,-1.41056,-1.38206,-1.35359,-1.32518,-1.29681,-1.26849,-1.24023,-1.21203,-1.18391,-1.15585,-1.12788,-1.09999,-1.0722,-1.04451,-1.01694,-0.989492,-0.962175,-0.935003,-0.907987,-0.881139,-0.854473,-0.828,-0.801736,-0.775695,-0.749891,-0.72434,-0.699057,-0.674057,-0.649356,-0.62497,-0.600914,-0.577202,-0.553849,-0.530869,-0.508275,-0.486081,-0.464297,-0.442934,-0.422003,-0.401511,-0.381466,-0.361874,-0.342741,-0.32407,-0.305864,-0.288123,-0.270849,-0.254039,-0.237691,-0.221802,-0.206368,-0.191382,-0.176839,-0.16273,-0.149049,-0.135785,-0.12293,-0.110473,-0.0984044,-0.0867132,-0.0753882,-0.0644181,-0.0537914,-0.0434966,-0.033522,-0.0238562,-0.0144877,-0.00540531,0.00340213,0.0119454,0.020235,0.0282811,0.0360936,0.0436819,0.0510551,0.0582217,0.0651899,0.0719673,0.0785612,0.0849783,0.0912248,0.0973064,0.103229,0.108996,0.114613,0.120083,0.125411,0.130598,0.135648,0.140563,0.145344,0.149994,0.154514,0.158905,0.163168,0.167303,0.17131,0.175192,0.178947,0.182576,0.18608,0.189459,0.192712,0.195842,0.198848,0.201731,0.204492,0.207133,0.209654,0.212057,0.214344,0.216516,0.218576,0.220526,0.222368,0.224105,0.225741,0.227277,0.228717,0.230064,0.231321,0.232492,0.233579,0.234587,0.235517,0.236375,0.237162,0.237881,0.238537,0.23913,0.239665,0.240142,0.240566,0.240936,0.241256,0.241527,0.241749,0.241924,0.242053,0.242135,0.24217,0.242159,0.242099,0.241989,0.241828,0.241614,0.241342,0.24101,0.240614,0.240148,0.239606,0.238983,0.238271,0.237461,0.236543,0.235507,0.23434,0.233029,0.231559,0.229912,0.228069,0.22601,0.22371,0.221144,0.218284,0.215096,0.211547,0.207598,0.203207,0.198328,0.19291,0.186899,0.180235,0.172853,0.164686,0.155659,0.145692,0.134703,0.122603,0.1093,0.0947016,0.0787072,0.06122,0.0421434,0.0213807,-0.00115796,-0.0255575,-0.0518898,-0.0802163},
	{-1.77384,-1.74501,-1.71619,-1.68738,-1.65859,-1.62981,-1.60105,-1.57231,-1.54359,-1.5149,-1.48622,-1.45758,-1.42896,-1.40038,-1.37183,-1.34332,-1.31486,-1.28644,-1.25807,-1.22976,-1.20151,-1.17333,-1.14522,-1.11719,-1.08925,-1.0614,-1.03366,-1.00603,-0.978528,-0.951157,-0.923932,-0.896862,-0.869962,-0.843244,-0.816722,-0.79041,-0.764321,-0.738472,-0.712877,-0.687552,-0.662513,-0.637774,-0.613351,-0.58926,-0.565514,-0.54213,-0.519119,-0.496496,-0.474273,-0.452461,-0.431071,-0.410112,-0.389592,-0.369519,-0.349898,-0.330734,-0.312031,-0.293791,-0.276014,-0.2587,-0.241848,-0.225455,-0.209518,-0.194031,-0.178989,-0.164385,-0.150211,-0.13646,-0.123122,-0.110188,-0.0976479,-0.0854913,-0.0737074,-0.0622851,-0.0512134,-0.0404807,-0.0300759,-0.0199874,-0.0102041,-0.000714775,0.00849148,0.0174254,0.0260974,0.0345175,0.0426956,0.050641,0.0583626,0.065869,0.0731681,0.0802677,0.0871748,0.093896,0.100438,0.106805,0.113003,0.119038,0.124912,0.13063,0.136196,0.141611,0.146879,0.152002,0.156982,0.16182,0.166518,0.171077,0.175497,0.17978,0.183925,0.187935,0.191808,0.195546,0.19915,0.202619,0.205954,0.209157,0.212227,0.215167,0.217978,0.220661,0.223217,0.225649,0.227959,0.230149,0.232221,0.234179,0.236024,0.237761,0.239392,0.240921,0.24235,0.243684,0.244925,0.246078,0.247146,0.248132,0.24904,0.249872,0.250633,0.251325,0.251952,0.252515,0.253017,0.253461,0.253849,0.254183,0.254463,0.254692,0.25487,0.254997,0.255074,0.2551,0.255075,0.254998,0.254867,0.25468,0.254433,0.254125,0.25375,0.253304,0.252782,0.252177,0.251482,0.250689,0.249787,0.248768,0.247618,0.246324,0.244871,0.243243,0.241421,0.239383,0.237107,0.234568,0.231736,0.228581,0.225068,0.22116,0.216814,0.211986,0.206626,0.200679,0.194088,0.186788,0.178713,0.169788,0.159935,0.149073,0.137113,0.123966,0.109538,0.093731,0.076448,0.0575925,0.0370683,0.0147847,-0.00934364,-0.0353905,-0.0634193,-0.0934801},
	{-1.81958,-1.7907,-1.76183,-1.73296,-1.70411,-1.67527,-1.64645,-1.61763,-1.58884,-1.56006,-1.5313,-1.50256,-1.47384,-1.44515,-1.41649,-1.38786,-1.35926,-1.33071,-1.30219,-1.27372,-1.2453,-1.21694,-1.18864,-1.1604,-1.13224,-1.10416,-1.07617,-1.04827,-1.02048,-0.992796,-0.965242,-0.937822,-0.910548,-0.883433,-0.856489,-0.829729,-0.803167,-0.776817,-0.750694,-0.724813,-0.699189,-0.673837,-0.648774,-0.624014,-0.599574,-0.575467,-0.551709,-0.528314,-0.505296,-0.482667,-0.46044,-0.438625,-0.417233,-0.396272,-0.375751,-0.355676,-0.336053,-0.316887,-0.298179,-0.279931,-0.262146,-0.24482,-0.227953,-0.211542,-0.195582,-0.180069,-0.164996,-0.150357,-0.136143,-0.122347,-0.108959,-0.0959693,-0.0833687,-0.0711464,-0.0592918,-0.0477938,-0.0366416,-0.0258238,-0.0153293,-0.00514723,0.00473355,0.0143238,0.023634,0.0326745,0.0414552,0.0499858,0.0582755,0.0663331,0.0741669,0.081785,0.0891948,0.0964032,0.103417,0.110241,0.116883,0.123346,0.129635,0.135755,0.141709,0.1475,0.153131,0.158605,0.163923,0.169088,0.174101,0.178964,0.183677,0.188242,0.192659,0.196929,0.201053,0.205032,0.208865,0.212555,0.216101,0.219505,0.222769,0.225892,0.228877,0.231725,0.234439,0.23702,0.239471,0.241794,0.243992,0.246068,0.248024,0.249865,0.251594,0.253214,0.254728,0.256141,0.257456,0.258676,0.259807,0.26085,0.261811,0.262692,0.263496,0.264228,0.26489,0.265485,0.266015,0.266483,0.266892,0.267243,0.267537,0.267777,0.267962,0.268093,0.268171,0.268195,0.268164,0.268078,0.267934,0.26773,0.267463,0.267129,0.266725,0.266245,0.265683,0.265033,0.264287,0.263435,0.262468,0.261375,0.260143,0.258758,0.257202,0.25546,0.253511,0.251333,0.248901,0.246188,0.243165,0.239799,0.236053,0.231888,0.227259,0.22212,0.216417,0.210096,0.203094,0.195347,0.186783,0.177327,0.166898,0.155413,0.142781,0.128911,0.113708,0.0970751,0.0789145,0.0591316,0.0376323,0.0143297,-0.0108567,-0.0379939,-0.0671373,-0.0983286},
	{-1.86335,-1.83442,-1.80551,-1.7766,-1.74769,-1.7188,-1.68991,-1.66103,-1.63217,-1.60332,-1.57448,-1.54566,-1.51685,-1.48807,-1.45931,-1.43057,-1.40186,-1.37319,-1.34454,-1.31593,-1.28737,-1.25885,-1.23038,-1.20196,-1.17361,-1.14532,-1.1171,-1.08897,-1.06092,-1.03298,-1.00513,-0.977407,-0.949806,-0.922342,-0.895027,-0.867872,-0.84089,-0.814096,-0.787503,-0.761126,-0.734979,-0.709078,-0.683437,-0.658073,-0.633001,-0.608237,-0.583796,-0.559692,-0.53594,-0.512555,-0.489549,-0.466936,-0.444725,-0.42293,-0.401558,-0.38062,-0.360121,-0.340068,-0.320467,-0.301321,-0.282633,-0.264403,-0.246632,-0.229319,-0.21246,-0.196054,-0.180095,-0.164578,-0.149497,-0.134844,-0.120611,-0.106791,-0.0933734,-0.0803491,-0.067708,-0.0554397,-0.0435337,-0.0319792,-0.0207654,-0.00988123,0.000684003,0.0109411,0.0209005,0.0305728,0.0399679,0.0490957,0.0579655,0.0665865,0.0749672,0.0831159,0.0910404,0.0987479,0.106245,0.113539,0.120634,0.127537,0.134252,0.140784,0.147136,0.153312,0.159316,0.16515,0.170816,0.176316,0.181653,0.186827,0.191841,0.196695,0.20139,0.205927,0.210308,0.214533,0.218602,0.222517,0.22628,0.22989,0.23335,0.236661,0.239824,0.242842,0.245717,0.24845,0.251045,0.253504,0.25583,0.258027,0.260097,0.262044,0.263872,0.265585,0.267185,0.268679,0.270068,0.271358,0.272552,0.273654,0.274669,0.275598,0.276447,0.277219,0.277917,0.278543,0.279102,0.279595,0.280024,0.280392,0.2807,0.280949,0.281141,0.281276,0.281354,0.281375,0.281337,0.28124,0.281082,0.280859,0.28057,0.280209,0.279773,0.279256,0.278652,0.277953,0.277152,0.276238,0.275201,0.27403,0.27271,0.271227,0.269562,0.267699,0.265615,0.263287,0.26069,0.257794,0.254569,0.250979,0.246987,0.24255,0.237622,0.232155,0.226093,0.219378,0.211947,0.20373,0.194656,0.184645,0.173616,0.161482,0.148153,0.133535,0.117532,0.100048,0.080986,0.0602536,0.0377593,0.0134208,-0.0128371,-0.0410735,-0.0713379,-0.103656},
	{-1.90531,-1.87635,-1.84739,-1.81843,-1.78948,-1.76054,-1.7316,-1.70266,-1.67374,-1.64482,-1.61592,-1.58702,-1.55814,-1.52928,-1.50043,-1.4716,-1.44279,-1.414,-1.38525,-1.35652,-1.32782,-1.29916,-1.27054,-1.24197,-1.21344,-1.18497,-1.15657,-1.12822,-1.09996,-1.07177,-1.04368,-1.01568,-0.987789,-0.960017,-0.932373,-0.904868,-0.877515,-0.850326,-0.823314,-0.796494,-0.769878,-0.743483,-0.717322,-0.691411,-0.665767,-0.640403,-0.615337,-0.590583,-0.566156,-0.542071,-0.518343,-0.494985,-0.472011,-0.449431,-0.427258,-0.405502,-0.384172,-0.363275,-0.34282,-0.322811,-0.303253,-0.284149,-0.2655,-0.247309,-0.229574,-0.212293,-0.195464,-0.179083,-0.163144,-0.147643,-0.132572,-0.117925,-0.103692,-0.089866,-0.076437,-0.0633955,-0.0507315,-0.0384348,-0.0264948,-0.0149011,-0.00364294,0.00729013,0.0179086,0.0282229,0.0382431,0.047979,0.0574402,0.0666359,0.075575,0.0842659,0.0927166,0.100935,0.108927,0.116701,0.124261,0.131615,0.138766,0.14572,0.152481,0.159052,0.165438,0.17164,0.177662,0.183507,0.189175,0.194669,0.199991,0.205141,0.210121,0.214932,0.219576,0.224053,0.228364,0.232511,0.236494,0.240316,0.243977,0.24748,0.250826,0.254018,0.257057,0.259946,0.262689,0.265287,0.267745,0.270065,0.272251,0.274308,0.276238,0.278046,0.279737,0.281313,0.28278,0.284141,0.285401,0.286565,0.287635,0.288616,0.289512,0.290327,0.291063,0.291725,0.292315,0.292835,0.293289,0.293678,0.294004,0.294269,0.294473,0.294617,0.294701,0.294725,0.294689,0.294589,0.294426,0.294195,0.293894,0.293519,0.293065,0.292526,0.291897,0.291169,0.290334,0.289381,0.288301,0.287081,0.285706,0.284161,0.282428,0.280488,0.278319,0.275897,0.273196,0.270185,0.266833,0.263103,0.258956,0.25435,0.249238,0.243568,0.237284,0.230327,0.222632,0.214129,0.204744,0.194398,0.183008,0.170485,0.156739,0.141677,0.125202,0.107217,0.0876289,0.0663435,0.0432739,0.0183389,-0.00853027,-0.0373914,-0.0682878,-0.101239},
	{-1.9456,-1.9166,-1.88761,-1.85861,-1.82962,-1.80063,-1.77165,-1.74266,-1.71368,-1.68471,-1.65575,-1.62679,-1.59784,-1.5689,-1.53997,-1.51106,-1.48216,-1.45328,-1.42442,-1.39558,-1.36677,-1.33799,-1.30924,-1.28052,-1.25185,-1.22322,-1.19464,-1.16611,-1.13765,-1.10926,-1.08094,-1.0527,-1.02455,-0.996507,-0.968572,-0.940756,-0.913072,-0.885531,-0.858144,-0.830927,-0.80389,-0.77705,-0.75042,-0.724015,-0.69785,-0.671941,-0.646303,-0.620953,-0.595905,-0.571175,-0.546777,-0.522727,-0.499038,-0.475724,-0.452798,-0.43027,-0.408152,-0.386453,-0.365182,-0.344347,-0.323953,-0.304007,-0.28451,-0.265467,-0.246878,-0.228743,-0.211062,-0.193832,-0.177049,-0.16071,-0.14481,-0.129342,-0.114298,-0.0996728,-0.0854565,-0.0716408,-0.0582164,-0.0451737,-0.0325027,-0.0201934,-0.00823555,0.00338122,0.0146672,0.0256325,0.0362872,0.0466411,0.056704,0.066485,0.0759931,0.0852369,0.0942246,0.102964,0.111462,0.119726,0.127762,0.135577,0.143174,0.15056,0.157739,0.164715,0.171492,0.178072,0.184459,0.190655,0.196663,0.202484,0.208121,0.213575,0.218847,0.223938,0.228851,0.233585,0.238144,0.242527,0.246737,0.250774,0.254641,0.25834,0.261872,0.265241,0.268448,0.271496,0.274389,0.277129,0.279721,0.282167,0.284471,0.286639,0.288672,0.290577,0.292358,0.294018,0.295562,0.296996,0.298322,0.299546,0.300672,0.301704,0.302646,0.303503,0.304276,0.304971,0.30559,0.306135,0.30661,0.307017,0.307357,0.307632,0.307843,0.307991,0.308075,0.308095,0.308051,0.30794,0.307761,0.307511,0.307187,0.306783,0.306296,0.305719,0.305045,0.304266,0.303373,0.302357,0.301204,0.299902,0.298436,0.29679,0.294944,0.292879,0.290571,0.287996,0.285124,0.281926,0.278366,0.274408,0.27001,0.265127,0.259712,0.253709,0.247062,0.239708,0.23158,0.222606,0.212711,0.201812,0.189825,0.17666,0.162226,0.146429,0.129171,0.110358,0.0898966,0.0676962,0.0436734,0.0177518,-0.0101309,-0.0400247,-0.0719648,-0.105961},
	{-1.98434,-1.95532,-1.92629,-1.89726,-1.86823,-1.83921,-1.81018,-1.78115,-1.75212,-1.7231,-1.69408,-1.66507,-1.63605,-1.60705,-1.57805,-1.54907,-1.52009,-1.49112,-1.46217,-1.43324,-1.40432,-1.37543,-1.34656,-1.31772,-1.28891,-1.26014,-1.23141,-1.20272,-1.17408,-1.1455,-1.11698,-1.08853,-1.06015,-1.03186,-1.00367,-0.975576,-0.947595,-0.919738,-0.892017,-0.864442,-0.837027,-0.809786,-0.782731,-0.755878,-0.72924,-0.702834,-0.676675,-0.650777,-0.625158,-0.599832,-0.574815,-0.550121,-0.525767,-0.501766,-0.478132,-0.454877,-0.432014,-0.409554,-0.387507,-0.365882,-0.344688,-0.32393,-0.303616,-0.283747,-0.264329,-0.245363,-0.226849,-0.208788,-0.191176,-0.174013,-0.157292,-0.141011,-0.125164,-0.109743,-0.094742,-0.080153,-0.0659675,-0.0521767,-0.0387714,-0.025742,-0.0130789,-0.00077193,0.0111887,0.0228129,0.0341107,0.0450918,0.0557657,0.0661417,0.0762288,0.0860357,0.0955706,0.104842,0.113856,0.122621,0.131143,0.139428,0.147482,0.15531,0.162917,0.170306,0.177483,0.18445,0.19121,0.197767,0.204122,0.210279,0.216238,0.222002,0.227572,0.232951,0.238139,0.243138,0.247949,0.252574,0.257015,0.261273,0.265351,0.26925,0.272974,0.276523,0.279902,0.283113,0.28616,0.289046,0.291774,0.294349,0.296775,0.299056,0.301196,0.303201,0.305074,0.306821,0.308445,0.309953,0.311348,0.312636,0.31382,0.314906,0.315897,0.316797,0.317611,0.318342,0.318993,0.319568,0.320068,0.320497,0.320855,0.321146,0.321369,0.321526,0.321616,0.32164,0.321596,0.321483,0.321298,0.32104,0.320703,0.320284,0.319778,0.319178,0.318477,0.317668,0.31674,0.315683,0.314485,0.313131,0.311608,0.309898,0.307981,0.305837,0.303441,0.300768,0.297789,0.294472,0.290782,0.286681,0.282126,0.277071,0.271467,0.265258,0.258386,0.250788,0.242395,0.233135,0.222929,0.211696,0.19935,0.185801,0.170956,0.154723,0.137004,0.117705,0.0967351,0.074004,0.0494316,0.0229445,-0.00551576,-0.0359948,-0.0685227,-0.103103},
	{-2.02166,-1.99261,-1.96355,-1.93449,-1.90543,-1.87637,-1.8473,-1.81823,-1.78917,-1.7601,-1.73103,-1.70196,-1.67289,-1.64383,-1.61477,-1.58572,-1.55667,-1.52763,-1.4986,-1.46957,-1.44057,-1.41158,-1.3826,-1.35365,-1.32472,-1.29582,-1.26695,-1.23811,-1.20932,-1.18057,-1.15187,-1.12323,-1.09465,-1.06614,-1.03771,-1.00937,-0.981123,-0.952983,-0.924959,-0.897063,-0.869307,-0.841702,-0.814263,-0.787003,-0.759935,-0.733076,-0.706439,-0.68004,-0.653895,-0.628019,-0.602428,-0.577138,-0.552164,-0.527521,-0.503223,-0.479284,-0.455718,-0.432537,-0.409753,-0.387375,-0.365415,-0.343879,-0.322776,-0.302112,-0.281891,-0.262116,-0.242792,-0.223917,-0.205493,-0.187519,-0.169991,-0.152908,-0.136264,-0.120054,-0.104272,-0.0889126,-0.0739671,-0.0594279,-0.0452865,-0.031534,-0.0181615,-0.00515932,0.00748191,0.0197718,0.0317201,0.0433362,0.0546295,0.0656093,0.0762846,0.0866639,0.0967557,0.106568,0.116108,0.125384,0.134401,0.143167,0.151687,0.159966,0.168009,0.175822,0.183407,0.190769,0.197911,0.204836,0.211547,0.218046,0.224335,0.230416,0.236292,0.241964,0.247433,0.252701,0.257771,0.262643,0.26732,0.271804,0.276097,0.280201,0.284118,0.287853,0.291407,0.294784,0.297987,0.301021,0.303888,0.306594,0.309143,0.311539,0.313787,0.315892,0.317859,0.319693,0.321398,0.32298,0.324444,0.325795,0.327037,0.328175,0.329213,0.330156,0.331008,0.331773,0.332454,0.333054,0.333575,0.334021,0.334394,0.334694,0.334924,0.335083,0.335172,0.335191,0.335137,0.335011,0.334809,0.334528,0.334164,0.333713,0.33317,0.332527,0.331777,0.330912,0.329921,0.328792,0.327514,0.326072,0.32445,0.322629,0.320589,0.318308,0.315762,0.312922,0.309758,0.306238,0.302324,0.297976,0.29315,0.287799,0.281869,0.275304,0.268044,0.260023,0.25117,0.241411,0.230665,0.218849,0.205876,0.191655,0.176094,0.159096,0.140567,0.120415,0.0985488,0.0748841,0.0493442,0.0218614,-0.00761525,-0.0391248,-0.0726834,-0.108289},
	{-2.05765,-2.02857,-1.99949,-1.9704,-1.94131,-1.91221,-1.88311,-1.85401,-1.8249,-1.79579,-1.76668,-1.73757,-1.70845,-1.67934,-1.65022,-1.6211,-1.59199,-1.56288,-1.53378,-1.50468,-1.47559,-1.44651,-1.41744,-1.38838,-1.35934,-1.33033,-1.30133,-1.27237,-1.24343,-1.21453,-1.18567,-1.15685,-1.12809,-1.09939,-1.07075,-1.04218,-1.01369,-0.985299,-0.957002,-0.928815,-0.90075,-0.872816,-0.845028,-0.817397,-0.789938,-0.762664,-0.73559,-0.708731,-0.682102,-0.655719,-0.629597,-0.603753,-0.578202,-0.552959,-0.528039,-0.503458,-0.479229,-0.455366,-0.431881,-0.408788,-0.386096,-0.363815,-0.341955,-0.320522,-0.299525,-0.278967,-0.258853,-0.239186,-0.219968,-0.201198,-0.182877,-0.165002,-0.147571,-0.13058,-0.114024,-0.0978975,-0.0821944,-0.0669077,-0.0520298,-0.0375526,-0.0234678,-0.00976649,0.00356025,0.0165216,0.0291268,0.041385,0.0533055,0.0648973,0.0761692,0.0871299,0.0977877,0.108151,0.118226,0.128022,0.137544,0.1468,0.155795,0.164534,0.173024,0.181268,0.189271,0.197037,0.204569,0.211871,0.218945,0.225794,0.23242,0.238826,0.245014,0.250986,0.256743,0.262287,0.267621,0.272746,0.277665,0.28238,0.286892,0.291206,0.295323,0.299246,0.30298,0.306526,0.30989,0.313076,0.316087,0.318927,0.321603,0.324118,0.326477,0.328686,0.33075,0.332674,0.334463,0.336123,0.337659,0.339077,0.34038,0.341574,0.342665,0.343655,0.34455,0.345354,0.34607,0.346701,0.34725,0.347721,0.348115,0.348434,0.348679,0.348851,0.348949,0.348975,0.348926,0.348801,0.348598,0.348314,0.347944,0.347484,0.346928,0.346271,0.345503,0.344616,0.3436,0.342443,0.341133,0.339654,0.33799,0.336123,0.334032,0.331694,0.329085,0.326175,0.322935,0.31933,0.315324,0.310875,0.305938,0.300466,0.294405,0.287698,0.280284,0.272095,0.263062,0.253109,0.242155,0.230118,0.216908,0.202435,0.186607,0.169329,0.150506,0.130046,0.107861,0.0838667,0.0579889,0.0301619,0.000337092,-0.0315218,-0.065428,-0.101376},
	{-2.0924,-2.0633,-2.03419,-2.00508,-1.97596,-1.94684,-1.91771,-1.88857,-1.85943,-1.83028,-1.80113,-1.77197,-1.74281,-1.71365,-1.68448,-1.65531,-1.62614,-1.59697,-1.56779,-1.53862,-1.50946,-1.4803,-1.45114,-1.42199,-1.39286,-1.36374,-1.33463,-1.30554,-1.27648,-1.24744,-1.21844,-1.18947,-1.16054,-1.13165,-1.10282,-1.07405,-1.04535,-1.01672,-0.988178,-0.959728,-0.931381,-0.903149,-0.875043,-0.847074,-0.819257,-0.791604,-0.764129,-0.736847,-0.709772,-0.682921,-0.656308,-0.629949,-0.603861,-0.578058,-0.552556,-0.527371,-0.502518,-0.478011,-0.453863,-0.430088,-0.406699,-0.383705,-0.361118,-0.338947,-0.3172,-0.295884,-0.275005,-0.254566,-0.234572,-0.215025,-0.195925,-0.177272,-0.159065,-0.141302,-0.123979,-0.107091,-0.0906346,-0.0746029,-0.0589895,-0.0437874,-0.0289888,-0.0145858,-0.000569982,0.0130673,0.0263347,0.0392411,0.0517953,0.0640062,0.0758824,0.0874324,0.0986643,0.109586,0.120206,0.13053,0.140566,0.150321,0.1598,0.169009,0.177953,0.186637,0.195066,0.203244,0.211174,0.21886,0.226304,0.233511,0.240481,0.247218,0.253725,0.260002,0.266052,0.271878,0.277482,0.282865,0.28803,0.292979,0.297715,0.302242,0.306561,0.310677,0.314593,0.318312,0.321839,0.325177,0.328333,0.331309,0.334112,0.336746,0.339217,0.341529,0.34369,0.345703,0.347575,0.349311,0.350917,0.352398,0.35376,0.355007,0.356144,0.357177,0.358109,0.358945,0.359688,0.360342,0.36091,0.361395,0.361798,0.362122,0.362367,0.362535,0.362626,0.362638,0.362571,0.362422,0.36219,0.361871,0.36146,0.360953,0.360343,0.359623,0.358784,0.357818,0.356712,0.355455,0.354033,0.352429,0.350627,0.348606,0.346344,0.343817,0.340999,0.337859,0.334364,0.330479,0.326164,0.321375,0.316067,0.310187,0.30368,0.296485,0.28854,0.279774,0.270113,0.259481,0.247794,0.234966,0.220908,0.205527,0.188732,0.170427,0.15052,0.128921,0.105545,0.080314,0.053159,0.024026,-0.0071258,-0.0403209,-0.0755586,-0.112819},
	{-2.12599,-2.09687,-2.06774,-2.03861,-2.00946,-1.98031,-1.95116,-1.92199,-1.89282,-1.86364,-1.83445,-1.80525,-1.77605,-1.74684,-1.71763,-1.68841,-1.65918,-1.62995,-1.60072,-1.57149,-1.54225,-1.51302,-1.48378,-1.45455,-1.42533,-1.39611,-1.3669,-1.33771,-1.30852,-1.27936,-1.25022,-1.22111,-1.19203,-1.16299,-1.13398,-1.10503,-1.07613,-1.04729,-1.01852,-0.989829,-0.961227,-0.932722,-0.904325,-0.876048,-0.847903,-0.819902,-0.792059,-0.764388,-0.736903,-0.709618,-0.68255,-0.655714,-0.629125,-0.6028,-0.576753,-0.551002,-0.525561,-0.500445,-0.47567,-0.451248,-0.427194,-0.40352,-0.380237,-0.357356,-0.334887,-0.312837,-0.291216,-0.270028,-0.249278,-0.22897,-0.209108,-0.189691,-0.17072,-0.152195,-0.134113,-0.116472,-0.0992668,-0.0824937,-0.0661471,-0.0502208,-0.0347081,-0.0196017,-0.00489414,0.00942264,0.0233568,0.0369166,0.0501107,0.0629473,0.0754348,0.0875816,0.0993956,0.110885,0.122057,0.132918,0.143477,0.153739,0.163711,0.173397,0.182805,0.191938,0.200802,0.209399,0.217736,0.225814,0.233637,0.241208,0.24853,0.255606,0.262437,0.269027,0.275377,0.281491,0.28737,0.293016,0.298433,0.303622,0.308588,0.313332,0.317859,0.322171,0.326273,0.330169,0.333863,0.337359,0.340662,0.343778,0.346712,0.349469,0.352054,0.354474,0.356734,0.35884,0.360798,0.362614,0.364293,0.365841,0.367264,0.368568,0.369756,0.370835,0.371809,0.372681,0.373457,0.37414,0.374732,0.375237,0.375657,0.375993,0.376248,0.376421,0.376513,0.376523,0.376449,0.376291,0.376045,0.375708,0.375275,0.37474,0.374098,0.373341,0.37246,0.371445,0.370284,0.368966,0.367474,0.365793,0.363904,0.361786,0.359418,0.356774,0.353825,0.350541,0.346889,0.34283,0.338324,0.333327,0.327789,0.32166,0.31488,0.30739,0.299122,0.290008,0.27997,0.268931,0.256807,0.24351,0.228952,0.213038,0.195678,0.176777,0.156243,0.133989,0.109932,0.0839987,0.0561232,0.026257,-0.00563475,-0.0395702,-0.0755411,-0.11352},
	{-2.1585,-2.12936,-2.10021,-2.07106,-2.04189,-2.01272,-1.98353,-1.95434,-1.92514,-1.89593,-1.86671,-1.83748,-1.80824,-1.77899,-1.74973,-1.72047,-1.69119,-1.66191,-1.63262,-1.60333,-1.57403,-1.54473,-1.51542,-1.48612,-1.45681,-1.4275,-1.3982,-1.36891,-1.33962,-1.31035,-1.28109,-1.25184,-1.22262,-1.19343,-1.16427,-1.13515,-1.10607,-1.07703,-1.04806,-1.01915,-0.990314,-0.96156,-0.932897,-0.904337,-0.875892,-0.847571,-0.81939,-0.791359,-0.763495,-0.73581,-0.708319,-0.681039,-0.653985,-0.627171,-0.600615,-0.574332,-0.548338,-0.522648,-0.497278,-0.472243,-0.447557,-0.423233,-0.399284,-0.375722,-0.352557,-0.329801,-0.307461,-0.285546,-0.264061,-0.243012,-0.222403,-0.202238,-0.182517,-0.163241,-0.14441,-0.126023,-0.108076,-0.0905667,-0.0734902,-0.0568418,-0.0406157,-0.0248056,-0.00940473,0.00559412,0.0201985,0.0344161,0.0482548,0.0617228,0.0748279,0.0875782,0.0999814,0.112045,0.123777,0.135184,0.146273,0.157051,0.167524,0.177697,0.187576,0.197166,0.206472,0.215498,0.224248,0.232725,0.240934,0.248878,0.256558,0.263979,0.271142,0.278051,0.284707,0.291114,0.297273,0.303188,0.308861,0.314296,0.319495,0.324461,0.329199,0.333711,0.338003,0.342078,0.345941,0.349597,0.353051,0.356309,0.359375,0.362256,0.364957,0.367485,0.369845,0.372044,0.374088,0.375983,0.377735,0.37935,0.380834,0.382192,0.38343,0.384553,0.385565,0.386471,0.387276,0.387983,0.388595,0.389115,0.389545,0.389887,0.390143,0.390313,0.390397,0.390394,0.390303,0.390122,0.389848,0.389476,0.389003,0.388421,0.387725,0.386907,0.385956,0.384862,0.383614,0.382196,0.380594,0.378791,0.376765,0.374497,0.371962,0.369133,0.36598,0.362472,0.358572,0.354241,0.349437,0.344113,0.338218,0.331697,0.32449,0.316535,0.307763,0.298101,0.287472,0.275794,0.262982,0.248949,0.233603,0.216852,0.198603,0.178763,0.157244,0.133962,0.108837,0.081802,0.0527986,0.0217871,-0.0112592,-0.0463435,-0.0834496,-0.122536},
	{-2.18999,-2.16084,-2.13167,-2.1025,-2.07331,-2.04412,-2.01491,-1.98569,-1.95646,-1.92722,-1.89797,-1.86871,-1.83944,-1.81015,-1.78085,-1.75155,-1.72223,-1.6929,-1.66356,-1.63422,-1.60486,-1.57549,-1.54612,-1.51675,-1.48736,-1.45798,-1.42859,-1.3992,-1.36982,-1.34044,-1.31107,-1.28171,-1.25236,-1.22303,-1.19373,-1.16445,-1.1352,-1.106,-1.07683,-1.04772,-1.01867,-0.989687,-0.960781,-0.931961,-0.903238,-0.874624,-0.846129,-0.817768,-0.789552,-0.761496,-0.733614,-0.705921,-0.678432,-0.651163,-0.624129,-0.597347,-0.570832,-0.544601,-0.51867,-0.493052,-0.467765,-0.442821,-0.418235,-0.39402,-0.370188,-0.34675,-0.323716,-0.301096,-0.278897,-0.257126,-0.235789,-0.21489,-0.194433,-0.174419,-0.15485,-0.135726,-0.117045,-0.0988047,-0.0810029,-0.0636355,-0.0466977,-0.0301843,-0.0140895,0.00159311,0.0168703,0.0317492,0.0462371,0.0603414,0.0740698,0.0874298,0.100429,0.113074,0.125374,0.137334,0.148961,0.160263,0.171244,0.181912,0.19227,0.202325,0.212082,0.221544,0.230715,0.2396,0.248203,0.256525,0.264572,0.272344,0.279846,0.28708,0.294048,0.300754,0.307201,0.31339,0.319325,0.32501,0.330447,0.33564,0.340593,0.34531,0.349795,0.354054,0.35809,0.36191,0.365518,0.368919,0.372121,0.375129,0.377949,0.380588,0.383051,0.385346,0.387478,0.389455,0.391282,0.392966,0.394512,0.395927,0.397217,0.398386,0.399439,0.400382,0.401218,0.401951,0.402586,0.403124,0.403568,0.40392,0.404181,0.404351,0.404432,0.404421,0.404318,0.404119,0.403823,0.403424,0.402919,0.402299,0.401559,0.400689,0.399681,0.398522,0.3972,0.3957,0.394006,0.3921,0.389961,0.387567,0.384892,0.381909,0.378587,0.374892,0.370787,0.366231,0.36118,0.355586,0.349396,0.342553,0.334997,0.326662,0.317478,0.30737,0.29626,0.284065,0.2707,0.256074,0.240097,0.222677,0.20372,0.183137,0.160841,0.13675,0.11079,0.0828958,0.0530193,0.0211245,-0.0128067,-0.0487688,-0.0867359,-0.126659},
	{-2.22053,-2.19136,-2.16218,-2.13299,-2.10378,-2.07457,-2.04534,-2.0161,-1.98685,-1.95758,-1.9283,-1.89901,-1.8697,-1.84039,-1.81105,-1.78171,-1.75235,-1.72298,-1.69359,-1.6642,-1.63479,-1.60536,-1.57593,-1.54649,-1.51704,-1.48758,-1.45811,-1.42864,-1.39917,-1.36969,-1.34022,-1.31075,-1.28129,-1.25183,-1.22239,-1.19297,-1.16358,-1.13421,-1.10487,-1.07557,-1.04632,-1.01713,-0.987998,-0.958938,-0.92996,-0.901074,-0.872291,-0.843623,-0.815082,-0.786682,-0.758435,-0.730358,-0.702464,-0.674768,-0.647287,-0.620037,-0.593033,-0.566291,-0.539828,-0.513659,-0.4878,-0.462266,-0.437072,-0.412231,-0.387757,-0.363663,-0.339959,-0.316656,-0.293764,-0.271291,-0.249244,-0.227628,-0.20645,-0.185711,-0.165416,-0.145564,-0.126157,-0.107193,-0.0886717,-0.0705892,-0.0529425,-0.0357274,-0.0189389,-0.00257171,0.0133802,0.028923,0.0440635,0.0588085,0.0731651,0.0871403,0.100741,0.113975,0.126849,0.139369,0.151542,0.163375,0.174873,0.186043,0.196889,0.207417,0.217631,0.227537,0.237138,0.246438,0.255441,0.26415,0.272569,0.280701,0.288548,0.296113,0.3034,0.310411,0.31715,0.323619,0.329822,0.335761,0.341441,0.346866,0.352039,0.356965,0.361648,0.366093,0.370307,0.374293,0.378058,0.381607,0.384948,0.388085,0.391026,0.393778,0.396346,0.398739,0.400962,0.403022,0.404926,0.40668,0.408291,0.409765,0.411107,0.412324,0.41342,0.4144,0.415269,0.416031,0.416689,0.417247,0.417706,0.418068,0.418336,0.418509,0.418588,0.418571,0.418458,0.418244,0.417929,0.417505,0.41697,0.416316,0.415535,0.414618,0.413556,0.412337,0.410946,0.40937,0.407591,0.40559,0.403346,0.400835,0.398032,0.394906,0.391428,0.38756,0.383266,0.378503,0.373225,0.367383,0.360923,0.353787,0.345911,0.33723,0.327671,0.317159,0.305614,0.292953,0.279088,0.26393,0.247388,0.22937,0.209783,0.18854,0.165556,0.140751,0.114056,0.0854109,0.0547715,0.0221088,-0.0125888,-0.0493079,-0.0880149,-0.128654},
	{-2.25017,-2.22099,-2.19179,-2.16258,-2.13336,-2.10413,-2.07488,-2.04562,-2.01634,-1.98705,-1.95775,-1.92843,-1.8991,-1.86975,-1.84038,-1.811,-1.7816,-1.75219,-1.72276,-1.69332,-1.66386,-1.63439,-1.6049,-1.5754,-1.54588,-1.51636,-1.48682,-1.45727,-1.42771,-1.39815,-1.36858,-1.33901,-1.30944,-1.27987,-1.25031,-1.22076,-1.19122,-1.1617,-1.1322,-1.10273,-1.0733,-1.04391,-1.01457,-0.985292,-0.956077,-0.92694,-0.897889,-0.868937,-0.840094,-0.811373,-0.782788,-0.754352,-0.726079,-0.697985,-0.670085,-0.642395,-0.614929,-0.587706,-0.56074,-0.534048,-0.507646,-0.48155,-0.455775,-0.430336,-0.405246,-0.38052,-0.356171,-0.332208,-0.308644,-0.285489,-0.26275,-0.240434,-0.21855,-0.1971,-0.176091,-0.155523,-0.135399,-0.11572,-0.0964844,-0.0776919,-0.0593399,-0.0414254,-0.0239445,-0.00689284,0.0097346,0.0259433,0.0417392,0.0571284,0.0721174,0.0867127,0.100921,0.114749,0.128203,0.14129,0.154016,0.166387,0.178409,0.190088,0.201429,0.212438,0.223118,0.233475,0.243513,0.253235,0.262646,0.271749,0.280547,0.289044,0.297242,0.305146,0.312757,0.320079,0.327116,0.33387,0.340345,0.346544,0.352472,0.358132,0.363529,0.368667,0.373551,0.378187,0.38258,0.386736,0.39066,0.39436,0.39784,0.401109,0.404173,0.407039,0.409714,0.412205,0.414519,0.416662,0.418643,0.420468,0.422143,0.423674,0.425068,0.426331,0.427468,0.428484,0.429383,0.43017,0.430848,0.431421,0.43189,0.432259,0.432527,0.432696,0.432766,0.432735,0.432602,0.432363,0.432017,0.431557,0.430978,0.430273,0.429434,0.428452,0.427315,0.426012,0.424528,0.422848,0.420952,0.418822,0.416435,0.413766,0.410787,0.407469,0.403778,0.399678,0.395128,0.390085,0.384501,0.378324,0.371499,0.363964,0.355657,0.346508,0.336443,0.325386,0.313254,0.299962,0.285423,0.269547,0.252242,0.233418,0.212983,0.190852,0.166943,0.141181,0.113501,0.0838506,0.0521935,0.0185091,-0.0172051,-0.0549246,-0.0946054,-0.136181},
	{-2.27896,-2.24977,-2.22056,-2.19133,-2.1621,-2.13284,-2.10358,-2.0743,-2.045,-2.01569,-1.98637,-1.95702,-1.92766,-1.89828,-1.86889,-1.83947,-1.81004,-1.78059,-1.75113,-1.72164,-1.69214,-1.66261,-1.63307,-1.60352,-1.57394,-1.54435,-1.51475,-1.48513,-1.45549,-1.42585,-1.39619,-1.36653,-1.33685,-1.30718,-1.2775,-1.24783,-1.21816,-1.1885,-1.15886,-1.12923,-1.09963,-1.07006,-1.04053,-1.01104,-0.981608,-0.952237,-0.922938,-0.893721,-0.864597,-0.835578,-0.806677,-0.777906,-0.74928,-0.720812,-0.692519,-0.664414,-0.636515,-0.608837,-0.581396,-0.554208,-0.527291,-0.500659,-0.47433,-0.448317,-0.422638,-0.397304,-0.372332,-0.347732,-0.323518,-0.299699,-0.276287,-0.25329,-0.230715,-0.208569,-0.186857,-0.165585,-0.144754,-0.124368,-0.104426,-0.084929,-0.0658761,-0.0472653,-0.0290939,-0.0113586,0.00594471,0.0228205,0.0392739,0.0553103,0.0709355,0.0861553,0.100976,0.115403,0.129444,0.143104,0.15639,0.169306,0.18186,0.194055,0.205899,0.217395,0.228549,0.239365,0.249847,0.259999,0.269825,0.279328,0.288513,0.297382,0.305939,0.314187,0.32213,0.329769,0.33711,0.344154,0.350907,0.357372,0.363552,0.369453,0.375079,0.380434,0.385525,0.390356,0.394933,0.399262,0.40335,0.407203,0.410828,0.414232,0.417423,0.420406,0.423191,0.425784,0.428192,0.430424,0.432485,0.434384,0.436127,0.43772,0.43917,0.440484,0.441666,0.442723,0.443658,0.444476,0.445181,0.445776,0.446264,0.446646,0.446925,0.4471,0.447172,0.447139,0.447,0.446751,0.44639,0.445911,0.445309,0.444576,0.443705,0.442684,0.441504,0.440151,0.438611,0.436867,0.434901,0.432692,0.430218,0.427452,0.424367,0.420932,0.417113,0.412872,0.408168,0.402956,0.397188,0.390811,0.383769,0.376,0.367438,0.358015,0.347655,0.33628,0.323808,0.310155,0.295231,0.278946,0.261211,0.241934,0.221025,0.198401,0.173982,0.147695,0.11948,0.089285,0.0570802,0.022849,-0.0134061,-0.0516557,-0.0918504,-0.133919},
	{-2.30695,-2.27775,-2.24852,-2.21929,-2.19003,-2.16077,-2.13149,-2.10219,-2.07287,-2.04354,-2.01419,-1.98483,-1.95544,-1.92604,-1.89661,-1.86717,-1.83771,-1.80822,-1.77872,-1.74919,-1.71965,-1.69008,-1.66049,-1.63088,-1.60126,-1.57161,-1.54194,-1.51225,-1.48255,-1.45282,-1.42309,-1.39333,-1.36357,-1.3338,-1.30401,-1.27423,-1.24444,-1.21465,-1.18487,-1.15509,-1.12534,-1.0956,-1.06589,-1.03621,-1.00657,-0.976984,-0.947453,-0.91799,-0.888604,-0.859307,-0.83011,-0.801027,-0.772069,-0.743252,-0.714589,-0.686095,-0.657786,-0.629679,-0.601788,-0.574131,-0.546723,-0.519582,-0.492723,-0.466163,-0.439917,-0.414001,-0.388428,-0.363214,-0.33837,-0.313909,-0.289842,-0.26618,-0.242931,-0.220103,-0.197704,-0.175739,-0.154212,-0.133127,-0.112487,-0.0922923,-0.0725437,-0.0532406,-0.0343815,-0.0159642,0.00201436,0.0195577,0.0366699,0.0533557,0.0696199,0.0854678,0.100905,0.115937,0.13057,0.144808,0.158659,0.172128,0.185219,0.197938,0.210291,0.222282,0.233917,0.245198,0.256131,0.26672,0.276968,0.286879,0.296457,0.305705,0.314627,0.323225,0.331504,0.339466,0.347116,0.354457,0.361493,0.368227,0.374665,0.38081,0.386669,0.392245,0.397544,0.402573,0.407337,0.411842,0.416095,0.420104,0.423875,0.427416,0.430734,0.433836,0.436731,0.439426,0.441929,0.444247,0.446388,0.448359,0.450168,0.451821,0.453325,0.454686,0.45591,0.457003,0.457969,0.458813,0.459538,0.460149,0.460647,0.461035,0.461313,0.461484,0.461545,0.461497,0.461336,0.461061,0.460667,0.460149,0.459501,0.458715,0.457782,0.456692,0.455434,0.453993,0.452354,0.450501,0.448414,0.44607,0.443447,0.440516,0.43725,0.433616,0.429577,0.425095,0.420128,0.414628,0.408546,0.401827,0.394412,0.386238,0.377238,0.36734,0.356469,0.344545,0.331484,0.3172,0.301604,0.284606,0.266117,0.246046,0.224306,0.200815,0.175497,0.148286,0.119125,0.0879702,0.0547998,0.0196042,-0.0176008,-0.0567798,-0.0978724,-0.140798},
	{-2.33419,-2.30497,-2.27573,-2.24649,-2.21722,-2.18794,-2.15864,-2.12933,-2.1,-2.07065,-2.04128,-2.01189,-1.98248,-1.95305,-1.9236,-1.89413,-1.86464,-1.83512,-1.80558,-1.77602,-1.74644,-1.71683,-1.6872,-1.65754,-1.62786,-1.59816,-1.56843,-1.53868,-1.50891,-1.47912,-1.4493,-1.41947,-1.38962,-1.35975,-1.32987,-1.29998,-1.27008,-1.24017,-1.21026,-1.18035,-1.15044,-1.12055,-1.09067,-1.06081,-1.03099,-1.0012,-0.97145,-0.941757,-0.912127,-0.88257,-0.853097,-0.82372,-0.794451,-0.765305,-0.736295,-0.707435,-0.67874,-0.650227,-0.62191,-0.593807,-0.565934,-0.538308,-0.510944,-0.48386,-0.457072,-0.430595,-0.404445,-0.378637,-0.353184,-0.3281,-0.303398,-0.279088,-0.255182,-0.231687,-0.208614,-0.185968,-0.163757,-0.141984,-0.120653,-0.0997678,-0.0793294,-0.0593386,-0.0397952,-0.0206981,-0.00204542,0.0161653,0.0339372,0.051274,0.0681798,0.0846591,0.100717,0.116358,0.131587,0.14641,0.160833,0.174859,0.188494,0.201744,0.214614,0.227107,0.239228,0.250983,0.262374,0.273407,0.284084,0.29441,0.304388,0.314021,0.323314,0.33227,0.340892,0.349183,0.357148,0.364791,0.372115,0.379125,0.385825,0.39222,0.398316,0.404117,0.40963,0.414861,0.419816,0.424501,0.428924,0.433092,0.437013,0.440694,0.444142,0.447367,0.450375,0.453175,0.455775,0.458184,0.460407,0.462455,0.464333,0.466049,0.46761,0.469022,0.470292,0.471425,0.472426,0.4733,0.474051,0.474682,0.475196,0.475596,0.475881,0.476054,0.476113,0.476058,0.475887,0.475595,0.47518,0.474636,0.473956,0.473132,0.472156,0.471017,0.469702,0.468197,0.466487,0.464554,0.462377,0.459935,0.457202,0.454151,0.450753,0.446972,0.442774,0.438117,0.432958,0.42725,0.42094,0.413974,0.406292,0.397828,0.388516,0.378281,0.367048,0.354736,0.34126,0.326535,0.310471,0.292979,0.273969,0.253352,0.231043,0.206961,0.181035,0.1532,0.123404,0.0916087,0.0577964,0.0219637,-0.0158678,-0.0556557,-0.0973358,-0.14082},
	{-2.36071,-2.33148,-2.30223,-2.27297,-2.24369,-2.2144,-2.18509,-2.15576,-2.12641,-2.09704,-2.06765,-2.03825,-2.00882,-1.97937,-1.94989,-1.92039,-1.89087,-1.86133,-1.83176,-1.80216,-1.77254,-1.74289,-1.71322,-1.68352,-1.65379,-1.62404,-1.59426,-1.56445,-1.53462,-1.50476,-1.47487,-1.44496,-1.41503,-1.38508,-1.35511,-1.32511,-1.29511,-1.26509,-1.23505,-1.20502,-1.17498,-1.14494,-1.1149,-1.08488,-1.05487,-1.02489,-0.994946,-0.965038,-0.935178,-0.905378,-0.875645,-0.845993,-0.816433,-0.786978,-0.75764,-0.728435,-0.699376,-0.670479,-0.641759,-0.613234,-0.584918,-0.556829,-0.528984,-0.501399,-0.474091,-0.447076,-0.42037,-0.39399,-0.367949,-0.342261,-0.316942,-0.292002,-0.267455,-0.243309,-0.219576,-0.196263,-0.173377,-0.150926,-0.128915,-0.107346,-0.0862247,-0.0655515,-0.0453278,-0.0255539,-0.00622903,0.0126482,0.0310799,0.0490687,0.066618,0.0837313,0.100413,0.116667,0.132498,0.14791,0.162909,0.177499,0.191685,0.205472,0.218864,0.231865,0.244481,0.256716,0.268572,0.280055,0.291168,0.301916,0.3123,0.322326,0.331997,0.341315,0.350286,0.358913,0.367199,0.375149,0.382766,0.390056,0.397024,0.403674,0.410011,0.416042,0.421773,0.427209,0.432358,0.437227,0.441822,0.446153,0.450225,0.454048,0.45763,0.460978,0.464102,0.467009,0.469708,0.472207,0.474515,0.476639,0.478587,0.480367,0.481985,0.483449,0.484764,0.485937,0.486973,0.487876,0.488651,0.489301,0.48983,0.490239,0.49053,0.490702,0.490757,0.490692,0.490506,0.490194,0.489754,0.489178,0.488461,0.487595,0.486569,0.485373,0.483993,0.482416,0.480625,0.478601,0.476324,0.473771,0.470915,0.467729,0.464181,0.460237,0.455859,0.451006,0.445633,0.439691,0.433127,0.425885,0.417903,0.409116,0.399455,0.388845,0.377208,0.364464,0.350527,0.335312,0.318728,0.300686,0.281099,0.259877,0.236939,0.212206,0.185608,0.157086,0.126592,0.0940954,0.0595811,0.0230531,-0.01546,-0.0559095,-0.0982241,-0.142309},
	{-2.38654,-2.3573,-2.32805,-2.29877,-2.26949,-2.24018,-2.21086,-2.18151,-2.15215,-2.12277,-2.09336,-2.06393,-2.03449,-2.00501,-1.97552,-1.94599,-1.91645,-1.88687,-1.85727,-1.82765,-1.79799,-1.76831,-1.7386,-1.70885,-1.67908,-1.64928,-1.61945,-1.58959,-1.5597,-1.52977,-1.49982,-1.46984,-1.43984,-1.4098,-1.37974,-1.34966,-1.31956,-1.28943,-1.25929,-1.22913,-1.19896,-1.16878,-1.1386,-1.10842,-1.07825,-1.04809,-1.01796,-0.987846,-0.957772,-0.927742,-0.897767,-0.867856,-0.838022,-0.808275,-0.778629,-0.749097,-0.719694,-0.690434,-0.661332,-0.632405,-0.603669,-0.57514,-0.546835,-0.518771,-0.490964,-0.463433,-0.436193,-0.40926,-0.382651,-0.35638,-0.330461,-0.304909,-0.279737,-0.254955,-0.230576,-0.206609,-0.183062,-0.159944,-0.13726,-0.115017,-0.0932194,-0.0718695,-0.0509702,-0.030523,-0.0105283,0.00901403,0.0281051,0.0467465,0.0649406,0.08269,0.0999982,0.116868,0.133305,0.149311,0.164892,0.180051,0.194794,0.209123,0.223044,0.236561,0.249678,0.262399,0.274728,0.286668,0.298224,0.309399,0.320197,0.330622,0.340676,0.350365,0.359691,0.368658,0.377271,0.385534,0.393451,0.401026,0.408266,0.415175,0.421759,0.428024,0.433976,0.439622,0.44497,0.450025,0.454797,0.459293,0.463521,0.467489,0.471206,0.474681,0.477923,0.480939,0.48374,0.486332,0.488726,0.490929,0.492949,0.494794,0.496472,0.497989,0.499351,0.500566,0.501638,0.502573,0.503375,0.504046,0.504592,0.505013,0.505311,0.505486,0.505539,0.505467,0.50527,0.504942,0.50448,0.503878,0.503129,0.502225,0.501155,0.499909,0.498472,0.496831,0.494968,0.492864,0.490497,0.487845,0.48488,0.481573,0.477893,0.473804,0.469267,0.464241,0.458679,0.452531,0.445745,0.438261,0.430017,0.420948,0.410982,0.400046,0.38806,0.374942,0.360609,0.344971,0.327942,0.309431,0.289352,0.267617,0.244146,0.218863,0.191701,0.162604,0.131528,0.0984468,0.0633502,0.0262473,-0.0128278,-0.0538208,-0.0966559,-0.141233},
	{-2.41173,-2.38248,-2.35322,-2.32393,-2.29464,-2.26532,-2.23598,-2.20663,-2.17725,-2.14785,-2.11843,-2.08899,-2.05952,-2.03003,-2.00051,-1.97096,-1.94139,-1.91179,-1.88217,-1.85251,-1.82282,-1.79311,-1.76336,-1.73358,-1.70376,-1.67392,-1.64404,-1.61413,-1.58418,-1.5542,-1.52419,-1.49414,-1.46406,-1.43395,-1.40381,-1.37364,-1.34345,-1.31322,-1.28297,-1.2527,-1.22241,-1.19211,-1.16179,-1.13146,-1.10114,-1.07081,-1.0405,-1.0102,-0.979921,-0.949676,-0.919472,-0.889318,-0.859224,-0.829203,-0.799266,-0.769426,-0.739697,-0.710093,-0.680629,-0.651321,-0.622184,-0.593235,-0.564491,-0.535969,-0.507685,-0.479658,-0.451904,-0.424439,-0.397281,-0.370444,-0.343945,-0.317799,-0.292018,-0.266616,-0.241605,-0.216997,-0.192801,-0.169026,-0.145681,-0.122772,-0.100305,-0.0782852,-0.0567156,-0.0355991,-0.0149377,0.00526776,0.0250171,0.044311,0.0631506,0.0815378,0.0994749,0.116965,0.13401,0.150615,0.166782,0.182515,0.197819,0.212697,0.227154,0.241192,0.254816,0.26803,0.280837,0.293242,0.305247,0.316857,0.328075,0.338904,0.349349,0.359413,0.3691,0.378414,0.38736,0.395941,0.404162,0.412028,0.419545,0.426717,0.433552,0.440055,0.446233,0.452092,0.457641,0.462887,0.467838,0.472501,0.476887,0.481003,0.484858,0.488461,0.491822,0.494949,0.497852,0.500539,0.503019,0.505301,0.507393,0.509303,0.511039,0.512609,0.514018,0.515273,0.51638,0.517344,0.51817,0.51886,0.519419,0.519849,0.52015,0.520323,0.520369,0.520285,0.520069,0.519718,0.519227,0.51859,0.517799,0.516846,0.515721,0.514411,0.512903,0.511181,0.509228,0.507024,0.504547,0.501773,0.498673,0.495218,0.491375,0.487107,0.482375,0.477135,0.471341,0.46494,0.457879,0.450097,0.441532,0.432115,0.421775,0.410437,0.398021,0.384444,0.369621,0.353465,0.335887,0.316798,0.296114,0.273748,0.249622,0.223663,0.195809,0.166006,0.134217,0.10042,0.064612,0.0268086,-0.0129498,-0.0546014,-0.098065,-0.143234},
	{-2.4363,-2.40705,-2.37777,-2.34848,-2.31917,-2.28984,-2.2605,-2.23113,-2.20174,-2.17233,-2.14289,-2.11343,-2.08395,-2.05444,-2.0249,-1.99533,-1.96574,-1.93612,-1.90646,-1.87678,-1.84706,-1.81731,-1.78753,-1.75771,-1.72786,-1.69797,-1.66805,-1.63809,-1.60809,-1.57806,-1.54799,-1.51788,-1.48774,-1.45756,-1.42734,-1.39709,-1.3668,-1.33649,-1.30614,-1.27576,-1.24536,-1.21493,-1.18449,-1.15402,-1.12355,-1.09307,-1.06258,-1.03211,-1.00164,-0.971192,-0.940771,-0.910388,-0.88005,-0.849769,-0.819557,-0.789426,-0.759388,-0.729458,-0.69965,-0.669979,-0.640461,-0.611112,-0.581948,-0.552988,-0.524247,-0.495743,-0.467494,-0.439517,-0.411829,-0.384445,-0.357384,-0.330659,-0.304286,-0.278279,-0.252652,-0.227416,-0.202583,-0.178163,-0.154167,-0.130601,-0.107473,-0.0847894,-0.062555,-0.0407739,-0.0194492,0.00141689,0.0218231,0.0417689,0.0612544,0.0802805,0.0988484,0.11696,0.134618,0.151824,0.168582,0.184895,0.200766,0.216198,0.231195,0.245761,0.259899,0.273612,0.286904,0.299779,0.31224,0.324291,0.335936,0.347177,0.358018,0.368464,0.378519,0.388185,0.397469,0.406374,0.414904,0.423066,0.430865,0.438307,0.445397,0.452142,0.45855,0.464627,0.470381,0.475821,0.480954,0.485789,0.490335,0.494601,0.498597,0.502331,0.505814,0.509054,0.512061,0.514844,0.517413,0.519776,0.521942,0.523919,0.525716,0.527339,0.528796,0.530093,0.531237,0.532232,0.533083,0.533794,0.534368,0.534807,0.535113,0.535287,0.535327,0.535233,0.535002,0.53463,0.534112,0.533442,0.532612,0.531614,0.530437,0.529068,0.527493,0.525696,0.52366,0.521362,0.518782,0.515893,0.512667,0.509074,0.505079,0.500645,0.495732,0.490294,0.484284,0.47765,0.470334,0.462278,0.453417,0.443681,0.432999,0.421293,0.408485,0.394489,0.379222,0.362596,0.344523,0.324914,0.303686,0.280755,0.256044,0.229483,0.201013,0.170586,0.138167,0.103741,0.0673091,0.0288931,-0.0114607,-0.053685,-0.0976935,-0.143375},
	{-2.46029,-2.43103,-2.40174,-2.37244,-2.34313,-2.31379,-2.28443,-2.25505,-2.22565,-2.19622,-2.16677,-2.1373,-2.1078,-2.07827,-2.04871,-2.01913,-1.98952,-1.95987,-1.93019,-1.90048,-1.87074,-1.84096,-1.81115,-1.78129,-1.75141,-1.72148,-1.69151,-1.66151,-1.63146,-1.60138,-1.57125,-1.54109,-1.51088,-1.48063,-1.45034,-1.42002,-1.38965,-1.35924,-1.3288,-1.29833,-1.26782,-1.23728,-1.20671,-1.17612,-1.1455,-1.11487,-1.08423,-1.05359,-1.02294,-0.992301,-0.961677,-0.931076,-0.900507,-0.869981,-0.839509,-0.809101,-0.778771,-0.748531,-0.718396,-0.68838,-0.658499,-0.628768,-0.599204,-0.569823,-0.540644,-0.511684,-0.482959,-0.454488,-0.426287,-0.398375,-0.370768,-0.343482,-0.316533,-0.289937,-0.263707,-0.237857,-0.2124,-0.187347,-0.162709,-0.138496,-0.114715,-0.0913754,-0.0684825,-0.0460419,-0.024058,-0.00253421,0.0185268,0.0391234,0.0592546,0.0789201,0.0981203,0.116856,0.135129,0.152941,0.170293,0.18719,0.203632,0.219624,0.235167,0.250266,0.264923,0.279142,0.292925,0.306277,0.319201,0.331699,0.343776,0.355434,0.366679,0.377513,0.38794,0.397965,0.407592,0.416826,0.425672,0.434134,0.44222,0.449934,0.457284,0.464276,0.470917,0.477215,0.483178,0.488814,0.494133,0.499142,0.503851,0.50827,0.512408,0.516275,0.51988,0.523234,0.526346,0.529226,0.531884,0.534328,0.536567,0.538611,0.540466,0.542142,0.543645,0.544983,0.54616,0.547183,0.548056,0.548783,0.549368,0.549812,0.550118,0.550285,0.550313,0.550201,0.549945,0.549542,0.548987,0.548273,0.547392,0.546334,0.545089,0.543643,0.541982,0.540089,0.537946,0.535529,0.532818,0.529784,0.526398,0.52263,0.518443,0.513799,0.508656,0.502968,0.496686,0.489756,0.48212,0.473717,0.464481,0.454342,0.443227,0.431057,0.417752,0.403228,0.387399,0.370179,0.351479,0.331213,0.309298,0.285652,0.260202,0.232882,0.203637,0.172423,0.139213,0.103997,0.066784,0.0276042,-0.0134904,-0.0564247,-0.101105,-0.147416},
	{-2.48372,-2.45445,-2.42516,-2.39585,-2.36652,-2.33718,-2.30781,-2.27842,-2.249,-2.21957,-2.1901,-2.16062,-2.1311,-2.10156,-2.07198,-2.04238,-2.01275,-1.98308,-1.95338,-1.92365,-1.89388,-1.86407,-1.83422,-1.80434,-1.77442,-1.74446,-1.71445,-1.68441,-1.65432,-1.62418,-1.59401,-1.56378,-1.53352,-1.50321,-1.47285,-1.44245,-1.41201,-1.38152,-1.35099,-1.32042,-1.28981,-1.25916,-1.22848,-1.19776,-1.16702,-1.13625,-1.10546,-1.07465,-1.04384,-1.01302,-0.982199,-0.951392,-0.920605,-0.889847,-0.859127,-0.828458,-0.797849,-0.767315,-0.736869,-0.706524,-0.676297,-0.646201,-0.616254,-0.586472,-0.556873,-0.527473,-0.49829,-0.469343,-0.440649,-0.412225,-0.384089,-0.356258,-0.328749,-0.301578,-0.27476,-0.24831,-0.222241,-0.196567,-0.171298,-0.146447,-0.122023,-0.0980342,-0.0744892,-0.0513945,-0.0287558,-0.00657776,0.0151357,0.0363816,0.0571578,0.0774631,0.0972966,0.116658,0.135549,0.153969,0.17192,0.189404,0.206423,0.222978,0.239074,0.254711,0.269894,0.284624,0.298905,0.31274,0.326132,0.339084,0.3516,0.363683,0.375336,0.386564,0.397371,0.40776,0.417737,0.427305,0.436471,0.44524,0.453618,0.46161,0.469224,0.476467,0.483346,0.48987,0.496045,0.501882,0.50739,0.512576,0.517452,0.522026,0.52631,0.530312,0.534043,0.537514,0.540734,0.543713,0.546462,0.548989,0.551304,0.553416,0.555334,0.557065,0.558616,0.559996,0.561209,0.562263,0.56316,0.563907,0.564505,0.564957,0.565265,0.56543,0.565449,0.565323,0.565047,0.564619,0.564031,0.563279,0.562353,0.561243,0.559938,0.558424,0.556687,0.554709,0.55247,0.549948,0.54712,0.543957,0.540431,0.536507,0.53215,0.527321,0.521975,0.516067,0.509546,0.502357,0.494441,0.485735,0.476173,0.465684,0.454193,0.441622,0.427889,0.412911,0.396601,0.378873,0.359639,0.338816,0.316318,0.29207,0.265999,0.238044,0.208151,0.176284,0.142418,0.10655,0.0686942,0.0288874,-0.012813,-0.0563261,-0.101554,-0.148376},
	{-2.50661,-2.47733,-2.44804,-2.41872,-2.38939,-2.36003,-2.33066,-2.30126,-2.27183,-2.24238,-2.21291,-2.18341,-2.15388,-2.12432,-2.09473,-2.06511,-2.03546,-2.00577,-1.97605,-1.94629,-1.9165,-1.88667,-1.85679,-1.82688,-1.79693,-1.76693,-1.73689,-1.7068,-1.67667,-1.64649,-1.61626,-1.58599,-1.55567,-1.5253,-1.49488,-1.46441,-1.43389,-1.40333,-1.37271,-1.34205,-1.31135,-1.2806,-1.24981,-1.21898,-1.18811,-1.15721,-1.12627,-1.09532,-1.06434,-1.03335,-1.00235,-0.971348,-0.940353,-0.909374,-0.87842,-0.847502,-0.816629,-0.785815,-0.755073,-0.724415,-0.693857,-0.663413,-0.633099,-0.602933,-0.57293,-0.543108,-0.513485,-0.484079,-0.454907,-0.425989,-0.397341,-0.368982,-0.340928,-0.313197,-0.285806,-0.258768,-0.2321,-0.205816,-0.179928,-0.154449,-0.12939,-0.10476,-0.0805702,-0.0568272,-0.0335385,-0.0107101,0.0116528,0.033546,0.0549663,0.075911,0.0963784,0.116367,0.135877,0.154908,0.173461,0.191536,0.209136,0.22626,0.242912,0.259094,0.274807,0.290055,0.304839,0.319163,0.33303,0.346442,0.359402,0.371915,0.383984,0.395612,0.406803,0.417563,0.427894,0.437803,0.447295,0.456375,0.465049,0.473324,0.481207,0.488705,0.495826,0.502578,0.50897,0.515011,0.52071,0.526076,0.531121,0.535853,0.540284,0.544423,0.548282,0.55187,0.555198,0.558278,0.561117,0.563728,0.566118,0.568298,0.570276,0.572061,0.573659,0.575079,0.576326,0.577407,0.578326,0.579088,0.579696,0.580151,0.580457,0.580612,0.580616,0.580468,0.580165,0.579701,0.579072,0.57827,0.577286,0.57611,0.57473,0.573133,0.571301,0.569217,0.566862,0.564211,0.561239,0.557919,0.55422,0.550107,0.545543,0.540487,0.534895,0.528719,0.521907,0.514402,0.506145,0.497071,0.487112,0.476197,0.464249,0.45119,0.436937,0.421406,0.404512,0.386167,0.366286,0.344785,0.321582,0.296604,0.269782,0.241058,0.210384,0.177728,0.143074,0.106424,0.0677986,0.0272435,-0.0151764,-0.0593758,-0.105249,-0.152669},
	{-2.529,-2.49971,-2.47041,-2.44109,-2.41175,-2.38238,-2.353,-2.32359,-2.29415,-2.26469,-2.23521,-2.20569,-2.17615,-2.14658,-2.11698,-2.08734,-2.05767,-2.02796,-1.99822,-1.96845,-1.93863,-1.90877,-1.87887,-1.84893,-1.81895,-1.78892,-1.75884,-1.72872,-1.69854,-1.66832,-1.63805,-1.60773,-1.57735,-1.54693,-1.51645,-1.48591,-1.45533,-1.42469,-1.394,-1.36325,-1.33246,-1.30161,-1.27072,-1.23977,-1.20879,-1.17776,-1.1467,-1.1156,-1.08447,-1.05331,-1.02214,-0.990952,-0.959761,-0.928572,-0.897395,-0.86624,-0.835116,-0.804035,-0.77301,-0.742053,-0.711179,-0.680402,-0.649738,-0.619202,-0.588812,-0.558585,-0.528538,-0.498689,-0.469057,-0.43966,-0.410516,-0.381644,-0.353061,-0.324786,-0.296834,-0.269223,-0.241969,-0.215086,-0.18859,-0.162493,-0.136807,-0.111545,-0.086717,-0.0623319,-0.0383984,-0.0149238,0.00808539,0.0306237,0.0526866,0.0742702,0.0953718,0.115989,0.13612,0.155765,0.174922,0.193593,0.211776,0.229474,0.246688,0.263419,0.279668,0.295439,0.310733,0.325552,0.339899,0.353777,0.367189,0.380139,0.392628,0.404663,0.416245,0.427381,0.438073,0.448328,0.458151,0.467547,0.476523,0.485086,0.493243,0.501001,0.508368,0.515354,0.521966,0.528215,0.534109,0.53966,0.544876,0.54977,0.554351,0.558631,0.56262,0.566329,0.569769,0.572951,0.575885,0.578581,0.58105,0.5833,0.585342,0.587183,0.588831,0.590294,0.591578,0.592689,0.593633,0.594414,0.595034,0.595497,0.595804,0.595955,0.59595,0.595786,0.59546,0.594968,0.594304,0.593461,0.592428,0.591196,0.589753,0.588083,0.58617,0.583996,0.581539,0.578777,0.575682,0.572227,0.568379,0.564104,0.559362,0.554113,0.548311,0.541906,0.534846,0.527073,0.518527,0.509142,0.498849,0.487576,0.475246,0.461779,0.447093,0.431104,0.413725,0.394872,0.374458,0.352402,0.328625,0.303053,0.275622,0.246277,0.214976,0.181688,0.146404,0.109132,0.069897,0.0287505,-0.0142372,-0.0589758,-0.105353,-0.153246},
	{-2.55089,-2.5216,-2.4923,-2.46297,-2.43362,-2.40425,-2.37485,-2.34543,-2.31599,-2.28652,-2.25703,-2.2275,-2.19795,-2.16836,-2.13874,-2.10909,-2.07941,-2.04968,-2.01992,-1.99013,-1.96029,-1.93041,-1.90048,-1.87051,-1.8405,-1.81044,-1.78033,-1.75017,-1.71996,-1.6897,-1.65939,-1.62902,-1.59859,-1.56811,-1.53758,-1.50698,-1.47633,-1.44562,-1.41485,-1.38403,-1.35315,-1.32221,-1.29122,-1.26017,-1.22908,-1.19793,-1.16674,-1.1355,-1.10423,-1.07292,-1.04158,-1.01022,-0.978837,-0.947449,-0.91606,-0.884679,-0.853315,-0.82198,-0.790685,-0.759443,-0.728267,-0.697172,-0.666171,-0.635282,-0.60452,-0.573903,-0.543447,-0.513172,-0.483095,-0.453236,-0.423611,-0.394241,-0.365144,-0.336338,-0.30784,-0.279669,-0.251841,-0.224372,-0.197278,-0.170573,-0.144271,-0.118385,-0.092926,-0.0679053,-0.0433325,-0.0192163,0.00443544,0.027616,0.0503196,0.0725412,0.0942767,0.115523,0.136277,0.156536,0.176301,0.195569,0.214341,0.232617,0.250396,0.267681,0.284472,0.300771,0.316579,0.331899,0.346733,0.361083,0.374953,0.388344,0.401261,0.413707,0.425687,0.437203,0.448262,0.458868,0.469027,0.478745,0.488028,0.496883,0.505317,0.513339,0.520957,0.528179,0.535015,0.541474,0.547567,0.553304,0.558695,0.563752,0.568485,0.572906,0.577027,0.580857,0.584409,0.587693,0.590721,0.593503,0.596049,0.598368,0.600471,0.602366,0.604061,0.605564,0.606882,0.60802,0.608984,0.609778,0.610405,0.610869,0.611169,0.611308,0.611283,0.611092,0.610733,0.6102,0.609487,0.608586,0.607488,0.606181,0.604653,0.602888,0.600869,0.598577,0.595989,0.593082,0.589829,0.586198,0.582159,0.577674,0.572703,0.567205,0.561131,0.554432,0.547053,0.538935,0.530016,0.52023,0.509507,0.497772,0.484948,0.470955,0.45571,0.439129,0.421125,0.401615,0.380514,0.357743,0.333223,0.306885,0.27867,0.248526,0.216416,0.182317,0.146226,0.108157,0.0681436,0.0262446,-0.0174622,-0.0628808,-0.109893,-0.158372},
	{-2.57232,-2.54303,-2.51371,-2.48438,-2.45502,-2.42565,-2.39625,-2.36682,-2.33737,-2.30789,-2.27838,-2.24885,-2.21928,-2.18968,-2.16005,-2.13039,-2.10068,-2.07095,-2.04117,-2.01135,-1.98149,-1.95159,-1.92164,-1.89165,-1.86161,-1.83152,-1.80138,-1.77119,-1.74094,-1.71064,-1.68029,-1.64988,-1.6194,-1.58887,-1.55828,-1.52763,-1.49692,-1.46614,-1.4353,-1.4044,-1.37344,-1.34242,-1.31133,-1.28019,-1.24898,-1.21772,-1.18641,-1.15504,-1.12363,-1.09217,-1.06068,-1.02915,-0.997591,-0.966013,-0.934422,-0.902825,-0.871233,-0.839655,-0.808103,-0.776588,-0.745123,-0.713722,-0.682399,-0.65117,-0.620051,-0.589059,-0.55821,-0.527523,-0.497017,-0.466709,-0.436619,-0.406766,-0.377169,-0.347846,-0.318816,-0.290098,-0.261708,-0.233665,-0.205984,-0.178682,-0.151773,-0.12527,-0.0991885,-0.0735388,-0.0483324,-0.0235795,0.000711068,0.024531,0.0478733,0.0707316,0.0931006,0.114976,0.136353,0.15723,0.177605,0.197474,0.216837,0.235694,0.254045,0.271888,0.289226,0.306059,0.322388,0.338214,0.353541,0.368369,0.382702,0.396542,0.409892,0.422756,0.435139,0.447043,0.458474,0.469437,0.479938,0.489983,0.499578,0.508731,0.517448,0.525739,0.533612,0.541075,0.54814,0.554814,0.56111,0.567037,0.572607,0.577831,0.58272,0.587287,0.591542,0.595497,0.599165,0.602555,0.605681,0.608551,0.611178,0.613571,0.615739,0.617693,0.619439,0.620987,0.622342,0.623512,0.624501,0.625314,0.625955,0.626426,0.626728,0.626862,0.626827,0.62662,0.626237,0.625675,0.624926,0.623983,0.622834,0.62147,0.619876,0.618037,0.615935,0.61355,0.61086,0.607839,0.604461,0.600694,0.596504,0.591855,0.586706,0.581014,0.574729,0.567802,0.560176,0.551792,0.542588,0.532495,0.521443,0.509357,0.496159,0.48177,0.466105,0.44908,0.430612,0.410614,0.389006,0.365707,0.340644,0.313748,0.284963,0.254241,0.22155,0.186869,0.150202,0.111567,0.071002,0.028571,-0.0156435,-0.0615416,-0.109002,-0.157896},
	{-2.59331,-2.56401,-2.53469,-2.50535,-2.47599,-2.4466,-2.41719,-2.38776,-2.3583,-2.32881,-2.2993,-2.26975,-2.24017,-2.21056,-2.18092,-2.15124,-2.12153,-2.09177,-2.06198,-2.03214,-2.00227,-1.97234,-1.94237,-1.91236,-1.88229,-1.85218,-1.82201,-1.79178,-1.7615,-1.73117,-1.70077,-1.67032,-1.6398,-1.60923,-1.57858,-1.54788,-1.5171,-1.48627,-1.45536,-1.42439,-1.39335,-1.36224,-1.33107,-1.29983,-1.26852,-1.23715,-1.20572,-1.17424,-1.14269,-1.11109,-1.07945,-1.04776,-1.01603,-0.984273,-0.952489,-0.920687,-0.888876,-0.857066,-0.825267,-0.793491,-0.76175,-0.730056,-0.698424,-0.666869,-0.635407,-0.604053,-0.572826,-0.541742,-0.51082,-0.48008,-0.449539,-0.419217,-0.389134,-0.359308,-0.32976,-0.300507,-0.271568,-0.242962,-0.214706,-0.186817,-0.15931,-0.1322,-0.105503,-0.0792317,-0.053398,-0.0280134,-0.00308855,0.0213672,0.0453455,0.0688387,0.0918402,0.114344,0.136346,0.157841,0.178827,0.1993,0.219258,0.2387,0.257624,0.276031,0.29392,0.311291,0.328146,0.344484,0.360308,0.37562,0.390422,0.404716,0.418505,0.431793,0.444583,0.456881,0.468689,0.480015,0.490862,0.501238,0.51115,0.520604,0.529609,0.538172,0.546303,0.554011,0.561306,0.568198,0.574699,0.580818,0.586568,0.59196,0.597006,0.601718,0.606108,0.610188,0.61397,0.617465,0.620686,0.623643,0.626347,0.62881,0.631039,0.633046,0.634838,0.636424,0.63781,0.639003,0.640009,0.640831,0.641474,0.641939,0.642229,0.642343,0.64228,0.642037,0.641611,0.640996,0.640186,0.639171,0.637942,0.636486,0.63479,0.632835,0.630606,0.628079,0.625232,0.62204,0.618472,0.614497,0.61008,0.605183,0.599764,0.593777,0.587173,0.5799,0.5719,0.563113,0.553473,0.542912,0.531359,0.518737,0.504968,0.48997,0.473661,0.455956,0.436771,0.416023,0.393631,0.369518,0.343614,0.315855,0.286189,0.254573,0.220981,0.185402,0.147845,0.108337,0.0669256,0.0236811,-0.0213056,-0.0679284,-0.116061,-0.165569},
	{-2.61386,-2.58456,-2.55523,-2.52589,-2.49652,-2.46713,-2.43771,-2.40827,-2.3788,-2.34931,-2.31979,-2.29023,-2.26064,-2.23102,-2.20137,-2.17168,-2.14195,-2.11218,-2.08237,-2.05252,-2.02262,-1.99268,-1.96269,-1.93265,-1.90256,-1.87242,-1.84222,-1.81197,-1.78166,-1.75129,-1.72086,-1.69036,-1.65981,-1.62918,-1.59849,-1.56773,-1.5369,-1.50601,-1.47504,-1.444,-1.41288,-1.3817,-1.35044,-1.31911,-1.28771,-1.25624,-1.2247,-1.19309,-1.16142,-1.12969,-1.0979,-1.06606,-1.03417,-1.00224,-0.970269,-0.938271,-0.906251,-0.874219,-0.842184,-0.810157,-0.778151,-0.746176,-0.714248,-0.682379,-0.650586,-0.618885,-0.587292,-0.555825,-0.524502,-0.493342,-0.462364,-0.431588,-0.401032,-0.370717,-0.340663,-0.310888,-0.281413,-0.252255,-0.223435,-0.194968,-0.166873,-0.139165,-0.111861,-0.0849747,-0.0585197,-0.0325089,-0.00695406,0.018134,0.0427456,0.0668718,0.0905048,0.113638,0.136264,0.158379,0.179978,0.201057,0.221612,0.241643,0.261146,0.28012,0.298565,0.31648,0.333865,0.350721,0.367049,0.382851,0.398127,0.412881,0.427115,0.440833,0.454038,0.466734,0.478926,0.490619,0.50182,0.512533,0.522767,0.532528,0.541825,0.550666,0.55906,0.567017,0.574548,0.581663,0.588372,0.594688,0.600622,0.606186,0.611393,0.616254,0.620783,0.624991,0.628891,0.632495,0.635815,0.638863,0.641649,0.644185,0.64648,0.648544,0.650386,0.652015,0.653437,0.654659,0.655686,0.656524,0.657175,0.657643,0.657928,0.65803,0.657949,0.657681,0.657222,0.656567,0.655709,0.654639,0.653345,0.651816,0.650036,0.647989,0.645656,0.643015,0.640041,0.636709,0.632987,0.628844,0.624244,0.619147,0.61351,0.607286,0.600427,0.592876,0.584578,0.575469,0.565484,0.554554,0.542605,0.529562,0.515345,0.499873,0.483063,0.464831,0.445094,0.423769,0.400778,0.376045,0.349503,0.321091,0.290761,0.258476,0.224213,0.187966,0.14975,0.109598,0.0675647,0.0237241,-0.0218275,-0.0689794,-0.117602,-0.167559},
	{-2.634,-2.60469,-2.57536,-2.54601,-2.51664,-2.48724,-2.45782,-2.42838,-2.3989,-2.3694,-2.33987,-2.3103,-2.28071,-2.25108,-2.22141,-2.19171,-2.16197,-2.13218,-2.10236,-2.07249,-2.04258,-2.01262,-1.98261,-1.95255,-1.92244,-1.89227,-1.86205,-1.83177,-1.80143,-1.77102,-1.74056,-1.71003,-1.67943,-1.64876,-1.61803,-1.58722,-1.55634,-1.52538,-1.49435,-1.46324,-1.43206,-1.4008,-1.36946,-1.33804,-1.30655,-1.27498,-1.24334,-1.21162,-1.17983,-1.14797,-1.11605,-1.08406,-1.05201,-1.01992,-0.98777,-0.955584,-0.923364,-0.891119,-0.858858,-0.826592,-0.79433,-0.762086,-0.729872,-0.697702,-0.665591,-0.633555,-0.60161,-0.569773,-0.538062,-0.506496,-0.475094,-0.443876,-0.412862,-0.38207,-0.351523,-0.321239,-0.291239,-0.261542,-0.232168,-0.203134,-0.174461,-0.146164,-0.11826,-0.0907662,-0.0636965,-0.0370652,-0.0108853,0.0148311,0.0400728,0.0648298,0.0890927,0.112853,0.136105,0.15884,0.181053,0.20274,0.223896,0.244518,0.264603,0.284148,0.303153,0.321617,0.339538,0.356916,0.373753,0.39005,0.405806,0.421026,0.435711,0.449864,0.463489,0.476589,0.48917,0.501236,0.512794,0.52385,0.53441,0.544483,0.554076,0.563199,0.571861,0.580071,0.58784,0.59518,0.602102,0.608616,0.614737,0.620475,0.625844,0.630856,0.635524,0.639862,0.64388,0.647593,0.651012,0.654149,0.657016,0.659624,0.661982,0.664102,0.665992,0.66766,0.669114,0.67036,0.671405,0.672253,0.672906,0.673369,0.673641,0.673724,0.673614,0.67331,0.672808,0.6721,0.67118,0.670038,0.668664,0.667042,0.665159,0.662997,0.660536,0.657753,0.654623,0.651119,0.64721,0.642861,0.638036,0.632694,0.626791,0.620279,0.613107,0.605219,0.596556,0.587055,0.57665,0.565268,0.552838,0.539281,0.524519,0.508469,0.491049,0.472175,0.451765,0.429739,0.406019,0.380533,0.353217,0.324015,0.292882,0.259786,0.224712,0.187662,0.148656,0.107733,0.0649568,0.0204074,-0.0258123,-0.0735873,-0.122783,-0.173264},
	{-2.65375,-2.62444,-2.5951,-2.56575,-2.53637,-2.50697,-2.47754,-2.44809,-2.41861,-2.3891,-2.35956,-2.32998,-2.30038,-2.27074,-2.24106,-2.21135,-2.1816,-2.1518,-2.12196,-2.09208,-2.06215,-2.03217,-2.00215,-1.97207,-1.94193,-1.91174,-1.8815,-1.85119,-1.82082,-1.79039,-1.75989,-1.72932,-1.69869,-1.66798,-1.6372,-1.60634,-1.57541,-1.5444,-1.51331,-1.48214,-1.45089,-1.41956,-1.38814,-1.35665,-1.32507,-1.2934,-1.26166,-1.22983,-1.19793,-1.16595,-1.13389,-1.10177,-1.06957,-1.03731,-1.005,-0.972633,-0.940221,-0.907773,-0.875295,-0.842798,-0.810292,-0.777789,-0.7453,-0.71284,-0.680423,-0.648064,-0.615778,-0.583583,-0.551497,-0.519539,-0.487726,-0.456079,-0.424618,-0.393364,-0.362335,-0.331554,-0.301041,-0.270816,-0.240899,-0.211309,-0.182066,-0.153189,-0.124694,-0.0966002,-0.0689224,-0.0416762,-0.0148762,0.0114644,0.037333,0.0627182,0.0876094,0.111997,0.135873,0.159229,0.182058,0.204354,0.226113,0.247329,0.268,0.288121,0.30769,0.326706,0.345168,0.363075,0.380426,0.397223,0.413466,0.429157,0.444298,0.458893,0.472943,0.486454,0.499429,0.511874,0.523795,0.535198,0.546091,0.55648,0.566375,0.575784,0.584718,0.593186,0.601198,0.608767,0.615905,0.622622,0.628932,0.634848,0.640383,0.645549,0.65036,0.654828,0.658968,0.662792,0.666313,0.669542,0.672492,0.675173,0.677598,0.679775,0.681714,0.683423,0.684911,0.686184,0.687248,0.688108,0.688766,0.689226,0.689488,0.689553,0.689418,0.689081,0.688538,0.687781,0.686803,0.685594,0.684142,0.682434,0.680454,0.678183,0.675601,0.672684,0.669407,0.665741,0.661655,0.657112,0.652076,0.646504,0.640352,0.63357,0.626106,0.617903,0.608901,0.599037,0.588241,0.576443,0.563568,0.549539,0.534275,0.517696,0.499719,0.48026,0.459239,0.436576,0.412198,0.386033,0.358022,0.328111,0.296261,0.262446,0.226653,0.188892,0.149189,0.10759,0.0641631,0.0189963,-0.0278061,-0.0761206,-0.125816,-0.176749},
	{-2.67311,-2.6438,-2.61446,-2.5851,-2.55572,-2.52631,-2.49688,-2.46742,-2.43793,-2.40842,-2.37887,-2.34929,-2.31968,-2.29003,-2.26034,-2.23062,-2.20085,-2.17104,-2.14119,-2.1113,-2.08135,-2.05136,-2.02132,-1.99122,-1.96106,-1.93085,-1.90058,-1.87025,-1.83985,-1.80939,-1.77886,-1.74826,-1.71759,-1.68684,-1.65602,-1.62512,-1.59414,-1.56308,-1.53193,-1.50071,-1.46939,-1.43799,-1.4065,-1.37493,-1.34326,-1.31151,-1.27967,-1.24774,-1.21573,-1.18363,-1.15145,-1.11919,-1.08685,-1.05444,-1.02197,-0.989427,-0.956831,-0.924186,-0.8915,-0.858782,-0.826041,-0.793289,-0.760536,-0.727797,-0.695084,-0.662413,-0.629799,-0.597259,-0.56481,-0.53247,-0.500259,-0.468197,-0.436302,-0.404596,-0.373099,-0.341833,-0.310818,-0.280076,-0.249626,-0.219491,-0.189689,-0.16024,-0.131163,-0.102476,-0.0741969,-0.0463421,-0.0189273,0.00803291,0.0345246,0.0605351,0.0860524,0.111066,0.135565,0.159542,0.182987,0.205895,0.228258,0.250071,0.27133,0.29203,0.312168,0.331741,0.350748,0.369187,0.387058,0.40436,0.421094,0.437262,0.452865,0.467905,0.482387,0.496312,0.509687,0.522516,0.534805,0.54656,0.557789,0.568499,0.578699,0.588399,0.597607,0.606335,0.614594,0.622395,0.62975,0.636672,0.643173,0.649268,0.654968,0.660288,0.665241,0.669841,0.674102,0.678035,0.681655,0.684973,0.688003,0.690755,0.69324,0.695469,0.697452,0.699197,0.700711,0.702002,0.703076,0.703936,0.704588,0.705032,0.70527,0.705301,0.705124,0.704736,0.70413,0.703301,0.702239,0.700935,0.699376,0.697548,0.695434,0.693014,0.690267,0.687169,0.683692,0.679808,0.675482,0.670678,0.665357,0.659476,0.652989,0.645843,0.637986,0.62936,0.619902,0.609547,0.598226,0.585866,0.572393,0.557727,0.541788,0.524495,0.505766,0.485519,0.463675,0.440155,0.41489,0.387812,0.358865,0.328003,0.295191,0.260411,0.223658,0.18495,0.144321,0.101826,0.0575425,0.0115655,-0.0359943,-0.085008,-0.135341,-0.186851},
	{-2.69211,-2.66279,-2.63345,-2.60409,-2.5747,-2.54529,-2.51585,-2.48639,-2.4569,-2.42737,-2.39782,-2.36823,-2.33861,-2.30895,-2.27926,-2.24952,-2.21975,-2.18993,-2.16007,-2.13016,-2.1002,-2.07019,-2.04013,-2.01002,-1.97984,-1.94961,-1.91932,-1.88896,-1.85854,-1.82805,-1.79749,-1.76686,-1.73615,-1.70537,-1.67451,-1.64356,-1.61254,-1.58143,-1.55023,-1.51894,-1.48757,-1.4561,-1.42455,-1.3929,-1.36115,-1.32931,-1.29738,-1.26536,-1.23324,-1.20103,-1.16872,-1.13634,-1.10386,-1.07131,-1.03867,-1.00597,-0.973197,-0.940364,-0.907478,-0.874546,-0.841579,-0.808587,-0.77558,-0.742571,-0.709574,-0.676601,-0.64367,-0.610795,-0.577995,-0.545286,-0.512689,-0.480222,-0.447905,-0.415759,-0.383806,-0.352066,-0.320561,-0.289312,-0.25834,-0.227668,-0.197316,-0.167304,-0.137653,-0.108381,-0.0795071,-0.0510495,-0.0230248,0.00455082,0.0316623,0.0582954,0.0844371,0.110075,0.135198,0.159796,0.183859,0.20738,0.23035,0.252763,0.274613,0.295896,0.316607,0.336742,0.3563,0.375277,0.393673,0.411487,0.428719,0.445369,0.461441,0.476934,0.491854,0.506202,0.519983,0.533203,0.545866,0.55798,0.569553,0.580591,0.591103,0.601099,0.61059,0.619585,0.628097,0.636136,0.643717,0.65085,0.65755,0.66383,0.669704,0.675186,0.68029,0.68503,0.689419,0.693471,0.697199,0.700617,0.703738,0.706571,0.709131,0.711426,0.713466,0.715262,0.71682,0.718147,0.719251,0.720135,0.720803,0.721258,0.7215,0.72153,0.721345,0.720942,0.720316,0.71946,0.718366,0.717022,0.715416,0.713534,0.711359,0.70887,0.706046,0.702863,0.699292,0.695304,0.690866,0.685939,0.680485,0.67446,0.667816,0.660503,0.652465,0.643646,0.633981,0.623406,0.611852,0.599244,0.585509,0.570568,0.55434,0.536745,0.5177,0.497126,0.474942,0.451074,0.425452,0.39801,0.368694,0.337459,0.304275,0.269124,0.232004,0.192936,0.151957,0.109124,0.0645163,0.018231,-0.0296196,-0.0789056,-0.129492,-0.181236},
	{-2.71076,-2.68144,-2.65209,-2.62273,-2.59334,-2.56392,-2.53448,-2.50501,-2.47551,-2.44598,-2.41642,-2.38683,-2.3572,-2.32753,-2.29783,-2.26809,-2.2383,-2.20847,-2.1786,-2.14868,-2.1187,-2.08868,-2.0586,-2.02847,-1.99828,-1.96803,-1.93772,-1.90734,-1.87689,-1.84638,-1.81579,-1.78512,-1.75439,-1.72357,-1.69267,-1.66168,-1.63061,-1.59946,-1.56821,-1.53687,-1.50544,-1.47391,-1.44229,-1.41056,-1.37874,-1.34682,-1.3148,-1.28268,-1.25046,-1.21815,-1.18573,-1.15322,-1.12061,-1.08792,-1.05514,-1.02227,-0.989328,-0.956314,-0.923235,-0.890099,-0.856915,-0.823691,-0.790439,-0.75717,-0.723898,-0.690635,-0.657396,-0.624198,-0.591058,-0.557992,-0.525019,-0.492159,-0.459432,-0.426859,-0.39446,-0.362258,-0.330273,-0.298529,-0.267046,-0.235848,-0.204955,-0.17439,-0.144172,-0.114323,-0.0848617,-0.0558078,-0.0271791,0.0010069,0.0287338,0.055986,0.0827491,0.10901,0.134755,0.159973,0.184654,0.208788,0.232367,0.255381,0.277826,0.299694,0.320981,0.341682,0.361793,0.381313,0.400238,0.418568,0.436302,0.453441,0.469985,0.485937,0.501298,0.516073,0.530265,0.54388,0.556922,0.5694,0.581319,0.592688,0.603516,0.613812,0.623587,0.632851,0.641617,0.649897,0.657703,0.665048,0.671947,0.678412,0.684459,0.690101,0.695353,0.700229,0.704743,0.70891,0.712742,0.716254,0.719458,0.722366,0.72499,0.727341,0.729428,0.731262,0.73285,0.734199,0.735315,0.736204,0.736868,0.737311,0.737532,0.737533,0.737309,0.736858,0.736175,0.735252,0.734079,0.732646,0.73094,0.728946,0.726644,0.724016,0.721038,0.717685,0.713929,0.709737,0.705076,0.699908,0.694191,0.68788,0.680927,0.67328,0.664883,0.655677,0.645597,0.634578,0.622549,0.609436,0.595163,0.579652,0.562823,0.544594,0.524886,0.503617,0.480712,0.456097,0.429704,0.401472,0.371351,0.339302,0.305298,0.269329,0.2314,0.191538,0.149787,0.10621,0.0608943,0.0139429,-0.0345276,-0.0843841,-0.13549,-0.187702},
	{-2.72907,-2.69974,-2.6704,-2.64102,-2.61163,-2.58221,-2.55276,-2.52329,-2.49379,-2.46425,-2.43468,-2.40508,-2.37545,-2.34578,-2.31606,-2.28631,-2.25652,-2.22668,-2.19679,-2.16686,-2.13688,-2.10684,-2.07675,-2.0466,-2.01639,-1.98612,-1.95579,-1.92539,-1.89492,-1.86438,-1.83377,-1.80307,-1.7723,-1.74145,-1.71052,-1.67949,-1.64838,-1.61718,-1.58589,-1.5545,-1.52301,-1.49142,-1.45973,-1.42794,-1.39605,-1.36405,-1.33194,-1.29973,-1.26742,-1.235,-1.20247,-1.16984,-1.13711,-1.10428,-1.07135,-1.03834,-1.00523,-0.972042,-0.938777,-0.905444,-0.872049,-0.838603,-0.805114,-0.771595,-0.738056,-0.704512,-0.670977,-0.637466,-0.603996,-0.570583,-0.537246,-0.504005,-0.470878,-0.437888,-0.405055,-0.372401,-0.339948,-0.307718,-0.275735,-0.24402,-0.212596,-0.181486,-0.15071,-0.120291,-0.0902494,-0.0606053,-0.0313782,-0.00258656,0.0257517,0.0536199,0.081002,0.107883,0.13425,0.160088,0.185387,0.210136,0.234324,0.257943,0.280985,0.303442,0.325308,0.346579,0.367249,0.387316,0.406776,0.425628,0.44387,0.461502,0.478525,0.494941,0.510751,0.525958,0.540567,0.554583,0.568011,0.580857,0.593129,0.604835,0.615984,0.626586,0.636651,0.64619,0.655216,0.663741,0.671778,0.67934,0.686442,0.693098,0.699322,0.705129,0.710534,0.715551,0.720196,0.724481,0.728423,0.732034,0.735327,0.738315,0.741009,0.743422,0.745563,0.747442,0.749067,0.750445,0.751583,0.752486,0.753156,0.753598,0.75381,0.753794,0.753546,0.753063,0.752339,0.751367,0.750137,0.748638,0.746856,0.744775,0.742378,0.739643,0.736547,0.733064,0.729164,0.724816,0.719985,0.714631,0.708712,0.702184,0.694996,0.687096,0.678427,0.668929,0.658538,0.647186,0.634803,0.621314,0.606644,0.590714,0.573444,0.554754,0.534563,0.512793,0.489368,0.464217,0.437275,0.408483,0.377794,0.345172,0.310594,0.274055,0.235563,0.195151,0.152866,0.108778,0.0629749,0.015565,-0.033333,-0.083584,-0.13505,-0.187592},
	{-2.74705,-2.71772,-2.68837,-2.659,-2.6296,-2.60017,-2.57072,-2.54124,-2.51174,-2.4822,-2.45262,-2.42302,-2.39338,-2.3637,-2.33398,-2.30422,-2.27441,-2.24457,-2.21467,-2.18473,-2.15473,-2.12468,-2.09458,-2.06441,-2.03419,-2.0039,-1.97355,-1.94313,-1.91264,-1.88207,-1.85143,-1.82072,-1.78992,-1.75903,-1.72806,-1.697,-1.66585,-1.63461,-1.60327,-1.57183,-1.54029,-1.50864,-1.47689,-1.44504,-1.41307,-1.381,-1.34881,-1.31652,-1.28411,-1.25159,-1.21896,-1.18621,-1.15336,-1.1204,-1.08734,-1.05417,-1.02091,-0.987553,-0.95411,-0.920586,-0.886989,-0.853328,-0.819611,-0.785849,-0.752054,-0.718239,-0.684416,-0.650602,-0.616812,-0.583062,-0.549372,-0.51576,-0.482245,-0.448849,-0.415593,-0.382498,-0.349587,-0.316883,-0.284408,-0.252187,-0.220241,-0.188594,-0.157269,-0.126288,-0.0956736,-0.065446,-0.0356263,-0.00623434,0.0227108,0.051191,0.079189,0.106689,0.133675,0.160133,0.186049,0.211413,0.236212,0.260436,0.284077,0.307125,0.329575,0.351419,0.372652,0.393269,0.413268,0.432646,0.451401,0.469531,0.487038,0.503922,0.520186,0.535831,0.550862,0.565284,0.579101,0.59232,0.604949,0.616996,0.62847,0.639381,0.649739,0.659557,0.668845,0.677618,0.685887,0.693669,0.700975,0.707822,0.714224,0.720196,0.725753,0.730911,0.735684,0.740088,0.744136,0.747842,0.751221,0.754284,0.757045,0.759514,0.761702,0.763619,0.765272,0.766671,0.76782,0.768724,0.769388,0.769814,0.770002,0.769952,0.769661,0.769125,0.768337,0.767291,0.765977,0.764381,0.76249,0.760288,0.757756,0.754871,0.75161,0.747946,0.743848,0.739284,0.734217,0.728606,0.72241,0.715581,0.708068,0.699819,0.690774,0.680872,0.670049,0.658236,0.645362,0.631351,0.616128,0.599614,0.581729,0.562393,0.541528,0.519055,0.494903,0.469001,0.441288,0.411709,0.380222,0.346796,0.311413,0.274075,0.234796,0.193615,0.150587,0.105788,0.0593133,0.0112743,-0.0382058,-0.0889892,-0.140938,-0.193913},
	{-2.76471,-2.73538,-2.70603,-2.67665,-2.64725,-2.61782,-2.58837,-2.55889,-2.52937,-2.49983,-2.47025,-2.44064,-2.41099,-2.38131,-2.35158,-2.32181,-2.292,-2.26214,-2.23224,-2.20228,-2.17228,-2.14222,-2.1121,-2.08192,-2.05168,-2.02138,-1.99101,-1.96057,-1.93006,-1.89947,-1.86881,-1.83806,-1.80723,-1.77632,-1.74532,-1.71422,-1.68304,-1.65175,-1.62037,-1.58888,-1.55729,-1.52559,-1.49378,-1.46186,-1.42983,-1.39768,-1.36542,-1.33304,-1.30055,-1.26793,-1.2352,-1.20234,-1.16937,-1.13629,-1.10309,-1.06979,-1.03637,-1.00285,-0.969238,-0.935531,-0.901738,-0.867869,-0.833932,-0.799936,-0.765892,-0.731814,-0.697713,-0.663605,-0.629505,-0.595429,-0.561395,-0.527422,-0.49353,-0.459738,-0.426068,-0.392543,-0.359184,-0.326015,-0.293059,-0.260341,-0.227882,-0.195708,-0.163842,-0.132307,-0.101125,-0.0703207,-0.0399141,-0.00992681,0.0196209,0.0487096,0.0773207,0.105437,0.133041,0.160117,0.186652,0.212631,0.238042,0.262874,0.287116,0.310759,0.333794,0.356215,0.378015,0.399189,0.419732,0.439641,0.458914,0.477548,0.495545,0.512904,0.529627,0.545716,0.561175,0.576009,0.590221,0.60382,0.616813,0.629207,0.641011,0.652237,0.662894,0.672995,0.682552,0.691577,0.700085,0.70809,0.715607,0.72265,0.729235,0.735377,0.741092,0.746395,0.751303,0.755829,0.759989,0.763797,0.767266,0.770411,0.773244,0.775776,0.778018,0.77998,0.78167,0.783097,0.784266,0.785183,0.785851,0.786273,0.786449,0.786379,0.78606,0.785487,0.784654,0.783554,0.782176,0.780508,0.778535,0.77624,0.773603,0.770604,0.767216,0.763412,0.759161,0.75443,0.749181,0.743374,0.736964,0.729905,0.722146,0.71363,0.7043,0.694094,0.682946,0.670787,0.657546,0.643147,0.627515,0.61057,0.592233,0.572426,0.551071,0.52809,0.503413,0.476974,0.448711,0.418573,0.386522,0.352529,0.316583,0.278686,0.238859,0.197145,0.153603,0.108313,0.0613733,0.012899,-0.036985,-0.0881397,-0.140427,-0.193709},
	{-2.78207,-2.75274,-2.72338,-2.694,-2.6646,-2.63517,-2.60571,-2.57622,-2.54671,-2.51716,-2.48758,-2.45796,-2.42831,-2.39861,-2.36888,-2.33911,-2.30929,-2.27942,-2.24951,-2.21954,-2.18953,-2.15945,-2.12932,-2.09913,-2.06888,-2.03856,-2.00817,-1.97771,-1.94718,-1.91658,-1.88589,-1.85512,-1.82427,-1.79332,-1.76229,-1.73116,-1.69994,-1.66862,-1.63719,-1.60566,-1.57402,-1.54227,-1.5104,-1.47843,-1.44633,-1.41411,-1.38178,-1.34932,-1.31673,-1.28403,-1.25119,-1.21824,-1.18516,-1.15195,-1.11863,-1.08518,-1.05162,-1.01795,-0.984167,-0.950282,-0.916302,-0.882232,-0.848081,-0.813859,-0.779575,-0.745242,-0.710872,-0.676479,-0.642078,-0.607685,-0.573317,-0.538993,-0.504733,-0.470555,-0.436483,-0.402537,-0.36874,-0.335116,-0.301689,-0.268482,-0.23552,-0.202827,-0.170427,-0.138346,-0.106605,-0.07523,-0.0442426,-0.0136652,0.0164805,0.0461738,0.0753947,0.104124,0.132345,0.160039,0.187191,0.213786,0.23981,0.265251,0.290096,0.314336,0.337961,0.360962,0.383333,0.405067,0.426158,0.446603,0.466399,0.485543,0.504035,0.521874,0.539062,0.555601,0.571493,0.586744,0.601358,0.615342,0.628703,0.641449,0.653589,0.665135,0.676096,0.686484,0.696313,0.705596,0.714346,0.722579,0.730309,0.737552,0.744323,0.750638,0.756514,0.761965,0.767009,0.77166,0.775934,0.779845,0.783408,0.786636,0.789542,0.792138,0.794435,0.796443,0.798171,0.799627,0.800817,0.801747,0.80242,0.802839,0.803003,0.802914,0.802566,0.801957,0.80108,0.799926,0.798486,0.796745,0.79469,0.792303,0.789564,0.78645,0.782936,0.778994,0.774592,0.769697,0.764269,0.758268,0.751649,0.744365,0.736362,0.727586,0.717978,0.707474,0.696009,0.683514,0.669917,0.655142,0.639113,0.621753,0.602981,0.582721,0.560894,0.537427,0.512248,0.485295,0.456508,0.42584,0.393254,0.358725,0.322246,0.283822,0.243481,0.201266,0.157243,0.111494,0.0641205,0.015239,-0.0350217,-0.0865228,-0.139127,-0.192696},
	{-2.79914,-2.7698,-2.74044,-2.71106,-2.68165,-2.65222,-2.62276,-2.59327,-2.56375,-2.5342,-2.50461,-2.47499,-2.44533,-2.41563,-2.38589,-2.35611,-2.32628,-2.29641,-2.26649,-2.23651,-2.20649,-2.1764,-2.14626,-2.11606,-2.08579,-2.05546,-2.02505,-1.99458,-1.96403,-1.9334,-1.90269,-1.8719,-1.84102,-1.81005,-1.77899,-1.74783,-1.71657,-1.68521,-1.65375,-1.62217,-1.59049,-1.55869,-1.52677,-1.49473,-1.46257,-1.43029,-1.39788,-1.36535,-1.33268,-1.29989,-1.26696,-1.2339,-1.20072,-1.1674,-1.13395,-1.10037,-1.06667,-1.03284,-0.998902,-0.964847,-0.930684,-0.89642,-0.862063,-0.827621,-0.793105,-0.758525,-0.723894,-0.689225,-0.654532,-0.619831,-0.585139,-0.550473,-0.515854,-0.481301,-0.446835,-0.412478,-0.378254,-0.344185,-0.310295,-0.276609,-0.243152,-0.209949,-0.177024,-0.144404,-0.112111,-0.080172,-0.0486096,-0.0174476,0.0132914,0.0435852,0.0734126,0.102753,0.131588,0.159898,0.187667,0.214878,0.241516,0.267566,0.293017,0.317857,0.342074,0.365659,0.388604,0.410902,0.432546,0.453532,0.473856,0.493515,0.512507,0.530832,0.54849,0.565484,0.581815,0.597489,0.61251,0.626884,0.640618,0.653722,0.666203,0.678073,0.689343,0.700024,0.71013,0.719674,0.72867,0.737135,0.745082,0.752527,0.759488,0.765979,0.772018,0.777621,0.782803,0.787582,0.791972,0.795988,0.799646,0.802959,0.80594,0.808602,0.810955,0.813011,0.814778,0.816264,0.817476,0.81842,0.819098,0.819514,0.819669,0.81956,0.819186,0.818542,0.817621,0.816414,0.814912,0.813101,0.810965,0.808487,0.805647,0.802421,0.798784,0.794707,0.790157,0.7851,0.779498,0.773308,0.766485,0.758981,0.750742,0.741714,0.731836,0.721045,0.709275,0.696456,0.682515,0.667379,0.650971,0.633213,0.614026,0.593334,0.571059,0.54713,0.521477,0.494039,0.464758,0.433591,0.400502,0.365472,0.328494,0.28958,0.248759,0.206079,0.161608,0.115433,0.0676583,0.0184006,-0.0322088,-0.0840309,-0.136929,-0.190765},
	{-2.81592,-2.78658,-2.75722,-2.72783,-2.69843,-2.66899,-2.63952,-2.61003,-2.58051,-2.55095,-2.52136,-2.49173,-2.46207,-2.43237,-2.40262,-2.37283,-2.343,-2.31312,-2.28319,-2.25321,-2.22317,-2.19308,-2.16292,-2.13271,-2.10243,-2.07208,-2.04166,-2.01117,-1.98061,-1.94996,-1.91923,-1.88842,-1.85751,-1.82652,-1.79543,-1.76424,-1.73295,-1.70155,-1.67005,-1.63843,-1.6067,-1.57485,-1.54289,-1.51079,-1.47858,-1.44623,-1.41375,-1.38115,-1.3484,-1.31552,-1.2825,-1.24935,-1.21606,-1.18263,-1.14906,-1.11535,-1.08152,-1.04755,-1.01345,-0.979229,-0.94489,-0.910439,-0.875882,-0.841229,-0.806487,-0.771669,-0.736784,-0.701847,-0.66687,-0.63187,-0.596862,-0.561865,-0.526896,-0.491977,-0.457127,-0.42237,-0.387726,-0.353221,-0.318879,-0.284723,-0.250781,-0.217076,-0.183634,-0.150482,-0.117645,-0.085149,-0.0530179,-0.0212768,0.0100503,0.0409402,0.0713703,0.101319,0.130766,0.15969,0.188075,0.215901,0.243152,0.269814,0.295871,0.321312,0.346123,0.370295,0.393818,0.416683,0.438884,0.460415,0.481271,0.501449,0.520946,0.539761,0.557894,0.575347,0.592123,0.608224,0.623656,0.638425,0.652537,0.666002,0.678828,0.691026,0.702607,0.713584,0.723969,0.733777,0.743022,0.75172,0.759885,0.767535,0.774685,0.781353,0.787554,0.793307,0.798627,0.803531,0.808034,0.812153,0.815902,0.819295,0.822346,0.825068,0.827471,0.829566,0.831364,0.832871,0.834095,0.83504,0.835712,0.836111,0.836239,0.836095,0.835675,0.834974,0.833987,0.832703,0.831112,0.8292,0.826952,0.824348,0.821369,0.817989,0.814184,0.809922,0.805171,0.799895,0.794055,0.787609,0.780509,0.772706,0.764147,0.754775,0.744529,0.733346,0.721158,0.707896,0.693486,0.677854,0.660924,0.642618,0.622859,0.601569,0.578675,0.554106,0.527794,0.499681,0.469713,0.43785,0.404061,0.368331,0.330659,0.29106,0.249572,0.206247,0.161158,0.114396,0.0660707,0.0163021,-0.0347757,-0.0870263,-0.14031,-0.194493},
	{-2.83242,-2.80308,-2.77372,-2.74433,-2.71492,-2.68548,-2.65602,-2.62652,-2.59699,-2.56743,-2.53784,-2.50821,-2.47854,-2.44883,-2.41908,-2.38929,-2.35945,-2.32956,-2.29962,-2.26963,-2.23958,-2.20948,-2.17932,-2.14909,-2.1188,-2.08844,-2.05801,-2.0275,-1.99692,-1.96625,-1.93551,-1.90467,-1.87374,-1.84272,-1.81161,-1.78039,-1.74907,-1.71764,-1.6861,-1.65444,-1.62267,-1.59078,-1.55876,-1.52662,-1.49434,-1.46194,-1.42939,-1.39671,-1.36389,-1.33093,-1.29783,-1.26458,-1.23119,-1.19765,-1.16397,-1.13014,-1.09617,-1.06206,-1.02781,-0.993433,-0.958924,-0.924291,-0.889541,-0.854682,-0.819723,-0.784672,-0.749542,-0.714344,-0.679092,-0.643801,-0.608486,-0.573166,-0.537857,-0.50258,-0.467356,-0.432206,-0.397154,-0.362222,-0.327435,-0.292819,-0.258399,-0.2242,-0.19025,-0.156574,-0.1232,-0.0901524,-0.0574586,-0.0251439,0.00676667,0.0382485,0.0692779,0.0998318,0.129888,0.159426,0.188425,0.216866,0.244732,0.272006,0.298672,0.324716,0.350124,0.374886,0.39899,0.422427,0.44519,0.46727,0.488664,0.509366,0.529373,0.548684,0.567299,0.585218,0.602443,0.618978,0.634827,0.649996,0.664493,0.678325,0.691502,0.704034,0.715933,0.727211,0.737882,0.747959,0.757458,0.766394,0.774783,0.782642,0.789988,0.796837,0.803207,0.809115,0.814578,0.819613,0.824236,0.828463,0.832309,0.835789,0.838916,0.841704,0.844165,0.846308,0.848144,0.849681,0.850926,0.851884,0.852559,0.852954,0.853069,0.852903,0.852452,0.851713,0.850678,0.849337,0.847679,0.845691,0.843357,0.840657,0.83757,0.834071,0.830135,0.82573,0.820824,0.815379,0.809356,0.802712,0.795399,0.787368,0.778564,0.768931,0.758407,0.746928,0.734426,0.720832,0.706073,0.690074,0.672759,0.65405,0.633872,0.612149,0.588807,0.563776,0.536992,0.508398,0.477942,0.445588,0.411307,0.375087,0.33693,0.296856,0.254905,0.211133,0.165616,0.118448,0.0697408,0.0196161,-0.0317912,-0.0843444,-0.137905,-0.19234},
	{-2.84866,-2.81932,-2.78996,-2.76057,-2.73115,-2.70171,-2.67224,-2.64274,-2.61321,-2.58365,-2.55405,-2.52441,-2.49474,-2.46503,-2.43527,-2.40547,-2.37563,-2.34574,-2.31579,-2.28579,-2.25574,-2.22563,-2.19545,-2.16522,-2.13491,-2.10454,-2.0741,-2.04358,-2.01298,-1.9823,-1.95153,-1.92067,-1.88973,-1.85868,-1.82754,-1.79629,-1.76494,-1.73348,-1.70191,-1.67021,-1.6384,-1.60647,-1.5744,-1.54221,-1.50988,-1.47741,-1.44481,-1.41206,-1.37917,-1.34613,-1.31294,-1.2796,-1.24611,-1.21247,-1.17868,-1.14473,-1.11064,-1.07639,-1.042,-1.00747,-0.972792,-0.937984,-0.903047,-0.867989,-0.832818,-0.797543,-0.762174,-0.726724,-0.691204,-0.65563,-0.620017,-0.584381,-0.548741,-0.513116,-0.477527,-0.441994,-0.406541,-0.371192,-0.33597,-0.300902,-0.266013,-0.231329,-0.196878,-0.162687,-0.128782,-0.0951914,-0.0619415,-0.0290592,0.00342929,0.0354982,0.0671225,0.098278,0.128941,0.159089,0.188702,0.217757,0.246237,0.274122,0.301397,0.328045,0.354053,0.379406,0.404094,0.428106,0.451432,0.474066,0.496,0.51723,0.537752,0.557563,0.576663,0.595051,0.61273,0.629702,0.645972,0.661546,0.67643,0.690632,0.704163,0.717031,0.72925,0.740831,0.751788,0.762135,0.771888,0.781063,0.789675,0.797742,0.805281,0.812309,0.818844,0.824903,0.830505,0.835664,0.8404,0.844727,0.848662,0.852219,0.855412,0.858254,0.860758,0.862934,0.864792,0.866341,0.867586,0.868534,0.869189,0.869552,0.869624,0.869404,0.868888,0.868071,0.866945,0.865501,0.863726,0.861607,0.859126,0.856264,0.852998,0.849304,0.845153,0.840514,0.835352,0.829631,0.823309,0.816342,0.808681,0.800276,0.791071,0.781008,0.770024,0.758056,0.745034,0.730889,0.715546,0.698932,0.680969,0.661583,0.640697,0.618237,0.594132,0.568313,0.540721,0.5113,0.480005,0.446802,0.411671,0.374603,0.335608,0.294713,0.251963,0.207423,0.161174,0.113317,0.0639679,0.0132515,-0.0386949,-0.0917346,-0.145731,-0.200553},
	{-2.86464,-2.8353,-2.80593,-2.77654,-2.74713,-2.71768,-2.68821,-2.65871,-2.62917,-2.59961,-2.57001,-2.54037,-2.51069,-2.48097,-2.45121,-2.42141,-2.39156,-2.36166,-2.33171,-2.3017,-2.27164,-2.24152,-2.21134,-2.18109,-2.15078,-2.1204,-2.08994,-2.0594,-2.02879,-1.99809,-1.96731,-1.93643,-1.90546,-1.8744,-1.84323,-1.81196,-1.78058,-1.74909,-1.71748,-1.68575,-1.6539,-1.62193,-1.58982,-1.55758,-1.52519,-1.49267,-1.46001,-1.4272,-1.39423,-1.36112,-1.32785,-1.29443,-1.26084,-1.2271,-1.1932,-1.15914,-1.12492,-1.09055,-1.05601,-1.02133,-0.986497,-0.951518,-0.916401,-0.88115,-0.845773,-0.81028,-0.77468,-0.738984,-0.703204,-0.667355,-0.631451,-0.595508,-0.559545,-0.52358,-0.487633,-0.451726,-0.415882,-0.380123,-0.344475,-0.308964,-0.273614,-0.238453,-0.203509,-0.168809,-0.134381,-0.100253,-0.0664529,-0.0330083,5.33333e-05,0.0327051,0.0649209,0.0966749,0.127943,0.1587,0.188924,0.218593,0.247687,0.276185,0.304071,0.331325,0.357934,0.383883,0.409158,0.433749,0.457644,0.480836,0.503316,0.52508,0.546121,0.566438,0.586029,0.604893,0.623032,0.640448,0.657146,0.67313,0.688408,0.702988,0.716878,0.73009,0.742635,0.754525,0.765775,0.776399,0.786413,0.795833,0.804674,0.812956,0.820695,0.827909,0.834616,0.840833,0.84658,0.851873,0.856729,0.861164,0.865196,0.868839,0.872107,0.875014,0.877572,0.879793,0.881685,0.883258,0.884519,0.885473,0.886124,0.886474,0.886524,0.886271,0.885714,0.884845,0.883657,0.882141,0.880283,0.87807,0.875484,0.872504,0.869108,0.865271,0.860964,0.856154,0.850807,0.844885,0.838346,0.831145,0.823234,0.814559,0.805067,0.794697,0.783387,0.771072,0.757684,0.743152,0.727402,0.710361,0.691952,0.6721,0.65073,0.627769,0.603148,0.576799,0.548666,0.518695,0.486845,0.453084,0.417394,0.379773,0.340234,0.298806,0.255541,0.210505,0.163784,0.115482,0.0657174,0.014616,-0.0376836,-0.0910455,-0.145334,-0.200421},
	{-2.88037,-2.85103,-2.82166,-2.79227,-2.76285,-2.7334,-2.70393,-2.67443,-2.64489,-2.61532,-2.58572,-2.55607,-2.52639,-2.49667,-2.46691,-2.4371,-2.40724,-2.37734,-2.34738,-2.31737,-2.2873,-2.25717,-2.22698,-2.19673,-2.1664,-2.13601,-2.10554,-2.07499,-2.04436,-2.01365,-1.98285,-1.95196,-1.92097,-1.88988,-1.85869,-1.8274,-1.79599,-1.76447,-1.73283,-1.70107,-1.66918,-1.63717,-1.60501,-1.57273,-1.5403,-1.50772,-1.475,-1.44212,-1.40909,-1.37591,-1.34256,-1.30905,-1.27538,-1.24154,-1.20754,-1.17337,-1.13903,-1.10453,-1.06986,-1.03503,-1.00004,-0.964902,-0.929608,-0.89417,-0.858595,-0.82289,-0.787065,-0.75113,-0.715098,-0.678981,-0.642794,-0.606553,-0.570275,-0.533978,-0.497683,-0.46141,-0.425183,-0.389024,-0.352958,-0.317011,-0.281209,-0.24558,-0.21015,-0.174949,-0.140006,-0.105347,-0.0710034,-0.0370027,-0.00337352,0.0298562,0.0626588,0.0950076,0.126876,0.15824,0.189074,0.219356,0.249063,0.278174,0.30667,0.334532,0.361744,0.388289,0.414154,0.439326,0.463793,0.487546,0.510576,0.532876,0.554442,0.575268,0.595354,0.614698,0.633301,0.651165,0.668294,0.684693,0.700368,0.715328,0.729581,0.743139,0.756012,0.768214,0.779759,0.790661,0.800937,0.810601,0.819673,0.828168,0.836106,0.843504,0.85038,0.856753,0.862642,0.868063,0.873034,0.877572,0.881694,0.885415,0.888749,0.891711,0.894312,0.896564,0.898477,0.90006,0.901319,0.90226,0.902887,0.903202,0.903205,0.902893,0.902264,0.901312,0.900028,0.898401,0.896419,0.894067,0.891326,0.888176,0.884592,0.880549,0.876017,0.870963,0.865352,0.859143,0.852294,0.84476,0.836489,0.827431,0.817526,0.806717,0.79494,0.782128,0.768213,0.753123,0.736786,0.719127,0.700071,0.679543,0.657469,0.633779,0.608405,0.581283,0.552358,0.521582,0.488917,0.454337,0.417829,0.379397,0.33906,0.296854,0.252837,0.207082,0.15968,0.11074,0.0603839,0.00874081,-0.0440504,-0.097854,-0.152538,-0.207977},
	{-2.89586,-2.86652,-2.83715,-2.80775,-2.77833,-2.74889,-2.71941,-2.6899,-2.66036,-2.63079,-2.60118,-2.57154,-2.54186,-2.51213,-2.48236,-2.45255,-2.42269,-2.39278,-2.36282,-2.3328,-2.30272,-2.27259,-2.24239,-2.21213,-2.18179,-2.15139,-2.12091,-2.09035,-2.05971,-2.02898,-1.99816,-1.96725,-1.93625,-1.90514,-1.87393,-1.84261,-1.81118,-1.77963,-1.74796,-1.71617,-1.68425,-1.65219,-1.62,-1.58767,-1.55519,-1.52256,-1.48978,-1.45685,-1.42375,-1.3905,-1.35708,-1.32349,-1.28973,-1.2558,-1.22169,-1.18742,-1.15297,-1.11834,-1.08354,-1.04858,-1.01344,-0.978135,-0.942672,-0.907052,-0.871284,-0.835373,-0.79933,-0.763163,-0.726885,-0.690508,-0.654045,-0.617512,-0.580927,-0.544306,-0.50767,-0.47104,-0.434437,-0.397886,-0.361411,-0.325036,-0.28879,-0.2527,-0.216793,-0.181098,-0.145645,-0.110463,-0.0755813,-0.0410299,-0.00683802,0.0269651,0.060351,0.0932913,0.125759,0.157727,0.18917,0.220063,0.250383,0.280108,0.309216,0.337687,0.365504,0.392649,0.419107,0.444864,0.469907,0.494225,0.51781,0.540653,0.562748,0.58409,0.604677,0.624507,0.643581,0.661899,0.679466,0.696285,0.712365,0.727711,0.742334,0.756244,0.769453,0.781973,0.793819,0.805006,0.815549,0.825466,0.834774,0.84349,0.851633,0.859222,0.866274,0.87281,0.878847,0.884404,0.889498,0.894147,0.898367,0.902175,0.905585,0.908611,0.911266,0.913561,0.915507,0.917112,0.918384,0.919327,0.919947,0.920244,0.920219,0.91987,0.919193,0.918182,0.916829,0.915122,0.913049,0.910593,0.907737,0.904459,0.900735,0.896537,0.891836,0.886599,0.880788,0.874365,0.867284,0.859501,0.850964,0.841621,0.831413,0.82028,0.80816,0.794986,0.780688,0.765195,0.748436,0.730334,0.710817,0.68981,0.667241,0.643039,0.617139,0.58948,0.560008,0.528678,0.495456,0.460318,0.423255,0.384276,0.343402,0.300673,0.256153,0.209916,0.162058,0.11269,0.061936,0.00992596,-0.0432009,-0.0973096,-0.15227,-0.207962},
	{-2.91112,-2.88177,-2.8524,-2.823,-2.79358,-2.76413,-2.73465,-2.70515,-2.67561,-2.64603,-2.61642,-2.58677,-2.55709,-2.52736,-2.49759,-2.46777,-2.4379,-2.40799,-2.37802,-2.348,-2.31792,-2.28777,-2.25757,-2.2273,-2.19695,-2.16654,-2.13605,-2.10548,-2.07482,-2.04408,-2.01325,-1.98233,-1.9513,-1.92018,-1.88895,-1.85761,-1.82615,-1.79458,-1.76288,-1.73105,-1.6991,-1.66701,-1.63478,-1.6024,-1.56988,-1.5372,-1.50437,-1.47138,-1.43822,-1.4049,-1.3714,-1.33774,-1.3039,-1.26988,-1.23568,-1.2013,-1.16674,-1.13199,-1.09707,-1.06197,-1.02668,-0.991226,-0.955597,-0.919802,-0.883846,-0.847736,-0.81148,-0.775088,-0.73857,-0.701939,-0.665208,-0.628392,-0.591506,-0.55457,-0.517601,-0.480621,-0.443652,-0.406717,-0.369839,-0.333046,-0.296364,-0.25982,-0.223443,-0.187262,-0.151307,-0.115608,-0.0801949,-0.0450988,-0.0103498,0.0240218,0.0579863,0.0915143,0.124577,0.157147,0.189196,0.2207,0.251632,0.28197,0.31169,0.340771,0.369194,0.396941,0.423994,0.450338,0.475961,0.500848,0.524991,0.54838,0.571009,0.592871,0.613963,0.634284,0.653832,0.672609,0.690618,0.707863,0.72435,0.740087,0.755084,0.76935,0.782897,0.795739,0.807889,0.819363,0.830177,0.840347,0.849892,0.85883,0.867179,0.874959,0.882188,0.888885,0.89507,0.90076,0.905975,0.910732,0.915047,0.918938,0.922418,0.925503,0.928206,0.930537,0.932508,0.934127,0.935401,0.936336,0.936936,0.937203,0.937136,0.936734,0.935992,0.934904,0.933462,0.931653,0.929464,0.92688,0.92388,0.920443,0.916545,0.912157,0.907249,0.901787,0.895732,0.889046,0.881682,0.873595,0.864733,0.855041,0.844462,0.832934,0.820395,0.806777,0.792011,0.776025,0.758747,0.740104,0.720021,0.698426,0.675248,0.650417,0.623871,0.595551,0.565408,0.533397,0.499489,0.463666,0.425922,0.386271,0.344739,0.301373,0.256238,0.209416,0.161007,0.111124,0.0598937,0.0074479,-0.0460743,-0.10054,-0.15582,-0.211799},
	{-2.92614,-2.8968,-2.86742,-2.83803,-2.8086,-2.77915,-2.74967,-2.72016,-2.69062,-2.66104,-2.63143,-2.60178,-2.57209,-2.54236,-2.51258,-2.48276,-2.45289,-2.42298,-2.393,-2.36297,-2.33289,-2.30274,-2.27253,-2.24225,-2.2119,-2.18147,-2.15097,-2.12039,-2.08972,-2.05897,-2.02813,-1.99719,-1.96615,-1.935,-1.90375,-1.87239,-1.84091,-1.80931,-1.77759,-1.74573,-1.71375,-1.68162,-1.64935,-1.61694,-1.58437,-1.55165,-1.51876,-1.48572,-1.4525,-1.41911,-1.38555,-1.35181,-1.31789,-1.28378,-1.24949,-1.21501,-1.18034,-1.14549,-1.11044,-1.07521,-1.03978,-1.00418,-0.968387,-0.932421,-0.896283,-0.859979,-0.823517,-0.786906,-0.750155,-0.713277,-0.676284,-0.639191,-0.602013,-0.564768,-0.527475,-0.490153,-0.452824,-0.415513,-0.378242,-0.341038,-0.303927,-0.266937,-0.230097,-0.193437,-0.156987,-0.120777,-0.0848393,-0.0492042,-0.0139034,0.021032,0.0555709,0.0896829,0.123338,0.156507,0.189161,0.221273,0.252817,0.283768,0.3141,0.343793,0.372824,0.401174,0.428825,0.45576,0.481965,0.507426,0.532131,0.556071,0.579238,0.601625,0.623228,0.644044,0.664072,0.683313,0.70177,0.719446,0.736347,0.75248,0.767856,0.782483,0.796374,0.809542,0.822001,0.833767,0.844856,0.855285,0.865072,0.874236,0.882795,0.890769,0.898178,0.905041,0.911377,0.917205,0.922543,0.927411,0.931825,0.935801,0.939356,0.942503,0.945256,0.947627,0.949626,0.951262,0.952543,0.953474,0.954058,0.954299,0.954195,0.953745,0.952944,0.951785,0.950259,0.948355,0.946059,0.943354,0.94022,0.936636,0.932575,0.92801,0.922909,0.917237,0.910957,0.904026,0.896401,0.888032,0.878869,0.868857,0.857936,0.846047,0.833124,0.819101,0.803908,0.787474,0.769728,0.750595,0.730002,0.707879,0.684155,0.658762,0.631641,0.602734,0.571994,0.539382,0.50487,0.468444,0.430104,0.389865,0.347759,0.303838,0.25817,0.210841,0.161954,0.111626,0.0599829,0.00715922,-0.0467067,-0.101484,-0.157044,-0.213277},
	{-2.94095,-2.9116,-2.88223,-2.85283,-2.82341,-2.79395,-2.76447,-2.73496,-2.70542,-2.67584,-2.64622,-2.61657,-2.58688,-2.55714,-2.52737,-2.49754,-2.46767,-2.43774,-2.40777,-2.37773,-2.34764,-2.31749,-2.28727,-2.25698,-2.22662,-2.19619,-2.16568,-2.13509,-2.10441,-2.07365,-2.04279,-2.01184,-1.98078,-1.94962,-1.91835,-1.88697,-1.85547,-1.82385,-1.7921,-1.76022,-1.7282,-1.69604,-1.66374,-1.63128,-1.59867,-1.5659,-1.53297,-1.49987,-1.4666,-1.43315,-1.39952,-1.36571,-1.33171,-1.29752,-1.26314,-1.22857,-1.1938,-1.15883,-1.12366,-1.0883,-1.05274,-1.01699,-0.981046,-0.944914,-0.908599,-0.872107,-0.835445,-0.79862,-0.761643,-0.724524,-0.687277,-0.649914,-0.61245,-0.574904,-0.537293,-0.499637,-0.461957,-0.424276,-0.38662,-0.349012,-0.31148,-0.274052,-0.236757,-0.199625,-0.162687,-0.125973,-0.0895163,-0.053348,-0.0175008,0.0179933,0.0531023,0.0877943,0.122038,0.155804,0.18906,0.22178,0.253935,0.285497,0.316443,0.346747,0.376388,0.405343,0.433594,0.461123,0.487913,0.51395,0.539222,0.563717,0.587426,0.610343,0.632461,0.653778,0.674291,0.694002,0.712911,0.731023,0.748342,0.764877,0.780636,0.79563,0.809869,0.823368,0.836141,0.848202,0.85957,0.870261,0.880293,0.889686,0.898458,0.90663,0.914221,0.921252,0.92774,0.933708,0.939172,0.944152,0.948665,0.952728,0.956357,0.959566,0.96237,0.96478,0.966807,0.968459,0.969745,0.970671,0.971239,0.971451,0.971309,0.970808,0.969945,0.968713,0.967102,0.9651,0.962693,0.959865,0.956594,0.952858,0.948631,0.943885,0.938587,0.932702,0.926191,0.919013,0.911122,0.902468,0.893001,0.882664,0.8714,0.859145,0.845837,0.831406,0.815786,0.798903,0.780687,0.761065,0.739964,0.717314,0.693047,0.667097,0.639405,0.609917,0.578589,0.545384,0.510278,0.473261,0.434335,0.393522,0.350856,0.306394,0.260208,0.212387,0.163037,0.112277,0.060236,0.00704882,-0.0471493,-0.102226,-0.158057,-0.214536},
	{-2.95554,-2.92619,-2.89682,-2.86742,-2.83799,-2.80854,-2.77906,-2.74954,-2.72,-2.69042,-2.6608,-2.63115,-2.60145,-2.57172,-2.54193,-2.51211,-2.48223,-2.4523,-2.42232,-2.39228,-2.36219,-2.33203,-2.3018,-2.27151,-2.24114,-2.2107,-2.18019,-2.14958,-2.1189,-2.08812,-2.05725,-2.02629,-1.99522,-1.96404,-1.93275,-1.90135,-1.86983,-1.83819,-1.80641,-1.77451,-1.74246,-1.71027,-1.67793,-1.64544,-1.61279,-1.57998,-1.547,-1.51385,-1.48052,-1.44702,-1.41332,-1.37944,-1.34537,-1.3111,-1.27663,-1.24197,-1.20709,-1.17202,-1.13674,-1.10126,-1.06557,-1.02967,-0.993578,-0.957285,-0.920799,-0.884124,-0.847266,-0.810234,-0.773036,-0.735684,-0.698187,-0.660561,-0.622819,-0.584979,-0.547057,-0.509074,-0.47105,-0.433008,-0.394973,-0.356969,-0.319024,-0.281165,-0.243423,-0.205826,-0.168406,-0.131195,-0.0942252,-0.0575296,-0.0211413,0.0149066,0.0505811,0.0858493,0.120679,0.155038,0.188895,0.22222,0.254984,0.287159,0.318718,0.349635,0.379886,0.409448,0.438302,0.466427,0.493806,0.520423,0.546264,0.571318,0.595575,0.619026,0.641664,0.663487,0.68449,0.704675,0.724042,0.742595,0.760339,0.77728,0.793428,0.808793,0.823386,0.83722,0.850311,0.862673,0.874324,0.88528,0.895562,0.905188,0.914177,0.92255,0.930327,0.937527,0.944172,0.950281,0.955873,0.960968,0.965582,0.969734,0.97344,0.976713,0.97957,0.98202,0.984077,0.985747,0.987041,0.987962,0.988515,0.988702,0.988522,0.987973,0.987051,0.985747,0.984053,0.981956,0.979443,0.976494,0.97309,0.969207,0.96482,0.959898,0.95441,0.948319,0.941586,0.934168,0.926021,0.917094,0.907335,0.896688,0.885094,0.872492,0.858815,0.843999,0.827972,0.810665,0.792005,0.771922,0.750343,0.7272,0.702425,0.675954,0.64773,0.617703,0.585829,0.552075,0.51642,0.478858,0.439394,0.398053,0.354875,0.309918,0.263259,0.214989,0.165217,0.114065,0.0616607,0.00814111,-0.0463599,-0.101712,-0.157792,-0.214498},
	{-2.96992,-2.94057,-2.9112,-2.8818,-2.85237,-2.82292,-2.79344,-2.76392,-2.73437,-2.70479,-2.67517,-2.64552,-2.61582,-2.58608,-2.5563,-2.52647,-2.49659,-2.46666,-2.43667,-2.40663,-2.37653,-2.34636,-2.31613,-2.28583,-2.25546,-2.22502,-2.19449,-2.16388,-2.13318,-2.1024,-2.07152,-2.04054,-2.00945,-1.97826,-1.94696,-1.91554,-1.884,-1.85234,-1.82054,-1.78861,-1.75653,-1.72431,-1.69194,-1.65942,-1.62673,-1.59388,-1.56085,-1.52765,-1.49428,-1.46071,-1.42696,-1.39301,-1.35887,-1.32452,-1.28997,-1.25521,-1.22025,-1.18507,-1.14968,-1.11407,-1.07826,-1.04223,-1.00599,-0.969539,-0.932886,-0.896033,-0.858986,-0.821752,-0.78434,-0.746759,-0.709021,-0.671138,-0.633124,-0.594996,-0.556771,-0.518468,-0.480108,-0.441713,-0.403307,-0.364915,-0.326564,-0.288282,-0.250098,-0.212044,-0.17415,-0.136448,-0.0989723,-0.0617557,-0.0248322,0.0117642,0.0479993,0.0838391,0.11925,0.154199,0.188654,0.222583,0.255955,0.28874,0.320911,0.35244,0.383302,0.413473,0.44293,0.471653,0.499623,0.526823,0.553237,0.578853,0.60366,0.627649,0.650811,0.673143,0.694641,0.715304,0.735133,0.754131,0.772302,0.789653,0.806193,0.821932,0.836881,0.851054,0.864464,0.877128,0.889064,0.900288,0.910819,0.920678,0.929883,0.938456,0.946417,0.953786,0.960584,0.966832,0.972548,0.977752,0.982463,0.986697,0.990472,0.993802,0.996702,0.999183,1.00126,1.00293,1.00422,1.00512,1.00565,1.00579,1.00555,1.00493,1.00393,1.00253,1.00072,0.998504,0.995851,0.992747,0.989173,0.985103,0.980511,0.975368,0.969638,0.963287,0.956274,0.948556,0.940085,0.930814,0.920687,0.909649,0.89764,0.884598,0.870457,0.855151,0.838611,0.820765,0.801544,0.780875,0.758689,0.734918,0.709496,0.682362,0.65346,0.622743,0.590173,0.555718,0.519363,0.481105,0.440955,0.398943,0.355112,0.309528,0.262271,0.213436,0.163137,0.111497,0.0586461,0.00472069,-0.0501463,-0.105826,-0.162202,-0.219172},
	{-2.98411,-2.95475,-2.92538,-2.89598,-2.86655,-2.8371,-2.80761,-2.7781,-2.74855,-2.71896,-2.68934,-2.65969,-2.62999,-2.60025,-2.57046,-2.54063,-2.51075,-2.48081,-2.45082,-2.42078,-2.39067,-2.3605,-2.33027,-2.29996,-2.26959,-2.23913,-2.2086,-2.17798,-2.14728,-2.11648,-2.08559,-2.0546,-2.0235,-1.9923,-1.96098,-1.92954,-1.89799,-1.8663,-1.83448,-1.80252,-1.77043,-1.73818,-1.70578,-1.67322,-1.64049,-1.6076,-1.57454,-1.54129,-1.50786,-1.47424,-1.44043,-1.40642,-1.37221,-1.33779,-1.30316,-1.26831,-1.23325,-1.19798,-1.16248,-1.12676,-1.09082,-1.05466,-1.01828,-0.981677,-0.944862,-0.907837,-0.870606,-0.833175,-0.795554,-0.757751,-0.719777,-0.681643,-0.643365,-0.604956,-0.566435,-0.527819,-0.489129,-0.450387,-0.411617,-0.372844,-0.334094,-0.295396,-0.256779,-0.218274,-0.179912,-0.141726,-0.10375,-0.066018,-0.0285645,0.00857538,0.0453667,0.0817743,0.117764,0.1533,0.18835,0.22288,0.256859,0.290255,0.323038,0.35518,0.386654,0.417435,0.447498,0.476822,0.505386,0.533172,0.560163,0.586346,0.611708,0.636239,0.659931,0.682778,0.704776,0.725923,0.746219,0.765668,0.784273,0.80204,0.818978,0.835097,0.850408,0.864925,0.878661,0.891634,0.903859,0.915356,0.926143,0.93624,0.945667,0.954446,0.962596,0.97014,0.977097,0.983488,0.989334,0.994654,0.999467,1.00379,1.00764,1.01103,1.01398,1.0165,1.0186,1.0203,1.02158,1.02248,1.02298,1.02309,1.02281,1.02213,1.02106,1.01958,1.01768,1.01536,1.01259,1.00935,1.00563,1.0014,0.996632,0.991297,0.985361,0.978787,0.971535,0.96356,0.954815,0.94525,0.934812,0.923444,0.911085,0.897674,0.883145,0.867432,0.850465,0.832174,0.812489,0.791339,0.768656,0.744372,0.718424,0.690752,0.661304,0.630034,0.596905,0.561891,0.524979,0.486169,0.445477,0.402935,0.358591,0.312515,0.264788,0.21551,0.164796,0.112772,0.0595669,0.00531842,-0.0498422,-0.105788,-0.162405,-0.219595},
	{-2.99809,-2.96874,-2.93936,-2.90996,-2.88053,-2.85108,-2.82159,-2.79208,-2.76253,-2.73294,-2.70332,-2.67366,-2.64396,-2.61422,-2.58443,-2.5546,-2.52471,-2.49477,-2.46478,-2.43473,-2.40462,-2.37445,-2.34421,-2.3139,-2.28352,-2.25306,-2.22252,-2.19189,-2.16118,-2.13038,-2.09947,-2.06847,-2.03736,-2.00615,-1.97482,-1.94336,-1.91179,-1.88008,-1.84824,-1.81627,-1.78414,-1.75187,-1.71944,-1.68685,-1.65409,-1.62116,-1.58805,-1.55476,-1.52129,-1.48762,-1.45375,-1.41967,-1.3854,-1.3509,-1.3162,-1.28127,-1.24612,-1.21075,-1.17515,-1.13932,-1.10326,-1.06697,-1.03045,-0.993705,-0.956733,-0.91954,-0.88213,-0.844509,-0.806684,-0.768665,-0.73046,-0.692083,-0.653545,-0.614863,-0.576052,-0.53713,-0.498118,-0.459037,-0.41991,-0.380763,-0.341622,-0.302515,-0.263471,-0.224522,-0.185699,-0.147036,-0.108567,-0.0703252,-0.0323477,0.00533036,0.0426728,0.0796437,0.116207,0.152327,0.187969,0.223099,0.257682,0.291687,0.325082,0.357837,0.389923,0.421314,0.451985,0.481911,0.511071,0.539446,0.567017,0.59377,0.61969,0.644768,0.668992,0.692358,0.714859,0.736494,0.757262,0.777166,0.796208,0.814395,0.831735,0.848237,0.863914,0.878777,0.892842,0.906125,0.918643,0.930414,0.941458,0.951795,0.961444,0.970428,0.978768,0.986484,0.993599,1.00013,1.00611,1.01154,1.01645,1.02086,1.02478,1.02823,1.03122,1.03377,1.03589,1.03759,1.03887,1.03974,1.04021,1.04028,1.03994,1.03919,1.03803,1.03645,1.03444,1.03199,1.02908,1.02568,1.02179,1.01737,1.0124,1.00684,1.00066,0.993826,0.986293,0.978018,0.968952,0.959046,0.948244,0.936489,0.923723,0.909881,0.894898,0.878709,0.861243,0.842431,0.822204,0.800492,0.777227,0.752344,0.725781,0.69748,0.667392,0.635474,0.601692,0.566025,0.528461,0.489007,0.447683,0.404525,0.359585,0.312937,0.264668,0.214881,0.163692,0.11123,0.0576242,0.00301273,-0.0524753,-0.108716,-0.165598,-0.223028},
	{-3.01188,-2.98253,-2.95316,-2.92375,-2.89433,-2.86487,-2.83538,-2.80587,-2.77631,-2.74673,-2.71711,-2.68745,-2.65774,-2.628,-2.59821,-2.56837,-2.53849,-2.50855,-2.47855,-2.4485,-2.41839,-2.38821,-2.35797,-2.32765,-2.29727,-2.2668,-2.23625,-2.20562,-2.1749,-2.14409,-2.11318,-2.08217,-2.05105,-2.01982,-1.98847,-1.95701,-1.92542,-1.89369,-1.86184,-1.82984,-1.79769,-1.76539,-1.73293,-1.70031,-1.66752,-1.63456,-1.60141,-1.56808,-1.53455,-1.50083,-1.46691,-1.43278,-1.39844,-1.36388,-1.3291,-1.29409,-1.25886,-1.22339,-1.18769,-1.15175,-1.11558,-1.07917,-1.04251,-1.00563,-0.968501,-0.931145,-0.893561,-0.855754,-0.817732,-0.779501,-0.741073,-0.702457,-0.663667,-0.624717,-0.585623,-0.546402,-0.507074,-0.467661,-0.428185,-0.388671,-0.349145,-0.309636,-0.270173,-0.230787,-0.19151,-0.152375,-0.113418,-0.0746732,-0.0361771,0.00203395,0.0399229,0.0774527,0.114586,0.151287,0.187519,0.223245,0.258432,0.293044,0.32705,0.360418,0.393118,0.42512,0.4564,0.48693,0.516689,0.545656,0.57381,0.601137,0.62762,0.653248,0.67801,0.701899,0.724909,0.747037,0.768283,0.788646,0.808131,0.826743,0.84449,0.861381,0.877428,0.892644,0.907042,0.92064,0.933455,0.945506,0.956811,0.967391,0.977267,0.986461,0.994993,1.00289,1.01016,1.01684,1.02295,1.02849,1.03351,1.038,1.042,1.04551,1.04855,1.05113,1.05327,1.05498,1.05626,1.05711,1.05755,1.05758,1.05718,1.05637,1.05513,1.05346,1.05135,1.04877,1.04572,1.04218,1.03813,1.03353,1.02836,1.02259,1.01618,1.0091,1.00131,0.992752,0.983387,0.973161,0.96202,0.949908,0.936762,0.922522,0.907121,0.890493,0.872568,0.853279,0.832557,0.810332,0.786538,0.761111,0.733992,0.705124,0.67446,0.64196,0.607594,0.571342,0.533199,0.493173,0.451287,0.407583,0.362117,0.314966,0.266219,0.215983,0.164376,0.111528,0.0575695,0.00263678,-0.0531422,-0.109646,-0.166767,-0.224414},
	{-3.02549,-2.99614,-2.96676,-2.93736,-2.90793,-2.87847,-2.84899,-2.81947,-2.78992,-2.76033,-2.73071,-2.70105,-2.67134,-2.6416,-2.61181,-2.58197,-2.55208,-2.52214,-2.49214,-2.46209,-2.43197,-2.40179,-2.37154,-2.34123,-2.31083,-2.28036,-2.24981,-2.21917,-2.18845,-2.15763,-2.12671,-2.09568,-2.06456,-2.03332,-2.00196,-1.97048,-1.93887,-1.90713,-1.87526,-1.84324,-1.81107,-1.77875,-1.74626,-1.71361,-1.68079,-1.64779,-1.61461,-1.58124,-1.54767,-1.5139,-1.47993,-1.44574,-1.41134,-1.37671,-1.34186,-1.30678,-1.27146,-1.23591,-1.20011,-1.16407,-1.12778,-1.09125,-1.05447,-1.01744,-0.98017,-0.942656,-0.904903,-0.866916,-0.828701,-0.790265,-0.751618,-0.71277,-0.673734,-0.634522,-0.595151,-0.555638,-0.516001,-0.476262,-0.436444,-0.39657,-0.356667,-0.316763,-0.276887,-0.237071,-0.197346,-0.157747,-0.118309,-0.0790667,-0.0400577,-0.00131909,0.0371113,0.0751956,0.112895,0.150173,0.186991,0.223312,0.2591,0.294319,0.328935,0.362915,0.396228,0.428843,0.460732,0.491868,0.522228,0.551789,0.58053,0.608433,0.635482,0.661664,0.686968,0.711385,0.734908,0.757534,0.77926,0.800088,0.82002,0.839062,0.85722,0.874504,0.890925,0.906497,0.921233,0.935149,0.948264,0.960597,0.972165,0.982992,0.993096,1.0025,1.01123,1.0193,1.02674,1.03356,1.0398,1.04546,1.05057,1.05515,1.05922,1.06279,1.06588,1.06849,1.07065,1.07236,1.07363,1.07447,1.07487,1.07485,1.0744,1.07351,1.07219,1.07041,1.06818,1.06548,1.06229,1.05859,1.05436,1.04957,1.04419,1.0382,1.03155,1.02421,1.01614,1.00728,0.997601,0.987039,0.975541,0.963051,0.949507,0.934848,0.919007,0.901918,0.883514,0.863725,0.842484,0.819723,0.795376,0.769382,0.741682,0.712224,0.680961,0.647857,0.612884,0.576027,0.537284,0.496666,0.454202,0.409936,0.363929,0.316261,0.267026,0.216332,0.1643,0.11106,0.0567446,0.00148737,-0.0545842,-0.111353,-0.168713,-0.226579},
	{-3.03892,-3.00956,-2.98019,-2.95079,-2.92136,-2.8919,-2.86241,-2.83289,-2.80334,-2.77375,-2.74413,-2.71447,-2.68476,-2.65502,-2.62523,-2.59539,-2.5655,-2.53555,-2.50555,-2.4755,-2.44538,-2.41519,-2.38494,-2.35462,-2.32423,-2.29375,-2.26319,-2.23255,-2.20182,-2.17099,-2.14006,-2.10903,-2.07789,-2.04664,-2.01527,-1.98378,-1.95216,-1.92041,-1.88852,-1.85648,-1.82429,-1.79194,-1.75943,-1.72676,-1.69391,-1.66088,-1.62766,-1.59425,-1.56064,-1.52683,-1.4928,-1.45856,-1.4241,-1.38941,-1.35449,-1.31934,-1.28394,-1.2483,-1.21241,-1.17627,-1.13987,-1.10322,-1.06632,-1.02916,-0.991744,-0.954076,-0.916159,-0.877996,-0.839593,-0.800958,-0.762099,-0.723024,-0.683747,-0.644281,-0.604639,-0.56484,-0.524901,-0.484843,-0.444689,-0.404462,-0.364188,-0.323896,-0.283614,-0.243375,-0.203209,-0.163152,-0.123239,-0.0835053,-0.0439891,-0.00472825,0.0342387,0.0728728,0.111135,0.148986,0.186387,0.2233,0.259686,0.29551,0.330736,0.365328,0.399254,0.432482,0.464982,0.496727,0.52769,0.557847,0.587177,0.61566,0.643279,0.67002,0.695869,0.720818,0.744859,0.767987,0.7902,0.811497,0.831881,0.851357,0.869931,0.887613,0.904413,0.920345,0.935423,0.949663,0.963082,0.9757,0.987536,0.998612,1.00895,1.01857,1.02749,1.03574,1.04335,1.05032,1.05669,1.06247,1.06768,1.07235,1.07649,1.08012,1.08325,1.0859,1.08808,1.0898,1.09106,1.09188,1.09225,1.09218,1.09167,1.09071,1.0893,1.08742,1.08508,1.08225,1.07892,1.07506,1.07065,1.06568,1.0601,1.05389,1.047,1.03941,1.03106,1.02192,1.01193,1.00104,0.989191,0.976334,0.962405,0.94734,0.931075,0.913543,0.894677,0.874408,0.85267,0.829397,0.804523,0.777989,0.749738,0.719719,0.687889,0.654215,0.61867,0.581244,0.541937,0.500765,0.45776,0.41297,0.366459,0.31831,0.268621,0.217501,0.165075,0.111472,0.0568241,0.0012653,-0.0550795,-0.112096,-0.169682,-0.227753},
	{-3.05217,-3.02281,-2.99344,-2.96404,-2.93461,-2.90515,-2.87566,-2.84614,-2.81659,-2.787,-2.75738,-2.72771,-2.69801,-2.66826,-2.63847,-2.60863,-2.57874,-2.54879,-2.51879,-2.48873,-2.45861,-2.42843,-2.39817,-2.36785,-2.33745,-2.30697,-2.27641,-2.24576,-2.21502,-2.18419,-2.15325,-2.12221,-2.09107,-2.05981,-2.02843,-1.99692,-1.96529,-1.93352,-1.90161,-1.86956,-1.83735,-1.80498,-1.77245,-1.73975,-1.70687,-1.67381,-1.64056,-1.60711,-1.57347,-1.53961,-1.50554,-1.47125,-1.43673,-1.40198,-1.367,-1.33177,-1.2963,-1.26057,-1.22459,-1.18836,-1.15186,-1.1151,-1.07807,-1.04078,-1.00323,-0.965409,-0.927333,-0.888999,-0.850415,-0.811585,-0.772518,-0.733224,-0.693712,-0.653997,-0.614091,-0.574012,-0.533778,-0.493408,-0.452924,-0.412351,-0.371714,-0.331041,-0.290361,-0.249704,-0.209105,-0.168597,-0.128215,-0.0879961,-0.0479788,-0.00820139,0.0312967,0.0704757,0.109295,0.147715,0.185696,0.223197,0.260181,0.296607,0.33244,0.367643,0.402182,0.436023,0.469135,0.501488,0.533055,0.563811,0.593732,0.622797,0.650989,0.678291,0.704689,0.730174,0.754735,0.778369,0.801071,0.822841,0.843681,0.863594,0.882588,0.90067,0.917853,0.934147,0.949569,0.964133,0.977859,0.990764,1.00287,1.01419,1.02476,1.0346,1.04372,1.05215,1.05991,1.06703,1.07353,1.07942,1.08473,1.08949,1.09369,1.09738,1.10055,1.10322,1.10541,1.10712,1.10836,1.10915,1.10947,1.10934,1.10875,1.1077,1.10619,1.1042,1.10172,1.09873,1.09523,1.09119,1.08659,1.08139,1.07558,1.06911,1.06195,1.05406,1.0454,1.03592,1.02557,1.0143,1.00205,0.988771,0.974395,0.958862,0.942106,0.924061,0.904661,0.883837,0.861524,0.837657,0.812174,0.785014,0.756124,0.725457,0.69297,0.658635,0.622429,0.584345,0.544386,0.502575,0.458947,0.413553,0.366464,0.317767,0.26756,0.215959,0.163088,0.109077,0.0540596,-0.00183255,-0.058477,-0.115763,-0.173593,-0.231886},
	{-3.06524,-3.03589,-3.00652,-2.97711,-2.94768,-2.91823,-2.88874,-2.85922,-2.82967,-2.80008,-2.77046,-2.74079,-2.71109,-2.68134,-2.65154,-2.6217,-2.59181,-2.56186,-2.53186,-2.5018,-2.47168,-2.44149,-2.41124,-2.38091,-2.3505,-2.32002,-2.28946,-2.2588,-2.22806,-2.19722,-2.16628,-2.13524,-2.10408,-2.07281,-2.04142,-2.00991,-1.97826,-1.94648,-1.91456,-1.88249,-1.85026,-1.81787,-1.78532,-1.7526,-1.71969,-1.6866,-1.65332,-1.61984,-1.58615,-1.55226,-1.51814,-1.4838,-1.44923,-1.41442,-1.37938,-1.34408,-1.30853,-1.27273,-1.23666,-1.20033,-1.16373,-1.12686,-1.08972,-1.05231,-1.01462,-0.976658,-0.938427,-0.899928,-0.861166,-0.822147,-0.782879,-0.743369,-0.703629,-0.66367,-0.623506,-0.583154,-0.54263,-0.501954,-0.461149,-0.420236,-0.379242,-0.338194,-0.297122,-0.256056,-0.215029,-0.174075,-0.133231,-0.0925328,-0.0520199,-0.0117311,0.0282933,0.0680126,0.107386,0.146372,0.184929,0.223016,0.260594,0.297622,0.334061,0.369875,0.405026,0.439481,0.473206,0.50617,0.538343,0.5697,0.600216,0.629867,0.658635,0.686503,0.713455,0.739479,0.764567,0.788712,0.811909,0.834158,0.855459,0.875815,0.895234,0.913724,0.931294,0.947957,0.963729,0.978624,0.992661,1.00586,1.01824,1.02982,1.04063,1.05068,1.06,1.06862,1.07655,1.08382,1.09045,1.09646,1.10188,1.10672,1.111,1.11475,1.11796,1.12067,1.12287,1.12459,1.12582,1.12658,1.12687,1.12669,1.12604,1.12492,1.12331,1.12122,1.11862,1.11551,1.11186,1.10766,1.10287,1.09749,1.09146,1.08477,1.07737,1.06922,1.06029,1.05052,1.03986,1.02826,1.01567,1.00202,0.987265,0.971332,0.954158,0.935678,0.915826,0.894534,0.871739,0.847375,0.821382,0.793703,0.764285,0.733081,0.700055,0.665178,0.628432,0.58981,0.549323,0.506993,0.46286,0.416979,0.369424,0.320283,0.269659,0.217667,0.164435,0.110092,0.0547719,-0.00139596,-0.0582917,-0.115805,-0.173843,-0.232327},
	{-3.07816,-3.0488,-3.01943,-2.99003,-2.9606,-2.93114,-2.90165,-2.87213,-2.84258,-2.81299,-2.78337,-2.7537,-2.724,-2.69425,-2.66445,-2.63461,-2.60472,-2.57477,-2.54477,-2.51471,-2.48458,-2.45439,-2.42414,-2.39381,-2.3634,-2.33291,-2.30234,-2.27169,-2.24094,-2.2101,-2.17915,-2.1481,-2.11694,-2.08566,-2.05426,-2.02274,-1.99108,-1.95929,-1.92735,-1.89526,-1.86302,-1.83062,-1.79804,-1.7653,-1.73237,-1.69925,-1.66594,-1.63243,-1.5987,-1.56477,-1.53061,-1.49622,-1.4616,-1.42674,-1.39164,-1.35628,-1.32066,-1.28478,-1.24863,-1.21221,-1.17551,-1.13854,-1.10128,-1.06375,-1.02593,-0.987827,-0.949445,-0.910785,-0.871852,-0.832649,-0.793184,-0.753465,-0.713502,-0.673306,-0.63289,-0.592271,-0.551464,-0.510489,-0.469368,-0.428123,-0.386779,-0.345363,-0.303905,-0.262436,-0.220988,-0.179596,-0.138296,-0.0971247,-0.0561222,-0.0153279,0.0252174,0.065472,0.105394,0.144941,0.184071,0.222741,0.26091,0.298537,0.335582,0.372004,0.407768,0.442836,0.477174,0.510749,0.543531,0.57549,0.606602,0.636842,0.666189,0.694625,0.722134,0.748702,0.77432,0.798979,0.822675,0.845406,0.867172,0.887975,0.907823,0.926722,0.944684,0.961719,0.977843,0.993072,1.00742,1.02092,1.03357,1.04541,1.05645,1.06673,1.07625,1.08505,1.09315,1.10057,1.10734,1.11347,1.11899,1.12391,1.12827,1.13206,1.13532,1.13805,1.14027,1.14198,1.1432,1.14393,1.14418,1.14394,1.14321,1.142,1.14029,1.13808,1.13535,1.13209,1.12829,1.1239,1.11893,1.11333,1.10708,1.10014,1.09248,1.08405,1.07482,1.06472,1.05373,1.04177,1.02881,1.01477,0.999592,0.983224,0.965596,0.946643,0.9263,0.904501,0.881182,0.85628,0.829735,0.801492,0.7715,0.739717,0.706106,0.670643,0.633312,0.594111,0.553053,0.510164,0.465489,0.419085,0.371029,0.321414,0.270345,0.21794,0.164326,0.109633,0.0539948,-0.00246174,-0.059618,-0.117367,-0.17562,-0.234303},
	{-3.0909,-3.06155,-3.03218,-3.00277,-2.97335,-2.94389,-2.9144,-2.88488,-2.85533,-2.82574,-2.79612,-2.76645,-2.73675,-2.707,-2.6772,-2.64736,-2.61747,-2.58752,-2.55752,-2.52745,-2.49733,-2.46714,-2.43688,-2.40655,-2.37614,-2.34565,-2.31508,-2.28442,-2.25366,-2.22282,-2.19187,-2.16081,-2.12964,-2.09836,-2.06695,-2.03542,-2.00375,-1.97195,-1.94,-1.9079,-1.87564,-1.84322,-1.81063,-1.77786,-1.74491,-1.71176,-1.67842,-1.64488,-1.61113,-1.57715,-1.54295,-1.50852,-1.47386,-1.43894,-1.40378,-1.36835,-1.33267,-1.29671,-1.26049,-1.22398,-1.18719,-1.15012,-1.11275,-1.0751,-1.03715,-0.998918,-0.960391,-0.921575,-0.882474,-0.843093,-0.803437,-0.763514,-0.723333,-0.682906,-0.642245,-0.601364,-0.560281,-0.519013,-0.477583,-0.436011,-0.394324,-0.352547,-0.310711,-0.268845,-0.226982,-0.185158,-0.143408,-0.10177,-0.0602838,-0.0189894,0.0220715,0.0628566,0.103323,0.143427,0.183126,0.222376,0.261134,0.299358,0.337006,0.374037,0.410412,0.446094,0.481045,0.515233,0.548624,0.581188,0.612899,0.64373,0.67366,0.702668,0.730737,0.757853,0.784005,0.809184,0.833383,0.856601,0.878837,0.900093,0.920374,0.939689,0.958046,0.975458,0.99194,1.00751,1.02218,1.03597,1.0489,1.061,1.07229,1.08279,1.09252,1.10151,1.10978,1.11735,1.12425,1.13051,1.13613,1.14115,1.14557,1.14943,1.15273,1.15549,1.15772,1.15944,1.16064,1.16135,1.16155,1.16126,1.16047,1.15917,1.15737,1.15505,1.1522,1.1488,1.14483,1.14028,1.13512,1.12933,1.12286,1.11569,1.10778,1.09909,1.08957,1.07918,1.06787,1.05558,1.04226,1.02785,1.01229,0.995517,0.977469,0.95808,0.937285,0.915019,0.89122,0.865825,0.838776,0.81002,0.779508,0.7472,0.71306,0.677069,0.639212,0.599491,0.55792,0.514532,0.469371,0.4225,0.373999,0.323962,0.272496,0.219721,0.165766,0.11076,0.0548362,-0.00188041,-0.0592732,-0.117238,-0.17569,-0.234556},
	{-3.10349,-3.07414,-3.04477,-3.01537,-2.98594,-2.95648,-2.92699,-2.89747,-2.86792,-2.83833,-2.80871,-2.77905,-2.74934,-2.71959,-2.6898,-2.65995,-2.63006,-2.60011,-2.57011,-2.54004,-2.50992,-2.47973,-2.44947,-2.41913,-2.38872,-2.35823,-2.32766,-2.29699,-2.26624,-2.23539,-2.20443,-2.17337,-2.1422,-2.11091,-2.07949,-2.04795,-2.01628,-1.98446,-1.9525,-1.92039,-1.88812,-1.85568,-1.82307,-1.79028,-1.75731,-1.72414,-1.69078,-1.65721,-1.62342,-1.58941,-1.55517,-1.5207,-1.48599,-1.45103,-1.41581,-1.38032,-1.34457,-1.30855,-1.27224,-1.23565,-1.19878,-1.1616,-1.12414,-1.08637,-1.0483,-1.00994,-0.971269,-0.932302,-0.893039,-0.853484,-0.813642,-0.773521,-0.733128,-0.692476,-0.651575,-0.610439,-0.569086,-0.527532,-0.485799,-0.443908,-0.401884,-0.359754,-0.317545,-0.27529,-0.23302,-0.19077,-0.148577,-0.106479,-0.0645151,-0.0227268,0.0188439,0.0601539,0.10116,0.141816,0.18208,0.221906,0.261249,0.300067,0.338316,0.375953,0.412938,0.449232,0.484797,0.519597,0.553597,0.586767,0.619078,0.650502,0.681016,0.710598,0.73923,0.766897,0.793585,0.819286,0.843992,0.8677,0.890408,0.912119,0.932837,0.952569,0.971324,0.989116,1.00596,1.02186,1.03685,1.05094,1.06416,1.07652,1.08805,1.09877,1.10871,1.11788,1.12632,1.13404,1.14108,1.14745,1.15317,1.15827,1.16276,1.16667,1.17,1.17278,1.17502,1.17673,1.17791,1.17857,1.17872,1.17835,1.17748,1.17608,1.17416,1.17171,1.16871,1.16514,1.161,1.15625,1.15088,1.14485,1.13813,1.13069,1.12249,1.11349,1.10364,1.09291,1.08123,1.06855,1.05482,1.03999,1.02398,1.00674,0.988202,0.968306,0.946985,0.924177,0.899818,0.87385,0.846214,0.81686,0.785742,0.75282,0.718064,0.681456,0.642985,0.602658,0.560491,0.516521,0.470797,0.423386,0.37437,0.323846,0.271926,0.21873,0.164389,0.10903,0.0527877,-0.00421648,-0.0618689,-0.120068,-0.178735,-0.237799},
	{-3.11593,-3.08658,-3.0572,-3.0278,-2.99837,-2.96892,-2.93943,-2.90991,-2.88036,-2.85077,-2.82115,-2.79149,-2.76178,-2.73203,-2.70224,-2.67239,-2.6425,-2.61255,-2.58255,-2.55248,-2.52236,-2.49216,-2.4619,-2.43157,-2.40116,-2.37066,-2.34009,-2.30942,-2.27867,-2.24781,-2.21685,-2.18579,-2.15461,-2.12331,-2.09189,-2.06034,-2.02866,-1.99684,-1.96486,-1.93274,-1.90046,-1.868,-1.83538,-1.80257,-1.76958,-1.73639,-1.703,-1.6694,-1.63559,-1.60155,-1.56728,-1.53276,-1.49801,-1.463,-1.42773,-1.39219,-1.35637,-1.32028,-1.28391,-1.24724,-1.21027,-1.17301,-1.13544,-1.09756,-1.05938,-1.02089,-0.982081,-0.942968,-0.903547,-0.863824,-0.823802,-0.783488,-0.742889,-0.702017,-0.660882,-0.619498,-0.577881,-0.536048,-0.494018,-0.451815,-0.409461,-0.366983,-0.324409,-0.281771,-0.2391,-0.196432,-0.153803,-0.11125,-0.0688154,-0.0265391,0.0155357,0.0573654,0.0989052,0.14011,0.180934,0.221332,0.261258,0.300666,0.339514,0.377755,0.41535,0.452255,0.488433,0.523845,0.558456,0.592233,0.625145,0.657164,0.688264,0.718423,0.747621,0.775842,0.80307,0.829296,0.854513,0.878714,0.901899,0.924068,0.945226,0.965379,0.984537,1.00271,1.01991,1.03616,1.05147,1.06587,1.07937,1.09199,1.10376,1.11471,1.12485,1.13421,1.14282,1.15069,1.15786,1.16434,1.17017,1.17535,1.1799,1.18386,1.18723,1.19003,1.19227,1.19396,1.19511,1.19573,1.19581,1.19537,1.19441,1.1929,1.19086,1.18827,1.18511,1.18138,1.17705,1.17209,1.1665,1.16023,1.15325,1.14553,1.13704,1.12772,1.11754,1.10645,1.09439,1.08132,1.06718,1.0519,1.03544,1.01772,0.998688,0.978277,0.956423,0.933065,0.908141,0.881592,0.853365,0.823408,0.791679,0.75814,0.722767,0.68554,0.646455,0.605521,0.56276,0.51821,0.471927,0.423979,0.374453,0.323449,0.271081,0.217472,0.162751,0.107047,0.0504925,-0.00679321,-0.0646994,-0.123129,-0.182006,-0.241265},
	{-3.12821,-3.09886,-3.06949,-3.04009,-3.01066,-2.9812,-2.95172,-2.9222,-2.89265,-2.86306,-2.83344,-2.80378,-2.77407,-2.74432,-2.71453,-2.68469,-2.65479,-2.62484,-2.59484,-2.56478,-2.53465,-2.50446,-2.47419,-2.44386,-2.41345,-2.38295,-2.35238,-2.32171,-2.29095,-2.26009,-2.22913,-2.19806,-2.16688,-2.13558,-2.10415,-2.0726,-2.04091,-2.00907,-1.97709,-1.94496,-1.91266,-1.8802,-1.84756,-1.81474,-1.78173,-1.74852,-1.71511,-1.68148,-1.64764,-1.61357,-1.57926,-1.54471,-1.50991,-1.47486,-1.43954,-1.40395,-1.36808,-1.33192,-1.29547,-1.25873,-1.22168,-1.18433,-1.14666,-1.10868,-1.07038,-1.03177,-0.992831,-0.953576,-0.914003,-0.874116,-0.833919,-0.793417,-0.752619,-0.711533,-0.67017,-0.628543,-0.586668,-0.544562,-0.502242,-0.459733,-0.417055,-0.374237,-0.331305,-0.28829,-0.245225,-0.202145,-0.159085,-0.116085,-0.073185,-0.0304266,0.0121469,0.0544909,0.0965601,0.138308,0.179689,0.220655,0.26116,0.301157,0.340601,0.379445,0.417647,0.455164,0.491954,0.527979,0.563202,0.597587,0.631103,0.663719,0.695408,0.726147,0.755915,0.784692,0.812465,0.839221,0.864951,0.889651,0.913317,0.935949,0.957552,0.978131,0.997695,1.01626,1.03383,1.05042,1.06606,1.08076,1.09455,1.10744,1.11946,1.13063,1.14098,1.15054,1.15931,1.16734,1.17465,1.18125,1.18717,1.19244,1.19706,1.20107,1.20448,1.20729,1.20954,1.21122,1.21234,1.21292,1.21296,1.21245,1.21139,1.20979,1.20763,1.2049,1.2016,1.1977,1.19319,1.18804,1.18223,1.17573,1.16851,1.16053,1.15175,1.14214,1.13164,1.12021,1.10781,1.09436,1.07983,1.06415,1.04725,1.02909,1.0096,0.988713,0.966367,0.942501,0.917056,0.889975,0.861205,0.830697,0.79841,0.764311,0.728375,0.690587,0.650947,0.609465,0.566168,0.521098,0.474312,0.425882,0.375899,0.324465,0.271696,0.217715,0.162653,0.106639,0.0498025,-0.00773871,-0.0658767,-0.124518,-0.183589,-0.24303},
	{-3.14035,-3.111,-3.08163,-3.05223,-3.0228,-2.99334,-2.96386,-2.93434,-2.90479,-2.87521,-2.84558,-2.81592,-2.78622,-2.75647,-2.72667,-2.69683,-2.66694,-2.63699,-2.60699,-2.57692,-2.5468,-2.51661,-2.48634,-2.45601,-2.42559,-2.3951,-2.36452,-2.33385,-2.30309,-2.27223,-2.24127,-2.2102,-2.17901,-2.1477,-2.11628,-2.08472,-2.05302,-2.02118,-1.98919,-1.95705,-1.92474,-1.89226,-1.85961,-1.82678,-1.79375,-1.76052,-1.72709,-1.69344,-1.65957,-1.62547,-1.59113,-1.55655,-1.52171,-1.48662,-1.45125,-1.41561,-1.37968,-1.34346,-1.30695,-1.27013,-1.23301,-1.19557,-1.15781,-1.11973,-1.08132,-1.04258,-1.00352,-0.96413,-0.92441,-0.884364,-0.843997,-0.803313,-0.76232,-0.721025,-0.67944,-0.637577,-0.595451,-0.553077,-0.510475,-0.467665,-0.424671,-0.381518,-0.338234,-0.29485,-0.251398,-0.207911,-0.164428,-0.120987,-0.0776275,-0.0343928,0.0086736,0.0515267,0.0941203,0.136407,0.17834,0.219871,0.260952,0.301534,0.341572,0.381018,0.419826,0.457953,0.495356,0.531995,0.567829,0.602824,0.636945,0.67016,0.702442,0.733764,0.764104,0.793442,0.821763,0.849053,0.875302,0.900504,0.924656,0.947756,0.969809,0.990818,1.01079,1.02975,1.04769,1.06464,1.08061,1.09562,1.1097,1.12286,1.13513,1.14654,1.1571,1.16685,1.1758,1.18399,1.19143,1.19816,1.20419,1.20954,1.21424,1.2183,1.22175,1.22459,1.22684,1.22852,1.22962,1.23016,1.23014,1.22957,1.22844,1.22674,1.22447,1.22162,1.21818,1.21413,1.20945,1.20412,1.19811,1.19139,1.18394,1.17571,1.16667,1.15677,1.14598,1.13424,1.1215,1.10772,1.09282,1.07676,1.05948,1.04091,1.02099,0.999666,0.97687,0.952542,0.926624,0.89906,0.869798,0.838793,0.806004,0.7714,0.73496,0.69667,0.656534,0.614563,0.570789,0.525255,0.478022,0.429164,0.378776,0.32696,0.273834,0.219522,0.164156,0.107864,0.0507732,-0.00699965,-0.0653494,-0.124185,-0.183437,-0.243047},
	{-3.15234,-3.12299,-3.09362,-3.06422,-3.0348,-3.00534,-2.97586,-2.94634,-2.91679,-2.88721,-2.85758,-2.82792,-2.79822,-2.76847,-2.73868,-2.70884,-2.67895,-2.649,-2.619,-2.58893,-2.55881,-2.52861,-2.49835,-2.46802,-2.4376,-2.40711,-2.37653,-2.34586,-2.3151,-2.28424,-2.25327,-2.2222,-2.19101,-2.1597,-2.12827,-2.0967,-2.065,-2.03316,-2.00116,-1.96901,-1.93669,-1.90421,-1.87154,-1.83869,-1.80565,-1.77241,-1.73895,-1.70528,-1.67139,-1.63726,-1.6029,-1.56828,-1.53341,-1.49827,-1.46286,-1.42717,-1.39119,-1.35492,-1.31834,-1.28145,-1.24425,-1.20673,-1.16888,-1.1307,-1.09219,-1.05334,-1.01416,-0.974632,-0.934769,-0.89457,-0.854038,-0.813178,-0.771995,-0.730498,-0.688697,-0.646603,-0.604231,-0.561596,-0.518717,-0.475614,-0.43231,-0.388829,-0.3452,-0.301453,-0.25762,-0.213734,-0.169834,-0.125957,-0.0821447,-0.0384395,0.00511437,0.0484712,0.0915843,0.134406,0.176887,0.218978,0.260631,0.301797,0.342426,0.382472,0.421885,0.460622,0.498638,0.53589,0.572338,0.607943,0.642671,0.676488,0.709365,0.741273,0.772189,0.802093,0.830966,0.858794,0.885567,0.911277,0.93592,0.959494,0.982001,1.00345,1.02384,1.04319,1.06151,1.07882,1.09512,1.11045,1.12483,1.13827,1.1508,1.16244,1.17323,1.18317,1.19231,1.20066,1.20825,1.2151,1.22124,1.22669,1.23147,1.2356,1.23909,1.24196,1.24423,1.24591,1.24701,1.24752,1.24747,1.24684,1.24564,1.24386,1.2415,1.23854,1.23498,1.2308,1.22597,1.22048,1.21429,1.20739,1.19973,1.19129,1.18202,1.17189,1.16084,1.14883,1.13581,1.12172,1.10652,1.09014,1.07252,1.0536,1.03333,1.01164,0.988467,0.963753,0.937442,0.909478,0.879811,0.848397,0.815197,0.780181,0.743331,0.704634,0.664095,0.62173,0.57757,0.531663,0.484069,0.434866,0.384149,0.332024,0.278607,0.224023,0.168405,0.11188,0.0545741,-0.00339707,-0.0619308,-0.120938,-0.180351,-0.240113},
	{-3.16419,-3.13485,-3.10547,-3.07608,-3.04665,-3.0172,-2.98771,-2.9582,-2.92865,-2.89907,-2.86945,-2.83979,-2.81008,-2.78034,-2.75055,-2.72071,-2.69081,-2.66087,-2.63087,-2.6008,-2.57068,-2.54049,-2.51023,-2.47989,-2.44948,-2.41898,-2.38841,-2.35774,-2.32697,-2.29611,-2.26514,-2.23407,-2.20288,-2.17157,-2.14013,-2.10856,-2.07686,-2.04501,-2.01301,-1.98085,-1.94852,-1.91603,-1.88335,-1.85049,-1.81743,-1.78417,-1.7507,-1.71702,-1.6831,-1.64895,-1.61456,-1.57991,-1.545,-1.50983,-1.47438,-1.43864,-1.40261,-1.36628,-1.32964,-1.29269,-1.25542,-1.21782,-1.17988,-1.14161,-1.103,-1.06404,-1.02474,-0.985088,-0.945088,-0.904741,-0.864049,-0.823017,-0.781651,-0.739957,-0.697946,-0.655628,-0.613016,-0.570127,-0.526978,-0.483588,-0.43998,-0.396179,-0.352212,-0.308109,-0.263901,-0.219624,-0.175313,-0.131008,-0.0867493,-0.0425803,0.00145464,0.0453092,0.0889358,0.132286,0.17531,0.217958,0.260179,0.301924,0.343142,0.383783,0.4238,0.463144,0.501771,0.539635,0.576696,0.612913,0.648248,0.682668,0.71614,0.748636,0.78013,0.8106,0.840028,0.868397,0.895696,0.921916,0.947052,0.971103,0.994068,1.01595,1.03677,1.05651,1.07521,1.09288,1.10953,1.12518,1.13985,1.15357,1.16636,1.17824,1.18924,1.19938,1.2087,1.21721,1.22495,1.23193,1.23817,1.24371,1.24857,1.25275,1.25629,1.25919,1.26147,1.26314,1.26422,1.2647,1.2646,1.2639,1.26263,1.26076,1.25829,1.25521,1.25152,1.24718,1.24219,1.23652,1.23014,1.22303,1.21515,1.20647,1.19695,1.18654,1.17521,1.16291,1.14957,1.13516,1.11962,1.10288,1.0849,1.0656,1.04493,1.02284,0.999246,0.974106,0.947359,0.918951,0.888835,0.856967,0.82331,0.787837,0.75053,0.71138,0.670394,0.62759,0.583002,0.536679,0.488686,0.439101,0.388022,0.335555,0.281819,0.22694,0.171049,0.114272,0.0567367,-0.00144552,-0.0601735,-0.11936,-0.178942,-0.238862},
	{-3.17591,-3.14656,-3.11719,-3.0878,-3.05837,-3.02892,-2.99944,-2.96992,-2.94038,-2.91079,-2.88117,-2.85151,-2.82181,-2.79207,-2.76228,-2.73244,-2.70255,-2.6726,-2.6426,-2.61254,-2.58242,-2.55223,-2.52197,-2.49163,-2.46122,-2.43073,-2.40015,-2.36948,-2.33872,-2.30786,-2.27689,-2.24581,-2.21462,-2.18331,-2.15187,-2.1203,-2.08859,-2.05673,-2.02473,-1.99256,-1.96023,-1.92773,-1.89505,-1.86217,-1.8291,-1.79583,-1.76235,-1.72864,-1.6947,-1.66053,-1.62611,-1.59144,-1.5565,-1.52129,-1.4858,-1.45002,-1.41395,-1.37757,-1.34087,-1.30386,-1.26652,-1.22884,-1.19083,-1.15247,-1.11376,-1.0747,-1.03528,-0.995507,-0.955375,-0.914885,-0.87404,-0.832843,-0.791299,-0.749416,-0.707201,-0.664666,-0.621823,-0.578687,-0.535275,-0.491607,-0.447704,-0.403591,-0.359295,-0.314845,-0.270272,-0.225611,-0.180899,-0.136175,-0.0914792,-0.0468554,-0.00234854,0.0419946,0.086126,0.129996,0.173554,0.21675,0.259532,0.301847,0.343646,0.384876,0.425489,0.465434,0.504665,0.543136,0.580803,0.617624,0.653561,0.688577,0.72264,0.755717,0.787784,0.818815,0.848791,0.877695,0.905513,0.932236,0.957858,0.982376,1.00579,1.0281,1.04932,1.06946,1.08853,1.10654,1.12351,1.13946,1.15441,1.16839,1.18141,1.19351,1.2047,1.21502,1.22448,1.23312,1.24097,1.24803,1.25434,1.25993,1.2648,1.26899,1.27251,1.27538,1.2776,1.2792,1.28017,1.28053,1.28029,1.27943,1.27797,1.27589,1.2732,1.26987,1.26589,1.26126,1.25594,1.24992,1.24316,1.23564,1.22733,1.21819,1.20818,1.19725,1.18537,1.17248,1.15853,1.14348,1.12725,1.10981,1.09108,1.07101,1.04954,1.02661,1.00216,0.976137,0.948478,0.919137,0.88807,0.855235,0.8206,0.784143,0.745851,0.705722,0.663767,0.620013,0.574499,0.527282,0.478432,0.428038,0.3762,0.323029,0.268648,0.213186,0.156771,0.09953,0.0415838,-0.0169593,-0.0760073,-0.135481,-0.195319,-0.255477},
	{-3.18749,-3.15815,-3.12878,-3.09938,-3.06996,-3.04051,-3.01103,-2.98151,-2.95197,-2.92239,-2.89277,-2.86311,-2.83341,-2.80367,-2.77388,-2.74404,-2.71415,-2.68421,-2.65421,-2.62415,-2.59403,-2.56384,-2.53358,-2.50325,-2.47284,-2.44234,-2.41177,-2.3811,-2.35034,-2.31947,-2.28851,-2.25743,-2.22624,-2.19493,-2.16349,-2.13191,-2.1002,-2.06834,-2.03633,-2.00416,-1.97183,-1.93932,-1.90663,-1.87374,-1.84066,-1.80738,-1.77388,-1.74016,-1.70621,-1.67201,-1.63757,-1.60287,-1.5679,-1.53266,-1.49714,-1.46132,-1.4252,-1.38877,-1.35202,-1.31495,-1.27754,-1.2398,-1.2017,-1.16326,-1.12446,-1.0853,-1.04577,-1.00588,-0.96562,-0.924993,-0.883999,-0.842642,-0.800926,-0.758857,-0.716445,-0.673697,-0.630628,-0.587251,-0.543582,-0.49964,-0.455448,-0.411028,-0.366408,-0.321616,-0.276683,-0.231645,-0.186536,-0.141397,-0.0962689,-0.0511945,-0.00621943,0.038609,0.0832421,0.12763,0.17172,0.215463,0.258803,0.30169,0.344069,0.38589,0.4271,0.467649,0.507488,0.546569,0.584847,0.622279,0.658824,0.694444,0.729104,0.762772,0.79542,0.827023,0.857558,0.887008,0.915358,0.942597,0.968718,0.993718,1.0176,1.04036,1.062,1.08254,1.102,1.12037,1.13769,1.15396,1.16922,1.18347,1.19676,1.2091,1.22051,1.23103,1.24068,1.24948,1.25746,1.26465,1.27107,1.27675,1.2817,1.28594,1.2895,1.29239,1.29462,1.2962,1.29715,1.29748,1.29718,1.29625,1.2947,1.29253,1.28972,1.28626,1.28214,1.27735,1.27186,1.26565,1.25869,1.25096,1.24242,1.23303,1.22276,1.21156,1.19939,1.1862,1.17194,1.15655,1.13999,1.12218,1.10309,1.08264,1.06078,1.03744,1.01258,0.986129,0.958041,0.928264,0.896756,0.863477,0.828397,0.791495,0.75276,0.712194,0.669808,0.625632,0.579709,0.532096,0.482867,0.432112,0.379932,0.326441,0.271762,0.216024,0.159354,0.101879,0.0437179,-0.0150238,-0.0742534,-0.133898,-0.193896,-0.254206},
	{-3.19894,-3.1696,-3.14023,-3.11084,-3.08141,-3.05196,-3.02248,-2.99297,-2.96343,-2.93385,-2.90423,-2.87458,-2.84488,-2.81514,-2.78535,-2.75551,-2.72563,-2.69569,-2.66569,-2.63563,-2.60551,-2.57532,-2.54507,-2.51473,-2.48433,-2.45384,-2.42326,-2.39259,-2.36183,-2.33097,-2.3,-2.26893,-2.23774,-2.20642,-2.17498,-2.14341,-2.1117,-2.07984,-2.04782,-2.01565,-1.98331,-1.95079,-1.9181,-1.88521,-1.85212,-1.81882,-1.78531,-1.75157,-1.71761,-1.68339,-1.64893,-1.61421,-1.57922,-1.54395,-1.50839,-1.47253,-1.43637,-1.3999,-1.3631,-1.32597,-1.2885,-1.25069,-1.21252,-1.174,-1.13511,-1.09585,-1.05622,-1.01622,-0.975836,-0.935076,-0.893939,-0.852427,-0.810545,-0.768298,-0.725693,-0.682741,-0.639451,-0.595839,-0.551921,-0.507713,-0.463238,-0.41852,-0.373582,-0.328456,-0.283171,-0.237762,-0.192265,-0.146719,-0.101165,-0.0556478,-0.0102118,0.0350948,0.0802229,0.125121,0.169738,0.214021,0.257915,0.301367,0.344324,0.38673,0.428534,0.469682,0.510126,0.549815,0.588702,0.626743,0.663894,0.700117,0.735375,0.769633,0.802863,0.835037,0.866131,0.896128,0.92501,0.952766,0.979388,1.00487,1.02921,1.05242,1.07449,1.09544,1.11528,1.13402,1.15167,1.16827,1.18383,1.19836,1.21191,1.22448,1.23611,1.24683,1.25665,1.26561,1.27373,1.28104,1.28756,1.29332,1.29833,1.30262,1.3062,1.3091,1.31132,1.31288,1.31379,1.31406,1.31369,1.31267,1.31102,1.30872,1.30577,1.30216,1.29787,1.29289,1.2872,1.28077,1.27358,1.26559,1.25678,1.24711,1.23653,1.22501,1.21251,1.19896,1.18433,1.16855,1.15158,1.13336,1.11383,1.09293,1.07061,1.0468,1.02144,0.994492,0.965893,0.935598,0.903564,0.869755,0.834142,0.796708,0.757445,0.716354,0.673453,0.628773,0.58236,0.534276,0.484596,0.433413,0.380831,0.326965,0.271938,0.215881,0.158921,0.101182,0.0427806,-0.0161797,-0.0756093,-0.135439,-0.19561,-0.256084},
	{-3.21027,-3.18092,-3.15156,-3.12216,-3.09274,-3.06329,-3.03382,-3.00431,-2.97476,-2.94519,-2.91557,-2.88592,-2.85622,-2.82648,-2.7967,-2.76686,-2.73698,-2.70704,-2.67704,-2.64699,-2.61687,-2.58668,-2.55643,-2.5261,-2.49569,-2.4652,-2.43463,-2.40397,-2.37321,-2.34235,-2.31138,-2.28031,-2.24912,-2.2178,-2.18636,-2.15479,-2.12308,-2.09121,-2.0592,-2.02702,-1.99468,-1.96216,-1.92946,-1.89656,-1.86347,-1.83016,-1.79664,-1.76289,-1.72891,-1.69468,-1.6602,-1.62546,-1.59044,-1.55514,-1.51955,-1.48366,-1.44747,-1.41095,-1.3741,-1.33692,-1.2994,-1.26152,-1.22329,-1.18469,-1.14572,-1.10637,-1.06664,-1.02652,-0.986024,-0.945136,-0.903861,-0.8622,-0.820157,-0.777737,-0.734947,-0.691795,-0.648292,-0.604452,-0.56029,-0.515824,-0.471073,-0.426062,-0.380815,-0.335361,-0.289731,-0.243959,-0.19808,-0.152134,-0.106162,-0.0602073,-0.0143167,0.0314621,0.0770792,0.122483,0.167621,0.212439,0.256883,0.300897,0.344426,0.387415,0.42981,0.471557,0.512604,0.5529,0.592396,0.631045,0.668804,0.705632,0.741488,0.776339,0.810153,0.842901,0.874559,0.905106,0.934525,0.962803,0.98993,1.0159,1.04071,1.06437,1.08687,1.10823,1.12846,1.14757,1.16558,1.1825,1.19837,1.21319,1.227,1.23982,1.25167,1.26259,1.2726,1.28172,1.28999,1.29742,1.30405,1.30989,1.31498,1.31932,1.32294,1.32586,1.32808,1.32964,1.33052,1.33074,1.33031,1.32922,1.32748,1.32508,1.32201,1.31827,1.31383,1.30869,1.30282,1.2962,1.2888,1.2806,1.27155,1.26163,1.25079,1.239,1.2262,1.21235,1.1974,1.1813,1.16399,1.14541,1.12551,1.10424,1.08152,1.05731,1.03155,1.00419,0.975167,0.944445,0.911979,0.877736,0.841689,0.803822,0.764128,0.722612,0.679294,0.634207,0.587398,0.538933,0.488888,0.437358,0.384448,0.330275,0.274963,0.218642,0.161438,0.103476,0.0448692,-0.0142808,-0.0738864,-0.13388,-0.194207,-0.25483},
	{-3.22147,-3.19212,-3.16276,-3.13337,-3.10395,-3.0745,-3.04502,-3.01552,-2.98598,-2.9564,-2.92679,-2.89713,-2.86744,-2.8377,-2.80792,-2.77809,-2.7482,-2.71827,-2.68828,-2.65822,-2.62811,-2.59792,-2.56767,-2.53734,-2.50694,-2.47645,-2.44588,-2.41522,-2.38446,-2.35361,-2.32264,-2.29157,-2.26038,-2.22907,-2.19763,-2.16606,-2.13434,-2.10248,-2.07047,-2.03829,-2.00595,-1.97342,-1.94072,-1.90782,-1.87471,-1.8414,-1.80787,-1.77411,-1.74012,-1.70588,-1.67138,-1.63662,-1.60158,-1.56626,-1.53064,-1.49472,-1.45849,-1.42193,-1.38504,-1.34781,-1.31024,-1.2723,-1.234,-1.19533,-1.15628,-1.11684,-1.07702,-1.0368,-0.99619,-0.95518,-0.913773,-0.87197,-0.829772,-0.787186,-0.744216,-0.700872,-0.657163,-0.613102,-0.568704,-0.523986,-0.478968,-0.433672,-0.388124,-0.342351,-0.296384,-0.250256,-0.204004,-0.157666,-0.111284,-0.0649015,-0.0185646,0.0276781,0.0737763,0.119678,0.165329,0.210676,0.255662,0.300231,0.344326,0.387892,0.430873,0.473213,0.514858,0.555756,0.595857,0.635112,0.673475,0.710903,0.747356,0.782797,0.817193,0.850513,0.882732,0.913827,0.943781,0.972578,1.00021,1.02666,1.05194,1.07605,1.09898,1.12075,1.14136,1.16084,1.17919,1.19644,1.2126,1.2277,1.24177,1.25482,1.26689,1.278,1.28818,1.29746,1.30585,1.3134,1.32011,1.32603,1.33116,1.33554,1.33917,1.34209,1.34429,1.3458,1.34662,1.34677,1.34624,1.34504,1.34317,1.34062,1.33739,1.33346,1.32882,1.32346,1.31735,1.31048,1.30281,1.29431,1.28495,1.2747,1.26352,1.25135,1.23817,1.22392,1.20854,1.19199,1.17422,1.15516,1.13477,1.11298,1.08973,1.06498,1.03866,1.01072,0.981115,0.949799,0.916732,0.881884,0.845229,0.806754,0.766455,0.724342,0.680438,0.634778,0.587416,0.538419,0.487866,0.435858,0.3825,0.327913,0.27222,0.215552,0.158034,0.099789,0.0409273,-0.0184532,-0.0782688,-0.138457,-0.198965,-0.259758},
	{-3.23254,-3.2032,-3.17384,-3.14445,-3.11503,-3.08559,-3.05611,-3.0266,-2.99707,-2.96749,-2.93788,-2.90823,-2.87854,-2.8488,-2.81902,-2.78919,-2.75931,-2.72938,-2.69939,-2.66934,-2.63923,-2.60905,-2.5788,-2.54847,-2.51807,-2.48759,-2.45702,-2.42636,-2.39561,-2.36475,-2.33379,-2.30272,-2.27153,-2.24022,-2.20879,-2.17722,-2.1455,-2.11364,-2.08163,-2.04945,-2.01711,-1.98458,-1.95187,-1.91897,-1.88586,-1.85255,-1.81901,-1.78524,-1.75124,-1.71698,-1.68247,-1.64769,-1.61264,-1.57729,-1.54165,-1.5057,-1.46944,-1.43285,-1.39592,-1.35864,-1.32102,-1.28303,-1.24466,-1.20592,-1.1668,-1.12728,-1.08737,-1.04705,-1.00633,-0.965207,-0.923673,-0.881732,-0.839386,-0.796638,-0.753496,-0.709965,-0.666056,-0.62178,-0.577152,-0.532189,-0.48691,-0.441336,-0.395493,-0.349407,-0.30311,-0.256634,-0.210014,-0.163291,-0.116505,-0.0696998,-0.0229222,0.0237791,0.0703534,0.116748,0.162909,0.208779,0.254304,0.299425,0.344086,0.388227,0.431792,0.474724,0.516968,0.558469,0.599176,0.639039,0.678009,0.716041,0.753095,0.789131,0.824114,0.858013,0.890799,0.922451,0.952946,0.982271,1.01041,1.03736,1.06312,1.08768,1.11105,1.13324,1.15426,1.17411,1.19282,1.21041,1.22688,1.24228,1.25662,1.26993,1.28223,1.29355,1.30392,1.31337,1.32192,1.3296,1.33643,1.34244,1.34766,1.35209,1.35578,1.35872,1.36094,1.36245,1.36325,1.36337,1.3628,1.36154,1.3596,1.35696,1.35363,1.34959,1.34483,1.33933,1.33307,1.32604,1.31819,1.30951,1.29996,1.2895,1.27809,1.2657,1.25228,1.23777,1.22214,1.20532,1.18727,1.16793,1.14724,1.12514,1.10159,1.07652,1.04988,1.02161,0.991681,0.960034,0.926634,0.891452,0.854464,0.815659,0.775034,0.732599,0.688379,0.642411,0.59475,0.545464,0.494635,0.442362,0.388754,0.33393,0.278015,0.221139,0.163426,0.104999,0.0459674,-0.0135733,-0.0735403,-0.133872,-0.194518,-0.255445},
	{-3.2435,-3.21416,-3.1848,-3.15541,-3.12599,-3.09655,-3.06708,-3.03757,-3.00804,-2.97847,-2.94886,-2.91921,-2.88952,-2.85979,-2.83001,-2.80018,-2.7703,-2.74037,-2.71039,-2.68034,-2.65023,-2.62005,-2.58981,-2.55949,-2.52909,-2.49861,-2.46804,-2.43739,-2.40664,-2.37578,-2.34483,-2.31376,-2.28258,-2.25127,-2.21984,-2.18827,-2.15656,-2.1247,-2.09269,-2.06051,-2.02817,-1.99564,-1.96293,-1.93003,-1.89692,-1.8636,-1.83005,-1.79628,-1.76227,-1.728,-1.69348,-1.65869,-1.62361,-1.58825,-1.55259,-1.51661,-1.48032,-1.4437,-1.40673,-1.36942,-1.33174,-1.2937,-1.25528,-1.21648,-1.17729,-1.13769,-1.09769,-1.05729,-1.01647,-0.975228,-0.933573,-0.891501,-0.849013,-0.806112,-0.762804,-0.719094,-0.674992,-0.63051,-0.585661,-0.540461,-0.494929,-0.449087,-0.402958,-0.356569,-0.309951,-0.263136,-0.216159,-0.169061,-0.12188,-0.0746629,-0.0274548,0.0196951,0.0667354,0.113613,0.160273,0.206659,0.252713,0.298377,0.343592,0.3883,0.43244,0.475957,0.518791,0.560888,0.602193,0.642656,0.682225,0.720856,0.758503,0.795127,0.83069,0.865161,0.898508,0.930708,0.961739,0.991584,1.02023,1.04767,1.07389,1.09891,1.12271,1.1453,1.16671,1.18693,1.20598,1.22389,1.24066,1.25634,1.27093,1.28447,1.29698,1.30849,1.31903,1.32862,1.33729,1.34507,1.35198,1.35806,1.36331,1.36777,1.37145,1.37437,1.37655,1.378,1.37873,1.37875,1.37806,1.37667,1.37457,1.37176,1.36824,1.36399,1.35899,1.35325,1.34672,1.33939,1.33124,1.32222,1.31232,1.30149,1.28969,1.27689,1.26303,1.24807,1.23196,1.21465,1.19608,1.1762,1.15496,1.13229,1.10814,1.08246,1.05519,1.02629,0.995705,0.963396,0.929326,0.893468,0.855803,0.816321,0.775024,0.731926,0.687056,0.640456,0.592185,0.542314,0.490931,0.438137,0.384044,0.328773,0.272451,0.215206,0.157162,0.098439,0.0391416,-0.020638,-0.080822,-0.141354,-0.202186,-0.26329},
	{-3.25434,-3.225,-3.19564,-3.16626,-3.13684,-3.1074,-3.07793,-3.04843,-3.01889,-2.98932,-2.95972,-2.93007,-2.90039,-2.87066,-2.84088,-2.81106,-2.78118,-2.75125,-2.72127,-2.69123,-2.66112,-2.63095,-2.6007,-2.57039,-2.53999,-2.50952,-2.47896,-2.4483,-2.41756,-2.38671,-2.35576,-2.32469,-2.29351,-2.26221,-2.23078,-2.19922,-2.16751,-2.13566,-2.10364,-2.07147,-2.03913,-2.0066,-1.97389,-1.94099,-1.90788,-1.87455,-1.84101,-1.80723,-1.77321,-1.73894,-1.70441,-1.6696,-1.63452,-1.59914,-1.56346,-1.52746,-1.49114,-1.45449,-1.41749,-1.38014,-1.34242,-1.30433,-1.26586,-1.227,-1.18774,-1.14807,-1.10799,-1.0675,-1.02658,-0.985238,-0.943469,-0.901271,-0.858646,-0.815598,-0.772129,-0.728246,-0.683958,-0.639276,-0.594212,-0.548782,-0.503004,-0.456899,-0.41049,-0.363805,-0.316872,-0.269724,-0.222397,-0.174929,-0.127361,-0.0797368,-0.0321035,0.0154897,0.0629911,0.110347,0.157502,0.204399,0.25098,0.297184,0.342953,0.388225,0.432942,0.477042,0.520467,0.563161,0.605068,0.646133,0.686306,0.725539,0.763785,0.801003,0.837154,0.872203,0.90612,0.938878,0.970454,1.00083,1.02999,1.05793,1.08464,1.11011,1.13436,1.15738,1.17918,1.19978,1.2192,1.23745,1.25454,1.27052,1.28539,1.29918,1.31193,1.32365,1.33438,1.34414,1.35297,1.36088,1.36791,1.37408,1.37942,1.38394,1.38767,1.39062,1.39281,1.39426,1.39497,1.39496,1.39422,1.39277,1.39059,1.3877,1.38407,1.37971,1.37459,1.3687,1.36203,1.35454,1.34621,1.33701,1.32691,1.31587,1.30386,1.29083,1.27673,1.26153,1.24516,1.22758,1.20874,1.18858,1.16705,1.14408,1.11964,1.09365,1.06607,1.03685,1.00595,0.973315,0.938922,0.902742,0.864755,0.824954,0.783341,0.739933,0.694759,0.647862,0.599303,0.549156,0.497507,0.444459,0.390126,0.334629,0.278094,0.22065,0.162419,0.103521,0.0440594,-0.0158751,-0.0762061,-0.136879,-0.197845,-0.25908},
	{-3.26507,-3.23573,-3.20637,-3.17699,-3.14758,-3.11814,-3.08867,-3.05917,-3.02964,-3.00007,-2.97047,-2.94082,-2.91114,-2.88141,-2.85164,-2.82182,-2.79195,-2.76202,-2.73204,-2.702,-2.6719,-2.64173,-2.61149,-2.58118,-2.55079,-2.52032,-2.48976,-2.45911,-2.42837,-2.39753,-2.36658,-2.33552,-2.30435,-2.27305,-2.24162,-2.21006,-2.17836,-2.14651,-2.1145,-2.08233,-2.04999,-2.01747,-1.98476,-1.95186,-1.91875,-1.88542,-1.85188,-1.8181,-1.78407,-1.7498,-1.71526,-1.68044,-1.64535,-1.60995,-1.57425,-1.53824,-1.5019,-1.46522,-1.42819,-1.39081,-1.35305,-1.31492,-1.2764,-1.23748,-1.19816,-1.15843,-1.11828,-1.0777,-1.03669,-0.995251,-0.953372,-0.911056,-0.868302,-0.825113,-0.781492,-0.737444,-0.692978,-0.648104,-0.602833,-0.557182,-0.511167,-0.464809,-0.418131,-0.371159,-0.323922,-0.276452,-0.228785,-0.180958,-0.133012,-0.0849923,-0.0369449,0.0110807,0.0590325,0.106856,0.154496,0.201894,0.24899,0.295724,0.342036,0.387863,0.433145,0.47782,0.521827,0.565109,0.607607,0.649267,0.690035,0.729861,0.768698,0.806502,0.843232,0.878853,0.913331,0.946638,0.978751,1.00965,1.03932,1.06774,1.09492,1.12085,1.14553,1.16896,1.19115,1.21213,1.23189,1.25046,1.26786,1.28411,1.29924,1.31327,1.32622,1.33813,1.34903,1.35894,1.36788,1.3759,1.38301,1.38924,1.39461,1.39915,1.40287,1.4058,1.40795,1.40934,1.40997,1.40985,1.409,1.40741,1.40508,1.40201,1.39819,1.39361,1.38826,1.38212,1.37517,1.36739,1.35875,1.34922,1.33877,1.32736,1.31496,1.30152,1.28699,1.27134,1.2545,1.23644,1.21709,1.1964,1.17432,1.15079,1.12576,1.09918,1.07099,1.04114,1.0096,0.976329,0.941286,0.90445,0.865806,0.825351,0.78309,0.739043,0.693245,0.645742,0.596599,0.545894,0.493717,0.440175,0.385384,0.329466,0.272547,0.214757,0.156216,0.0970393,0.0373282,-0.0228319,-0.0833688,-0.144233,-0.20538,-0.266784},
	{-3.27568,-3.24635,-3.21699,-3.18761,-3.1582,-3.12876,-3.0993,-3.0698,-3.04027,-3.0107,-2.9811,-2.95146,-2.92178,-2.89206,-2.86229,-2.83247,-2.80261,-2.77269,-2.74271,-2.71267,-2.68257,-2.65241,-2.62217,-2.59187,-2.56148,-2.53101,-2.50046,-2.46982,-2.43908,-2.40824,-2.3773,-2.34625,-2.31508,-2.28379,-2.25237,-2.22081,-2.18911,-2.15727,-2.12527,-2.0931,-2.06076,-2.02825,-1.99554,-1.96264,-1.92953,-1.89621,-1.86266,-1.82888,-1.79486,-1.76058,-1.72603,-1.69121,-1.6561,-1.6207,-1.58499,-1.54896,-1.5126,-1.4759,-1.43884,-1.40143,-1.36364,-1.32547,-1.28691,-1.24794,-1.20856,-1.16877,-1.12854,-1.08789,-1.0468,-1.00526,-0.96328,-0.920851,-0.877973,-0.834649,-0.790882,-0.746675,-0.702038,-0.656978,-0.611508,-0.565642,-0.519397,-0.472793,-0.425852,-0.3786,-0.331066,-0.283281,-0.23528,-0.1871,-0.138784,-0.0903743,-0.0419186,0.00653377,0.0549305,0.103217,0.151337,0.199231,0.246839,0.294101,0.340953,0.387334,0.433179,0.478428,0.523018,0.566888,0.60998,0.652237,0.693603,0.734027,0.77346,0.811855,0.849171,0.88537,0.920418,0.954284,0.986942,1.01837,1.04856,1.07749,1.10515,1.13154,1.15667,1.18053,1.20313,1.22449,1.24462,1.26353,1.28125,1.29781,1.31321,1.3275,1.34069,1.35282,1.36391,1.37399,1.3831,1.39125,1.39847,1.4048,1.41025,1.41485,1.41862,1.42158,1.42374,1.42513,1.42574,1.42559,1.42469,1.42304,1.42063,1.41748,1.41355,1.40886,1.40339,1.39711,1.39001,1.38207,1.37326,1.36355,1.35291,1.3413,1.32869,1.31502,1.30027,1.28437,1.26729,1.24897,1.22935,1.20839,1.18604,1.16222,1.1369,1.11003,1.08154,1.05139,1.01955,0.985963,0.950611,0.913467,0.874517,0.833757,0.791194,0.746851,0.700763,0.652977,0.60356,0.552591,0.500161,0.446378,0.391359,0.335224,0.278103,0.220123,0.161403,0.102059,0.0421902,-0.0181194,-0.0787987,-0.139799,-0.201078,-0.262611},
	{-3.28619,-3.25686,-3.2275,-3.19812,-3.16871,-3.13928,-3.10982,-3.08032,-3.05079,-3.02123,-2.99163,-2.962,-2.93232,-2.9026,-2.87283,-2.84302,-2.81316,-2.78324,-2.75327,-2.72323,-2.69314,-2.66298,-2.63275,-2.60245,-2.57207,-2.54161,-2.51106,-2.48042,-2.44969,-2.41886,-2.38792,-2.35688,-2.32571,-2.29443,-2.26301,-2.23146,-2.19977,-2.16793,-2.13593,-2.10377,-2.07144,-2.03893,-2.00623,-1.97333,-1.94023,-1.90691,-1.87337,-1.83959,-1.80556,-1.77128,-1.73673,-1.70191,-1.66679,-1.63138,-1.59566,-1.55962,-1.52324,-1.48652,-1.44945,-1.41201,-1.37419,-1.33598,-1.29738,-1.25837,-1.21894,-1.17909,-1.1388,-1.09808,-1.0569,-1.01528,-0.973203,-0.930667,-0.887674,-0.844223,-0.800316,-0.755959,-0.711158,-0.665922,-0.620261,-0.57419,-0.527724,-0.480883,-0.433689,-0.386167,-0.338345,-0.290255,-0.24193,-0.193409,-0.144733,-0.0959443,-0.0470909,0.00177746,0.0506085,0.0993471,0.147936,0.196316,0.244427,0.292205,0.339589,0.386513,0.432913,0.478727,0.523891,0.568341,0.612019,0.654865,0.696822,0.737836,0.777858,0.816838,0.854734,0.891505,0.927115,0.961533,0.994733,1.02669,1.05739,1.08681,1.11495,1.1418,1.16737,1.19165,1.21465,1.23639,1.25687,1.27612,1.29415,1.31099,1.32666,1.34119,1.35461,1.36694,1.37821,1.38844,1.39768,1.40595,1.41327,1.41967,1.42517,1.42981,1.43359,1.43655,1.43869,1.44003,1.44059,1.44036,1.43937,1.43761,1.43508,1.43177,1.42769,1.42282,1.41716,1.41067,1.40335,1.39517,1.3861,1.37612,1.3652,1.35328,1.34035,1.32635,1.31125,1.29499,1.27753,1.25881,1.23879,1.21741,1.19462,1.17036,1.14458,1.11723,1.08826,1.05763,1.02528,0.991198,0.955338,0.917685,0.878225,0.836958,0.793895,0.749059,0.702488,0.654235,0.604368,0.552969,0.500132,0.445967,0.390592,0.334129,0.276707,0.218453,0.159485,0.0999151,0.0398405,-0.0206582,-0.0815141,-0.142679,-0.204116,-0.265799},
	{-3.29659,-3.26726,-3.2379,-3.20853,-3.17912,-3.14969,-3.12023,-3.09074,-3.06121,-3.03165,-3.00206,-2.97242,-2.94275,-2.91303,-2.88327,-2.85346,-2.8236,-2.79369,-2.76372,-2.73369,-2.7036,-2.67345,-2.64322,-2.61293,-2.58255,-2.5521,-2.52156,-2.49093,-2.4602,-2.42937,-2.39844,-2.3674,-2.33625,-2.30497,-2.27356,-2.24202,-2.21034,-2.1785,-2.14651,-2.11436,-2.08203,-2.04953,-2.01684,-1.98394,-1.95085,-1.91753,-1.88399,-1.85021,-1.81619,-1.78191,-1.74736,-1.71254,-1.67742,-1.64201,-1.60628,-1.57022,-1.53384,-1.4971,-1.46001,-1.42254,-1.3847,-1.34646,-1.30782,-1.26877,-1.2293,-1.18939,-1.14905,-1.10826,-1.06701,-1.02531,-0.983141,-0.940506,-0.897402,-0.853831,-0.809793,-0.765292,-0.720335,-0.674929,-0.629085,-0.582816,-0.536137,-0.489068,-0.441629,-0.393845,-0.345744,-0.297357,-0.248717,-0.199863,-0.150834,-0.101675,-0.052432,-0.00315529,0.0461025,0.095286,0.144337,0.193197,0.241803,0.290093,0.338002,0.385465,0.432416,0.478791,0.524525,0.569554,0.613815,0.657249,0.699796,0.741401,0.782012,0.821579,0.860056,0.897402,0.933579,0.968554,1.0023,1.03479,1.066,1.09593,1.12455,1.15187,1.17788,1.20259,1.226,1.24812,1.26897,1.28857,1.30692,1.32406,1.34002,1.3548,1.36845,1.381,1.39246,1.40287,1.41226,1.42066,1.42809,1.43459,1.44017,1.44486,1.44868,1.45166,1.45381,1.45514,1.45568,1.45541,1.45436,1.45253,1.44992,1.44652,1.44233,1.43734,1.43154,1.4249,1.41742,1.40907,1.39982,1.38965,1.37852,1.36639,1.35323,1.339,1.32365,1.30713,1.28941,1.27042,1.25012,1.22845,1.20536,1.1808,1.15471,1.12704,1.09775,1.06679,1.03411,0.999696,0.963502,0.925513,0.88572,0.844121,0.80073,0.755572,0.708685,0.660126,0.609962,0.558277,0.505168,0.450744,0.395125,0.338434,0.280797,0.222344,0.163191,0.103448,0.0432111,-0.0174406,-0.0784417,-0.139746,-0.201317,-0.263131},
	{-3.30688,-3.27755,-3.2482,-3.21883,-3.18942,-3.15999,-3.13054,-3.10105,-3.07153,-3.04197,-3.01238,-2.98275,-2.95308,-2.92336,-2.89361,-2.8638,-2.83395,-2.80404,-2.77407,-2.74405,-2.71397,-2.68382,-2.6536,-2.62331,-2.59294,-2.56249,-2.53195,-2.50133,-2.47061,-2.43979,-2.40887,-2.37784,-2.34669,-2.31542,-2.28402,-2.25248,-2.22081,-2.18898,-2.157,-2.12486,-2.09254,-2.06004,-2.02736,-1.99447,-1.96138,-1.92807,-1.89454,-1.86077,-1.82675,-1.79247,-1.75793,-1.7231,-1.68799,-1.65257,-1.61683,-1.58078,-1.54438,-1.50763,-1.47052,-1.43304,-1.39518,-1.35691,-1.31825,-1.27916,-1.23965,-1.19969,-1.1593,-1.11845,-1.07713,-1.03535,-0.993102,-0.950375,-0.907169,-0.863485,-0.819323,-0.774688,-0.729583,-0.684016,-0.637998,-0.591541,-0.544659,-0.497371,-0.449697,-0.401662,-0.353292,-0.304619,-0.255675,-0.206498,-0.157128,-0.10761,-0.0579882,-0.00831457,0.0413586,0.0909757,0.140479,0.189807,0.238898,0.287688,0.336112,0.384103,0.431595,0.478522,0.524817,0.570415,0.615251,0.659264,0.702394,0.744582,0.785775,0.825921,0.864973,0.902887,0.939624,0.975149,1.00943,1.04245,1.07417,1.10459,1.13369,1.16147,1.18793,1.21306,1.23687,1.25937,1.28058,1.30051,1.31918,1.33661,1.35283,1.36786,1.38174,1.39448,1.40612,1.41669,1.42622,1.43473,1.44226,1.44882,1.45446,1.45919,1.46303,1.466,1.46813,1.46942,1.4699,1.46956,1.46842,1.46648,1.46374,1.4602,1.45586,1.4507,1.4447,1.43787,1.43017,1.42158,1.41209,1.40165,1.39024,1.37782,1.36435,1.34979,1.33411,1.31724,1.29915,1.27978,1.25909,1.23701,1.2135,1.18851,1.16197,1.13385,1.1041,1.07267,1.03951,1.00461,0.967927,0.929448,0.889164,0.847079,0.803207,0.757575,0.710226,0.661219,0.610625,0.558529,0.505032,0.450244,0.394287,0.337284,0.279364,0.22065,0.16126,0.101302,0.0408694,-0.0199637,-0.0811349,-0.142599,-0.204323,-0.266282},
	{-3.31707,-3.28775,-3.2584,-3.22902,-3.19963,-3.1702,-3.14074,-3.11126,-3.08174,-3.05219,-3.0226,-2.99297,-2.9633,-2.93359,-2.90384,-2.87404,-2.84419,-2.81429,-2.78433,-2.75431,-2.72423,-2.69409,-2.66387,-2.63359,-2.60323,-2.57278,-2.54226,-2.51164,-2.48093,-2.45012,-2.4192,-2.38818,-2.35704,-2.32578,-2.29439,-2.26286,-2.23119,-2.19938,-2.16741,-2.13527,-2.10296,-2.07047,-2.0378,-2.00492,-1.97184,-1.93854,-1.90501,-1.87125,-1.83724,-1.80297,-1.76843,-1.7336,-1.69849,-1.66307,-1.62734,-1.59128,-1.55487,-1.51812,-1.481,-1.44351,-1.40562,-1.36734,-1.32865,-1.28953,-1.24998,-1.20999,-1.16954,-1.12864,-1.08727,-1.04542,-1.00309,-0.960275,-0.916975,-0.873185,-0.828907,-0.784144,-0.7389,-0.693181,-0.646997,-0.60036,-0.553283,-0.505785,-0.457886,-0.409608,-0.360979,-0.312029,-0.26279,-0.2133,-0.163599,-0.11373,-0.0637391,-0.0136774,0.0364024,0.0864445,0.13639,0.186179,0.235747,0.28503,0.333962,0.382476,0.430503,0.477976,0.524827,0.570989,0.616397,0.660987,0.704696,0.747466,0.78924,0.829965,0.869592,0.908075,0.945373,0.98145,1.01627,1.04982,1.08206,1.11298,1.14256,1.17081,1.19771,1.22327,1.24748,1.27037,1.29195,1.31222,1.33121,1.34895,1.36544,1.38073,1.39484,1.4078,1.41963,1.43037,1.44005,1.44869,1.45633,1.46299,1.46869,1.47348,1.47735,1.48035,1.48247,1.48375,1.4842,1.48382,1.48261,1.4806,1.47777,1.47413,1.46967,1.46437,1.45824,1.45125,1.44338,1.43462,1.42493,1.41429,1.40266,1.39002,1.37632,1.36151,1.34557,1.32844,1.31007,1.29041,1.26942,1.24704,1.22322,1.1979,1.17104,1.14259,1.11249,1.08071,1.0472,1.01195,0.974907,0.93607,0.895428,0.852988,0.808763,0.762786,0.715099,0.665764,0.614852,0.562453,0.508667,0.453606,0.397392,0.34015,0.282008,0.223088,0.163509,0.103374,0.0427763,-0.018212,-0.0795306,-0.141136,-0.202996,-0.265087},
	{-3.32716,-3.29784,-3.26849,-3.23912,-3.20973,-3.1803,-3.15085,-3.12137,-3.09185,-3.0623,-3.03272,-3.00309,-2.97343,-2.94373,-2.91398,-2.88418,-2.85434,-2.82444,-2.79448,-2.76447,-2.7344,-2.70426,-2.67405,-2.64378,-2.61342,-2.58299,-2.55247,-2.52186,-2.49115,-2.46035,-2.42945,-2.39843,-2.3673,-2.33604,-2.30466,-2.27315,-2.24149,-2.20969,-2.17772,-2.1456,-2.1133,-2.08083,-2.04816,-2.01529,-1.98222,-1.94893,-1.91542,-1.88166,-1.84766,-1.8134,-1.77886,-1.74405,-1.70894,-1.67352,-1.63779,-1.60173,-1.56533,-1.52857,-1.49144,-1.45394,-1.41604,-1.37774,-1.33903,-1.29989,-1.26031,-1.22028,-1.1798,-1.13884,-1.09742,-1.05551,-1.01311,-0.970216,-0.926828,-0.882942,-0.838556,-0.793674,-0.7483,-0.702438,-0.656098,-0.609291,-0.56203,-0.514333,-0.466218,-0.417709,-0.368832,-0.319617,-0.270095,-0.220304,-0.170283,-0.120075,-0.069728,-0.0192908,0.0311832,0.0816379,0.132014,0.182251,0.232284,0.282048,0.331477,0.380501,0.429051,0.477059,0.524456,0.571172,0.617141,0.662298,0.706577,0.749919,0.792265,0.83356,0.873753,0.912797,0.950649,0.987271,1.02263,1.05669,1.08944,1.12085,1.15091,1.17962,1.20696,1.23293,1.25755,1.28082,1.30275,1.32336,1.34266,1.36068,1.37745,1.39298,1.40731,1.42047,1.43248,1.44337,1.45318,1.46194,1.46966,1.47639,1.48215,1.48696,1.49085,1.49384,1.49594,1.49717,1.49756,1.49709,1.4958,1.49367,1.49071,1.48692,1.4823,1.47683,1.4705,1.4633,1.45521,1.4462,1.43625,1.42534,1.41343,1.40048,1.38646,1.37133,1.35504,1.33754,1.3188,1.29875,1.27735,1.25455,1.2303,1.20454,1.17722,1.14829,1.11772,1.08545,1.05145,1.01569,0.978139,0.938793,0.897643,0.854697,0.809973,0.763506,0.715341,0.665543,0.614189,0.561367,0.507184,0.451753,0.395196,0.337639,0.27921,0.22003,0.160215,0.0998667,0.0390723,-0.0220954,-0.0835828,-0.145348,-0.207361,-0.2696},
	{-3.33716,-3.30784,-3.27849,-3.24912,-3.21973,-3.19031,-3.16086,-3.13138,-3.10187,-3.07232,-3.04274,-3.01312,-2.98346,-2.95376,-2.92402,-2.89423,-2.86439,-2.83449,-2.80455,-2.77454,-2.74447,-2.71434,-2.68414,-2.65387,-2.62352,-2.59309,-2.56258,-2.53198,-2.50129,-2.47049,-2.4396,-2.40859,-2.37747,-2.34623,-2.31486,-2.28335,-2.25171,-2.21991,-2.18796,-2.15585,-2.12356,-2.0911,-2.05845,-2.02559,-1.99253,-1.95926,-1.92575,-1.89201,-1.85801,-1.82376,-1.78924,-1.75443,-1.71933,-1.68393,-1.6482,-1.61214,-1.57574,-1.53898,-1.50185,-1.46434,-1.42644,-1.38813,-1.3494,-1.31024,-1.27063,-1.23058,-1.19006,-1.14906,-1.10759,-1.06562,-1.02316,-0.980199,-0.936731,-0.892755,-0.84827,-0.803277,-0.75778,-0.711784,-0.665297,-0.618329,-0.570894,-0.523006,-0.474686,-0.425956,-0.37684,-0.327369,-0.277574,-0.227492,-0.177161,-0.126625,-0.0759307,-0.0251275,0.0257313,0.0765893,0.127387,0.178063,0.228553,0.278791,0.328708,0.378235,0.427303,0.47584,0.523777,0.571043,0.617569,0.663288,0.708136,0.752048,0.794965,0.83683,0.87759,0.917195,0.955603,0.992771,1.02867,1.06326,1.09652,1.12842,1.15897,1.18814,1.21592,1.24233,1.26735,1.29101,1.3133,1.33426,1.35388,1.37221,1.38925,1.40505,1.41961,1.43298,1.44519,1.45626,1.46622,1.47511,1.48295,1.48977,1.4956,1.50047,1.5044,1.50741,1.50952,1.51074,1.5111,1.5106,1.50925,1.50705,1.50401,1.50012,1.49539,1.4898,1.48334,1.47599,1.46774,1.45857,1.44845,1.43735,1.42523,1.41208,1.39783,1.38247,1.36594,1.34819,1.32919,1.30888,1.2872,1.26412,1.23957,1.2135,1.18588,1.15664,1.12574,1.09314,1.05881,1.02271,0.984826,0.945141,0.903652,0.860369,0.815313,0.768517,0.720032,0.669923,0.618268,0.565159,0.510702,0.455012,0.398212,0.340429,0.28179,0.222415,0.162419,0.101902,0.0409497,-0.0203673,-0.0819975,-0.1439,-0.206046,-0.268414},
	{-3.34705,-3.31773,-3.28839,-3.25903,-3.22964,-3.20022,-3.17077,-3.14129,-3.11179,-3.08224,-3.05267,-3.02305,-2.9934,-2.9637,-2.93396,-2.90418,-2.87434,-2.84446,-2.81451,-2.78451,-2.75445,-2.72433,-2.69414,-2.66387,-2.63353,-2.60311,-2.57261,-2.54202,-2.51133,-2.48055,-2.44966,-2.41867,-2.38755,-2.35632,-2.32496,-2.29347,-2.26184,-2.23006,-2.19812,-2.16602,-2.13375,-2.1013,-2.06866,-2.03582,-2.00277,-1.96951,-1.93602,-1.90229,-1.86831,-1.83407,-1.79956,-1.76477,-1.72968,-1.69428,-1.65856,-1.62251,-1.58612,-1.54936,-1.51223,-1.47472,-1.43681,-1.3985,-1.35976,-1.32058,-1.28096,-1.24088,-1.20033,-1.1593,-1.11778,-1.07577,-1.03326,-0.99023,-0.94669,-0.902633,-0.858057,-0.812963,-0.767353,-0.721232,-0.674608,-0.627489,-0.579888,-0.531822,-0.483307,-0.434366,-0.385023,-0.335308,-0.285251,-0.234889,-0.184261,-0.133408,-0.0823784,-0.031221,0.0200108,0.0712605,0.122468,0.173573,0.224508,0.275208,0.325603,0.375624,0.425198,0.474255,0.522723,0.57053,0.617605,0.66388,0.709288,0.753764,0.797245,0.839675,0.880997,0.92116,0.960119,0.997831,1.03426,1.06937,1.10314,1.13555,1.16657,1.1962,1.22444,1.25127,1.2767,1.30074,1.32341,1.34471,1.36466,1.38329,1.40061,1.41666,1.43147,1.44506,1.45746,1.4687,1.47882,1.48784,1.4958,1.50272,1.50863,1.51356,1.51753,1.52056,1.52268,1.5239,1.52423,1.52369,1.52229,1.52003,1.51691,1.51294,1.5081,1.5024,1.49581,1.48832,1.47993,1.47059,1.4603,1.44902,1.43671,1.42335,1.4089,1.39332,1.37656,1.35857,1.33932,1.31875,1.29681,1.27345,1.24862,1.22226,1.19434,1.16479,1.13358,1.10067,1.06601,1.02959,0.991377,0.951363,0.909545,0.865934,0.820553,0.773438,0.724641,0.674228,0.62228,0.568889,0.514165,0.458223,0.401187,0.343182,0.284338,0.224772,0.164599,0.103917,0.0428091,-0.0186551,-0.080426,-0.142464,-0.204742,-0.267238},
	{-3.35685,-3.32754,-3.2982,-3.26884,-3.23945,-3.21003,-3.18059,-3.15112,-3.12161,-3.09207,-3.0625,-3.03289,-3.00324,-2.97355,-2.94382,-2.91404,-2.88421,-2.85433,-2.82439,-2.7944,-2.76434,-2.73423,-2.70404,-2.67378,-2.64345,-2.61304,-2.58255,-2.55197,-2.52129,-2.49052,-2.45964,-2.42866,-2.39756,-2.36634,-2.33499,-2.30351,-2.27189,-2.24012,-2.2082,-2.17611,-2.14386,-2.11142,-2.0788,-2.04597,-2.01294,-1.9797,-1.94622,-1.91251,-1.87854,-1.84432,-1.80983,-1.77505,-1.73997,-1.70459,-1.66888,-1.63284,-1.59645,-1.55971,-1.52259,-1.48508,-1.44717,-1.40885,-1.37011,-1.33092,-1.29129,-1.25119,-1.21062,-1.16956,-1.12801,-1.08596,-1.0434,-1.00032,-0.956712,-0.912583,-0.867925,-0.822739,-0.777026,-0.73079,-0.684039,-0.63678,-0.589026,-0.540791,-0.492093,-0.442954,-0.393396,-0.343449,-0.293144,-0.242515,-0.191601,-0.140445,-0.0890934,-0.0375952,0.0139962,0.0656241,0.117229,0.168748,0.220116,0.271265,0.322125,0.372626,0.422696,0.472261,0.521248,0.569584,0.617198,0.664018,0.709976,0.755006,0.799044,0.842029,0.883904,0.924617,0.96412,1.00237,1.03932,1.07495,1.10923,1.14212,1.17362,1.20371,1.23239,1.25964,1.28548,1.3099,1.33293,1.35457,1.37484,1.39376,1.41137,1.42767,1.44271,1.45651,1.4691,1.48051,1.49077,1.49992,1.50798,1.51499,1.52097,1.52595,1.52995,1.533,1.53512,1.53632,1.53662,1.53603,1.53457,1.53223,1.52902,1.52493,1.51997,1.51413,1.5074,1.49975,1.49118,1.48167,1.47117,1.45968,1.44716,1.43356,1.41887,1.40303,1.386,1.36773,1.34819,1.32731,1.30506,1.28138,1.25621,1.22951,1.20123,1.17132,1.13974,1.10644,1.0714,1.03458,0.995976,0.955564,0.913349,0.869342,0.823569,0.77607,0.726897,0.676121,0.623824,0.570101,0.515062,0.458825,0.401515,0.343258,0.284181,0.224402,0.164034,0.103171,0.041896,-0.0197252,-0.0816457,-0.143827,-0.206244,-0.268873},
	{-3.36656,-3.33725,-3.30791,-3.27855,-3.24917,-3.21976,-3.19032,-3.16085,-3.13135,-3.10181,-3.07224,-3.04264,-3.013,-2.98331,-2.95358,-2.92381,-2.89398,-2.86411,-2.83418,-2.80419,-2.77415,-2.74404,-2.71386,-2.68361,-2.65329,-2.62289,-2.5924,-2.56183,-2.53116,-2.5004,-2.46954,-2.43856,-2.40748,-2.37627,-2.34494,-2.31347,-2.28187,-2.25011,-2.21821,-2.18614,-2.15389,-2.12148,-2.08887,-2.05606,-2.02305,-1.98982,-1.95636,-1.92267,-1.88872,-1.85452,-1.82004,-1.78528,-1.75022,-1.71485,-1.67916,-1.64314,-1.60676,-1.57003,-1.53292,-1.49542,-1.45752,-1.4192,-1.38045,-1.34127,-1.30163,-1.26152,-1.22093,-1.17985,-1.13828,-1.09619,-1.05359,-1.01046,-0.966801,-0.922608,-0.877877,-0.832608,-0.786801,-0.740461,-0.693592,-0.646203,-0.598306,-0.549914,-0.501044,-0.451717,-0.401956,-0.351788,-0.301245,-0.25036,-0.199173,-0.147726,-0.0960631,-0.0442354,0.00770439,0.0596996,0.11169,0.163613,0.215403,0.266992,0.318309,0.369281,0.419837,0.469902,0.519401,0.56826,0.616407,0.663767,0.710272,0.755852,0.800443,0.843981,0.886409,0.927672,0.967719,1.00651,1.04399,1.08014,1.11492,1.14831,1.18029,1.21085,1.23997,1.26765,1.2939,1.31872,1.34212,1.36411,1.38471,1.40395,1.42184,1.43842,1.4537,1.46773,1.48053,1.49213,1.50256,1.51186,1.52005,1.52718,1.53325,1.53831,1.54238,1.54548,1.54763,1.54885,1.54916,1.54856,1.54708,1.5447,1.54145,1.53732,1.5323,1.52638,1.51956,1.51183,1.50316,1.49353,1.48293,1.47131,1.45865,1.44492,1.43008,1.41409,1.3969,1.37847,1.35876,1.33771,1.31527,1.29139,1.26603,1.23913,1.21065,1.18053,1.14873,1.11521,1.07995,1.04291,1.00407,0.963425,0.920977,0.876738,0.830735,0.783007,0.733609,0.682612,0.630099,0.576168,0.52093,0.464501,0.407008,0.348577,0.289336,0.229402,0.168885,0.107882,0.0464715,-0.0152799,-0.0773269,-0.139632,-0.202169,-0.264917},
	{-3.37618,-3.34687,-3.31754,-3.28818,-3.2588,-3.22939,-3.19995,-3.17049,-3.14099,-3.11146,-3.0819,-3.0523,-3.02266,-2.99298,-2.96325,-2.93349,-2.90367,-2.8738,-2.84388,-2.8139,-2.78386,-2.75376,-2.72359,-2.69335,-2.66304,-2.63264,-2.60217,-2.57161,-2.54095,-2.5102,-2.47935,-2.44839,-2.41732,-2.38612,-2.3548,-2.32335,-2.29176,-2.26003,-2.22814,-2.19608,-2.16386,-2.13146,-2.09887,-2.06608,-2.03309,-1.99988,-1.96644,-1.93277,-1.89884,-1.86466,-1.8302,-1.79546,-1.76042,-1.72507,-1.6894,-1.6534,-1.61704,-1.58032,-1.54323,-1.50574,-1.46785,-1.42954,-1.3908,-1.35162,-1.31198,-1.27186,-1.23127,-1.19018,-1.14858,-1.10647,-1.06384,-1.02067,-0.976964,-0.932717,-0.887923,-0.842581,-0.796691,-0.750256,-0.703282,-0.655775,-0.607746,-0.559209,-0.51018,-0.460678,-0.410726,-0.360352,-0.309585,-0.258459,-0.207013,-0.155288,-0.103329,-0.0511866,0.00108674,0.0534343,0.105796,0.158109,0.210306,0.262319,0.314078,0.365508,0.416536,0.467087,0.517085,0.566455,0.615121,0.66301,0.710049,0.756168,0.801301,0.845383,0.888354,0.930156,0.970739,1.01006,1.04806,1.08472,1.12001,1.15388,1.18634,1.21735,1.24692,1.27502,1.30168,1.32688,1.35065,1.37299,1.39392,1.41346,1.43164,1.44848,1.46401,1.47826,1.49126,1.50304,1.51363,1.52308,1.5314,1.53862,1.54479,1.54992,1.55404,1.55718,1.55936,1.56059,1.56089,1.56028,1.55876,1.55634,1.55303,1.54883,1.54373,1.53773,1.53081,1.52296,1.51417,1.50442,1.49367,1.4819,1.46909,1.45519,1.44017,1.42399,1.40661,1.38798,1.36805,1.34678,1.32411,1.3,1.2744,1.24724,1.21849,1.1881,1.15603,1.12223,1.08668,1.04934,1.01021,0.969266,0.926519,0.88198,0.835678,0.787657,0.73797,0.686693,0.63391,0.579718,0.524232,0.467569,0.409857,0.35122,0.291788,0.231675,0.170992,0.109833,0.0482752,-0.0136167,-0.0757989,-0.138235,-0.2009,-0.263772},
	{-3.38571,-3.3564,-3.32707,-3.29772,-3.26834,-3.23893,-3.2095,-3.18004,-3.15055,-3.12102,-3.09146,-3.06187,-3.03223,-3.00256,-2.97284,-2.94308,-2.91327,-2.88341,-2.85349,-2.82352,-2.79349,-2.76339,-2.73323,-2.703,-2.6727,-2.64232,-2.61185,-2.5813,-2.55066,-2.51992,-2.48908,-2.45814,-2.42708,-2.3959,-2.3646,-2.33316,-2.30159,-2.26987,-2.23799,-2.20596,-2.17376,-2.14137,-2.1088,-2.07604,-2.04307,-2.00988,-1.97646,-1.94281,-1.90891,-1.87475,-1.84031,-1.8056,-1.77058,-1.73526,-1.69961,-1.66362,-1.62729,-1.59059,-1.55352,-1.51605,-1.47817,-1.43988,-1.40115,-1.36198,-1.32234,-1.28223,-1.24163,-1.20053,-1.15893,-1.1168,-1.07414,-1.03095,-0.987206,-0.942914,-0.898067,-0.852663,-0.806701,-0.760184,-0.713116,-0.665503,-0.617356,-0.568687,-0.519511,-0.469848,-0.41972,-0.369154,-0.318178,-0.266826,-0.215136,-0.163149,-0.11091,-0.058469,-0.0058782,0.0468055,0.099522,0.152208,0.204796,0.257218,0.309402,0.361274,0.412759,0.463781,0.514262,0.564127,0.613298,0.6617,0.70926,0.755905,0.801567,0.84618,0.889681,0.932012,0.973119,1.01295,1.05147,1.08863,1.12441,1.15876,1.19168,1.22315,1.25314,1.28167,1.30872,1.3343,1.35843,1.38111,1.40235,1.42219,1.44065,1.45774,1.4735,1.48796,1.50116,1.51311,1.52386,1.53343,1.54186,1.54918,1.55542,1.56061,1.56477,1.56794,1.57012,1.57134,1.57162,1.57097,1.5694,1.56691,1.56352,1.55923,1.55402,1.5479,1.54085,1.53286,1.52391,1.51398,1.50305,1.49109,1.47807,1.46396,1.44871,1.43229,1.41465,1.39576,1.37555,1.35399,1.33102,1.30659,1.28065,1.25316,1.22406,1.19331,1.16086,1.12668,1.09074,1.05301,1.01347,0.972123,0.92897,0.884027,0.837326,0.788912,0.738842,0.687194,0.634056,0.579527,0.523722,0.466763,0.408776,0.349887,0.290223,0.2299,0.169024,0.107687,0.0459644,-0.016083,-0.0784135,-0.140991,-0.203796,-0.266801},
	{-3.39515,-3.36584,-3.33652,-3.30717,-3.27779,-3.24839,-3.21896,-3.18951,-3.16002,-3.1305,-3.10094,-3.07135,-3.04172,-3.01205,-2.98234,-2.95259,-2.92278,-2.89293,-2.86302,-2.83306,-2.80303,-2.77295,-2.7428,-2.71258,-2.68228,-2.65191,-2.62146,-2.59092,-2.56029,-2.52957,-2.49874,-2.46781,-2.43676,-2.4056,-2.37431,-2.34289,-2.31134,-2.27964,-2.24778,-2.21577,-2.18358,-2.15122,-2.11868,-2.08593,-2.05298,-2.01982,-1.98643,-1.9528,-1.91893,-1.88479,-1.85038,-1.81569,-1.78071,-1.74541,-1.70979,-1.67383,-1.63752,-1.60085,-1.56379,-1.52635,-1.4885,-1.45022,-1.41151,-1.37235,-1.33273,-1.29262,-1.25203,-1.21094,-1.16933,-1.12719,-1.08452,-1.0413,-0.997534,-0.953207,-0.908317,-0.86286,-0.816837,-0.770248,-0.723097,-0.67539,-0.627135,-0.578346,-0.529036,-0.479225,-0.428933,-0.378186,-0.327014,-0.275449,-0.223528,-0.171292,-0.118785,-0.066058,-0.0131622,0.0398454,0.0929047,0.145952,0.19892,0.25174,0.304339,0.356642,0.408575,0.460059,0.511016,0.561369,0.611039,0.659949,0.708025,0.755193,0.801382,0.846524,0.890555,0.933415,0.975048,1.0154,1.05444,1.09211,1.12838,1.16322,1.19661,1.22853,1.25898,1.28793,1.31539,1.34137,1.36587,1.3889,1.41049,1.43065,1.4494,1.46677,1.48279,1.49749,1.51091,1.52306,1.534,1.54374,1.55232,1.55978,1.56614,1.57143,1.57568,1.57892,1.58116,1.58243,1.58275,1.58212,1.58057,1.5781,1.5747,1.5704,1.56517,1.55902,1.55194,1.5439,1.53491,1.52493,1.51394,1.50192,1.48883,1.47463,1.45931,1.4428,1.42507,1.40607,1.38577,1.3641,1.34102,1.31647,1.29041,1.26279,1.23356,1.20267,1.17008,1.13576,1.09966,1.06178,1.02208,0.980574,0.937256,0.892146,0.845276,0.796693,0.746455,0.69464,0.641336,0.586645,0.530683,0.473569,0.415434,0.356401,0.296599,0.236142,0.175137,0.113675,0.0518299,-0.0103368,-0.0727844,-0.135477,-0.198395,-0.261512},
	{-3.4045,-3.3752,-3.34588,-3.31653,-3.28716,-3.25776,-3.22834,-3.19889,-3.1694,-3.13989,-3.11034,-3.08075,-3.05113,-3.02147,-2.99176,-2.96201,-2.93221,-2.90237,-2.87247,-2.84251,-2.8125,-2.78242,-2.75228,-2.72207,-2.69179,-2.66143,-2.63099,-2.60046,-2.56984,-2.53913,-2.50832,-2.4774,-2.44637,-2.41523,-2.38395,-2.35255,-2.32102,-2.28934,-2.2575,-2.22551,-2.19335,-2.16101,-2.12848,-2.09577,-2.06284,-2.0297,-1.99634,-1.96274,-1.92889,-1.89479,-1.86041,-1.82575,-1.79079,-1.75552,-1.71993,-1.684,-1.64772,-1.61108,-1.57405,-1.53664,-1.49881,-1.46056,-1.42187,-1.38274,-1.34313,-1.30305,-1.26247,-1.22138,-1.17977,-1.13764,-1.09496,-1.05174,-1.00795,-0.963599,-0.918677,-0.873181,-0.827108,-0.78046,-0.733239,-0.685452,-0.637105,-0.58821,-0.538782,-0.488838,-0.438399,-0.387489,-0.336138,-0.284378,-0.232244,-0.179777,-0.127023,-0.0740292,-0.0208483,0.032463,0.0858444,0.139232,0.192559,0.245756,0.298749,0.351463,0.403821,0.455746,0.507158,0.557977,0.608125,0.657524,0.706095,0.753764,0.800459,0.84611,0.890651,0.934021,0.976161,1.01702,1.05655,1.0947,1.13145,1.16676,1.20061,1.23297,1.26384,1.2932,1.32106,1.34741,1.37226,1.39563,1.41753,1.43798,1.45701,1.47464,1.4909,1.50581,1.51942,1.53175,1.54284,1.55271,1.56141,1.56897,1.57541,1.58076,1.58506,1.58833,1.59059,1.59186,1.59217,1.59151,1.58992,1.58739,1.58392,1.57953,1.57421,1.56795,1.56075,1.55258,1.54344,1.53331,1.52215,1.50994,1.49665,1.48225,1.4667,1.44996,1.43198,1.41273,1.39215,1.37019,1.3468,1.32194,1.29555,1.26759,1.238,1.20674,1.17377,1.13905,1.10255,1.06426,1.02414,0.982214,0.938474,0.892944,0.845658,0.796665,0.746028,0.693828,0.640156,0.585116,0.528826,0.471409,0.412992,0.353702,0.293666,0.232996,0.171797,0.110157,0.0481464,-0.0141762,-0.0767727,-0.139609,-0.202666,-0.265917}
};

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

//LS coefficient for A=atomic weight * 1.05
const G4double G4AtimaEnergyLossModel::ls_coefficients_ahi[110][200]=
{
	{-0.0286005,-0.0269762,-0.0254384,-0.0239826,-0.0226044,-0.0212998,-0.0200648,-0.0188958,-0.0177891,-0.0167414,-0.0157495,-0.0148105,-0.0139213,-0.0130793,-0.0122818,-0.0115265,-0.0108109,-0.0101329,-0.00949027,-0.00888115,-0.0083036,-0.00775584,-0.00723615,-0.00674295,-0.0062747,-0.00582997,-0.00540738,-0.00500564,-0.00462354,-0.00425989,-0.00391361,-0.00358365,-0.00326903,-0.0029688,-0.00268208,-0.00240804,-0.00214586,-0.00189481,-0.00165416,-0.00142324,-0.0012014,-0.000988041,-0.000782573,-0.000584452,-0.000393159,-0.0002082,-2.91055e-05,0.000144568,0.000313242,0.000477319,0.000637177,0.000793179,0.000945667,0.00109497,0.00124139,0.00138522,0.00152675,0.00166624,0.00180394,0.00194008,0.0020749,0.00220859,0.00234138,0.00247345,0.00260497,0.00273611,0.00286704,0.00299791,0.00312884,0.00325998,0.00339143,0.00352331,0.00365572,0.00378874,0.00392247,0.00405695,0.00419227,0.00432847,0.00446558,0.00460364,0.00474267,0.00488268,0.00502366,0.00516561,0.00530849,0.00545228,0.00559694,0.00574239,0.00588858,0.00603543,0.00618284,0.00633071,0.00647894,0.0066274,0.00677595,0.00692445,0.00707275,0.0072207,0.00736812,0.00751483,0.00766066,0.00780541,0.0079489,0.00809094,0.00823133,0.00836987,0.00850638,0.00864065,0.00877251,0.00890177,0.00902826,0.00915181,0.00927227,0.00938949,0.00950334,0.00961369,0.00972045,0.00982352,0.00992282,0.0100183,0.0101099,0.0101976,0.0102814,0.0103612,0.0104372,0.0105093,0.0105776,0.010642,0.0107028,0.01076,0.0108136,0.0108637,0.0109105,0.010954,0.0109943,0.0110316,0.0110659,0.0110974,0.0111262,0.0111523,0.0111759,0.011197,0.0112157,0.0112322,0.0112464,0.0112584,0.0112682,0.011276,0.0112816,0.0112851,0.0112865,0.0112857,0.0112827,0.0112774,0.0112698,0.0112596,0.0112467,0.011231,0.0112123,0.0111902,0.0111645,0.0111349,0.0111009,0.0110622,0.0110182,0.0109684,0.0109121,0.0108486,0.0107772,0.0106967,0.0106064,0.0105049,0.010391,0.0102631,0.0101197,0.00995886,0.00977852,0.00957633,0.00934967,0.00909558,0.00881078,0.00849156,0.00813377,0.00773276,0.00728332,0.00677961,0.00621509,0.00558242,0.00487342,0.00407938,0.00318934,0.00219215,0.00107501,-0.000176368,-0.00157794,-0.00314752,-0.00490495,-0.00687157,-0.00907295,-0.0115359},
	{-0.108683,-0.102922,-0.097437,-0.0922162,-0.0872489,-0.0825243,-0.0780321,-0.073762,-0.0697043,-0.0658493,-0.0621879,-0.0587109,-0.0554098,-0.0522762,-0.049302,-0.0464796,-0.0438014,-0.0412604,-0.0388496,-0.0365626,-0.0343929,-0.0323346,-0.0303819,-0.0285294,-0.0267717,-0.0251038,-0.023521,-0.0220186,-0.0205924,-0.0192382,-0.017952,-0.0167301,-0.015569,-0.0144651,-0.0134154,-0.0124167,-0.0114661,-0.0105608,-0.00969815,-0.00887573,-0.00809113,-0.00734208,-0.00662645,-0.0059422,-0.00528739,-0.00466021,-0.0040589,-0.00348182,-0.0029274,-0.00239416,-0.00188068,-0.00138563,-0.000907748,-0.000445819,1.29566e-06,0.000434677,0.00085535,0.00126429,0.00166242,0.0020506,0.00242967,0.00280041,0.00316356,0.00351982,0.00386984,0.00421425,0.00455363,0.00488854,0.00521949,0.00554697,0.00587143,0.00619329,0.00651293,0.00683073,0.007147,0.00746206,0.00777618,0.00808959,0.00840253,0.00871516,0.00902767,0.00934017,0.00965278,0.00996557,0.0102786,0.0105919,0.0109054,0.0112191,0.011533,0.011847,0.0121609,0.0124746,0.012788,0.0131009,0.013413,0.0137241,0.014034,0.0143424,0.014649,0.0149535,0.0152555,0.0155547,0.0158509,0.0161435,0.0164323,0.016717,0.016997,0.0172722,0.0175421,0.0178064,0.0180647,0.0183169,0.0185625,0.0188013,0.019033,0.0192574,0.0194744,0.0196838,0.0198853,0.020079,0.0202647,0.0204423,0.0206119,0.0207735,0.0209271,0.0210727,0.0212105,0.0213406,0.0214631,0.0215781,0.0216858,0.0217865,0.0218802,0.0219673,0.0220478,0.022122,0.0221902,0.0222524,0.022309,0.02236,0.0224057,0.0224462,0.0224817,0.0225122,0.0225379,0.0225589,0.0225751,0.0225867,0.0225936,0.0225956,0.0225929,0.0225851,0.0225721,0.0225537,0.0225296,0.0224994,0.0224628,0.0224191,0.0223678,0.0223083,0.0222398,0.0221614,0.0220721,0.0219707,0.021856,0.0217264,0.0215804,0.0214161,0.0212313,0.0210237,0.0207907,0.0205292,0.020236,0.0199071,0.0195385,0.0191254,0.0186624,0.0181438,0.0175626,0.0169117,0.0161824,0.015366,0.0144514,0.0134272,0.0122803,0.0109964,0.00955909,0.00795034,0.00615,0.00413605,0.00188305,-0.000636415,-0.00345299,-0.00660058,-0.0101166,-0.0140447,-0.0184274,-0.0233159,-0.0287653,-0.0348352},
	{-0.223884,-0.213009,-0.202568,-0.192553,-0.182953,-0.173755,-0.16495,-0.156525,-0.148469,-0.140769,-0.133415,-0.126394,-0.119694,-0.113303,-0.10721,-0.101404,-0.0958714,-0.0906027,-0.0855866,-0.0808123,-0.0762693,-0.0719475,-0.0678369,-0.063928,-0.0602115,-0.0566782,-0.0533196,-0.0501272,-0.0470929,-0.0442089,-0.0414678,-0.0388623,-0.0363856,-0.034031,-0.0317921,-0.029663,-0.0276377,-0.0257108,-0.0238769,-0.0221309,-0.0204681,-0.0188837,-0.0173735,-0.0159331,-0.0145586,-0.0132462,-0.0119923,-0.0107934,-0.00964629,-0.00854775,-0.00749487,-0.00648481,-0.0055149,-0.00458258,-0.00368545,-0.0028212,-0.00198765,-0.00118273,-0.000404483,0.00034896,0.00107936,0.00178839,0.00247764,0.00314859,0.00380267,0.00444122,0.00506549,0.00567668,0.00627591,0.00686422,0.00744262,0.00801201,0.00857327,0.0091272,0.00967454,0.010216,0.0107521,0.0112836,0.0118109,0.0123346,0.0128549,0.0133724,0.0138872,0.0143997,0.0149101,0.0154186,0.0159253,0.0164302,0.0169334,0.0174348,0.0179344,0.0184321,0.0189277,0.0194211,0.0199119,0.0203999,0.0208849,0.0213663,0.021844,0.0223174,0.0227862,0.0232499,0.023708,0.0241601,0.0246056,0.025044,0.0254749,0.0258978,0.0263122,0.0267176,0.0271135,0.0274995,0.0278752,0.0282403,0.0285943,0.028937,0.0292681,0.0295873,0.0298945,0.0301896,0.0304723,0.0307428,0.0310009,0.0312467,0.0314802,0.0317017,0.0319111,0.0321088,0.0322949,0.0324696,0.0326333,0.0327862,0.0329286,0.0330608,0.0331831,0.0332959,0.0333994,0.0334941,0.03358,0.0336577,0.0337273,0.033789,0.0338431,0.0338898,0.0339292,0.0339615,0.0339867,0.0340048,0.0340158,0.0340197,0.0340164,0.0340055,0.0339869,0.0339602,0.033925,0.0338807,0.0338268,0.0337624,0.0336869,0.033599,0.0334978,0.0333819,0.0332498,0.0330999,0.0329301,0.0327385,0.0325225,0.0322794,0.032006,0.031699,0.0313544,0.0309677,0.0305341,0.0300481,0.0295034,0.0288931,0.0282095,0.0274437,0.0265864,0.0256263,0.0245514,0.0233482,0.0220015,0.0204944,0.0188081,0.0169217,0.0148122,0.0124533,0.00981647,0.00686995,0.00357859,-9.64072e-05,-0.00419959,-0.0087756,-0.0138772,-0.0195608,-0.0258898,-0.0329293,-0.0407528,-0.0494389},
	{-0.356328,-0.340686,-0.325539,-0.310886,-0.296723,-0.283047,-0.269853,-0.257134,-0.244884,-0.233095,-0.221759,-0.210866,-0.200406,-0.19037,-0.180747,-0.171526,-0.162694,-0.154242,-0.146157,-0.138426,-0.131039,-0.123984,-0.117247,-0.110818,-0.104685,-0.0988357,-0.0932592,-0.0879442,-0.0828796,-0.0780547,-0.073459,-0.0690824,-0.0649147,-0.0609465,-0.0571684,-0.0535713,-0.0501465,-0.0468855,-0.0437803,-0.040823,-0.0380061,-0.0353223,-0.0327648,-0.0303268,-0.028002,-0.0257842,-0.0236677,-0.0216467,-0.0197161,-0.0178706,-0.0161054,-0.0144159,-0.0127976,-0.0112463,-0.00975799,-0.00832879,-0.00695508,-0.00563338,-0.0043604,-0.00313301,-0.00194821,-0.000803192,0.000304733,0.00137811,0.00241935,0.00343072,0.00441439,0.0053724,0.00630667,0.007219,0.00811112,0.0089846,0.00984097,0.0106816,0.0115079,0.0123209,0.0131219,0.0139119,0.0146917,0.0154624,0.0162247,0.0169793,0.0177268,0.0184678,0.0192028,0.0199321,0.0206561,0.021375,0.022089,0.0227982,0.0235028,0.0242025,0.0248974,0.0255873,0.026272,0.0269512,0.0276247,0.028292,0.0289527,0.0296064,0.0302525,0.0308906,0.0315201,0.0321404,0.0327509,0.033351,0.0339401,0.0345176,0.035083,0.0356355,0.0361747,0.0367,0.0372109,0.037707,0.0381877,0.0386529,0.039102,0.0395348,0.0399512,0.0403509,0.0407338,0.0410999,0.0414493,0.0417819,0.0420979,0.0423974,0.0426807,0.0429481,0.0431998,0.0434362,0.0436577,0.0438646,0.0440575,0.0442366,0.0444025,0.0445555,0.0446962,0.044825,0.0449423,0.0450484,0.0451438,0.0452288,0.0453038,0.045369,0.0454246,0.0454708,0.0455079,0.0455357,0.0455545,0.045564,0.0455642,0.0455549,0.0455357,0.0455064,0.0454663,0.0454149,0.0453514,0.045275,0.0451846,0.045079,0.044957,0.0448168,0.0446568,0.0444748,0.0442686,0.0440355,0.0437726,0.0434765,0.0431435,0.0427694,0.0423493,0.041878,0.0413495,0.0407571,0.0400932,0.0393495,0.0385167,0.0375841,0.0365399,0.0353712,0.0340631,0.0325995,0.030962,0.0291304,0.0270824,0.0247926,0.0222334,0.019374,0.0161802,0.0126146,0.00863444,0.0041956,-0.000752743,-0.0062655,-0.0124026,-0.0192308,-0.0268188,-0.0352438,-0.0445879,-0.0549357},
	{-0.492346,-0.472912,-0.453947,-0.435458,-0.417451,-0.399931,-0.382902,-0.366367,-0.350327,-0.334782,-0.319731,-0.305173,-0.291103,-0.277517,-0.264411,-0.251777,-0.239609,-0.227898,-0.216636,-0.205814,-0.195421,-0.185447,-0.175882,-0.166715,-0.157933,-0.149525,-0.141479,-0.133784,-0.126427,-0.119396,-0.11268,-0.106266,-0.100143,-0.0942985,-0.0887222,-0.0834024,-0.0783281,-0.0734886,-0.0688734,-0.0644723,-0.0602755,-0.0562733,-0.0524563,-0.0488157,-0.0453427,-0.0420289,-0.0388662,-0.0358468,-0.0329632,-0.0302083,-0.0275751,-0.025057,-0.0226477,-0.020341,-0.0181313,-0.0160128,-0.0139804,-0.0120289,-0.0101536,-0.0083497,-0.00661296,-0.00493915,-0.00332431,-0.00176466,-0.000256614,0.00120324,0.00261812,0.0039911,0.00532505,0.00662272,0.00788668,0.00911936,0.010323,0.0114999,0.0126519,0.0137809,0.0148887,0.0159769,0.0170469,0.0181003,0.0191382,0.0201618,0.0211721,0.02217,0.0231564,0.0241319,0.0250972,0.0260528,0.0269991,0.0279364,0.0288649,0.0297847,0.0306959,0.0315985,0.0324922,0.0333769,0.0342523,0.0351181,0.0359738,0.0368189,0.037653,0.0384755,0.0392858,0.0400831,0.040867,0.0416366,0.0423912,0.0431303,0.0438531,0.0445589,0.045247,0.045917,0.0465681,0.0471999,0.0478118,0.0484035,0.0489745,0.0495245,0.0500534,0.0505609,0.051047,0.0515115,0.0519547,0.0523765,0.0527772,0.0531569,0.0535161,0.053855,0.0541741,0.0544738,0.0547546,0.055017,0.0552616,0.0554889,0.0556995,0.055894,0.0560729,0.0562369,0.0563864,0.0565221,0.0566443,0.0567535,0.0568503,0.0569349,0.0570077,0.0570689,0.0571188,0.0571575,0.057185,0.0572014,0.0572064,0.0572,0.0571818,0.0571514,0.0571082,0.0570515,0.0569806,0.0568945,0.0567919,0.0566715,0.0565318,0.056371,0.0561869,0.0559773,0.0557395,0.0554705,0.0551669,0.0548247,0.0544398,0.0540072,0.0535214,0.0529763,0.0523651,0.0516799,0.0509123,0.0500525,0.0490896,0.0480117,0.0468052,0.0454552,0.0439447,0.0422552,0.0403658,0.0382536,0.0358927,0.0332547,0.0303081,0.0270179,0.0233458,0.0192485,0.0146804,0.00959004,0.00392131,-0.00238673,-0.00940171,-0.0171937,-0.0258408,-0.035426,-0.0460352,-0.0577651},
	{-0.623826,-0.601642,-0.579854,-0.558476,-0.537521,-0.517,-0.496926,-0.477307,-0.458152,-0.439471,-0.421268,-0.403551,-0.386322,-0.369586,-0.353343,-0.337595,-0.32234,-0.307577,-0.293302,-0.279513,-0.266202,-0.253366,-0.240995,-0.229084,-0.217623,-0.206603,-0.196014,-0.185847,-0.176089,-0.166731,-0.157761,-0.149166,-0.140936,-0.133058,-0.125521,-0.118311,-0.111418,-0.104829,-0.0985324,-0.0925163,-0.0867694,-0.0812804,-0.076038,-0.0710314,-0.0662501,-0.0616837,-0.0573221,-0.0531557,-0.0491749,-0.0453707,-0.0417341,-0.0382567,-0.0349302,-0.0317467,-0.0286985,-0.0257785,-0.0229794,-0.0202947,-0.0177178,-0.0152426,-0.0128632,-0.0105739,-0.00836933,-0.00624439,-0.00419417,-0.00221402,-0.000299511,0.00155356,0.00334917,0.0050911,0.00678294,0.00842805,0.0100296,0.0115907,0.013114,0.0146022,0.0160578,0.0174831,0.0188803,0.0202512,0.0215978,0.0229217,0.0242245,0.0255074,0.0267718,0.0280188,0.0292492,0.030464,0.0316639,0.0328493,0.0340208,0.0351786,0.0363231,0.0374543,0.0385721,0.0396766,0.0407674,0.0418444,0.042907,0.043955,0.0449877,0.0460046,0.047005,0.0479884,0.0489539,0.0499008,0.0508284,0.051736,0.0526228,0.053488,0.0543309,0.0551509,0.0559473,0.0567196,0.0574671,0.0581895,0.0588864,0.0595573,0.0602022,0.0608208,0.061413,0.0619788,0.0625184,0.063032,0.0635197,0.0639818,0.0644189,0.0648313,0.0652196,0.0655843,0.0659261,0.0662456,0.0665434,0.0668204,0.0670771,0.0673144,0.0675329,0.0677333,0.0679164,0.0680828,0.0682331,0.0683679,0.0684878,0.0685932,0.0686846,0.0687624,0.0688267,0.068878,0.0689161,0.0689413,0.0689534,0.0689523,0.0689376,0.068909,0.0688658,0.0688074,0.0687329,0.0686413,0.0685312,0.0684013,0.0682498,0.0680748,0.067874,0.0676449,0.0673846,0.0670898,0.0667567,0.0663812,0.0659584,0.0654831,0.0649493,0.0643503,0.0636785,0.0629254,0.0620816,0.0611367,0.0600787,0.0588944,0.0575691,0.0560865,0.0544281,0.0525738,0.0505008,0.0481841,0.0455957,0.0427048,0.0394772,0.0358754,0.031857,0.0273774,0.022386,0.0168279,0.0106434,0.00376658,-0.00387188,-0.0123484,-0.0217445,-0.0321454,-0.0436452,-0.0563395},
	{-0.747011,-0.722922,-0.699148,-0.675705,-0.652609,-0.629874,-0.607515,-0.585548,-0.563985,-0.542839,-0.522123,-0.501848,-0.482024,-0.462662,-0.443768,-0.425351,-0.407415,-0.389966,-0.373007,-0.35654,-0.340566,-0.325085,-0.310095,-0.295594,-0.281577,-0.26804,-0.254978,-0.242383,-0.230248,-0.218565,-0.207324,-0.196516,-0.186132,-0.176159,-0.166588,-0.157406,-0.148603,-0.140166,-0.132084,-0.124343,-0.116933,-0.109841,-0.103055,-0.0965628,-0.0903532,-0.0844142,-0.0787344,-0.0733026,-0.0681079,-0.0631394,-0.0583866,-0.0538395,-0.0494881,-0.0453229,-0.0413345,-0.037514,-0.0338527,-0.0303422,-0.0269745,-0.023742,-0.020637,-0.0176526,-0.0147819,-0.0120184,-0.00935571,-0.00678798,-0.00430949,-0.00191482,0.00040121,0.0026435,0.0048167,0.00692523,0.00897326,0.0109647,0.0129034,0.0147927,0.0166359,0.0184362,0.0201963,0.0219189,0.0236066,0.0252615,0.0268858,0.0284814,0.03005,0.0315932,0.0331122,0.0346084,0.0360828,0.0375363,0.0389696,0.0403833,0.0417779,0.0431536,0.0445106,0.045849,0.0471686,0.0484694,0.0497509,0.0510129,0.0522548,0.0534761,0.0546761,0.0558543,0.0570098,0.0581419,0.0592498,0.0603328,0.06139,0.0624207,0.0634241,0.0643994,0.0653462,0.0662636,0.0671511,0.0680083,0.0688348,0.0696302,0.0703943,0.0711269,0.0718281,0.0724979,0.0731364,0.0737438,0.0743206,0.074867,0.0753837,0.0758711,0.0763299,0.0767609,0.0771647,0.0775421,0.077894,0.0782212,0.0785246,0.078805,0.0790633,0.0793003,0.0795169,0.0797139,0.0798919,0.0800518,0.0801941,0.0803195,0.0804285,0.0805215,0.0805989,0.080661,0.0807079,0.0807397,0.0807563,0.0807576,0.0807432,0.0807128,0.0806656,0.0806008,0.0805176,0.0804147,0.0802908,0.0801441,0.0799727,0.0797746,0.079547,0.0792872,0.0789918,0.0786571,0.0782789,0.0778524,0.0773723,0.0768325,0.0762263,0.075546,0.0747832,0.0739283,0.0729707,0.0718984,0.0706982,0.069355,0.0678525,0.0661722,0.0642935,0.0621937,0.0598474,0.0572266,0.0543001,0.0510336,0.0473892,0.0433251,0.0387945,0.0337483,0.0281309,0.0218822,0.0149364,0.00722378,-0.00133222,-0.0108131,-0.0213044,-0.0328998,-0.0456949,-0.05979},
	{-0.860779,-0.835389,-0.810244,-0.785358,-0.760745,-0.736422,-0.712404,-0.688706,-0.665343,-0.642332,-0.619687,-0.597422,-0.575552,-0.554091,-0.53305,-0.512443,-0.492279,-0.47257,-0.453323,-0.434546,-0.416247,-0.39843,-0.381101,-0.36426,-0.347911,-0.332054,-0.316687,-0.301809,-0.287417,-0.273507,-0.260073,-0.247109,-0.234608,-0.222562,-0.210963,-0.199802,-0.189067,-0.17875,-0.168839,-0.159323,-0.150191,-0.14143,-0.133028,-0.124975,-0.117257,-0.109862,-0.102778,-0.0959938,-0.0894966,-0.0832748,-0.0773169,-0.0716114,-0.0661471,-0.0609131,-0.0558987,-0.0510936,-0.0464875,-0.0420708,-0.037834,-0.0337678,-0.0298633,-0.0261121,-0.0225059,-0.0190368,-0.0156971,-0.0124797,-0.00937742,-0.00638371,-0.00349216,-0.000696695,0.00200846,0.00462882,0.00716959,0.00963573,0.0120319,0.0143625,0.0166318,0.0188435,0.0210015,0.023109,0.0251692,0.0271852,0.0291596,0.0310948,0.0329933,0.034857,0.0366878,0.0384874,0.0402572,0.0419984,0.0437122,0.0453994,0.0470607,0.0486968,0.0503079,0.0518944,0.0534562,0.0549934,0.0565058,0.057993,0.0594547,0.0608904,0.0622995,0.0636814,0.0650352,0.0663604,0.0676561,0.0689215,0.0701557,0.071358,0.0725277,0.0736639,0.074766,0.0758334,0.0768654,0.0778616,0.0788217,0.0797452,0.080632,0.0814819,0.0822951,0.0830716,0.0838115,0.0845153,0.0851834,0.0858162,0.0864143,0.0869786,0.0875096,0.0880084,0.0884756,0.0889123,0.0893195,0.0896981,0.0900491,0.0903736,0.0906725,0.0909469,0.0911977,0.0914259,0.0916323,0.0918177,0.091983,0.0921288,0.0922558,0.0923644,0.0924551,0.0925283,0.0925841,0.0926227,0.092644,0.0926478,0.0926339,0.0926016,0.0925504,0.0924793,0.0923873,0.0922731,0.0921351,0.0919715,0.0917802,0.0915586,0.091304,0.0910132,0.0906824,0.0903075,0.0898838,0.089406,0.0888681,0.0882634,0.0875842,0.0868222,0.0859679,0.0850107,0.0839387,0.0827385,0.0813955,0.0798932,0.0782132,0.076335,0.074236,0.071891,0.0692719,0.0663479,0.0630849,0.059445,0.0553867,0.0508637,0.0458268,0.040221,0.0339866,0.0270588,0.0193665,0.0108354,0.00138403,-0.00907391,-0.0206277,-0.0333759,-0.0474169,-0.0628505},
	{-0.965337,-0.939056,-0.912961,-0.887067,-0.861386,-0.835932,-0.81072,-0.785764,-0.761079,-0.736681,-0.712586,-0.688808,-0.665364,-0.642268,-0.619536,-0.597182,-0.57522,-0.553665,-0.532528,-0.511822,-0.491557,-0.471744,-0.452392,-0.433509,-0.415101,-0.397174,-0.379731,-0.362777,-0.346312,-0.330337,-0.314851,-0.299853,-0.28534,-0.271306,-0.257748,-0.244658,-0.232031,-0.219857,-0.208129,-0.196837,-0.185971,-0.175521,-0.165476,-0.155825,-0.146556,-0.137657,-0.129117,-0.120923,-0.113063,-0.105525,-0.0982961,-0.0913652,-0.08472,-0.0783486,-0.0722393,-0.0663807,-0.0607614,-0.0553707,-0.0501977,-0.0452319,-0.0404634,-0.0358822,-0.0314788,-0.0272441,-0.0231692,-0.0192454,-0.0154647,-0.011819,-0.00830087,-0.00490304,-0.00161858,0.00155912,0.00463634,0.00761906,0.0105129,0.0133234,0.0160554,0.0187138,0.0213031,0.0238275,0.0262909,0.028697,0.0310491,0.0333505,0.0356039,0.0378121,0.0399773,0.0421018,0.0441874,0.0462358,0.0482484,0.0502266,0.0521713,0.0540833,0.0559634,0.057812,0.0596293,0.0614154,0.0631705,0.0648942,0.0665863,0.0682464,0.069874,0.0714684,0.073029,0.0745551,0.0760459,0.0775006,0.0789185,0.0802986,0.0816403,0.0829427,0.0842053,0.0854273,0.0866083,0.0877476,0.0888451,0.0899003,0.0909131,0.0918834,0.0928114,0.0936972,0.0945411,0.0953435,0.0961048,0.0968259,0.0975072,0.0981498,0.0987544,0.0993221,0.0998538,0.100351,0.100814,0.101244,0.101643,0.102012,0.102352,0.102664,0.102949,0.103208,0.103442,0.103652,0.103839,0.104005,0.104148,0.104271,0.104373,0.104455,0.104518,0.10456,0.104583,0.104586,0.104569,0.10453,0.10447,0.104387,0.10428,0.104148,0.103988,0.103799,0.103578,0.103322,0.103029,0.102693,0.102312,0.10188,0.101393,0.100843,0.100224,0.099528,0.0987471,0.0978712,0.0968896,0.09579,0.094559,0.0931815,0.0916406,0.0899177,0.0879919,0.08584,0.0834365,0.0807528,0.0777575,0.0744157,0.0706892,0.0665358,0.0619083,0.0567571,0.0510262,0.0446551,0.0375785,0.0297245,0.021018,0.0113771,0.000714721,-0.0110588,-0.0240422,-0.0383341,-0.0540341,-0.0712486},
	{-1.06145,-1.03454,-1.00778,-0.981172,-0.95473,-0.928466,-0.902393,-0.876522,-0.850868,-0.825445,-0.800267,-0.775349,-0.750707,-0.726356,-0.702311,-0.678587,-0.655201,-0.632168,-0.609501,-0.587217,-0.565328,-0.543848,-0.52279,-0.502165,-0.481984,-0.462256,-0.442991,-0.424195,-0.405874,-0.388035,-0.37068,-0.353811,-0.337432,-0.32154,-0.306135,-0.291215,-0.276777,-0.262815,-0.249324,-0.236297,-0.223728,-0.211608,-0.199929,-0.188679,-0.177851,-0.167432,-0.157412,-0.147779,-0.138523,-0.12963,-0.121088,-0.112887,-0.105012,-0.0974528,-0.0901963,-0.0832307,-0.0765439,-0.0701241,-0.0639597,-0.0580393,-0.0523519,-0.0468864,-0.0416323,-0.0365792,-0.0317172,-0.0270366,-0.0225279,-0.0181823,-0.0139908,-0.00994533,-0.0060377,-0.00226026,0.00139432,0.00493305,0.00836259,0.0116893,0.0149191,0.0180578,0.0211107,0.0240828,0.0269788,0.0298033,0.0325601,0.0352533,0.0378862,0.0404622,0.0429841,0.0454546,0.0478762,0.0502509,0.0525806,0.0548671,0.0571115,0.0593152,0.061479,0.0636037,0.0656897,0.0677375,0.0697471,0.0717186,0.0736517,0.0755463,0.0774019,0.0792179,0.0809938,0.0827289,0.0844224,0.0860737,0.0876819,0.0892462,0.0907659,0.0922403,0.0936686,0.0950504,0.096385,0.097672,0.0989111,0.100102,0.101245,0.102339,0.103385,0.104383,0.105334,0.106238,0.107095,0.107907,0.108674,0.109398,0.110078,0.110717,0.111316,0.111875,0.112396,0.112881,0.11333,0.113745,0.114128,0.114479,0.1148,0.115093,0.115357,0.115595,0.115808,0.115995,0.116159,0.116299,0.116417,0.116513,0.116586,0.116638,0.116668,0.116677,0.116663,0.116626,0.116565,0.11648,0.116369,0.11623,0.116062,0.115862,0.115627,0.115356,0.115043,0.114686,0.11428,0.11382,0.113299,0.112712,0.112052,0.111309,0.110476,0.109541,0.108493,0.10732,0.106006,0.104536,0.102893,0.101056,0.0990027,0.0967094,0.0941488,0.0912907,0.088102,0.084546,0.0805825,0.0761666,0.0712503,0.0657802,0.0596983,0.0529417,0.0454417,0.0371254,0.0279137,0.0177224,0.0064637,-0.00595771,-0.0196388,-0.0346803,-0.0511821,-0.0692502},
	{-1.15003,-1.12268,-1.09543,-1.06831,-1.04132,-1.01446,-0.987758,-0.961215,-0.934845,-0.908659,-0.882671,-0.856894,-0.831343,-0.80603,-0.780972,-0.756183,-0.731678,-0.707474,-0.683585,-0.660027,-0.636815,-0.613965,-0.59149,-0.569406,-0.547724,-0.52646,-0.505623,-0.485225,-0.465276,-0.445786,-0.42676,-0.408208,-0.390132,-0.372539,-0.35543,-0.338807,-0.322671,-0.30702,-0.291853,-0.277167,-0.262957,-0.249217,-0.235942,-0.223125,-0.210757,-0.19883,-0.187334,-0.17626,-0.165596,-0.155333,-0.145458,-0.135959,-0.126826,-0.118046,-0.109606,-0.101495,-0.0937001,-0.0862092,-0.0790101,-0.0720908,-0.0654395,-0.0590445,-0.0528943,-0.0469777,-0.0412838,-0.0358019,-0.0305217,-0.025433,-0.0205262,-0.0157919,-0.011221,-0.00680481,-0.002535,0.00159645,0.00559716,0.00947443,0.0132352,0.016886,0.0204331,0.0238824,0.0272393,0.0305091,0.0336966,0.0368064,0.0398426,0.042809,0.0457092,0.0485465,0.0513237,0.0540435,0.0567081,0.0593197,0.0618801,0.0643907,0.0668527,0.0692673,0.0716352,0.0739569,0.0762328,0.0784632,0.0806479,0.0827869,0.0848798,0.0869263,0.0889258,0.0908777,0.0927814,0.0946362,0.0964412,0.0981959,0.0998994,0.101551,0.10315,0.104697,0.106189,0.107628,0.109013,0.110343,0.111619,0.11284,0.114007,0.11512,0.116181,0.117188,0.118143,0.119048,0.119902,0.120708,0.121465,0.122176,0.122842,0.123464,0.124044,0.124583,0.125083,0.125544,0.12597,0.12636,0.126717,0.127042,0.127335,0.1276,0.127835,0.128043,0.128224,0.12838,0.12851,0.128615,0.128696,0.128753,0.128786,0.128794,0.128777,0.128734,0.128666,0.128569,0.128444,0.128287,0.128098,0.127873,0.12761,0.127305,0.126955,0.126555,0.126099,0.125583,0.125001,0.124343,0.123604,0.122773,0.121841,0.120795,0.119624,0.118313,0.116845,0.115204,0.11337,0.11132,0.10903,0.106474,0.103622,0.10044,0.096892,0.0929383,0.0885346,0.0836321,0.0781789,0.072117,0.0653841,0.0579119,0.0496281,0.0404543,0.0303069,0.0190975,0.00673476,-0.00688063,-0.0218478,-0.0382666,-0.0562419,-0.0758735},
	{-1.23199,-1.2043,-1.1767,-1.14919,-1.12179,-1.0945,-1.06732,-1.04028,-1.01337,-0.986609,-0.960009,-0.93358,-0.907333,-0.881282,-0.855441,-0.829823,-0.804442,-0.779313,-0.754451,-0.729873,-0.705592,-0.681624,-0.657986,-0.634692,-0.611758,-0.589197,-0.567025,-0.545253,-0.523896,-0.502966,-0.482472,-0.462425,-0.442835,-0.423708,-0.405051,-0.38687,-0.369169,-0.35195,-0.335216,-0.318966,-0.3032,-0.287915,-0.273109,-0.258777,-0.244914,-0.231514,-0.218569,-0.206071,-0.194012,-0.182382,-0.171172,-0.160369,-0.149965,-0.139947,-0.130304,-0.121023,-0.112093,-0.103501,-0.095235,-0.0872831,-0.0796329,-0.0722721,-0.0651888,-0.0583711,-0.0518074,-0.0454862,-0.0393964,-0.033527,-0.0278675,-0.0224075,-0.0171371,-0.0120466,-0.00712682,-0.00236877,0.00223609,0.00669597,0.0110187,0.0152117,0.0192821,0.0232365,0.0270814,0.0308226,0.0344658,0.0380162,0.0414785,0.0448575,0.0481571,0.0513813,0.0545334,0.0576166,0.0606338,0.0635874,0.0664796,0.0693122,0.072087,0.0748052,0.077468,0.080076,0.0826301,0.0851304,0.0875773,0.0899707,0.0923105,0.0945965,0.0968281,0.0990049,0.101126,0.103192,0.105201,0.107152,0.109045,0.11088,0.112656,0.114372,0.116027,0.117622,0.119156,0.12063,0.122043,0.123394,0.124686,0.125918,0.12709,0.128204,0.12926,0.13026,0.131204,0.132094,0.132931,0.133716,0.134452,0.135139,0.135779,0.136374,0.136926,0.137436,0.137906,0.138338,0.138732,0.139091,0.139417,0.139709,0.13997,0.140201,0.140403,0.140576,0.140722,0.140841,0.140933,0.140999,0.141038,0.141051,0.141037,0.140995,0.140925,0.140825,0.140694,0.140529,0.14033,0.140092,0.139813,0.139489,0.139117,0.138691,0.138207,0.137658,0.137038,0.136338,0.135551,0.134667,0.133674,0.132562,0.131315,0.12992,0.128359,0.126614,0.124663,0.122485,0.120052,0.117337,0.114308,0.110931,0.107168,0.102975,0.0983079,0.0931155,0.0873428,0.0809299,0.0738119,0.0659181,0.0571739,0.0474982,0.0368055,0.0250063,0.0120045,-0.00229751,-0.0180005,-0.0352038,-0.0540121,-0.0745228},
	{-1.30813,-1.28018,-1.25231,-1.22451,-1.1968,-1.16917,-1.14164,-1.11421,-1.08689,-1.05969,-1.03262,-1.00569,-0.978906,-0.952283,-0.925832,-0.899565,-0.873494,-0.847635,-0.822,-0.796603,-0.771461,-0.746587,-0.721998,-0.697708,-0.673733,-0.650089,-0.626791,-0.603854,-0.581292,-0.559119,-0.537348,-0.515993,-0.495064,-0.474573,-0.454529,-0.434941,-0.415817,-0.397162,-0.378981,-0.361279,-0.344059,-0.32732,-0.311064,-0.295289,-0.279993,-0.265172,-0.250823,-0.236938,-0.223512,-0.210538,-0.198007,-0.18591,-0.174238,-0.16298,-0.152126,-0.141664,-0.131584,-0.121874,-0.11252,-0.103512,-0.0948377,-0.0864838,-0.0784386,-0.0706897,-0.0632252,-0.056033,-0.0491014,-0.0424189,-0.0359741,-0.029756,-0.0237539,-0.0179573,-0.0123561,-0.00694059,-0.0017013,0.0033708,0.00828436,0.0130477,0.0176686,0.0221547,0.0265131,0.0307505,0.0348732,0.0388871,0.0427978,0.0466105,0.05033,0.0539606,0.0575064,0.060971,0.0643577,0.0676696,0.0709092,0.0740789,0.0771805,0.0802159,0.0831863,0.0860929,0.0889366,0.091718,0.0944374,0.0970952,0.0996912,0.102225,0.104698,0.107107,0.109454,0.111737,0.113956,0.116111,0.1182,0.120223,0.12218,0.12407,0.125893,0.127649,0.129337,0.130957,0.13251,0.133996,0.135415,0.136768,0.138055,0.139277,0.140436,0.141533,0.142568,0.143544,0.144462,0.145323,0.146129,0.146882,0.147583,0.148235,0.14884,0.149398,0.149913,0.150385,0.150817,0.151209,0.151565,0.151885,0.15217,0.152423,0.152643,0.152832,0.152991,0.15312,0.15322,0.153291,0.153333,0.153346,0.15333,0.153282,0.153204,0.153093,0.152947,0.152765,0.152545,0.152282,0.151974,0.151617,0.151207,0.150738,0.150204,0.149599,0.148916,0.148147,0.147281,0.146308,0.145216,0.143993,0.142624,0.141091,0.139377,0.137462,0.135322,0.132933,0.130266,0.127292,0.123976,0.12028,0.116164,0.111583,0.106486,0.10082,0.0945266,0.0875416,0.0797962,0.0712158,0.0617221,0.0512304,0.0396515,0.0268934,0.0128571,-0.00255656,-0.019447,-0.0379176,-0.0580665,-0.07999},
	{-1.37915,-1.351,-1.32292,-1.29489,-1.26693,-1.23904,-1.21123,-1.18351,-1.15587,-1.12833,-1.10089,-1.07357,-1.04636,-1.01929,-0.99236,-0.965581,-0.938965,-0.912523,-0.88627,-0.860217,-0.834379,-0.80877,-0.783403,-0.758295,-0.73346,-0.708914,-0.684673,-0.660751,-0.637165,-0.613928,-0.591057,-0.568565,-0.546466,-0.524773,-0.503498,-0.482653,-0.462247,-0.44229,-0.422791,-0.403755,-0.38519,-0.367099,-0.349485,-0.332351,-0.315697,-0.299523,-0.283828,-0.268607,-0.253858,-0.239575,-0.225752,-0.212383,-0.19946,-0.186973,-0.174914,-0.163273,-0.15204,-0.141203,-0.130752,-0.120674,-0.110958,-0.101592,-0.0925642,-0.0838614,-0.0754718,-0.0673832,-0.0595835,-0.0520607,-0.044803,-0.0377989,-0.0310369,-0.0245059,-0.0181952,-0.0120943,-0.00619299,-0.000481517,0.00504956,0.0104093,0.0156064,0.020649,0.0255451,0.0303021,0.034927,0.0394266,0.0438069,0.0480739,0.0522329,0.0562889,0.0602466,0.0641101,0.0678833,0.0715695,0.075172,0.0786933,0.0821358,0.0855017,0.0887926,0.09201,0.0951549,0.0982283,0.101231,0.104163,0.107025,0.109816,0.112537,0.115188,0.117767,0.120275,0.122712,0.125076,0.127367,0.129584,0.131728,0.133798,0.135793,0.137714,0.13956,0.141331,0.143029,0.144652,0.146201,0.147678,0.149084,0.150418,0.151683,0.152879,0.154008,0.155072,0.156073,0.157012,0.15789,0.158711,0.159476,0.160187,0.160846,0.161455,0.162016,0.162531,0.163002,0.163431,0.163819,0.164168,0.164481,0.164757,0.164999,0.165207,0.165382,0.165525,0.165637,0.165718,0.165768,0.165786,0.165772,0.165727,0.165647,0.165533,0.165382,0.165192,0.164961,0.164686,0.164363,0.163987,0.163556,0.163062,0.1625,0.161863,0.161143,0.160332,0.159419,0.158394,0.157244,0.155955,0.154513,0.152898,0.151094,0.149077,0.146825,0.144312,0.141507,0.13838,0.134895,0.131013,0.126691,0.121882,0.116535,0.110595,0.104,0.0966849,0.0885788,0.079606,0.0696851,0.0587301,0.0466513,0.0333528,0.0187373,0.00270451,-0.0148446,-0.0340126,-0.0548957,-0.0775876},
	{-1.44565,-1.41735,-1.38909,-1.36088,-1.33273,-1.30464,-1.2766,-1.24864,-1.22075,-1.19293,-1.1652,-1.13757,-1.11003,-1.0826,-1.05529,-1.0281,-1.00105,-0.974139,-0.947387,-0.920804,-0.894401,-0.868192,-0.84219,-0.816408,-0.790862,-0.765566,-0.740535,-0.715785,-0.691331,-0.667189,-0.643373,-0.6199,-0.596784,-0.57404,-0.551681,-0.52972,-0.508171,-0.487045,-0.466353,-0.446104,-0.426307,-0.406969,-0.388096,-0.369694,-0.351767,-0.334316,-0.317343,-0.300848,-0.284829,-0.269285,-0.254212,-0.239604,-0.225457,-0.211764,-0.198517,-0.185709,-0.173329,-0.161369,-0.149818,-0.138666,-0.127901,-0.117512,-0.107487,-0.0978146,-0.0884823,-0.079478,-0.0707894,-0.0624045,-0.0543112,-0.0464974,-0.0389514,-0.0316616,-0.0246167,-0.0178057,-0.0112176,-0.0048422,0.00133073,0.0073109,0.0131077,0.01873,0.0241865,0.0294852,0.0346339,0.0396399,0.0445102,0.049251,0.0538685,0.0583683,0.0627555,0.0670348,0.0712106,0.0752869,0.0792671,0.0831544,0.0869516,0.0906611,0.094285,0.0978249,0.101282,0.104659,0.107955,0.111171,0.114307,0.117365,0.120343,0.123242,0.126062,0.128802,0.131463,0.134042,0.136541,0.138959,0.141295,0.143549,0.145721,0.147811,0.149819,0.151745,0.15359,0.155354,0.157037,0.158641,0.160167,0.161615,0.162987,0.164285,0.16551,0.166664,0.167748,0.168766,0.169718,0.170607,0.171436,0.172206,0.172919,0.173578,0.174186,0.174743,0.175252,0.175716,0.176136,0.176513,0.17685,0.177148,0.177409,0.177633,0.177821,0.177975,0.178094,0.178179,0.178231,0.178248,0.178231,0.178178,0.178089,0.177962,0.177794,0.177584,0.17733,0.177026,0.17667,0.176258,0.175783,0.175241,0.174624,0.173925,0.173136,0.172247,0.171247,0.170124,0.168865,0.167455,0.165877,0.164112,0.162139,0.159936,0.157476,0.154733,0.151674,0.148264,0.144466,0.140239,0.135536,0.130307,0.124497,0.118048,0.110896,0.10297,0.0941956,0.0844945,0.0737816,0.0619677,0.0489609,0.0346625,0.0189742,0.00179661,-0.0169718,-0.037428,-0.0596665,-0.0837716},
	{-1.50815,-1.47972,-1.45132,-1.42297,-1.39466,-1.36639,-1.33818,-1.31002,-1.28192,-1.25389,-1.22593,-1.19804,-1.17023,-1.14251,-1.11489,-1.08737,-1.05996,-1.03268,-1.00552,-0.978506,-0.951642,-0.924941,-0.898415,-0.872077,-0.845941,-0.82002,-0.794329,-0.768882,-0.743694,-0.718781,-0.694158,-0.66984,-0.645844,-0.622184,-0.598875,-0.575932,-0.553369,-0.531199,-0.509436,-0.48809,-0.467173,-0.446695,-0.426665,-0.40709,-0.387977,-0.369331,-0.351156,-0.333455,-0.316229,-0.299479,-0.283204,-0.267402,-0.252068,-0.2372,-0.222791,-0.208835,-0.195325,-0.182253,-0.16961,-0.157386,-0.145572,-0.134156,-0.123129,-0.112477,-0.102191,-0.0922571,-0.0826644,-0.0734005,-0.0644533,-0.0558106,-0.0474606,-0.0393912,-0.0315908,-0.0240478,-0.0167511,-0.00968968,-0.00285285,0.0037697,0.0101879,0.0164114,0.0224494,0.0283106,0.0340035,0.0395358,0.0449153,0.0501488,0.055243,0.0602042,0.0650379,0.0697496,0.074344,0.0788255,0.0831982,0.0874657,0.0916311,0.0956972,0.0996664,0.103541,0.107322,0.111012,0.114612,0.118122,0.121543,0.124875,0.12812,0.131276,0.134343,0.137323,0.140214,0.143016,0.145728,0.148352,0.150885,0.153329,0.155683,0.157947,0.160122,0.162207,0.164203,0.166112,0.167932,0.169667,0.171316,0.172881,0.174364,0.175766,0.177089,0.178335,0.179506,0.180605,0.181633,0.182593,0.183487,0.184318,0.185089,0.1858,0.186456,0.187058,0.187608,0.188109,0.188563,0.188971,0.189336,0.189659,0.189941,0.190184,0.19039,0.190557,0.190689,0.190784,0.190843,0.190865,0.190851,0.190799,0.190708,0.190577,0.190403,0.190185,0.189918,0.189601,0.189228,0.188795,0.188297,0.187728,0.18708,0.186347,0.185518,0.184584,0.183533,0.182354,0.181032,0.179551,0.177895,0.176042,0.173972,0.171662,0.169083,0.166207,0.163001,0.15943,0.155454,0.151029,0.14611,0.140642,0.134571,0.127836,0.12037,0.112103,0.102957,0.0928521,0.0817021,0.0694163,0.0559019,0.0410595,0.0247904,0.00699609,-0.0124243,-0.0335659,-0.0565204,-0.0813683},
	{-1.56708,-1.53854,-1.51003,-1.48156,-1.45312,-1.42471,-1.39635,-1.36803,-1.33977,-1.31155,-1.28339,-1.2553,-1.22727,-1.19931,-1.17144,-1.14365,-1.11595,-1.08835,-1.06086,-1.03349,-1.00625,-0.979139,-0.952179,-0.92538,-0.898752,-0.872308,-0.846062,-0.820028,-0.794219,-0.76865,-0.743336,-0.718294,-0.693537,-0.669082,-0.644944,-0.621138,-0.597679,-0.574583,-0.551862,-0.529531,-0.507602,-0.486088,-0.464999,-0.444346,-0.424137,-0.40438,-0.385082,-0.366248,-0.347883,-0.329989,-0.312567,-0.29562,-0.279144,-0.263139,-0.247602,-0.232527,-0.217911,-0.203745,-0.190024,-0.176739,-0.163882,-0.151443,-0.139412,-0.127779,-0.116533,-0.105662,-0.0951551,-0.0850002,-0.0751855,-0.065699,-0.0565287,-0.0476625,-0.0390886,-0.0307953,-0.0227708,-0.015004,-0.00748358,-0.000198898,0.00686055,0.0137049,0.0203439,0.0267869,0.0330428,0.0391201,0.045027,0.050771,0.0563593,0.0617987,0.0670954,0.0722552,0.0772834,0.0821851,0.0869645,0.0916258,0.0961725,0.100608,0.104935,0.109155,0.113272,0.117286,0.121199,0.125013,0.128727,0.132343,0.135862,0.139282,0.142605,0.145831,0.148959,0.15199,0.154922,0.157757,0.160493,0.163132,0.165672,0.168115,0.17046,0.172707,0.174859,0.176914,0.178875,0.180743,0.182518,0.184202,0.185797,0.187305,0.188728,0.190068,0.191327,0.192508,0.193612,0.194643,0.195604,0.196496,0.197323,0.198086,0.198789,0.199435,0.200025,0.200561,0.201047,0.201484,0.201873,0.202218,0.202519,0.202777,0.202995,0.203172,0.203309,0.203407,0.203466,0.203486,0.203465,0.203403,0.203299,0.203151,0.202957,0.202714,0.202418,0.202067,0.201655,0.201178,0.200629,0.200002,0.19929,0.198483,0.197572,0.196546,0.195394,0.1941,0.19265,0.191027,0.189212,0.187184,0.184919,0.182391,0.179572,0.176429,0.172929,0.169031,0.164695,0.159874,0.154516,0.148568,0.141969,0.134654,0.126554,0.117594,0.107695,0.0967708,0.0847334,0.0714905,0.0569444,0.0409971,0.0235499,0.00450286,-0.01624,-0.0387711,-0.0631735,-0.0895214},
	{-1.62282,-1.59419,-1.56558,-1.537,-1.50845,-1.47993,-1.45145,-1.423,-1.39459,-1.36622,-1.3379,-1.30964,-1.28142,-1.25327,-1.22518,-1.19717,-1.16923,-1.14137,-1.1136,-1.08594,-1.05837,-1.03093,-1.00361,-0.976421,-0.94938,-0.922497,-0.895783,-0.869251,-0.842915,-0.816787,-0.790883,-0.765217,-0.739804,-0.714659,-0.689798,-0.665236,-0.640988,-0.617071,-0.593499,-0.570286,-0.547447,-0.524996,-0.502944,-0.481304,-0.460088,-0.439304,-0.418963,-0.399071,-0.379636,-0.360663,-0.342156,-0.324117,-0.30655,-0.289453,-0.272826,-0.256667,-0.240973,-0.22574,-0.210962,-0.196632,-0.182745,-0.169292,-0.156264,-0.143651,-0.131445,-0.119634,-0.108207,-0.0971543,-0.0864629,-0.0761216,-0.0661187,-0.0564421,-0.04708,-0.0380206,-0.0292521,-0.0207629,-0.0125417,-0.00457729,0.0031412,0.0106243,0.0178822,0.0249248,0.0317614,0.0384012,0.0448527,0.0511241,0.057223,0.0631568,0.0689321,0.0745554,0.0800323,0.0853684,0.0905684,0.0956369,0.100578,0.105395,0.110091,0.114669,0.119132,0.123481,0.127718,0.131844,0.135861,0.13977,0.14357,0.147264,0.15085,0.154329,0.157701,0.160966,0.164124,0.167176,0.170121,0.172959,0.17569,0.178315,0.180835,0.183249,0.185559,0.187766,0.18987,0.191874,0.193777,0.195583,0.197293,0.198909,0.200434,0.201869,0.203217,0.204481,0.205663,0.206766,0.207794,0.208748,0.209631,0.210447,0.211198,0.211886,0.212515,0.213087,0.213604,0.214068,0.214482,0.214847,0.215165,0.215437,0.215664,0.215848,0.215989,0.216087,0.216142,0.216154,0.216123,0.216046,0.215924,0.215752,0.21553,0.215254,0.214921,0.214525,0.214063,0.213529,0.212916,0.212216,0.211422,0.210523,0.209509,0.208369,0.207088,0.205651,0.204042,0.202241,0.200229,0.197981,0.195473,0.192676,0.189558,0.186085,0.182219,0.177918,0.173136,0.167824,0.161927,0.155386,0.148136,0.14011,0.131233,0.121427,0.110607,0.0986852,0.0855707,0.0711668,0.0553759,0.0380989,0.0192389,-0.0013026,-0.0236166,-0.0477896,-0.0738946,-0.101997},
	{-1.67568,-1.64697,-1.61829,-1.58962,-1.56098,-1.53236,-1.50377,-1.4752,-1.44668,-1.41818,-1.38973,-1.36131,-1.33294,-1.30462,-1.27636,-1.24815,-1.22001,-1.19193,-1.16393,-1.13602,-1.10819,-1.08046,-1.05283,-1.02532,-0.99793,-0.970675,-0.943564,-0.916609,-0.889823,-0.863218,-0.836806,-0.810603,-0.784623,-0.758879,-0.733388,-0.708164,-0.683223,-0.65858,-0.634252,-0.610252,-0.586597,-0.5633,-0.540376,-0.517839,-0.4957,-0.473972,-0.452666,-0.431791,-0.411357,-0.391372,-0.37184,-0.352769,-0.334162,-0.316021,-0.298349,-0.281145,-0.264409,-0.248138,-0.232328,-0.216977,-0.202078,-0.187624,-0.17361,-0.160026,-0.146864,-0.134114,-0.121768,-0.109813,-0.0982395,-0.0870364,-0.076192,-0.0656947,-0.0555328,-0.0456947,-0.0361685,-0.0269426,-0.0180054,-0.00934567,-0.000952186,0.00718589,0.0150791,0.0227378,0.0301716,0.0373902,0.0444026,0.0512174,0.0578428,0.0642864,0.0705556,0.0766571,0.0825973,0.0883818,0.0940162,0.0995051,0.104853,0.110064,0.115142,0.120089,0.124908,0.129603,0.134174,0.138623,0.142952,0.147162,0.151254,0.155227,0.159084,0.162824,0.166447,0.169954,0.173345,0.17662,0.179779,0.182822,0.18575,0.188563,0.191263,0.193848,0.196322,0.198684,0.200935,0.203079,0.205115,0.207047,0.208875,0.210603,0.212233,0.213767,0.215208,0.216559,0.217823,0.219002,0.220101,0.221121,0.222066,0.222938,0.223742,0.224479,0.225153,0.225766,0.226321,0.22682,0.227265,0.227659,0.228003,0.228298,0.228547,0.22875,0.228908,0.229021,0.229089,0.229113,0.229091,0.229022,0.228906,0.228739,0.228521,0.228247,0.227914,0.227517,0.227053,0.226515,0.225896,0.22519,0.224387,0.223478,0.222452,0.221298,0.220001,0.218546,0.216917,0.215094,0.213057,0.210782,0.208243,0.205413,0.202258,0.198744,0.194834,0.190485,0.185652,0.180283,0.174325,0.167718,0.160398,0.152296,0.143338,0.133445,0.122533,0.110515,0.0972987,0.082788,0.0668857,0.0494933,0.0305139,0.00985017,-0.0125885,-0.0368877,-0.0631198,-0.0913488},
	{-1.72594,-1.69717,-1.66841,-1.63967,-1.61094,-1.58224,-1.55356,-1.5249,-1.49627,-1.46767,-1.4391,-1.41056,-1.38206,-1.35359,-1.32518,-1.29681,-1.26849,-1.24023,-1.21203,-1.18391,-1.15585,-1.12788,-1.09999,-1.0722,-1.04451,-1.01694,-0.989492,-0.962175,-0.935003,-0.907987,-0.88114,-0.854473,-0.828,-0.801736,-0.775695,-0.749891,-0.72434,-0.699057,-0.674057,-0.649357,-0.62497,-0.600914,-0.577202,-0.553849,-0.530869,-0.508276,-0.486081,-0.464297,-0.442934,-0.422003,-0.401511,-0.381466,-0.361874,-0.342741,-0.32407,-0.305864,-0.288123,-0.270849,-0.254039,-0.237692,-0.221803,-0.206368,-0.191383,-0.176839,-0.162731,-0.149049,-0.135785,-0.12293,-0.110473,-0.0984049,-0.0867137,-0.0753887,-0.0644186,-0.053792,-0.0434972,-0.0335226,-0.0238569,-0.0144884,-0.00540602,0.00340138,0.0119446,0.0202342,0.0282803,0.0360927,0.043681,0.0510541,0.0582207,0.0651888,0.0719662,0.07856,0.0849771,0.0912235,0.0973051,0.103227,0.108994,0.114611,0.120082,0.125409,0.130596,0.135646,0.140561,0.145342,0.149992,0.154512,0.158903,0.163165,0.1673,0.171308,0.175189,0.178944,0.182573,0.186077,0.189455,0.192709,0.195838,0.198844,0.201727,0.204488,0.207128,0.209648,0.212051,0.214337,0.216509,0.218569,0.220518,0.22236,0.224097,0.225731,0.227267,0.228706,0.230052,0.231308,0.232478,0.233564,0.234571,0.2355,0.236356,0.237141,0.237859,0.238512,0.239103,0.239635,0.24011,0.24053,0.240898,0.241214,0.24148,0.241698,0.241868,0.241991,0.242067,0.242095,0.242075,0.242007,0.241888,0.241716,0.241489,0.241204,0.240858,0.240445,0.23996,0.239399,0.238753,0.238015,0.237177,0.236228,0.235157,0.233952,0.232598,0.231079,0.229379,0.227478,0.225353,0.22298,0.220334,0.217384,0.214098,0.210439,0.206369,0.201844,0.196817,0.191236,0.185046,0.178185,0.170589,0.162185,0.1529,0.142653,0.131358,0.118928,0.105269,0.0902849,0.0738789,0.0559524,0.0364098,0.015155,-0.00789989,-0.0328375,-0.0597261,-0.0886251},
	{-1.77384,-1.74501,-1.71619,-1.68738,-1.65859,-1.62981,-1.60105,-1.57231,-1.54359,-1.5149,-1.48622,-1.45758,-1.42896,-1.40038,-1.37183,-1.34332,-1.31486,-1.28644,-1.25807,-1.22976,-1.20151,-1.17333,-1.14522,-1.11719,-1.08925,-1.0614,-1.03366,-1.00603,-0.978528,-0.951157,-0.923932,-0.896863,-0.869963,-0.843245,-0.816722,-0.79041,-0.764321,-0.738472,-0.712878,-0.687553,-0.662513,-0.637774,-0.613351,-0.58926,-0.565514,-0.54213,-0.519119,-0.496497,-0.474273,-0.452461,-0.431071,-0.410112,-0.389592,-0.369519,-0.349898,-0.330735,-0.312031,-0.293791,-0.276014,-0.2587,-0.241848,-0.225456,-0.209518,-0.194031,-0.178989,-0.164385,-0.150211,-0.13646,-0.123122,-0.110188,-0.0976484,-0.0854918,-0.073708,-0.0622858,-0.051214,-0.0404814,-0.0300766,-0.0199882,-0.0102049,-0.000715615,0.00849061,0.0174245,0.0260964,0.0345165,0.0426946,0.0506399,0.0583615,0.0658678,0.0731669,0.0802664,0.0871734,0.0938946,0.100436,0.106803,0.113002,0.119036,0.12491,0.130628,0.136194,0.141609,0.146877,0.152,0.15698,0.161818,0.166515,0.171074,0.175494,0.179777,0.183922,0.187931,0.191805,0.195543,0.199146,0.202614,0.205949,0.209152,0.212222,0.215162,0.217972,0.220655,0.223211,0.225642,0.227952,0.230141,0.232213,0.23417,0.236015,0.237751,0.239381,0.240908,0.242337,0.24367,0.24491,0.246062,0.247128,0.248113,0.249019,0.249849,0.250608,0.251298,0.251922,0.252482,0.252982,0.253422,0.253806,0.254136,0.254412,0.254635,0.254807,0.254928,0.254998,0.255016,0.254983,0.254896,0.254754,0.254555,0.254295,0.253972,0.253581,0.253117,0.252574,0.251947,0.251227,0.250405,0.249473,0.248419,0.24723,0.245894,0.244394,0.242713,0.240832,0.23873,0.236382,0.233762,0.230842,0.227589,0.223968,0.21994,0.215462,0.210488,0.204967,0.198844,0.192058,0.184546,0.176237,0.167057,0.156927,0.145763,0.133477,0.119977,0.105169,0.088955,0.0712375,0.0519207,0.0309091,0.00811375,-0.0165485,-0.043148,-0.071744,-0.102383},
	{-1.81958,-1.7907,-1.76183,-1.73296,-1.70411,-1.67527,-1.64645,-1.61763,-1.58884,-1.56006,-1.5313,-1.50256,-1.47384,-1.44515,-1.41649,-1.38786,-1.35926,-1.33071,-1.30219,-1.27372,-1.2453,-1.21694,-1.18864,-1.1604,-1.13224,-1.10416,-1.07617,-1.04827,-1.02048,-0.992796,-0.965242,-0.937822,-0.910549,-0.883433,-0.856489,-0.829729,-0.803167,-0.776817,-0.750694,-0.724813,-0.699189,-0.673837,-0.648774,-0.624014,-0.599574,-0.575467,-0.551709,-0.528315,-0.505296,-0.482667,-0.46044,-0.438625,-0.417233,-0.396272,-0.375751,-0.355677,-0.336054,-0.316887,-0.298179,-0.279932,-0.262146,-0.244821,-0.227954,-0.211543,-0.195583,-0.18007,-0.164997,-0.150357,-0.136144,-0.122347,-0.108959,-0.0959699,-0.0833694,-0.0711471,-0.0592925,-0.0477946,-0.0366424,-0.0258246,-0.0153302,-0.00514814,0.0047326,0.0143228,0.023633,0.0326734,0.0414541,0.0499846,0.0582742,0.0663318,0.0741656,0.0817836,0.0891933,0.0964017,0.103415,0.11024,0.116881,0.123344,0.129633,0.135753,0.141707,0.147498,0.153129,0.158602,0.163921,0.169086,0.174099,0.178961,0.183674,0.188239,0.192656,0.196926,0.201049,0.205028,0.208861,0.21255,0.216096,0.2195,0.222763,0.225886,0.228871,0.231719,0.234432,0.237013,0.239463,0.241785,0.243983,0.246058,0.248014,0.249854,0.251582,0.253201,0.254714,0.256126,0.257439,0.258659,0.259788,0.260829,0.261788,0.262667,0.263469,0.264199,0.264858,0.265449,0.265977,0.266441,0.266846,0.267192,0.267482,0.267715,0.267894,0.268019,0.268089,0.268105,0.268065,0.267968,0.267813,0.267596,0.267314,0.266965,0.266543,0.266044,0.26546,0.264786,0.264013,0.263131,0.262132,0.261002,0.259729,0.258297,0.256692,0.254893,0.252882,0.250634,0.248126,0.245329,0.242212,0.238741,0.23488,0.230588,0.225819,0.220526,0.214653,0.208145,0.200939,0.192968,0.184159,0.174435,0.163716,0.151915,0.138943,0.124706,0.109109,0.092054,0.0734446,0.0531868,0.0311873,0.0073616,-0.0183685,-0.0460659,-0.0757839,-0.107553},
	{-1.86335,-1.83442,-1.80551,-1.7766,-1.74769,-1.7188,-1.68991,-1.66103,-1.63217,-1.60332,-1.57448,-1.54566,-1.51685,-1.48807,-1.45931,-1.43057,-1.40186,-1.37319,-1.34454,-1.31593,-1.28737,-1.25885,-1.23038,-1.20196,-1.17361,-1.14532,-1.1171,-1.08897,-1.06092,-1.03298,-1.00513,-0.977407,-0.949806,-0.922342,-0.895027,-0.867872,-0.840891,-0.814096,-0.787503,-0.761126,-0.734979,-0.709078,-0.683437,-0.658073,-0.633002,-0.608237,-0.583796,-0.559692,-0.535941,-0.512555,-0.48955,-0.466936,-0.444726,-0.42293,-0.401559,-0.38062,-0.360121,-0.340069,-0.320468,-0.301322,-0.282633,-0.264404,-0.246633,-0.229319,-0.212461,-0.196055,-0.180096,-0.164579,-0.149497,-0.134844,-0.120612,-0.106792,-0.0933741,-0.0803498,-0.0677088,-0.0554405,-0.0435346,-0.0319801,-0.0207663,-0.00988221,0.000682971,0.01094,0.0208994,0.0305716,0.0399667,0.0490944,0.0579641,0.0665851,0.0749657,0.0831144,0.0910388,0.0987463,0.106244,0.113537,0.120632,0.127535,0.13425,0.140782,0.147134,0.15331,0.159314,0.165147,0.170813,0.176313,0.18165,0.186824,0.191838,0.196691,0.201386,0.205923,0.210304,0.214528,0.218597,0.222513,0.226275,0.229885,0.233344,0.236655,0.239818,0.242835,0.245709,0.248442,0.251036,0.253495,0.25582,0.258016,0.260085,0.262032,0.263859,0.26557,0.26717,0.268662,0.270051,0.271339,0.272532,0.273632,0.274644,0.275572,0.276418,0.277187,0.277882,0.278506,0.27906,0.279549,0.279974,0.280337,0.28064,0.280884,0.281069,0.281197,0.281267,0.281278,0.281231,0.281123,0.280952,0.280716,0.280411,0.280033,0.279578,0.27904,0.278413,0.277688,0.276858,0.275913,0.274841,0.27363,0.272266,0.270734,0.269016,0.267093,0.264943,0.262542,0.259863,0.256877,0.253552,0.249852,0.245737,0.241166,0.236091,0.23046,0.224219,0.217307,0.20966,0.201208,0.191877,0.181586,0.170253,0.15779,0.144106,0.129107,0.112695,0.0947747,0.0752511,0.0540299,0.0310235,0.00615029,-0.0206589,-0.0494618,-0.0803027,-0.113201},
	{-1.90531,-1.87635,-1.84739,-1.81843,-1.78948,-1.76054,-1.7316,-1.70266,-1.67374,-1.64482,-1.61592,-1.58702,-1.55814,-1.52928,-1.50043,-1.4716,-1.44279,-1.414,-1.38525,-1.35652,-1.32782,-1.29916,-1.27054,-1.24197,-1.21344,-1.18497,-1.15657,-1.12822,-1.09996,-1.07177,-1.04368,-1.01568,-0.987789,-0.960017,-0.932373,-0.904868,-0.877515,-0.850326,-0.823315,-0.796494,-0.769878,-0.743483,-0.717322,-0.691412,-0.665767,-0.640404,-0.615337,-0.590583,-0.566156,-0.542072,-0.518344,-0.494986,-0.472011,-0.449431,-0.427258,-0.405502,-0.384172,-0.363276,-0.34282,-0.322811,-0.303253,-0.284149,-0.265501,-0.24731,-0.229574,-0.212294,-0.195464,-0.179083,-0.163145,-0.147644,-0.132573,-0.117925,-0.103693,-0.0898668,-0.0764379,-0.0633964,-0.0507324,-0.0384357,-0.0264958,-0.0149021,-0.00364403,0.00728899,0.0179074,0.0282217,0.0382418,0.0479776,0.0574388,0.0666344,0.0755735,0.0842643,0.092715,0.100933,0.108925,0.116699,0.124259,0.131612,0.138764,0.145717,0.152478,0.159049,0.165435,0.171637,0.177659,0.183504,0.189172,0.194666,0.199987,0.205137,0.210117,0.214928,0.219572,0.224048,0.228359,0.232506,0.236489,0.24031,0.243971,0.247474,0.250819,0.25401,0.257049,0.259938,0.26268,0.265277,0.267734,0.270054,0.27224,0.274295,0.276224,0.278032,0.279721,0.281296,0.282761,0.284121,0.28538,0.286541,0.287609,0.288588,0.289482,0.290294,0.291027,0.291685,0.292271,0.292788,0.293237,0.293621,0.293942,0.2942,0.294397,0.294534,0.29461,0.294624,0.294577,0.294466,0.294289,0.294045,0.293728,0.293335,0.292862,0.292301,0.291647,0.290893,0.290027,0.289042,0.287925,0.286664,0.285243,0.283648,0.281859,0.279857,0.277619,0.275121,0.272335,0.269231,0.265776,0.261932,0.257659,0.252914,0.247649,0.24181,0.235342,0.228182,0.220266,0.21152,0.201872,0.191238,0.179536,0.166677,0.152568,0.137116,0.120224,0.101796,0.0817383,0.0599581,0.0363705,0.0108966,-0.0165285,-0.0459586,-0.0774325,-0.110964},
	{-1.9456,-1.9166,-1.88761,-1.85861,-1.82962,-1.80063,-1.77165,-1.74266,-1.71368,-1.68471,-1.65575,-1.62679,-1.59784,-1.5689,-1.53997,-1.51106,-1.48216,-1.45328,-1.42442,-1.39558,-1.36677,-1.33799,-1.30924,-1.28052,-1.25185,-1.22322,-1.19464,-1.16611,-1.13765,-1.10926,-1.08094,-1.0527,-1.02455,-0.996507,-0.968572,-0.940756,-0.913072,-0.885531,-0.858145,-0.830927,-0.803891,-0.77705,-0.75042,-0.724015,-0.69785,-0.671941,-0.646304,-0.620953,-0.595905,-0.571175,-0.546777,-0.522727,-0.499039,-0.475725,-0.452798,-0.43027,-0.408152,-0.386453,-0.365183,-0.344347,-0.323954,-0.304007,-0.284511,-0.265467,-0.246878,-0.228744,-0.211062,-0.193832,-0.17705,-0.160711,-0.144811,-0.129342,-0.114299,-0.0996736,-0.0854574,-0.0716418,-0.0582174,-0.0451747,-0.0325038,-0.0201946,-0.00823673,0.00337999,0.0146659,0.0256311,0.0362858,0.0466397,0.0567025,0.0664834,0.0759914,0.0852352,0.0942228,0.102962,0.11146,0.119724,0.12776,0.135574,0.143172,0.150558,0.157737,0.164712,0.171489,0.178069,0.184456,0.190652,0.19666,0.202481,0.208117,0.213571,0.218842,0.223934,0.228846,0.23358,0.238138,0.242521,0.246731,0.250768,0.254635,0.258333,0.261865,0.265233,0.268439,0.271487,0.274379,0.277119,0.27971,0.282155,0.284459,0.286625,0.288658,0.290562,0.292341,0.293999,0.295542,0.296974,0.298299,0.299521,0.300645,0.301674,0.302614,0.303467,0.304237,0.304928,0.305543,0.306084,0.306555,0.306956,0.30729,0.307559,0.307762,0.307902,0.307977,0.307987,0.307932,0.307809,0.307616,0.307351,0.307009,0.306587,0.306079,0.305478,0.304779,0.303971,0.303047,0.301995,0.300803,0.299458,0.297944,0.296244,0.294339,0.292208,0.289827,0.287171,0.28421,0.280913,0.277244,0.273165,0.268635,0.263606,0.258029,0.251849,0.245008,0.237442,0.229081,0.219854,0.209683,0.198485,0.186174,0.172659,0.15785,0.141649,0.123962,0.104692,0.0837493,0.061043,0.0364923,0.0100238,-0.0184211,-0.0488893,-0.0814063,-0.11598},
	{-1.98434,-1.95532,-1.92629,-1.89726,-1.86823,-1.83921,-1.81018,-1.78115,-1.75212,-1.7231,-1.69408,-1.66507,-1.63605,-1.60705,-1.57805,-1.54907,-1.52009,-1.49112,-1.46217,-1.43324,-1.40432,-1.37543,-1.34656,-1.31772,-1.28891,-1.26014,-1.23141,-1.20272,-1.17408,-1.1455,-1.11698,-1.08853,-1.06015,-1.03186,-1.00367,-0.975576,-0.947595,-0.919739,-0.892017,-0.864442,-0.837028,-0.809786,-0.782731,-0.755878,-0.72924,-0.702834,-0.676675,-0.650777,-0.625158,-0.599832,-0.574815,-0.550122,-0.525768,-0.501766,-0.478132,-0.454877,-0.432014,-0.409554,-0.387507,-0.365883,-0.344688,-0.323931,-0.303616,-0.283748,-0.26433,-0.245364,-0.22685,-0.208788,-0.191177,-0.174013,-0.157293,-0.141012,-0.125165,-0.109744,-0.0947429,-0.0801539,-0.0659685,-0.0521778,-0.0387725,-0.0257432,-0.0130801,-0.000773218,0.0111873,0.0228115,0.0341093,0.0450903,0.0557641,0.06614,0.076227,0.0860338,0.0955687,0.10484,0.113854,0.122619,0.131141,0.139426,0.147479,0.155307,0.162914,0.170303,0.17748,0.184446,0.191207,0.197763,0.204119,0.210275,0.216234,0.221998,0.227568,0.232946,0.238134,0.243132,0.247943,0.252568,0.257009,0.261267,0.265344,0.269243,0.272966,0.276515,0.279893,0.283104,0.28615,0.289035,0.291763,0.294337,0.296762,0.299042,0.301181,0.303184,0.305056,0.306801,0.308425,0.309931,0.311324,0.312609,0.313792,0.314875,0.315863,0.31676,0.317571,0.318298,0.318945,0.319515,0.32001,0.320433,0.320785,0.321069,0.321285,0.321433,0.321514,0.321527,0.321472,0.321346,0.321147,0.320872,0.320518,0.320079,0.319551,0.318928,0.3182,0.317361,0.3164,0.315306,0.314068,0.312669,0.311096,0.30933,0.307352,0.30514,0.302669,0.299913,0.296841,0.293422,0.28962,0.285394,0.280702,0.275496,0.269726,0.263335,0.256264,0.248448,0.239816,0.230296,0.219808,0.208268,0.195591,0.181685,0.166458,0.149814,0.131659,0.111897,0.0904386,0.0671961,0.0420911,0.0150532,-0.0139717,-0.0450263,-0.0781308,-0.113287},
	{-2.02166,-1.99261,-1.96355,-1.93449,-1.90543,-1.87637,-1.8473,-1.81823,-1.78917,-1.7601,-1.73103,-1.70196,-1.67289,-1.64383,-1.61477,-1.58572,-1.55667,-1.52763,-1.4986,-1.46957,-1.44057,-1.41158,-1.3826,-1.35365,-1.32472,-1.29582,-1.26695,-1.23811,-1.20932,-1.18057,-1.15187,-1.12323,-1.09465,-1.06614,-1.03771,-1.00937,-0.981123,-0.952983,-0.924959,-0.897063,-0.869307,-0.841702,-0.814263,-0.787003,-0.759935,-0.733076,-0.706439,-0.68004,-0.653895,-0.628019,-0.602429,-0.577139,-0.552164,-0.527521,-0.503223,-0.479284,-0.455718,-0.432537,-0.409753,-0.387376,-0.365415,-0.34388,-0.322777,-0.302112,-0.281891,-0.262117,-0.242792,-0.223918,-0.205494,-0.18752,-0.169992,-0.152909,-0.136264,-0.120055,-0.104273,-0.0889137,-0.0739682,-0.059429,-0.0452877,-0.0315353,-0.0181628,-0.00516071,0.00748046,0.0197703,0.0317185,0.0433345,0.0546278,0.0656075,0.0762827,0.0866619,0.0967537,0.106566,0.116106,0.125381,0.134399,0.143164,0.151684,0.159963,0.168006,0.175819,0.183404,0.190766,0.197908,0.204832,0.211543,0.218042,0.224331,0.230412,0.236287,0.241958,0.247427,0.252696,0.257765,0.262637,0.267314,0.271797,0.276089,0.280193,0.28411,0.287844,0.291397,0.294773,0.297976,0.301009,0.303876,0.306581,0.309129,0.311524,0.313771,0.315875,0.31784,0.319672,0.321376,0.322956,0.324418,0.325766,0.327006,0.328141,0.329177,0.330117,0.330965,0.331726,0.332402,0.332997,0.333513,0.333953,0.334319,0.334612,0.334833,0.334984,0.335063,0.33507,0.335004,0.334864,0.334647,0.334349,0.333966,0.333495,0.332928,0.33226,0.331482,0.330585,0.329558,0.328391,0.32707,0.32558,0.323905,0.322025,0.31992,0.317567,0.314941,0.312013,0.308752,0.305124,0.301091,0.296612,0.291642,0.286132,0.280028,0.273272,0.265803,0.257554,0.248451,0.238421,0.22738,0.215246,0.20193,0.187339,0.171381,0.15396,0.134982,0.114355,0.0919899,0.0678038,0.0417219,0.0136828,-0.0163627,-0.0484482,-0.0825835,-0.11876},
	{-2.05765,-2.02857,-1.99949,-1.9704,-1.94131,-1.91221,-1.88311,-1.85401,-1.8249,-1.79579,-1.76668,-1.73757,-1.70845,-1.67934,-1.65022,-1.6211,-1.59199,-1.56288,-1.53378,-1.50468,-1.47559,-1.44651,-1.41744,-1.38838,-1.35934,-1.33033,-1.30133,-1.27237,-1.24343,-1.21453,-1.18567,-1.15686,-1.12809,-1.09939,-1.07075,-1.04218,-1.01369,-0.985299,-0.957002,-0.928816,-0.90075,-0.872817,-0.845028,-0.817397,-0.789938,-0.762664,-0.73559,-0.708731,-0.682102,-0.655719,-0.629598,-0.603753,-0.578202,-0.552959,-0.528039,-0.503458,-0.479229,-0.455366,-0.431882,-0.408788,-0.386096,-0.363815,-0.341955,-0.320523,-0.299525,-0.278968,-0.258854,-0.239187,-0.219968,-0.201199,-0.182878,-0.165003,-0.147572,-0.130581,-0.114025,-0.0978986,-0.0821956,-0.0669089,-0.052031,-0.0375539,-0.0234692,-0.00976793,0.00355875,0.01652,0.0291251,0.0413833,0.0533037,0.0648954,0.0761673,0.0871279,0.0977856,0.108148,0.118224,0.128019,0.137542,0.146797,0.155792,0.164531,0.173021,0.181265,0.189268,0.197034,0.204566,0.211867,0.218941,0.22579,0.232416,0.238822,0.245009,0.25098,0.256737,0.262281,0.267615,0.27274,0.277658,0.282372,0.286884,0.291197,0.295314,0.299237,0.30297,0.306516,0.309879,0.313064,0.316074,0.318914,0.321588,0.324102,0.32646,0.328668,0.33073,0.332652,0.33444,0.336098,0.337632,0.339047,0.340348,0.34154,0.342627,0.343614,0.344505,0.345305,0.346016,0.346642,0.347186,0.347651,0.348038,0.348349,0.348586,0.348748,0.348837,0.348851,0.348789,0.34865,0.348431,0.348129,0.34774,0.347259,0.34668,0.345996,0.345199,0.34428,0.343228,0.342031,0.340677,0.339149,0.337431,0.335503,0.333346,0.330934,0.328243,0.325244,0.321904,0.31819,0.314062,0.309479,0.304396,0.298762,0.292524,0.285623,0.277996,0.269575,0.260289,0.250061,0.238809,0.226448,0.21289,0.198044,0.181815,0.16411,0.144834,0.123896,0.101209,0.0766909,0.0502692,0.0218845,-0.00850968,-0.0409443,-0.0754262,-0.111943},
	{-2.0924,-2.0633,-2.03419,-2.00508,-1.97596,-1.94684,-1.91771,-1.88857,-1.85943,-1.83028,-1.80113,-1.77197,-1.74281,-1.71365,-1.68448,-1.65531,-1.62614,-1.59697,-1.56779,-1.53862,-1.50946,-1.4803,-1.45114,-1.42199,-1.39286,-1.36374,-1.33463,-1.30554,-1.27648,-1.24744,-1.21844,-1.18947,-1.16054,-1.13165,-1.10282,-1.07405,-1.04535,-1.01672,-0.988178,-0.959728,-0.931381,-0.903149,-0.875043,-0.847074,-0.819257,-0.791604,-0.764129,-0.736847,-0.709773,-0.682921,-0.656308,-0.62995,-0.603861,-0.578058,-0.552557,-0.527372,-0.502519,-0.478011,-0.453864,-0.430089,-0.406699,-0.383706,-0.361119,-0.338948,-0.317201,-0.295885,-0.275005,-0.254567,-0.234573,-0.215026,-0.195926,-0.177273,-0.159066,-0.141303,-0.12398,-0.107092,-0.0906359,-0.0746042,-0.0589909,-0.0437888,-0.0289903,-0.0145874,-0.000571624,0.0130655,0.0263329,0.0392392,0.0517934,0.0640042,0.0758803,0.0874301,0.098662,0.109584,0.120203,0.130528,0.140564,0.150318,0.159797,0.169005,0.177949,0.186634,0.195062,0.20324,0.21117,0.218855,0.2263,0.233506,0.240476,0.247213,0.253719,0.259996,0.266046,0.271872,0.277475,0.282857,0.288022,0.292971,0.297707,0.302233,0.306552,0.310667,0.314582,0.3183,0.321826,0.325164,0.328319,0.331294,0.334096,0.336729,0.339198,0.341509,0.343668,0.34568,0.34755,0.349284,0.350888,0.352366,0.353725,0.354969,0.356103,0.357132,0.35806,0.358891,0.359629,0.360278,0.36084,0.361318,0.361714,0.36203,0.362266,0.362424,0.362503,0.362503,0.362422,0.362258,0.362009,0.361671,0.361239,0.360709,0.360073,0.359325,0.358455,0.357453,0.356309,0.355009,0.353539,0.351883,0.350022,0.347936,0.345603,0.342997,0.340091,0.336854,0.333253,0.32925,0.324805,0.319874,0.314409,0.308357,0.301661,0.294261,0.286089,0.277078,0.26715,0.256227,0.244226,0.231059,0.216636,0.200865,0.183653,0.164905,0.144529,0.122437,0.098545,0.0727784,0.0450709,0.0153729,-0.0163518,-0.0501226,-0.085932,-0.123753},
	{-2.12599,-2.09687,-2.06774,-2.03861,-2.00946,-1.98031,-1.95116,-1.92199,-1.89282,-1.86364,-1.83445,-1.80525,-1.77605,-1.74684,-1.71763,-1.68841,-1.65918,-1.62995,-1.60072,-1.57149,-1.54225,-1.51302,-1.48378,-1.45455,-1.42533,-1.39611,-1.3669,-1.33771,-1.30852,-1.27936,-1.25022,-1.22111,-1.19203,-1.16299,-1.13398,-1.10503,-1.07613,-1.04729,-1.01852,-0.98983,-0.961227,-0.932722,-0.904325,-0.876049,-0.847903,-0.819903,-0.79206,-0.764388,-0.736903,-0.709619,-0.682551,-0.655714,-0.629125,-0.6028,-0.576754,-0.551002,-0.525561,-0.500446,-0.47567,-0.451249,-0.427194,-0.40352,-0.380237,-0.357356,-0.334887,-0.312838,-0.291217,-0.270029,-0.249279,-0.228971,-0.209109,-0.189692,-0.170721,-0.152196,-0.134114,-0.116473,-0.0992681,-0.0824951,-0.0661486,-0.0502223,-0.0347097,-0.0196034,-0.00489587,0.00942082,0.0233549,0.0369147,0.0501086,0.0629451,0.0754325,0.0875792,0.0993931,0.110882,0.122054,0.132916,0.143474,0.153736,0.163707,0.173394,0.182801,0.191934,0.200798,0.209395,0.217731,0.225809,0.233632,0.241203,0.248525,0.2556,0.262431,0.269021,0.275371,0.281484,0.287362,0.293008,0.298425,0.303614,0.308579,0.313323,0.317849,0.322161,0.326262,0.330157,0.33385,0.337345,0.340648,0.343762,0.346695,0.34945,0.352034,0.354453,0.356711,0.358815,0.360771,0.362585,0.364262,0.365808,0.367228,0.368528,0.369713,0.370788,0.371757,0.372625,0.373396,0.374072,0.374658,0.375156,0.375568,0.375896,0.376141,0.376303,0.376384,0.37638,0.376293,0.376119,0.375855,0.375498,0.375043,0.374484,0.373815,0.373029,0.372114,0.371063,0.369862,0.368498,0.366957,0.36522,0.36327,0.361086,0.358643,0.355916,0.352876,0.349491,0.345728,0.341547,0.336906,0.331761,0.326061,0.319753,0.312778,0.305074,0.296574,0.287205,0.276893,0.265555,0.253107,0.239463,0.22453,0.208218,0.190431,0.171078,0.150067,0.127313,0.102734,0.0762595,0.0478274,0.0173938,-0.0150729,-0.0495812,-0.0861204,-0.124655},
	{-2.1585,-2.12936,-2.10021,-2.07106,-2.04189,-2.01272,-1.98353,-1.95434,-1.92514,-1.89593,-1.86671,-1.83748,-1.80824,-1.77899,-1.74973,-1.72047,-1.69119,-1.66191,-1.63262,-1.60333,-1.57403,-1.54473,-1.51542,-1.48612,-1.45681,-1.4275,-1.3982,-1.36891,-1.33962,-1.31035,-1.28109,-1.25184,-1.22262,-1.19343,-1.16427,-1.13515,-1.10607,-1.07704,-1.04806,-1.01915,-0.990314,-0.96156,-0.932897,-0.904338,-0.875892,-0.847572,-0.81939,-0.79136,-0.763495,-0.73581,-0.70832,-0.68104,-0.653985,-0.627172,-0.600615,-0.574332,-0.548338,-0.522649,-0.497279,-0.472244,-0.447557,-0.423233,-0.399284,-0.375722,-0.352558,-0.329802,-0.307462,-0.285547,-0.264062,-0.243013,-0.222405,-0.202239,-0.182518,-0.163242,-0.144412,-0.126024,-0.108078,-0.0905682,-0.0734918,-0.0568435,-0.0406174,-0.0248074,-0.00940661,0.00559215,0.0201964,0.0344139,0.0482526,0.0617204,0.0748255,0.0875756,0.0999788,0.112042,0.123774,0.135181,0.14627,0.157048,0.16752,0.177693,0.187572,0.197162,0.206467,0.215493,0.224243,0.23272,0.240929,0.248872,0.256552,0.263973,0.271136,0.278044,0.2847,0.291106,0.297265,0.30318,0.308853,0.314287,0.319485,0.324451,0.329188,0.3337,0.337991,0.342065,0.345927,0.349582,0.353035,0.356292,0.359356,0.362236,0.364936,0.367462,0.369821,0.372018,0.37406,0.375952,0.377702,0.379314,0.380794,0.382149,0.383383,0.384501,0.385509,0.386411,0.38721,0.38791,0.388515,0.389027,0.389449,0.389783,0.390028,0.390187,0.390258,0.390241,0.390135,0.389937,0.389643,0.38925,0.388754,0.388146,0.387422,0.386571,0.385585,0.384452,0.38316,0.381695,0.38004,0.378177,0.376087,0.373747,0.371133,0.368216,0.364966,0.361351,0.357333,0.352873,0.347926,0.342446,0.336379,0.329669,0.322257,0.314078,0.305061,0.295132,0.284214,0.272224,0.259076,0.244681,0.228947,0.211782,0.193092,0.172788,0.15078,0.126985,0.101328,0.0737429,0.0441783,0.0125963,-0.0210239,-0.056679,-0.0943459,-0.133978},
	{-2.18999,-2.16084,-2.13167,-2.1025,-2.07331,-2.04412,-2.01491,-1.98569,-1.95646,-1.92722,-1.89797,-1.86871,-1.83944,-1.81015,-1.78085,-1.75155,-1.72223,-1.6929,-1.66356,-1.63422,-1.60486,-1.57549,-1.54612,-1.51675,-1.48736,-1.45798,-1.42859,-1.3992,-1.36982,-1.34044,-1.31107,-1.28171,-1.25236,-1.22303,-1.19373,-1.16445,-1.1352,-1.106,-1.07683,-1.04772,-1.01867,-0.989687,-0.960781,-0.931961,-0.903238,-0.874624,-0.84613,-0.817768,-0.789552,-0.761496,-0.733614,-0.705921,-0.678432,-0.651163,-0.62413,-0.597347,-0.570833,-0.544602,-0.51867,-0.493053,-0.467765,-0.442822,-0.418236,-0.394021,-0.370189,-0.346751,-0.323717,-0.301097,-0.278898,-0.257127,-0.23579,-0.214891,-0.194434,-0.174421,-0.154852,-0.135727,-0.117046,-0.0988063,-0.0810046,-0.0636372,-0.0466995,-0.0301863,-0.0140915,0.00159101,0.0168681,0.0317469,0.0462347,0.0603389,0.0740672,0.087427,0.100426,0.113071,0.125371,0.137331,0.148958,0.160259,0.171241,0.181908,0.192266,0.202321,0.212077,0.221539,0.23071,0.239595,0.248197,0.25652,0.264566,0.272338,0.279839,0.287073,0.294041,0.300747,0.307192,0.313381,0.319316,0.325,0.330436,0.335629,0.340581,0.345298,0.349782,0.35404,0.358075,0.361894,0.3655,0.368901,0.372102,0.375108,0.377926,0.380563,0.383025,0.385317,0.387447,0.389422,0.391246,0.392927,0.39447,0.395881,0.397167,0.398331,0.39938,0.400317,0.401147,0.401874,0.402501,0.403031,0.403466,0.403808,0.404059,0.404217,0.404284,0.404259,0.404139,0.403923,0.403606,0.403185,0.402655,0.402008,0.401237,0.400334,0.399289,0.398088,0.396721,0.39517,0.393421,0.391453,0.389246,0.386776,0.384018,0.380943,0.377519,0.373712,0.369484,0.364792,0.359593,0.353835,0.347466,0.340427,0.332657,0.324088,0.31465,0.304266,0.292858,0.28034,0.266627,0.251628,0.235252,0.217408,0.198001,0.176943,0.154148,0.129537,0.103038,0.07459,0.0441482,0.0116819,-0.022822,-0.0593511,-0.0978732,-0.138333},
	{-2.22053,-2.19136,-2.16218,-2.13299,-2.10378,-2.07457,-2.04534,-2.0161,-1.98685,-1.95758,-1.9283,-1.89901,-1.8697,-1.84039,-1.81105,-1.78171,-1.75235,-1.72298,-1.69359,-1.6642,-1.63479,-1.60536,-1.57593,-1.54649,-1.51704,-1.48758,-1.45811,-1.42864,-1.39917,-1.36969,-1.34022,-1.31075,-1.28129,-1.25183,-1.22239,-1.19297,-1.16358,-1.13421,-1.10487,-1.07557,-1.04632,-1.01713,-0.987998,-0.958939,-0.929961,-0.901075,-0.872291,-0.843623,-0.815082,-0.786682,-0.758436,-0.730358,-0.702464,-0.674769,-0.647288,-0.620037,-0.593033,-0.566291,-0.539828,-0.513659,-0.4878,-0.462267,-0.437072,-0.412232,-0.387758,-0.363664,-0.33996,-0.316657,-0.293765,-0.271292,-0.249245,-0.22763,-0.206451,-0.185713,-0.165417,-0.145566,-0.126159,-0.107195,-0.0886735,-0.0705911,-0.0529444,-0.0357294,-0.018941,-0.00257393,0.0133778,0.0289206,0.044061,0.0588059,0.0731623,0.0871374,0.100738,0.113972,0.126845,0.139365,0.151539,0.163371,0.174869,0.186038,0.196884,0.207412,0.217627,0.227532,0.237133,0.246433,0.255435,0.264144,0.272563,0.280694,0.288541,0.296106,0.303392,0.310403,0.317141,0.32361,0.329812,0.335751,0.34143,0.346854,0.352026,0.356951,0.361634,0.366079,0.370291,0.374276,0.37804,0.381588,0.384927,0.388063,0.391002,0.393752,0.396319,0.398709,0.400929,0.402987,0.404888,0.406639,0.408247,0.409716,0.411055,0.412267,0.413358,0.414332,0.415195,0.41595,0.4166,0.417149,0.417599,0.417951,0.418207,0.418368,0.418433,0.4184,0.418269,0.418037,0.4177,0.417254,0.416692,0.416009,0.415196,0.414245,0.413144,0.411881,0.410443,0.408814,0.406977,0.404911,0.402596,0.400006,0.397116,0.393894,0.39031,0.386326,0.381903,0.376999,0.371567,0.365555,0.35891,0.35157,0.343473,0.33455,0.324729,0.313933,0.302081,0.289087,0.274866,0.259326,0.242375,0.223923,0.203878,0.182152,0.158662,0.133331,0.106092,0.0768887,0.0456821,0.0124478,-0.0228205,-0.0601025,-0.0993582,-0.140525},
	{-2.25017,-2.22099,-2.19179,-2.16258,-2.13336,-2.10413,-2.07488,-2.04562,-2.01634,-1.98705,-1.95775,-1.92843,-1.8991,-1.86975,-1.84038,-1.811,-1.7816,-1.75219,-1.72276,-1.69332,-1.66386,-1.63439,-1.6049,-1.5754,-1.54588,-1.51636,-1.48682,-1.45727,-1.42771,-1.39815,-1.36858,-1.33901,-1.30944,-1.27987,-1.25031,-1.22076,-1.19122,-1.1617,-1.1322,-1.10273,-1.0733,-1.04391,-1.01457,-0.985292,-0.956078,-0.92694,-0.897889,-0.868937,-0.840094,-0.811373,-0.782788,-0.754352,-0.72608,-0.697986,-0.670086,-0.642395,-0.61493,-0.587706,-0.560741,-0.534049,-0.507647,-0.481551,-0.455776,-0.430337,-0.405247,-0.380521,-0.356172,-0.332209,-0.308646,-0.28549,-0.262751,-0.240436,-0.218551,-0.197102,-0.176092,-0.155524,-0.135401,-0.115721,-0.0964863,-0.0776939,-0.059342,-0.0414275,-0.0239468,-0.00689522,0.00973211,0.0259407,0.0417365,0.0571255,0.0721144,0.0867096,0.100918,0.114745,0.128199,0.141286,0.154012,0.166383,0.178405,0.190084,0.201425,0.212433,0.223113,0.23347,0.243507,0.253229,0.26264,0.271742,0.28054,0.289037,0.297235,0.305138,0.312749,0.320071,0.327107,0.33386,0.340334,0.346533,0.35246,0.358119,0.363515,0.368653,0.373536,0.378171,0.382563,0.386718,0.390641,0.394339,0.397818,0.401085,0.404148,0.407012,0.409684,0.412173,0.414484,0.416625,0.418603,0.420424,0.422095,0.423622,0.425012,0.42627,0.427401,0.428411,0.429303,0.430083,0.430753,0.431316,0.431776,0.432133,0.43239,0.432545,0.4326,0.432552,0.432401,0.432142,0.431773,0.431288,0.430681,0.429946,0.429073,0.428054,0.426876,0.425527,0.423993,0.422256,0.420299,0.4181,0.415638,0.412885,0.409815,0.406396,0.402593,0.39837,0.393685,0.388494,0.382747,0.376392,0.369372,0.361626,0.353087,0.343686,0.333348,0.321994,0.309543,0.295907,0.280999,0.264728,0.247003,0.227733,0.206828,0.184203,0.159778,0.133482,0.105252,0.0750395,0.0428132,0.00855657,-0.0277232,-0.065999,-0.106219,-0.14831},
	{-2.27896,-2.24977,-2.22056,-2.19133,-2.1621,-2.13284,-2.10358,-2.0743,-2.045,-2.01569,-1.98637,-1.95702,-1.92766,-1.89828,-1.86889,-1.83947,-1.81004,-1.78059,-1.75113,-1.72164,-1.69214,-1.66261,-1.63307,-1.60352,-1.57394,-1.54435,-1.51475,-1.48513,-1.45549,-1.42585,-1.39619,-1.36653,-1.33685,-1.30718,-1.2775,-1.24783,-1.21816,-1.1885,-1.15886,-1.12923,-1.09963,-1.07006,-1.04053,-1.01104,-0.981608,-0.952237,-0.922938,-0.893721,-0.864597,-0.835579,-0.806677,-0.777907,-0.74928,-0.720813,-0.692519,-0.664415,-0.636516,-0.608837,-0.581396,-0.554209,-0.527291,-0.50066,-0.47433,-0.448318,-0.422639,-0.397305,-0.372333,-0.347733,-0.323519,-0.299701,-0.276288,-0.253291,-0.230716,-0.20857,-0.186859,-0.165587,-0.144756,-0.124369,-0.104428,-0.0849311,-0.0658783,-0.0472676,-0.0290963,-0.0113611,0.0059421,0.0228178,0.0392711,0.0553074,0.0709324,0.086152,0.100972,0.1154,0.12944,0.1431,0.156385,0.169302,0.181855,0.194051,0.205894,0.21739,0.228544,0.239359,0.249841,0.259992,0.269818,0.279321,0.288506,0.297375,0.305931,0.314179,0.322121,0.32976,0.3371,0.344144,0.350896,0.35736,0.36354,0.36944,0.375065,0.38042,0.385509,0.390339,0.394915,0.399243,0.403329,0.407181,0.410805,0.414207,0.417396,0.420377,0.42316,0.42575,0.428156,0.430384,0.432443,0.434338,0.436077,0.437666,0.439112,0.44042,0.441596,0.442646,0.443574,0.444385,0.445081,0.445667,0.446144,0.446515,0.446781,0.446942,0.446999,0.446949,0.44679,0.446521,0.446136,0.445632,0.445001,0.444237,0.44333,0.442271,0.441048,0.439647,0.438055,0.436253,0.434223,0.431944,0.429391,0.42654,0.423361,0.419821,0.415887,0.411519,0.406676,0.401312,0.395377,0.388818,0.381576,0.373589,0.36479,0.355109,0.34447,0.332793,0.319995,0.305991,0.290691,0.274006,0.255844,0.236115,0.214731,0.191608,0.166669,0.139844,0.111076,0.0803181,0.047544,0.0127418,-0.0240762,-0.0628779,-0.103606,-0.146184},
	{-2.30695,-2.27775,-2.24852,-2.21929,-2.19003,-2.16077,-2.13149,-2.10219,-2.07287,-2.04354,-2.01419,-1.98483,-1.95544,-1.92604,-1.89661,-1.86717,-1.83771,-1.80822,-1.77872,-1.74919,-1.71965,-1.69008,-1.66049,-1.63088,-1.60126,-1.57161,-1.54194,-1.51225,-1.48255,-1.45282,-1.42309,-1.39333,-1.36357,-1.3338,-1.30401,-1.27423,-1.24444,-1.21465,-1.18487,-1.15509,-1.12534,-1.0956,-1.06589,-1.03621,-1.00657,-0.976984,-0.947453,-0.91799,-0.888604,-0.859307,-0.830111,-0.801027,-0.77207,-0.743252,-0.714589,-0.686096,-0.657787,-0.629679,-0.601789,-0.574131,-0.546724,-0.519583,-0.492724,-0.466164,-0.439918,-0.414002,-0.38843,-0.363215,-0.338371,-0.31391,-0.289843,-0.266181,-0.242932,-0.220105,-0.197706,-0.17574,-0.154214,-0.133129,-0.112489,-0.0922945,-0.072546,-0.053243,-0.0343841,-0.0159668,0.00201157,0.0195548,0.0366669,0.0533525,0.0696165,0.0854643,0.100901,0.115933,0.130566,0.144804,0.158655,0.172123,0.185214,0.197933,0.210286,0.222277,0.233911,0.245192,0.256125,0.266713,0.276961,0.286872,0.296449,0.305697,0.314618,0.323216,0.331494,0.339456,0.347106,0.354446,0.361481,0.368215,0.374652,0.380796,0.386654,0.392229,0.397527,0.402555,0.407317,0.411821,0.416074,0.420081,0.42385,0.427389,0.430705,0.433805,0.436698,0.43939,0.44189,0.444205,0.446342,0.44831,0.450115,0.451763,0.453262,0.454618,0.455836,0.456922,0.45788,0.458716,0.459432,0.460033,0.46052,0.460895,0.46116,0.461316,0.461361,0.461294,0.461114,0.460816,0.460397,0.459852,0.459173,0.458354,0.457384,0.456253,0.454949,0.453459,0.451765,0.44985,0.447695,0.445277,0.442572,0.439551,0.436185,0.432441,0.428282,0.423667,0.418554,0.412895,0.406638,0.399727,0.392104,0.383703,0.374456,0.364291,0.353129,0.340891,0.327493,0.312847,0.296864,0.279453,0.260526,0.239993,0.217767,0.193768,0.167923,0.140167,0.110448,0.0787285,0.0449889,0.00922605,-0.0285377,-0.0682599,-0.109876,-0.153295},
	{-2.33419,-2.30497,-2.27573,-2.24649,-2.21722,-2.18794,-2.15864,-2.12933,-2.1,-2.07065,-2.04128,-2.01189,-1.98248,-1.95305,-1.9236,-1.89413,-1.86464,-1.83512,-1.80558,-1.77602,-1.74644,-1.71683,-1.6872,-1.65754,-1.62786,-1.59816,-1.56843,-1.53868,-1.50891,-1.47912,-1.4493,-1.41947,-1.38962,-1.35975,-1.32987,-1.29998,-1.27008,-1.24017,-1.21026,-1.18035,-1.15044,-1.12055,-1.09067,-1.06081,-1.03099,-1.0012,-0.971451,-0.941758,-0.912127,-0.88257,-0.853097,-0.82372,-0.794452,-0.765306,-0.736295,-0.707435,-0.678741,-0.650227,-0.621911,-0.593808,-0.565935,-0.538309,-0.510945,-0.483861,-0.457073,-0.430596,-0.404446,-0.378638,-0.353185,-0.328102,-0.303399,-0.27909,-0.255183,-0.231689,-0.208616,-0.18597,-0.163759,-0.141986,-0.120655,-0.0997701,-0.0793318,-0.0593411,-0.0397978,-0.0207009,-0.00204835,0.0161622,0.033934,0.0512706,0.0681763,0.0846554,0.100713,0.116354,0.131583,0.146406,0.160828,0.174854,0.188489,0.201739,0.214608,0.227101,0.239222,0.250976,0.262367,0.2734,0.284076,0.294402,0.30438,0.314013,0.323305,0.33226,0.340882,0.349173,0.357137,0.364779,0.372102,0.379111,0.385811,0.392205,0.3983,0.404101,0.409613,0.414842,0.419796,0.42448,0.428901,0.433068,0.436987,0.440665,0.444112,0.447334,0.45034,0.453137,0.455735,0.458139,0.46036,0.462403,0.464277,0.465988,0.467544,0.468951,0.470214,0.47134,0.472333,0.473198,0.47394,0.47456,0.475063,0.475449,0.475721,0.475878,0.47592,0.475846,0.475653,0.475339,0.474898,0.474325,0.473613,0.472755,0.47174,0.470558,0.469196,0.467639,0.465872,0.463875,0.461628,0.459109,0.456291,0.453147,0.449645,0.445751,0.441427,0.436633,0.431324,0.425451,0.418961,0.411799,0.403902,0.395205,0.385638,0.375129,0.363598,0.350965,0.337145,0.32205,0.305592,0.287679,0.268224,0.247138,0.224336,0.199742,0.173284,0.144901,0.114545,0.082184,0.0478035,0.011406,-0.02698,-0.0673059,-0.109501,-0.15347},
	{-2.36071,-2.33148,-2.30223,-2.27297,-2.24369,-2.2144,-2.18509,-2.15576,-2.12641,-2.09704,-2.06765,-2.03825,-2.00882,-1.97937,-1.94989,-1.92039,-1.89087,-1.86133,-1.83176,-1.80216,-1.77254,-1.74289,-1.71322,-1.68352,-1.65379,-1.62404,-1.59426,-1.56445,-1.53462,-1.50476,-1.47487,-1.44496,-1.41503,-1.38508,-1.35511,-1.32511,-1.29511,-1.26509,-1.23506,-1.20502,-1.17498,-1.14494,-1.1149,-1.08488,-1.05487,-1.02489,-0.994947,-0.965038,-0.935179,-0.905378,-0.875646,-0.845994,-0.816434,-0.786978,-0.757641,-0.728435,-0.699376,-0.670479,-0.64176,-0.613234,-0.584919,-0.55683,-0.528985,-0.5014,-0.474092,-0.447077,-0.420372,-0.393991,-0.36795,-0.342263,-0.316943,-0.292004,-0.267456,-0.243311,-0.219577,-0.196265,-0.173379,-0.150929,-0.128917,-0.107349,-0.0862273,-0.0655542,-0.0453306,-0.0255568,-0.00623212,0.012645,0.0310765,0.0490652,0.0666143,0.0837274,0.100409,0.116662,0.132493,0.147905,0.162904,0.177494,0.19168,0.205466,0.218858,0.231859,0.244475,0.256709,0.268565,0.280048,0.291161,0.301907,0.312292,0.322317,0.331987,0.341305,0.350276,0.358902,0.367187,0.375136,0.382753,0.390042,0.397009,0.403658,0.409994,0.416024,0.421754,0.427189,0.432337,0.437204,0.441798,0.446127,0.450197,0.454018,0.457598,0.460944,0.464065,0.466969,0.469665,0.472161,0.474465,0.476584,0.478528,0.480303,0.481915,0.483373,0.484681,0.485847,0.486874,0.487769,0.488534,0.489173,0.48969,0.490085,0.490361,0.490518,0.490554,0.490469,0.490261,0.489925,0.489457,0.488852,0.488102,0.487199,0.486133,0.484893,0.483464,0.481833,0.479982,0.477892,0.475542,0.472908,0.469964,0.46668,0.463025,0.458963,0.454455,0.44946,0.443931,0.437819,0.43107,0.423624,0.415421,0.406394,0.396471,0.385578,0.373636,0.360563,0.346274,0.33068,0.313693,0.295223,0.275183,0.253485,0.230048,0.204796,0.177662,0.148589,0.117534,0.0844705,0.0493893,0.0122999,-0.0267626,-0.0677428,-0.110564,-0.155124},
	{-2.38654,-2.3573,-2.32805,-2.29877,-2.26949,-2.24018,-2.21086,-2.18151,-2.15215,-2.12277,-2.09336,-2.06393,-2.03449,-2.00501,-1.97552,-1.94599,-1.91645,-1.88687,-1.85727,-1.82765,-1.79799,-1.76831,-1.7386,-1.70885,-1.67908,-1.64928,-1.61945,-1.58959,-1.5597,-1.52977,-1.49982,-1.46985,-1.43984,-1.4098,-1.37974,-1.34966,-1.31956,-1.28943,-1.25929,-1.22913,-1.19896,-1.16878,-1.1386,-1.10842,-1.07825,-1.04809,-1.01796,-0.987847,-0.957772,-0.927743,-0.897767,-0.867857,-0.838022,-0.808276,-0.77863,-0.749098,-0.719695,-0.690435,-0.661333,-0.632406,-0.60367,-0.575141,-0.546836,-0.518772,-0.490966,-0.463434,-0.436194,-0.409262,-0.382652,-0.356381,-0.330463,-0.304911,-0.279739,-0.254957,-0.230578,-0.206611,-0.183064,-0.159946,-0.137263,-0.11502,-0.093222,-0.0718723,-0.0509732,-0.0305261,-0.0105316,0.00901064,0.0281015,0.0467428,0.0649367,0.082686,0.0999939,0.116864,0.1333,0.149306,0.164887,0.180046,0.194788,0.209117,0.223038,0.236555,0.249671,0.262392,0.27472,0.28666,0.298216,0.309391,0.320188,0.330612,0.340666,0.350354,0.35968,0.368647,0.377259,0.385521,0.393437,0.401012,0.408251,0.415159,0.421742,0.428005,0.433956,0.439601,0.444947,0.450002,0.454772,0.459266,0.463492,0.467458,0.471173,0.474645,0.477884,0.480897,0.483694,0.486284,0.488673,0.490872,0.492887,0.494727,0.496399,0.497909,0.499265,0.500472,0.501536,0.502461,0.503252,0.503913,0.504445,0.504852,0.505135,0.505293,0.505327,0.505235,0.505014,0.504661,0.504171,0.503538,0.502755,0.501813,0.500702,0.499409,0.497922,0.496224,0.494299,0.492126,0.489684,0.486949,0.483892,0.480485,0.476694,0.472484,0.467814,0.462641,0.456918,0.450596,0.443618,0.435925,0.427455,0.418139,0.407906,0.39668,0.384382,0.370929,0.356236,0.340214,0.322775,0.30383,0.283292,0.261075,0.2371,0.211294,0.183592,0.153942,0.122303,0.0886555,0.0529938,0.0153327,-0.0242871,-0.0658045,-0.109138,-0.154182},
	{-2.41173,-2.38248,-2.35322,-2.32393,-2.29464,-2.26532,-2.23598,-2.20663,-2.17725,-2.14785,-2.11843,-2.08899,-2.05952,-2.03003,-2.00051,-1.97096,-1.94139,-1.91179,-1.88217,-1.85251,-1.82282,-1.79311,-1.76336,-1.73358,-1.70376,-1.67392,-1.64404,-1.61413,-1.58418,-1.5542,-1.52419,-1.49414,-1.46406,-1.43396,-1.40381,-1.37364,-1.34345,-1.31322,-1.28297,-1.2527,-1.22241,-1.19211,-1.16179,-1.13146,-1.10114,-1.07081,-1.0405,-1.0102,-0.979922,-0.949677,-0.919472,-0.889318,-0.859225,-0.829204,-0.799267,-0.769427,-0.739698,-0.710094,-0.68063,-0.651322,-0.622185,-0.593236,-0.564492,-0.53597,-0.507687,-0.479659,-0.451905,-0.424441,-0.397282,-0.370446,-0.343947,-0.3178,-0.29202,-0.266618,-0.241607,-0.216999,-0.192803,-0.169029,-0.145684,-0.122775,-0.100308,-0.0782882,-0.0567187,-0.0356024,-0.0149411,0.00526419,0.0250134,0.0443071,0.0631465,0.0815335,0.0994704,0.11696,0.134005,0.150609,0.166776,0.18251,0.197813,0.212691,0.227147,0.241185,0.254809,0.268022,0.280829,0.293233,0.305238,0.316848,0.328065,0.338894,0.349339,0.359402,0.369089,0.378402,0.387347,0.395927,0.404147,0.412012,0.419528,0.4267,0.433534,0.440035,0.446212,0.45207,0.457618,0.462862,0.467811,0.472473,0.476856,0.48097,0.484822,0.488423,0.491781,0.494905,0.497804,0.500487,0.502963,0.50524,0.507328,0.509232,0.510962,0.512525,0.513927,0.515174,0.516272,0.517226,0.518041,0.518719,0.519265,0.51968,0.519965,0.52012,0.520146,0.520041,0.519801,0.519423,0.518903,0.518233,0.517407,0.516414,0.515245,0.513887,0.512326,0.510546,0.508528,0.506253,0.503697,0.500836,0.497641,0.494082,0.490124,0.48573,0.480859,0.475468,0.469508,0.462926,0.455667,0.44767,0.43887,0.4292,0.418585,0.40695,0.394213,0.380292,0.365101,0.348552,0.330556,0.311026,0.289875,0.26702,0.242384,0.215898,0.187499,0.157141,0.124788,0.0904252,0.0540538,0.0156976,-0.024599,-0.0667671,-0.110719,-0.156344},
	{-2.4363,-2.40705,-2.37777,-2.34848,-2.31917,-2.28984,-2.2605,-2.23113,-2.20174,-2.17233,-2.14289,-2.11343,-2.08395,-2.05444,-2.0249,-1.99533,-1.96574,-1.93612,-1.90646,-1.87678,-1.84706,-1.81731,-1.78753,-1.75771,-1.72786,-1.69797,-1.66805,-1.63809,-1.60809,-1.57806,-1.54799,-1.51788,-1.48774,-1.45756,-1.42734,-1.39709,-1.3668,-1.33649,-1.30614,-1.27576,-1.24536,-1.21493,-1.18449,-1.15402,-1.12355,-1.09307,-1.06258,-1.03211,-1.00164,-0.971192,-0.940772,-0.910388,-0.880051,-0.84977,-0.819558,-0.789427,-0.759389,-0.729459,-0.699651,-0.66998,-0.640462,-0.611113,-0.581949,-0.552989,-0.524248,-0.495745,-0.467496,-0.439519,-0.41183,-0.384447,-0.357385,-0.330661,-0.304288,-0.278281,-0.252654,-0.227418,-0.202585,-0.178166,-0.154169,-0.130604,-0.107476,-0.0847924,-0.0625582,-0.0407773,-0.0194528,0.00141314,0.0218192,0.0417648,0.0612501,0.0802759,0.0988437,0.116955,0.134613,0.151819,0.168577,0.184889,0.20076,0.216192,0.231188,0.245754,0.259891,0.273604,0.286896,0.299771,0.312231,0.324282,0.335926,0.347166,0.358007,0.368453,0.378506,0.388172,0.397455,0.406359,0.414889,0.42305,0.430848,0.438288,0.445377,0.452121,0.458528,0.464603,0.470356,0.475794,0.480925,0.485759,0.490303,0.494567,0.49856,0.502291,0.505771,0.509007,0.512011,0.51479,0.517355,0.519713,0.521873,0.523845,0.525635,0.527251,0.528701,0.529989,0.531124,0.532108,0.532948,0.533646,0.534206,0.53463,0.53492,0.535075,0.535094,0.534977,0.534721,0.534321,0.533773,0.533069,0.532203,0.531163,0.529941,0.528522,0.526892,0.525034,0.52293,0.520559,0.517897,0.514919,0.511594,0.507893,0.503779,0.499215,0.494158,0.488564,0.482384,0.475563,0.468044,0.459767,0.450665,0.440669,0.429704,0.417695,0.404559,0.390213,0.37457,0.357544,0.339045,0.318989,0.297287,0.273861,0.248635,0.221543,0.192526,0.161541,0.128558,0.0935675,0.056575,0.0176107,-0.0232752,-0.0660085,-0.110497,-0.156624},
	{-2.46029,-2.43103,-2.40174,-2.37244,-2.34313,-2.31379,-2.28443,-2.25505,-2.22565,-2.19622,-2.16677,-2.1373,-2.1078,-2.07827,-2.04871,-2.01913,-1.98952,-1.95987,-1.93019,-1.90048,-1.87074,-1.84096,-1.81115,-1.78129,-1.75141,-1.72148,-1.69151,-1.66151,-1.63147,-1.60138,-1.57125,-1.54109,-1.51088,-1.48063,-1.45035,-1.42002,-1.38965,-1.35925,-1.3288,-1.29833,-1.26782,-1.23728,-1.20671,-1.17612,-1.1455,-1.11487,-1.08423,-1.05359,-1.02294,-0.992301,-0.961677,-0.931076,-0.900508,-0.869982,-0.839509,-0.809102,-0.778772,-0.748532,-0.718397,-0.688381,-0.6585,-0.628769,-0.599205,-0.569825,-0.540646,-0.511685,-0.48296,-0.454489,-0.426289,-0.398377,-0.37077,-0.343484,-0.316535,-0.289939,-0.263709,-0.237859,-0.212402,-0.187349,-0.162712,-0.138498,-0.114718,-0.0913786,-0.0684859,-0.0460455,-0.0240617,-0.00253817,0.0185227,0.039119,0.05925,0.0789153,0.0981152,0.116851,0.135123,0.152935,0.170287,0.187183,0.203625,0.219617,0.23516,0.250258,0.264915,0.279133,0.292917,0.306268,0.319191,0.331689,0.343765,0.355423,0.366667,0.3775,0.387927,0.397951,0.407578,0.416811,0.425655,0.434117,0.442202,0.449915,0.457263,0.464254,0.470894,0.47719,0.483152,0.488786,0.494103,0.49911,0.503817,0.508233,0.512368,0.516232,0.519834,0.523185,0.526293,0.529169,0.531822,0.534261,0.536494,0.538532,0.540381,0.542049,0.543544,0.544873,0.54604,0.547052,0.547913,0.548627,0.549197,0.549626,0.549913,0.550061,0.550067,0.549931,0.549649,0.549217,0.54863,0.54788,0.54696,0.545859,0.544567,0.543069,0.54135,0.539393,0.537179,0.534686,0.531889,0.528761,0.525273,0.521392,0.517081,0.512302,0.50701,0.50116,0.4947,0.487577,0.479731,0.471099,0.461614,0.451207,0.439801,0.427318,0.413676,0.398792,0.382579,0.364949,0.345815,0.325093,0.302697,0.278549,0.252578,0.224721,0.194926,0.163153,0.129379,0.0936005,0.0558312,0.0161084,-0.0255102,-0.0689432,-0.114092,-0.160836},
	{-2.48372,-2.45445,-2.42516,-2.39585,-2.36652,-2.33718,-2.30781,-2.27842,-2.249,-2.21957,-2.1901,-2.16062,-2.1311,-2.10156,-2.07198,-2.04238,-2.01275,-1.98308,-1.95338,-1.92365,-1.89388,-1.86407,-1.83422,-1.80434,-1.77442,-1.74446,-1.71445,-1.68441,-1.65432,-1.62418,-1.59401,-1.56378,-1.53352,-1.50321,-1.47285,-1.44245,-1.41201,-1.38152,-1.35099,-1.32042,-1.28981,-1.25916,-1.22848,-1.19776,-1.16702,-1.13625,-1.10546,-1.07465,-1.04384,-1.01302,-0.982199,-0.951393,-0.920606,-0.889848,-0.859128,-0.828458,-0.79785,-0.767316,-0.73687,-0.706525,-0.676298,-0.646202,-0.616255,-0.586474,-0.556874,-0.527474,-0.498292,-0.469345,-0.44065,-0.412227,-0.384091,-0.35626,-0.328752,-0.301581,-0.274763,-0.248312,-0.222244,-0.196569,-0.171301,-0.14645,-0.122026,-0.0980376,-0.0744928,-0.0513983,-0.0287597,-0.00658193,0.0151313,0.036377,0.057153,0.077458,0.0972913,0.116653,0.135543,0.153963,0.171914,0.189397,0.206416,0.222971,0.239066,0.254703,0.269885,0.284615,0.298896,0.31273,0.326122,0.339073,0.351588,0.363671,0.375323,0.386551,0.397357,0.407745,0.417721,0.427289,0.436454,0.445222,0.453598,0.46159,0.469203,0.476444,0.483322,0.489844,0.496018,0.501853,0.507358,0.512542,0.517416,0.521988,0.526268,0.530267,0.533995,0.537462,0.540678,0.543653,0.546397,0.548919,0.551228,0.553333,0.555244,0.556967,0.55851,0.55988,0.561083,0.562125,0.56301,0.563743,0.564326,0.564761,0.565051,0.565195,0.565192,0.56504,0.564737,0.564278,0.563658,0.562868,0.561901,0.560746,0.559392,0.557824,0.556027,0.553982,0.55167,0.549069,0.546152,0.542892,0.539259,0.535219,0.530734,0.525765,0.520266,0.51419,0.507486,0.500098,0.491965,0.483024,0.473207,0.462442,0.450653,0.437762,0.423685,0.408339,0.391637,0.373492,0.353818,0.33253,0.309547,0.284791,0.258195,0.229699,0.199254,0.166825,0.132397,0.0959691,0.0575611,0.0172169,-0.0249994,-0.0690034,-0.114688,-0.161933},
	{-2.50661,-2.47733,-2.44804,-2.41872,-2.38939,-2.36003,-2.33066,-2.30126,-2.27183,-2.24238,-2.21291,-2.18341,-2.15388,-2.12432,-2.09473,-2.06511,-2.03546,-2.00577,-1.97605,-1.94629,-1.9165,-1.88667,-1.85679,-1.82688,-1.79693,-1.76693,-1.73689,-1.7068,-1.67667,-1.64649,-1.61626,-1.58599,-1.55567,-1.5253,-1.49488,-1.46441,-1.43389,-1.40333,-1.37271,-1.34205,-1.31135,-1.2806,-1.24981,-1.21898,-1.18811,-1.15721,-1.12627,-1.09532,-1.06434,-1.03335,-1.00235,-0.971348,-0.940354,-0.909375,-0.878421,-0.847502,-0.81663,-0.785816,-0.755074,-0.724416,-0.693858,-0.663414,-0.633101,-0.602934,-0.572931,-0.543109,-0.513486,-0.48408,-0.454909,-0.425991,-0.397343,-0.368984,-0.34093,-0.3132,-0.285808,-0.258771,-0.232103,-0.205819,-0.179931,-0.154452,-0.129393,-0.104764,-0.080574,-0.0568312,-0.0335427,-0.0107145,0.0116481,0.0335412,0.0549612,0.0759056,0.0963728,0.116361,0.135871,0.154902,0.173454,0.191529,0.209128,0.226253,0.242904,0.259085,0.274798,0.290045,0.304829,0.319153,0.333019,0.34643,0.35939,0.371903,0.383971,0.395598,0.406789,0.417547,0.427878,0.437786,0.447277,0.456355,0.465028,0.473302,0.481184,0.48868,0.4958,0.50255,0.50894,0.514979,0.520676,0.52604,0.531082,0.535812,0.540239,0.544376,0.548231,0.551815,0.555139,0.558214,0.561048,0.563653,0.566037,0.568211,0.570181,0.571957,0.573547,0.574957,0.576193,0.577262,0.578168,0.578915,0.579506,0.579944,0.58023,0.580364,0.580345,0.58017,0.579838,0.579342,0.578678,0.577837,0.576811,0.575588,0.574156,0.572501,0.570607,0.568454,0.566022,0.563287,0.560224,0.556803,0.552992,0.548758,0.544061,0.538859,0.533108,0.526758,0.519756,0.512045,0.503564,0.494247,0.484025,0.472825,0.460571,0.447182,0.432576,0.416669,0.399373,0.380603,0.360272,0.338299,0.314604,0.289112,0.26176,0.232491,0.201262,0.168045,0.132829,0.0956221,0.0564502,0.0153649,-0.0275622,-0.0722407,-0.118557,-0.166388},
	{-2.529,-2.49971,-2.47041,-2.44109,-2.41175,-2.38238,-2.353,-2.32359,-2.29415,-2.26469,-2.23521,-2.20569,-2.17615,-2.14658,-2.11698,-2.08734,-2.05767,-2.02796,-1.99822,-1.96845,-1.93863,-1.90877,-1.87887,-1.84893,-1.81895,-1.78892,-1.75884,-1.72872,-1.69854,-1.66832,-1.63805,-1.60773,-1.57735,-1.54693,-1.51645,-1.48592,-1.45533,-1.42469,-1.394,-1.36325,-1.33246,-1.30161,-1.27072,-1.23977,-1.20879,-1.17776,-1.1467,-1.1156,-1.08447,-1.05331,-1.02214,-0.990953,-0.959761,-0.928573,-0.897396,-0.86624,-0.835117,-0.804036,-0.773011,-0.742054,-0.71118,-0.680403,-0.649739,-0.619204,-0.588814,-0.558586,-0.52854,-0.498691,-0.469059,-0.439662,-0.410519,-0.381646,-0.353064,-0.324788,-0.296837,-0.269226,-0.241972,-0.215089,-0.188593,-0.162496,-0.136811,-0.111549,-0.086721,-0.0623361,-0.0384028,-0.0149284,0.00808054,0.0306186,0.0526812,0.0742646,0.0953659,0.115983,0.136114,0.155758,0.174915,0.193585,0.211768,0.229466,0.246679,0.26341,0.279659,0.295429,0.310722,0.325541,0.339887,0.353765,0.367177,0.380125,0.392614,0.404648,0.41623,0.427364,0.438056,0.44831,0.458131,0.467527,0.476502,0.485063,0.493218,0.500975,0.508341,0.515324,0.521935,0.528181,0.534074,0.539622,0.544836,0.549727,0.554305,0.558581,0.562566,0.566271,0.569707,0.572883,0.575812,0.578503,0.580965,0.583208,0.585242,0.587074,0.588713,0.590165,0.591438,0.592537,0.593467,0.594232,0.594836,0.595281,0.595567,0.595696,0.595665,0.595474,0.595118,0.594593,0.593893,0.593009,0.591932,0.590651,0.589154,0.587424,0.585446,0.5832,0.580665,0.577815,0.574626,0.571066,0.567103,0.562702,0.557823,0.552424,0.546457,0.539873,0.532618,0.524633,0.515857,0.506222,0.495659,0.484095,0.471451,0.457648,0.442602,0.426229,0.408442,0.389156,0.368287,0.345754,0.321478,0.295388,0.267423,0.237531,0.205674,0.171825,0.135981,0.0981549,0.0583781,0.0167077,-0.0267794,-0.0719881,-0.118801,-0.167091},
	{-2.55089,-2.5216,-2.4923,-2.46297,-2.43362,-2.40425,-2.37485,-2.34543,-2.31599,-2.28652,-2.25703,-2.2275,-2.19795,-2.16836,-2.13874,-2.10909,-2.07941,-2.04968,-2.01992,-1.99013,-1.96029,-1.93041,-1.90048,-1.87051,-1.8405,-1.81044,-1.78033,-1.75017,-1.71996,-1.6897,-1.65939,-1.62902,-1.59859,-1.56811,-1.53758,-1.50698,-1.47633,-1.44562,-1.41485,-1.38403,-1.35315,-1.32221,-1.29122,-1.26017,-1.22908,-1.19793,-1.16674,-1.1355,-1.10423,-1.07292,-1.04158,-1.01022,-0.978838,-0.94745,-0.916061,-0.884679,-0.853316,-0.821981,-0.790686,-0.759444,-0.728268,-0.697173,-0.666173,-0.635283,-0.604522,-0.573904,-0.543449,-0.513174,-0.483097,-0.453238,-0.423613,-0.394244,-0.365146,-0.33634,-0.307843,-0.279672,-0.251844,-0.224375,-0.197281,-0.170577,-0.144275,-0.118389,-0.0929302,-0.0679097,-0.0433371,-0.0192212,0.0044303,0.0276106,0.050314,0.0725353,0.0942704,0.115516,0.13627,0.156529,0.176293,0.195561,0.214333,0.232608,0.250387,0.267672,0.284462,0.30076,0.316568,0.331888,0.346721,0.361071,0.374939,0.38833,0.401246,0.413692,0.42567,0.437186,0.448244,0.458849,0.469007,0.478723,0.488005,0.496859,0.505292,0.513312,0.520928,0.528148,0.534982,0.541439,0.54753,0.553264,0.558652,0.563706,0.568436,0.572853,0.57697,0.580796,0.584343,0.587622,0.590644,0.593419,0.595958,0.598271,0.600365,0.602251,0.603936,0.605428,0.606734,0.607858,0.608808,0.609586,0.610195,0.610639,0.610919,0.611033,0.610982,0.610763,0.610372,0.609804,0.609053,0.60811,0.606965,0.605607,0.604022,0.602195,0.600108,0.59774,0.59507,0.592072,0.588719,0.58498,0.580821,0.576205,0.571091,0.565436,0.559192,0.552307,0.544725,0.536388,0.527231,0.517187,0.506185,0.49415,0.481004,0.466665,0.451051,0.434077,0.415656,0.395705,0.374141,0.350883,0.325858,0.298997,0.270243,0.239549,0.206882,0.172223,0.135574,0.0969559,0.0564076,0.0139937,-0.0302019,-0.0760782,-0.123513,-0.172374},
	{-2.57232,-2.54303,-2.51371,-2.48438,-2.45502,-2.42565,-2.39625,-2.36682,-2.33737,-2.30789,-2.27838,-2.24885,-2.21928,-2.18968,-2.16005,-2.13039,-2.10068,-2.07095,-2.04117,-2.01135,-1.98149,-1.95159,-1.92164,-1.89165,-1.86161,-1.83152,-1.80138,-1.77119,-1.74094,-1.71064,-1.68029,-1.64988,-1.6194,-1.58887,-1.55828,-1.52763,-1.49692,-1.46614,-1.4353,-1.4044,-1.37344,-1.34242,-1.31133,-1.28019,-1.24898,-1.21772,-1.18641,-1.15504,-1.12363,-1.09217,-1.06068,-1.02915,-0.997592,-0.966014,-0.934422,-0.902826,-0.871234,-0.839656,-0.808104,-0.776589,-0.745124,-0.713723,-0.6824,-0.651172,-0.620053,-0.58906,-0.558212,-0.527525,-0.497019,-0.466711,-0.436622,-0.406769,-0.377171,-0.347849,-0.318819,-0.290101,-0.261711,-0.233668,-0.205988,-0.178686,-0.151777,-0.125275,-0.0991929,-0.0735434,-0.0483373,-0.0235846,0.000705698,0.0245254,0.0478673,0.0707253,0.093094,0.114969,0.136346,0.157223,0.177597,0.197466,0.216829,0.235685,0.254035,0.271878,0.289215,0.306048,0.322376,0.338202,0.353528,0.368355,0.382687,0.396527,0.409876,0.42274,0.435121,0.447025,0.458455,0.469417,0.479917,0.48996,0.499554,0.508705,0.517421,0.52571,0.533581,0.541043,0.548105,0.554777,0.56107,0.566995,0.572562,0.577783,0.582668,0.587231,0.591482,0.595433,0.599095,0.602481,0.6056,0.608464,0.611084,0.613469,0.615628,0.617572,0.619308,0.620844,0.622187,0.623343,0.624317,0.625114,0.625736,0.626186,0.626466,0.626576,0.626513,0.626276,0.625861,0.625262,0.624474,0.623486,0.622289,0.620872,0.619219,0.617316,0.615143,0.612681,0.609905,0.606791,0.60331,0.59943,0.595118,0.590334,0.585037,0.579184,0.572724,0.565605,0.557772,0.549163,0.539715,0.529358,0.518021,0.505629,0.492102,0.47736,0.46132,0.443896,0.425004,0.40456,0.382483,0.358694,0.33312,0.305698,0.276372,0.245099,0.21185,0.17661,0.139388,0.100206,0.0591124,0.016173,-0.0285225,-0.0748696,-0.122743,-0.172009},
	{-2.59331,-2.56401,-2.53469,-2.50535,-2.47599,-2.4466,-2.41719,-2.38776,-2.3583,-2.32881,-2.2993,-2.26975,-2.24017,-2.21056,-2.18092,-2.15124,-2.12153,-2.09177,-2.06198,-2.03214,-2.00227,-1.97234,-1.94237,-1.91236,-1.88229,-1.85218,-1.82201,-1.79178,-1.7615,-1.73117,-1.70077,-1.67032,-1.6398,-1.60923,-1.57858,-1.54788,-1.5171,-1.48627,-1.45536,-1.42439,-1.39335,-1.36224,-1.33107,-1.29983,-1.26852,-1.23715,-1.20572,-1.17424,-1.14269,-1.1111,-1.07945,-1.04776,-1.01603,-0.984274,-0.95249,-0.920688,-0.888877,-0.857067,-0.825269,-0.793492,-0.761751,-0.730057,-0.698426,-0.666871,-0.635408,-0.604055,-0.572828,-0.541744,-0.510823,-0.480082,-0.449541,-0.41922,-0.389136,-0.359311,-0.329763,-0.30051,-0.271572,-0.242966,-0.21471,-0.18682,-0.159314,-0.132205,-0.105508,-0.0792366,-0.0534031,-0.0280189,-0.00309427,0.0213612,0.0453392,0.068832,0.0918332,0.114337,0.136338,0.157833,0.178818,0.199291,0.219249,0.23869,0.257614,0.27602,0.293909,0.311279,0.328133,0.344471,0.360295,0.375606,0.390407,0.4047,0.418488,0.431775,0.444565,0.456861,0.468669,0.479993,0.490839,0.501214,0.511124,0.520577,0.52958,0.538141,0.54627,0.553976,0.561269,0.568159,0.574656,0.580773,0.58652,0.591908,0.59695,0.601658,0.606044,0.610119,0.613895,0.617385,0.6206,0.62355,0.626247,0.628701,0.630921,0.632918,0.634699,0.636272,0.637645,0.638823,0.639813,0.640617,0.641241,0.641685,0.641951,0.642038,0.641947,0.641672,0.641212,0.640559,0.639706,0.638646,0.637365,0.635854,0.634095,0.632073,0.629769,0.627161,0.624225,0.620934,0.617258,0.613166,0.60862,0.603582,0.598009,0.591854,0.585068,0.577596,0.56938,0.560359,0.550466,0.539632,0.527784,0.514846,0.500738,0.485378,0.468683,0.450568,0.43095,0.409746,0.386877,0.362266,0.335845,0.307554,0.277343,0.245174,0.211025,0.174888,0.13678,0.0967327,0.0548008,0.011059,-0.0343958,-0.0814528,-0.129981,-0.179846},
	{-2.61386,-2.58456,-2.55523,-2.52589,-2.49652,-2.46713,-2.43771,-2.40827,-2.3788,-2.34931,-2.31979,-2.29023,-2.26064,-2.23102,-2.20137,-2.17168,-2.14195,-2.11218,-2.08237,-2.05252,-2.02262,-1.99268,-1.96269,-1.93265,-1.90256,-1.87242,-1.84222,-1.81197,-1.78166,-1.75129,-1.72086,-1.69036,-1.65981,-1.62918,-1.59849,-1.56773,-1.53691,-1.50601,-1.47504,-1.444,-1.41288,-1.3817,-1.35044,-1.31911,-1.28771,-1.25624,-1.2247,-1.19309,-1.16142,-1.12969,-1.0979,-1.06606,-1.03417,-1.00224,-0.97027,-0.938271,-0.906252,-0.87422,-0.842185,-0.810159,-0.778152,-0.746178,-0.714249,-0.682381,-0.650588,-0.618887,-0.587294,-0.555827,-0.524504,-0.493345,-0.462367,-0.43159,-0.401035,-0.37072,-0.340666,-0.310891,-0.281416,-0.252259,-0.223439,-0.194972,-0.166877,-0.13917,-0.111866,-0.0849798,-0.0585251,-0.0325146,-0.00696006,0.0181277,0.042739,0.0668649,0.0904975,0.11363,0.136256,0.158371,0.179969,0.201047,0.221603,0.241633,0.261135,0.280109,0.298553,0.316467,0.333852,0.350707,0.367035,0.382835,0.398111,0.412864,0.427098,0.440814,0.454018,0.466713,0.478904,0.490596,0.501795,0.512507,0.522739,0.532499,0.541794,0.550633,0.559025,0.566981,0.574509,0.581621,0.588328,0.59464,0.600571,0.606132,0.611334,0.616191,0.620715,0.624918,0.628813,0.632411,0.635724,0.638765,0.641543,0.64407,0.646355,0.648409,0.65024,0.651855,0.653263,0.65447,0.655481,0.6563,0.656931,0.657376,0.657636,0.657711,0.657599,0.657298,0.656804,0.656109,0.655207,0.654088,0.652742,0.651154,0.649311,0.647193,0.644783,0.642057,0.63899,0.635556,0.631723,0.627458,0.622724,0.617481,0.611685,0.605289,0.598241,0.590486,0.581965,0.572616,0.562371,0.551161,0.538911,0.525544,0.51098,0.495139,0.477935,0.459287,0.439109,0.417322,0.393847,0.368611,0.341548,0.312601,0.281724,0.248885,0.214064,0.177263,0.1385,0.0978147,0.0552673,0.0109388,-0.0350718,-0.0826473,-0.131654,-0.181957},
	{-2.634,-2.60469,-2.57536,-2.54601,-2.51664,-2.48725,-2.45782,-2.42838,-2.3989,-2.3694,-2.33987,-2.3103,-2.28071,-2.25108,-2.22141,-2.19171,-2.16197,-2.13218,-2.10236,-2.07249,-2.04258,-2.01262,-1.98261,-1.95255,-1.92244,-1.89227,-1.86205,-1.83177,-1.80143,-1.77102,-1.74056,-1.71003,-1.67943,-1.64876,-1.61803,-1.58722,-1.55634,-1.52538,-1.49435,-1.46324,-1.43206,-1.4008,-1.36946,-1.33804,-1.30655,-1.27498,-1.24334,-1.21162,-1.17983,-1.14797,-1.11605,-1.08406,-1.05202,-1.01992,-0.987771,-0.955585,-0.923365,-0.89112,-0.858859,-0.826593,-0.794332,-0.762088,-0.729874,-0.697704,-0.665593,-0.633557,-0.601612,-0.569775,-0.538064,-0.506499,-0.475097,-0.443879,-0.412864,-0.382073,-0.351526,-0.321243,-0.291243,-0.261546,-0.232172,-0.203139,-0.174465,-0.146169,-0.118265,-0.0907717,-0.0637023,-0.0370712,-0.0108916,0.0148244,0.0400658,0.0648224,0.0890849,0.112845,0.136096,0.158831,0.181044,0.20273,0.223885,0.244507,0.264591,0.284136,0.303141,0.321603,0.339524,0.356902,0.373738,0.390033,0.405789,0.421008,0.435692,0.449844,0.463468,0.476567,0.489147,0.501212,0.512768,0.523822,0.534381,0.544452,0.554044,0.563164,0.571824,0.580032,0.587799,0.595136,0.602054,0.608566,0.614683,0.620417,0.625782,0.630789,0.635453,0.639785,0.643797,0.647504,0.650916,0.654045,0.656904,0.659502,0.66185,0.663959,0.665836,0.667491,0.66893,0.67016,0.671187,0.672015,0.672647,0.673086,0.673333,0.673386,0.673245,0.672907,0.672366,0.671617,0.670651,0.669458,0.668028,0.666346,0.664396,0.66216,0.659618,0.656746,0.65352,0.649909,0.645884,0.641408,0.636444,0.63095,0.624882,0.61819,0.610823,0.602723,0.59383,0.584081,0.573407,0.561737,0.548997,0.535107,0.51999,0.503561,0.485739,0.46644,0.445582,0.423085,0.398874,0.372879,0.345037,0.315296,0.283613,0.249963,0.214333,0.176732,0.137184,0.095735,0.0524537,0.00742785,-0.0392385,-0.0874217,-0.13699,-0.187801},
	{-2.65375,-2.62444,-2.5951,-2.56575,-2.53637,-2.50697,-2.47754,-2.44809,-2.41861,-2.3891,-2.35956,-2.32998,-2.30038,-2.27074,-2.24106,-2.21135,-2.1816,-2.1518,-2.12196,-2.09208,-2.06215,-2.03217,-2.00215,-1.97207,-1.94193,-1.91174,-1.8815,-1.85119,-1.82082,-1.79039,-1.75989,-1.72932,-1.69869,-1.66798,-1.6372,-1.60634,-1.57541,-1.5444,-1.51331,-1.48214,-1.45089,-1.41956,-1.38814,-1.35665,-1.32507,-1.2934,-1.26166,-1.22983,-1.19793,-1.16595,-1.13389,-1.10177,-1.06957,-1.03732,-1.005,-0.972634,-0.940222,-0.907774,-0.875296,-0.842799,-0.810293,-0.77779,-0.745302,-0.712842,-0.680425,-0.648065,-0.61578,-0.583586,-0.5515,-0.519541,-0.487729,-0.456082,-0.424621,-0.393367,-0.362339,-0.331558,-0.301045,-0.27082,-0.240903,-0.211314,-0.182071,-0.153194,-0.1247,-0.0966059,-0.0689284,-0.0416826,-0.0148829,0.0114574,0.0373256,0.0627104,0.0876012,0.111989,0.135864,0.159219,0.182048,0.204344,0.226102,0.247318,0.267987,0.288108,0.307677,0.326692,0.345153,0.363059,0.38041,0.397206,0.413448,0.429138,0.444278,0.458872,0.472921,0.48643,0.499404,0.511848,0.523768,0.535169,0.54606,0.556448,0.566341,0.575748,0.584679,0.593144,0.601154,0.60872,0.615854,0.622569,0.628875,0.634787,0.640317,0.645478,0.650284,0.654747,0.658881,0.662698,0.666211,0.669432,0.672373,0.675045,0.677458,0.679624,0.68155,0.683245,0.684718,0.685974,0.687019,0.687858,0.688493,0.688928,0.689163,0.689198,0.689031,0.688658,0.688074,0.687273,0.686247,0.684986,0.683476,0.681704,0.679654,0.677306,0.67464,0.671631,0.668253,0.664477,0.66027,0.655596,0.650416,0.644687,0.638364,0.631396,0.623731,0.615309,0.606071,0.59595,0.584879,0.572784,0.559591,0.545222,0.529595,0.512629,0.494242,0.47435,0.452874,0.429735,0.40486,0.378181,0.34964,0.319187,0.286787,0.252417,0.21607,0.177761,0.137522,0.0954035,0.0514805,0.00584639,-0.0413903,-0.090102,-0.140155,-0.191406},
	{-2.67311,-2.6438,-2.61446,-2.5851,-2.55572,-2.52631,-2.49688,-2.46742,-2.43793,-2.40842,-2.37887,-2.34929,-2.31968,-2.29003,-2.26034,-2.23062,-2.20085,-2.17104,-2.14119,-2.1113,-2.08135,-2.05136,-2.02132,-1.99122,-1.96106,-1.93085,-1.90058,-1.87025,-1.83985,-1.80939,-1.77886,-1.74826,-1.71759,-1.68684,-1.65602,-1.62512,-1.59414,-1.56308,-1.53193,-1.50071,-1.46939,-1.43799,-1.4065,-1.37493,-1.34326,-1.31151,-1.27967,-1.24774,-1.21573,-1.18363,-1.15145,-1.11919,-1.08685,-1.05444,-1.02197,-0.989428,-0.956832,-0.924187,-0.891501,-0.858783,-0.826042,-0.79329,-0.760538,-0.727799,-0.695086,-0.662415,-0.629801,-0.597261,-0.564812,-0.532473,-0.500262,-0.4682,-0.436305,-0.404599,-0.373103,-0.341837,-0.310822,-0.28008,-0.249631,-0.219495,-0.189694,-0.160245,-0.131168,-0.102482,-0.0742033,-0.0463489,-0.0189344,0.0080254,0.0345167,0.0605268,0.0860437,0.111057,0.135556,0.159532,0.182977,0.205884,0.228246,0.250059,0.271317,0.292016,0.312153,0.331726,0.350732,0.36917,0.38704,0.404341,0.421075,0.437241,0.452843,0.467883,0.482363,0.496287,0.509661,0.522488,0.534776,0.546529,0.557756,0.568464,0.578662,0.588359,0.597565,0.606291,0.614547,0.622344,0.629696,0.636614,0.643112,0.649202,0.654898,0.660213,0.66516,0.669754,0.674008,0.677934,0.681546,0.684855,0.687875,0.690617,0.693091,0.695308,0.697277,0.699006,0.700504,0.701777,0.702831,0.70367,0.704297,0.704715,0.704924,0.704924,0.704712,0.704285,0.703637,0.702761,0.701649,0.700289,0.698669,0.696773,0.694585,0.692085,0.689249,0.686054,0.682472,0.678471,0.674019,0.669077,0.663606,0.657561,0.650894,0.643555,0.635488,0.626634,0.61693,0.60631,0.594703,0.582037,0.568234,0.553217,0.536904,0.519215,0.500065,0.479376,0.457066,0.433062,0.407292,0.379693,0.350212,0.318804,0.285441,0.250107,0.212803,0.173552,0.132394,0.0893915,0.0446243,-0.00180542,-0.0497833,-0.099177,-0.14985,-0.20166},
	{-2.69211,-2.66279,-2.63345,-2.60409,-2.5747,-2.54529,-2.51585,-2.48639,-2.4569,-2.42737,-2.39782,-2.36823,-2.33861,-2.30895,-2.27926,-2.24952,-2.21975,-2.18993,-2.16007,-2.13016,-2.1002,-2.07019,-2.04013,-2.01002,-1.97984,-1.94961,-1.91932,-1.88896,-1.85854,-1.82805,-1.79749,-1.76686,-1.73615,-1.70537,-1.67451,-1.64356,-1.61254,-1.58143,-1.55023,-1.51894,-1.48757,-1.4561,-1.42455,-1.3929,-1.36115,-1.32931,-1.29738,-1.26536,-1.23324,-1.20103,-1.16873,-1.13634,-1.10386,-1.07131,-1.03868,-1.00597,-0.973198,-0.940365,-0.907479,-0.874548,-0.841581,-0.808588,-0.775582,-0.742573,-0.709575,-0.676603,-0.643672,-0.610798,-0.577997,-0.545289,-0.512692,-0.480225,-0.447908,-0.415763,-0.38381,-0.35207,-0.320565,-0.289316,-0.258345,-0.227673,-0.197321,-0.16731,-0.137659,-0.108387,-0.0795138,-0.0510565,-0.0230322,0.00454305,0.0316541,0.0582868,0.084428,0.110065,0.135188,0.159785,0.183848,0.207368,0.230338,0.25275,0.2746,0.295882,0.316592,0.336727,0.356283,0.375259,0.393654,0.411467,0.428698,0.445348,0.461418,0.476911,0.491829,0.506176,0.519955,0.533173,0.545836,0.557948,0.569518,0.580554,0.591064,0.601059,0.610547,0.619539,0.628048,0.636084,0.643661,0.65079,0.657486,0.663762,0.669631,0.675108,0.680206,0.684939,0.689321,0.693366,0.697086,0.700495,0.703606,0.706429,0.708976,0.711258,0.713285,0.715065,0.716606,0.717915,0.718998,0.719859,0.720503,0.72093,0.721143,0.72114,0.720919,0.720477,0.719808,0.718904,0.717757,0.716356,0.714688,0.712737,0.710486,0.707914,0.705,0.701718,0.698039,0.693933,0.689365,0.684298,0.67869,0.672497,0.665672,0.658162,0.649911,0.64086,0.630946,0.620102,0.608258,0.59534,0.581272,0.565975,0.54937,0.531373,0.511905,0.490885,0.468234,0.443879,0.41775,0.389785,0.359934,0.328154,0.294418,0.258714,0.221045,0.181438,0.139933,0.0965957,0.0515087,0.00477553,-0.0434879,-0.0931485,-0.144069,-0.196109},
	{-2.71076,-2.68144,-2.65209,-2.62273,-2.59334,-2.56392,-2.53448,-2.50501,-2.47551,-2.44598,-2.41642,-2.38683,-2.3572,-2.32753,-2.29783,-2.26809,-2.2383,-2.20847,-2.1786,-2.14868,-2.1187,-2.08868,-2.0586,-2.02847,-1.99828,-1.96803,-1.93772,-1.90734,-1.87689,-1.84638,-1.81579,-1.78512,-1.75439,-1.72357,-1.69267,-1.66168,-1.63061,-1.59946,-1.56821,-1.53687,-1.50544,-1.47391,-1.44229,-1.41056,-1.37874,-1.34682,-1.3148,-1.28268,-1.25046,-1.21815,-1.18573,-1.15322,-1.12061,-1.08792,-1.05514,-1.02227,-0.989329,-0.956315,-0.923237,-0.890101,-0.856916,-0.823693,-0.790441,-0.757172,-0.7239,-0.690637,-0.657399,-0.624201,-0.59106,-0.557994,-0.525022,-0.492163,-0.459436,-0.426863,-0.394464,-0.362262,-0.330277,-0.298533,-0.267051,-0.235853,-0.204961,-0.174396,-0.144178,-0.114329,-0.0848687,-0.0558151,-0.0271869,0.000998678,0.0287251,0.0559769,0.0827395,0.108999,0.134744,0.159962,0.184642,0.208776,0.232353,0.255368,0.277812,0.299679,0.320965,0.341665,0.361776,0.381294,0.400219,0.418548,0.436281,0.453418,0.469961,0.485912,0.501272,0.516045,0.530236,0.543849,0.55689,0.569365,0.581282,0.592649,0.603474,0.613768,0.62354,0.632802,0.641565,0.649841,0.657643,0.664985,0.671879,0.67834,0.684381,0.690018,0.695263,0.700133,0.70464,0.708798,0.712622,0.716124,0.719318,0.722214,0.724826,0.727163,0.729236,0.731053,0.732623,0.733952,0.735047,0.735912,0.73655,0.736964,0.737154,0.73712,0.736859,0.736367,0.735638,0.734665,0.733437,0.731945,0.730173,0.728106,0.725726,0.723011,0.719939,0.716483,0.712613,0.708298,0.703503,0.698188,0.692311,0.685827,0.678685,0.670834,0.662216,0.652771,0.642434,0.631137,0.618809,0.605377,0.590763,0.574888,0.557672,0.539034,0.518893,0.497171,0.473791,0.448682,0.421776,0.393018,0.362359,0.329762,0.295207,0.258687,0.220213,0.179816,0.137547,0.0934759,0.0476926,0.00030618,-0.0485634,-0.0987797,-0.150205,-0.202698},
	{-2.72907,-2.69974,-2.6704,-2.64102,-2.61163,-2.58221,-2.55276,-2.52329,-2.49379,-2.46425,-2.43468,-2.40508,-2.37545,-2.34578,-2.31606,-2.28631,-2.25652,-2.22668,-2.19679,-2.16686,-2.13688,-2.10684,-2.07675,-2.0466,-2.01639,-1.98612,-1.95579,-1.92539,-1.89492,-1.86438,-1.83377,-1.80307,-1.7723,-1.74145,-1.71052,-1.67949,-1.64838,-1.61718,-1.58589,-1.5545,-1.52301,-1.49142,-1.45973,-1.42794,-1.39605,-1.36405,-1.33194,-1.29973,-1.26742,-1.235,-1.20247,-1.16984,-1.13711,-1.10428,-1.07135,-1.03834,-1.00523,-0.972043,-0.938779,-0.905445,-0.872051,-0.838604,-0.805116,-0.771596,-0.738058,-0.704515,-0.67098,-0.637469,-0.603998,-0.570586,-0.537249,-0.504008,-0.470882,-0.437892,-0.405059,-0.372405,-0.339952,-0.307723,-0.27574,-0.244025,-0.212602,-0.181492,-0.150717,-0.120298,-0.0902567,-0.060613,-0.0313863,-0.00259514,0.0257427,0.0536103,0.0809919,0.107873,0.134238,0.160077,0.185375,0.210123,0.234311,0.257929,0.28097,0.303426,0.325292,0.346561,0.367231,0.387297,0.406756,0.425606,0.443847,0.461478,0.4785,0.494914,0.510723,0.525929,0.540537,0.55455,0.567976,0.58082,0.59309,0.604794,0.615941,0.62654,0.636602,0.646138,0.655161,0.663682,0.671715,0.679273,0.686371,0.693022,0.69924,0.705041,0.71044,0.71545,0.720087,0.724365,0.728297,0.731898,0.73518,0.738156,0.740838,0.743236,0.745362,0.747224,0.74883,0.750188,0.751303,0.752181,0.752824,0.753236,0.753416,0.753364,0.753077,0.75255,0.751779,0.750755,0.749469,0.747907,0.746058,0.743902,0.741423,0.738599,0.735406,0.731816,0.7278,0.723325,0.718355,0.71285,0.706767,0.700061,0.69268,0.684571,0.675675,0.665933,0.655278,0.643643,0.630956,0.617141,0.602124,0.585824,0.568161,0.549055,0.528427,0.506198,0.482294,0.456645,0.429187,0.399866,0.368637,0.335468,0.300339,0.263251,0.224217,0.183275,0.140478,0.0959022,0.0496396,0.00180342,-0.0474844,-0.0980856,-0.149863,-0.202675},
	{-2.74705,-2.71772,-2.68837,-2.659,-2.6296,-2.60017,-2.57072,-2.54124,-2.51174,-2.4822,-2.45262,-2.42302,-2.39338,-2.3637,-2.33398,-2.30422,-2.27441,-2.24457,-2.21467,-2.18473,-2.15473,-2.12468,-2.09458,-2.06441,-2.03419,-2.0039,-1.97355,-1.94313,-1.91264,-1.88207,-1.85143,-1.82072,-1.78992,-1.75903,-1.72806,-1.697,-1.66585,-1.63461,-1.60327,-1.57183,-1.54029,-1.50864,-1.47689,-1.44504,-1.41307,-1.381,-1.34882,-1.31652,-1.28411,-1.25159,-1.21896,-1.18621,-1.15336,-1.1204,-1.08734,-1.05417,-1.02091,-0.987554,-0.954111,-0.920588,-0.886991,-0.853329,-0.819613,-0.785851,-0.752056,-0.718241,-0.684419,-0.650605,-0.616815,-0.583066,-0.549376,-0.515763,-0.482249,-0.448853,-0.415597,-0.382502,-0.349592,-0.316888,-0.284414,-0.252192,-0.220247,-0.188601,-0.157276,-0.126296,-0.0956813,-0.0654542,-0.0356349,-0.00624342,0.0227012,0.0511809,0.0791784,0.106677,0.133663,0.16012,0.186036,0.211399,0.236197,0.260421,0.284061,0.307109,0.329557,0.3514,0.372632,0.393249,0.413247,0.432623,0.451377,0.469506,0.487012,0.503895,0.520156,0.5358,0.550829,0.565249,0.579064,0.592282,0.604908,0.616953,0.628424,0.639332,0.649688,0.659502,0.668787,0.677555,0.685821,0.693598,0.700899,0.707741,0.714137,0.720103,0.725653,0.730804,0.735569,0.739964,0.744002,0.747698,0.751065,0.754116,0.756863,0.759317,0.761488,0.763387,0.765021,0.766398,0.767523,0.768402,0.769037,0.769432,0.769586,0.769498,0.769165,0.768584,0.767747,0.766647,0.765273,0.763612,0.761651,0.75937,0.756753,0.753775,0.750412,0.746637,0.742418,0.737721,0.73251,0.726743,0.720376,0.713362,0.70565,0.697184,0.687905,0.677751,0.666656,0.654552,0.641364,0.62702,0.61144,0.594547,0.576261,0.556502,0.535191,0.512253,0.487614,0.461209,0.432976,0.402865,0.370835,0.336861,0.300929,0.263043,0.223228,0.181524,0.137993,0.0927172,0.0457955,-0.00265777,-0.0525124,-0.103633,-0.155879,-0.20911},
	{-2.76471,-2.73538,-2.70603,-2.67665,-2.64725,-2.61782,-2.58837,-2.55889,-2.52937,-2.49983,-2.47025,-2.44064,-2.41099,-2.38131,-2.35158,-2.32181,-2.292,-2.26214,-2.23224,-2.20228,-2.17228,-2.14222,-2.1121,-2.08192,-2.05168,-2.02138,-1.99101,-1.96057,-1.93006,-1.89947,-1.86881,-1.83806,-1.80723,-1.77632,-1.74532,-1.71422,-1.68304,-1.65175,-1.62037,-1.58888,-1.55729,-1.52559,-1.49378,-1.46186,-1.42983,-1.39769,-1.36542,-1.33304,-1.30055,-1.26793,-1.2352,-1.20234,-1.16938,-1.13629,-1.10309,-1.06979,-1.03637,-1.00285,-0.969239,-0.935532,-0.90174,-0.867871,-0.833933,-0.799938,-0.765895,-0.731816,-0.697716,-0.663608,-0.629508,-0.595432,-0.561399,-0.527426,-0.493534,-0.459742,-0.426073,-0.392548,-0.359189,-0.326021,-0.293065,-0.260347,-0.227888,-0.195715,-0.163849,-0.132314,-0.101133,-0.0703292,-0.0399231,-0.00993629,0.0196109,0.048699,0.0773095,0.105425,0.133028,0.160104,0.186638,0.212616,0.238027,0.262858,0.287099,0.310741,0.333776,0.356196,0.377995,0.399167,0.419709,0.439617,0.458888,0.477522,0.495517,0.512875,0.529596,0.545684,0.561141,0.575972,0.590183,0.60378,0.61677,0.629161,0.640963,0.652186,0.66284,0.672937,0.68249,0.691512,0.700015,0.708016,0.715527,0.722565,0.729143,0.735279,0.740987,0.746283,0.751182,0.755699,0.759848,0.763645,0.767103,0.770235,0.773053,0.77557,0.777794,0.779737,0.781407,0.782812,0.783956,0.784846,0.785485,0.785874,0.786015,0.785905,0.785543,0.784923,0.78404,0.782884,0.781444,0.779708,0.777662,0.775286,0.772562,0.769466,0.765973,0.762054,0.757678,0.752811,0.747414,0.741446,0.734862,0.727614,0.719649,0.710911,0.701342,0.690878,0.679453,0.666997,0.653437,0.638698,0.622703,0.605374,0.586631,0.566395,0.544589,0.521138,0.495971,0.469024,0.440239,0.409567,0.376973,0.342432,0.305938,0.267496,0.227137,0.184906,0.140867,0.0951068,0.0477285,-0.00115118,-0.0514003,-0.102883,-0.155458,-0.208989},
	{-2.78207,-2.75274,-2.72338,-2.694,-2.6646,-2.63517,-2.60571,-2.57622,-2.54671,-2.51716,-2.48758,-2.45796,-2.42831,-2.39861,-2.36888,-2.33911,-2.30929,-2.27942,-2.24951,-2.21954,-2.18953,-2.15945,-2.12932,-2.09913,-2.06888,-2.03856,-2.00817,-1.97772,-1.94718,-1.91658,-1.88589,-1.85512,-1.82427,-1.79332,-1.76229,-1.73116,-1.69994,-1.66862,-1.63719,-1.60566,-1.57402,-1.54227,-1.5104,-1.47843,-1.44633,-1.41411,-1.38178,-1.34932,-1.31673,-1.28403,-1.2512,-1.21824,-1.18516,-1.15195,-1.11863,-1.08518,-1.05162,-1.01795,-0.984168,-0.950284,-0.916303,-0.882233,-0.848083,-0.813861,-0.779577,-0.745244,-0.710875,-0.676482,-0.642081,-0.607688,-0.573321,-0.538997,-0.504737,-0.47056,-0.436487,-0.402542,-0.368745,-0.335122,-0.301695,-0.268488,-0.235526,-0.202834,-0.170435,-0.138354,-0.106614,-0.0752389,-0.0442519,-0.0136751,0.0164701,0.0461628,0.0753831,0.104112,0.132332,0.160025,0.187176,0.213771,0.239794,0.265234,0.290078,0.314318,0.337941,0.360942,0.383311,0.405044,0.426134,0.446578,0.466373,0.485515,0.504006,0.521843,0.53903,0.555566,0.571457,0.586706,0.601318,0.615299,0.628658,0.641401,0.653539,0.665081,0.676039,0.686424,0.696249,0.705527,0.714273,0.722501,0.730226,0.737462,0.744227,0.750536,0.756404,0.761847,0.766882,0.771524,0.775787,0.779687,0.783237,0.786452,0.789343,0.791922,0.794202,0.79619,0.797897,0.799329,0.800494,0.801396,0.802038,0.802423,0.802551,0.80242,0.802029,0.801371,0.800441,0.799229,0.797725,0.795915,0.793784,0.791314,0.788484,0.785271,0.781649,0.777589,0.773059,0.768023,0.762443,0.756277,0.749479,0.742001,0.733788,0.724786,0.714933,0.704166,0.692419,0.679621,0.665699,0.650579,0.634183,0.616432,0.597249,0.576555,0.554273,0.530331,0.504659,0.477196,0.447884,0.416681,0.383551,0.348475,0.311449,0.272485,0.231615,0.188888,0.144373,0.0981606,0.0503562,0.00107854,-0.0495386,-0.101359,-0.154242,-0.208052},
	{-2.79914,-2.7698,-2.74044,-2.71106,-2.68165,-2.65222,-2.62276,-2.59327,-2.56375,-2.5342,-2.50461,-2.47499,-2.44533,-2.41563,-2.38589,-2.35611,-2.32628,-2.29641,-2.26649,-2.23651,-2.20649,-2.1764,-2.14626,-2.11606,-2.08579,-2.05546,-2.02506,-1.99458,-1.96403,-1.9334,-1.90269,-1.8719,-1.84102,-1.81005,-1.77899,-1.74783,-1.71657,-1.68521,-1.65375,-1.62217,-1.59049,-1.55869,-1.52677,-1.49473,-1.46258,-1.43029,-1.39789,-1.36535,-1.33268,-1.29989,-1.26696,-1.23391,-1.20072,-1.1674,-1.13395,-1.10037,-1.06667,-1.03285,-0.998903,-0.964848,-0.930685,-0.896422,-0.862065,-0.827623,-0.793108,-0.758528,-0.723897,-0.689228,-0.654535,-0.619834,-0.585142,-0.550477,-0.515858,-0.481305,-0.44684,-0.412483,-0.378259,-0.34419,-0.310301,-0.276616,-0.243159,-0.209956,-0.177032,-0.144412,-0.11212,-0.0801812,-0.0486193,-0.0174578,0.0132805,0.0435737,0.0734005,0.102741,0.131575,0.159884,0.187652,0.214862,0.241499,0.267549,0.292999,0.317837,0.342053,0.365637,0.388582,0.410878,0.432521,0.453506,0.473828,0.493486,0.512476,0.530799,0.548456,0.565448,0.581777,0.597449,0.612467,0.626839,0.640571,0.653672,0.66615,0.678017,0.689283,0.69996,0.710062,0.719602,0.728594,0.737053,0.744994,0.752434,0.759388,0.765872,0.771903,0.777498,0.782671,0.787439,0.791818,0.795823,0.799468,0.802766,0.805732,0.808377,0.810712,0.812747,0.814492,0.815954,0.817139,0.818054,0.818701,0.819082,0.819198,0.819047,0.818627,0.817933,0.816957,0.815691,0.814123,0.81224,0.810026,0.807463,0.804529,0.801201,0.797453,0.793254,0.788573,0.783373,0.777615,0.771256,0.76425,0.756547,0.748094,0.738834,0.728707,0.717648,0.70559,0.692463,0.678193,0.662707,0.645926,0.627773,0.60817,0.587039,0.564306,0.539898,0.513749,0.485798,0.455991,0.424287,0.390655,0.355079,0.317557,0.278104,0.236758,0.193571,0.148615,0.101983,0.0537842,0.00413816,-0.0468197,-0.0989532,-0.152122,-0.206192},
	{-2.81592,-2.78658,-2.75722,-2.72783,-2.69843,-2.66899,-2.63952,-2.61003,-2.58051,-2.55095,-2.52136,-2.49173,-2.46207,-2.43237,-2.40262,-2.37283,-2.343,-2.31312,-2.28319,-2.25321,-2.22317,-2.19308,-2.16292,-2.13271,-2.10243,-2.07208,-2.04166,-2.01117,-1.98061,-1.94996,-1.91923,-1.88842,-1.85751,-1.82652,-1.79543,-1.76424,-1.73295,-1.70155,-1.67005,-1.63843,-1.6067,-1.57485,-1.54289,-1.51079,-1.47858,-1.44623,-1.41376,-1.38115,-1.3484,-1.31552,-1.28251,-1.24935,-1.21606,-1.18263,-1.14906,-1.11536,-1.08152,-1.04755,-1.01345,-0.979231,-0.944892,-0.910441,-0.875884,-0.841231,-0.80649,-0.771671,-0.736787,-0.70185,-0.666873,-0.631873,-0.596866,-0.561869,-0.526901,-0.491981,-0.457132,-0.422375,-0.387732,-0.353227,-0.318885,-0.28473,-0.250788,-0.217083,-0.183642,-0.150491,-0.117655,-0.0851586,-0.0530281,-0.0212876,0.0100389,0.0409281,0.0713575,0.101306,0.130751,0.159675,0.188059,0.215884,0.243135,0.269795,0.295852,0.321291,0.346102,0.370272,0.393794,0.416658,0.438858,0.460387,0.481242,0.501418,0.520913,0.539727,0.557858,0.575309,0.592083,0.608182,0.623611,0.638377,0.652487,0.665949,0.678772,0.690967,0.702544,0.713517,0.723898,0.733701,0.742941,0.751633,0.759793,0.767436,0.774579,0.78124,0.787433,0.793177,0.798487,0.803381,0.807873,0.811979,0.815714,0.819092,0.822127,0.82483,0.827214,0.829289,0.831063,0.832544,0.83374,0.834656,0.835294,0.835656,0.835745,0.835556,0.835089,0.834336,0.833291,0.831945,0.830286,0.828299,0.825969,0.823277,0.8202,0.816715,0.812794,0.808407,0.803519,0.798095,0.792095,0.785473,0.778185,0.770177,0.761398,0.751788,0.741286,0.729828,0.717345,0.703767,0.689021,0.673032,0.655722,0.637014,0.616831,0.595098,0.571739,0.546685,0.519873,0.491243,0.460746,0.428345,0.394013,0.357737,0.319524,0.279392,0.237384,0.193559,0.147994,0.100785,0.052047,0.0019019,-0.0495135,-0.102062,-0.155606,-0.210012},
	{-2.83242,-2.80308,-2.77372,-2.74433,-2.71492,-2.68548,-2.65602,-2.62652,-2.59699,-2.56743,-2.53784,-2.50821,-2.47854,-2.44883,-2.41908,-2.38929,-2.35945,-2.32956,-2.29962,-2.26963,-2.23958,-2.20948,-2.17932,-2.14909,-2.1188,-2.08844,-2.05801,-2.0275,-1.99692,-1.96625,-1.93551,-1.90467,-1.87374,-1.84272,-1.81161,-1.78039,-1.74907,-1.71764,-1.6861,-1.65444,-1.62267,-1.59078,-1.55876,-1.52662,-1.49434,-1.46194,-1.42939,-1.39671,-1.3639,-1.33093,-1.29783,-1.26458,-1.23119,-1.19765,-1.16397,-1.13014,-1.09617,-1.06206,-1.02782,-0.993435,-0.958926,-0.924293,-0.889544,-0.854685,-0.819725,-0.784675,-0.749545,-0.714347,-0.679096,-0.643805,-0.60849,-0.57317,-0.537861,-0.502585,-0.467361,-0.432212,-0.397159,-0.362228,-0.327442,-0.292826,-0.258406,-0.224208,-0.190258,-0.156583,-0.123209,-0.0901624,-0.0574692,-0.0251551,0.0067548,0.038236,0.0692646,0.0998178,0.129873,0.15941,0.188409,0.216849,0.244714,0.271987,0.298651,0.324694,0.350102,0.374862,0.398965,0.422401,0.445162,0.467241,0.488633,0.509333,0.529339,0.548648,0.567261,0.585178,0.602401,0.618933,0.63478,0.649947,0.664441,0.67827,0.691443,0.703972,0.715867,0.727141,0.737807,0.74788,0.757373,0.766304,0.774687,0.782539,0.789878,0.796719,0.803081,0.808979,0.814432,0.819456,0.824067,0.828281,0.832113,0.835577,0.838687,0.841457,0.843897,0.846018,0.84783,0.849341,0.850557,0.851483,0.852124,0.852481,0.852555,0.852343,0.851844,0.85105,0.849956,0.848551,0.846823,0.844758,0.842339,0.839548,0.836361,0.832754,0.828699,0.824166,0.81912,0.813523,0.807335,0.800512,0.793007,0.784767,0.775738,0.765862,0.755077,0.743318,0.730517,0.716603,0.701502,0.68514,0.66744,0.648326,0.62772,0.605549,0.581738,0.55622,0.528933,0.49982,0.468835,0.435942,0.401118,0.364354,0.325658,0.285055,0.242588,0.198321,0.152334,0.104726,0.055613,0.00511932,-0.0466178,-0.0994617,-0.153275,-0.207926},
	{-2.84866,-2.81932,-2.78996,-2.76057,-2.73115,-2.70171,-2.67224,-2.64274,-2.61321,-2.58365,-2.55405,-2.52441,-2.49474,-2.46503,-2.43527,-2.40547,-2.37563,-2.34574,-2.31579,-2.28579,-2.25574,-2.22563,-2.19545,-2.16522,-2.13492,-2.10454,-2.0741,-2.04358,-2.01298,-1.9823,-1.95153,-1.92067,-1.88973,-1.85868,-1.82754,-1.79629,-1.76494,-1.73348,-1.70191,-1.67021,-1.6384,-1.60647,-1.5744,-1.54221,-1.50988,-1.47741,-1.44481,-1.41206,-1.37917,-1.34613,-1.31294,-1.27961,-1.24612,-1.21247,-1.17868,-1.14474,-1.11064,-1.07639,-1.042,-1.00747,-0.972794,-0.937986,-0.903049,-0.867991,-0.832821,-0.797546,-0.762177,-0.726727,-0.691208,-0.655634,-0.620021,-0.584386,-0.548746,-0.513121,-0.477532,-0.442,-0.406547,-0.371198,-0.335977,-0.300909,-0.266021,-0.231338,-0.196887,-0.162696,-0.128792,-0.095202,-0.0619527,-0.0290711,0.00341672,0.0354849,0.0671085,0.0982631,0.128925,0.159073,0.188684,0.217739,0.246217,0.274102,0.301375,0.328022,0.354029,0.379381,0.404067,0.428078,0.451403,0.474035,0.495968,0.517196,0.537716,0.557525,0.576622,0.595008,0.612685,0.629655,0.645922,0.661493,0.676374,0.690573,0.7041,0.716965,0.729179,0.740756,0.751708,0.76205,0.771798,0.780966,0.789572,0.797632,0.805163,0.812183,0.818709,0.824758,0.830349,0.835497,0.84022,0.844533,0.848453,0.851994,0.855169,0.857992,0.860474,0.862627,0.864459,0.86598,0.867195,0.86811,0.868728,0.869052,0.86908,0.868813,0.868245,0.867371,0.866183,0.864671,0.862823,0.860623,0.858054,0.855096,0.851726,0.847918,0.843644,0.838871,0.833564,0.827685,0.821192,0.814039,0.806178,0.797557,0.788119,0.777805,0.766552,0.754296,0.740966,0.726491,0.710798,0.693812,0.675457,0.655656,0.634334,0.611418,0.586837,0.560526,0.532425,0.502482,0.470654,0.436912,0.401238,0.363629,0.3241,0.28268,0.239424,0.194398,0.147691,0.0994073,0.0496666,-0.00140382,-0.053665,-0.10698,-0.161215,-0.216242},
	{-2.86464,-2.8353,-2.80593,-2.77654,-2.74713,-2.71768,-2.68821,-2.65871,-2.62917,-2.59961,-2.57001,-2.54037,-2.51069,-2.48097,-2.45121,-2.42141,-2.39156,-2.36166,-2.33171,-2.3017,-2.27164,-2.24152,-2.21134,-2.18109,-2.15078,-2.1204,-2.08994,-2.0594,-2.02879,-1.99809,-1.96731,-1.93643,-1.90546,-1.8744,-1.84323,-1.81196,-1.78058,-1.74909,-1.71748,-1.68576,-1.6539,-1.62193,-1.58982,-1.55758,-1.5252,-1.49267,-1.46001,-1.4272,-1.39423,-1.36112,-1.32785,-1.29443,-1.26084,-1.2271,-1.1932,-1.15914,-1.12492,-1.09055,-1.05602,-1.02133,-0.986499,-0.951521,-0.916403,-0.881152,-0.845776,-0.810283,-0.774683,-0.738987,-0.703208,-0.667359,-0.631455,-0.595513,-0.55955,-0.523585,-0.487639,-0.451732,-0.415888,-0.38013,-0.344483,-0.308971,-0.273622,-0.238462,-0.203518,-0.168819,-0.134391,-0.100264,-0.0664645,-0.0330207,4.02134e-05,0.0326913,0.0649062,0.0966594,0.127926,0.158683,0.188906,0.218574,0.247667,0.276164,0.304048,0.331302,0.357909,0.383856,0.40913,0.433719,0.457613,0.480803,0.503282,0.525043,0.546083,0.566398,0.585987,0.604849,0.622985,0.640399,0.657093,0.673075,0.688349,0.702925,0.716812,0.73002,0.74256,0.754446,0.765691,0.77631,0.786318,0.795731,0.804566,0.81284,0.820571,0.827776,0.834474,0.840681,0.846417,0.851697,0.85654,0.860961,0.864977,0.868603,0.871853,0.874739,0.877275,0.879471,0.881337,0.882881,0.884111,0.88503,0.885643,0.885952,0.885956,0.885655,0.885043,0.884116,0.882864,0.881278,0.879344,0.877048,0.874371,0.871292,0.867789,0.863835,0.8594,0.854453,0.848957,0.842872,0.836158,0.828767,0.82065,0.811755,0.802024,0.791398,0.779814,0.767205,0.753503,0.738636,0.722531,0.705112,0.686305,0.666033,0.644223,0.620802,0.595701,0.568857,0.540213,0.509718,0.477334,0.443033,0.406802,0.368642,0.328571,0.286624,0.242857,0.197343,0.150172,0.101451,0.0513032,-0.000143021,-0.0527485,-0.106377,-0.160896,-0.216182},
	{-2.88037,-2.85103,-2.82166,-2.79227,-2.76285,-2.7334,-2.70393,-2.67443,-2.64489,-2.61532,-2.58572,-2.55607,-2.52639,-2.49667,-2.46691,-2.4371,-2.40724,-2.37734,-2.34738,-2.31737,-2.2873,-2.25717,-2.22698,-2.19673,-2.1664,-2.13601,-2.10554,-2.07499,-2.04436,-2.01365,-1.98285,-1.95196,-1.92097,-1.88988,-1.85869,-1.8274,-1.79599,-1.76447,-1.73283,-1.70107,-1.66918,-1.63717,-1.60501,-1.57273,-1.5403,-1.50772,-1.475,-1.44212,-1.40909,-1.37591,-1.34256,-1.30905,-1.27538,-1.24154,-1.20754,-1.17337,-1.13903,-1.10453,-1.06986,-1.03503,-1.00005,-0.964904,-0.92961,-0.894173,-0.858598,-0.822893,-0.787068,-0.751134,-0.715102,-0.678985,-0.642798,-0.606558,-0.57028,-0.533983,-0.497688,-0.461416,-0.425189,-0.389031,-0.352965,-0.317019,-0.281218,-0.245589,-0.21016,-0.17496,-0.140016,-0.105359,-0.0710158,-0.0370158,-0.0033874,0.0298415,0.0626433,0.0949911,0.126859,0.158222,0.189055,0.219335,0.249041,0.278151,0.306646,0.334507,0.361717,0.388261,0.414124,0.439294,0.46376,0.487511,0.510539,0.532838,0.554401,0.575226,0.595309,0.614651,0.633251,0.651112,0.668238,0.684634,0.700305,0.715261,0.729511,0.743064,0.755933,0.76813,0.77967,0.790566,0.800835,0.810493,0.819558,0.828045,0.835974,0.843363,0.850229,0.856592,0.862468,0.867876,0.872833,0.877356,0.881462,0.885164,0.888479,0.891419,0.893997,0.896224,0.898109,0.899661,0.900887,0.901791,0.902379,0.90265,0.902605,0.902242,0.901557,0.900543,0.899192,0.897492,0.89543,0.892991,0.890155,0.886902,0.883207,0.879042,0.874378,0.86918,0.863413,0.857036,0.850006,0.842274,0.833792,0.824505,0.814355,0.803282,0.791222,0.778108,0.76387,0.748438,0.731737,0.713692,0.69423,0.673274,0.650754,0.626596,0.600737,0.573113,0.543674,0.512371,0.47917,0.444051,0.407004,0.368036,0.327172,0.284455,0.239945,0.193721,0.145881,0.0965363,0.0458104,-0.00616338,-0.0592459,-0.113303,-0.168204,-0.22383},
	{-2.89586,-2.86652,-2.83715,-2.80775,-2.77833,-2.74889,-2.71941,-2.6899,-2.66036,-2.63079,-2.60118,-2.57154,-2.54186,-2.51213,-2.48236,-2.45255,-2.42269,-2.39278,-2.36282,-2.3328,-2.30272,-2.27259,-2.24239,-2.21213,-2.18179,-2.15139,-2.12091,-2.09035,-2.05971,-2.02898,-1.99816,-1.96725,-1.93625,-1.90514,-1.87393,-1.84261,-1.81118,-1.77963,-1.74796,-1.71617,-1.68425,-1.65219,-1.62,-1.58767,-1.55519,-1.52256,-1.48978,-1.45685,-1.42375,-1.3905,-1.35708,-1.32349,-1.28973,-1.2558,-1.2217,-1.18742,-1.15297,-1.11834,-1.08355,-1.04858,-1.01344,-0.978138,-0.942674,-0.907055,-0.871287,-0.835376,-0.799333,-0.763167,-0.726889,-0.690512,-0.654049,-0.617517,-0.580932,-0.544312,-0.507676,-0.471046,-0.434444,-0.397893,-0.361418,-0.325045,-0.288799,-0.252709,-0.216803,-0.181109,-0.145656,-0.110475,-0.0755942,-0.0410435,-0.00685249,0.0269498,0.0603347,0.0932741,0.125741,0.157708,0.18915,0.220042,0.250361,0.280084,0.30919,0.337661,0.365476,0.392619,0.419076,0.444831,0.469872,0.494189,0.517772,0.540612,0.562705,0.584045,0.60463,0.624457,0.643528,0.661843,0.679407,0.696223,0.712299,0.727641,0.74226,0.756166,0.76937,0.781885,0.793725,0.804906,0.815443,0.825353,0.834653,0.843361,0.851495,0.859074,0.866116,0.87264,0.878665,0.884208,0.889288,0.893921,0.898124,0.901913,0.905302,0.908306,0.910936,0.913205,0.915122,0.916695,0.917932,0.918838,0.919416,0.919668,0.919594,0.919192,0.918456,0.917382,0.915959,0.914177,0.912021,0.909476,0.906522,0.903138,0.899298,0.894975,0.890138,0.884754,0.878783,0.872187,0.86492,0.856936,0.848182,0.838605,0.828146,0.816744,0.804336,0.790854,0.776229,0.760388,0.743258,0.724767,0.704838,0.683399,0.660378,0.635705,0.609316,0.581153,0.551164,0.519307,0.485549,0.449872,0.412272,0.372759,0.331362,0.288128,0.243121,0.196423,0.148134,0.09837,0.0472548,-0.00507688,-0.0584862,-0.112841,-0.16801,-0.22388},
	{-2.91112,-2.88177,-2.8524,-2.823,-2.79358,-2.76413,-2.73465,-2.70515,-2.67561,-2.64603,-2.61642,-2.58677,-2.55709,-2.52736,-2.49759,-2.46777,-2.4379,-2.40799,-2.37802,-2.348,-2.31792,-2.28777,-2.25757,-2.2273,-2.19695,-2.16654,-2.13605,-2.10548,-2.07482,-2.04408,-2.01325,-1.98233,-1.9513,-1.92018,-1.88895,-1.85761,-1.82615,-1.79458,-1.76288,-1.73105,-1.6991,-1.66701,-1.63478,-1.6024,-1.56988,-1.5372,-1.50437,-1.47138,-1.43822,-1.4049,-1.37141,-1.33774,-1.3039,-1.26988,-1.23568,-1.2013,-1.16674,-1.132,-1.09707,-1.06197,-1.02669,-0.991228,-0.955599,-0.919804,-0.883848,-0.847739,-0.811483,-0.775092,-0.738574,-0.701943,-0.665213,-0.628396,-0.591512,-0.554575,-0.517607,-0.480628,-0.443659,-0.406724,-0.369848,-0.333055,-0.296373,-0.25983,-0.223454,-0.187273,-0.151319,-0.115621,-0.0802084,-0.0451131,-0.010365,0.0240057,0.0579692,0.0914962,0.124558,0.157127,0.189175,0.220677,0.251608,0.281944,0.311663,0.340743,0.369165,0.39691,0.423961,0.450304,0.475924,0.50081,0.52495,0.548337,0.570964,0.592823,0.613913,0.634231,0.653776,0.67255,0.690555,0.707797,0.72428,0.740014,0.755006,0.769267,0.782809,0.795645,0.80779,0.819257,0.830064,0.840227,0.849764,0.858693,0.867033,0.874803,0.88202,0.888706,0.894877,0.900554,0.905753,0.910493,0.914791,0.918661,0.92212,0.925182,0.927859,0.930162,0.932102,0.933688,0.934926,0.935822,0.936379,0.936599,0.936481,0.936023,0.93522,0.934066,0.932551,0.930664,0.928389,0.925712,0.922611,0.919064,0.915046,0.910529,0.90548,0.899866,0.893647,0.886782,0.879227,0.870932,0.861847,0.851915,0.841078,0.829276,0.816441,0.802509,0.787408,0.771067,0.753414,0.734373,0.713873,0.69184,0.668204,0.642898,0.615859,0.587032,0.556368,0.523828,0.489385,0.453024,0.414745,0.374564,0.332515,0.288649,0.243036,0.195762,0.146932,0.0966642,0.0450844,-0.00767037,-0.0614648,-0.116165,-0.171645,-0.227794},
	{-2.92614,-2.8968,-2.86742,-2.83803,-2.8086,-2.77915,-2.74967,-2.72016,-2.69062,-2.66104,-2.63143,-2.60178,-2.57209,-2.54236,-2.51258,-2.48276,-2.45289,-2.42298,-2.393,-2.36297,-2.33289,-2.30274,-2.27253,-2.24225,-2.2119,-2.18147,-2.15097,-2.12039,-2.08972,-2.05897,-2.02813,-1.99719,-1.96615,-1.935,-1.90375,-1.87239,-1.84091,-1.80931,-1.77759,-1.74574,-1.71375,-1.68162,-1.64935,-1.61694,-1.58437,-1.55165,-1.51876,-1.48572,-1.4525,-1.41911,-1.38555,-1.35181,-1.31789,-1.28378,-1.24949,-1.21501,-1.18035,-1.14549,-1.11044,-1.07521,-1.03979,-1.00418,-0.96839,-0.932424,-0.896286,-0.859982,-0.82352,-0.786909,-0.750159,-0.713282,-0.676289,-0.639196,-0.602019,-0.564774,-0.527481,-0.49016,-0.452832,-0.41552,-0.37825,-0.341047,-0.303936,-0.266947,-0.230108,-0.193449,-0.157,-0.120791,-0.0848534,-0.0492192,-0.0139193,0.0210151,0.055553,0.0896639,0.123318,0.156486,0.189139,0.22125,0.252792,0.283741,0.314073,0.343764,0.372793,0.401142,0.428791,0.455724,0.481927,0.507385,0.532088,0.556026,0.57919,0.601575,0.623175,0.643988,0.664013,0.683251,0.701704,0.719376,0.736273,0.752403,0.767773,0.782396,0.796281,0.809444,0.821897,0.833656,0.844737,0.855158,0.864937,0.874092,0.882641,0.890605,0.898002,0.904852,0.911175,0.916988,0.922311,0.927161,0.931556,0.935511,0.939043,0.942166,0.944892,0.947234,0.949201,0.950803,0.952046,0.952936,0.953476,0.953667,0.95351,0.953002,0.952138,0.95091,0.94931,0.947325,0.94494,0.942139,0.9389,0.935202,0.931019,0.92632,0.921074,0.915246,0.908796,0.901682,0.89386,0.885279,0.875888,0.86563,0.854446,0.842275,0.829052,0.814709,0.799175,0.782381,0.764252,0.744717,0.723702,0.701135,0.676949,0.651077,0.62346,0.594043,0.562782,0.529639,0.494592,0.45763,0.418757,0.377993,0.335374,0.290959,0.24482,0.197048,0.147749,0.0970447,0.0450625,-0.00806018,-0.0621888,-0.117191,-0.172944,-0.22934},
	{-2.94095,-2.9116,-2.88223,-2.85283,-2.82341,-2.79395,-2.76447,-2.73496,-2.70542,-2.67584,-2.64622,-2.61657,-2.58688,-2.55714,-2.52737,-2.49754,-2.46767,-2.43774,-2.40777,-2.37773,-2.34764,-2.31749,-2.28727,-2.25698,-2.22662,-2.19619,-2.16568,-2.13509,-2.10441,-2.07365,-2.04279,-2.01184,-1.98078,-1.94962,-1.91835,-1.88697,-1.85547,-1.82385,-1.7921,-1.76022,-1.7282,-1.69604,-1.66374,-1.63128,-1.59867,-1.56591,-1.53297,-1.49987,-1.4666,-1.43315,-1.39952,-1.36571,-1.33171,-1.29752,-1.26314,-1.22857,-1.1938,-1.15883,-1.12366,-1.0883,-1.05274,-1.01699,-0.981049,-0.944917,-0.908602,-0.87211,-0.835448,-0.798624,-0.761647,-0.724529,-0.687282,-0.649919,-0.612456,-0.57491,-0.537299,-0.499644,-0.461964,-0.424285,-0.386628,-0.349021,-0.31149,-0.274063,-0.236769,-0.199637,-0.1627,-0.125987,-0.089531,-0.0533637,-0.0175174,0.0179757,0.0530836,0.0877745,0.122017,0.155781,0.189037,0.221755,0.253908,0.28547,0.316414,0.346716,0.376355,0.405309,0.433558,0.461085,0.487873,0.513908,0.539177,0.56367,0.587376,0.61029,0.632406,0.653719,0.674229,0.693936,0.712842,0.730949,0.748265,0.764795,0.78055,0.795538,0.809772,0.823264,0.83603,0.848085,0.859445,0.870128,0.880151,0.889535,0.898297,0.906457,0.914036,0.921054,0.927528,0.93348,0.938927,0.943889,0.948382,0.952423,0.956029,0.959213,0.961989,0.964368,0.966362,0.967979,0.969226,0.970108,0.97063,0.970792,0.970594,0.970033,0.969105,0.967801,0.966113,0.964027,0.961529,0.9586,0.955222,0.951369,0.947015,0.942132,0.936685,0.930639,0.923954,0.916588,0.908495,0.899624,0.889923,0.879335,0.867802,0.855261,0.841646,0.82689,0.810922,0.793673,0.77507,0.75504,0.733511,0.710414,0.685681,0.659248,0.631057,0.601058,0.569206,0.535471,0.49983,0.462278,0.422822,0.381487,0.338314,0.293364,0.246714,0.198458,0.148705,0.0975791,0.0452084,-0.00826905,-0.0627198,-0.118013,-0.174029,-0.230664},
	{-2.95554,-2.92619,-2.89682,-2.86742,-2.83799,-2.80854,-2.77906,-2.74954,-2.72,-2.69042,-2.6608,-2.63115,-2.60145,-2.57172,-2.54193,-2.51211,-2.48223,-2.4523,-2.42232,-2.39228,-2.36219,-2.33203,-2.3018,-2.27151,-2.24114,-2.2107,-2.18019,-2.14958,-2.1189,-2.08812,-2.05725,-2.02629,-1.99522,-1.96404,-1.93275,-1.90135,-1.86983,-1.83819,-1.80641,-1.77451,-1.74246,-1.71027,-1.67793,-1.64544,-1.61279,-1.57998,-1.547,-1.51385,-1.48052,-1.44702,-1.41332,-1.37944,-1.34537,-1.3111,-1.27663,-1.24197,-1.2071,-1.17202,-1.13674,-1.10126,-1.06557,-1.02967,-0.993581,-0.957288,-0.920802,-0.884127,-0.84727,-0.810238,-0.773041,-0.735688,-0.698192,-0.660566,-0.622825,-0.584985,-0.547064,-0.509081,-0.471058,-0.433017,-0.394982,-0.356979,-0.319035,-0.281177,-0.243434,-0.205838,-0.168419,-0.131209,-0.0942405,-0.0575459,-0.0211586,0.0148882,0.0505616,0.0858286,0.120657,0.155014,0.18887,0.222194,0.254957,0.28713,0.318687,0.349602,0.379851,0.409412,0.438264,0.466386,0.493763,0.520378,0.546217,0.571269,0.595522,0.61897,0.641606,0.663425,0.684425,0.704606,0.72397,0.742518,0.760258,0.777194,0.793337,0.808696,0.823283,0.837111,0.850195,0.86255,0.874192,0.885141,0.895413,0.905029,0.914007,0.922369,0.930133,0.93732,0.94395,0.950042,0.955617,0.960692,0.965286,0.969416,0.973097,0.976344,0.979171,0.98159,0.983612,0.985246,0.986498,0.987375,0.98788,0.988014,0.987777,0.987166,0.986176,0.984799,0.983025,0.980842,0.978233,0.975182,0.971667,0.967664,0.963146,0.958083,0.952442,0.946186,0.939274,0.931665,0.923311,0.914162,0.904164,0.893261,0.881394,0.868499,0.854511,0.839363,0.822985,0.805306,0.786255,0.765759,0.743748,0.720153,0.694908,0.667952,0.639227,0.608686,0.576288,0.542003,0.505815,0.46772,0.427729,0.385871,0.34219,0.296751,0.249634,0.200937,0.15077,0.0992602,0.0465355,-0.0072659,-0.0620113,-0.117572,-0.17383,-0.230686},
	{-2.96992,-2.94057,-2.9112,-2.8818,-2.85237,-2.82292,-2.79344,-2.76392,-2.73437,-2.70479,-2.67517,-2.64552,-2.61582,-2.58608,-2.5563,-2.52647,-2.49659,-2.46666,-2.43667,-2.40663,-2.37653,-2.34636,-2.31613,-2.28583,-2.25546,-2.22502,-2.19449,-2.16388,-2.13318,-2.1024,-2.07152,-2.04054,-2.00945,-1.97826,-1.94696,-1.91554,-1.884,-1.85234,-1.82054,-1.78861,-1.75653,-1.72431,-1.69194,-1.65942,-1.62673,-1.59388,-1.56085,-1.52766,-1.49428,-1.46071,-1.42696,-1.39301,-1.35887,-1.32452,-1.28997,-1.25521,-1.22025,-1.18507,-1.14968,-1.11408,-1.07826,-1.04223,-1.00599,-0.969542,-0.932889,-0.896036,-0.85899,-0.821756,-0.784344,-0.746764,-0.709026,-0.671143,-0.63313,-0.595003,-0.556778,-0.518476,-0.480117,-0.441722,-0.403316,-0.364925,-0.326575,-0.288294,-0.250111,-0.212057,-0.174164,-0.136463,-0.0989884,-0.0617728,-0.0248503,0.0117449,0.0479788,0.0838174,0.119227,0.154175,0.188628,0.222555,0.255926,0.288709,0.320879,0.352406,0.383266,0.413435,0.44289,0.471611,0.499578,0.526775,0.553187,0.578801,0.603605,0.62759,0.650749,0.673078,0.694572,0.715231,0.735056,0.754049,0.772216,0.789562,0.806097,0.82183,0.836772,0.850938,0.864341,0.876998,0.888925,0.90014,0.910662,0.92051,0.929704,0.938264,0.946212,0.953567,0.960349,0.96658,0.972277,0.977462,0.98215,0.986361,0.99011,0.993412,0.996282,0.99873,1.00077,1.00241,1.00365,1.00451,1.00498,1.00507,1.00477,1.00409,1.00301,1.00154,0.999649,0.997337,0.994586,0.991377,0.987687,0.983492,0.978766,0.973476,0.967589,0.961067,0.95387,0.945954,0.937272,0.927771,0.9174,0.906099,0.89381,0.880469,0.86601,0.850366,0.833468,0.815244,0.795624,0.774537,0.751913,0.727685,0.701788,0.674164,0.644758,0.613526,0.58043,0.545444,0.508557,0.469768,0.429095,0.386571,0.342245,0.296187,0.248481,0.19923,0.148547,0.096561,0.0434011,-0.0107944,-0.0658947,-0.121774,-0.178318,-0.235431},
	{-2.98411,-2.95475,-2.92538,-2.89598,-2.86655,-2.8371,-2.80761,-2.7781,-2.74855,-2.71896,-2.68934,-2.65969,-2.62999,-2.60025,-2.57046,-2.54063,-2.51075,-2.48081,-2.45082,-2.42078,-2.39067,-2.3605,-2.33027,-2.29996,-2.26959,-2.23913,-2.2086,-2.17798,-2.14728,-2.11648,-2.08559,-2.0546,-2.0235,-1.9923,-1.96098,-1.92954,-1.89799,-1.8663,-1.83448,-1.80252,-1.77043,-1.73818,-1.70578,-1.67322,-1.6405,-1.6076,-1.57454,-1.54129,-1.50786,-1.47424,-1.44043,-1.40642,-1.37221,-1.33779,-1.30316,-1.26831,-1.23326,-1.19798,-1.16248,-1.12676,-1.09082,-1.05466,-1.01828,-0.98168,-0.944866,-0.90784,-0.870609,-0.83318,-0.795559,-0.757756,-0.719782,-0.681649,-0.643371,-0.604963,-0.566442,-0.527827,-0.489138,-0.450397,-0.411627,-0.372855,-0.334106,-0.295408,-0.256792,-0.218287,-0.179926,-0.141742,-0.103767,-0.0660358,-0.0285835,0.00855525,0.0453453,0.0817516,0.117739,0.153274,0.188323,0.222852,0.256829,0.290223,0.323004,0.355145,0.386617,0.417395,0.447456,0.476777,0.505339,0.533122,0.560111,0.586291,0.61165,0.636178,0.659866,0.682709,0.704703,0.725846,0.746138,0.765582,0.784182,0.801944,0.818877,0.834989,0.850294,0.864803,0.878532,0.891497,0.903713,0.915201,0.925977,0.936063,0.945479,0.954244,0.962381,0.969909,0.97685,0.983224,0.98905,0.994349,0.999139,1.00344,1.00726,1.01063,1.01354,1.01603,1.01809,1.01974,1.02099,1.02183,1.02228,1.02234,1.02199,1.02125,1.02011,1.01855,1.01656,1.01415,1.01127,1.00793,1.00409,0.999729,0.994823,0.989339,0.983241,0.976493,0.969052,0.960875,0.951912,0.942115,0.931426,0.919791,0.907146,0.893431,0.878579,0.862522,0.845192,0.826518,0.806429,0.784857,0.761731,0.736987,0.710562,0.682397,0.652443,0.620655,0.587001,0.551457,0.514014,0.474676,0.433464,0.390415,0.345581,0.299037,0.250869,0.201183,0.150094,0.0977317,0.0442271,-0.0102831,-0.0656689,-0.121807,-0.178586,-0.235913},
	{-2.99809,-2.96874,-2.93936,-2.90996,-2.88053,-2.85108,-2.82159,-2.79208,-2.76253,-2.73294,-2.70332,-2.67366,-2.64396,-2.61422,-2.58443,-2.5546,-2.52471,-2.49477,-2.46478,-2.43473,-2.40462,-2.37445,-2.34421,-2.3139,-2.28352,-2.25306,-2.22252,-2.19189,-2.16118,-2.13038,-2.09947,-2.06847,-2.03736,-2.00615,-1.97482,-1.94336,-1.91179,-1.88008,-1.84824,-1.81627,-1.78414,-1.75187,-1.71944,-1.68685,-1.65409,-1.62116,-1.58805,-1.55476,-1.52129,-1.48762,-1.45375,-1.41968,-1.3854,-1.35091,-1.3162,-1.28127,-1.24613,-1.21075,-1.17515,-1.13932,-1.10326,-1.06697,-1.03045,-0.993708,-0.956737,-0.919544,-0.882134,-0.844513,-0.806689,-0.76867,-0.730466,-0.692089,-0.653552,-0.61487,-0.576059,-0.537138,-0.498127,-0.459046,-0.419921,-0.380774,-0.341634,-0.302528,-0.263485,-0.224537,-0.185715,-0.147053,-0.108584,-0.0703439,-0.0323675,0.00530926,0.0426504,0.0796199,0.116182,0.1523,0.187941,0.223069,0.25765,0.291653,0.325046,0.357799,0.389884,0.421272,0.45194,0.481864,0.511021,0.539393,0.566962,0.593711,0.619629,0.644702,0.668924,0.692285,0.714782,0.736413,0.757177,0.777075,0.796112,0.814294,0.831627,0.848123,0.863793,0.878649,0.892706,0.90598,0.918489,0.93025,0.941283,0.951608,0.961245,0.970216,0.978541,0.986242,0.993339,0.999854,1.00581,1.01122,1.0161,1.02049,1.02438,1.0278,1.03076,1.03328,1.03536,1.03701,1.03825,1.03907,1.03948,1.03949,1.03909,1.03827,1.03704,1.03537,1.03328,1.03073,1.02771,1.0242,1.02018,1.01563,1.01052,1.0048,0.998459,0.991444,0.983718,0.975235,0.965946,0.955801,0.944743,0.932715,0.919657,0.905505,0.890193,0.873654,0.855818,0.836618,0.815982,0.793841,0.770129,0.744782,0.717737,0.688941,0.658344,0.625907,0.5916,0.555402,0.51731,0.477331,0.435491,0.391831,0.346408,0.299301,0.2506,0.200414,0.148861,0.0960715,0.0421774,-0.0126854,-0.0683885,-0.124813,-0.18185,-0.23941},
	{-3.01188,-2.98253,-2.95316,-2.92375,-2.89433,-2.86487,-2.83538,-2.80587,-2.77631,-2.74673,-2.71711,-2.68745,-2.65774,-2.628,-2.59821,-2.56837,-2.53849,-2.50855,-2.47855,-2.4485,-2.41839,-2.38821,-2.35797,-2.32765,-2.29727,-2.2668,-2.23625,-2.20562,-2.1749,-2.14409,-2.11318,-2.08217,-2.05105,-2.01982,-1.98847,-1.95701,-1.92542,-1.89369,-1.86184,-1.82984,-1.79769,-1.76539,-1.73293,-1.70031,-1.66752,-1.63456,-1.60141,-1.56808,-1.53456,-1.50083,-1.46691,-1.43278,-1.39844,-1.36388,-1.3291,-1.29409,-1.25886,-1.22339,-1.18769,-1.15176,-1.11558,-1.07917,-1.04252,-1.00563,-0.968505,-0.931149,-0.893565,-0.855759,-0.817737,-0.779507,-0.741079,-0.702464,-0.663674,-0.624724,-0.58563,-0.54641,-0.507083,-0.467671,-0.428195,-0.388682,-0.349157,-0.309649,-0.270187,-0.230802,-0.191526,-0.152392,-0.113436,-0.0746927,-0.0361978,0.00201192,0.0398995,0.0774278,0.11456,0.151259,0.187489,0.223214,0.258398,0.293009,0.327013,0.360379,0.393076,0.425076,0.456353,0.486881,0.516637,0.545601,0.573752,0.601075,0.627555,0.653179,0.677938,0.701823,0.724829,0.746952,0.768192,0.78855,0.80803,0.826636,0.844377,0.861261,0.877301,0.892508,0.906899,0.920488,0.933293,0.945333,0.956627,0.967195,0.977058,0.986237,0.994755,1.00263,1.00989,1.01655,1.02263,1.02816,1.03314,1.03761,1.04158,1.04506,1.04806,1.05061,1.05271,1.05437,1.0556,1.05641,1.05679,1.05675,1.0563,1.05541,1.05409,1.05234,1.05013,1.04746,1.0443,1.04064,1.03646,1.03172,1.02641,1.02048,1.0139,1.00664,0.998647,0.989878,0.980284,0.969814,0.958412,0.946021,0.932579,0.918022,0.902286,0.885304,0.867005,0.847322,0.826186,0.803528,0.779283,0.753388,0.725784,0.696418,0.665244,0.632225,0.597333,0.560552,0.521882,0.481334,0.438937,0.394736,0.348794,0.30119,0.25202,0.201394,0.149432,0.0962661,0.042028,-0.0131471,-0.0691328,-0.125813,-0.183083,-0.240855},
	{-3.02549,-2.99614,-2.96676,-2.93736,-2.90793,-2.87847,-2.84899,-2.81947,-2.78992,-2.76033,-2.73071,-2.70105,-2.67134,-2.6416,-2.61181,-2.58197,-2.55208,-2.52214,-2.49214,-2.46209,-2.43197,-2.40179,-2.37154,-2.34123,-2.31083,-2.28036,-2.24981,-2.21917,-2.18845,-2.15763,-2.12671,-2.09568,-2.06456,-2.03332,-2.00196,-1.97048,-1.93887,-1.90713,-1.87526,-1.84324,-1.81107,-1.77875,-1.74626,-1.71361,-1.68079,-1.64779,-1.61461,-1.58124,-1.54767,-1.5139,-1.47993,-1.44574,-1.41134,-1.37672,-1.34186,-1.30678,-1.27146,-1.23591,-1.20011,-1.16407,-1.12779,-1.09125,-1.05447,-1.01745,-0.980174,-0.94266,-0.904907,-0.86692,-0.828706,-0.790271,-0.751624,-0.712777,-0.673741,-0.63453,-0.595159,-0.555647,-0.516011,-0.476273,-0.436455,-0.396582,-0.35668,-0.316776,-0.276902,-0.237086,-0.197363,-0.157765,-0.118328,-0.079087,-0.0400793,-0.00134211,0.0370868,0.0751695,0.112868,0.150144,0.18696,0.223279,0.259064,0.294281,0.328895,0.362873,0.396184,0.428796,0.460683,0.491817,0.522173,0.551731,0.580468,0.608368,0.635414,0.661592,0.686892,0.711304,0.734823,0.757444,0.779165,0.799987,0.819914,0.838949,0.857101,0.874378,0.890791,0.906354,0.921081,0.934989,0.948094,0.960415,0.971972,0.982785,0.992876,1.00227,1.01098,1.01903,1.02645,1.03325,1.03947,1.04511,1.05019,1.05475,1.05878,1.06232,1.06537,1.06795,1.07006,1.07173,1.07295,1.07373,1.07408,1.07399,1.07347,1.07251,1.0711,1.06924,1.06692,1.06411,1.06081,1.05699,1.05263,1.0477,1.04217,1.03601,1.02919,1.02166,1.01338,1.00431,0.994397,0.983585,0.97182,0.959046,0.9452,0.930219,0.914038,0.896589,0.877805,0.857617,0.835957,0.812759,0.787957,0.761491,0.733304,0.703345,0.67157,0.637946,0.602448,0.565064,0.525795,0.48466,0.441689,0.396932,0.350456,0.302343,0.252693,0.201618,0.149241,0.0956931,0.0411072,-0.014383,-0.0706541,-0.12759,-0.185093,-0.243078},
	{-3.03892,-3.00956,-2.98019,-2.95079,-2.92136,-2.8919,-2.86241,-2.83289,-2.80334,-2.77375,-2.74413,-2.71447,-2.68476,-2.65502,-2.62523,-2.59539,-2.5655,-2.53555,-2.50555,-2.4755,-2.44538,-2.41519,-2.38494,-2.35462,-2.32423,-2.29375,-2.26319,-2.23255,-2.20182,-2.17099,-2.14006,-2.10903,-2.07789,-2.04664,-2.01527,-1.98378,-1.95216,-1.92041,-1.88852,-1.85648,-1.82429,-1.79194,-1.75944,-1.72676,-1.69391,-1.66088,-1.62766,-1.59425,-1.56064,-1.52683,-1.4928,-1.45856,-1.4241,-1.38942,-1.3545,-1.31934,-1.28394,-1.2483,-1.21241,-1.17627,-1.13988,-1.10323,-1.06632,-1.02916,-0.991747,-0.95408,-0.916163,-0.878001,-0.839599,-0.800964,-0.762105,-0.723031,-0.683755,-0.644288,-0.604648,-0.564849,-0.524911,-0.484854,-0.4447,-0.404474,-0.364201,-0.32391,-0.28363,-0.243391,-0.203227,-0.163171,-0.123259,-0.0835264,-0.0440117,-0.00475225,0.0342132,0.0728456,0.111106,0.148955,0.186354,0.223265,0.25965,0.295471,0.330695,0.365284,0.399208,0.432433,0.464931,0.496672,0.527632,0.557786,0.587113,0.615592,0.643207,0.669944,0.695789,0.720734,0.74477,0.767892,0.7901,0.811391,0.831769,0.851238,0.869806,0.88748,0.904272,0.920195,0.935264,0.949494,0.962902,0.975509,0.987333,0.998395,1.00872,1.01832,1.02723,1.03546,1.04304,1.05,1.05634,1.0621,1.06729,1.07193,1.07603,1.07963,1.08272,1.08533,1.08747,1.08913,1.09035,1.09111,1.09142,1.09128,1.0907,1.08966,1.08817,1.08621,1.08376,1.08083,1.07738,1.0734,1.06886,1.06374,1.05801,1.05162,1.04456,1.03677,1.02822,1.01885,1.00862,0.997477,0.985361,0.972215,0.957978,0.942586,0.925975,0.908079,0.888828,0.868156,0.845995,0.82228,0.796948,0.769939,0.741198,0.710676,0.678334,0.644139,0.608069,0.570118,0.530288,0.488602,0.445095,0.399819,0.352845,0.30426,0.254164,0.202672,0.14991,0.0960069,0.0410986,-0.0146856,-0.0712222,-0.128399,-0.186121,-0.244308},
	{-3.05217,-3.02281,-2.99344,-2.96404,-2.93461,-2.90515,-2.87566,-2.84614,-2.81659,-2.787,-2.75738,-2.72771,-2.69801,-2.66826,-2.63847,-2.60863,-2.57874,-2.54879,-2.51879,-2.48873,-2.45861,-2.42843,-2.39817,-2.36785,-2.33745,-2.30697,-2.27641,-2.24576,-2.21502,-2.18419,-2.15325,-2.12221,-2.09107,-2.05981,-2.02843,-1.99692,-1.96529,-1.93352,-1.90161,-1.86956,-1.83735,-1.80498,-1.77245,-1.73975,-1.70687,-1.67381,-1.64056,-1.60712,-1.57347,-1.53961,-1.50554,-1.47125,-1.43673,-1.40198,-1.367,-1.33177,-1.2963,-1.26057,-1.22459,-1.18836,-1.15186,-1.1151,-1.07807,-1.04078,-1.00323,-0.965413,-0.927337,-0.889004,-0.85042,-0.811591,-0.772525,-0.733231,-0.69372,-0.654005,-0.6141,-0.574021,-0.533788,-0.493419,-0.452936,-0.412364,-0.371728,-0.331056,-0.290376,-0.249721,-0.209123,-0.168616,-0.128235,-0.0880182,-0.0480024,-0.00822654,0.0312699,0.0704471,0.109265,0.147683,0.185662,0.223161,0.260142,0.296566,0.332397,0.367597,0.402133,0.435971,0.46908,0.50143,0.532994,0.563747,0.593664,0.622726,0.650913,0.678211,0.704605,0.730084,0.754641,0.778269,0.800965,0.822729,0.843562,0.863469,0.882455,0.90053,0.917703,0.933989,0.949401,0.963955,0.977669,0.990562,1.00265,1.01397,1.02452,1.03434,1.04344,1.05185,1.05959,1.06669,1.07316,1.07903,1.08432,1.08904,1.09321,1.09686,1.09999,1.10262,1.10476,1.10642,1.10761,1.10834,1.1086,1.1084,1.10774,1.10661,1.10501,1.10292,1.10034,1.09725,1.09363,1.08946,1.08472,1.07938,1.0734,1.06676,1.05941,1.05133,1.04245,1.03274,1.02215,1.01062,0.998095,0.984519,0.96983,0.953964,0.936856,0.918441,0.89865,0.877417,0.854677,0.830364,0.804417,0.776779,0.747397,0.716224,0.683224,0.648368,0.611638,0.57303,0.532553,0.490231,0.446106,0.400234,0.35269,0.303565,0.252962,0.200998,0.147802,0.0935029,0.0382356,-0.0178719,-0.0746994,-0.132138,-0.190097,-0.2485},
	{-3.06524,-3.03589,-3.00652,-2.97711,-2.94768,-2.91823,-2.88874,-2.85922,-2.82967,-2.80008,-2.77046,-2.74079,-2.71109,-2.68134,-2.65154,-2.6217,-2.59181,-2.56186,-2.53186,-2.5018,-2.47168,-2.44149,-2.41124,-2.38091,-2.3505,-2.32002,-2.28946,-2.2588,-2.22806,-2.19722,-2.16628,-2.13524,-2.10408,-2.07281,-2.04142,-2.00991,-1.97826,-1.94648,-1.91456,-1.88249,-1.85026,-1.81787,-1.78532,-1.7526,-1.71969,-1.6866,-1.65332,-1.61984,-1.58615,-1.55226,-1.51814,-1.4838,-1.44923,-1.41443,-1.37938,-1.34408,-1.30854,-1.27273,-1.23667,-1.20034,-1.16374,-1.12687,-1.08973,-1.05231,-1.01462,-0.976662,-0.938431,-0.899933,-0.861171,-0.822153,-0.782885,-0.743376,-0.703636,-0.663678,-0.623515,-0.583163,-0.54264,-0.501966,-0.461161,-0.420249,-0.379256,-0.33821,-0.297138,-0.256073,-0.215047,-0.174095,-0.133252,-0.0925558,-0.0520444,-0.0117573,0.0282654,0.0679829,0.107354,0.146338,0.184893,0.222978,0.260553,0.297579,0.334016,0.369827,0.404975,0.439427,0.473149,0.506109,0.53828,0.569633,0.600144,0.629792,0.658556,0.686419,0.713366,0.739385,0.764468,0.788607,0.811798,0.83404,0.855334,0.875684,0.895095,0.913576,0.931137,0.947791,0.963552,0.978436,0.992462,1.00565,1.01801,1.02958,1.04037,1.0504,1.05971,1.0683,1.07621,1.08346,1.09007,1.09606,1.10144,1.10625,1.1105,1.1142,1.11738,1.12004,1.1222,1.12386,1.12504,1.12574,1.12597,1.12572,1.12499,1.12378,1.12209,1.11989,1.11719,1.11397,1.1102,1.10586,1.10094,1.0954,1.08921,1.08234,1.07476,1.06641,1.05725,1.04725,1.03634,1.02448,1.0116,0.997657,0.982581,0.966311,0.948781,0.929925,0.909678,0.887974,0.864747,0.839934,0.813475,0.785315,0.755402,0.723693,0.690152,0.654755,0.617485,0.578342,0.537339,0.494503,0.449878,0.403524,0.355521,0.305959,0.254946,0.202601,0.149052,0.0944291,0.0388668,-0.0175091,-0.0745802,-0.13224,-0.190403,-0.248994},
	{-3.07816,-3.0488,-3.01943,-2.99003,-2.9606,-2.93114,-2.90165,-2.87213,-2.84258,-2.81299,-2.78337,-2.7537,-2.724,-2.69425,-2.66445,-2.63461,-2.60472,-2.57477,-2.54477,-2.51471,-2.48458,-2.45439,-2.42414,-2.39381,-2.3634,-2.33291,-2.30234,-2.27169,-2.24094,-2.2101,-2.17915,-2.1481,-2.11694,-2.08566,-2.05426,-2.02274,-1.99108,-1.95929,-1.92735,-1.89526,-1.86302,-1.83062,-1.79804,-1.7653,-1.73237,-1.69925,-1.66594,-1.63243,-1.59871,-1.56477,-1.53061,-1.49623,-1.46161,-1.42674,-1.39164,-1.35628,-1.32066,-1.28478,-1.24863,-1.21221,-1.17551,-1.13854,-1.10129,-1.06375,-1.02593,-0.987831,-0.94945,-0.91079,-0.871857,-0.832655,-0.793191,-0.753472,-0.713509,-0.673314,-0.632899,-0.592281,-0.551475,-0.510501,-0.46938,-0.428136,-0.386793,-0.345379,-0.303922,-0.262454,-0.221007,-0.179617,-0.138318,-0.0971486,-0.0561478,-0.0153552,0.0251882,0.0654409,0.105361,0.144906,0.184033,0.222701,0.260868,0.298492,0.335534,0.371954,0.407715,0.442779,0.477114,0.510686,0.543463,0.575419,0.606527,0.636763,0.666105,0.694536,0.72204,0.748603,0.774215,0.798868,0.822558,0.845282,0.867041,0.887837,0.907676,0.926567,0.944519,0.961544,0.977657,0.992875,1.00721,1.02069,1.03333,1.04516,1.05618,1.06644,1.07595,1.08472,1.0928,1.1002,1.10694,1.11304,1.11853,1.12342,1.12774,1.1315,1.13471,1.1374,1.13956,1.14123,1.14239,1.14305,1.14323,1.14292,1.14212,1.14082,1.13902,1.13671,1.13387,1.1305,1.12656,1.12205,1.11692,1.11117,1.10475,1.09763,1.08978,1.08114,1.07168,1.06135,1.0501,1.03788,1.02462,1.01028,0.99478,0.978069,0.96008,0.940747,0.920005,0.897788,0.874033,0.848678,0.821664,0.792938,0.76245,0.730159,0.696034,0.66005,0.622198,0.582478,0.540907,0.497516,0.452354,0.405484,0.356988,0.30696,0.255511,0.202761,0.148839,0.0938756,0.0380041,-0.0186516,-0.0759759,-0.133866,-0.192239,-0.251024},
	{-3.0909,-3.06155,-3.03218,-3.00277,-2.97335,-2.94389,-2.9144,-2.88488,-2.85533,-2.82574,-2.79612,-2.76645,-2.73675,-2.707,-2.6772,-2.64736,-2.61747,-2.58752,-2.55752,-2.52745,-2.49733,-2.46714,-2.43688,-2.40655,-2.37614,-2.34565,-2.31508,-2.28442,-2.25366,-2.22282,-2.19187,-2.16081,-2.12964,-2.09836,-2.06695,-2.03542,-2.00375,-1.97195,-1.94,-1.9079,-1.87564,-1.84322,-1.81063,-1.77786,-1.74491,-1.71176,-1.67842,-1.64488,-1.61113,-1.57715,-1.54295,-1.50852,-1.47386,-1.43894,-1.40378,-1.36836,-1.33267,-1.29672,-1.26049,-1.22398,-1.18719,-1.15012,-1.11276,-1.0751,-1.03716,-0.998922,-0.960396,-0.92158,-0.88248,-0.843099,-0.803443,-0.763521,-0.723341,-0.682914,-0.642254,-0.601374,-0.560292,-0.519025,-0.477595,-0.436025,-0.394339,-0.352563,-0.310728,-0.268863,-0.227002,-0.185179,-0.143431,-0.101795,-0.0603103,-0.0190178,0.0220411,0.0628242,0.103289,0.143391,0.183087,0.222335,0.26109,0.299311,0.336956,0.373984,0.410356,0.446034,0.480982,0.515166,0.548553,0.581114,0.61282,0.643647,0.673572,0.702575,0.730639,0.757749,0.783895,0.809067,0.83326,0.856471,0.878699,0.899947,0.92022,0.939525,0.957873,0.975275,0.991745,1.0073,1.02196,1.03574,1.04866,1.06074,1.07201,1.08249,1.0922,1.10116,1.10941,1.11696,1.12384,1.13006,1.13565,1.14063,1.14502,1.14884,1.15209,1.15481,1.15699,1.15865,1.1598,1.16043,1.16057,1.1602,1.15933,1.15795,1.15605,1.15362,1.15066,1.14714,1.14305,1.13836,1.13305,1.12709,1.12046,1.1131,1.10499,1.09609,1.08634,1.07571,1.06414,1.05158,1.03796,1.02324,1.00736,0.990238,0.971824,0.952051,0.930853,0.908166,0.883928,0.858077,0.830557,0.801315,0.770306,0.737489,0.702835,0.666324,0.627947,0.587709,0.545629,0.501743,0.456101,0.40877,0.359834,0.309392,0.257554,0.204442,0.150187,0.0949186,0.0387689,-0.0181398,-0.0756944,-0.133795,-0.192361,-0.251327},
	{-3.10349,-3.07414,-3.04477,-3.01536,-2.98594,-2.95648,-2.92699,-2.89747,-2.86792,-2.83833,-2.80871,-2.77905,-2.74934,-2.71959,-2.6898,-2.65995,-2.63006,-2.60011,-2.57011,-2.54004,-2.50992,-2.47973,-2.44946,-2.41913,-2.38872,-2.35823,-2.32766,-2.29699,-2.26624,-2.23539,-2.20443,-2.17337,-2.1422,-2.11091,-2.07949,-2.04795,-2.01628,-1.98446,-1.9525,-1.92039,-1.88812,-1.85568,-1.82307,-1.79028,-1.75731,-1.72414,-1.69078,-1.65721,-1.62342,-1.58941,-1.55518,-1.5207,-1.48599,-1.45103,-1.41581,-1.38033,-1.34458,-1.30855,-1.27225,-1.23566,-1.19878,-1.16161,-1.12414,-1.08637,-1.04831,-1.00994,-0.971273,-0.932307,-0.893044,-0.85349,-0.813649,-0.773528,-0.733137,-0.692485,-0.651584,-0.61045,-0.569097,-0.527545,-0.485812,-0.443923,-0.4019,-0.35977,-0.317563,-0.275309,-0.233041,-0.190792,-0.148601,-0.106505,-0.0645428,-0.0227565,0.0188122,0.0601201,0.101124,0.141778,0.182039,0.221862,0.261203,0.300018,0.338263,0.375898,0.41288,0.44917,0.484731,0.519527,0.553523,0.586689,0.618995,0.650415,0.680923,0.7105,0.739127,0.766788,0.79347,0.819164,0.843863,0.867563,0.890263,0.911965,0.932674,0.952397,0.971142,0.988922,1.00575,1.02164,1.03662,1.0507,1.0639,1.07624,1.08775,1.09845,1.10837,1.11752,1.12593,1.13363,1.14064,1.14697,1.15266,1.15773,1.16218,1.16605,1.16934,1.17207,1.17425,1.1759,1.17702,1.17761,1.17769,1.17725,1.17629,1.1748,1.17278,1.17022,1.16711,1.16342,1.15914,1.15425,1.14873,1.14253,1.13564,1.12801,1.1196,1.11039,1.10031,1.08933,1.07738,1.06443,1.0504,1.03525,1.01891,1.00132,0.982409,0.962123,0.940395,0.917161,0.892359,0.865931,0.837822,0.80798,0.776362,0.742931,0.707661,0.670533,0.631544,0.590703,0.548031,0.503568,0.45737,0.409505,0.360062,0.309143,0.25686,0.203337,0.148706,0.0930949,0.0366355,-0.0205525,-0.0783591,-0.136688,-0.195464,-0.254623},
	{-3.11593,-3.08658,-3.0572,-3.0278,-2.99837,-2.96892,-2.93943,-2.90991,-2.88036,-2.85077,-2.82115,-2.79149,-2.76178,-2.73203,-2.70224,-2.67239,-2.6425,-2.61255,-2.58255,-2.55248,-2.52236,-2.49216,-2.4619,-2.43157,-2.40116,-2.37066,-2.34009,-2.30942,-2.27867,-2.24781,-2.21685,-2.18578,-2.15461,-2.12331,-2.09189,-2.06034,-2.02866,-1.99684,-1.96486,-1.93274,-1.90046,-1.868,-1.83538,-1.80257,-1.76958,-1.73639,-1.703,-1.6694,-1.63559,-1.60155,-1.56728,-1.53277,-1.49801,-1.463,-1.42773,-1.39219,-1.35638,-1.32028,-1.28391,-1.24724,-1.21027,-1.17301,-1.13544,-1.09757,-1.05938,-1.02089,-0.982086,-0.942973,-0.903553,-0.86383,-0.823809,-0.783495,-0.742898,-0.702026,-0.660892,-0.619509,-0.577893,-0.536061,-0.494032,-0.45183,-0.409477,-0.367,-0.324428,-0.281791,-0.239122,-0.196455,-0.153828,-0.111277,-0.0688443,-0.0265701,0.0155026,0.05733,0.0988675,0.14007,0.180891,0.221286,0.261209,0.300615,0.339459,0.377697,0.415288,0.45219,0.488363,0.523771,0.558378,0.592151,0.625058,0.657072,0.688167,0.71832,0.747512,0.775726,0.802948,0.829167,0.854376,0.878569,0.901746,0.923906,0.945054,0.965197,0.984344,1.00251,1.0197,1.03593,1.05123,1.06561,1.07909,1.0917,1.10345,1.11438,1.12449,1.13383,1.14241,1.15026,1.1574,1.16385,1.16964,1.17478,1.1793,1.18321,1.18653,1.18928,1.19146,1.19309,1.19418,1.19473,1.19474,1.19422,1.19317,1.19157,1.18943,1.18672,1.18345,1.17959,1.17512,1.17002,1.16427,1.15783,1.15067,1.14276,1.13405,1.12451,1.1141,1.10275,1.09043,1.07707,1.06262,1.04703,1.03022,1.01215,0.992743,0.971938,0.949672,0.925884,0.900513,0.873502,0.844797,0.81435,0.78212,0.748072,0.712183,0.674439,0.634838,0.593393,0.550131,0.505095,0.458343,0.409949,0.360005,0.308615,0.255894,0.201967,0.146966,0.0910198,0.0342572,-0.023204,-0.0812572,-0.13981,-0.198791,-0.258141},
	{-3.12821,-3.09886,-3.06949,-3.04009,-3.01066,-2.9812,-2.95172,-2.9222,-2.89265,-2.86306,-2.83344,-2.80378,-2.77407,-2.74432,-2.71453,-2.68469,-2.65479,-2.62484,-2.59484,-2.56478,-2.53465,-2.50446,-2.47419,-2.44386,-2.41345,-2.38295,-2.35238,-2.32171,-2.29095,-2.26009,-2.22913,-2.19806,-2.16688,-2.13558,-2.10415,-2.0726,-2.04091,-2.00907,-1.97709,-1.94496,-1.91266,-1.8802,-1.84756,-1.81474,-1.78173,-1.74852,-1.71511,-1.68148,-1.64764,-1.61357,-1.57926,-1.54471,-1.50992,-1.47486,-1.43954,-1.40395,-1.36808,-1.33192,-1.29547,-1.25873,-1.22168,-1.18433,-1.14666,-1.10868,-1.07039,-1.03177,-0.992836,-0.953581,-0.914009,-0.874123,-0.833926,-0.793425,-0.752627,-0.711542,-0.67018,-0.628554,-0.58668,-0.544575,-0.502257,-0.459748,-0.417072,-0.374255,-0.331324,-0.288311,-0.245248,-0.202169,-0.159111,-0.116113,-0.0732151,-0.0304588,0.0121124,0.0544541,0.0965208,0.138266,0.179644,0.220607,0.261109,0.301103,0.340543,0.379384,0.417583,0.455095,0.491882,0.527902,0.56312,0.597501,0.631011,0.663622,0.695306,0.726039,0.7558,0.784571,0.812337,0.839085,0.864808,0.889499,0.913156,0.935779,0.957372,0.97794,0.997493,1.01604,1.0336,1.05018,1.0658,1.08049,1.09426,1.10713,1.11913,1.13028,1.14061,1.15014,1.15889,1.16689,1.17416,1.18073,1.18662,1.19184,1.19643,1.20039,1.20375,1.20651,1.2087,1.21032,1.21138,1.21188,1.21184,1.21124,1.2101,1.2084,1.20614,1.2033,1.19988,1.19585,1.1912,1.1859,1.17993,1.17325,1.16584,1.15767,1.14868,1.13884,1.1281,1.11641,1.10373,1.08999,1.07515,1.05915,1.04191,1.02339,1.00352,0.98223,0.959467,0.935169,0.909274,0.881727,0.852477,0.821478,0.788689,0.75408,0.717629,0.679326,0.639171,0.597182,0.553388,0.507836,0.460587,0.411718,0.361325,0.309513,0.256399,0.202109,0.146777,0.0905285,0.0334919,-0.0242173,-0.0824967,-0.141257,-0.200428,-0.259956},
	{-3.14035,-3.111,-3.08163,-3.05223,-3.0228,-2.99334,-2.96386,-2.93434,-2.90479,-2.87521,-2.84558,-2.81592,-2.78622,-2.75647,-2.72667,-2.69683,-2.66694,-2.63699,-2.60699,-2.57692,-2.5468,-2.51661,-2.48634,-2.45601,-2.42559,-2.3951,-2.36452,-2.33385,-2.30309,-2.27223,-2.24127,-2.2102,-2.17901,-2.1477,-2.11628,-2.08472,-2.05302,-2.02118,-1.98919,-1.95705,-1.92474,-1.89226,-1.85961,-1.82678,-1.79375,-1.76052,-1.72709,-1.69344,-1.65957,-1.62547,-1.59113,-1.55655,-1.52171,-1.48662,-1.45125,-1.41561,-1.37968,-1.34346,-1.30695,-1.27013,-1.23301,-1.19557,-1.15781,-1.11973,-1.08132,-1.04259,-1.00353,-0.964135,-0.924416,-0.884371,-0.844004,-0.803321,-0.762328,-0.721035,-0.679451,-0.637589,-0.595463,-0.55309,-0.510489,-0.467681,-0.424688,-0.381537,-0.338254,-0.294872,-0.251421,-0.207937,-0.164455,-0.121016,-0.0776587,-0.0344262,0.00863786,0.0514885,0.0940794,0.136364,0.178294,0.219821,0.260899,0.301478,0.341512,0.380954,0.419759,0.457882,0.49528,0.531914,0.567744,0.602734,0.636849,0.670059,0.702334,0.73365,0.763983,0.793315,0.821628,0.84891,0.875151,0.900345,0.924487,0.947578,0.96962,0.990618,1.01058,1.02952,1.04745,1.06438,1.08034,1.09533,1.10939,1.12254,1.13479,1.14617,1.15671,1.16643,1.17536,1.18351,1.19093,1.19762,1.20361,1.20892,1.21357,1.21759,1.22099,1.22377,1.22597,1.22758,1.22861,1.22908,1.22898,1.22832,1.2271,1.2253,1.22292,1.21996,1.21639,1.21221,1.20738,1.2019,1.19572,1.18883,1.18119,1.17276,1.1635,1.15338,1.14234,1.13034,1.11732,1.10324,1.08803,1.07164,1.05401,1.03508,1.01478,0.993051,0.969835,0.945071,0.9187,0.890668,0.860926,0.829427,0.796136,0.761022,0.724068,0.685264,0.644616,0.602141,0.557874,0.511863,0.464172,0.414883,0.36409,0.311903,0.25844,0.203827,0.148198,0.0916785,0.0343943,-0.0235403,-0.0820263,-0.140977,-0.200325,-0.260019},
	{-3.15234,-3.12299,-3.09362,-3.06422,-3.0348,-3.00534,-2.97586,-2.94634,-2.91679,-2.88721,-2.85758,-2.82792,-2.79822,-2.76847,-2.73868,-2.70884,-2.67895,-2.649,-2.619,-2.58893,-2.55881,-2.52861,-2.49835,-2.46802,-2.4376,-2.40711,-2.37653,-2.34586,-2.3151,-2.28424,-2.25327,-2.2222,-2.19101,-2.1597,-2.12827,-2.0967,-2.065,-2.03316,-2.00116,-1.96901,-1.93669,-1.90421,-1.87154,-1.83869,-1.80565,-1.77241,-1.73895,-1.70528,-1.67139,-1.63727,-1.6029,-1.56828,-1.53341,-1.49827,-1.46286,-1.42717,-1.39119,-1.35492,-1.31834,-1.28145,-1.24425,-1.20673,-1.16888,-1.13071,-1.09219,-1.05335,-1.01416,-0.974637,-0.934775,-0.894577,-0.854046,-0.813186,-0.772004,-0.730508,-0.688707,-0.646615,-0.604244,-0.56161,-0.518732,-0.47563,-0.432327,-0.388848,-0.345221,-0.301475,-0.257644,-0.21376,-0.169861,-0.125987,-0.0821768,-0.0384739,0.00507746,0.0484317,0.091542,0.13436,0.176838,0.218927,0.260577,0.301739,0.342364,0.382406,0.421815,0.460548,0.498559,0.535806,0.572249,0.607849,0.642572,0.676383,0.709253,0.741155,0.772064,0.80196,0.830825,0.858646,0.88541,0.911111,0.935744,0.959307,0.981804,1.00324,1.02362,1.04295,1.06126,1.07855,1.09484,1.11016,1.12451,1.13793,1.15044,1.16206,1.17282,1.18274,1.19185,1.20017,1.20772,1.21454,1.22064,1.22605,1.23078,1.23486,1.2383,1.24112,1.24333,1.24494,1.24596,1.24641,1.24627,1.24555,1.24426,1.24238,1.23991,1.23683,1.23314,1.22882,1.22385,1.2182,1.21185,1.20476,1.19692,1.18827,1.17878,1.16841,1.15711,1.14484,1.13153,1.11715,1.10163,1.08492,1.06695,1.04767,1.02701,1.00491,0.981319,0.956167,0.929401,0.900968,0.87082,0.838912,0.805209,0.769684,0.732321,0.693111,0.652063,0.609196,0.564547,0.518165,0.470118,0.420489,0.369373,0.316881,0.263133,0.208255,0.152379,0.0956315,0.0381373,-0.0199915,-0.0786575,-0.137777,-0.197283,-0.257126},
	{-3.16419,-3.13485,-3.10547,-3.07608,-3.04665,-3.0172,-2.98771,-2.9582,-2.92865,-2.89907,-2.86945,-2.83979,-2.81008,-2.78034,-2.75055,-2.72071,-2.69081,-2.66087,-2.63087,-2.6008,-2.57068,-2.54049,-2.51023,-2.47989,-2.44948,-2.41898,-2.38841,-2.35774,-2.32697,-2.29611,-2.26514,-2.23407,-2.20288,-2.17157,-2.14013,-2.10856,-2.07686,-2.04501,-2.01301,-1.98085,-1.94852,-1.91603,-1.88335,-1.85049,-1.81743,-1.78417,-1.7507,-1.71702,-1.6831,-1.64895,-1.61456,-1.57991,-1.545,-1.50983,-1.47438,-1.43864,-1.40261,-1.36628,-1.32965,-1.29269,-1.25542,-1.21782,-1.17989,-1.14162,-1.10301,-1.06405,-1.02475,-0.985094,-0.945094,-0.904747,-0.864056,-0.823025,-0.78166,-0.739967,-0.697957,-0.65564,-0.613029,-0.570141,-0.526993,-0.483604,-0.439998,-0.396199,-0.352233,-0.308132,-0.263926,-0.219651,-0.175342,-0.131039,-0.0867825,-0.0426159,0.00141646,0.0452683,0.088892,0.132239,0.17526,0.217904,0.260122,0.301863,0.343077,0.383714,0.423727,0.463067,0.501689,0.539548,0.576603,0.612814,0.648144,0.682557,0.716023,0.748512,0.779999,0.810462,0.839881,0.868242,0.895532,0.921743,0.946868,0.970908,0.993862,1.01573,1.03653,1.05627,1.07495,1.0926,1.10923,1.12486,1.13952,1.15322,1.16598,1.17784,1.18881,1.19893,1.20822,1.2167,1.2244,1.23134,1.23755,1.24304,1.24785,1.25199,1.25547,1.25831,1.26053,1.26214,1.26314,1.26354,1.26335,1.26257,1.2612,1.25922,1.25664,1.25344,1.24962,1.24514,1.24,1.23417,1.22762,1.22033,1.21225,1.20336,1.19362,1.18298,1.17139,1.15882,1.1452,1.13049,1.11463,1.09755,1.07921,1.05955,1.03849,1.01599,0.991973,0.966392,0.939189,0.910311,0.879712,0.847349,0.813189,0.777207,0.739388,0.699728,0.658234,0.614931,0.569857,0.523065,0.474623,0.424618,0.373146,0.320319,0.266258,0.211092,0.154949,0.0979565,0.0402379,-0.0180968,-0.0769527,-0.136248,-0.195919,-0.25592},
	{-3.17591,-3.14656,-3.11719,-3.08779,-3.05837,-3.02892,-2.99944,-2.96992,-2.94038,-2.91079,-2.88117,-2.85151,-2.82181,-2.79207,-2.76228,-2.73244,-2.70255,-2.6726,-2.6426,-2.61254,-2.58242,-2.55223,-2.52197,-2.49163,-2.46122,-2.43073,-2.40015,-2.36948,-2.33872,-2.30785,-2.27689,-2.24581,-2.21462,-2.18331,-2.15187,-2.1203,-2.08859,-2.05673,-2.02473,-1.99256,-1.96023,-1.92773,-1.89505,-1.86217,-1.8291,-1.79583,-1.76235,-1.72864,-1.69471,-1.66053,-1.62611,-1.59144,-1.5565,-1.52129,-1.4858,-1.45002,-1.41395,-1.37757,-1.34087,-1.30386,-1.26652,-1.22884,-1.19083,-1.15247,-1.11376,-1.0747,-1.03529,-0.995513,-0.955381,-0.914892,-0.874047,-0.832851,-0.791309,-0.749426,-0.707212,-0.664678,-0.621837,-0.578702,-0.535291,-0.491624,-0.447723,-0.403612,-0.359317,-0.314869,-0.270298,-0.225639,-0.18093,-0.136207,-0.0915143,-0.0468931,-0.00238905,0.0419512,0.0860794,0.129946,0.173501,0.216693,0.259471,0.301782,0.343577,0.384803,0.425411,0.465351,0.504577,0.543042,0.580704,0.617519,0.65345,0.688459,0.722514,0.755585,0.787643,0.818666,0.848634,0.877528,0.905337,0.93205,0.957661,0.982167,1.00557,1.02787,1.04908,1.0692,1.08825,1.10624,1.1232,1.13913,1.15406,1.16801,1.18101,1.19308,1.20425,1.21454,1.22397,1.23258,1.24038,1.2474,1.25368,1.25921,1.26404,1.26818,1.27164,1.27444,1.2766,1.27812,1.27902,1.2793,1.27897,1.27802,1.27646,1.27427,1.27146,1.268,1.26389,1.25911,1.25364,1.24745,1.24051,1.2328,1.22429,1.21493,1.20469,1.19352,1.18137,1.16821,1.15397,1.1386,1.12205,1.10426,1.08517,1.06473,1.04286,1.01951,0.994632,0.968158,0.940037,0.910221,0.878666,0.845333,0.810194,0.773228,0.734426,0.693788,0.651331,0.607085,0.561094,0.513418,0.464134,0.413332,0.361116,0.307603,0.252915,0.197182,0.140532,0.083092,0.0249775,-0.033707,-0.0928707,-0.152443,-0.212366,-0.272599},
	{-3.18749,-3.15815,-3.12878,-3.09938,-3.06996,-3.04051,-3.01103,-2.98151,-2.95197,-2.92239,-2.89277,-2.86311,-2.83341,-2.80367,-2.77388,-2.74404,-2.71415,-2.68421,-2.65421,-2.62415,-2.59403,-2.56384,-2.53358,-2.50325,-2.47284,-2.44234,-2.41177,-2.3811,-2.35034,-2.31947,-2.28851,-2.25743,-2.22624,-2.19492,-2.16349,-2.13191,-2.1002,-2.06834,-2.03633,-2.00416,-1.97183,-1.93932,-1.90663,-1.87374,-1.84066,-1.80738,-1.77388,-1.74016,-1.70621,-1.67201,-1.63757,-1.60287,-1.56791,-1.53266,-1.49714,-1.46132,-1.4252,-1.38877,-1.35202,-1.31495,-1.27754,-1.2398,-1.20171,-1.16326,-1.12446,-1.0853,-1.04578,-1.00589,-0.965627,-0.925,-0.884006,-0.84265,-0.800935,-0.758868,-0.716456,-0.67371,-0.630642,-0.587266,-0.543598,-0.499658,-0.455467,-0.411049,-0.366431,-0.32164,-0.27671,-0.231673,-0.186567,-0.141431,-0.0963051,-0.0512334,-0.00626126,0.0385641,0.083194,0.127578,0.171665,0.215404,0.25874,0.301623,0.343998,0.385814,0.427019,0.467563,0.507396,0.546471,0.584744,0.622169,0.658708,0.694321,0.728973,0.762634,0.795274,0.826868,0.857394,0.886834,0.915174,0.942402,0.968512,0.9935,1.01736,1.04011,1.06174,1.08227,1.1017,1.12006,1.13736,1.15361,1.16885,1.18308,1.19634,1.20865,1.22004,1.23052,1.24014,1.24891,1.25685,1.264,1.27038,1.27601,1.2809,1.28509,1.28859,1.29142,1.29358,1.29509,1.29597,1.2962,1.29581,1.29479,1.29314,1.29085,1.28792,1.28433,1.28008,1.27514,1.26949,1.26311,1.25597,1.24805,1.2393,1.22969,1.21918,1.20774,1.1953,1.18183,1.16727,1.15157,1.13468,1.11653,1.09707,1.07623,1.05397,1.03022,1.00492,0.978024,0.949472,0.919219,0.887222,0.853446,0.817861,0.780452,0.741208,0.700135,0.65725,0.612585,0.566188,0.518122,0.468463,0.417306,0.364755,0.310927,0.255947,0.199945,0.143046,0.0853778,0.0270537,-0.0318239,-0.0911664,-0.150906,-0.210986,-0.27137},
	{-3.19894,-3.1696,-3.14023,-3.11084,-3.08141,-3.05196,-3.02248,-2.99297,-2.96343,-2.93385,-2.90423,-2.87458,-2.84488,-2.81514,-2.78535,-2.75551,-2.72563,-2.69569,-2.66569,-2.63563,-2.60551,-2.57532,-2.54506,-2.51473,-2.48433,-2.45383,-2.42326,-2.39259,-2.36183,-2.33097,-2.3,-2.26893,-2.23774,-2.20642,-2.17498,-2.14341,-2.1117,-2.07984,-2.04782,-2.01565,-1.98331,-1.95079,-1.91809,-1.88521,-1.85212,-1.81882,-1.78531,-1.75157,-1.71761,-1.68339,-1.64893,-1.61421,-1.57922,-1.54395,-1.50839,-1.47253,-1.43637,-1.3999,-1.3631,-1.32597,-1.2885,-1.25069,-1.21253,-1.174,-1.13512,-1.09586,-1.05623,-1.01622,-0.975842,-0.935083,-0.893946,-0.852436,-0.810554,-0.768308,-0.725705,-0.682753,-0.639465,-0.595855,-0.551937,-0.507731,-0.463258,-0.418541,-0.373606,-0.328481,-0.283199,-0.237792,-0.192297,-0.146754,-0.101203,-0.0556881,-0.0102552,0.0350483,0.0801729,0.125068,0.169681,0.21396,0.25785,0.301297,0.344249,0.38665,0.428449,0.469592,0.51003,0.549713,0.588594,0.626628,0.663773,0.699988,0.735238,0.769488,0.802709,0.834874,0.865959,0.895946,0.924817,0.952562,0.979172,1.00464,1.02897,1.05216,1.07422,1.09515,1.11497,1.13369,1.15133,1.16791,1.18344,1.19795,1.21147,1.22402,1.23562,1.2463,1.25609,1.26501,1.2731,1.28036,1.28684,1.29254,1.2975,1.30173,1.30526,1.30809,1.31024,1.31173,1.31256,1.31274,1.31227,1.31116,1.3094,1.30698,1.30391,1.30017,1.29574,1.29061,1.28475,1.27815,1.27077,1.26259,1.25357,1.24367,1.23286,1.22108,1.20831,1.19448,1.17954,1.16345,1.14615,1.12758,1.10768,1.08639,1.06366,1.03943,1.01364,0.986237,0.957173,0.926399,0.893876,0.859569,0.823452,0.785511,0.745739,0.704145,0.660748,0.615583,0.568702,0.52017,0.470067,0.418489,0.365543,0.311349,0.25603,0.199717,0.142535,0.0846097,0.0260522,-0.0330383,-0.0925763,-0.152497,-0.212747,-0.273293},
	{-3.21027,-3.18092,-3.15156,-3.12216,-3.09274,-3.06329,-3.03382,-3.00431,-2.97476,-2.94519,-2.91557,-2.88592,-2.85622,-2.82648,-2.7967,-2.76686,-2.73698,-2.70704,-2.67704,-2.64699,-2.61687,-2.58668,-2.55643,-2.5261,-2.49569,-2.4652,-2.43463,-2.40396,-2.37321,-2.34235,-2.31138,-2.28031,-2.24911,-2.2178,-2.18636,-2.15479,-2.12308,-2.09121,-2.0592,-2.02702,-1.99468,-1.96216,-1.92946,-1.89656,-1.86347,-1.83016,-1.79664,-1.76289,-1.72891,-1.69468,-1.6602,-1.62546,-1.59044,-1.55514,-1.51955,-1.48366,-1.44747,-1.41095,-1.3741,-1.33692,-1.2994,-1.26153,-1.22329,-1.18469,-1.14572,-1.10637,-1.06664,-1.02653,-0.98603,-0.945143,-0.903869,-0.862209,-0.820167,-0.777748,-0.734958,-0.691808,-0.648306,-0.604468,-0.560307,-0.515842,-0.471093,-0.426084,-0.380839,-0.335387,-0.289759,-0.243989,-0.198113,-0.152169,-0.1062,-0.0602488,-0.0143614,0.0314141,0.0770276,0.122428,0.167562,0.212376,0.256815,0.300824,0.344349,0.387333,0.429722,0.471463,0.512504,0.552794,0.592283,0.630926,0.668678,0.705497,0.741346,0.776188,0.809993,0.842732,0.87438,0.904916,0.934324,0.96259,0.989704,1.01566,1.04046,1.0641,1.08659,1.10793,1.12814,1.14723,1.16522,1.18212,1.19796,1.21276,1.22654,1.23933,1.25116,1.26204,1.27201,1.2811,1.28932,1.29671,1.30329,1.30909,1.31412,1.3184,1.32196,1.32481,1.32697,1.32844,1.32924,1.32937,1.32884,1.32766,1.3258,1.32328,1.32009,1.31621,1.31163,1.30634,1.3003,1.2935,1.28592,1.27751,1.26825,1.25811,1.24703,1.23498,1.22191,1.20777,1.19252,1.17609,1.15845,1.13952,1.11925,1.09758,1.07446,1.04983,1.02363,0.99581,0.966322,0.93512,0.902165,0.867424,0.830873,0.7925,0.752299,0.710282,0.66647,0.620902,0.573629,0.52472,0.474257,0.422337,0.36907,0.314575,0.258977,0.202406,0.144987,0.0868435,0.0280852,-0.0311907,-0.0909013,-0.150984,-0.211388,-0.27208},
	{-3.22147,-3.19212,-3.16276,-3.13337,-3.10395,-3.0745,-3.04502,-3.01552,-2.98597,-2.9564,-2.92679,-2.89713,-2.86744,-2.8377,-2.80792,-2.77809,-2.7482,-2.71827,-2.68827,-2.65822,-2.62811,-2.59792,-2.56767,-2.53734,-2.50694,-2.47645,-2.44588,-2.41522,-2.38446,-2.3536,-2.32264,-2.29157,-2.26038,-2.22907,-2.19763,-2.16606,-2.13434,-2.10248,-2.07047,-2.03829,-2.00595,-1.97342,-1.94072,-1.90782,-1.87471,-1.8414,-1.80787,-1.77411,-1.74012,-1.70588,-1.67138,-1.63662,-1.60158,-1.56626,-1.53064,-1.49472,-1.45849,-1.42193,-1.38504,-1.34782,-1.31024,-1.2723,-1.234,-1.19533,-1.15628,-1.11685,-1.07702,-1.03681,-0.996196,-0.955187,-0.913781,-0.871978,-0.829782,-0.787196,-0.744228,-0.700885,-0.657177,-0.613118,-0.568721,-0.524005,-0.478989,-0.433695,-0.388148,-0.342378,-0.296413,-0.250288,-0.204038,-0.157703,-0.111324,-0.0649445,-0.018611,0.0276282,0.0737227,0.11962,0.165268,0.21061,0.255591,0.300155,0.344245,0.387806,0.430781,0.473115,0.514753,0.555645,0.595739,0.634986,0.673342,0.710762,0.747206,0.782638,0.817024,0.850335,0.882543,0.913627,0.943569,0.972353,0.99997,1.02641,1.05168,1.07576,1.09868,1.12043,1.14103,1.16048,1.17881,1.19604,1.21218,1.22725,1.24129,1.25431,1.26635,1.27743,1.28757,1.2968,1.30516,1.31266,1.31932,1.32519,1.33026,1.33458,1.33815,1.34099,1.34312,1.34455,1.34529,1.34534,1.34472,1.34341,1.34143,1.33876,1.33539,1.33133,1.32654,1.32102,1.31475,1.30769,1.29983,1.29112,1.28155,1.27107,1.25964,1.24721,1.23375,1.21921,1.20352,1.18665,1.16854,1.14912,1.12835,1.10616,1.08251,1.05733,1.03057,1.00217,0.972096,0.940299,0.906742,0.871394,0.834235,0.795255,0.754453,0.711842,0.667448,0.621312,0.573492,0.524058,0.473095,0.420706,0.367001,0.312101,0.256131,0.199223,0.141499,0.0830812,0.0240755,-0.0354248,-0.0953402,-0.155613,-0.216195,-0.277055},
	{-3.23254,-3.2032,-3.17384,-3.14445,-3.11503,-3.08559,-3.05611,-3.0266,-2.99707,-2.96749,-2.93788,-2.90823,-2.87854,-2.8488,-2.81902,-2.78919,-2.75931,-2.72938,-2.69939,-2.66934,-2.63922,-2.60904,-2.57879,-2.54847,-2.51807,-2.48759,-2.45702,-2.42636,-2.3956,-2.36475,-2.33379,-2.30272,-2.27153,-2.24022,-2.20879,-2.17722,-2.1455,-2.11364,-2.08163,-2.04945,-2.01711,-1.98458,-1.95187,-1.91897,-1.88586,-1.85255,-1.81901,-1.78524,-1.75124,-1.71698,-1.68247,-1.64769,-1.61264,-1.57729,-1.54165,-1.5057,-1.46944,-1.43285,-1.39592,-1.35865,-1.32102,-1.28303,-1.24467,-1.20593,-1.1668,-1.12729,-1.08737,-1.04706,-1.00634,-0.965214,-0.92368,-0.88174,-0.839395,-0.796649,-0.753507,-0.709978,-0.66607,-0.621796,-0.57717,-0.532208,-0.486931,-0.441359,-0.395518,-0.349434,-0.303139,-0.256666,-0.210049,-0.163329,-0.116546,-0.0697438,-0.0229697,0.0237279,0.0702983,0.116689,0.162845,0.208711,0.254231,0.299347,0.344002,0.388138,0.431697,0.474622,0.51686,0.558354,0.599054,0.638909,0.677871,0.715895,0.75294,0.788966,0.823939,0.857828,0.890603,0.922243,0.952726,0.982038,1.01017,1.0371,1.06284,1.08739,1.11074,1.13291,1.15391,1.17374,1.19243,1.20999,1.22644,1.24181,1.25612,1.2694,1.28167,1.29295,1.30329,1.31269,1.3212,1.32883,1.33561,1.34157,1.34673,1.35111,1.35472,1.35759,1.35974,1.36116,1.36188,1.3619,1.36123,1.35987,1.35781,1.35505,1.35159,1.34741,1.34249,1.33683,1.33041,1.32319,1.31515,1.30626,1.29648,1.28579,1.27414,1.26148,1.24778,1.23298,1.21704,1.19989,1.1815,1.16179,1.14073,1.11824,1.09427,1.06877,1.04169,1.01297,0.982565,0.950436,0.916546,0.880865,0.843375,0.804065,0.762937,0.720006,0.675298,0.628857,0.580741,0.531022,0.479787,0.427137,0.373186,0.318053,0.261865,0.204753,0.146839,0.088242,0.029069,-0.0305888,-0.0906533,-0.151069,-0.211787,-0.272779},
	{-3.2435,-3.21416,-3.1848,-3.15541,-3.12599,-3.09655,-3.06708,-3.03757,-3.00804,-2.97846,-2.94886,-2.91921,-2.88952,-2.85979,-2.83001,-2.80018,-2.7703,-2.74037,-2.71039,-2.68034,-2.65023,-2.62005,-2.5898,-2.55948,-2.52909,-2.49861,-2.46804,-2.43739,-2.40663,-2.37578,-2.34483,-2.31376,-2.28258,-2.25127,-2.21984,-2.18827,-2.15656,-2.1247,-2.09269,-2.06051,-2.02816,-1.99564,-1.96293,-1.93002,-1.89692,-1.8636,-1.83005,-1.79628,-1.76227,-1.728,-1.69348,-1.65869,-1.62361,-1.58825,-1.55259,-1.51662,-1.48032,-1.4437,-1.40673,-1.36942,-1.33175,-1.2937,-1.25529,-1.21648,-1.17729,-1.1377,-1.0977,-1.05729,-1.01647,-0.975235,-0.933581,-0.89151,-0.849023,-0.806123,-0.762815,-0.719107,-0.675007,-0.630526,-0.585679,-0.540481,-0.494951,-0.44911,-0.402983,-0.356597,-0.309981,-0.263169,-0.216195,-0.1691,-0.121923,-0.0747086,-0.0275041,0.0196418,0.066678,0.113551,0.160207,0.206587,0.252636,0.298295,0.343505,0.388206,0.432341,0.47585,0.518678,0.560767,0.602065,0.642519,0.68208,0.720702,0.758339,0.794953,0.830506,0.864966,0.898302,0.930489,0.961507,0.991339,1.01997,1.04739,1.0736,1.0986,1.12238,1.14496,1.16634,1.18654,1.20557,1.22345,1.2402,1.25584,1.27041,1.28391,1.29639,1.30786,1.31836,1.32791,1.33653,1.34427,1.35113,1.35714,1.36234,1.36673,1.37034,1.37319,1.37529,1.37666,1.3773,1.37722,1.37643,1.37492,1.37271,1.36977,1.36611,1.36172,1.35657,1.35066,1.34395,1.33644,1.32808,1.31886,1.30873,1.29766,1.28561,1.27254,1.2584,1.24314,1.22671,1.20907,1.19015,1.1699,1.14828,1.12521,1.10064,1.07453,1.04682,1.01746,0.986397,0.953603,0.919041,0.882684,0.844516,0.804532,0.762735,0.719144,0.673792,0.626724,0.578005,0.52771,0.475929,0.422769,0.368344,0.312776,0.256192,0.198722,0.140487,0.0816031,0.0221727,-0.0377172,-0.0979941,-0.158605,-0.219507,-0.280673},
	{-3.25434,-3.225,-3.19564,-3.16626,-3.13684,-3.1074,-3.07793,-3.04843,-3.01889,-2.98932,-2.95972,-2.93007,-2.90039,-2.87066,-2.84088,-2.81106,-2.78118,-2.75125,-2.72127,-2.69123,-2.66112,-2.63095,-2.6007,-2.57039,-2.53999,-2.50952,-2.47896,-2.4483,-2.41756,-2.38671,-2.35576,-2.32469,-2.29351,-2.26221,-2.23078,-2.19922,-2.16751,-2.13565,-2.10364,-2.07147,-2.03913,-2.0066,-1.97389,-1.94099,-1.90788,-1.87455,-1.84101,-1.80723,-1.77321,-1.73894,-1.70441,-1.6696,-1.63452,-1.59914,-1.56346,-1.52746,-1.49114,-1.45449,-1.41749,-1.38014,-1.34242,-1.30433,-1.26586,-1.227,-1.18774,-1.14808,-1.108,-1.0675,-1.02659,-0.985245,-0.943476,-0.901279,-0.858656,-0.815608,-0.77214,-0.728259,-0.683973,-0.639292,-0.594229,-0.548801,-0.503025,-0.456922,-0.410516,-0.363833,-0.316903,-0.269758,-0.222434,-0.174969,-0.127404,-0.0797834,-0.0321539,0.0154352,0.0629324,0.110284,0.157434,0.204326,0.250902,0.297101,0.342863,0.388129,0.432839,0.476932,0.520351,0.563037,0.604935,0.645992,0.686156,0.725379,0.763616,0.800823,0.836963,0.872001,0.905906,0.938651,0.970214,1.00058,1.02972,1.05764,1.08433,1.10979,1.13402,1.15702,1.1788,1.19938,1.21877,1.23699,1.25406,1.27001,1.28484,1.29861,1.31131,1.323,1.33369,1.34341,1.35219,1.36005,1.36703,1.37314,1.37841,1.38287,1.38653,1.3894,1.39152,1.39288,1.3935,1.39338,1.39254,1.39098,1.38868,1.38566,1.3819,1.37738,1.37211,1.36605,1.3592,1.35152,1.34299,1.33358,1.32325,1.31197,1.2997,1.2864,1.27202,1.25651,1.23983,1.22192,1.20272,1.1822,1.16028,1.13691,1.11205,1.08563,1.0576,1.02792,0.996542,0.963428,0.928543,0.891864,0.853375,0.813073,0.770962,0.727062,0.681408,0.634047,0.585043,0.534474,0.482431,0.429022,0.37436,0.318569,0.261777,0.204111,0.145692,0.0866372,0.0270455,-0.0329969,-0.0934187,-0.154168,-0.215204,-0.276499},
	{-3.26507,-3.23573,-3.20637,-3.17699,-3.14758,-3.11814,-3.08867,-3.05917,-3.02964,-3.00007,-2.97046,-2.94082,-2.91114,-2.88141,-2.85164,-2.82182,-2.79195,-2.76202,-2.73204,-2.702,-2.6719,-2.64173,-2.61149,-2.58118,-2.55079,-2.52032,-2.48976,-2.45911,-2.42837,-2.39753,-2.36658,-2.33552,-2.30435,-2.27305,-2.24162,-2.21006,-2.17836,-2.14651,-2.1145,-2.08233,-2.04999,-2.01747,-1.98476,-1.95186,-1.91875,-1.88542,-1.85188,-1.8181,-1.78407,-1.7498,-1.71526,-1.68044,-1.64534,-1.60995,-1.57425,-1.53824,-1.5019,-1.46522,-1.42819,-1.39081,-1.35305,-1.31492,-1.2764,-1.23749,-1.19817,-1.15843,-1.11828,-1.07771,-1.0367,-0.995257,-0.95338,-0.911064,-0.868312,-0.825123,-0.781503,-0.737457,-0.692993,-0.64812,-0.602851,-0.557202,-0.511189,-0.464833,-0.418158,-0.371188,-0.323954,-0.276487,-0.228822,-0.180999,-0.133057,-0.0850406,-0.0369971,0.0110243,0.0589716,0.106791,0.154425,0.201818,0.248908,0.295637,0.341942,0.387763,0.433038,0.477705,0.521705,0.564979,0.607468,0.649119,0.689878,0.729694,0.76852,0.806313,0.843032,0.87864,0.913106,0.946399,0.978498,1.00938,1.03903,1.06744,1.0946,1.12051,1.14517,1.16858,1.19075,1.2117,1.23144,1.24998,1.26735,1.28357,1.29867,1.31266,1.32558,1.33745,1.3483,1.35817,1.36707,1.37503,1.38208,1.38825,1.39356,1.39803,1.40168,1.40453,1.4066,1.4079,1.40843,1.40822,1.40725,1.40555,1.40309,1.39989,1.39592,1.39119,1.38568,1.37937,1.37224,1.36427,1.35542,1.34567,1.33499,1.32334,1.31067,1.29696,1.28214,1.26618,1.24902,1.23062,1.21091,1.18985,1.16738,1.14345,1.118,1.09098,1.06234,1.03203,1.00001,0.966253,0.930715,0.893381,0.854237,0.813281,0.770525,0.725991,0.679717,0.631756,0.582175,0.531056,0.478496,0.424602,0.369493,0.313292,0.256128,0.198126,0.139407,0.0800817,0.0202483,-0.0400141,-0.100638,-0.161574,-0.222787,-0.284249},
	{-3.27568,-3.24635,-3.21699,-3.18761,-3.1582,-3.12876,-3.0993,-3.0698,-3.04027,-3.0107,-2.9811,-2.95146,-2.92178,-2.89206,-2.86229,-2.83247,-2.80261,-2.77268,-2.74271,-2.71267,-2.68257,-2.65241,-2.62217,-2.59186,-2.56148,-2.53101,-2.50046,-2.46982,-2.43908,-2.40824,-2.3773,-2.34625,-2.31508,-2.28379,-2.25237,-2.22081,-2.18911,-2.15727,-2.12526,-2.0931,-2.06076,-2.02824,-1.99554,-1.96264,-1.92953,-1.89621,-1.86266,-1.82888,-1.79485,-1.76057,-1.72603,-1.69121,-1.6561,-1.6207,-1.58499,-1.54896,-1.5126,-1.4759,-1.43884,-1.40143,-1.36364,-1.32547,-1.28691,-1.24794,-1.20856,-1.16877,-1.12855,-1.08789,-1.0468,-1.00527,-0.963287,-0.920858,-0.877982,-0.834659,-0.790893,-0.746688,-0.702052,-0.656994,-0.611526,-0.565662,-0.519419,-0.472817,-0.425879,-0.37863,-0.331098,-0.283316,-0.235318,-0.187142,-0.138829,-0.0904233,-0.0419717,0.00647626,0.0548683,0.10315,0.151264,0.199153,0.246756,0.294011,0.340857,0.38723,0.433069,0.47831,0.522892,0.566754,0.609837,0.652084,0.693441,0.733854,0.773276,0.81166,0.848964,0.88515,0.920185,0.954037,0.986681,1.0181,1.04826,1.07718,1.10482,1.13119,1.1563,1.18014,1.20272,1.22405,1.24415,1.26304,1.28073,1.29725,1.31262,1.32687,1.34003,1.35211,1.36316,1.3732,1.38225,1.39035,1.39752,1.40378,1.40917,1.4137,1.4174,1.42028,1.42235,1.42365,1.42416,1.42391,1.4229,1.42113,1.4186,1.41531,1.41124,1.4064,1.40076,1.39431,1.38703,1.37889,1.36987,1.35994,1.34906,1.33721,1.32433,1.31039,1.29534,1.27914,1.26173,1.24306,1.22309,1.20176,1.17901,1.15479,1.12905,1.10174,1.0728,1.04219,1.00986,0.975796,0.939951,0.902309,0.862858,0.821599,0.778542,0.733714,0.687152,0.638911,0.589059,0.53768,0.48487,0.430739,0.375405,0.318992,0.261629,0.20344,0.144545,0.0850556,0.0250669,-0.0353428,-0.0961074,-0.157179,-0.218522,-0.280111},
	{-3.28619,-3.25686,-3.2275,-3.19812,-3.16871,-3.13928,-3.10981,-3.08032,-3.05079,-3.02123,-2.99163,-2.962,-2.93232,-2.9026,-2.87283,-2.84302,-2.81316,-2.78324,-2.75327,-2.72323,-2.69314,-2.66298,-2.63275,-2.60245,-2.57207,-2.54161,-2.51106,-2.48042,-2.44969,-2.41886,-2.38792,-2.35687,-2.32571,-2.29442,-2.26301,-2.23146,-2.19977,-2.16793,-2.13593,-2.10377,-2.07144,-2.03893,-2.00623,-1.97333,-1.94023,-1.90691,-1.87336,-1.83958,-1.80556,-1.77128,-1.73673,-1.70191,-1.66679,-1.63138,-1.59566,-1.55962,-1.52324,-1.48652,-1.44945,-1.41201,-1.37419,-1.33598,-1.29738,-1.25837,-1.21894,-1.17909,-1.13881,-1.09808,-1.05691,-1.01529,-0.973209,-0.930675,-0.887682,-0.844232,-0.800327,-0.755972,-0.711172,-0.665938,-0.620279,-0.574209,-0.527746,-0.480907,-0.433716,-0.386197,-0.338378,-0.29029,-0.241969,-0.193452,-0.144779,-0.0959944,-0.0471454,0.00171842,0.0505446,0.0992781,0.147862,0.196236,0.24434,0.292112,0.339489,0.386406,0.432799,0.478605,0.52376,0.568202,0.61187,0.654706,0.696653,0.737656,0.777666,0.816635,0.854518,0.891275,0.926872,0.961276,0.994459,1.0264,1.05708,1.08648,1.11461,1.14144,1.16698,1.19124,1.21422,1.23593,1.25638,1.2756,1.2936,1.31041,1.32605,1.34054,1.35392,1.3662,1.37743,1.38762,1.3968,1.40501,1.41227,1.41861,1.42405,1.42861,1.43232,1.43519,1.43725,1.4385,1.43895,1.43862,1.43752,1.43563,1.43297,1.42953,1.42531,1.42028,1.41445,1.40779,1.40028,1.3919,1.38262,1.37242,1.36125,1.34909,1.33589,1.32161,1.30621,1.28964,1.27185,1.25279,1.23241,1.21065,1.18746,1.16279,1.13659,1.1088,1.07938,1.04828,1.01546,0.980883,0.944529,0.906378,0.866418,0.824654,0.781099,0.735781,0.688741,0.640037,0.589742,0.537938,0.484729,0.430223,0.374541,0.317808,0.260152,0.201697,0.142561,0.0828512,0.0226605,-0.0379328,-0.09887,-0.160104,-0.221603,-0.283341},
	{-3.29659,-3.26726,-3.2379,-3.20853,-3.17912,-3.14969,-3.12023,-3.09074,-3.06121,-3.03165,-3.00206,-2.97242,-2.94275,-2.91303,-2.88327,-2.85346,-2.8236,-2.79369,-2.76372,-2.73369,-2.7036,-2.67345,-2.64322,-2.61293,-2.58255,-2.5521,-2.52156,-2.49092,-2.4602,-2.42937,-2.39844,-2.3674,-2.33625,-2.30497,-2.27356,-2.24202,-2.21033,-2.1785,-2.14651,-2.11436,-2.08203,-2.04953,-2.01683,-1.98394,-1.95084,-1.91753,-1.88399,-1.85021,-1.81619,-1.78191,-1.74736,-1.71254,-1.67742,-1.642,-1.60627,-1.57022,-1.53383,-1.4971,-1.46001,-1.42254,-1.3847,-1.34646,-1.30782,-1.26877,-1.2293,-1.1894,-1.14905,-1.10826,-1.06702,-1.02531,-0.983147,-0.940513,-0.89741,-0.85384,-0.809803,-0.765304,-0.720349,-0.674945,-0.629103,-0.582835,-0.536159,-0.489092,-0.441656,-0.393875,-0.345777,-0.297393,-0.248756,-0.199906,-0.150881,-0.101726,-0.0524872,-0.0032153,0.0460374,0.0952155,0.144261,0.193115,0.241715,0.289998,0.3379,0.385355,0.432299,0.478665,0.52439,0.56941,0.613662,0.657085,0.699622,0.741216,0.781815,0.821369,0.859833,0.897165,0.933327,0.968287,1.00201,1.03449,1.06568,1.09559,1.12419,1.15149,1.17748,1.20217,1.22555,1.24765,1.26847,1.28803,1.30636,1.32346,1.33938,1.35413,1.36774,1.38024,1.39165,1.40202,1.41135,1.41969,1.42707,1.4335,1.43901,1.44363,1.44738,1.45027,1.45233,1.45357,1.454,1.45363,1.45247,1.45051,1.44777,1.44423,1.43989,1.43474,1.42877,1.42196,1.41429,1.40574,1.39627,1.38587,1.3745,1.36212,1.34869,1.33418,1.31853,1.3017,1.28364,1.26431,1.24365,1.2216,1.19811,1.17314,1.14662,1.11851,1.08877,1.05734,1.02419,0.989282,0.952595,0.914109,0.873816,0.831722,0.787841,0.742202,0.694849,0.645841,0.595252,0.543167,0.489689,0.434929,0.379007,0.322049,0.264184,0.205534,0.146216,0.0863367,0.0259867,-0.0347572,-0.0958379,-0.15721,-0.218842,-0.280709},
	{-3.30688,-3.27755,-3.2482,-3.21883,-3.18942,-3.15999,-3.13054,-3.10105,-3.07152,-3.04197,-3.01238,-2.98275,-2.95308,-2.92336,-2.89361,-2.8638,-2.83395,-2.80404,-2.77407,-2.74405,-2.71397,-2.68382,-2.6536,-2.62331,-2.59294,-2.56249,-2.53195,-2.50133,-2.47061,-2.43979,-2.40887,-2.37784,-2.34669,-2.31542,-2.28402,-2.25248,-2.22081,-2.18898,-2.157,-2.12486,-2.09254,-2.06004,-2.02735,-1.99447,-1.96138,-1.92807,-1.89454,-1.86076,-1.82675,-1.79247,-1.75792,-1.7231,-1.68798,-1.65257,-1.61683,-1.58077,-1.54438,-1.50763,-1.47052,-1.43304,-1.39518,-1.35691,-1.31825,-1.27916,-1.23965,-1.1997,-1.1593,-1.11845,-1.07714,-1.03536,-0.993108,-0.950381,-0.907177,-0.863494,-0.819334,-0.774699,-0.729596,-0.684032,-0.638015,-0.59156,-0.54468,-0.497395,-0.449724,-0.401691,-0.353325,-0.304655,-0.255715,-0.206541,-0.157176,-0.107661,-0.0580446,-0.00837585,0.0412921,0.0909036,0.140401,0.189722,0.238807,0.28759,0.336006,0.38399,0.431474,0.478392,0.524678,0.570266,0.615092,0.659095,0.702213,0.744389,0.78557,0.825703,0.864741,0.902641,0.939362,0.974871,1.00914,1.04213,1.07384,1.10424,1.13332,1.16108,1.18751,1.21262,1.2364,1.25887,1.28005,1.29995,1.31859,1.33598,1.35217,1.36716,1.38099,1.39369,1.40528,1.4158,1.42527,1.43373,1.44119,1.4477,1.45326,1.45791,1.46167,1.46456,1.4666,1.4678,1.46817,1.46772,1.46646,1.4644,1.46153,1.45784,1.45335,1.44802,1.44186,1.43484,1.42695,1.41816,1.40845,1.39778,1.38613,1.37345,1.35971,1.34487,1.32888,1.3117,1.29327,1.27355,1.25249,1.23003,1.20612,1.18072,1.15375,1.12519,1.09498,1.06308,1.02945,0.994053,0.956877,0.917901,0.87712,0.83454,0.790179,0.74407,0.696259,0.646809,0.595795,0.543305,0.489446,0.434329,0.378077,0.320815,0.262672,0.203769,0.144222,0.0841336,0.0235918,-0.0373295,-0.0985767,-0.160106,-0.221889,-0.283901},
	{-3.31707,-3.28775,-3.2584,-3.22902,-3.19962,-3.1702,-3.14074,-3.11126,-3.08174,-3.05218,-3.0226,-2.99297,-2.9633,-2.93359,-2.90384,-2.87404,-2.84419,-2.81429,-2.78433,-2.75431,-2.72423,-2.69409,-2.66387,-2.63359,-2.60323,-2.57278,-2.54226,-2.51164,-2.48093,-2.45012,-2.4192,-2.38818,-2.35704,-2.32577,-2.29438,-2.26286,-2.23119,-2.19938,-2.1674,-2.13527,-2.10296,-2.07047,-2.0378,-2.00492,-1.97184,-1.93854,-1.90501,-1.87125,-1.83723,-1.80296,-1.76842,-1.7336,-1.69849,-1.66307,-1.62734,-1.59128,-1.55487,-1.51812,-1.481,-1.44351,-1.40562,-1.36734,-1.32865,-1.28953,-1.24998,-1.20999,-1.16955,-1.12864,-1.08727,-1.04542,-1.00309,-0.960281,-0.916982,-0.873193,-0.828917,-0.784155,-0.738913,-0.693196,-0.647014,-0.600379,-0.553304,-0.505809,-0.457912,-0.409638,-0.361012,-0.312065,-0.26283,-0.213344,-0.163646,-0.113782,-0.063796,-0.0137395,0.0363349,0.0863712,0.136311,0.186093,0.235654,0.28493,0.333854,0.38236,0.430378,0.477842,0.524684,0.570836,0.616233,0.660812,0.70451,0.747267,0.789028,0.829739,0.869352,0.90782,0.945102,0.981163,1.01597,1.04949,1.08171,1.11261,1.14218,1.1704,1.19728,1.22281,1.247,1.26986,1.2914,1.31164,1.3306,1.3483,1.36476,1.38001,1.39407,1.40698,1.41877,1.42945,1.43908,1.44766,1.45523,1.46182,1.46746,1.47216,1.47596,1.47887,1.4809,1.48208,1.48242,1.48193,1.48061,1.47846,1.4755,1.47172,1.4671,1.46164,1.45534,1.44816,1.4401,1.43113,1.42122,1.41035,1.39848,1.38557,1.3716,1.3565,1.34026,1.3228,1.3041,1.28409,1.26273,1.23997,1.21575,1.19001,1.16272,1.13382,1.10327,1.07102,1.03703,1.00129,0.963752,0.924418,0.88328,0.840345,0.795635,0.749183,0.701037,0.651261,0.599934,0.547145,0.493001,0.437616,0.381112,0.323617,0.265256,0.206152,0.146419,0.0861573,0.0254536,-0.0356205,-0.0970131,-0.158682,-0.2206,-0.282743},
	{-3.32716,-3.29784,-3.26849,-3.23912,-3.20973,-3.1803,-3.15085,-3.12137,-3.09185,-3.0623,-3.03272,-3.00309,-2.97343,-2.94373,-2.91398,-2.88418,-2.85433,-2.82444,-2.79448,-2.76447,-2.7344,-2.70426,-2.67405,-2.64377,-2.61342,-2.58298,-2.55246,-2.52185,-2.49115,-2.46035,-2.42944,-2.39843,-2.3673,-2.33604,-2.30466,-2.27315,-2.24149,-2.20968,-2.17772,-2.1456,-2.1133,-2.08082,-2.04816,-2.01529,-1.98222,-1.94893,-1.91541,-1.88166,-1.84766,-1.81339,-1.77886,-1.74404,-1.70894,-1.67352,-1.63779,-1.60173,-1.56533,-1.52857,-1.49144,-1.45394,-1.41604,-1.37774,-1.33903,-1.29989,-1.26031,-1.22028,-1.1798,-1.13884,-1.09742,-1.05551,-1.01311,-0.970221,-0.926834,-0.882949,-0.838565,-0.793685,-0.748312,-0.702452,-0.656114,-0.609309,-0.562051,-0.514356,-0.466244,-0.417738,-0.368865,-0.319653,-0.270135,-0.220348,-0.170331,-0.120128,-0.0697858,-0.0193538,0.0311145,0.0815631,0.131933,0.182163,0.232189,0.281946,0.331366,0.380381,0.428923,0.476921,0.524308,0.571014,0.616972,0.662117,0.706384,0.749713,0.792045,0.833326,0.873504,0.912533,0.950368,0.986972,1.02231,1.05636,1.08908,1.12047,1.15051,1.17919,1.20651,1.23246,1.25705,1.28029,1.30218,1.32276,1.34203,1.36001,1.37674,1.39223,1.40651,1.41962,1.43158,1.44242,1.45217,1.46087,1.46853,1.47519,1.48087,1.48561,1.48941,1.49231,1.49431,1.49545,1.49572,1.49515,1.49373,1.49147,1.48837,1.48444,1.47966,1.47402,1.46751,1.46013,1.45184,1.44262,1.43245,1.42131,1.40915,1.39594,1.38164,1.36621,1.34961,1.3318,1.31271,1.29231,1.27054,1.24735,1.2227,1.19651,1.16876,1.13939,1.10835,1.07561,1.04113,1.00488,0.966836,0.926993,0.885347,0.841909,0.796702,0.749763,0.701143,0.650911,0.599146,0.545943,0.491409,0.435661,0.378823,0.32102,0.262381,0.203024,0.143062,0.0825923,0.0216979,-0.0395528,-0.101112,-0.162939,-0.225008,-0.287296},
	{-3.33716,-3.30784,-3.27849,-3.24912,-3.21973,-3.19031,-3.16086,-3.13138,-3.10187,-3.07232,-3.04274,-3.01312,-2.98346,-2.95376,-2.92402,-2.89423,-2.86439,-2.83449,-2.80454,-2.77454,-2.74447,-2.71434,-2.68414,-2.65387,-2.62352,-2.59309,-2.56258,-2.53198,-2.50129,-2.47049,-2.4396,-2.40859,-2.37747,-2.34622,-2.31485,-2.28335,-2.2517,-2.21991,-2.18796,-2.15585,-2.12356,-2.0911,-2.05844,-2.02559,-1.99253,-1.95925,-1.92575,-1.892,-1.85801,-1.82376,-1.78924,-1.75443,-1.71933,-1.68392,-1.6482,-1.61214,-1.57574,-1.53898,-1.50185,-1.46434,-1.42644,-1.38813,-1.3494,-1.31023,-1.27063,-1.23058,-1.19006,-1.14906,-1.10759,-1.06562,-1.02316,-0.980203,-0.936736,-0.892761,-0.848278,-0.803287,-0.757792,-0.711797,-0.665312,-0.618347,-0.570913,-0.523029,-0.474711,-0.425984,-0.376872,-0.327405,-0.277614,-0.227535,-0.177209,-0.126678,-0.0759886,-0.0251909,0.025662,0.0765138,0.127305,0.177974,0.228457,0.278686,0.328595,0.378114,0.427172,0.475699,0.523625,0.57088,0.617395,0.663103,0.707937,0.751836,0.794739,0.836589,0.877333,0.916923,0.955313,0.992464,1.02834,1.06291,1.09615,1.12803,1.15856,1.1877,1.21546,1.24184,1.26683,1.29046,1.31272,1.33364,1.35323,1.37152,1.38852,1.40427,1.41879,1.43211,1.44426,1.45528,1.46518,1.47401,1.48178,1.48853,1.49429,1.49908,1.50292,1.50584,1.50785,1.50898,1.50922,1.5086,1.50713,1.5048,1.50162,1.49759,1.49269,1.48694,1.4803,1.47276,1.46431,1.45493,1.44458,1.43324,1.42088,1.40746,1.39294,1.37728,1.36044,1.34237,1.32302,1.30235,1.2803,1.25683,1.23187,1.20539,1.17733,1.14764,1.11628,1.08321,1.04839,1.0118,0.973424,0.933242,0.891259,0.847485,0.801946,0.754681,0.705744,0.655203,0.603142,0.549654,0.494851,0.438849,0.381773,0.323749,0.264904,0.205356,0.145217,0.0845818,0.0235319,-0.037866,-0.0995658,-0.161528,-0.22373,-0.286147},
	{-3.34705,-3.31773,-3.28839,-3.25903,-3.22963,-3.20022,-3.17077,-3.14129,-3.11179,-3.08224,-3.05267,-3.02305,-2.9934,-2.9637,-2.93396,-2.90418,-2.87434,-2.84445,-2.81451,-2.78451,-2.75445,-2.72433,-2.69413,-2.66387,-2.63353,-2.60311,-2.57261,-2.54202,-2.51133,-2.48055,-2.44966,-2.41866,-2.38755,-2.35632,-2.32496,-2.29347,-2.26184,-2.23005,-2.19812,-2.16602,-2.13375,-2.10129,-2.06865,-2.03582,-2.00277,-1.96951,-1.93601,-1.90228,-1.86831,-1.83407,-1.79956,-1.76476,-1.72967,-1.69428,-1.65856,-1.62251,-1.58611,-1.54936,-1.51223,-1.47472,-1.43681,-1.39849,-1.35975,-1.32058,-1.28096,-1.24088,-1.20033,-1.1593,-1.11778,-1.07577,-1.03326,-0.990233,-0.946695,-0.902639,-0.858064,-0.812971,-0.767363,-0.721244,-0.674622,-0.627505,-0.579907,-0.531843,-0.483331,-0.434393,-0.385054,-0.335343,-0.28529,-0.234932,-0.184308,-0.133461,-0.0824362,-0.0312844,0.0199413,0.0711845,0.122386,0.173483,0.22441,0.275102,0.325489,0.3755,0.425065,0.474112,0.522569,0.570364,0.617428,0.66369,0.709085,0.753547,0.797014,0.839428,0.880733,0.92088,0.959821,0.997514,1.03392,1.06902,1.10276,1.13515,1.16615,1.19575,1.22396,1.25076,1.27617,1.30018,1.32281,1.34407,1.36399,1.38257,1.39986,1.41587,1.43062,1.44416,1.45651,1.46769,1.47775,1.48671,1.4946,1.50144,1.50728,1.51213,1.51601,1.51895,1.52097,1.52209,1.52231,1.52166,1.52013,1.51773,1.51448,1.51035,1.50535,1.49948,1.49271,1.48504,1.47644,1.46689,1.45637,1.44485,1.43229,1.41867,1.40393,1.38805,1.37098,1.35267,1.33307,1.31214,1.28982,1.26607,1.24083,1.21406,1.18569,1.1557,1.12403,1.09064,1.0555,1.01859,0.979879,0.939368,0.897055,0.852955,0.807093,0.759511,0.710263,0.659422,0.607071,0.553306,0.498241,0.441991,0.384682,0.326442,0.267396,0.207661,0.147348,0.0865513,0.0253486,-0.0361944,-0.0980334,-0.16013,-0.222463,-0.285006},
	{-3.35685,-3.32754,-3.2982,-3.26884,-3.23945,-3.21003,-3.18059,-3.15112,-3.12161,-3.09207,-3.0625,-3.03289,-3.00324,-2.97355,-2.94382,-2.91404,-2.88421,-2.85432,-2.82439,-2.7944,-2.76434,-2.73422,-2.70404,-2.67378,-2.64345,-2.61304,-2.58255,-2.55196,-2.52129,-2.49051,-2.45964,-2.42865,-2.39755,-2.36633,-2.33499,-2.30351,-2.27189,-2.24012,-2.2082,-2.17611,-2.14385,-2.11142,-2.07879,-2.04597,-2.01294,-1.97969,-1.94622,-1.9125,-1.87854,-1.84432,-1.80982,-1.77504,-1.73997,-1.70458,-1.66888,-1.63284,-1.59645,-1.5597,-1.52258,-1.48508,-1.44717,-1.40885,-1.3701,-1.33092,-1.29128,-1.25119,-1.21062,-1.16956,-1.12801,-1.08596,-1.0434,-1.00032,-0.956715,-0.912587,-0.867931,-0.822746,-0.777035,-0.730801,-0.684052,-0.636795,-0.589043,-0.540811,-0.492117,-0.44298,-0.393426,-0.343483,-0.293181,-0.242557,-0.191648,-0.140497,-0.0891509,-0.0376585,0.0139266,0.0655479,0.117146,0.168657,0.220017,0.271157,0.322009,0.372501,0.42256,0.472114,0.52109,0.569415,0.617016,0.663823,0.709768,0.754784,0.798806,0.841775,0.883634,0.924329,0.963814,1.00204,1.03898,1.07459,1.10884,1.14171,1.17318,1.20325,1.23189,1.25912,1.28492,1.30932,1.33231,1.35391,1.37414,1.39303,1.41059,1.42685,1.44184,1.45558,1.46812,1.47947,1.48967,1.49876,1.50675,1.51369,1.51959,1.52448,1.5284,1.53135,1.53337,1.53447,1.53465,1.53395,1.53235,1.52988,1.52652,1.52229,1.51717,1.51116,1.50424,1.4964,1.48763,1.47789,1.46717,1.45544,1.44266,1.4288,1.41382,1.39767,1.38033,1.36174,1.34185,1.32061,1.29798,1.2739,1.24832,1.2212,1.19248,1.16212,1.13007,1.0963,1.06077,1.02347,0.984362,0.943453,0.900743,0.856248,0.809997,0.762033,0.712414,0.661213,0.608518,0.554426,0.499051,0.442513,0.384936,0.326449,0.267176,0.207234,0.14673,0.085757,0.02439,-0.0373077,-0.0992944,-0.161533,-0.224003,-0.286679},
	{-3.36656,-3.33725,-3.30791,-3.27855,-3.24917,-3.21976,-3.19032,-3.16085,-3.13135,-3.10181,-3.07224,-3.04264,-3.01299,-2.98331,-2.95358,-2.9238,-2.89398,-2.86411,-2.83418,-2.80419,-2.77414,-2.74403,-2.71386,-2.68361,-2.65329,-2.62288,-2.5924,-2.56183,-2.53116,-2.5004,-2.46953,-2.43856,-2.40747,-2.37627,-2.34493,-2.31347,-2.28186,-2.25011,-2.2182,-2.18613,-2.15389,-2.12147,-2.08886,-2.05606,-2.02304,-1.98981,-1.95636,-1.92266,-1.88872,-1.85451,-1.82003,-1.78527,-1.75021,-1.71485,-1.67916,-1.64313,-1.60676,-1.57002,-1.53291,-1.49541,-1.45751,-1.4192,-1.38045,-1.34126,-1.30162,-1.26151,-1.22093,-1.17985,-1.13827,-1.09619,-1.05359,-1.01046,-0.966803,-0.922611,-0.877881,-0.832614,-0.786809,-0.74047,-0.693603,-0.646217,-0.598322,-0.549933,-0.501066,-0.451742,-0.401984,-0.35182,-0.301281,-0.250401,-0.199219,-0.147777,-0.0961196,-0.0442979,0.00763555,0.0596239,0.111607,0.163523,0.215305,0.266884,0.318192,0.369155,0.419701,0.469754,0.519242,0.568089,0.616223,0.66357,0.71006,0.755626,0.800201,0.843723,0.886133,0.927378,0.967406,1.00617,1.04364,1.07976,1.11452,1.14789,1.17984,1.21037,1.23946,1.26712,1.29334,1.31812,1.34148,1.36344,1.384,1.4032,1.42105,1.43758,1.45281,1.46679,1.47953,1.49107,1.50144,1.51067,1.5188,1.52584,1.53184,1.53682,1.54079,1.5438,1.54585,1.54696,1.54715,1.54644,1.54483,1.54232,1.53892,1.53463,1.52945,1.52336,1.51636,1.50843,1.49956,1.48971,1.47888,1.46702,1.45411,1.44011,1.42498,1.40869,1.39118,1.37242,1.35236,1.33095,1.30813,1.28386,1.25809,1.23076,1.20183,1.17126,1.13899,1.105,1.06925,1.03172,0.992386,0.951246,0.908305,0.863578,0.817096,0.768904,0.719061,0.667641,0.614733,0.560435,0.504862,0.448135,0.390379,0.33172,0.272286,0.21219,0.151541,0.0904285,0.028928,-0.0328988,-0.0950109,-0.157372,-0.219962,-0.282756},
	{-3.37618,-3.34687,-3.31754,-3.28818,-3.2588,-3.22939,-3.19995,-3.17049,-3.14099,-3.11146,-3.0819,-3.0523,-3.02266,-2.99298,-2.96325,-2.93348,-2.90367,-2.8738,-2.84388,-2.8139,-2.78386,-2.75376,-2.72359,-2.69335,-2.66303,-2.63264,-2.60217,-2.57161,-2.54095,-2.5102,-2.47935,-2.44839,-2.41731,-2.38612,-2.3548,-2.32335,-2.29176,-2.26002,-2.22813,-2.19608,-2.16386,-2.13145,-2.09886,-2.06608,-2.03308,-1.99987,-1.96644,-1.93276,-1.89884,-1.86465,-1.8302,-1.79545,-1.76042,-1.72507,-1.6894,-1.65339,-1.61703,-1.58032,-1.54322,-1.50573,-1.46784,-1.42954,-1.3908,-1.35161,-1.31197,-1.27186,-1.23126,-1.19017,-1.14858,-1.10647,-1.06383,-1.02067,-0.976964,-0.932718,-0.887925,-0.842584,-0.796696,-0.750264,-0.703291,-0.655787,-0.607761,-0.559226,-0.5102,-0.460701,-0.410753,-0.360382,-0.309619,-0.258498,-0.207057,-0.155337,-0.103384,-0.051248,0.00101888,0.0533595,0.105714,0.158019,0.210207,0.262212,0.313961,0.365381,0.416399,0.466939,0.516925,0.566282,0.614935,0.662809,0.709834,0.755938,0.801055,0.84512,0.888073,0.929857,0.97042,1.00972,1.0477,1.08434,1.1196,1.15345,1.18588,1.21686,1.2464,1.27448,1.3011,1.32627,1.35,1.3723,1.39319,1.41269,1.43082,1.44762,1.46309,1.47729,1.49023,1.50195,1.51249,1.52186,1.53011,1.53726,1.54335,1.54839,1.55243,1.55547,1.55754,1.55866,1.55885,1.55812,1.55647,1.55392,1.55046,1.5461,1.54084,1.53466,1.52756,1.51952,1.51052,1.50055,1.48957,1.47756,1.46448,1.45031,1.43501,1.41853,1.40083,1.38186,1.36159,1.33995,1.3169,1.29239,1.26637,1.23879,1.2096,1.17875,1.14621,1.11194,1.0759,1.03807,0.99844,0.957,0.913758,0.868732,0.821954,0.77347,0.723341,0.671643,0.618467,0.563912,0.508096,0.451139,0.393167,0.334307,0.274685,0.214415,0.153602,0.0923369,0.0306911,-0.0312745,-0.0935207,-0.156012,-0.218729,-0.281646},
	{-3.38571,-3.3564,-3.32707,-3.29772,-3.26834,-3.23893,-3.2095,-3.18004,-3.15055,-3.12102,-3.09146,-3.06187,-3.03223,-3.00256,-2.97284,-2.94308,-2.91327,-2.8834,-2.85349,-2.82352,-2.79349,-2.76339,-2.73323,-2.703,-2.6727,-2.64232,-2.61185,-2.5813,-2.55066,-2.51992,-2.48908,-2.45813,-2.42707,-2.3959,-2.36459,-2.33316,-2.30158,-2.26986,-2.23799,-2.20595,-2.17375,-2.14137,-2.1088,-2.07603,-2.04306,-2.00987,-1.97646,-1.9428,-1.9089,-1.87474,-1.84031,-1.80559,-1.77057,-1.73525,-1.6996,-1.66362,-1.62728,-1.59058,-1.55351,-1.51604,-1.47817,-1.43987,-1.40114,-1.36197,-1.32233,-1.28222,-1.24162,-1.20053,-1.15892,-1.1168,-1.07414,-1.03094,-0.987204,-0.942913,-0.898068,-0.852665,-0.806705,-0.760189,-0.713123,-0.665513,-0.617368,-0.568702,-0.519529,-0.469869,-0.419745,-0.369182,-0.31821,-0.266863,-0.215178,-0.163197,-0.110964,-0.058529,-0.00594485,0.0467317,0.0994405,0.152118,0.204698,0.257111,0.309285,0.361146,0.41262,0.46363,0.5141,0.563951,0.613109,0.661497,0.709042,0.755671,0.801317,0.845912,0.889394,0.931706,0.972793,1.01261,1.0511,1.08824,1.12399,1.15832,1.19121,1.22265,1.25261,1.28111,1.30813,1.33367,1.35776,1.3804,1.40161,1.4214,1.43981,1.45685,1.47257,1.48697,1.50011,1.512,1.52268,1.53219,1.54055,1.54779,1.55395,1.55905,1.56312,1.56618,1.56826,1.56937,1.56954,1.56876,1.56706,1.56444,1.5609,1.55645,1.55107,1.54477,1.53754,1.52935,1.52019,1.51005,1.49888,1.48668,1.4734,1.45901,1.44347,1.42674,1.40879,1.38955,1.36899,1.34706,1.32371,1.29888,1.27253,1.2446,1.21506,1.18385,1.15093,1.11627,1.07984,1.04162,1.00158,0.959738,0.916092,0.870663,0.823487,0.774613,0.724104,0.672041,0.618514,0.563627,0.507499,0.450252,0.392011,0.332905,0.273059,0.212583,0.151582,0.0901431,0.0283351,-0.0337836,-0.0961764,-0.158809,-0.221663,-0.284712},
	{-3.39515,-3.36584,-3.33652,-3.30717,-3.27779,-3.24839,-3.21896,-3.1895,-3.16002,-3.1305,-3.10094,-3.07135,-3.04172,-3.01205,-2.98234,-2.95258,-2.92278,-2.89293,-2.86302,-2.83306,-2.80303,-2.77295,-2.7428,-2.71258,-2.68228,-2.65191,-2.62146,-2.59092,-2.56029,-2.52956,-2.49874,-2.4678,-2.43676,-2.4056,-2.37431,-2.34289,-2.31133,-2.27963,-2.24778,-2.21576,-2.18358,-2.15122,-2.11867,-2.08593,-2.05298,-2.01981,-1.98642,-1.9528,-1.91892,-1.88478,-1.85038,-1.81569,-1.7807,-1.7454,-1.70978,-1.67382,-1.63751,-1.60084,-1.56378,-1.52634,-1.48849,-1.45021,-1.4115,-1.37234,-1.33272,-1.29262,-1.25202,-1.21093,-1.16932,-1.12718,-1.08451,-1.0413,-0.99753,-0.953204,-0.908315,-0.86286,-0.816838,-0.770251,-0.723102,-0.675397,-0.627145,-0.578358,-0.529052,-0.479243,-0.428955,-0.378212,-0.327044,-0.275484,-0.223567,-0.171337,-0.118837,-0.0661154,-0.0132264,0.039774,0.0928255,0.145864,0.198824,0.251634,0.304223,0.356516,0.408437,0.459909,0.510854,0.561194,0.61085,0.659746,0.707806,0.754958,0.801129,0.846253,0.890266,0.933106,0.974719,1.01505,1.05406,1.09171,1.12795,1.16277,1.19613,1.22803,1.25844,1.28736,1.31479,1.34073,1.36519,1.38819,1.40973,1.42984,1.44855,1.46587,1.48184,1.49649,1.50984,1.52194,1.53281,1.54248,1.55099,1.55837,1.56464,1.56985,1.57401,1.57715,1.57929,1.58045,1.58065,1.5799,1.57821,1.5756,1.57206,1.56759,1.5622,1.55587,1.5486,1.54037,1.53117,1.52097,1.50974,1.49747,1.48412,1.46965,1.45403,1.43722,1.41917,1.39984,1.37918,1.35714,1.33367,1.30873,1.28225,1.2542,1.22452,1.19317,1.16011,1.12531,1.08872,1.05034,1.01015,0.968141,0.924329,0.878733,0.831389,0.782345,0.731669,0.679438,0.625747,0.5707,0.514415,0.457015,0.398628,0.339379,0.279396,0.218788,0.157659,0.0960955,0.0341663,-0.0280709,-0.0905801,-0.153327,-0.216294,-0.279454},
	{-3.4045,-3.3752,-3.34588,-3.31653,-3.28716,-3.25776,-3.22834,-3.19888,-3.1694,-3.13989,-3.11034,-3.08075,-3.05113,-3.02146,-2.99176,-2.96201,-2.93221,-2.90236,-2.87246,-2.84251,-2.8125,-2.78242,-2.75228,-2.72207,-2.69178,-2.66142,-2.63098,-2.60046,-2.56984,-2.53913,-2.50832,-2.4774,-2.44637,-2.41522,-2.38395,-2.35255,-2.32101,-2.28933,-2.2575,-2.2255,-2.19334,-2.161,-2.12848,-2.09576,-2.06283,-2.0297,-1.99633,-1.96273,-1.92888,-1.89478,-1.8604,-1.82574,-1.79078,-1.75551,-1.71992,-1.68399,-1.64771,-1.61107,-1.57404,-1.53663,-1.4988,-1.46055,-1.42186,-1.38273,-1.34312,-1.30303,-1.26246,-1.22137,-1.17977,-1.13763,-1.09495,-1.05173,-1.00794,-0.963594,-0.918673,-0.873177,-0.827106,-0.78046,-0.733241,-0.685456,-0.637111,-0.58822,-0.538794,-0.488854,-0.438418,-0.387512,-0.336165,-0.284409,-0.232281,-0.17982,-0.127072,-0.074084,-0.02091,0.0323938,0.0857674,0.139147,0.192465,0.245652,0.298634,0.351337,0.403684,0.455597,0.506996,0.557802,0.607936,0.657319,0.705874,0.753527,0.800204,0.845837,0.890358,0.933707,0.975826,1.01666,1.05617,1.0943,1.13102,1.1663,1.20012,1.23246,1.26329,1.29262,1.32044,1.34675,1.37157,1.3949,1.41676,1.43716,1.45614,1.47372,1.48992,1.50478,1.51833,1.5306,1.54162,1.55143,1.56005,1.56753,1.57389,1.57915,1.58336,1.58653,1.58868,1.58984,1.59002,1.58925,1.58752,1.58484,1.58123,1.57668,1.57119,1.56475,1.55736,1.549,1.53964,1.52928,1.51789,1.50543,1.49188,1.4772,1.46136,1.4443,1.426,1.40641,1.38547,1.36314,1.33936,1.3141,1.28729,1.25889,1.22885,1.19713,1.16368,1.12848,1.09149,1.0527,1.01209,0.969659,0.925425,0.87941,0.831651,0.782201,0.731129,0.678519,0.624464,0.569074,0.512468,0.454771,0.39611,0.336611,0.2764,0.215585,0.154267,0.0925296,0.0304376,-0.0319538,-0.0946098,-0.1575,-0.220604,-0.283896}
};

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
   
const G4double G4AtimaEnergyLossModel::ionisation_potentials_z[121] = {
    9999999,  // 0   
    19.2, // H
    41.8, // He
    40.0,
    63.7,
    76.0,
    78.0, // C
    82.0,
    95.0,
    115.0,
    137.0,
    149.0,
    156.0,
    166.0,
    173.0,
    173.0,
    180.0,
    174.0,
    188.0,
    190.0,
    191.0, //Ca (20)
    216.0,
    233.0,
    245.0,
    257.0,
    272.0,
    286.0, //Fe 26
    297.0,
    311.0,
    322.0,
    330.0,
    334.0,
    350.0,
    347.0,
    348.0,
    343.0,
    352.0,
    363.0,
    366.0,
    379.0,
    393.0,
    417.0,
    424.0,
    428.0,
    441.0,
    449.0,
    470.0,
    470.0,
    469.0,
    488.0,
    488.0, //Sn 50
    487.0,
    485.0,
    491.0,
    482.0, // Xe 54
    488.0,
    491.0,
    501.0,
    523.0,
    535.0,
    546.0,
    560.0,
    574.0,
    580.0,
    591.0,
    614.0,
    628.0,
    650.0,
    658.0,
    674.0,
    684.0,
    694.0,
    705.0,
    718.0,
    727.0,
    736.0,
    746.0,
    757.0,
    790.0,
    790.0, // Au 79
    800.0,
    810.0,
    823.0, //Pb 82
    823.0,
    830.0,
    825.0,
    794.0,
    827.0,
    826.0,
    841.0,
    847.0,
    878.0,
    890.0, // U 92
      900.0,
      910.0,
      920.0,
      930.0,
      940.0,
      950.0,
      960.0,
     970.0,
     980.0,
     990.0,
    1000.0,
    1010.0,
    1020.0,
    1030.0,
    1040.0,
    1050.0,
    1060.0,
    1070.0,
    1080.0,
    1090.0,
    1100.0,
    1110.0,
    1120.0,
    1130.0,
    1140.0,
    1150.0,
    1160.0,
    1170.0,
    };

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

const G4double G4AtimaEnergyLossModel::x0[92]= {
1.8639, 2.2017,  0.1304, 0.0592, 0.0305, -.0178, 1.7378, 1.7541, 1.8433, 2.0735,
0.2880, 0.1499,  0.1708, 0.2014, 0.1696, 0.1580, 1.5555, 1.7635, 0.3851, 0.3228,
0.1640, 0.0957,  0.0691, 0.0340, 0.0447, -.0012, -.0187, -.0566, -.0254, 0.0049,
0.2267, 0.3376,  0.1767, 0.2258, 1.5262, 1.7158, 0.5737, 0.4585, 0.3608, 0.2957,
0.1785, 0.2267,  0.0949, 0.0599, 0.0576, 0.0563, 0.0657, 0.1281, 0.2406, 0.2879,
0.3189, 0.3296,  0.0549, 1.5630, 0.5473, 0.4190, 0.3161, 0.2713, 0.2333, 0.1984,
0.1627, 0.1520,  0.1888, 0.1058, 0.0947, 0.0822, 0.0761, 0.0648, 0.0812, 0.1199,
0.1560, 0.1965,  0.2117, 0.2167, 0.0559, 0.0891, 0.0819, 0.1484, 0.2021, 0.2756,
0.3491, 0.3776,  0.4152, 0.4267, 0.4300, 1.5368, 0.6000, 0.5991, 0.4559, 0.4202,
0.3144, 0.2260 };

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

const G4double G4AtimaEnergyLossModel::x1[92]= {
3.2718, 3.6122,  1.6397, 1.6922, 1.9688, 2.3415, 4.1323, 4.3213, 4.4096, 4.6421,
3.1962, 3.0668,  3.0127, 2.8715, 2.7815, 2.7159, 4.2994, 4.4855, 3.1724, 3.1191,
3.0593, 3.0386,  3.0322, 3.0451, 3.1074, 3.1531, 3.1790, 3.1851, 3.2792, 3.3668,
3.5434, 3.6096,  3.5702, 3.6264, 4.9899, 5.0748, 3.7995, 3.6778, 3.5542, 3.4890,
3.2201, 3.2784,  3.1253, 3.0834, 3.1069, 3.0555, 3.1074, 3.1667, 3.2032, 3.2959,
3.3489, 3.4418,  3.2596, 4.7371, 3.5914, 3.4547, 3.3293, 3.3432, 3.2773, 3.3063,
3.3199, 3.3460,  3.4633, 3.3932, 3.4224, 3.4474, 3.4782, 3.4922, 3.5085, 3.6246,
3.5218, 3.4337,  3.4805, 3.4960, 3.4845, 3.5414, 3.5480, 3.6212, 3.6979, 3.7275,
3.8044, 3.8073,  3.8248, 3.8293, 4.0000, 4.9889, 4.0000, 3.9428, 3.7966, 3.7681,
3.5079, 3.3721 };

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

const G4double G4AtimaEnergyLossModel::afermi[92]= {
 0.14092, 0.13443,  0.95136, 0.80392, 0.56224, 0.26142, 0.15349, 0.11778, 0.11083, .08064,
 0.07772, 0.08163,  0.08024, 0.14921, 0.23610, 0.33992, 0.19849, 0.19714, 0.19827, .15643,
 0.15754, 0.15662,  0.15436, 0.15419, 0.14973, 0.14680, 0.14474, 0.16496, 0.14339, .14714,
 0.09440, 0.07188,  0.06633, 0.06568, 0.06335, 0.07446, 0.07261, 0.07165, 0.07138, .07177,
 0.13883, 0.10525,  0.16572, 0.19342, 0.19205, 0.24178, 0.24585, 0.24609, 0.23879, .18689,
 0.16652, 0.13815,  0.23766, 0.23314, 0.18233, 0.18268, 0.18591, 0.18885, 0.23265, .23530,
 0.24280, 0.24698,  0.24448, 0.25109, 0.24453, 0.24665, 0.24638, 0.24823, 0.24889, .25295,
 0.24033, 0.22918,  0.17798, 0.15509, 0.15184, 0.12751, 0.12690, 0.11128, 0.09756, .11014,
 0.09455, 0.09359,  0.09410, 0.09282, 0.09000, 0.20798, 0.08000, 0.08804, 0.08567, 0.08655,
 0.14770,  0.19677 };

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

const G4double G4AtimaEnergyLossModel::c[92]= {
9.5835, 11.1393, 3.1221, 2.7847, 2.8477, 2.8680, 10.5400, 10.7004, 10.9653, 11.9041,
5.0526, 4.5297,  4.2395, 4.4351, 4.5214, 4.6659, 11.1421, 11.9480, 5.6423, 5.0396,
4.6949, 4.4450,  4.2659, 4.1781, 4.2702, 4.2911, 4.2601, 4.3115, 4.4190, 4.6906,
4.9353, 5.1411,  5.0510, 5.3210, 11.7307,12.5115,6.4776, 5.9867, 5.4801 , 5.1774,
5.0141, 4.8793,  4.7769, 4.7694, 4.8008, 4.9358, 5.0630, 5.2727, 5.5211, 5.5340,
5.6241, 5.7131,  5.9488, 12.7281, 6.9135, 6.3153, 5.7850, 5.7837, 5.8096, 5.8290,
5.8224, 5.8597,  6.2278, 5.8738, 5.9045, 5.9183, 5.9587, 5.9521, 5.9677, 6.3325,
5.9785, 5.7139,  5.5262, 5.4059, 5.3445, 5.3083, 5.3418, 5.4732, 5.5747, 5.9605,
6.1365, 6.2018,  6.3505, 6.4003, 6.4, 13.2839, 7., 7.0452, 6.3742, 6.2473,
6.0327, 5.8694 };

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

const G4double G4AtimaEnergyLossModel::m0[92]= {
5.7273, 5.8347,  2.4993, 2.4339, 2.4512, 2.8697, 3.2125, 3.2913, 3.2962, 3.5771,
3.6452, 3.6166,  3.6345, 3.2546, 2.9158, 2.6456, 2.9702, 2.9618, 2.9233, 3.0745,
3.0517, 3.0302,  3.0163, 2.9896, 2.9796, 2.9632, 2.9502, 2.8430, 2.9044, 2.8652,
3.1314, 3.3306,  3.4176, 3.4317, 3.4670, 3.4051, 3.4177, 3.4435, 3.4585, 3.4533,
3.0930, 3.2549,  2.9738, 2.8707, 2.8633, 2.7239, 2.6899, 2.6772, 2.7144, 2.8576,
2.9319, 3.0354,  2.7276, 2.7414, 2.8866, 2.8906, 2.8828, 2.8592, 2.7331, 2.7050,
2.6674, 2.6403,  2.6245, 2.5977, 2.6056, 2.5849, 2.5726, 2.5573, 2.5469, 2.5141,
2.5643, 2.6155,  2.7623, 2.8447, 2.8627, 2.9608, 2.9658, 3.0417, 3.1101, 3.0519,
3.1450, 3.1608,  3.1671, 3.1830, 1.1111, 2.7409, 1.1111, 3.2454, 3.2683, 3.2610,
2.9845, 2.8171 };

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

const G4double G4AtimaEnergyLossModel::del_0[92]= {
0.0,  0.00,  0.14, 0.14, 0.14, 0.12, 0.00, 0.00, 0.00, 0.00,
0.08, 0.08,  0.12, 0.14, 0.14, 0.14, 0.00, 0.00, 0.10, 0.14,
0.10, 0.12,  0.14, 0.14, 0.14, 0.12, 0.12, 0.10, 0.08, 0.08,
0.14, 0.14,  0.08, 0.10, 0.00, 0.00, 0.14, 0.14, 0.14, 0.14,
0.14, 0.14,  0.14, 0.14, 0.14, 0.14, 0.14, 0.14, 0.14, 0.14,
0.14, 0.14,  0.00, 0.00, 0.14, 0.14, 0.14, 0.14, 0.14, 0.14,
0.14, 0.14,  0.14, 0.14, 0.14, 0.14, 0.14, 0.14, 0.14, 0.14,
0.14, 0.14,  0.14, 0.14, 0.08, 0.10, 0.10, 0.12, 0.14, 0.14,
0.14, 0.14,  0.14, 0.14, 0.00, 0.00, 0.00, 0.14, 0.14, 0.14,
0.14, 0.14 };

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
