/* vim: set tabstop=4: */
/*
 * This file is part of TraceProto.
 * Copyright 2004-2005 Eric Hope and others; see the AUTHORS file for details.
 *
 * TraceProto is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * TraceProto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TraceProto; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <libnet.h>
#include <pcap.h>
#include <unistd.h>
#include <errno.h>
#include <sys/time.h>
#include <stdio.h>
#include <sys/types.h>
#include <arpa/inet.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <ctype.h>
#include <time.h>
#include <assert.h>

#include "tp_miscfunc.h"
#include "traceproto.h"
#include "config.h"
#include "tp_as.h"

#ifdef HAVE_LIBCAP
# include <sys/capability.h>
#endif /* HAVE_LIBCAP */

/*
#ifdef HAVE_LIBCURSES
#include <curses.h>
#endif / * HAVE_LIBCURSES */

#ifdef HAVE_LIBDMALLOC
#include <dmalloc.h>
#endif /* HAVE_LIBDMALLOC */

/*
 * take a comma separated list (sans spaces) and give them a miss
 * (ie don't send probes with the ttl set to these numbers)
 */
int parse_skips ( char * skip_ptr )
{
	size_t c;
	char tmp[ 4 ];
	int tmp_c = 0;

	behavior.skips = ( int * ) calloc ( behavior.max_ttl + 1, sizeof ( int ) );
	if ( behavior.skips == NULL )
	{
		printf ( "memory allocation error" );
		tixe ( tixe_cleanup, 1 );
	}

	tixe_cleanup.skips_free = YES;

	/*
	 * loop through the str from the cmd line
	 * if its a digit remember it
	 * if its a comma, add it to the list of skipped hops
	 */
	memset ( & tmp, '\0', sizeof ( tmp ) );
	for ( c = 0; c < strlen ( skip_ptr ); c++ )
	{
		if ( isdigit ( skip_ptr[ c ] ) && tmp_c < 4 )
		{
			tmp[ tmp_c++ ] = skip_ptr[ c ];
		}
		else if ( isdigit ( skip_ptr[ c ] ) && tmp_c >= 4 )
		{
			return 1;
		}
		else if ( skip_ptr[ c ] == ',' )
		{
			tmp_c = atoi ( tmp );
			if ( tmp_c > 0 && tmp_c <= behavior.max_ttl )
			{
				behavior.skips[ tmp_c ] = 1;
			}
			else
			{
				return 1;
			}
			tmp_c = 0;
			memset ( & tmp, '\0', sizeof ( tmp ) );
		}
		else
		{
			return 1;
		}
	} /* end of for loop */

	/* one more time so it doesn't require a trailing comma */
	if ( tmp_c != 0 )
	{
		tmp_c = atoi ( tmp );
		if ( tmp_c > 0 && tmp_c <= behavior.max_ttl )
		{
			behavior.skips[ tmp_c ] = 1;
		}
		else
		{
			return 1;
		}
		tmp_c = 0;
		memset ( & tmp, '\0', sizeof ( tmp ) );
	}

	return 0;
}

/*
 * remember if the user hits it
 */
void ctrl_c ( __attribute__((__unused__)) int unused )
{
	behavior.do_audit_exit = YES;
}
void ctrl_z( __attribute__((__unused__)) int unused )
{
	behavior.do_audit = YES;
}

/*
 * one timeval - another timeval
 */
double diff_time ( struct timeval * start, struct timeval * end )
{
	register double diff;

	diff = ( double )( end->tv_sec  - start->tv_sec  ) * 1000.0
	     + ( double )( end->tv_usec - start->tv_usec ) / 1000.0;

	return diff;
}

/*
 * create and apply the Berkley Packet Filter filter
 */
int do_filter( void )
{
	char filter [ FILTERSIZE ];
	struct bpf_program compiled_filter;
	bpf_u_int32 netmask = 0;

	if ( packet.protocol_number == IPPROTO_ICMP )
	{
		sprintf ( filter,			/* the buffer fed to pcap */
			behavior.filter_text,	/* the static text set on traceproto.c */
			packet.ip_id,			/* the randomly generated id */
			behavior.target );
	} else {
		sprintf ( filter,			/* the buffer fed to pcap */
			behavior.filter_text,	/* the static text set on traceproto.c */
			behavior.target,
			packet.src_port,
			packet.dst_port,
			packet.ip_id );			/* the randomly generated id */
	}

	if ( debug.interface == YES )
		printf ( "debug: BPF filter: %s\n",
			filter );

	if ( pcap_compile(
		state.psocket,		 /* pcap_t */
		& compiled_filter,
		filter,
		0,			/* don't optimise */
		netmask ) == -1 )
	{
		printf("pcap error: %s\n", pcap_geterr( state.psocket ) );
		return 1;
	}

	if ( pcap_setfilter(
		state.psocket,
		& compiled_filter ) != 0 )
	{
		printf("pcap error: %s\n", pcap_geterr( state.psocket ) );
		return 1;
	}

	pcap_freecode ( & compiled_filter );

	return 0;
}

/*
 * shove the details into memory
 */
int account_packet ( double return_time )
{

/*
 * record total for all hops
 */
	if ( return_time == ( double ) 0 )
		state.hop_record[ 0 ].lost_packets++;
	else
		state.hop_record[ 0 ].num_packets++;

	if ( state.hop_record[ 0 ].min_time == ( double ) 0
			|| state.hop_record[ 0 ].min_time > return_time )
		state.hop_record[ 0 ].min_time = return_time;

	if ( state.hop_record[ 0 ].max_time < return_time )
		state.hop_record[ 0 ].max_time = return_time;

	state.hop_record[ 0 ].ave_time =
		      ( state.hop_record[ 0 ].ave_time
		*	state.hop_record[ 0 ].num_packets
		+	return_time )
		/     ( state.hop_record[ 0 ].num_packets
		+	1 );

/*
 * record the individual hop stats
 */
	state.hop_record[ state.current_hop ].distance = state.current_hop;

	/*
	 * if there was no response, record the lost packet, but not the time
	 */
	if ( return_time == ( double ) 0 )
	{
		state.hop_record[ state.current_hop ].lost_packets++;
		return 0;
	}
	else
		state.hop_record[ state.current_hop ].num_packets++;

	if ( state.hop_record[ state.current_hop ].min_time == (double)0
		|| state.hop_record[ state.current_hop ].min_time > return_time )
	{
		state.hop_record[ state.current_hop ].min_time = return_time;
	}

	if ( state.hop_record[ state.current_hop ].max_time < return_time )
		state.hop_record[ state.current_hop ].max_time = return_time;

	/*
	 * math: ave = ( ave * (#pkts - 1) + response ) / #pkts
	 */
	state.hop_record[ state.current_hop ].ave_time =
			( state.hop_record[ state.current_hop ].ave_time
		*	( state.hop_record[ state.current_hop ].num_packets
		-	  1 )
		+	  return_time )
		/	( state.hop_record[ state.current_hop ].num_packets );

	return 1;
}

/*
 * formal info on all packets seen so far
 * ( may or may not indicate the end of the run )
 */
void hop_audit( void )
{
	int hop;

	if ( behavior.account_level > TP_ACCOUNT_NONE )
	{
		printf ( "\nhop :  min   /  ave   /  max   :  # packets  :  # lost\n" );
	}

	if ( behavior.account_level == TP_ACCOUNT_FULL )
	{
		printf (     "-------------------------------------------------------\n" );
		for ( hop = behavior.min_ttl; hop <= behavior.max_ttl; hop++ )
		{
			if ( state.hop_record[ hop ].num_packets != 0
				|| state.hop_record[ hop ].lost_packets != 0 )
			{
				printf ( "%3d : %#.5g / %#.5g / %#.5g : %3d packets : %3d lost\n",
					hop,
					state.hop_record[ hop ].min_time,
					state.hop_record[ hop ].ave_time,
					state.hop_record[ hop ].max_time,
					state.hop_record[ hop ].num_packets,
					state.hop_record[ hop ].lost_packets );
			}

		} /* end for hop loop */
	} /* end if full accounting */

	if ( behavior.account_level > TP_ACCOUNT_NONE )
	{
		printf (     "------------------------Total--------------------------\n" );
		if ( state.hop_record[ 0 ].num_packets != 0
			|| state.hop_record[ 0 ].lost_packets != 0 )
		{
			printf ( "total %#.5g / %#.5g / %#.5g : %3d packets : %3d lost\n",
				state.hop_record[ 0 ].min_time,
				state.hop_record[ 0 ].ave_time,
				state.hop_record[ 0 ].max_time,
				state.hop_record[ 0 ].num_packets,
				state.hop_record[ 0 ].lost_packets );
		}
		printf ( "\n" );
	} /* end if ( account level > TP_ACCOUNT_NONE ) */

	if ( behavior.do_audit_exit == YES )
	{
		printf ( "\n" );
		tixe ( tixe_cleanup, 0 );
	}

	behavior.do_audit = NO;
}

/*
 * take the name used to call and make it canonical
 */
int reg_name ( char * arg0 )
{
	size_t c;
	state.prog = & arg0 [ 0 ];
	for ( c = 0; c < strlen ( arg0 ); c++ )
	{
		if ( arg0[ c ] == '/' )
		{
			state.prog = & arg0[ c+1 ];
		}
	}

	return strlen ( state.prog );
}

/*
 * cleanup our mess and vamoos
 */
void tixe ( struct cleanup clean_list, int exit_status )
{
	if ( clean_list.payload_free      == YES ) { free ( packet.payload     ); }
	if ( clean_list.skips_free        == YES ) { free ( behavior.skips     ); }
	if ( clean_list.hop_record_free   == YES ) { free ( state.hop_record   ); }
	if ( clean_list.fake_psocket_free == YES ) { free ( state.fake_psocket ); }

	if ( clean_list.libnet_cleanup  == YES ) { libnet_destroy ( state.packet ); }
	if ( clean_list.pcap_cleanup    == YES ) { pcap_close ( state.psocket );    }

#ifdef HAVE_GETADDRINFO
	if ( clean_list.addrinfo_cleanup  == YES ) { freeaddrinfo( behavior.target_addrinfo_list_start ); }
#endif /* HAVE_GETADDRINFO */

/*
#ifdef HAVE_LIBCURSES
	if ( clean_list.curses_end == YES )
	{
		endwin ( );
	}
#endif / * HAVE_LIBCURSES */

#ifdef HAVE_LIBDMALLOC
	/* dmalloc_error(); */
#endif /* HAVE_LIBDMALLOC */

	exit ( exit_status );
}

/*
 * who are we
 */
void version( void )
{
        printf("%s version %s, released %s\n", PACKAGE, VERSION, RELEASE_DATE );
}

/*
 * sort out which tcp flags to use
 * called from the cmd line flags parsing
 */
u_char parse_flags ( char * user_flags )
{
	int i, errors = 0;
	u_char packet_flags = '\0';

	for ( i = strlen ( user_flags ); i > 0; i-- )
	{
		switch ( user_flags [ 0 ] )
		{
		case 'S':
		case 's':
			packet_flags |= TH_SYN;
			break;
		case 'A':
		case 'a':
			packet_flags |= TH_ACK;
			break;
		case 'R':
		case 'r':
			packet_flags |= TH_RST;
			break;
		case 'U':
		case 'u':
			packet_flags |= TH_URG;
			break;
		case 'p':
		case 'P':
			packet_flags |= TH_PUSH;
			break;
		case 'F':
		case 'f':
			packet_flags |= TH_FIN;
			break;
		case 'E':
		case 'e':
			packet_flags |= TH_ECE;
			break;
		case 'C':
		case 'c':
			packet_flags |= TH_CWR;
			break;
		default:
			errors++;
			break;
		}
		user_flags++;
	}

	return packet_flags;
}


/*
 * make a human readable timestamp
 */
int make_timestamp ( char * str_buf )
{
	time_t epoch_time;
	struct tm * time_struct;
	char time_string [ TP_TIMESTAMP_LEN ];
	const char * month [ ] = {
			"Jan", "Feb", "Mar", "Apr",
			"May", "Jun", "Jul", "Aug",
			"Sep", "Oct", "Nov", "Dec" };

	if ( ( epoch_time = time ( NULL ) ) == -1 )
	{
		perror ( "error determining time" );
		tixe ( tixe_cleanup, 1 );
	}

	time_struct = localtime ( & epoch_time );

	memset ( time_string, '\0', TP_TIMESTAMP_LEN );

	if ( debug.timestamp == YES )
		printf ( "timestamp_style: %d\n", behavior.timestamp_style );

	switch ( behavior.timestamp_style )
	{
	case TP_TIMESTAMP_US:
		snprintf ( time_string,
			TP_TIMESTAMP_LEN - 1,
			"%d/%0.2d/%d:%d:%d:%d", 
			time_struct->tm_mon + 1,
			time_struct->tm_mday,
			/* time_struct->tm_mon, */
			time_struct->tm_year + 1900,
			time_struct->tm_hour,
			time_struct->tm_min,
			time_struct->tm_sec );
		break;
	case TP_TIMESTAMP_DESCEND:
		snprintf ( time_string,
			TP_TIMESTAMP_LEN - 1,
			"%d/%d/%d:%d:%d:%d",
			time_struct->tm_year + 1900,
			time_struct->tm_mon + 1,
			time_struct->tm_mday,
			time_struct->tm_hour,
			time_struct->tm_min,
			time_struct->tm_sec );
		break;
	case TP_TIMESTAMP_EPOCH:
		snprintf ( time_string,
			TP_TIMESTAMP_LEN - 1,
			"%d",
			epoch_time );
		break;
	case TP_TIMESTAMP_STD:
	default:
		snprintf ( time_string,
			TP_TIMESTAMP_LEN - 1,
			"%0.2d/%s/%d:%d:%d:%d", 
			time_struct->tm_mday,
			month [ time_struct->tm_mon ],
			/* time_struct->tm_mon, */
			time_struct->tm_year + 1900,
			time_struct->tm_hour,
			time_struct->tm_min,
			time_struct->tm_sec );
		break;
	}

	strncpy ( str_buf, time_string, TP_TIMESTAMP_LEN );

	return strlen ( time_string );
}

/*
 * function to align the buffers on machines that need aligned buffers
 */
u_char * tp_align ( const u_char * buf, unsigned int offset, unsigned int size )
{
	unsigned char * new_buf;
	struct tp_align_ref * new_ref;
	struct tp_align_ref * ref_ptr;

	new_buf = ( u_char * ) malloc ( size );
	assert(new_buf != NULL);

	/* paranioa check in case the buf is NULL, shouldn't ever happen */
	if ( buf == NULL )
	{
		printf ( "error: reading from empty receive buffer\n" );
		tixe ( tixe_cleanup, 1 );
	}

	memset ( new_buf, '\0', size );
	memcpy ( new_buf, buf + offset, size );

	new_ref = ( struct tp_align_ref * ) malloc ( sizeof ( struct tp_align_ref ) );
	assert(new_ref != NULL);
	new_ref->next = NULL;
	new_ref->ref = new_buf;
	for (	ref_ptr = & tp_align_freelist;
		ref_ptr->next != NULL;
		ref_ptr = ref_ptr->next ) { /* nothing happens here */; }
	ref_ptr->next = new_ref;

	if ( debug.memory == YES )
	{
		printf ( "debug: memory: allocated new_buf %p\n", new_buf );
		printf ( "debug: memory: allocated new_ref %p\n", (void *) new_ref );
	}

	return new_buf;
}

/*
 * drop any and all unneeded root privs -- called ASAP after start
 * patch from jdassen
 */
void dropexcesscapabilities( void )
{
#ifdef HAVE_LIBCAP
	cap_t cap_d = cap_init();
	cap_value_t cap_values[] = {
		/* capabilities we need to keep */
		CAP_NET_RAW
	};

	if (!cap_d) {
		fprintf(stderr, "Could not alloc cap struct\n");
		exit(-1);
	}

	cap_clear(cap_d);
	cap_set_flag(cap_d, CAP_PERMITTED, 1, cap_values, CAP_SET);
	cap_set_flag(cap_d, CAP_EFFECTIVE, 1, cap_values, CAP_SET);

	if (cap_set_proc(cap_d) != 0) {
		fprintf(stderr, "Could not set capabilities: %s\n", strerror(errno));
		exit(1);
	}
	cap_free(&cap_d);
#endif /* HAVE_LIBCAP */
}


/*************************
 * Debugging starts here *
 *************************/

void parse_debug ( char * optarg )
{
	size_t i;

	for ( i = 0; i < strlen ( optarg ); i++ )
	{
		switch ( optarg [ i ] )
		{
		case 'a':
			printf ( "debug: all\n" );
			debug.loop = YES;
			debug.interface = YES;
			debug.send_buf = YES;
			debug.recv_buf = YES;
			debug.packet_length = YES;
			debug.memory = YES;
			break;
		case 'i':
			printf ( "debug: interface\n" );
			debug.interface = YES;
			break;
		case 'l':
			printf ( "debug: packet_length\n" );
			debug.packet_length = YES;
			break;
		case 'r':
			printf ( "debug: recv_buf\n" );
			debug.recv_buf = YES;
			break;
		case 's':
			printf ( "debug: send_buf\n" );
			debug.send_buf = YES;
			break;
		case 't':
			printf ( "debug: timestamp\n" );
			debug.timestamp = YES;
			break;
		case 'A':
			printf ( "debug: as_lookup\n" );
			debug.as_lookup = YES;
			break;
		case 'm':
			printf ( "debug: memory\n" );
			debug.memory = YES;
			break;
		default:
			printf ( "debug: invalid debug flag: %c\n", optarg [ i ] );
			break;
		}
	}
	
}

/*
 * Debugging function
 */
void debug_packet ( const u_int8_t * const raw, const u_int bytes )
{
	u_int i;

	/* set i to 1 rather than 0 so that the mod for spaces
	 * and \n's are easier */
	for ( i = 1; i <= bytes; i++ )
	{
		printf ( "%02x", raw [ i - 1 ] );
		if ( ( i % 2  ) == 0 ) { printf ( " "  ); }
		if ( ( i % 16 ) == 0 ) { printf ( "\n" ); }
	}
	if ( ( i % 16 ) != 0 ) { printf ( "\n" ); }

	return;
}
