--
-- (C) 2019-22 - ntop.org
--

-- ##############################################

local other_alert_keys = require "other_alert_keys"

-- Import the classes library.
local classes = require "classes"
-- Make sure to import the Superclass!
local alert = require "alert"
local alert_entities = require "alert_entities"

-- ##############################################

local alert_vulnerability_scan = classes.class(alert)

alert_vulnerability_scan.meta = {
  alert_key = other_alert_keys.alert_vulnerability_scan,
  i18n_title = "alerts_dashboard.vulnerability_scan_title",
  icon = "fas fa-fw fa-exclamation-triangle",
  entities = {
     alert_entities.am_host,
  },
}

-- ##############################################

function alert_vulnerability_scan:init(differences_list)
   -- Call the parent constructor
   self.super:init()

   self.alert_type_params = differences_list
   -- Trick to set this alert as an active monitoring alert
   self.alert_type_params.threshold = 0
   self.alert_type_params.value = 0
   self.alert_type_params.measurement = differences_list.measurement
end

-- #######################################################

-- Function to normalize values (check nil cases)
local function normalize_values(primary_key, secondary_key) 
   if (primary_key == nil) then
      return nil
   elseif (primary_key[secondary_key] == nil) then
      return nil
   else
      return primary_key[secondary_key]
   end
end

-- #######################################################

-- @brief Format an alert into a human-readable string
-- @param ifid The integer interface id of the generated alert
-- @param alert The alert description table, including alert data such as the generating entity, timestamp, granularity, type
-- @param alert_type_params Table `alert_type_params` as built in the `:init` method
-- @return A human-readable string
function alert_vulnerability_scan.format(ifid, alert, alert_type_params)
   local msg = ""

   if (alert_type_params.scan_type == "tcp_portscan" or alert_type_params.scan_type == "tcp_openports") then
      if (not isEmptyString(alert_type_params.tcp_ports_case)) then
         msg = msg .. i18n('vulnerability_scan.ports_changed_cases.'..alert_type_params.tcp_ports_case, {
            open_ports_num = normalize_values(alert_type_params.tcp_open_ports,"num"),
            open_ports = normalize_values(alert_type_params.tcp_open_ports,"ports"),
            closed_ports_num = normalize_values(alert_type_params.tcp_closed_ports,"num"),
            closed_ports = normalize_values(alert_type_params.tcp_closed_ports,"ports"),
            protocol = i18n("tcp")
         })
         msg = msg:gsub("%,", ", ")
      end

   elseif (alert_type_params.scan_type == "udp_portscan") then
      if (not isEmptyString(alert_type_params.udp_ports_case)) then
      msg = msg .. i18n('vulnerability_scan.ports_changed_cases.'..alert_type_params.udp_ports_case, {
         open_ports_num = normalize_values(alert_type_params.udp_open_ports,"num"),
         open_ports = normalize_values(alert_type_params.udp_open_ports,"ports"),
         closed_ports_num = normalize_values(alert_type_params.udp_closed_ports,"num"),
         closed_ports = normalize_values(alert_type_params.udp_closed_ports,"ports"),
         protocol = i18n("udp")
      })
      msg = msg:gsub("%,", ", ")
      end

   end

   if alert_type_params.num_new_cve_issues then
      local new_cve = table.concat(alert_type_params.new_cve or {}, ", ")
      if alert_type_params.num_new_cve_issues > 5 then
         new_cve = new_cve .. " " .. i18n('vulnerability_scan.and_other_n', { n = alert_type_params.num_new_cve_issues - 5 })
      end 
      msg = msg .. i18n('vulnerability_scan.new_issues', { num_issues = alert_type_params.num_new_cve_issues, new_cve = new_cve })
   end

   if alert_type_params.num_cve_solved then
      local cve_solved = table.concat(alert_type_params.cve_solved or {}, ", ")
      if alert_type_params.num_cve_solved > 5 then
         cve_solved = cve_solved .. " " .. i18n('vulnerability_scan.and_other_n', { n = alert_type_params.num_cve_solved - 5 })
      end 
      msg = msg .. i18n('vulnerability_scan.solved_issues', { num_issues = alert_type_params.num_cve_solved, cve_solved = cve_solved }) .. " "
   end

   local host = alert_type_params.host_name
   if isEmptyString(host) then
      host = alert_type_params.host
   end

   local report_url = getHttpHost() .. ntop.getHttpPrefix() .. 
      "/lua/vulnerability_scan.lua?page=report&report_template=vs_result"

   local alert_descr = i18n('vulnerability_scan.host_alert', { host = host, msg = msg, url = report_url })

   return alert_descr
end

-- #######################################################

return alert_vulnerability_scan
