/*
 *  Copyright (C) 2005-2007 Renier Roth
 *  www.renier.de
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package de.renier.vdr.channel.editor.container;

import java.awt.Point;
import java.awt.dnd.DnDConstants;
import java.util.Enumeration;
import java.util.Vector;

import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreePath;

import de.renier.vdr.channel.ChannelElement;
import de.renier.vdr.channel.editor.ChannelEditor;

/**
 * DefaultTreeTransferHandler
 * 
 * @author <a href="mailto:editor@renier.de">Renier Roth</a>
 */
public class DefaultTreeTransferHandler extends AbstractTreeTransferHandler {

  public DefaultTreeTransferHandler(DNDTree tree, int action) {
    super(tree, action, true);
  }

  public boolean canPerformAction(DNDTree target, DefaultMutableTreeNode draggedNode, int action, Point location) {
    TreePath pathTarget = target.getPathForLocation(location.x, location.y);
    if (pathTarget == null) {
      target.setSelectionPath(null);
      return (false);
    }
    // get channelElement
    DefaultMutableTreeNode node = (DefaultMutableTreeNode) pathTarget.getLastPathComponent();
    if (node.getUserObject() instanceof ChannelElement && draggedNode.getUserObject() instanceof ChannelElement) {
      ChannelElement channelElementTarget = (ChannelElement) node.getUserObject();
      ChannelElement channelElementSource = (ChannelElement) draggedNode.getUserObject();
      if (channelElementSource.isCategory() && channelElementTarget.isRadioOrTelevisionOrService()) {
        target.setSelectionPath(null);
        return (false);
      }
    }

    if (action == DnDConstants.ACTION_MOVE) {
      if (node == draggedNode.getParent() || draggedNode.isNodeDescendant(node)) {
        target.setSelectionPath(null);
        return (false);
      }
      target.setSelectionPath(pathTarget);
      return (true);
    }
    target.setSelectionPath(null);
    return (false);
  }

  public boolean executeDrop(DNDTree target, DefaultMutableTreeNode draggedNode, DefaultMutableTreeNode newParentNode,
      Vector expandedStates, int action) {

    if (action == DnDConstants.ACTION_MOVE) {
      // special for treenode, we remove always the parent
      draggedNode.removeFromParent();
      DefaultTreeModel treeModel = (DefaultTreeModel) target.getModel();

      // get channel Element
      ChannelElement channelElementTarget = (ChannelElement) newParentNode.getUserObject();
      ChannelElement channelElementSource = (ChannelElement) draggedNode.getUserObject();
      // check special insert condition
      if (channelElementSource.isRadioOrTelevisionOrService() && channelElementTarget.isCategory()) {
        // if we move a channel into a category
        treeModel.insertNodeInto(draggedNode, newParentNode, newParentNode.getChildCount());
      } else if (channelElementSource.isRadioOrTelevisionOrService() && newParentNode.isRoot()) {
        // if we move a channel into the root
        treeModel.insertNodeInto(draggedNode, newParentNode, 0);
      } else if (channelElementSource.isCategory() && newParentNode.isRoot()) {
        // if we move a category into the root
        treeModel.insertNodeInto(draggedNode, newParentNode, newParentNode.getChildCount());
      } else {
        // all other get parent and insert node into position
        DefaultMutableTreeNode parentNode = (DefaultMutableTreeNode) newParentNode.getParent();
        int pos = parentNode.getIndex(newParentNode);
        // insert node into parent on selected position
        treeModel.insertNodeInto(draggedNode, parentNode, pos);
      }
      // mark modified
      ChannelEditor.application.setModified(true);

      // scroll to node
      int i = 0;
      if (expandedStates != null) {
        for (Enumeration enumeration = draggedNode.depthFirstEnumeration(); enumeration.hasMoreElements(); i++) {
          DefaultMutableTreeNode element = (DefaultMutableTreeNode) enumeration.nextElement();
          TreePath path = new TreePath(element.getPath());
          if (((Boolean) expandedStates.get(i)).booleanValue()) {
            target.expandPath(path);
          }
        }
      }
      TreePath treePath = new TreePath(draggedNode.getPath());
      target.scrollPathToVisible(treePath);
      target.setSelectionPath(treePath);
      return (true);
    }
    return (false);
  }

  /*
   * (non-Javadoc)
   * 
   * @see de.renier.vdr.channel.editor.tree.AbstractTreeTransferHandler#afterDrop(de.renier.vdr.channel.editor.tree.DNDTree,
   *      javax.swing.tree.DefaultMutableTreeNode, int)
   */
  public void afterDrop(DNDTree dndTree, DefaultMutableTreeNode draggedNode, int action) {

  }
}