/*  job_setupsectormappings.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_setupsectormappings.h"
#include "functions.h"
#include "sectormappings.h"
#include "files.h"
#include "graph.h"
#include "kinematics.h"
#include "job_setupcrossings.h"
#include "job_verifypermutationsymmetries.h"

using namespace std;

namespace Reduze {

// register job type
namespace {
JobProxy<SetupSectorMappings> dummy;
}

SetupSectorMappings::SetupSectorMappings() :
		find_sector_relations_(true), find_sector_symmetries_(true), examine_twists_for_symmetries_(
				true), find_zero_sectors_(true), find_graphless_zero_sectors_(
				false), minimize_graphs_by_twists_(false), construct_minimal_graphs_(
				true), setup_crossings_(true), minimize_target_crossings_(true), allow_general_crossings_(
				true), verify_permutation_symmetries_(true) {
	add_auto_options();
}

void SetupSectorMappings::make_graph_directories(const IntegralFamily* ic,
		std::string& all_dir, std::string& shift_targets_dir) const {
	string graph_dir = Files::instance()->get_graph_directory(ic);
	remove_directory_with_files(graph_dir);
	make_directory(graph_dir);
	all_dir = graph_dir + "all/";
	shift_targets_dir = graph_dir + "shift_targets/";
	make_directory(all_dir);
	make_directory(shift_targets_dir);
}

void SetupSectorMappings::find_crossed_relations(const IntegralFamily* ic,
		const std::set<SectorGL>& sectors) {
	// find crossed counter parts for sector mappings (one per equiv. class)
	list<SectorMappings*> ms;
	// it is enough to consider minimal crossing representatives as sources
	set<Crossing> xs =
			Files::instance()->crossings(ic->kinematics()->name())->equivalent_crossing_classes();
	for (set<Crossing>::const_iterator x = xs.begin(); x != xs.end(); ++x)
		if (!x->is_equivalent_to_identity()) {
			string xfname = x->name_for_crossed_family(ic);
			ms.push_back(Files::instance()->sectormappings(xfname));
		}
	// shift crossed versions of target sectors to uncrossed ones
	ProgressBar pbar(2, "crossed relations:", sectors.size());
	pbar.start();
	for (set<SectorGL>::const_iterator s = sectors.begin(); s != sectors.end();
			++s) {
		ASSERT(*ic == *(s->sector().integralfamily()));
		for (list<SectorMappings*>::iterator m = ms.begin(); m != ms.end();
				++m) {
			// construct crossed sector and its graph
			Sector xsec((*m)->integralfamily(), s->sector().id());
			list<SectorGL> gs;
			set<Sector> xsecset;
			xsecset.insert(xsec);
			SectorMappings::get_graphs(gs, xsecset, minimize_graphs_by_twists_,
					construct_minimal_graphs_, false);
			ASSERT(gs.size() == 1);
			VERIFY(gs.front().sector() == xsec);
			// find shift
			set<SectorGL> from, to;
			from.insert(gs.front());
			to.insert(*s);
			map<SectorGL, list<SectorGL> > dummy;
			(*m)->find_sector_shifts(to, from, dummy, true, false, true, true,
					cached_node_perm_, false);
			map<int, Sector>::const_iterator rel = //
					(*m)->sector_relations().find(s->sector().id());
			if (rel != (*m)->sector_relations().end()) {
				const Sector& target = rel->second;
				if (Crossing::to_minimal_crossing(target) == //
						Crossing::to_minimal_crossing(xsec)) {
					// delete shift, is covered by crossing equivalences
					(*m)->remove_shift(s->sector().id());
				} else {
					LOGX(xsec << "=>" << s->sector() << ": " << target);
				}
			}
		}
		pbar.print();
	}
	pbar.end();
}

void SetupSectorMappings::setup(SectorMappings* sm,
		std::set<SectorGL>& shift_targets) {

	const IntegralFamily* fam = sm->integralfamily();
	const Kinematics* kin = fam->kinematics();
	Files* files = Files::instance();
	const OrderedCrossings* oco = files->crossings(kin->name());

	LOG("Setup sector mappings for " << fam->name() << "\n");
    kin->print_info();
	oco->print_info();
    fam->print_info();

	// sector tree (using global permutation symmetry)
	LOG("Finding non-equivalent sectors using permutation symmetries:");
	bool have_perm_syms = !fam->permutation_symmetries().empty();
	set<Sector> sectors;
	set<int> sector_ids;
    const int max_sec_id = fam->max_sector_id();
	for (int i = 0; i <= max_sec_id; ++i) {
		int id = have_perm_syms ? fam->get_equivalent(i) : i;
		sectors.insert(Sector(fam, id));
		sector_ids.insert(id);
	}
	// print non-equivalent sectors to log-file
	LOGX("\nNon-equivalent sectors:");
	LOGX(fam->t_tree_nice_string(sector_ids));
	LOG(sector_ids.size() << " non-equivalent sectors were found");

	set<SectorGL> local_sectors;
	// print graphs as dot files in standard directory ./graphs/family_name/
	string all_dir(""), shift_targets_dir(""); // suppress valgrind message
	{
		// find graphs of sectors and add sectors with vanishing jacobian determinant
		// of the loop momenta matrix to zero_sectors_ and set the sector_without_graph_
		list<SectorGraph> sectorgraphs;
		bool verbose = true;
		sm->find_graphs(sectorgraphs, sectors, minimize_graphs_by_twists_,
				construct_minimal_graphs_, verbose);

		// print graphs as dot files in standard directory ./graphs/family_name/
		make_graph_directories(fam, all_dir, shift_targets_dir);
		LOG("Printing all graphs from sectors in " << all_dir << ":");
		for (list<SectorGraph>::const_iterator s = sectorgraphs.begin();
				s != sectorgraphs.end(); ++s)
			s->print_dot(
					all_dir + s->sector().get_safe_string_for_filename(true)
							+ ".dot");

		LOGX("\nSectors without a graph:");
		LOGX(fam->t_tree_nice_string(sm->sectors_without_graph()));

		// canonical labeling

		LOG("Finding canonical labels:");
		ProgressBar pbar(2, "canonical labels:", sectorgraphs.size());
		pbar.start();

		for (; !sectorgraphs.empty(); sectorgraphs.pop_front()) {
			local_sectors.insert(SectorGL(sectorgraphs.front()));
			pbar.print();
		}
		pbar.end();
	}

	// the sectors with graphs of this family which are not shifted to any sector of 'shift_targets'
	std::set<SectorGL> local_shift_targets;
	// list of sectors by their shift target
	// all sectors are from this family, the target itself is not included in the list
	map<SectorGL, list<SectorGL> > local_targets_by_equiv;
    // quick check we map to targets of lower families
	ASSERT(shift_targets.empty() ||
		   *(shift_targets.rbegin()->sector().integralfamily()) < *fam);
	// try to shift the 'local_sectors' to the shift_targets or to them self
	// updates the 'shift_targets'
	// sets the 'local_shift_targets'
	if (find_sector_relations_) {
		sm->find_sector_shifts(shift_targets, local_sectors,
				local_targets_by_equiv, minimize_target_crossings_,
				find_zero_sectors_, true, allow_general_crossings_,
				cached_node_perm_, true);
	} else {
		// move local sectors to local_shift_targets
		map<SectorGL, list<SectorGL> >::iterator in =
				local_targets_by_equiv.end();
		list<SectorGL> empty;
		for (set<SectorGL>::iterator s = local_sectors.begin();
				s != local_sectors.end();) {
			in = local_targets_by_equiv.insert(in, make_pair(*s, empty));
			local_sectors.erase(s++);
		}
	}
	LOG("Found " << local_targets_by_equiv.size() << " shift target sectors.");
	// print the number of different graphs in the shift targets
	set<CanonicalLabel> different_graphs;
	for (map<SectorGL, list<SectorGL> >::const_iterator s =
			local_targets_by_equiv.begin(); s != local_targets_by_equiv.end();
			++s)
		different_graphs.insert(s->first.label_pair().first);
	LOG("  shift targets contain " << different_graphs.size() //
			<< " different graph(s)");

	// zero test for the local_targets

	const bool allow_long_zero_test = (kin->external_momenta().nops() != 0);
	if (find_zero_sectors_) {
		LOGN("Finding zero sectors of sectors with a graph:");
		map<Sector, list<Sector> > zeros;
		map<Sector, list<Sector> >::iterator in = zeros.end();
		map<SectorGL, list<SectorGL> >::const_iterator s;
		list<SectorGL>::const_iterator t;
		s = local_targets_by_equiv.begin();
		for (; s != local_targets_by_equiv.end(); ++s) {
			in = zeros.insert(in, make_pair(s->first.sector(), list<Sector>()));
			for (t = s->second.begin(); t != s->second.end(); ++t)
				in->second.push_back(t->sector());
		}
		sm->find_zero_sectors(zeros, allow_long_zero_test);
		LOG(
				"Found " << sm->zero_sectors().size() << " zero sectors (isomorphics included).");
	}
	if (find_zero_sectors_ && find_graphless_zero_sectors_) {
		LOGN("Finding zero sectors for sectors without a graph:");
		map<Sector, list<Sector> > zeros;
		map<Sector, list<Sector> >::iterator in = zeros.end();
		int num_phys_zeros = sm->zero_sectors().size();
		set<int>::const_iterator s = sm->sectors_without_graph_.begin();
		for (; s != sm->sectors_without_graph_.end(); ++s)
			in = zeros.insert(in, make_pair(Sector(fam, *s), list<Sector>()));
		sm->find_zero_sectors(zeros, allow_long_zero_test);
		int num_zeros = sm->zero_sectors().size();
		LOG(
				"Found additional " << num_zeros - num_phys_zeros << " zero sectors.");
	}

	set<int> non_zero_shift_targets;
	set<CanonicalLabel> non_zero_different_graphs;
	for (map<SectorGL, list<SectorGL> >::const_iterator s =
			local_targets_by_equiv.begin(); s != local_targets_by_equiv.end();
			++s) {
		if (sm->zero_sectors().find(s->first.sector().id())
				== sm->zero_sectors().end()) {
			non_zero_shift_targets.insert(s->first.sector().id());
			non_zero_different_graphs.insert(s->first.label_pair().first);
		}
		local_shift_targets.insert(s->first);
	}
	LOGX("\nNon-vanishing shift targets:");
	LOGX(fam->t_tree_nice_string(non_zero_shift_targets));
	LOG("Found " << non_zero_shift_targets.size() //
			<< " non-vanishing shift target sectors");
	LOG("  non-vanishing shift targets contain " //
			<< non_zero_different_graphs.size() << " different graph(s)");

	// symmetry shifts

	if (find_sector_symmetries_) {
		LOG("Finding symmetry shift relations:");
		sm->find_symmetry_shifts(local_shift_targets,
				examine_twists_for_symmetries_);
	}
	//LOG("");

	// pure crossing relations: crossed to uncrossed sector
	// setup of uncrossed mappings done, it is safe now to construct
	// crossed mappings based on them (implicitely via Files::sectormappings())
	// and enrich them
	if (allow_general_crossings_) {
		LOG("Finding crossed sector relations for non-zero shift targets");
		find_crossed_relations(fam, local_shift_targets);
	}

	LOG("Printing graphs of shift targets in " << shift_targets_dir << ":");
	set<SectorGL>::const_iterator s;
	for (s = local_shift_targets.begin(); s != local_shift_targets.end(); ++s)
		s->sectorgraph().print_dot(
				shift_targets_dir
						+ s->sector().get_safe_string_for_filename(true)
						+ ".dot");
	string dot_file_prefix = all_dir + safe_filename(fam->name());
	string dot_file_prefix2 = shift_targets_dir + safe_filename(fam->name());
	string ps_file = safe_filename(fam->name()) + ".ps";
	string ps_file2 = safe_filename(fam->name()) + "_shift_targets.ps";
	LOG("To generate a postscript file from the graphs in the dot format type e.g.");
	LOG("  dot -Tps -o " << ps_file << " " << dot_file_prefix << "_*.dot");
	LOG("  neato -Tps -o " << ps_file2 << " " << dot_file_prefix2 << "_*.dot");

	string mmagraphs = Files::instance()->get_graph_directory(fam) + "shift_targets.m";
	LOG("Printing graphs in mma format to " << mmagraphs);
	ofstream of(mmagraphs.c_str());
	of << "{\n";
	for (s = local_shift_targets.begin(); s != local_shift_targets.end(); ++s) {
		if (s != local_shift_targets.begin())
			of << ",\n";
		of << "{\"" << s->sector().integralfamily()->name() << "\", "
				<< s->sector().t() << ", " << s->sector().id() << ", ";
		s->sectorgraph().print_mma(of);
		of << "}";
	}
	of << "\n}\n";
	of.close();

	LOG("Done setup sector mappings for " << fam->name());
}

void SetupSectorMappings::run_serial() {

	// sort families by loop

	Files* files = Files::instance();
	map<int, list<IntegralFamily*> > families_by_loop;
	list<IntegralFamily*> ifs = files->integralfamilies();
	for (list<IntegralFamily*>::const_iterator f = ifs.begin(); f != ifs.end();
			++f)
		if (!(*f)->is_crossed())
			families_by_loop[(*f)->loop_momenta().nops()].push_back(*f);

	// print what will be done

	map<int, list<IntegralFamily*> >::const_reverse_iterator fbl; // heavy work first
	LOG("Setup sector mappings for the following integral families requested:");
	for (fbl = families_by_loop.rbegin(); fbl != families_by_loop.rend();
			++fbl) {
		list<IntegralFamily*>::const_iterator fam;
		LOG("  " << fbl->first << "-loop:");
		for (fam = fbl->second.begin(); fam != fbl->second.end(); ++fam) {
			LOG("    " << (*fam)->name());
		}
	}
	LOG("");

	// setup the sector mappings for each class of families with the same number of loop momenta

	for (fbl = families_by_loop.rbegin(); fbl != families_by_loop.rend();
			++fbl) {
		// cached shift target sectors
		set<SectorGL> shift_targets;
		// cached node permutations by sector
		//map<Sector, list<map<int, int> > > cached_node_perm;
		bool lower_family_changed = false;
		list<IntegralFamily*>::const_iterator fam;
		for (fam = fbl->second.begin(); fam != fbl->second.end(); ++fam) {
			bool sm_already_setup = is_readable_file(
					files->get_filename_sectormappings((*fam)->name()));
			if (sm_already_setup && is_conditional()) {
				// case of where a family is already setup: we must find the shift target sectors of this family
				SectorMappings* mp = files->sectormappings((*fam)->name());
				LOG("Integral family " << (*fam)->name() << " already setup.");
				// case where lower family has been changed not handled
				if (lower_family_changed)
					WARNING("Lower integral family has been changed." //
							<< " This might invalidate the already existent setup" //
							<< " of integral family " << (*fam)->name() << "\n");
				if (find_sector_relations_) {
					list<SectorGL> st_list;
					mp->find_shift_targets_with_graph(st_list,
							minimize_graphs_by_twists_,
							construct_minimal_graphs_);
					shift_targets.insert(st_list.begin(), st_list.end());
				}
			} else {
				// clear and create empty sector mappings and set it up
				files->clear_sectormappings((*fam)->name());
				files->save(SectorMappings(*fam));
				SectorMappings* m = files->sectormappings((*fam)->name());
				setup(m, shift_targets);
				LOG("Saving sector mappings for " << (*fam)->name());
				files->save(*m);
				lower_family_changed = true;
				LOG("");
			}
		} // t
	} // loop
}

bool SetupSectorMappings::find_dependencies(const set<string>& outothers, //
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	bool res = SetupSectorMappings::get_dependencies(outothers, in, out,
			auxjobs, setup_crossings_, verify_permutation_symmetries_,
			is_conditional());
	return res;
}

namespace {
std::string dummy_output_file_for_perm_verification(const std::string& intfam) {
	string tmp = Files::instance()->get_tmp_directory();
	string dummy_file = string("permutation_symmetries_verified_for_") + intfam;
	return cat_directory_and_filename(tmp, dummy_file);
}
}


bool SetupSectorMappings::get_dependencies(
		const set<string>& outothers, //
		list<string>& in, list<string>& out, list<Job*>& auxjobs,
		bool setup_crossings, bool verify_perms, bool is_conditional) {

	Files* files = Files::instance();
	bool check = files->crossings_loaded(); // \todo remove this check

	// remove empty sector mappings files (empty is if all maps and sequences are empty)
	files->remove_empty_sectormappings_files();

	// auxiliary job setup_crossings
	if (setup_crossings) {
		Job* cr_job = new SetupCrossings();
		cr_job->set_conditional(is_conditional);
		auxjobs.push_back(cr_job);
		in.push_back(files->get_filename_crossings());
		LOGX("Created auxiliary job setup_crossings");
	}

	// auxiliary job verify_permutation_symmetries: perform if
	// verify_permutation_symmetries_ = true
	// AND
	// conditional = false  OR  sector mappings file does not exist
	list<IntegralFamily*> ics = files->uncrossed_integralfamilies();
	list<IntegralFamily*>::const_iterator ic;
	for (ic = ics.begin(); ic != ics.end(); ++ic) {
		const string& name = (*ic)->name();
		const string& sm_file = files->get_filename_sectormappings(name);
		bool sm_is_readable = is_readable_file(sm_file);
		string dummy = dummy_output_file_for_perm_verification(name);
		if (verify_perms && (!is_conditional || !sm_is_readable)) {
			Job* job = new VerifyPermutationSymmetries();
			dynamic_cast<VerifyPermutationSymmetries*>(job)->set_family(name);
			dynamic_cast<VerifyPermutationSymmetries*>(job)->set_dummy_output_file(
					dummy);
			job->set_conditional(false);
			auxjobs.push_back(job);
			in.push_back(dummy);
			LOGX("Created auxiliary job verify_permutation_symmetries" //
					" for family: " << name);
		}
	}

	// add output files of this job
	for (ic = ics.begin(); ic != ics.end(); ++ic)
		out.push_back(files->get_filename_sectormappings((*ic)->name()));

	if (!check)
		VERIFY(!files->crossings_loaded());
	return true;
}

void SetupSectorMappings::init() {
}

std::string SetupSectorMappings::get_description() const {
	return string("setup sector mappings");
}

}
