/*  job_generateidentities.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_generateidentities.h"
//#include "functions_yaml.h"
#include "equationlist.h"
#include "files.h"
#include "filedata.h"
#include "identitygenerator.h"
//#include "ginac/ginac.h"

using namespace std;

namespace Reduze {

// register job types at JobFactory
namespace {
JobProxy<GenerateIdentities> dummy;
}

void GenerateIdentities::run_serial() {
	make_directory(get_directory_of_filename(output_file_));
	set<INT> seeds;
	InFileINTs in(input_file_);
	in.get_all(seeds);
	in.close();

	int num_seeds = seeds.size();
	if (to_ > num_seeds) {
		LOG("  Adjusting last integral " << to_ << " to " << num_seeds);
		to_ = num_seeds;
	}

	// translate the inclusive range [from_, to_] into the half inclusive and
	// half exclusive range of iterators [it_from, it_to)
	set<INT>::const_iterator it_from = seeds.begin(), it_to = seeds.end();
	int lauf = 1;
	if (from_ > 0) {
		for (; it_from != seeds.end() && lauf < from_; ++it_from, ++lauf) {
		}
	}
	if (to_ > 0) {
		it_to = it_from;
		for (; it_to != seeds.end() && lauf <= to_; ++it_to, ++lauf) {
		}
	}
	compute(it_from, it_to);
}

void GenerateIdentities::compute(const std::set<INT>::const_iterator& it_from,
		const std::set<INT>::const_iterator& it_to) {
	Timer timer;
	int to_minus_from = distance(it_from, it_to);
	LOG("  Generating identities of type " << type_ << ": " << get_filename_without_directory(output_file_));

	// if from_ > to_ still produce an empty output file
	if (from_ > to_) {
		LOG("  Nothing to do.");
		OutFileEquations out(output_file_);
		out.finalize();
		return;
	}

	string range;
	if (from_ < 0 && to_ < 0)
		range = "full range";
	else if (from_ > 0 && to_ < 0)
		range = "[" + to_string(from_) + ", end]";
	else if (from_ < 0 && to_ > 0)
		range = "[begin, " + to_string(to_) + "]";
	else
		range = "[" + to_string(from_) + ", " + to_string(to_) + "]";

	ProgressBar pbar(2, "identities from " + to_string(to_minus_from)
			+ " seeds, range: " + range, to_minus_from);
	IdentityGenerator generator(options_);
	int num_equations = 0;
	OutFileEquations out(output_file_);
	pbar.start();
	size_t num_integrals = 0;
	for (set<INT>::const_iterator s = it_from; s != it_to; ++s) {
		IdentityList eqs;
		generator.generate(*s, eqs, type_);
		for (IdentityList::const_iterator it = eqs.begin(); it != eqs.end(); ++it, ++num_equations) {
			num_integrals += it->size();
			out << *it;
		}
		pbar.print();
	}
	out.finalize();
	pbar.end();
	LOG("  Identities generated: " << num_equations << " " << timer.get_wall_time_nice_string());
	LOGX("  Average number of integrals per equation: " << double(num_integrals) / double(num_equations));
}

bool GenerateIdentities::find_dependencies(const set<string>& outothers,//
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	//find_dependencies_all_sectormappings(outothers, in, auxjobs);
	in.push_back(input_file_);
	out.push_back(output_file_);
	return true;
}

std::string GenerateIdentities::get_description() const {
	ostringstream ss;
	ss << "generate identities: " << get_filename_without_directory(
			output_file_);
	return ss.str();
}

void GenerateIdentities::init() {
	if (from_ > 0 && to_ > 0 && from_ > to_) {
		ostringstream ss;
		ss << "Invalid range [" << from_ << ", " << to_ << "]";
		throw std::runtime_error(ss.str());
	}
	if (input_file_.empty() || output_file_.empty())
		throw runtime_error("input or output file name undefined.");
	if (input_file_ == output_file_)
		throw runtime_error("same input and output file name.");
}

} // namespace Reduze
