/*  job_export.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_export.h"
#include "functions.h"
#include "amplitude.h"
#include "files.h"
#include "filedata.h"
#include "yamlutils.h"
#include "ginacutils.h"
#include "kinematics.h"
#include "equationlist.h"
#include "feynmanrules.h"
#include "globalsymbols.h"

using namespace std;

namespace Reduze {

// register job types at JobFactory
namespace {
JobProxy<Export> dummy;
}

void Export::run_serial() {
	INT::load_preferred(preferred_masters_filename_);

	LOG("Checking type of input file \"" << input_filename_ << "\"");
	if (InFileEquations::check_filetype(input_filename_)) {
		LOG("  file contains equations");
		export_equations();
	} else if (InFileLinearCombinations::check_filetype(input_filename_)) {
		LOG("  file contains linear combinations");
		export_linearcombinations();
	} else {
		export_INTs();
		LOG("  assuming file contains a list of integrals");
	}
}

void Export::write_export_header(OutFileData& of) {
	write_header(of);
	YAML::Emitter emitter2;
	print(emitter2);
	of.write_comment(string(emitter2.c_str()) + "\n\n");
}

void Export::export_INTs() {
	LOG("Exporting contents of file '" << input_filename_ << "'");
	OutFileINTs out(output_filename_, output_format_);
	write_export_header(out);
	InFileINTs in(input_filename_);
	size_t s = in.file_size();
	ProgressBar pbar(2, "processing " + to_string(s) + " bytes", s);
	pbar.start();
	while (in) { // = while (!in.fail())
		INT i = in.get_next();
		pbar.print(in.tellg());
		if (toggle_metric_convention_)
			i.set_metric_mostly_plus(!i.uses_metric_mostly_plus());
		out << i;
	}
	in.close();
	out.finalize();
	pbar.end();
	LOG("Exported to file '" << output_filename_ << "'");
}

void Export::export_equations() {
	LOG("Exporting contents of file '" << input_filename_ << "'");
	OutFileEquations out(output_filename_, output_format_);
	write_export_header(out);
	InFileEquations in(input_filename_);
	size_t s = in.file_size();
	ProgressBar pbar(2, "processing " + to_string(s) + " bytes", s);
	pbar.start();
	Identity eq;
	while (in.get(eq)) {
		pbar.print(in.tellg());
		eq.reconstruct_symbol_replaced_by_one();
		eq.apply(reparametrization_rules_);
		if (toggle_metric_convention_)
			eq.toggle_metric_convention();
		eq.quick_solve(); // for preferred masters, don't divide by 0 etc.
		if (series_parameter_ != undefined)
			eq.series(series_parameter_, series_point_, series_order_);
		out << eq;
	}
	in.close();
	out.finalize();
	pbar.end();
	LOG("Exported to file '" << output_filename_ << "'");
}

void Export::export_linearcombinations() {
	LOG("Exporting contents of file '" << input_filename_ << "'");
	OutFileLinearCombinations out(output_filename_, output_format_);
	write_export_header(out);
	InFileLinearCombinations in(input_filename_);
	size_t s = in.file_size();
	ProgressBar pbar(2, "processing " + to_string(s) + " bytes", s);
	pbar.start();
	LinearCombination lc;
	while (in.get(lc)) {
		pbar.print(in.tellg());
		lc.apply(reparametrization_rules_);
		if (toggle_metric_convention_)
			lc.toggle_metric_convention();
		if (series_parameter_ != undefined)
			lc.series(series_parameter_, series_point_, series_order_);
		out << lc;
	}
	in.close();
	out.finalize();
	pbar.end();
	LOG("Exported to file '" << output_filename_ << "'");
}

bool Export::find_dependencies(const set<string>& outothers,//
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	//find_dependencies_all_sectormappings(outothers, in, auxjobs);
	in.push_back(input_filename_);
	out.push_back(output_filename_);
	if (!preferred_masters_filename_.empty())
		in.push_back(preferred_masters_filename_);
	return true;
}

void Export::print_manual_options(YAML::Emitter& os) const {
	using namespace YAML;
	os << Key << "new_parameters" << Value << Flow << new_parameters_;
	os << Key << "reparametrization_rules" << Value << Flow
			<< reparametrization_rules_;
	os << Key << "toggle_metric_convention" << Value
			<< (toggle_metric_convention_ ? "true" : "false");
	os << Key << "series_expansion" << Value << Flow << BeginSeq;
	if (series_parameter_ != undefined)
		os << series_parameter_ << series_point_ << series_order_;
	os << EndSeq;
}

void Export::read_manual_options(const YAML::Node& node) {
	if (output_filename_.empty())
		output_filename_ = input_filename_
				+ OutFileData::default_filename_suffix(output_format_);
	if (node.FindValue("new_parameters")) {
		const YAML::Node& n = node["new_parameters"];
		if (n.Type() != YAML::NodeType::Sequence)
			throw runtime_error("new_parameters must be a sequence "
					+ position_info(n));
		for (YAML::Iterator it = n.begin(); it != n.end(); ++it) {
			GiNaC::symbol s;
			*it >> s;
			new_parameters_.append(s);
		}
	}

	Files* files = Files::instance();
	Kinematics* kin = files->kinematics();
	GiNaC::lst syms = add_lst(kin->all_symbols(), new_parameters_);

	GiNaC::lst s = files->globalsymbols()->coupling_constants();
	syms = add_lst(syms, s);
	syms.append(files->globalsymbols()->Nc());

	if (node.FindValue("reparametrization_rules"))
		Reduze::read(node["reparametrization_rules"], reparametrization_rules_,
				syms);
	if (node.FindValue("toggle_metric_convention"))
		node["toggle_metric_convention"] >> toggle_metric_convention_;
	if (node.FindValue("series_expansion")) {
		const YAML::Node& n = node["series_expansion"];
		if (n.size() > 0) {
			Reduze::read(n[0], series_parameter_, syms);
			Reduze::read(n[1], series_point_, syms);
			n[2] >> series_order_;
		}
	}
}

void Export::init() {
	if (input_filename_.empty() || output_filename_.empty())
		ABORT("Input or output file undefined");
	if (output_filename_ == input_filename_)
		ABORT("Input and output files are identical: " << input_filename_);
	OutFileData::default_filename_suffix(output_format_); // format check
}

std::string Export::get_description() const {
	std::stringstream s;
	s << "export file " << short_filename(input_filename_);
	return s.str();
}

}
