/*  job.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef JOB_H_
#define JOB_H_

#include <string>
#include <list>
#include <set>
#ifdef HAVE_MPI
#include "mpi.h"
#endif

#include "yamlconfigurable.h"

namespace YAML {
class Node;
class Emitter;
}

namespace Reduze {

/// Task which can be performed after its dependencies are satisfied
/** By default, a Job is configured for serial execution. Derive from
 *  DistributedJob to supply a parallizable Job.
 *  In addition to the pure abstract methods defined below a Job
 *  must implement some static members like keyword() and register to the
 *  JobFactory, see e.g. job_catfiles.h and job_catfiles.cpp.
 *  A Job is not intended to be copied.
 */
class Job: public YAMLAutoConfigurable {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec o;
		o.add_option("conditional", false, "boolean",
				"Whether to skip the job if all output files are already present.");
		return o;
	}

	/// status of a job
	enum Status {
		Pending, Running, Skipped, Done
	};

	/// default constructor
	Job() :
		job_id(0),//
		conditional_(false)
#ifdef HAVE_MPI
	, jobcenter_rank_(-1), manager_rank_(-1), communicator_(0)
#endif
	{
		add_auto_io("conditional", conditional_);
	}
	/// destructor
	virtual ~Job() {
	}

	/// returns a description for the specific job
	virtual std::string get_description() const = 0;

	/// returns a text representation for logging purposes
	/** max_id can be given to fill in spaces to nicely format lists **/
	std::string to_short_string(unsigned max_id = 0) const;

	/// performs the job autonomously
	virtual void run_serial();

#ifdef HAVE_MPI
	virtual void run_manager(MPI::Intracomm* comm, int jobcenter_rank);
	virtual void run_worker(MPI::Intracomm* comm, int manager_rank);
	virtual void set_communication_params(MPI::Intracomm* comm, int jobcenter_rank, int manager_rank);
	/// returns the rank of the manager for this job
	int manager_rank() const;
#endif

	/// sets an id for the job
	void set_id(unsigned);

	/// returns a id for the job
	int id() const;

	/// returns the minimal number of workers needed for this job
	virtual int min_workers() const {
		return 0;
	}

	/// returns the maximal number of workers allowed for this job
	virtual int max_workers() const {
		return 0;
	}

	/// finds input and output files, create auxiliary jobs if necessary
	/** \param output_others names of output files of preceeding pending jobs
	 ** \param input         names of input files to be used by this job
	 ** \param output        names of output files provided by this job
	 ** \param aux_jobs      auxiliary jobs created on the heap by the method
	 **                      (job queue will take ownership and delete them)
	 ** \return              whether all input and output file could be
	 **                      determined (job queue will call this method again
	 **                      at a later time if and only if false) **/
	virtual bool find_dependencies(//
			const std::set<std::string>& output_others,//
			std::list<std::string>& input,//
			std::list<std::string>& output,//
			std::list<Job*>& aux_jobs) = 0;

	/// sets whether job should only be run if any of the output files are missing
	virtual void set_conditional(bool flag);

	/// returns whether job must be run only if outputfiles are not present
	bool is_conditional() const;

protected:
	// hide copy constructor
	Job(const Job&);

private:
	int job_id;
	bool conditional_;
#ifdef HAVE_MPI
	int jobcenter_rank_, manager_rank_;
	MPI::Intracomm* communicator_;
#endif
};

/// Job which may be executed in a distributed way employing worker processes
class DistributedJob: public Job {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec o(Job::yaml_spec());
		o.add_option("min_workers", false, "integer",
				"Minimal number of required workers (in addition to employer).");
		o.add_option("max_workers", false, "integer",
				"Maximal number of allowed workers (in addition to employer).");
		return o;
	}

	DistributedJob() :
		Job(), min_workers_(1), max_workers_(1000) {
		add_auto_io("min_workers", min_workers_);
		add_auto_io("max_workers", max_workers_);
	}

	/// sets a range for the minimal and maximal number of workers
	virtual void set_num_workers_range(unsigned min, unsigned max);

	/// returns the minimal number of workers needed for this job
	virtual int min_workers() const;

	/// returns the maximal number of workers allowed for this job
	virtual int max_workers() const;

protected:
	DistributedJob(const DistributedJob&);
	//	DistributedJob(const DistributedJob& o) :
	//		Job(o), min_workers_(o.min_workers_), max_workers_(o.max_workers_) {
	//	}

private:
	int min_workers_, max_workers_;
};

template<class T>
class JobProxy: public YAMLProxy<T> {
};

/// finds dependencies on all sectormappings, creates aux jobs if necessary
/** this is intendended as a helper method for implementations of find_dependencies(),
 ** return true if no sector mappings file in output_of_others found **/
bool find_dependencies_all_sectormappings(
		const std::set<std::string>& output_of_others,
		std::list<std::string>& input, std::list<Job*>& auxiliary_jobs);

/// finds dependencies on all preceeding reductions
/** this is intendended as a helper method for implementations of find_dependencies() **/
void find_dependencies_reductions(
		const std::set<std::string>& output_of_others,
		std::list<std::string>& input);

}
// namespace Reduze

#endif /* JOB_H_ */
