/*  identitygenerator.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef IDENTITYGENERATOR_H_
#define IDENTITYGENERATOR_H_

#include "ginac/ginac.h"
#include "yamlconfigurable.h"
#include "equationlist.h" /* STL container with IdentityGenericList */
#include "shiftoperators.h"  /* STL container with OPSUM */

class Propagator;

namespace Reduze {

class IntegralFamily;
class INT;
class LinearCombination;
class Identity;
class IdentityList;
class IdentityGeneric;

class IdentityGeneratorOptions: public YAMLConfigurable {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("identity_generator_options");
		s.set_short_description("Options for the equation generator.");
		s.set_long_description(""//
					"Options for the equation generator.");
		s.add_option("transform_integrals_to_equivalent", false, "boolean", ""//
					"Whether to replace integrals by lowest equivalent"
					" according to permutation symmetries.");
		s.add_option("solve_equations", false, "boolean", "" //
						"Whether to solve equations for highest integral.");
		s.add_option("set_subsectors_to_zero", false, "boolean", ""//
					"Whether to set sub-sector integrals to zero"
					" (equations are only valid modulo sub-sectors then).");
		s.add_option("eliminate_subsectors_by_shift", false, "boolean", ""//
					"Whether to eliminate sub-sectors by known shift relations."
					" If set to true the sub-sector integrals are always replaced"
					" by their minimal equivalent.");
		s.add_option("map_on_coeff", false, "sequence of 2-element sequences",
				""//,
					"Substitutions to perform in coefficients."
					" Each element of the sequence is of the form"
					" [SYMBOLTOREPLACE, EXPRESSIONTOREPLACEWITH], e.g. [ [m, 1] ]."
					" The symbol \"symbol_to_replace_by_one\" defined in the"
					" kinematics file is set to one anyway.");
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	IdentityGeneratorOptions() :
		transform_integrals_to_equivalent_(true), /**/
		solve_equations_(true), /* b.c. of MMA output */
		set_subsectors_to_zero_(false), /* otherwise conflict when setting eliminate_subsectors_by_shift_ */
		eliminate_subsectors_by_shift_(true), /**/
		map_on_integral_(0) {
		init();
	}
	virtual ~IdentityGeneratorOptions() {
	}

	void set_map_on_coeff(const GiNaC::exmap& m);
	void set_transform_integrals_to_equivalent(bool b);
	void set_solve_equations(bool b);
	void set_subsectors_to_zero(bool b);
	void set_eliminate_subsectors_by_shift(bool b);
	virtual void print(YAML::Emitter& os) const;
	virtual void read(const YAML::Node&);

private:
	/// substitutions applied on the coefficients
	GiNaC::exmap map_on_coeff_;
	/// whether integrals should be replaced by the minimal equivalents
	bool transform_integrals_to_equivalent_;
	/// divides all coefficients by the coeff. of the most complicated integral
	bool solve_equations_;
	/// whether sub-sector integrals should be set to zero
	bool set_subsectors_to_zero_;
	/// replace integrals by integrals of a simpler sector, if shifts are available
	bool eliminate_subsectors_by_shift_;

	friend class IdentityGenerator;

private:
	void init();
	INT (*map_on_integral_)(const INT&);
};

inline void operator>>(const YAML::Node& n, IdentityGeneratorOptions& r) {
	r.read(n);
}
inline YAML::Emitter& operator<<(YAML::Emitter& ye,
		const IdentityGeneratorOptions& r) {
	r.print(ye);
	return ye;
}

//
//
//

class GenericIdentityGenerator;
class ShiftGenerator;

/// generator for identities (integration-by-parts, lorentz, shift, symmetry)
class IdentityGenerator {
public:
	IdentityGenerator() {
	}
	IdentityGenerator(const IdentityGeneratorOptions& options) :
		options_(options) {
	}
	virtual ~IdentityGenerator() {
	}

	/// generates identities from integral 'in' of type 'type'
	/** equations are appended to out,
	 ** valid types are ibp, ibp_dim, ibp_dim_free, ibp_lee, lorentz,
	 ** symbol_to_replace_by_one is replaced by one **/
	void generate(const INT& in, IdentityList& out, const std::string& type);
	/// generates identities from the integrals in 's'
	/** equations are appended to out
	 ** symbol_to_replace_by_one is replaced by one */
	void generate(const std::set<INT>& s, IdentityList& out,
			const std::string& type);

	/// returns the shift identity for the integral
	/** removes zero-sector integrals, employs global permutation symmetry
	 ** and shift relations */
	static Identity get_shift_identity(const INT& integral,
			bool replace_symbol_to_replace_by_one, bool set_subsecs_to_zero);

private:
	/// Options for generating the identities
	IdentityGeneratorOptions options_;
	/// generate shift identities
	void genid_shift(const INT& in, IdentityList& out, bool no_options = false);
	/// generate shift identities
	void genid_shift(const std::set<INT>& in, IdentityList& out);
	/// generate symmetry identities
	void genid_symmetry(const INT& in, IdentityList& out, bool no_options =
			false);
	/// generate the IBPs and LIs
	void genid_ibp_lorentz(const INT& in, IdentityList& out,
			const std::string& type);

	/// apply remaining options
	/** applies all options on the identities except "set_subsectors_to_zero_"
	 ** and "eliminate_subsectors_by_shift_" */
	void apply_remaining_options(IdentityList& idlist) const;

	/// access and construction of the generic equations
	/** valid types are: ibp, ibp_lee, ibp_dim, ibp_dim_free, lorentz */
	static const LinearCombinationGenericList& generic_equations(
			const IntegralFamily* ic, const std::string& type);
	/// the generic IBP generators by the integral family name
	static std::map<std::string, GenericIdentityGenerator> generic_generators_;
	static ShiftGenerator shift_generator_;
};

//
//
//

/// generates shift identities based on loop momentum
/** Caches auxiliary expressions for each sector.
 ** Note: in contrast to IBPs we can't generate an explicit generic identity
 ** here. **/
class ShiftGenerator {
public:
	/// returns recursively cleaned shift identity, empty equation if input minimal
	Identity find_shift(const INT&, bool set_symbol_one, bool zero_subsecs);
	/// returns raw shift identity without recursion, assumes input to be minimal wrt permutations
	bool find_raw_shift(const INT&, LinearCombination&);
private:
	const std::pair<Sector, std::vector<OPSUM> >& prepare_shift(const Sector& srcsec);
	// we story also dummy entries for sectors which don't have a shift:
	// target sector equals source sector in this case
	std::map<Sector, std::pair<Sector, std::vector<OPSUM> > > shifts_;
};

/// generator for generic identities (ibp, lorentz)
class GenericIdentityGenerator {
public:
	/// construction generates and/or loads user-defined generic IBP and LI identities
	GenericIdentityGenerator(const IntegralFamily* f);
	virtual ~GenericIdentityGenerator() {
	}

	/// access to the generic equations
	/** valid types are: ibp, ibp_lee, ibp_dim, ibp_dim_free, lorentz */
	const LinearCombinationGenericList& generic_equations( //
			const std::string& type) const;
	/// returns (q_i d/dq_k integral) for ext. mom. q_i, q_k with i,k in [0,NoEM)
	LinearCombination get_derivative_wrt_momentum(const INT& integral,
			int i, int k);
	/// returns (d/da integral) for explicit a-dependencies of propagators
	LinearCombination get_derivative_wrt_parameter(const INT& integral,
			const GiNaC::symbol& a);

private:
	/// the integral family
	const IntegralFamily* integralfamily_;
	///	generic ibps: ibp, ibp_dim, ibp_dim_free, ibp_lee, lorentz
	std::map<std::string, LinearCombinationGenericList> generic_equations_;
	/// the derivatives q_i * d/dq_k in terms of shift operators
	/** Map: [i,k]  ==>  q_i * d/dq_k  with  0 <= i < NoLM+NoEM, 0 <= k < NoLM+NoEM
	 ** The vector q_j is a an element of { k_1, ... , k_NoLM, p_1, ... , p_NoEM }
	 ** where the loop momenta k_m are sorted before the independent external
	 ** momenta p_n **/
	std::map<std::vector<int>, OPSUM> derivative_ops_;

private:
	GenericIdentityGenerator() :
		integralfamily_(0) {
	}
	/// constructs generic identities: ibp, ibp_dim, ibp_dim_free, ibp_lee, lorentz
	void construct_generic_identities();
	/// constructs shift operators for creating generic identities
	/** determine effect of q_i * d/dq_k on general integrand,
	 ** expressed in terms of shift operators **/
	void construct_shift_operators();
};

/// transforms combinations of Propagators to OPSUMs
OPSUM ex_to_OPSUM(const GiNaC::ex& e, const std::vector<Propagator>& props);

} // namespace Reduze


#endif /* IDENTITYGENERATOR_H_ */
