/*  identitygenerator.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "identitygenerator.h"
#include "shiftoperators.h"
#include "propagator.h"
#include "kinematics.h"
#include "integralfamily.h"
#include "sectormappings.h"
#include "functions.h"
#include "files.h"
#include "equation.h"
#include "equationlist.h"
#include "ginacutils.h"
#include "yamlutils.h"
#include "filedata.h"

using namespace std;
using namespace GiNaC;

namespace Reduze {

// register type
namespace {
YAMLProxy<IdentityGeneratorOptions> dummy;
}

// TODO: remove this once we moved symmetry generation to new scheme (ShiftGenerator)
/// applies the shift on the momenta of the propagators and tries to match the result with the integral family 'ic'
// multiplies result with the integer 'mult'
LinearCombination shift_INT(const INT& si, const GiNaC::exmap& shift,
		const IntegralFamily* f, int mult, bool set_subsec_to_zero) {
	const exmap& sp2prop = f->rules_sp_to_prop();
	const exmap& sp2inv = f->kinematics()->rules_sp_to_invariants();
	const vector<Propagator>& props = si.integralfamily()->propagators();
	const unsigned num_props = props.size();
	ASSERT(num_props == si.v().size());
	ex integrand = 1;
	for (unsigned i = 0; i < num_props; ++i) {
		int vi = static_cast<int> (si.v_i(i));
		if (vi > 0) {
			ex pishift = props[i].subs(shift);
			integrand *= pow(pishift, vi);
		} else if (vi < 0) {
			ex pishiftinv = props[i].inverse_ex().subs(shift).expand().subs(sp2prop).subs(sp2inv);
			integrand *= pow(pishiftinv, -vi);
		}
	}
	LinearCombination result;
	result.set_name("shift_identity");
	try {
		integrand *= mult;
		propagator_products_to_integrals(integrand.expand(), result, f);
	} catch (exception& e) {
		stringstream ss;
		ss << "Shifted integral cannot be matched to integral family "
				<< f->name() << "\n";
		ss << "shift:    " << shift << "\nintegral: " << si;
		ss << "\n" << e.what();
		//ABORT(ss.str()); // \todo abort ?
		throw runtime_error(ss.str());
	}

	if (set_subsec_to_zero) {
		LinearCombination tmp_result;
		tmp_result.swap_terms(result);
		LinearCombination::const_iterator it;
		for (it = tmp_result.begin(); it != tmp_result.end(); ++it) {
			if (!(it->first.t() < static_cast<int> (si.t())))
				result.insert(it->first, it->second);
		}
	}
	// result.remove_zero_integral();
	return result;
}


// IdentityGeneratorOptions


void IdentityGeneratorOptions::set_map_on_coeff(const GiNaC::exmap& m) {
	map_on_coeff_ = m;
}
void IdentityGeneratorOptions::set_transform_integrals_to_equivalent(bool b) {
	transform_integrals_to_equivalent_ = b;
	if (transform_integrals_to_equivalent_)
		map_on_integral_ = &SectorMappings::get_equivalent_or_unique_zero;
	else
		map_on_integral_ = 0; // has been set to non zero value in init()
}
void IdentityGeneratorOptions::set_solve_equations(bool b) {
	solve_equations_ = b;
}
void IdentityGeneratorOptions::set_subsectors_to_zero(bool b) {
	set_subsectors_to_zero_ = b;
	if (set_subsectors_to_zero_)
		eliminate_subsectors_by_shift_ = false;
}
void IdentityGeneratorOptions::set_eliminate_subsectors_by_shift(bool b) {
	eliminate_subsectors_by_shift_ = b;
	if (set_subsectors_to_zero_)
		eliminate_subsectors_by_shift_ = false;
}
void IdentityGeneratorOptions::init() {
	if (transform_integrals_to_equivalent_)
		map_on_integral_ = &SectorMappings::get_equivalent_or_unique_zero;
}

void IdentityGeneratorOptions::print(YAML::Emitter& os) const {
	using namespace YAML;
	os << BeginMap;
	os << Key << "transform_integrals_to_equivalent" << Value
			<< transform_integrals_to_equivalent_;
	os << Key << "solve_equations" << Value << solve_equations_;
	os << Key << "set_subsectors_to_zero" << Value << set_subsectors_to_zero_;
	os << Key << "eliminate_subsectors_by_shift" << Value
			<< eliminate_subsectors_by_shift_;
	os << Key << "map_on_coeff" << Value << Flow << map_on_coeff_;
	os << EndMap;
}

void IdentityGeneratorOptions::read(const YAML::Node& node) {
	verify_yaml_spec(node);
	const YAML::Node* n;
	if ((n = node.FindValue("transform_integrals_to_equivalent")))
		*n >> transform_integrals_to_equivalent_;
	if ((n = node.FindValue("solve_equations")))
		*n >> solve_equations_;
	if ((n = node.FindValue("set_subsectors_to_zero")))
		*n >> set_subsectors_to_zero_;
	if ((n = node.FindValue("eliminate_subsectors_by_shift")))
		*n >> eliminate_subsectors_by_shift_;
	if ((n = node.FindValue("map_on_coeff"))) {
		lst l = Files::instance()->kinematics()->kinematic_invariants_and_dimension();
		Reduze::read(*n, map_on_coeff_, l);
	}
	init();
}

// IdentityGenerator

std::map<std::string, GenericIdentityGenerator>
		IdentityGenerator::generic_generators_;
ShiftGenerator IdentityGenerator::shift_generator_;

void IdentityGenerator::generate(const INT& in, IdentityList& out,
		const string& type) {
	IdentityList identities;
	if (type == "sector_relations")
		genid_shift(in, identities);
	else if (type == "sector_symmetries")
		genid_symmetry(in, identities);
	else
		genid_ibp_lorentz(in, identities, type);

	if (options_.eliminate_subsectors_by_shift_) {
		set<INT> ints;
		identities.find_subsector_INT(static_cast<int> (in.t()), ints);
		IdentityList replace;
		genid_shift(ints, replace); // note: permutation symmetries have been applied, currently we do it again (which is a waste)
		identities.replace(replace);
	}
	out.splice(identities);
}

void IdentityGenerator::generate(const set<INT>& s, IdentityList& out,
		const string& type) {
	set<INT>::const_iterator it;
	for (it = s.begin(); it != s.end(); ++it)
		generate(*it, out, type);
}

void IdentityGenerator::genid_ibp_lorentz(const INT& in, IdentityList& result,
		const string& type) {
	const IntegralFamily* f = in.integralfamily();
	const LinearCombinationGenericList& eqs = generic_equations(f, type);
	IdentityList identities;
	LinearCombinationGenericList::const_iterator it;
	for (it = eqs.begin(); it != eqs.end(); ++it) {
		LinearCombination lincomb = it->get_linear_combination(in,
				options_.set_subsectors_to_zero_);
		identities.push_back(Identity());
		identities.back().swap_terms(lincomb);
	}
	apply_remaining_options(identities);
	result.splice(identities);
}

Identity IdentityGenerator::get_shift_identity(const INT& integin,
		bool replace_symbol_to_replace_by_one, bool set_subsecs_to_zero) {

	Identity res;

#ifndef USE_OLD_SHIFTS
	return shift_generator_.find_shift(integin, replace_symbol_to_replace_by_one, set_subsecs_to_zero);
#else

// old slow but well tested implementation:

	// minimal crossing, permutation symmetries, obvious zero
	INT imin = SectorMappings::get_equivalent_or_unique_zero(integin);

	LinearCombination rhs;

	// check if we find a shift to uncrossed version of same sector
	bool found_shift = false;
	if (imin.integralfamily()->is_crossed()) {
		const SectorMappings* m = Files::instance()->sectormappings(
				imin.integralfamily()->name());
		map<int, Sector>::const_iterator target = m->sector_relations().find(
				imin.id());
		if (target != m->sector_relations().end()) {
			const exmap& shift =
					m->sector_relation_shifts().find(imin.id())->second;
			rhs = shift_INT(imin, shift, target->second.integralfamily(), -1,
					set_subsecs_to_zero);
			found_shift = true;
		}
	}

	// search shift (crossed families inherit from uncrossed ones)
	if (!found_shift) {
		pair<INT, Crossing> unx = Crossing::uncross(imin);
		INT inox = unx.first;
		const string famnox = inox.integralfamily()->name();
		const SectorMappings* m = Files::instance()->sectormappings(famnox);
		map<int, Sector>::const_iterator target = //
				m->sector_relations().find(inox.id());
		if (target != m->sector_relations().end()) {
			const exmap& shift =
					m->sector_relation_shifts().find(inox.id())->second;
			LinearCombination rhsnox = shift_INT(inox, shift,
					target->second.integralfamily(), -1, set_subsecs_to_zero);
			// cross before recursion to find minimal cross in case of chained crossings
			rhs = unx.second.transform(rhsnox);
			found_shift = true;
		}
	}
	// also provide other simplifications
	if (!found_shift) {
		rhs.insert(imin, -1, false);
	}
	rhs.remove_zero_integral();

	// set scale to 1 (at this late point to work correctly for crossings)
	if (replace_symbol_to_replace_by_one) {
		const Kinematics* kin = integin.integralfamily()->kinematics();
		const exmap& s2o = kin->get_rule_symbol_to_replace_by_one();
		rhs.apply(s2o, subs_options::no_pattern);
	}

	if (found_shift) {
		IdentityList replace;
		for (LinearCombination::const_iterator t = rhs.begin(); t != rhs.end(); ++t)
			replace.push_back(get_shift_identity(t->first,
					replace_symbol_to_replace_by_one, set_subsecs_to_zero));
		for (IdentityList::const_iterator r = replace.begin(); r != replace.end(); ++r)
			rhs.replace(*r);
	}

	// build up identity
	Identity outeq;
	// rhs: minus the shifted expression
	outeq.swap_terms(rhs);
	// lhs: the original integral
	outeq.insert(integin, 1);

	return outeq;
#endif
}

void IdentityGenerator::genid_shift(const INT& integral, IdentityList& out,
		bool no_options) {
	IdentityList ident;
	ident.push_back(get_shift_identity(integral, true, options_.set_subsectors_to_zero_));
	if (!no_options)
		apply_remaining_options(ident);
	if (!ident.back().empty())
		out.splice(ident);
}

void IdentityGenerator::genid_shift(const set<INT>& in, IdentityList& out) {
	IdentityList identities;
	set<INT>::const_iterator it;
	for (it = in.begin(); it != in.end(); ++it)
		genid_shift(*it, identities, true);
	apply_remaining_options(identities);
	out.splice(identities);
}

void IdentityGenerator::genid_symmetry(const INT& integral, IdentityList& out,
		bool no_options) {
	IdentityList identities;
	int id = integral.id();
	const SectorMappings* m = Files::instance()->sectormappings(
			integral.integralfamily()->name());
	const map<int, list<exmap> >& all_shifts = m->sector_symmetries();
	map<int, list<exmap> >::const_iterator it = all_shifts.find(id);
	if (it == all_shifts.end())
		return;
	const list<exmap>& shifts = it->second;
	for (list<exmap>::const_iterator s = shifts.begin(); s != shifts.end(); ++s) {
		// rhs
		LinearCombination lc = shift_INT(integral, *s,
				integral.integralfamily(), -1,
				options_.set_subsectors_to_zero_);
		// set scale to 1
		const Kinematics* kin = integral.integralfamily()->kinematics();
		const exmap& s2o = kin->get_rule_symbol_to_replace_by_one();
		lc.apply(s2o, subs_options::no_pattern);
		// rhs
		identities.push_back(Identity());
		identities.back().swap_terms(lc);
		// lhs
		identities.back().insert(integral, 1);
	}

	if (!no_options)
		apply_remaining_options(identities);

	out.splice(identities);
}

const LinearCombinationGenericList& IdentityGenerator::generic_equations(
		const IntegralFamily* f, const std::string& type) {
	map<string, GenericIdentityGenerator>::const_iterator g;
	g = generic_generators_.find(f->name());
	if (g == generic_generators_.end()) {
		generic_generators_.insert(make_pair(f->name(),
				GenericIdentityGenerator(f)));
		return generic_equations(f, type);
	}
	return g->second.generic_equations(type);
}

void IdentityGenerator::apply_remaining_options(IdentityList& identities) const {
	if (options_.map_on_integral_)
		identities.transform_integrals(options_.map_on_integral_);
	if (!options_.map_on_coeff_.empty())
		identities.apply(options_.map_on_coeff_);
	identities.remove_zero_integral();
	if (options_.solve_equations_)
		identities.solve();
	const Kinematics* kin = Files::instance()->kinematics();
	const exmap& s2o = kin->get_rule_symbol_to_replace_by_one();
	identities.apply(s2o, subs_options::no_pattern);
}


const pair<Sector, vector<OPSUM> >& ShiftGenerator::prepare_shift(const Sector& sec) {
	//cout << "checking if shift for sec " << sec << endl;
	map<Sector, pair<Sector, vector<OPSUM> > >::iterator pmap = shifts_.find(sec);
	if (pmap != shifts_.end())
		return pmap->second;
	const IntegralFamily* f = sec.integralfamily();
	const SectorMappings* m = Files::instance()->sectormappings(f->name());
	map<int, Sector>::const_iterator target = m->sector_relations().find(sec.id());
	if (target == m->sector_relations().end()) {
		//cout << "nope " << endl;
		shifts_.insert(make_pair(sec, make_pair(sec, vector<OPSUM>())));
		return shifts_.find(sec)->second;
	}
	const exmap& shift = m->sector_relation_shifts().find(sec.id())->second;
	const exmap& sp2prop = target->second.integralfamily()->rules_sp_to_prop();
	const exmap& sp2inv = f->kinematics()->rules_sp_to_invariants();
	vector<OPSUM> ops;
	ops.reserve(f->propagators().size());
//	cout << "doing " << sec << " -> " << target->second << endl;
	for (size_t i = 0; i < f->propagators().size(); ++i) {
		ex sp = f->propagators()[i].inverse_ex().subs(shift).expand();
		ex props = sp.subs(sp2prop).subs(sp2inv);
		ops.push_back(ex_to_OPSUM(props, target->second.integralfamily()->propagators()));
		//cout << " prop " << i << ": " << ops[i] << endl;
	}
	shifts_.insert(make_pair(sec, make_pair(target->second, ops)));
	return shifts_.find(sec)->second;
}

bool ShiftGenerator::find_raw_shift(const INT& integral, LinearCombination& minusrhs) {
	Sector srcsec = integral.get_sector();
	const pair<Sector, vector<OPSUM> >& pmap = prepare_shift(srcsec);
	if (pmap.second.empty()) // no shift available
		return false;
	OPSUM o(OPPROD(),-1);
	for (size_t i = 0 ; i < integral.size(); ++i)
		o *= pow(pmap.second[i], -(int)integral.v_i(i));
	const IntegralFamily* f = pmap.first.integralfamily();
	LinearCombinationGeneric lcg = INTGeneric(f).shift(o);
	minusrhs = lcg.get_linear_combination(INT(f));
	return true;
}

Identity ShiftGenerator::find_shift(const INT& integin, bool set_sym_one, bool zero_subsecs) {
	INT imin = SectorMappings::get_equivalent_or_unique_zero(integin);
	Identity outeq;
	LinearCombination minusrhs;
	while (true) {
		// standard shift to lower sector
		if (find_raw_shift(imin, minusrhs))
			break;
		// crossed families inherit from uncrossed ones
		pair<INT, Crossing> unx = Crossing::uncross(imin);
		if (find_raw_shift(unx.first, minusrhs)) {
			minusrhs = unx.second.transform(minusrhs);
			break;
		}
		// just conversion to minimal permutation
		if (integin != imin) {
			minusrhs.insert(imin, -1, false);
			break;
		}
		return outeq;
	}
	if (zero_subsecs) {
		LinearCombination tmp;
		tmp.swap_terms(minusrhs);
		LinearCombination::const_iterator it;
		for (it = tmp.begin(); it != tmp.end(); ++it)
			if (!(it->first.t() < static_cast<int> (integin.t())))
				minusrhs.insert(it->first, it->second);
	}
	minusrhs.remove_zero_integral();
	outeq.swap_terms(minusrhs); // minusrhs: minus the shifted expression
	outeq.insert(integin, 1);   // lhs: the original integral

	// set scale to 1 (at this late point to work correctly for crossings)
	if (set_sym_one) {
		const Kinematics* kin = integin.integralfamily()->kinematics();
		const exmap& s2o = kin->get_rule_symbol_to_replace_by_one();
		minusrhs.apply(s2o, subs_options::no_pattern);
	}

	IdentityList replace;
	for (LinearCombination::const_iterator t = minusrhs.begin(); t != minusrhs.end(); ++t)
		replace.push_back(find_shift(t->first, set_sym_one, zero_subsecs));
	for (IdentityList::const_iterator r = replace.begin(); r != replace.end(); ++r)
		minusrhs.replace(*r);
	return outeq;
}


// generic IBP generation

GenericIdentityGenerator::GenericIdentityGenerator(const IntegralFamily* f) :
	integralfamily_(f) {
	ASSERT(f != 0);

	// construct differential operators and default identities
	construct_shift_operators();
	construct_generic_identities();

	// replace default identities by user-defined identities if available
	map<string, LinearCombinationGenericList>::iterator it;
	it = generic_equations_.begin();
	bool have_user_defined_ids = false;
	for (; it != generic_equations_.end(); ++it) {
		string type = it->first;
		LinearCombinationGenericList& eqs = it->second;
		string file = Files::instance()->get_filename_generic_identities(
				f->name(), type);
		if (!is_readable_file(file))
			continue;
		LOG("Loading generic identities of type '" << type
				<< "' for integral family '" << f->name() << "'");
		eqs.clear();
		InFileLinearCombinationsGeneric in(file);
		LinearCombinationGeneric lc;
		while (in.get(lc))
			eqs.push_back(lc);
		in.close();
		have_user_defined_ids = true;
	}
	if (!have_user_defined_ids)
		LOGX("Loaded only default generic IBP and LI identities");

	// test remove me (just a temp hack)
    /*
		it = generic_equations_.begin();
		for (; it != generic_equations_.end(); ++it) {
			string file = "generic_equations_" + integralfamily_->name() + "_"
					+ it->first;
			file += ".m";
			OutFileLinearCombinationsGeneric out(file, "mma");
			LinearCombinationGenericList::const_iterator l;
		for (l = it->second.begin(); l != it->second.end(); ++l)
				out << *l;
			out.finalize();
		}
	*/
}

const LinearCombinationGenericList& GenericIdentityGenerator::generic_equations(
		const std::string& type) const {
	map<string, LinearCombinationGenericList>::const_iterator e;
	e = generic_equations_.find(type);
	if (e == generic_equations_.end())
		ABORT("Unknown generic equations of type " << type);
	return e->second;
}

void GenericIdentityGenerator::construct_shift_operators() {
	// determine effect of q_i d/dq_k on general integrand,
	// expressed in terms of shift operators
	LOGXX("determine differential operators: q_i * d/dq_k * " << INTGeneric(integralfamily_));
	vector<int> v(2, 0);
	const Kinematics* kin = integralfamily_->kinematics();
	const lst& loop_mom = integralfamily_->loop_momenta();
	const lst& iext_mom = kin->independent_external_momenta();
	const lst all_mom = add_lst(loop_mom, iext_mom);
	const vector<Propagator>& propagators = integralfamily_->propagators();
	const exmap& s2i = kin->rules_sp_to_invariants();
	const exmap& s2p = integralfamily_->rules_sp_to_prop();
	for (size_t i = 0; i < all_mom.nops(); i++) {
		for (size_t k = 0; k < all_mom.nops(); k++) {
			ASSERT(is_a<symbol>(all_mom[i]) && is_a<symbol>(all_mom[k]));
			const symbol qi = ex_to<symbol> (all_mom[i]);
			const symbol qk = ex_to<symbol> (all_mom[k]);
			v[0] = i;
			v[1] = k;
			LOGXX("(" << qi << " * d/d" << qk << ") * " << INTGeneric(integralfamily_));
			derivative_ops_[v] = OPSUM();
			for (size_t j = 0; j < propagators.size(); j++) {
				// qi (d/dqk)propj^aj = (aj propj^(aj-1)) * (qi dpropj/dqk)
				ex aj = integralfamily_->propagator_exponents()[j];
				OP yjm = OP(-(j + 1)); // decrements aj by one
				ex qi_dpropj_dqk = propagators[j].derivative_contracted(qi, qk);
				qi_dpropj_dqk = qi_dpropj_dqk.expand();
				int opt = subs_options::algebraic;
				qi_dpropj_dqk = qi_dpropj_dqk.subs(s2p, opt);
				qi_dpropj_dqk = qi_dpropj_dqk.subs(s2i, opt);
				OPSUM opsum(yjm, aj);
				opsum *= ex_to_OPSUM(qi_dpropj_dqk, propagators);
				derivative_ops_[v] += opsum;
			}
			LOGXX("  = " << derivative_ops_[v]);
		}
	}
}

void GenericIdentityGenerator::construct_generic_identities() {
	// references to the data
	LinearCombinationGenericList& ibp /*     */= generic_equations_["ibp"];
	LinearCombinationGenericList& ibp_dim /* */= generic_equations_["ibp_dim"];
	LinearCombinationGenericList& ibp_dim_free =
			generic_equations_["ibp_dim_free"];
	LinearCombinationGenericList& ibp_lee /* */= generic_equations_["ibp_lee"];
	LinearCombinationGenericList& lorentz /* */= generic_equations_["lorentz"];

	const Kinematics* kin = integralfamily_->kinematics();
	const lst& loop_mom = integralfamily_->loop_momenta();
	const lst& iext_mom = kin->independent_external_momenta();
	const exmap& s2i = kin->rules_sp_to_invariants();
	const size_t num_loop_mom = loop_mom.nops();
	const size_t num_iext_mom = iext_mom.nops();
	const lst all_mom = add_lst(loop_mom, iext_mom);

	// the generic integral
	const INTGeneric gint(integralfamily_);

	// generate the ibp_
	// (k,i)  ==>  d/dq_k * q_i  (X) == dim (X) + q_i * d/dq_k (X)
	map<vector<int> , OPSUM> ibp_operators;
	vector<int> v(2, 0), dv(2, 0);
	// derivative:
	for (size_t k = 0; k < num_loop_mom; k++) {
		v[0] = dv[1] = k;
		// multiply:
		for (size_t i = 0; i < num_loop_mom + num_iext_mom; i++) {
			v[1] = dv[0] = i;
			OPSUM opsum = derivative_ops_[dv]; // (i, k)
			if (i == k)
				opsum += OPSUM(OPPROD(), kin->dimension());
			LinearCombinationGeneric equibp = gint.shift(opsum);
			ibp.push_back(equibp);
			if (i != k)
				ibp_dim_free.push_back(equibp);
			else
				ibp_dim.push_back(equibp);
			ibp_operators[v] = opsum;
		}
	}
	if (num_loop_mom > 1) {
		// d/dk_i k_{i+1} , k_{l+1} == k_1, k_i = loop mom  ==> NoLM equations
		for (size_t i = 0; i < num_loop_mom; i++) {
			v[0] = i;
			v[1] = (i + 1) % num_loop_mom;
			ibp_lee.push_back(gint.shift(ibp_operators[v]));
		}
		// d/dk_1 p_j , k_1 = loop mom, p_j ext. mom.   ==> NoEM equations
		for (size_t i = 0; i < num_iext_mom; i++) {
			v[0] = 1;
			v[1] = num_loop_mom + i;
			ibp_lee.push_back(gint.shift(ibp_operators[v]));
		}
		// Sum_{i=0}^{NoLM-1} d/dk_i k_i , k_i = loop mom.   ==> 1 equation
		OPSUM opsum;
		for (size_t i = 0; i < num_loop_mom; i++) {
			v[0] = v[1] = i;
			opsum += ibp_operators[v];
		}
		ibp_lee.push_back(gint.shift(opsum));
	}

	// generic lorentz identities

	//  Akj - Ajk; 0 <= k < NoEM,  k+1 <= j < NoEM
	// 	Akj = Sum_{i=0}^{NoEM-1}  ( p_j*p_i   p_k*d/dp_i   -   p_k*p_i   p_j*d/dp_i    )
	vector<int> v1(2, 0), v2(2, 0);
	for (size_t k = num_loop_mom; k < num_iext_mom + num_loop_mom; k++) {
		v1[0] = k;
		for (size_t j = k + 1; j < num_iext_mom + num_loop_mom; j++) {
			v2[0] = j;
			OPSUM opsum;
			// calculate Akj and Ajk
			for (size_t i = num_loop_mom; i < num_iext_mom + num_loop_mom; i++) {
				ex mom_i = all_mom[i];
				ex mom_j = all_mom[j];
				ex mom_k = all_mom[k];
				int opts = subs_options::algebraic;
				ex pjpi = ScalarProduct(mom_j, mom_i).eval().subs(s2i, opts);
				ex pkpi = ScalarProduct(mom_k, mom_i).eval().subs(s2i, opts);
				if (!freeof(pkpi, iext_mom) || !freeof(pjpi, iext_mom))
					ABORT("Failed to generate lorentz identities\n"
							<< pjpi << '\n' << pkpi << '\n' << s2i);
				v1[1] = v2[1] = i;
				OPSUM Akj = OPSUM(OPPROD(), pjpi) * derivative_ops_[v1];
				OPSUM Ajk = OPSUM(OPPROD(), pkpi) * derivative_ops_[v2];
				opsum += Akj - Ajk;
			}
			lorentz.push_back(gint.shift(opsum));
		}
	}
}

LinearCombination GenericIdentityGenerator::get_derivative_wrt_momentum(
		const INT& integral, int i, int k) {
	VERIFY(*integral.integralfamily() == *integralfamily_);
	vector<int> v(2);
	v[0] = integralfamily_->loop_momenta().nops() + i;
	v[1] = integralfamily_->loop_momenta().nops() + k;
	INTGeneric genint(integral.integralfamily());
	LinearCombinationGeneric genlc = genint.shift(derivative_ops_[v]);
	return genlc.get_linear_combination(integral);
}

LinearCombination GenericIdentityGenerator::get_derivative_wrt_parameter(
		const INT& integral, const GiNaC::symbol& a) {
	LinearCombination res;
    const IntegralFamily* fam = integral.integralfamily();
	size_t n = fam->num_propagators();
	const vector<Propagator>& props = fam->propagators();
	for (size_t i = 0 ; i < n; ++i) {
		if (integral.v_i(i) == 0)
			continue;
		// just in case we allow scalar symbolic factor for momenta some day
		VERIFY(!props[i].scalarproduct().has(a));
		ex coeff = integral.v_i(i) * props[i].squaredmass().diff(a);
		vector<int8> vnew = integral.v();
		++vnew[i];
		res.insert(INT(fam, vnew), coeff, true);
	}
	return res;
}


// application of e to integrand is implied (e integrand = OPSUM integrand)
OPSUM ex_to_OPSUM(const GiNaC::ex& e, const vector<Propagator>& propagators) {
	exmap s;
	OPSUM res; // is zero !!!
	if (is_a<add> (e)) {
		for (size_t i = 0; i < e.nops(); ++i)
			res += ex_to_OPSUM(e.op(i), propagators);
	} else if (is_a<mul> (e)) {
		res = OPSUM(OPPROD(), 1); // is one
		for (size_t i = 0; i < e.nops(); ++i)
			res *= ex_to_OPSUM(e.op(i), propagators);
	} else if (e.has(Propagator(wild(1), wild(2), wild(3)))) {
		ex base, expo;
		if (e.match(pow(Propagator(wild(1), wild(2), wild(3)), wild(4)), s)) {
			base = Propagator(wild(1).subs(s), wild(2).subs(s),//
					wild(3).subs(s));
			expo = wild(4).subs(s);
		} else if (e.match(Propagator(wild(1), wild(2), wild(3)), s)) {
			base = Propagator(wild(1).subs(s), wild(2).subs(s), //
					wild(3).subs(s));
			expo = 1;
		} else {
			ABORT("Unknown structure containing propagator: " << e);
		}
		// check integer expo > 0 or expo < 0
		if (!(expo.info(info_flags::posint) || expo.info(info_flags::negint)))
			ABORT("Failed to parse propagator exponent in " << e);
		int n = ex_to<numeric> (expo).to_int();
		size_t i;
		for (i = 0; i < propagators.size(); ++i)
			if (base.match(propagators[i]))
				break;
		if (i == propagators.size())
			ABORT("Unknown propagator " << base);
		res = OPSUM(OPPROD(), 1);
		for (size_t j = 0; j < static_cast<size_t> (abs(n)); ++j)
			res *= OPSUM(OPPROD(OP(n > 0 ? (i + 1) : -(i + 1))), 1);
	} else if (!e.has(Propagator(wild(1), wild(2), wild(3)))) {
		res = OPSUM(OPPROD(), e);
	} else {
		ABORT("Failed to parse " << e);
	}
	return res;
}

} // namespace Reduze
