/*  files.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "files.h"

#include <fstream>
#include <dirent.h>
#include <errno.h>
#include <sys/stat.h>   // mkdir etc
#include <sys/types.h>  // mkdir etc
#include <ftw.h> // nftw
#include <unistd.h> // unlink etc
#include "functions.h"
#include "integralfamily.h"
#include "sectormappings.h"
#include "kinematics.h"
#include "ginacutils.h"
#include "yamlutils.h"
#include "int.h"
#include "jobqueue.h"
#include "streamutils.h"
#include "feynmanrules.h"
#include "ginac/ginac.h"
#include "globalsymbols.h"

using namespace std;

namespace Reduze {

/// iterator over files in a directory
class DirectoryIterator {
public:
	/// constructs an "end" iterator
	DirectoryIterator() :
			dir_(0), entry_(0) {
	}
	/// constructs a "begin" iterator (opens the directory)
	DirectoryIterator(const char* path) :
			dir_(0), entry_(0) {
		dir_ = opendir(path);
		if (!dir_)
			WARNING("can't open directory " << path);
		else
			++*this; // go to first entry
	}
	/// closes the directory
	virtual ~DirectoryIterator() {
		if (dir_ != 0)  closedir(dir_); // for Mac
	}
	DirectoryIterator& operator++() {
		errno = 0;
		entry_ = readdir(dir_);
		if (errno != 0)
			ABORT("readdir() failure");
		return *this;
	}
	bool operator!=(const DirectoryIterator& rhs) {
		return entry_ != rhs.entry_;
	}
	std::string operator*() {
		if (!entry_)
			ABORT("Illegal DirectoryIterator dereference");
		return std::string(entry_->d_name);
	}
private:
	DirectoryIterator(const DirectoryIterator&);
	DIR *dir_;
	struct dirent *entry_;
};

// Files

Files::Files() :
		kinematics_loaded_(false), uncrossed_intfam_loaded_(false), intfam_loaded_(
				false), crossings_loaded_(false), project_directory_("./"), paths_(
				0), globalsymbols_(0), globaloptions_(0), feynmanrules_(0) {
}

Files::~Files() {
	delete paths_;
	delete globalsymbols_;
	delete globaloptions_;
	delete feynmanrules_;
	map<string, OrderedCrossings*>::iterator c;
	for (c = crossings_.begin(); c != crossings_.end(); ++c)
		delete c->second;
	map<string, IntegralFamily*>::iterator m;
	for (m = integralfamilies_.begin(); m != integralfamilies_.end(); ++m) {
		if (m->second != 0)
			delete m->second;
		m->second = 0;
	}
	for (m = uncrossed_integralfamilies_.begin();
			m != uncrossed_integralfamilies_.end(); ++m) {
		if (m->second != 0)
			delete m->second;
		m->second = 0;
	}
	map<string, Kinematics*>::iterator k;
	for (k = kinematics_.begin(); k != kinematics_.end(); ++k)
		delete k->second;
	map<string, SectorMappings*>::iterator s;
	for (s = sectormappings_.begin(); s != sectormappings_.end(); ++s)
		delete s->second;
}

Files* Files::instance() {
	static Files* files = 0;
	if (files == 0) {
		files = new Files();
	}
	return files;
}

void Files::set_project_directory(const std::string& name) {
	if (name.size() == 0)
		project_directory_ = "./";
	else if (name.size() > 0 && name[name.size() - 1] == '/')
		project_directory_ = name;
	else
		project_directory_ = name + "/";
	int mkdir_rights = S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;
	mkdir(get_tmp_directory().c_str(), mkdir_rights);
	mkdir(get_log_base_directory().c_str(), mkdir_rights);
	mkdir(get_log_run_directory().c_str(), mkdir_rights);
	mkdir(get_reductions_directory().c_str(), mkdir_rights);
	mkdir(get_graph_directory().c_str(), mkdir_rights);
	mkdir(sectormappings_directory().c_str(), mkdir_rights);
	mkdir(crossings_directory().c_str(), mkdir_rights);
	// check for config directory
	if (!is_readable_directory(get_config_directory()))
		ABORT("No config directory found. Expected: " //
				<< get_config_directory());
}

const std::string& Files::project_directory() const {
	return project_directory_;
}

std::string Files::get_filename_global_yaml() const {
	string name = "global.yaml";
	return cat_directory_and_filename(get_config_directory(), name);
}

std::string Files::get_filename_paths() const {
	return get_filename_global_yaml();
}

std::string Files::get_filename_globalsymbols() const {
	return get_filename_global_yaml();
}

std::string Files::get_filename_globaloptions() const {
	return get_filename_global_yaml();
}

std::string Files::get_filename_kinematics() const {
	string name = "kinematics.yaml";
	return cat_directory_and_filename(get_config_directory(), name);
}

std::string Files::get_filename_crossings() const {
	string name = "crossings.yaml";
	return cat_directory_and_filename(crossings_directory(), name);
}

std::string Files::get_filename_integralfamilies() const {
	string name = "integralfamilies.yaml";
	return cat_directory_and_filename(get_config_directory(), name);
}

std::string Files::get_filename_feynmanrules() const {
	string name = "feynmanrules.yaml";
	return cat_directory_and_filename(get_config_directory(), name);
}

std::string Files::get_filename_sectormappings(const std::string& smname) {
	string name = "sectormappings_" + smname + ".yaml";
	load_uncrossed_integralfamilies();
	map<string, IntegralFamily*>::const_iterator found;
	found = uncrossed_integralfamilies_.find(smname);
	if (found == uncrossed_integralfamilies_.end())
		ERROR("Integral family '" << smname << "' does not exist.");
	return cat_directory_and_filename(sectormappings_directory(), name);
}

std::list<std::string> Files::get_filenames_all_sectormappings() {
	load_uncrossed_integralfamilies();
	list<string> res;
	list<IntegralFamily*>::const_iterator it;
	it = ordered_uncrossed_integralfamilies_.begin();
	for (; it != ordered_uncrossed_integralfamilies_.end(); ++it)
		res.push_back(get_filename_sectormappings((*it)->name()));
	return res;
}

std::string Files::get_filename_sectorexternal(const Sector& sector) const {
	stringstream ss;
	ss << "external_" << sector.get_safe_string_for_filename();
	return cat_directory_and_filename(get_reductions_directory(), ss.str());
}

std::string Files::get_filename_sectorexternal_shared() const {
	string s = "external_shared";
	return cat_directory_and_filename(get_reductions_directory(), s);
}

std::string Files::get_filename_sectorreduction(const Sector& sector) const {
	return get_filename_sectorreduction(sector, get_reductions_directory());
}

std::string Files::get_filename_sectorreduction(const Sector& sector,
		const string& dir) const {
	stringstream ss;
	ss << "reduction_" << sector.get_safe_string_for_filename();
	return cat_directory_and_filename(dir, ss.str());

}

std::string Files::get_filename_generic_identities(const std::string& family,
		const std::string& type) const {
	string filename = "generic_identities_" + family + '_' + type;
	return cat_directory_and_filename(get_config_directory(), filename);
}

std::string Files::get_safe_log_filename_for_job(const Job* j) const {
	stringstream ss;
/*	ss << RunIdentification::instance()->string();
	ss << "_j" << setfill('0') << setw(2) << j->id();
	ss << '_' << safe_filename(j->get_description()) << ".log";
*/
	ss << "j" << setfill('0') << setw(2) << j->id();
	ss << '_' << safe_filename(j->get_description()) << ".log";
	return cat_directory_and_filename(get_log_run_directory(), ss.str());
}

std::string Files::get_tmp_directory() const {
	return project_directory_ + "tmp/";
}

std::string Files::get_log_base_directory() const {
	return project_directory_ + "log/";
}

std::string Files::get_log_run_directory() const {
	return get_log_base_directory()
			+ RunIdentification::instance()->long_string() + "/";
}

std::string Files::get_reductions_directory() const {
	return project_directory_ + "reductions/";
}

std::string Files::get_graph_directory() const {
	return project_directory_ + "graphs/";
}

std::string Files::get_graph_directory(const IntegralFamily* ic) const {
	return get_graph_directory() + safe_filename(ic->name()) + "/";
}

std::string Files::get_config_directory() const {
	return project_directory_ + "config/";
}

std::string Files::sectormappings_directory() const {
	return project_directory_ + "sectormappings/";
}

std::string Files::crossings_directory() const {
	return project_directory_ + "sectormappings/";
}

// helper to read YAML configuration from a file
namespace {

template<class T>
void get(T& result, const string& file, const string& key) {
	std::ifstream fin(file.c_str());
	if (!fin)
		throw missing_file_error(string("Can't open file ") + file);

	YAML::Parser parser(fin);
	YAML::Node doc;
	parser.GetNextDocument(doc);

	if (key.empty()) { // feed top_level_node to object
		try {
			result.read(doc);
		} catch (exception& e) {
			throw runtime_error(
					string("Failed to read from file '") + file + "':\n"
							+ e.what());
		}
	} else { // search object's key according to its YAML spec
		if (!doc.FindValue(key))
			throw missing_key_error(string("Failed to read from file '") + file
					+ "':\n" + "key " + key + " does not exist.");
		try {
			result.read(doc[key]);
		} catch (exception& e) {
			ERROR("Failed to read key '" + key + "' from file '"
					+ file + "':\n"	+ e.what());
		}
	}

	fin.close();
	LOGX("read key '" << key << "' in file '" << file << "'");
}

} // unnamed namespace

void Files::load_kinematics() {
	if (kinematics_loaded_)
		return;

	std::string fullfilename = get_filename_kinematics();
	std::ifstream fin(fullfilename.c_str());
	LOGX("Loading kinematics from file " << fullfilename);
	try {
		if (!fin)
			throw runtime_error(string("Can't open file ") + fullfilename);
		YAML::Parser parser(fin);
		YAML::Node doc;
		parser.GetNextDocument(doc);
		const YAML::Node& node = doc["kinematics"];
		verify_is_valid_map_or_non_empty_sequence(node);
		if (node.Type() == YAML::NodeType::Map) {
			// default kinematic (allowed to have empty name)
			Kinematics* kin = new Kinematics();
			kin->read(node);
			kin->set_id(kinematics_.size() + 1);
			kinematics_[kin->name()] = kin;
			default_kinematics_ = kin->name();
		} else {
			std::map<std::string, GiNaC::symbol> shared_symbols;
			for (YAML::Iterator it = node.begin(); it != node.end(); ++it) {
				Kinematics* kin = new Kinematics();
				kin->set_shared_symbols(shared_symbols);
				kin->read(*it);
				kin->set_id(kinematics_.size() + 1);
				shared_symbols = kin->shared_symbols();
				if (kinematics_.find(kin->name()) != kinematics_.end())
					throw std::runtime_error(
							"Kinematics '" + kin->name() + "' already exists.");
				if (it == node.begin())
					default_kinematics_ = kin->name();
				else if (kin->name().empty())
					throw std::runtime_error(
							"Only first defined kinematics (default kinematics) is allowed to have an empty name.");
				kinematics_[kin->name()] = kin;
			}
		}
		fin.close();
	} catch (std::exception& e) {
		throw runtime_error(
				string("Failed to parse ") + fullfilename + ":\n" + e.what());
	}
	kinematics_loaded_ = true;
	LOGX("Loaded " << kinematics_.size() << " kinematics");
}

void Files::load_uncrossed_integralfamilies() {
	if (uncrossed_intfam_loaded_)
		return;
	std::string fullfilename = get_filename_integralfamilies();
	std::ifstream fin(fullfilename.c_str());
	LOGX("Loading uncrossed integral families from file " << fullfilename);
	try {
		if (!fin)
			throw runtime_error(string("Can't open file ") + fullfilename);
		YAML::Parser parser(fin);
		YAML::Node doc;
		parser.GetNextDocument(doc);
		const YAML::Node& node = doc["integralfamilies"];
		if (node.Type() != YAML::NodeType::Sequence)
			throw runtime_error(
					"node is not a sequence " + position_info(node));
		for (YAML::Iterator it = node.begin(); it != node.end(); ++it) {
			IntegralFamily* f = new IntegralFamily(kinematics());
			f->read(*it);
			f->set_id(ordered_uncrossed_integralfamilies_.size() + 1);
			ordered_uncrossed_integralfamilies_.push_back(f);
			if (uncrossed_integralfamilies_.find(f->name())
					!= uncrossed_integralfamilies_.end())
				ERROR("Integral family '" << f->name() << "' already exists.");
			uncrossed_integralfamilies_.insert(make_pair(f->name(), f));
			LOGX("Loaded integral family " << f->name());
		}
		LOGX("Loaded " << ordered_uncrossed_integralfamilies_.size() //
				<< " (uncrossed) integral families");
		fin.close();
	} catch (std::exception& e) {
		throw runtime_error(
				string("Failed to parse ") + fullfilename + ":\n" + e.what());
	}
	uncrossed_intfam_loaded_ = true;
	LOGX("Done loading uncrossed integral families");
}

void Files::setup_crossed_integralfamilies(IntegralFamily* f) {
	f->set_id(ordered_integralfamilies_.size() + 1);
	ordered_integralfamilies_.push_back(f);
	if (integralfamilies_.find(f->name()) != integralfamilies_.end())
		ERROR("Integral family '" << f->name() << "' already exists.");
	integralfamilies_.insert(make_pair(f->name(), f));
	LOGX("Using integral family " << f->name() << " (id reset to " //
			<< f->id() << ")");
   // generate crossed families
	const list<Crossing>& crossings = Files::instance()->crossings(
			f->kinematics()->name())->ordered_crossings();
	list<Crossing>::const_iterator c;
	for (c = crossings.begin(); c != crossings.end(); ++c) {
		CrossedIntegralFamily* cic = new CrossedIntegralFamily(f, *c);
		cic->set_id(ordered_integralfamilies_.size() + 1);
		ordered_integralfamilies_.push_back(cic);
		integralfamilies_.insert(make_pair(cic->name(), cic));
		LOGX("  generated crossed integral family " << cic->name() //
				<< " (id set to " << cic->id() << ")");
	}
}

void Files::load_integralfamilies() {
	if (intfam_loaded_)
		return;
	load_uncrossed_integralfamilies();
	list<IntegralFamily*>::iterator it;
	it = ordered_uncrossed_integralfamilies_.begin();
	for (; it != ordered_uncrossed_integralfamilies_.end(); ++it)
		setup_crossed_integralfamilies(*it);
	intfam_loaded_ = true;
	LOGX("Done loading integral families");
}

void Files::load_crossings() {
	if (crossings_loaded_)
		return;
	const map<string, Kinematics*>& kins = all_kinematics();
	std::string fullfilename = get_filename_crossings();

	if (!is_readable_file(fullfilename)) {
		LOGX("No crossings file defined");
		// construct default crossings for all kinematics
		map<string, Kinematics*>::const_iterator k;
		for (k = kins.begin(); k != kins.end(); ++k) {
			OrderedCrossings crs(k->second);
			crossings_[k->first] = new OrderedCrossings(crs);
			LOGX("Created default crossings for kinematics '" //
					<< k->first << "'");
		}
		crossings_loaded_ = true;
		return;
	}

	std::ifstream fin(fullfilename.c_str());
	LOGX("Loading crossings from file " << fullfilename);
	try {
		YAML::Parser parser(fin);
		YAML::Node doc;
		parser.GetNextDocument(doc);
		const YAML::Node& node = doc["crossings"];
		if (node.Type() != YAML::NodeType::Sequence)
			throw runtime_error(
					"node is not a sequence " + position_info(node));
		for (YAML::Iterator it = node.begin(); it != node.end(); ++it) {
			string kin_name;
			(*it)["name"] >> kin_name;
			if (crossings_.find(kin_name) != crossings_.end())
				throw runtime_error(
						"Crossings for kinematics '" + kin_name
								+ "' already defined.");
			const Kinematics* kin = kinematics(kin_name);
			OrderedCrossings crs(kin);
			crs.read((*it)["ordered_crossings"]);
			crossings_[kin_name] = new OrderedCrossings(crs);
			LOGX("Loaded crossings for kinematics '" << kin_name << "'");
		}
		// set default crossings for remaining kinematics
		map<string, Kinematics*>::const_iterator k;
		for (k = kins.begin(); k != kins.end(); ++k) {
			if (crossings_.find(k->first) == crossings_.end()) {
				OrderedCrossings crs(k->second);
				crossings_[k->first] = new OrderedCrossings(crs);
				LOGX("Created default crossings for kinematics '" //
						<< k->first << "'");
			}
		}
		LOGX("Loaded crossings for " << crossings_.size() << " kinematics");
		fin.close();
	} catch (std::exception& e) {
		throw runtime_error(
				string("Failed to parse ") + fullfilename + ":\n" + e.what());
	}
	crossings_loaded_ = true;
	LOGX("Done loading crossings");
}

Kinematics* Files::kinematics(const std::string& name0) {
    LOGXX("looking for kinematics '" << name0 << "'");
	load_kinematics();
	const string& name = (name0.empty() ? default_kinematics_ : name0);
	map<string, Kinematics*>::const_iterator k;
	k = kinematics_.find(name);
	if (k != kinematics_.end())
		return k->second;
	LOGX("trying to derive kinematics '" << name << "'");
	Kinematics* kin = Kinematics::create_derived_kinematics(name);
	if (kin != 0 && kin->name() == name) {
		kin->set_id(kinematics_.size() + 1);
		kinematics_[name] = kin;
		return kin;
	}
	stringstream ss;
	ss << "\nKnown kinematics:\n";
	map<string, Kinematics*>::const_iterator k1;
	for (k1 = kinematics_.begin(); k1 != kinematics_.end(); ++k1)
		ss << "  '" << k1->second->name() << "'\n";
	ss << "\n";
	throw runtime_error(
			"Encountered unknown kinematics '" + name + "'" + ss.str());
	return 0;
}

const std::map<std::string, Kinematics*>& Files::all_kinematics() {
	load_kinematics();
	return kinematics_;
}

IntegralFamily* Files::integralfamily(const std::string& name) {
	load_integralfamilies();
	map<string, IntegralFamily*>::iterator ic;
	ic = integralfamilies_.find(name);
	if (ic != integralfamilies_.end())
		return ic->second;
	LOGX("trying to derive integral family '" << name << "'");
	IntegralFamily* f = IntegralFamily::create_derived_integralfamily(name);
	if (f != 0)
		LOG("created integral family '" << f->name() << "'");
	if (f == 0 || f->name() != name)
		throw runtime_error("Encountered unknown integral family '" + name + "'");
	ordered_uncrossed_integralfamilies_.push_back(f);
	uncrossed_integralfamilies_.insert(make_pair(f->name(), f));
	map<string, OrderedCrossings*>::const_iterator bc =
			crossings_.find(f->source_integralfamily()->kinematics()->name());
	if (bc == crossings_.end())
		throw runtime_error("Can't find crossings for base kinematics");
	crossings_[f->kinematics()->name()] = new OrderedCrossings(*bc->second);
	setup_crossed_integralfamilies(f);
	ic = integralfamilies_.find(name);
	if (ic == integralfamilies_.end())
	    throw runtime_error("Unable to generate integral family '" + name + "'");
	return ic->second;
}

IntegralFamily* Files::uncrossed_integralfamily(const std::string& name) {
	load_uncrossed_integralfamilies();
	map<string, IntegralFamily*>::iterator ic;
	ic = uncrossed_integralfamilies_.find(name);
	if (ic == uncrossed_integralfamilies_.end())
		throw runtime_error(
				"Encountered unknown (uncrossed) integral family '" + name
						+ "'");
	return ic->second;
}

const IntegralFamily* Files::integralfamily(const IntegralFamily* ic1,
		const IntegralFamily* ic2) {
	using namespace GiNaC;

	if (*ic1 < *ic2) // make *ic2 the lower IntegralFamily
		swap(ic1, ic2);

	// for product of n-loop times tree: return the n-loop family
	if (ic2->loop_momenta().nops() == 0)
		return ic1;
	else if (ic1->loop_momenta().nops() == 0)
		return ic2;

	// non-trivial product families
	ABORT("non-trivial product of integral families not fully implemented yet");
	// note: creating one on the fly is not enough, those integrals can't
	// be read in again from files in next run; also crossings are not handled
	return 0; // suppress compiler warning
}

std::list<IntegralFamily*> Files::integralfamilies() {
	load_integralfamilies();
	return ordered_integralfamilies_;
}

std::list<IntegralFamily*> Files::uncrossed_integralfamilies() {
	load_uncrossed_integralfamilies();
	return ordered_uncrossed_integralfamilies_;
}

SectorMappings* Files::sectormappings(const std::string& name) {
	if (sectormappings_.find(name) == sectormappings_.end()) {
		IntegralFamily* ic = integralfamily(name);
		SectorMappings m(ic);
		if (ic->is_crossed()
				&& dynamic_cast<const CrossedIntegralFamily*>(ic)) {
			m = SectorMappings(static_cast<const CrossedIntegralFamily*>(ic));
		} else if (!is_readable_file(get_filename_sectormappings(name))) {
			LOG("Writing empty sector mappings file for '" << name << "'");
			save(m);
		} else {
			get(m, get_filename_sectormappings(name), "sectormappings");
			LOGX("Found valid sector mappings for " << name);
		}
		// only store if no exception during read
		sectormappings_[name] = new SectorMappings(m);
	}
	return sectormappings_[name];
}

Paths* Files::paths() {
	if (paths_ == 0) {
		Paths p;
		get(p, get_filename_paths(), p.yaml_spec_link().keyword());
		// only store if read was successful (no exception)
		paths_ = new Paths(p);
	}
	return paths_;
}

GlobalSymbols* Files::globalsymbols() {
	if (globalsymbols_ != 0)
		return globalsymbols_;
	string fullfilename = get_filename_globalsymbols();
	GlobalSymbols gs;
	if (is_readable_file(fullfilename)) {
		std::ifstream fin(fullfilename.c_str());
		if (!fin)
			throw runtime_error(string("Can't open file ") + fullfilename);
		try {
			YAML::Parser parser(fin);
			YAML::Node doc;
			parser.GetNextDocument(doc);
			string key = gs.yaml_spec_link().keyword();
			if (doc.FindValue(key)) {
				const YAML::Node& node = doc[key];
				gs.read(node);
				LOGX("Found file with global symbols");
			} else {
				LOGX("Using default global symbols");
			}
			fin.close();
		} catch (std::exception& e) {
			throw runtime_error(
					string("Failed to read ") + " global symbols from file '"
							+ fullfilename + "':\n" + e.what());
		}
	} else {
		LOGX("No file with global symbols found, using default");
	}
	globalsymbols_ = new GlobalSymbols(gs);
	// to be backward compatible: overwrite symbols by symbols defined in Kinematics and FeynmanRules
	globalsymbols_->set_external_symbols();
	return globalsymbols_;
}

GlobalOptions* Files::globaloptions() {
	if (globaloptions_)
		return globaloptions_;
	try {
		GlobalOptions o;
		get(o, get_filename_globaloptions(), o.yaml_spec_link().keyword());
		LOG("Found global_options");
		// only store if read was successful (no exception)
		globaloptions_ = new GlobalOptions(o);
	} catch (missing_file_error& e) {
		LOG(e.what());
		LOG("Failed to read global_options, using default values");
		globaloptions_ = new GlobalOptions();
	} catch (missing_key_error& e) {
		LOG(e.what());
		LOG("Failed to read global_options, using default values");
		globaloptions_ = new GlobalOptions();
	}
	return globaloptions_;
}

FeynmanRules* Files::feynmanrules() {
	if (feynmanrules_)
		return feynmanrules_;
	string file = get_filename_feynmanrules();
	FeynmanRules fr;
	try {
		get(fr, file, fr.yaml_spec_link().keyword());
	} catch (exception& e) {
		throw runtime_error(
				"Failed to read Feynman rules from file " + file + "\n"
						+ e.what());
	}
	feynmanrules_ = new FeynmanRules(fr);
	return feynmanrules_;
}

OrderedCrossings* Files::crossings(const std::string& kin) {
	load_crossings();
	const string& name = (kin.empty() ? default_kinematics_ : kin);
	map<string, OrderedCrossings*>::const_iterator c = crossings_.find(name);
	if (c != crossings_.end())
		return c->second;
	throw runtime_error("No crossing defined for kinematics '" + name + "'");
}

//void Files::read(JobQueue& j, const std::string& filename) {
//	get(j, filename, string(""));
//}

void Files::save_crossings(
		const std::map<std::string, OrderedCrossings>& crs) const {
	using namespace YAML;
	string file_name = get_filename_crossings();
	string file_name_tmp = file_name + ".tmp";
	ofstream file(file_name_tmp.c_str());
	if (!file)
		ABORT("Can't write to " << file_name_tmp);
	Emitter os;
	os << BeginMap;
	os << Key << "crossings" << Value;
	os << BeginSeq;
	map<string, OrderedCrossings>::const_iterator c;
	for (c = crs.begin(); c != crs.end(); ++c) {
		os << BeginMap;
		os << Key << "name" << Value << c->first;
		os << Key << "ordered_crossings" << Value;
		os << c->second;
		os << EndMap;
	}
	os << EndSeq;
	os << EndMap;
	file << os.c_str();
	file.close();
	rename(file_name_tmp, file_name);
}

void Files::save(const SectorMappings& m) {
	if (m.integralfamily()->is_crossed())
		return;
	YAML::Emitter os;
	string file_name = //
			get_filename_sectormappings(m.integralfamily()->name());
	string file_name_tmp = file_name + ".tmp";
	ofstream file(file_name_tmp.c_str());
	if (!file)
		ABORT("Can't write to " << file_name_tmp);
	os << YAML::BeginMap << YAML::Key << "sectormappings" << YAML::Value;
	os << m << YAML::EndMap;
	file << os.c_str();
	file.close();
	rename(file_name_tmp, file_name);
}

//void Files::clear_crossings() {
//	LOGX("Clear crossings cache");
//	map<string, OrderedCrossings*>::iterator c;
//	for (c = crossings_.begin(); c != crossings_.end(); ++c)
//		delete c->second;
//	crossings_.clear();
//	crossings_loaded_ = false;
//	clear_integralfamilies();
//}
//
//void Files::clear_integralfamilies() {
//	LOGX("Clear integral families cache");
//	map<string, IntegralFamily*>::iterator m;
//	for (m = integralfamilies_.begin(); m != integralfamilies_.end(); ++m)
//		delete m->second;
//	integralfamilies_.clear();
//	ordered_integralfamilies_.clear();
//	intfam_loaded_ = false;
//	clear_sectormappings();
//}

bool Files::has_empty_sectormappings_file(const std::string& name) {
	string file = get_filename_sectormappings(name);
	ifstream fin(file.c_str());
	if (!fin)
		ERROR("Cannot open file: " << file);
	YAML::Parser parser(fin);
	YAML::Node doc;
	parser.GetNextDocument(doc);
	const YAML::Node& node = doc["sectormappings"];
	VERIFY(node.Type() == YAML::NodeType::Map);
	for (YAML::Iterator n = node.begin(); n != node.end(); ++n) {
		const YAML::Node& value = n.second();
		if ((value.Type() == YAML::NodeType::Map
				|| value.Type() == YAML::NodeType::Sequence) && value.size() > 0)
			return false;
	}
	return true;
}

bool Files::sectormappings_already_setup(const std::string& name) {
	string f = get_filename_sectormappings(name);
	if (!is_readable_file(f))
		return false; // not readable --> not setup
	if (!has_empty_sectormappings_file(name))
		return true; // not empty file --> already setup
	// file is now readable and empty: the sector mappings have not been setup
	// for all families except the tree level family which still can be set up
	// if there are no crossings defined.
	// However, to test this we had to access and load the crossings, which we
	// are not allowed to, since the crossings could be setup at a later stage by
	// e.g. the job setup_crossing. Also, it almost takes no time to resetup
	// already setup empty tree level families.
	if (crossings_loaded()) {
		IntegralFamily* fam = integralfamily(name);
		if (fam->loop_momenta().nops() == 0
				&& crossings(fam->kinematics()->name())->ordered_crossings().empty())
			return true; // no crossings for the tree 0-sector possible --> already setup
	}
	return false;
}

void Files::remove_empty_sectormappings_files() {
	bool check = crossings_loaded(); // \todo remove check
	list<IntegralFamily*> ics = uncrossed_integralfamilies();
	list<IntegralFamily*>::const_iterator ic;
	for (ic = ics.begin(); ic != ics.end(); ++ic) {
		string name = (*ic)->name();
		const string& sm_file = get_filename_sectormappings(name);
		if (!sectormappings_already_setup(name) && is_readable_file(sm_file)) {
			LOGX("removing (empty) file " << sm_file);
			remove(sm_file);
		}
	}
	if (!check)
		ASSERT(!crossings_loaded());
}

void Files::clear_sectormappings() {
	LOGX("Clear sector mappings cache");
	map<string, SectorMappings*>::iterator s;
	for (s = sectormappings_.begin(); s != sectormappings_.end();)
		delete s->second;
	sectormappings_.clear();
}

void Files::clear_sectormappings(const std::string& name) {
	LOGX("Clear sector mappings and crossed mappings of family '" //
			<< name << "'");
	// delete sector mappings
	map<string, SectorMappings*>::iterator s = sectormappings_.find(name);
	if (s != sectormappings_.end()) {
		delete s->second;
		sectormappings_.erase(s);
	}
	// delete crossed sector mappings
	IntegralFamily* ic = integralfamily(name);
	const std::list<Crossing>& cs = Files::instance()->crossings(
			ic->kinematics()->name())->ordered_crossings();
	for (list<Crossing>::const_iterator c = cs.begin(); c != cs.end(); ++c) {
		string n = c->name_for_crossed_family(ic);
		s = sectormappings_.find(n);
		if (s != sectormappings_.end()) {
			delete s->second;
			sectormappings_.erase(s);
		}
	}
}

void Files::verify_equivalent_sector_mapping_files(
		const std::string& intfam_name, const std::string& secmap1_file,
		const std::string& secmap2_file) {

	// setup two new project directories in the ./tmp directory

	string tmp_dir = Files::instance()->get_tmp_directory();
	string project1_dir = tmp_dir + "project1/";
	string project2_dir = tmp_dir + "project2/";
	make_directory(project1_dir);
	make_directory(project2_dir);
	// delete old files
	remove_directory_with_files(project1_dir);
	remove_directory_with_files(project2_dir);
	// directory itself is not going to be erased, anyway, create them again
	make_directory(project1_dir);
	make_directory(project2_dir);
	string sm1_dir = project1_dir + "sectormappings/";
	string sm2_dir = project2_dir + "sectormappings/";
	make_directory(sm1_dir);
	make_directory(sm2_dir);
	string sm1_file = sm1_dir + "sectormappings_" + intfam_name + ".yaml";
	string sm2_file = sm2_dir + "sectormappings_" + intfam_name + ".yaml";
	copy_file(secmap1_file, sm1_file);
	copy_file(secmap2_file, sm2_file);

	string config1_dir = project1_dir + "config/";
	string config2_dir = project2_dir + "config/";
	make_directory(config1_dir);
	make_directory(config2_dir);

	string kin_file = Files::instance()->get_config_directory()
			+ "kinematics.yaml";
	string kin1_file = config1_dir + "kinematics.yaml";
	string kin2_file = config2_dir + "kinematics.yaml";
	string intfam_file = Files::instance()->get_config_directory()
			+ "integralfamilies.yaml";
	string intfam1_file = config1_dir + "integralfamilies.yaml";
	string intfam2_file = config2_dir + "integralfamilies.yaml";

	copy_file(kin_file, kin1_file);
	copy_file(kin_file, kin2_file);
	copy_file(intfam_file, intfam1_file);
	copy_file(intfam_file, intfam2_file);

	Files f1, f2;
	f1.set_project_directory(project1_dir);
	f2.set_project_directory(project2_dir);

	// reading a Sector from a node reads from Files::instance and this
	// we don't want here. A quick hack is to turn off verification of shifts
	SectorMappings::no_explicit_shift_verification_ = true;
	const SectorMappings* sm1 = f1.sectormappings(intfam_name);
	const SectorMappings* sm2 = f2.sectormappings(intfam_name);
	SectorMappings::no_explicit_shift_verification_ = false;

	const IntegralFamily* fam1 = sm1->integralfamily();
	const IntegralFamily* fam2 = sm2->integralfamily();
	const Kinematics* kin1 = fam1->kinematics();
	const Kinematics* kin2 = fam2->kinematics();

	string error = "Sectormapping files '" + secmap1_file + "' and '"
			+ secmap2_file + "' differ:";
	try {
		sm1->verify_is_equivalent(*sm2);
	} catch (exception& e) {
		throw runtime_error(error + "\n" + e.what());
	}

	const list<Crossing>& cross1 =
			f1.crossings(kin1->name())->ordered_crossings();
	const list<Crossing>& cross2 =
			f2.crossings(kin2->name())->ordered_crossings();
	if (cross1.size() != cross2.size())
		throw runtime_error(error + " mismatch in the number of crossings");
	list<Crossing>::const_iterator c1 = cross1.begin(), c2 = cross2.begin();
	for (; c1 != cross1.end(); ++c1, ++c2) {
		const SectorMappings* csm1 = f1.sectormappings(
				c1->name_for_crossed_family(sm1->integralfamily()));
		const SectorMappings* csm2 = f2.sectormappings(
				c2->name_for_crossed_family(sm2->integralfamily()));
		try {
			csm1->verify_is_equivalent(*csm2);
		} catch (exception& e) {
			throw runtime_error(error + "\n" + e.what());
		}
	}
	remove_directory_with_files(project1_dir);
	remove_directory_with_files(project2_dir);
	remove_empty_dir(f1.get_config_directory());
	remove_empty_dir(f2.get_config_directory());
	remove_empty_dir(f1.get_log_run_directory());
	remove_empty_dir(f2.get_log_run_directory());
	remove_empty_dir(f1.get_log_base_directory());
	remove_empty_dir(f2.get_log_base_directory());
	remove_empty_dir(f1.get_tmp_directory());
	remove_empty_dir(f2.get_tmp_directory());
	remove_empty_dir(f1.crossings_directory());
	remove_empty_dir(f2.crossings_directory());
	remove_empty_dir(f1.sectormappings_directory());
	remove_empty_dir(f2.sectormappings_directory());
	remove_empty_dir(f1.get_reductions_directory());
	remove_empty_dir(f2.get_reductions_directory());
	remove_empty_dir(f1.get_graph_directory());
	remove_empty_dir(f2.get_graph_directory());
	remove_empty_dir(project1_dir);
	remove_empty_dir(project2_dir);
}

// static
std::string Files::family_in_sector_mapping_file(const std::string& filename) {
	using namespace YAML;
	std::ifstream fin(filename.c_str());
	if (!fin)
		ABORT("Can't open file '" + filename + "'");
	try {
		Parser parser(fin);
		Node doc;
		parser.GetNextDocument(doc);
		if (const Node* node = doc.FindValue("sectormappings")) {
			if (const Node* n = node->FindValue("name")) {
				string name;
				*n >> name;
				return name;
			}
		}
	} catch (exception&) {
		return string();
	}
	return string();
}

string get_temporary_dir_name(const string& directory) {
	string res;
	if (directory.size() > 1 && *(directory.rbegin()) == '/') {
		res = directory.substr(0, directory.size() - 1);
		res += ".tmp/";
	} else {
		ostringstream ss;
		ss << "Unable to determine temporary directory name from directory "
				<< directory;
		throw runtime_error(ss.str());
	}
	return res;
}

void make_directory(const std::string& dirname) {
	ASSERT(!dirname.empty());
	int mkdir_rights = S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;
	int e = mkdir(dirname.c_str(), mkdir_rights);
	if (e == 0 || errno == EEXIST) {
		return;
	} else if (errno == ENOENT && dirname[dirname.size() - 1] == '/') {
		string tmp = dirname.substr(0, dirname.size() - 1);
		string subdir = get_directory_of_filename(tmp);
		make_directory(subdir);
		make_directory(dirname);
		return;
	} else {
		throw std::runtime_error(
				"failed to create directory " + dirname + "\nerrno: "
						+ to_string(errno));
	}
}

bool is_readable_file(const string& filename) {
	ifstream is(filename.c_str());
	bool ok = (bool) is;
	is.close();
	return ok && !is_readable_directory(filename);
}

bool are_readable_files(const list<string>& filenames) {
	if (filenames.empty())
		return false;
	list<string>::const_iterator f;
	for (f = filenames.begin(); f != filenames.end(); ++f) {
		if (!is_readable_file(*f))
			return false;
	}
	return true;
}

bool is_readable_directory(const std::string& path) {
	DIR* dir = opendir(path.c_str());
	bool ok = (dir != 0);
	if (dir != 0)  closedir(dir); // for Mac
	errno = 0;
	return ok;
}

void rename(const string& fn1, const string& fn2) {
	int res = std::rename(fn1.c_str(), fn2.c_str());
	if (res != 0)
		ABORT("Failed to rename file from: " << fn1 << " to: " << fn2);
}

void remove(const string& fn) {
	int res = std::remove(fn.c_str());
	if (res != 0)
		ABORT("Failed to remove file: " << fn);
}

void remove(const list<string>& filenames) {
	for (list<string>::const_iterator it = filenames.begin();
			it != filenames.end(); ++it)
		if (is_readable_file(*it))
			remove(*it);
}

void remove_empty_dir(const std::string& dir) {
	if (get_files_from_directory(dir).empty())
		remove_directory_with_files(dir);
}

void remove_empty_dir(const std::list<std::string>& dir) {
	for (list<string>::const_iterator f = dir.begin(); f != dir.end(); ++f)
		remove_empty_dir(*f);
}

std::string get_directory_of_filename(const std::string& filename) {
	size_t pos = filename.find_last_of("/\\");
	if (pos == string::npos) // relative path or empty filename
		return "./";
	return filename.substr(0, pos) + '/';
}

string get_filename_without_directory(const std::string& filename) {
	size_t pos = filename.find_last_of("/\\");
	if (pos == string::npos) // relative path or empty filename
		return filename;
	else
		return filename.substr(pos + 1);
}

string get_filename_without_suffix(const string& filename,
		const string& suffix) {
	if (filename.substr(filename.size() - suffix.length()) == suffix)
		return filename.substr(0, filename.size() - suffix.length());
	return filename;
}

string get_filename_without_suffix(const string& filename) {
	size_t pos = filename.find_last_of(".");
	if (pos == string::npos) // no dot found
		return filename;
	return filename.substr(0, pos);
}

std::string get_canonical_filename(const std::string& filename) {
	/// \todo: support windows
	if (filename.empty())
		return "";
	if (filename[0] != '/' && filename[0] != '.')
		return "./" + filename;
	return filename;
}

std::string short_filename(const std::string& str) {
	const size_t max_len = 50;
	if (str.size() <= max_len)
		return str;
	else
		return "[...]" + str.substr(str.size() - max_len, max_len);
}

std::string safe_filename(const std::string& str) {
	// version with exact substitution of dangerous characters by '_'
	/*
	 const char* allowed =
	 "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789_-.";
	 string filename = str;
	 string::size_type pos = 0;
	 while ((pos = filename.find_first_not_of(allowed, pos)) != string::npos)
	 filename[pos] = '_';
	 */
	// version with no multiple '_':
	string allowed =
			"abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789_-";
	string filename;
	for (string::const_iterator c = str.begin(); c != str.end(); ++c)
		if (allowed.find(*c) != string::npos)
			filename.push_back(*c);
		else if (filename.empty() || filename[filename.size() - 1] != '_')
			filename.push_back('_');
	return filename;
}

std::string cat_directory_and_filename(const std::string& dir,
		const std::string& filename) {
	if (!dir.empty() && dir[dir.size() - 1] != '/')
		return dir + '/' + filename;
	else
		return dir + filename;
}

list<string> get_files_from_directory(const string& dirname) {
	list<string> files;
	for (DirectoryIterator f(dirname.c_str()), end; f != end; ++f)
		if (*f != "." && *f != "..")
			files.push_back(dirname + *f);
	return files;
}

namespace {
extern "C" {
// remove regular file or visited directory, to be called by nftw
int rm_regfile(const char *path, const struct stat *s, int flag,
		struct FTW *f) {
	switch (flag) {
	case FTW_DP: // visited directory if nftw was called with FTW_DEPTH
		return 0;
	case FTW_F: // regular file if nftw was called with FTW_PHYS
		return ::unlink(path);
	default:
		return 1;
	}
}
}
}

bool remove_directory_with_files(const std::string& dirname) {
	const char* dir = dirname.c_str();
	struct stat s;
	stat(dir, &s);
	if (S_ISDIR(s.st_mode) && //
			nftw(dir, rm_regfile, 1 /*depth*/, FTW_DEPTH | FTW_PHYS) == 0 && //
			::rmdir(dir) == 0)
		return true;
	return false;
}

void copy_file(const std::string& from, const std::string& to) {
	string totmp = to + ".tmp";
	ofstream out(totmp.c_str());
	if (!out)
		ABORT("Error opening file: '" << totmp << "'");
	ifstream in(from.c_str());
	if (!in)
		ABORT("Error opening file: '" << from << "'");
	string line;
	while (getline(in, line))
		out << line << "\n";
	out.close();
	in.close();
	rename(totmp, to);
}

void write_header(OutFileData& of) {
	/// \todo: reactivate
	/*
	 of.write_comment(string("Generated by Reduze ") + PACKAGE_VERSION + " on:");
	 of.write_comment(time_to_string());
	 of.write_comment("\n");
	 list<IntegralFamily*> ics = Files::instance()->get_integralfamilies();
	 using namespace YAML;
	 YAML::Emitter ye1, ye2, ye3;

	 ye1 << YAML::BeginMap << Key << "kinematics" << Value << BeginMap
	 << *Files::instance()->kinematics() << EndMap << EndMap;

	 ye2 << BeginMap << Key << "integralfamilies" << Value << BeginSeq;
	 list<IntegralFamily*>::const_iterator ic;
	 for (ic = ics.begin(); ic != ics.end(); ++ic)
	 (*ic)->print(ye2);
	 ye1 << EndSeq << EndMap;

	 of.write_comment(ye1.c_str());
	 of.write_comment("");
	 of.write_comment(ye2.c_str());
	 of.write_comment("");
	 */
}

} // namespace Reduze
