/*  equation.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef REDUZE_EQUATION_H
#define REDUZE_EQUATION_H

#include "ginac/ginac.h"
#include "functions.h"
#include "int.h"

namespace Reduze {

/// gives a normalized form of the expression, must recognize zero
GiNaC::ex normal_form(const GiNaC::ex&);

/// alias for unsigned integer used for integral indices
typedef unsigned int INTIndex;

/// alias for unsigned integer used for coefficient indices
typedef unsigned int CoeffIndex;

// comparison function for e.g. std::string and GiNaC::ex
template<class TC>
inline int reduze_compare(const TC& e1, const TC& e2) {
	return e1.compare(e2);
}

template<>
inline int reduze_compare(const signed char& e1, const signed char& e2) {
	if (e1 == e2)
		return 0;
	return (e1 < e2 ? -1 : 1);
}

template<>
inline int reduze_compare(const int& e1, const int& e2) {
	if (e1 == e2)
		return 0;
	return (e1 < e2 ? -1 : 1);
}

template<>
inline int reduze_compare(const unsigned int& e1, const unsigned int& e2) {
	if (e1 == e2)
		return 0;
	return (e1 < e2 ? -1 : 1);
}

// predeclaration

template<class TI, class TC> class LinCombBasic;
template<class TI, class TC>
std::ostream& operator<<(std::ostream&, const LinCombBasic<TI, TC>&);

/// linear combination of integrals:  coeff1*integral1 + coeff2*integral2 + ...
/** Linear combination may be non-zero, coefficient type need not be algebraic.
 **
 ** General remarks for this and more specific derived classes like Identity:
 ** - zero integrals:
 **   A linear combination may contain integrals which are obviously zero,
 **   see INT::is_zero(). This is needed e.g. if the user explicitely selects
 **   reductions of such integrals. Such integrals can be removed with
 **   remove_zero_integrals().
 ** - zero coefficients:
 **   Generally it is preferred to remove terms with zero coefficients.
 **   However, it might be computationally expensive to detect hidden zeros
 **   or not possible to decide for non-algebraic coefficient types. Therefore
 **   a linear combination is allowed to contain terms with zero coefficients.
 **   Some methods of algebraic linear combinations have an argument
 **   'normalize'. If true, modified coefficients are normalized and terms with
 **   zero coefficients are immediately removed, such that no _new_ zero
 **   coefficients are introduced by the operation. This is useful to
 **   selectively normalize modified terms in case all terms were normalized
 **   before. Methods which generally alter all coefficients typically
 **   perform no normalization at all.
 **   To normalize all coefficients and ensure absence of any (explicit or
 **   hidden) zero coefficient the methods normalize() or solve() (for
 **   equations) should be used.
 **/

template<class TI, class TC>
class LinCombBasic {
public:
	typedef TI integral_type;
	typedef TC coeff_type;

	/// type of the internal implementation of the equation
	typedef std::map<integral_type, coeff_type> terms_type;
	/// a const_iterator over terms, deref gives a pair<integral, coeff>
	typedef typename terms_type::const_iterator const_iterator;

	virtual ~LinCombBasic() {
	}

	const_iterator begin() const {
		return terms.begin();
	}

	const_iterator end() const {
		return terms.end();
	}

	/// returns the size of an equation
	size_t size() const {
		return terms.size();
	}

	/// return true if equation is empty
	bool empty() const {
		return terms.empty();
	}

	/// swaps the terms of the equations
	void swap_terms(LinCombBasic<integral_type, coeff_type> & from) {
		terms.swap(from.terms);
	}

	/// inserts an integral 'in' with corresponding coefficient 'e' in the equation
	/* aborts if integral already inserted */
	void insert(const integral_type & in, const coeff_type & e);
	/// tries first to insert at the beginning
	void insert_begin(const integral_type & in, const coeff_type & e);
	/// tries first to insert at the end
	void insert_end(const integral_type & in, const coeff_type & e);

	/// returns the average length of the prefactors (string)
	double average_coeff_length();

	/// removes all terms from the identity
	void clear() {
		terms.clear();
	}

	/// compares equations
	bool operator <(const LinCombBasic<TI, TC> & other) const;
	/// returns true if *this is equal to other (each integral and each prefactor is equal) and false otherwise
	bool operator ==(const LinCombBasic<TI, TC> & other) const;
	/// returns true if operator==  returns false, false otherwise
	bool operator !=(const LinCombBasic<TI, TC> & other) const;

	/// returns the most complicated integral of the equation
	integral_type max_INT() const {
		if (empty())
			ABORT("request for leading term of empty equation");
		return terms.rbegin()->first;
	}

	/// finds the different integrals in the Identity and inserts them in the set
	template<class TSet>
	void find_INT(TSet& myset) const;

	/// finds all subleading integrals in the Identity and inserts them in the set
	template<class TSet>
	void find_subleading_INT(TSet& myset) const;

	/// returns the name
	const std::string& name() const {
		return name_;
	}

	friend class Identity;
	friend class EquationLight;
	friend class Equation;
	/// prints the linear combination in Reduze format; no newline at the end
	friend std::ostream& operator<<<> (std::ostream& os, const LinCombBasic<TI,
			TC>& lc);

protected:
	/// map containing pairs of integrals and prefactors
	terms_type terms;

	/// sets the name
	void set_name(const std::string& n) {
		name_ = n;
	}

private:
	std::string name_;
};

template<class TI, class TC> class EqAlgebraic;

/// linear combination of integrals with algebraic coefficient type
/** lin. comb. may be non-zero **/
template<class TI, class TC>
class LinCombAlgebraic: public LinCombBasic<TI, TC> {
public:
	typedef TI integral_type;
	typedef TC coeff_type;
	typedef typename LinCombBasic<TI, TC>::terms_type terms_type;
	/// a const_iterator over terms, deref gives a pair<integral, coeff>
	typedef typename terms_type::const_iterator const_iterator;

	virtual ~LinCombAlgebraic() {
	}

	/// inserts an integral 'in' with corresponding coefficient 'e' in the equation
	/** the flag 'normalize' chooses whether to normalize coefficients of
	 ** already contained integrals, coefficients of new integrals are never
	 ** normalized **/
	void insert(const integral_type & in, const coeff_type & e, bool normalize =
			false);

	/// inserts terms specified via iterators
	void insert(const_iterator first, const_iterator last);

	/// brings all coefficients into normal form
	/** detects all zero coefficients and removes corresponding terms,
	 ** for equations it is recommended to use solve() instead **/
	void normalize(bool factorize);

	/// transforms integrals of an equation
	/** the flag 'normalize' chooses whether to normalize coefficients of
	 ** terms which need to be combined, other coefficients are never
	 ** normalized **/
	// would save a lot of special cases but bind1st has problems with
	// pass by const ref argument functors :(
	template<typename IntegralToIntegralOperation>
	void transform_integrals(IntegralToIntegralOperation mapping,
			bool normalize = false);

	/// returns true if a coefficient in the equation contains the symbol 'symb'
	bool has_symbol(const GiNaC::symbol & symb) const;

	/// does a Laurent expansion of the prefactors
	/** order gives the highest order to calculate, order + 1 is neglected */
	void series(const GiNaC::symbol& param, const GiNaC::ex& point, int order);

	/// applies the map 'm' on each coefficient
	/** no normalization of coefficients is performed */
	void apply(const GiNaC::exmap & m, unsigned options = 0);

	/// multiplies with (-1)
	//void to_negative();

	/// replaces the equation eq in the equation
	void replace(const EqAlgebraic<TI, TC> & eq, bool normalize = false);

	/// writes the lin. comb. to stream out in Mathematica format; no newline at the end
	/** writes in form of a rule: name -> a*INT[...] + ... **/
	virtual void to_mma_stream(std::ostream & out) const;

	/// writes the lin. comb. to stream out in Maple format; no newline at the end
	/** writes in form of a definition: name := a*INT[...] + ... **/
	virtual void to_maple_stream(std::ostream & out) const;

	/// writes the lin. comb. to the stream in the FORM format; no newline at the end
	/** writes in form of an id: id name = a*INTA[...] + ... ; **/
	virtual void to_form_stream(std::ostream & out) const;
};

/// linear combination of integrals which is zero
template<class TI, class TC>
class EqAlgebraic: public LinCombAlgebraic<TI, TC> {
public:
	typedef TI integral_type;
	typedef TC coeff_type;
	typedef typename LinCombBasic<TI, TC>::terms_type terms_type;

	virtual ~EqAlgebraic() {
	}

	/// solves an equation for the most complicated integral it contains
	/** sets the coefficient of the most complicated integral equal to one and
	 ** divides the other coefficients by it; all coefficients are normalized
	 ** and zero coefficient terms are removed (no need to normalize() before) */
	void solve(bool factorize = false);

	/// solves an equation for the most complicated integral it contains
	/** similar to solve(), but does not normalize subleading coefficients **/
	void quick_solve();

	/// writes the equation to the stream in the Mathematica format; no newline at the end
	virtual void to_mma_stream(std::ostream & out) const;

	/// writes the equation to the stream in the Maple format; no newline at the end
	virtual void to_maple_stream(std::ostream & out) const;

	/// writes the equation to the stream in the FORM format; no newline at the end
	virtual void to_form_stream(std::ostream & out) const;
};

class EquationLight;
class Equation;
class Identity;

/// equation with indexed coefficients
class EquationXLight: public LinCombBasic<INTIndex, CoeffIndex> {
public:
	/// constructs an empty equation
	EquationXLight() {
	}
};

/// linear combination with string coefficients but full integrals
class LinCombHLight: public LinCombBasic<INT, std::string> {
public:
	/// constructs an empty equation
	LinCombHLight() {
	}
	LinCombHLight(std::istream&, bool discard_coeff = false);
};

/// equation with string coefficients but full integrals
class EquationHLight: public LinCombBasic<INT, std::string> {
public:
	/// constructs an empty equation
	EquationHLight() {
	}
	//EquationHLight(const LinCombBasic<INT, std::string>&);
};

/// equation with light-weight coefficients optimized for message passing
class EquationLight: public LinCombBasic<INTIndex, std::string> {
public:
	/// constructs an empty equation
	EquationLight() {
	}
	// conversion constructors
	EquationLight(const Equation& eq);
	EquationLight(const Identity& eq, const std::map<INT, INTIndex>& m);
};

/// Linear combination of integrals
class LinearCombination: public LinCombAlgebraic<INT, GiNaC::ex> {
public:
	LinearCombination() {
	}
	LinearCombination(std::istream&, const GiNaC::lst& symbols);

	/// substitutes each integral I(t,r,s) by (-1)^(r+s)*I(t,r,s)
	void toggle_metric_convention();

	/// removes the zero integral with r=s=t=0
	void remove_zero_integral();

	/// employs shift relations to substitute integrals
	void apply_shifts(bool normalize = false);

	/// sets the name
	void set_name(const std::string& n) {
		LinCombAlgebraic<INT, GiNaC::ex>::set_name(n);
	}
};

/// Linear combination of generic integrals
class LinearCombinationGeneric: public LinCombAlgebraic<INTGeneric, GiNaC::ex> {
public:
	LinearCombinationGeneric() {
	}
	LinearCombinationGeneric(std::istream& is, const GiNaC::lst& symbols);
	/// replaces the generic propagator exponents by the integer values of the integral
	LinearCombination get_linear_combination(const INT&,
			bool set_sub_secs_to_zero = false) const;
	/// sets the name
	void set_name(const std::string& n) {
		LinCombAlgebraic<INTGeneric, GiNaC::ex>::set_name(n);
	}
};

/// equation with coefficient type GiNaC::ex and integral type INTIndex
class Equation: public EqAlgebraic<INTIndex, GiNaC::ex> {
public:
	/// constructs an empty equation
	Equation() {
	}
	/// converts the equation by using the symbols from the default Kinematics
	Equation(const EquationLight& eq);
	/// converts the equation by using the symbols
	Equation(const EquationLight& eq, const GiNaC::lst& symbols);
	Equation(const Identity& eq, const std::map<INT, INTIndex>& m);
};

class Sector;

/// equation with coefficient type GiNaC::ex and integral type INT
class Identity: public EqAlgebraic<INT, GiNaC::ex> {
public:
	/// constructs an empty equation
	Identity() {
	}
	// conversion constructors
	Identity(const EquationHLight& eq);
	Identity(const EquationLight& eq, const std::map<INTIndex, INT>& m);
	Identity(const Equation& eq, const std::map<INTIndex, INT>& m);
	//Identity(const LinCombAlgebraic<INT, GiNaC::ex>&);

	/// substitutes each integral I(t,r,s) by (-1)^(r+s)*I(t,r,s)
	void toggle_metric_convention();

	/// reconstruct the dependence on the parameter that has been set to one
	void reconstruct_symbol_replaced_by_one();

	/// removes the zero integral with r=s=t=0 for 1 or more loops
	void remove_zero_integral();

	/// replaces the integral family 'old_ic' by the family 'new_ic' and applies the map 'm' on the coefficients
	/** assumes that all integrals of *this belong to the family 'ic_old', otherwise aborts */
	//Identity replace_family_and_coefficients(const IntegralFamily* old_ic,
	//		const IntegralFamily* new_ic, const GiNaC::exmap& m) const;

	/// finds the different integrals with t smaller than "t" in the Identity and inserts them in the set
	template<class TSet>
	void find_subsector_INT(int t, TSet& myset) const;
};

// global functions

// NEEDS an EXPANDED expression, adds to lc (without clearing it before)
void propagator_products_to_integrals(const GiNaC::ex& e, LinCombAlgebraic<INT,
		GiNaC::ex>& lc, const IntegralFamily* ic);

// friend template operator

template<class TI, class TC>
std::ostream& operator<<(std::ostream& os, const LinCombBasic<TI, TC>& lc) {
	typename LinCombBasic<TI, TC>::terms_type::const_reverse_iterator it;
	if (!lc.name_.empty())
		os << "name = " << lc.name_ << "\n";
	for (it = lc.terms.rbegin(); it != lc.terms.rend(); ++it) {
		os << (it == lc.terms.rbegin() ? "  " : "    ") //
				<< it->first << "\n" //
				<< (it == lc.terms.rbegin() ? "   " : "     ") //
				<< it->second << "\n";
	}
	os << ";";
	return os;
}

/// linear combination of integrals with generic exponents
class IdentityGeneric: public EqAlgebraic<INTGeneric, GiNaC::ex> {
public:
	IdentityGeneric() {
	}
};

// methods with additional template parameters

template<class TI, class TC>
template<typename IntegralToIntegralOperation>
void LinCombAlgebraic<TI, TC>::transform_integrals(
		IntegralToIntegralOperation mapping, bool normalize) {
	terms_type oldterms;
	oldterms.swap(this->terms);
	typename terms_type::const_iterator t;
	for (t = oldterms.begin(); t != oldterms.end(); ++t)
		insert(mapping(t->first), t->second, normalize);
}

template<class TI, class TC> template<class TSet>
void LinCombBasic<TI, TC>::find_INT(TSet & myset) const {
	typename terms_type::const_iterator it;
	for (it = terms.begin(); it != terms.end(); ++it)
		myset.insert(it->first);
}

template<class TI, class TC> template<class TSet>
void LinCombBasic<TI, TC>::find_subleading_INT(TSet & myset) const {
	if (terms.end() == terms.begin())
		return;
	typename terms_type::const_iterator it, it_last = --terms.end();
	for (it = terms.begin(); it != it_last; ++it)
		myset.insert(it->first);
}

template<class TSet>
void Identity::find_subsector_INT(int t, TSet& myset) const {
	typename terms_type::const_iterator it;
	for (it = terms.begin(); it != terms.end(); ++it) {
		if (static_cast<int> (it->first.t()) < t)
			myset.insert(it->first);
	}
}

} // namespace Reduze


#endif

